/*-
 * Copyright (c) 1999-2004 Andrey Simonenko
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "config.h"

#ifndef lint
static const char rcsid[] ATTR_UNUSED =
  "@(#)$Id: ipa_conf.c,v 1.4.2.2 2012/07/09 20:28:19 simon Exp $";
#endif /* !lint */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/un.h>

#include <ctype.h>
#include <dirent.h>
#include <errno.h>
#include <fnmatch.h>
#include <limits.h>
#include <regex.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "ipa_mod.h"

#include "queue.h"

#include "dlapi.h"
#include "confcommon.h"
#include "memfunc.h"
#include "parser.h"
#include "pathnames.h"

#include "ipa_ac.h"
#include "ipa_db.h"
#include "ipa_time.h"

#include "ipa_ctl.h"
#include "ipa_cmd.h"
#include "ipa_conf.h"
#include "ipa_log.h"
#include "ipa_main.h"
#include "ipa_rules.h"
#include "ipa_autorules.h"


char		*ipa_conf_file = IPA_CONF_FILE;	/* -f conf_file */

char		mimic_real_config = 0;	/* Set if -tt. */

/* Give limitation on the depth of included files. */
#define INCLUDE_DEPTH_MAX 100
static unsigned int include_depth;

#ifdef CTL_CHECK_CREDS
static regex_t	re_acl;
static regex_t	re_ctl_acl_class;
#endif

/*
 * IPA_CONF_SECT_* with 1..21 values are defined in ipa_mod.h.
 * These are local values, modules should not use them.
 */
#define SECT_SUBLIMIT	22

#ifdef WITH_RULES
static struct rule *currule;		/* Current rule. */
#endif

static struct rulepat *currulepat;	/* Current rulepat. */
static unsigned int rulepatno;

#ifdef WITH_AUTORULES
static struct autorule *curautorule;	/* Current autorule. */
#endif

#ifdef WITH_LIMITS
static struct limit *curlimit;		/* Current limit. */
static unsigned int limitno;		/* Current limit ordinal number. */
static regex_t	re_texp;		/* Regex for PAT_TIME_EXP. */
static struct limits_list *limits_list;
#endif

#ifdef WITH_SUBLIMITS
static struct sublimit *cursublimit;	/* Current sublimit. */
#endif

#ifdef WITH_THRESHOLDS
static struct threshold *curthreshold;	/* Current threshold. */
static unsigned int thresholdno;	/* Current threshold ordinal number. */
static regex_t	re_thr_balance;		/* Regex for "threshold_balance". */
static struct thresholds_list *thresholds_list;
#endif

static struct cmds *curcmds;		/* Current commands section. */

/* Built-in "null" accounting system and database. */
static struct ac_list ac_list_null = STAILQ_HEAD_INITIALIZER(ac_list_null);
static struct db_list db_list_null = STAILQ_HEAD_INITIALIZER(db_list_null);

/* Empty time interval. */
static struct tint_list tint_list_empty =
	STAILQ_HEAD_INITIALIZER(tint_list_empty);

static signed char posix_re_pattern;	/* posix_re_pattern parameter. */
static signed char only_abs_paths;	/* only_abs_paths parameter. */

struct debug_param {
	const char	*name;
	signed char	*value;
};

/* debug_* parameters in IPA_CONF_SECT_ROOT. */
static struct debug_param root_debug_param[] = {
#ifdef WITH_AUTORULES
	{ "debug_autorule",		&debug_autorule			},
#endif
	{ "debug_ctl",			&debug_ctl			},
	{ "debug_time",			&debug_time			},
	{ "debug_worktime",		&debug_worktime			},
	{ "debug_ac_null",		&debug_ac_null			},
	{ "debug_db_null",		&debug_db_null			}
};

#define ROOT_DEBUG_PARAM_TBL_SIZE \
	(sizeof(root_debug_param) / sizeof(root_debug_param[0]))

/* debug_* parameters in IPA_CONF_SECT_GLOBAL. */
static struct debug_param global_debug_param[] = {
#ifdef WITH_LIMITS
	{ "debug_limit",		&global_debug_limit		},
	{ "debug_limit_init",		&global_debug_limit_init	},
#endif
#ifdef WITH_THRESHOLDS
	{ "debug_threshold",		&global_debug_threshold		},
	{ "debug_threshold_init",	&global_debug_threshold_init	},
#endif
	{ "debug_exec",			&global_debug_exec		}
};

#define GLOBAL_DEBUG_PARAM_TBL_SIZE \
	(sizeof(global_debug_param) / sizeof(global_debug_param[0]))

struct time_param {
	const char	*name;
	unsigned int	*value;
};

/* *_time parameters in IPA_CONF_SECT_ROOT. */
static struct time_param root_time_param[] = {
	{ "wakeup_time",		&wakeup_time			},
	{ "freeze_time",		&freeze_time			},
	{ "sleep_after_dump",		&sleep_after_dump		},
	{ "sensitive_time",		&sensitive_time			}
};

#define ROOT_TIME_PARAM_TBL_SIZE \
	(sizeof(root_time_param) / sizeof(root_time_param[0]))

static const struct {
	const char	*str;
	int		val;
} value_units_tbl[] = {
	{ "bytes",			IPA_CONF_TYPE_BYTES		},
	{ "time",			IPA_CONF_TYPE_TIME		},
	{ "number",			IPA_CONF_TYPE_UINT64		},
	{ "any",			0				}
};

#define VALUE_UNITS_TBL_SIZE \
	(sizeof(value_units_tbl) / sizeof(value_units_tbl[0]))

static char	use_log;		/* 0, if -t or -x switch was used. */

static PARSING_MODE parsing_mode;	/* xxx_PARSING. */

static unsigned int section;		/* Current section ID. */
static unsigned int section_first;	/* root autorule rulepat rule */
static unsigned int section_top;	/* root autorule rulepat rule limit
					   sublimit threshold */

static unsigned int section_rc;		/* RC_{STARTUP,SHUTDOWN,NOTSET} */

static char	global_section_set;	/* Set if has global{}. */

static unsigned int conf_event_no;	/* Current section ordinal number. */
static const void *conf_event_arg;	/* Current argument for conf_event(). */

static const unsigned char conf_event_begin[] = {
	0,					/* Does not exist.	*/
	0,					/* Root section.	*/
	IPA_CONF_EVENT_GLOBAL_BEGIN,		/* global{}		*/
	IPA_CONF_EVENT_RULE_BEGIN,		/* rule{}		*/
	IPA_CONF_EVENT_LIMIT_BEGIN,		/* limit{}		*/
	IPA_CONF_EVENT_THRESHOLD_BEGIN,		/* threshold{}		*/
	IPA_CONF_EVENT_AUTORULE_BEGIN,		/* autorule{}		*/
	IPA_CONF_EVENT_RULEPAT_BEGIN,		/* rulepat{}		*/
	IPA_CONF_EVENT_STARTUP_BEGIN,		/* startup{}		*/
	IPA_CONF_EVENT_SHUTDOWN_BEGIN,		/* shutdown{}		*/
	IPA_CONF_EVENT_RESTART_BEGIN,		/* restart{}		*/
	IPA_CONF_EVENT_REACH_BEGIN,		/* reach{}		*/
	IPA_CONF_EVENT_EXPIRE_BEGIN,		/* expire{}		*/
	IPA_CONF_EVENT_IF_REACHED_BEGIN,	/* if_reached{}		*/
	IPA_CONF_EVENT_IF_NOT_REACHED_BEGIN,	/* if_not_reached{}	*/
	IPA_CONF_EVENT_IF_ALL_REACHED_BEGIN,	/* if_all_reached{}	*/
	IPA_CONF_EVENT_IF_ANY_REACHED_BEGIN,	/* if_any_reached{}	*/
	IPA_CONF_EVENT_IF_ANY_NOT_REACHED_BEGIN,/* if_any_not_reached{}	*/
	IPA_CONF_EVENT_IF_ALL_NOT_REACHED_BEGIN,/* if_all_not_reached{}	*/
	IPA_CONF_EVENT_BELOW_THRESHOLD_BEGIN,	/* below_threshold{}	*/
	IPA_CONF_EVENT_EQUAL_THRESHOLD_BEGIN,	/* equal_threshold{}	*/
	IPA_CONF_EVENT_ABOVE_THRESHOLD_BEGIN	/* above_threshold{}	*/
};

static regex_t	re_worktime;		/* Regexp for "worktime". */
static regex_t	re_list;		/* Regexp for any list. */

static void	confvlogmsgx(int, const char *, const char *, va_list)
		    ATTR_FORMAT(printf, 3, 0);

/*
 * Exported support functions for modules.
 */
static const ipa_suppfunc suppfunc = {
	print_string,		/* print_string		*/
	print_bytes,		/* print_bytes		*/
	print_time,		/* print_time		*/
	print_value,		/* print_value		*/
	print_boolean,		/* print_boolean	*/
	print_space,		/* print_space		*/
	mod_print_param_name,	/* print_param_name	*/
	mod_print_args,		/* print_param_args	*/
	mod_print_param_end,	/* print_param_end	*/
	mod_print_sect_name,	/* print_sect_name	*/
	mod_print_args,		/* print_sect_args	*/
	print_sect_begin,	/* print_sect_begin	*/
	mod_print_sect_end,	/* print_sect_end	*/
	open_log,		/* open_log		*/
	close_log,		/* close_log		*/
	mod_logmsg,		/* logmsg		*/
	mod_logconferr,		/* logconferr		*/
	parser_local_sym_add,	/* local_sym_add	*/
	parser_local_sym_del,	/* local_sym_del	*/
	parser_global_sym_add,	/* global_sym_add	*/
	parser_global_sym_del	/* global_sym_del	*/
};

/*
 * Log message prepending it with the prefix.
 * If use_log is set, then log is used, else printf(3) is used.
 * Do not use mem_*() functions in this function.
 */
static void
confvlogmsgx(int priority, const char *prefix, const char *format, va_list ap)
{
	if (use_log) {
		char buf[LOG_BUF_SIZE];
		char *msg;
		int rv;

		rv = vsnprintf(buf, sizeof(buf), format, ap);
		if (rv < 0) {
			logmsg(IPA_LOG_ERR, "confvlogmsgx: vsnprintf failed");
			goto log_unformated;
		}
		if (rv < sizeof(buf)) {
			logmsgx(priority, "%s: %s", prefix, buf);
			return;
		}
		msg = malloc(++rv);
		if (msg == NULL) {
			logmsgx(IPA_LOG_ERR, "confvlogmsgx: malloc failed");
			goto log_unformated;
		}
		if (vsnprintf(msg, rv, format, ap) < 0) {
			logmsg(IPA_LOG_ERR, "confvlogmsgx: vsnprintf failed");
			free(msg);
			goto log_unformated;
		}
		logmsgx(priority, "%s: %s", prefix, msg);
		free(msg);
		return;
log_unformated:
		logmsgx(priority, "%s: unformated message: %s",
		    prefix, format);
	} else {
		fflush(stdout);
		fprintf(stderr, "%s: ", prefix);
		vfprintf(stderr, format, ap);
		fprintf(stderr, "\n");
	}
}

/*
 * The wrapper for parser_vlogmsgx.
 */
static void
parser_vlogmsgx_wrapper(const char *format, va_list ap)
{
	confvlogmsgx(IPA_LOG_ERR, "parsing error", format, ap);
}

/*
 * Wrappers for logging about configuration errors.
 */
void
vlogconfe(const char *format, va_list ap)
{
	confvlogmsgx(IPA_LOG_ERR, "configuration error", format, ap);
}

void
logconfe(const char *format, ...)
{
	va_list ap;

	va_start(ap, format);
	vlogconfe(format, ap);
	va_end(ap);
}

/*
 * The same as above one, but with priority argument and va_list.
 */
static void
vlogconfx_priority(int priority, const char *format, va_list ap)
{
	confvlogmsgx(priority, "configuration error", format, ap);
}

/*
 * Try to find tevent with the given value of event_step.
 * If such tevent does not exist, then allocate new one.
 * Return NULL if new tevent cannot be allocated.
 */
static struct tevent *
find_tevent(unsigned int event_step)
{
	struct tevent *tevent;

	SLIST_FOREACH(tevent, &tevents_list, link)
		if (tevent->event_step == event_step)
			return (tevent);
	tevent = mzone_alloc(tevent_mzone);
	if (tevent != NULL) {
		tevent->event_step = event_step;
		SLIST_INSERT_HEAD(&tevents_list, tevent, link);
	} else
		logconfx("find_tevent: mzone_alloc failed");
	return (tevent);
}

/*
 * Register a configuration event in ac_mod module.
 */
static int
ac_mod_conf_event(const struct ac_mod *ac_mod, unsigned int event,
    unsigned int no, const void *arg)
{
	if (ac_mod->ipa_ac_mod->conf_event(event, no, arg) < 0) {
		logconfx("module %s: conf_event(IPA_CONF_EVENT_%s) failed",
		    ac_mod->mod_file, conf_event_msg[event]);
		return (-1);
	}
	return (0);
}

/*
 * Register a configuration event in db_mod module.
 */
static int
db_mod_conf_event(const struct db_mod *db_mod, unsigned int event,
    unsigned int no, const void *arg)
{
	if (db_mod->ipa_db_mod->conf_event(event, no, arg) < 0) {
		logconfx("module %s: conf_event(IPA_CONF_EVENT_%s) failed",
		    db_mod->mod_file, conf_event_msg[event]);
		return (-1);
	}
	return (0);
}

/*
 * Register a configuration event in each module.
 */
static int
mod_conf_event(unsigned int event, unsigned int no, const void *arg)
{
	const struct ac_mod *ac_mod;
	const struct db_mod *db_mod;

	SLIST_FOREACH(ac_mod, &ac_mod_list, link)
		if (ac_mod_conf_event(ac_mod, event, no, arg) < 0)
			return (-1);

	SLIST_FOREACH(db_mod, &db_mod_list, link)
		if (db_mod_conf_event(db_mod, event, no, arg) < 0)
			return (-1);

	return (0);
}

/*
 * Parse the "global" section.
 */
/* ARGSUSED */
static int
parse_global(void *arg ATTR_UNUSED)
{
	if (global_section_set) {
		logconfx("this section is duplicated");
		return (-1);
	}
	global_section_set = 1;
	return (0);
}

#ifdef WITH_RULES
/*
 * Parse the "rule" section.
 */
static int
parse_rule(void *arg)
{
	const char *name;
	struct rule *rule;

	name = *(char **)arg;
	if (conf_validate_name(name) < 0)
		return (-1);
	if (rule_by_name(name) != NULL) {
		logconfx("this section is duplicated");
		return (-1);
	}

	rule = mzone_alloc(rule_mzone);
	if (rule == NULL) {
		logconfx("mzone_alloc failed");
		return (-1);
	}
	rule->name = mem_strdup(name, m_anon);
	if (rule->name == NULL) {
		logconfx("mem_strdup failed");
		return (-1);
	}
	rule->info = NULL;
	rule->no = nstatrules++;

	rule->cnt = rule->cnt_neg = 0;

	rule->rule_flags = RULE_FLAG_ACTIVE
#ifdef WITH_LIMITS
	    | RULE_FLAG_FREE_LIMITS
#endif
#ifdef WITH_THRESHOLDS
	    | RULE_FLAG_FREE_THRESHOLDS
#endif
	;

	rule->update_tevent = rule->append_tevent = NULL;
	rule->worktime = NULL;

	rule->ac_list = NULL;
	rule->db_list = NULL;

	rule->acg_add_pat = rule->acg_sub_pat = NULL;
	SLIST_INIT(&rule->acgs);

	rule->debug_exec = -1;
	rule_init_cmds(rule);

#ifdef WITH_LIMITS
	rule->debug_limit = rule->debug_limit_init = -1;
	limits_list = &rule->limits;
	STAILQ_INIT(limits_list);
	limitno = 0;
#endif

#ifdef WITH_THRESHOLDS
	rule->debug_threshold = rule->debug_threshold_init = -1;
	thresholds_list = &rule->thresholds;
	STAILQ_INIT(thresholds_list);
	thresholdno = 0;
#endif

#ifdef CTL_CHECK_CREDS
	rule->ctl_rule_acl = NULL;
#endif

	section_first = section_top = IPA_CONF_SECT_RULE;
	if (parser_local_sym_add("rule", rule->name, 0) < 0)
		return (-1);

	TAILQ_INSERT_TAIL(&rules_list, rule, list);
	rules_hash_add(rule);
	currule = rule;

	conf_event_no = rule->no;
	conf_event_arg = rule->name;

	return (0);
}
#endif /* WITH_RULES */

/*
 * Parse the "rulepat" section.
 */
static int
parse_rulepat(void *arg)
{
	struct rulepat *rulepat;
	char *pat;
	int error;

	pat = *(char **)arg;
	STAILQ_FOREACH(rulepat, &rulepats_list, link)
		if (strcmp(rulepat->pat, pat) == 0) {
			logconfx("this section is duplicated");
			return (-1);
		}

	rulepat = mzone_alloc(rulepat_mzone);
	if (rulepat == NULL) {
		logconfx("mzone_alloc failed");
		return (-1);
	}
	STAILQ_INSERT_TAIL(&rulepats_list, rulepat, link);

	error = regcomp(&rulepat->re, pat, REG_EXTENDED|REG_NOSUB);
	if (error != 0) {
		logconfx("regcomp(\"%s\"): %s", pat, regerrbuf(error));
		return (-1);
	}

	rulepat->pat = pat;
	rulepat->no = rulepatno++;

	rulepat->check_next = -1;

	rulepat->update_tevent = rulepat->append_tevent = NULL;
	rulepat->worktime = NULL;

	rulepat->ac_list = NULL;
	rulepat->db_list = NULL;

	rulepat->debug_exec = -1;

	cmds_rule_init(&rulepat->rc[RC_STARTUP]);
	cmds_rule_init(&rulepat->rc[RC_SHUTDOWN]);

#ifdef WITH_LIMITS
	rulepat->debug_limit = rulepat->debug_limit_init = -1;
	limits_list = &rulepat->limits;
	STAILQ_INIT(limits_list);
	limitno = 0;
#endif

#ifdef WITH_THRESHOLDS
	rulepat->debug_threshold = rulepat->debug_threshold_init = -1;
	thresholds_list = &rulepat->thresholds;
	STAILQ_INIT(thresholds_list);
	thresholdno = 0;
#endif

#ifdef CTL_CHECK_CREDS
	rulepat->ctl_rule_acl = NULL;
#endif

	section_first = section_top = IPA_CONF_SECT_RULEPAT;
	currulepat = rulepat;

	conf_event_no = rulepat->no;
	conf_event_arg = rulepat->pat;

	return (0);
}

/*
 * Parse the "check_next_rulepat" parameter.
 */
static int
parse_check_next_rulepat(void *arg)
{
	currulepat->check_next = *(int *)arg;
	return (0);
}

/*
 * Parse an argument of some "debug_*" parameter.
 */
static signed char
check_debug_level(uint32_t level, unsigned int maxlevel)
{
	if (level > maxlevel) {
		logconfx("too big debug level, max level is %u", maxlevel);
		return (-1);
	}
	return ((signed char)level);
}

/*
 * Parse the "debug_time" parameter.
 */
static int
parse_debug_time(void *arg)
{
	debug_time = check_debug_level(*(uint32_t *)arg, 2);
	return (debug_time);
}

/*
 * Parse the "debug_ac_null" parameter.
 */
static int
parse_debug_ac_null(void *arg)
{
	debug_ac_null = check_debug_level(*(uint32_t *)arg, 1);
	return (debug_ac_null);
}

/*
 * Parse the "debug_db_null" parameter.
 */
static int
parse_debug_db_null(void *arg)
{
	debug_db_null = check_debug_level(*(uint32_t *)arg, 1);
	return (debug_db_null);
}

/*
 * Parse the "debug_worktime" parameter.
 */
static int
parse_debug_worktime(void *arg)
{
	debug_worktime = check_debug_level(*(uint32_t *)arg, 1);
	return (debug_worktime);
}

/*
 * Parse the "debug_exec" parameter.
 */
static int
parse_debug_exec(void *arg)
{
	signed char level;

	level = check_debug_level(*(uint32_t *)arg, 2);
	if (level < 0)
		return (-1);
	switch (section) {
	case IPA_CONF_SECT_GLOBAL:
		global_debug_exec = level;
		break;
#ifdef WITH_RULES
	case IPA_CONF_SECT_RULE:
		currule->debug_exec = level;
		break;
#endif
#ifdef WITH_AUTORULES
	case IPA_CONF_SECT_AUTORULE:
		curautorule->debug_exec = level;
		break;
#endif
	case IPA_CONF_SECT_RULEPAT:
		currulepat->debug_exec = level;
		break;
	}
	return (0);
}

/*
 * Parse the "update_time" parameter.
 */
static int
parse_update_time(void *arg)
{
	uint64_t timeval;
	struct tevent *tevent;

	timeval = *(uint64_t *)arg;
	if (timeval == 0 || timeval > SECONDS_IN_DAY) {
		logconfx("argument should be greater than zero seconds "
		    "and less than or equal to 1 day");
		return (-1);
	}
	tevent = find_tevent((unsigned int)timeval);
	if (tevent == NULL)
		return (-1);
	switch (section) {
	case IPA_CONF_SECT_GLOBAL:
		global_update_tevent = tevent;
		break;
#ifdef WITH_RULES
	case IPA_CONF_SECT_RULE:
		currule->update_tevent = tevent;
		break;
#endif
#ifdef WITH_AUTORULES
	case IPA_CONF_SECT_AUTORULE:
		curautorule->update_tevent = tevent;
		break;
#endif
	case IPA_CONF_SECT_RULEPAT:
		currulepat->update_tevent = tevent;
		break;
	}
	return (0);
}

/*
 * Parse the "freeze_time" parameter.
 */
static int
parse_freeze_time(void *arg)
{
	uint64_t timeval;

	timeval = *(uint64_t *)arg;
	if (timeval == 0 || timeval >= 5 * SECONDS_IN_MINUTE) {
		logconfx("argument should be greater than zero seconds "
		    "and less than 5 minutes");
		return (-1);
	}
	freeze_time = (unsigned int)timeval;
	return (0);
}

/*
 * Parse the "append_time" parameter.
 */
static int
parse_append_time(void *arg)
{
	uint64_t timeval;
	struct tevent *tevent;

	timeval = *(uint64_t *)arg;
	if (timeval == 0 || timeval > SECONDS_IN_DAY) {
		logconfx("argument should be greater than zero seconds and "
		    "less than or equal to 1 day");
		return (-1);
	}
	tevent = find_tevent((unsigned int)timeval);
	if (tevent == NULL)
		return (-1);
	switch (section) {
	case IPA_CONF_SECT_GLOBAL:
		global_append_tevent = tevent;
		break;
#ifdef WITH_RULES
	case IPA_CONF_SECT_RULE:
		currule->append_tevent = tevent;
		break;
#endif
#ifdef WITH_AUTORULES
	case IPA_CONF_SECT_AUTORULE:
		curautorule->append_tevent = tevent;
		break;
#endif
	case IPA_CONF_SECT_RULEPAT:
		currulepat->append_tevent = tevent;
		break;
	}
	return (0);
}

/*
 * Parse the "wakeup_time" parameter.
 */
static int
parse_wakeup_time(void *arg)
{
	uint64_t timeval;

	timeval = *(uint64_t *)arg;
	if (timeval == 0 || timeval > SECONDS_IN_DAY) {
		logconfx("argument should be greater than zero seconds and "
		    "less than or equal to 1 day");
		return (-1);
	}
	wakeup_time = (unsigned int)timeval;
	return (0);
}

/*
 * Parse the "value_units" parameter.
 */
static int
parse_value_units(void *arg)
{
	unsigned int i;

	if (value_units >= 0) {
		logconfx("cannot re-define this parameter");
		return (-1);
	}
	if (got_arg_value) {
		logconfx("this parameter must be used before other parameters "
		    "and sections, that accept the IPA_CONF_TYPE_VALUE data "
		    "type");
		return (-1);
	}
	for (i = 0; i < VALUE_UNITS_TBL_SIZE; ++i)
		if (strcmp(*(char **)arg, value_units_tbl[i].str) == 0) {
			value_units = value_units_tbl[i].val;
			return (0);
		}
	logconfx("wrong value");
	return (-1);
}

#if defined(WITH_THRESHOLDS) || defined(WITH_SUBLIMITS)
/*
 * get_arg_value() plus get_arg_per_cent().
 */
static int
get_value_or_per_cent(uint64_t *val_ptr, unsigned char *type_ptr)
{
	uint64_t val;
	char *args;
	unsigned char type;

	args = parser_args;
	if (regexec_simple(&re_bytes, args) == 0) {
		if (get_arg_bytes(&val) < 0)
			return (-1);
		type = IPA_CONF_TYPE_BYTES;
	} else if (regexec_simple(&re_time, args) == 0) {
		if (get_arg_time(&val) < 0)
			return (-1);
		type = IPA_CONF_TYPE_TIME;
	} else {
		args += parser_args_len - 1;
		if (*args == '%') {
			*args = '\0';
			type = IPA_CONF_TYPE_PER_CENT;
		} else
			type = IPA_CONF_TYPE_UINT64;
		if (get_arg_uint64(&val) < 0)
			return (-1);
		if (type == IPA_CONF_TYPE_PER_CENT && val > 100) {
			logconfx("per cent value should less than or "
			    "equal to 100%%");
			return (-1);
		}
	}
	*val_ptr = val;
	*type_ptr = type;
	return (0);
}
#endif /* WITH_THRESHOLDS || WITH_SUBLIMITS */

#ifdef WITH_LIMITS
/*
 * Parse the "debug_limit" parameter.
 */
static int
parse_debug_limit(void *arg)
{
	signed char level;

	level = check_debug_level(*(uint32_t *)arg, 1);
	if (level < 0)
		return (-1);
	switch (section) {
	case IPA_CONF_SECT_GLOBAL:
		global_debug_limit = level;
		break;
#ifdef WITH_RULES
	case IPA_CONF_SECT_RULE:
		currule->debug_limit = level;
		break;
#endif
#ifdef WITH_AUTORULES
	case IPA_CONF_SECT_AUTORULE:
		curautorule->debug_limit = level;
		break;
#endif
	case IPA_CONF_SECT_RULEPAT:
		currulepat->debug_limit = level;
		break;
	}
	return (0);
}

/*
 * Parse the "debug_limit_init" parameter.
 */
static int
parse_debug_limit_init(void *arg)
{
	signed char level;

	level = check_debug_level(*(uint32_t *)arg, 1);
	if (level < 0)
		return (-1);
	switch (section) {
	case IPA_CONF_SECT_GLOBAL:
		global_debug_limit_init = level;
		break;
#ifdef WITH_RULES
	case IPA_CONF_SECT_RULE:
		currule->debug_limit_init = level;
		break;
#endif
#ifdef WITH_AUTORULES
	case IPA_CONF_SECT_AUTORULE:
		curautorule->debug_limit_init = level;
		break;
#endif
	case IPA_CONF_SECT_RULEPAT:
		currulepat->debug_limit_init = level;
		break;
	}
	return (0);
}

/*
 * Parse the "load_limit" parameter.
 */
static int
parse_load_limit(void *arg)
{
	if (section == IPA_CONF_SECT_LIMIT)
		curlimit->load_limit = (signed char)*(int *)arg;
	else /* IPA_CONF_SECT_GLOBAL */
		global_load_limit = (signed char)*(int *)arg;
	return (0);
}

/*
 * Parse the "limit" section.
 */
static int
parse_limit_sect(void *arg)
{
	const char *name;
	struct limit *limit;

	name = *(char **)arg;
	if (conf_validate_name(name) < 0)
		return (-1);
	STAILQ_FOREACH(limit, limits_list, link)
		if (strcmp(limit->name, name) == 0) {
			logconfx("this section is duplicated");
			return (-1);
		}
	limit = mzone_alloc(limit_mzone);
	if (limit == NULL) {
		logconfx("mzone_alloc failed");
		return (-1);
	}
	limit->name = mem_strdup(name, m_anon);
	if (limit->name == NULL) {
		logconfx("mem_strdup failed");
		return (-1);
	}
	limit->info = NULL;

	STAILQ_INSERT_TAIL(limits_list, limit, link);
	limit->no = limitno++;

	limit->lim_flags = LIMIT_FLAG_ACTIVE;

	limit->cnt_neg = 0;

	limit->db_list = NULL;

#ifdef WITH_SUBLIMITS
	STAILQ_INIT(&limit->sublimits);
#endif

	limit->restart.restart.upto = limit->expire.expire.upto =
	    TEXP_UPTO_NOTSET;
	limit_init_cmds(limit);

	limit->worktime = NULL;

	limit->wpid.type = WPID_TYPE_LIMIT;
	limit->wpid.pid = 0;
	limit->wpid.u.limit = limit;

#ifdef WITH_RULES
	limit->rule = currule;
#endif

	limit->load_limit = -1;

	if (section_top == IPA_CONF_SECT_RULE)
		++nstatlimits;
	section_top = IPA_CONF_SECT_LIMIT;

	if (parser_local_sym_add("limit", limit->name, 0) < 0)
		return (-1);

	curlimit = limit;
	conf_event_no = limit->no;
	conf_event_arg = limit->name;

	return (0);
}

/*
 * Parse the "limit" parameter.
 */
static int
parse_limit_param(void *arg)
{
	struct limit *limit;
	uint64_t *ptr;

	limit = curlimit;
	if (LIMIT_IS_SET(limit)) {
		logconfx("cannot re-define this parameter");
		return (-1);
	}

	limit->lim_flags |= LIMIT_FLAG_SET;
	ptr = (uint64_t *)arg;
	limit->lim = ptr[1];
	limit->cnt_type = (unsigned char)ptr[0];
	return (0);
}

#ifdef WITH_SUBLIMITS
/*
 * Parse the "sublimit" section.
 */
static int
parse_sublimit(void *arg)
{
	uint64_t lim;
	struct limit *limit;
	struct sublimit *sublimit;
	char *name;
	unsigned char cnt_type;

	limit = curlimit;
	if (LIMIT_IS_NOTSET(limit)) {
		logconfx("\"limit\" parameter in previous section should "
		    "be specified before");
		return (-1);
	}

	if (conf_validate_name(*(char **)arg) < 0)
		return (-1);
	name = mem_strdup(*(char **)arg, m_anon);
	if (name == NULL) {
		logconfx("mem_strdup failed");
		return (-1);
	}

	if (get_value_or_per_cent(&lim, &cnt_type) < 0)
		return (-1);
	if (lim == 0) {
		logconfx("argument should be greater than zero");
		return (-1);
	}
	if (cnt_type != IPA_CONF_TYPE_PER_CENT &&
	    cnt_type != limit->cnt_type) {
		logconfx("different arguments types for sublimit and limit");
		return (-1);
	}

	STAILQ_FOREACH(sublimit, &limit->sublimits, link)
		if (cnt_type == IPA_CONF_TYPE_PER_CENT) {
			if ((unsigned int)lim == sublimit->lim_pc) {
				logconfx("this section is duplicated");
				return (-1);
			}
		} else {
			if (lim == sublimit->lim) {
				logconfx("this section is duplicated");
				return (-1);
			}
		}

	sublimit = mzone_alloc(sublimit_mzone);
	if (sublimit == NULL) {
		logconfx("mzone_alloc failed");
		return (-1);
	}

	STAILQ_INSERT_TAIL(&limit->sublimits, sublimit, link);

	sublimit->name = name;
	if (cnt_type == IPA_CONF_TYPE_PER_CENT)
		sublimit->lim_pc = (unsigned int)lim;
	else {
		sublimit->lim = lim;
		sublimit->lim_pc = 0;
	}
	sublimit->cnt_type = cnt_type;

	sublimit_init_cmds(sublimit);

	sublimit->wpid.type = WPID_TYPE_SUBLIMIT;
	sublimit->wpid.pid = 0;
	sublimit->wpid.u.sublimit = sublimit;

	sublimit->limit = limit;

	if (section_top == IPA_CONF_SECT_RULE)
		++nstatsublimits;
	section_top = SECT_SUBLIMIT;

	if (parser_local_sym_add("sublimit", sublimit->name, 0) < 0)
		return (-1);

	cursublimit = sublimit;

	return (0);
}
#endif /* WITH_SUBLIMITS */

/*
 * Parse "expired time" value.
 */
static int
parse_texp(struct texp *texp, const void *arg)
{
	const char *ptr;
	char *endptr;
	uint32_t value;
	char level, error, overflow;

	ptr = (const char *)arg;
	level = error = overflow = 0;

	texp->seconds = 0;
	texp->upto = TEXP_UPTO_SIMPLE;
	for (;;) {
		if (*ptr == ' ')
			++ptr;
		if (*ptr == '+') {
			texp->upto = *++ptr;
			texp->side = (char)(texp->seconds > 0);
			if (*++ptr == '\0') {
				/* EOL */
				break;
			}
			continue;
		}
		if (strto_uint32(&value, ptr, &endptr) < 0)
			return (-1);
		ptr = endptr;
		switch (*ptr) {
		case 'W':
			if (level > 1)
				error = 1;
			else if (value > UINT32_MAX / SECONDS_IN_WEEK)
				overflow = 1;
			else {
				level = 2;
				value *= SECONDS_IN_WEEK;
			}
			break;
		case 'D':
			if (level > 2)
				error = 1;
			else if (value > UINT32_MAX / SECONDS_IN_DAY)
				overflow = 1;
			else {
				level = 3;
				value *= SECONDS_IN_DAY;
			}
			break;
		case 'h':
			if (level > 3)
				error = 1;
			else if (value > UINT32_MAX / SECONDS_IN_HOUR)
				overflow = 1;
			else {
				level = 4;
				value *= SECONDS_IN_HOUR;
			}
			break;
		case 'm':
			if (level > 4)
				error = 1;
			else if (value > UINT32_MAX / SECONDS_IN_MINUTE)
				overflow = 1;
			else {
				level = 5;
				value *= SECONDS_IN_MINUTE;
			}
			break;
		default: /* 's' */
			if (level > 5)
				error = 1;
			else
				level = 6;
		}
		if (error) {
			logconfx("wrong format of an argument");
			return (-1);
		}
		if (overflow || texp->seconds > UINT_MAX - value) {
			logconfx("too big value for 'unsigned int'");
			return (-1);
		}
		texp->seconds += (unsigned int)value;
		if (*++ptr == '\0') {
			/* EOL */
			break;
		}
	}
	return (0);
}

/*
 * Parse the "restart" parameter.
 */
static int
parse_restart(void *arg)
{
	struct texp *texp;

	texp = &curlimit->restart.restart;
	if (parse_texp(texp, *(char **)arg) < 0)
		return (-1);
	if (texp->seconds == 0 && texp->upto == TEXP_UPTO_SIMPLE) {
		logconfx("argument should be greater than zero");
		return (-1);
	}
	return (0);
}

/*
 * Parse the "expire" parameter.
 */
static int
parse_expire(void *arg)
{
	return (parse_texp(&curlimit->expire.expire, *(char **)arg));
}
#endif /* WITH_LIMITS */

/*
 * Parse the "exec" parameter.
 */
static int
parse_exec(void *arg)
{
	struct cmd *cmd;
	char *start, *ptr, *str;
	int allow_subst;

	if (parser_nargs > 2) {
		logconfx("one or two arguments are expected");
		return (-1);
	}

	/* Allocate new structure for command. */
	cmd = mzone_alloc(cmd_mzone);
	if (cmd == NULL) {
		logconfx("mzone_alloc failed");
		return (-1);
	}
	STAILQ_INSERT_TAIL(&curcmds->cmd_list, cmd, link);
	curcmds->has_cmd = 1;

	/* user "..." or "..." */
	str = *(char **)arg;
	if (parser_nargs == 2) {
		if (*str == '\"') {
			logconfx("wrong format of an argument");
			return (-1);
		}
		ptr = str;
		str = strchr(str, ' ');
		*str++ = '\0';
		cmd->user = mem_strdup(ptr, m_cmd);
		if (cmd->user == NULL) {
			logconfx("mem_strdup failed");
			return (-1);
		}
		cmd->free_mask = CMD_FREE_STR|CMD_FREE_USER;
	} else {
		cmd->user = NULL;
		cmd->free_mask = CMD_FREE_STR;
	}

	/* Check for "..." */
	if (!parser_buf_is_str(str)) {
		logconfx("wrong format of an argument");
		return (-1);
	}

	/* Copy "..." */
	cmd->str = parser_strdup(str, m_cmd);
	if (cmd->str == NULL)
		return (-1);

	/* Validate "..." */
	str = cmd->str;
	if (*str == '\0') {
		logconfx("command should be a non-empty string");
		return (-1);
	}
	if (*str != '/' && only_abs_paths != 0) {
		logconfx("command should be given with absolute path");
		return (-1);
	}

	switch (section_first) {
#ifdef WITH_AUTORULES
	case IPA_CONF_SECT_AUTORULE:
#endif
	case IPA_CONF_SECT_RULEPAT:
		allow_subst = 1;
		break;
	default:
		allow_subst = 0;
	}

	/* Check if a string has correct substitutions and count them. */
	start = NULL;
	cmd->subst_per_cent = cmd->subst_rule = 0;
	cmd->str_size = 1;
	for (ptr = str; *ptr != '\0'; cmd->str_size++, ++ptr) {
		if (allow_subst && *ptr == '%') {
			if (start == NULL)
				start = ptr + 1;
			else if (ptr != start) {
				/* %xxx% */
				*ptr = '\0';
				if (strcmp(start, "rule") != 0) {
					logconfx("unknown substitution "
					    "%%\"%s\"%% in a command", start);
					return (-1);
				}
				*ptr = '%';
				cmd->subst_rule++;
				start = NULL;
			} else {
				/* %% */
				cmd->subst_per_cent++;
				start = NULL;
			}
		}
	}
	if (start != NULL) {
		logconfx("not closed substitution in a command");
		return (-1);
	}

	return (0);
}

#ifdef WITH_RULES
/*
 * Find the next field in a string starting from *end.  If there is
 * a next field, then return 1 and return the start of the next field
 * in *beg and a pointer to the next character after the end of the
 * next field in *end.  Otherwise return 0.
 */
static int
next_field(char **beg, char **end)
{
	char *ptr;
	int in_string;

	/* Skip spaces. */
	for (ptr = *end;; ++ptr) {
		if (*ptr == '\0')
			return (0);
		if (*ptr != ' ' && *ptr != '\t')
			break;
	}

	*beg = ptr;

	/* Skip data. */
	for (in_string = 0; *ptr != '\0'; ++ptr)
		switch (*ptr) {
		case ' ':
		case '\t':
			if (!in_string)
				goto done;
			break;
		case '\"':
			in_string = !in_string;
			break;
		case '\\':
			if (in_string)
				++ptr;
		}
done:
	if (*ptr != '\0') {
		*ptr = '\0';
		*end = ptr + 1;
	} else
		*end = ptr;
	return (1);
}

/*
 * Parse the "ictl" parameter.
 * Possible combinations:
 *  "-r <rule> set ..."
 *  "-r <rule> -l <limit> restart"
 *  "-r <rule> -l <limit> expire"
 *  "-r <rule> -l <limit> set ..."
 *  "-r <rule> -t <threshold> set ..."
 */
static int
parse_ictl(void *arg)
{
	uint64_t value[2];
	const char *name1;
	struct ictl *ictl;
	char *beg, *end, *cmd_str;
#ifdef WITH_ANY_LIMITS
	char *name2;
#endif
	unsigned int cmd, qflags, qflagset;
	int this_rule;

	if (section_first != IPA_CONF_SECT_RULE &&
	    section_first != IPA_CONF_SECT_ROOT) {
		logconfx("this parameter can be used only inside static "
		    "rules and inside global startup{} section");
		return (-1);
	}
	if (section_rc == RC_SHUTDOWN) {
		logconfx("this parameter cannot be used inside "
		    "shutdown{} section");
		return (-1);
	}
	if (!STAILQ_EMPTY(&curcmds->cmd_list)) {
		logconfx("this parameter should be used before \"exec\" "
		    "parameters in one section");
		return (-1);
	}

	ictl = mzone_alloc(ictl_mzone);
	if (ictl == NULL) {
		logconfx("mzone_alloc failed");
		return (-1);
	}
	STAILQ_INSERT_TAIL(&curcmds->ictl_list, ictl, link);
	curcmds->has_cmd = 1;

	cmd_str = mem_strdup(*(char **)arg, m_ctl);
	if (cmd_str == NULL) {
		logconfx("mem_strdup failed");
		return (-1);
	}

	/* Check "-r ..." and get rule's name. */
	end = *(char **)arg;
	if (!next_field(&beg, &end))
		goto short_command;
	if (beg[0] != '-' || beg[1] != 'r') {
		logconfx("command should begin with -r");
		return (-1);
	}
	if (beg[2] == '\0') {
		if (!next_field(&beg, &end))
			goto short_command;
	} else
		beg += 2;
	name1 = beg;
	if (conf_validate_name(name1) < 0)
		return (-1);

	/* Next field can be command or "-l" or "-t" option. */
	if (next_field(&beg, &end) == 0)
		goto short_command;
	if (beg[0] == '-' && beg[1] == 'l') {
#ifdef WITH_LIMITS
		qflags = CTL_CFLAG_RULE|CTL_CFLAG_LIMIT;
#else
		logconfx("limits support was disabled");
		return (-1);
#endif
	} else if (beg[0] == '-' && beg[1] == 't') {
#ifdef WITH_THRESHOLDS
		qflags = CTL_CFLAG_RULE|CTL_CFLAG_THRESHOLD;
#else
		logconfx("thresholds support was disabled");
		return (-1);
#endif
	} else
		qflags = CTL_CFLAG_RULE;

#ifdef WITH_ANY_LIMITS
	if (qflags & (CTL_CFLAG_LIMIT|CTL_CFLAG_THRESHOLD)) {
		/* Get limit's or threshold's name. */
		if (beg[2] == '\0') {
			if (!next_field(&beg, &end))
				goto short_command;
		} else
			beg += 2;
		name2 = beg;
		if (conf_validate_name(name2) < 0)
			return (-1);
		/* Get command. */
		if (!next_field(&beg, &end))
			goto short_command;
	} else
		name2 = NULL;
#endif

	/* Here beg points to command. */
	if (strcmp(beg, "set") == 0)
		cmd = CTL_CMD_SET;
#ifdef WITH_LIMITS
	else if (strcmp(beg, "restart") == 0)
		cmd = CTL_CMD_RESTART;
	else if (strcmp(beg, "expire") == 0)
		cmd = CTL_CMD_EXPIRE;
#endif
	else {
		logconfx("unknown or unsupported command \"%s\"", beg);
		return (-1);
	}

	if (cmd == CTL_CMD_SET) {
		/* Parse arguments of a "set" command. */
		while (next_field(&beg, &end)) {
			if (strcmp(beg, "counter") == 0)
				qflagset = CTL_CFLAG_VALUE2;
#ifdef WITH_LIMITS
			else if (strcmp(beg, "limit") == 0) {
				if (!(qflags & CTL_CFLAG_LIMIT))
					goto wrong_keyword;
				qflagset = CTL_CFLAG_VALUE1;
			}
#endif
#ifdef WITH_THRESHOLDS
			else if (strcmp(beg, "threshold") == 0) {
				if (!(qflags & CTL_CFLAG_THRESHOLD))
					goto wrong_keyword;
				qflagset = CTL_CFLAG_VALUE1;
			}
#endif
			else
				goto wrong_keyword;

			if (qflags & qflagset)
				goto duped_keyword;
			qflags |= qflagset;

			if (!next_field(&beg, &end))
				goto short_command;
			switch (*beg) {
			case '+':
				qflags |= qflagset == CTL_CFLAG_VALUE1 ?
				    CTL_CFLAG_VALUE1_INC : CTL_CFLAG_VALUE2_INC;
				++beg;
				break;
			case '-':
				qflags |= qflagset == CTL_CFLAG_VALUE1 ?
				    CTL_CFLAG_VALUE1_DEC : CTL_CFLAG_VALUE2_DEC;
				++beg;
				break;
			default:
				if (!(qflags &
				    (CTL_CFLAG_LIMIT|CTL_CFLAG_THRESHOLD))) {
					logconfx("it is allowed only to "
					    "increase or decrease statistics "
					    "of a rule");
					return (-1);
				}
			}
			parser_args = beg;
			if (get_arg_value(value) < 0)
				return (-1);
			if (qflagset == CTL_CFLAG_VALUE2)
				ictl->values.value2 = value[1];
#ifdef WITH_ANY_LIMITS
			else
				ictl->values.value1 = value[1];
#endif
		}
		if (!(qflags & (CTL_CFLAG_VALUE1|CTL_CFLAG_VALUE2)))
			goto short_command;
	}
#ifdef WITH_LIMITS
	else if (!(qflags & CTL_CFLAG_LIMIT)) {
		logconfx("command \"%s\" can be used only with limits", beg);
		return (-1);
	} else if (next_field(&beg, &end)) {
		logconfx("too many arguments for command \"%s\"", beg);
		return (-1);
	}
#endif

	/* At this point command was successfully parsed. */
	ictl->rule.name = mem_strdup(name1, m_ctl);
	if (ictl->rule.name == NULL) {
		logconfx("mem_strdup failed");
		return (-1);
	}

	this_rule = section_first == IPA_CONF_SECT_RULE &&
	    strcmp(currule->name, name1) == 0;
	if (this_rule && !(qflags & (CTL_CFLAG_LIMIT|CTL_CFLAG_THRESHOLD))) {
		logconfx("rule is not allowed to change own statistics");
		return (-1);
	}

#ifdef WITH_ANY_LIMITS
	if (name2 != NULL) {
		name2 = mem_strdup(name2, m_ctl);
		if (name2 == NULL) {
			logconfx("mem_strdup failed");
			return (-1);
		}
# ifdef WITH_LIMITS
		if (qflags & CTL_CFLAG_LIMIT) {
			if (this_rule &&
			    section_top == IPA_CONF_SECT_LIMIT &&
			    strcmp(curlimit->name, name2) == 0) {
				logconfx("limit is not allowed to change "
				    "own statistics");
				return (-1);
			}
			ictl->limit.name = name2;
		}
# endif
# ifdef WITH_THRESHOLDS
		if (qflags & CTL_CFLAG_THRESHOLD) {
			if (this_rule &&
			    section_top == IPA_CONF_SECT_THRESHOLD &&
			    strcmp(curthreshold->name, name2) == 0) {
				logconfx("threshold is not allowed to change "
				    "own statistics");
				return (-1);
			}
			ictl->threshold.name = name2;
		}
# endif
	}
#endif /* WITH_ANY_LIMITS */

	ictl->cmdq.cmd = cmd;
	ictl->cmdq.flags = qflags;
	ictl->str = cmd_str;

	mem_free(*(char **)arg, m_parser);
	return (0);

duped_keyword:
	logconfx("duplicated keyword \"%s\" in a command", beg);
	return (-1);

short_command:
	logconfx("command is incomplete");
	return (-1);

wrong_keyword:
	logconfx("wrong keyword \"%s\" in a command", beg);
	return (-1);
}
#endif /* WITH_RULES */

/*
 * Parse the "sync_exec" parameter.
 */
static int
parse_sync_exec(void *arg)
{
	curcmds->sync = (signed char)*(int *)arg;
	return (0);
}

#if defined(WITH_RULES) || defined(WITH_ANY_LIMITS)
/*
 * Parse the "info" parameter.
 */
static int
parse_info(void *arg)
{
	char *ptr, **pptr;

	for (ptr = *(char **)arg; *ptr != '\0'; ++ptr)
		switch (*ptr) {
		case '\t':
		case '\n':
			logconfx("'\\t' and '\\n' are not allowed");
			return (-1);
		}

	switch (section) {
#ifdef WITH_RULES
	case IPA_CONF_SECT_RULE:
		pptr = &currule->info;
		break;
#endif
#ifdef WITH_LIMITS
	case IPA_CONF_SECT_LIMIT:
		pptr = &curlimit->info;
		break;
#endif
#ifdef WITH_THRESHOLDS
	default: /* IPA_CONF_SECT_THRESHOLD */
		pptr = &curthreshold->info;
		break;
#endif
	}

	if (*pptr != NULL) {
		logconfx("cannot re-define this parameter");
		return (-1);
	}

	*pptr = *(char **)arg;
	return (0);
}
#endif /* WITH_RULES || WITH_ANY_LIMITS */

static const char wdays_short[DAYS_IN_WEEK] = {
	'S', 'M', 'T', 'W', 'H', 'F', 'A'
};

static const struct worktime *
parse_generic_worktime(void *arg)
{
	struct worktime *wt;
	struct tint *tint1, *tint2;
	struct tint_set *set1, *set2;
	struct tint_list *list;
	char *ptr;
	unsigned int wday, h1, m1, h2, m2, h2_prev, m2_prev;

	wt = mzone_alloc(worktime_mzone);
	if (wt == NULL) {
		logconfx("mzone_alloc failed");
		return (NULL);
	}

	for (wday = 0; wday < DAYS_IN_WEEK; ++wday)
		wt->tint_list[wday] = &tint_list_empty;

	for (ptr = *(char **)arg; *ptr != '\0';) {
		wday = strchr(wdays_short, *ptr) - wdays_short;
		if (wt->tint_list[wday] != &tint_list_empty) {
			logconfx("each day (currently %s) should be "
			    "specified only one time", wdays[wday]);
			return (NULL);
		}

		set1 = mem_malloc(sizeof(*set1), m_anon);
		if (set1 == NULL) {
			logconfx("mem_malloc failed");
			return (NULL);
		}
		list = &set1->list;
		STAILQ_INIT(list);

		h2_prev = m2_prev = 0;
		ptr += 2;
		for (;;) {
			if (*ptr == '*') {
				h1 = m1 = m2 = 0;
				h2 = HOURS_IN_DAY;
			} else if (!isdigit((unsigned char)*ptr)) {
				/* Next day in worktime. */
				break;
			} else {
				errno = 0;
				if (sscanf(ptr, "%u:%u-%u:%u", &h1, &m1, &h2,
				    &m2) != 4) {
					logconf("sscanf(%s) failed", ptr);
					return (NULL);
				}
				if ((h1 > 23 || h2 > 23 || m1 > 59 || m2 > 59)
				    && !(h2 == HOURS_IN_DAY && m2 == 0)) {
					logconfx("wrong value of hours or "
					    "minutes in time interval for %s",
					    wdays[wday]);
					return (NULL);
				}
				if (h1 == h2 && m1 == m2) {
					logconfx("zero seconds time intervals "
					    "are not allowed");
					return (NULL);
				}
				if ((h1 * MINUTES_IN_HOUR + m1) >
				    (h2 * MINUTES_IN_HOUR + m2)) {
					logconfx("wrong time interval for "
					   "%s: %02u:%02u-%02u:%02u",
					    wdays[wday], h1, m1, h2, m2);
					return (NULL);
				}
				if (h2_prev * MINUTES_IN_HOUR + m2_prev >
				    h1 * MINUTES_IN_HOUR + m1) {
					logconfx("nonsuccessive time interval "
					    "for %s", wdays[wday]);
					return (NULL);
				}
			}

			tint1 = mzone_alloc(tint_mzone);
			if (tint1 == NULL) {
				logconfx("mzone_alloc failed");
				return (NULL);
			}
			tint1->sec1 = h1 * SECONDS_IN_HOUR +
			    m1 * SECONDS_IN_MINUTE;
			tint1->sec2 = h2 * SECONDS_IN_HOUR +
			    m2 * SECONDS_IN_MINUTE;
			STAILQ_INSERT_TAIL(&set1->list, tint1, link);

			for (; *ptr != ' '; ++ptr)
				if (*ptr == '\0')
					goto done;
			++ptr;
			h2_prev = h2;
			m2_prev = m2;
		}
done:
		/*
		 * If there is already the same tint_list,
		 * then free memory used by just allocated one.
		 */
		STAILQ_FOREACH(set2, &tint_sets, link) {
			tint1 = STAILQ_FIRST(&set1->list);
			tint2 = STAILQ_FIRST(&set2->list);
			while (tint1 != NULL && tint2 != NULL) {
				if (tint1->sec1 != tint2->sec1 ||
				    tint1->sec2 != tint2->sec2)
					break;
				tint1 = STAILQ_NEXT(tint1, link);
				tint2 = STAILQ_NEXT(tint2, link);
			}
			if (tint1 == NULL && tint2 == NULL) {
				/* Duplicate tint_list was found. */
				free_tint_set(set1);
				set1 = set2;
				break;
			}
		}
		if (set2 == NULL)
			/* New tint_list --> add it to tint_sets. */
			STAILQ_INSERT_TAIL(&tint_sets, set1, link);

		wt->tint_list[wday] = &set1->list;
	}

	/*
	 * Since "ictl" can be used before newday() is called, need to mark
	 * worktime as active, since by default all static rules are active
	 * on startup.
	 */
	wt->wt_flags = WT_FLAG_ACTIVE;

	return (find_worktime(wt));
}

/*
 * Parse the "worktime" parameter.
 */
static int
parse_worktime(void *arg)
{
	const struct worktime **wtpp;

	switch (section) {
#ifdef WITH_RULES
	case IPA_CONF_SECT_RULE:
		wtpp = &currule->worktime;
		break;
#endif
#ifdef WITH_LIMITS
	case IPA_CONF_SECT_LIMIT:
		wtpp = &curlimit->worktime;
		break;
#endif
#ifdef WITH_THRESHOLDS
	case IPA_CONF_SECT_THRESHOLD:
		wtpp = &curthreshold->worktime;
		break;
#endif
#ifdef WITH_AUTORULES
	case IPA_CONF_SECT_AUTORULE:
		wtpp = &curautorule->worktime;
		break;
#endif
	case IPA_CONF_SECT_RULEPAT:
		wtpp = &currulepat->worktime;
		break;
	default: /* IPA_CONF_SECT_GLOBAL */
		wtpp = &global_worktime;
	}

	if (*wtpp != NULL) {
		logconfx("cannot re-define this parameter");
		return (-1);
	}

	*wtpp = parse_generic_worktime(arg);
	return (*wtpp == NULL ? -1 : 0);
}

#ifdef WITH_AUTORULES
/*
 * Parse the "worktime_rule" parameter.
 */
static int
parse_worktime_rule(void *arg)
{
	struct autorule *autorule;

	autorule = curautorule;
	if (autorule->worktime_rule != NULL) {
		logconfx("cannot re-define this parameter");
		return (-1);
	}
	autorule->worktime_rule = parse_generic_worktime(arg);
	return (autorule->worktime_rule == NULL ? -1 : 0);
}
#endif

/*
 * Check security of a configuration file: absolute path, regular file,
 * is owned by the user who run ipa, writable only for the user.
 */
static int
check_conf_file(const char *fname, int ignore_non_regular)
{
	struct stat statbuf;

	if (stat(fname, &statbuf) < 0) {
		logconfe("stat(%s): %s", fname, strerror(errno));
		return (-1);
	}
	if (!S_ISREG(statbuf.st_mode)) {
		if (ignore_non_regular)
			return (0);
		logconfe("configuration file \"%s\" should be a regular file",
		    fname);
		return (-1);
	}
	if (statbuf.st_uid != myuid) {
		logconfe("configuration file \"%s\" should be owned by the "
		    "user, who run this program", fname);
		return (-1);
	}
	if (statbuf.st_mode & (S_IWGRP|S_IWOTH)) {
		logconfe("configuration file \"%s\" should not have write "
		    "permissions for group or other users", fname);
		return (-1);
	}
	return (1);
}

/*
 * Check file path: non empty string and should start with '/'.
 */
static int
check_file_path(const char *fname, const char *what)
{
	if (*fname == '\0') {
		logconfx("argument should be a non-empty string");
		return (-1);
	}
	if (*fname != '/') {
		logconfx("%s should be given with absolute path", what);
		return (-1);
	}
	return (0);
}

static int
check_include_depth(void)
{
	if (include_depth > INCLUDE_DEPTH_MAX) {
		logconfx("too big (> %u) depth of included files",
		    INCLUDE_DEPTH_MAX);
		return (-1);
	}
	return (0);
}

/*
 * Parse the "include" parameter.
 */
static int
parse_include(void *arg)
{
	struct parser_pb *pb;
	char *fname;

	/* Save offset of current configuration file and close it. */
	fname = *(char **)arg;
	pb = parser_top_pb();
	pb->foff = ftell(pb->fp);
	if (pb->foff < 0) {
		logconf("ftell(%s)", pb->fname);
		return (-1);
	}
	if (fclose(pb->fp) != 0) {
		logconf("fclose(%s)", pb->fname);
		return (-1);
	}

	/* Validate an argument. */
	if (check_file_path(fname, "file") < 0)
		return (-1);

	/* Check security of configuration file. */
	if (check_conf_file(fname, 0) < 0)
		return (-1);

	++include_depth;
	if (check_include_depth() < 0)
		return (-1);

	/* Open included configuration file and put it to stack. */
	pb = parser_new_pb(0);
	if (pb == NULL)
		return (-1);
	pb->fp = fopen(fname, "r");
	if (pb->fp == NULL) {
		logconf("fopen(%s, \"r\")", fname);
		return (-1);
	}
	pb->fname = fname;
	return (parser_push_pb(pb));
}

/*
 * Parse the "include_files" parameter.
 */
static int
parse_include_files(void *arg)
{
	struct stat statbuf;
	regex_t re;
	const struct dirent *dp;
	struct parser_pb *pb, *tpb;
	char *dir, *pat, *fname;
	DIR *dirp;

	/* Validate an argument. */
	dir = *(char **)arg;
	if (check_file_path(dir, "directory") < 0)
		return (-1);

	pat = strrchr(dir, '/');
	*pat++ = '\0';
	if (posix_re_pattern > 0) {
		int error;

		error = regcomp(&re, pat, REG_EXTENDED|REG_NOSUB);
		if (error != 0) {
			logconfx("cannot compile regular expression: "
			    "regcomp(\"%s\"): %s", pat, regerrbuf(error));
			return (-1);
		}
	}

	/* Check security of the given directory. */
	if (lstat(dir, &statbuf) < 0) {
		logconf("lstat(%s)", dir);
		return (-1);
	}
	if (!S_ISDIR(statbuf.st_mode)) {
		logconfx("given pathname is not a directory");
		return (-1);
	}
	if (statbuf.st_uid != myuid) {
		logconfx("directory should be owned by the user, who run "
		    "this program");
		return (-1);
	}
	if (statbuf.st_mode & (S_IWGRP|S_IWOTH)) {
		logconfx("directory should not have write permissions for "
		    "group and other users");
		return (-1);
	}

	/* Read directory and push pb to stack. */
	dirp = opendir(dir);
	if (dirp == NULL) {
		logconf("opendir(%s)", dir);
		return (-1);
	}
	pb = NULL;
	tpb = parser_top_pb();
	for (;;) {
		errno = 0;
		dp = readdir(dirp);
		if (dp == NULL) {
			if (errno != 0) {
				logconf("readdir(%s)", dir);
				return (-1);
			}
			break;
		}
		/* Ignore "." and ".." directories. */
		if (dp->d_name[0] == '.')
			switch (dp->d_name[1]) {
			case '\0':
				continue;
			case '.':
				if (dp->d_name[2] == '\0')
					continue;
			}
		if (posix_re_pattern != 1) {
			if (fnmatch(pat, dp->d_name, FNM_PERIOD) == FNM_NOMATCH)
				continue;
		} else {
			if (regexec_simple(&re, dp->d_name) != 0)
				continue;
		}
		if (mem_asprintf(m_parser, &fname, "%s/%s", dir,
		    dp->d_name) < 0) {
			logconfx("mem_asprintf failed");
			return (-1);
		}
		switch (check_conf_file(fname, 1)) {
		case -1:
			return (-1);
		case 0:
			mem_free(fname, m_parser);
			continue;
		}
		pb = parser_new_pb(0);
		if (pb == NULL)
			return (-1);
		pb->fname = fname;
		pb->foff = 0;
		if (parser_push_pb(pb) < 0)
			return (-1);
	}
	if (posix_re_pattern > 0)
		regfree(&re);
	if (closedir(dirp) < 0) {
		logconf("closedir(%s)", dir);
		return (-1);
	}
	mem_free(dir, m_parser);
	if (pb != NULL) {
		tpb->foff = ftell(tpb->fp);
		if (tpb->foff < 0) {
			logconf("ftell(%s)", tpb->fname);
			return (-1);
		}
		if (fclose(tpb->fp) != 0) {
			logconf("fclose(%s)", tpb->fname);
			return (-1);
		}
		++include_depth;
		if (check_include_depth() < 0)
			return (-1);
		pb->fp = fopen(pb->fname, "r");
		if (pb->fp == NULL) {
			logconf("fopen(%s, \"r\")", pb->fname);
			return (-1);
		}
	}
	return (0);
}

/*
 * Parse the "keep_rules_order" parameter.
 */
static int
parse_keep_rules_order(void *arg)
{
	keep_rules_order = (signed char)*(int *)arg;
	return (0);
}

/*
 * Parse the "only_abs_paths" parameter.
 */
static int
parse_only_abs_paths(void *arg)
{
	only_abs_paths = (signed char)*(int *)arg;
	return (0);
}

/*
 * Parse the "ac_mod" parameter.
 */
static int
parse_ac_mod(void *arg)
{
	const struct ac_mod *ac_mod2;
	struct ipa_ac_mod *ipa_ac_mod;
	struct ac_mod *ac_mod;
	char *mod_name, *sym;

	ac_mod = mem_malloc(sizeof(*ac_mod), m_anon);
	if (ac_mod == NULL) {
		logconfx("mem_malloc failed");
		return (-1);
	}
	ac_mod->mod_file = *(char **)arg;
	ac_mod->mod_ref_count = 0;
	ac_mod->mod_handle = dl_open(ac_mod->mod_file);
	if (ac_mod->mod_handle == NULL) {
		logconfx("dl_open(%s): %s", ac_mod->mod_file, dl_error());
		return (-1);
	}
	mod_name = get_mod_name(ac_mod->mod_file);
	if (mod_name == NULL)
		return (-1);
	if (mem_asprintf(m_anon, &sym, "%s_ac_mod", mod_name) < 0) {
		logconfx("mem_asprintf failed");
		return (-1);
	}
	ac_mod->ipa_ac_mod =
	    (struct ipa_ac_mod *)dl_lookup_sym(ac_mod->mod_handle, sym);
	if (ac_mod->ipa_ac_mod == NULL) {
		logconfx("given module is not an IPA accounting module "
		    "or unknown symbol naming scheme is used");
		return (-1);
	}
	mem_free(sym, m_anon);
	mem_free(mod_name, m_anon);
	ipa_ac_mod = ac_mod->ipa_ac_mod;

	/* Check ipa_ac_mod API version. */
	if (ipa_ac_mod->api_ver != IPA_AC_MOD_API_VERSION) {
		logconfx("module %s uses ipa_ac_mod API version %u, "
		    "my ipa_ac_mod API version is %u", ac_mod->mod_file,
		    ipa_ac_mod->api_ver, IPA_AC_MOD_API_VERSION);
		return (-1);
	}

	/* Check if module is thread-safe or vice versa. */
#ifdef WITH_PTHREAD
	if (!(ipa_ac_mod->mod_flags & IPA_MOD_FLAG_PTHREAD_SAFE)) {
		logconfx("module %s must be thread-safe", ac_mod->mod_file);
		return (-1);
	}
#else
	if (ipa_ac_mod->mod_flags & IPA_MOD_FLAG_PTHREAD_SAFE) {
		logconfx("module %s must not be thread-safe",
		    ac_mod->mod_file);
		return (-1);
	}
#endif /* WITH_PTHREAD */

	if (strcmp(ipa_ac_mod->ac_name, "null") == 0) {
		logconfx("module's accounting system name is \"null\", "
		    "this is a name of the built-in accounting system");
		return (-1);
	}

	ac_mod2 = ac_mod_by_name(ipa_ac_mod->ac_name);
	if (ac_mod2 != NULL) {
		logconfx("duplicated accounting system name \"%s\" in %s and "
		    "%s modules", ipa_ac_mod->ac_name, ac_mod->mod_file,
		    ac_mod2->mod_file);
		return (-1);
	}
	if (ipa_ac_mod->conf_prefix != NULL) {
		const struct db_mod *db_mod;

		ac_mod2 = ac_mod_by_prefix(ipa_ac_mod->conf_prefix);
		if (ac_mod2 != NULL) {
			logconfx("duplicated configuration prefix \"%s\" in "
			    "%s and %s modules", ipa_ac_mod->conf_prefix,
			    ac_mod->mod_file, ac_mod2->mod_file);
			return (-1);
		}
		db_mod = db_mod_by_prefix(ipa_ac_mod->conf_prefix);
		if (db_mod != NULL) {
			logconfx("duplicated configuration prefix \"%s\" in "
			    "%s and %s modules", ipa_ac_mod->conf_prefix,
			    ac_mod->mod_file, db_mod->mod_file);
			return (-1);
		}
	}

	ipa_ac_mod->suppfunc = &suppfunc;
	ipa_ac_mod->memfunc = &memfunc;
	ipa_ac_mod->ac_create_rule = create_rule;
	ipa_ac_mod->ac_delete_rule = delete_rule;

	if (init_conf_tbls(ac_mod->mod_file, 1,
	    ipa_ac_mod->conf_sect_tbl, &ac_mod->conf_sect_hash,
	    ipa_ac_mod->conf_param_tbl, &ac_mod->conf_param_hash) < 0)
		return (-1);

	if (ipa_ac_mod->conf_init() < 0) {
		logconfx("module %s: conf_init failed", ac_mod->mod_file);
		return (-1);
	}

	SLIST_INSERT_HEAD(&ac_mod_list, ac_mod, link);
	++nac_mods;
	return (0);
}

/*
 * Parse the "db_mod" parameter.
 */
static int
parse_db_mod(void *arg)
{
	const struct db_mod *db_mod2;
	struct ipa_db_mod *ipa_db_mod;
	struct db_mod *db_mod;
	char *mod_name, *sym;

	db_mod = mem_malloc(sizeof(*db_mod), m_anon);
	if (db_mod == NULL) {
		logconfx("mem_malloc failed");
		return (-1);
	}
	db_mod->mod_file = *(char **)arg;
	db_mod->mod_handle = dl_open(db_mod->mod_file);
	if (db_mod->mod_handle == NULL) {
		logconfx("dl_open(%s): %s", db_mod->mod_file, dl_error());
		return (-1);
	}
	mod_name = get_mod_name(db_mod->mod_file);
	if (mod_name == NULL)
		return (-1);
	if (mem_asprintf(m_anon, &sym, "%s_db_mod", mod_name) < 0) {
		logconfx("mem_asprintf failed");
		return (-1);
	}
	db_mod->ipa_db_mod =
	    (struct ipa_db_mod *)dl_lookup_sym(db_mod->mod_handle, sym);
	if (db_mod->ipa_db_mod == NULL) {
		logconfx("given module is not an IPA database module or "
		    "unknown symbol naming scheme is used");
		return (-1);
	}
	mem_free(sym, m_anon);
	mem_free(mod_name, m_anon);
	ipa_db_mod = db_mod->ipa_db_mod;

	/* Check ipa_db_mod API version. */
	if (ipa_db_mod->api_ver != IPA_DB_MOD_API_VERSION) {
		logconfx("module %s uses ipa_db_mod API version %u, "
		    "my ipa_db_mod API version is %u", db_mod->mod_file,
		    ipa_db_mod->api_ver, IPA_DB_MOD_API_VERSION);
		return (-1);
	}

	/* Check if module is thread-safe or vice versa. */
#ifdef WITH_PTHREAD
	if (!(ipa_db_mod->mod_flags & IPA_MOD_FLAG_PTHREAD_SAFE)) {
		logconfx("module %s must be thread-safe", db_mod->mod_file);
		return (-1);
	}
#else
	if (ipa_db_mod->mod_flags & IPA_MOD_FLAG_PTHREAD_SAFE) {
		logconfx("module %s must not be thread-safe",
		    db_mod->mod_file);
		return (-1);
	}
#endif /* WITH_PTHREAD */

	if (strcmp(ipa_db_mod->db_name, "null") == 0) {
		logconfx("module's database name is \"null\","
		    "this is a name of the built-in database");
		return (-1);
	}

	db_mod2 = db_mod_by_name(ipa_db_mod->db_name);
	if (db_mod2 != NULL) {
		logconfx("duplicated database name \"%s\" in %s and %s modules",
		    ipa_db_mod->db_name, db_mod->mod_file, db_mod2->mod_file);
		return (-1);
	}
	if (ipa_db_mod->conf_prefix != NULL) {
		const struct ac_mod *ac_mod;

		db_mod2 = db_mod_by_prefix(ipa_db_mod->conf_prefix);
		if (db_mod2 != NULL) {
			logconfx("duplicated configuration prefix \"%s\" in "
			    "%s and %s modules", ipa_db_mod->conf_prefix,
			    db_mod->mod_file, db_mod2->mod_file);
			return (-1);
		}
		ac_mod = ac_mod_by_prefix(ipa_db_mod->conf_prefix);
		if (ac_mod != NULL) {
			logconfx("duplicated configuration prefix \"%s\" in "
			    "%s and %s modules", ipa_db_mod->conf_prefix,
			    db_mod->mod_file, ac_mod->mod_file);
			return (-1);
		}
	}

	ipa_db_mod->suppfunc = &suppfunc;
	ipa_db_mod->memfunc = &memfunc;

	if (init_conf_tbls(db_mod->mod_file, 1,
	    ipa_db_mod->conf_sect_tbl, &db_mod->conf_sect_hash,
	    ipa_db_mod->conf_param_tbl, &db_mod->conf_param_hash) < 0)
		return (-1);

	if (ipa_db_mod->conf_init() < 0) {
		logconfx("module %s: conf_init failed", db_mod->mod_file);
		return (-1);
	}

	SLIST_INSERT_HEAD(&db_mod_list, db_mod, link);
	++ndb_mods;
	return (0);
}

/*
 * Parse the "ac_list" parameter.
 */
static int
parse_ac_list(void *arg)
{
	const struct ac_list **list_ptr;
	const char *ac_name;
	struct ac_elem *ac1, *ac2;
	struct ac_set *set1, *set2;
	struct ac_list *list;
	struct ac_mod *ac_mod;
	char *ptr;

	switch (section) {
#ifdef WITH_RULES
	case IPA_CONF_SECT_RULE:
		list_ptr = &currule->ac_list;
		break;
#endif
#ifdef WITH_AUTORULES
	case IPA_CONF_SECT_AUTORULE:
		list_ptr = &curautorule->ac_list;
		break;
#endif
	case IPA_CONF_SECT_RULEPAT:
		list_ptr = &currulepat->ac_list;
		break;
	default: /* IPA_CONF_SECT_GLOBAL */
		list_ptr = &global_ac_list;
	}

	if (*list_ptr != NULL) {
		logconfx("cannot re-define this parameter");
		return (-1);
	}

	set1 = NULL;
	list = NULL;

	for (ptr = *(char **)arg; ptr != NULL;) {
		/* Get the name of the next accounting system. */
		ac_name = ptr;
		ptr = strchr(ptr, ' ');
		if (ptr != NULL)
			*ptr++ = '\0';

		/* Handle "null" accounting system. */
		if (strcmp(ac_name, "null") == 0) {
			if (list != NULL || ptr != NULL) {
				logconfx("built-in accounting system \"null\" "
				    "cannot be used together with another "
				    "accounting systems");
				return (-1);
			}
			*list_ptr = &ac_list_null;
			return (0);
		}

		ac_mod = ac_mod_by_name(ac_name);
		if (ac_mod == NULL) {
			logconfx("cannot find module with \"%s\" "
			    "accounting system name", ac_name);
			return (-1);
		}

		if (set1 != NULL) {
			/* We already have set for current ac_list parameter. */
			STAILQ_FOREACH(ac1, list, link)
				if (strcmp(ac_name,
				    ac1->ipa_ac_mod->ac_name) == 0) {
					logconfx("duplicated accounting system "
					    "\"%s\"", ac_name);
					return (-1);
				}
		} else {
			/* Create new set for ac_list parameter. */
			set1 = mem_malloc(sizeof(*set1), m_anon);
			if (set1 == NULL) {
				logconfx("mem_malloc failed");
				return (-1);
			}
			list = &set1->list;
			STAILQ_INIT(list);
		}

		/* Add new ac element to ac_list. */
		ac1 = mem_malloc(sizeof(*ac1), m_anon);
		if (ac1 == NULL) {
			logconfx("mem_malloc failed");
			return (-1);
		}
		ac1->ipa_ac_mod = ac_mod->ipa_ac_mod;
		ac1->mod_file = ac_mod->mod_file;

		/* Increase reference counter if needed. */
		ac1->mod_ref_count = &ac_mod->mod_ref_count;
		switch (section) {
#ifdef WITH_RULES
		case IPA_CONF_SECT_RULE:
			/* FALLTHROUGH */
#endif
#ifdef WITH_AUTORULES
		case IPA_CONF_SECT_AUTORULE:
#endif
			ac_mod->mod_ref_count++;
			break;
		}

		STAILQ_INSERT_TAIL(list, ac1, link);
	}

	/*
	 * If we already have the same ac_list (with elements exactly
	 * in the same order), then free memory used by just allocated
	 * one (mod_ref_count was updated above).
	 */
	SLIST_FOREACH(set2, &ac_sets, link) {
		ac1 = STAILQ_FIRST(&set1->list);
		ac2 = STAILQ_FIRST(&set2->list);
		while (ac1 != NULL && ac2 != NULL) {
			if (ac1->ipa_ac_mod != ac2->ipa_ac_mod)
				break;
			ac1 = STAILQ_NEXT(ac1, link);
			ac2 = STAILQ_NEXT(ac2, link);
		}
		if (ac1 == NULL && ac2 == NULL) {
			/* Duplicated ac_list was found. */
			*list_ptr = &set2->list;
			free_ac_set(set1);
			return (0);
		}
	}

	/* New ac_list --> add it to ac_sets. */
	*list_ptr = list;
	SLIST_INSERT_HEAD(&ac_sets, set1, link);
	return (0);
}

/*
 * Parse the "db_list" parameter.
 */
static int
parse_db_list(void *arg)
{
	const struct db_list **list_ptr;
	const struct db_mod *db_mod;
	const char *db_name;
	struct db_elem *db1, *db2;
	struct db_set *set1, *set2;
	struct db_list *list;
	char *ptr;

	switch (section) {
#ifdef WITH_RULES
	case IPA_CONF_SECT_RULE:
		list_ptr = &currule->db_list;
		break;
#endif
#ifdef WITH_AUTORULES
	case IPA_CONF_SECT_AUTORULE:
		list_ptr = &curautorule->db_list;
		break;
#endif
#ifdef WITH_LIMITS
	case IPA_CONF_SECT_LIMIT:
		list_ptr = &curlimit->db_list;
		break;
#endif
#ifdef WITH_THRESHOLDS
	case IPA_CONF_SECT_THRESHOLD:
		list_ptr = &curthreshold->db_list;
		break;
#endif
	case IPA_CONF_SECT_RULEPAT:
		list_ptr = &currulepat->db_list;
		break;
	default: /* IPA_CONF_SECT_GLOBAL */
		list_ptr = &global_db_list;
	}

	if (*list_ptr != NULL) {
		logconfx("cannot re-define this parameter");
		return (-1);
	}

	set1 = NULL;
	list = NULL;

	for (ptr = *(char **)arg; ptr != NULL;) {
		/* Get the name of the next database. */
		db_name = ptr;
		ptr = strchr(ptr, ' ');
		if (ptr != NULL)
			*ptr++ = '\0';

		/* Handle "null" database. */
		if (strcmp(db_name, "null") == 0) {
			if (list != NULL || ptr != NULL) {
				logconfx("built-in \"null\" database cannot "
				    "be used together with another databases");
				return (-1);
			}
			*list_ptr = &db_list_null;
			return (0);
		}

		db_mod = db_mod_by_name(db_name);
		if (db_mod == NULL) {
			logconfx("cannot find module with \"%s\" database "
			   "name", db_name);
			return (-1);
		}

		if (set1 != NULL) {
			/* We already have set for current db_list parameter. */
			STAILQ_FOREACH(db1, list, link)
				if (strcmp(db_name,
				    db1->ipa_db_mod->db_name) == 0) {
					logconfx("duplicated database \"%s\"",
					    db_name);
					return (-1);
				}
		} else {
			/* Create new set for db_list parameter. */
			set1 = mem_malloc(sizeof(*set1), m_anon);
			if (set1 == NULL) {
				logconfx("mem_malloc failed");
				return (-1);
			}
			list = &set1->list;
			STAILQ_INIT(list);
		}

		/* Add new db element to db_list. */
		db1 = mem_malloc(sizeof(*db1), m_anon);
		if (db1 == NULL) {
			logconfx("mem_malloc failed");
			return (-1);
		}
		db1->ipa_db_mod = db_mod->ipa_db_mod;
		db1->mod_file = db_mod->mod_file;

		STAILQ_INSERT_TAIL(list, db1, link);
	}

	/*
	 * If we already have the same db_list (with elements exactly
	 * in the same order), then free memory used by just allocated one.
	 */
	SLIST_FOREACH(set2, &db_sets, link) {
		db1 = STAILQ_FIRST(&set1->list);
		db2 = STAILQ_FIRST(&set2->list);
		while (db1 != NULL && db2 != NULL) {
			if (db1->ipa_db_mod != db2->ipa_db_mod)
				break;
			db1 = STAILQ_NEXT(db1, link);
			db2 = STAILQ_NEXT(db2, link);
		}
		if (db1 == NULL && db2 == NULL) {
			/* Duplicated db_list was found. */
			*list_ptr = &set2->list;
			free_db_set(set1);
			return (0);
		}
	}

	/* New db_list --> add it to db_sets. */
	*list_ptr = list;
	SLIST_INSERT_HEAD(&db_sets, set1, link);
	return (0);
}

#ifdef WITH_RULES
/*
 * Parse the "ac_gather_add" parameter.
 */
static int
parse_ac_gather_add(void *arg)
{
	struct rule *rule;
	char *pat;
	int error;

	rule = currule;
	if (rule->acg_add_pat != NULL) {
		logconfx("cannot re-define this parameter");
		return (-1);
	}

	rule->acg_add_pat = pat = *(char **)arg;
	error = regcomp(&rule->acg_add_re, pat, REG_EXTENDED|REG_NOSUB);
	if (error != 0) {
		logconfx("regcomp(\"%s\"): %s", pat, regerrbuf(error));
		return (-1);
	}

	if (rule->acg_sub_pat == NULL)
		SLIST_INSERT_HEAD(&acg_list, rule, acg_link);
	return (0);
}

/*
 * Parse the "ac_gather_sub" parameter.
 */
static int
parse_ac_gather_sub(void *arg)
{
	struct rule *rule;
	char *pat;
	int error;

	rule = currule;
	if (rule->acg_sub_pat != NULL) {
		logconfx("cannot re-define this parameter");
		return (-1);
	}

	rule->acg_sub_pat = pat = *(char **)arg;
	error = regcomp(&rule->acg_sub_re, pat, REG_EXTENDED|REG_NOSUB);
	if (error != 0) {
		logconfx("regcomp(\"%s\"): %s", pat, regerrbuf(error));
		return (-1);
	}

	if (rule->acg_add_pat == NULL)
		SLIST_INSERT_HEAD(&acg_list, rule, acg_link);
	return (0);
}
#endif /* WITH_RULES */

/*
 * Parse "posix_re_pattern" parameter.
 */
static int
parse_posix_re_pattern(void *arg)
{
	posix_re_pattern = (signed char)*(int *)arg;
	return (0);
}

/*
 * Parse "shell_path" parameter.
 */
static int
parse_shell_path(void *arg)
{
	shell_path = *(char **)arg;
	if (*shell_path != '/') {
		logconfx("shell path should be absolute");
		return (-1);
	}
	return (0);
}

/*
 * Parse "shell_arg1" parameter.
 */
static int
parse_shell_arg1(void *arg)
{
	shell_arg1 = *(char **)arg;
	return (0);
}

/*
 * Parse "ctl_enable" parameter.
 */
static int
parse_ctl_enable(void *arg)
{
	ctl_enable = (signed char)*(int *)arg;
	return (0);
}

/*
 * Parse "ctl_socket_path" parameter.
 */
static int
parse_ctl_socket_path(void *arg)
{
	struct sockaddr_un addr;

	ctl_socket_path = *(char **)arg;
	if (*ctl_socket_path != '/') {
		logconfx("control socket path should be absolute");
		return (-1);
	}
	if (strlen(ctl_socket_path) + 1 > sizeof(addr.sun_path)) {
		logconfx("too long path (> %lu) for control socket path",
		    (unsigned long)sizeof(addr.sun_path) - 1);
		return (-1);
	}
	return (0);
}

/*
 * Parse "ctl_socket_perm" parameter.
 */
static int
parse_ctl_socket_perm(void *arg)
{
	const char *ptr;
	mode_t val;
	int error;

	ptr = *(char **)arg;
	val = 0;
	error = 0;
	for (; *ptr != '\0'; ++ptr)
		switch (*ptr) {
		case 'u':
			if (!(val & S_IWUSR))
				val |= S_IWUSR;
			else
				error = 1;
			break;
		case 'g':
			if (!(val & S_IWGRP))
				val |= S_IWGRP;
			else
				error = 1;
			break;
		case 'o':
#ifdef CTL_CHECK_CREDS
			if (!(val & S_IWOTH))
				val |= S_IWOTH;
			else
				error = 1;
			break;
#else
			logconfx("cannot allow to write to socket for "
			    "other users because messages credentials are "
			    "disabled");
			return (-1);
#endif
		default:
			error = 1;
		}
	if (error) {
		logconfx("wrong value");
		return (-1);
	}
	ctl_socket_perm = val | S_IRUSR;
	return (0);
}

/*
 * Parse "ctl_timeout" parameter.
 */
static int
parse_ctl_timeout(void *arg)
{
	uint64_t timeval;

	timeval = *(uint64_t *)arg;
	if (timeval == 0 || timeval >= 30) {
		logconfx("argument should be greater than zero seconds and "
		    "less than 30 seconds");
		return (-1);
	}
	ctl_timeout = (unsigned int)timeval;
	return (0);
}

/*
 * Parse "ctl_query_max_size" parameter.
 */
static int
parse_ctl_query_max_size(void *arg)
{
	uint64_t val;

	val = *(uint64_t *)arg;
	if (val > 10 * MBYTE) {
		logconfx("too big value (> 10M)");
		return (-1);
	}
	ctl_query_max_size = (size_t)val;
	return (0);
}

/*
 * Parse "debug_ctl" parameter.
 */
static int
parse_debug_ctl(void *arg)
{
	debug_ctl = check_debug_level(*(uint32_t *)arg, 1);
	return (debug_ctl);
}

#ifdef CTL_CHECK_CREDS
/*
 * Find ctl class with the given name.
 */
static const struct ctl_acl_class *
ctl_class_by_name(const char *class_name)
{
	const struct ctl_acl_class *cl;

	STAILQ_FOREACH(cl, &ctl_acl_classes, link)
		if (strcmp(cl->class_name, class_name) == 0)
			return (cl);
	return (NULL);
}

/*
 * Parse "ctl_acl_class" parameter.
 */
static int
parse_ctl_acl_class(void *arg)
{
	struct ctl_acl_elem *elem;
	struct ctl_acl_class *class;
	char *ptr, *str;

	/* class [elem1 elem2 ...] */
	str = *(char **)arg;
	ptr = strchr(str, ' ');
	if (ptr != NULL)
		*ptr = '\0';

	/* Check if we already have class with the same name. */
	if (ctl_class_by_name(str) != NULL) {
		logconfx("duplicated ctl class \"%s\"", str);
		return (-1);
	}

	/* New class. */
	class = mzone_alloc(ctl_acl_class_mzone);
	if (class == NULL) {
		logconfx("mzone_alloc failed");
		return (-1);
	}
	class->class_name = mem_strdup(str, m_ctl);
	if (class->class_name == NULL) {
		logconfx("mem_strdup failed");
		return (-1);
	}

	/* Initialize ACL for new class. */
	STAILQ_INIT(&class->list);

	/* Link new class to all classes list. */
	STAILQ_INSERT_TAIL(&ctl_acl_classes, class, link);

	if (ptr == NULL)
		/* Empty class. */
		return (0);

	str = ptr + 1;

	do {
		elem = mzone_alloc(ctl_acl_elem_mzone);
		if (elem == NULL) {
			logconfx("mzone_alloc failed");
			return (-1);
		}
		STAILQ_INSERT_TAIL(&class->list, elem, link);

		if (*str == '!') {
			elem->allowed = 0;
			++str;
		} else
			elem->allowed = 1;

		ptr = str;
		str = strchr(str, ' ');
		if (str != NULL)
			*str++ = '\0';
		if (*ptr == '%') {
			/* Group. */
			elem->user = NULL;
			elem->group = mem_strdup(++ptr, m_ctl);
			if (elem->group == NULL) {
				logconfx("mem_strdup failed");
				return (-1);
			}
		} else {
			/* User. */
			elem->user = mem_strdup(ptr, m_ctl);
			if (elem->user == NULL) {
				logconfx("mem_strdup failed");
				return (-1);
			}
			elem->group = NULL;
		}
	} while (str != NULL);

	return (0);
}

static int
parse_ctl_xxx_acl(const struct ctl_acl_class **classpp)
{
	*classpp = ctl_class_by_name(parser_args);
	if (*classpp == NULL) {
		logconfx("cannot find ctl class \"%s\"", parser_args);
		return (-1);
	}
	return (0);
}

/*
 * Parse "ctl_dump_acl" parameter.
 */
/* ARGSUSED */
static int
parse_ctl_dump_acl(void *arg ATTR_UNUSED)
{
	return (parse_ctl_xxx_acl(&ctl_dump_acl));
}

/*
 * Parse "ctl_freeze_acl" parameter.
 */
/* ARGSUSED */
static int
parse_ctl_freeze_acl(void *arg ATTR_UNUSED)
{
	return (parse_ctl_xxx_acl(&ctl_freeze_acl));
}

/*
 * Parse "ctl_stat_acl" parameter.
 */
/* ARGSUSED */
static int
parse_ctl_stat_acl(void *arg ATTR_UNUSED)
{
	return (parse_ctl_xxx_acl(&ctl_stat_acl));
}

/*
 * Parse "ctl_rule_acl" parameter.
 */
/* ARGSUSED */
static int
parse_ctl_rule_acl(void *arg ATTR_UNUSED)
{
	const struct ctl_acl_class **classpp;

	switch (section) {
#ifdef WITH_RULES
	case IPA_CONF_SECT_RULE:
		classpp = &currule->ctl_rule_acl;
		break;
#endif
#ifdef WITH_AUTORULES
	case IPA_CONF_SECT_AUTORULE:
		classpp = &curautorule->ctl_rule_acl;
		break;
#endif
	case IPA_CONF_SECT_RULEPAT:
		classpp = &currulepat->ctl_rule_acl;
		break;
	default: /* IPA_CONF_SECT_GLOBAL */
		classpp = &global_ctl_rule_acl;
	}

	return (parse_ctl_xxx_acl(classpp));
}

#endif /* CTL_CHECK_CREDS */

#ifdef WITH_THRESHOLDS
/*
 * Parse "debug_threshold" parameter.
 */
static int
parse_debug_threshold(void *arg)
{
	signed char level;

	level = check_debug_level(*(uint32_t *)arg, 1);
	if (level < 0)
		return (-1);

	switch (section) {
	case IPA_CONF_SECT_GLOBAL:
		global_debug_threshold = level;
		break;
#ifdef WITH_RULES
	case IPA_CONF_SECT_RULE:
		currule->debug_threshold = level;
		break;
#endif
#ifdef WITH_AUTORULES
	case IPA_CONF_SECT_AUTORULE:
		curautorule->debug_threshold = level;
		break;
#endif
	case IPA_CONF_SECT_RULEPAT:
		currulepat->debug_threshold = level;
		break;
	}

	return (0);
}

/*
 * Parse "debug_threshold_init" parameter.
 */
static int
parse_debug_threshold_init(void *arg)
{
	signed char level;

	level = check_debug_level(*(uint32_t *)arg, 1);
	if (level < 0)
		return (-1);

	switch (section) {
	case IPA_CONF_SECT_GLOBAL:
		global_debug_threshold_init = level;
		break;
#ifdef WITH_RULES
	case IPA_CONF_SECT_RULE:
		currule->debug_threshold_init = level;
		break;
#endif
#ifdef WITH_AUTORULES
	case IPA_CONF_SECT_AUTORULE:
		curautorule->debug_threshold_init = level;
		break;
#endif
	case IPA_CONF_SECT_RULEPAT:
		currulepat->debug_threshold_init = level;
		break;
	}

	return (0);
}

/*
 * Parse "load_threshold" parameter.
 */
static int
parse_load_threshold(void *arg)
{
	if (section == IPA_CONF_SECT_THRESHOLD)
		curthreshold->load_thr = (char)*(int *)arg;
	else /* IPA_CONF_SECT_GLOBAL */
		global_load_threshold = (char)*(int *)arg;
	return (0);
}

/*
 * Parse "threshold" section.
 */
static int
parse_threshold_sect(void *arg)
{
	const char *name;
	struct threshold *threshold;

	name = *(char **)arg;
	if (conf_validate_name(name) < 0)
		return (-1);
	STAILQ_FOREACH(threshold, thresholds_list, link)
		if (strcmp(threshold->name, name) == 0) {
			logconfx("this section is duplicated");
			return (-1);
		}
	threshold = mzone_alloc(threshold_mzone);
	if (threshold == NULL) {
		logconfx("mzone_alloc failed");
		return (-1);
	}
	threshold->name = mem_strdup(name, m_anon);
	if (threshold->name == NULL) {
		logconfx("mem_strdup failed");
		return (-1);
	}
	STAILQ_INSERT_TAIL(thresholds_list, threshold, link);
	threshold->no = thresholdno++;

	threshold->thr_flags = THRESHOLD_FLAG_ACTIVE;

	threshold->cnt_neg = 0;
	threshold->cnt_slice = NULL;
	threshold->cnt_slice_sign = NULL;

	threshold->time_width = 0;
	threshold->time_slice = NULL;
	threshold->worktime = NULL;

	threshold->db_list = NULL;

	threshold->below_lim = threshold->equal_lim = threshold->above_lim = 0;

	threshold_init_cmds(threshold);

	threshold->wpid.type = WPID_TYPE_THRESHOLD;
	threshold->wpid.pid = 0;
	threshold->wpid.u.threshold = threshold;

#ifdef WITH_RULES
	threshold->rule = currule;
#endif

	threshold->thr = threshold->thr_dev = 0;
	threshold->thr_dev_pc = 0;

	threshold->info = NULL;

	threshold->load_thr = -1;
	threshold->thr_type = -1;

	if (section_top == IPA_CONF_SECT_RULE)
		++nstatthresholds;
	section_top = IPA_CONF_SECT_THRESHOLD;

	if (parser_local_sym_add("threshold", threshold->name, 0) < 0)
		return (-1);

	curthreshold = threshold;
	conf_event_no = threshold->no;
	conf_event_arg = threshold->name;

	return (0);
}

/*
 * Parse "threshold" parameter.
 */
static int
parse_threshold_param(void *arg)
{
	uint64_t *ptr;

	if (THRESHOLD_IS_SET(curthreshold)) {
		logconfx("cannot re-define this parameter");
		return (-1);
	}
	curthreshold->thr_flags |= THRESHOLD_FLAG_SET;

	ptr = (uint64_t *)arg;
	curthreshold->thr = *(ptr + 1);
	curthreshold->cnt_type = (unsigned char)*ptr;
	return (0);
}

/*
 * Parse "threshold_deviation" parameter.
 */
/* ARGSUSED */
static int
parse_threshold_deviation(void *arg ATTR_UNUSED)
{
	uint64_t thr_dev;
	struct threshold *threshold;
	unsigned char cnt_type;

	threshold = curthreshold;
	if (THRESHOLD_IS_NOTSET(threshold)) {
		logconfx("parameter \"threshold\" should be specified before");
		return (-1);
	}

	if (get_value_or_per_cent(&thr_dev, &cnt_type) < 0)
		return (-1);
	if (thr_dev == 0) {
		logconfx("argument should be greater than zero");
		return (-1);
	}
	if (cnt_type == IPA_CONF_TYPE_PER_CENT)
		threshold->thr_dev_pc = (unsigned int)thr_dev;
	else {
		if (cnt_type != threshold->cnt_type) {
			logconfx("different arguments types for \"threshold\" "
			    "and \"threshold_deviation\" parameters");
			return (-1);
		}
		threshold->thr_dev_pc = 0;
	}
	threshold->thr_dev = thr_dev;

	return (0);
}

/*
 * Parse "threshold_time_width" parameter.
 */
static int
parse_threshold_time_width(void *arg)
{
	uint64_t timeval;

	timeval = *(uint64_t *)arg;
	if (timeval == 0) {
		logconfx("argument should be greater than zero seconds");
		return (-1);
	}
	if (timeval > UINT_MAX) {
		logconfx("too big value %"PRIu64" for 'unsigned int' type",
		    timeval);
		return (-1);
	}

	if (section == IPA_CONF_SECT_THRESHOLD)
		curthreshold->time_width = (unsigned int)timeval;
	else /* IPA_CONF_SECT_GLOBAL */
		global_threshold_time_width = (unsigned int)timeval;

	return (0);
}

/*
 * Parse "threshold_time_slice" parameter.
 */
static int
parse_threshold_time_slice(void *arg)
{
	uint64_t timeval;
	struct tevent *tevent;

	timeval = *(uint64_t *)arg;
	if (timeval == 0) {
		logconfx("argument should be greater than zero seconds");
		return (-1);
	}
	if (timeval > UINT_MAX) {
		logconfx("too big value %"PRIu64" for 'unsigned int' type",
		    timeval);
		return (-1);
	}
	if (SECONDS_IN_DAY % timeval != 0) {
		logconfx("number of seconds in one day must be divisible "
		    "by the value of \"threshold_time_slice\" parameter");
		return (-1);
	}

	tevent = find_tevent((unsigned int)timeval);
	if (tevent == NULL)
		return (-1);

	if (section == IPA_CONF_SECT_THRESHOLD)
		curthreshold->time_slice = tevent;
	else /* IPA_CONF_SECT_GLOBAL */
		global_threshold_time_slice = tevent;

	return (0);
}

/*
 * Parse "threshold_type" parameter.
 */
static int
parse_threshold_type(void *arg)
{
	uint32_t val;

	val = *(uint32_t *)arg;
	if (val > THRESHOLD_JUMP_OVER_ALLBITS) {
		logconfx("incorrect value for this parameter");
		return (-1);
	}

	if (section == IPA_CONF_SECT_THRESHOLD)
		curthreshold->thr_type = (char)val;
	else /* IPA_CONF_SECT_GLOBAL */
		global_threshold_type = (char)val;

	return (0);
}

/*
 * Parse "threshold_balance" parameter.
 */
static int
parse_threshold_balance(void *arg)
{
	char *ptr1, *ptr2;
	unsigned int i, val[3];

	ptr1 = *(char **)arg;
	for (i = 0, ptr2 = ptr1; i < 3; ptr1 = ptr2, ++i) {
		if (i != 2) {
			ptr2 = strchr(ptr2, ':');
			*ptr2 = '\0';
			++ptr2;
		}
		if (*ptr1 != '-') {
			if (strto_u_int(&val[i], ptr1, (char **)NULL) < 0)
				return (-1);
			switch (val[i]) {
			case 0:
			case UINT_MAX:
				logconfx("values should be greater than zero "
				    "and less than %u", UINT_MAX);
				return (-1);
			}
		} else
			val[i] = UINT_MAX;
	}

	if (section == IPA_CONF_SECT_THRESHOLD) {
		curthreshold->below_lim = curthreshold->below_cnt = val[0];
		curthreshold->equal_lim = curthreshold->equal_cnt = val[1];
		curthreshold->above_lim = curthreshold->above_cnt = val[2];
	} else { /* IPA_CONF_SECT_GLOBAL */
		global_threshold_below_lim = val[0];
		global_threshold_equal_lim = val[1];
		global_threshold_above_lim = val[2];
	}

	return (0);
}

/*
 * Check if threshold's time_width and time_slice can be used together.
 */
static int
check_threshold_times(unsigned int time_width,
    const struct tevent *time_slice, int must_exist)
{
	if (time_width == 0 && time_slice == NULL) {
		if (must_exist && global_threshold_time_width == 0) {
			logconfx("parameter \"threshold_time_width\" is not "
			    "defined neigher in this threshold{} section nor "
			    "in the global{} section");
			return (-1);
		}
		return (0);
	}

	if ((time_width != 0 && time_slice == NULL) ||
	    (time_width == 0 && time_slice != NULL)) {
		logconfx("parameters \"threshold_time_width\" and "
		    "\"threshold_time_slice\" should be defined together "
		    "in one section");
		return (-1);
	}

	if (time_width < time_slice->event_step) {
		logconfx("the value of \"threshold_time_width\" parameter must "
		    "be greater than the value of \"threshold_time_slice\" "
		    "parameter");
		return (-1);
	}

	if (time_width % time_slice->event_step != 0) {
		logconfx("the value of \"threshold_time_width\" parameter must "
		    "be divisible by the value of \"threshold_time_slice\" "
		    "parameter");
		return (-1);
	}

	return (0);
}
#endif /* WITH_THRESHOLDS */

#ifdef WITH_AUTORULES
/*
 * Parse "debug_autorule" parameter.
 */
static int
parse_debug_autorule(void *arg)
{
	debug_autorule = check_debug_level(*(uint32_t *)arg, 1);
	return (debug_autorule);
}

/*
 * Parse the "autorule" section.
 */
static int
parse_autorule(void *arg)
{
	const char *name;
	struct autorule *autorule;

	name = *(char **)arg;
	if (conf_validate_name(name) < 0)
		return (-1);
	if (autorule_by_name(name) != NULL) {
		logconfx("this section is duplicated");
		return (-1);
	}
	if (marray_alloc(autorules_marray, &nautorules, 1) < 0) {
		logconfx("marray_alloc failed");
		return (-1);
	}
	autorule = autorules + nautorules;

	autorule->name = mem_strdup(name, m_anon);
	if (autorule->name == NULL) {
		logconfx("mem_strdup failed");
		return (-1);
	}
	autorule->no = nautorules++;

	autorule->arule_flags = AUTORULE_FLAG_ACTIVE;

	autorule->update_tevent = autorule->append_tevent = NULL;
	autorule->worktime = autorule->worktime_rule = NULL;

	autorule->ac_list = NULL;
	autorule->db_list = NULL;

	autorule->debug_exec = -1;

#ifdef WITH_LIMITS
	autorule->debug_limit = autorule->debug_limit_init = -1;
	limits_list = &autorule->limits;
	STAILQ_INIT(limits_list);
	limitno = 0;
#endif

#ifdef WITH_THRESHOLDS
	autorule->debug_threshold = autorule->debug_threshold_init = -1;
	thresholds_list = &autorule->thresholds;
	STAILQ_INIT(thresholds_list);
	thresholdno = 0;
#endif

#ifdef CTL_CHECK_CREDS
	autorule->ctl_rule_acl = NULL;
#endif

	autorule->nrules = 0;

	cmds_rule_init(&autorule->rc[RC_STARTUP]);
	cmds_rule_init(&autorule->rc[RC_SHUTDOWN]);

	section_first = section_top = IPA_CONF_SECT_AUTORULE;

	if (parser_local_sym_add("autorule", autorule->name, 0) < 0)
		return (-1);

	curautorule = autorule;
	conf_event_no = autorule->no;
	conf_event_arg = autorule->name;

	return (0);
}

#endif /* WITH_AUTORULES */

/*
 * Parse "sleep_after_dump" parameter.
 */
static int
parse_sleep_after_dump(void *arg)
{
	uint64_t timeval;

	timeval = *(uint64_t *)arg;
	if (timeval == 0 || timeval >= 5 * SECONDS_IN_MINUTE) {
		logconfx("argument should be greater than zero seconds and "
		    "less than 5 minutes");
		return (-1);
	}
	sleep_after_dump = (unsigned int)timeval;
	return (0);
}

/*
 * Parse "sensitive_time" parameter.
 */
static int
parse_sensitive_time(void *arg)
{
	uint64_t timeval;

	timeval = *(uint64_t *)arg;
	if (timeval == 0) {
		logconfx("argument should be greater than zero");
		return (-1);
	}
	if (timeval >= 5 * SECONDS_IN_MINUTE) {
		logconfx("argument should be less than 5 minutes");
		return (-1);
	}
	sensitive_time = (unsigned int)timeval;
	return (0);
}

/*
 * Log information about chain of included files.
 */
static void
log_include_history(void)
{
	const struct parser_pb *pb;
	const char *fname;

	pb = parser_top_pb();
	if (pb != NULL) {
		fname = pb->fname;
		while ((pb = parser_top_pb()) != NULL) {
			if (fname != pb->fname) {
				fname = pb->fname;
				logconfe("included from %s:%u",
				    fname, pb->lineno);
			}
			(void)parser_pop_pb();
		}
	}
}

static int
set_curcmdl(void)
{
#ifdef WITH_LIMITS
	static void *ptr;
#endif
	unsigned int x;

	switch (section) {
	case IPA_CONF_SECT_STARTUP:
	case IPA_CONF_SECT_SHUTDOWN:
		x = section_rc;
		switch (section_top) {
#ifdef WITH_RULES
		case IPA_CONF_SECT_RULE:
			curcmds = &currule->rc[x].cmds;
# ifdef WITH_LIMITS
			ptr = &currule->rc[x];
# endif
			break;
#endif
#ifdef WITH_AUTORULES
		case IPA_CONF_SECT_AUTORULE:
			curcmds = &curautorule->rc[x].cmds;
# ifdef WITH_LIMITS
			ptr = &curautorule->rc[x];
# endif
			break;
#endif
#ifdef WITH_LIMITS
		case IPA_CONF_SECT_LIMIT:
			curcmds = &curlimit->rc[x].cmds;
			ptr = &curlimit->rc[x];
			break;
#endif
#ifdef WITH_SUBLIMITS
		case SECT_SUBLIMIT:
			curcmds = &cursublimit->rc[x].cmds;
			ptr = &cursublimit->rc[x];
			break;
#endif
#ifdef WITH_THRESHOLDS
		case IPA_CONF_SECT_THRESHOLD:
			curcmds = &curthreshold->rc[x];
			break;
#endif
		case IPA_CONF_SECT_RULEPAT:
			curcmds = &currulepat->rc[x].cmds;
#ifdef WITH_LIMITS
			ptr = &currulepat->rc[x];
#endif
			break;
		default: /* IPA_CONF_SECT_ROOT */
			curcmds = &cmds_global[x];
		}
		break;
#ifdef WITH_LIMITS
	case IPA_CONF_SECT_RESTART:
		curcmds = &curlimit->restart.cmds;
		break;
	case IPA_CONF_SECT_REACH:
# ifdef WITH_SUBLIMITS
		if (section_top == SECT_SUBLIMIT)
			curcmds = &cursublimit->reach;
		else
# endif
			curcmds = &curlimit->reach;
		break;
	case IPA_CONF_SECT_EXPIRE:
		curcmds = &curlimit->expire.cmds;
		break;
	case IPA_CONF_SECT_IF_REACHED:
		curcmds = &((struct cmds_limit *)ptr)->cmds_reached;
		break;
	case IPA_CONF_SECT_IF_NOT_REACHED:
		curcmds = &((struct cmds_limit *)ptr)->cmds_not_reached;
		break;
	case IPA_CONF_SECT_IF_ALL_REACHED:
		curcmds = &((struct cmds_rule *)ptr)->cmds_all_reached;
		break;
	case IPA_CONF_SECT_IF_ALL_NOT_REACHED:
		curcmds = &((struct cmds_rule *)ptr)->cmds_all_not_reached;
		break;
	case IPA_CONF_SECT_IF_ANY_REACHED:
		curcmds = &((struct cmds_rule *)ptr)->cmds_any_reached;
		break;
	case IPA_CONF_SECT_IF_ANY_NOT_REACHED:
		curcmds = &((struct cmds_rule *)ptr)->cmds_any_not_reached;
		break;
#endif /* WITH_LIMITS */
#ifdef WITH_THRESHOLDS
	case IPA_CONF_SECT_BELOW_THRESHOLD:
		curcmds = &curthreshold->below_thr;
		break;
	case IPA_CONF_SECT_ABOVE_THRESHOLD:
		curcmds = &curthreshold->above_thr;
		break;
	case IPA_CONF_SECT_EQUAL_THRESHOLD:
		curcmds = &curthreshold->equal_thr;
		break;
#endif /* WITH_THRESHOLDS */
	}

	if (curcmds->sect_set) {
		logconfx("this section is duplicated");
		return (-1);
	}
	curcmds->sect_set = 1;

	return (0);
}

static int
set_global_params(void)
{
	unsigned int i;

	global_section_set = 1;
	if (only_abs_paths < 0)
		only_abs_paths = 0;
	if (keep_rules_order < 0)
		keep_rules_order = 0;
	if (wakeup_time == 0)
		wakeup_time = WAKEUP_TIME_DEF;
	if (sensitive_time == 0)
		sensitive_time = SENSITIVE_TIME_DEF;
	if (global_update_tevent == NULL) {
		global_update_tevent = find_tevent(UPDATE_TIME_DEF);
		if (global_update_tevent == NULL) {
			logconfe("find_tevent failed for default value of"
			    "\"update_time\" parameter");
			return (-1);
		}
	}
	if (global_ac_list == NULL)
		global_ac_list = &ac_list_null;
	if (global_db_list == NULL)
		global_db_list = &db_list_null;
	if (global_worktime == NULL)
		global_worktime = &worktime_default;
#ifdef WITH_LIMITS
	if (global_load_limit < 0)
		global_load_limit = 0;
#endif
#ifdef WITH_THRESHOLDS
	if (global_threshold_type < 0)
		global_threshold_type = 0;
	if (global_load_threshold < 0)
		global_load_threshold = 0;
	if (global_threshold_below_lim == 0)
		global_threshold_below_lim = global_threshold_equal_lim =
		    global_threshold_above_lim = UINT_MAX;
#endif
	for (i = 0; i < GLOBAL_DEBUG_PARAM_TBL_SIZE; ++i)
		if (*global_debug_param[i].value < 0)
			*global_debug_param[i].value = 0;
	for (i = 0; i < ROOT_DEBUG_PARAM_TBL_SIZE; ++i)
		if (*root_debug_param[i].value < 0)
			*root_debug_param[i].value = 0;

	if (shell_path == NULL)
		shell_path = shell_path_default;
	if (shell_arg1 == NULL)
		shell_arg1 = shell_arg1_default;
	if (cmds_global[RC_STARTUP].sync < 0)
		cmds_global[RC_STARTUP].sync = 1;
	if (cmds_global[RC_SHUTDOWN].sync < 0)
		cmds_global[RC_SHUTDOWN].sync = 1;
	if (value_units < 0)
		value_units = 0;

	if (ctl_enable < 0)
		ctl_enable = 0;
	if (ctl_socket_path == NULL)
		ctl_socket_path = ctl_socket_path_default;
	if (ctl_socket_perm == 0)
		ctl_socket_perm = CTL_SOCKET_PERM_DEF;
	if (ctl_timeout == 0)
		ctl_timeout = CTL_TIMEOUT_DEF;
	if (ctl_query_max_size == 0)
		ctl_query_max_size = CTL_QUERY_MAX_SIZE;

	return (0);
}

static const unsigned int sect_root[] = { IPA_CONF_SECT_ROOT, 0 };
#ifdef WITH_RULES
static const unsigned int sect_rule[] = { IPA_CONF_SECT_RULE, 0 };
#endif
static const unsigned int sect_rulepat[] = { IPA_CONF_SECT_RULEPAT, 0 };
static const unsigned int sect_any_rule[] = { IPA_CONF_SECT_GLOBAL,
#ifdef WITH_RULES
	IPA_CONF_SECT_RULE,
#endif
#ifdef WITH_AUTORULES
	IPA_CONF_SECT_AUTORULE,
#endif
	IPA_CONF_SECT_RULEPAT, 0
};
static const unsigned int sect_for_exec[] = { IPA_CONF_SECT_STARTUP,
	IPA_CONF_SECT_SHUTDOWN,
#ifdef WITH_LIMITS
	IPA_CONF_SECT_RESTART, IPA_CONF_SECT_REACH, IPA_CONF_SECT_EXPIRE,
	IPA_CONF_SECT_IF_REACHED, IPA_CONF_SECT_IF_NOT_REACHED,
	IPA_CONF_SECT_IF_ANY_REACHED, IPA_CONF_SECT_IF_ANY_NOT_REACHED,
	IPA_CONF_SECT_IF_ALL_REACHED, IPA_CONF_SECT_IF_ALL_NOT_REACHED,
#endif
#ifdef WITH_THRESHOLDS
	IPA_CONF_SECT_BELOW_THRESHOLD, IPA_CONF_SECT_ABOVE_THRESHOLD,
	IPA_CONF_SECT_EQUAL_THRESHOLD,
#endif
	0
};
#ifdef WITH_ANY_LIMITS
static const unsigned int sect_for_any_limit[] = {
# ifdef WITH_RULES
	IPA_CONF_SECT_RULE,
# endif
# ifdef WITH_AUTORULES
	IPA_CONF_SECT_AUTORULE,
# endif
	IPA_CONF_SECT_RULEPAT, 0
};
#endif
#ifdef WITH_THRESHOLDS
static const unsigned int sect_threshold[] = { IPA_CONF_SECT_THRESHOLD, 0 };
static const unsigned int sect_global_threshold[] = { IPA_CONF_SECT_GLOBAL,
	IPA_CONF_SECT_THRESHOLD, 0
};
#endif
#ifdef WITH_LIMITS
static const unsigned int sect_limit[] = { IPA_CONF_SECT_LIMIT, 0 };
static const unsigned int sect_global_limit[] = { IPA_CONF_SECT_GLOBAL,
	IPA_CONF_SECT_LIMIT, 0
};
static const unsigned int sect_restart[] = { IPA_CONF_SECT_RESTART, 0 };
static const unsigned int sect_expire[] = { IPA_CONF_SECT_EXPIRE, 0 };
static const unsigned int sect_startup_shutdown[] = { IPA_CONF_SECT_STARTUP,
	IPA_CONF_SECT_SHUTDOWN, 0
};
static const unsigned int sect_for_reach[] = { IPA_CONF_SECT_LIMIT,
# ifdef WITH_SUBLIMITS
	SECT_SUBLIMIT,
# endif
	0
};
#endif /* WITH_LIMITS */
static const unsigned int sect_for_startup_shutdown[] = { IPA_CONF_SECT_ROOT,
#ifdef WITH_RULES
	IPA_CONF_SECT_RULE,
#endif
#ifdef WITH_AUTORULES
	IPA_CONF_SECT_AUTORULE,
#endif
#ifdef WITH_LIMITS
	IPA_CONF_SECT_LIMIT,
#endif
#ifdef WITH_SUBLIMITS
	SECT_SUBLIMIT,
#endif
#ifdef WITH_THRESHOLDS
	IPA_CONF_SECT_THRESHOLD,
#endif
	IPA_CONF_SECT_RULEPAT, 0
};
#if defined(WITH_RULES) || defined(WITH_ANY_LIMITS)
static const unsigned int sect_for_info[] = {
# ifdef WITH_RULES
	IPA_CONF_SECT_RULE,
# endif
# ifdef WITH_LIMITS
	IPA_CONF_SECT_LIMIT,
# endif
# ifdef WITH_THRESHOLDS
	IPA_CONF_SECT_THRESHOLD,
#endif
	0
};
#endif
static const unsigned int sect_for_db_list[] = { IPA_CONF_SECT_GLOBAL,
#ifdef WITH_RULES
	IPA_CONF_SECT_RULE,
#endif
#ifdef WITH_AUTORULES
	IPA_CONF_SECT_AUTORULE,
#endif
#ifdef WITH_LIMITS
	IPA_CONF_SECT_LIMIT,
#endif
#ifdef WITH_THRESHOLDS
	IPA_CONF_SECT_THRESHOLD,
#endif
	IPA_CONF_SECT_RULEPAT, 0
};
static const unsigned int sect_for_worktime[] = { IPA_CONF_SECT_GLOBAL,
#ifdef WITH_RULES
	IPA_CONF_SECT_RULE,
#endif
#ifdef WITH_AUTORULES
	IPA_CONF_SECT_AUTORULE,
#endif
#ifdef WITH_LIMITS
	IPA_CONF_SECT_LIMIT,
#endif
#ifdef WITH_THRESHOLDS
	IPA_CONF_SECT_THRESHOLD,
#endif
	IPA_CONF_SECT_RULEPAT, 0
};
#ifdef WITH_AUTORULES
static const unsigned int sect_autorule[] = { IPA_CONF_SECT_AUTORULE, 0 };
#endif

/*
 * Sections in ipa.conf.
 */
static ipa_conf_sect conf_sect_tbl[] = {
#ifdef WITH_RULES
	{ "rule", IPA_CONF_SECT_RULE, 1, NULL, NULL,
	  IPA_CONF_TYPE_MISC, sect_root, parse_rule
	},
#endif
	{ "rulepat", IPA_CONF_SECT_RULEPAT, 1, NULL, NULL,
	   IPA_CONF_TYPE_STRING, sect_root, parse_rulepat
	},
	{ "global", IPA_CONF_SECT_GLOBAL, 0, NULL, NULL,
	  IPA_CONF_TYPE_MISC, sect_root, parse_global
	},
#ifdef WITH_AUTORULES
	{ "autorule", IPA_CONF_SECT_AUTORULE, 1, NULL, NULL,
	  IPA_CONF_TYPE_MISC, sect_root, parse_autorule
	},
#endif
#ifdef WITH_LIMITS
	{ "limit", IPA_CONF_SECT_LIMIT, 1, NULL, NULL,
	   IPA_CONF_TYPE_MISC, sect_for_any_limit, parse_limit_sect
	},
	{ "restart", IPA_CONF_SECT_RESTART, 0, NULL, NULL,
	  IPA_CONF_TYPE_MISC, sect_limit, NULL
	},
	{ "reach", IPA_CONF_SECT_REACH, 0, NULL, NULL,
	   IPA_CONF_TYPE_MISC, sect_for_reach, NULL
	},
	{ "expire", IPA_CONF_SECT_EXPIRE, 0, NULL, NULL,
	   IPA_CONF_TYPE_MISC, sect_limit, NULL
	},
# ifdef WITH_SUBLIMITS
	{ "sublimit", SECT_SUBLIMIT, -1, NULL, NULL,
	  IPA_CONF_TYPE_MISC, sect_limit, parse_sublimit
	},
# endif
	{ "if_reached", IPA_CONF_SECT_IF_REACHED, 0, NULL, NULL,
	   IPA_CONF_TYPE_MISC, sect_startup_shutdown, NULL
	},
	{ "if_not_reached", IPA_CONF_SECT_IF_NOT_REACHED, 0, NULL, NULL,
	   IPA_CONF_TYPE_MISC, sect_startup_shutdown, NULL
	},
	{ "if_any_reached", IPA_CONF_SECT_IF_ANY_REACHED, 0, NULL, NULL,
	  IPA_CONF_TYPE_MISC, sect_startup_shutdown, NULL
	},
	{ "if_all_reached", IPA_CONF_SECT_IF_ALL_REACHED, 0, NULL, NULL,
	  IPA_CONF_TYPE_MISC, sect_startup_shutdown, NULL
	},
	{ "if_any_not_reached", IPA_CONF_SECT_IF_ANY_NOT_REACHED, 0, NULL, NULL,
	  IPA_CONF_TYPE_MISC, sect_startup_shutdown, NULL
	},
	{ "if_all_not_reached", IPA_CONF_SECT_IF_ALL_NOT_REACHED, 0, NULL, NULL,
	  IPA_CONF_TYPE_MISC, sect_startup_shutdown, NULL
	},
#endif /* WITH_LIMITS */
#ifdef WITH_THRESHOLDS
	{ "threshold", IPA_CONF_SECT_THRESHOLD, 1, NULL, NULL,
	  IPA_CONF_TYPE_MISC, sect_for_any_limit, parse_threshold_sect
	},
	{ "below_threshold", IPA_CONF_SECT_BELOW_THRESHOLD, 0, NULL, NULL,
	  IPA_CONF_TYPE_MISC, sect_threshold, NULL
	},
	{ "above_threshold", IPA_CONF_SECT_ABOVE_THRESHOLD, 0, NULL, NULL,
	  IPA_CONF_TYPE_MISC, sect_threshold, NULL
	},
	{ "equal_threshold", IPA_CONF_SECT_EQUAL_THRESHOLD, 0, NULL, NULL,
	  IPA_CONF_TYPE_MISC, sect_threshold, NULL
	},
#endif
	{ "startup", IPA_CONF_SECT_STARTUP, 0, NULL, NULL,
	  IPA_CONF_TYPE_MISC, sect_for_startup_shutdown, NULL
	},
	{ "shutdown", IPA_CONF_SECT_SHUTDOWN, 0, NULL, NULL,
	  IPA_CONF_TYPE_MISC, sect_for_startup_shutdown, NULL
	},
	{ NULL, 0, 0, NULL, NULL,
	  IPA_CONF_TYPE_MISC, NULL, NULL
	}
};

#ifdef WITH_LIMITS
#define PAT_TIME_EXP "\
^((\\+[mhDWM] ?)?([[:digit:]]+[smhDW]( ?[[:digit:]]+[smhDW])*)?|\
([[:digit:]]+[smhDW]( ?[[:digit:]]+[smhDW])*)?( ?\\+[mhDWM])?)$"
#endif

#ifdef WITH_THRESHOLDS
#define PAT_THR_BALANCE	"\
^(-|[[:digit:]]+):(-|[[:digit:]]+):(-|[[:digit:]]+)$"
#endif

#define PAT_WORKTIME1 "\
[SMTWHFA] (\\*|[[:digit:]]{1,2}:[[:digit:]]{1,2}-\
[[:digit:]]{1,2}:[[:digit:]]{1,2}( [[:digit:]]{1,2}:[[:digit:]]{1,2}-\
[[:digit:]]{1,2}:[[:digit:]]{1,2})*)"
#define PAT_WORKTIME "^"PAT_WORKTIME1"( "PAT_WORKTIME1")*$"
#define PAT_LIST "^[^ \"]+( [^ \"]+)*$"

#ifdef CTL_CHECK_CREDS
# define PAT_ACL "^[[:alnum:]]+$"
# define PAT_CTL_ACL_CLASS "^[[:alnum:]]+( !?%?[-_[:alnum:]]+)*$"
#endif

/*
 * Parameters in ipa.conf.
 */
static ipa_conf_param conf_param_tbl[] = {
	{ "update_time", -1, NULL, NULL, IPA_CONF_TYPE_TIME,
	  sect_any_rule, parse_update_time
	},
	{ "append_time", -1, NULL, NULL, IPA_CONF_TYPE_TIME,
	  sect_any_rule, parse_append_time
	},
	{ "wakeup_time", -1, NULL, NULL, IPA_CONF_TYPE_TIME,
	  sect_root, parse_wakeup_time
	},
	{ "freeze_time", -1, NULL, NULL, IPA_CONF_TYPE_TIME,
	  sect_root, parse_freeze_time
	},
	{ "sleep_after_dump", -1, NULL, NULL, IPA_CONF_TYPE_TIME,
	  sect_root, parse_sleep_after_dump
	},
	{ "sensitive_time", -1, NULL, NULL, IPA_CONF_TYPE_TIME,
	  sect_root, parse_sensitive_time
	},
	{ "exec", -1, NULL, NULL, IPA_CONF_TYPE_MISC,
	  sect_for_exec, parse_exec
	},
	{ "sync_exec", -1, NULL, NULL, IPA_CONF_TYPE_BOOLEAN,
	  sect_for_exec, parse_sync_exec
	},
	{ "ac_mod", 1, NULL, NULL, IPA_CONF_TYPE_STRING,
	  sect_root, parse_ac_mod
	},
	{ "db_mod", 1, NULL, NULL, IPA_CONF_TYPE_STRING,
	  sect_root, parse_db_mod
	},
	{ "ac_list", -1, PAT_LIST, &re_list, IPA_CONF_TYPE_MISC,
	  sect_any_rule, parse_ac_list
	},
	{ "db_list", -1, NULL, &re_list, IPA_CONF_TYPE_MISC,
	  sect_for_db_list, parse_db_list
	},
	{ "include", 1, NULL, NULL,IPA_CONF_TYPE_STRING,
	  NULL, parse_include
	},
	{ "include_files", 1, NULL, NULL, IPA_CONF_TYPE_STRING,
	  NULL, parse_include_files
	},
	{ "value_units", 1, NULL, NULL, IPA_CONF_TYPE_MISC,
	  sect_root, parse_value_units
	},
#if defined(WITH_RULES) || defined(WITH_ANY_LIMITS)
	{ "info", 1, NULL, NULL, IPA_CONF_TYPE_STRING,
	  sect_for_info, parse_info
	},
#endif
#ifdef WITH_RULES
	{ "ac_gather_add", 1, NULL, NULL, IPA_CONF_TYPE_STRING,
	  sect_rule, parse_ac_gather_add
	},
	{ "ac_gather_sub", 1, NULL, NULL, IPA_CONF_TYPE_STRING,
	  sect_rule, parse_ac_gather_sub
	},
	{ "ictl", 1, NULL, NULL, IPA_CONF_TYPE_STRING,
	  sect_for_exec, parse_ictl
	},
#endif
#ifdef WITH_LIMITS
	{ "limit", -1, NULL, NULL, IPA_CONF_TYPE_VALUE,
	  sect_limit, parse_limit_param
	},
	{ "restart", -1, PAT_TIME_EXP, &re_texp, IPA_CONF_TYPE_MISC,
	  sect_restart, parse_restart
	},
	{ "expire", -1, NULL, &re_texp, IPA_CONF_TYPE_MISC,
	  sect_expire, parse_expire
	},
	{ "load_limit", 1, NULL, NULL, IPA_CONF_TYPE_BOOLEAN,
	  sect_global_limit, parse_load_limit
	},
	{ "debug_limit", 1, NULL, NULL, IPA_CONF_TYPE_UINT32,
	  sect_any_rule, parse_debug_limit
	},
	{ "debug_limit_init", 1, NULL, NULL, IPA_CONF_TYPE_UINT32,
	  sect_any_rule, parse_debug_limit_init
	},
#endif
#ifdef WITH_THRESHOLDS
	{ "threshold", -1, NULL, NULL, IPA_CONF_TYPE_VALUE,
	  sect_threshold, parse_threshold_param
	},
	{ "threshold_deviation", -1, NULL, NULL, IPA_CONF_TYPE_MISC,
	  sect_threshold, parse_threshold_deviation
	},
	{ "threshold_time_width", -1, NULL, NULL, IPA_CONF_TYPE_TIME,
	  sect_global_threshold, parse_threshold_time_width
	},
	{ "threshold_time_slice", -1, NULL, NULL, IPA_CONF_TYPE_TIME,
	  sect_global_threshold, parse_threshold_time_slice
	},
	{ "threshold_type", 1, NULL, NULL, IPA_CONF_TYPE_UINT32,
	  sect_global_threshold, parse_threshold_type
	},
	{ "threshold_balance", 1, PAT_THR_BALANCE, &re_thr_balance,
	  IPA_CONF_TYPE_MISC, sect_global_threshold, parse_threshold_balance
	},
	{ "load_threshold", 1, NULL, NULL, IPA_CONF_TYPE_BOOLEAN,
	  sect_global_threshold, parse_load_threshold
	},
	{ "debug_threshold", 1, NULL, NULL, IPA_CONF_TYPE_UINT32,
	  sect_any_rule, parse_debug_threshold
	},
	{ "debug_threshold_init", 1, NULL, NULL, IPA_CONF_TYPE_UINT32,
	  sect_any_rule, parse_debug_threshold_init
	},
#endif
	{ "worktime", -1, PAT_WORKTIME, &re_worktime, IPA_CONF_TYPE_MISC,
	  sect_for_worktime, parse_worktime
	},
	{ "posix_re_pattern", 1, NULL, NULL, IPA_CONF_TYPE_BOOLEAN,
	  sect_root, parse_posix_re_pattern
	},
	{ "only_abs_paths", 1, NULL, NULL, IPA_CONF_TYPE_BOOLEAN,
	  sect_root, parse_only_abs_paths
	},
	{ "keep_rules_order", 1, NULL, NULL, IPA_CONF_TYPE_BOOLEAN,
	  sect_root, parse_keep_rules_order
	},
	{ "check_next_rulepat", 1, NULL, NULL, IPA_CONF_TYPE_BOOLEAN,
	  sect_rulepat, parse_check_next_rulepat
	},
	{ "shell_path", 1, NULL, NULL, IPA_CONF_TYPE_STRING,
	  sect_root, parse_shell_path
	},
	{ "shell_arg1", 1, NULL, NULL, IPA_CONF_TYPE_STRING,
	  sect_root, parse_shell_arg1
	},
	{ "debug_time", 1, NULL, NULL, IPA_CONF_TYPE_UINT32,
	  sect_root, parse_debug_time
	},
	{ "debug_ac_null", 1, NULL, NULL, IPA_CONF_TYPE_UINT32,
	  sect_root, parse_debug_ac_null
	},
	{ "debug_db_null", 1, NULL, NULL, IPA_CONF_TYPE_UINT32,
	  sect_root, parse_debug_db_null
	},
	{ "debug_exec", 1, NULL, NULL, IPA_CONF_TYPE_UINT32,
	  sect_any_rule, parse_debug_exec
	},
	{ "debug_worktime", 1, NULL, NULL, IPA_CONF_TYPE_UINT32,
	  sect_root, parse_debug_worktime
	},
	{ "ctl_enable", 1, NULL, NULL, IPA_CONF_TYPE_BOOLEAN,
	  sect_root, parse_ctl_enable
	},
	{ "ctl_socket_path", 1, NULL, NULL, IPA_CONF_TYPE_STRING,
	  sect_root, parse_ctl_socket_path
	},
	{ "ctl_socket_perm", 1, NULL, NULL, IPA_CONF_TYPE_MISC,
	  sect_root, parse_ctl_socket_perm
	},
	{ "ctl_timeout", -1, NULL, NULL, IPA_CONF_TYPE_TIME,
	  sect_root, parse_ctl_timeout
	},
	{ "ctl_query_max_size", -1, NULL, NULL, IPA_CONF_TYPE_BYTES,
	  sect_root, parse_ctl_query_max_size
	},
	{ "debug_ctl", 1, NULL, NULL, IPA_CONF_TYPE_UINT32,
	  sect_root, parse_debug_ctl
	},
#ifdef CTL_CHECK_CREDS
	{ "ctl_rule_acl", 1, PAT_ACL, &re_acl, IPA_CONF_TYPE_MISC,
	  sect_any_rule, parse_ctl_rule_acl
	},
	{ "ctl_dump_acl", 1, NULL, &re_acl, IPA_CONF_TYPE_MISC,
	  sect_root, parse_ctl_dump_acl
	},
	{ "ctl_freeze_acl", 1, NULL, &re_acl, IPA_CONF_TYPE_MISC,
	  sect_root, parse_ctl_freeze_acl
	},
	{ "ctl_stat_acl", 1, NULL, &re_acl, IPA_CONF_TYPE_MISC,
	  sect_root, parse_ctl_stat_acl
	},
	{ "ctl_acl_class", -1, PAT_CTL_ACL_CLASS, &re_ctl_acl_class,
	  IPA_CONF_TYPE_MISC, sect_root, parse_ctl_acl_class
	},
#endif
#ifdef WITH_AUTORULES
	{ "worktime_rule", -1, NULL, &re_worktime, IPA_CONF_TYPE_MISC,
	  sect_autorule, parse_worktime_rule
	},
	{ "debug_autorule", 1, NULL, NULL, IPA_CONF_TYPE_UINT32,
	  sect_root, parse_debug_autorule
	},
#endif
	{ NULL, 0, NULL, NULL, IPA_CONF_TYPE_MISC,
	  NULL, NULL
	}
};

/* Module which own current custom section. */
static const struct ac_mod *cur_ac_mod;
static const struct db_mod *cur_db_mod;

/* Hash tables for ipa.conf configuration. */
static struct conf_sect_hash *conf_sect_hash;
static struct conf_param_hash *conf_param_hash;

/* Pointers to configuration tables. */
static const ipa_conf_sect *conf_sect_tbl_ptr;
static const ipa_conf_param *conf_param_tbl_ptr;
static const struct conf_sect_hash *conf_sect_hash_ptr;
static const struct conf_param_hash *conf_param_hash_ptr;

/* Set if in own ipa.conf section. */
static char	in_own_sect;

#ifdef WITH_LIMITS
/* Previous commands section. */
static struct cmds *prevcmds;
#endif

/* Stack of nested configuration sections. */
struct sect_stack {
	SLIST_ENTRY(sect_stack) link;
	unsigned int	section;
};

SLIST_HEAD(EMPTY, sect_stack) sect_stack_list =
	SLIST_HEAD_INITIALIZER(sect_stack_list);

static int
init_config_data(PARSING_MODE mode)
{
	unsigned int i;

	if ((m_anon = mem_type_new_local(MTYPE_NAME(anonymous),
	    "Anonymous memory", 0)) == NULL ||
	    (m_tmp = mem_type_new_local(MTYPE_NAME(tmp),
	    "Temporal memory", 0)) == NULL ||
	    (m_ctl = mem_type_new_local(MTYPE_NAME(ctl),
	    "Memory for ctl", 0)) == NULL ||
	    (m_cmd = mem_type_new_local(MTYPE_NAME(cmd),
	    "Memory for commands", 0)) == NULL ||
	    (m_parser = mem_type_new_local(MTYPE_NAME(parser),
	    "Memory allocated by parser", MEMTYPE_FLAGS)) == NULL)
		return (-1);

	nac_mods = ndb_mods = 0;

	value_units = -1;
	got_arg_value = 0;

	global_ac_list = NULL;
	global_db_list = NULL;

	SLIST_INIT(&ac_mod_list);
	SLIST_INIT(&db_mod_list);

	SLIST_INIT(&ac_sets);
	SLIST_INIT(&db_sets);

	conf_sect_he_mzone = mzone_init(MZONE_NAME(conf_sect_he),
	    "Config sections hash entries", 0, sizeof(struct conf_sect_he),
	    CONF_SECT_HE_NSIZE, CONF_SECT_HE_NALLOC);
	if (conf_sect_he_mzone == NULL)
		return (-1);

	conf_param_he_mzone = mzone_init(MZONE_NAME(conf_param_he),
	    "Config parameters hash entries", 0, sizeof(struct conf_param_he),
	    CONF_PARAM_HE_NSIZE, CONF_PARAM_HE_NALLOC);
	if (conf_param_he_mzone == NULL)
		return (-1);

	if (mode != RECONFIG_PARSING) {
		rules_hash_init();
#ifdef WITH_ANY_LIMITS
		wpid_hash_init();
#endif
	}

	rule_mzone = mzone_init(MZONE_NAME(rule), "Rules", MEMFLAG_OPTIMIZE,
	    sizeof(struct rule), RULE_NSIZE, RULE_NALLOC);
	if (rule_mzone == NULL)
		return (-1);

	rulepat_mzone = mzone_init(MZONE_NAME(rulepat), "Rules patterns",
	    0, sizeof(struct rulepat), RULEPAT_NSIZE, RULEPAT_NALLOC);
	if (rulepat_mzone == NULL)
		return (-1);
	STAILQ_INIT(&rulepats_list);
	rulepatno = 0;

#ifdef WITH_RULES
	SLIST_INIT(&acg_list);
#endif

	tint_mzone = mzone_init(MZONE_NAME(tint), "Time intervals", 0,
	    sizeof(struct tint), TINT_NSIZE, TINT_NALLOC);
	if (tint_mzone == NULL)
		return (-1);
	STAILQ_INIT(&tint_sets);

	worktime_mzone = mzone_init(MZONE_NAME(worktime), "Worktimes", 0,
	    sizeof(struct worktime), WORKTIME_NSIZE, WORKTIME_NALLOC);
	if (worktime_mzone == NULL)
		return (-1);
	init_worktime_default();
	SLIST_INIT(&worktimes_list);
	SLIST_INSERT_HEAD(&worktimes_list, &worktime_default, link);

	tevent_mzone = mzone_init(MZONE_NAME(tevent), "Time events", 0,
	    sizeof(struct tevent), TEVENT_NSIZE, TEVENT_NALLOC);
	if (tevent_mzone == NULL)
		return (-1);
	SLIST_INIT(&tevents_list);

	cmd_mzone = mzone_init(MZONE_NAME(cmd), "Commands", 0,
	    sizeof(struct cmd), CMD_NSIZE, CMD_NALLOC);
	if (cmd_mzone == NULL)
		return (-1);

#ifdef WITH_RULES
	acg_mzone = NULL;
	ictl_mzone = mzone_init(MZONE_NAME(ictl), "Ictl", 0,
	    sizeof(struct ictl), CMD_NSIZE, CMD_NALLOC);
	if (ictl_mzone == NULL)
		return (-1);
#endif

	cmds_init(&cmds_global[RC_STARTUP]);
	cmds_init(&cmds_global[RC_SHUTDOWN]);

#ifdef WITH_LIMITS
	limit_mzone = mzone_init(MZONE_NAME(limit), "Limits", MEMFLAG_OPTIMIZE,
	    sizeof(struct limit), LIMIT_NSIZE, LIMIT_NALLOC);
	if (limit_mzone == NULL)
		return (-1);
	global_load_limit = -1;
#endif

#ifdef WITH_SUBLIMITS
	sublimit_mzone = mzone_init(MZONE_NAME(sublimit), "Sublimits", 0,
	    sizeof(struct sublimit), SUBLIMIT_NSIZE, SUBLIMIT_NALLOC);
	if (sublimit_mzone == NULL)
		return (-1);
#endif

	nautorules = nstatrules = ndynrules = nstatlimits = ndynlimits =
	    nstatsublimits = ndynsublimits = 0;

#ifdef WITH_THRESHOLDS
	threshold_mzone = mzone_init(MZONE_NAME(threshold), "Thresholds",
	    MEMFLAG_OPTIMIZE, sizeof(struct threshold), THRESHOLD_NSIZE,
	    THRESHOLD_NALLOC);
	if (threshold_mzone == NULL)
		return (-1);
	global_threshold_type = -1;
	global_load_threshold = -1;
	global_threshold_time_width = 0;
	global_threshold_time_slice = NULL;
	global_threshold_below_lim = global_threshold_equal_lim =
	    global_threshold_above_lim = 0;
#endif
	nstatthresholds = ndynthresholds = 0;

#ifdef WITH_AUTORULES
	autorules_marray = marray_init(MARRAY_NAME(autorules), "Autorules", 0,
	    (void *)&autorules, sizeof(struct autorule), AUTORULE_NSIZE,
	    AUTORULE_NALLOC);
	if (autorules_marray == NULL)
		return (-1);
	rules_ptr_marray = NULL;
#endif

	global_update_tevent = global_append_tevent = NULL;
	global_worktime = NULL;
	global_section_set = 0;
	only_abs_paths = keep_rules_order = posix_re_pattern = -1;

	for (i = 0; i < GLOBAL_DEBUG_PARAM_TBL_SIZE; ++i)
		*global_debug_param[i].value = -1;
	for (i = 0; i < ROOT_DEBUG_PARAM_TBL_SIZE; ++i)
		*root_debug_param[i].value = -1;
	for (i = 0; i < ROOT_TIME_PARAM_TBL_SIZE; ++i)
		*root_time_param[i].value = 0;

	ctl_enable = -1;
	ctl_socket_path = NULL;
	ctl_socket_perm = 0;
	ctl_timeout = 0;
#ifdef CTL_CHECK_CREDS
	ctl_acl_elem_mzone = mzone_init(MZONE_NAME(ctl_acl_elem),
	    "ACL elements", 0, sizeof(struct ctl_acl_elem), CTL_ACL_ELEM_NSIZE,
	    CTL_ACL_ELEM_NALLOC);
	if (ctl_acl_elem_mzone == NULL)
		return (-1);
	ctl_acl_class_mzone = mzone_init(MZONE_NAME(ctl_acl_class),
	    "ACL classes", 0, sizeof(struct ctl_acl_class), CTL_ACL_CLASS_NSIZE,
	    CTL_ACL_CLASS_NALLOC);
	if (ctl_acl_class_mzone == NULL)
		return (-1);
	global_ctl_rule_acl = ctl_dump_acl = ctl_freeze_acl =
	    ctl_stat_acl = NULL;
	STAILQ_INIT(&ctl_acl_classes);
#endif

	if (build_conf_re() < 0)
		return (-1);
	if (init_conf_tbls((char *)NULL, mode != RECONFIG_PARSING,
	    conf_sect_tbl, &conf_sect_hash, conf_param_tbl,
	    &conf_param_hash) < 0)
		return (-1);

	return (0);
}

/*
 * Deinitialize not needed configuration data structures.
 */
static void
deinit_config_data(void)
{
	deinit_conf_tbls(0, conf_sect_tbl, conf_sect_hash,
	    conf_param_tbl, conf_param_hash);

	mzone_deinit(conf_sect_he_mzone);
	mzone_deinit(conf_param_he_mzone);

	if (mzone_is_empty(worktime_mzone)) {
		mzone_deinit(worktime_mzone);
		mzone_deinit(tint_mzone);
		worktime_mzone = tint_mzone = NULL;
	}

	if (mzone_is_empty(cmd_mzone)) {
		mzone_deinit(cmd_mzone);
		cmd_mzone = NULL;
	}
#ifdef WITH_RULES
	if (mzone_is_empty(ictl_mzone)) {
		mzone_deinit(ictl_mzone);
		ictl_mzone = NULL;
	}
#endif
	if (mzone_is_empty(rulepat_mzone)) {
		mzone_deinit(rulepat_mzone);
		rulepat_mzone = NULL;
	}
#ifdef WITH_LIMITS
	if (mzone_is_empty(limit_mzone)) {
		mzone_deinit(limit_mzone);
		limit_mzone = NULL;
	}
#endif
#ifdef WITH_SUBLIMITS
	if (mzone_is_empty(sublimit_mzone)) {
		mzone_deinit(sublimit_mzone);
		sublimit_mzone = NULL;
	}
#endif
#ifdef WITH_THRESHOLDS
	if (mzone_is_empty(threshold_mzone)) {
		mzone_deinit(threshold_mzone);
		threshold_mzone = NULL;
	}
#endif

#ifdef CTL_CHECK_CREDS
	if (mzone_is_empty(ctl_acl_elem_mzone)) {
		mzone_deinit(ctl_acl_elem_mzone);
		ctl_acl_elem_mzone = NULL;
	}
	if (mzone_is_empty(ctl_acl_class_mzone)) {
		mzone_deinit(ctl_acl_class_mzone);
		ctl_acl_class_mzone = NULL;
	}
#endif
}

static int
token_section_begin(void)
{
	const unsigned int *id;
	const char *prefix;
	const ipa_conf_sect *conf_sect;
	const struct get_arg *get_arg;
	struct sect_stack *sect_stack;
	struct ac_mod *ac_mod;
	struct db_mod *db_mod;
	char *ptr;
	int nargs;

	/* Put previous section on top of sections stack. */
	sect_stack = mem_malloc(sizeof(*sect_stack), m_anon);
	if (sect_stack == NULL) {
		logconfe("token_section_begin: mem_malloc failed");
		return (-1);
	}
	sect_stack->section = section;
	SLIST_INSERT_HEAD(&sect_stack_list, sect_stack, link);

	/* Get name of current section: "section" or "prefix:section". */
	cursect = parser_token;
	if (in_own_sect && (ptr = strchr(cursect, ':')) != NULL) {
		/*
		 * Previous section is not a custom section and
		 * new custom section appears.
		 */
		*ptr = '\0';
		prefix = cursect;
		cursect = ptr + 1;
		if ((db_mod = db_mod_by_prefix(prefix)) != NULL) {
			conf_sect_hash_ptr = db_mod->conf_sect_hash;
			conf_param_hash_ptr = db_mod->conf_param_hash;
			conf_sect_tbl_ptr = db_mod->ipa_db_mod->conf_sect_tbl;
			conf_param_tbl_ptr = db_mod->ipa_db_mod->conf_param_tbl;
			curmodfile = db_mod->mod_file;
			cur_db_mod = db_mod;
		} else if ((ac_mod = ac_mod_by_prefix(prefix)) != NULL) {
			conf_sect_hash_ptr = ac_mod->conf_sect_hash;
			conf_param_hash_ptr = ac_mod->conf_param_hash;
			conf_sect_tbl_ptr = ac_mod->ipa_ac_mod->conf_sect_tbl;
			conf_param_tbl_ptr = ac_mod->ipa_ac_mod->conf_param_tbl;
			curmodfile = ac_mod->mod_file;
			cur_ac_mod = ac_mod;
			cur_db_mod = NULL;
		} else {
			logconfx("cannot find module with \"%s\" "
			    "configuration prefix", prefix);
			return (-1);
		}
		in_own_sect = 0;
	}

	/* Find section. */
	conf_sect = find_conf_sect(conf_sect_tbl_ptr, conf_sect_hash_ptr,
	    cursect);
	if (conf_sect == NULL) {
		logconfx("unknown section");
		return (-1);
	}

	/* Validate type of argument. */
	if (conf_sect->arg_type > IPA_CONF_TYPE_MISC) {
		logconfx("internal error: unknown type %u of function's "
		    "argument", conf_sect->arg_type);
		return (-1);
	}

	/* Check whether current section is used in correct place. */
	id = conf_sect->sect_where;
	if (id != NULL)
		for (;; ++id) {
			if (*id == section)
				break;
			if (*id == 0)
				goto not_expected;
		}

	/* Check number of section's arguments. */
	nargs = conf_sect->arg_nargs;
	if (nargs >= 0) {
		if (parser_nargs != nargs) {
			logconfx("wrong number of arguments (has %d, "
			    "should have %d)", parser_nargs, nargs);
			return (-1);
		}
	} else {
		nargs = -nargs;
		if (parser_nargs < nargs) {
			logconfx("this section should have at "
			    "least %d argument%s", nargs,
			    plural_form((unsigned int)nargs));
			return (-1);
		}
	}

	/* Validate arguments if needed. */
	if (conf_sect->arg_regexp != NULL)
		if (regexec_simple(conf_sect->arg_regexp, parser_args) != 0) {
			logconfx("wrong format of an argument");
			return (-1);
		}

	section = conf_sect->sect_id;
	if (!in_own_sect) {
		/* Register configuration event in module for its section. */
		if (cur_db_mod != NULL) {
			if (db_mod_conf_event(cur_db_mod,
			    IPA_CONF_EVENT_CUSTOM_SECT_BEGIN, section,
			    (void *)NULL) < 0)
				return (-1);
		} else {
			if (ac_mod_conf_event(cur_ac_mod,
			    IPA_CONF_EVENT_CUSTOM_SECT_BEGIN, section,
			    (void *)NULL) < 0)
				return (-1);
		}
	}

	/* Parse it. */
	conf_event_no = 0;
	conf_event_arg = NULL;
	get_arg = &get_arg_tbl[conf_sect->arg_type];
	if (get_arg->reg != NULL)
		if (regexec_simple(get_arg->reg, parser_args) != 0) {
			logconfx("wrong format of an argument");
			return (-1);
		}
	if (get_arg->parse(get_arg->argp) < 0)
		return (-1);
	if (conf_sect->arg_parse != NULL)
		if (conf_sect->arg_parse(get_arg->argp) < 0)
			return (-1);

	if (in_own_sect && section != SECT_SUBLIMIT) {
		/*
		 * Register configuration event in all modules for
		 * ipa.conf's section.
		 */
		if (mod_conf_event(conf_event_begin[section], conf_event_no,
		    conf_event_arg) < 0)
			return (-1);
	}

	if (in_own_sect && section >= IPA_CONF_SECT_RESTART &&
	    section <= IPA_CONF_SECT_ABOVE_THRESHOLD) {
		/* Do additional settings for ipa.conf's sections. */
		switch (section) {
		case IPA_CONF_SECT_STARTUP:
			section_rc = RC_STARTUP;
			break;
		case IPA_CONF_SECT_SHUTDOWN:
			section_rc = RC_SHUTDOWN;
			break;
#ifdef WITH_LIMITS
		case IPA_CONF_SECT_IF_ALL_REACHED:
		case IPA_CONF_SECT_IF_ALL_NOT_REACHED:
		case IPA_CONF_SECT_IF_ANY_REACHED:
		case IPA_CONF_SECT_IF_ANY_NOT_REACHED:
			switch (section_top) {
# ifdef WITH_RULES
			case IPA_CONF_SECT_RULE:
# endif
# ifdef WITH_AUTORULES
			case IPA_CONF_SECT_AUTORULE:
# endif
			case IPA_CONF_SECT_RULEPAT:
				break;
			default:
				goto not_expected;
			}
			prevcmds = curcmds;
			break;
		case IPA_CONF_SECT_IF_REACHED:
		case IPA_CONF_SECT_IF_NOT_REACHED:
			switch (section_top) {
			case IPA_CONF_SECT_LIMIT:
# ifdef WITH_SUBLIMITS
			case SECT_SUBLIMIT:
# endif
				break;
			default:
				goto not_expected;
			}
			prevcmds = curcmds;
			break;
#endif /* WITH_LIMITS */
		}
		if (set_curcmdl() < 0)
			return (-1);
	}

	return (0);

not_expected:
	logconfx("this section is not expected here");
	return (-1);
}

static int
token_parameter(void)
{
	const unsigned int *id;
	const char *prefix;
	const ipa_conf_param *conf_param;
	const struct get_arg *get_arg;
	const struct ac_mod *ac_mod;
	const struct db_mod *db_mod;
	char *ptr;
	int nargs;

	/* Get name of current parameter: "parameter" or "prefix:parameter". */
	curparam = parser_token;
	if (in_own_sect && (ptr = strchr(curparam, ':')) != NULL) {
		/*
		 * Current section is not a custom section
		 * and custom parameter appears.
		 */
		*ptr = '\0';
		prefix = curparam;
		curparam = ptr + 1;
		if ((db_mod = db_mod_by_prefix(prefix)) != NULL) {
			conf_param_hash_ptr = db_mod->conf_param_hash;
			conf_param_tbl_ptr = db_mod->ipa_db_mod->conf_param_tbl;
			curmodfile = db_mod->mod_file;
		} else if ((ac_mod = ac_mod_by_prefix(prefix)) != NULL) {
			conf_param_hash_ptr = ac_mod->conf_param_hash;
			conf_param_tbl_ptr = ac_mod->ipa_ac_mod->conf_param_tbl;
			curmodfile = ac_mod->mod_file;
			db_mod = NULL;
		} else {
			logconfx("cannot find module with \"%s\" "
			    "configuration prefix", prefix);
			return (-1);
		}
	}

	/* Find parameter. */
	conf_param = find_conf_param(conf_param_tbl_ptr, conf_param_hash_ptr,
	    curparam);
	if (conf_param == NULL) {
		logconfx("unknown parameter");
		return (-1);
	}

	/* Validate type of argument. */
	if (conf_param->arg_type > IPA_CONF_TYPE_MISC) {
		logconfx("internal error: unknown type %u of an argument",
		    conf_param->arg_type);
		return (-1);
	}

	/* Check whether parameter is used in correct place. */
	id = conf_param->param_where;
	if (id != NULL)
		for (;; ++id) {
			if (*id == section)
				break;
			if (*id == 0) {
				logconfx("this parameter is not expected here");
				return (-1);
			}
		}

	/* Check number of parameter's arguments. */
	nargs = conf_param->arg_nargs;
	if (nargs >= 0) {
		if (parser_nargs != nargs) {
			logconfx("wrong number of arguments (has %d, "
			    "should have %d)", parser_nargs, nargs);
			return (-1);
		}
	} else {
		nargs = -nargs;
		if (parser_nargs < nargs) {
			logconfx("this parameter should have at "
			    "least %d argument%s", nargs,
			    plural_form((unsigned int)nargs));
			return (-1);
		}
	}

	/* Validate arguments if needed. */
	if (conf_param->arg_regexp != NULL)
		if (regexec_simple(conf_param->arg_regexp, parser_args) != 0) {
			logconfx("wrong format of an argument");
			return (-1);
		}

	/* Parse it. */
	get_arg = &get_arg_tbl[conf_param->arg_type];
	if (get_arg->reg != NULL)
		if (regexec_simple(get_arg->reg, parser_args) != 0) {
			logconfx("wrong format of an argument");
			return (-1);
		}
	if (get_arg->parse(get_arg->argp) < 0)
		return (-1);
	if (conf_param->arg_parse != NULL)
		if (conf_param->arg_parse(get_arg->argp) < 0)
			return (-1);

	curparam = NULL;
	if (curmodfile != NULL && in_own_sect) {
		/* Restore ipa.conf parameters tables. */
		curmodfile = NULL;
		conf_param_hash_ptr = conf_param_hash;
		conf_param_tbl_ptr = conf_param_tbl;
	}

	return (0);
}

static int
token_section_end(void)
{
	struct sect_stack *sect_stack;

	cursect = NULL;
	if (!in_own_sect) {
		/* Register configuration event in module for its section. */
		if (cur_db_mod != NULL) {
			if (db_mod_conf_event(cur_db_mod,
			    IPA_CONF_EVENT_CUSTOM_SECT_END, section,
			    (void *)NULL) < 0)
				return (-1);
		} else {
			if (ac_mod_conf_event(cur_ac_mod,
			    IPA_CONF_EVENT_CUSTOM_SECT_END, section,
			    (void *)NULL) < 0)
				return (-1);
		}
	} else {
		/*
		 * Register configuration event in all modules for
		 * ipa.conf's section.
		 */
		if (mod_conf_event(conf_event_begin[section] + 1,
		    conf_event_no, conf_event_arg) < 0)
			return (-1);
		switch (section) {
#ifdef WITH_RULES
		case IPA_CONF_SECT_RULE:
#endif
#ifdef WITH_AUTORULES
		case IPA_CONF_SECT_AUTORULE:
#endif
		case IPA_CONF_SECT_RULEPAT:
			section_top = IPA_CONF_SECT_ROOT;
			break;
		case IPA_CONF_SECT_GLOBAL:
			section_top = IPA_CONF_SECT_ROOT;
#ifdef WITH_THRESHOLDS
			if (check_threshold_times(global_threshold_time_width,
			    global_threshold_time_slice, 0) < 0)
				return (-1);
#endif
			break;
#ifdef WITH_LIMITS
		case IPA_CONF_SECT_LIMIT:
			section_top = section_first;
			if (LIMIT_IS_NOTSET(curlimit)) {
				logconfx("parameter \"limit\" is absent");
				return (-1);
			}
			if (curlimit->lim == 0 &&
			    curlimit->expire.expire.upto == TEXP_UPTO_SIMPLE &&
			    curlimit->expire.expire.seconds == 0) {
				logconfx("\"limit\" parameter cannot be equal "
				    "to zero, since \"expire\" parameter is "
				    "equal to 0s");
				return (-1);
			}
			(void)parser_local_sym_del("limit");
			break;
# ifdef WITH_SUBLIMITS
		case SECT_SUBLIMIT:
			section_top = IPA_CONF_SECT_LIMIT;
			(void)parser_local_sym_del("sublimit");
			break;
# endif
		case IPA_CONF_SECT_IF_REACHED:
		case IPA_CONF_SECT_IF_NOT_REACHED:
		case IPA_CONF_SECT_IF_ALL_REACHED:
		case IPA_CONF_SECT_IF_ALL_NOT_REACHED:
		case IPA_CONF_SECT_IF_ANY_REACHED:
		case IPA_CONF_SECT_IF_ANY_NOT_REACHED:
			curcmds = prevcmds;
			break;
#endif /* WITH_LIMITS */
#ifdef WITH_THRESHOLDS
		case IPA_CONF_SECT_THRESHOLD:
			section_top = section_first;
			if (THRESHOLD_IS_NOTSET(curthreshold)) {
				logconfx("parameter \"threshold\" is absent");
				return (-1);
			}
			if (check_threshold_times(curthreshold->time_width,
			    curthreshold->time_slice, 1) < 0)
				return (-1);
			(void)parser_local_sym_del("threshold");
			break;
#endif /* WITH_THRESHOLDS */
		case IPA_CONF_SECT_STARTUP:
		case IPA_CONF_SECT_SHUTDOWN:
			section_rc = RC_NOTSET;
			break;
		}
	}

	sect_stack = SLIST_FIRST(&sect_stack_list);
	SLIST_REMOVE_HEAD(&sect_stack_list, link);
	section = sect_stack->section;
	mem_free(sect_stack, m_anon);
	if (!in_own_sect && section < IPA_CONF_SECT_CUSTOM_OFFSET) {
		/*
		 * We were in custom section and previous section
		 * is not a custom section.  Restore ipa.conf tables.
		 */
		curmodfile = NULL;
		conf_sect_tbl_ptr = conf_sect_tbl;
		conf_param_tbl_ptr = conf_param_tbl;
		conf_sect_hash_ptr = conf_sect_hash;
		conf_param_hash_ptr = conf_param_hash;
		in_own_sect = 1;
	}

	return (0);
}

static int
deinit_config_mods(void)
{
	const struct ipa_ac_mod *ipa_ac_mod;
	const struct ipa_db_mod *ipa_db_mod;
	const struct ac_mod *ac_mod;
	const struct db_mod *db_mod;

	SLIST_FOREACH(ac_mod, &ac_mod_list, link) {
		ipa_ac_mod = ac_mod->ipa_ac_mod;
		if (mimic_real_config)
			if (ipa_ac_mod->conf_mimic_real() < 0) {
				logconfe("module %s: conf_mimic_real failed",
				    ac_mod->mod_file);
				return (-1);
			}
		deinit_conf_tbls(1,
		    ipa_ac_mod->conf_sect_tbl, ac_mod->conf_sect_hash,
		    ipa_ac_mod->conf_param_tbl, ac_mod->conf_param_hash);
		if (ipa_ac_mod->conf_deinit() < 0) {
			logconfe("module %s: conf_deinit failed",
			    ac_mod->mod_file);
			return (-1);
		}
	}

	SLIST_FOREACH(db_mod, &db_mod_list, link) {
		ipa_db_mod = db_mod->ipa_db_mod;
		if (mimic_real_config)
			if (ipa_db_mod->conf_mimic_real() < 0) {
				logconfe("module %s: conf_mimic_real failed",
				    db_mod->mod_file);
				return (-1);
			}
		deinit_conf_tbls(1,
		    ipa_db_mod->conf_sect_tbl, db_mod->conf_sect_hash,
		    ipa_db_mod->conf_param_tbl, db_mod->conf_param_hash);
		if (ipa_db_mod->conf_deinit() < 0) {
			logconfe("module %s: conf_deinit failed",
			    db_mod->mod_file);
			return (-1);
		}
	}
	return (0);
}

/*
 * Main function for parsing configuration file(s).
 */
int
configure(PARSING_MODE mode)
{
	struct parser_pb *pb;

	switch (mode) {
	case TEST_PARSING:
	case CMD_PARSING:
		use_log = 0;
		break;
	default:
		use_log = 1;
	}

	/* Set wrappers for log functions during configuration. */
	xvlogmsgx = vlogconfx_priority;
	mvlogmsgx = vlogconfe;
	parser_vlogmsgx = parser_vlogmsgx_wrapper;

	if (memfunc_init() < 0)
		goto failed;

	if (init_config_data(mode) < 0)
		goto failed;

	section = section_first = section_top = IPA_CONF_SECT_ROOT;
	section_rc = RC_NOTSET;
	parsing_mode = mode;
	memfunc.m_parser = m_parser;

	if (mode != TEST_PARSING && mode != CMD_PARSING)
		logmsgx(IPA_LOG_INFO, "use configuration file %s, parsing...",
		    ipa_conf_file);

	if (check_conf_file(ipa_conf_file, 0) < 0)
		goto failed;

	/* Initialize parser and first pb. */
	if (parser_init() < 0)
		goto failed;
	pb = parser_new_pb(0);
	if (pb == NULL)
		goto failed;
	pb->fname = ipa_conf_file;
	pb->fp = fopen(ipa_conf_file, "r");
	if (pb->fp == NULL) {
		logconfe("fopen(%s, \"r\"): %s", ipa_conf_file,
		    strerror(errno));
		goto failed;
	}
	if (parser_push_pb(pb) < 0)
		goto failed;
	include_depth = 1;

	/* Needed for log messages. */
	curparam = cursect = curmodfile = NULL;

	/* Set ipa.conf configuration tables. */
	conf_sect_tbl_ptr = conf_sect_tbl;
	conf_param_tbl_ptr = conf_param_tbl;
	conf_sect_hash_ptr = conf_sect_hash;
	conf_param_hash_ptr = conf_param_hash;
	in_own_sect = 1;

	for (;;) {
		switch (parser_read_string()) {
		case 1:
			/* Successfully read one logical line. */
			break;
		case 0:
			/* EOF of current configuration file. */
			--include_depth;
			pb = parser_top_pb();
			if (fclose(pb->fp) != 0) {
				logconfe("fclose(%s): %s", pb->fname,
				    strerror(errno));
				goto failed;
			}
			if (pb->fname != ipa_conf_file)
				mem_free(pb->fname, m_parser);
			pb = parser_pop_pb();
			if (pb == NULL)
				goto end_of_parsing;
			/* Initialize previous file for parsing. */
			pb->fp = fopen(pb->fname, "r");
			if (pb->fp == NULL) {
				logconfe("fopen(%s, \"r\"): %s", pb->fname,
				    strerror(errno));
				goto failed;
			}
			if (fseek(pb->fp, pb->foff, SEEK_SET) < 0) {
				logconfe("fseek(%s, %ld, SEEK_SET): %s",
				    pb->fname, pb->foff, strerror(errno));
				goto failed;
			}
			continue;
		default: /* -1 */
			goto failed;
		}
		switch (parser_token_id) {
		case TOKEN_ID_SECTION_BEGIN:
			if (token_section_begin() < 0)
				goto failed;
			break;
		case TOKEN_ID_SECTION_END:
			if (token_section_end() < 0)
				goto failed;
			break;
		case TOKEN_ID_PARAMETER:
			if (token_parameter() < 0)
				goto failed;
			break;
		}
	}

end_of_parsing:
	if (deinit_config_mods() < 0)
		goto failed;

	if (parser_deinit() < 0)
		goto failed;

	deinit_config_data();

	if (mode != TEST_PARSING)
		mimic_real_config = 1;

	if (mimic_real_config) {
		if (set_global_params() < 0)
			goto failed;
		rulepats_inherit();
	}

#ifdef WITH_AUTORULES
	STAILQ_INIT(&autorules_list);
	if (nautorules != 0) {
		if (autorules_inherit() < 0)
			goto failed;
	} else
		marray_deinit(autorules_marray);
#endif

#ifdef WITH_RULES
	if (mimic_real_config) {
		if (rules_inherit() < 0)
			goto failed;
		if (setup_all_ictl() < 0)
			goto failed;
	}
#endif

	/* Set wrappers for log functions after configuration. */
	mvlogmsgx = mvlogmsgx_wrapper;
	xvlogmsgx = vlogmsgx;
	return (0);

failed:
	log_include_history();
	logconfe("configuration file parsing failed!");
	return (-1);
}

/*
 * Unload all modules and free memory used by structures which
 * describe loaded modules.  Note that any pointer should not
 * references any data in unloaded modules' memory.  So, the best
 * place when to call this function: after memfunc_deinit_1() and
 * before memfunc_deinit_2().
 */
int
unload_all_mods(void)
{
	struct ac_mod *ac_mod, *ac_mod_next;
	struct db_mod *db_mod, *db_mod_next;
	int rv;

	rv = 0;
	SLIST_FOREACH_SAFE(ac_mod, &ac_mod_list, link, ac_mod_next) {
		if (dl_close(ac_mod->mod_handle) < 0) {
			logmsgx(IPA_LOG_ERR, "module %s: dl_close failed: %s",
			    ac_mod->mod_file, dl_error());
			rv = -1;
		}
		mem_free(ac_mod->mod_file, m_parser);
		mem_free(ac_mod, m_anon);
	}

	SLIST_FOREACH_SAFE(db_mod, &db_mod_list, link, db_mod_next) {
		if (dl_close(db_mod->mod_handle) < 0) {
			logmsgx(IPA_LOG_ERR, "module %s: dl_close failed: %s",
			    db_mod->mod_file, dl_error());
			rv = -1;
		}
		mem_free(db_mod->mod_file, m_parser);
		mem_free(db_mod, m_anon);
	}

	return (rv);
}

static void
print_time_u_int(unsigned int a)
{
	const uint64_t t = a;

	print_time(&t);
}

/*
 * Table of sections names, only used entries are defined.
 */
static const char *const sect_name_tbl[] = {
	NULL,			/*  0 | undefined			*/
	NULL,			/*  1 | IPA_CONF_SECT_ROOT		*/
	NULL,			/*  2 | IPA_CONF_SECT_GLOBAL		*/
	NULL,			/*  3 | IPA_CONF_SECT_RULE		*/
	NULL,			/*  4 | IPA_CONF_SECT_LIMIT		*/
	NULL,			/*  5 | IPA_CONF_SECT_THRESHOLD		*/
	NULL,			/*  6 | IPA_CONF_SECT_AUTORULE		*/
	NULL,			/*  7 | IPA_CONF_SECT_RULEPAT		*/
	"restart",		/*  8 | IPA_CONF_SECT_RESTART		*/
	"reach",		/*  9 | IPA_CONF_SECT_REACH		*/
	"expire",		/* 10 | IPA_CONF_SECT_EXPIRE		*/
	"startup",		/* 11 | IPA_CONF_SECT_STARTUP		*/
	"shutdown",		/* 12 | IPA_CONF_SECT_SHUTDOWN		*/
	"if_reached",		/* 13 | IPA_CONF_SECT_IF_REACHED	*/
	"if_not_reached",	/* 14 | IPA_CONF_SECT_IF_NOT_REACHED	*/
	"if_all_reached",	/* 15 | IPA_CONF_SECT_IF_ALL_REACHED	*/
	"if_any_reached",	/* 16 | IPA_CONF_SECT_IF_ANY_REACHED	*/
	"if_all_not_reached",	/* 17 | IPA_CONF_SECT_IF_ALL_NOT_REACHED*/
	"if_any_not_reached",	/* 18 | IPA_CONF_SECT_IF_ANY_NOT_REACHED*/
	"below_threshold",	/* 19 | IPA_CONF_SECT_BELOW_THRESHOLD	*/
	"equal_threshold",	/* 20 | IPA_CONF_SECT_EQUAL_THRESHOLD	*/
	"above_threshold"	/* 21 | IPA_CONF_SECT_ABOVE_THRESHOLD	*/
};

static char	sect_is_visible;	/* Set if section is visible for
					   modules. */

/*
 * Call conf_show() for all modules if current section is
 * visible for modules.
 */
static void
mod_conf_show(unsigned int sect_id, unsigned int no)
{
	if (sect_is_visible) {
		const struct ac_mod *ac_mod;
		const struct db_mod *db_mod;

		need_nl = 0;
		SLIST_FOREACH(ac_mod, &ac_mod_list, link)
			ac_mod->ipa_ac_mod->conf_show(sect_id, no);
		if (sect_id == IPA_CONF_SECT_ROOT)
			print_nl_cond();
		SLIST_FOREACH(db_mod, &db_mod_list, link)
			db_mod->ipa_db_mod->conf_show(sect_id, no);
		if (sect_id == IPA_CONF_SECT_ROOT)
			print_nl_cond();
	}
}

static void
show_cmds(const struct cmds *cmds)
{
#ifdef WITH_RULES
	const struct ictl_list *ictl_list;
	const struct ictl *ictl;
#endif
	const struct cmd_list *cmd_list;
	const struct cmd *cmd;

	if (cmds->sync >= 0) {
		print_param_name("sync_exec");
		print_boolean(cmds->sync);
		print_param_end();
	}

#ifdef WITH_RULES
	ictl_list = &cmds->ictl_list;
	STAILQ_FOREACH(ictl, ictl_list, link) {
		print_param_name0("ictl");
		printf("\"%s\"", ictl->str);
		print_param_end();
	}
#endif

	cmd_list = &cmds->cmd_list;
	STAILQ_FOREACH(cmd, cmd_list, link) {
		print_param_name0("exec");
		if (cmd->user != NULL)
			printf("%s ", cmd->user);
		print_string(cmd->str);
		print_param_end();
	}
}

static void
show_commands(unsigned int sect_id, const struct cmds *cmds)
{
	if (cmds->sect_set) {
		print_sect_name(sect_name_tbl[sect_id]);
		print_sect_begin();
		show_cmds(cmds);
		mod_conf_show(sect_id, 0);
		print_sect_end();
	}
}

static void
show_rule_rc(const struct cmds_rule *rc)
{
	unsigned int x, sect_id;

	sect_id = IPA_CONF_SECT_STARTUP;
	for (x = 0; x < 2; ++rc, ++x) {
		if (rc->cmds.sect_set) {
			print_sect_name(rc_sect_name[x]);
			print_sect_begin();
			show_cmds(&rc->cmds);
			mod_conf_show(sect_id, 0);
#ifdef WITH_LIMITS
			show_commands(IPA_CONF_SECT_IF_ALL_REACHED,
			    &rc->cmds_all_reached);
			show_commands(IPA_CONF_SECT_IF_ALL_NOT_REACHED,
			    &rc->cmds_all_not_reached);
			show_commands(IPA_CONF_SECT_IF_ANY_REACHED,
			    &rc->cmds_any_reached);
			show_commands(IPA_CONF_SECT_IF_ANY_NOT_REACHED,
			    &rc->cmds_any_not_reached);
#endif
			print_sect_end();
		}
		sect_id = IPA_CONF_SECT_SHUTDOWN;
	}
}

static void
show_debug_exec(signed char val)
{
	if (val >= 0) {
		print_param_name("debug_exec");
		printf("%d", val);
		print_param_end();
	}
}

static void
show_worktime_generic(const struct worktime *wt, const char *param_name)
{
	const struct tint_list *list;
	const struct tint *tint;
	unsigned int wday;

	if (wt == NULL)
		return;

	print_param_name(param_name);
	for (wday = 0; wday < DAYS_IN_WEEK; ++wday) {
		list = wt->tint_list[wday];
		if (!STAILQ_EMPTY(list)) {
			if (wday > 0)
				printf(" ");
			printf("%c", wdays_short[wday]);
			tint = STAILQ_FIRST(list);
			if (tint->sec1 == 0 && tint->sec2 == SECONDS_IN_DAY)
				printf(" *");
			else
				for (; tint != NULL;
				    tint = STAILQ_NEXT(tint, link))
					printf(" %s", tint_str(tint));
		}
	}
	print_param_end();
}

static void
show_worktime(const struct worktime *wt)
{
	show_worktime_generic(wt, "worktime");
}

static void
show_ac_list(const struct ac_list *list)
{
	if (list != NULL) {
		print_param_name("ac_list");
		if (list != &ac_list_null) {
			const struct ac_elem *ac;

			for (ac = STAILQ_FIRST(list);;) {
				printf("%s", ac->ipa_ac_mod->ac_name);
				ac = STAILQ_NEXT(ac, link);
				if (ac == NULL)
					break;
				printf(" ");
			}
		} else
			printf("null");
		print_param_end();
	}
}

static void
show_db_list(const struct db_list *list)
{
	if (list != NULL) {
		print_param_name("db_list");
		if (list != &db_list_null) {
			const struct db_elem *db;

			for (db = STAILQ_FIRST(list);;) {
				printf("%s", db->ipa_db_mod->db_name);
				db = STAILQ_NEXT(db, link);
				if (db == NULL)
					break;
				printf(" ");
			}
		} else
			printf("null");
		print_param_end();
	}
}

#if defined(WITH_RULES) || defined(WITH_ANY_LIMITS)
static void
print_info(const char *info)
{
	if (info != NULL) {
		print_param_name("info");
		print_string(info);
		print_param_end();
	}
}
#endif /* WITH_RULES || WITH_ANY_LIMITS */

#ifdef WITH_RULES
static void
show_ac_gather_xxx(const char *pat, const char *param_name)
{
	if (pat != NULL) {
		print_param_name(param_name);
		print_string(pat);
		print_param_end();
	}
}
#endif /* WITH_RULES */

#ifdef WITH_LIMITS
/*
 * Convert expired time to human readable string.
 */
static void
print_texp(const struct texp *texp)
{
	unsigned int t, a;
	int need_space;

	a = texp->seconds;
	if (texp->upto != TEXP_UPTO_SIMPLE && texp->side == 0) {
		printf("+%c", texp->upto);
		need_space = 1;
	} else
		need_space = 0;
	t = a / SECONDS_IN_DAY;
	if (t != 0) {
		if (need_space)
			printf(" ");
		else
			need_space = 1;
		printf("%uD", t);
		a -= t * SECONDS_IN_DAY;
	}
	if (a != 0 || (a == 0 && t == 0 && texp->upto == TEXP_UPTO_SIMPLE)) {
		if (need_space)
			printf(" ");
		else
			need_space = 1;
		print_time_u_int(a);
	}
	if (texp->upto != TEXP_UPTO_SIMPLE && texp->side == 1) {
		if (need_space)
			printf(" ");
		printf("+%c", texp->upto);
	}
}

static void
show_limit_rc(const struct cmds_limit *rc)
{
	unsigned int x, sect_id;

	sect_id = IPA_CONF_SECT_STARTUP;
	for (x = 0; x < 2; ++rc, ++x) {
		if (rc->cmds.sect_set) {
			print_sect_name(rc_sect_name[x]);
			print_sect_begin();
			show_cmds(&rc->cmds);
			mod_conf_show(sect_id, 0);
			show_commands(IPA_CONF_SECT_IF_REACHED,
			    &rc->cmds_reached);
			show_commands(IPA_CONF_SECT_IF_NOT_REACHED,
			    &rc->cmds_not_reached);
			print_sect_end();
		}
		sect_id = IPA_CONF_SECT_SHUTDOWN;
	}
}

static void
show_load_limit(signed char val)
{
	if (val >= 0) {
		print_param_name("load_limit");
		print_boolean(val);
		print_param_end();
	}
}

static void
show_debug_limit(signed char val1, signed char val2)
{
	if (val1 >= 0) {
		print_param_name("debug_limit");
		printf("%d", val1);
		print_param_end();
	}
	if (val2 >= 0) {
		print_param_name("debug_limit_init");
		printf("%d", val2);
		print_param_end();
	}
}

static void
show_limits(const struct limits_list *list)
{
	const struct limit *limit;
#ifdef WITH_SUBLIMITS
	const struct sublimit *sublimit;
#endif

	STAILQ_FOREACH(limit, list, link) {
		print_sect_name("limit");
		printf("%s ", limit->name);
		print_sect_begin();
		show_db_list(limit->db_list);
		print_info(limit->info);
		print_param_name("limit");
		print_value(&limit->lim, limit->cnt_type);
		print_param_end();
		show_load_limit(limit->load_limit);
		show_worktime(limit->worktime);
		mod_conf_show(IPA_CONF_SECT_LIMIT, limit->no);
		if (limit->restart.cmds.sect_set) {
			print_sect_name("restart");
			print_sect_begin();
			if (limit->restart.restart.upto != TEXP_UPTO_NOTSET) {
				print_param_name("restart");
				print_texp(&limit->restart.restart);
				print_param_end();
			}
			show_cmds(&limit->restart.cmds);
			mod_conf_show(IPA_CONF_SECT_RESTART, 0);
			print_sect_end();
		}
		show_commands(IPA_CONF_SECT_REACH, &limit->reach);
		if (limit->expire.cmds.sect_set) {
			print_sect_name("expire");
			print_sect_begin();
			if (limit->expire.expire.upto != TEXP_UPTO_NOTSET) {
				print_param_name("expire");
				print_texp(&limit->expire.expire);
				print_param_end();
			}
			show_cmds(&limit->expire.cmds);
			mod_conf_show(IPA_CONF_SECT_EXPIRE, 0);
			print_sect_end();
		}
		show_limit_rc(limit->rc);
# ifdef WITH_SUBLIMITS
		sect_is_visible = 0;
		STAILQ_FOREACH(sublimit, &limit->sublimits, link) {
			print_sect_name("sublimit");
			if (sublimit->lim_pc != 0)
				printf("%u%%", sublimit->lim_pc);
			else
				print_value(&sublimit->lim, sublimit->cnt_type);
			printf(" ");
			print_sect_begin();
			show_commands(IPA_CONF_SECT_REACH, &sublimit->reach);
			show_limit_rc(sublimit->rc);
			print_sect_end();
		}
		sect_is_visible = 1;
# endif /* WITH_SUBLIMITS */
		print_sect_end();
	}
}
#endif /* WITH_LIMITS */

#ifdef WITH_THRESHOLDS
static void
show_load_threshold(signed char val)
{
	if (val >= 0) {
		print_param_name("load_threshold");
		print_boolean(val);
		print_param_end();
	}
}

static void
show_threshold_type(signed char val)
{
	if (val >= 0) {
		print_param_name("threshold_type");
		printf("%d", val);
		print_param_end();
	}
}

static void
show_threshold_time_width(unsigned int val)
{
	if (val != 0) {
		print_param_name("threshold_time_width");
		print_time_u_int(val);
		print_param_end();
	}
}

static void
show_threshold_time_slice(const struct tevent *tevent)
{
	if (tevent != NULL) {
		print_param_name("threshold_time_slice");
		print_time_u_int(tevent->event_step);
		print_param_end();
	}
}

static void
show_threshold_balance(unsigned int a, unsigned int b, unsigned int c)
{
	if (a != 0) {
		print_param_name("threshold_balance");
		if (a != UINT_MAX)
			printf("%u:", a);
		else
			printf("-:");
		if (b != UINT_MAX)
			printf("%u:", b);
		else
			printf("-:");
		if (c != UINT_MAX)
			printf("%u", c);
		else
			printf("-");
		print_param_end();
	}
}

static void
show_debug_threshold(signed char val1, signed char val2)
{
	if (val1 >= 0) {
		print_param_name("debug_threshold");
		printf("%d", val1);
		print_param_end();
	}
	if (val2 >= 0) {
		print_param_name("debug_threshold_init");
		printf("%d", val2);
		print_param_end();
	}
}

static void
show_thresholds(const struct thresholds_list *list)
{
	const struct threshold *threshold;

	STAILQ_FOREACH(threshold, list, link) {
		print_sect_name("threshold");
		printf("%s ", threshold->name);
		print_sect_begin();
		show_db_list(threshold->db_list);
		print_info(threshold->info);
		print_param_name("threshold");
		print_value(&threshold->thr, threshold->cnt_type);
		print_param_end();
		if (threshold->thr_dev != 0) {
			print_param_name("threshold_deviation");
			if (threshold->thr_dev_pc != 0)
				printf("%u%%", threshold->thr_dev_pc);
			else
				print_value(&threshold->thr_dev,
				    threshold->cnt_type);
			print_param_end();
		}
		show_threshold_type(threshold->thr_type);
		show_threshold_balance(threshold->below_lim,
		    threshold->equal_lim, threshold->above_lim);
		show_load_threshold(threshold->load_thr);
		show_worktime(threshold->worktime);
		show_threshold_time_width(threshold->time_width);
		show_threshold_time_slice(threshold->time_slice);
		mod_conf_show(IPA_CONF_SECT_THRESHOLD, threshold->no);
		show_commands(IPA_CONF_SECT_STARTUP,
		    &threshold->rc[RC_STARTUP]);
		show_commands(IPA_CONF_SECT_SHUTDOWN,
		    &threshold->rc[RC_SHUTDOWN]);
		show_commands(IPA_CONF_SECT_BELOW_THRESHOLD,
		    &threshold->below_thr);
		show_commands(IPA_CONF_SECT_EQUAL_THRESHOLD,
		    &threshold->equal_thr);
		show_commands(IPA_CONF_SECT_ABOVE_THRESHOLD,
		    &threshold->above_thr);
		print_sect_end();
	}
}
#endif /* WITH_THRESHOLDS */

static void
show_ctl_socket_perm(void)
{
	if (ctl_socket_perm != 0) {
		print_param_name("ctl_socket_perm");
		if (ctl_socket_perm & S_IWUSR)
			printf("u");
		if (ctl_socket_perm & S_IWGRP)
			printf("g");
#ifdef CTL_CHECK_CREDS
		if (ctl_socket_perm & S_IWOTH)
			printf("o");
#endif
		print_param_end();
		need_nl = 1;
	}
}

#ifdef CTL_CHECK_CREDS
static void
show_ctl_xxx_acl(const struct ctl_acl_class *class, const char *param_name)
{
	if (class != NULL) {
		print_param_name(param_name);
		printf("%s", class->class_name);
		print_param_end();
	}
}

static void
show_ctl_rule_acl(const struct ctl_acl_class *class)
{
	show_ctl_xxx_acl(class, "ctl_rule_acl");
}

static void
show_ctl_acl_classes(void)
{
	const struct ctl_acl_class *class;
	const struct ctl_acl_elem *elem;

	STAILQ_FOREACH(class, &ctl_acl_classes, link) {
		print_param_name("ctl_acl_class");
		printf("%s", class->class_name);
		STAILQ_FOREACH(elem, &class->list, link) {
			printf(elem->allowed ? " " : " !");
			if (elem->user != NULL)
				printf("%s", elem->user);
			else
				printf("%%%s", elem->group);
		}
		print_param_end();
		need_nl = 1;
	}
}
#endif /* CTL_CHECK_CREDS */

static void
show_update_time(const struct tevent *tevent)
{
	if (tevent != NULL) {
		print_param_name("update_time");
		print_time_u_int(tevent->event_step);
		print_param_end();
	}
}

static void
show_append_time(const struct tevent *tevent)
{
	if (tevent != NULL) {
		print_param_name("append_time");
		print_time_u_int(tevent->event_step);
		print_param_end();
	}
}

/*
 * Main function for outputting configuration.
 */
void
show_config(void)
{
#ifdef WITH_RULES
	const struct rule *rule;
#endif
#ifdef WITH_AUTORULES
	const struct autorule *autorule;
#endif
	const struct rulepat *rulepat;
	const struct ac_mod *ac_mod;
	const struct db_mod *db_mod;
	const struct time_param *time_param;
	const struct debug_param *debug_param;
	uint64_t val;
	unsigned int i;

	printf(
"/*\n\
 * This output is not identical to the original content of the configuration\n\
 * file(s), this is just how ipa(8) and IPA modules see their configurations.\n\
 * Any \"include\" or \"include_files\" parameters are not printed and all\n\
 * macro variables are expanded.\n\
 */\n\n");

	if (mimic_real_config)
		printf("/* Mimic real configuration regime (" IPA_NAME "-"
		    PACKAGE_VERSION "). */\n\n");

	sect_is_visible = 1;

	SLIST_FOREACH(ac_mod, &ac_mod_list, link) {
		print_param_name0("ac_mod");
		print_string(ac_mod->mod_file);
		print_param_end();
		need_nl = 1;
	}
	SLIST_FOREACH(db_mod, &db_mod_list, link) {
		print_param_name0("db_mod");
		print_string(db_mod->mod_file);
		print_param_end();
		need_nl = 1;
	}
	print_nl_cond();

	for (i = 0; i < ROOT_DEBUG_PARAM_TBL_SIZE; ++i) {
		debug_param = &root_debug_param[i];
		if (*debug_param->value >= 0) {
			print_param_name(debug_param->name);
			printf("%d", *debug_param->value);
			print_param_end();
			need_nl = 1;
		}
	}
	print_nl_cond();

	if (shell_path != NULL) {
		print_param_name("shell_path");
		print_string(shell_path);
		print_param_end();
		need_nl = 1;
	}
	if (shell_arg1 != NULL) {
		print_param_name("shell_arg1");
		print_string(shell_arg1);
		print_param_end();
		need_nl = 1;
	}
	print_nl_cond();

	if (posix_re_pattern >= 0) {
		print_param_name("posix_re_pattern");
		print_boolean(posix_re_pattern);
		print_param_end();
		need_nl = 1;
	}
	print_nl_cond();

	for (i = 0; i < ROOT_TIME_PARAM_TBL_SIZE; ++i) {
		time_param = &root_time_param[i];
		if (*time_param->value > 0) {
			print_param_name(time_param->name);
			print_time_u_int(*time_param->value);
			print_param_end();
			need_nl = 1;
		}
	}
	print_nl_cond();

	if (only_abs_paths >= 0) {
		print_param_name("only_abs_paths");
		print_boolean(only_abs_paths);
		print_param_end();
		need_nl = 1;
	}
	print_nl_cond();

	if (keep_rules_order >= 0) {
		print_param_name("keep_rules_order");
		print_boolean(keep_rules_order);
		print_param_end();
		need_nl = 1;
	}
	print_nl_cond();

	if (ctl_enable >= 0) {
		print_param_name("ctl_enable");
		print_boolean(ctl_enable);
		print_param_end();
		need_nl = 1;
	}
	if (ctl_socket_path != NULL) {
		print_param_name("ctl_socket_path");
		print_string(ctl_socket_path);
		print_param_end();
		need_nl = 1;
	}
	if (ctl_timeout != 0) {
		print_param_name("ctl_timeout");
		print_time_u_int(ctl_timeout);
		print_param_end();
		need_nl = 1;
	}
	if (ctl_query_max_size != 0) {
		print_param_name("ctl_query_max_size");
		val = (uint64_t)ctl_query_max_size;
		print_bytes(&val);
		print_param_end();
		need_nl = 1;
	}
	show_ctl_socket_perm();
#ifdef CTL_CHECK_CREDS
	show_ctl_acl_classes();
	show_ctl_xxx_acl(ctl_dump_acl, "ctl_dump_acl");
	show_ctl_xxx_acl(ctl_freeze_acl, "ctl_freeze_acl");
	show_ctl_xxx_acl(ctl_stat_acl, "ctl_stat_acl");
#endif
	print_nl_cond();

	if (value_units >= 0) {
		print_param_name("value_units");
		for (i = 0; i < VALUE_UNITS_TBL_SIZE; ++i)
			if (value_units == value_units_tbl[i].val) {
				printf("%s", value_units_tbl[i].str);
				break;
			}
		print_param_end();
		print_nl();
	}

	mod_conf_show(IPA_CONF_SECT_ROOT, 0);

	if (cmds_global[RC_STARTUP].sect_set) {
		show_commands(IPA_CONF_SECT_STARTUP,
		    &cmds_global[RC_STARTUP]);
		print_nl();
	}
	if (cmds_global[RC_SHUTDOWN].sect_set) {
		show_commands(IPA_CONF_SECT_SHUTDOWN,
		    &cmds_global[RC_SHUTDOWN]);
		print_nl();
	}

	if (global_section_set) {
		print_sect_name("global");
		print_sect_begin();
		show_ac_list(global_ac_list);
		show_db_list(global_db_list);
		show_append_time(global_append_tevent);
		show_update_time(global_update_tevent);
		show_worktime(global_worktime);
#ifdef WITH_LIMITS
		show_load_limit(global_load_limit);
#endif
#ifdef WITH_THRESHOLDS
		show_threshold_type(global_threshold_type);
		show_threshold_balance(global_threshold_below_lim,
		    global_threshold_equal_lim, global_threshold_above_lim);
		show_load_threshold(global_load_threshold);
		show_threshold_time_width(global_threshold_time_width);
		show_threshold_time_slice(global_threshold_time_slice);
#endif
#ifdef CTL_CHECK_CREDS
		show_ctl_rule_acl(global_ctl_rule_acl);
#endif
		for (i = 0; i < GLOBAL_DEBUG_PARAM_TBL_SIZE; ++i) {
			debug_param = &global_debug_param[i];
			if (*debug_param->value >= 0) {
				print_param_name(debug_param->name);
				printf("%d", *debug_param->value);
				print_param_end();
				need_nl = 1;
			}
		}
		mod_conf_show(IPA_CONF_SECT_GLOBAL, 0);
		print_sect_end();
		print_nl();
	}

#ifdef WITH_AUTORULES
	STAILQ_FOREACH(autorule, &autorules_list, link) {
		print_sect_name("autorule");
		printf("%s ", autorule->name);
		print_sect_begin();
		show_ac_list(autorule->ac_list);
		show_db_list(autorule->db_list);
		show_append_time(autorule->append_tevent);
		show_update_time(autorule->update_tevent);
		show_worktime(autorule->worktime);
		show_worktime_generic(autorule->worktime_rule, "worktime_rule");
# ifdef CTL_CHECK_CREDS
		show_ctl_rule_acl(autorule->ctl_rule_acl);
# endif
		mod_conf_show(IPA_CONF_SECT_AUTORULE, autorule->no);
		show_debug_exec(autorule->debug_exec);
		show_rule_rc(autorule->rc);
# ifdef WITH_LIMITS
		show_debug_limit(autorule->debug_limit,
		    autorule->debug_limit_init);
		show_limits(&autorule->limits);
# endif
# ifdef WITH_THRESHOLDS
		show_debug_threshold(autorule->debug_threshold,
		    autorule->debug_threshold_init);
		show_thresholds(&autorule->thresholds);
# endif
		print_sect_end();
		print_nl();
	}
#endif /* WITH_AUTORULES */

	STAILQ_FOREACH(rulepat, &rulepats_list, link) {
		print_sect_name("rulepat");
		print_string(rulepat->pat);
		printf(" ");
		print_sect_begin();
		if (rulepat->check_next >= 0) {
			print_param_name("check_next_rulepat");
			print_boolean(rulepat->check_next);
			print_param_end();
		}
		show_ac_list(rulepat->ac_list);
		show_db_list(rulepat->db_list);
		show_append_time(rulepat->append_tevent);
		show_update_time(rulepat->update_tevent);
		show_worktime(rulepat->worktime);
#ifdef CTL_CHECK_CREDS
		show_ctl_rule_acl(rulepat->ctl_rule_acl);
#endif
		mod_conf_show(IPA_CONF_SECT_RULEPAT, rulepat->no);
		show_debug_exec(rulepat->debug_exec);
		show_rule_rc(rulepat->rc);
# ifdef WITH_LIMITS
		show_debug_limit(rulepat->debug_limit,
		    rulepat->debug_limit_init);
		show_limits(&rulepat->limits);
# endif
# ifdef WITH_THRESHOLDS
		show_debug_threshold(rulepat->debug_threshold,
		    rulepat->debug_threshold_init);
		show_thresholds(&rulepat->thresholds);
# endif
		print_sect_end();
		print_nl();
	}

#ifdef WITH_RULES
	TAILQ_FOREACH(rule, &rules_list, list) {
		print_sect_name("rule");
		printf("%s ", rule->name);
		print_sect_begin();
		show_ac_list(rule->ac_list);
		show_ac_gather_xxx(rule->acg_add_pat, "ac_gather_add");
		show_ac_gather_xxx(rule->acg_sub_pat, "ac_gather_sub");
		show_db_list(rule->db_list);
		print_info(rule->info);
		show_append_time(rule->append_tevent);
		show_update_time(rule->update_tevent);
		show_worktime(rule->worktime);
# ifdef CTL_CHECK_CREDS
		show_ctl_rule_acl(rule->ctl_rule_acl);
# endif
		mod_conf_show(IPA_CONF_SECT_RULE, rule->no);
		show_debug_exec(rule->debug_exec);
		show_rule_rc(rule->rc);
# ifdef WITH_LIMITS
		show_debug_limit(rule->debug_limit,
		    rule->debug_limit_init);
		show_limits(&rule->limits);
# endif
# ifdef WITH_THRESHOLDS
		show_debug_threshold(rule->debug_threshold,
		    rule->debug_threshold_init);
		show_thresholds(&rule->thresholds);
# endif
		print_sect_end();
		print_nl();
	}
#endif /* WITH_RULES */
}
