/*-
 * Copyright (c) 2000-2003 Andrey Simonenko
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "config.h"

#ifndef lint
static const char rcsid[] ATTR_UNUSED =
  "@(#)$Id: ipa_cmd.c,v 1.3.2.1 2011/11/15 18:12:29 simon Exp $";
#endif /* !lint */

#include <sys/types.h>
#include <sys/time.h>
#include <sys/wait.h>

#include <errno.h>
#include <grp.h>
#include <limits.h>
#include <pwd.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#ifdef HAVE_PATHS_H
# include <paths.h>
#endif
#ifndef _PATH_BSHELL
# ifdef __GNUC__
#  warning "defining _PATH_BSHELL to \"/bin/sh\""
# endif
# define _PATH_BSHELL "/bin/sh"
#endif

#include "ipa_mod.h"

#include "queue.h"

#include "dlapi.h"
#include "confcommon.h"
#include "memfunc.h"
#include "parser.h"

#include "ipa_ac.h"
#include "ipa_db.h"
#include "ipa_ctl.h"
#include "ipa_cmd.h"
#include "ipa_time.h"

#include "ipa_conf.h"
#include "ipa_log.h"
#include "ipa_main.h"
#include "ipa_rules.h"
#include "ipa_autorules.h"

#ifndef WCOREDUMP
# define WCOREDUMP(x) (0)
#endif

#if defined(HAVE_CLOSEFROM) && \
	(STDIN_FILENO > 2 || STDOUT_FILENO > 2 || STDERR_FILENO > 2)
# undef HAVE_CLOSEFROM
#endif

/*
 * run_cmd() and run_cmd_cons() return -1 only if they cannot fork()
 * a new process or if they get an error from waitpid(), if any error
 * occurred in a child forked from these functions, then such error
 * is invisible for run_cmds*() functions.
 */

/*
 * If run_cmds() function returns -1, then we should terminate ipa(8).
 */

/*
 * If return code from a child forked in run_cmdl_bg() or
 * in run_cmdl_fg() is non-zero, then we should terminate ipa(8).
 */

/* Mzone for all struct cmd{}. */
ipa_mzone	*cmd_mzone;

/* Memory for commands. */
ipa_mem_type	*m_cmd;

/* shell_path parameter and its default value. */
char		*shell_path = NULL;
char		*shell_path_default = _PATH_BSHELL;

/* shell_arg1 parameter and its default value. */
char		*shell_arg1 = NULL;
char		*shell_arg1_default = "-c";

/* Section names for RC_STARTUP and RC_SHUTDOWN. */
const char *const rc_sect_name[] = { "startup", "shutdown" };

signed char	global_debug_exec;	/* global { debug_exec } */

struct cmds	cmds_global[2];		/* Global commands. */

#ifdef WITH_ANY_LIMITS
struct wpid_hash wpid_hash[WPID_HASH_BUCKETS];
#endif

static void	outerr(const char *, ...) ATTR_FORMAT(printf, 1, 2);
static void	outerrx(const char *, ...) ATTR_FORMAT(printf, 1, 2);
static void	outwarn(const char *, ...) ATTR_FORMAT(printf, 1, 2);
static void	run_cmd_child(const struct cmd *) ATTR_NORETURN;
static void	run_cmd_child_cons(const struct cmd *) ATTR_NORETURN;
static void	run_cmdl_child(const struct cmd_list *, int, const char *,
		    va_list) ATTR_FORMAT(printf, 3, 0) ATTR_NORETURN;

/*
 * Set supplementary groups for process, gid is a GID from getpwnam(),
 * NGROUPS_MAX is defined in SUSv3.
 */
int
setsuppgids(const char *user, gid_t gid)
{
	gid_t gids[NGROUPS_MAX + 1];
	const struct group *grp;
	char *const *gr_user;
	int i, ngids;

	gids[0] = gid;
	ngids = 1;

	errno = 0;
	while ((grp = getgrent()) != NULL) {
		for (i = 0; i < ngids; ++i)
			if (grp->gr_gid == gids[i])
				goto next;
		for (gr_user = grp->gr_mem; *gr_user != NULL; ++gr_user) {
			if (strcmp(user, *gr_user) == 0) {
				if (ngids < NGROUPS_MAX + 1) {
					gids[ngids++] = grp->gr_gid;
					break;
				}
			}
		}
next:		;
	}
	if (errno != 0) {
		xlogmsgx(IPA_LOG_ERR, "setsuppgids: getgrent: %s",
		    strerror(errno));
		return (-1);
	}
	if (setgroups(ngids, gids) < 0) {
		xlogmsgx(IPA_LOG_ERR, "setsuppgids: setgroups: %s",
		    strerror(errno));
		return (-1);
	}
	return (0);
}

static void
run_cmd_child(const struct cmd *cmd)
{
	const struct passwd *pwd;

	if (cmd->user != NULL) {
		errno = 0;
		pwd = getpwnam(cmd->user);
		if (pwd == NULL) {
			int errno_save;

			errno_save = errno;
			open_log();
			if (errno_save != 0) {
				errno = errno_save;
				logmsg(IPA_LOG_ERR, "run_cmd_child: "
				    "getpwnam(%s)", cmd->user);
			} else
				logmsgx(IPA_LOG_ERR, "run_cmd_child: cannot "
				    "find user \"%s\"", cmd->user);
			goto failed;
		}
		if (setgid(pwd->pw_gid) < 0) {
			open_log();
			logmsg(IPA_LOG_ERR, "run_cmd_child: setgid(%lu)",
			    (unsigned long)pwd->pw_gid);
			goto failed;
		}
		if (setsuppgids(cmd->user, pwd->pw_gid) < 0) {
			open_log();
			logmsgx(IPA_LOG_ERR, "run_cmd_child: cannot set all "
			    "groups for user %s", cmd->user);
			goto failed;
		}
		if (setuid(pwd->pw_uid) < 0) {
			open_log();
			logmsg(IPA_LOG_ERR, "run_cmd_child: setuid(%lu)",
			    (unsigned long)pwd->pw_uid);
			goto failed;
		}
		/* Just, to be sure that descriptors are closed. */
		endpwent();
		endgrent();
	}

	execl(shell_path, shell_path, shell_arg1, cmd->str, (char *)NULL);
	open_log();
	logmsg(IPA_LOG_ERR, "run_cmd_child: execl(\"%s\", \"%s\", \"%s\", %s)",
	    shell_path, shell_path, shell_arg1, parser_stringify(cmd->str));
failed:
	close_log();
	_exit(127);
}

/*
 * Run command changing user if needed, this function is
 * called with disabled log system.  Return -1 if some error
 * occurs and do not close log system in this case.
 */
static int
run_cmd(const struct cmd *cmd)
{
	pid_t pid;
	int status;

	pid = fork();
	if (pid == (pid_t)-1) {
		open_log();
		logmsg(IPA_LOG_ERR, "run_cmd: fork");
		return (-1);
	}

	if (pid == 0) {
		/* Child. */
		run_cmd_child(cmd);
		/* NOTREACHED */
	}

	/* Parent. */
	while (waitpid(pid, &status, 0) == (pid_t)-1) {
		open_log();
		if (errno != EINTR) {
			logmsg(IPA_LOG_ERR, "run_cmd: waitpid(%ld)", (long)pid);
			return (-1);
		}
		logmsg(IPA_LOG_WARNING, "run_cmd: waitpid(%ld)", (long)pid);
		close_log();
	}

	if (WIFEXITED(status)) {
		if (WEXITSTATUS(status) != 0) {
			open_log();
			logmsgx(IPA_LOG_WARNING, "run_cmd: exit status of "
			    "child %ld (%s) is %d%s", (long)pid,
			    parser_stringify(cmd->str), WEXITSTATUS(status),
			    WEXITSTATUS(status) == 127 ?
			    ", execution of the shell probably failed" : "");
			close_log();
		}
	} else {
		open_log();
		if (WIFSIGNALED(status))
			logmsgx(IPA_LOG_ERR, "run_cmd: abnormal termination "
			    "of child %ld (%s), signal %d%s", (long)pid,
			    parser_stringify(cmd->str), WTERMSIG(status),
			    WCOREDUMP(status) ? " (core file generated)" : "");
		else
			logmsgx(IPA_LOG_ERR, "run_cmd: termination of child "
			    "%ld (%s), unknown status", (long)pid,
			    parser_stringify(cmd->str));
		close_log();
	}

	return (0);
}

#ifndef HAVE_CLOSEFROM
static void
close_my_fd(int nofile_max)
{
	int fd;

# if STDIN_FILENO < 3 && STDOUT_FILENO < 3 && STDERR_FILENO < 3
	for (fd = 3; fd < nofile_max; ++fd)
		(void)close(fd);
# else
	for (fd = 0; fd < nofile_max; ++fd)
		switch (fd) {
		case STDIN_FILENO:
		case STDOUT_FILENO:
		case STDERR_FILENO:
			break;
		default:
			(void)close(fd);
		}
# endif
}
#endif /* !HAVE_CLOSEFROM */

static void
log_failed_cmd(const struct cmd *cmd, const char *format, va_list ap)
{
	char *buf;

	mem_vasprintf(m_tmp, &buf, format, ap);
	logmsgx(IPA_LOG_ERR, "execution of command %s from section %s failed",
	    parser_stringify(cmd->str), buf != NULL ? buf : "(no memory)");
	mem_free(buf, m_tmp);
}

static void
run_cmdl_child(const struct cmd_list *cmd_list, int debug_exec,
    const char *format, va_list ap)
{
	const struct cmd *cmd;
#ifndef HAVE_CLOSEFROM
	long val;
#endif

	/* Close all descriptors, except stdin, stdout and stderr. */
#ifdef HAVE_CLOSEFROM
	close_log();
	(void)closefrom(3);
#else
	/*
	 * Since a module can call setrlimit(), it is necessary
	 * to call sysconf(_SC_OPEN_MAX).
	 */
	errno = 0;
	val = sysconf(_SC_OPEN_MAX);
	if (val < 0) {
		if (errno != 0)
			logmsg(IPA_LOG_ERR, "run_cmdl_child: "
			    "sysconf(_SC_OPEN_MAX)");
		else
			logmsgx(IPA_LOG_ERR, "run_cmdl_child: infinite limit "
			    "for _SC_OPEN_MAX is not supported");
		goto failed;
	}
# if LONG_MAX > INT_MAX
	if (val > INT_MAX) {
		logmsgx(IPA_LOG_ERR, "run_cmdl_child: too big value %ld for "
		    "'int' type from sysconf(_SC_OPEN_MAX)", val);
		goto failed;
	}
# endif
	close_log();
	close_my_fd((int)val);
#endif /* HAVE_CLOSEFROM */

	STAILQ_FOREACH(cmd, cmd_list, link) {
		if (debug_exec > 1) {
			open_log();
			logdbg("exec %s %s",
			    cmd->user == NULL ? "" : cmd->user,
			    parser_stringify(cmd->str));
			close_log();
		}
		if (run_cmd(cmd) < 0) {
			/* Log was opened in exec_cmd(). */
			logbt("run_cmdl_child");
			log_failed_cmd(cmd, format, ap);
			goto failed;
		}
	}
	_exit(EXIT_SUCCESS);

failed:
	close_log();
	_exit(EXIT_FAILURE);
}

/*
 * Run commands list in background and return PID of a child
 * (internal version).
 */
static pid_t
run_cmdl(const struct cmd_list *cmd_list, int debug_exec,
    const char *format, va_list ap)
{
	pid_t pid;

	pid = fork();
	if (pid == (pid_t)-1) {
		logmsg(IPA_LOG_ERR, "run_cmdl: fork");
		return ((pid_t)-1);
	}

	if (pid == 0) {
		/* Child. */
		run_cmdl_child(cmd_list, debug_exec, format, ap);
		/* NOTREACHED */
	}

	/* Parent. */
	return (pid);

}

/*
 * Run commands list in background and wait.
 */
static int
run_cmdl_fg(const struct cmd_list *cmd_list, int debug_exec,
    const char *format, va_list ap)
{
	pid_t pid;
	int status;

	pid = run_cmdl(cmd_list, debug_exec, format, ap);
	if (pid == (pid_t)-1) {
		logbt("run_cmdl_fg");
		return (-1);
	}

	while (waitpid(pid, &status, 0) == (pid_t)-1) {
		if (errno != EINTR) {
			logmsg(IPA_LOG_ERR, "run_cmdl_fg: waitpid(%ld)",
			    (long)pid);
			return (-1);
		}
		logmsg(IPA_LOG_WARNING, "run_cmdl_fg: waitpid(%ld)",
		    (long)pid);
	}
	log_stdall();
	if (WIFEXITED(status)) {
		if (WEXITSTATUS(status) != EXIT_SUCCESS) {
			logmsgx(IPA_LOG_WARNING, "run_cmdl_fg: exit status of "
			    "child %ld is %d",(long)pid, WEXITSTATUS(status));
			return (-1);
		}
	} else {
		if (WIFSIGNALED(status))
			logmsgx(IPA_LOG_ERR, "run_cmdl_fg: abnormal "
			    "termination of child %ld, signal %d%s", (long)pid,
			    WTERMSIG(status), WCOREDUMP(status) ?
			    " (core file generated)" : "");
		else
			logmsgx(IPA_LOG_ERR, "run_cmdl_fg: termination of "
			    "child %ld, unknown status", (long)pid);
		return (-1);
	}

	return (0);
}

/*
 * Run commands list in background and return PID of a child.
 */
static pid_t
run_cmdl_bg(const struct cmd_list *cmd_list, int debug_exec,
    const char *format, va_list ap)
{
	pid_t pid;

	pid = run_cmdl(cmd_list, debug_exec, format, ap);
	if (pid == (pid_t)-1)
		logbt("run_cmdl_bg");
	return (pid);
}

#ifdef WITH_ANY_LIMITS

#define WPID_ARG_ATTR

/*
 * Initialize lists of wpid_hash;
 */
void
wpid_hash_init(void)
{
	struct wpid_hash *h;

	for (h = wpid_hash; h < wpid_hash + WPID_HASH_BUCKETS; ++h)
		LIST_INIT(h);
}

/*
 * Return non-zero if all lists in wpid_hash are empty,
 * otherwise return zero.
 */
int
wpid_hash_is_empty(void)
{
	const struct wpid_hash *h;

	for (h = wpid_hash; h < wpid_hash + WPID_HASH_BUCKETS; ++h)
		if (!LIST_EMPTY(h))
			return (0);
	return (1);
}

/*
 * Remove wpid structure from the hash, wpid is there
 * if its pid field is not equal to zero.
 */
void
wpid_hash_rem(struct wpid *wpid)
{
	if (wpid->pid != 0) {
		wpid->pid = 0;
		LIST_REMOVE(wpid, link);
	}
}

/*
 * Quick version of wpid_hash_rem().
 */
# define wpid_hash_rem_quick(w)	\
	LIST_REMOVE((w), link)

/*
 * Add wpid to wpid_hash.
 */
# define wpid_hash_add(w) \
	LIST_INSERT_HEAD(&wpid_hash[wpid_hash_bucket((w)->pid)], wpid, link)

/*
 * Lookup wpid for the given PID.
 */
struct wpid *
wpid_lookup_pid(pid_t pid)
{
	struct wpid_hash *h;
	struct wpid *wpid;

	h = &wpid_hash[wpid_hash_bucket(pid)];
	LIST_FOREACH(wpid, h, link)
		if (wpid->pid == pid)
			break;
	return (wpid);
}

/*
 * Log message about previous wpid.
 */
static void
log_prev_wpid(const struct wpid *wpid, const char *format, va_list ap)
{
	char *buf;

	mem_vasprintf(m_tmp, &buf, format, ap);
	logmsgx(IPA_LOG_WARNING, "previously run process with PID %ld "
	    "for %s has not exited yet", (long)wpid->pid,
	    buf != NULL ? buf : "(no memory)");
	mem_free(buf, m_tmp);
}

void
log_term_wpid(const struct wpid *wpid)
{
	const struct rule *rule;
	long pid;

	pid = (long)wpid->pid;
	switch (wpid->type) {
# ifdef WITH_LIMITS
	case WPID_TYPE_LIMIT:
		{
			const struct limit *limit;

			limit = wpid->u.limit;
			rule = limit->rule;
			logdbg("rule %s, limit %s: child %ld exited",
			    rule->name, limit->name, pid);
		}
		break;
# endif
# ifdef WITH_SUBLIMITS
	case WPID_TYPE_SUBLIMIT:
		{
			const struct sublimit *sublimit;
			const struct limit *limit;

			sublimit = wpid->u.sublimit;
			limit = sublimit->limit;
			rule = limit->rule;
			logdbg("rule %s, limit %s, sublimit %s: "
			    "child %ld exited", rule->name, limit->name,
			    sublimit->name, pid);
		}
		break;
# endif
# ifdef WITH_THRESHOLDS
	case WPID_TYPE_THRESHOLD:
		{
			const struct threshold *threshold;

			threshold = wpid->u.threshold;
			rule = threshold->rule;
			logdbg("rule %s, threshold %s: child %ld exited",
			    rule->name, threshold->name, pid);
		}
		break;
# endif
	}
}

#else

# define WPID_ARG_ATTR ATTR_UNUSED

#endif /* WITH_ANY_LIMITS */

/*
 * Log debug message about running commands.
 */
static void
log_run_cmds(const struct cmds *cmds, const char *format, va_list ap)
{
	char *buf;

	mem_vasprintf(m_tmp, &buf, format, ap);
	logdbg("running %s (%ssync)", buf != NULL ? buf : "(no memory)",
	    cmds->sync ? "" : "a");
	mem_free(buf, m_tmp);
}

/*
 * Log debug message about failed commands.
 */
static void
log_failed_cmds(const struct cmds *cmds, const char *format, va_list ap)
{
	char *buf;

	mem_vasprintf(m_tmp, &buf, format, ap);
	logmsgx(IPA_LOG_ERR, "failed %s (%ssync)",
	    buf != NULL ? buf : "(no memory)", cmds->sync ? "" : "a");
	mem_free(buf, m_tmp);
}

/*
 * Run commands from cmds, rule and format with
 * rest arguments are used for debugging.
 */
int
run_cmds(const struct rule *rule, struct wpid *wpid WPID_ARG_ATTR,
    const struct cmds *cmds, const char *format, ...)
{
	const struct cmd_list *cmd_list;
	va_list ap;
	pid_t pid;
	int debug_exec;

	va_start(ap, format);
	debug_exec = rule->debug_exec;
	if (debug_exec)
		log_run_cmds(cmds, format, ap);

#ifdef WITH_RULES
	if (!STAILQ_EMPTY(&cmds->ictl_list))
		if (run_ictl_list(&cmds->ictl_list) < 0)
			goto failed;
#endif

	cmd_list = &cmds->cmd_list;
	if (!STAILQ_EMPTY(cmd_list)) {
		if (cmds->sync) {
			if (run_cmdl_fg(cmd_list, debug_exec, format, ap) < 0)
				goto failed;
		} else {
#ifdef WITH_ANY_LIMITS
			if (wpid != NULL && wpid->pid != 0) {
				log_prev_wpid(wpid, format, ap);
				wpid_hash_rem_quick(wpid);
			}
#endif
			pid = run_cmdl_bg(cmd_list, debug_exec, format, ap);
			if (pid == (pid_t)-1)
				goto failed;
#ifdef WITH_ANY_LIMITS
			if (wpid != NULL) {
				wpid->pid = pid;
				wpid_hash_add(wpid);
			}
#endif
		}
	}

	va_end(ap);
	return (0);

failed:
	logbt("run_cmds");
	log_failed_cmds(cmds, format, ap);
	va_end(ap);
	return (-1);
}

/*
 * Output a message to stderr and an error message if errno != 0.
 */
static void
outerr(const char *format, ...)
{
	va_list ap;
	int errno_save;

	errno_save = errno;
	va_start(ap, format);
	vlogmsgx_stderr(IPA_LOG_ERR, format, ap);
	va_end(ap);
	if (errno_save != 0)
		fprintf(stderr, "Error: %s.\n", strerror(errno_save));
}

/*
 * Output a message to stderr.
 */
static void
outerrx(const char *format, ...)
{
	va_list ap;

	va_start(ap, format);
	vlogmsgx_stderr(IPA_LOG_ERR, format, ap);
	va_end(ap);
}

/*
 * Output a warning message to stderr.
 */
static void
outwarn(const char *format, ...)
{
	va_list ap;

	va_start(ap, format);
	vlogmsgx_stderr(IPA_LOG_WARNING, format, ap);
	va_end(ap);
}

static void
run_cmd_child_cons(const struct cmd *cmd)
{
	const struct passwd *pwd;

	if (cmd->user != NULL) {
		errno = 0;
		pwd = getpwnam(cmd->user);
		if (pwd == NULL) {
			if (errno != 0)
				outerr("run_cmd_child_cons: getpwnam(%s)",
				    cmd->user);
			else
				outerrx("run_cmd_child_cons: cannot find "
				    "user \"%s\"", cmd->user);
			goto failed;
		}
		if (setgid(pwd->pw_gid) < 0) {
			outerr("run_cmd_child_cons: setgid(%lu)",
			    (unsigned long)pwd->pw_gid);
			goto failed;
		}
		if (setsuppgids(cmd->user, pwd->pw_gid) < 0) {
			outerrx("run_cmd_child_cons: cannot set all groups "
			    "for user %s", cmd->user);
			goto failed;
		}
		if (setuid(pwd->pw_uid) < 0) {
			outerr("run_cmd_child_cons: setuid(%lu)",
			    (unsigned long)pwd->pw_uid);
			goto failed;
		}
		/* Just, to be sure that descriptors are closed. */
		endpwent();
		endgrent();
	}

	execl(shell_path, shell_path, shell_arg1, cmd->str, (char *)NULL);
	outerr("run_cmd_child_cons: execl(\"%s\", \"%s\", \"%s\", %s)",
	    shell_path, shell_path, shell_arg1, parser_stringify(cmd->str));
failed:
	_exit(127);
}

/*
 * Run a command in foreground.  Return -1 if some error occurs.
 */
static int
run_cmd_cons(const struct cmd *cmd)
{
	pid_t pid;
	int status;

	pid = fork();
	if (pid == (pid_t)-1) {
		outerr("run_cmd_cons: fork");
		return (-1);
	}

	if (pid == 0) {
		/* Child. */
		run_cmd_child_cons(cmd);
		/* NOTREACHED */
	}

	/* Parent. */
	while (waitpid(pid, &status, 0) == (pid_t)-1)
		if (errno != EINTR) {
			outerr("run_cmd_cons: waitpid");
			return (-1);
		}

	if (WIFEXITED(status)) {
		if (WEXITSTATUS(status) != 0)
			outwarn("run_cmd_cons: exit status of child is %d%s",
			    WEXITSTATUS(status), WEXITSTATUS(status) == 127 ?
			    ", execution of the shell probably failed" : "");
	} else if (WIFSIGNALED(status))
		outwarn("run_cmd_cons: abnormal termination of child, signal "
		    "%d%s", WTERMSIG(status), WCOREDUMP(status) ?
		    " (core file generated)" : "");
	else
		outwarn("run_cmd_cons: termination of child, unknown status");

	return (0);
}

/*
 * Run commands from cmd_list in foreground, ignoring if
 * some command cannot be run, user does not exist, etc.
 */
int
run_cmds_cons(const struct cmds *cmds, const char *message, ...)
{
	const struct cmd_list *cmd_list;
	const struct cmd *cmd;
	va_list ap;
#ifndef HAVE_CLOSEFROM
	long val;
#endif

	printf("Running commands from ");
	va_start(ap, message);
	vprintf(message, ap);
	va_end(ap);
	printf(" section:\n");

#ifdef WITH_RULES
	if (!STAILQ_EMPTY(&cmds->ictl_list))
		printf("- skipping all \"ictl\" commands\n");
#endif

	cmd_list = &cmds->cmd_list;
	if (STAILQ_EMPTY(cmd_list)) {
		printf("- nothing to run\n");
		return (0);
	}

	/* Close all descriptors, except stdin, stdout and stderr. */
#ifdef HAVE_CLOSEFROM
	(void)closefrom(3);
#else
	errno = 0;
	val = sysconf(_SC_OPEN_MAX);
	if (val < 0) {
		if (errno != 0)
			outerr("run_cmds_cons: sysconf(_SC_OPEN_MAX)");
		else
			outerrx("run_cmds_cons: infinite limit for "
			    "_SC_OPEN_MAX is not supported");
		return (-1);
	}
# if LONG_MAX > INT_MAX
	if (val > INT_MAX) {
		outerrx("run_cmds_cons: too big value %ld for 'int' type "
		    "from sysconf(_SC_OPEN_MAX)", val);
		return (-1);
	}
# endif
	close_my_fd((int)val);
#endif /* HAVE_CLOSEFROM */

	STAILQ_FOREACH(cmd, cmd_list, link) {
		printf("* exec ");
		if (cmd->user != NULL)
			printf("%s ", cmd->user);
		print_string(cmd->str);
		printf("\n");
		fflush(stdout);
		if (run_cmd_cons(cmd) < 0) {
			outerrx("stop running commands");
			return (-1);
		}
	}

	return (0);
}

#ifdef WITH_SUBLIMITS
static int
run_cmds_sublimit(const struct rule *rule, const struct limit *limit,
    struct sublimit *sublimit, unsigned int x)
{
	const char *rule_name, *limit_name, *sublimit_name, *sect_name;
	const struct cmds *cmds;
	struct cmds_limit *cmds_limit;

	rule_name = rule->name;
	limit_name = limit->name;
	sublimit_name = sublimit->name;
	sect_name = rc_sect_name[x];
	cmds_limit = &sublimit->rc[x];
	cmds = &cmds_limit->cmds;
	if (cmds->has_cmd && run_cmds(rule, (struct wpid *)NULL, cmds,
	    "rule %s { limit %s { sublimit %s { %s {}}}}",
	    rule_name, limit_name, sublimit_name, sect_name) < 0)
		goto failed;
	if (SUBLIMIT_IS_REACHED(sublimit)) {
		cmds = &cmds_limit->cmds_reached;
		if (cmds->has_cmd && run_cmds(rule, (struct wpid *)NULL, cmds,
		    "rule %s { limit %s { sublimit %s { %s { "
		    "if_reached{}}}}}", rule_name, limit_name,
		    sublimit_name, sect_name) < 0)
			goto failed;
	} else {
		cmds = &cmds_limit->cmds_not_reached;
		if (cmds->has_cmd && run_cmds(rule, (struct wpid *)NULL, cmds,
		    "rule %s { limit %s { sublimit %s { %s { "
		    "if_not_reached{}}}}}", rule_name, limit_name,
		    sublimit_name, sect_name) < 0)
			goto failed;
	}
	cmds_limit_free(cmds_limit);
	return (0);

failed:
	logbt("run_cmds_sublimit");
	return (-1);
}
#endif

#ifdef WITH_LIMITS
static int
run_cmds_limit(const struct rule *rule, struct limit *limit, unsigned int x)
{
	const struct cmds *cmds;
	const char *rule_name, *limit_name, *sect_name;
	struct cmds_limit *cmds_limit;
#ifdef WITH_SUBLIMITS
	struct sublimit	*sublimit;
#endif

	rule_name = rule->name;
	limit_name = limit->name;
	sect_name = rc_sect_name[x];
	cmds_limit = &limit->rc[x];
	cmds = &cmds_limit->cmds;
	if (cmds->has_cmd && run_cmds(rule, (struct wpid *)NULL, cmds,
	    "rule %s { limit %s { %s {}}}", rule_name, limit_name,
	    sect_name) < 0)
		goto failed;
	if (LIMIT_IS_REACHED(limit)) {
		cmds = &cmds_limit->cmds_reached;
		if (cmds->has_cmd && run_cmds(rule, (struct wpid *)NULL,
		    cmds, "rule %s { limit %s { %s { if_reached{}}}}",
		    rule_name, limit_name, sect_name) < 0)
			goto failed;
		if (ac_limit_event(rule, limit, x == RC_STARTUP ?
		    IPA_LIMIT_EVENT_STARTUP_IF_REACHED :
		    IPA_LIMIT_EVENT_SHUTDOWN_IF_REACHED) < 0)
			goto failed;
	} else {
		cmds = &cmds_limit->cmds_not_reached;
		if (cmds->has_cmd && run_cmds(rule, (struct wpid *)NULL,
		    cmds, "rule %s { limit %s { %s { if_not_reached{}}}}",
		    rule_name, limit_name, sect_name) < 0)
			goto failed;
		if (ac_limit_event(rule, limit, x == RC_STARTUP ?
		    IPA_LIMIT_EVENT_STARTUP_IF_NOT_REACHED :
		    IPA_LIMIT_EVENT_SHUTDOWN_IF_NOT_REACHED) < 0)
			goto failed;
	}
	cmds_limit_free(cmds_limit);
# ifdef WITH_SUBLIMITS
	STAILQ_FOREACH(sublimit, &limit->sublimits, link)
		if (run_cmds_sublimit(rule, limit, sublimit, x) < 0)
			goto failed;
# endif
	return (0);

failed:
	logbt("run_cmds_limit");
	return (-1);
}
#endif /* WITH_LIMITS */

#ifdef WITH_THRESHOLDS
static int
run_cmds_threshold(const struct rule *rule, struct threshold *threshold,
    unsigned int x)
{
	struct cmds *cmds;

	cmds = &threshold->rc[x];
	if (cmds->has_cmd && run_cmds(rule, (struct wpid *)NULL, cmds,
	    "rule %s { threshold %s { %s {}}}", rule->name, threshold->name,
	    rc_sect_name[x]) < 0) {
		logbt("run_cmds_threshold");
		return (-1);
	}
	cmds_free(cmds);
	return (0);
}
#endif

/*
 * Run startup or shutdown commands for a rule, its limits,
 * sublimits and thresholds.
 */
int
run_cmds_rule(struct rule *rule, unsigned int x)
{
	const char *rule_name, *sect_name;
	struct cmds *cmds;
	struct cmds_rule *cmds_rule;
#ifdef WITH_THRESHOLDS
	struct threshold *threshold;
#endif
#ifdef WITH_LIMITS
	struct limit *limit;
	char any_reached, all_reached;
#endif

	rule_name = rule->name;
	sect_name = rc_sect_name[x];
	cmds_rule = &rule->rc[x];
	cmds = &cmds_rule->cmds;
	if (cmds->has_cmd && run_cmds(rule, (struct wpid *)NULL, cmds,
	    "rule %s { %s {}}", rule_name, sect_name) < 0)
		goto failed;

#ifdef WITH_LIMITS
	any_reached = 0;
	all_reached = 1;
	STAILQ_FOREACH(limit, &rule->limits, link)
		if (LIMIT_IS_REACHED(limit))
			any_reached = 1;
		else
			all_reached = 0;
	if (any_reached) {
		cmds = &cmds_rule->cmds_any_reached;
		if (cmds->has_cmd && run_cmds(rule, (struct wpid *)NULL, cmds,
		    "rule %s { %s { if_any_reached {}}}", rule_name,
		    sect_name) < 0)
			goto failed;
	} else {
		cmds = &cmds_rule->cmds_any_not_reached;
		if (cmds->has_cmd && run_cmds(rule, (struct wpid *)NULL, cmds,
		    "rule %s { %s { if_any_not_reached {}}}", rule_name,
		    sect_name) < 0)
			goto failed;
	}
	if (all_reached) {
		cmds = &cmds_rule->cmds_all_reached;
		if (cmds->has_cmd && run_cmds(rule, (struct wpid *)NULL, cmds,
		    "rule %s { %s { if_all_reached {}}}", rule_name,
		    sect_name) < 0)
			goto failed;
	} else {
		cmds = &cmds_rule->cmds_all_not_reached;
		if (cmds->has_cmd && run_cmds(rule, (struct wpid *)NULL, cmds,
		    "rule %s { %s { if_all_not_reached {}}}", rule_name,
		    sect_name) < 0)
			goto failed;
	}
	STAILQ_FOREACH(limit, &rule->limits, link)
		if (run_cmds_limit(rule, limit, x) < 0)
			goto failed;
#endif /* WITH_LIMITS */

#ifdef WITH_THRESHOLDS
	STAILQ_FOREACH(threshold, &rule->thresholds, link)
		if (run_cmds_threshold(rule, threshold, x) < 0)
			goto failed;
#endif

	cmds_rule_free(cmds_rule);
	return (0);

failed:
	logbt("run_cmds_rule");
	return (-1);
}

/*
 * Run startup or shutdown commands for everything.
 */
int
run_all_cmds(unsigned int x)
{
	struct rule *rule;

	logmsgx(IPA_LOG_INFO, "running %s commands...", rc_sect_name[x]);

	if (cmds_global[x].has_cmd) {
		struct rule rule_arg;

		rule_arg.debug_exec = global_debug_exec;
		if (run_cmds(&rule_arg, (struct wpid *)NULL, &cmds_global[x],
		    "%s {}", rc_sect_name[x]) < 0)
			goto failed;
		cmds_free(&cmds_global[x]);
	}

	TAILQ_FOREACH(rule, &rules_list, list)
		if (run_cmds_rule(rule, x) < 0)
			goto failed;

	return (0);

failed:
	logbt("run_all_cmds");
	logmsgx(IPA_LOG_ERR, "stop running commands");
	return (-1);
}

/*
 * Release memory used by commands list.
 */
void
cmds_free(struct cmds *cmds)
{
	struct cmd *cmd, *cmd_next;
	struct cmd_list *cmd_list;
	unsigned int mask;

#ifdef WITH_RULES
	if (!STAILQ_EMPTY(&cmds->ictl_list))
		free_ictl_list(&cmds->ictl_list);
#endif
	cmd_list = &cmds->cmd_list;
	STAILQ_FOREACH_SAFE(cmd, cmd_list, link, cmd_next) {
		mask = cmd->free_mask;
		if (mask & CMD_FREE_STR)
			mem_free(cmd->str, m_cmd);
		if (mask & CMD_FREE_USER)
			mem_free(cmd->user, m_cmd);
		mzone_free(cmd_mzone, cmd);
	}
	STAILQ_INIT(cmd_list);
}

/*
 * Set sync_exec to -1, this means that it is undefined and
 * initialize heads of actual lists of commands.
 */
void
cmds_init(struct cmds *cmds)
{
	cmds->sect_set = cmds->has_cmd = 0;
	cmds->sync = -1;
#ifdef WITH_RULES
	STAILQ_INIT(&cmds->ictl_list);
#endif
	STAILQ_INIT(&cmds->cmd_list);
}

/*
 * Initialize lists of commands in struct cmds_rule{}.
 */
void
cmds_rule_init(struct cmds_rule *cmds_rule)
{
	cmds_init(&cmds_rule->cmds);
#ifdef WITH_LIMITS
	cmds_init(&cmds_rule->cmds_all_reached);
	cmds_init(&cmds_rule->cmds_any_reached);
	cmds_init(&cmds_rule->cmds_all_not_reached);
	cmds_init(&cmds_rule->cmds_any_not_reached);
#endif
}

void
cmds_rule_set_sync(struct cmds_rule *cmds_rule)
{
#ifdef WITH_LIMITS
	signed char sync_exec;
#endif

	if (cmds_rule->cmds.sync < 0)
		cmds_rule->cmds.sync = 1;
#ifdef WITH_LIMITS
	sync_exec = cmds_rule->cmds.sync;
	if (cmds_rule->cmds_all_reached.sync < 0)
		cmds_rule->cmds_all_reached.sync = sync_exec;
	if (cmds_rule->cmds_any_reached.sync < 0)
		cmds_rule->cmds_any_reached.sync = sync_exec;
	if (cmds_rule->cmds_all_not_reached.sync < 0)
		cmds_rule->cmds_all_not_reached.sync = sync_exec;
	if (cmds_rule->cmds_any_not_reached.sync < 0)
		cmds_rule->cmds_any_not_reached.sync = sync_exec;
#endif
}

/*
 * Release memory used by struct cmds_rule{}.
 */
void
cmds_rule_free(struct cmds_rule *cmds_rule)
{
	cmds_free(&cmds_rule->cmds);
#ifdef WITH_LIMITS
	cmds_free(&cmds_rule->cmds_any_reached);
	cmds_free(&cmds_rule->cmds_all_reached);
	cmds_free(&cmds_rule->cmds_any_not_reached);
	cmds_free(&cmds_rule->cmds_all_not_reached);
#endif
}

#ifdef WITH_LIMITS
/*
 * Initialize lists of commands in struct cmds_limit{}.
 */
void
cmds_limit_init(struct cmds_limit *cmds_limit)
{
	cmds_init(&cmds_limit->cmds);
	cmds_init(&cmds_limit->cmds_reached);
	cmds_init(&cmds_limit->cmds_not_reached);
}

void
cmds_limit_set_sync(struct cmds_limit *cmds_limit)
{
	signed char sync_exec;

	if (cmds_limit->cmds.sync < 0)
		cmds_limit->cmds.sync = 1;
	sync_exec = cmds_limit->cmds.sync;
	if (cmds_limit->cmds_reached.sync < 0)
		cmds_limit->cmds_reached.sync = sync_exec;
	if (cmds_limit->cmds_not_reached.sync < 0)
		cmds_limit->cmds_not_reached.sync = sync_exec;
}

/*
 * Release memory used by struct cmds_limit{}.
 */
void
cmds_limit_free(struct cmds_limit *cmds_limit)
{
	cmds_free(&cmds_limit->cmds);
	cmds_free(&cmds_limit->cmds_reached);
	cmds_free(&cmds_limit->cmds_not_reached);
}
#endif /* WITH_LIMITS */

static void
cmd_substitute(char *dst, const char *src, const char *rule_name)
{
	const char *ptr;

	/*
	 * Command string substitution algorithm:
	 *   %% -> %
	 *   %rule% -> rule_name
	 */
	for (; *src != '\0'; ++src)
		if (*src == '%') {
			if (*++src == '%') {
				/* %% */
				*dst++ = '%';
			} else {
				/* %rule% */
				for (ptr = rule_name; *ptr != '\0';)
					*dst++ = *ptr++;
				src += sizeof("rule") - 1;
			}
		} else
			*dst++ = *src;
	*dst = '\0';
}

/*
 * Copy command list substituting "%..%" substrings.
 */
int
cmds_copy(const struct rule *rule, struct cmds *cmds_dst,
    const struct cmds *cmds_src)
{
	const struct cmd_list *cmd_list_src;
	const struct cmd *cmd_src;
	const char *rule_name;
	struct cmd *cmd_dst;
	struct cmd_list *cmd_list_dst;

	if (!cmds_src->has_cmd)
		return (0);

	rule_name = rule->name;

	cmds_dst->sect_set = cmds_dst->has_cmd = 1;
	cmds_dst->sync = cmds_src->sync;

	cmd_list_dst = &cmds_dst->cmd_list;
	cmd_list_src = &cmds_src->cmd_list;

	/* Copy struct cmd{} from cmd_list_src to cmd_list_dst one by one. */
	STAILQ_FOREACH(cmd_src, cmd_list_src, link) {
		cmd_dst = mzone_alloc(cmd_mzone);
		if (cmd_dst == NULL) {
			xlogmsgx(IPA_LOG_ERR, "cmds_copy: mzone_alloc failed");
			return (-1);
		}
		STAILQ_INSERT_TAIL(cmd_list_dst, cmd_dst, link);
		if (cmd_src->subst_per_cent == 0 && cmd_src->subst_rule == 0) {
			cmd_dst->str = cmd_src->str;
			cmd_dst->free_mask = 0;
		} else {
			cmd_dst->str = mem_malloc(cmd_src->str_size -
			    cmd_src->subst_per_cent +
			    (strlen(rule_name) - 6) * cmd_src->subst_rule,
			    m_cmd);
			if (cmd_dst->str == NULL) {
				xlogmsgx(IPA_LOG_ERR, "cmds_copy: "
				    "mem_malloc failed");
				return (-1);
			}
			cmd_substitute(cmd_dst->str, cmd_src->str, rule_name);
			cmd_dst->free_mask = CMD_FREE_STR;
		}
		/* Do not set CMD_FREE_USER, since we share user pointer. */
		cmd_dst->user = cmd_src->user;
	}
	return (0);
}

/*
 * Copy rule's command list substituting "%...%" substrings.
 */
int
cmds_rule_copy(const struct rule *rule, struct cmds_rule *cmds_rule_dst,
    const struct cmds_rule *cmds_rule_src)
{
	int rv;

	rv = cmds_copy(rule, &cmds_rule_dst->cmds, &cmds_rule_src->cmds);
#ifdef WITH_LIMITS
	rv +=
	    cmds_copy(rule, &cmds_rule_dst->cmds_all_reached,
		&cmds_rule_src->cmds_all_reached) +
	    cmds_copy(rule, &cmds_rule_dst->cmds_all_not_reached,
		&cmds_rule_src->cmds_all_not_reached) +
	    cmds_copy(rule, &cmds_rule_dst->cmds_any_reached,
		&cmds_rule_src->cmds_any_reached) +
	    cmds_copy(rule, &cmds_rule_dst->cmds_any_not_reached,
		&cmds_rule_src->cmds_any_not_reached);
#endif
	if (rv < 0) {
		xlogmsgx(IPA_LOG_ERR, "rule %s: cmds_rule_copy: "
		    "cmds_copy failed", rule->name);
		return (-1);
	}
	return (0);
}

#ifdef WITH_LIMITS
/*
 * Copy limit's command list substituting "%...%" substrings.
 */
int
cmds_limit_copy(const struct rule *rule, struct cmds_limit *cmds_limit_dst,
    const struct cmds_limit *cmds_limit_src)
{
	int rv;

	rv =
	    cmds_copy(rule, &cmds_limit_dst->cmds,
		&cmds_limit_src->cmds) +
	    cmds_copy(rule, &cmds_limit_dst->cmds_reached,
		&cmds_limit_src->cmds_reached) +
	    cmds_copy(rule, &cmds_limit_dst->cmds_not_reached,
		&cmds_limit_src->cmds_not_reached);
	if (rv < 0) {
		xlogmsgx(IPA_LOG_ERR, "rule %s: cmds_limit_copy: "
		    "cmds_copy failed", rule->name);
		return (-1);
	}
	return (0);
}
#endif /* WITH_LIMITS */
