/* Code to read Garmin GPS 25/35LP binary output
 * 
 * For license terms, see the file COPYING in this directory
 * 
 * Last update: 11 Dec 1999 by Joe Desbonnet, joe@wombat.ie
 *
 * I'd like to thank Sam Storm van Leeuwen <samsvl@nlr.nl> for providing me 
 * with some sample Garmin binary data in the days before I had a  garmin 
 * unit to play with.
 * 
 * Usage: ./garminbin < garbin.dat
 * Options:
 * -b 
 * Echo bell whenever a set of receiver records with more than 4 PRNs
 * detected. I added this feature when gathering data on a laptop which was
 * being carried around in its carry case. The beeps reassured me that I 
 * was getting good data.
 * 
 * -f<x>
 * Determine output format. Currently the GPS25PM.EXE style format
 * is hardwired -- working on a RINEX output also.
 *
 * -o 
 * Output device. "-" means stdout. Defaults to stdout.
 * 
 * -p 
 * Input port. "-" means stdin. Defaults to stdin.
 * FIXME: Would -i be a better choice for input??
 * 
 * Currently tested only with Linux (RedHat 6.0 on Intel box).
 * 
 * BUGS:
 *  Endian issues... 
 *  Currently assumes that MSB is stored at the higher address (Intel style)
 *  Checksums not working. Checksums are defined in the manual as 
 *  "the addition of bytes between the delimiters should equal 0"
 * 
 * UPDATES:
 * 
 * 11 Dec 1999
 * Integrate with libgps to process pseudorange/phase data 
 * on the fly.
 * 
 * 29 Nov 1999
 * Added -i (input device), -o (output device).
 * Added signal handling. 
 * 
 * 
 * 28 Nov 1999
 * Now understands ephemeris records. Outputs in the same format as the
 * GPS25PM.EXE DOS program. Other formats (eg RINEX) available soon.
 * 
 * 19 Nov 1999
 * Some code tidy up. Start adding RINEX output format option.
 * 
 * 4 Nov 1998
 * Stuff I can't remember
 * Joe Desbonnet, joe@wombat.ie
 * 
 * 10 July 1999
 * Minor mods
 */


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <signal.h>

#include "libgps.h"
#include "garmin.h"


/* Name & Version of the software */
#define SWVERSION "garminbin V0.3"


/* Output formats */
#define FORMAT_GPS25PM 1
#define FORMAT_RINEX 2
#define FORMAT_VT100 3 /* output suitable for vt100 terminals */
#define FORMAT_CSV 4

/* Define functions */
void usage ();
void display_record (garmin_rcv_type *);
void output_gps25pm_rcv_record (garmin_ch_type ch, FILE *fp);
void output_gps25pm_rcv_recordset (garmin_rcv_type rcv, FILE *fp);
void output_rinex_rcv_recordset (garmin_rcv_type rcv, FILE *fp);
void output_csv (FILE *fp);

void read_rcv_record (unsigned char *, FILE *fp);
void read_pvt_record (unsigned char *, FILE *fp);
void read_eph_record (unsigned char *, FILE *fp);
int verify_checksum (void *memaddr, int len, int expected_checksum);
int retrieve_record (unsigned char *buf, int length, FILE *fp);
void calc_posn (vec3 lla, garmin_rcv_type rcv_rec);
void output_to_screen ();
void skymap (FILE *);

/* Define signal handlers */
void sig_hup_handler (int);
void sig_pipe_handler (int);
void sig_term_handler (int);

/* Globals */

int bell_flag;
int output_format;
unsigned char chksum=0;
int get_ephemeris_flag=0;

/* Array of latest ephemeris records -- very experimental */
eph_type current_eph[32];
boolean  got_eph[32];

/* Current SV range/phase records */
garmin_rcv_type rcv_rec;

/* Current PVT record */
garmin_pvt_type pvt_rec;

/* current position estimate */
vec3     current_Xr;
vec3     mean_Xr;
int      mean_Xr_count;

/* current calculated clock error (in range meters;  *NOT* seconds) */
double   cclke_m;

/* current position as reported by the Garmin unit */
vec3     current_reported_lla;

/* Reference station position. -- experimental stuff -- ignore */
vec3     ref_Xr;


double   current_tow;

boolean screen_mode;

/* experimental stuff -- ignore */
int  skymap_sum[8][60];
int  skymap_count[8][60];

#define MASK_ANGLE 20

/* Main Program */
int main (int argc, char **argv) 
{
    char *input_device=NULL, *output_device=NULL;
    unsigned char buf[512];
    int i;
    boolean end_of_dataset=false;
    vec3 lla;
    FILE *input_fp=NULL, *output_fp=NULL;
    
    int rec_len,rec_type,checksum,exit_loop=0;

    /* Setup signal handlers */
    signal (SIGHUP, sig_hup_handler);
    signal (SIGPIPE, sig_pipe_handler);
    signal (SIGTERM, sig_term_handler);
    signal (SIGINT, sig_term_handler);

    /* Initialize global variables */
    bzero (skymap_sum, sizeof(int)*8*60);
    bzero (skymap_count, sizeof(int)*8*60);
    bzero (current_eph, sizeof (eph_type) * 32);
    bzero (got_eph, sizeof (boolean) * 32);
    bzero (&pvt_rec, sizeof (garmin_pvt_type));
    bzero (&rcv_rec, sizeof (garmin_rcv_type));
    current_tow=0;
    screen_mode=false;
    cclke_m=0;
    lla[0] = 53 * DEG_TO_RAD;
    lla[1] = -6 * DEG_TO_RAD;
    lla[2] = 70;
    
    /* This is where my units sit when I do zero basline tests. +/- 20m or so */
    ref_Xr[0]= 3795025;
    ref_Xr[1]= -416238;
    ref_Xr[2]= 5092264;

    gps_lla2xyz(lla,current_Xr);
    bzero (mean_Xr, sizeof(vec3));
    mean_Xr_count=0;


    /* Parse switches. Will move to getopt some day. */
    bell_flag=0;
    output_format=FORMAT_GPS25PM;
    for (i=1; i<argc; i++) 
    {
	if (*argv[i] == '-') 
	{
	    switch (*(argv[i]+1)) {
	     case 'b':
		bell_flag = 1;
		break;
	     case 'f':
		switch (*(argv[i]+2)) {
		 case 'c':
		    output_format=FORMAT_CSV;
		    break;
		 case 'r':
		    output_format=FORMAT_RINEX;
		    break;
		 case 't':
		    output_format=FORMAT_GPS25PM;
		    break;
		 case 'v':
		    output_format=FORMAT_VT100;
		    break;
		}
		break;
		
	     case 'h':
		usage();
		exit(0);

	     case 'o':
		i++;
		if (i == argc)
		{
		    break;
		}
		output_device = argv[i];
		break;
		
	     case 'i':
		i++;
		if (i == argc)
		{
		    break;
		}
		input_device = argv[i];
		break;
		
	     case 's':
		screen_mode=true;
		break;
	    }
	}
    }
    
    /* If input_device was unspecified or was "-" then use stdin */
    /* FIXME: if input device is a tty then we should set it up properly */
    if ( input_device==NULL || strcmp(input_device,"-")==0 )
    {
	input_fp = stdin;
	if (DEBUG)
	{
	    fprintf (stderr,"Input device: stdin\n");
	}
    } 
    else 
    {
	input_fp = fopen (input_device,"rw");
	if (input_fp==NULL)
	{
	    fprintf (stderr,"Error opening %s for input\n",input_device);
	    exit(1);
	}
	if (DEBUG)
	{
	    fprintf (stderr,"Input device: %s\n",input_device);
	}
    }
    
    /* If output_device was unspecified or was "-" then use stdout */
    if ( output_device==NULL || strcmp(output_device,"-")==0 )
    {
	output_fp = stdout;
	if (DEBUG)
	{
	    fprintf (stderr,"Output device: stdout\n");
	}
    } 
    else 
    {
	output_fp = fopen (output_device,"rw");
	if (output_fp==NULL)
	{
	    fprintf (stderr,"Error opening %s for output\n",output_device);
	    exit(1);
	}
	if (DEBUG)
	{
	    fprintf (stderr,"Input device: %s\n",output_device);
	}
    }
    
    
    /* Skip to first valid record */
    garmin_skip_to_next_record(input_fp);

    /* Main loop */
    do 
    {
	if (DEBUG) 
	{
	    if (feof(input_fp)) 
	    {
		fprintf (stderr,"eof on input_fp\n");
	    }
	    
	    fprintf (stderr,"fpos = %ld\n",ftell(input_fp));
	}

	if (feof(input_fp)) 
	{
	    break;
	}
	
	/* This flag can be set by HUP or ALARM */
	if (get_ephemeris_flag == 1)
	{
	    fwrite (GPS35_GETEPH_CMD, 1, strlen(GPS35_GETEPH_CMD), input_fp);
	    fflush (input_fp);
	    get_ephemeris_flag=0;
	}
	
	/* Read record type and length */

	rec_type = garmin_get_data_byte(input_fp);
	rec_len = garmin_get_data_byte(input_fp);
	if (DEBUG) 
	{
	    fprintf (stderr,"rec_type = 0x%02x, rec_len = %d\n",rec_type,rec_len);
	}
       

	retrieve_record(buf,rec_len,input_fp);

       
	if (DEBUG) 
	{
	    fprintf (stderr,"checksum = %d (0x%02x)\n",checksum,checksum);
	}
	switch (rec_type) {
	 case RCV_REC:
	    if (DEBUG) fprintf (stderr,"Receiver record found.\n");
	    
	    if (end_of_dataset)
	    {
		end_of_dataset=false;
	    }
	    read_rcv_record(buf, output_fp);
	    break;
	    
	 case POS_REC:
	    if (DEBUG) fprintf (stderr,"Position record found.\n");
	    read_pvt_record(buf, output_fp);
	    end_of_dataset=true; /* this is the last record of a 1 sec set */
	    
	    if (output_format == FORMAT_VT100)
	    {
		output_to_screen ();
	    }
	    if (output_format == FORMAT_CSV) 
	    {
		output_csv (output_fp);
	    }
	    
	    end_of_dataset=false;
	    break;
	    
	 case EPH_REC:
	    if (DEBUG) fprintf (stderr,"Ephemeris record found. len=%d\n",rec_len);
	    read_eph_record(buf, output_fp);
	    break;
	    
	 default:
	    printf ("Unrecognized record type 0x%02x of length %d at %ld\n",
		    rec_type,
		    rec_len,
		    ftell(input_fp));
	    
	    for (i=0; i<rec_len; i++) 
	    {
	       printf (" %02X ", buf[i]);
	    }
	    printf ("\n");
	    
	    garmin_skip_to_next_record(input_fp);
	    
	    /* exit(2); */
	    break;
	}
	
    } while (exit_loop==0);

    return 0; /* FIXME */
}

int retrieve_record (unsigned char *buf, int length, FILE *fp) 
{
   unsigned char checksum;
    int i;
    
    /* reset checksum counter used in get_data_byte (this is a global!) */
    chksum=0;
   
    for (i=0; i<length; i++) 
    {
      buf[i] = garmin_get_data_byte (fp);
    }
    
    checksum = garmin_get_data_byte (fp);
   
    return 0; /* FIXME */
}



void read_eph_record (unsigned char *buf, FILE *fd)
{
    
    /*
     * FIXME:
     * It would be nice to just do "eph=buf" but I don't know how to
     * guarantee that the C compiler will back the eph structure 
     * the way I'd expect
     */
    
    eph_type *eph = (eph_type *)malloc(sizeof(eph_type));
    
    eph->prn = buf[0];
    eph->wn = (short) *(buf + 1);
    eph->toc = * ( (float *) (buf + 3) );
    eph->toe = * ( (float *) (buf + 7) );
    eph->af0 = * ( (float *) (buf +11) );
    eph->af1 = * ( (float *) (buf +15) );
    eph->af2 = * ( (float *) (buf +19) );
    eph->ura = * ( (float *) (buf +23) );

    eph->e   = * ( (double *) (buf +27) );
    eph->sqrta= * ( (double *) (buf +35) );
    eph->dn   = * ( (double *) (buf +43) );
    eph->m0   = * ( (double *) (buf +51) );
    eph->w    = * ( (double *) (buf +59) );
    eph->omg0 = * ( (double *) (buf +67) );
    eph->i0   = * ( (double *) (buf +75) );
    
    eph->odot = * ( (float *) (buf +83) );
    eph->idot = * ( (float *) (buf +87) );
    eph->cus = * ( (float *) (buf +91) );
    eph->cuc = * ( (float *) (buf +95) );
    eph->cis = * ( (float *) (buf +99) );
    eph->cic = * ( (float *) (buf +103) );
    eph->crs = * ( (float *) (buf +107) );
    eph->crc = * ( (float *) (buf +111) );
    
    eph->iod = buf[115];
    
    if (eph->prn < 0 || eph->prn > 31)
    {
	return;
    }

    got_eph[(int)eph->prn]=true;
    
    /* Update current_eph records */
    if (eph->iod == current_eph[(int)eph->prn].iod)
    {
	return;
    }
    else
    {
	fprintf (stderr,"iod[%d]=%d\n",
		eph->prn + 1, 
		(int)eph->iod);
	memcpy (&current_eph[(int)eph->prn], eph, sizeof (eph_type));
    } 
    
    
    fprintf (fd,"\n");
    fprintf (fd,"**** Week %d.  Ephemeris for PRN-%02d **********\n",
	    eph->wn, eph->prn+1);
    
    fprintf (fd,"Ref Time of Clk Parms  (s): %e\n", eph->toc);
    fprintf (fd,"Ref Time of Eph Parms  (s): %e\n", eph->toe);
    fprintf (fd,"Clk Cor, Group Dly     (s): %e\n", eph->af0);
    fprintf (fd,"Clk Correction af1   (s/s): %e\n", eph->af1);
    fprintf (fd,"Clk Correction af2 (s/s/s): %e\n", eph->af2);
    fprintf (fd,"User Range Accuracy    (m): %e\n", eph->ura);
    fprintf (fd,"Eccentricity           (-): %e\n", eph->e);
    fprintf (fd,"SQRT(A)           (m**1/2): %e\n", eph->sqrta);
    fprintf (fd,"Mean Motion Cor      (r/s): %e\n", eph->dn);
    fprintf (fd,"Mean Anomaly           (r): %e\n", eph->m0);
    fprintf (fd,"Argument of Perigee    (r): %f\n", eph->w);
    fprintf (fd,"Right Ascension        (r): %e\n", eph->omg0);
    fprintf (fd,"Inclination Angle      (r): %e\n", eph->i0);
    fprintf (fd,"Rate of Right Asc    (r/s): %e\n", eph->odot);
    fprintf (fd,"Rate of Inc Angle    (r/s): %e\n", eph->idot);
    fprintf (fd,"Lat Cor, Sine          (r): %e\n", eph->cus);
    fprintf (fd,"Lat Cor, Cosine        (r): %e\n", eph->cuc);
    fprintf (fd,"Inc Cor, Sine          (r): %e\n", eph->cis);
    fprintf (fd,"Inc Cor, Cosine        (r): %e\n", eph->cic);
    fprintf (fd,"Radius Cor, Sine       (m): %e\n", eph->crs);
    fprintf (fd,"Radius Cor, Cosine     (m): %e\n", eph->crc);
    fprintf (fd,"Issue of Data             : %d\n", eph->iod);
}

/* Read receiver record from a memory buffer to an
 * instance of the garmin_rcv_type structure.
 */
void read_rcv_record (unsigned char *buf, FILE *fp)
{
    int i,offset,nvalid;
    
    rcv_rec.tow = (double) * ( (double *) buf );
    rcv_rec.wn = (short)  * ( (short *) (buf+8) );

    nvalid=0;

    /* Loop for each of the 12 channels */
    for (i=0; i<12; i++) 
    {
	offset = 10+i*18;
	rcv_rec.sv[i].cycles = *((int *)&buf[offset]);
	rcv_rec.sv[i].pr = *((double *)&buf[offset + 4]);
	rcv_rec.sv[i].phase = *((short *)&buf[offset + 12]);
	rcv_rec.sv[i].slip = buf[offset+14];
	rcv_rec.sv[i].snr  = buf[offset+15];
	rcv_rec.sv[i].prn = buf[offset+16];
	rcv_rec.sv[i].valid = buf[offset+17];
	
	if (rcv_rec.sv[i].valid == 1) 
	{
	    nvalid++;
	}
    }
    
    switch (output_format)
    {
     case FORMAT_GPS25PM:
	output_gps25pm_rcv_recordset (rcv_rec, fp);
	break;
     case FORMAT_RINEX:
	output_rinex_rcv_recordset (rcv_rec, fp);
	break;
    }

    if (bell_flag) {
	/* beep if insufficient SVs */
	if (nvalid < 4) {
	    fprintf (stderr,"\007");
	}
	/* reassurance bell */
	if ( ((int)rcv_rec.tow)%5 == 0) {
	    fprintf (stderr,"\007");
	}
	fprintf (stderr,"nvalid = %d\n", nvalid);
    }
    
}

/* Read PVT (Position Velocity Time) record into an instance of the
 * garmin_pvt_type structure
 */
void read_pvt_record (unsigned char *buf, FILE *fd)
{

   pvt_rec.alt = *((float *)&buf[0]);
   pvt_rec.epe = *((float *)&buf[4]);
   pvt_rec.eph = *((float *)&buf[8]);
   pvt_rec.epv = *((float *)&buf[12]);
   pvt_rec.fix = *((short *)&buf[16]);
   pvt_rec.tow = *((double *)&buf[18]);
   pvt_rec.lon = *((double *)&buf[26]);
   pvt_rec.lat = *((double *)&buf[34]);
   pvt_rec.lon_vel = *((float *)&buf[42]);
   pvt_rec.lat_vel = *((float *)&buf[46]);
   pvt_rec.alt_vel = *((float *)&buf[50]);
   
    current_reported_lla[0]=pvt_rec.lon;  /* FIXME!! Something wrong here */
    current_reported_lla[1]=pvt_rec.lat;
    current_reported_lla[2]=pvt_rec.alt;
    
    current_tow = pvt_rec.tow;
    
    if  (output_format == FORMAT_GPS25PM) 
    {
	/* Not the full PVT record -- will do for the moment however */
	fprintf (fd,"PVT %f %f %f %f\n",
		 pvt_rec.tow,
		 pvt_rec.lat * 180.0 / M_PI,
		 pvt_rec.lon * 180.0 / M_PI,
		 pvt_rec.alt);
    }
    
}


void output_gps25pm_rcv_recordset (garmin_rcv_type rcv_rec, FILE *fp)
{
    int i;
    
    fprintf (fp,"TIM %f %hd\n",rcv_rec.tow,rcv_rec.wn);

    for (i=0; i<12; i++)
    {
	if (rcv_rec.sv[i].valid == 1)
	{
	    output_gps25pm_rcv_record (rcv_rec.sv[i],fp);
	}
    }
    fflush (fp);
}

void output_gps25pm_rcv_record (garmin_ch_type ch, FILE *fp)
{
      fprintf (fp,"RCV %2d %2d %c %3.1f %15.2f %12u \n",
	      ch.prn + 1,
	      ch.snr,
	      ch.slip==1 ? 'C':'T',
	      ( (double)ch.phase  * (360.0/2048.0) ),
	      ch.pr,
	      ch.cycles);
}


void output_rinex_rcv_recordset (garmin_rcv_type rcv_rec, FILE *fd)
{
    /* See rinex.html in doc directory for information on the RINEX V2 standard */

    int i;
    /* FIXME temporary hardcode this stuff. Should get these from optional 
     * getopt command line args eg --rinex-observer="Bill Smith" 
     */
    char current_date[]="01-JAN-1970 00:00";
    char rinex_run_by[]="run_by";
    char rinex_marker_name[]="marker_name";
    char rinex_marker_number[]="marker_number";
    
    char rinex_observer[]="observer";
    char rinex_agency[]="agency";
    
    char rinex_receiver_number[]="receiver_number";
    char rinex_receiver_type[]="receiver_number";
    char rinex_receiver_version[]="receiver_number";

    double rinex_antenna_height=0.0;
    double rinex_antenna_e=0.0;
    double rinex_antenna_n=0.0;
    
    double rinex_antenna_position_x=0.0;
    double rinex_antenna_position_y=0.0;
    double rinex_antenna_position_z=0.0;

    int rinex_epoch_year;
    int rinex_epoch_month;
    int rinex_epoch_mday;
    int rinex_epoch_hour;
    int rinex_epoch_min;
    double rinex_epoch_sec;
    int rinex_epoch_flag;
    int rinex_epoch_nsat;
    
    fprintf (fd,"      2              OBSERVATION DATA    G (GPS)             RINEX VERSION / TYPE\n");
    
    /* GARMINBIN V0.3      XX                  22-APR-93 12:43     PGM / RUN BY / DATE */
    fprintf (fd," %20s%20s%20sPGM / RUN BY / DATE\n",SWVERSION, rinex_run_by, current_date);

    if (rinex_marker_name != NULL)
    {
	fprintf (fd," %60sMARKER NAME\n",rinex_marker_name);
    }
    
    if (rinex_marker_number != NULL) 
    {
	fprintf (fd," %20sMARKER NUMBER\n", rinex_marker_number);
    }
    
    /* BILL SMITH          ABC INSTITUTE                           OBSERVER / AGENCY */
    fprintf (fd," %20s%40sOBSERVER / AGENCY\n",rinex_observer, rinex_agency);
    
    /* X1234A123           XX                 ZZZ                  REC # / TYPE / VERS */
    fprintf (fd," %20s%20s%20sREC # / TYPE / VERS\n",
	    rinex_receiver_number,
	    rinex_receiver_type,
	    rinex_receiver_version);
    
    /* 234                 YY                                      ANT # / TYPE */
    fprintf (fd," 234                 YY                                      ANT # / TYPE\n");
    
    /*   4375274.       587466.      4589095.                      APPROX POSITION XYZ */
    fprintf (fd," %14.4f%14.4f%14.4fAPPROX POSITION XYZ\n",
	    rinex_antenna_position_x,
	    rinex_antenna_position_y,
	    rinex_antenna_position_z
	    );
    
    /*           .9030         .0000         .0000                  ANTENNA: DELTA H/E/N */
    fprintf (fd," %14.4f%14.4f%14.4fANTENNA: DELTA H/E/N\n",
	    rinex_antenna_height,
	    rinex_antenna_e, /* Eccentricities of antenna center relative to marker to the east (meters) */
	    rinex_antenna_n  /* Eccentricities of antenna center relative to marker to the north (meters) */
	   );
    
    fprintf (fd,"      1     1                                                WAVELENGTH FACT L1/2\n");
    fprintf (fd,"      1     1                                                WAVELENGTH FACT L1/2\n");
    
    fprintf (fd,"      1     2     6   G14   G15   G16   G17   G18   G19      WAVELENGTH FACT L1/2\n");
    fprintf (fd,"      4    P1    L1    L2    P2                              # / TYPES OF OBSEVR\n");
    fprintf (fd,"     18                                                      INTERVAL\n");
    fprintf (fd,"   1990     3    24    13    10   36.000000                  TIME OF FIRST OBS\n");
    fprintf (fd,"                                                             END OF HEADER\n");

    /* 90  3 24 13 10 36.0000000  0  3G12G 9G 6                            -.123456789 */
    fprintf (fd," %3d%3d%3d%3d%3d%14.7f%3d%3d",
	    rinex_epoch_year,
	    rinex_epoch_month,
	    rinex_epoch_mday,
	    rinex_epoch_hour,
	    rinex_epoch_min,
	    rinex_epoch_sec,
	    rinex_epoch_flag,
	    rinex_epoch_nsat);
	    
    fprintf (fd,"   23629347.915            .300 8         -.353    23629364.158\n");
	    
    for (i=0; i<12; i++)
    {
	if (rcv_rec.sv[i].valid == 1)
	{
	    
	}
    }
}


/**
 * Output some info in a format that I can use with GNU Plot. Its supposed
 * to be "Output to Comma Separated Value format" but I prefer spaces.
 * 
 * @param out_fp The file descriptor to send output to
 */

void output_csv (FILE *out_fp)
{
    vec3 Xs,errorvec, Xrg, current_lla, dXrXs;
    double Trel,az,el,cclke_t,errormag;
    boolean status;
    int i,prn,nvalid;
    
    

    /* How many SV records? */
    nvalid=0;
    for (i=0; i<12; i++) 
    {
	if (!rcv_rec.sv[i].valid) 
	{
	    continue;
	}
	
	prn = rcv_rec.sv[i].prn;
	
	/* Where is the satellite (SV) in space and where in the sky? */
	gps_satpos (current_eph[prn],current_tow,&Trel,Xs);
	gps_calcAzEl (Xs, current_Xr, &az, &el, &status);
	
	if (el < MASK_ANGLE * DEG_TO_RAD) 
	{
	    rcv_rec.sv[i].valid = false;
	    continue;
	}
	nvalid++;
    }

    if (nvalid >= 4)
    {
	calc_posn (current_Xr, rcv_rec);
    }
    
    gps_xyz2lla (current_Xr, current_lla);
    gps_lla2xyz (current_reported_lla, Xrg);
    
    vec3sub (errorvec, current_Xr, Xrg);
    errormag = vec3mag (errorvec);
    
    cclke_t = cclke_m / const_c;

    fprintf (out_fp, "%f  %9.6f %9.6f  %9.6f %9.6f  %9.6f %9.6f 999666999\n",
	     current_tow,
	     current_reported_lla[0] * RAD_TO_DEG,
	     current_lla[0] * RAD_TO_DEG,
	     current_reported_lla[1] * RAD_TO_DEG,
	     current_lla[1] * RAD_TO_DEG,
	     current_reported_lla[2],
	     current_lla[2]
	     );
}

void output_to_screen ()
{
    FILE *out_fp=stdout;
    vec3 Xs,errorvec, Xrg, current_lla, dXrXs;
    double Trel,az,el,errormag,real_range, range_error;
    double gclke_m, gclke_s, cclke_t,gre,cre;
    boolean status;
    int i,prn,nvalid;
    
    
    /* How many SV records? */
    nvalid=0;
    for (i=0; i<12; i++) 
    {
	if (rcv_rec.sv[i].valid) 
	{
	    nvalid++;
	}
    }

    if (nvalid >= 4)
    {
	/* sanity check on current_Xr (FIXME why are we finding this necessary */
	if (vec3mag(current_Xr) > 100000000) 
	{
	    current_Xr[0]=40000000;
	    current_Xr[1]=40000000;
	    current_Xr[2]=40000000;
	}
	calc_posn (current_Xr, rcv_rec);
	if (vec3mag(current_Xr) > 100000000)
	{
	  fprintf (stderr,"Crazy result!!!!!!!\n");
	    exit(1);
	}
	mean_Xr[0] += current_Xr[0];
	mean_Xr[1] += current_Xr[1];
	mean_Xr[2] += current_Xr[2];
	mean_Xr_count++;
    }
    
    gps_xyz2lla (current_Xr, current_lla);
    gps_lla2xyz (current_reported_lla, Xrg);
    
    vec3sub (errorvec, current_Xr, Xrg);
    errormag = vec3mag (errorvec);
    
    cclke_t = cclke_m / const_c;
    
    /* Clear screen and home cursor. Will move to ncurses soon! */
    fprintf (out_fp,"\033[H\033\[2J");

    fprintf (out_fp, "GPST: %d/%15.9f  #Sat:%2d\n",
	     rcv_rec.wn,
	     current_tow,
	     nvalid
	     );
    
    fprintf (out_fp, "Calc LLA: %8.5f %9.5f %5.0f   ErrorMag: %7.2f CClkE: %10.9f\n",
	     current_lla[0] * RAD_TO_DEG,
	     current_lla[1] * RAD_TO_DEG,
	     current_lla[2],
	     errormag,
	     cclke_t);
    fprintf (out_fp, "PVT  LLA: %8.5f %9.5f %5.0f   ErrorVec: (%7.2f %7.2f %7.2f)\n",
	     current_reported_lla[0] * RAD_TO_DEG,
	     current_reported_lla[1] * RAD_TO_DEG,
	     current_reported_lla[2],
	     errorvec[0],
	     errorvec[1],
	     errorvec[2]);
    
    fprintf (out_fp, "Mean XYZ: %7.0f %7.0f %7.0f   GClkE: %10.9f\n",
	     mean_Xr[0]/ (double)mean_Xr_count,
	     mean_Xr[1]/ (double)mean_Xr_count,
	     mean_Xr[2]/ (double)mean_Xr_count,
	     rcv_rec.tow - current_tow
	     );
    
    fprintf (out_fp, "\n");
    
    
    /* Display channel records */
    
    fprintf (out_fp, "PRN  PseudoRange(m) SNR Az(d) El(d)   Pc           Pr           Ecr\n");
    for (i=0; i<12; i++)
    {
	
	prn = rcv_rec.sv[i].prn;

	/* Only continue with this channel record if it contains valid 
	 * data AND we have a valid ephemeris record for this SV */
	if (rcv_rec.sv[i].valid == false  || got_eph[prn] == false) 
	{
	    continue;
	}
	

	/* Where is the satellite (SV) in space and where in the sky? */
	gps_satpos (current_eph[prn],current_tow,&Trel,Xs);
	gps_calcAzEl (Xs, current_Xr, &az, &el, &status);

	/* Displacement between ref position and SV. This sould be very
	 * accurate as the only error is that in ephemeris and our
	 * surveyed position (and the latter is a constant error) */
	vec3sub (dXrXs, Xs, ref_Xr);
	real_range = vec3mag (dXrXs);
	
	/* Clock error (according to Garmin data) */
	gclke_s = rcv_rec.tow - current_tow; /* Clock error in seconds */
	gclke_m = gclke_s * const_c;         /* Clock error in range meters */
	gre = rcv_rec.sv[i].pr - gclke_m;    /* Range error */
	
	/* Clock error (accounding to my calculations) */
	cre = rcv_rec.sv[i].pr + cclke_m -  real_range; /* Calculated Range Error */
	
	range_error = rcv_rec.sv[i].pr - (rcv_rec.tow - current_tow)*const_c - real_range ;

	fprintf (out_fp,"%2d %14.2f   %2d %6.2f  %5.2f %12.2f %12.2f %12.2f\n",
		 prn + 1, 
		 rcv_rec.sv[i].pr, 
		 /* (double) (rcv_rec.sv[i].phase * (360.0/2048)), */
		 rcv_rec.sv[i].snr, 
		 az * RAD_TO_DEG,
		 el * RAD_TO_DEG,
		 /* rcv_rec.sv[i].pr + clock_error, */
		 rcv_rec.sv[i].pr - (rcv_rec.tow - current_tow)*const_c,
		 real_range,
		 range_error
		 );
	
	skymap_sum[(int)(el * 16 / pi)][(int)(az * 30 / pi)] += (int)rcv_rec.sv[i].snr;
	skymap_count[(int)(el * 16 / pi)][(int)(az * 30 / pi)]++;
	
    }
    
    for (i=nvalid; i<12; i++) 
    {
	fprintf (out_fp, "\n");
    }
	     

    skymap(out_fp);

}

	    
void skymap (FILE *out_fp)
{
    int i,j;
    double mean_snr;
    char buf[61];
    
    buf[60]='\n';
    
    for (i=7; i>=0; i--)
    {
	for (j=0; j<60; j++)
	{
	    if (skymap_count[i][j]==0)
	    {
		buf[j]=' ';
		continue;
	    }
	    mean_snr = (double)skymap_sum[i][j] / (double)skymap_count[i][j];
	    if (mean_snr < 30) 
	    {
		buf[j]=' ';
		continue;
	    }
	    if (mean_snr < 35)
	    {
		buf[j]='.';
		continue;
	    }
	    if (mean_snr < 40)
	    {
		buf[j]='x';
		continue;
	    }
	    if (mean_snr < 45)
	    {
		buf[j]='8';
		continue;
	    }
	    buf[j]='@';
	}
	fwrite (buf , 1, 61, out_fp);
    }
}

		
void calc_posn (vec3 lla, garmin_rcv_type  rcv_rec)
{
    int i,prn;
    double pseudorange[32];
    vec8 iono_coeff;
    boolean pseudorange_valid[32];
    
    /* Zero arrays */
    for (i=0; i<32; i++)
    {
	pseudorange[i]=0.0;
	pseudorange_valid[i]=0;
    }
    
    for (i=0; i<12; i++)
    {
	if (rcv_rec.sv[i].valid == false) 
	{
	    continue;
	}
	
	prn = rcv_rec.sv[i].prn;
	
	if (got_eph[prn] == false)
	{
	    continue;
	}
	
	pseudorange[prn] = rcv_rec.sv[i].pr;
	pseudorange_valid[prn] = true;
    }
    
    /* These are not available from the Garmin GPS25/35. Make'em up
     * for the moment. FIXME. 
     */
    iono_coeff[0]= -3.3528e-08;
    iono_coeff[1]=  2.5611e-08;
    iono_coeff[2]= -1.0803e-07;
    iono_coeff[3]=  5.5879e-08;
    iono_coeff[4]= -2.9696e+04;
    iono_coeff[5]=  1.0240e+03;
    iono_coeff[6]= -1.1469e+05;
    iono_coeff[7]= -1.6794e+05;
    
    gps_sps_calc (lla, 
		  current_tow, 
		  pseudorange_valid,
		  current_eph, 
		  iono_coeff, 
		  pseudorange,
		  &cclke_m);
    
}

/* Handle the HUP signal */
void sig_hup_handler (int i)
{
    if (DEBUG)
    {
	fprintf (stderr,"Caught SIGHUP\n");
	fflush (stderr);
    }

    /* reestablish signal handler */
    signal (SIGHUP, sig_hup_handler);
}
void sig_pipe_handler (int i)
{
    if (DEBUG)
    {
	fprintf (stderr,"Caught SIGPIPE.\n");
	fflush (stderr);
    }
    
    exit(0);
}

void sig_term_handler (int i)
{
    fprintf (stderr,"Caught SIGTERM\n");
    fflush (stderr);
    exit(0);
}


void usage ()
{
   fprintf (stderr,"\
Usage: garminbin [-b] [-i <portname>]\n\n\
-b  Echo bell character if more than 4 SVs detected \n\
-s  Screen mode (do simple formatting etc. assumes vt100) \n\
-o  Output device. \"-\" means stdout. Defaults to stdout if not specified.\n\
-i  Input device to expect data from (eg -i /dev/ttyS0)\n\
    \"-\" means stdin. Defaults to stdin if not specified.\n\
\n\
Text output is on stdout\n\n");
}
