/*****************************************************************
 * fbm2pod.c: FBM Release 1.0 25-Feb-90 Michael Mauldin
 *
 * Copyright (C) 1989,1990 by Michael Mauldin.  Permission is granted
 * to use this file in whole or in part for any purpose, educational,
 * recreational or commercial, provided that this copyright notice
 * is retained unchanged.  This software is available to all free of
 * charge by anonymous FTP and in the UUNET archives.
 *
 * fbm2pod.c: Take an 8bit gray image, resize it to a maximum total
 *	      number of pixels, optionally sharpen it with a digital
 *	      Laplacian filter, and halftone it using one of three
 *	      standard algorithms.  Output the result in Diablo
 *	      graphics format.
 *
 * USAGE
 *	% fbm2pod [ -args ]  [ size ] < foo.fbm > foo.pod
 *
 *	size	Choose a width and height as large as possible so that
 *		width is a factor of 8 and width*height <= size (default
 *		is width and height of original 8bit file, ignoring aspect
 *		ratio).
 *
 *	-f	Do Floyd-Steinberg halftoning (the default algorithm)
 *	-bNNN	Do Blue noise halftoning (-b50 or 50% noise is default)
 *	-cNNN	Do Constained average halftoning (-c4 is the default)
 *	-sNNN	Sharpen the image with a given beta (-s2.0 is default)
 *
 * EDITLOG
 *	LastEditDate = Mon Jun 25 00:03:25 1990 - Michael Mauldin
 *	LastFileName = /usr2/mlm/src/misc/fbm/fbm2pod.c
 *
 * HISTORY
 * 25-Jun-90  Michael Mauldin (mlm@cs.cmu.edu) Carnegie Mellon
 *	Package for Release 1.0
 *
 * 07-Mar-89  Michael Mauldin (mlm) at Carnegie Mellon University
 *	Beta release (version 0.9) mlm@cs.cmu.edu
 *
 *  8-Sep-88  Michael Mauldin (mlm) at Carnegie-Mellon University
 *	Created.
 *****************************************************************/

# include <stdio.h>
# include <math.h>
# include "fbm.h"

# define PODASPECT 1.25

# define USAGE \
"Usage: fbm2pod [ -fbc<parm> ] [-s<sharpen> ] [ size ] < 8bit > pod"

#ifndef lint
static char *fbmid =
"$FBM fbm2pod.c <1.0> 25-Jun-90  (C) 1989,1990 by Michael Mauldin, source \
code available free from MLM@CS.CMU.EDU and from UUNET archives$";
#endif

main (argc, argv)
char *argv[];
{ int w, h, ow = -1, oh = -1, size = -1, alg = 'b';
  double beta = -1e9, parm = -1e9;
  char *title;
  FBM input, resized, sharpened, output, *image;

  /* Clear the memory pointers so alloc_fbm won't be confused */
  input.cm     = input.bm     = (unsigned char *) NULL;
  resized.cm   = resized.bm   = (unsigned char *) NULL;
  sharpened.cm = sharpened.bm = (unsigned char *) NULL;
  output.cm    = output.bm    = (unsigned char *) NULL;

  /* Read the image */
  if (read_bitmap (&input, (char *) NULL))
  {
    if (input.hdr.bits != 8 || input.hdr.physbits != 8)
    { fprintf (stderr,
	       "Can't handle images with %d bits and %d physbits per pixel\n",
	       input.hdr.bits, input.hdr.physbits);
      exit (1);
    }

    if (input.hdr.title[0]) title = input.hdr.title;

    /* Get the options */
    while (--argc > 0 && (*++argv)[0] == '-')
    { while (*++(*argv))
      { switch (**argv)
        { case 's':	if (argv[0][1]) { beta = atof (*argv+1); SKIPARG; }
			else		beta = 2.0;
			break;
			
	  case 'f':	alg = 'f'; break;
			
	  case 'b':	alg = 'b';
			if (argv[0][1])	{ parm = atof (*argv+1); SKIPARG; }
			break;
			
	  case 'c':	alg = 'c';
			if (argv[0][1])	{ parm = atof (*argv+1); SKIPARG; }
			break;
			
	  default:	fprintf (stderr, "%s", USAGE);
			exit (1);
        }
      }
    }
    
    if (argc > 0)	size = atoi (argv[0]);

    /* Default parms for algorithms */
    if (parm <= -1e9)
    { if      (alg == 'b') parm = 50.0;
      else if (alg == 'c') parm = 4.0;
    }

    /* Determine output height & width (oh*ow <= size) */
    h = input.hdr.rows;
    w = input.hdr.cols;

    if (size < 0)
    { oh = h; ow = w; }
    else
    { ow = sqrt ((double) size * w / (h * input.hdr.aspect / PODASPECT));
      ow &= ~7;			/* Make width multiple of 8 */
      oh = ow * input.hdr.aspect/PODASPECT * h / w;
    }

    fprintf (stderr,
	     "Halftone \"%s\" size [%dx%d] => %d pixels\n",
	     input.hdr.title[0] ? input.hdr.title : "(untitled)",
	     ow, oh, ow*oh);

    /* Start with image in variable 'input' */
    image = &input;

    /* If necessary, resize it */
    if (w != ow || h != oh)
    { if (extract_fbm (&input, &resized, 0, 0, w, h, ow, oh, title, (char *) NULL))
      { image = &resized; }
      else
      { exit (1); }
    }

    /* Sharpen the image if requested */    
    if (beta > -1e9)
    { if (sharpen_fbm (image, &sharpened, beta))
      { image = &sharpened; }
      else
      { exit (1); }
    }
    
    /* Now use the appropriate algorithm to halftone it */
    switch (alg)
    { case 'b':	bluenoise_fbm (image, &output, parm); break;
      case 'c': constravg_fbm (image, &output, parm); break;
      default:	floyd_fbm (image, &output);
    }

    if (write_pod (&output, stdout)) exit (0);
  }
  
  exit (1);
}

/****************************************************************
 * write_pod: Write out a binary bitmap as a Diablo file, for use
 *	      by the podtype or mp programs.
 ****************************************************************/

# define FF "\014"
# define LF "\012"
# define CR "\015"
# define GON	"\033\037\005\033\036\003"
# define GOFF	"\033\037\015\033\036\011"
# define ABSTAB "\033\011"
# define STARTCOL 10

write_pod (image, stream)
FBM *image;
FILE *stream;
{ register int i, j, h, w;

  h = image->hdr.rows;
  w = image->hdr.cols;

  /* Bracket commands with form feeds (for podtype) */
  fprintf (stream, "%s%s%s", FF, CR, GOFF);
  
  for (j=0; j<h; j++)
  { fprintf (stream, "%s%c%s", ABSTAB, STARTCOL+1, GON);
    for (i=0; i<w; i++)
    { putchar (image->bm[j*w + i] ? ' ' : '.'); }
    fprintf (stream, "%s%s%s", LF, GOFF, CR);
    
  }
  
  fprintf (stream, "%s", FF);
  
  return (1);
}
