/* $Header: /usr/local/src/dync-1.0/RCS/dync.c,v 1.4 1998/08/13 14:32:58 root Exp root $ */

/*
 * Copyright © 1998 Alistair G. Crooks.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Alistair G. Crooks.
 * 4. The name of the author may not be used to endorse or promote
 *    products derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
 * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>

#ifdef HAVE_GRP_H
#include <grp.h>
#endif

#ifdef HAVE_PWD_H
#include <pwd.h>
#endif

#include <string.h>
#include <stdarg.h>

#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif

#ifdef HAVE_DLFCN_H
#include <dlfcn.h>
#endif

#ifndef DL_LAZY
#define DL_LAZY	RTLD_LAZY
#endif

#define VERSION	"1.1"
#define AUTHOR	"Alistair G. Crooks (agc@uts.amdahl.com)"

enum {
	MaxFileNameLen = 2048,
	MaxCmdLen = 4096
};

#ifndef TMPDIR
#define TMPDIR	"/tmp"
#endif

#define OBJ_SUFFIX	".o"

#define FLAGS_FOR_PIC	"-fPIC -DPIC"

static int	verbose;
static char	*progname;

/* ====================================================================== */

/* a replacement for system(3) */
int
asystem(char *buf, int n, char *fmt, ...)
{
	va_list	vp;

	va_start(vp, fmt);
	vsnprintf(buf, n, fmt, vp);
	va_end(vp);
	if (verbose) {
		printf("Command: %s\n", buf);
	}
	return system(buf);
}

/* like Perl's die... */
void
die(char *fmt, ...)
{
	va_list	vp;

	(void) fprintf(stderr, "%s: ", progname);
	va_start(vp, fmt);
	vfprintf(stderr, fmt, vp);
	va_end(vp);
	(void) fputc('\n', stderr);
	exit(1);
}

/* ====================================================================== */

#ifndef HAVE_RANDOM
int
random(void)
{
	return rand();
}
#endif

/* like mkstemp, but uses XXXXXX.c in template, for obvious reasons */
static int
nmkstemp(char *template)
{
	char	randv[7];
	char	*rand;
	int	fd;

	rand = &template[strlen(template) - 8];
	for (;;) {
		(void) sprintf(randv, "%06.06d", random() % 999999);
		(void) memcpy(rand, randv, 6);
		if ((fd = open(template, O_EXCL | O_CREAT | O_RDWR, 0600)) >= 0) {
			if (verbose) {
				fprintf(stderr, "nmkstemp: `%s'\n", template);
			}
			return fd;
		}
	}
}

/* return 1 if we're on an ELF platform */
static int
iself(char *prog)
{
	char	cmd[MaxCmdLen];

	return (asystem(cmd, sizeof(cmd), "%s %s | %s ELF", FILE_CMD, prog, GREP_CMD) == 0);
}

/* print usage message and die */
static void
usage(void)
{
	(void) fprintf(stderr, "Usage:\t%s [compilation args] -f sourcefile [runtimeargs], or\n", progname);
	(void) fprintf(stderr, "\t%s [compilation args] program-text [runtimeargs], or\n", progname);
	exit(1);
}

extern char	*optarg;
extern int	optind;

int
main(int argc, char **argv)
{
	char	mainname[MaxFileNameLen];
	char	basename[MaxFileNameLen];
	char	libname[MaxFileNameLen];
	char	file[MaxFileNameLen];
	char	cflags[MaxCmdLen];
	char	cmd[MaxCmdLen];
	char	*dot;
	char	*s;
	void	*handle;
	int	tempfile;
	int	cflagc;
	int	(*newmain)(int argc, char **argv);
	int	elf;
	int	fd;
	int	i;

	if ((progname = strrchr(*argv, '/')) == (char *) NULL) {
		progname = *argv;
	} else {
		progname++;
	}
	verbose = cflagc = 0;
	cflags[cflagc] = 0;
	tempfile = 1;
	while ((i = getopt(argc, argv, "D:I:L:U:l:f:v")) != -1) {
		switch(i) {
		case 'D':
		case 'I':
		case 'L':
		case 'U':
		case 'l':
			cflagc += snprintf(&cflags[cflagc], sizeof(cflags) - cflagc, "-%c%s ", i, optarg);
			break;
		case 'f':
			(void) strncpy(file, optarg, sizeof(file) - 1);
			file[sizeof(file) - 1] = 0;
			tempfile = 0;
			break;
		case 'v':
			verbose = 1;
		}
	}
	if (verbose) {
		(void) fprintf(stderr, "%s, version %s, %s\n", *argv, VERSION, __DATE__);
		(void) fprintf(stderr, "Please mail bug-reports to %s\n", AUTHOR);
	}
	if (optind == argc) {
		usage();
	}
	if (tempfile) {
		(void) snprintf(file, sizeof(file), "%s/tempdync.XXXXXX.c", TMPDIR);
		fd = nmkstemp(file);
		i = strlen(argv[optind]);
		if (write(fd, argv[optind], i) != i) {
			die("short write to `%s'\n", file);
		}
		(void) close(fd);
	}
	if ((dot = strrchr(file, '.')) == (char *) NULL) {
		usage();
	}
	if ((s = strrchr(file, '/')) == (char *) NULL) {
		s = file;
	} else {
		s++;
	}
	elf = iself(*argv);
	(void) strncpy(basename, s, dot - s);
	(void) snprintf(mainname, sizeof(mainname), "%s__newmain", (elf) ? "" : "_");
	(void) snprintf(libname, sizeof(libname), "lib%s.so.1.0", basename);

	/* compile the source file */
	if (asystem(cmd, sizeof(cmd), "%s -c %s -Dmain=%s %s %s -o %s%s",
			CC_CMD,
			cflags,
			&mainname[elf ? 0 : 1],
			FLAGS_FOR_PIC,
			file,
			basename,
			OBJ_SUFFIX) != 0) {
		die("`%s' failed", cmd);
	}

	/* make it into a shared object */
	if (asystem(cmd, sizeof(cmd), "%s -x -shared --whole-archive %s%s -o %s",
			LD_CMD, basename, OBJ_SUFFIX, libname) != 0) {
		die("`%s' failed", cmd);
	}

	if (!elf) {
		/* remember to add it to the cache */
		if (asystem(cmd, sizeof(cmd), "%s -m .", LDCONFIG_CMD) != 0) {
			die("`%s' failed", cmd);
		}
	}

	/* get a handle on the shared lib */
	if ((handle = dlopen(libname, DL_LAZY)) == (void *) NULL) {
		die("can't dlopen `%s'", libname);
	}

	/* remove the object file and shared object lib */
	if (asystem(cmd, sizeof(cmd), "rm -f %s%s %s", basename, OBJ_SUFFIX, libname) != 0) {
		die("`%s' failed", cmd);
	}

	/* remove any temp file */
	if (tempfile && asystem(cmd, sizeof(cmd), "rm -f %s", file) != 0) {
		die("`%s' failed", cmd);
	}

	/* find the renamed main */
	if ((newmain = dlsym(handle, mainname)) == NULL) {
		die("can't get symbol `%s'", mainname);
	}

	/* execute it */
	argv[optind - 1] = mainname;
	i = (*newmain)(argc - optind + 1, argv + optind - 1);

	/* close the shared lib and exit */
	dlclose(handle);

	exit(i);
}
