#include "dde_drive.h"

#define DDE_BLOCK_BUFFER_SIZE (32768)
#define DDE_PAGE_SIZE (4096)

static struct dde_minix_drive gbl_drives[MAX_DRIVES];
static struct dde_minix_drive *gbl_sel_drv;

static int gbl_selected_drive_idx;			/* selected drive */
static struct device *gbl_selected_part;


static char *buffer;
ddekit_sem_t *gbl_bio_sem;
static ddekit_thread_t *gbl_thread;

/*****************************************************************************/
/**  PROTOTYPES                                                             **/
/*****************************************************************************/

static int            dde_block_open(dev_t minor, int access);
static int            dde_block_close(dev_t minor);
static struct device *dde_block_prepare(dev_t minor);

static int            dde_block_transfer(dev_t minor,
                                         int do_write,
										 u64_t pos,
										 endpoint_t endpt,
										 iovec_t *iov,
										 unsigned count,
										 int flags);

static int            dde_block_do_read(struct dde_minix_drive *drv,
                                        unsigned long block,
										iovec_t *iov,
										unsigned nr_req,
										int proc_nr,
										unsigned nbytes);

static int            dde_block_do_write(struct dde_minix_drive *drv,
                                         unsigned long block,
										 iovec_t *iov,
										 unsigned nr_req,
										 int proc_nr,
										 unsigned nbytes);

static int            dde_block_ioctl(dev_t minor,
                                      unsigned int request,
									  endpoint_t endpt,
									  cp_grant_id_t grant);

static void           dde_block_geometry(dev_t minor,
                                         struct part_geom *entry);


/* Entry points to this driver. */
static struct blockdriver dde_block_driver  = {
  BLOCKDRIVER_TYPE_DISK,
  dde_block_open,		/* open or mount request, initialize device */
  dde_block_close,		/* release device */
  dde_block_transfer,		/* do the I/O */
  dde_block_ioctl,		/* get or set a partition's geometry */
  NULL,		/* nothing to clean up */
  dde_block_prepare,		/* prepare for I/O on a given minor device */
  dde_block_geometry,		/* tell the geometry of the disk */
  NULL,		/* ignore interrupts (handled by ddekit main_loop) */
  NULL,		/* ignore leftover alarms */
  NULL,		/* ignore CANCELs */
  NULL,		/* catch-all for unrecognized commands and ioctls */
};



/*****************************************************************************/
/**  BLOCKDRIVER CALLBACK IMPLEMENTATION                                    **/
/*****************************************************************************/

static void dde_block_geometry(dev_t minor, struct part_geom *entry)
{
	int result;
	struct ddekit_bd_geometry geom;

	if ( (dde_block_prepare(minor)) == NULL)
		return;

	DDEBUG_MSG_VERBOSE(" for partition: %p",entry);

	result = ddekit_bd_get_geom(gbl_sel_drv->disk, &geom);
	if (!result) {
		entry->heads = geom.heads;
		entry->cylinders = geom.cylinders;
		entry->sectors = geom.sectors;
	} else {
		entry->heads=0;
		entry->cylinders=0;
		entry->sectors=0;
	}
}

static int dde_block_open(dev_t minor, int access)
{
	struct dde_minix_drive * drv;

	if ( (dde_block_prepare(minor)) == NULL) return(ENXIO);

	drv = gbl_sel_drv;

	DDEBUG_MSG_VERBOSE("%s", gbl_sel_drv->name);

	if (drv->open_ct == 0) {

		/* TODO: only disks supported for now */

		/* try to open disk */

		gbl_sel_drv->disk =
		    ddekit_bd_create(gbl_sel_drv->name, 0 /*whole disk*/ );
		if (gbl_sel_drv->disk == NULL)
			return (ENXIO);

		/* determine size of disk */
		struct ddekit_bd_geometry geom;
		ddekit_bd_get_geom(gbl_sel_drv->disk, &geom);

		gbl_sel_drv->part[0].dv_size =
			mul64u(ddekit_bd_get_sectorcount(gbl_sel_drv->disk), SECTOR_SIZE);

		DDEBUG_MSG_VERBOSE("calling partition()...");

		partition(&dde_block_driver, gbl_selected_drive_idx * DEV_PER_DRIVE,
			P_PRIMARY, 0);

		DDEBUG_MSG_VERBOSE("partition() done.");
	}

	gbl_sel_drv->open_ct++;

	return (OK);
}

static int dde_block_close(dev_t minor)
{
	if (dde_block_prepare(minor) == NULL) return(ENXIO);

	DDEBUG_MSG_VERBOSE("%s ",gbl_sel_drv->name);

	gbl_sel_drv->open_ct -= 1;

	if (gbl_sel_drv->open_ct == 0) {
		/* release disk */
		ddekit_bd_destroy(gbl_sel_drv->disk);
	}

	return (OK);
}

static struct device *dde_block_prepare(dev_t device)
{
	/* Prepare for I/O on a device. */
	gbl_selected_drive_idx = device;

	DDEBUG_MSG_VERBOSE("%d", device);

	if (device < NR_MINORS) {			/* d0, d0p[0-3], d1, ... */
		gbl_selected_drive_idx = device / DEV_PER_DRIVE;	/* save drive number */
		gbl_sel_drv = &gbl_drives[gbl_selected_drive_idx];
		gbl_selected_part = &gbl_sel_drv->part[device % DEV_PER_DRIVE];
	} else {
		if ((unsigned) (device -= MINOR_d0p0s0) < NR_SUBDEVS) {/*d[0-7]p[0-3]s[0-3]*/
			gbl_selected_drive_idx = device / SUB_PER_DRIVE;
			gbl_sel_drv = &gbl_drives[gbl_selected_drive_idx];
			gbl_selected_part = &gbl_sel_drv->subpart[device % SUB_PER_DRIVE];
		} else {
			gbl_selected_drive_idx = -1;
			return(NULL);
		}
	}
	return(gbl_selected_part);
}


static int dde_block_transfer(
  dev_t minor,			/* minor device to perform the transfer on */
  int do_write,			/* read or write? */
  u64_t position,		/* offset on device to read or write */
  endpoint_t proc_nr,		/* process doing the request */
  iovec_t *iov,			/* pointer to read or write request vector */
  unsigned int nr_req,		/* length of request vector */
  int UNUSED(flags)		/* transfer flags */
)
{

	struct dde_minix_drive *drv = NULL;
	unsigned long block;
	unsigned nbytes =0;
	iovec_t *iop, *iov_end = iov + nr_req;
	DDEBUG_MSG_VERBOSE("");

	if (dde_block_prepare(minor) == NULL) return(ENXIO);

	drv = gbl_sel_drv;

	DDEBUG_MSG_VERBOSE("proc_nr: %d, %s, do_write: %d,  req_nr: %d",
		proc_nr, gbl_sel_drv->name, do_write, nr_req);

	DDEBUG_MSG_VERBOSE("Address of first iovec: %p", iov->iov_addr);

    /*
	 * we have to posibilities here:
	 * (1) the userbuffer is DMA-able. Then we have to map the userbuffer and
	 *     add it's pages to the ddepage_cache.
	 * (2) the userbuffer is not DMA-able. Than we have to copy the buffer to
	 *     memory, which was prevously added to the dde_page_cache
	 *
	 */

	/* Check disk address. */
	if (rem64u(position, SECTOR_SIZE) != 0) {
		DDEBUG_MSG_VERBOSE("(rem64u(position, SECTOR_SIZE) != 0");
		return(EINVAL);
	}

	/* as position is the sector on a partition we need to convert it to
	 * a sector of the disk */
	/* Note: SECTOR_SIZE in Minix 3 is 512 -> fits to Linux sector size */
	block = div64u(add64(gbl_selected_part->dv_base, position), SECTOR_SIZE);

	/* would the transfer exceed partition limits? */
	for (iop = iov; iop < iov_end; iop++) {
		DDEBUG_MSG_VERBOSE("iovec: %p, %d", iop->iov_addr,  iop->iov_size);
		nbytes += iop->iov_size;
	}

	/* ggc schould actually know how to do 64 bit calculations.... */

	if (position + nbytes > gbl_selected_part->dv_size) {
		nbytes -= position + nbytes - gbl_selected_part->dv_size;
		if (nbytes < 0)
			return 0;
	}


	if (do_write) {
			return dde_block_do_write(drv, block, iov, nr_req, proc_nr, nbytes);
	} else {
			return dde_block_do_read(drv, block, iov, nr_req, proc_nr, nbytes);
	}
}

static int dde_block_do_read(struct dde_minix_drive *drv,
                             unsigned long block,
                             iovec_t *iov,
                             unsigned nr_req,
                             int proc_nr,
							 unsigned nbytes)
{
	int i;
	iovec_t *iop = iov;

	DDEBUG_MSG_VERBOSE("block: %ld", block);

	/* how much to transfer? */
	unsigned transfered_bytes = 0;

	DDEBUG_MSG_VERBOSE("%d bytes", nbytes);

	if ((nbytes & SECTOR_MASK) != 0) return(EINVAL);

	while (nbytes > 0) {
		unsigned read_size  = 0;
		unsigned iov_offset = 0;

		/* read as much as you can! */
		if (nbytes < DDE_BLOCK_BUFFER_SIZE) {
			read_size = nbytes;
		} else {
			read_size = DDE_BLOCK_BUFFER_SIZE;
		}

		DDEBUG_MSG_VERBOSE("reading: %d bytes", read_size);

		unsigned nr_pages = read_size/DDE_PAGE_SIZE;

		if (nr_pages * DDE_PAGE_SIZE < read_size)
			nr_pages++;

		DDEBUG_MSG_VERBOSE("%d pread_sizeages", nr_pages);

		struct bio *b= ddekit_bd_create_bio(drv->disk,
		               block,
		   			   nr_pages);

		/* add buffer to bio */
		DDEBUG_MSG_VERBOSE("Setting up bio");

		unsigned chunk_size = read_size;

		for (i=0; i < nr_pages; i++) {

			unsigned size = DDE_PAGE_SIZE;

			chunk_size -= DDE_PAGE_SIZE;

			if(chunk_size < 0 )
				size += chunk_size;

			DDEBUG_MSG_VERBOSE("ddekit_bd_add_bio_io %d, %p, %d", i,
				buffer+(i*DDE_PAGE_SIZE), size);

			ddekit_bd_add_bio_io(b, i,
				(unsigned long) buffer + (i * DDE_PAGE_SIZE), size);
		}

		/* transmit */
		DDEBUG_MSG_VERBOSE("submitting bio");

		ddekit_bd_submit_bio(b, NULL, DDEKIT_BIO_READ);

		/* wait for completion */
		DDEBUG_MSG_VERBOSE("waiting for completion");
		ddekit_sem_down(gbl_bio_sem);

		/* calculate new block address
		 * ... only necessary when there is more data to write
		 */
		block  += read_size/SECTOR_SIZE;
		nbytes -= read_size;

		DDEBUG_MSG_VERBOSE("still to read: %d bytes", nbytes);

		/* COPY OUT */
		unsigned buffer_offset =0;
		while(read_size > 0){
			int s;
			unsigned copy_size;
			/* is the vector bigger than what we have here */
			if ( (iop->iov_size) > read_size) {
				copy_size = read_size;
			} else {
				copy_size = iop->iov_size;
			}

			if (proc_nr != SELF) {

				DDEBUG_MSG_VERBOSE("sys_safecopyto(%d, %p, %d, %p, %d);", proc_nr,
					 iop->iov_addr, iov_offset, buffer+buffer_offset, copy_size);

				s= sys_safecopyto(proc_nr, iop->iov_addr,
						iov_offset,
						(vir_bytes)(buffer+buffer_offset), copy_size);

				if (s != OK) {
					panic("w_transfer: sys_vircopy failed: %d", s);
				}

			} else {

				DDEBUG_MSG_VERBOSE("memcpy(%p, %p, %d);",
					(char *) iop->iov_addr + iov_offset,buffer, read_size);

				memcpy((char *) iop->iov_addr + iov_offset,
						buffer, copy_size);
			}

			iop->iov_size -= copy_size;

			/* next iov? */
			if (iop->iov_size == 0) {
				iop++;
				iov_offset=0;
			} else {
				iov_offset += copy_size;
			}

			read_size -= copy_size;
			buffer_offset += copy_size;
			transfered_bytes += copy_size;
		}

	} /* while (nbytes > 0) */

	return transfered_bytes;
}


static int dde_block_do_write(
	struct dde_minix_drive *drv,
	unsigned long block,
	iovec_t *iov,
	unsigned nr_req,
	int proc_nr,
	unsigned nbytes)
{
	iovec_t *iop = iov;

	unsigned buffer_avail_bytes = DDE_BLOCK_BUFFER_SIZE;
	unsigned iov_offset         = 0;
	unsigned buffer_offset      = 0;
	unsigned bytes_transfered   = 0;


	/* only whole sectors? */
	if ((nbytes & SECTOR_MASK) != 0) {
		return(EINVAL);
	}

	while (nbytes > 0) {

		int res;
		unsigned copy_size=0;

		/* how much of current iovec fits into buffer */
		copy_size = iop->iov_size;
		if (copy_size > buffer_avail_bytes) {
			copy_size = buffer_avail_bytes;
		}

		/* Copy the stuff */
		if (proc_nr != SELF) {

			DDEBUG_MSG_VERBOSE("sys_safecopyfrom(%d, %d, %d, %p, %d)",
				proc_nr, iop->iov_addr, iov_offset, buffer+buffer_offset, copy_size);

			res = sys_safecopyfrom(proc_nr, iop->iov_addr, iov_offset,
					  (vir_bytes) (buffer + buffer_offset),
					  copy_size);

			if (res != OK)
				ddekit_panic("copy in failed");

		} else {

			memcpy(buffer + buffer_offset,
				(char *) iop->iov_addr + iov_offset, copy_size);

		}

		iov_offset += copy_size;
		nbytes     -= copy_size;
		iop->iov_size -= copy_size;

		buffer_avail_bytes -= copy_size;
		buffer_offset += copy_size;

		/* if either the buffer is full or we have copied all... */
		if (buffer_avail_bytes == 0 || nbytes == 0) {

			unsigned transfer_bytes =  DDE_BLOCK_BUFFER_SIZE-buffer_avail_bytes;

			/* alloc bio */
			DDEBUG_MSG_VERBOSE("creating bio for block %d", block);

			/* how many pages of the transfer buffer to we have to write? */
			unsigned nr_pages = transfer_bytes / DDE_PAGE_SIZE;

			if (nr_pages * DDE_PAGE_SIZE < transfer_bytes)
				nr_pages++;

			struct bio *b= ddekit_bd_create_bio(drv->disk,
				block, nr_pages);

			unsigned chunk_size = transfer_bytes;

			int i;

			for (i=0; i < nr_pages; i++) {

				unsigned size = DDE_PAGE_SIZE;
				chunk_size   -= DDE_PAGE_SIZE;

				if(chunk_size < 0 )
					size += chunk_size;

				DDEBUG_MSG_VERBOSE("ddekit_bd_add_bio_io %d, %p, %d", i,
					buffer + (i * DDE_PAGE_SIZE), size);

				ddekit_bd_add_bio_io(b, i,
					(unsigned long) buffer + (i * DDE_PAGE_SIZE), size);
			}

			/* transmit */
			ddekit_bd_submit_bio(b, NULL, DDEKIT_BIO_WRITE);

			/* wait for completion */
			ddekit_sem_down(gbl_bio_sem);

			/* calculate new block address
			 * ... only necessary when there is more data to write
			 */

			block += DDE_BLOCK_BUFFER_SIZE/SECTOR_SIZE;
			buffer_offset = 0;
			buffer_avail_bytes = DDE_BLOCK_BUFFER_SIZE;
			bytes_transfered += transfer_bytes;
		}

		/* did we copy all of the iovec? */
		if(iop->iov_size == 0) {
			iop++;
			iov_offset = 0;
		}

	} /* while (nbytes > 0) */

	return bytes_transfered;
}

static void dde_block_end_bio_callback(void *priv_data, unsigned int err) 
{
	ddekit_sem_up(gbl_bio_sem);
}

void l4dde26_process_add_worker();

static void dde_block_thread_func(void *unused) {

	struct ddekit_minix_msg_q *mq = ddekit_minix_create_msg_q(BDEV_RQ_BASE, 
	                                    BDEV_RQ_BASE + 0xff);

	message m;
	int ipc_status;
	/* as we call linux function in the context of this thread, we have to
	 * make linux aware of this thread. */
	l4dde26_process_add_worker();

	while(1) {
		DDEBUG_MSG_VERBOSE("block: waiting for message.");
		ddekit_minix_rcv(mq, &m, &ipc_status);
		DDEBUG_MSG_VERBOSE("block: received message.");
		blockdriver_process(&dde_block_driver, &m, ipc_status);
		DDEBUG_MSG_VERBOSE("block: message handled.");
	}
}

void dde_block_init()
{
	static char name[] = "sda";

	int i;

	for (i=0; i < MAX_DRIVES ; i++) {
		memset(&gbl_drives[i],0,sizeof(gbl_drives[i]));
		memcpy(gbl_drives[i].name, name, strlen(name));
		gbl_drives[i].name[2] += i;
		gbl_drives[i].disk = NULL;
		gbl_drives[i].open_ct = 0;
	}

	/* allocate buffer */
	buffer = (char *)ddekit_large_malloc(DDE_BLOCK_BUFFER_SIZE);

	/* add our buffer to dde page cache */
	ddekit_bd_page_cache_add((unsigned long)buffer, DDE_BLOCK_BUFFER_SIZE);

	DDEBUG_MSG_VERBOSE("Buffer at: %p, size: %d\n", buffer, DDE_BLOCK_BUFFER_SIZE);

	/* init semaphore */
	gbl_bio_sem = ddekit_sem_init(0);

	/* set callback function */
	ddekit_bd_register_callback(dde_block_end_bio_callback);

	/* start message thread */
	DDEBUG_MSG_VERBOSE("Starting storage thread..\n");
	gbl_thread = ddekit_thread_create(dde_block_thread_func, NULL, "blockthread");

	DDEBUG_MSG_INFO("DDE block layer intialized.");
}

static int dde_block_ioctl(dev_t minor,
                    unsigned int request,
					endpoint_t endpt,
					cp_grant_id_t grant)
{
	int r, count;

	switch (request) {
		case DIOCTIMEOUT:
			return OK;

		case DIOCOPENCT:
			if (dde_block_prepare(minor) == NULL) return ENXIO;
			count = gbl_sel_drv->open_ct;
			r = sys_safecopyto(endpt, grant, 0, (vir_bytes)&count,
					sizeof(count));

			if(r != OK)
				return r;

			return OK;

		case DIOCFLUSH:
			if (dde_block_prepare(minor) == NULL) return ENXIO;

			if (gbl_sel_drv)
				return EIO;

			return ddekit_bd_flush(gbl_sel_drv->disk);
	}

	return EINVAL;
}

