/*
 * \brief   DDEUSB26 library - Virtual host controller for DDELinux 
 *          based USB drivers
 * \date    2009-04-07
 * \author  Dirk Vogt <dvogt@os.inf.tu-dresden.de>
 */

/*
 * This file is part of the DDEUSB package, which is distributed under
 * the  terms  of the  GNU General Public Licence 2.  Please see the
 * COPYING file for details.
 */

#define WARN_UNIMPL 

#include <asm/io.h>
#include <linux/kthread.h>
#include <linux/init.h>
#include <linux/platform_device.h>
#include <linux/workqueue.h>
#include <linux/usb.h>

#include <ddekit/memory.h>
#include <ddekit/usb.h>

#include "../core/hcd.h"
#include "../core/usb.h"
	

MODULE_AUTHOR("Dirk Vogt <dvogt@few.vu.nl>");
MODULE_DESCRIPTION("Virtual Host Controller for ddekit_usb");
MODULE_LICENSE("GPL");

static int urb_counter = 0;

static struct kmem_cache * priv_cache;

static const char driver_name[] = "ddeusb_vhcd";
static const char driver_desc[] = "DDEUSB Virtual Host Controller";

struct usb_hcd  *ddeusb_vhcd_hcd = NULL;
static u64 ddeusb_dma_mask = 0;

#define PORT_C_MASK ((USB_PORT_STAT_C_CONNECTION | USB_PORT_STAT_C_ENABLE | \
        USB_PORT_STAT_C_SUSPEND | USB_PORT_STAT_C_OVERCURRENT | \
        USB_PORT_STAT_C_RESET) << 16)


#define DDEUSB_VHCD_PORT_COUNT 8
#if 0
#define DEBUG_MSG(fmt, ...) ddekit_printf("%s : "fmt"\n", __func__, ##__VA_ARGS__ ) 
#else
#define DEBUG_MSG(fmt, ...) do{}while(0)
#endif

#if 0
static void dump_linux_urb(struct urb *urb) {
	printk("================\n");
	printk("DUMP: LX urb (0x%p)\n", urb);
	printk("================\n");
	printk("= usb_device: %p\n", urb->dev);
	printk("= pipe %d\n", urb->pipe);
	printk("=    type: %d\n", usb_pipetype(urb->pipe));
	printk("=    direction: %d\n", usb_pipein(urb->pipe));
	printk("= status: %d\n", urb->status);
	printk("= error_count: %d\n", urb->error_count);
	printk("= size: %d\n", urb->transfer_buffer_length);
	printk("= actual_length: %d\n", urb->actual_length);
	printk("= transfer_flags %x\n", urb->transfer_flags);
	printk("= setup_packet: \n");
	if (urb->setup_packet) {
		printk("=   bRequestType: 0x%x \n", 
		    ((struct usb_ctrlrequest *)urb->setup_packet)->bRequestType);
		printk("=   bRequest 0x%x \n",
		    ((struct usb_ctrlrequest *)urb->setup_packet)->bRequest);
		printk("=   wValue: 0x%x \n",
		    ((struct usb_ctrlrequest *)urb->setup_packet)->wValue);
		printk("=   wIndex: 0x%x \n",
		    ((struct usb_ctrlrequest *)urb->setup_packet)->wIndex);
		printk("=   wLength: 0x%x \n",
		    ((struct usb_ctrlrequest *)urb->setup_packet)->wLength);
	} else {
		printk("+ None. \n");
	}
	printk("===============\n");

}
#else
#define dump_linux_urb(x) do{}while(0)
#endif

struct vhcd_port {
	struct ddekit_usb_dev *dev;
	struct usb_device *udev;
	unsigned int interfaces;
	int speed;
	int status;
	unsigned long port_status;

};

struct ddeusb_vhcd_urbpriv {
	struct ddekit_usb_urb *d_urb;
    int status;
    struct urb *urb;
};

struct ddeusb_vhcd_priv_data {
	spinlock_t lock;
	struct workqueue_struct * wq;
   	struct vhcd_port ports[DDEUSB_VHCD_PORT_COUNT];
};


/*
 * this function is part of the interface hack
 * (see usb/core/message.c:1735)
 */
int ddekit_usbintf_hack(int portnum, int interface)
{
	struct ddeusb_vhcd_priv_data *data = NULL;
	data = (struct ddeusb_vhcd_priv_data *) ddeusb_vhcd_hcd->hcd_priv;
	return data->ports[portnum].interfaces & (1 << interface);
}

static void
ddeusb_register_device
(struct ddekit_usb_dev *dev, unsigned int interfaces)
{
    unsigned long flags;
	int i;
	int ret= ENOMEM;

	struct ddeusb_vhcd_priv_data *data = NULL;
	
	/* check if hcd is allready setup */
	if (ddeusb_vhcd_hcd == NULL) {
		return ;
	}

	DEBUG_MSG("Registering new device to vhcd");
	data = (struct ddeusb_vhcd_priv_data *) ddeusb_vhcd_hcd->hcd_priv;

	/*find an unused port */
    spin_lock_irqsave(&data->lock, flags);

	for(i=0; i < DDEUSB_VHCD_PORT_COUNT ; i++ ) {

		if (data->ports[i].dev == NULL) {

			/* found unused port */
			data->ports[i].dev = dev;
			data->ports[i].interfaces = interfaces;
			ddekit_usb_dev_set_data(dev, data);

			data->ports[i].port_status |= USB_PORT_STAT_CONNECTION
			                               | (1 << USB_PORT_FEAT_C_CONNECTION);
			ret = 0;

			break;
		}
	}

    spin_unlock_irqrestore(&data->lock, flags);
}

/*
 * context: in_interrupt()
 */
static void ddeusb_disconnect_device(struct ddekit_usb_dev *dev) {

    unsigned long flags;
	int i;

	struct ddeusb_vhcd_priv_data *data = NULL;

	/* check if hcd is allready setup */
	if(!ddeusb_vhcd_hcd) 
	{
		return;
	}

	data = (struct ddeusb_vhcd_priv_data *) ddeusb_vhcd_hcd->hcd_priv;

	/*find an unused port */
    spin_lock_irqsave(&data->lock, flags);

	struct vhcd_port *port = (struct vhcd_port *) ddekit_usb_dev_get_data(dev);

	port->dev = NULL;

	data->ports[i].port_status &= ~USB_PORT_STAT_CONNECTION ;
	data->ports[i].port_status |= (1 << USB_PORT_FEAT_C_CONNECTION);

    spin_unlock_irqrestore(&data->lock, flags);
}


/*
 * context: in_interrupt()
 */
static void ddeusb_vhcd_urb_complete (void *priv)
{
	struct ddeusb_vhcd_urbpriv *urb_priv = (struct ddeusb_vhcd_urbpriv *) priv;
	struct urb *urb=NULL;

	// TODO: change that into an assert!
	if(!ddeusb_vhcd_hcd)
    {
		DEBUG_MSG("if(!ddeusb_vhcd_hcd)");
		return;
    }

	struct ddekit_usb_urb *d_urb = urb_priv->d_urb;

    /* check if hcd is allready setup */
    urb = urb_priv->urb;

	urb->status         = d_urb->status;
	urb->actual_length  = d_urb->actual_length;
	urb->start_frame    = d_urb->start_frame;
	urb->interval       = d_urb->interval;
	urb->error_count    = d_urb->error_count;
    urb->transfer_flags = d_urb->transfer_flags;

    kmem_cache_free(priv_cache, urb_priv);
       urb->hcpriv = NULL;

	//dump_linux_urb(urb);
	DEBUG_MSG("usb_hcd_giveback_urb");
	usb_hcd_giveback_urb(ddeusb_vhcd_hcd, urb, urb->status);

	kfree(d_urb);

	usb_put_urb(urb);
}


static int ddeusb_vhcd_reset(struct usb_hcd *hcd)
{
	/* should not be called...*/
	WARN_UNIMPL;
    return 0;
}


static int ddeusb_vhcd_start(struct usb_hcd *hcd)
{
	int i;

    struct ddeusb_vhcd_priv_data *data;
	DEBUG_MSG("");

	data = (struct ddeusb_vhcd_priv_data *) hcd->hcd_priv;


	for (i=0; i<DDEUSB_VHCD_PORT_COUNT;i++)	{
		data->ports[i].dev = NULL;
		data->ports[i].port_status = 0;
	}

	spin_lock_init(&data->lock);

	/* We let the usbcore do the mapping not the driver... */
	hcd->self.uses_dma=0;

	hcd->state  = HC_STATE_RUNNING;

	ddeusb_vhcd_hcd = hcd;

	return 0;
}

/******************************************************************************/
/*                                                                            */
/*           power management functions are non implemented                   */
/*                                                                            */
/******************************************************************************/
static int ddeusb_vhcd_stop(struct usb_hcd *hcd)
{
	WARN_UNIMPL;
	return 0;
}
static void ddeusb_vhcd_shutdown(struct usb_hcd *hcd)
{
	WARN_UNIMPL;
}

/******************************************************************************/


/* get frame number should return the current usr frame */
/* not many drivers use it, so not supported yet        */
static int ddeusb_vhcd_get_frame_number(struct usb_hcd *hcd)
{
	WARN_UNIMPL;
	return 0;
}



static int ddeusb_vhcd_urb_enqueue(struct usb_hcd *hcd,
                                   struct urb *urb,
                                   gfp_t mem_flags)
{
    struct ddeusb_vhcd_priv_data *data;
	int number_of_iso_packets=0;
	int ret=0;
	unsigned int transfer_flags = 0 ;
    struct ddeusb_vhcd_urbpriv * priv=NULL; 

	struct usb_device * udev = urb->dev;
	data = (struct ddeusb_vhcd_priv_data *) hcd->hcd_priv;
	/* is our VHCD in a running state? */
	if (!HC_IS_RUNNING(hcd->state)) 
	{
		DEBUG_MSG("HC is not running");
		return  -ENODEV;
	}	

	/* Is there a device connected to this port? */
	if (data->ports[udev->portnum-1].dev == NULL) {
		DEBUG_MSG("Trying to send URB to non existent dev...");
		return -ENODEV;
	}
	/*
	 * we have to trap some control messages, i.e. USB_REQ_SET_ADDRESS...
	 * Is request for endpoint zero? 
	 */
	if (usb_pipedevice(urb->pipe) == 0) {

		__u8 type = usb_pipetype(urb->pipe);

		struct usb_ctrlrequest *ctrlreq =
		    (struct usb_ctrlrequest *) urb->setup_packet;

		if (type != PIPE_CONTROL || !ctrlreq ) {
			DEBUG_MSG("invalid request to devnum 0\n");
			ret = -EINVAL;
			goto no_need_xmit;
		}
		
		switch (ctrlreq->bRequest) {
			
			case USB_REQ_SET_ADDRESS:
				data->ports[udev->portnum].udev = urb->dev;
				DEBUG_MSG("SetAddress Request (%d) to port %d\n",
						ctrlreq->wValue, urb->dev->portnum);

				if (urb->status == -EINPROGRESS) {
					/* This request is successfully completed. */
					/* If not -EINPROGRESS, possibly unlinked. */
					urb->status = 0;
				}

				goto no_need_xmit;

			case USB_REQ_GET_DESCRIPTOR:
				
				if (ctrlreq->wValue == (USB_DT_DEVICE << 8))
					DEBUG_MSG("Get_Descriptor to device 0 "
					          "(get max pipe size)\n");
				goto out;

			default:
				
				/* NOT REACHED */
				DEBUG_MSG("invalid request to devnum 0"
				          "bRequest %u, wValue %u\n",
						   ctrlreq->bRequest, ctrlreq->wValue);

				ret =  -EINVAL;
				goto no_need_xmit;
		} /* switch */
	} /* if ... */

out:
	if (urb->status != -EINPROGRESS) {
		DEBUG_MSG("URB already unlinked!, status %d\n", urb->status);
		return urb->status;
	}
	
	/* is it an iso urb? */
	if (usb_pipeisoc(urb->pipe))
		number_of_iso_packets =  urb->number_of_packets;

	priv = ( struct ddeusb_vhcd_urbpriv *)
	    kmem_cache_alloc (priv_cache, GFP_KERNEL);	

	urb->hcpriv = (void *)priv;

	priv->urb=urb;
    
	transfer_flags = urb->transfer_flags;

    /* don't free the urb, we need it yet */
    usb_get_urb(urb);

    ret=0;
	
	struct ddekit_usb_urb * d_urb = (struct ddekit_usb_urb *) 
	    kmalloc(sizeof(struct ddekit_usb_urb), GFP_KERNEL);
	
	priv->d_urb=d_urb;

    if (ret == 0) {
        d_urb->type              = usb_pipetype(urb->pipe);
        d_urb->dev               = data->ports[urb->dev->portnum-1].dev;
        d_urb->endpoint          = usb_pipeendpoint(urb->pipe);
        d_urb->direction         = !!usb_pipein(urb->pipe);
        d_urb->interval          = urb->interval;
        d_urb->transfer_flags    = urb->transfer_flags;
        d_urb->number_of_packets = urb->number_of_packets;
        d_urb->priv              = priv;
        d_urb->size              = urb->transfer_buffer_length; 
        d_urb->data			     = urb->transfer_buffer;
		d_urb->setup_packet      = urb->setup_packet;
		d_urb->iso_desc          = (struct ddekit_usb_iso_packet_desc*)
		                               urb->iso_frame_desc;
		//dump_linux_urb(urb);	
        ret = ddekit_usb_submit_urb(d_urb);

        DEBUG_MSG("tried to send urb : %d",ret);
    }

	if (ret) {
	    DEBUG_MSG("URB SUBMIT FAILED (%d).",ret);
		
		/* s.t. went wrong. */	
        usb_put_urb(urb);
        urb->status = ret;
        urb->hcpriv=NULL;
	
		/* free the d_urb  */
		kfree(d_urb);

		return ret;
	}
    
    return 0;

no_need_xmit:
    
	usb_hcd_giveback_urb(hcd, urb, urb->status);
	return 0;
}


/*TODO: */
static int ddeusb_vhcd_urb_dequeue(struct usb_hcd *hcd, struct urb *urb, int status)
{
	WARN_UNIMPL;
	return 0;
}

/* See hub_configure in hub.c */
static inline void hub_descriptor(struct usb_hub_descriptor *desc)
{
	memset(desc, 0, sizeof(*desc));
	desc->bDescriptorType = 0x29;
	desc->bDescLength = 9;
	desc->wHubCharacteristics = (__force __u16)
		(__constant_cpu_to_le16 (0x0001));
	desc->bNbrPorts = DDEUSB_VHCD_PORT_COUNT;
	desc->bitmap [0] = 0xff;
	desc->bitmap [1] = 0xff;
}

static void ddeusb_vhcd_endpoint_disable (struct usb_hcd *hcd,
                        struct usb_host_endpoint *ep)
{
	WARN_UNIMPL;
}


/*
 * hubd is polling this funtion. it returns wich on wich port events occured
 */
static int ddeusb_vhcd_hub_status_data (struct usb_hcd *hcd, char *buf)
{
    struct ddeusb_vhcd_priv_data *data;	int		ret = 0;
    unsigned long flags;
	unsigned long	*event_bits = (unsigned long *) buf;
	int		port;
	int		changed = 0;

	data = (struct ddeusb_vhcd_priv_data *) hcd->hcd_priv;

	*event_bits = 0;
	
	spin_lock_irqsave(&data->lock, flags);

	if (!test_bit(HCD_FLAG_HW_ACCESSIBLE, &hcd->flags))	{
		DEBUG_MSG("hw accessible flag in on?");
		goto done;
	}

	/* check status register for each port */
	for (port = 0; port < DDEUSB_VHCD_PORT_COUNT; port++) {
		if (data->ports[port].port_status & PORT_C_MASK) 
		{
			/* The status of a port has been changed, */
			DEBUG_MSG("port %d has changed\n", port);

			*event_bits |= 1 << ( port + 1);
			changed = 1;
		}
	}

	if (changed)
		ret = 1 + (DDEUSB_VHCD_PORT_COUNT / 8);
	else
		ret = 0;

done:
	spin_unlock_irqrestore(&data->lock, flags);
	return ret;	
}




static int ddeusb_vhcd_hub_control (struct usb_hcd *hcd,
                                u16 typeReq, u16 wValue, u16 wIndex,
                                char *buf, u16 wLength) 
{
   struct ddeusb_vhcd_priv_data *data =  (struct ddeusb_vhcd_priv_data *)  hcd->hcd_priv;
    unsigned long flags;	
    int ret = 0;
	int	port;	
	
    if (!test_bit(HCD_FLAG_HW_ACCESSIBLE, &hcd->flags))
		return -ETIMEDOUT;

	if (wIndex > DDEUSB_VHCD_PORT_COUNT)
	{
		DEBUG_MSG("invalid port number %d\n", wIndex);
		return -ENODEV;
	}
	
    /*
  	 * NOTE:
  	 * wIndex shows the port number and begins from 1.
  	 */	

	port = ((__u8 ) (wIndex & 0x00ff)) -1;
	DEBUG_MSG("port = %d \n", port);

	spin_lock_irqsave(&data->lock, flags);
	switch (typeReq)
	{
		case ClearHubFeature:
			DEBUG_MSG("unimplemented hub control request: ClearHubFeature");
			break;
		case ClearPortFeature:
			switch (wValue) 
			{
				case USB_PORT_FEAT_SUSPEND:
					DEBUG_MSG("unimplemented hub control request: ClearPortFeature USB_PORT_FEAT_SUSPEND");
					break;

 				case USB_PORT_FEAT_C_RESET:
 
					DEBUG_MSG("ClearPortFeature USB_PORT_FEAT_C_RESET");
 					
					switch (data->ports[port].speed) {
 						case USB_SPEED_HIGH:
 							data->ports[port].port_status |= USB_PORT_STAT_HIGH_SPEED;
 							break;
						case USB_SPEED_LOW:
							data->ports[port].port_status |= USB_PORT_STAT_LOW_SPEED;
							break;
						default:
							break;
					}	
				case USB_PORT_FEAT_POWER:
					DEBUG_MSG("ClearPortFeature USB_PORT_FEAT_POWER");
					
					/* FALLS THROUGH */

				default:
					DEBUG_MSG(" ClearPortFeature: default %x\n", wValue);
					data->ports[port].port_status &= ~(1 << wValue);
					break;
			}
			break;
		case GetHubDescriptor:
			DEBUG_MSG("hub_control_request: GetHubDescriptor");
			/* return the hub descriptor */ 
			hub_descriptor ((struct usb_hub_descriptor *) buf);
			
			break;
		case GetHubStatus:
			*(u32 *) buf = __constant_cpu_to_le32 (0);
			break;
		case GetPortStatus:
			if (wIndex > DDEUSB_VHCD_PORT_COUNT || wIndex < 1) {
				DEBUG_MSG(" invalid port number %d\n", wIndex);
				ret = -EPIPE;
			}


			if ((data->ports[port].port_status & (1 << USB_PORT_FEAT_RESET)) != 0
					/* && time_after (jiffies, dum->re_timeout) */) 
			{
				data->ports[port].port_status |= (1 << USB_PORT_FEAT_C_RESET);
				data->ports[port].port_status &= ~(1 << USB_PORT_FEAT_RESET);
				
				if (data->ports[port].dev !=NULL)
				{
					data->ports[port].port_status |= USB_PORT_STAT_ENABLE;
				}
				DEBUG_MSG(" enable rhport %d (status)\n", port);
				data->ports[port].port_status |= USB_PORT_STAT_ENABLE;
		
			}
			((__le16 *) buf)[0] = cpu_to_le16 (data->ports[port].port_status);
			((__le16 *) buf)[1] = cpu_to_le16 (data->ports[port].port_status >> 16);

			break;
		case SetHubFeature:
			DEBUG_MSG("unimplemented hub control request: SetHubFeature");
			ret = -EPIPE;
			break;
		case SetPortFeature:
			switch (wValue) {
				case USB_PORT_FEAT_SUSPEND:
		            DEBUG_MSG("unimplemented hub control request: SetPortFeature USB_PORT_FEAT_SUSPEND");
					break;
				case USB_PORT_FEAT_RESET:
					/* if it's already running, disconnect first */
					if (data->ports[port].port_status & USB_PORT_STAT_ENABLE) 
					{
						data->ports[port].port_status &= ~(USB_PORT_STAT_ENABLE
								| USB_PORT_STAT_LOW_SPEED
								| USB_PORT_STAT_HIGH_SPEED);

					}
				default:
				data->ports[port].port_status |= (1 << wValue);
					break;
			}
			break;
		

		default:	
			DEBUG_MSG("no such request.");
	}
	spin_unlock_irqrestore(&data->lock, flags);
	return ret;
}



static int ddeusb_vhcd_bus_suspend (struct usb_hcd * hcd) {
	WARN_UNIMPL;
	return 0;
}
static int ddeusb_vhcd_bus_resume (struct usb_hcd * hcd) {
	WARN_UNIMPL;
	return 0;
}
static int ddeusb_vhcd_start_port_reset(struct usb_hcd *hcd, unsigned port_num)
{
	WARN_UNIMPL;
	return 0;
}


static struct hc_driver ddeusb_vhcd_driver = {
	.description=driver_name,	/* "ehci-hcd" etc */
	.product_desc=driver_desc,	/* product/vendor string */
	.hcd_priv_size= sizeof(struct ddeusb_vhcd_priv_data),
	.flags = HCD_USB2,
	.reset = ddeusb_vhcd_reset,
	.start = ddeusb_vhcd_start,
	.shutdown = ddeusb_vhcd_shutdown,
	.get_frame_number = ddeusb_vhcd_get_frame_number,
	.urb_enqueue = ddeusb_vhcd_urb_enqueue,
	.urb_dequeue = ddeusb_vhcd_urb_dequeue,
	.endpoint_disable = ddeusb_vhcd_endpoint_disable,
	.hub_status_data = ddeusb_vhcd_hub_status_data,
	.hub_control = ddeusb_vhcd_hub_control,
	.bus_suspend = ddeusb_vhcd_bus_suspend,
	.bus_resume = ddeusb_vhcd_bus_resume,
	.start_port_reset = ddeusb_vhcd_start_port_reset,
};


static int ddeusb_vhcd_pd_probe(struct platform_device *pdev)
{
	int ret,err;
	struct usb_hcd* hcd;
	/* check if it is our device... */
	if(strcmp(pdev->name,driver_name)) {
		return -ENODEV;
	}
	/* create the hcd */
	hcd = usb_create_hcd(&ddeusb_vhcd_driver, &pdev->dev, pdev->dev.bus_id);
	if (!hcd) {
		DEBUG_MSG("create hcd failed\n");
		return -ENOMEM;
	}
	/* make it known to the world! */
	ret = usb_add_hcd(hcd, 0, 0);
	return ret;
}


static int ddeusb_vhcd_pd_remove(struct platform_device *pdev)
{
	WARN_UNIMPL;
	return 0;
}


static int ddeusb_vhcd_pd_suspend(struct platform_device *pdev,pm_message_t state)
{
	WARN_UNIMPL;
	return 0;
}


static int ddeusb_vhcd_pd_resume(struct platform_device *pdev)
{
    WARN_UNIMPL;
	return 0;
}


void ddeusb_vhcd_pd_release(struct device *dev)
{
    WARN_UNIMPL;
}


static struct platform_driver ddeusb_vhcd_pdriver ={
	.probe	= ddeusb_vhcd_pd_probe,
	.remove	= ddeusb_vhcd_pd_remove,
	.suspend = ddeusb_vhcd_pd_suspend,
	.resume	= ddeusb_vhcd_pd_resume,
	.driver	= {
		.name = (char *) driver_name,
		.owner = THIS_MODULE,
	},
};


static struct platform_device ddeusb_vhcd_pdev = {
	/* should be the same name as driver_name */
	.name = (char *) driver_name,
	.id = -1,
	.dev = {
		.release = ddeusb_vhcd_pd_release,
		.dma_mask = 0
	},
};


static struct  ddekit_usb_driver this_driver = 
{
	.connect    = ddeusb_register_device,
	.disconnect = ddeusb_disconnect_device,
	.completion = ddeusb_vhcd_urb_complete
};


static int ddeusb_thread(void *unused)
{
	printk("I'm the IPC thread");
    ddekit_usb_init(&this_driver, NULL, NULL);
}

 int __init ddeusb_vhcd_init(void)
{
	int ret;

	if (usb_disabled())
		return -ENODEV;

	/* creating cache for privs */
    priv_cache  = kmem_cache_create("privcache",
	    sizeof(struct ddeusb_vhcd_urbpriv),0,0,0);


    DEBUG_MSG("...registering platform device...");
	ret = platform_device_register(&ddeusb_vhcd_pdev);
	if (ret < 0)
		goto err_platform_device_register;

	DEBUG_MSG("...registering platform driver...");
	ret = platform_driver_register(&ddeusb_vhcd_pdriver);
	kthread_run(ddeusb_thread, NULL, "ddekit_usb_thread");
	if (ret < 0)
		goto err_driver_register;

return ret;
	/* error occurred */

err_driver_register:
err_platform_device_register:
	platform_device_unregister(&ddeusb_vhcd_pdev);
	return ret;
}

fs_initcall(ddeusb_vhcd_init);
