/*
 * Copyright (c) 2012 Emmanuel Dreyfus
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *        This product includes software developed by Emmanuel Dreyfus
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <getopt.h>
#include <err.h>
#include <errno.h>
#include <sysexits.h>
#include <time.h>

#include <sys/param.h>
#include <sys/stat.h>
#include <ufs/ufs/dinode.h>
#include <protocols/dumprestore.h>

#ifndef _PATH_DUMPDATES
#define     _PATH_DUMPDATES "/etc/dumpdates"
#endif

#ifndef NAME_MAX
#define NAME_MAX 255
#endif

#ifndef STATE_OK
#define STATE_OK        0
#define STATE_WARNING   1
#define STATE_CRITICAL  2
#define STATE_UNKNOWN   3
#endif

int
main(argc, argv)
	int argc;
	char **argv;
{
	char *pname;
	char *dumpdates = _PATH_DUMPDATES;
	char output[4096];
	struct stat st;
	FILE *fp;
	char *want_device = NULL;
	int device_failed = 0;
	int device_found = 0;
	int want_level = 0;
	int days_crit = 7;
	int days_warn = 3;
	time_t now, warn, crit;
	int ch;
	int retval = STATE_OK;

	pname = argv[0];

	while ((ch = getopt(argc, argv, "f:d:l:w:c:h")) != -1) {
		switch (ch) {
		case 'f':
			dumpdates = optarg;
			break;
		case 'd':
			want_device = optarg;
			break;
		case 'l':
			want_level = atoi(optarg);
			break;
		case 'w':
			days_warn = atoi(optarg);
			break;
		case 'c':
			days_crit = atoi(optarg);
			break;
		case 'h': /* FALLTHROUGH */
		default:
			err(EX_USAGE, "%s [-d device ] [-l level ] "
				      "[-w warn_days] [-c crit_days]", pname);
			/* NOTREACHED */
			break;
		}
	}

	now = time(NULL);
	warn = now - (days_warn * 24 * 3600);
	crit = now - (days_crit * 24 * 3600);

	if (stat(dumpdates, &st) != 0) {
		printf("FAILED: cannot open %s: %s",
		       dumpdates, strerror(errno));
		exit(STATE_UNKNOWN);
	}

	if (st.st_size == 0) {
		printf("FAILED: %s is empty", dumpdates);
		exit(STATE_UNKNOWN);
	}

	if ((fp = fopen(dumpdates, "r")) == NULL) {
		printf("FAILED: cannot open \"%s\": %s",
		       dumpdates, strerror(errno));
		exit(STATE_UNKNOWN);
	}

	while (!feof(fp)) {
		char timebuf[128];
		char *res;
		char name[NAME_MAX + 3];
		char level;
		time_t ddate;
		struct tm tm;

		(void)fscanf(fp, DUMPINFMT, name, &level, timebuf);
		level -= '0';

		res = strptime(timebuf, "%a %b %e %H:%M:%S %Y", &tm);
		if (res == NULL || (*res != '\n' && *res != '\0')) {
			printf("FAILED; cannot parse date \"%s\"",
			    timebuf);
			exit(STATE_UNKNOWN);
		}

		ddate = mktime(&tm);
		
		if ((want_device != NULL) && (strcmp(want_device, name) != 0))
			continue;

		if (level > want_level)
			continue;

		device_found++;

		if (ddate > now) {
			printf("FAILED: %s level %d dump is in the future ",
			       name, level);
			device_failed++;
		} else if (ddate < crit) {
			printf("FAILED: %s level %d dump is %d days old ",
			       name, level, (int)((now - ddate) / 86400));
			device_failed++;
			retval = STATE_CRITICAL;
		} else if (ddate < warn) {
			printf("WARNING: %s level %d dump is %d days old ",
			       name, level, (int)((now - ddate) / 86400));
			device_failed++;
			if (retval != STATE_CRITICAL)
				retval = STATE_WARNING;
		} else {
			snprintf(output, sizeof(output), 
				 "%s%s: level %d backup %d days ago ", 
				 output, name,level, 
				(int)((now - ddate) / 86400));
		}

	}

	(void)fclose(fp);

	
	if (device_found == 0) {
		printf("FAILED: no backup found\n");
		return STATE_CRITICAL;
	} else if (device_failed == 0) {
		printf("OK %s\n", output);
		retval = STATE_OK;
	}

	printf("\n%s", output);

	return retval;
}
