/*
 *  Copyright (C) 1999,2001,2004,2005,2007  Anders Gavare.  All rights reserved.
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions are met:
 *
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *  3. The name of the author may not be used to endorse or promote products
 *     derived from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 *  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 *  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *  ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 *  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 *  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 *  OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 *  HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 *  LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 *  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 *  SUCH DAMAGE.
 *
 *
 *  $Id: relative.c,v 1.4 2005/03/03 20:40:15 debug Exp $
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>


/*
 *  relative():
 *
 *  This function gives a relative path to a file.
 *
 *  Example:   src = "sys/kern/init_main.c"
 *             dst = "sys/sys/syslimits.h"
 *  Result:    "../sys/syslimits.h"
 *
 *  How to do this:
 *
 *  1)	Substitute any double-slashed to single
 *	ie sys//kern/ ==> sys/kern/
 *
 *  2)	Find the longest posible starting substring
 *	match   (ie  sys/)  (must be empty, or end with a slash)
 *
 *  3)	Calculate nr of levels we need to go down from
 *	src to reach dst.  If Ns = nr of slashes in src
 *	and Nm = nr of slashes in the matched substring
 *	from step 2, then the nr of levels to go down
 *	is Ns - Nm.
 *
 *  4)	The result is composed of a number of "../" strings
 *	and the rest of the dst (the part after the matched
 *	substring)
 *
 *  NOTE:  *reslen should be set to the length of the character
 *	   buffer at result. At most *reslen - 1 bytes will be
 *	   written to the result buffer.
 *
 *  Returns 0 on success, -1 on error. *reslen is set to the
 *  actual strlen of the result string (not valid if there was
 *  an error).
 */
int relative(char *orig_src, char *orig_dst, char *result, int *reslen)
{
	int Ns, Nm, Ndown;
	int reslen_max;
	int srclen, dstlen;
	int i, matchlen, done;
	int baselen;
	char *src = NULL;
	char *dst = NULL;
	char *tmp;

	if (reslen == NULL || orig_src == NULL ||
	    orig_dst == NULL || result == NULL)
		return -1;

	if ((orig_src[0] == '/' && orig_dst[0] != '/') ||
	    (orig_src[0] != '/' && orig_dst[0] == '/')) {
		fprintf(stderr, "relative(): src='%s', dst='%s'; if one is ab"
		    "solute, the other needs to be absolute too!\n", src, dst);
		goto error;
	}

	reslen_max = *reslen;

	srclen = strlen(orig_src);
	dstlen = strlen(orig_dst);

	src = (char *) malloc(srclen+1);
	if (src == NULL)
		goto error;
	dst = (char *) malloc(dstlen+1);
	if (dst == NULL)
		goto error;
	strcpy(src, orig_src);
	strcpy(dst, orig_dst);

	for (i=0; i<srclen-1; i++)
		if (src[i] == '/' && src[i+1] == '/') {
			memcpy(src+i, src+i+1, srclen-i+1);
			srclen --;
			i --;
		}

    for (i=0; i<dstlen-1; i++)
      if (dst[i]=='/' && dst[i+1]=='/')
	memcpy (dst+i, dst+i+1, dstlen-i+1), dstlen--, i--;

    /*  Recache string lengths:  */
    srclen = strlen (src);
    dstlen = strlen (dst);

    /*  Find longest match:  */
    matchlen = 1;
    done = 0;
    while (!done)
      {
	/*  Try this matchlen:  */
	if (!strncmp(src, dst, matchlen))
	  matchlen++;
	else
	  {
	    /*  This matchlen didn't work. Decrease by one,
		and we're done.  */
	    matchlen --;
	    done = 1;
	  }

	if (matchlen > srclen || matchlen > dstlen)
	  done=1, matchlen--;
      }

/*
printf ("src='%s' dst='%s'\n", src, dst);
printf ("matchlen=%i\n", matchlen);
*/

    /*  Find last slash (withing the first matchlen bytes) of
	the matched part:  */

    strlcpy (result, src, reslen_max);
    result[matchlen] = 0;

/*  printf ("result='%s'\n", result);  */

    tmp = strrchr (result, '/');
    if (!tmp)
      tmp = result - 1;

    baselen = (int) (tmp - result) + 1;

/*  printf ("baselen=%i\n", baselen);  */

    result[baselen] = 0;

    /*  Count nr of slashes in base:  */
    Nm = 0;
    for (i=0; i<baselen; i++)
      if (result[i]=='/')
	Nm ++;

    /*  Count nr of slashes in src:  */
    Ns = 0;
    for (i=strlen(src)-1; i>=0; i--)
      if (result[i]=='/')
	Ns ++;

/*
printf ("Nm=%i, Ns=%i\n", Nm, Ns);
*/

    Ndown = Ns - Nm;

    result[0] = 0;
    for (i=0; i<Ndown; i++)
      strlcat (result, "../", reslen_max);

    strlcat (result, dst+baselen, reslen_max);

/*
printf ("result='%s'\n", result);
*/

    if (src)      free (src);
    if (dst)      free (dst);
    (*reslen) = strlen (result);
    return 0;

error:
    if (src)      free (src);
    if (dst)      free (dst);
    return -1;
  }

