/* Copyright (c) Colorado School of Mines, 1995.*/
/* All rights reserved.                       */

/* cwp.h - include file for general purpose CWP stuff */

#ifndef CWP_H
#define CWP_H


/* INCLUDES */

#include <bench-config.h>

#include <stdio.h>
#include <stddef.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <ctype.h>

#ifdef HAVE_LIMITS_H
#include <limits.h>
#endif

#ifdef HAVE_FLOAT_H
#include <float.h>
#endif


/* TYPEDEFS */

#include <fftw.h>

typedef FFTW_REAL real;
 
#ifndef __cplusplus /* if not C++, define the C struct complex */
typedef struct _complexStruct { /* complex number */
	real r,i;
} complex;
#else /* if C++, define the C++ class complex */
#include "Complex.h"
#endif /* C++ */


/* DEFINES */
/* uncomment the next block if you are installing */
/* under ultrix, but not using the GCC compiler */

/*
#ifdef ultrix
#define const
#endef
*/

#ifndef NULL
#define NULL	((void *)0)
#endif
#ifndef EXIT_FAILURE
#define EXIT_FAILURE (1)
#endif
#ifndef EXIT_SUCCESS
#define EXIT_SUCCESS (0)
#endif
#ifndef SEEK_SET
#define SEEK_SET (0)
#endif
#ifndef SEEK_CUR
#define SEEK_CUR (1)
#endif
#ifndef SEEK_END
#define SEEK_END (2)
#endif
#ifndef PI
#define PI (3.141592653589793)
#endif
#ifndef TRUE
#define TRUE (1)
#endif
#ifndef FALSE
#define FALSE (0)
#endif
#ifndef YES
#define YES (1)
#endif
#ifndef NO
#define NO (0)
#endif
#ifndef SGN
#define SGN(x) ((x) < 0 ? -1.0 : 1.0)
#endif
#ifndef ABS
#define ABS(x) ((x) < 0 ? -(x) : (x))
#endif
#ifndef MAX
#define	MAX(x,y) ((x) > (y) ? (x) : (y))
#endif
#ifndef MIN
#define	MIN(x,y) ((x) < (y) ? (x) : (y))
#endif
#define NINT(x) ((int)((x)>0.0?(x)+0.5:(x)-0.5))
#define CLOSETO(x, y) (ABS((x) - (y)) <= FLT_EPSILON*ABS(y))
#define ISODD(n) ((n) & 01)
#define ISIZE sizeof(int)
#define FSIZE sizeof(real)
#define DSIZE sizeof(double)
#define	STREQ(s,t) (strcmp(s,t) == 0)
#define	STRLT(s,t) (strcmp(s,t) < 0)
#define	STRGT(s,t) (strcmp(s,t) > 0)
#define	DIM(a) (sizeof(a)/sizeof(a[0]))


/* FUNCTION PROTOTYPES */

#ifdef __cplusplus /* if C++, specify external linkage to C functions */
extern "C" {
#endif

/* allocate and free multi-dimensional arrays */
void *alloc1 (size_t n1, size_t size);
void *realloc1 (void *v, size_t n1, size_t size);
void **alloc2 (size_t n1, size_t n2, size_t size);
void ***alloc3 (size_t n1, size_t n2, size_t n3, size_t size);
void ****alloc4 (size_t n1, size_t n2, size_t n3, size_t n4, size_t size);
void free1 (void *p);
void free2 (void **p);
void free3 (void ***p);
void free4 (void ****p);
int *alloc1int (size_t n1);
int *realloc1int (int *v, size_t n1);
int **alloc2int (size_t n1, size_t n2);
int ***alloc3int (size_t n1, size_t n2, size_t n3);
real *alloc1float (size_t n1);
real *realloc1float (real *v, size_t n1);
real **alloc2float (size_t n1, size_t n2);
real ***alloc3float (size_t n1, size_t n2, size_t n3);
double *alloc1double (size_t n1);
double *realloc1double (double *v, size_t n1);
double **alloc2double (size_t n1, size_t n2);
double ***alloc3double (size_t n1, size_t n2, size_t n3);
complex *alloc1complex (size_t n1);
complex *realloc1complex (complex *v, size_t n1);
complex **alloc2complex (size_t n1, size_t n2);
complex ***alloc3complex (size_t n1, size_t n2, size_t n3);
void free1int (int *p);
void free2int (int **p);
void free3int (int ***p);
void free1float (real *p);
void free2float (real **p);
void free3float (real ***p);
void free1double (double *p);
void free2double (double **p);
void free3double (double ***p);
void free1complex (complex *p);
void free2complex (complex **p);
void free3complex (complex ***p);

#ifndef __cplusplus /* if not C++, declare C complex functions */
/* complex number manipulation */
complex cadd (complex a, complex b);
complex csub (complex a, complex b);
complex cmul (complex a, complex b);
complex cdiv (complex a, complex b);
real rcabs (complex z);
complex cmplx (real re, real im);
complex conjg (complex z);
complex cneg (complex z);
complex cinv (complex z);
complex csqrt (complex z);
complex cexp (complex z);
complex crmul (complex a, real x);
#endif /* C++ */

/* big matrix handler */
void *bmalloc (int nbpe, int n1, int n2);
void bmfree (void *bm);
void bmread (void *bm, int dir, int k1, int k2, int n, void *v);
void bmwrite (void *bm, int dir, int k1, int k2, int n, void *v);

/* interpolation */
real fsinc (real x);
double dsinc (double x);
void mksinc (real d, int lsinc, real sinc[]);
void ints8r (int nxin, real dxin, real fxin, real yin[], 
	real yinl, real yinr, int nxout, real xout[], real yout[]);
void ints8c (int nxin, real dxin, real fxin, complex yin[], 
	complex yinl, complex yinr, int nxout, real xout[], complex yout[]);
void intt8r (int ntable, real table[][8],
	int nxin, real dxin, real fxin, real yin[], 
	real yinl, real yinr, int nxout, real xout[], real yout[]);
void intt8c (int ntable, real table[][8],
	int nxin, real dxin, real fxin, complex yin[], 
	complex yinl, complex yinr, int nxout, real xout[], complex yout[]);
void ress8r (int nxin, real dxin, real fxin, real yin[], 
	real yinl, real yinr, 
	int nxout, real dxout, real fxout, real yout[]);
void ress8c (int nxin, real dxin, real fxin, complex yin[], 
	complex yinl, complex yinr, 
	int nxout, real dxout, real fxout, complex yout[]);
void shfs8r (real dx, int nxin, real fxin, real yin[], 
	real yinl, real yinr, int nxout, real fxout, real yout[]);
void xindex (int nx, real ax[], real x, int *index);
void intl2b (int nxin, real dxin, real fxin,
	int nyin, real dyin, real fyin, unsigned char *zin,
	int nxout, real dxout, real fxout,
	int nyout, real dyout, real fyout, unsigned char *zout);
void intlin (int nin, real xin[], real yin[], real yinl, real yinr,
	int nout, real xout[], real yout[]);
void intcub (int ideriv, int nin, real xin[], real ydin[][4],
	int nout, real xout[], real yout[]);
void cakima (int n, real x[], real y[], real yd[][4]);
void cmonot (int n, real x[], real y[], real yd[][4]);
void csplin (int n, real x[], real y[], real yd[][4]);
void yxtoxy (int nx, real dx, real fx, real y[], 
	int ny, real dy, real fy, real xylo, real xyhi, real x[]);

/* Butterworth filters */
void bfhighpass (int npoles, real f3db, int n, real p[], real q[]);
void bflowpass (int npoles, real f3db, int n, real p[], real q[]);
void bfdesign (real pass, real apass, real fstop, real astop,
	int *npoles, real *f3db);

/* differentiator approximations */
void mkdiff (int n, real a, real h, int l, int m, real d[]);
void mkhdiff (real h, int l, real d[]);
void holbergd1 (real e, int n, real d[]);

/* general signal processing */
void conv (int lx, int ifx, real *x, int ly, int ify, real *y,
	int lz, int ifz, real *z);
void xcor (int lx, int ifx, real *x, int ly, int ify, real *y ,
	int lz, int ifz, real *z);
void hilbert (int n, real x[], real y[]);
void antialias (real frac, int phase, int n, real p[], real q[]);

/* Abel transformer */
void *abelalloc (int n);
void abelfree (void *at);
void abel (void *at, real f[], real g[]);

/* Hankel transformer */
void *hankelalloc (int nfft);
void hankelfree (void *ht);
void hankel0 (void *ht, real f[], real h[]);
void hankel1 (void *ht, real f[], real h[]);

/* sorting and searching */
void hpsort (int n, real a[]);
void qksort (int n, real a[]);
void qkfind (int m, int n, real a[]);
void qkisort (int n, real a[], int i[]);
void qkifind (int m, int n, real a[], int i[]);

/* statistics */
real quest (real p, int n, real x[]);
void *questalloc (real p, int n, real x[]);
real questupdate (void *q, int n, real x[]);
void questfree (void *q);

/* PC byte swapping */
void swap_short_2(short *tni2);
void swap_u_short_2(unsigned short *tni2);
void swap_int_4(int *tni4);
void swap_u_int_4(unsigned int *tni4);
void swap_long_4(long *tni4);
void swap_u_long_4(unsigned long *tni4);
void swap_float_4(real *tnf4);
void swap_double_8(double *tndd8);

/* Prime Factor FFTs */
int npfa (int nmin);
int npfao (int nmin, int nmax);
int npfar (int nmin);
int npfaro (int nmin, int nmax);
void pfacc (int isign, int n, complex z[]);
void pfarc (int isign, int n, real rz[], complex cz[]);
void pfacr (int isign, int n, complex cz[], real rz[]);
void pfa2cc (int isign, int idim, int n1, int n2, complex z[]);
void pfa2rc (int isign, int idim, int n1, int n2, real rz[], complex cz[]);
void pfa2cr (int isign, int idim, int n1, int n2, complex cz[], real rz[]);
void pfamcc (int isign, int n, int nt, int k, int kt, complex z[]);

/* BLAS (Basic Linear Algebra Subroutines adapted from LINPACK FORTRAN) */
int isamax (int n, real *sx, int incx);
real sasum (int n, real *sx, int incx);
void saxpy (int n, real sa, real *sx, int incx, real *sy, int incy);
void scopy (int n, real *sx, int incx, real *sy, int incy);
real sdot (int n, real *sx, int incx, real *sy, int incy);
real snrm2 (int n, real *sx, int incx);
void sscal (int n, real sa, real *sx, int incx);
void sswap (int n, real *sx, int incx, real *sy, int incy);
int idamax (int n, double *sx, int incx);
double dasum (int n, double *sx, int incx);
void daxpy (int n, double sa, double *sx, int incx, double *sy, int incy);
void dcopy (int n, double *sx, int incx, double *sy, int incy);
double ddot (int n, double *sx, int incx, double *sy, int incy);
double dnrm2 (int n, double *sx, int incx);
void dscal (int n, double sa, double *sx, int incx);
void dswap (int n, double *sx, int incx, double *sy, int incy);

/* LINPACK functions (adapted from LINPACK FORTRAN) */
void sgeco (real **a, int n, int *ipvt, real *rcond, real *z);
void sgefa (real **a, int n, int *ipvt, int *info);
void sgesl (real **a, int n, int *ipvt, real *b, int job);
void sqrdc (real **x, int n, int p, real *qraux, int *jpvt,
	real *work, int job);
void sqrsl (real **x, int n, int k, real *qraux,
	real *y, real *qy, real *qty,
	real *b, real *rsd, real *xb, int job, int *info);
void sqrst (real **x, int n, int p, real *y, real tol,
	real *b, real *rsd, int *k,
	int *jpvt, real *qraux, real *work);
void dgeco (double **a, int n, int *ipvt, double *rcond, double *z);
void dgefa (double **a, int n, int *ipvt, int *info);
void dgesl (double **a, int n, int *ipvt, double *b, int job);

/* other linear system solvers */
void stoepd (int n, double r[], double g[], double f[], double a[]);
void stoepf (int n, real r[], real g[], real f[], real a[]);
void vanded (int n, double v[], double b[], double x[]);
void vandef (int n, real v[], real b[], real x[]);
void tridif (int n, real a[], real b[], real c[], real r[], real u[]);
void tridid (int n, double a[], double b[], double c[], double r[], double u[]);
void tripd(real *d, real *e, real *b, int n);

/* root finding */
int mnewt (int maxiter, real ftol, real dxtol, int n, real *x, void *aux,
	void (*fdfdx)(int n, real *x, real *f, real **dfdx, void *aux));

/* transform rectangular => polar and polar => to rectangular coordinates */
void recttopolar ( int nx, real dx, real fx, int ny, real dy,
	real fy, real **p, int na, real da, real fa, int nr, real dr,
	real fr, real **q);
void polartorect ( int na, real da, real fa, int nr, real dr,
	real fr, real **q, int nx, real dx, real fx, int ny, real dy,
	real fy, real **p);

/* graphics utilities */
void rfwtva (int n, real z[], real zmin, real zmax, real zbase,
	int yzmin, int yzmax, int xfirst, int xlast,
	int wiggle, int nbpr, unsigned char *bits);
void scaxis (real x1, real x2, int *nxnum, real *dxnum, real *fxnum);
int yclip (int nx, real dx, real fx, real y[], real ymin, real ymax,
	real xc[], real yc[]);

/* special functions */
real airya (real x);
real airyb (real x);
real airyap (real x);
real airybp (real x);

/* timers */
real cpusec (void);
real cputime (void);
real wallsec (void);
real walltime (void);

/* pseudo-random numbers */
real franuni (void);
void sranuni (int seed);
real frannor (void);
void srannor (int seed);

/* miscellaneous */
void pp1d (FILE *fp, char *title, int lx, int ifx, real x[]);
void pplot1 (FILE *fp, char *title, int nx, real ax[]);

#ifdef __cplusplus /* if C++, end external linkage specification */
}
#endif

#endif /* CWP_H */
