/******************************************************************************
This file is part of the argtable library. It contains a simple example program
showing how to use the argtable library to process command line arguments.

Copyright (C) 1998, 1999 Stewart Heitmann (Stewart.Heitmann@tip.csiro.au)

The argtable library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public License as
published by the Free Software Foundation; either version 2 of the
License, or (at your option) any later version.

This software is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, 
USA.

How it works
------------
When called without any command line arguments, this program will just 
display some help text about the command line usage and exit. Otherwise 
it processes the arguments, checking them for correctness. Any errors are
displayed to stdout and cause the program to terminate with error code 1.
If no errors are detected, the scanned argument values are printed to
stdout.
The -verbose flag can be used to toggle the verbosity of the output.
******************************************************************************/
#include "argtable.h"

/*-- the values of the command line args will be stored in these variables --*/
int x,y,z,verbose,debug;
double radius;
char infname[100];
char outfname[100];


int main(int argc, char**argv)
  {
  arg_rec argtable[] =
    {
    /*-tag---------name---------type-----storage---default---descrip --*/
    { NULL,        "x",         arg_int,  &x,       NULL,    "\t\t is the x coord" },
    { NULL,        "y",         arg_int,  &y,       NULL,    "\t\t is the y coord" },
    { NULL,        "z",         arg_int,  &z,       "0",     "\t\t is the z coord (default=0)" },
    { "-r ",       NULL,        arg_dbl,  &radius,  "1.0",   "\t is the radius (default=1.0)" },
    { "-o ",       "<outfile>", arg_str,  outfname, "-",     "\t is the output file, use \"-\" for stdout (default=\"-\")" },
    { "-verbose",  NULL,        arg_lit,  &verbose, "0",     "\t specifies verbose output" },
    { NULL,       "<infile>",   arg_str,  infname,  NULL,    "\t is the input file, use \"-\" for stdin" },
    { "debug=",   "<on/off>",   arg_bool, &debug,   "off",   NULL },
    };
  const size_t narg = sizeof(argtable)/sizeof(arg_rec);

  /*-- process the command line args --*/
  if (argc==1)
     {
     /*-- no command line arguments, so display program usage and exit. --*/
     printf("This program parses the values of the command line arguments,\n"
            "prints the given values, and then exits.\n");
     printf("Usage: %s %s\n",argv[0],arg_syntax(argtable,narg));
     printf("%s\n",arg_glossary(argtable,narg,"   "));
     return 0;
     }
  else
     {
     /*-- scan command line arguments from argv[] as per argtable[] specs --*/
     char cmdline[200], errmsg[200], errmark[200]; 
     if (!arg_scanargv(argc,argv,argtable,narg,cmdline,errmsg,errmark))
	{
	/*-- scan error occurred, print error message and exit --*/
        printf("ERROR: %s\n", cmdline);
        printf("       %s %s\n", errmark, errmsg);
        return 1;
	}
     }


  /*-- display the args in one of two forms --*/
  if (verbose)
     /*-- print argument values referenced by argtable[] --*/
     arg_dump(stdout,argtable,narg);
  else
     /*-- print arguments in an abbreviated form --*/
     printf("x=%d\ny=%d\nz=%d\nradius=%f\noutfname=\"%s\"\ninfname=\"%s\"\ndebug=%d\n",
             x,y,z,radius,outfname,infname,debug);

  return 0; 
  }


