/*************************************************************************

   Program:    ProFit
   File:       main.c
   
   Version:    V2.2
   Date:       20.12.01
   Function:   Protein Fitting program. Main routines.
   
   Copyright:  SciTech Software 1992-2001
   Author:     Dr. Andrew C. R. Martin
   Address:    SciTech Software
               23, Stag Leys,
               Ashtead,
               Surrey,
               KT21 2TD.
   Phone:      +44 (0)1372 275775
   EMail:      andrew@bioinf.org.uk
               
**************************************************************************

   This program is not in the public domain.

   It may not be copied or made available to third parties, but may be
   freely used by non-profit-making organisations who have obtained it
   directly from the author or by FTP.

   You are requested to send EMail to the author to say that you are 
   using this code so that you may be informed of future updates.

   The code may not be made available on other FTP sites without express
   permission from the author.

   The code may be modified as required, but any modifications must be
   documented so that the person responsible can be identified. If
   someone else breaks this code, the author doesn't want to be blamed
   for code that does not work! You may not distribute any
   modifications, but are encouraged to send them to the author so
   that they may be incorporated into future versions of the code.

   Such modifications become the property of Dr. Andrew C.R. Martin and
   SciTech Software though their origin will be acknowledged.

   The code may not be sold commercially or used for commercial purposes
   without prior permission from the author.
   
**************************************************************************

   Description:
   ============

**************************************************************************

   Usage:
   ======

**************************************************************************

   Revision History:
   =================
   V0.1  25.09.92 Original
   V0.2  02.10.92 Added CURSES support
   V0.3  07.10.92 Added Amiga windows and paging support
   V0.4  09.10.92 Added N&W alignment support & fixed bug in multi-zones
   V0.5  08.10.93 Various tidying for Unix & chaned for booklib 
   V0.6  05.01.94 Reads HELPDIR and DATADIR environment variables under
                  Unix
   V0.7  24.11.94 Uses ReadPDBAtoms()
   V0.8  17.07.95 Stripped out the windowing support (never used!). Will
                  add a Tcl/Tk interface later!
                  Multiple chains now work correctly.
   V1.0  18.07.95 Insert codes now work.
                  First official release (at last!).
   V1.1  20.07.95 Added WEIGHT command support and translation vector
                  output from MATRIX command
   V1.1a 22.07.95 Stop crash on writing when not fitted
   V1.2  25.07.95 Added GAPPEN command
                  Added chain label printing in Status
                  Added optional filename parameter to RESIDUE command
   V1.3  31.07.95 Fixed bug in fitting.c where end of zone=end of chain
                  other than the last chain
   V1.4  14.08.95 Fixed bug in fitting.c with RESIDUE command was not
                  printing RMS for last residue
   V1.5  21.08.95 Fixed bug in NWAlign.c (when last zone not at end of
                  chain) and Bioplib library bug in align()
   V1.5a 02.10.95 Added printing of CofG information with MATRIX command
   V1.5b 15.11.95 NWAlign.c: Prints normalised score
   V1.6  20.11.95 Added READALIGNMENT command and code
   V1.6a 21.11.95 Fixed a couple of warnings under gcc
   V1.6b 22.11.95 Modified code in SetNWZones() such that deletions at
                  the same position in both sequences don't cause a 
                  problem.
   V1.6c 13.12.95 Further fix to double deletions. Added info when
                  zones mismatch in fitting.c
   V1.6d 24.01.96 Fixed bug in status command when printing atom names
                  containing spaces.
   V1.6e 31.05.96 Added BVAL command and code
   V1.6f 13.06.96 Added BWEIGHT command and code
   V1.6g 18.06.96 Replaced MODE_* with ZONE_MODE_* and use FindZonePDB()
                  from bioplib rather than own version
   V1.7  23.07.96 Supports atom wildcards. Some comment tidying.
   V1.7b 11.11.96 Added REF keyword to end of BVAL command allowing
                  only the reference structure to be considered
   V1.7c 18.11.96 Added IGNOREMISSING option
   V1.7d 20.12.96 Added NFITTED command
   V1.7e 27.06.97 Allows WRITE and RESIDUE to output to a pipe
   V1.7f 03.07.97 Added break into CreateFitArrays() to fix core dump
                  on bad multiple-occupancy PDB files
   V1.7g 06.05.98 Rewrite of NWAlign/SetNWZones()
   V1.8  07.05.98 Skipped for release
   V2.0  01.03.01 Now supports multiple structure fitting and iterative
                  zone updating
   V2.1  28.03.01 Parameter for ITERATE and added CENTRE command
   V2.2  20.02.01 Fixed some Bioplib problems related to raw atom names
                  and multiple occupancies

*************************************************************************/
#define MAIN 
#include "ProFit.h"


/************************************************************************/
/*>void logo(void)
   ---------------
   Displays the ProFit logo.
   
   25.09.92 Original
   01.10.92 Added version
   10.10.93 Version 0.5
   05.01.94 Version 0.6
   24.11.94 Version 0.7
   17.07.95 Version 0.8; removed screen() version
   18.07.95 Version 1.0
   20.07.95 Version 1.1
   22.07.95 Version 1.2
   31.07.95 Version 1.3
   14.08.95 Version 1.4
   21.08.95 Version 1.5
   20.11.95 Version 1.6
   23.07.96 Version 1.7
   07.05.98 Version 1.8
   15.02.01 Version 2.0
   20.03.01 Version 2.1
   20.12.01 Version 2.2
*/
void logo(void)
{
   printf("\n                  PPPPP                FFFFFF ii   tt\n");
   printf("                  PP  PP               FF          tt\n");
   printf("                  PP  PP rrrrr   oooo  FF     ii  ttttt\n");
   printf("                  PPPPP  rr  rr oo  oo FFFF   ii   tt\n");
   printf("                  PP     rr     oo  oo FF     ii   tt\n");
   printf("                  PP     rr     oo  oo FF     ii   tt\n");
   printf("                  PP     rr      oooo  FF      ii   ttt\n\n");
   printf("                      Protein Least Squares Fitting\n\n");
   printf("                               Version 2.2\n\n");
   printf("      Copyright (c) Dr. Andrew C.R. Martin, SciTech Software \
1992-2001\n\n");
}


/************************************************************************/
/*>int main(int  argc, char **argv)
   --------------------------------
   The main program. Checks parameters, sets some defaults, reads files if
   specified, initialises command parser and starts the command loop.
   
   25.09.92 Original
   29.09.92 Added gCurrentMode, Changed to call ReadStructure()
   02.10.92 Calls Cleanup() as it should have done!
   05.10.92 Added AMIGA_WINDOWS support
   06.10.92 Changed argc check for icon start
   08.10.93 Changed version number
            Chaned CURSES calls for book library
   17.07.95 Removed windowing support
   19.07.95 Handles -h flag
   20.07.95 Initialise gDoWeights
   13.06.96 Changed initialisation of gDoWeights since it now has 3 states
   18.06.96 Replaced MODE_* with ZONE_MODE_*
   01.03.01 Added -x handling
*/
int main(int  argc, char **argv)
{
   BOOL CmdError = FALSE;
   BOOL xmasFormat = FALSE;
   int  i;
   
   logo();

   argc--;
   argv++;

   /* See if a flag has been given                                      */
   while(argc>0 && argv[0][0] == '-')
   {
      switch(argv[0][1])
      {
      case 'h':
         gHetAtoms = TRUE;
         break;
      case 'x':
#ifdef USE_XMAS
         xmasFormat = TRUE;
#else
         fprintf(stderr,"Error: XMAS support not compiled in this \
version\n");
         return(1);
#endif
         break;
      default:
         CmdError = TRUE;
         break;
      }
      argc--;
      argv++;
   }
   

   if(CmdError || (argc>0 && argc!=2))
   {
      printf("\n\nSyntax: ProFit [-h] [<reference.pdb> <mobile.pdb>]\n");
      printf("        -h Include HETATM records when reading PDB \
files\n\n");
      
      exit(1);
   }
   
   /* Initialise various things                                         */
   strcpy(gFitAtoms[0],"*");
   strcpy(gRMSAtoms[0],"*");
   gRefFilename[0] = '\0';
   gCurrentMode     = ZONE_MODE_RESNUM;
   gUserRMSAtoms    = FALSE;
   gUserRMSZone     = FALSE;
   gUserFitZone     = FALSE;
   gFitted          = FALSE;
   gDoWeights       = WEIGHT_NONE;
   for(i=0; i<MAXSTRUC; i++)
   {
      gMobPDB[i]         = NULL;
      gFitPDB[i]         = NULL;
      gMobCoor[i]        = NULL;
      gMobFilename[i][0] = '\0';
      gZoneList[i]       = NULL;
      gRZoneList[i]      = NULL;
   }
   gLimit[0] = gLimit[1] = (-1);
   
   if(argc==2)
   {
      /* Filenames have been specified, so open and read files          */
      ReadStructure(STRUC_REFERENCE, argv[0], 0, xmasFormat);
      ReadStructure(STRUC_MOBILE,    argv[1], 0, xmasFormat);
      gMultiCount = 1;
   }
   
   InitParser();

   DoCommandLoop();
   
   Cleanup();

   return(0);
}


/************************************************************************/
/*>void SetRMSAtoms(char *command)
   -------------------------------
   Responds to a command to set the atoms for calculation of RMS

   28.09.92 Original.
   29.09.92 Added gUserRMSAtoms
   30.09.92 Added NOT option; pad gRMSAtoms[]
   17.07.95 Replaced calls to screen() with printf()
   19.07.95 Added parameter to ShowRMS()
   25.07.95 Added another parameter to ShowRMS()
   23.07.96 Changed to call PADMINTERM() rather than padterm()
            Checks for legal wildcard specifications
            Improved out-of-bounds error checking
*/
void SetRMSAtoms(char *command)
{
   int   comptr,
         atmnum,
         atmpos,
         comstart;
   char  *cmnd;
   
   /* Put to upper case                                                 */
   UPPER(command);
   
   /* Assume this is not a NOT selection                                */
   gNOTRMSAtoms = FALSE;
   cmnd        = command;
   
   /* Set NOT flag if required and step over the symbol                 */
   if(command[0] == '~' || command[0] == '^')
   {
      gNOTRMSAtoms = TRUE;
      cmnd++;
   }

   /* Blank all rmsatoms                                                */
   for(atmnum=0; atmnum<NUMTYPES; atmnum++)
      for(atmpos=0; atmpos<8; atmpos++)
         gRMSAtoms[atmnum][atmpos] = '\0';
   
   atmnum=0;
   atmpos=0;
   comstart=0;
   for(comptr=0;comptr<strlen(cmnd);comptr++)
   {
      if(cmnd[comptr]==',')
      {
         comstart=comptr+1;
         gRMSAtoms[atmnum][atmpos]   = '\0';  /* Terminate string       */
         PADMINTERM(gRMSAtoms[atmnum],4);
         if(++atmnum >= NUMTYPES)
         {
            if(!gQuiet)
            {
               printf("   Warning==> Too many atoms in specification. \
Only %d used\n",NUMTYPES);
            }
            break;
         }
         atmpos=0;
      }
      else
      {
         if(atmpos >= MAXATSPEC)
         {
            TERMAT(cmnd+comptr, ',');
            if(!gQuiet)
            {
               printf("   Warning==> Atom name specification too long \
(%s)\n", cmnd+comstart);
               printf("              Using all atoms.\n");
            }
            gRMSAtoms[0][0] = '*';
            gRMSAtoms[0][1] = '\0';
            break;
         }
         gRMSAtoms[atmnum][atmpos++] = cmnd[comptr];
      }
   }
   
   /* Terminate last one                                                */
   gRMSAtoms[atmnum][atmpos] = '\0';
   PADMINTERM(gRMSAtoms[atmnum],4);

   /* See if a * was specified not in first position; move it if so.
      Also check for legal atom wildcard specifications
   */
   for(atmnum=0; atmnum<NUMTYPES; atmnum++)
   {
      if(!LegalAtomSpec(gRMSAtoms[atmnum]))
      {
         if(!gQuiet)
         {
            printf("   Warning==> Illegal atom specification (%s). Using \
all atoms.\n",
                   gRMSAtoms[atmnum]);
         }
         gRMSAtoms[0][0] = '*';
         gRMSAtoms[0][1] = '\0';
         break;
      }
      
      if(gRMSAtoms[atmnum][0] == '*')
      {
         if(gNOTRMSAtoms)
         {
            if(!gQuiet)
            {
               printf("   Warning==> NOT option ignored.\n");
            }
            gNOTRMSAtoms = FALSE;
         }
         gRMSAtoms[0][0] = '*';
         gRMSAtoms[0][1] = '\0';
         break;
      }
   }
   
   gUserRMSAtoms = TRUE;
   
   ShowRMS(FALSE,NULL,0,FALSE);

   return;
}


/************************************************************************/
/*>void SetRMSZone(char *command)
   ------------------------------
   Responds to a command to set the zone for RMS calculation

   29.09.92 Original. 
   17.07.95 Replaced calls to screen() with printf()
   18.07.95 Added initialisation of inserts in zones
   19.07.95 Added parameter to ShowRMS()
            * on its own equivalent to CLEAR
   25.07.95 Added another parameter to ShowRMS()
   18.06.96 Replaced MODE_* with ZONE_MODE_*
   01.02.01 Added multiple structures
   28.02.01 Error message from ParseZone() when trying to specify multiple
            zones for multi-structure now here instead of in ParseZone()
*/
void SetRMSZone(char *command)
{
   int   start1,  stop1,
         start2,  stop2,
         SeqZone, strucnum;
   char  chain1,  chain2,
         startinsert1, stopinsert1,
         startinsert2, stopinsert2;
   ZONE  *z;
   int   warned = 0;

   
   /* See if this is clearing the zones                                 */
   UPPER(command);
   
   if(!strncmp(command,"CLEAR",5) || !strcmp(command,"*"))
   {
      gUserRMSZone = FALSE;
      
      for(strucnum=0; strucnum<gMultiCount; strucnum++)
      {
         if(gRZoneList[strucnum]!=NULL)
         {
            FREELIST(gRZoneList[strucnum],ZONE);
            gRZoneList[strucnum] = NULL;
         }
      }
      
      ShowRMS(FALSE,NULL,0,FALSE);

      return;
   }

   for(strucnum=0; strucnum<gMultiCount; strucnum++)
   {
      SeqZone = ParseZone(command, &start1, &stop1, &chain1, 
                          &startinsert1, &stopinsert1, 
                          &start2, &stop2, &chain2,
                          &startinsert2, &stopinsert2,
                          strucnum);

      if((SeqZone == (-2)) && (!warned))
      {
         printf("   Error==> You cannot specify zones for each \
structure when performing\n");
         printf("            multiple structure fitting.\n");
         warned = 1;
      }
      
      if(SeqZone > -1)
      {
         /* If the user has not already specified an RMS zone, blank
            the current list.
         */
         if(!gUserRMSZone)
         {
            if(gRZoneList[strucnum]!=NULL)
            {
               FREELIST(gRZoneList[strucnum],ZONE);
               gRZoneList[strucnum] = NULL;
            }
         }
         
         /* Allocate an entry in zone list                              */
         if(gRZoneList[strucnum])
         {
            /* Move to end of zone list                                 */
            z=gRZoneList[strucnum];
            LAST(z);
            ALLOCNEXT(z,ZONE);
         }
         else
         {
            INIT(gRZoneList[strucnum],ZONE);
            z = gRZoneList[strucnum];
         }
         
         if(z==NULL)
         {
            printf("   Error==> No memory for zone!\n");
         }
         else
         {
            /* Add this zone to the zone list                           */
            z->chain1       = chain1;
            z->start1       = start1;
            z->startinsert1 = startinsert1;
            z->stop1        = stop1;
            z->stopinsert1  = stopinsert1;
            z->chain2       = chain2;
            z->start2       = start2;
            z->startinsert2 = startinsert2;
            z->stop2        = stop2;
            z->stopinsert2  = stopinsert2;
            z->mode         = SeqZone ? ZONE_MODE_SEQUENTIAL : gCurrentMode;
         }
         
         gUserRMSZone = TRUE;
         
         ShowRMS(FALSE,NULL,0,FALSE);
      }
   }

   return;
}


/************************************************************************/
/*>void Cleanup(void)
   ------------------
   Clean up screen and any malloc'd memory

   28.09.92 Original
   29.09.92 Added sequences
   01.10.92 Freed strparam and *Coor memory.
   05.10.92 Added libraries for AMIGA_WINDOWS
   08.10.93 Changed CURSES calls for book library
   17.07.95 Removed CURSES clean up and Amiga Windows cleanup
   20.07.95 Frees gWeights
   12.01.01 gMobPDB[] now an array
   01.02.01 Various other arrays now freed for multi-structure support
*/
void Cleanup(void)
{
   int   i;
   
   for(i=0; i<MAXSTRUC; i++)
   {
      if(gMobPDB[i])  FREELIST(gMobPDB[i], PDB);
      gMobPDB[i]     = NULL;
   }
   
   if(gRefPDB)          FREELIST(gRefPDB, PDB);
   gRefPDB     = NULL;
   
   for(i=0; i<MAXSTRUC; i++)
   {
      if(gFitPDB[i])          FREELIST(gFitPDB[i], PDB);
      gFitPDB[i]     = NULL;

      if(gMobCoor[i])         free(gMobCoor[i]);
      gMobCoor[i]    = NULL;

      if(gMobSeq[i])          free(gMobSeq[i]);
      gMobSeq[i]     = NULL;

      if(gZoneList[i])        FREELIST(gZoneList[i], ZONE);
      gZoneList[i]   = NULL;
   
      if(gRZoneList[i])       FREELIST(gRZoneList[i], ZONE);
      gRZoneList[i]  = NULL;

   }

   if(gRefSeq)          free(gRefSeq);
   gRefSeq     = NULL;
   
   if(gRefCoor)         free(gRefCoor);
   gRefCoor    = NULL;
   
   if(gWeights)         free(gWeights);
   gWeights    = NULL;
   
   for(i=0; i<MAXSTRPARAM; i++)
      free(gStrParam[i]);
   
   Help("Dummy","CLOSE");
}


/************************************************************************/
/*>void Die(char *message)
   -----------------------
   Program death. Writes message and cleans up before exit.

   25.09.92 Original
   17.07.95 Replaced calls to screen() with puts()
*/
void Die(char *message)
{
   puts(message);
   
   Cleanup();
   
   exit(0);
}


/************************************************************************/
/*>BOOL ReadStructure(int structure, char *filename, int strucnum,
                      int xmasFormat)
   ---------------------------------------------------------------
   Reads one of the 2 PDB structures.

   28.09.92 Original
   30.09.92 Added coordinate array allocation & check for inserts
   08.10.93 Modified for new ReadPDB()
   24.11.94 Changed to call ReadPDBAtoms() rather than ReadPDB()
   17.07.95 Replaced calls to screen() with printf()
   18.07.95 Uses fopen() rather than OpenRead()
            Uses fclose() rather than CloseFile()
   19.07.95 Selects whether to read HETATM records depending on
            gHetAtoms
   24.01.96 Improved error messages when no atoms read since ReadPDB()
            can distinguish between no memory and no atoms.
   03.07.97 Warning message if finds multiple occupancies
            Changed to CD1 rather than CD for ILE
   12.01.01 gMobPDB[] now an array
   01.02.01 Added strucnum param; returns BOOL
   20.02.01 Frees all previously loaded structures if we had a multiple
            set loaded previously.
   01.03.01 Added xmasFormat parameter
*/
BOOL ReadStructure(int  structure,
                   char *filename,
                   int  strucnum,
                   int  xmasFormat)
{
   file  *fp;
   int   natoms;
   PDB   *p;
   int   i;
   static int sLastStrucNum = (-1);

   gPDBPartialOcc = FALSE;
   
   /* Open the file                                                     */
   if((fp = fopen(filename,"r"))==NULL)
   {
      printf("   Unable to open file %s\n",filename);
      return(FALSE);
   }

   /* Handle appropriate pdb linked list and give message               */
   if(structure==STRUC_REFERENCE)
   {
      printf("   Reading reference structure...\n");
      strcpy(gRefFilename,filename);

      /* If there is something in the current list, free it             */
      if(gRefPDB)
      {
         FREELIST(gRefPDB,PDB);
         gRefPDB = NULL;
      }
   
      /* Read the structure                                             */
      if(xmasFormat)
      {
         if(gHetAtoms)
            gRefPDB = ReadXMAS(fp, &natoms);
         else
            gRefPDB = ReadXMASAtoms(fp, &natoms);
      }
      else
      {
         if(gHetAtoms)
            gRefPDB = ReadPDB(fp, &natoms);
         else
            gRefPDB = ReadPDBAtoms(fp, &natoms);
      }

      if(gRefPDB==NULL)
      {
         if(!natoms)
         {
            printf("   Error==> No atoms read from reference PDB \
file!\n");
         }
         else
         {
            printf("   Error==> No memory to read reference PDB file!\n");
         }
            
         gRefFilename[0] = '\0';
         fclose(fp);
         return(FALSE);
      }

      /* 03.07.97 Warning about multiple-occupancies                    */
      if(gPDBPartialOcc && !gQuiet)
      {
         printf("   Warning==> Reference set contains multiple occupancy \
atoms.\n");
         printf("              Only the first atom will be \
considered.\n");
      }      

      /* Allocate coordinate array                                      */
      if(gRefCoor) free(gRefCoor);
      if((gRefCoor = (COOR *)malloc(natoms * sizeof(COOR))) == NULL)
         printf("   Error==> Unable to allocate reference coordinate \
memory!\n");

      /* Convert to sequence                                            */
      if(gRefSeq != NULL) free(gRefSeq);
      if((gRefSeq = PDB2Seq(gRefPDB))==NULL)
         printf("   Error==> Unable to read sequence for reference \
structure!\n");
      
      /* Check for inserts                                              */
      for(p=gRefPDB; p!=NULL; NEXT(p))
      {
         if(p->insert[0] != ' ')
         {
            if(!gQuiet)
            {
               printf("   Warning==> Reference protein contains \
insertions.\n");
            }
            break;
         }
      }
      
      /* Fix ILE CD to CD1 (03.07.97 was other way round)               */
      for(p=gRefPDB; p!=NULL; NEXT(p))
      {
         /* 28.02.01 added ->atnam_raw                                  */
         if(!strncmp(p->resnam,"ILE ",4) && !strncmp(p->atnam,"CD  ",4))
         {
            strcpy(p->atnam,"CD1 ");
            strcpy(p->atnam_raw," CD1 ");
         }
      }
   }
   else if(structure==STRUC_MOBILE)
   {
      printf("   Reading mobile structure...\n");
      strcpy(gMobFilename[strucnum],filename);

      /* If there is something in the current list, free it             */
      for(i=strucnum; i<=sLastStrucNum; i++)
      {
         if(gMobPDB[i])
         {
            FREELIST(gMobPDB[i],PDB);
            gMobPDB[i] = NULL;
         }
      }
      sLastStrucNum = strucnum;
   
      /* Read the structure                                             */
      if(xmasFormat)
      {
         if(gHetAtoms)
            gMobPDB[strucnum] = ReadXMAS(fp, &natoms);
         else
            gMobPDB[strucnum] = ReadXMASAtoms(fp, &natoms);
      }
      else
      {
         if(gHetAtoms)
            gMobPDB[strucnum] = ReadPDB(fp, &natoms);
         else
            gMobPDB[strucnum] = ReadPDBAtoms(fp, &natoms);
      }


      if(gMobPDB[strucnum]==NULL)
      {
         if(!natoms)
         {
            printf("   Error==> No atoms read from mobile PDB file!\n");
         }
         else
         {
            printf("   Error==> No memory to read mobile PDB file!\n");
         }
            
         gMobFilename[strucnum][0] = '\0';
         fclose(fp);
         return(FALSE);
      }

      /* 03.07.97 Warning about multiple-occupancies                    */
      if(gPDBPartialOcc && !gQuiet)
      {
         printf("   Warning==> Mobile set contains multiple occupancy \
atoms.\n");
         printf("              Only the first atom will be \
considered.\n");
      }      

      /* Allocate coordinate array                                      */
      if(gMobCoor[strucnum]) free(gMobCoor[strucnum]);
      if((gMobCoor[strucnum] = (COOR *)malloc(natoms * sizeof(COOR))) 
         == NULL)
         printf("   Error==> Unable to allocate mobile coordinate \
memory!\n");

      /* Convert to sequence                                            */
      if(gMobSeq[strucnum] != NULL) free(gMobSeq[strucnum]);
      if((gMobSeq[strucnum] = PDB2Seq(gMobPDB[strucnum]))==NULL)
         printf("   Error==> Unable to read sequence for reference \
structure!\n");
      
      /* Check for inserts                                              */
      for(p=gMobPDB[strucnum]; p!=NULL; NEXT(p))
      {
         if(p->insert[0] != ' ')
         {
            if(!gQuiet)
            {
               printf("   Warning==> Mobile protein contains \
insertions.\n");
            }
            break;
         }
      }
      
      /* Fix ILE CD to CD1 (03.07.97 was other way round)               */
      for(p=gMobPDB[strucnum]; p!=NULL; NEXT(p))
      {
         /* 28.02.01 added ->atnam_raw                                  */
         if(!strncmp(p->resnam,"ILE ",4) && !strncmp(p->atnam,"CD  ",4))
         {
            strcpy(p->atnam,"CD1 ");
            strcpy(p->atnam_raw," CD1 ");
         }
      }
   }
   else
   {
      printf("   ReadStructure(): Internal Error!\n");
      return(FALSE);
   }

   gUserRMSAtoms = FALSE;
   gUserRMSZone  = FALSE;
   gFitted       = FALSE;

   /* and close file                                                    */
   fclose(fp);

   return(TRUE);
}


/************************************************************************/
/*>int InitParser(void)
   --------------------
   Initialise the command parser.
   
   25.09.92 Original
   09.10.92 Changed ALIGN to 0 parameters
   19.07.95 Added RESIDUE, HETATOMS & NOHETATOMS
   20.07.95 Added WEIGHT/NOWEIGHT
   22.07.95 Added GAPPEN
   20.11.95 Added READALIGNMENT
   31.05.96 Added BVALUE
   13.06.96 Added BWEIGHT
   11.11.96 BVALUE command now takes 1 or 2 params
   18.11.96 Added (NO)IGNOREMISSING
   20.12.96 Added NFITTED
   12.01.01 Added ITERATE
   01.02.01 Added MULTI, QUIET and MWRITE
   20.02.01 Added LIMIT
   01.03.01 Added optional 'xmas' parameter to REF, MOB, MULTI if
            XMAS support is compiled in
   20.03.01 Added CENTER/CENTRE commands
   28.03.01 Added optional 'reference' parameter to WRITE
*/
int InitParser(void)
{
   int   i;
   
   /* Initialise returned string array                                  */
   for(i=0; i<MAXSTRPARAM; i++)
      gStrParam[i] = (char *)malloc(MAXSTRLEN * sizeof(char));

   /* Construct the gKeyWords                                           */
#ifdef USE_XMAS
   MAKEMKEY(gKeyWords[0],  "REFERENCE",       STRING,1,2);
   MAKEMKEY(gKeyWords[1],  "MOBILE",          STRING,1,2);
#else
   MAKEMKEY(gKeyWords[0],  "REFERENCE",       STRING,1,1);
   MAKEMKEY(gKeyWords[1],  "MOBILE",          STRING,1,1);
#endif
   MAKEMKEY(gKeyWords[2],  "FIT",             NUMBER,0,0);
   MAKEMKEY(gKeyWords[3],  "ATOMS",           STRING,1,1);
   MAKEMKEY(gKeyWords[4],  "ZONE",            STRING,1,1);
   MAKEMKEY(gKeyWords[5],  "GRAPHIC",         NUMBER,0,0);
   MAKEMKEY(gKeyWords[6],  "ALIGN",           NUMBER,0,0);
   MAKEMKEY(gKeyWords[7],  "RATOMS",          STRING,1,1);
   MAKEMKEY(gKeyWords[8],  "RZONE",           STRING,1,1);
   MAKEMKEY(gKeyWords[9],  "WRITE",           STRING,1,2);
   MAKEMKEY(gKeyWords[10], "MATRIX",          NUMBER,0,0);
   MAKEMKEY(gKeyWords[11], "STATUS",          NUMBER,0,0);
   MAKEMKEY(gKeyWords[12], "QUIT",            NUMBER,0,0);
   MAKEMKEY(gKeyWords[13], "NUMBER",          STRING,1,1);
   MAKEMKEY(gKeyWords[14], "RMS",             NUMBER,0,0);
   MAKEMKEY(gKeyWords[15], "RESIDUE",         STRING,0,1);
   MAKEMKEY(gKeyWords[16], "HETATOMS",        NUMBER,0,0);
   MAKEMKEY(gKeyWords[17], "NOHETATOMS",      NUMBER,0,0);
   MAKEMKEY(gKeyWords[18], "WEIGHT",          NUMBER,0,0);
   MAKEMKEY(gKeyWords[19], "NOWEIGHT",        NUMBER,0,0);
   MAKEMKEY(gKeyWords[20], "GAPPEN",          NUMBER,1,1);
   MAKEMKEY(gKeyWords[21], "READALIGNMENT",   STRING,1,1);
   MAKEMKEY(gKeyWords[22], "BVALUE",          STRING,1,2);
   MAKEMKEY(gKeyWords[23], "BWEIGHT",         NUMBER,0,0);
   MAKEMKEY(gKeyWords[24], "IGNOREMISSING",   NUMBER,0,0);
   MAKEMKEY(gKeyWords[25], "NOIGNOREMISSING", NUMBER,0,0);
   MAKEMKEY(gKeyWords[26], "NFITTED",         NUMBER,0,0);
   MAKEMKEY(gKeyWords[27], "ITERATE",         STRING,0,1);
#ifdef USE_XMAS
   MAKEMKEY(gKeyWords[28], "MULTI",           STRING,1,2);
#else
   MAKEMKEY(gKeyWords[28], "MULTI",           STRING,1,1);
#endif
   MAKEMKEY(gKeyWords[29], "QUIET",           STRING,0,1);
   MAKEMKEY(gKeyWords[30], "MWRITE",          STRING,0,1);
   MAKEMKEY(gKeyWords[31], "LIMIT",           STRING,1,2);
   MAKEMKEY(gKeyWords[32], "CENTRE",          STRING,0,1);
   MAKEMKEY(gKeyWords[33], "CENTER",          STRING,0,1);
   
   return(0);
}


/************************************************************************/
/*>int DoCommandLoop(void)
   -----------------------
   Sit and wait for commands to be processed by the parser. Also checks 
   for help commands.
   
   25.09.92 Original
   28.09.92 Added ReadStructure(), OS commands
   02.10.92 Changed gets() to GetKybdString()
   17.07.95 Replaced calls to screen() with printf()
            Back to fgets() rather than GetKybdString()
   19.07.95 Added ByRes parameter to ShowRMS() and added RESIDUE, HETATOMS
            and NOHETATOMS
   20.07.95 Added WEIGHT/NOWEIGHT
   25.07.95 Changed to use mparse() and RESIDUE command may now take
            a filename for output. Added filename parameter to ShowRMS()
   20.11.95 Added READALIGNMENT handling
   21.11.95 Corrected `defaut:' to `default:'
   31.05.96 Added BVALUE handling
   13.06.96 Added BWEIGHT handling
   11.11.96 Handles second parameter to BVALUE
   18.11.96 Added IGNOREMISSING handling
   20.12.96 Added NFITTED handling
   15.01.01 Added ITER handling
   01.02.01 Added MULTI, QUIET, MWRITE handling
   20.02.01 Added LIMIT handling
   01.03.01 Added xmas parameters to REF, MOB, MULTI
   20.03.01 Added numeric parameter for ITERATE
            Added CENTRE/CENTER
   28.03.01 Added REFERENCE parameter to WRITE command
*/
int DoCommandLoop(void)
{
   int   nletters,
         NParam,
         i;
   char  comline[MAXBUFF],
         *ptr;
   
   for(;;)
   {
      stdprompt("ProFit");
      if(!fgets(comline, MAXBUFF, stdin))
         return(0);
      
      TERMINATE(comline);
      
      /* Any line which starts with a $ is passed to the OS             */
      if(comline[0] == '$')
      {
         ptr = comline+1;
         system(ptr);
         continue;
      }

      /* We need to check HELP outside the main parser as it has an 
         optional parameter (N.B. mparse() could now handle this)
      */
      if(match(comline,"HELP",&nletters))
      {
         if(nletters == 4)
         {
            DoHelp(comline,HELPFILE);
            continue;
         }
      }
      
      /* Main parser                                                    */
      switch(mparse(comline,NCOMM,gKeyWords,gNumParam,gStrParam,&NParam))
      {
      case PARSE_ERRC:
         printf("   Unrecognised keyword: %s\n",comline);
         break;
      case PARSE_ERRP:
         printf("   Invalid parameters: %s\n",comline);
         break;
      case PARSE_COMMENT:
         break;
      case 0:  /* REFERENCE                                             */
         gMultiCount = 1;
         if(NParam==1)
            ReadStructure(STRUC_REFERENCE,gStrParam[0],0,FALSE);
         else
            ReadStructure(STRUC_REFERENCE,gStrParam[1],0,TRUE);
         break;
      case 1:  /* MOBILE                                                */
         gMultiCount = 1;
         if(NParam==1)
            ReadStructure(STRUC_MOBILE,gStrParam[0],0,FALSE);
         else
            ReadStructure(STRUC_MOBILE,gStrParam[1],0,TRUE);
         break;
      case 2:  /* FIT                                                   */
         FitStructures();
         break;
      case 3:  /* ATOMS                                                 */
         if(gIterate)
         {
            printf("   Warning==> You cannot change the atoms when \
ITERATE is set.\n");
            printf("              Command ignored\n");
         }
         else
         {
            SetFitAtoms(gStrParam[0]);
         }
         break;
      case 4:  /* ZONE                                                  */
         SetFitZone(gStrParam[0], -1);
         break;
      case 5:  /* GRAPHIC                                               */
         GraphicAlign();
         break;
      case 6:  /* ALIGN                                                 */
         for(i=0; i< gMultiCount; i++)
            NWAlign(i);
         break;
      case 7:  /* RATOMS                                                */
         SetRMSAtoms(gStrParam[0]);
         break;
      case 8:  /* RZONE                                                 */
         SetRMSZone(gStrParam[0]);
         break;   
      case 9:  /* WRITE                                                 */
         if(NParam == 2)
         {
            if(!upstrncmp(gStrParam[0],"REF", 3))
            {
               WriteCoordinates(gStrParam[1], -1);
            }
            else
            {
               printf("   Error==> Invalid qualifier for WRITE: %s\n",
                      gStrParam[0]);
            }
         }
         else
         {
            WriteCoordinates(gStrParam[0], 0);
         }
         break;
      case 10: /* MATRIX                                                */
         ShowMatrix();
         break;
      case 11: /* STATUS                                                */
         ShowStatus();
         break;
      case 12: /* QUIT                                                  */
         return(0);
         break;
      case 13: /* NUMBER                                                */
         SetZoneStatus(gStrParam[0]);
         break;
      case 14: /* RMS                                                   */
         ShowRMS(FALSE,NULL,0,FALSE);
         break;
      case 15: /* RESIDUE                                               */
         ShowRMS(TRUE,(NParam?gStrParam[0]:NULL),0,FALSE);
         break;
      case 16: /* HETATOMS                                              */
         gHetAtoms = TRUE;
         printf("   Hetatoms will be read with future MOBILE or \
REFERENCE commands\n");
         break;
      case 17: /* NOHETATOMS                                            */
         gHetAtoms = FALSE;
         printf("   Hetatoms will be ignored with future MOBILE or \
REFERENCE commands\n");
         break;
      case 18: /* WEIGHT                                                */
         gDoWeights = WEIGHT_BVAL;
         break;
      case 19: /* NOWEIGHT                                              */
         gDoWeights = WEIGHT_NONE;
         break;
      case 20: /* GAPPEN                                                */
         gGapPen = (int)gNumParam[0];
         break;
      case 21: /* READALIGNMENT                                         */
         ReadAlignment(gStrParam[0]);
         break;
      case 22: /* BVALUE                                                */
         if((!isdigit(gStrParam[0][0])) && 
            (gStrParam[0][0] != '-')    &&
            (gStrParam[0][0] != '+')    &&
            (gStrParam[0][0] != '.'))
         {
            gUseBVal = 0;
            printf("   Atoms will be included regardless of B-value\n");
         }
         else
         {
            if(NParam == 2)
            {
               if(!upstrncmp(gStrParam[1],"REF",3))
               {
                  gUseBVal = 2;
               }
               else if(!upstrncmp(gStrParam[1],"MOB",3))
               {
                  gUseBVal = 3;
               }
               else
               {
                  printf("   %s is not a valid parameter to BVALUE. \
Command ignored.\n",gStrParam[1]);
                  break;
               }
            }
            else
            {
               gUseBVal = 1;
            }
            sscanf(gStrParam[0], "%lf", &gBValue);
         }
         break;
      case 23: /* BWEIGHT                                               */
         gDoWeights = WEIGHT_INVBVAL;
         break;
      case 24: /* IGNOREMISSING                                         */
         gIgnoreMissing = TRUE;
         break;
      case 25: /* NOIGNOREMISSING                                       */
         gIgnoreMissing = FALSE;
         break;
      case 26: /* NFITTED                                               */
         ShowNFitted();
         break;
      case 27: /* ITERATE                                               */
         if((NParam == 1) && !upstrncmp(gStrParam[0], "OFF",3))
         {
            gIterate = FALSE;
         }
         else
         {
            int ok = TRUE;

            /* If a parameter specified use this as the cutoff for adding
               or removing equivalenced pairs
            */
            if(NParam == 1)
            {
               if(!sscanf(gStrParam[0], "%lf", &gMaxEquivDistSq))
               {
                  printf("   Error==> Couldn't read cutoff from \
parameter\n");
                  break;
               }
               gMaxEquivDistSq *= gMaxEquivDistSq;
            }
            
            /* Check for numbers of chains                              */
            if(countchar(gRefSeq,'*') > 0)
            {
               printf("   Error==> Structures must have only one chain \
for iterative zones\n");
               ok = FALSE;
            }
            for(i=0; i<gMultiCount; i++)
            {
               if(countchar(gMobSeq[i],'*') > 0)
               {
                  printf("   Error==> Structures must have only one \
chain for iterative zones\n");
                  ok = FALSE;
                  break;
               }
            }

            if(ok)
            {
               gIterate = TRUE;
               SetFitAtoms("CA");
               if(!gQuiet)
               {
                  printf("   Info==> Setting atom selection to CA \
only\n");
               }
               
            }
         }
         break;
      case 28: /* MULTI                                                 */
         if(NParam==1)
            ReadMulti(gStrParam[0], FALSE);
         else
            ReadMulti(gStrParam[0], TRUE);
         break;
      case 29: /* QUIET                                                 */
         if((NParam == 1) && !upstrncmp(gStrParam[0], "OFF",3))
         {
            gQuiet = FALSE;
         }
         else
         {
            gQuiet = TRUE;
         }
         break;
      case 30: /* MWRITE                                                */
         if(NParam == 1)
         {
            WriteMulti(gStrParam[0]);
         }
         else
         {
            WriteMulti("fit");
         }
         break;
      case 31: /* LIMIT                                                 */
         if((NParam == 1) && (!upstrncmp(gStrParam[0], "OFF",3)))
         {
            gLimit[0] = gLimit[1] = (-1);
            break;
         }
         else if(NParam == 2)
         {
            if(sscanf(gStrParam[0], "%d", &(gLimit[0])) &&
               sscanf(gStrParam[1], "%d", &(gLimit[1])))
            {
               break;
            }
         }

         /* Error message                                               */
         printf("   Invalid parameters: %s\n",comline);
         break;
      case 32: /* CENTER & CENTRE                                       */
      case 33:
         if((NParam == 1) && (!upstrncmp(gStrParam[0], "OFF",3)))
         {
            gCentre = FALSE;
         }
         else
         {
            gCentre = TRUE;
         }
         break;
      default:
         break;
      }
   }
   return(0);
}


/************************************************************************/
/*>void SetFitAtoms(char *command)
   -------------------------------
   This splits up a list of atoms names and inserts them in the 
   fitatoms list.
   
   28.09.92 Framework
   29.09.92 Original
   30.09.92 Added NOT option; pad fitatoms[]
   17.07.95 Replaced calls to screen() with printf()
   23.07.96 Changed to call PADMINTERM() rather than padterm()
            Checks for legal wildcard specifications
            Improved out-of-bounds error checking
*/
void SetFitAtoms(char *command)
{
   int   comptr,
         comstart,
         atmnum,
         atmpos;
   char  *cmnd;
   
   /* Put to upper case                                                 */
   UPPER(command);
   
   /* Assume this is not a NOT selection                                */
   gNOTFitAtoms = FALSE;
   cmnd = command;
   
   /* Set NOT flag if required and step over the symbol                 */
   if(command[0] == '~' || command[0] == '^')
   {
      gNOTFitAtoms = TRUE;
      cmnd++;
   }

   /* Blank all fitatoms                                                */
   for(atmnum=0; atmnum<NUMTYPES; atmnum++)
      for(atmpos=0; atmpos<8; atmpos++)
         gFitAtoms[atmnum][atmpos] = '\0';
   
   atmnum=0;
   atmpos=0;
   comstart=0;
   for(comptr=0;comptr<strlen(cmnd);comptr++)
   {
      if(cmnd[comptr]==',')
      {
         comstart=comptr+1;
         gFitAtoms[atmnum][atmpos]   = '\0';  /* Terminate string       */
         PADMINTERM(gFitAtoms[atmnum],4);
         if(++atmnum >= NUMTYPES)
         {
            if(!gQuiet)
            {
               printf("   Warning==> Too many atoms in specification. \
Only %d used\n",NUMTYPES);
            }
            break;
         }
         atmpos=0;
      }
      else
      {
         if(atmpos >= MAXATSPEC)
         {
            TERMAT(cmnd+comptr, ',');
            if(!gQuiet)
            {
               printf("   Warning==> Atom name specification too long \
(%s)\n", cmnd+comstart);
               printf("              Using all atoms.\n");
            }
            gFitAtoms[0][0] = '*';
            gFitAtoms[0][1] = '\0';
            break;
         }

         gFitAtoms[atmnum][atmpos++] = cmnd[comptr];
      }
   }
   
   /* Terminate last one                                                */
   gFitAtoms[atmnum][atmpos] = '\0';
   PADMINTERM(gFitAtoms[atmnum],4);

   /* See if a * was specified not in first position; move it if so.
      Also check for legal atom wildcard specifications
   */
   for(atmnum=0; atmnum<NUMTYPES; atmnum++)
   {
      if(!LegalAtomSpec(gFitAtoms[atmnum]))
      {
         if(!gQuiet)
         {
            printf("   Warning==> Illegal atom specification (%s). Using \
all atoms\n",
                   gFitAtoms[atmnum]);
         }
         gFitAtoms[0][0] = '*';
         gFitAtoms[0][1] = '\0';
         break;
      }
      
      if(gFitAtoms[atmnum][0] == '*')
      {
         if(gNOTFitAtoms)
         {
            if(!gQuiet)
            {
               printf("   Warning==> NOT option ignored.\n");
            }
            gNOTFitAtoms = FALSE;
         }
         gFitAtoms[0][0] = '*';
         gFitAtoms[0][1] = '\0';
         break;
      }
   }
   
   gFitted = FALSE;
   
   return;
}


/************************************************************************/
/*>void SetFitZone(char *command, int strucnum)
   --------------------------------------------
   Processes commands to specify the zone for fitting.
   
   28.09.92 Original
   29.09.92 Added gCurrentMode
   01.10.92 Added gUserFitZone flag
   17.07.95 Replaced calls to screen() with printf()
   18.07.95 Added initialisation of inserts in zones
   19.07.95 * on its own equivalent to CLEAR
   18.06.96 Replaced MODE_* with ZONE_MODE_*
   01.02.01 Added multi-structure support incl. strucnum param
   28.02.01 Error message from ParseZone() when trying to specify multiple
            zones for multi-structure now here instead of in ParseZone()
*/
void SetFitZone(char *command, int strucnum)
{
   int   start1,  stop1,
         start2,  stop2,
         SeqZone, maxstruc,
         snum;
   char  chain1,  chain2,
         startinsert1, stopinsert1,
         startinsert2, stopinsert2;
   ZONE  *z;
   int   warned = 0;
   

   UPPER(command);

   snum = strucnum;
   
   /* See if this is clearing the zones                                 */
   if(!gUserFitZone || !strncmp(command,"CLEAR",5) || 
      !strcmp(command,"*"))
   {
      if(strucnum > (-1))
      {
         if(gZoneList[snum] != NULL)
         {
            FREELIST(gZoneList[snum],ZONE);
            gZoneList[snum] = NULL;
         }
      }
      else
      {
         for(snum=0; snum<gMultiCount; snum++)
         {
            if(gZoneList[snum] != NULL)
            {
               FREELIST(gZoneList[snum],ZONE);
               gZoneList[snum] = NULL;
            }
         }
      }
      gUserFitZone = FALSE;
      if(!strncmp(command,"CLEAR",5) || !strcmp(command,"*"))  
         return;
   }

   /* If strucnum is -1 then set range so we will do all structures     */
   if(strucnum == (-1))
   {
      snum = 0;
      maxstruc = gMultiCount;
   }
   else  /* Just do the one structure specified                         */
   {
      maxstruc = strucnum+1;
   }

   for(; snum<maxstruc; snum++)
   {
      SeqZone = ParseZone(command, &start1, &stop1, &chain1, 
                          &startinsert1, &stopinsert1,
                          &start2, &stop2, &chain2,
                          &startinsert2, &stopinsert2, snum);
      if((SeqZone == (-2)) && (!warned))
      {
         printf("   Error==> You cannot specify zones for each \
structure when performing\n");
         printf("            multiple structure fitting.\n");
         warned = 1;
      }
      
      if(SeqZone > -1)
      {
         /* Allocate an entry in zone list                              */
         if(gZoneList[snum])
         {
            /* Move to end of zone list                                 */
            z=gZoneList[snum];
            LAST(z);
            ALLOCNEXT(z,ZONE);
         }
         else
         {
            INIT(gZoneList[snum],ZONE);
            z = gZoneList[snum];
         }
         
         if(z==NULL)
         {
            printf("   Error==> No memory for zone!\n");
         }
         else
         {
            /* Add this zone to the zone list                           */
            z->chain1       = chain1;
            z->start1       = start1;
            z->startinsert1 = startinsert1;
            z->stop1        = stop1;
            z->stopinsert1  = stopinsert1;
            z->chain2       = chain2;
            z->start2       = start2;
            z->startinsert2 = startinsert2;
            z->stop2        = stop2;
            z->stopinsert2  = stopinsert2;
            z->mode         = SeqZone ? 
                              ZONE_MODE_SEQUENTIAL : gCurrentMode;
         }
      }
   }
   
   gFitted      = FALSE;
   gUserFitZone = TRUE;
   
   return;
}


/************************************************************************/
/*>void SetZoneStatus(char *status)
   --------------------------------
   Processes the command to set the mode for interpretation of residue
   numbers.
   
   28.09.92 Framework
   29.09.92 Original
   08.10.93 Modified only to toupper lowercase letters
   17.07.95 Replaced calls to screen() with printf()
   18.06.96 Replaced MODE_* with ZONE_MODE_*
*/
void SetZoneStatus(char *status)
{
   char ch;

   ch = status[0];
   if(islower(ch)) ch = toupper(ch);

   if(ch == 'R')
      gCurrentMode = ZONE_MODE_RESNUM;
   else if(ch == 'S') 
      gCurrentMode = ZONE_MODE_SEQUENTIAL;
   else
      printf("   Error==> Invalid numbering mode. Must be RESIDUE or \
SEQUENTIAL\n");

   return;
}


/************************************************************************/
/*>int GetResSpec(char *resspec, int *resnum, char *chain, char *insert)
   ---------------------------------------------------------------------
   Extracts residue number and chain name from a string. If the string is
   blank, the residue number will be specified as -999 and the chain name
   will be unmodified.

   29.09.92 Original
   17.07.95 Returns insert code. Uses ParseResSpec()
   20.02.01 Now makes its own copy of the string with an \ characters
            removed
   20.02.01 -999 for start or end of structure rather than -1
*/
int GetResSpec(char *resspec, int *resnum, char *chain, char *insert)
{
   char  *ptr, *buffer;
   int   i,
         retval = 0;

   if((buffer = (char *)malloc(strlen(resspec)+1))==NULL)
   {
      printf("   Error==> no memory for copy of residue spec\n");
      return(2);
   }
   
   /* Move pointer over any spaces                                      */
   for(ptr=resspec; *ptr==' '||*ptr=='\t'; ptr++) ;

   /* Copy the resspec string skipping \ characters                     */
   for(i=0; *ptr; ptr++)
   {
      if(*ptr != '\\')
         buffer[i++] = *ptr;
   }
   buffer[i] = '\0';

   /* Assume blank insert code                                          */
   *insert = ' ';
   
   /* If the zone is blank, it will be all residues                     */
   if(*buffer == '\0')
   {
      *resnum = -999;
      free(buffer);
      return(0);
   }

   if(!ParseResSpec(buffer, chain, resnum, insert))
      retval = 1;              /* Indicates an error                    */

   free(buffer);
   return(retval);
}


/************************************************************************/
/*>int ParseZone(char *zonespec, 
                 int *start1, int *stop1, char *chain1, 
                 char *startinsert1, char *stopinsert1,
                 int *start2, int *stop2, char *chain2,
                 char *startinsert2, char *stopinsert2,
                 int strucnum)
   ----------------------------------------------------
   Returns: 0 for numeric zones,
            1 for sequence specified zones,
           -1 for error.
           -2 for error in multi-zone

   Sorts out zone specification. The specified residue ranges will be
   returned in start1/stop1/chain1 and start2/stop2/chain2. The first or 
   last residue present will be specified by -999. Zones may be specified 
   by number or by sequence.

   28.09.92 Original
   29.09.92 Added Sequence zones and return flag
   17.07.95 Replaced calls to screen() with printf()
            Added inserts
   01.02.01 Added strucnum parameter
   20.02.01 Allow negative residue numbers to be escaped with a \ 
            Code realises this - is not a range spcifier.
   20.02.01 -999 for start or end of structure rather than -1
   28.02.01 -2 return for multi-zone error - moved message out to calling
            routine
*/
int  ParseZone(char *zonespec,
               int  *start1, 
               int  *stop1,
               char *chain1,
               char *startinsert1,
               char *stopinsert1,
               int  *start2, 
               int  *stop2,
               char *chain2,
               char *startinsert2,
               char *stopinsert2,
               int  strucnum)
{
   char *zone1,
        *zone2,
        *ptr,
        *dash = NULL,
        buffer[80];
   int  retval = 0;
   

   /* Blank the chain and insert names                                  */
   *chain1       = *chain2       = ' ';
   *startinsert1 = *startinsert2 = ' ';
   *stopinsert1  = *stopinsert2  = ' ';
   
   /* First split into 2 parts if required
      17.07.95 Added calls to KILLLEADSPACES()
   */
   KILLLEADSPACES(zone1, zonespec);
   zone2 = zone1;
   if((ptr=strchr(zonespec,':'))!=NULL)
   {
      /* 20.02.01 We don't allow this type of zone spec when we have 
         multiple structures
      */
      if(gMultiCount > 1)
      {
         return(-2);
      }
      
      KILLLEADSPACES(zone2, ptr+1);
      *ptr  = '\0';
   }
   
/*
*** Do zone1 first                                                     ***
*/

   /* See if there is a dash representing a range, skipping any - sign
      escaped with a \ (\- is used to represent a negative residue
      number)
   */
   dash=FindDash(zone1);

   /* If there's a * in the zone spec. it's all residues                */
   if((ptr=strchr(zone1,'*'))!=NULL)
   {
      *start1 = -999;
      *stop1  = -999;
      
      /* If * was not the first char in the zone spec., then the first
         character was the chain name.
      */
      if(ptr != zone1)
         *chain1 = *zone1;
   }
   else if(dash != NULL)                  /* - indicates a numeric zone */
   {
      /* Make a temporary copy                                          */
      strcpy(buffer,zone1);
      /* Terminate copy at the -                                        */
      *(FindDash(buffer)) = '\0';
      /* Read the start of the zone                                     */
      if(GetResSpec(buffer,start1,chain1,startinsert1))
      {
         printf("   Invalid zone specification: ");
         printf("%s\n",buffer);
         return(-1);
      }

      /* Read end of zone                                               */
      if(GetResSpec(dash+1,stop1,chain1,stopinsert1))
      {
         printf("   Invalid zone specification: ");
         printf("%s\n",dash+1);
         return(-1);
      }
   }
   else                                /* It's a sequence specified zone*/
   {
      if(FindSeq(zone1,gRefSeq,start1,stop1,chain1))
      {
         printf("   Sequence zone specification not found: ");
         printf("%s\n",zone1);
         return(-1);
      }
      else
      {
         retval = 1;
      }
   }


/*
*** Do zone2 if different from zone1                                   ***
*/
   if((zone1 == zone2) && retval==0)
   {
      *start2       = *start1;
      *stop2        = *stop1;
      *chain2       = *chain1;
      *startinsert2 = *startinsert1;
      *stopinsert2  = *stopinsert1;
   }
   else
   {
      dash=FindDash(zone2);

      /* If there's a * in the zone spec. it's all residues             */
      if((ptr=strchr(zone2,'*'))!=NULL)
      {
         *start2 = -999;
         *stop2  = -999;
         
         /* If * was not the first char in the zone spec., then the first
            character was the chain name.
            */
         if(ptr != zone2)
            *chain2 = *zone2;
      }
      else if(dash!=NULL)                    /* - shows a numeric zone  */
      {
         /* Make a temporary copy                                       */
         strcpy(buffer,zone2);
         /* Terminate copy at the -                                     */
         *(FindDash(buffer)) = '\0';
         /* Read the start of the zone                                  */
         if(GetResSpec(buffer,start2,chain2,startinsert2))
         {
            printf("   Invalid zone specification: ");
            printf("%s\n",buffer);
            return(-1);
         }
         
         /* Read end of zone                                            */
         if(GetResSpec(dash+1,stop2,chain2,stopinsert2))
         {
            printf("   Invalid zone specification: ");
            printf("%s\n",dash+1);
            return(-1);
         }
      }
      else                             /* It's a sequence specified zone*/
      {
         if(FindSeq(zone2,gMobSeq[strucnum],start2,stop2,chain2))
         {
            printf("   Sequence zone specification not found: ");
            printf("%s\n",zone2);
            return(-1);
         }
         else
         {
            retval = 1;
         }
      }
   }
   
   return(retval);
}


/************************************************************************/
/*>int FindSeq(char *zonespec, char *sequence, int *start, int *stop, 
               int *chain)
   ------------------------------------------------------------------
   Returns: 0 Sequence found
            1 Sequence not found

   Finds the start and stop on the basis of sequence.
            
   29.09.92 Original
   30.09.92 +1 Correction to length
*/
int FindSeq(char *zonespec,
            char *sequence, 
            int  *start, 
            int  *stop, 
            char *chain)
{
   char  zoneseq[40],
         *ptr;
   int   length,
         occurence,
         noccur,
         j;
   
   /* Extract the sequence part                                         */
   strcpy(zoneseq,zonespec);
   ptr = strchr(zoneseq,',');
   if(ptr!=NULL) *ptr = '\0';
   ptr = strchr(zoneseq,'/');
   if(ptr!=NULL) *ptr = '\0';
   
   /* Extract the length                                                */
   ptr = strchr(zonespec,',');
   if(ptr!=NULL)    sscanf(ptr+1,"%d",&length);
   else             length = strlen(zoneseq);
   
   /* Extract the occurence number                                      */
   ptr = strchr(zonespec,'/');
   if(ptr!=NULL)    sscanf(ptr+1,"%d",&occurence);
   else             occurence = 1;
   
   /* Now search for the occurence'th occurence of zoneseq              */
   noccur = 0;
   for(j=0; j<strlen(sequence)-length+1; j++)
   {
      if(!strncmp(zoneseq,sequence+j,strlen(zoneseq)))   /* 30.09.92    */
      {
         if(++noccur == occurence)
         {
            *start = j+1;
            *stop  = j+length;
            *chain = ' ';
            return(0);
         }
      }
   }

   *start = -2;
   *stop  = -2;
   *chain = 'X';

   return(1);
}


/************************************************************************/
/*>void ShowMatrix(void)
   ---------------------
   Displays the rotation matrix.
   
   28.09.92 Framework
   30.09.92 Original
   17.07.95 Replaced calls to screen() with printf()
   02.10.95 Added printing of CofGs
   20.02.01 gMobCofG now an array
*/
void ShowMatrix(void)
{
   int   i,
         strucnum;

   if(gFitted)
   {
      if(gMultiCount == 1)
      {
         printf("   Reference CofG...\n");
         printf("   %8.4f %8.4f %8.4f\n",gRefCofG.x,
                                         gRefCofG.y,
                                         gRefCofG.z);
         printf("   Mobile CofG...\n");
      }

      for(strucnum=0; strucnum<gMultiCount; strucnum++)
      {
         if(gMultiCount > 1)
         {
            printf("   Structure %d CofG...\n", strucnum+1);
         }
         printf("   %8.4f %8.4f %8.4f\n",gMobCofG[strucnum].x,
                                         gMobCofG[strucnum].y,
                                         gMobCofG[strucnum].z);
      }

      printf("   Rotation matrix...\n");
      for(strucnum=0; strucnum<gMultiCount; strucnum++)
      {
         if(gMultiCount > 1)
         {
            printf("   Structure %d:\n", strucnum+1);
         }
         for(i=0; i<3; i++)
         {
            printf("   %8.4f %8.4f %8.4f\n",gRotMat[strucnum][i][0],
                                            gRotMat[strucnum][i][1],
                                            gRotMat[strucnum][i][2]);
         }
      }
      

      printf("   Translation vector (between CofGs)...\n");
      for(strucnum=0; strucnum<gMultiCount; strucnum++)
      {
         if(gMultiCount > 1)
         {
            printf("   Structure %d:\n", strucnum+1);
         }
         printf("   %8.4f %8.4f %8.4f\n",gRefCofG.x - gMobCofG[strucnum].x,
                                         gRefCofG.y - gMobCofG[strucnum].y,
                                         gRefCofG.z - gMobCofG[strucnum].z);
      }
   }
   else
   {
      printf("   Warning==> Structures have not yet been fitted.\n");
   }
   
   return;
}


/************************************************************************/
/*>void ShowStatus(void)
   ---------------------
   Shows the current program status.
   
   28.09.92 Framework
   29.09.92 Original
   30.09.92 Modified for padding of atom names
   01.10.92 Modification to All zone printing
   17.07.95 Replaced calls to screen() with printf()
   18.07.95 Prints zones with chain first and with inserts
            Prints *'s rather than -1 for all residues
            Added calls to FormatZone()
   20.07.95 Added WEIGHTS
   25.07.95 Added code to print chain labels
   24.01.96 Fixed bug in printing atom names containing spaces
   31.06.96 Added BValue cutoff 
   13.06.96 Modified B-value weighting message
   18.06.96 Replaced MODE_* with ZONE_MODE_*
   11.11.96 BValue cutoff message reflects new REF and MOB 
            parameters
   12.01.01 gMobPDB[] now an array
            Added iterate mode printing
   01.02.01 Added printing of multi structure data
   20.02.01 -999 for start or end of structure rather than -1
   28.03.01 Reports CENTRE mode
*/
void ShowStatus(void)
{
   char  buffer[240],
         atm[8],
         *chains;
   int   i, j, strucnum;
   ZONE  *z;
   
   printf("   Reference structure:        %s\n",
          (gRefFilename[0]?gRefFilename:"Undefined"));

   for(strucnum=0; strucnum<gMultiCount; strucnum++)
   {
      printf("   Mobile structure:           %s\n",
             (gMobFilename[strucnum][0] ?
              gMobFilename[strucnum]:"Undefined"));
   }

   printf("   HETATM records are:         %s\n",
          (gHetAtoms?"Included":"Ignored"));

   printf("   Fitting will be:            %s\n",
          (gDoWeights==WEIGHT_BVAL?"Weighted by B-value":
           (gDoWeights==WEIGHT_INVBVAL?"Weighted by 1/B-value":
            "Normal (unweighted)")));

   if(((chains = GetPDBChainLabels(gRefPDB)) != NULL) &&
      ((strlen(chains) > 1) || (chains[0] != ' ')))
      printf("   Reference structure Chains: %s\n",chains);
   if(chains != NULL)
      free(chains);
   
   if(((chains = GetPDBChainLabels(gMobPDB[0])) != NULL) &&
      ((strlen(chains) > 1) || (chains[0] != ' ')))
      printf("   Mobile structure Chains:    %s\n",chains);
   if(chains != NULL)
      free(chains);

   printf("   Current numbering mode:     ");
   if(gCurrentMode == ZONE_MODE_RESNUM)
      printf("Residue\n");
   else if(gCurrentMode == ZONE_MODE_SEQUENTIAL)
      printf("Sequential\n");

   printf("   Iterative zone updating:    ");
   printf("%s\n", ((gIterate)?"On":"Off"));

   printf("   Atoms being fitted:         ");
   if(gFitAtoms[0][0] == '*')
   {
      printf("All\n");
   }
   else
   {
      if(gNOTFitAtoms) printf("NOT ");
      
      for(i=0;i<NUMTYPES;i++)
      {
         if(gFitAtoms[i][0] == '\0') break;
         strcpy(atm,gFitAtoms[i]);
         /* Remove trailing spaces                                      */
         /* 24.01.96 Was terminating at the first space. This broke
            atom names containing spaces (e.g. "N A" in heme groups)
         */
         for(j=strlen(atm)-1; j>=0; j--)
         {
            if(atm[j] == ' ')
               atm[j] = '\0';
            else
               break;
         }

         if(i)
            sprintf(buffer, ", %s", atm);
         else
            sprintf(buffer, "%s",   atm);
         printf(buffer);
      }
      printf("\n");
   }
   if(gUseBVal)
   {
      printf("   Atoms will be discarded if their B-value is > %.2f in \
%s structure\n", gBValue, 
             (gUseBVal==2?"the reference":
              (gUseBVal==3?"the mobile":
               "either")));
   }
   else
   {
      printf("   Atoms will be included regardless of B-value\n");
   }
      
   
   printf("   Zones being fitted:         ");

   for(strucnum=0; strucnum<gMultiCount; strucnum++)
   {
      if(gMultiCount > 2)
         printf("\n   (Mobile Structure: %d)   ", strucnum+1);

      if(gZoneList[strucnum]==NULL || !gUserFitZone)
      {
         printf("All\n");
      }
      else
      {
         printf("\n");
         
         for(z=gZoneList[strucnum]; z!=NULL; NEXT(z))
         {
            char zone1[64],
                 zone2[64];
            
            FormatZone(zone1, z->chain1, 
                       z->start1, z->startinsert1, 
                       z->stop1,  z->stopinsert1);
            
            FormatZone(zone2, z->chain2, 
                       z->start2, z->startinsert2, 
                       z->stop2,  z->stopinsert2);
            
            printf("      %-16s with %-16s %s\n",
                   zone1, zone2,
                   ((z->mode == ZONE_MODE_RESNUM)?"(Residue numbering)"
                    :"(Sequential numbering)"));
         }
      }
   }
   
   if(gFitted)  /* Only display this when its definitely valid          */
   {
      printf("   Atoms for RMS calculation:  ");
   
      if(gUserRMSAtoms)
      {
         if(gRMSAtoms[0][0] == '*')
         {
            printf("All\n");
         }
         else
         {
            if(gNOTRMSAtoms) printf("NOT ");

            for(i=0;i<NUMTYPES;i++)
            {
               if(gRMSAtoms[i][0] == '\0') break;
               strcpy(atm,gRMSAtoms[i]);
               if(strchr(atm,' ')) *strchr(atm,' ') = '\0';
               
               if(i)
                  sprintf(buffer, ", %s", atm);
               else
                  sprintf(buffer, "%s",   atm);
               printf(buffer);
            }
            printf("\n");
         }
      }
      else
      {
         if(gFitAtoms[0][0] == '*')
         {
            printf("All\n");
         }
         else
         {
            if(gNOTFitAtoms) printf("NOT ");

            for(i=0;i<NUMTYPES;i++)
            {
               if(gFitAtoms[i][0] == '\0') break;
               strcpy(atm,gFitAtoms[i]);
               if(strchr(atm,' ')) *strchr(atm,' ') = '\0';

               if(i)
                  sprintf(buffer, ", %s", atm);
               else
                  sprintf(buffer, "%s",   atm);
               printf(buffer);
            }
            printf("\n");
         }
      }
      
      printf("   Zones for RMS calculation:  ");
   
      if(gUserRMSZone)
      {
         for(strucnum=0; strucnum<gMultiCount; strucnum++)
         {
            if(gMultiCount > 2)
               printf("\n   (Mobile Structure: %d)   ", strucnum+1);

            if(gRZoneList[strucnum]==NULL) printf("All\n");
            else                printf("\n");
            
            for(z=gRZoneList[strucnum]; z!=NULL; NEXT(z))
            {
               char zone1[64],
                    zone2[64];
               
               FormatZone(zone1, z->chain1, 
                          z->start1, z->startinsert1, 
                          z->stop1,  z->stopinsert1);
               
               FormatZone(zone2, z->chain2, 
                          z->start2, z->startinsert2, 
                          z->stop2,  z->stopinsert2);
               
               printf("      %-16s with %-16s %s\n",
                      zone1, zone2,
                      ((z->mode == ZONE_MODE_RESNUM)?"(Residue numbering)"
                       :"(Sequential numbering)"));
            }
         }
      }
      else
      {
         for(strucnum=0; strucnum<gMultiCount; strucnum++)
         {
            if(gMultiCount > 2)
               printf("\n   (Mobile Structure: %d)   ", strucnum+1);
            if(gZoneList[0]==NULL || !gUserFitZone)
            {
               printf("All\n");
            }
            else
            {
               printf("\n");
               
               for(z=gZoneList[strucnum]; z!=NULL; NEXT(z))
               {
                  char zone1[64],
                       zone2[64];
                  
                  FormatZone(zone1, z->chain1, 
                             z->start1, z->startinsert1, 
                             z->stop1,  z->stopinsert1);
                  
                  FormatZone(zone2, z->chain2, 
                             z->start2, z->startinsert2, 
                             z->stop2,  z->stopinsert2);
                  
                  printf("      %-16s with %-16s %s\n",
                         zone1, zone2,
                         ((z->mode == ZONE_MODE_RESNUM)?"(Residue numbering)"
                          :"(Sequential numbering)"));
                  
               }
            }
         }
      }
   }
   
   if(gCentre)
   {
      printf("\n   Coordinates written centred on: ORIGIN\n\n");
   }
   else
   {
      printf("\n   Coordinates written centred on: REFERENCE SET\n\n");
   }
   

   printf("   Reference sequence:         ");
   if(gRefSeq)
   {
      for(i=0, j=0; i<strlen(gRefSeq); i++)
      {
         buffer[j++] = gRefSeq[i];
         
         if(j>59 || i==strlen(gRefSeq)-1)
         {
            buffer[j] = '\0';
            printf("\n   ");
            printf(buffer);
            j=0;
         }
      }
      printf("\n");
   }
   else
   {
      printf("Undefined\n");
   }

   for(strucnum=0; strucnum<gMultiCount; strucnum++)
   {
      printf("   Mobile sequence:            ");
      if(gMobSeq[strucnum])
      {
         for(i=0, j=0; i<strlen(gMobSeq[strucnum]); i++)
         {
            buffer[j++] = gMobSeq[strucnum][i];
            
            if(j>59 || i==strlen(gMobSeq[strucnum])-1)
            {
               buffer[j] = '\0';
               printf("\n   ");
               printf(buffer);
               j=0;
            }
         }
         printf("\n");
      }
      else
      {
         printf("Undefined\n");
      }
   }

   return;
}


/************************************************************************/
/*>void stdprompt(char *string)
   ----------------------------
   Issues a prompt to stdout providing stdin is a tty

   18.07.95 Original    By: ACRM
*/
#include <unistd.h>
void stdprompt(char *string)
{
#if (unix || __unix__)
   if(!isatty(0))
      return;
#endif
   
   printf("%s> ",string);
   fflush(stdout);
}


/************************************************************************/
/*>void FormatZone(char *zone, char chain, int start, char startinsert, 
                   int stop,  char stopinsert)
   --------------------------------------------------------------------
   Formats a zone specification accounting for -999 values which represent
   all residues

   18.07.95 Original    By: ACRM
   20.02.01 -999 for start or end of structure rather than -1
*/
void FormatZone(char *zone, char chain, int start, char startinsert, 
                int stop,  char stopinsert)
{
   char part1[16],
        part2[16];
   
   if((start == (-999)) && (stop  == (-999)))
   {
      if(chain == ' ')
         sprintf(zone,"All residues");
      else
         sprintf(zone,"Chain %c",chain);
   }
   else
   {
      if(start==(-999))
      {
         sprintf(part1,"%c*",    chain);
         sprintf(part2,"%c%d%c", chain, stop, stopinsert);
      }
      else if(stop==(-999))
      {
         sprintf(part1,"%c%d%c", chain, start, startinsert);
         sprintf(part2,"%c*",    chain);
      }
      else
      {
         sprintf(part1,"%c%d%c", chain, start, startinsert);
         sprintf(part2,"%c%d%c", chain, stop, stopinsert);
      }

      sprintf(zone,"%-6s to %-6s", part1, part2);
   }
}


/************************************************************************/
/*>void ReadMulti(char *filename, BOOL xmasFormat)
   -----------------------------------------------
   Reads in multiple structures as given by the MULTI command

   01.02.01 Original   By: ACRM
   01.03.01 Added xmasFormat parameter
*/
void ReadMulti(char *filename, BOOL xmasFormat)
{
   FILE *fof = NULL;
   BOOL GotRef = FALSE;
   char buffer[MAXBUFF], *ch;

   gMultiCount = 0;

   if((fof=fopen(filename,"r"))==NULL)
   {
      printf("   Error==> Can't open list of files: %s\n",
             filename);
   }
   else
   {
      while(fgets(buffer, MAXBUFF, fof))
      {
         TERMINATE(buffer);
         KILLTRAILSPACES(buffer);
         KILLLEADSPACES(ch, buffer);
         
         if((*ch != '#') && (*ch != '!') && (strlen(ch)))
         {
            if(!GotRef)
            {
               ReadStructure(STRUC_REFERENCE, ch, 0, xmasFormat);
               GotRef=TRUE;
            }
            
            if(ReadStructure(STRUC_MOBILE, ch, gMultiCount, xmasFormat))
            {
               if(++gMultiCount >= MAXSTRUC)
               {
                  printf("   Error==> Maximum structure count (%d) \
exceeded. Increase MAXSTRUC\n", MAXSTRUC);
                  printf("            Skipped structure: %s\n",
                         ch);
               }
            }
         }
      }
   }
}


/************************************************************************/
/*>void WriteCoordinates(char *filename, int strucnum)
   ---------------------------------------------------
   Writes a coordinate file.

   28.09.92 Framework
   01.10.92 Original
   17.07.95 Replaced calls to screen() with printf()
   18.07.95 Uses fopen() rather than OpenWrite()
            Uses fclose() rather than CloseFile()
   21.07.95 Corrected logic of test for un-fitted PDB
   27.06.97 Changed call to fopen() to OpenOrPipe
   11.01.01 gFitPDB now an array
   01.02.01 Added strucnum parameter
   28.03.01 Added code to support strucnum < 0 to write the centred
            reference set and centering of coordinates
*/
void WriteCoordinates(char *filename, int strucnum)
{
   FILE  *fp;
   PDB   *pdb,
         *pdbc;
   VEC3F CofG;

   /* Point pdb to the coordinate set of interest                       */
   if(strucnum < 0)  /* use the reference set                           */
   {
      pdb = gRefPDB;
   }
   else              /* use a mobile set                                */
   {
      pdb = gFitPDB[strucnum];
   }

   CofG.x = -1.0 * gRefCofG.x;
   CofG.y = -1.0 * gRefCofG.y;
   CofG.z = -1.0 * gRefCofG.z;
   
   /* If centering, then make a copy of the coordinates and translate
      to the origin
   */
   if(gCentre && (pdb!=NULL))
   {
      if((pdbc = DupePDB(pdb))==NULL)
      {
         printf("   Error==> No memory for translating the reference \
coordinates\n");
         printf("            Centred reference coordinates not \
written.\n");
         return;
      }

      /* Point pdb to this copy of the coordinates                      */
      pdb = pdbc;
      TranslatePDB(pdb, CofG);
   }
   
   if(!gFitted || pdb == NULL)
   {
      printf("   Error==> Fitting has not yet been performed.\n");
   }
   else
   {
      if((fp=OpenOrPipe(filename))==NULL)
      {
         printf("   Error==> Enable to open file for writing.\n");
      }
      else
      {
         printf("   Writing coordinates...\n");
         WritePDB(fp, pdb);
         CloseOrPipe(fp);
      }
   }

   /* If centering, free the copy of the reference set                  */
   if(gCentre && pdb!=NULL)
   {
      FREELIST(pdb, PDB);
   }

   return;
}


/************************************************************************/
/*>void WriteMulti(char *ext)
   --------------------------
   Writes a set of fitted files from multi-fitting

   01.02.01 Original   By: ACRM
   20.02.01 Added more defensive checks on string lengths and handle case
            where input filename didn't contain a .
*/
void WriteMulti(char *ext)
{
   char filename[MAXBUFF];
   int  i,
        j;

   for(i=0; i<gMultiCount; i++)
   {
      strncpy(filename, gMobFilename[i], MAXBUFF);
      /* Work from the end of string to strip off the extension         */
      j = strlen(filename)-1;


      /* Step back until we find a . delimiting the extension           */
      while((filename[j] != '.') && (j>=0))
      {
         /* Break out if we find a / \ ] or : since we are in the path  */
         if(filename[j] == '/'  || 
            filename[j] == '\\' ||
            filename[j] == ']'  ||
            filename[j] == ':')
         {
            j=0;
            break;
         }
         j--;
      }
      
      /* There was a . in the filename, so truncate the string there    */
      if(j)
      {
         filename[j] = '\0';
      }

      /* Report error if filename too long                              */
      if((strlen(filename) + strlen(ext) + 1) >= MAXBUFF)
      {
         printf("   Error==> Filename too long to add new extension\n");
         printf("            Fitted file not written: %s\n", 
                gMobFilename[i]);
         return;
      }

      /* Append extension                                               */
      strcat(filename,".");
      strcat(filename, ext);

      /* Write the file                                                 */
      WriteCoordinates(filename, i);
   }
}


/************************************************************************/
/*>char *FindDash(char *buffer)
   ----------------------------
   Find a dash representing a range, skipping any - sign escaped with a 
   \ (\- is used to represent a negative residue number)

   20.02.01 Original   By: ACRM
*/
char *FindDash(char *buffer)
{
   char *dash = NULL;

   dash=strchr(buffer,'-');
   while((dash != NULL) && (dash > buffer) && (*(dash-1) == '\\'))
   {
      buffer = dash+1;
      dash=strchr(buffer,'-');
   }

   return(dash);
}


/************************************************************************/
/*>PDB *ReadXMAS(FILE *fp, int *natoms)
   ------------------------------------
   Like ReadPDB() but reads from an XMAS file instead of a PDB file

   01.03.01 Original   By: ACRM
*/
PDB *ReadXMAS(FILE *fp, int *natoms)
{
   return(doReadXMAS(fp, natoms, TRUE));
}


/************************************************************************/
/*>PDB *ReadXMASAtoms(FILE *fp, int *natoms)
   -----------------------------------------
   Like ReadPDBAtoms() but reads from an XMAS file instead of a PDB file

   01.03.01 Original   By: ACRM
*/
PDB *ReadXMASAtoms(FILE *fp, int *natoms)
{
   return(doReadXMAS(fp, natoms, FALSE));
}


/************************************************************************/
#ifdef USE_XMAS
#  define _LIBACRM_H          /* Stops libacrm.h being included        */
#  define FDWRAP_MAX_BUFF 1024
   typedef struct
   {
      int fd;
      char buffer[FDWRAP_MAX_BUFF];
      int  buffpos,
           maxbuff,
           eof,
           socket;
   }
   FDWRAP;

#  include "xmas.h"
#endif


/************************************************************************/
/*>PDB *doReadXMAS(FILE *fp, int *natoms, int readhet)
   ---------------------------------------------------
   Reads an XMAS file into a PDB linked list

   01.03.01 Original   By: ACRM
   15.03.01 Removed special call to ReadXmasData() - no longer needed as
            XMAS column index is now stored in the XMAS structure
*/
PDB *doReadXMAS(FILE *fp, int *natoms, int readhet)
{
   PDB  *pdb = NULL;

#ifdef USE_XMAS
   XMAS *xmas = NULL;
   PDB  *p;
   char  atnum[16], 
         atnam[16], 
         x[16], 
         y[16], 
         z[16], 
         occup[16], 
         bval[16], 
         resnam[16], 
         resnum[16], 
         chain[16], 
         type[16];

   *natoms = 0;
   
   /* Read the XMAS header                                              */
   if((xmas = ReadXmasHeader(fp))==NULL)
   {
      printf("   Error==> Couldn't read XMAS header: %s\n", gXMASError);
      return(NULL);
   }

   /* Read in the XMAS data                                             */
   if(!CacheXmasData(xmas))
   {
      printf("   Error==> Couldn't read XMAS data: %s\n", gXMASError);
      FreeXmasData(xmas);
      return(NULL);
   }
   
   /* Check the data contains atom records                              */
   if(!DoesXmasContain(xmas, "atoms"))
   {
      fprintf(stderr,"   Error==> XMAS file does not have ATOM \
records!\n");
      FreeXmasData(xmas);
      return(NULL);
   }
   
   /* All looks OK, read the ATOM data into a PDB linked list           */
   while(ReadXmasData(xmas, "atoms",
                      "atnum, atnam, x, y, z, occup, bval, resnam, \
                       resnum, chain, type",
                      atnum, atnam, x, y, z, occup, 
                      bval, resnam, resnum, chain, type))
   {
      if((!strcmp(type, "ATOM")) ||
         (!strcmp(type, "HETATM") && readhet))
      {
         
         /* Allocate memory in the PDB linked list                      */
         if(pdb==NULL)
         {
            INIT(pdb, PDB);
            p=pdb;
         }
         else
         {
            ALLOCNEXT(p, PDB);
         }
         if(p==NULL)
         {
            FREELIST(pdb, PDB);
            *natoms = 0;
            return(NULL);
         }
         
         sscanf(atnum, "%d", &(p->atnum));

         DEDOTIFY(atnam);
         strcat(atnam, " ");
         strcpy(p->atnam_raw, atnam);
         strcpy(p->atnam, FixAtomName(atnam));
         p->atnam[4] = '\0';
         sscanf(x, "%lf", &(p->x));
         sscanf(y, "%lf", &(p->y));
         sscanf(z, "%lf", &(p->z));
         sscanf(occup, "%lf", &(p->occ));
         sscanf(bval,  "%lf", &(p->bval));
         strncpy(p->resnam, resnam, 3);
         strcat(p->resnam, " ");
         
         DEDOTIFY(resnum);
         p->insert[0] = resnum[4];
         p->insert[1] = '\0';
         resnum[4] = '\0';
         sscanf(resnum,  "%d", &(p->resnum));
         
         p->chain[0] = chain[0];
         p->chain[1] = '\0';
         
         strcpy(p->junk, type);
         PADCHARMINTERM(p->junk, ' ', 6);

         (*natoms)++;
      }
   }

   /* Free the XMAS data                                                */
   FreeXmasData(xmas);
   
#endif   

   return(pdb);
}
