/* 
 * Copyright (C) 2002-2005  Simon Baldwin, simon_baldwin@yahoo.com
 * Mac portions Copyright (C) 2002  Ben Hines
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of version 2 of the GNU General Public License 
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  
 * USA 
 */

/*
 * Glk interface for Level9
 * ------------------------
 *
 * This module contains the the Glk porting layer for the Level9
 * interpreter.  It defines the Glk arguments list structure, the
 * entry points for the Glk library framework to use, and all
 * platform-abstracted I/O to link to Glk's I/O.
 *
 * The following items are omitted from this Glk port:
 *
 *  o Glk tries to assert control over _all_ file I/O.  It's just too
 *    disruptive to add it to existing code, so for now, the Level9
 *    interpreter is still dependent on stdio and the like.
 *
 */

#include <assert.h>
#include <errno.h>
#include <stdio.h>
#include <ctype.h>
#include <time.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <stddef.h>

#include "level9.h"

#include "glk.h"

/* File path delimiter, used to be #defined in v2 interpreter. */
#if defined(_Windows) || defined(__MSDOS__) || defined (_WIN32) \
		|| defined (__WIN32__)
static const char	GLN_FILE_DELIM		= '\\';
#else
static const char	GLN_FILE_DELIM		= '/';
#endif


/*---------------------------------------------------------------------*/
/*  Module variables, miscellaneous other stuff                        */
/*---------------------------------------------------------------------*/

/* Glk Level9 port version number. */
static const glui32	GLN_PORT_VERSION		= 0x00020100;

/*
 * We use a maximum of three Glk windows, one for status, one for pictures,
 * and one for everything else.  The status and pictures windows may be
 * NULL, depending on user selections and the capabilities of the Glk
 * library.
 */
static winid_t		gln_main_window			= NULL;
static winid_t		gln_status_window		= NULL;
static winid_t		gln_graphics_window		= NULL;

/*
 * Transcript stream and input log.  These are NULL if there is no
 * current collection of these strings.
 */
static strid_t		gln_transcript_stream		= NULL;
static strid_t		gln_inputlog_stream		= NULL;

/* Input read log stream, for reading back an input log. */
static strid_t		gln_readlog_stream		= NULL;

/* Note about whether graphics is possible, or not. */
static int		gln_graphics_possible		= TRUE;

/* Options that may be turned off by command line flags. */
static int		gln_graphics_enabled		= TRUE;
static int		gln_intercept_enabled		= TRUE;
static int		gln_prompt_enabled		= TRUE;
static int		gln_loopcheck_enabled		= TRUE;
static int		gln_abbreviations_enabled	= TRUE;
static int		gln_commands_enabled		= TRUE;

/* Note indicating whether the Glk library has timers, or not. */
static int		gln_timeouts_possible		= TRUE;

/* Reason for stopping the game, used to detect restarts and ^C exits. */
static enum		{ STOP_NONE, STOP_FORCE, STOP_RESTART, STOP_EXIT }
			gln_stop_reason			= STOP_NONE;

/* Level9 standard input prompt string. */
static const char	*GLN_INPUT_PROMPT		= "> ";

/* Internal interpreter symbols used for our own deviant purposes. */
extern void		save (void);
extern void		restore (void);
extern L9BOOL		Cheating;
extern L9BYTE		*startdata;
extern L9UINT32		FileSize;

/* Forward declaration of event wait functions. */
static void		gln_event_wait (glui32 wait_type, event_t *event);
static void		gln_event_wait_2 (glui32 wait_type_1,
					glui32 wait_type_2, event_t *event);

/* Forward declaration of watchdog tick and standout string functions. */
static void		gln_watchdog_tick (void);
static void		gln_standout_string (const char *message);

/* Forward declaration of the confirmation function. */
static int		gln_confirm (const char *prompt);


/*---------------------------------------------------------------------*/
/*  Glk port utility functions                                         */
/*---------------------------------------------------------------------*/

/*
 * gln_fatal()
 *
 * Fatal error handler.  The function returns, expecting the caller to
 * abort() or otherwise handle the error.
 */
void
gln_fatal (const char *string)
{
	/*
	 * If the failure happens too early for us to have a window, print
	 * the message to stderr.
	 */
	if (gln_main_window == NULL)
	    {
		fprintf (stderr, "\n\nINTERNAL ERROR: ");
		fprintf (stderr, "%s", string);
		fprintf (stderr, "\n");

		fprintf (stderr, "\nPlease record the details of this error,");
		fprintf (stderr, " try to note down everything you did to");
		fprintf (stderr, " cause it, and email this information to");
		fprintf (stderr, " simon_baldwin@yahoo.com.\n\n");
		return;
	    }

	/* Cancel all possible pending window input events. */
	glk_cancel_line_event (gln_main_window, NULL);
	glk_cancel_char_event (gln_main_window);

	/* Print a message indicating the error, and exit. */
	glk_set_window (gln_main_window);
	glk_set_style (style_Normal);
	glk_put_string ("\n\nINTERNAL ERROR: ");
	glk_put_string ((char *) string);
	glk_put_string ("\n");

	glk_put_string ("\nPlease record the details of this error,");
	glk_put_string (" try to note down everything you did to");
	glk_put_string (" cause it, and email this information to");
	glk_put_string (" simon_baldwin@yahoo.com.\n\n");
}


/*
 * gln_malloc()
 *
 * Non-failing malloc and realloc; call gln_fatal and exit if memory
 * allocation fails.
 */
static void *
gln_malloc (size_t size)
{
	void		*pointer;		/* Return value pointer. */

	/* Malloc, and call gln_fatal if the malloc fails. */
	pointer = malloc (size);
	if (pointer == NULL)
	    {
		gln_fatal ("GLK: Out of system memory");
		glk_exit ();
	    }

	/* Return the allocated pointer. */
	return pointer;
}

static void *
gln_realloc (void *ptr, size_t size)
{
	void		*pointer;		/* Return value pointer. */

	/* Realloc, and call gln_fatal() if the realloc fails. */
	pointer = realloc (ptr, size);
	if (pointer == NULL)
	    {
		gln_fatal ("GLK: Out of system memory");
		glk_exit ();
	    }

	/* Return the allocated pointer. */
	return pointer;
}


/*
 * gln_strncasecmp()
 * gln_strcasecmp()
 *
 * Strncasecmp and strcasecmp are not ANSI functions, so here are local
 * definitions to do the same jobs.
 *
 * They're global here so that the core interpreter can use them; otherwise
 * it tries to use the non-ANSI str[n]icmp() functions.
 */
int
gln_strncasecmp (const char *s1, const char *s2, size_t n)
{
	size_t		index;			/* Strings iterator. */

	/* Compare each character, return +/- 1 if not equal. */
	for (index = 0; index < n; index++)
	    {
		if (glk_char_to_lower (s1[ index ])
				< glk_char_to_lower (s2[ index ]))
			return -1;
		else if (glk_char_to_lower (s1[ index ])
				> glk_char_to_lower (s2[ index ]))
			return  1;
	    }

	/* Strings are identical to n characters. */
	return 0;
}
int
gln_strcasecmp (const char *s1, const char *s2)
{
	size_t		s1len, s2len;		/* String lengths. */
	int		result;			/* Result of strncasecmp. */

	/* Note the string lengths. */
	s1len = strlen (s1);
	s2len = strlen (s2);

	/* Compare first to shortest length, return any definitive result. */
	result = gln_strncasecmp (s1, s2, (s1len < s2len) ? s1len : s2len);
	if (result != 0)
		return result;

	/* Return the longer string as being the greater. */
	if (s1len < s2len)
		return -1;
	else if (s1len > s2len)
		return  1;

	/* Same length, and same characters -- identical strings. */
	return 0;
}


/*---------------------------------------------------------------------*/
/*  Glk port stub graphics functions                                   */
/*---------------------------------------------------------------------*/

/*
 * If we're working with a very stripped down, old, or lazy Glk library
 * that neither offers Glk graphics nor graphics stubs functions, here
 * we define our own stubs, to avoid link-time errors.
 */
#ifndef GLK_MODULE_IMAGE
static glui32	glk_image_draw (winid_t win, glui32 image,
			glsi32 val1, glsi32 val2)	 { return FALSE; }
static glui32	glk_image_draw_scaled (winid_t win, glui32 image, 
			glsi32 val1, glsi32 val2,
			glui32 width, glui32 height)	 { return FALSE; }
static glui32	glk_image_get_info (glui32 image,
			glui32 *width, glui32 *height)	 { return FALSE; }
static void	glk_window_flow_break (winid_t win)			{}
static void	glk_window_erase_rect (winid_t win, 
			glsi32 left, glsi32 top,
			glui32 width, glui32 height)			{}
static void	glk_window_fill_rect (winid_t win, glui32 color, 
			glsi32 left, glsi32 top,
			glui32 width, glui32 height)			{}
static void	glk_window_set_background_color (winid_t win,
			glui32 color)					{}
#endif


/*---------------------------------------------------------------------*/
/*  Glk port CRC functions                                             */
/*---------------------------------------------------------------------*/

/* CRC table initialization magic number. */
static const L9UINT16	GLN_CRC_MAGIC			= 0xA001;


/*
 * gln_update_crc()
 *
 * Update a running CRC with the bytes buffer[0..length-1] -- the CRC should
 * be initialized to all 1's, and the transmitted value is the 1's complement
 * of the final running CRC.
 */
static L9UINT16
gln_update_crc (L9UINT16 crc, const L9BYTE *buffer, size_t length)
{
	static int	initialized	= FALSE;/* First call flag. */
	static L9UINT16	crc_table[UCHAR_MAX+1];	/* CRC table for all bytes. */

	L9UINT16	c;			/* Updated CRC, for return. */
	size_t		index;			/* Buffer index. */

	/* Build the CRC lookup table if this is the first call. */
	if (!initialized)
	    {
		int	n, k;			/* General loop indexes. */

		/* Create a table entry for each byte value. */
		for (n = 0; n < UCHAR_MAX + 1; n++)
		    {
			c = (L9UINT16) n;
			for (k = 0; k < CHAR_BIT; k++)
			    {
				if (c & 1)
					c = GLN_CRC_MAGIC ^ (c >> 1);
				else
					c = c >> 1;
			    }
			crc_table[ n ] = c;
		    }

		/* Note the table is built. */
		initialized = TRUE;
	    }

	/* Update the CRC with each character in the buffer. */
	c = crc;
	for (index = 0; index < length; index++)
		c = crc_table[ (c ^ buffer[ index ]) & UCHAR_MAX ]
							^ (c >> CHAR_BIT);

	/* Return the updated CRC. */
	return c;
}


/*
 * gln_buffer_crc()
 *
 * Return the CRC of the bytes buffer[0..length-1].
 *
 * Ordinarily, a CRC would begin at 0xFFFF, and the final value be XORed
 * with 0xFFFF before return.
 * 
 * However, Level9 CRCs aren't done that way; instead, we just return the
 * base value obtained by running the data through the table.
 */
static L9UINT16
gln_buffer_crc (const L9BYTE *buffer, size_t length)
{
	return gln_update_crc (0, buffer, length);
}


/*---------------------------------------------------------------------*/
/*  Glk port game identification data                                  */
/*---------------------------------------------------------------------*/

/*
 * The following game database is obtained from L9cut's l9data_d.h, and
 * allows a game's name to be found from its data CRC.  Entries marked
 * "WANTED" in l9data_d.h, and file commentary, have been removed for
 * brevity, and the file has been reformatted slightly.
 *
 * The version of l9data_d.h used is 050 (22 Oct 2002).
 */
typedef struct {
	const L9BYTE	version;	/* V2/V3/V4, V5 is split V4 */
	const L9UINT16	length;		/* datafile length, bytes */
	const L9BYTE	checksum;	/* 8-bit checksum, last datafile byte */
	const L9UINT16	crcvalue;	/* 16-bit CRC, l9cut-internal */
	const L9BYTE	patchstate;	/* 0=unprotected original,
					   1=patched, ---- REMOVED
					   2=original with Lenslok protection,
					   3=original with manual protection */
	const char	*game;		/* Game title and platform */
} gln_version_table_t;
typedef const gln_version_table_t	*gln_version_tableref_t;

static const gln_version_table_t	GLN_VERSION_TABLE[] = {
	{ 2, 0x5323, 0xb7, 0x8af7, 0, "Adventure Quest (CPC/Spectrum)" },

	{ 2, 0x630e, 0x8d, 0x7d7d, 0, "Dungeon Adventure (CPC)" },
	{ 2, 0x630e, 0xbe, 0x3374, 0, "Dungeon Adventure (MSX)" },

	{ 2, 0x5eb9, 0x30, 0xe99a, 0, "Lords of Time (CPC)" },
	{ 2, 0x5eb9, 0x5d, 0xc098, 0, "Lords of Time (MSX)" },
	{ 2, 0x5eb9, 0x6e, 0xc689, 0, "Lords of Time (Spectrum)" },

	{ 2, 0x5fab, 0x5c, 0xa309, 0, "Snowball (CPC)" },
	{ 2, 0x5fab, 0x2f, 0x8aa2, 0, "Snowball (MSX)" },

	{ 2, 0x60c4, 0x28, 0x0154, 0, "Return to Eden (CPC/C-64[v1])" },
	{ 2, 0x6064, 0x01, 0x5b3c, 0, "Return to Eden (BBC[v1])" },
	{ 2, 0x6064, 0x95, 0x510c, 0, "Return to Eden (C-64[v2])" },
	{ 2, 0x6064, 0xda, 0xe610, 0, "Return to Eden (C-64[v2] *corrupt*)" },
	{ 2, 0x6064, 0xbd, 0x73ec, 0, "Return to Eden (Atari *corrupt*)" },
	{ 2, 0x6047, 0x6c, 0x17ab, 0, "Return to Eden (BBC[v2])" },
	{ 2, 0x5ca1, 0x33, 0x1c43, 0, "Return to Eden (Spectrum[v1])" },
	{ 2, 0x5cb7, 0x64, 0x0790, 0, "Return to Eden (Spectrum[v2])" },
	{ 2, 0x5cb7, 0xfe, 0x3533, 0, "Return to Eden (MSX)" },

	{ 2, 0x34b3, 0x20, 0xccda, 0, "Erik the Viking (BBC/C-64)" },
	{ 2, 0x34b3, 0x53, 0x8f00, 0, "Erik the Viking (Spectrum)" },
	{ 2, 0x34b3, 0xc7, 0x9058, 0, "Erik the Viking (CPC)" },

	{ 2, 0x63be, 0xd6, 0xcf5d, 0,
		"Emerald Isle (Atari/C-64/CPC/Spectrum)" },
	{ 2, 0x63be, 0x0a, 0x21ed, 0, "Emerald Isle (MSX *corrupt*)" },
	{ 2, 0x378c, 0x8d, 0x3a21, 0, "Emerald Isle (BBC)" },

	{ 2, 0x506c, 0xf0, 0xba72, 0, "Red Moon (BBC/C-64/CPC/MSX)" },
	{ 2, 0x505d, 0x32, 0x2dcf, 0, "Red Moon (Spectrum)" },

	{ 3, 0x772b, 0xcd, 0xa503, 0, "Worm in Paradise (Spectrum128)" },
	{ 3, 0x546c, 0xb7, 0x9420, 0, "Worm in Paradise (Spectrum48)" },
	{ 3, 0x6d84, 0xf9, 0x49ae, 0, "Worm in Paradise (C-64 *corrupt*)" },
	{ 3, 0x6d84, 0xc8, 0x943f, 0, "Worm in Paradise (C-64 *fixed*)" },
	{ 3, 0x6030, 0x47, 0x46ad, 0, "Worm in Paradise (CPC)" },
	{ 3, 0x5828, 0xbd, 0xe7cb, 0, "Worm in Paradise (BBC)" },

	{ 3, 0x7410, 0x5e, 0x60be, 2, "Price of Magik (Spectrum128)" },
	{ 3, 0x5aa4, 0xc1, 0x10a0, 2, "Price of Magik (Spectrum48[v1])" },
	{ 3, 0x5aa4, 0xc1, 0xeda4, 2, "Price of Magik (Spectrum48[v2])" },
	{ 3, 0x6fc6, 0x14, 0xf9b6, 2, "Price of Magik (C-64)" },
	{ 3, 0x5aa4, 0xc1, 0xbbf4, 2, "Price of Magik (CPC)" },
	{ 3, 0x5671, 0xbc, 0xff35, 0, "Price of Magik (BBC)" },

	{ 3, 0x76f4, 0x5e, 0x1fe5, 3, "Colossal Adventure /JoD (Amiga/PC)" },
	{ 3, 0x76f4, 0x5a, 0xcf4b, 3, "Colossal Adventure /JoD (ST)" },
	{ 3, 0x6e60, 0x83, 0x18e0, 3, "Adventure Quest /JoD (Amiga/PC)" },
	{ 3, 0x6e5c, 0xf6, 0xd356, 3, "Adventure Quest /JoD (ST)" },
	{ 3, 0x6f0c, 0x95, 0x1f64, 3, "Dungeon Adventure /JoD (Amiga/PC/ST)" },

	{ 3, 0x6f70, 0x40, 0xbd91, 2, "Colossal Adventure /JoD (MSX)" },

	{ 3, 0x6f6e, 0x78, 0x28cd, 2, "Colossal Adventure /JoD (Spectrum128)" },
	{ 3, 0x6970, 0xd6, 0xa820, 2, "Adventure Quest /JoD (Spectrum128)" },
	{ 3, 0x6de8, 0x4c, 0xd795, 2, "Dungeon Adventure /JoD (Spectrum128)" },

	{ 3, 0x6f4d, 0xcb, 0xe8f2, 2,
		"Colossal Adventure /JoD (CPC128[v1]/Spectrum+3)" },
	{ 3, 0x6f6a, 0xa5, 0x8dd2, 2, "Colossal Adventure /JoD (CPC128[v2])" },
	{ 3, 0x6968, 0x32, 0x0c01, 2,
		"Adventure Quest /JoD (CPC128/Spectrum+3)" },
	{ 3, 0x6dc0, 0x63, 0x5d95, 2,
		"Dungeon Adventure /JoD (CPC128/Spectrum+3)" },

	{ 3, 0x5e31, 0x7c, 0xaa54, 2, "Colossal Adventure /JoD (CPC64)" },
	{ 3, 0x5b50, 0x66, 0x1800, 2, "Adventure Quest /JoD (CPC64)" },
	{ 3, 0x58a6, 0x24, 0xb50f, 2, "Dungeon Adventure /JoD (CPC64)" },

	{ 3, 0x6c8e, 0xb6, 0x9be3, 2, "Colossal Adventure /JoD (C-64)" },
	{ 3, 0x63b6, 0x2e, 0xef38, 2, "Adventure Quest /JoD (C-64)" },
	{ 3, 0x6bd2, 0x65, 0xa41f, 2, "Dungeon Adventure /JoD (C-64)" },

	{ 3, 0x5b16, 0x3b, 0xe2aa, 2, "Colossal Adventure /JoD (Atari)" },
	{ 3, 0x5b58, 0x50, 0x332e, 2, "Adventure Quest /JoD (Atari)" },
	{ 3, 0x593a, 0x80, 0x7a34, 2, "Dungeon Adventure /JoD (Atari)" },

	{ 3, 0x5a8e, 0xf2, 0x7cca, 2, "Colossal Adventure /JoD (Spectrum48)" },
	{ 3, 0x5ace, 0x11, 0xdc12, 2, "Adventure Quest /JoD (Spectrum48)" },
	{ 3, 0x58a3, 0x38, 0x8ce4, 2, "Dungeon Adventure /JoD (Spectrum48)" },

	{ 3, 0x7b31, 0x6e, 0x2e2b, 3, "Snowball /SD (Amiga/ST)" },
	{ 3, 0x7d16, 0xe6, 0x5438, 3, "Return to Eden /SD (Amiga/ST)" },
	{ 3, 0x7cd9, 0x0c, 0x4df1, 3, "Worm in Paradise /SD (Amiga/ST)" },

	{ 3, 0x7b2f, 0x70, 0x6955, 3, "Snowball /SD (Mac/PC/Spectrum128)" },
	{ 3, 0x7b2f, 0x70, 0x6f6c, 3, "Snowball /SD (CPC/Spectrum+3)" },
	{ 3, 0x7d14, 0xe8, 0xfbab, 3, "Return to Eden /SD (PC)" },
	{ 3, 0x7cff, 0xf8, 0x6044, 3, "Return to Eden /SD (CPC/Spectrum+3)" },
	{ 3, 0x7cf8, 0x24, 0x9c1c, 3, "Return to Eden /SD (Mac)" },
	{ 3, 0x7c55, 0x18, 0xdaee, 3, "Return to Eden /SD (Spectrum128)" },
	{ 3, 0x7cd7, 0x0e, 0x4feb, 3,
		"Worm in Paradise /SD (CPC/Mac/PC/Spectrum128/Spectrum+3)" },

	{ 3, 0x7363, 0x65, 0xa0ab, 3, "Snowball /SD (C-64)" },
	{ 3, 0x772f, 0xca, 0x8602, 3, "Return to Eden /SD (C-64)" },
	{ 3, 0x788d, 0x72, 0x888a, 3, "Worm in Paradise /SD (C-64)" },

	{ 3, 0x6bf8, 0x3f, 0xc9f7, 3, "Snowball /SD (Atari)" },
	{ 3, 0x60f7, 0x68, 0xc2bc, 3, "Return to Eden /SD (Atari)" },
	{ 3, 0x6161, 0xf3, 0xe6d7, 3, "Worm in Paradise /SD (Atari)" },

	{ 3, 0x67a3, 0x9d, 0x1d05, 3, "Snowball /SD (Apple ][)" },
	{ 3, 0x639c, 0x8b, 0x06e2, 3, "Return to Eden /SD (Apple ][)" },
	{ 3, 0x60dd, 0xf2, 0x5bb8, 3, "Worm in Paradise /SD (Apple ][)" },

	{ 3, 0x6541, 0x02, 0x2e6c, 3, "Snowball /SD (Spectrum48)" },
	{ 3, 0x5f43, 0xca, 0x828c, 3, "Return to Eden /SD (Spectrum48)" },
	{ 3, 0x5ebb, 0xf1, 0x4dec, 3, "Worm in Paradise /SD (Spectrum48)" },

	{ 3, 0x8333, 0xb7, 0xe2ac, 0, "Adrian Mole I, pt. 1 (C-64)" },
	{ 3, 0x844d, 0x50, 0x5353, 0, "Adrian Mole I, pt. 2 (C-64)" },
	{ 3, 0x8251, 0x5f, 0x862a, 0, "Adrian Mole I, pt. 3 (C-64)" },
	{ 3, 0x7a78, 0x5e, 0x6ea3, 0, "Adrian Mole I, pt. 4 (C-64)" },

	{ 3, 0x7c6f, 0x0f, 0xba24, 0, "Adrian Mole I, pt. 1 (CPC)" },

	{ 3, 0x72fa, 0x8b, 0x6f12, 0, "Adrian Mole I, pt. 1 (Spectrum)" },
	{ 3, 0x738e, 0x5b, 0x7e3d, 0, "Adrian Mole I, pt. 2 (Spectrum)" },
	{ 3, 0x7375, 0xe5, 0x3f3e, 0, "Adrian Mole I, pt. 3 (Spectrum)" },
	{ 3, 0x78d5, 0xe3, 0xcd7d, 0, "Adrian Mole I, pt. 4 (Spectrum)" },

	{ 3, 0x3a31, 0xe5, 0x0bdb, 0, "Adrian Mole I, pt. 1 (BBC)" },
	{ 3, 0x37f1, 0x77, 0xd231, 0, "Adrian Mole I, pt. 2 (BBC)" },
	{ 3, 0x3900, 0x1c, 0x5d9a, 0, "Adrian Mole I, pt. 3 (BBC)" },
	{ 3, 0x3910, 0xac, 0x07f9, 0, "Adrian Mole I, pt. 4 (BBC)" },
	{ 3, 0x3ad6, 0xa7, 0x95d2, 0, "Adrian Mole I, pt. 5 (BBC)" },
	{ 3, 0x38a5, 0x0f, 0xdefc, 0, "Adrian Mole I, pt. 6 (BBC)" },
	{ 3, 0x361e, 0x7e, 0xfd9f, 0, "Adrian Mole I, pt. 7 (BBC)" },
	{ 3, 0x3934, 0x75, 0xe141, 0, "Adrian Mole I, pt. 8 (BBC)" },
	{ 3, 0x3511, 0xcc, 0xd829, 0, "Adrian Mole I, pt. 9 (BBC)" },
	{ 3, 0x38dd, 0x31, 0x2534, 0, "Adrian Mole I, pt. 10 (BBC)" },
	{ 3, 0x39c0, 0x44, 0x89df, 0, "Adrian Mole I, pt. 11 (BBC)" },
	{ 3, 0x3a12, 0x8f, 0xc2bd, 0, "Adrian Mole I, pt. 12 (BBC)" },

	{ 3, 0x7931, 0xb9, 0xc51b, 0, "Adrian Mole II, pt. 1 (C-64/CPC)" },
	{ 3, 0x7cdf, 0xa5, 0x43e3, 0, "Adrian Mole II, pt. 2 (C-64/CPC)" },
	{ 3, 0x7a0c, 0x97, 0x4bea, 0, "Adrian Mole II, pt. 3 (C-64/CPC)" },
	{ 3, 0x7883, 0xe2, 0xee0e, 0, "Adrian Mole II, pt. 4 (C-64/CPC)" },

	{ 3, 0x6841, 0x4a, 0x94e7, 0, "Adrian Mole II, pt. 1 (Spectrum)" },
	{ 3, 0x6bc0, 0x62, 0xab3d, 0, "Adrian Mole II, pt. 2 (Spectrum)" },
	{ 3, 0x692c, 0x21, 0x2015, 0, "Adrian Mole II, pt. 3 (Spectrum)" },
	{ 3, 0x670a, 0x94, 0xa2a6, 0, "Adrian Mole II, pt. 4 (Spectrum)" },

	{ 3, 0x593a, 0xaf, 0x30e9, 0, "Adrian Mole II, pt. 1 (BBC)" },
	{ 3, 0x57e6, 0x8a, 0xc41a, 0, "Adrian Mole II, pt. 2 (BBC)" },
	{ 3, 0x5819, 0xcd, 0x1ba0, 0, "Adrian Mole II, pt. 3 (BBC)" },
	{ 3, 0x579b, 0xad, 0xa723, 0, "Adrian Mole II, pt. 4 (BBC)" },

	{ 3, 0x765d, 0xcd, 0xfc02, 0, "The Archers, pt. 1 (C-64)" },
	{ 3, 0x6e58, 0x07, 0xbffc, 0, "The Archers, pt. 2 (C-64)" },
	{ 3, 0x7e98, 0x6a, 0x95e5, 0, "The Archers, pt. 3 (C-64)" },
	{ 3, 0x81e2, 0xd5, 0xb278, 0, "The Archers, pt. 4 (C-64)" },

	{ 3, 0x6ce5, 0x58, 0x46de, 0, "The Archers, pt. 1 (Spectrum)" },
	{ 3, 0x68da, 0xc1, 0x3b8e, 0, "The Archers, pt. 2 (Spectrum)" },
	{ 3, 0x6c67, 0x9a, 0x9a6a, 0, "The Archers, pt. 3 (Spectrum)" },
	{ 3, 0x6d91, 0xb9, 0x12a7, 0, "The Archers, pt. 4 (Spectrum)" },

	{ 3, 0x5834, 0x42, 0xcc9d, 0, "The Archers, pt. 1 (BBC)" },
	{ 3, 0x56dd, 0x51, 0xe582, 0, "The Archers, pt. 2 (BBC)" },
	{ 3, 0x5801, 0x53, 0xf2ef, 0, "The Archers, pt. 3 (BBC)" },
	{ 3, 0x54a4, 0x01, 0xc0ab, 0, "The Archers, pt. 4 (BBC)" },

	{ 5, 0x579e, 0x97, 0x9faa, 0, "Lords of Time /T&M GD (BBC)" },
	{ 5, 0x5500, 0x50, 0xca75, 0, "Red Moon /T&M GD (BBC)" },
	{ 5, 0x579a, 0x2a, 0x9373, 0, "Price of Magik /T&M GD (BBC)" },

	{ 5, 0x4fd2, 0x9d, 0x799a, 0, "Lancelot, pt. 1 GD (BBC)" },
	{ 5, 0x4dac, 0xa8, 0x86ed, 0, "Lancelot, pt. 2 GD (BBC)" },
	{ 5, 0x4f96, 0x22, 0x30f8, 0, "Lancelot, pt. 3 GD (BBC)" },

	{ 5, 0x55ce, 0xa1, 0xba12, 0, "Scapeghost, pt. 1 GD (BBC)" },
	{ 5, 0x54a6, 0xa9, 0xc9f3, 0, "Scapeghost, pt. 2 GD (BBC)" },
	{ 5, 0x51bc, 0xe3, 0x89c3, 0, "Scapeghost, pt. 3 GD (BBC)" },

	{ 5, 0x46ec, 0x64, 0x2300, 0, "Knight Orc, pt. 1 GD (CPC/Spectrum+3)" },
	{ 5, 0x6140, 0x18, 0x4f66, 0, "Knight Orc, pt. 2 GD (CPC/Spectrum+3)" },
	{ 5, 0x640e, 0xc1, 0xfc69, 0, "Knight Orc, pt. 3 GD (CPC/Spectrum+3)" },

	{ 5, 0x5ff0, 0xf8, 0x3a13, 0,
		"Gnome Ranger, pt. 1 GD (CPC/Spectrum+3)" },
	{ 5, 0x6024, 0x01, 0xaaa9, 0,
		"Gnome Ranger, pt. 2 GD (CPC/Spectrum+3)" },
	{ 5, 0x6036, 0x3d, 0x6c6c, 0,
		"Gnome Ranger, pt. 3 GD (CPC/Spectrum+3)" },

	{ 5, 0x69fe, 0x56, 0xecfb, 0,
		"Lords of Time /T&M GD (CPC/Spectrum+3)" },
	{ 5, 0x6888, 0x8d, 0x7f6a, 0, "Red Moon /T&M GD (CPC/Spectrum+3)" },
	{ 5, 0x5a50, 0xa9, 0xa5fa, 0,
		"Price of Magik /T&M GD (CPC/Spectrum+3)" },

	{ 5, 0x5c7a, 0x44, 0x460e, 0, "Lancelot, pt. 1 GD (CPC/Spectrum+3)" },
	{ 5, 0x53a2, 0x1e, 0x2fae, 0, "Lancelot, pt. 2 GD (CPC/Spectrum+3)" },
	{ 5, 0x5914, 0x22, 0x4a31, 0, "Lancelot, pt. 3 GD (CPC/Spectrum+3)" },

	{ 5, 0x5a38, 0xf7, 0x876e, 0,
		"Ingrid's Back, pt. 1 GD (CPC/Spectrum+3)" },
	{ 5, 0x531a, 0xed, 0xcf3f, 0,
		"Ingrid's Back, pt. 2 GD (CPC/Spectrum+3)" },
	{ 5, 0x57e4, 0x19, 0xb354, 0,
		"Ingrid's Back, pt. 3 GD (CPC/Spectrum+3)" },

	{ 5, 0x5cbc, 0xa5, 0x0dbe, 0, "Scapeghost, pt. 1 GD (CPC/Spectrum+3)" },
	{ 5, 0x5932, 0x4e, 0xb2f5, 0, "Scapeghost, pt. 2 GD (CPC/Spectrum+3)" },
	{ 5, 0x5860, 0x95, 0x3227, 0, "Scapeghost, pt. 3 GD (CPC/Spectrum+3)" },

	{ 5, 0x74e0, 0x92, 0x885e, 0, "Knight Orc, pt. 1 GD (Spectrum128)" },
	{ 5, 0x6dbc, 0x97, 0x6f55, 0, "Knight Orc, pt. 2 GD (Spectrum128)" },
	{ 5, 0x7402, 0x07, 0x385f, 0, "Knight Orc, pt. 3 GD (Spectrum128)" },

	{ 5, 0x52aa, 0xdf, 0x7b5b, 0, "Gnome Ranger, pt. 1 GD (Spectrum128)" },
	{ 5, 0x6ffa, 0xdb, 0xdde2, 0, "Gnome Ranger, pt. 2 GD (Spectrum128)" },
	{ 5, 0x723a, 0x69, 0x039b, 0, "Gnome Ranger, pt. 3 GD (Spectrum128)" },

	{ 5, 0x6f1e, 0xda, 0x2ce0, 0, "Lords of Time /T&M GD (Spectrum128)" },
	{ 5, 0x6da0, 0xb8, 0x3802, 0, "Red Moon /T&M GD (Spectrum128)" },
	{ 5, 0x6108, 0xdd, 0xefe7, 0, "Price of Magik /T&M GD (Spectrum128)" },

	{ 5, 0x768c, 0xe8, 0x8fc6, 0, "Lancelot, pt. 1 GD (Spectrum128)" },
	{ 5, 0x76b0, 0x1d, 0x0fcd, 0, "Lancelot, pt. 2 GD (Spectrum128)" },
	{ 5, 0x765e, 0x4f, 0x3b73, 0, "Lancelot, pt. 3 GD (Spectrum128)" },

	{ 5, 0x76a0, 0x3a, 0xb803, 0, "Ingrid's Back, pt. 1 GD (Spectrum128)" },
	{ 5, 0x7674, 0x0b, 0xe92f, 0, "Ingrid's Back, pt. 2 GD (Spectrum128)" },
	{ 5, 0x765e, 0xba, 0x086d, 0, "Ingrid's Back, pt. 3 GD (Spectrum128)" },

	{ 5, 0x762e, 0x82, 0x8848, 0, "Scapeghost, pt. 1 GD (Spectrum128)" },
	{ 5, 0x5bd6, 0x35, 0x79ef, 0, "Scapeghost, pt. 2 GD (Spectrum128)" },
	{ 5, 0x6fa8, 0xa4, 0x62c2, 0, "Scapeghost, pt. 3 GD (Spectrum128)" },

	{ 4, 0xbb93, 0x36, 0x6a05, 3, "Knight Orc, pt. 1 (Amiga)" },
	{ 4, 0xbb6e, 0xad, 0x4d40, 3, "Knight Orc, pt. 1 (ST)" },
	{ 4, 0xc58e, 0x4a, 0x4e9d, 3, "Knight Orc, pt. 2 (Amiga/ST)" },
	{ 4, 0xcb9a, 0x0f, 0x0804, 3, "Knight Orc, pt. 3 (Amiga/ST)" },

	{ 4, 0xbb6e, 0xa6, 0x9753, 3, "Knight Orc, pt. 1 (PC)" },
	{ 4, 0xc58e, 0x43, 0xe9ce, 3, "Knight Orc, pt. 2 (PC)" },
	{ 4, 0xcb9a, 0x08, 0x6c36, 3, "Knight Orc, pt. 3 (PC)" },

	{ 4, 0x898a, 0x43, 0xfc8b, 3, "Knight Orc, pt. 1 (Apple ][)" },
	{ 4, 0x8b9f, 0x61, 0x7288, 0, "Knight Orc, pt. 2 (Apple ][)" },
	{ 4, 0x8af9, 0x61, 0x7542, 0, "Knight Orc, pt. 3 (Apple ][)" },

	{ 4, 0x8970, 0x6b, 0x3c7b, 3, "Knight Orc, pt. 1 (C-64 Gfx)" },
	{ 4, 0x8b90, 0x4e, 0x098c, 0, "Knight Orc, pt. 2 (C-64 Gfx)" },
	{ 4, 0x8aea, 0x4e, 0xca54, 0, "Knight Orc, pt. 3 (C-64 Gfx)" },

	{ 4, 0x86d0, 0xb7, 0xadbd, 3, "Knight Orc, pt. 1 (Spectrum48)" },
	{ 4, 0x8885, 0x22, 0xe293, 0, "Knight Orc, pt. 2 (Spectrum48)" },
	{ 4, 0x87e5, 0x0e, 0xdc33, 0, "Knight Orc, pt. 3 (Spectrum48)" },

	{ 4, 0xb1a9, 0x80, 0x5fb7, 3, "Gnome Ranger, pt. 1 (Amiga/ST)" },
	{ 4, 0xab9d, 0x31, 0xbe6d, 3, "Gnome Ranger, pt. 2 (Amiga/ST)" },
	{ 4, 0xae28, 0x87, 0xb6b6, 3, "Gnome Ranger, pt. 3 (Amiga/ST)" },

	{ 4, 0xb0ec, 0xc2, 0x0053, 3, "Gnome Ranger, pt. 1 (ST[v1])" },
	{ 4, 0xaf82, 0x83, 0x19f7, 3, "Gnome Ranger, pt. 2 (ST[v1])" },

	{ 4, 0xb1aa, 0xad, 0xaf47, 3, "Gnome Ranger, pt. 1 (PC)" },
	{ 4, 0xb19e, 0x92, 0x8f96, 3, "Gnome Ranger, pt. 1 (ST[v2])" },
	{ 4, 0xab8b, 0xbf, 0x31e6, 3, "Gnome Ranger, pt. 2 (PC/ST[v2])" },
	{ 4, 0xae16, 0x81, 0x8741, 3, "Gnome Ranger, pt. 3 (PC/ST[v2])" },

	{ 4, 0xad41, 0xa8, 0x42c5, 3, "Gnome Ranger, pt. 1 (C-64 TO)" },
	{ 4, 0xa735, 0xf7, 0x2e08, 3, "Gnome Ranger, pt. 2 (C-64 TO)" },
	{ 4, 0xa9c0, 0x9e, 0x0d70, 3, "Gnome Ranger, pt. 3 (C-64 TO)" },

	{ 4, 0x908e, 0x0d, 0x58a7, 3, "Gnome Ranger, pt. 1 (C-64 Gfx)" },
	{ 4, 0x8f6f, 0x0a, 0x411a, 3, "Gnome Ranger, pt. 2 (C-64 Gfx)" },
	{ 4, 0x9060, 0xbb, 0xe75d, 3, "Gnome Ranger, pt. 3 (C-64 Gfx)" },

	{ 4, 0x8aab, 0xc0, 0xde5f, 3, "Gnome Ranger, pt. 1 (Spectrum48)" },
	{ 4, 0x8ac8, 0x9a, 0xc89b, 3, "Gnome Ranger, pt. 2 (Spectrum48)" },
	{ 4, 0x8a93, 0x4f, 0x10cc, 3, "Gnome Ranger, pt. 3 (Spectrum48)" },

	{ 4, 0xb57c, 0x44, 0x7779, 3, "Lords of Time /T&M (PC)" },
	{ 4, 0xa69e, 0x6c, 0xb268, 3, "Red Moon /T&M (PC)" },
	{ 4, 0xbac7, 0x7f, 0xddb2, 3, "Price of Magik /T&M (PC)" },

	{ 4, 0xb579, 0x89, 0x3e89, 3, "Lords of Time /T&M (ST)" },
	{ 4, 0xa698, 0x41, 0xcaca, 3, "Red Moon /T&M (ST)" },
	{ 4, 0xbac4, 0x80, 0xa750, 3, "Price of Magik /T&M (ST)" },

	{ 4, 0xb576, 0x2a, 0x7239, 3, "Lords of Time /T&M (Amiga)" },
	{ 4, 0xa692, 0xd1, 0x6a99, 3, "Red Moon /T&M (Amiga)" },
	{ 4, 0xbaca, 0x3a, 0x221b, 3, "Price of Magik /T&M (Amiga)" },

	{ 4, 0xb563, 0x6a, 0x0c5c, 3, "Lords of Time /T&M (Mac)" },
	{ 4, 0xa67c, 0xb8, 0xff41, 3, "Red Moon /T&M (Mac)" },
	{ 4, 0xbab2, 0x87, 0x09f5, 3, "Price of Magik /T&M (Mac)" },

	{ 4, 0xb38c, 0x37, 0x9f8e, 3, "Lords of Time /T&M (C-64 TO)" },
	{ 4, 0xa4e2, 0xa6, 0x016d, 3, "Red Moon /T&M (C-64 TO)" },
	{ 4, 0xb451, 0xa8, 0x2682, 3, "Price of Magik /T&M (C-64 TO)" },

	{ 4, 0x9070, 0x43, 0x45d4, 3, "Lords of Time /T&M (C-64 Gfx)" },
	{ 4, 0x903f, 0x6b, 0x603e, 3, "Red Moon /T&M (C-64 Gfx)" },
	{ 4, 0x8f51, 0xb2, 0x6c9a, 3, "Price of Magik /T&M (C-64 Gfx)" },

	{ 4, 0x8950, 0xa1, 0xbb16, 3, "Lords of Time /T&M (Spectrum48)" },
	{ 4, 0x8813, 0x11, 0x22de, 3, "Red Moon /T&M (Spectrum48)" },
	{ 4, 0x8a60, 0x2a, 0x29ed, 3, "Price of Magik /T&M (Spectrum48)" },

	{ 4, 0xb260, 0xe5, 0xc5b2, 0, "Lords of Time /T&M (PC/ST *USA*)" },
	{ 4, 0xa3a4, 0xdf, 0x6732, 0, "Red Moon /T&M (PC/ST *USA*)" },
	{ 4, 0xb7a0, 0x7e, 0x2226, 0, "Price of Magik /T&M (PC/ST *USA*)" },

	{ 4, 0xb257, 0xf8, 0xfbd5, 0, "Lords of Time /T&M (Amiga *USA*)" },
	{ 4, 0xa398, 0x82, 0xd031, 0, "Red Moon /T&M (Amiga *USA*)" },
	{ 4, 0xb797, 0x1f, 0x84a9, 0, "Price of Magik /T&M (Amiga *USA*)" },

	{ 4, 0x8d78, 0x3a, 0xba6e, 0, "Lords of Time /T&M (C-64 Gfx *USA*)" },
	{ 4, 0x8d56, 0xd3, 0x146a, 0, "Red Moon /T&M (C-64 Gfx *USA*)" },
	{ 4, 0x8c46, 0xf0, 0xcaf6, 0, "Price of Magik /T&M (C-64 Gfx *USA*)" },

	{ 4, 0xc0cf, 0x4e, 0xb7fa, 3, "Lancelot, pt. 1 (Amiga/PC/ST)" },
	{ 4, 0xd5e9, 0x6a, 0x4192, 3, "Lancelot, pt. 2 (Amiga/PC/ST)" },
	{ 4, 0xbb8f, 0x1a, 0x7487, 3, "Lancelot, pt. 3 (Amiga/PC/ST)" },

	{ 4, 0xc0bd, 0x57, 0x6ef1, 3, "Lancelot, pt. 1 (Mac)" },
	{ 4, 0xd5d7, 0x99, 0x770b, 3, "Lancelot, pt. 2 (Mac)" },
	{ 4, 0xbb7d, 0x17, 0xbc42, 3, "Lancelot, pt. 3 (Mac)" },

	{ 4, 0xb4c9, 0x94, 0xd784, 3, "Lancelot, pt. 1 (C-64 TO)" },
	{ 4, 0xb729, 0x51, 0x8ee5, 3, "Lancelot, pt. 2 (C-64 TO)" },
	{ 4, 0xb702, 0xe4, 0x1809, 3, "Lancelot, pt. 3 (C-64 TO)" },

	{ 4, 0x8feb, 0xba, 0xa800, 3, "Lancelot, pt. 1 (C-64 Gfx)" },
	{ 4, 0x8f6b, 0xfa, 0x0f7e, 3, "Lancelot, pt. 2 (C-64 Gfx)" },
	{ 4, 0x8f71, 0x2f, 0x0ddc, 3, "Lancelot, pt. 3 (C-64 Gfx)" },

	{ 4, 0x8ade, 0xf2, 0xfffb, 0, "Lancelot, pt. 1 (Spectrum48)" },
	{ 4, 0x8b0e, 0xfb, 0x0bab, 0, "Lancelot, pt. 2 (Spectrum48)" },
	{ 4, 0x8ab3, 0xc1, 0xcb62, 0, "Lancelot, pt. 3 (Spectrum48)" },

	{ 4, 0xbba4, 0x94, 0x0871, 0, "Lancelot, pt. 1 (Amiga/PC *USA*)" },
	{ 4, 0xd0c0, 0x56, 0x8c48, 0, "Lancelot, pt. 2 (Amiga/PC *USA*)" },
	{ 4, 0xb6ac, 0xc6, 0xaea0, 0, "Lancelot, pt. 3 (Amiga/PC *USA*)" },

	{ 4, 0x8afc, 0x07, 0x8321, 0, "Lancelot, pt. 1 (C-64 Gfx *USA*)" },
	{ 4, 0x8aec, 0x13, 0x6791, 0, "Lancelot, pt. 2 (C-64 Gfx *USA*)" },
	{ 4, 0x8aba, 0x0d, 0x5602, 0, "Lancelot, pt. 3 (C-64 Gfx *USA*)" },

	{ 4, 0xd19b, 0xad, 0x306d, 3, "Ingrid's Back, pt. 1 (PC)" },
	{ 4, 0xc5a5, 0xfe, 0x3c98, 3, "Ingrid's Back, pt. 2 (PC)" },
	{ 4, 0xd7ae, 0x9e, 0x1878, 3, "Ingrid's Back, pt. 3 (PC)" },

	{ 4, 0xd188, 0x13, 0xdc60, 3, "Ingrid's Back, pt. 1 (Amiga)" },
	{ 4, 0xc594, 0x03, 0xea95, 3, "Ingrid's Back, pt. 2 (Amiga)" },
	{ 4, 0xd79f, 0xb5, 0x1661, 3, "Ingrid's Back, pt. 3 (Amiga)" },

	{ 4, 0xd183, 0x83, 0xef72, 3, "Ingrid's Back, pt. 1 (ST)" },
	{ 4, 0xc58f, 0x65, 0xf337, 3, "Ingrid's Back, pt. 2 (ST)" },
	{ 4, 0xd79a, 0x57, 0x49c5, 3, "Ingrid's Back, pt. 3 (ST)" },

	{ 4, 0xb770, 0x03, 0x9a03, 3, "Ingrid's Back, pt. 1 (C-64 TO)" },
	{ 4, 0xb741, 0xb6, 0x2aa5, 3, "Ingrid's Back, pt. 2 (C-64 TO)" },
	{ 4, 0xb791, 0xa1, 0xd065, 3, "Ingrid's Back, pt. 3 (C-64 TO)" },

	{ 4, 0x9089, 0xce, 0xc5e2, 3, "Ingrid's Back, pt. 1 (C-64 Gfx)" },
	{ 4, 0x908d, 0x80, 0x30c7, 3, "Ingrid's Back, pt. 2 (C-64 Gfx)" },
	{ 4, 0x909e, 0x9f, 0xdecc, 3, "Ingrid's Back, pt. 3 (C-64 Gfx)" },

	{ 4, 0x8ab7, 0x68, 0xee57, 0, "Ingrid's Back, pt. 1 (Spectrum48)" },
	{ 4, 0x8b1e, 0x84, 0x2538, 0, "Ingrid's Back, pt. 2 (Spectrum48)" },
	{ 4, 0x8b1c, 0xa8, 0x9262, 0, "Ingrid's Back, pt. 3 (Spectrum48)" },

	{ 4, 0xbeab, 0x2d, 0x94d9, 0, "Scapeghost, pt. 1 (Amiga)" },
	{ 4, 0xc132, 0x14, 0x7adc, 0, "Scapeghost, pt. 1 (Amiga *bak*)" },
	{ 4, 0xbe94, 0xcc, 0x04b8, 0, "Scapeghost, pt. 1 (PC/ST)" },
	{ 4, 0x99bd, 0x65, 0x032e, 0, "Scapeghost, pt. 2 (Amiga/PC/ST)" },
	{ 4, 0xbcb6, 0x7a, 0x7d4f, 0, "Scapeghost, pt. 3 (Amiga/PC/ST)" },

	{ 4, 0x9058, 0xcf, 0x9748, 0, "Scapeghost, pt. 1 (C-64 Gfx)" },
	{ 4, 0x8f43, 0xc9, 0xeefd, 0, "Scapeghost, pt. 2 (C-64 Gfx)" },
	{ 4, 0x90ac, 0x68, 0xb4a8, 0, "Scapeghost, pt. 3 (C-64 Gfx)" },

	{ 4, 0x8a21, 0xf4, 0xd9e4, 0, "Scapeghost, pt. 1 (Spectrum48)" },
	{ 4, 0x8a12, 0xe3, 0xc2ff, 0, "Scapeghost, pt. 2 (Spectrum48)" },
	{ 4, 0x8a16, 0xcc, 0x4f3b, 0, "Scapeghost, pt. 3 (Spectrum48)" },

	{ 5, 0x3ebb, 0x00, 0xf6dc, 0,
		"Champion of the Raj (English) 1/2 GD (Amiga)" },
	{ 5, 0x0fd8, 0x00, 0xf250, 0,
		"Champion of the Raj (English) 2/2 GD (Amiga)" },

	{ 5, 0x3e8f, 0x00, 0x5599, 0,
		"Champion of the Raj (English) 1/2 GD (ST)" },

	{ 5, 0x3e4f, 0x00, 0xb202, 0,
		"Champion of the Raj (English) 1/2 GD (PC)" },
	{ 5, 0x14a3, 0x00, 0xa288, 0,
		"Champion of the Raj (English) 2/2 GD (PC)" },

	{ 5, 0x1929, 0x00, 0xd4b2, 0,
		"Champion of the Raj (demo), 1/2 GD (ST)" },
	{ 5, 0x40e0, 0x02, 0x080d, 0,
		"Champion of the Raj (demo), 2/2 GD (ST)" },

	{ 5, 0x4872, 0x00, 0x9515, 0,
		"Champion of the Raj (German) 1/2 GD (Amiga)" },
	{ 5, 0x11f5, 0x00, 0xbf39, 0,
		"Champion of the Raj (German) 2/2 GD (Amiga)" },

	{ 5, 0x4846, 0x00, 0xd9c1, 0,
		"Champion of the Raj (German) 1/2 GD (ST)" },
	{ 5, 0x11f5, 0x00, 0x7aa4, 0,
		"Champion of the Raj (German) 2/2 GD (ST)" },

	{ 5, 0x110f, 0x00, 0x4b57, 0,
		"Champion of the Raj (French) 2/2 GD (ST)" },

	{ 0, 0x0000, 0x00, 0x0000, 0, "*" }
};


/*
 * The following patch database is obtained from L9cut's l9data_p.h, and
 * allows CRCs from patched games to be translated into original CRCs for
 * lookup in the game database above.  Apart from minor reformatting, some
 * file commentary has been removed for brevity.
 *
 * The version of l9data_p.h used is 012 (22 May 2001).
 */
typedef struct {
	const L9UINT16	length;		/* datafile length, bytes */
	const L9BYTE	origchk;	/* 8-bit checksum, last datafile byte */
	const L9UINT16	origcrc;	/* 16-bit CRC, l9cut-internal */
	const L9BYTE	patchchk;	/* 8-bit checksum, last datafile byte */
	const L9UINT16	patchcrc;	/* 16-bit CRC, l9cut-internal */
	const L9UINT16	patchloc1;	/* patch locations and values */
	const L9BYTE	patch1;
	const L9UINT16	patchloc2;
	const L9BYTE	patch2;
	const L9UINT16	patchloc3;
	const L9BYTE	patch3;
	const L9UINT16	patchloc4;
	const L9BYTE	patch4;
} gln_patch_table_t;
typedef const gln_patch_table_t		*gln_patch_tableref_t;

static const gln_patch_table_t		GLN_PATCH_TABLE[] = {
	/* Price of Magik (Spectrum128) */
	{ 0x7410, 0x5e, 0x60be, 0x70, 0x6cef,
		0x732c, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Price of Magik (C-64) */
	{ 0x6fc6, 0x14, 0xf9b6, 0x26, 0x3326,
		0x6ee2, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Price of Magik (Spectrum48) */
	{ 0x5aa4, 0xc1, 0xeda4, 0xd3, 0xed35,
		0x59c0, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },
	{ 0x5aa4, 0xc1, 0xeda4, 0xc1, 0x8a65,
		0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Colossal Adventure /JoD (Amiga/PC) */
	{ 0x76f4, 0x5e, 0x1fe5, 0xea, 0x1305,
		0x7567, 0x20, 0x7568, 0x35, 0xffff, 0x00, 0xffff, 0x00 },
	{ 0x76f4, 0x5e, 0x1fe5, 0xb5, 0x901f,
		0x7567, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },
	{ 0x76f4, 0x5e, 0x1fe5, 0x5e, 0x6ea1,
		0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Colossal Adventure /JoD (ST) */
	{ 0x76f4, 0x5a, 0xcf4b, 0xe6, 0x012a,
		0x7567, 0x20, 0x7568, 0x35, 0xffff, 0x00, 0xffff, 0x00 },
	{ 0x76f4, 0x5a, 0xcf4b, 0xb1, 0x40b1,
		0x7567, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Adventure Quest /JoD (Amiga/PC) */
	{ 0x6e60, 0x83, 0x18e0, 0x4c, 0xcfb0,
		0x6d15, 0x20, 0x6d16, 0x33, 0xffff, 0x00, 0xffff, 0x00 },
	{ 0x6e60, 0x83, 0x18e0, 0xfa, 0x9b3b,
		0x6d15, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },
	{ 0x6e60, 0x83, 0x18e0, 0x83, 0x303d,
		0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Adventure Quest /JoD (ST) */
	{ 0x6e5c, 0xf6, 0xd356, 0xbf, 0xede7,
		0x6d11, 0x20, 0x6d12, 0x33, 0xffff, 0x00, 0xffff, 0x00 },
	{ 0x6e5c, 0xf6, 0xd356, 0x6d, 0x662d,
		0x6d11, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Dungeon Adventure /JoD (Amiga/PC/ST) */
	{ 0x6f0c, 0x95, 0x1f64, 0x6d, 0x2443,
		0x6db3, 0x20, 0x6db4, 0x33, 0xffff, 0x00, 0xffff, 0x00 },
	{ 0x6f0c, 0x95, 0x1f64, 0x0c, 0x6066,
		0x6db3, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },
	{ 0x6f0c, 0x95, 0x1f64, 0x96, 0xdaca,
		0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },
	{ 0x6f0c, 0x95, 0x1f64, 0x95, 0x848d,
		0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Colossal Adventure /JoD (Spectrum128) */
	{ 0x6f6e, 0x78, 0x28cd, 0xf8, 0xda5f,
		0x5d71, 0x48, 0x5d72, 0x01, 0x5d73, 0x5e, 0x6d49, 0x02 },
	{ 0x6f6e, 0x78, 0x28cd, 0x77, 0x5b4e,
		0x6d49, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Adventure Quest /JoD (Spectrum128) */
	{ 0x6970, 0xd6, 0xa820, 0x3b, 0x1870,
		0x5d9d, 0x48, 0x5d9e, 0x01, 0x5d9f, 0x79, 0x6763, 0x02 },
	{ 0x6970, 0xd6, 0xa820, 0xd5, 0x13c4,
		0x6763, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Dungeon Adventure /JoD (Spectrum128) */
	{ 0x6de8, 0x4c, 0xd795, 0xa2, 0x3eea,
		0x5d9f, 0x48, 0x5da0, 0x01, 0x5da1, 0x88, 0x6bdb, 0x02 },
	{ 0x6de8, 0x4c, 0xd795, 0x4b, 0xad30,
		0x6bdb, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Colossal Adventure /JoD (CPC) */
	{ 0x6f4d, 0xcb, 0xe8f2, 0x4b, 0xb384,
		0x5d61, 0x48, 0x5d62, 0x01, 0x5d63, 0x5e, 0x6d2a, 0x02 },
	{ 0x6f4d, 0xcb, 0xe8f2, 0xca, 0x96e7,
		0x6d2a, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Adventure Quest /JoD (CPC) */
	{ 0x6968, 0x32, 0x0c01, 0x97, 0xdded,
		0x5d97, 0x48, 0x5d98, 0x01, 0x5d99, 0x79, 0x675d, 0x02 },
	{ 0x6968, 0x32, 0x0c01, 0x31, 0xe8c2,
		0x675d, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Dungeon Adventure /JoD (CPC) */
	{ 0x6dc0, 0x63, 0x5d95, 0xb9, 0xc963,
		0x5d79, 0x48, 0x5d7a, 0x01, 0x5d7b, 0x88, 0x6bb5, 0x02 },
	{ 0x6dc0, 0x63, 0x5d95, 0x62, 0x79f7,
		0x6bb5, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Colossal Adventure /JoD (C-64) */
	{ 0x6c8e, 0xb6, 0x9be3, 0x36, 0x6971,
		0x5a91, 0x48, 0x5a92, 0x01, 0x5a93, 0x5e, 0x6a69, 0x02 },
	{ 0x6c8e, 0xb6, 0x9be3, 0xb5, 0xeba0,
		0x6a69, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Adventure Quest /JoD (C-64) */
	{ 0x63b6, 0x2e, 0xef38, 0x93, 0x4e68,
		0x57e3, 0x48, 0x57e4, 0x01, 0x57e5, 0x79, 0x61a9, 0x02 },
	{ 0x63b6, 0x2e, 0xef38, 0x2d, 0x54dc,
		0x61a9, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Dungeon Adventure /JoD (C-64) */
	{ 0x6bd2, 0x65, 0xa41f, 0xbb, 0x4260,
		0x5b89, 0x48, 0x5b8a, 0x01, 0x5b8b, 0x88, 0x69c5, 0x02 },
	{ 0x6bd2, 0x65, 0xa41f, 0x64, 0xdf5a,
		0x69c5, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Colossal Adventure /JoD (Spectrum48) */
	{ 0x5a8e, 0xf2, 0x7cca, 0x72, 0x8e58,
		0x4891, 0x48, 0x4892, 0x01, 0x4893, 0x5e, 0x5869, 0x02 },
	{ 0x5a8e, 0xf2, 0x7cca, 0xf1, 0x0c89,
		0x5869, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },
	{ 0x5a8e, 0xf2, 0x7cca, 0xf2, 0x2c96,
		0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Adventure Quest /JoD (Spectrum48) */
	{ 0x5ace, 0x11, 0xdc12, 0x76, 0x8663,
		0x4efb, 0x48, 0x4efc, 0x01, 0x4efd, 0x79, 0x58c1, 0x02 },
	{ 0x5ace, 0x11, 0xdc12, 0x10, 0xa757,
		0x58c1, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },
	{ 0x5ace, 0x11, 0xdc12, 0x11, 0xf118,
		0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Dungeon Adventure /JoD (Spectrum48) */
	{ 0x58a3, 0x38, 0x8ce4, 0x8e, 0xb61a,
		0x485a, 0x48, 0x485b, 0x01, 0x485c, 0x88, 0x5696, 0x02 },
	{ 0x58a3, 0x38, 0x8ce4, 0x37, 0x34c0,
		0x5696, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },
	{ 0x58a3, 0x38, 0x8ce4, 0x38, 0xa1ee,
		0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Snowball /SD (Amiga/ST) */
	{ 0x7b31, 0x6e, 0x2e2b, 0xe5, 0x6017,
		0x79b9, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Return to Eden /SD (Amiga/ST) */
	{ 0x7d16, 0xe6, 0x5438, 0x5d, 0xc770,
		0x7b74, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Worm in Paradise /SD (Amiga/ST) */
	{ 0x7cd9, 0x0c, 0x4df1, 0x83, 0xe997,
		0x7b5d, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Snowball /SD (PC/Spectrum128) */
	{ 0x7b2f, 0x70, 0x6955, 0xe7, 0x0af4,
		0x79b9, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },
	{ 0x7b2f, 0x70, 0x6955, 0x70, 0x1179,
		0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Return to Eden /SD (PC) */
	{ 0x7d14, 0xe8, 0xfbab, 0x5f, 0xeab9,
		0x7b74, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },
	{ 0x7d14, 0xe8, 0xfbab, 0xe8, 0xe216,
		0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Return to Eden /SD (CPC) */
	{ 0x7cff, 0xf8, 0x6044, 0x6f, 0xbb57,
		0x7b5f, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Return to Eden /SD (Spectrum128) */
	{ 0x7c55, 0x18, 0xdaee, 0x8f, 0x01fd,
		0x7ab5, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Worm in Paradise /SD (CPC/PC/Spectrum128) */
	{ 0x7cd7, 0x0e, 0x4feb, 0x85, 0x4eae,
		0x7b5d, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },
	{ 0x7cd7, 0x0e, 0x4feb, 0x0e, 0xb02c,
		0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Snowball /SD (C-64) */
	{ 0x7363, 0x65, 0xa0ab, 0xdc, 0xca6a,
		0x71ed, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Return to Eden /SD (C-64) */
	{ 0x772f, 0xca, 0x8602, 0x41, 0x9bd0,
		0x758f, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Worm in Paradise /SD (C-64) */
	{ 0x788d, 0x72, 0x888a, 0xe9, 0x4cce,
		0x7713, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Snowball /SD (Atari) */
	{ 0x6bf8, 0x3f, 0xc9f7, 0x96, 0x1908,
		0x6a5d, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Return to Eden /SD (Atari) */
	{ 0x60f7, 0x68, 0xc2bc, 0xdf, 0xd3ae,
		0x5f57, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Worm in Paradise /SD (Atari) */
	{ 0x6161, 0xf3, 0xe6d7, 0x6a, 0xe232,
		0x5fe7, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Snowball /SD (Spectrum48) */
	{ 0x6541, 0x02, 0x2e6c, 0x79, 0xb80c,
		0x63cb, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },
	{ 0x6541, 0x02, 0x2e6c, 0x02, 0x028a,
		0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Return to Eden /SD (Spectrum48) */
	{ 0x5f43, 0xca, 0x828c, 0x41, 0x9f5e,
		0x5da3, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },
	{ 0x5f43, 0xca, 0x828c, 0xca, 0x6e1b,
		0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Worm in Paradise /SD (Spectrum48) */
	{ 0x5ebb, 0xf1, 0x4dec, 0x68, 0x4909,
		0x5d41, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },
	{ 0x5ebb, 0xf1, 0x4dec, 0xf1, 0xcc1a,
		0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Knight Orc, pt. 1 (Amiga) */
	{ 0xbb93, 0x36, 0x6a05, 0xad, 0xe52d,
		0xba30, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Knight Orc, pt. 1 (ST) */
	{ 0xbb6e, 0xad, 0x4d40, 0x24, 0x3bcd,
		0xba13, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Knight Orc, pt. 2 (Amiga/ST) */
	{ 0xc58e, 0x4a, 0x4e9d, 0xc1, 0xe2bf,
		0x93e2, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Knight Orc, pt. 3 (Amiga/ST) */
	{ 0xcb9a, 0x0f, 0x0804, 0x86, 0x6487,
		0x99ee, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Knight Orc, pt. 1 (PC) */
	{ 0xbb6e, 0xa6, 0x9753, 0x1d, 0x2e7f,
		0xba13, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },
	{ 0xbb6e, 0xa6, 0x9753, 0xa6, 0x001d,
		0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Knight Orc, pt. 2 (PC) */
	{ 0xc58e, 0x43, 0xe9ce, 0xba, 0x5e4c,
		0x93e2, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },
	{ 0xc58e, 0x43, 0xe9ce, 0x43, 0xa8f0,
		0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Knight Orc, pt. 3 (PC) */
	{ 0xcb9a, 0x08, 0x6c36, 0x7f, 0xf0d4,
		0x99ee, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },
	{ 0xcb9a, 0x08, 0x6c36, 0x08, 0x2d08,
		0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Knight Orc, pt. 1 (C-64 Gfx) */
	{ 0x8970, 0x6b, 0x3c7b, 0xe2, 0xb6f3,
		0x880d, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Knight Orc, pt. 1 (Spectrum48) */
	{ 0x86d0, 0xb7, 0xadbd, 0x2e, 0x43e1,
		0x8567, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Gnome Ranger, pt. 1 (Amiga/ST) */
	{ 0xb1a9, 0x80, 0x5fb7, 0xf7, 0x5c6c,
		0xb093, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Gnome Ranger, pt. 2 (Amiga/ST) */
	{ 0xab9d, 0x31, 0xbe6d, 0xa8, 0xcb96,
		0x95bf, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Gnome Ranger, pt. 3 (Amiga/ST) */
	{ 0xae28, 0x87, 0xb6b6, 0xfe, 0x760c,
		0xad12, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Gnome Ranger, pt. 1 (PC) */
	{ 0xb1aa, 0xad, 0xaf47, 0x24, 0x5cfd,
		0xb094, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },
	{ 0xb1aa, 0xad, 0xaf47, 0xad, 0xe0ed,
		0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Gnome Ranger, pt. 1 (ST-var) */
	{ 0xb19e, 0x92, 0x8f96, 0x09, 0x798c,
		0xb088, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Gnome Ranger, pt. 2 (PC/ST-var) */
	{ 0xab8b, 0xbf, 0x31e6, 0x36, 0x811c,
		0x95ad, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },
	{ 0xab8b, 0xbf, 0x31e6, 0xbf, 0x8ff3,
		0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Gnome Ranger, pt. 3 (PC/ST-var) */
	{ 0xae16, 0x81, 0x8741, 0xf8, 0x47fb,
		0xad00, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },
	{ 0xae16, 0x81, 0x8741, 0x81, 0xc8eb,
		0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Gnome Ranger, pt. 1 (C-64 TO) */
	{ 0xad41, 0xa8, 0x42c5, 0x1f, 0x7d1e,
		0xac2b, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Gnome Ranger, pt. 2 (C-64 TO) */
	{ 0xa735, 0xf7, 0x2e08, 0x6e, 0x780e,
		0x919b, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Gnome Ranger, pt. 3 (C-64 TO) */
	{ 0xa9c0, 0x9e, 0x0d70, 0x15, 0x3e6b,
		0xa8aa, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Gnome Ranger, pt. 1 (C-64 Gfx) */
	{ 0x908e, 0x0d, 0x58a7, 0x84, 0xab1d,
		0x8f78, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Gnome Ranger, pt. 2 (C-64 Gfx) */
	{ 0x8f6f, 0x0a, 0x411a, 0x81, 0x12bc,
		0x79d5, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Gnome Ranger, pt. 3 (C-64 Gfx) */
	{ 0x9060, 0xbb, 0xe75d, 0x32, 0x14e7,
		0x8f4a, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Lords of Time /T&M (PC) */
	{ 0xb57c, 0x44, 0x7779, 0xbb, 0x31a6,
		0xb453, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },
	{ 0xb57c, 0x44, 0x7779, 0x44, 0xea72,
		0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Red Moon /T&M (PC) */
	{ 0xa69e, 0x6c, 0xb268, 0xe3, 0x4cef,
		0xa578, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },
	{ 0xa69e, 0x6c, 0xb268, 0x6c, 0x3799,
		0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Price of Magik /T&M (PC) */
	{ 0xbac7, 0x7f, 0xddb2, 0xf6, 0x6ab3,
		0xb96d, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },
	{ 0xbac7, 0x7f, 0xddb2, 0x7f, 0x905c,
		0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Lords of Time /T&M (ST) */
	{ 0xb579, 0x89, 0x3e89, 0x00, 0xa2b7,
		0xb450, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Red Moon /T&M (ST) */
	{ 0xa698, 0x41, 0xcaca, 0xb8, 0xeeac,
		0xa572, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Price of Magik /T&M (ST) */
	{ 0xbac4, 0x80, 0xa750, 0xf7, 0xe030,
		0xb96a, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Lords of Time /T&M (Amiga) */
	{ 0xb576, 0x2a, 0x7239, 0xa1, 0x2ea6,
		0xb44d, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Red Moon /T&M (Amiga) */
	{ 0xa692, 0xd1, 0x6a99, 0x48, 0x50ff,
		0xa56c, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Price of Magik /T&M (Amiga) */
	{ 0xbaca, 0x3a, 0x221b, 0xb1, 0x55bb,
		0xb970, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Lords of Time /T&M (C-64 TO) */
	{ 0xb38c, 0x37, 0x9f8e, 0xae, 0xc6b1,
		0xb263, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Red Moon /T&M (C-64 TO) */
	{ 0xa4e2, 0xa6, 0x016d, 0x1d, 0x31ab,
		0xa3bc, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Price of Magik /T&M (C-64 TO) */
	{ 0xb451, 0xa8, 0x2682, 0x1f, 0x5de2,
		0xb2f7, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Lords of Time /T&M (C-64 Gfx) */
	{ 0x9070, 0x43, 0x45d4, 0xba, 0x02eb,
		0x8f47, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Red Moon /T&M (C-64 Gfx) */
	{ 0x903f, 0x6b, 0x603e, 0xe2, 0x9f59,
		0x8f19, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Price of Magik /T&M (C-64 Gfx) */
	{ 0x8f51, 0xb2, 0x6c9a, 0x29, 0xde3b,
		0x8df7, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Lords of Time /T&M (Spectrum48) */
	{ 0x8950, 0xa1, 0xbb16, 0x18, 0x2828,
		0x8827, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },
	{ 0x8950, 0xa1, 0xbb16, 0xa1, 0x1ea2,
		0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Red Moon /T&M (Spectrum48) */
	{ 0x8813, 0x11, 0x22de, 0x88, 0x18b8,
		0x86ed, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },
	{ 0x8813, 0x11, 0x22de, 0x11, 0xd0cd,
		0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Price of Magik /T&M (Spectrum48) */
	{ 0x8a60, 0x2a, 0x29ed, 0xa1, 0x5e4d,
		0x8906, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Lancelot, pt. 1 (Amiga/PC/ST) */
	{ 0xc0cf, 0x4e, 0xb7fa, 0xc5, 0x4400,
		0xbf7f, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Lancelot, pt. 2 (Amiga/PC/ST) */
	{ 0xd5e9, 0x6a, 0x4192, 0xe1, 0x3b1e,
		0xd411, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Lancelot, pt. 3 (Amiga/PC/ST) */
	{ 0xbb8f, 0x1a, 0x7487, 0x91, 0x877d,
		0xba3f, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Lancelot, pt. 1 (C-64 TO) */
	{ 0xb4c9, 0x94, 0xd784, 0x0b, 0x203e,
		0xb379, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Lancelot, pt. 2 (C-64 TO) */
	{ 0xb729, 0x51, 0x8ee5, 0xc8, 0xf1c9,
		0xb551, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Lancelot, pt. 3 (C-64 TO) */
	{ 0xb702, 0xe4, 0x1809, 0x5b, 0x25b2,
		0xb5b2, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Lancelot, pt. 1 (C-64 Gfx) */
	{ 0x8feb, 0xba, 0xa800, 0x31, 0x5bfa,
		0x8e9b, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Lancelot, pt. 2 (C-64 Gfx) */
	{ 0x8f6b, 0xfa, 0x0f7e, 0x71, 0x75f2,
		0x8d93, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Lancelot, pt. 3 (C-64 Gfx) */
	{ 0x8f71, 0x2f, 0x0ddc, 0xa6, 0x3e87,
		0x8e21, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Ingrid's Back, pt. 1 (PC) */
	{ 0xd19b, 0xad, 0x306d, 0x24, 0x4504,
		0xcff0, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },
	{ 0xd19b, 0xad, 0x306d, 0xad, 0x878e,
		0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Ingrid's Back, pt. 2 (PC) */
	{ 0xc5a5, 0xfe, 0x3c98, 0x75, 0x8950,
		0xc3fa, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },
	{ 0xc5a5, 0xfe, 0x3c98, 0xfe, 0x8b7b,
		0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Ingrid's Back, pt. 3 (PC) */
	{ 0xd7ae, 0x9e, 0x1878, 0x15, 0xadb0,
		0xd603, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },
	{ 0xd7ae, 0x9e, 0x1878, 0x9e, 0xaf9b,
		0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Ingrid's Back, pt. 1 (Amiga) */
	{ 0xd188, 0x13, 0xdc60, 0x8a, 0x755c,
		0xcfe2, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Ingrid's Back, pt. 2 (Amiga) */
	{ 0xc594, 0x03, 0xea95, 0x7a, 0xb5a8,
		0xc3ee, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Ingrid's Back, pt. 3 (Amiga) */
	{ 0xd79f, 0xb5, 0x1661, 0x2c, 0xbf5d,
		0xd5f9, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Ingrid's Back, pt. 1 (ST) */
	{ 0xd183, 0x83, 0xef72, 0xfa, 0xb04f,
		0xcfdd, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Ingrid's Back, pt. 2 (ST) */
	{ 0xc58f, 0x65, 0xf337, 0xdc, 0x900a,
		0xc3e9, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Ingrid's Back, pt. 3 (ST) */
	{ 0xd79a, 0x57, 0x49c5, 0xce, 0xe0f9,
		0xd5f4, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Ingrid's Back, pt. 1 (C-64 TO) */
	{ 0xb770, 0x03, 0x9a03, 0x7a, 0xdc6a,
		0xb5c5, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Ingrid's Back, pt. 2 (C-64 TO) */
	{ 0xb741, 0xb6, 0x2aa5, 0x2d, 0x5a6c,
		0xb596, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Ingrid's Back, pt. 3 (C-64 TO) */
	{ 0xb791, 0xa1, 0xd065, 0x18, 0xaa0c,
		0xb5e6, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Ingrid's Back, pt. 1 (C-64 Gfx) */
	{ 0x9089, 0xce, 0xc5e2, 0x44, 0xeff4,
		0x8ed9, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Ingrid's Back, pt. 2 (C-64 Gfx) */
	{ 0x908d, 0x80, 0x30c7, 0xf6, 0x2a11,
		0x8edd, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	/* Ingrid's Back, pt. 3 (C-64 Gfx) */
	{ 0x909e, 0x9f, 0xdecc, 0x15, 0xf4da,
		0x8eee, 0x02, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 },

	{ 0x0000, 0x00, 0x0000, 0x00, 0x0000,
		0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00, 0xffff, 0x00 }
};


/*---------------------------------------------------------------------*/
/*  Glk port game identification functions                             */
/*---------------------------------------------------------------------*/

/*
 * The game's name, suitable for printing out on a status line, or other
 * location where game information is relevant.  It's allocated and
 * generated on demand, and may be re-requested when, say, the game
 * changes, perhaps by moving to the next of a multipart game.
 */
static int		gln_gameid_identification_done	= FALSE;
static char		*gln_gameid_game_name		= NULL;


/*
 * gln_gameid_lookup_version()
 * gln_gameid_lookup_patch()
 *
 * Look up and return game version table and patch table entries given a
 * game's length, checksum, and CRC.  Returns the entry, or NULL if not
 * found.
 */
static gln_version_tableref_t
gln_gameid_lookup_version (L9UINT16 length, L9BYTE checksum,
				L9UINT16 crcvalue, int ignore_crc)
{
	gln_version_tableref_t	entry;		/* Version table iterator */

	/* Scan the table for a match, breaking the loop if found. */
	for (entry = GLN_VERSION_TABLE; entry->version != 0; entry++)
	    {
	    	if (ignore_crc)
			crcvalue = entry->crcvalue;

		if (entry->length == length
			&& entry->checksum == checksum
			&& entry->crcvalue == crcvalue)
		    {
			break;
		    }
	    }

	/* Return the index, or NULL if not found. */
	return entry->version == 0 ? NULL : entry;
}
static gln_patch_tableref_t
gln_gameid_lookup_patch (L9UINT16 length, L9BYTE checksum, L9UINT16 crcvalue)
{
	gln_patch_tableref_t	entry;		/* Patch table iterator */

	/* Scan the table for a match, breaking the loop if found. */
	for (entry = GLN_PATCH_TABLE; entry->length != 0; entry++)
	    {
		if (entry->length == length
			&& entry->patchchk == checksum
			&& entry->patchcrc == crcvalue)
		    {
			break;
		    }
	    }

	/* Return the index, or NULL if not found. */
	return entry->length == 0 ? NULL : entry;
}


/*
 * gln_gameid_identify_game()
 *
 * Identify a game from its data length, checksum, and CRC.  Returns the
 * entry of the game in the version table, or NULL if not found.
 *
 * This function uses startdata and FileSize from the core interpreter.
 * These aren't advertised symbols, so be warned.
 */
static gln_version_tableref_t
gln_gameid_identify_game (void)
{
	L9UINT16		length;		/* Length of data for CRC */
	L9BYTE			check;		/* Game check byte */
	L9UINT16		crc;		/* Game data CRC */
	gln_version_tableref_t	game;		/* Version table entry */
	gln_patch_tableref_t	patch;		/* Patch table entry */

	/*
	 * Find the length of game data; this logic is taken from L9cut,
	 * with calcword() replaced by simple byte comparisons.  If the
	 * length exceeds the available data, fail.  This code also sets
	 * up the game check byte.
	 */
	assert (startdata != NULL);
	if (startdata[4] == 0x20 && startdata[5] == 0x00
			&& startdata[10] == 0x00 && startdata[11] == 0x80
			&& startdata[20] == startdata[22]
			&& startdata[21] == startdata[23])
	    {
		int		index;		/* Check byte loop */

	    	/* V2 length. */
		length = startdata[28] | startdata[29] << CHAR_BIT;
		if (length >= FileSize)
			return NULL;

		/* V2 check. */
		check = 0;
		for (index = 0; index <= length; index++)
			check += startdata[ index ];
	    }
	else
	    {
	    	/* V3 (and later) length. */
		length = startdata[0] | startdata[1] << CHAR_BIT;
		if (length >= FileSize)
			return NULL;

		/* V3 (and later) check. */
		check = startdata[ length ];
	    }

	/*
	 * Generate a CRC for this data.  When L9cut calculates a CRC, it's
	 * using a copy taken up to length + 1 and then padded with two
	 * zero bytes, so we mimic that here.
	 */
	crc = gln_buffer_crc (startdata, length + 1);
	crc = gln_update_crc (crc, "\0\0", 2);

	/*
	 * See if this is a patched file.  If it is, look up the game based
	 * on the original CRC; if not, use the current CRC.
	 */
	patch = gln_gameid_lookup_patch (length, check, crc);
	if (patch != NULL)
	    {
		game = gln_gameid_lookup_version (length, patch->origchk,
							patch->origcrc, FALSE);
	    }
	else
		game = gln_gameid_lookup_version (length, check, crc, FALSE);

	/*
	 * If no game identified, retry without the CRC.  This is a bit
	 * of guesswork.
	 */
	if (game == NULL)
		game = gln_gameid_lookup_version (length, check, crc, TRUE);

	/* Return the game's version table entry, or NULL if not found. */
	return game;
}


/*
 * gln_gameid_get_game_name()
 *
 * Return the name of the game, or NULL if not identifiable.
 *
 * This function uses startdata from the core interpreter.  This isn't an
 * advertised symbol, so be warned.
 */
static const char *
gln_gameid_get_game_name (void)
{
	gln_version_tableref_t	game;		/* Version table entry */

	/*
	 * If the interpreter hasn't yet loaded a game, startdata is NULL
	 * (uninitialized, global).  In this case, we return NULL, but
	 * don't set the flag that notes identification as done.  This way,
	 * we'll retry until a game is loaded.
	 */
	if (startdata == NULL)
		return NULL;

	/* Return any previous result of game lookup. */
	if (gln_gameid_identification_done)
		return gln_gameid_game_name;

	/*
	 * Look up the game's details, and note as having been done, no
	 * matter if it succeeded or not.
	 */
	game = gln_gameid_identify_game ();
	gln_gameid_identification_done = TRUE;

	/* If the lookup succeeded, generate a string describing the game. */
	if (game != NULL)
	    {
		/* Allocate enough space for the string, and build it. */
		assert (gln_gameid_game_name == NULL);
		gln_gameid_game_name = gln_malloc (strlen (game->game) + 7);
		sprintf (gln_gameid_game_name,
				"%s  (v%1d)", game->game, game->version);
	    }

	/* Return the newly constructed game name string. */
	return gln_gameid_game_name;
}


/*
 * gln_gameid_game_name_reset()
 *
 * Clear the saved game name, forcing a new lookup when next queried.  This
 * function should be called by actions that may cause the interpreter to
 * change game file, for example os_set_filenumber().
 */
static void
gln_gameid_game_name_reset (void)
{
	/* Free any existing game name string. */
	if (gln_gameid_game_name != NULL)
	    {
		free (gln_gameid_game_name);
		gln_gameid_game_name = NULL;
	    }

	/* Reset the identification flag to force re-lookup. */
	gln_gameid_identification_done = FALSE;
}


/*---------------------------------------------------------------------*/
/*  Glk port bitmap picture functions                                  */
/*---------------------------------------------------------------------*/

/* R,G,B color triple definition. */
typedef struct {
	int	red, green, blue;		/* Color attributes. */
} gln_rgb_t;
typedef	gln_rgb_t*		gln_rgbref_t;

/*
 * Maximum number of regions to consider in a single repaint pass.  A
 * couple of hundred seems to strike the right balance between not too
 * sluggardly picture updates, and responsiveness to input during graphics
 * rendering, when combined with short timeouts.
 */
static const int	GLN_REPAINT_LIMIT		= 256;

/*
 * Graphics timeout; we like an update call after this period (ms).  In
 * practice, this timeout may actually be shorter than the time taken
 * to reach the limit on repaint regions, but because Glk guarantees that
 * user interactions (in this case, line events) take precedence over
 * timeouts, this should be okay; we'll still see a game that responds to
 * input each time the background repaint function yields.
 *
 * Setting this value is tricky.  We'd like it to be the shortest possible
 * consistent with getting other stuff done, say 10ms.  However, Xglk has
 * a granularity of 50ms on checking for timeouts, as it uses a 1/20s
 * timeout on X select.  This means that the shortest timeout we'll ever
 * get from Xglk will be 50ms, so there's no point in setting this shorter
 * than that.  With luck, other Glk libraries will be more efficient than
 * this, and can give us higher timer resolution; we'll set 50ms here, and
 * hope that no other Glk library is worse.
 */
static const glui32	GLN_GRAPHICS_TIMEOUT		= 50;

/*
 * Count of timeouts to wait on.  Waiting after a repaint smooths the
 * display where the frame is being resized, by helping to avoid graphics
 * output while more resize events are received; around 1/2 second seems
 * okay.
 */
static const int	GLN_GRAPHICS_ANIMATION_WAIT	= 2;
static const int	GLN_GRAPHICS_REPAINT_WAIT	= 10;

/* Pixel size multiplier for image size scaling. */
static const int	GLN_GRAPHICS_PIXEL		= 2;

/* Proportion of the display to use for graphics. */
static const glui32	GLN_GRAPHICS_PROPORTION		= 60;

/*
 * Special title picture number, requiring its own handling, and count of
 * timeouts to wait on after fully rendering the title picture (~2 seconds).
 */
static const int	GLN_GRAPHICS_TITLE_PICTURE	= 0;
static const int	GLN_GRAPHICS_TITLE_WAIT		= 40;

/*
 * Border and shading control.  For cases where we can't detect the back-
 * ground color of the main window, there's a default, white, background.
 * Bordering is black, with a 1 pixel border, 2 pixel shading, and 8 steps
 * of shading fade.
 */
static const glui32	GLN_GRAPHICS_DEFAULT_BACKGROUND	= 0x00FFFFFF;
static const glui32	GLN_GRAPHICS_BORDER_COLOR	= 0x00000000;
static const int	GLN_GRAPHICS_BORDER		= 1;
static const int	GLN_GRAPHICS_SHADING		= 2;
static const int	GLN_GRAPHICS_SHADE_STEPS	= 8;

/*
 * Guaranteed unused pixel value.  This value is used to fill the on-screen
 * buffer on new pictures or repaints, resulting in a full paint of all
 * pixels since no off-screen, real picture, pixel will match it.
 */
static const int	GLN_GRAPHICS_UNUSED_PIXEL	= 0xFF;

/* Graphics file directory, and type of graphics found in it. */
static char		*gln_graphics_bitmap_directory	= NULL;
static BitmapType	gln_graphics_bitmap_type	= NO_BITMAPS;

/* The current picture id being displayed. */
enum {			GLN_PALETTE_SIZE		= 32 };
static L9BYTE		*gln_graphics_bitmap		= NULL;
static L9UINT16		gln_graphics_width		= 0;
static L9UINT16		gln_graphics_height		= 0;
static Colour		gln_graphics_palette[GLN_PALETTE_SIZE];
						/*	= { 0, ... }; */
static int		gln_graphics_picture		= -1;

/*
 * Flags set on new picture, and on resize or arrange events, and a flag
 * to indicate whether background repaint is stopped or active.
 */
static int		gln_graphics_new_picture	= FALSE;
static int		gln_graphics_repaint		= FALSE;
static int		gln_graphics_active		= FALSE;

/*
 * State to monitor the state of interpreter graphics.  The values of the
 * enumerations match the modes supplied by os_graphics().
 */
static enum {		GLN_GRAPHICS_OFF		= 0,
			GLN_GRAPHICS_LINE_MODE		= 1,
			GLN_GRAPHICS_BITMAP_MODE	= 2 }
			gln_graphics_interpreter_state	= GLN_GRAPHICS_OFF;


/*
 * Pointer to the two graphics buffers, one the off-screen representation
 * of pixels, and the other tracking on-screen data.  These are temporary
 * graphics malloc'ed memory, and should be free'd on exit.
 */
static L9BYTE		*gln_graphics_off_screen	= NULL;
static L9BYTE		*gln_graphics_on_screen		= NULL;

/*
 * The number of colors used in the palette by the current picture.  Because
 * of the way it's queried, we risk a race, with admittedly a very low
 * probability, with the updater.  So, it's initialized instead to the
 * largest possible value.  The real value in use is inserted on the first
 * picture update timeout call for a new picture.
 */
static int		gln_graphics_color_count	= GLN_PALETTE_SIZE;


/*
 * gln_graphics_open()
 *
 * If it's not open, open the graphics window.  Returns TRUE if graphics
 * was successfully started, or already on.
 */
static int
gln_graphics_open (void)
{
	/* If graphics are off, turn them back on now. */
	if (gln_graphics_window == NULL)
	    {
		/* (Re-)open a graphics window. */
		gln_graphics_window = glk_window_open
				(gln_main_window,
				winmethod_Above|winmethod_Proportional,
				GLN_GRAPHICS_PROPORTION,
				wintype_Graphics, 0);

		/* If the graphics fails, return FALSE. */
		if (gln_graphics_window == NULL)
			return FALSE;
	    }

	/* Graphics opened successfully, or already open. */
	return TRUE;
}


/*
 * gln_graphics_close()
 *
 * If it is open, close the graphics window.
 */
static void
gln_graphics_close (void)
{
	/* If the graphics window is open, close it and set NULL. */
	if (gln_graphics_window != NULL)
	    {
		glk_window_close (gln_graphics_window, NULL);
		gln_graphics_window = NULL;
	    }
}


/*
 * gln_graphics_start()
 *
 * Start any background picture update processing.
 */
static void
gln_graphics_start (void)
{
	/* Ignore the call if pictures are disabled. */
	if (!gln_graphics_enabled)
		return;

	/* If not running, start the updating "thread". */
	if (!gln_graphics_active)
	    {
		glk_request_timer_events (GLN_GRAPHICS_TIMEOUT);
		gln_graphics_active = TRUE;
	    }
}


/*
 * gln_graphics_stop()
 *
 * Stop any background picture update processing.
 */
static void
gln_graphics_stop (void)
{
	/* If running, stop the updating "thread". */
	if (gln_graphics_active)
	    {
		glk_request_timer_events (0);
		gln_graphics_active = FALSE;
	    }
}


/*
 * gln_graphics_are_displayed()
 *
 * Return TRUE if graphics are currently being displayed, FALSE otherwise.
 */
static int
gln_graphics_are_displayed (void)
{
	return gln_graphics_window != NULL;
}


/*
 * gln_graphics_paint()
 *
 * Set up a complete repaint of the current picture in the graphics window.
 * This function should be called on the appropriate Glk window resize and
 * arrange events.
 */
static void
gln_graphics_paint (void)
{
	/*
	 * Ignore the call if pictures are disabled, or if there is no
	 * graphics window currently displayed.
	 */
	if (!gln_graphics_enabled
			|| !gln_graphics_are_displayed ())
		return;

	/* Set the repaint flag, and start graphics. */
	gln_graphics_repaint = TRUE;
	gln_graphics_start ();
}


/*
 * gln_graphics_restart()
 *
 * Restart graphics as if the current picture is a new picture.  This
 * function should be called whenever graphics is re-enabled after being
 * disabled.
 */
static void
gln_graphics_restart (void)
{
	/*
	 * Ignore the call if pictures are disabled, or if there is no
	 * graphics window currently displayed.
	 */
	if (!gln_graphics_enabled
			|| !gln_graphics_are_displayed ())
		return;

	/* Set the new picture flag, and start graphics. */
	gln_graphics_new_picture = TRUE;
	gln_graphics_start ();
}


/*
 * gln_graphics_count_colors()
 *
 * Analyze an image, and return an overall count of how many colors out of
 * the palette are used.
 */
static int
gln_graphics_count_colors (L9BYTE bitmap[], L9UINT16 width, L9UINT16 height)
{
	int	index;				/* Palette iterator index */
	int	x, y;				/* Image iterators */
	long	usage[GLN_PALETTE_SIZE];	/* Color usage counts */
	int	count;				/* Colors count */
	long	index_row;			/* Optimization variable */
	assert (bitmap != NULL);

	/* Clear initial usage counts. */
	for (index = 0; index < GLN_PALETTE_SIZE; index++)
		usage[ index ] = 0;

	/*
	 * Traverse the image, counting each pixel usage.  For the y
	 * iterator, maintain an index row as an optimization to avoid
	 * multiplications in the loop.
	 */
	for (y = 0, index_row = 0;
			y < height; y++, index_row += width)
	    {
		for (x = 0; x < width; x++)
		    {
			long	index;		/* x,y pixel index */

			/* Get the index for this pixel. */
			index = index_row + x;

			/* Update the count for this palette color. */
			usage[ bitmap[ index ]]++;
		    }
	    }

	/* Count and return colors used overall in the palette. */
	count = 0;
	for (index = 0; index < GLN_PALETTE_SIZE; index++)
	    {
		if (usage[ index ] > 0)
			count++;
	    }
	return count;
}


/*
 * gln_graphics_split_color()
 * gln_graphics_combine_color()
 *
 * General graphics helper functions, to convert between RGB and Glk
 * glui32 color representations.
 */
static void
gln_graphics_split_color (glui32 color, gln_rgbref_t rgb_color)
{
	assert (rgb_color != NULL);

	/* Split color into its three components. */
	rgb_color->red		= (color >> 16) & 0xFF;
	rgb_color->green	= (color >>  8) & 0xFF;
	rgb_color->blue		= (color      ) & 0xFF;
}

static glui32
gln_graphics_combine_color (gln_rgbref_t rgb_color)
{
	glui32		color;			/* Return color. */
	assert (rgb_color != NULL);

	/* Combine RGB color from its three components. */
	color =		  (rgb_color->red   << 16)
			| (rgb_color->green << 8 )
			| (rgb_color->blue       );
	return color;
}


/*
 * gln_graphics_swap_array()
 *
 * Swap two elements of an array of long integers.
 */
static void
gln_graphics_swap_array (long longint_array[], int index_a, int index_b)
{
	long	temporary;			/* Temporary swap value */

	/* Swap values. */
	temporary			= longint_array[ index_a ];
	longint_array[ index_a ]	= longint_array[ index_b ];
	longint_array[ index_b ]	= temporary;
}


/*
 * gln_graphics_clear_and_border()
 *
 * Clear the graphics window, and border and shade the area where the
 * picture is going to be rendered.  This attempts a small raised effect
 * for the picture, in keeping with modern trends.
 */
static void
gln_graphics_clear_and_border (winid_t glk_window,
			int x_offset, int y_offset,
			int pixel_size, L9UINT16 width, L9UINT16 height)
{
	glui32		background;		/* Window background color */
	gln_rgb_t	rgb_background;		/* RGB background color */
	gln_rgb_t	rgb_border;		/* RGB border color */
	gln_rgb_t	rgb_fade;		/* RGB fade decremental */
	glui32		fade_color;		/* Fade decremental */
	int		index;			/* Shading iterator */
	glui32		shading_color;		/* Shading color */
	assert (glk_window != NULL);

	/*
	 * Try to detect the background color of the main window, by getting
	 * the background for Normal style (Glk offers no way to directly
	 * get a window's background color).  If we can get it, we'll match
	 * the graphics window background to it.  If we can't, we'll default
	 * the color to white.
	 */
	if (!glk_style_measure (gln_main_window,
			style_Normal, stylehint_BackColor, &background))
	    {
		/*
		 * Unable to get the main window background, so assume, and
		 * default graphics to white.
		 */
		background = GLN_GRAPHICS_DEFAULT_BACKGROUND;
	    }

	/*
	 * Set the graphics window background to match the main window
	 * background, as best as we can tell, and clear the window.
	 */
	glk_window_set_background_color (glk_window, background);
	glk_window_clear (glk_window);

	/*
	 * For very small pictures, just border them, but don't try and
	 * do any shading.  Failing this check is probably highly unlikely.
	 */
	if (width < 2 * GLN_GRAPHICS_SHADE_STEPS
			|| height < 2 * GLN_GRAPHICS_SHADE_STEPS)
	    {
		/*
		 * Paint a rectangle bigger than the picture by border
		 * pixels all round.
		 */
		glk_window_fill_rect (glk_window,
				GLN_GRAPHICS_BORDER_COLOR,
				x_offset - GLN_GRAPHICS_BORDER,
				y_offset - GLN_GRAPHICS_BORDER,
				width  * pixel_size
						+ GLN_GRAPHICS_BORDER * 2,
				height * pixel_size
						+ GLN_GRAPHICS_BORDER * 2);
		return;
	    }

	/*
	 * Paint a rectangle bigger than the picture by border pixels all
	 * round, and with additional shading pixels right and below.  Some
	 * of these shading pixels are later overwritten by the fading
	 * loop below.  The picture will sit over this rectangle.
	 */
	glk_window_fill_rect (glk_window,
				GLN_GRAPHICS_BORDER_COLOR,
				x_offset - GLN_GRAPHICS_BORDER,
				y_offset - GLN_GRAPHICS_BORDER,
				width  * pixel_size + GLN_GRAPHICS_BORDER * 2
							+ GLN_GRAPHICS_SHADING,
				height * pixel_size + GLN_GRAPHICS_BORDER * 2
							+ GLN_GRAPHICS_SHADING);

	/*
	 * Split the main window background color and the border color into
	 * components.
	 */
	gln_graphics_split_color (background, &rgb_background);
	gln_graphics_split_color (GLN_GRAPHICS_BORDER_COLOR, &rgb_border);

	/*
	 * Generate the incremental color to use in fade steps.  Here we're
	 * assuming that the border is always darker than the main window
	 * background (currently valid, as we're using black).
	 */
	rgb_fade.red	= (rgb_background.red - rgb_border.red)
						/ GLN_GRAPHICS_SHADE_STEPS;
	rgb_fade.green	= (rgb_background.green - rgb_border.green)
						/ GLN_GRAPHICS_SHADE_STEPS;
	rgb_fade.blue	= (rgb_background.blue - rgb_border.blue)
						/ GLN_GRAPHICS_SHADE_STEPS;

	/* Combine RGB fade into a single incremental Glk color. */
	fade_color = gln_graphics_combine_color (&rgb_fade);

	/* Fade in edge, from background to border, shading in stages. */
	shading_color = background;
	for (index = 0; index < GLN_GRAPHICS_SHADE_STEPS; index++)
	    {
		/* Shade the two border areas with this color. */
		glk_window_fill_rect (glk_window,
					shading_color,
					x_offset + width * pixel_size
							+ GLN_GRAPHICS_BORDER,
					y_offset + index
							- GLN_GRAPHICS_BORDER,
					GLN_GRAPHICS_SHADING, 1);
		glk_window_fill_rect (glk_window,
					shading_color,
					x_offset + index 
							- GLN_GRAPHICS_BORDER,
					y_offset + height * pixel_size
							+ GLN_GRAPHICS_BORDER,
					1, GLN_GRAPHICS_SHADING);

		/* Update the shading color for the fade next iteration. */
		shading_color -= fade_color;
	    }
}


/*
 * gln_graphics_convert_palette()
 *
 * Convert a Level9 bitmap color palette to a Glk one.
 */
static void
gln_graphics_convert_palette (Colour ln_palette[], glui32 glk_palette[])
{
	int		index;			/* Palette color index */
	assert (ln_palette != NULL && glk_palette != NULL);

	/* Convert each color in the palette. */
	for (index = 0; index < GLN_PALETTE_SIZE; index++)
	    {
		Colour		color;		/* Raw picture color. */
		gln_rgb_t	gln_color;	/* Internal picture color. */

		/* Convert color from Level9 to internal RGB. */
		color = ln_palette[ index ];
		gln_color.red	= color.red;
		gln_color.green	= color.green;
		gln_color.blue	= color.blue;

		/* Combine internal RGB into a Glk color. */
		glk_palette[ index ] = gln_graphics_combine_color (&gln_color);
	    }
}


/*
 * gln_graphics_position_picture()
 *
 * Given a picture width and height, return the x and y offsets to center
 * this picture in the current graphics window.
 */
static void
gln_graphics_position_picture (winid_t glk_window,
			int pixel_size, L9UINT16 width, L9UINT16 height,
			int *x_offset, int *y_offset)
{
	glui32		window_width, window_height;
					/* Graphics window dimensions */
	assert (glk_window != NULL);
	assert (x_offset != NULL && y_offset != NULL);

	/* Measure the current graphics window dimensions. */
	glk_window_get_size (glk_window, &window_width, &window_height);

	/*
	 * Calculate and return an x and y offset to use on point plotting,
	 * so that the image centers inside the graphical window.
	 */
	*x_offset = ((int) window_width  - width  * pixel_size) / 2;
	*y_offset = ((int) window_height - height * pixel_size) / 2;
}


/*
 * gln_graphics_is_vertex()
 *
 * Given a point, return TRUE if that point is the vertex of a fillable
 * region.  This is a helper function for layering pictures.  When assign-
 * ing layers, we want to weight the colors that have the most complex
 * shapes, or the largest count of isolated areas, heavier than simpler
 * areas.
 *
 * By painting the colors with the largest number of isolated areas or
 * the most complex shapes first, we help to minimize the number of fill
 * regions needed to render the complete picture.
 */
static int
gln_graphics_is_vertex (L9BYTE off_screen[],
			L9UINT16 width, L9UINT16 height, int x, int y)
{
	L9BYTE	pixel;				/* Reference pixel */
	int	above, below, left, right;	/* Neighbor difference flags */
	long	index_row;			/* Optimization variable */
	assert (off_screen != NULL);

	/* Use an index row to cut down on multiplications. */
	index_row = y * width;

	/* Find the color of the reference pixel. */
	pixel = off_screen[ index_row + x ];
	assert (pixel < GLN_PALETTE_SIZE);

	/*
	 * Detect differences between the reference pixel and its upper,
	 * lower, left and right neighbors.  Mark as different if the
	 * neighbor doesn't exist, that is, at the edge of the picture.
	 */
	above = (y == 0
		|| off_screen[ index_row - width + x ] != pixel);
	below = (y == height - 1
		|| off_screen[ index_row + width + x ] != pixel);
	left  = (x == 0
		|| off_screen[ index_row + x - 1 ] != pixel);
	right = (x == width - 1
		|| off_screen[ index_row + x + 1 ] != pixel);

	/*
	 * Return TRUE if this pixel lies at the vertex of a rectangular,
	 * fillable, area.  That is, if two adjacent neighbors aren't the
	 * same color (or if absent -- at the edge of the picture).
	 */
	return ((above || below) && (left || right));
}


/*
 * gln_graphics_assign_layers()
 *
 * Given two sets of image bitmaps, and a palette, this function will
 * assign layers palette colors.
 *
 * Layers are assigned by first counting the number of vertices in the
 * color plane, to get a measure of the complexity of shapes displayed in
 * this color, and also the raw number of times each palette color is
 * used.  This is then sorted, so that layers are assigned to colors, with
 * the lowest layer being the color with the most complex shapes, and
 * within this (or where the count of vertices is zero) the most used color.
 *
 * The function compares pixels in the two image bitmaps given, these
 * being the off-screen and on-screen buffers, and generates counts only
 * where these bitmaps differ.  This ensures that only pixels not yet
 * painted are included in layering.
 *
 * As well as assigning layers, this function returns a set of layer usage
 * flags, to help the rendering loop to terminate as early as possible.
 *
 * By painting lower layers first, the paint can take in larger areas if
 * it's permitted to include not-yet-validated higher levels.  This helps
 * minimize the amount of Glk areas fills needed to render a picture.
 */
static void
gln_graphics_assign_layers (L9BYTE off_screen[], L9BYTE on_screen[],
			L9UINT16 width, L9UINT16 height,
			int layers[], long layer_usage[])
{
	int	index, outer;			/* Palette iterators */
	int	x, y;				/* Image iterators */
	long	usage[GLN_PALETTE_SIZE];	/* Color use counts */
	long	complexity[GLN_PALETTE_SIZE];	/* Color vertex counts */
	long	colors[GLN_PALETTE_SIZE];	/* Temporary color indexes */
	long	index_row;			/* Optimization variable */
	assert (off_screen != NULL && on_screen != NULL);
	assert (layers != NULL && layer_usage != NULL);

	/* Clear initial complexity and usage counts. */
	for (index = 0; index < GLN_PALETTE_SIZE; index++)
	    {
		complexity[ index ] = 0;
		usage[ index ]      = 0;
	    }

	/*
	 * Traverse the image, counting vertices and pixel usage where the
	 * pixels differ between the off-screen and on-screen buffers.
	 * Optimize by maintaining an index row to avoid multiplications.
	 */
	for (y = 0, index_row = 0;
			y < height; y++, index_row += width)
	    {
		for (x = 0; x < width; x++)
		    {
			long	index;		/* x,y pixel index */

			/* Get the index for this pixel. */
			index = index_row + x;

			/* Update complexity and usage if pixels differ. */
			if (on_screen[ index ] != off_screen[ index ])
			    {
				if (gln_graphics_is_vertex (off_screen,
							width, height, x, y))
					complexity[ off_screen[ index ]]++;

				usage[ off_screen[ index ]]++;
			    }
		    }
	    }

	/*
	 * Sort counts to form color indexes.  The primary sort is on the
	 * shape complexity, and within this, on color usage.
	 *
	 * A colors might have no vertices at all, but rendering optimization
	 * relies on the first layer that contains no areas to fill halting
	 * the rendering loop.  So it's important here that we order indexes
	 * so that colors that render complex shapes come first, non-empty,
	 * but simpler shaped colors next, and finally all genuinely empty
	 * layers.
	 */
	for (index = 0; index < GLN_PALETTE_SIZE; index++)
		colors[ index ] = index;
	for (outer = 0; outer < GLN_PALETTE_SIZE - 1; outer++)
	    {
		int	is_sorted;		/* Array sorted flag */

		is_sorted = TRUE;
		for (index = 0;
			index < GLN_PALETTE_SIZE - outer - 1; index++)
		    {
			if (complexity[ index + 1 ] > complexity[ index ]
			    || (complexity[ index + 1 ] == complexity[ index ]
					&& usage[ index + 1 ] > usage[ index ]))
			    {
				/* Swap colors and matching counts. */
				gln_graphics_swap_array
						(colors, index, index + 1);
				gln_graphics_swap_array
						(complexity, index, index + 1);
				gln_graphics_swap_array
						(usage, index, index + 1);

				/* Flag as not yet sorted. */
				is_sorted = FALSE;
			    }
		    }
		if (is_sorted)
			break;
	    }

	/*
	 * Assign a layer to each palette color, and also return the layer
	 * usage for each layer.
	 */
	for (index = 0; index < GLN_PALETTE_SIZE; index++)
	    {
		layers[ colors[ index ]] = index;
		layer_usage[ index ]     = usage[ index ];
	    }
}


/*
 * gln_graphics_paint_region()
 *
 * This is a partially optimized point plot.  Given a point in the
 * graphics bitmap, it tries to extend the point to a color region, and
 * fill a number of pixels in a single Glk rectangle fill.  The goal
 * here is to reduce the number of Glk rectangle fills, which tend to be
 * extremely inefficient operations for generalized point plotting.
 *
 * The extension works in image layers; each palette color is assigned
 * a layer, and we paint each layer individually, starting at the lowest.
 * So, the region is free to fill any invalidated pixel in a higher layer,
 * and all pixels, invalidated or already validated, in the same layer.
 * In practice, it is good enough to look for either invalidated pixels
 * or pixels in the same layer, and construct a region as large as
 * possible from these, then on marking points as validated, mark only
 * those in the same layer as the initial point.
 *
 * The optimization here is not the best possible, but is reasonable.
 * What we do is to try and stretch the region horizontally first, then
 * vertically.  In practice, we might find larger areas by stretching
 * vertically and then horizontally, or by stretching both dimensions at
 * the same time.  In mitigation, the number of colors in a picture is
 * small (16), and the aspect ratio of pictures makes them generally
 * wider than they are tall.
 *
 * Once we've found the region, we render it with a single Glk rectangle
 * fill, and mark all the pixels in this region that match the layer of
 * the initial given point as validated.
 */
static void
gln_graphics_paint_region (winid_t glk_window,
			glui32 palette[], int layers[],
			L9BYTE off_screen[], L9BYTE on_screen[],
			int x, int y, int x_offset, int y_offset,
			int pixel_size, L9UINT16 width, L9UINT16 height)
{
	L9BYTE		pixel;			/* Reference pixel color */
	int		layer;			/* Reference pixel layer */
	int		x_min, x_max;		/* X region extent */
	int		y_min, y_max;		/* Y region extent */
	int		stop;			/* Stretch stop flag */
	int		x_index, y_index;	/* Region iterator indexes */
	long		index_row;		/* Optimization variable */
	assert (glk_window != NULL && palette != NULL && layers != NULL);
	assert (off_screen != NULL && on_screen != NULL);

	/* Find the color and layer for the initial pixel. */
	pixel = off_screen[ y * width + x ];
	layer = layers[ pixel ];
	assert (pixel < GLN_PALETTE_SIZE);

	/*
	 * Start by finding the extent to which we can pull the x coordinate
	 * and still find either invalidated pixels, or pixels in this layer.
	 *
	 * Use an index row to remove multiplications from the loops.
	 */
	index_row = y * width;
	for (x_min = x, stop = FALSE; x_min - 1 >= 0 && !stop; )
	    {
		long	index = index_row + x_min - 1;

		if (on_screen[ index ] == off_screen[ index ]
				&& layers[ off_screen[ index ]] != layer)
			stop = TRUE;
		if (!stop)
			x_min--;
	    }
	for (x_max = x, stop = FALSE; x_max + 1 < width && !stop; )
	    {
		long	index = index_row + x_max + 1;

		if (on_screen[ index ] == off_screen[ index ]
				&& layers[ off_screen[ index ]] != layer)
			stop = TRUE;
		if (!stop)
			x_max++;
	    }

	/*
	 * Now try to stretch the height of the region, by extending the
	 * y coordinate as much as possible too.  Again, we're looking
	 * for pixels that are invalidated or ones in the same layer.  We
	 * need to check across the full width of the current region.
	 *
	 * As above, an index row removes multiplications from the loops.
	 */
	for (y_min = y, index_row = (y - 1) * width, stop = FALSE;
			y_min - 1 >= 0 && !stop; index_row -= width)
	    {
		for (x_index = x_min; x_index <= x_max && !stop; x_index++)
		    {
			long	index = index_row + x_index;

			if (on_screen[ index ] == off_screen[ index ]
				    && layers[ off_screen[ index ]] != layer)
				stop = TRUE;
		    }
		if (!stop)
			y_min--;
	    }
	for (y_max = y, index_row = (y + 1) * width, stop = FALSE;
			y_max + 1 < height && !stop; index_row += width)
	    {
		for (x_index = x_min; x_index <= x_max && !stop; x_index++)
		    {
			long	index = index_row + x_index;

			if (on_screen[ index ] == off_screen[ index ]
				    && layers[ off_screen[ index ]] != layer)
				stop = TRUE;
		    }
		if (!stop)
			y_max++;
	    }

	/* Fill the region using Glk's rectangle fill. */
	glk_window_fill_rect (glk_window,
			palette[ pixel ],
			x_min * pixel_size + x_offset,
			y_min * pixel_size + y_offset,
			(x_max - x_min + 1) * pixel_size,
			(y_max - y_min + 1) * pixel_size);

	/*
	 * Validate each pixel in the reference layer that was rendered by
	 * the rectangle fill.  We don't validate pixels that are not in
	 * this layer (and are by definition in higher layers, as we've
	 * validated all lower layers), since although we colored them,
	 * we did it for optimization reasons, and they're not yet colored
	 * correctly.
	 *
	 * Maintain an index row as an optimization to avoid multiplication.
	 */
	index_row = y_min * width;
	for (y_index = y_min; y_index <= y_max; y_index++)
	    {
		for (x_index = x_min; x_index <= x_max; x_index++)
		    {
			long	index;			/* x,y pixel index */

			/* Get the index for x_index,y_index. */
			index = index_row + x_index;

			/* If the layers match, update the on-screen buffer. */
			if (layers[ off_screen[ index ]] == layer)
			    {
				assert (off_screen[ index ] == pixel);
				on_screen[ index ] = off_screen[ index ];
			    }
		    }

		/* Update row index component on change of y. */
		index_row += width;
	    }
}


/*
 * gln_graphics_timeout()
 *
 * This is a background function, called on Glk timeouts.  Its job is to
 * repaint some of the current graphics image.  On successive calls, it
 * does a part of the repaint, then yields to other processing.  This is
 * useful since the Glk primitive to plot points in graphical windows is
 * extremely slow; this way, the repaint doesn't block game play.
 *
 * The function should be called on Glk timeout events.  When the repaint
 * is complete, the function will turn off Glk timers.
 *
 * The function uses double-buffering to track how much of the graphics
 * buffer has been rendered.  This helps to minimize the amount of point
 * plots required, as only the differences between the two buffers need
 * to be rendered.
 */
static void
gln_graphics_timeout (void)
{
	static glui32	palette[GLN_PALETTE_SIZE];
						/* Precomputed Glk palette */
	static int	layers[GLN_PALETTE_SIZE];
						/* Assigned image layers */
	static long	layer_usage[GLN_PALETTE_SIZE];
						/* Image layer occupancies */
	static L9BYTE	*on_screen	= NULL;	/* On-screen image buffer */
	static L9BYTE	*off_screen	= NULL;	/* Off-screen image buffer */

	static int	deferred_repaint = FALSE;
						/* Local delayed repaint flag */
	static int	ignore_counter;		/* Count of calls ignored */

	static int	x_offset, y_offset;	/* Point plot offsets */
	static int	yield_counter;		/* Yields in rendering */
	static int	saved_layer;		/* Saved current layer */
	static int	saved_x, saved_y;	/* Saved x,y coord */

	static int	total_regions;		/* Debug statistic */

	long		picture_size;		/* Picture size in pixels */
	int		layer;			/* Image layer iterator */
	int		x, y;			/* Image iterators */
	int		regions;		/* Count of regions painted */

	/* Ignore the call if the current graphics state is inactive. */
	if (!gln_graphics_active)
		return;
	assert (gln_graphics_window != NULL);

	/*
	 * On detecting a repaint request, note the flag in a local static
	 * variable, then set up a graphics delay to wait until, hopefully,
	 * the resize, if that's what caused it, is complete, and return.
	 * This makes resizing the window a lot smoother, since it prevents
	 * unnecessary region paints where we are receiving consecutive Glk
	 * arrange or redraw events.
	 */
	if (gln_graphics_repaint)
	    {
		deferred_repaint	= TRUE;
		gln_graphics_repaint	= FALSE;
		ignore_counter		= GLN_GRAPHICS_REPAINT_WAIT - 1;
		return;
	    }

	/*
	 * If asked to ignore a given number of calls, decrement the ignore
	 * counter and return having done nothing more.  This lets us delay
	 * graphics operations by a number of timeouts, providing partial
	 * protection from resize event "storms".
	 *
	 * Note -- to wait for N timeouts, set the count of timeouts to be
	 * ignored to N-1.
	 */
	assert (ignore_counter >= 0);
	if (ignore_counter > 0)
	    {
		ignore_counter--;
		return;
	    }

	/* Calculate the picture size, useful throughout. */
	picture_size = gln_graphics_width * gln_graphics_height;

	/*
	 * If we received a new picture, set up the local static variables
	 * for that picture -- convert the color palette, and initialize
	 * the off_screen buffer to be the base picture.
	 */
	if (gln_graphics_new_picture)
	    {
		/*
		 * Initialize the off_screen buffer to be a copy of the base
		 * picture.
		 */
		if (off_screen != NULL)
			free (off_screen);
		off_screen = gln_malloc (picture_size * sizeof (*off_screen));
		memcpy (off_screen, gln_graphics_bitmap,
					picture_size * sizeof (*off_screen));

		/* Note the buffer for freeing on cleanup. */
		gln_graphics_off_screen = off_screen;


		/*
		 * Pre-convert all the picture palette colors into their
		 * corresponding Glk colors.
		 */
		gln_graphics_convert_palette (gln_graphics_palette, palette);

		/* Save the color count for possible queries later. */
		gln_graphics_color_count = gln_graphics_count_colors
						(off_screen,
						gln_graphics_width,
						gln_graphics_height);
	    }

	/*
	 * For a new picture, or a repaint of a prior one, calculate new
	 * values for the x and y offsets used to draw image points, and
	 * set the on-screen buffer to an unused pixel value, in effect
	 * invalidating all on-screen data.  Also, reset the saved image
	 * scan coordinates so that we scan for unpainted pixels from top
	 * left starting at layer zero, and clear the graphics window.
	 */
	if (gln_graphics_new_picture
			|| deferred_repaint)
	    {
		/*
		 * Calculate the x and y offset to center the picture in
		 * the graphics window.
		 */
		gln_graphics_position_picture (gln_graphics_window,
				GLN_GRAPHICS_PIXEL,
				gln_graphics_width, gln_graphics_height,
				&x_offset, &y_offset);

		/*
		 * Reset all on-screen pixels to an unused value, guaranteed
		 * not to match any in a real picture.  This forces all
		 * pixels to be repainted on a buffer/on-screen comparison.
		 */
		if (on_screen != NULL)
			free (on_screen);
		on_screen = gln_malloc (picture_size * sizeof (*on_screen));
		memset (on_screen, GLN_GRAPHICS_UNUSED_PIXEL,
					picture_size * sizeof (*on_screen));

		/* Note the buffer for freeing on cleanup. */
		gln_graphics_on_screen = on_screen;

		/*
		 * Assign new layers to the current image.  This sorts
		 * colors by usage and puts the most used colors in the
		 * lower layers.  It also hands us a count of pixels in
		 * each layer, useful for knowing when to stop scanning for
		 * layers in the rendering loop.
		 */
		gln_graphics_assign_layers (off_screen, on_screen,
				gln_graphics_width, gln_graphics_height,
							layers, layer_usage);

		/* Clear the graphics window. */
		gln_graphics_clear_and_border (gln_graphics_window,
						x_offset, y_offset,
						GLN_GRAPHICS_PIXEL,
						gln_graphics_width,
						gln_graphics_height);

		/* Start a fresh picture rendering pass. */
		yield_counter	= 0;
		saved_layer	= 0;
		saved_x		= 0;
		saved_y		= 0;
		total_regions	= 0;

		/* Clear the new picture and deferred repaint flags. */
		gln_graphics_new_picture	= FALSE;
		deferred_repaint		= FALSE;
	    }

	/*
	 * Make a portion of an image pass, from lower to higher image layers,
	 * scanning for invalidated pixels that are in the current image layer
	 * we are painting.  Each invalidated pixel gives rise to a region
	 * paint, which equates to one Glk rectangle fill.
	 *
	 * When the limit on regions is reached, save the current image pass
	 * layer and coordinates, and yield control to the main game playing
	 * code by returning.  On the next call, pick up where we left off.
	 *
	 * As an optimization, we can leave the loop on the first empty layer
	 * we encounter.  Since layers are ordered by complexity and color
	 * usage, all layers higher than the first unused one will also be
	 * empty, so we don't need to scan them.
	 */
	regions = 0;
	for (layer = saved_layer;
		layer < GLN_PALETTE_SIZE && layer_usage[ layer ] > 0;
		layer++)
	    {
		long	index_row;		/* Optimization variable */

		/*
		 * As an optimization to avoid multiplications in the loop,
		 * maintain a separate index row.
		 */
		index_row = saved_y * gln_graphics_width;
		for (y = saved_y; y < gln_graphics_height; y++)
		    {
			for (x = saved_x; x < gln_graphics_width; x++)
			    {
				long	index;		/* x,y pixel index */

				/* Get the index for this pixel. */
				index = index_row + x;
				assert (index < picture_size
							* sizeof (*off_screen));

				/*
				 * Ignore pixels not in the current layer, and
				 * pixels not currently invalid (that is, ones
				 * whose on-screen representation matches the
				 * off-screen buffer).
				 */
				if (layers[ off_screen[ index ]] == layer
					&& on_screen[ index ]
							!= off_screen[ index ])
				    {
					/*
					 * Rather than painting just one pixel,
					 * here we try to paint the maximal
					 * region we can for the layer of the
					 * given pixel.
					 */
					gln_graphics_paint_region
						(gln_graphics_window,
						palette, layers,
						off_screen, on_screen,
						x, y, x_offset, y_offset,
						GLN_GRAPHICS_PIXEL,
						gln_graphics_width,
						gln_graphics_height);

					/*
					 * Increment count of regions handled,
					 * and yield, by returning, if the
					 * limit on paint regions is reached.
					 * Before returning, save the current
					 * layer and scan coordinates, so we
					 * can pick up here on the next call.
					 */
					regions++;
					if (regions >= GLN_REPAINT_LIMIT)
					    {
						yield_counter++;
						saved_layer	= layer;
						saved_x		= x;
						saved_y		= y;
						total_regions	+= regions;
						return;
					    }
				    }
			    }

			/* Reset the saved x coordinate on y increment. */
			saved_x = 0;

			/* Update the index row on change of y. */
			index_row += gln_graphics_width;
		    }

		/* Reset the saved y coordinate on layer change. */
		saved_y = 0;
	    }

	/*
	 * If we reach this point, then we didn't get to the limit on regions
	 * painted on this pass.  In that case, we've finished rendering the
	 * image.
	 */
	assert (regions < GLN_REPAINT_LIMIT);
	total_regions += regions;

	/*
	 * Stop graphics.  There's no more to be done until something
	 * restarts us.
	 */
	gln_graphics_stop ();
}


/*
 * gln_graphics_locate_bitmaps()
 *
 * Given the name of the game file being run, try to set up the graphics
 * directory and bitmap type for that game.  If none available, leave
 * the directory as NULL, and bitmap type as NO_BITMAPS.
 */
static void
gln_graphics_locate_bitmaps (const char *gamefile)
{
	const char	*basename;		/* Base of gamefile. */
	char		*dirname;		/* Directory gamefile. */
	BitmapType	bitmap_type;		/* Detected bitmap type. */

	/* Find the start of the last element of the filename passed in. */
	basename = strrchr (gamefile, GLN_FILE_DELIM);
	if (basename == NULL)
		basename = gamefile;
	else
		basename++;

	/* Take a copy of the directory part of the filename. */
	dirname = gln_malloc (basename - gamefile + 1);
	strncpy (dirname, gamefile, basename - gamefile);
	dirname[ basename - gamefile ] = '\0';

	/*
	 * Use the core interpreter to search for suitable bitmaps.  If
	 * none found, free allocated memory and return noting none available.
	 */
	bitmap_type = DetectBitmaps (dirname);
	if (bitmap_type == NO_BITMAPS)
	    {
		free (dirname);
		gln_graphics_bitmap_directory	= NULL;
		gln_graphics_bitmap_type	= NO_BITMAPS;
		return;
	    }

	/* Record the bitmap details for later use. */
	gln_graphics_bitmap_directory	= dirname;
	gln_graphics_bitmap_type	= bitmap_type;
}


/*
 * gln_graphics_handle_title_picture()
 *
 * Picture 0 is special, normally the title picture.  Unless we handle it
 * specially, the next picture comes along and instantly overwrites it.
 * Here, then, we try to delay until the picture has rendered, allowing the
 * delay to be broken with a keypress.
 */
static void
gln_graphics_handle_title_picture (void)
{
	event_t		event;			/* Glk event buffer. */
	int		count;			/* Timeout counter. */

	/* Explain what's going on. */
	gln_standout_string
		("\n[ Press any key to skip the title picture... ]\n\n");

	/* Wait until a keypress or graphics rendering is complete. */
	glk_request_char_event (gln_main_window);
	do
	    {
		gln_event_wait_2 (evtype_CharInput, evtype_Timer, &event);

		/*
		 * If a character was pressed, return.  This will let
		 * the game progress, probably into showing the next bitmap.
		 */
		 if (event.type == evtype_CharInput)
		     {
			gln_watchdog_tick ();
			return;
		     }
	    }
	while (gln_graphics_active);

	/* Now wait another couple of seconds. */
	glk_request_timer_events (GLN_GRAPHICS_TIMEOUT);
	for (count = 0; count < GLN_GRAPHICS_TITLE_WAIT; count++)
	    {
		gln_event_wait_2 (evtype_CharInput, evtype_Timer, &event);

		/*
		 * Again, if a character was pressed, return.  Remember to
		 * stop timer events in this case.
		 */
		 if (event.type == evtype_CharInput)
		     {
			glk_request_timer_events (0);
			gln_watchdog_tick ();
			return;
		     }
	    }

	/* Cancel timers and pending character event, and continue on. */
	glk_request_timer_events (0);
	glk_cancel_char_event (gln_main_window);
	gln_watchdog_tick ();
}


/*
 * os_show_bitmap()
 *
 * Called by the main interpreter when it wants us to display a picture.
 *
 * The function gets the picture bitmap, palette, and dimensions, and
 * saves them, and the picture id, in module variables for the background
 * rendering function.
 */
void
os_show_bitmap(int picture, int x, int y)
{
	Bitmap		*bitmap;		/* New picture bitmap */
	long		picture_bytes;		/* Picture size in bytes */

	/*
	 * If interpreter graphics are disabled, the only way we can get
	 * into here is using #picture.  It seems that the interpreter
	 * won't always deliver correct bitmaps with #picture when in
	 * text mode, so it's simplest here if we just ignore those calls.
	 */
	if (gln_graphics_interpreter_state != GLN_GRAPHICS_BITMAP_MODE)
		return;

	/* Ignore repeat calls for the currently displayed picture. */
	if (picture == gln_graphics_picture)
		return;

	/*
	 * Get the core interpreter's bitmap for the requested picture.  If
	 * this returns NULL, the picture doesn't exist, so ignore the call
	 * silently.
	 */
	bitmap = DecodeBitmap (gln_graphics_bitmap_directory,
				gln_graphics_bitmap_type,
				picture, x, y);
	if (bitmap == NULL)
		return;

	/*
	 * Note the last thing passed to os_show_bitmap, to avoid possible
	 * repaints of the current picture.
	 */
	gln_graphics_picture = picture;

	/* Calculate the picture size in bytes. */
	picture_bytes = bitmap->width * bitmap->height
					* sizeof (*bitmap->bitmap);

	/*
	 * Save the picture details for the update code.  Here we take a
	 * complete local copy of the bitmap, dimensions, and palette.
	 * The core interpreter may return a palette with fewer colors than
	 * our maximum, so unused local palette entries are set to zero.
	 */
	if (gln_graphics_bitmap != NULL)
		free (gln_graphics_bitmap);
	gln_graphics_bitmap		= gln_malloc (picture_bytes);
	memcpy (gln_graphics_bitmap, bitmap->bitmap, picture_bytes);
	gln_graphics_width		= bitmap->width;
	gln_graphics_height		= bitmap->height;
	memset (gln_graphics_palette, 0, sizeof (gln_graphics_palette));
	memcpy (gln_graphics_palette, bitmap->palette,
			bitmap->npalette * sizeof (bitmap->palette[0]));

	/*
	 * If graphics are enabled, both at the Glk level and in the core
	 * interpreter, ensure the window is displayed, set the appropriate
	 * flags, and start graphics update.  If they're not enabled, the
	 * picture details will simply stick around in module variables
	 * until they are required.
	 */
	if (gln_graphics_enabled
		&& gln_graphics_interpreter_state == GLN_GRAPHICS_BITMAP_MODE)
	    {
		/* Ensure graphics on, and return (fail) if not possible. */
		if (!gln_graphics_open ())
			return;

		/*
		 * Set the new picture flag, and start the updating
		 * "thread".
		 */
		gln_graphics_new_picture = TRUE;
		gln_graphics_start ();

		/* If this is the title picture, start special handling. */
		if (picture == GLN_GRAPHICS_TITLE_PICTURE)
			gln_graphics_handle_title_picture ();
	    }
}


/*
 * gln_graphics_picture_is_available()
 *
 * Return TRUE if the graphics module data is loaded with a usable picture,
 * FALSE if there is no picture available to display.
 */
static int
gln_graphics_picture_is_available (void)
{
	return gln_graphics_bitmap != NULL;
}


/*
 * gln_graphics_get_picture_details()
 *
 * Return the width and height of the currently loaded picture.  The function
 * returns FALSE if no picture is loaded, otherwise TRUE, with picture
 * details in the return arguments.
 */
static int
gln_graphics_get_picture_details (int *width, int *height)
{
	/* If no picture available, return FALSE. */
	if (!gln_graphics_picture_is_available ())
		return FALSE;

	/* If requested, return width and height. */
	if (width != NULL)
		*width		= gln_graphics_width;
	if (height != NULL)
		*height		= gln_graphics_height;

	/* Details returned. */
	return TRUE;
}


/*
 * gln_graphics_get_rendering_details()
 *
 * Returns the type of bitmap in use (if any), as a string, the count of
 * colors in the picture, and a flag indicating if graphics is active (busy).
 * The function return FALSE if graphics is not enabled or if not being
 * displayed, otherwise TRUE with the bitmap type, color count, and active
 * flag in the return arguments.
 *
 * This function races with the graphics timeout, as it returns information
 * set up by the first timeout following a new picture.  There's a very
 * very small chance that it might win the race, in which case out-of-date
 * values are returned.
 */
static int
gln_graphics_get_rendering_details (const char **bitmap_type,
			int *color_count, int *is_active)
{
	/* If graphics isn't enabled and displayed, return FALSE. */
	if (!gln_graphics_enabled
			|| !gln_graphics_are_displayed ())
		return FALSE;

	/*
	 * Convert the detected bitmap type into a string and return it.
	 * A NULL bitmap string implies no bitmaps.
	 */
	if (bitmap_type != NULL)
	    {
		switch (gln_graphics_bitmap_type)
		    {
		    case NO_BITMAPS:	*bitmap_type = NULL;		break;
		    case AMIGA_BITMAPS:	*bitmap_type = "Amiga";		break;
		    case PC1_BITMAPS:	*bitmap_type = "IBM PC(1)";	break;
		    case PC2_BITMAPS:	*bitmap_type = "IBM PC(2)";	break;
		    case C64_BITMAPS:	*bitmap_type = "C64";		break;
		    case BBC_BITMAPS:	*bitmap_type = "BBC B";		break;
		    case CPC_BITMAPS:	*bitmap_type = "CPC/Spectrum";	break;
		    case MAC_BITMAPS:	*bitmap_type = "Macintosh";	break;
		    case ST1_BITMAPS:	*bitmap_type = "Atari ST(1)";	break;
		    case ST2_BITMAPS:	*bitmap_type = "Atari ST(2)";	break;
		    default:		*bitmap_type = NULL;		break;
		    }
	    }

	/*
	 * Return the color count noted by timeouts on the first timeout
	 * following a new picture.  We might return the one for the prior
	 * picture.
	 */
	if (color_count != NULL)
		*color_count = gln_graphics_color_count;

	/* Return graphics active flag. */
	if (is_active != NULL)
		*is_active = gln_graphics_active;

	/* Details returned. */
	return TRUE;
}


/*
 * gln_graphics_interpreter_enabled()
 *
 * Return TRUE if it looks like interpreter graphics are turned on, FALSE
 * otherwise.
 */
static int
gln_graphics_interpreter_enabled (void)
{
	return gln_graphics_interpreter_state != GLN_GRAPHICS_OFF;
}


/*
 * gln_graphics_cleanup()
 *
 * Free memory resources allocated by graphics functions.  Called on game
 * end.
 */
static void
gln_graphics_cleanup (void)
{
	if (gln_graphics_bitmap != NULL)
	    {
		free (gln_graphics_bitmap);
		gln_graphics_bitmap = NULL;
	    }
	if (gln_graphics_off_screen != NULL)
	    {
		free (gln_graphics_off_screen);
		gln_graphics_off_screen = NULL;
	    }
	if (gln_graphics_on_screen != NULL)
	    {
		free (gln_graphics_on_screen);
		gln_graphics_on_screen = NULL;
	    }
	if (gln_graphics_bitmap_directory == NULL)
	    {
		free (gln_graphics_bitmap_directory);
		gln_graphics_bitmap_directory = NULL;
	    }
}


/*---------------------------------------------------------------------*/
/*  Glk port line drawing picture adapter functions                    */
/*---------------------------------------------------------------------*/

/*
 * Graphics colour table.  These eight colors are selected into the four-
 * color palette by os_setcolour().  The standard Amiga palette is rather
 * over-vibrant, so to soften it a bit this table uses non-primary colors.
 */
static const gln_rgb_t GLN_LINEGRAPHICS_COLOR_TABLE[] = {
	{  47,  79,  79 },	/* DarkSlateGray	[Black] */
	{ 238,  44,  44 },	/* Firebrick2		[Red] */
	{  67, 205, 128 },	/* SeaGreen3		[Green] */
	{ 238, 201,   0 },	/* Gold2		[Yellow] */
	{  92, 172, 238 },	/* SteelBlue2		[Blue] */
	{ 139,  87,  66 },	/* LightSalmon4		[Brown] */
	{ 175, 238, 238 },	/* PaleTurquoise	[Cyan] */
	{ 245, 245, 245 },	/* WhiteSmoke		[White] */
};

/*
 * Structure of a Seed Fill segment entry, and a growable stack-based array
 * of segments pending fill.  When length exceeds size, size is increased
 * and the array grown.
 */
typedef struct {
	int	y;		/* Segment y coordinate */
	int	xl;		/* Segment x left hand side coordinate */
	int	xr;		/* Segment x right hand side coordinate */
	int	dy;		/* Segment y delta */
} gln_linegraphics_segment_t;

static gln_linegraphics_segment_t
			*gln_linegraphics_fill_segments		= NULL;
static int		gln_linegraphics_fill_segments_size	= 0;
static int		gln_linegraphics_fill_segments_length	= 0;


/*
 * gln_linegraphics_create_context()
 *
 * Initialize a new constructed bitmap graphics context for line drawn
 * graphics.
 */
static void
gln_linegraphics_create_context (void)
{
	int		width, height;		/* Picture dimensions */
	long		picture_bytes;		/* Picture size in bytes */

	/* Get the picture size, and calculate the bytes in the bitmap. */
	GetPictureSize (&width, &height);
	picture_bytes = width * height * sizeof (*gln_graphics_bitmap);

	/*
	 * Destroy any current bitmap, and begin a fresh one.  Here we set
	 * the bitmap and the palette to all zeroes; this equates to all
	 * black.
	 */
	if (gln_graphics_bitmap != NULL)
		free (gln_graphics_bitmap);
	gln_graphics_bitmap		= gln_malloc (picture_bytes);
	memset (gln_graphics_bitmap, 0, picture_bytes);
	gln_graphics_width		= width;
	gln_graphics_height		= height;
	memset (gln_graphics_palette, 0, sizeof (gln_graphics_palette));

	/*
	 * Set the graphics picture number to -1, as this is not a real
	 * game bitmap.
	 */
	gln_graphics_picture		= -1;
}


/*
 * gln_linegraphics_clear_context()
 *
 * Clear the complete graphical drawing area, setting all pixels to zero,
 * and resetting the palette to all black as well.
 */
static void
gln_linegraphics_clear_context (void)
{
	long		picture_bytes;		/* Picture size in bytes */

	/* Get the picture size, and zero all bytes in the bitmap. */
	picture_bytes = gln_graphics_width * gln_graphics_height
				* sizeof (*gln_graphics_bitmap);
	memset (gln_graphics_bitmap, 0, picture_bytes);

	/* Clear the palette colors too. */
	memset (gln_graphics_palette, 0, sizeof (gln_graphics_palette));
}


/*
 * gln_linegraphics_set_palette_color()
 *
 * Copy the indicated main color table entry into the palette.
 */
static void
gln_linegraphics_set_palette_color (int colour, int index)
{
	gln_rgb_t	const *entry;		/* Color table entry */
	assert (colour < GLN_PALETTE_SIZE);
	assert (index < sizeof (GLN_LINEGRAPHICS_COLOR_TABLE)
			/ sizeof (GLN_LINEGRAPHICS_COLOR_TABLE[0]));

	/* Copy the colour table entry to the constructed game palette. */
	entry = GLN_LINEGRAPHICS_COLOR_TABLE + index;
	gln_graphics_palette[ colour ].red	= entry->red;
	gln_graphics_palette[ colour ].green	= entry->green;
	gln_graphics_palette[ colour ].blue	= entry->blue;
}


/*
 * gln_linegraphics_get_pixel()
 * gln_linegraphics_set_pixel()
 *
 * Return and set the bitmap pixel at x,y.
 */
static L9BYTE
gln_linegraphics_get_pixel (int x, int y)
{
	assert (x >= 0 && x < gln_graphics_width
		&& y >= 0 && y < gln_graphics_height);

	return gln_graphics_bitmap[ y * gln_graphics_width + x ];
}
static void
gln_linegraphics_set_pixel (int x, int y, L9BYTE color)
{
	assert (x >= 0 && x < gln_graphics_width
		&& y >= 0 && y < gln_graphics_height);

	gln_graphics_bitmap[ y * gln_graphics_width + x ] = color;
}


/*
 * gln_linegraphics_plot_clip()
 * gln_linegraphics_draw_line_if()
 *
 * Draw a line from x1,y1 to x2,y2 in colour1, where the existing pixel
 * colour is colour2.  The function uses Bresenham's algorithm.  The second
 * function, gln_graphics_plot_clip, is a line drawing helper; it handles
 * clipping, and the requirement to plot a point only if it matches colour2.
 */
static void
gln_linegraphics_plot_clip (int x, int y, int colour1, int colour2)
{
	/* Clip the plot if the value is outside the context. */
	if (x < 0 || x >= gln_graphics_width
			|| y < 0 || y >= gln_graphics_height)
		return;

	/* Plot the pixel as colour1 if it is currently colour2. */
	if (gln_linegraphics_get_pixel (x, y) == colour2)
		gln_linegraphics_set_pixel (x, y, colour1);
}
static void
gln_linegraphics_draw_line_if (int x1, int y1, int x2, int y2,
				int colour1, int colour2)
{
	int		x, y, dx, dy;		/* Line coordinates */
	int		incx, incy;		/* Increments */
	int             balance;		/* Line remainder */

	/* Ignore any odd request where there will be no colour changes. */
	if (colour1 == colour2)
		return;

	/* Normalize the line into deltas and increments. */
	if (x2 >= x1)
	    {
		dx = x2 - x1;
		incx =  1;
	    }
	else
	    {
		dx = x1 - x2;
		incx = -1;
	    }

	if (y2 >= y1)
	    {
		dy = y2 - y1;
		incy =  1;
	    }
	else
	    {
		dy = y1 - y2;
		incy = -1;
	    }

	/* Start at x1,y1. */
	x = x1;
	y = y1;

	/* Decide on a direction to progress in. */
        if (dx >= dy)
	    {
		dy <<= 1;
		balance = dy - dx;
		dx <<= 1;

		/* Loop until we reach the end point of the line. */
		while (x != x2)
		    {
			gln_linegraphics_plot_clip (x, y, colour1, colour2);
			if (balance >= 0)
			    {
				y += incy;
				balance -= dx;
			    }
			balance += dy;
			x += incx;
		    }
		gln_linegraphics_plot_clip (x, y, colour1, colour2);
	    }
	else
	    {
		dx <<= 1;
		balance = dx - dy;
		dy <<= 1;

		/* Loop until we reach the end point of the line. */
		while (y != y2)
		    {
			gln_linegraphics_plot_clip (x, y, colour1, colour2);
			if (balance >= 0)
			    {
				x += incx;
				balance -= dy;
			    }
			balance += dx;
			y += incy;
		    }
		gln_linegraphics_plot_clip (x, y, colour1, colour2);
	    }
}


/*
 * gln_linegraphics_push_fill_segment()
 * gln_linegraphics_pop_fill_segment()
 * gln_linegraphics_fill_4way_if()
 *
 * Area fill algorithm, set a region to color1 if it is currently set to
 * color2.  This function is a derivation of Paul Heckbert's Seed Fill,
 * from "Graphics Gems", Academic Press, 1990, which fills 4-connected
 * neighbors.
 * 
 * The main modification is to make segment stacks growable, through the
 * helper push and pop functions.  There is also a small adaptation to
 * check explicitly for color2, to meet the Level9 API.
 */
static void
gln_linegraphics_push_fill_segment (int y, int xl, int xr, int dy)
{
	int		length, size;		/* Local dimensions copy */

	/* Clip points outside the graphics context. */
	if (y + dy < 0 || y + dy >= gln_graphics_height)
		return;

	length = ++gln_linegraphics_fill_segments_length;
	size   = gln_linegraphics_fill_segments_size;

	/* Grow the segments stack if required, successively doubling. */
	if (length > size)
	    {
		size = size == 0 ? gln_graphics_height : size << 1;

		gln_linegraphics_fill_segments = gln_realloc
			(gln_linegraphics_fill_segments,
			 size * sizeof (*gln_linegraphics_fill_segments));
	    }

	/* Push top of segments stack. */
	gln_linegraphics_fill_segments[ length - 1 ].y  = y;
	gln_linegraphics_fill_segments[ length - 1 ].xl = xl;
	gln_linegraphics_fill_segments[ length - 1 ].xr = xr;
	gln_linegraphics_fill_segments[ length - 1 ].dy = dy;

	/* Write back local dimensions copies. */
	gln_linegraphics_fill_segments_length	= length;
	gln_linegraphics_fill_segments_size	= size;
}
static void
gln_linegraphics_pop_fill_segment (int *y, int *xl, int *xr, int *dy)
{
	int		length;			/* Local stack length copy */
	assert (gln_linegraphics_fill_segments_length > 0);

	length = --gln_linegraphics_fill_segments_length;

	/* Pop top of segments stack. */
	*y  = gln_linegraphics_fill_segments[ length ].y;
	*xl = gln_linegraphics_fill_segments[ length ].xl;
	*xr = gln_linegraphics_fill_segments[ length ].xr;
	*dy = gln_linegraphics_fill_segments[ length ].dy;
}
static void
gln_linegraphics_fill_4way_if (int x, int y, int colour1, int colour2)
{
	int		left, x1, x2, dy;	/* Fill variables */
	int		x_lo, x_hi;		/* Inclusive x dimensions */

	/* Ignore requests where colours 1 and 2 are the same. */
	if (colour1 == colour2)
		return;

	/* Clip fill requests to visible graphics region. */
	if (x < 0 || x >= gln_graphics_width)
		return;
	if (y < 0 || y >= gln_graphics_height)
		return;

	/*
	 * Level9 API; explicit check for a match against colour2.  This
	 * also covers the standard Seed Fill check that old pixel value
	 * should not equal colour1, because of the color1 == colour2
	 * comparison above.
	 */
	if (gln_linegraphics_get_pixel (x, y) != colour2)
		return;

	/*
	 * Set up inclusive window dimension to ease algorithm translation.
	 * The original worked with inclusive rectangle limits.
	 */
	x_lo = 0;
	x_hi = gln_graphics_width - 1;

	/*
	 * The first of these is "needed in some cases", the second is the
	 * seed segment, popped first.
	 */
	gln_linegraphics_push_fill_segment (y, x, x, 1);
	gln_linegraphics_push_fill_segment (y + 1, x, x, -1);

	while (gln_linegraphics_fill_segments_length > 0)
	    {
		/* Pop segment off stack and add delta to y coord. */
		gln_linegraphics_pop_fill_segment (&y, &x1, &x2, &dy);
		y += dy;

		/*
		 * Segment of scan line y-dy for x1<=x<=x2 was previously
		 * filled, now explore adjacent pixels in scan line y.
		 */
		for (x = x1;
			x >= x_lo
			&& gln_linegraphics_get_pixel (x, y) == colour2; x--)
		    {
			gln_linegraphics_set_pixel (x, y, colour1);
		    }

		if (x >= x1)
			goto skip;

		left = x + 1;
		if (left < x1)
		    {
		    	/* Leak on left? */
			gln_linegraphics_push_fill_segment
						(y, left, x1 - 1, -dy);
		    }

		x = x1 + 1;
		do
		    {
			for ( ;
				x <= x_hi
				&& gln_linegraphics_get_pixel (x, y) == colour2;
				x++)
			    {
				gln_linegraphics_set_pixel (x, y, colour1);
			    }

			gln_linegraphics_push_fill_segment (y, left, x - 1, dy);

			if (x > x2 + 1)
			    {
				/* Leak on right? */
				gln_linegraphics_push_fill_segment
							(y, x2 + 1, x - 1, -dy);
			    }

skip:			for (x++;
				x <= x2
				&& gln_linegraphics_get_pixel (x, y) != colour2;
				x++);

			left = x;
		    }
		while (x <= x2);
	    }
}


/*
 * os_cleargraphics()
 * os_setcolour()
 * os_drawline()
 * os_fill()
 *
 * Interpreter entry points for line drawing graphics.  All calls to these
 * are ignored if line drawing mode is not set.
 */
void
os_cleargraphics (void)
{
	if (gln_graphics_interpreter_state == GLN_GRAPHICS_LINE_MODE)
		gln_linegraphics_clear_context ();

}
void
os_setcolour (int colour, int index)
{
	if (gln_graphics_interpreter_state == GLN_GRAPHICS_LINE_MODE)
		gln_linegraphics_set_palette_color (colour, index);

}
void
os_drawline (int x1, int y1, int x2, int y2, int colour1, int colour2)
{
	if (gln_graphics_interpreter_state == GLN_GRAPHICS_LINE_MODE)
		gln_linegraphics_draw_line_if (x1, y1, x2, y2,
						colour1, colour2);
}
void
os_fill (int x, int y, int colour1, int colour2)
{
	if (gln_graphics_interpreter_state == GLN_GRAPHICS_LINE_MODE)
		gln_linegraphics_fill_4way_if (x, y, colour1, colour2);

}


/*
 * gln_linegraphics_process()
 *
 * Process as many graphics opcodes as are available, constructing the
 * resulting image as a bitmap.  When complete, treat as normal bitmaps.
 */
static void
gln_linegraphics_process (void)
{
	int		opcodes_count;		/* Graphics opcode count. */

	/*
	 * If interpreter graphics are not set to line mode, ignore any
	 * call that arrives here.
	 */
	if (gln_graphics_interpreter_state != GLN_GRAPHICS_LINE_MODE)
		return;

	/* Run all the available graphics opcodes. */
	opcodes_count = 0;
	while (RunGraphics ())
		opcodes_count++;

	/* 
	 * If graphics is enabled and we created an image with graphics
	 * opcodes above, open a graphics window and start bitmap display.
	 */
	if (gln_graphics_enabled && opcodes_count > 0)
	    {
		/* Ensure graphics on, and return (fail) if not possible. */
		if (!gln_graphics_open ())
			return;

		/*
		 * Set the new picture flag, and start the updating
		 * "thread".
		 */
		gln_graphics_new_picture = TRUE;
		gln_graphics_start ();
	    }
}


/*
 * gln_linegraphics_cleanup()
 *
 * Free memory resources allocated by line graphics functions.  Called on
 * game end.
 */
static void
gln_linegraphics_cleanup (void)
{
	if (gln_linegraphics_fill_segments != NULL)
	    {
		free (gln_linegraphics_fill_segments);
		gln_linegraphics_fill_segments = NULL;
	    }
}


/*---------------------------------------------------------------------*/
/*  Glk picture dispatch (bitmap or line), and timer arbitration       */
/*---------------------------------------------------------------------*/

/*
 * Note of the current set graphics mode, to detect changes in mode from
 * the core interpreter.
 */
static int	gln_graphics_current_mode	= -1;

/* Note indicating if the graphics "thread" is temporarily suspended. */
static int	gln_graphics_suspended		= FALSE;


/*
 * os_graphics()
 *
 * Called by the main interpreter to turn graphics on and off.  Mode 0
 * turns graphics off, mode 1 is line drawing graphics, and mode 2 is
 * bitmap graphics.
 *
 * This function tracks the current state of interpreter graphics setting
 * using gln_graphics_interpreter_state.
 */
void
os_graphics (int mode)
{
	/* Ignore the call unless it changes the graphics mode. */
	if (mode == gln_graphics_current_mode)
		return;

	/* Set tracked interpreter state given the input mode. */
	switch (mode)
	    {
	    case 0:
		gln_graphics_interpreter_state = GLN_GRAPHICS_OFF;
		break;

	    case 1:
		gln_graphics_interpreter_state = GLN_GRAPHICS_LINE_MODE;
		break;

	    case 2:
		/* If no graphics bitmaps were detected, ignore this call. */
		if (gln_graphics_bitmap_directory == NULL
				|| gln_graphics_bitmap_type == NO_BITMAPS)
			return;

		gln_graphics_interpreter_state = GLN_GRAPHICS_BITMAP_MODE;
		break;
	    }

	/* Given the interpreter state, update graphics activities. */
	switch (gln_graphics_interpreter_state)
	    {
	    case GLN_GRAPHICS_OFF:

		/* If currently displaying graphics, stop and close window. */
		if (gln_graphics_enabled
				&& gln_graphics_are_displayed ())
		    {
			gln_graphics_stop ();
			gln_graphics_close ();
		    }
		break;

	    case GLN_GRAPHICS_LINE_MODE:
	    case GLN_GRAPHICS_BITMAP_MODE:

		/* Create a new graphics context on switch to line mode. */
		if (gln_graphics_interpreter_state == GLN_GRAPHICS_LINE_MODE)
			gln_linegraphics_create_context ();

		/*
		 * If we have a picture loaded already, restart graphics.
		 * If not, we'll delay this until one is supplied by a call
		 * to os_show_bitmap().
		 */
		if (gln_graphics_enabled
				&& gln_graphics_bitmap != NULL)
		    {
			if (gln_graphics_open ())
				gln_graphics_restart ();
		    }
		break;
	    }

	/* Note the current mode so changes can be detected. */
	gln_graphics_current_mode = mode;
}


/*
 * gln_arbitrate_request_timer_events()
 *
 * Shim function for glk_request_timer_events(), this function should be
 * called by other functional areas in place of the main timer event setting
 * function.  It suspends graphics if busy when setting timer events, and
 * resumes graphics if necessary when clearing timer events.
 *
 * On resuming, it calls the graphics timeout function to simulate the
 * timeout that has (probably) been missed.  This also ensures that tight
 * loops that enable then disable timers using this function don't lock out
 * the graphics completely.
 *
 * Use only in paired calls, the first non-zero, the second zero, and use
 * no graphics functions between calls.
 */
static void
gln_arbitrate_request_timer_events (glui32 millisecs)
{
	/* See if we are setting or clearing timer events. */
	if (millisecs > 0)
	    {
		/* Suspend graphics if currently active. */
		if (gln_graphics_active)
		    {
			gln_graphics_suspended = TRUE;
			gln_graphics_stop ();
		    }

		/* Set timer events as requested. */
		glk_request_timer_events (millisecs);
	    }
	else
	    {
		/*
		 * Resume graphics if currently suspended, otherwise cancel
		 * timer events as requested by the caller.
		 */
		if (gln_graphics_suspended)
		    {
			gln_graphics_suspended = FALSE;
			gln_graphics_start ();

			/* Simulate the "missed" graphics timeout. */
			gln_graphics_timeout ();
		    }
		else
			glk_request_timer_events (0);
	    }
}


/*---------------------------------------------------------------------*/
/*  Glk port infinite loop detection functions                         */
/*---------------------------------------------------------------------*/

/* Short timeout to wait purely in order to get the display updated. */
static const glui32	GLN_WATCHDOG_FIXUP		= 50;

/*
 * Timestamp of the last watchdog tick call, and timeout.  This is used to
 * monitor the elapsed time since the interpreter made an I/O call.  If it
 * remains silent for long enough, set by the timeout, we'll offer the
 * option to end the game.  A timeout of zero disables the watchdog.
 */
static time_t		gln_watchdog_monitor		= 0;
static int		gln_watchdog_timeout_secs	= 0;

/*
 * To save thrashing in time(), we want to check for timeouts less frequently
 * than we're polled.  Here's the control for that.
 */
static int		gln_watchdog_check_period	= 0;
static int		gln_watchdog_check_counter	= 0;


/*
 * gln_watchdog_start()
 * gln_watchdog_stop()
 *
 * Start and stop watchdog monitoring.
 */
static void
gln_watchdog_start (int timeout, int period)
{
	assert (timeout > 0 && period > 0);

	gln_watchdog_timeout_secs	= timeout;
	gln_watchdog_check_period	= period;
	gln_watchdog_check_counter	= period;
	gln_watchdog_monitor		= time (NULL);
}

static void
gln_watchdog_stop (void)
{
	gln_watchdog_timeout_secs	= 0;
}


/*
 * gln_watchdog_tick()
 *
 * Set the watchdog timestamp to the current system time.
 *
 * This function should be called just before almost every os_* function
 * returns to the interpreter, as a means of timing how long the interp-
 * reter dwells in running game code.
 */
static void
gln_watchdog_tick (void)
{
	gln_watchdog_monitor = time (NULL);
}


/*
 * gln_watchdog_has_timed_out()
 *
 * Check to see if too much time has elapsed since the last tick.  If it
 * has, offer the option to stop the game, and if accepted, return TRUE.
 * Otherwise, if no timeout, or if the watchdog is disabled, return FALSE.
 *
 * This function only checks every N calls; it's called extremely frequently
 * from opcode handling, and will thrash in time() if it checks on each call.
 */
static int
gln_watchdog_has_timed_out (void)
{
	double		delta_time;		/* Time difference. */

	/*
	 * If loop detection is off, or if the timeout is set to zero,
	 * do nothing.
	 */
	if (!gln_loopcheck_enabled
			|| gln_watchdog_timeout_secs == 0)
		return FALSE;

	/*
	 * Wait until we've seen enough calls to make a timeout check.
	 * If we haven't, return, otherwise reset the counter and continue.
	 */
	if (--gln_watchdog_check_counter > 0)
		return FALSE;
	else
		gln_watchdog_check_counter = gln_watchdog_check_period;

	/* If too much time has passed, offer to end the game. */
	delta_time = difftime (time (NULL), gln_watchdog_monitor);
	if (delta_time >= (double) gln_watchdog_timeout_secs)
	    {
		/* It looks like the game is in an endless loop. */
		if (gln_confirm ("\nThe game may be in an infinite loop."
				"  Do you want to stop it? [Y or N] "))
		    {
			/* Return TRUE -- timed out, and stop requested. */
			gln_watchdog_tick ();
			return TRUE;
		    }

		/*
		 * If we have timers, set a really short timeout and let it
		 * expire.  This is to force a display update with the
		 * response of the confirm -- without this, we may not get
		 * a screen update for a while since at this point the
		 * game isn't, by definition, doing any input or output.
		 * If we don't have timers, no biggie.
		 */
		if (gln_timeouts_possible)
		    {
			event_t		event;	/* Glk event buffer. */

			gln_arbitrate_request_timer_events (GLN_WATCHDOG_FIXUP);
			gln_event_wait (evtype_Timer, &event);
			gln_arbitrate_request_timer_events (0);
		    }

		/* Reset the monitor and return FALSE -- stop rejected. */
		gln_watchdog_tick ();
		return FALSE;
	    }

	/* No timeout yet. */
	return FALSE;
}


/*---------------------------------------------------------------------*/
/*  Glk port status line functions                                     */
/*---------------------------------------------------------------------*/

/* Default width used for non-windowing Glk status lines. */
static const int	GLN_DEFAULT_STATUS_WIDTH	= 74;


/*
 * gln_status_update()
 *
 * Update the information in the status window with the current contents of
 * the current game identity string, or a default string if no game identity
 * could be established.
 */
static void
gln_status_update (void)
{
	glui32		width, height;		/* Status window dimensions. */
	strid_t		status_stream;		/* Status window stream. */
	const char	*game_name;		/* Game identity. */
	assert (gln_status_window != NULL);

	/* Measure the status window, and do nothing if height is zero. */
	glk_window_get_size (gln_status_window, &width, &height);
	if (height == 0)
		return;

	/* Clear the current status window contents, and position cursor. */
	glk_window_clear (gln_status_window);
	glk_window_move_cursor (gln_status_window, 0, 0);
	status_stream = glk_window_get_stream (gln_status_window);

	/*
	 * Try to establish a game identity to display; if none, use a
	 * standard message instead.
	 */
	game_name = gln_gameid_get_game_name ();
	if (game_name == NULL)
		game_name = "Glk Level9 version 4.0";

	/* Output the game identity. */
	glk_put_string_stream (status_stream, (char *) game_name);
}


/*
 * gln_status_print()
 *
 * Print the current contents of the game identity out in the main window,
 * if it has changed since the last call.  This is for non-windowing Glk
 * libraries.
 *
 * To save memory management hassles, this function uses the CRC functions
 * to detect changes of game identity string, and gambles a little on the
 * belief that two games' strings won't have the same CRC.
 */
static void
gln_status_print (void)
{
	static int	initialized	= FALSE;/* First time call flag. */
	static L9UINT16	crc		= 0;	/* Retained CRC. */

	const char	*game_name;		/* Game identity. */
	L9UINT16	new_crc;		/* New game id string CRC. */
	int		index;			/* Padding index. */

	/*
	 * Get the current game name, and do nothing if none available or
	 * if not the first call and the game identity string has not changed.
	 */
	game_name = gln_gameid_get_game_name ();
	if (game_name == NULL)
		return;
	new_crc = gln_buffer_crc (game_name, strlen (game_name));
	if (initialized
			&& new_crc == crc)
		return;

	/* Set fixed width font to try to preserve status line formatting. */
	glk_set_style (style_Preformatted);

	/* Bracket, and output the extracted game name. */
	glk_put_string ("[ ");
	glk_put_string ((char *) game_name);

	/* Pad to default status width. */
	for (index = strlen (game_name);
			index <= GLN_DEFAULT_STATUS_WIDTH; index++)
		glk_put_char (' ');
	glk_put_string (" ]\n");

	/* Save the CRC for comparisons on the next call. */
	crc = new_crc;
	initialized = TRUE;
}


/*
 * gln_status_notify()
 *
 * Front end function for updating status.  Either updates the status window
 * or prints the status line to the main window.
 */
static void
gln_status_notify (void)
{
	/*
	 * If there is a status window, update it; if not, print the status
	 * line to the main window.
	 */
	if (gln_status_window != NULL)
		gln_status_update ();
	else
		gln_status_print ();
}


/*
 * gln_status_redraw()
 *
 * Redraw the contents of any status window with the buffered status string.
 * This function should be called on the appropriate Glk window resize and
 * arrange events.
 */
static void
gln_status_redraw (void)
{
	/* If there is a status window, update it. */
	if (gln_status_window != NULL)
	    {
		winid_t		parent;		/* Status window parent. */

		/*
		 * Rearrange the status window, without changing its actual
		 * arrangement in any way.  This is a hack to work round
		 * incorrect window repainting in Xglk; it forces a complete
		 * repaint of affected windows on Glk window resize and
		 * arrange events, and works in part because Xglk doesn't
		 * check for actual arrangement changes in any way before
		 * invalidating its windows.  The hack should be harmless to
		 * Glk libraries other than Xglk, moreover, we're careful to
		 * activate it only on resize and arrange events.
		 */
		parent = glk_window_get_parent (gln_status_window);
		glk_window_set_arrangement (parent,
					winmethod_Above|winmethod_Fixed,
		 			1, NULL);

		/* ...now update the status window. */
		gln_status_update ();
	    }
}


/*---------------------------------------------------------------------*/
/*  Glk port output functions                                          */
/*---------------------------------------------------------------------*/

/*
 * Output buffer.  We receive characters one at a time, and it's a bit
 * more efficient for everyone if we buffer them, and output a complete
 * string on a flush call.
 */
static const int	GLN_BUFFER_INCREMENT		= 1024;
static char		*gln_output_buffer		= NULL;
static int		gln_output_size			= 0;
static int		gln_output_length		= 0;

/*
 * Output activity flag.  Set when os_printchar() is called, and queried
 * periodically by os_readchar().  Helps os_readchar() judge whether it must
 * request input, or when it's being used as a crude scroll control.
 */
static int		gln_output_activity		= FALSE;

/*
 * Flag to indicate if the last buffer flushed looked like it ended in a
 * "> " prompt.  Some later games switch to this mode after a while, and
 * it's nice not to duplicate this prompt with our own.
 */
static int		gln_output_prompt		= FALSE;


/*
 * gln_output_notify()
 *
 * Register recent text output from the interpreter.  This function is
 * called by os_printchar().
 */
static void
gln_output_notify (void)
{
	/* Set the output activity flag. */
	gln_output_activity = TRUE;
}


/*
 * gln_recent_output()
 *
 * Return TRUE if the interpreter has recently output text, FALSE otherwise.
 * Clears the flag, so that more output text is required before the next
 * call returns TRUE.
 */
static int
gln_recent_output (void)
{
	int		result;				/* Return value. */

	/* Save the current flag value, and reset the main flag. */
	result = gln_output_activity;
	gln_output_activity = FALSE;

	/* Return the old value. */
	return result;
}


/*
 * gln_game_prompted()
 *
 * Return TRUE if the last game output appears to have been a "> " prompt.
 * Once called, the flag is reset to FALSE, and requires more game output
 * to set it again.
 */
static int
gln_game_prompted (void)
{
	int		result;				/* Return value. */

	/* Save the current flag value, and reset the main flag. */
	result = gln_output_prompt;
	gln_output_prompt = FALSE;

	/* Return the old value. */
	return result;
}


/*
 * gln_detect_game_prompt()
 *
 * See if the last non-newline-terminated line in the output buffer seems
 * to be a prompt, and set the game prompted flag if it does, otherwise
 * clear it.
 */
static void
gln_detect_game_prompt (void)
{
	int		index;			/* Output buffer iterator. */

	/* Begin with a clear prompt flag. */
	gln_output_prompt = FALSE;

	/* Search across any last unterminated buffered line. */
	for (index = gln_output_length - 1;
		index >= 0 && gln_output_buffer[ index ] != '\n';
		index--)
	    {
		/* Looks like a prompt if a non-space character found. */
		if (gln_output_buffer[ index ] != ' ')
		    {
			gln_output_prompt = TRUE;
			break;
		    }
	    }
}


/*
 * gln_output_delete()
 *
 * Delete all buffered output text.  Free all malloc'ed buffer memory, and
 * return the buffer variables to their initial values.
 */
static void
gln_output_delete (void)
{
	/* Clear and free the buffer of current contents. */
	if (gln_output_buffer != NULL)
		free (gln_output_buffer);
	gln_output_buffer = NULL;
	gln_output_size   = 0;
	gln_output_length = 0;
}


/*
 * gln_output_flush()
 *
 * Flush any buffered output text to the Glk main window, and clear the
 * buffer.  Check in passing for game prompts that duplicate our's.
 */
static void
gln_output_flush (void)
{
	assert (glk_stream_get_current () != NULL);

	/* Do nothing if the buffer is currently empty. */
	if (gln_output_length > 0)
	    {
		/* See if the game issued a standard prompt. */
		gln_detect_game_prompt ();

		/*
		 * Print the buffer to the stream for the main window, in
		 * game output style.
		 */
		glk_set_style (style_Normal);
		glk_put_buffer (gln_output_buffer, gln_output_length);

		/* Clear and free the buffer of current contents. */
		gln_output_delete ();
	    }
}


/*
 * os_printchar()
 *
 * Buffer a character for eventual printing to the main window.
 */
void
os_printchar (char c)
{
	assert (gln_output_length <= gln_output_size);

	/* Note that the game created some output. */
	gln_output_notify ();

	/* Grow the output buffer if necessary. */
	if (gln_output_length == gln_output_size)
	    {
		gln_output_size += GLN_BUFFER_INCREMENT;
		gln_output_buffer = gln_realloc (gln_output_buffer,
							gln_output_size);
	    }

	/* Handle return as a newline. */
	if (c == '\r')
	    {
		gln_output_buffer[ gln_output_length++ ] = '\n';
		return;
	    }

	/* Add the character to the buffer. */
	gln_output_buffer[ gln_output_length++ ] = c;
}


/*
 * gln_standout_string()
 * gln_standout_char()
 *
 * Print a string in a style that stands out from game text.
 */
static void
gln_standout_string (const char *message)
{
	assert (message != NULL);

	/*
	 * Print the message, in a style that hints that it's from the
	 * interpreter, not the game.
	 */
	glk_set_style (style_Emphasized);
	glk_put_string ((char *) message);
	glk_set_style (style_Normal);
}

static void
gln_standout_char (char c)
{
	/* Print the character, in a message style. */
	glk_set_style (style_Emphasized);
	glk_put_char (c);
	glk_set_style (style_Normal);
}


/*
 * gln_normal_string()
 * gln_normal_char()
 *
 * Print a string in normal text style.
 */
static void
gln_normal_string (const char *message)
{
	assert (message != NULL);

	/* Print the message, in normal text style. */
	glk_set_style (style_Normal);
	glk_put_string ((char *) message);
}

static void
gln_normal_char (char c)
{
	/* Print the character, in normal text style. */
	glk_set_style (style_Normal);
	glk_put_char (c);
}


/*
 * gln_header_string()
 * gln_banner_string()
 *
 * Print text messages for the banner at the start of a game run.
 */
static void
gln_header_string (const char *banner)
{
	assert (banner != NULL);

	/* Print the string in a header style. */
	glk_set_style (style_Header);
	glk_put_string ((char *) banner);
	glk_set_style (style_Normal);
}

static void
gln_banner_string (const char *banner)
{
	assert (banner != NULL);

	/* Print the banner in a subheader style. */
	glk_set_style (style_Subheader);
	glk_put_string ((char *) banner);
	glk_set_style (style_Normal);
}


/*
 * os_flush()
 *
 * Handle a core interpreter call to flush the output buffer.  Because
 * Glk only flushes its buffers and displays text on glk_select(), we
 * can ignore these calls as long as we call glk_output_flush() when
 * reading line or character input.
 */
void os_flush (void)
{
}


/*---------------------------------------------------------------------*/
/*  Glk command escape functions                                       */
/*---------------------------------------------------------------------*/

/* Valid command control values. */
static const char	*GLN_COMMAND_ON			= "on";
static const char	*GLN_COMMAND_OFF		= "off";


/*
 * gln_command_script()
 *
 * Turn game output scripting (logging) on and off.
 */
static void
gln_command_script (const char *argument)
{
	assert (argument != NULL);

	/* Set up a transcript according to the argument given. */
	if (!gln_strcasecmp (argument, GLN_COMMAND_ON))
	    {
		frefid_t	fileref;	/* Glk file reference. */

		/* See if a transcript is already active. */
		if (gln_transcript_stream != NULL)
		    {
			gln_normal_string ("Glk transcript is already ");
			gln_normal_string (GLN_COMMAND_ON);
			gln_normal_string (".\n");
			return;
		    }

		/* Get a Glk file reference for a transcript. */
		fileref = glk_fileref_create_by_prompt
				(fileusage_Transcript | fileusage_TextMode,
					filemode_WriteAppend, 0);
		if (fileref == NULL)
		    {
			gln_standout_string ("Glk transcript failed.\n");
			return;
		    }

		/* Open a Glk stream for the fileref. */
		gln_transcript_stream = glk_stream_open_file
					(fileref, filemode_WriteAppend, 0);
		if (gln_transcript_stream == NULL)
		    {
			glk_fileref_destroy (fileref);
			gln_standout_string ("Glk transcript failed.\n");
			return;
		    }
		glk_fileref_destroy (fileref);

		/* Set the new transcript stream as the main echo stream. */
		glk_window_set_echo_stream (gln_main_window,
						gln_transcript_stream);

		/* Confirm action. */
		gln_normal_string ("Glk transcript is now ");
		gln_normal_string (GLN_COMMAND_ON);
		gln_normal_string (".\n");
	    }
	else if (!gln_strcasecmp (argument, GLN_COMMAND_OFF))
	    {
		/* If a transcript is active, close it. */
		if (gln_transcript_stream != NULL)
		    {
			glk_stream_close (gln_transcript_stream, NULL);
			gln_transcript_stream = NULL;

			/* Clear the main echo stream. */
			glk_window_set_echo_stream (gln_main_window, NULL);

			/* Confirm action. */
			gln_normal_string ("Glk transcript is now ");
			gln_normal_string (GLN_COMMAND_OFF);
			gln_normal_string (".\n");
		    }
		else
		    {
			/* Note that scripts are already disabled. */
			gln_normal_string ("Glk transcript is already ");
			gln_normal_string (GLN_COMMAND_OFF);
			gln_normal_string (".\n");
		    }
	    }
	else if (strlen (argument) == 0)
	    {
		/*
		 * There was no argument on the line, so print out the current
		 * transcript mode.
		 */
		gln_normal_string ("Glk transcript is ");
		if (gln_transcript_stream != NULL)
			gln_normal_string (GLN_COMMAND_ON);
		else
			gln_normal_string (GLN_COMMAND_OFF);
		gln_normal_string (".\n");
	    }
	else
	    {
		/*
		 * The command argument isn't a valid one, so print a list of
		 * valid command arguments.
		 */
		gln_normal_string ("Glk transcript can be ");
		gln_standout_string (GLN_COMMAND_ON);
		gln_normal_string (", or ");
		gln_standout_string (GLN_COMMAND_OFF);
		gln_normal_string (".\n");
	    }
}



/*
 * gln_command_inputlog()
 *
 * Turn game input logging on and off.
 */
static void
gln_command_inputlog (const char *argument)
{
	assert (argument != NULL);

	/* Set up an input log according to the argument given. */
	if (!gln_strcasecmp (argument, GLN_COMMAND_ON))
	    {
		frefid_t	fileref;	/* Glk file reference. */

		/* See if an input log is already active. */
		if (gln_inputlog_stream != NULL)
		    {
			gln_normal_string ("Glk input logging is already ");
			gln_normal_string (GLN_COMMAND_ON);
			gln_normal_string (".\n");
			return;
		    }

		/* Get a Glk file reference for an input log. */
		fileref = glk_fileref_create_by_prompt
				(fileusage_InputRecord | fileusage_BinaryMode,
					filemode_WriteAppend, 0);
		if (fileref == NULL)
		    {
			gln_standout_string ("Glk input logging failed.\n");
			return;
		    }

		/* Open a Glk stream for the fileref. */
		gln_inputlog_stream = glk_stream_open_file
					(fileref, filemode_WriteAppend, 0);
		if (gln_inputlog_stream == NULL)
		    {
			glk_fileref_destroy (fileref);
			gln_standout_string ("Glk input logging failed.\n");
			return;
		    }
		glk_fileref_destroy (fileref);

		/* Confirm action. */
		gln_normal_string ("Glk input logging is now ");
		gln_normal_string (GLN_COMMAND_ON);
		gln_normal_string (".\n");
	    }
	else if (!gln_strcasecmp (argument, GLN_COMMAND_OFF))
	    {
		/* If an input log is active, close it. */
		if (gln_inputlog_stream != NULL)
		    {
			glk_stream_close (gln_inputlog_stream, NULL);
			gln_inputlog_stream = NULL;

			/* Confirm action. */
			gln_normal_string ("Glk input log is now ");
			gln_normal_string (GLN_COMMAND_OFF);
			gln_normal_string (".\n");
		    }
		else
		    {
			/* Note that there is no current input log. */
			gln_normal_string ("Glk input logging is already ");
			gln_normal_string (GLN_COMMAND_OFF);
			gln_normal_string (".\n");
		    }
	    }
	else if (strlen (argument) == 0)
	    {
		/*
		 * There was no argument on the line, so print out the current
		 * input logging mode.
		 */
		gln_normal_string ("Glk input logging is ");
		if (gln_inputlog_stream != NULL)
			gln_normal_string (GLN_COMMAND_ON);
		else
			gln_normal_string (GLN_COMMAND_OFF);
		gln_normal_string (".\n");
	    }
	else
	    {
		/*
		 * The command argument isn't a valid one, so print a list of
		 * valid command arguments.
		 */
		gln_normal_string ("Glk input logging can be ");
		gln_standout_string (GLN_COMMAND_ON);
		gln_normal_string (", or ");
		gln_standout_string (GLN_COMMAND_OFF);
		gln_normal_string (".\n");
	    }
}


/*
 * gln_command_readlog()
 *
 * Set the game input log, to read input from a file.
 */
static void
gln_command_readlog (const char *argument)
{
	assert (argument != NULL);

	/* Set up a read log according to the argument given. */
	if (!gln_strcasecmp (argument, GLN_COMMAND_ON))
	    {
		frefid_t	fileref;	/* Glk file reference. */

		/* See if a read log is already active. */
		if (gln_readlog_stream != NULL)
		    {
			gln_normal_string ("Glk read log is already ");
			gln_normal_string (GLN_COMMAND_ON);
			gln_normal_string (".\n");
			return;
		    }

		/* Get a Glk file reference for a read log. */
		fileref = glk_fileref_create_by_prompt
				(fileusage_InputRecord | fileusage_BinaryMode,
					filemode_Read, 0);
		if (fileref == NULL)
		    {
			gln_standout_string ("Glk read log failed.\n");
			return;
		    }

		/*
		 * Reject the file reference if we're expecting to read
		 * from it, and the referenced file doesn't exist.
		 */
		if (!glk_fileref_does_file_exist (fileref))
		    {
			glk_fileref_destroy (fileref);
			gln_standout_string ("Glk read log failed.\n");
			return;
		    }

		/* Open a Glk stream for the fileref. */
		gln_readlog_stream = glk_stream_open_file
					(fileref, filemode_Read, 0);
		if (gln_readlog_stream == NULL)
		    {
			glk_fileref_destroy (fileref);
			gln_standout_string ("Glk read log failed.\n");
			return;
		    }
		glk_fileref_destroy (fileref);

		/* Confirm action. */
		gln_normal_string ("Glk read log is now ");
		gln_normal_string (GLN_COMMAND_ON);
		gln_normal_string (".\n");
	    }
	else if (!gln_strcasecmp (argument, GLN_COMMAND_OFF))
	    {
		/* If a read log is active, close it. */
		if (gln_readlog_stream != NULL)
		    {
			glk_stream_close (gln_readlog_stream, NULL);
			gln_readlog_stream = NULL;

			/* Confirm action. */
			gln_normal_string ("Glk read log is now ");
			gln_normal_string (GLN_COMMAND_OFF);
			gln_normal_string (".\n");
		    }
		else
		    {
			/* Note that there is no current read log. */
			gln_normal_string ("Glk read log is already ");
			gln_normal_string (GLN_COMMAND_OFF);
			gln_normal_string (".\n");
		    }
	    }
	else if (strlen (argument) == 0)
	    {
		/*
		 * There was no argument on the line, so print out the current
		 * read logging mode.
		 */
		gln_normal_string ("Glk read log is ");
		if (gln_readlog_stream != NULL)
			gln_normal_string (GLN_COMMAND_ON);
		else
			gln_normal_string (GLN_COMMAND_OFF);
		gln_normal_string (".\n");
	    }
	else
	    {
		/*
		 * The command argument isn't a valid one, so print a list of
		 * valid command arguments.
		 */
		gln_normal_string ("Glk read log can be ");
		gln_standout_string (GLN_COMMAND_ON);
		gln_normal_string (", or ");
		gln_standout_string (GLN_COMMAND_OFF);
		gln_normal_string (".\n");
	    }
}


/*
 * gln_command_abbreviations()
 *
 * Turn abbreviation expansions on and off.
 */
static void
gln_command_abbreviations (const char *argument)
{
	assert (argument != NULL);

	/* Set up abbreviation expansions according to the argument given. */
	if (!gln_strcasecmp (argument, GLN_COMMAND_ON))
	    {
		/* See if expansions are already on. */
		if (gln_abbreviations_enabled)
		    {
			gln_normal_string ("Glk abbreviation expansions"
					" are already ");
			gln_normal_string (GLN_COMMAND_ON);
			gln_normal_string (".\n");
			return;
		    }

		/* The user has turned expansions on. */
		gln_abbreviations_enabled = TRUE;
		gln_normal_string ("Glk abbreviation expansions are now ");
		gln_normal_string (GLN_COMMAND_ON);
		gln_normal_string (".\n");
	    }
	else if (!gln_strcasecmp (argument, GLN_COMMAND_OFF))
	    {
		/* See if expansions are already off. */
		if (!gln_abbreviations_enabled)
		    {
			gln_normal_string ("Glk abbreviation expansions"
					" are already ");
			gln_normal_string (GLN_COMMAND_OFF);
			gln_normal_string (".\n");
			return;
		    }

		/* The user has turned expansions off. */
		gln_abbreviations_enabled = FALSE;
		gln_normal_string ("Glk abbreviation expansions are now ");
		gln_normal_string (GLN_COMMAND_OFF);
		gln_normal_string (".\n");
	    }
	else if (strlen (argument) == 0)
	    {
		/*
		 * There was no argument on the line, so print out the current
		 * expansion mode.
		 */
		gln_normal_string ("Glk abbreviation expansions are ");
		if (gln_abbreviations_enabled)
			gln_normal_string (GLN_COMMAND_ON);
		else
			gln_normal_string (GLN_COMMAND_OFF);
		gln_normal_string (".\n");
	    }
	else
	    {
		/*
		 * The command argument isn't a valid one, so print a list of
		 * valid command arguments.
		 */
		gln_normal_string ("Glk abbreviation expansions can be ");
		gln_standout_string (GLN_COMMAND_ON);
		gln_normal_string (", or ");
		gln_standout_string (GLN_COMMAND_OFF);
		gln_normal_string (".\n");
	    }
}


/*
 * gln_command_print_version_number()
 * gln_command_version()
 *
 * Print out the Glk library version number.
 */
static void
gln_command_print_version_number (glui32 version)
{
	char		buffer[64];		/* Output buffer string. */

	/* Print out the three version number component parts. */
	sprintf (buffer, "%lu.%lu.%lu",
			(version & 0xFFFF0000) >> 16,
			(version & 0x0000FF00) >>  8,
			(version & 0x000000FF)      );
	gln_normal_string (buffer);
}
static void
gln_command_version (const char *argument)
{
	glui32		version;		/* Glk lib version number. */

	/* Get the Glk library version number. */
	version = glk_gestalt (gestalt_Version, 0);

	/* Print the Glk library and port version numbers. */
	gln_normal_string ("The Glk library version is ");
	gln_command_print_version_number (version);
	gln_normal_string (".\n");
	gln_normal_string ("This is version ");
	gln_command_print_version_number (GLN_PORT_VERSION);
	gln_normal_string (" of the Glk Level9 port.\n");
}


/*
 * gln_command_loopchecks()
 *
 * Turn loop checking (for game infinite loops) on and off.
 */
static void
gln_command_loopchecks (const char *argument)
{
	assert (argument != NULL);

	/* Set up loopcheck according to the argument given. */
	if (!gln_strcasecmp (argument, GLN_COMMAND_ON))
	    {
		/* See if loop checks are already on. */
		if (gln_loopcheck_enabled)
		    {
			gln_normal_string ("Glk loop detection"
					" is already ");
			gln_normal_string (GLN_COMMAND_ON);
			gln_normal_string (".\n");
			return;
		    }

		/* The user has turned loop checking on. */
		gln_loopcheck_enabled = TRUE;
		gln_normal_string ("Glk loop detection is now ");
		gln_normal_string (GLN_COMMAND_ON);
		gln_normal_string (".\n");
	    }
	else if (!gln_strcasecmp (argument, GLN_COMMAND_OFF))
	    {
		/* See if loop checks are already off. */
		if (!gln_loopcheck_enabled)
		    {
			gln_normal_string ("Glk loop detection"
					" is already ");
			gln_normal_string (GLN_COMMAND_OFF);
			gln_normal_string (".\n");
			return;
		    }

		/* The user has turned loop checks off. */
		gln_loopcheck_enabled = FALSE;
		gln_normal_string ("Glk loop detection is now ");
		gln_normal_string (GLN_COMMAND_OFF);
		gln_normal_string (".\n");
	    }
	else if (strlen (argument) == 0)
	    {
		/*
		 * There was no argument on the line, so print out the current
		 * loop check mode.
		 */
		gln_normal_string ("Glk loop detection is ");
		if (gln_loopcheck_enabled)
			gln_normal_string (GLN_COMMAND_ON);
		else
			gln_normal_string (GLN_COMMAND_OFF);
		gln_normal_string (".\n");
	    }
	else
	    {
		/*
		 * The command argument isn't a valid one, so print a list of
		 * valid command arguments.
		 */
		gln_normal_string ("Glk loop detection can be ");
		gln_standout_string (GLN_COMMAND_ON);
		gln_normal_string (", or ");
		gln_standout_string (GLN_COMMAND_OFF);
		gln_normal_string (".\n");
	    }
}


/*
 * gln_command_locals()
 *
 * Turn local interpretation of "quit" etc. on and off.
 */
static void
gln_command_locals (const char *argument)
{
	assert (argument != NULL);

	/* Set up local commands according to the argument given. */
	if (!gln_strcasecmp (argument, GLN_COMMAND_ON))
	    {
		/* See if local commands are already on. */
		if (gln_intercept_enabled)
		    {
			gln_normal_string ("Glk local commands"
					" are already ");
			gln_normal_string (GLN_COMMAND_ON);
			gln_normal_string (".\n");
			return;
		    }

		/* The user has turned local commands on. */
		gln_intercept_enabled = TRUE;
		gln_normal_string ("Glk local commands are now ");
		gln_normal_string (GLN_COMMAND_ON);
		gln_normal_string (".\n");
	    }
	else if (!gln_strcasecmp (argument, GLN_COMMAND_OFF))
	    {
		/* See if local commands are already off. */
		if (!gln_intercept_enabled)
		    {
			gln_normal_string ("Glk local commands"
					" are already ");
			gln_normal_string (GLN_COMMAND_OFF);
			gln_normal_string (".\n");
			return;
		    }

		/* The user has turned local commands off. */
		gln_intercept_enabled = FALSE;
		gln_normal_string ("Glk local commands are now ");
		gln_normal_string (GLN_COMMAND_OFF);
		gln_normal_string (".\n");
	    }
	else if (strlen (argument) == 0)
	    {
		/*
		 * There was no argument on the line, so print out the current
		 * local command mode.
		 */
		gln_normal_string ("Glk local commands are ");
		if (gln_intercept_enabled)
			gln_normal_string (GLN_COMMAND_ON);
		else
			gln_normal_string (GLN_COMMAND_OFF);
		gln_normal_string (".\n");
	    }
	else
	    {
		/*
		 * The command argument isn't a valid one, so print a list of
		 * valid command arguments.
		 */
		gln_normal_string ("Glk local commands can be ");
		gln_standout_string (GLN_COMMAND_ON);
		gln_normal_string (", or ");
		gln_standout_string (GLN_COMMAND_OFF);
		gln_normal_string (".\n");
	    }
}


/*
 * gln_command_graphics()
 *
 * Enable or disable graphics more permanently than is done by the
 * main interpreter.  Also, print out a few brief details about the
 * graphics state of the program.
 */
static void
gln_command_graphics (const char *argument)
{
	assert (argument != NULL);

	/* If graphics is not possible, simply say so and return. */
	if (!gln_graphics_possible)
	    {
		gln_normal_string ("Glk graphics are not available.\n");
		return;
	    }

	/* Set up command handling according to the argument given. */
	if (!gln_strcasecmp (argument, GLN_COMMAND_ON))
	    {
		/* See if graphics are already on. */
		if (gln_graphics_enabled)
		    {
			gln_normal_string ("Glk graphics are already ");
			gln_normal_string (GLN_COMMAND_ON);
			gln_normal_string (".\n");
			return;
		    }

		/* Set the graphics enabled flag to on. */
		gln_graphics_enabled = TRUE;

		/*
		 * If there is a picture loaded and ready, call the
		 * restart function to repaint it.
		 */
		if (gln_graphics_picture_is_available ())
		    {
			if (!gln_graphics_open ())
			    {
				gln_normal_string ("Glk graphics error.\n");
				return;
			    }
			gln_graphics_restart ();
		    }

		/* Confirm actions. */
		gln_normal_string ("Glk graphics are now ");
		gln_normal_string (GLN_COMMAND_ON);
		gln_normal_string (".\n");
	    }
	else if (!gln_strcasecmp (argument, GLN_COMMAND_OFF))
	    {
		/* See if graphics are already off. */
		if (!gln_graphics_enabled)
		    {
			gln_normal_string ("Glk graphics are already ");
			gln_normal_string (GLN_COMMAND_OFF);
			gln_normal_string (".\n");
			return;
		    }

		/*
		 * Set graphics to disabled, and stop any graphics
		 * processing.  Close the graphics window.
		 */
		gln_graphics_enabled = FALSE;
		gln_graphics_stop ();
		gln_graphics_close ();

		/* Confirm actions. */
		gln_normal_string ("Glk graphics are now ");
		gln_normal_string (GLN_COMMAND_OFF);
		gln_normal_string (".\n");
	    }
	else if (strlen (argument) == 0)
	    {
		/* Print details about graphics availability. */
		gln_normal_string ("Glk graphics are available,");
		if (gln_graphics_enabled)
			gln_normal_string (" and enabled.\n");
		else
			gln_normal_string (" but disabled.\n");

		/* See if there is any current loaded picture. */
		if (gln_graphics_picture_is_available ())
		    {
			int	status;		/* Status of picture query */
			int	width, height;	/* Current picture size */

			/* Print the picture's dimensions and animation. */
			status = gln_graphics_get_picture_details
							(&width, &height);
			if (status)
			    {
				char	buffer[16];

				gln_normal_string ("There is a"
							" picture loaded, ");
				sprintf (buffer, "%d", width);
				gln_normal_string (buffer);
				gln_normal_string (" by ");
				sprintf (buffer, "%d", height);
				gln_normal_string (buffer);
				gln_normal_string (" pixels.\n");
			    }
		    }

		/* Indicate the state of interpreter graphics. */
		if (!gln_graphics_interpreter_enabled ())
			gln_normal_string ("Interpreter graphics are"
						" disabled.\n");

		/* Show if graphics are displayed, or not. */
		if (gln_graphics_enabled
				&& gln_graphics_are_displayed ())
		    {
			int		status;		/* Query status */
			const char	*bitmap_type;	/* Bitmap type */
			int		color_count;	/* Count of colors */
			int		is_active;	/* Graphics busy flag */

			/* Indicate graphics display. */
			status = gln_graphics_get_rendering_details
						(&bitmap_type,
						&color_count, &is_active);
			if (status)
			    {
				char	buffer[16];

				/* Print activity and colors count. */
				gln_normal_string ("Graphics are ");
				if (is_active)
					gln_normal_string ("active, ");
				else
					gln_normal_string ("displayed, ");
				sprintf (buffer, "%d", color_count);
				gln_normal_string (buffer);
				gln_normal_string (" colours");

				/* Print bitmap type if given. */
				if (bitmap_type != NULL)
				    {
					gln_normal_string (", ");
					gln_normal_string (bitmap_type);
					gln_normal_string (" bitmaps");
				    }
				gln_normal_string (".\n");
			    }
			else
				gln_normal_string ("Graphics are"
							" being displayed.\n");
		    }
		if (gln_graphics_enabled
				&& !gln_graphics_are_displayed ())
		    {
			/* Indicate no current graphics display. */
			gln_normal_string ("Graphics are"
						" not being displayed.\n");
		    }
	    }
	else
	    {
		/*
		 * The command argument isn't a valid one, so print a list of
		 * valid command arguments.
		 */
		gln_normal_string ("Glk graphics can be ");
		gln_standout_string (GLN_COMMAND_ON);
		gln_normal_string (", or ");
		gln_standout_string (GLN_COMMAND_OFF);
		gln_normal_string (".\n");
	    }
}


/*
 * gln_command_prompts()
 *
 * Turn the extra "> " prompt output on and off.
 */
static void
gln_command_prompts (const char *argument)
{
	assert (argument != NULL);

	/* Set up prompt according to the argument given. */
	if (!gln_strcasecmp (argument, GLN_COMMAND_ON))
	    {
		/* See if prompt is already on. */
		if (gln_prompt_enabled)
		    {
			gln_normal_string ("Glk extra prompts are already ");
			gln_normal_string (GLN_COMMAND_ON);
			gln_normal_string (".\n");
			return;
		    }

		/* The user has turned prompt on. */
		gln_prompt_enabled = TRUE;
		gln_normal_string ("Glk extra prompts are now ");
		gln_normal_string (GLN_COMMAND_ON);
		gln_normal_string (".\n");

		/* Check for a game prompt to clear the flag. */
		gln_game_prompted ();
	    }
	else if (!gln_strcasecmp (argument, GLN_COMMAND_OFF))
	    {
		/* See if prompt is already off. */
		if (!gln_prompt_enabled)
		    {
			gln_normal_string ("Glk extra prompts are already ");
			gln_normal_string (GLN_COMMAND_OFF);
			gln_normal_string (".\n");
			return;
		    }

		/* The user has turned prompt off. */
		gln_prompt_enabled = FALSE;
		gln_normal_string ("Glk extra prompts are now ");
		gln_normal_string (GLN_COMMAND_OFF);
		gln_normal_string (".\n");
	    }
	else if (strlen (argument) == 0)
	    {
		/*
		 * There was no argument on the line, so print out the current
		 * extra prompt mode.
		 */
		gln_normal_string ("Glk extra prompts are ");
		if (gln_prompt_enabled)
			gln_normal_string (GLN_COMMAND_ON);
		else
			gln_normal_string (GLN_COMMAND_OFF);
		gln_normal_string (".\n");
	    }
	else
	    {
		/*
		 * The command argument isn't a valid one, so print a list of
		 * valid command arguments.
		 */
		gln_normal_string ("Glk extra prompts can be ");
		gln_standout_string (GLN_COMMAND_ON);
		gln_normal_string (", or ");
		gln_standout_string (GLN_COMMAND_OFF);
		gln_normal_string (".\n");
	    }
}


/*
 * gln_command_commands()
 *
 * Turn command escapes off.  Once off, there's no way to turn them back on.
 * Commands must be on already to enter this function.
 */
static void
gln_command_commands (const char *argument)
{
	assert (argument != NULL);

	/* Set up command handling according to the argument given. */
	if (!gln_strcasecmp (argument, GLN_COMMAND_ON))
	    {
		/* Commands must already be on. */
		gln_normal_string ("Glk commands are already ");
		gln_normal_string (GLN_COMMAND_ON);
		gln_normal_string (".\n");
	    }
	else if (!gln_strcasecmp (argument, GLN_COMMAND_OFF))
	    {
		/* The user has turned commands off. */
		gln_commands_enabled = FALSE;
		gln_normal_string ("Glk commands are now ");
		gln_normal_string (GLN_COMMAND_OFF);
		gln_normal_string (".\n");
	    }
	else if (strlen (argument) == 0)
	    {
		/*
		 * There was no argument on the line, so print out the current
		 * command mode.
		 */
		gln_normal_string ("Glk commands are ");
		if (gln_commands_enabled)
			gln_normal_string (GLN_COMMAND_ON);
		else
			gln_normal_string (GLN_COMMAND_OFF);
		gln_normal_string (".\n");
	    }
	else
	    {
		/*
		 * The command argument isn't a valid one, so print a list of
		 * valid command arguments.
		 */
		gln_normal_string ("Glk commands can be ");
		gln_standout_string (GLN_COMMAND_ON);
		gln_normal_string (", or ");
		gln_standout_string (GLN_COMMAND_OFF);
		gln_normal_string (".\n");
	    }
}


/* Escape introducer string, and special intercepted commands. */
static const char	*GLN_COMMAND_ESCAPE		= "glk";
static const char	*GLN_COMMAND_QUIT		= "quit";
static const char	*GLN_COMMAND_RESTART		= "restart";
static const char	*GLN_COMMAND_SAVE		= "save";
static const char	*GLN_COMMAND_RESTORE		= "restore";
static const char	*GLN_COMMAND_LOAD		= "load";

/* Small table of Glk subcommands and handler functions. */
struct gln_command {
	const char	*command;		/* Glk subcommand. */
	void		(*handler) (const char *argument);
						/* Subcommand handler. */
	const int	takes_argument;		/* Argument flag. */
};
typedef const struct gln_command*	gln_commandref_t;
static const struct gln_command		GLN_COMMAND_TABLE[] = {
	{ "script",		gln_command_script,		TRUE  },
	{ "inputlog",		gln_command_inputlog,		TRUE  },
	{ "readlog",		gln_command_readlog,		TRUE  },
	{ "abbreviations",	gln_command_abbreviations,	TRUE  },
	{ "graphics",		gln_command_graphics,		TRUE  },
	{ "loopchecks",		gln_command_loopchecks,		TRUE  },
	{ "locals",		gln_command_locals,		TRUE  },
	{ "prompts",		gln_command_prompts,		TRUE  },
	{ "version",		gln_command_version,		FALSE },
	{ "commands",		gln_command_commands,		TRUE  },
	{ NULL,			NULL,				FALSE }
};

/* List of whitespace command-argument separator characters. */
static const char	*GLN_COMMAND_WHITESPACE		= "\t ";


/*
 * gln_command_dispatch()
 *
 * Given a command string and an argument, this function finds and runs any
 * handler for it.
 *
 * It searches for the first unambiguous command to match the string passed
 * in.  The return is the count of command matches found; zero represents no
 * match (fail), one represents an unambiguous match (success, and handler
 * run), and more than one represents an ambiguous match (fail).
 */
static int
gln_command_dispatch (const char *command, const char *argument)
{
	gln_commandref_t	entry;		/* Table search entry. */
	gln_commandref_t	matched;	/* Matched table entry. */
	int			matches;	/* Count of command matches. */
	assert (command != NULL && argument != NULL);

	/*
	 * Search for the first unambiguous table command string matching
	 * the command passed in.
	 */
	matches = 0;
	matched = NULL;
	for (entry = GLN_COMMAND_TABLE;
				entry->command != NULL; entry++)
	    {
		if (!gln_strncasecmp (command,
					entry->command, strlen (command)))
		    {
			matches++;
			matched = entry;
		    }
	    }

	/* If the match was unambiguous, call the command handler. */
	if (matches == 1)
	    {
		gln_normal_char ('\n');
		(matched->handler) (argument);

		/* Issue a brief warning if an argument was ignored. */
		if (!matched->takes_argument && strlen (argument) > 0)
		    {
			gln_normal_string ("[The ");
			gln_standout_string (matched->command);
			gln_normal_string (" command ignores arguments.]\n");
		    }
	    }

	/* Return the count of matching table entries. */
	return matches;
}


/*
 * gln_command_usage()
 *
 * On an empty, invalid, or ambiguous command, print out a list of valid
 * commands and perhaps some Glk status information.
 */
static void
gln_command_usage (const char *command, int is_ambiguous)
{
	gln_commandref_t	entry;	/* Command table iteration entry. */
	assert (command != NULL);

	/* Print a blank line separator. */
	gln_normal_char ('\n');

	/* If the command isn't empty, indicate ambiguous or invalid. */
	if (strlen (command) > 0)
	    {
		gln_normal_string ("The Glk command ");
		gln_standout_string (command);
		if (is_ambiguous)
			gln_normal_string (" is ambiguous.\n");
		else
			gln_normal_string (" is not valid.\n");
	    }

	/* Print out a list of valid commands. */
	gln_normal_string ("Glk commands are");
	for (entry = GLN_COMMAND_TABLE; entry->command != NULL; entry++)
	    {
		gln_commandref_t	next;	/* Next command table entry. */

		next = entry + 1;
		gln_normal_string (next->command != NULL ? " " : " and ");
		gln_standout_string (entry->command);
		gln_normal_string (next->command != NULL ? "," : ".\n");
	    }

	/* Write a note about abbreviating commands. */
	gln_normal_string ("Glk commands may be abbreviated, as long as");
	gln_normal_string (" the abbreviations are unambiguous.\n");

	/*
	 * If no command was given, call each command handler function with
	 * an empty argument to prompt each to report the current setting.
	 */
	if (strlen (command) == 0)
	    {
		gln_normal_char ('\n');
		for (entry = GLN_COMMAND_TABLE;
					entry->command != NULL; entry++)
			(entry->handler) ("");
	    }
}


/*
 * gln_skip_characters()
 *
 * Helper function for command escapes.  Skips over either whitespace or
 * non-whitespace in string, and returns the revised string pointer.
 */
static char *
gln_skip_characters (char *string, int skip_whitespace)
{
	char	*result;			/* Return string pointer. */
	assert (string != NULL);

	/* Skip over leading characters of the specified type. */
	for (result = string; *result != '\0'; result++)
	    {
		int	is_whitespace;		/* Whitespace flag. */

		/* Break if encountering a character not the required type. */
		is_whitespace =
			(strchr (GLN_COMMAND_WHITESPACE, *result) != NULL);
		if ((skip_whitespace && !is_whitespace)
				|| (!skip_whitespace && is_whitespace))
			break;
	    }

	/* Return the revised pointer. */
	return result;
}


/*
 * gln_command_escape()
 *
 * This function is handed each input line.  If the line contains a specific
 * Glk port command, handle it and return TRUE, otherwise return FALSE.
 */
static int
gln_command_escape (char *string)
{
	char		*temporary;		/* Temporary string pointer */
	char		*string_copy;		/* Destroyable string copy. */
	char		*command;		/* Glk subcommand. */
	char		*argument;		/* Glk subcommand argument. */
	int		matches;		/* Dispatcher matches. */
	assert (string != NULL);

	/*
	 * Return FALSE if the string doesn't begin with the Glk command
	 * escape introducer.
	 */
	temporary = gln_skip_characters (string, TRUE);
	if (gln_strncasecmp (temporary, GLN_COMMAND_ESCAPE,
					strlen (GLN_COMMAND_ESCAPE)))
		return FALSE;

	/* Take a copy of the string, without any leading space. */
	string_copy = gln_malloc (strlen (temporary) + 1);
	strcpy (string_copy, temporary);

	/* Find the subcommand; the word after the introducer. */
	command = gln_skip_characters (string_copy
					+ strlen (GLN_COMMAND_ESCAPE), TRUE);

	/* Skip over command word, be sure it terminates with NUL. */
	temporary = gln_skip_characters (command, FALSE);
	if (*temporary != '\0')
	    {
		*temporary = '\0';
		temporary++;
	    }

	/* Now find any argument data for the command. */
	argument = gln_skip_characters (temporary, TRUE);

	/* Ensure that argument data also terminates with a NUL. */
	temporary = gln_skip_characters (argument, FALSE);
	*temporary = '\0';

	/*
	 * Try to handle the command and argument as a Glk subcommand.  If
	 * it doesn't run unambiguously, print command usage.
	 */
	matches = gln_command_dispatch (command, argument);
	if (matches != 1)
	    {
		if (matches == 0)
			gln_command_usage (command, FALSE);
		else
			gln_command_usage (command, TRUE);
	    }

	/* Done with the copy of the string. */
	free (string_copy);

	/* Return TRUE to indicate string contained a Glk command. */
	return TRUE;
}


/*
 * gln_command_intercept()
 *
 * The Level 9 games handle the commands "quit" and "restart" oddly, and
 * somewhat similarly.  Both prompt "Press SPACE to play again", and
 * then ignore all characters except space.  This makes it especially
 * hard to exit from a game without killing the interpreter process.
 * It also handles "restore" via an odd security mechanism which has no
 * real place here (the base Level9 interpreter sidesteps this with its
 * "#restore" command), and has some bugs in "save".
 *
 * To try to improve these, here we'll catch and special case the input
 * lines "quit", "save", "restore", and "restart".  "Load" is a synonym
 * for "restore".
 *
 * On "quit" or "restart", the function sets the interpreter stop reason
 * code, stops the current game run.  On "save" or "restore" it calls the
 * appropriate internal interpreter function.
 *
 * The return value is TRUE if an intercepted command was found, otherwise
 * FALSE.
 */
static int
gln_command_intercept (char *string)
{
	char	*first, *trailing;	/* Start and end of the first word */
	char	*end;			/* Skipped spaces after first word */
	assert (string != NULL);

	/*
	 * Find the first significant character in string, and the space
	 * or NUL after the first word.
	 */
	first    = gln_skip_characters (string, TRUE);
	trailing = gln_skip_characters (first, FALSE);

	/*
	 * Find the first character, or NUL, following the whitespace
	 * after the first word.
	 */
	end = gln_skip_characters (trailing, TRUE);

	/* Forget it if string isn't a single word only. */
	if (strlen (end) != 0)
		return FALSE;

	/* If this command was "quit", confirm, then call StopGame(). */
	if (!gln_strncasecmp (first, GLN_COMMAND_QUIT,
						strlen (GLN_COMMAND_QUIT))
			&& first + strlen (GLN_COMMAND_QUIT) == trailing)
	    {
		/* Confirm quit just as a Level 9 game does. */
		if (gln_confirm ("\nDo you really want to stop? [Y or N] "))
		    {
			gln_stop_reason = STOP_EXIT;
			StopGame ();
		    }
		return TRUE;
	    }

	/* If this command was "restart", confirm, then call StopGame(). */
	if (!gln_strncasecmp (first, GLN_COMMAND_RESTART,
						strlen (GLN_COMMAND_RESTART))
			&& first + strlen (GLN_COMMAND_RESTART) == trailing)
	    {
		/* Confirm restart somewhat as a Level 9 game does. */
		if (gln_confirm ("\nDo you really want to restart? [Y or N] "))
		    {
			gln_stop_reason = STOP_RESTART;
			StopGame ();
		    }
		return TRUE;
	    }

	/* If this command was "save", simply call save(). */
	if (!gln_strncasecmp (first, GLN_COMMAND_SAVE,
						strlen (GLN_COMMAND_SAVE))
			&& first + strlen (GLN_COMMAND_SAVE) == trailing)
	    {
		/* Print a message and call the level9 internal save. */
		gln_standout_string ("\nSaving using interpreter\n\n");
		save ();
		return TRUE;
	    }

	/* If this command was "restore" or "load", call restore(). */
	if ((!gln_strncasecmp (first, GLN_COMMAND_RESTORE,
						strlen (GLN_COMMAND_RESTORE))
			&& first + strlen (GLN_COMMAND_RESTORE) == trailing)
		|| (!gln_strncasecmp (first, GLN_COMMAND_LOAD,
						strlen (GLN_COMMAND_LOAD))
			&& first + strlen (GLN_COMMAND_LOAD) == trailing))
	    {
		/*
		 * Print a message and call the level9 restore.  There is no
		 * need for confirmation since the file selection can be
		 * canceled.
		 */
		gln_standout_string ("\nRestoring using interpreter\n\n");
		restore ();
		return TRUE;
	    }

	/* No special buffer contents found. */
	return FALSE;
}


/*---------------------------------------------------------------------*/
/*  Glk port input functions                                           */
/*---------------------------------------------------------------------*/

/* Ctrl-C and Ctrl-U character constants. */
static const char	GLN_CONTROL_C			= '\003';
static const char	GLN_CONTROL_U			= '\025';

/*
 * os_readchar() call count limit, after which we really read a character.
 * Also, call count limit on os_stoplist calls, after which we poll for a
 * character press to stop the listing, and a stoplist poll timeout.
 */
static const int	GLN_READCHAR_LIMIT		= 1024;
static const int	GLN_STOPLIST_LIMIT		= 10;
static const glui32	GLN_STOPLIST_TIMEOUT		= 50;

/* Quote used to suppress abbreviation expansion and local commands. */
static const char	GLN_QUOTED_INPUT		= '\'';

/* Definition of whitespace characters to skip over. */
static const char	*GLN_WHITESPACE			= "\t ";

/*
 * Note of when the interpreter is in list output.  The last element of
 * any list generally lacks a terminating newline, and unless we do
 * something special with it, it'll look like a valid prompt to us.
 */
static int		gln_inside_list			= FALSE;


/*
 * gln_char_is_whitespace()
 *
 * Check for ASCII whitespace characters.  Returns TRUE if the character
 * qualifies as whitespace (NUL is not whitespace).
 */
static int
gln_char_is_whitespace (char c)
{
	return (c != '\0' && strchr (GLN_WHITESPACE, c) != NULL);
}


/*
 * gln_skip_leading_whitespace()
 *
 * Skip over leading whitespace, returning the address of the first non-
 * whitespace character.
 */
static char *
gln_skip_leading_whitespace (char *string)
{
	char	*result;			/* Return string pointer. */
	assert (string != NULL);

	/* Move result over leading whitespace. */
	for (result = string; gln_char_is_whitespace (*result); )
		result++;
	return result;
}


/* Table of single-character command abbreviations. */
struct gln_abbreviation {
	const char	abbreviation;		/* Abbreviation character. */
	const char	*expansion;		/* Expansion string. */
};
typedef const struct gln_abbreviation*		gln_abbreviationref_t;
static const struct gln_abbreviation		GLN_ABBREVIATIONS[] = {
	{ 'c',	"close"   },	{ 'g',	"again" },	{ 'i',	"inventory" },
	{ 'k',	"attack"  },	{ 'l',	"look"  },	{ 'p',	"open"      },
	{ 'q',	"quit"    },	{ 'r',	"drop"  },	{ 't',	"take"      },
	{ 'x',	"examine" },	{ 'y',	"yes"   },	{ 'z',	"wait"      },
	{ '\0',	NULL }
};

/*
 * gln_expand_abbreviations()
 *
 * Expand a few common one-character abbreviations commonly found in other
 * game systems, but not always normal in Level9 games.
 */
static void
gln_expand_abbreviations (char *buffer, int size)
{
	char		*command;	/* Single-character command. */
	gln_abbreviationref_t
			entry, match;	/* Table search entry, and match. */
	assert (buffer != NULL);

	/* Skip leading spaces to find command, and return if nothing. */
	command = gln_skip_leading_whitespace (buffer);
	if (strlen (command) == 0)
		return;

	/* If the command is not a single letter one, do nothing. */
	if (strlen (command) > 1
			&& !gln_char_is_whitespace (command[1]))
		return;

	/* Scan the abbreviations table for a match. */
	match = NULL;
	for (entry = GLN_ABBREVIATIONS; entry->expansion != NULL; entry++)
	    {
		if (entry->abbreviation == glk_char_to_lower
					((unsigned char) command[0]))
		    {
			match = entry;
			break;
		    }
	    }
	if (match == NULL)
		return;

	/* Match found, check for a fit. */
	if (strlen (buffer) + strlen (match->expansion) - 1 >= size)
		return;

	/* Replace the character with the full string. */
	memmove (command + strlen (match->expansion) - 1,
					command, strlen (command) + 1);
	memcpy (command, match->expansion, strlen (match->expansion));

	/* Provide feedback on the expansion. */
	gln_standout_string ("[");
	gln_standout_char   (match->abbreviation);
	gln_standout_string (" -> ");
	gln_standout_string (match->expansion);
	gln_standout_string ("]\n");
}


/*
 * gln_output_endlist()
 *
 * The core interpreter doesn't terminate lists with a newline, so we take
 * care of that here; a fixup for input functions.
 */
static void
gln_output_endlist (void)
{
	/* Ignore the call if not inside a list. */
	if (gln_inside_list)
	    {
		/*
		 * Supply the missing newline, using os_printchar() so that
		 * list output doesn't look like a prompt when we come to
		 * flush it.
		 */
		os_printchar ('\n');

		/* Clear the list indicator. */
		gln_inside_list = FALSE;
	    }
}


/*
 * os_input()
 *
 * Read a line from the keyboard.  This function makes a special case of
 * some command strings, and will also perform abbreviation expansion.
 */
L9BOOL
os_input (char *buffer, int size)
{
	event_t		event;			/* Glk event buffer. */
	assert (buffer != NULL);

	/* If doing linemode graphics, run all graphic opcodes available. */
	gln_linegraphics_process ();

	/*
	 * Update the current status line display, flush any pending
	 * buffered output, and terminate any open list.
	 */
	gln_status_notify ();
	gln_output_endlist ();
	gln_output_flush ();

	/*
	 * Level 9 games tend not to issue a prompt after reading an empty
	 * line of input, and the Adrian Mole games don't issue a prompt at
	 * all when outside the 1/2/3 menuing system.  This can make for a
	 * very blank looking screen.
	 *
	 * To slightly improve things, if it looks like we didn't get a
	 * prompt from the game, do our own.
	 */
	if (gln_prompt_enabled
			&& !gln_game_prompted ())
	    {
		gln_normal_char ('\n');
		gln_normal_string (GLN_INPUT_PROMPT);
	    }

	/*
	 * If we have an input log to read from, use that until it is
	 * exhausted.  On end of file, close the stream and resume input
	 * from line requests.
	 */
	if (gln_readlog_stream != NULL)
	    {
		glui32		chars;		/* Characters read. */

		/* Get the next line from the log stream. */
		chars = glk_get_line_stream
				(gln_readlog_stream, buffer, size);
		if (chars > 0)
		    {
			/* Echo the line just read in input style. */
			glk_set_style (style_Input);
			glk_put_buffer (buffer, chars);
			glk_set_style (style_Normal);

			/* Tick the watchdog, and return. */
			gln_watchdog_tick ();
			return TRUE;
		    }

		/*
		 * We're at the end of the log stream.  Close it, and then
		 * continue on to request a line from Glk.
		 */
		glk_stream_close (gln_readlog_stream, NULL);
		gln_readlog_stream = NULL;
	    }

	/* Set up the read buffer for the main window, and wait. */
	glk_request_line_event (gln_main_window, buffer, size - 1, 0);
	gln_event_wait (evtype_LineInput, &event);

	/* Terminate the input line with a NUL. */
	assert (event.val1 <= size - 1);
	buffer[ event.val1 ] = '\0';

	/*
	 * If neither abbreviations nor local commands are enabled, nor
	 * game command interceptions, then return the data read above
	 * without further massaging.
	 */
	if (gln_abbreviations_enabled
				|| gln_commands_enabled
				|| gln_intercept_enabled)
	    {
		char		*command;	/* Command part of buffer. */

		/* Find the first non-space character in the input buffer. */
		command = gln_skip_leading_whitespace (buffer);

		/*
		 * If the first non-space input character is a quote, bypass
		 * all abbreviation expansion and local command recognition,
		 * and use the unadulterated input, less introductory quote.
		 */
		if (command[0] == GLN_QUOTED_INPUT)
		    {
			/* Delete the quote with memmove(). */
			memmove (command, command + 1, strlen (command));
		    }
		else
		    {
			/* Check for, and expand, and abbreviated commands. */
			if (gln_abbreviations_enabled)
				gln_expand_abbreviations (buffer, size);

			/*
			 * Check for Glk port special commands, and if found
			 * then suppress the interpreter's use of this input.
			 */
			if (gln_commands_enabled
					&& gln_command_escape (buffer))
			    {
				gln_watchdog_tick ();
				return FALSE;
			    }

			/*
			 * Check for intercepted commands, and if found then
			 * suppress the interpreter's use of this input.
			 */
			if (gln_intercept_enabled
					&& gln_command_intercept (buffer))
			    {
				gln_watchdog_tick ();
				return FALSE;
			    }
		    }
	    }

	/*
	 * If there is an input log active, log this input string to it.
	 * Note that by logging here we get any abbreviation expansions but
	 * we won't log glk special commands, nor any input read from a
	 * current open input log.
	 */
	if (gln_inputlog_stream != NULL)
	    {
		glk_put_string_stream (gln_inputlog_stream, buffer);
		glk_put_char_stream (gln_inputlog_stream, '\n');
	    }

	/* Return TRUE since data buffered. */
	gln_watchdog_tick ();
	return TRUE;
}


/*
 * os_readchar()
 *
 * Poll the keyboard for characters, and return the character code of any key
 * pressed, or 0 if none pressed.
 *
 * Simple though this sounds, it's tough to do right in a timesharing OS, and
 * requires something close to an abuse of Glk.
 *
 * The initial, tempting, implementation is to wait inside this function for
 * a key press, then return the code.  Unfortunately, this causes problems in
 * the Level9 interpreter.  Here's why: the interpreter is a VM emulating a
 * single-user microprocessor system.  On such a system, it's quite okay for
 * code to spin in a loop waiting for a keypress; there's nothing else
 * happening on the system, so it can burn CPU.  To wait for a keypress, game
 * code might first wait for no-keypress (0 from this function), then a
 * keypress (non-0), then no-keypress again (and it does indeed seem to do
 * just this).  If, in os_readchar(), we simply wait for and return key codes,
 * we'll never return a 0, so the above wait for a keypress in the game will
 * hang forever.
 *
 * To make matters more complex, some Level 9 games poll for keypresses as a
 * way for a user to halt scrolling.  For these polls, we really want to
 * return 0, otherwise the output grinds to a halt.  Moreover, some games even
 * use key polling as a crude form of timeout - poll and increment a counter,
 * and exit when either os_readchar() returns non-0, or after some 300 or so
 * polls.
 *
 * So, this function MUST return 0 sometimes, and real key codes other times.
 * The solution adopted is best described as expedient.  Depending on what Glk
 * provides in the way of timers, we'll do one of two things:
 *
 *   o If we have timers, we'll set up a timeout, and poll for a key press
 *     within that timeout.  As a way to smooth output for games that use key
 *     press polling for scroll control, we'll ignore calls until we get two
 *     in a row without intervening character output.
 *
 *   o If we don't have timers, then we'll return 0 most of the time, and then
 *     really wait for a key one time out of some number.  A game polling for
 *     keypresses to halt scrolling will probably be to the point where it
 *     cannot continue without user input at this juncture, and once we've
 *     rejected a few hundred calls we can now really wait for Glk key press
 *     event, and avoid a spinning loop.  A game using key polling as crude
 *     timing may, or may not, time out in the calls for which we return 0.
 *
 * Empirically, this all seems to work.  The only odd behaviour is with the
 * DEMO mode of Adrian Mole where Glk has no timers, and this is primarily
 * because the DEMO mode relies on the delay of keyboard polling for part of
 * its effect; on a modern system, the time to call through here is nowhere
 * near the time consumed by the original platform.  The other point of note
 * is that this all means that we can't return characters from any readlog
 * with this function; its timing stuff and its general polling nature make
 * it impossible to connect to readlog, so it just won't work at all with the
 * Adrian Mole games, Glk timers or otherwise.
 */
char
os_readchar (int millis)
{
	static int	call_count = 0;		/* Calls count (no timers). */

	event_t		event;			/* Glk event buffer. */
	char		character;		/* Character read. */

	/* If doing linemode graphics, run all graphic opcodes available. */
	gln_linegraphics_process ();

	/*
	 * Here's the way we try to emulate keyboard polling for the case of
	 * no Glk timers.  We'll say nothing is pressed for some number of
	 * consecutive calls, then continue after that number of calls.
	 */
	if (!gln_timeouts_possible)
	    {
		if (++call_count < GLN_READCHAR_LIMIT)
		    {
			/* Call tick as we may be outside an opcode loop. */
			glk_tick ();
			gln_watchdog_tick ();
			return 0;
		    }
		else
			call_count = 0;
	    }

	/*
	 * If we have Glk timers, we can smooth game output with games that
	 * continuously use this input function by pretending that there is
	 * no keypress if the game printed output since the last call.  This
	 * helps with the Adrian Mole games, which check for a keypress at
	 * the end of a line as a way to temporarily halt scrolling.
	 */
	if (gln_timeouts_possible)
	    {
		if (gln_recent_output ())
		    {
			/* Call tick, and return no keypress. */
			glk_tick ();
			gln_watchdog_tick ();
			return 0;
		    }
	    }

	/*
	 * Now flush any pending buffered output.  We do it here rather
	 * than earlier as it only needs to be done when we're going to
	 * request Glk input, and we may have avoided this with the checks
	 * above.
	 */
	gln_status_notify ();
	gln_output_endlist ();
	gln_output_flush ();

	/*
	 * Set up a character event request, and a timeout if the Glk
	 * library can do them, and wait until one or the other occurs.
	 * Loop until we read an acceptable ASCII character (if we don't
	 * time out).
	 */
	do
	    {
		glk_request_char_event (gln_main_window);
		if (gln_timeouts_possible)
		    {
			/* Wait for a character or a timeout event. */
			gln_arbitrate_request_timer_events (millis);
			gln_event_wait_2 (evtype_CharInput,
							evtype_Timer, &event);
			gln_arbitrate_request_timer_events (0);

			/*
			 * If the event was a timeout, cancel the unfilled
			 * character request, and return no-keypress value.
			 */
			if (event.type == evtype_Timer)
			    {
				glk_cancel_char_event (gln_main_window);
				gln_watchdog_tick ();
				return 0;
			    }
		    }
		else
		    {
			/* Wait for only character events. */
			gln_event_wait (evtype_CharInput, &event);
		    }
	    }
	while (event.val1 > UCHAR_MAX && event.val1 != keycode_Return);

	/* Extract the character from the event, converting Return, no echo. */
	character = (event.val1 == keycode_Return) ? '\n' : event.val1;

	/*
	 * Special case ^U as a way to press a key on a wait, yet return a
	 * code to the interpreter as if no key was pressed.  Useful if
	 * scrolling stops where there are no Glk timers, to get scrolling
	 * started again.  ^U is always active.
	 */
	if (character == GLN_CONTROL_U)
	    {
		/* Pretend there was no key press after all. */
		gln_watchdog_tick ();
		return 0;
	    }

	/*
	 * Special case ^C to quit the program.  Without this, there's no
	 * easy way to exit from a game that never uses os_input(), but
	 * instead continually uses just os_readchar().  ^C handling can be
	 * disabled with command line options.
	 */
	if (gln_intercept_enabled
			&& character == GLN_CONTROL_C)
	    {
		/* Confirm the request to quit the game. */
		if (gln_confirm ("\n\nDo you really want to stop? [Y or N] "))
		    {
			gln_stop_reason = STOP_EXIT;
			StopGame ();

			/* Pretend there was no key press after all. */
			gln_watchdog_tick ();
			return 0;
		    }
	    }

	/*
	 * If there is a transcript stream, send the input to it as a one-
	 * line string, otherwise it won't be visible in the transcript.
	 */
	if (gln_transcript_stream != NULL)
	    {
		glk_put_char_stream (gln_transcript_stream, character);
		glk_put_char_stream (gln_transcript_stream, '\n');
	    }

	/* Return the single character read. */
	gln_watchdog_tick ();
	return character;
}


/*
 * os_stoplist()
 *
 * This is called from #dictionary listings to poll for a request to stop
 * the listing.  A check for keypress is usual at this point.  However, Glk
 * cannot check for keypresses without a delay, which slows listing consid-
 * erably, since it also adjusts and renders the display.  As a compromise,
 * then, we'll check for keypresses on a small percentage of calls, say one
 * in ten, which means that listings happen with only a short delay, but
 * there's still an opportunity to stop them.
 *
 * This applies only where the Glk library has timers.  Where it doesn't, we
 * can't check for keypresses without blocking, so we do no checks at all,
 * and let lists always run to completion.
 */
L9BOOL
os_stoplist (void)
{
	static int	call_count = 0;		/* Calls count (timers only). */

	event_t		event;			/* Glk event buffer. */
	int		status;			/* Confirm status. */

	/* Note that the interpreter is producing a list. */
	gln_inside_list = TRUE;

	/*
	 * If there are no Glk timers, then polling for a keypress but
	 * continuing on if there isn't one is not an option.  So flush
	 * output, return FALSE, and just keep listing on to the end.
	 */
	if (!gln_timeouts_possible)
	    {
		gln_output_flush ();
		gln_watchdog_tick ();
		return FALSE;
	    }

	/* Increment the call count, and return FALSE if under the limit. */
	if (++call_count < GLN_STOPLIST_LIMIT)
	    {
		/* Call tick as we may be outside an opcode loop. */
		glk_tick ();
		gln_watchdog_tick ();
		return FALSE;
	    }
	else
		call_count = 0;

	/*
	 * Flush any pending buffered output, delayed to here in case it's
	 * avoidable.
	 */
	gln_output_flush ();

	/*
	 * Look for a keypress, with a very short timeout in place, in a
	 * similar way as done for os_readchar() above.
	 */
	glk_request_char_event (gln_main_window);
	gln_arbitrate_request_timer_events (GLN_STOPLIST_TIMEOUT);
	gln_event_wait_2 (evtype_CharInput, evtype_Timer, &event);
	gln_arbitrate_request_timer_events (0);

	/*
	 * If the event was a timeout, cancel the unfilled character
	 * request, and return FALSE to continue listing.
	 */
	if (event.type == evtype_Timer)
	    {
		glk_cancel_char_event (gln_main_window);
		gln_watchdog_tick ();
		return FALSE;
	    }

	/* Keypress detected, so offer to stop listing. */
	assert (event.type == evtype_CharInput);
	status = gln_confirm ("\n\nStop listing? [Y or N] ");

	/*
	 * As we've output a newline, we no longer consider that we're
	 * inside a list.  Clear the flag, and also and prompt detection.
	 */
	gln_inside_list = FALSE;
	gln_game_prompted ();

	/* Return TRUE if stop was confirmed, FALSE to keep listing. */
	gln_watchdog_tick ();
	return status;
}


/*
 * gln_confirm()
 *
 * Print a confirmation prompt, and read a single input character, taking
 * only [YyNn] input.  If the character is 'Y' or 'y', return TRUE.
 */
static int
gln_confirm (const char *prompt)
{
	event_t		event;			/* Glk event buffer. */
	unsigned char	response;		/* Response character. */
	assert (prompt != NULL);

	/*
	 * Print the confirmation prompt, in a style that hints that it's
	 * from the interpreter, not the game.
	 */
	gln_standout_string (prompt);

	/* Wait for a single 'Y' or 'N' character response. */
	do
	    {
		/* Wait for a standard key, ignoring Glk special keys. */
		do
		    {
			glk_request_char_event (gln_main_window);
			gln_event_wait (evtype_CharInput, &event);
		    }
		while (event.val1 > UCHAR_MAX);
		response = glk_char_to_upper (event.val1);
	    }
	while (response != 'Y' && response != 'N');

	/* Echo the confirmation response, and a blank line. */
	glk_set_style (style_Input);
	glk_put_string (response == 'Y' ? "Yes" : "No");
	glk_set_style (style_Normal);
	glk_put_string ("\n\n");

	/* Return TRUE if 'Y' was entered. */
	return (response == 'Y');
}


/*---------------------------------------------------------------------*/
/*  Glk port event functions                                           */
/*---------------------------------------------------------------------*/

/*
 * gln_event_wait_2()
 * gln_event_wait()
 *
 * Process Glk events until one of the expected type, or types, arrives.
 * Return the event of that type.
 */
static void
gln_event_wait_2 (glui32 wait_type_1, glui32 wait_type_2, event_t *event)
{
	assert (event != NULL);

	/* Get events, until one matches one of the requested types. */
	do
	    {
		/* Get next event. */
		glk_select (event);

		/* Handle events of interest locally. */
		switch (event->type)
		    {
		    case evtype_Arrange:
		    case evtype_Redraw:
			/* Refresh any sensitive windows on size events. */
			gln_status_redraw ();
			gln_graphics_paint ();
			break;

		    case evtype_Timer:
			/* Do background graphics updates on timeout. */
			gln_graphics_timeout ();
			break;
		    }
	    }
	while (event->type != wait_type_1
				&& event->type != wait_type_2);
}

static void
gln_event_wait (glui32 wait_type, event_t *event)
{
	assert (event != NULL);
	gln_event_wait_2 (wait_type, evtype_None, event);
}


/*---------------------------------------------------------------------*/
/*  Glk port file functions                                            */
/*---------------------------------------------------------------------*/

/*
 * os_save_file ()
 * os_load_file ()
 *
 * Save the current game state to a file, and load a game state.
 */
L9BOOL
os_save_file (L9BYTE *ptr, int bytes)
{
	frefid_t	fileref;		/* Glk file reference. */
	strid_t		stream;			/* Glk stream reference. */
	assert (ptr != NULL);

	/* Flush any pending buffered output. */
	gln_output_flush ();

	/* Get a Glk file reference for a game save file. */
	fileref = glk_fileref_create_by_prompt
				(fileusage_SavedGame, filemode_Write, 0);
	if (fileref == NULL)
	    {
		gln_watchdog_tick ();
		return FALSE;
	    }

	/* Open a Glk stream for the fileref. */
	stream = glk_stream_open_file (fileref, filemode_Write, 0);
	if (stream == NULL)
	    {
		glk_fileref_destroy (fileref);
		gln_watchdog_tick ();
		return FALSE;
	    }

	/* Write the game state data. */
	glk_put_buffer_stream (stream, ptr, bytes);

	/* Close and destroy the Glk stream and fileref. */
	glk_stream_close (stream, NULL);
	glk_fileref_destroy (fileref);

	/* All done. */
	gln_watchdog_tick ();
	return TRUE;
}

L9BOOL
os_load_file (L9BYTE *ptr, int *bytes, int max)
{
	frefid_t	fileref;		/* Glk file reference. */
	strid_t		stream;			/* Glk stream reference. */
	assert (ptr != NULL && bytes != NULL);

	/* Flush any pending buffered output. */
	gln_output_flush ();

	/* Get a Glk file reference for a game save file. */
	fileref = glk_fileref_create_by_prompt
				(fileusage_SavedGame, filemode_Read, 0);
	if (fileref == NULL)
	    {
		gln_watchdog_tick ();
		return FALSE;
	    }

	/*
	 * Reject the file reference if we're expecting to read from it,
	 * and the referenced file doesn't exist.
	 */
	if (!glk_fileref_does_file_exist (fileref))
	    {
		glk_fileref_destroy (fileref);
		gln_watchdog_tick ();
		return FALSE;
	    }

	/* Open a Glk stream for the fileref. */
	stream = glk_stream_open_file (fileref, filemode_Read, 0);
	if (stream == NULL)
	    {
		glk_fileref_destroy (fileref);
		gln_watchdog_tick ();
		return FALSE;
	    }

	/* Read back the game state data. */
	*bytes = glk_get_buffer_stream (stream, ptr, max);

	/* Close and destroy the Glk stream and fileref. */
	glk_stream_close (stream, NULL);
	glk_fileref_destroy (fileref);

	/* All done. */
	gln_watchdog_tick ();
	return TRUE;
}


/*---------------------------------------------------------------------*/
/*  Glk port multi-file game functions                                 */
/*---------------------------------------------------------------------*/

/*
 * os_get_game_file ()
 *
 * This function is a bit of a cheat.  It's called when the emulator has
 * detected a request from the game to restart the tape, on a tape-based
 * game.  Ordinarily, we should prompt the player for the name of the
 * system file containing the next game part.  Unfortunately, Glk doesn't
 * make this at all easy.  The requirement is to return a filename, but Glk
 * hides these inside fileref_t's, and won't let them out.
 *
 * Theoretically, according to the porting guide, this function should
 * prompt the user for a new game file name, that being the next part of the
 * game just (presumably) completed.
 *
 * However, the newname passed in is always the current game file name, as
 * level9.c ensures this for us.  If we search for, and find, and then inc-
 * rement, the last digit in the filename passed in, we wind up with, in
 * all likelihood, the right file path.  This is icky.
 *
 * This function is likely to be a source of portability problems on
 * platforms that don't implement a file path/name mechanism that matches
 * the expectations of the Level9 base interpreter fairly closely.
 */
L9BOOL
os_get_game_file (char *newname, int size)
{
	char		*basename;			/* Base of newname. */
	int		index;				/* Char iterator. */
	int		digit;				/* Digit index. */
	int		file_number;			/* Filename's number. */
	FILE		*stream;			/* Test open stream. */
	assert (newname != NULL);

	/* Find the last element of the filename passed in. */
	basename = strrchr (newname, GLN_FILE_DELIM);
	if (basename == NULL)
		basename = newname;
	else
		basename++;

	/* Search for the last numeric character in the basename. */
	digit = -1;
	for (index = strlen (basename) - 1; index >= 0; index--)
	    {
		if (isdigit (basename[ index ]))
		    {
			digit = index;
			break;
		    }
	    }
	if (digit == -1)
	    {
		/* No numeric character, but still note watchdog tick. */
		gln_watchdog_tick ();
		return FALSE;
	    }

	/* Convert the digit and increment it. */
	file_number = basename[ digit ] - '0' + 1;

	/* Refuse values outside of the range 1 to 9. */
	if (file_number < 1 || file_number > 9)
	    {
		gln_watchdog_tick ();
		return FALSE;
	    }

	/* Write the new number back into the file. */
	basename[ digit ] = file_number + '0';

	/* Flush pending output, then display the filename generated. */
	gln_output_flush ();
	gln_game_prompted ();
	gln_standout_string ("\nNext load file: ");
	gln_standout_string (basename);
	gln_standout_string ("\n\n");

	/*
	 * Try to confirm access to the file.  Otherwise, if we return TRUE
	 * but the interpreter can't open the file, it stops the game, and
	 * we then lose any chance to save it before quitting.
	 */
	stream = fopen (newname, "rb");
	if (stream == NULL)
	    {
		/* Restore newname to how it was, and return fail. */
		basename[ digit ] = file_number - 1 + '0';
		gln_watchdog_tick ();
		return FALSE;
	    }
	fclose (stream);

	/* Encourage game name re-lookup, and return success. */
	gln_gameid_game_name_reset ();
	gln_watchdog_tick ();
	return TRUE;
}


/*
 * os_set_filenumber()
 *
 * This function returns the next file in a game series for a disk-based
 * game (typically, gamedat1.dat, gamedat2.dat...).  It finds a single digit
 * in a filename, and resets it to the new value passed in.  The implemen-
 * tation here is based on the generic interface version, and with the same
 * limitations, specifically being limited to file numbers in the range 0
 * to 9, since it works on only the last digit character in the filename
 * buffer passed in.
 *
 * This function may also be a source of portability problems on platforms
 * that don't use "traditional" file path schemes.
 */
void
os_set_filenumber (char *newname, int size, int file_number)
{
	char		*basename;			/* Base of newname. */
	int		index;				/* Char iterator. */
	int		digit;				/* Digit index. */
	assert (newname != NULL);

	/* Do nothing if the file number is beyond what we can handle. */
	if (file_number < 0 || file_number > 9)
	    {
		gln_watchdog_tick ();
		return;
	    }

	/* Find the last element of the new filename. */
	basename = strrchr (newname, GLN_FILE_DELIM);
	if (basename == NULL)
		basename = newname;
	else
		basename++;

	/* Search for the last numeric character in the basename. */
	digit = -1;
	for (index = strlen (basename) - 1; index >= 0; index--)
	    {
		if (isdigit (basename[ index ]))
		    {
			digit = index;
			break;
		    }
	    }
	if (digit == -1)
	    {
		/* No numeric character, but still note watchdog tick. */
		gln_watchdog_tick ();
		return;
	    }

	/* Reset the digit in the file name. */
	basename[ digit ] = file_number + '0';

	/* Flush pending output, then display the filename generated. */
	gln_output_flush ();
	gln_game_prompted ();
	gln_standout_string ("\nNext disk file: ");
	gln_standout_string (basename);
	gln_standout_string ("\n\n");

	/* Encourage game name re-lookup. */
	gln_gameid_game_name_reset ();

	/* Note watchdog tick and return. */
	gln_watchdog_tick ();
}


/*---------------------------------------------------------------------*/
/*  Functions intercepted by link-time wrappers                        */
/*---------------------------------------------------------------------*/

/*
 * __wrap_toupper()
 * __wrap_tolower()
 *
 * Wrapper functions around toupper() and tolower().  The Linux linker's
 * --wrap option will convert calls to mumble() to __wrap_mumble() if we
 * give it the right options.  We'll use this feature to translate all
 * toupper() and tolower() calls in the interpreter code into calls to
 * Glk's versions of these functions.
 *
 * It's not critical that we do this.  If a linker, say a non-Linux one,
 * won't do --wrap, then just do without it.  It's unlikely that there
 * will be much noticeable difference.
 */
int
__wrap_toupper (int ch)
{
	unsigned char		uch;

	uch = glk_char_to_upper ((unsigned char) ch);
	return (int) uch;
}
int
__wrap_tolower (int ch)
{
	unsigned char		lch;

	lch = glk_char_to_lower ((unsigned char) ch);
	return (int) lch;
}


/*---------------------------------------------------------------------*/
/*  main() and options parsing                                         */
/*---------------------------------------------------------------------*/

/*
 * Watchdog timeout -- we'll wait for five seconds of silence from the core
 * interpreter before offering to stop the game forcibly, and we'll check
 * it every 10,240 opcodes.
 */
static const int	GLN_WATCHDOG_TIMEOUT		= 5;
static const int	GLN_WATCHDOG_PERIOD		= 10240;

/*
 * The following values need to be passed between the startup_code and main
 * functions.
 */
static char	*gln_gamefile		= NULL;		/* Name of game file. */
static char	*gln_game_message	= NULL;		/* Error message. */


/*
 * gln_establish_picture_filename()
 *
 * Given a game name, try to create an (optional) graphics data file.  Given
 * an input "file" X, the function looks for X.PIC or X.pic, then for
 * for PICTURE.DAT or picture.dat in the same directory as X.  If the input
 * file already ends with a three-letter extension, it's stripped first.
 *
 * The function returns NULL if a graphics file is not available.  It's not
 * fatal for this to be the case.  Filenames are malloc'ed, and need to be
 * freed by the caller.
 *
 * The function uses fopen() rather than access() since fopen() is an ANSI
 * standard function, and access() isn't.
 */
static void
gln_establish_picture_filename (char *name, char **graphics)
{
	char		*base;			/* Copy of the base string. */
	char		*directory_end;		/* End of path directory. */
	char		*graphics_file;		/* Game graphics file path. */
	FILE		*stream;		/* Test file stream. */
	assert (name != NULL && graphics != NULL);

	/* First, take a destroyable copy of the input filename. */
	base = gln_malloc (strlen (name) + 1);
	strcpy (base, name);

	/* Now, if base has an extension .xxx, remove it. */
	if (strlen (base) > strlen (".XXX"))
	    {
		if (base[ strlen (base) - strlen (".XXX")] == '.')
			base[ strlen (base) - strlen (".XXX") ] = '\0';
	    }

	/* Malloc space for the return graphics file. */
	graphics_file = gln_malloc (strlen (base) + strlen (".PIC") + 1);

	/* Form a candidate graphics file, using a .PIC extension. */
	strcpy (graphics_file, base);
	strcat (graphics_file, ".PIC");
	stream = fopen (graphics_file, "rb");
	if (stream == NULL)
	    {
		/* Retry, using a .pic extension instead. */
		strcpy (graphics_file, base);
		strcat (graphics_file, ".pic");
		stream = fopen (graphics_file, "rb");
		if (stream == NULL)
		    {
			/*
			 * No access to this graphics file.  In this case,
			 * free memory and reset graphics file to NULL.
			 */
			free (graphics_file);
			graphics_file = NULL;
		    }
	    }
	if (stream != NULL)
		fclose (stream);

	/* If we found a graphics file, return its name immediately. */
	if (graphics_file != NULL)
	    {
		*graphics = graphics_file;
		free (base);
		return;
	    }

	/* Retry with base set to the game file directory part only. */
	directory_end = strrchr (base, GLN_FILE_DELIM);
	if (directory_end == NULL)
		directory_end = base;
	else
		directory_end++;
	base[ directory_end - base ] = '\0';

	/* Again, malloc space for the return graphics file. */
	graphics_file = gln_malloc (strlen (base) + strlen ("PICTURE.DAT") + 1);

	/* As above, form a candidate graphics file. */
	strcpy (graphics_file, base);
	strcat (graphics_file, "PICTURE.DAT");
	stream = fopen (graphics_file, "rb");
	if (stream == NULL)
	    {
		/* Retry, using picture.dat extension instead. */
		strcpy (graphics_file, base);
		strcat (graphics_file, "picture.dat");
		stream = fopen (graphics_file, "rb");
		if (stream == NULL)
		    {
			/*
			 * No access to this graphics file.  In this case,
			 * free memory and reset graphics file to NULL.
			 */
			free (graphics_file);
			graphics_file = NULL;
		    }
	    }
	if (stream != NULL)
		fclose (stream);

	/*
	 * Return whatever we found for the graphics file (NULL if none
	 * found, and free base.
	 */
	*graphics = graphics_file;
	free (base);
}


/*
 * gln_startup_code()
 * gln_main()
 *
 * Together, these functions take the place of the original main().
 * The first one is called from glkunix_startup_code(), to parse and
 * generally handle options.  The second is called from glk_main(), and
 * does the real work of running the game.
 */
static int
gln_startup_code (int argc, char *argv[])
{
	int		argv_index;			/* Argument iterator. */

	/* Handle command line arguments. */
	for (argv_index = 1;
		argv_index < argc && argv[ argv_index ][0] == '-'; argv_index++)
	    {
		if (strcmp (argv[ argv_index ], "-ni") == 0)
		    {
			gln_intercept_enabled = FALSE;
			continue;
		    }
		if (strcmp (argv[ argv_index ], "-nc") == 0)
		    {
			gln_commands_enabled = FALSE;
			continue;
		    }
		if (strcmp (argv[ argv_index ], "-na") == 0)
		    {
			gln_abbreviations_enabled = FALSE;
			continue;
		    }
		if (strcmp (argv[ argv_index ], "-np") == 0)
		    {
			gln_graphics_enabled = FALSE;
			continue;
		    }
		if (strcmp (argv[ argv_index ], "-ne") == 0)
		    {
			gln_prompt_enabled = FALSE;
			continue;
		    }
		if (strcmp (argv[ argv_index ], "-nl") == 0)
		    {
			gln_loopcheck_enabled = FALSE;
			continue;
		    }
		return FALSE;
	    }

	/*
	 * Get the name of the game file.  Since we need this in our call
	 * from glk_main, we need to keep it in a module static variable.
	 * If the game file name is omitted, then here we'll set the pointer
	 * to NULL, and complain about it later in main.  Passing the
	 * message string around like this is a nuisance...
	 */
	if (argv_index == argc - 1)
	    {
		gln_gamefile = argv[ argv_index ];
		gln_game_message = NULL;
	    }
	else
	    {
		gln_gamefile = NULL;
		if (argv_index < argc - 1)
			gln_game_message = "More than one game file"
					" was given on the command line.";
		else
			gln_game_message = "No game file was given"
					" on the command line.";
	    }

	/* All startup options were handled successfully. */
	return TRUE;
}

static void
gln_main (void)
{
	char		*graphics_file	= NULL;	/* Picture file path */

	/* Verify CRC tables build correctly, testing a value from L9cut. */
	assert (gln_buffer_crc ("123456789", 9) == 0xBB3D);

	/* Ensure Level9 types have the right sizes. */
	if (sizeof (L9BYTE) != 1
			|| sizeof (L9UINT16) != 2
			|| sizeof (L9UINT32) != 4)
	    {
		gln_fatal ( "GLK: Types sized incorrectly,"
					" recompilation is needed");
		glk_exit ();
	    }

	/* Create the Glk window, and set its stream as the current one. */
	gln_main_window = glk_window_open (0, 0, 0, wintype_TextBuffer, 0);
	if (gln_main_window == NULL)
	    {
		gln_fatal ("GLK: Can't open main window");
		glk_exit ();
	    }
	glk_window_clear (gln_main_window);
	glk_set_window (gln_main_window);
	glk_set_style (style_Normal);

	/* If there's a problem with the game file, complain now. */
	if (gln_gamefile == NULL)
	    {
		assert (gln_game_message != NULL);
		gln_header_string ("Glk Level9 Error\n\n");
		gln_normal_string (gln_game_message);
		gln_normal_char ('\n');
		glk_exit ();
	    }

	/*
	 * Given the basic game name, try to come up with a usable graphics
	 * filenames.  The graphics file may be null.
	 */
	gln_establish_picture_filename (gln_gamefile, &graphics_file);

	/* If Glk can't do timers, note that timeouts are not possible. */
	if (glk_gestalt (gestalt_Timer, 0))
		gln_timeouts_possible = TRUE;
	else
		gln_timeouts_possible = FALSE;

	/*
	 * Check Glk library capabilities, and note pictures are impossible
	 * if the library can't offer both graphics and timers.  We need
	 * timers to create the background "thread" for picture updates.
	 */
	if (glk_gestalt (gestalt_Graphics, 0)
			&& glk_gestalt (gestalt_Timer, 0))
		gln_graphics_possible = TRUE;
	else
		gln_graphics_possible = FALSE;

	/*
	 * If pictures are impossible, clear pictures enabled flag.  That
	 * is, act as if -np was given on the command line, even though
	 * it may not have been.  If pictures are impossible, they can
	 * never be enabled.
	 */
	if (!gln_graphics_possible)
		gln_graphics_enabled = FALSE;

	/* If pictures are possible, search for bitmap graphics. */
	if (gln_graphics_possible)
		gln_graphics_locate_bitmaps (gln_gamefile);

	/* Try to create a one-line status window.  We can live without it. */
	gln_status_window = glk_window_open (gln_main_window,
					winmethod_Above|winmethod_Fixed,
					1, wintype_TextGrid, 0);
	gln_status_redraw ();

	/*
	 * The main interpreter uses rand(), but never seeds the random
	 * number generator.  This can lead to predictability in games
	 * that might be better with a little less, so here, we'll seed
	 * the random number generator ourselves.
	 */
	srand (time (NULL));

	/* Repeat this game until no more restarts requested. */
	do
	    {
		/* Clear the Glk screen. */
		glk_window_clear (gln_main_window);

		/*
		 * In a multi-file game, restarting may mean reverting back
		 * to part one of the game.  So we have to encourage a re-
		 * lookup of the game name at this point.
		 */
		gln_gameid_game_name_reset ();

		/* Load the game, sending in any established graphics file. */
		errno = 0;
		if (!LoadGame (gln_gamefile, graphics_file))
		    {
			/* Report the error, including any details in errno. */
			if (gln_status_window != NULL)
				glk_window_close (gln_status_window, NULL);
			gln_header_string ("Glk Level9 Error\n\n");
			gln_normal_string ("Can't find, open,"
						" or load game file '");
			gln_normal_string (gln_gamefile);
			gln_normal_char ('\'');
			if (errno != 0)
			    {
				gln_normal_string (": ");
				gln_normal_string (strerror (errno));
			    }
			gln_normal_char ('\n');

			/* Free interpreter allocated memory. */
			FreeMemory ();

			/*
			 * Nothing more to be done, so we'll break rather
			 * than exit, to run memory cleanup and close any
			 * open streams.
			 */
			break;
		    }

		/* Print out a short banner. */
		gln_header_string ("\nLevel 9 Interpreter, version 4.0\n");
		gln_banner_string ("Written by Glen Summers and David Kinder\n"
					"Glk interface by Simon Baldwin\n\n");

		/*
		 * Set the stop reason indicator to none.  A game will then
		 * exit with a reason if we call StopGame(), or none if it
		 * exits of its own accord (or with the "#quit" command, say).
		 */
		gln_stop_reason = STOP_NONE;

		/* Start, or restart, watchdog checking. */
		gln_watchdog_start (GLN_WATCHDOG_TIMEOUT, GLN_WATCHDOG_PERIOD);

		/* Run the game until StopGame called. */
		while (TRUE)
		    {
			/* Execute an opcode - returns FALSE on StopGame. */
			if (!RunGame ())
				break;
			glk_tick ();

			if (gln_watchdog_has_timed_out ())
			    {
				gln_stop_reason = STOP_FORCE;
				StopGame ();
				break;
			    }
		    }

		/*
		 * Stop watchdog functions, and flush any pending buffered
		 * output.
		 */
		gln_watchdog_stop ();
		gln_status_notify ();
		gln_output_flush ();

		/* Free interpreter allocated memory. */
		FreeMemory ();

		/*
		 * Unset any "stuck" game 'cheating' flag.  This can get
		 * stuck on if exit is forced from the #cheat mode in the
		 * Adrian Mole games, which otherwise loop infinitely.  Un-
		 * setting the flag here permits restarts; without this,
		 * the core interpreter remains permanently in silent
		 * #cheat mode.
		 */
		Cheating = FALSE;

		/*
		 * If the stop reason is none, something in the game stopped
		 * itself, or the user entered "#quit".  If the stop reason
		 * is force, the user terminated because of an apparent inf-
		 * inite loop.  For both of these, offer the choice to restart,
		 * or not (equivalent to exit).
		 */
		if (gln_stop_reason == STOP_NONE
				|| gln_stop_reason == STOP_FORCE)
		    {
			if (gln_stop_reason == STOP_NONE)
				gln_standout_string (
					"\nThe game has exited.\n");
			else
				gln_standout_string (
					"\nGame exit was forced.  The current"
					" game state is unrecoverable."
					"  Sorry.\n");
			if (gln_confirm
					("\nDo you want to restart? [Y or N] "))
				gln_stop_reason = STOP_RESTART;
			else
				gln_stop_reason = STOP_EXIT;
		    }
	    }
	while (gln_stop_reason == STOP_RESTART);

	/* Free any temporary memory that may have been used by graphics. */
	gln_graphics_cleanup ();
	gln_linegraphics_cleanup ();

	/* And any temporary memory used by game id functions. */
	gln_gameid_game_name_reset ();

	/* Close any open transcript, input log, and/or read log. */
	if (gln_transcript_stream != NULL)
		glk_stream_close (gln_transcript_stream, NULL);
	if (gln_inputlog_stream != NULL)
		glk_stream_close (gln_inputlog_stream, NULL);
	if (gln_readlog_stream != NULL)
		glk_stream_close (gln_readlog_stream, NULL);

	/* Free any graphics file path. */
	if (graphics_file != NULL)
		free (graphics_file);
}


/*---------------------------------------------------------------------*/
/*  Linkage between Glk entry/exit calls and the real interpreter      */
/*---------------------------------------------------------------------*/

/*
 * Safety flags, to ensure we always get startup before main, and that
 * we only get a call to main once.
 */
static int		gln_startup_called	= FALSE;
static int		gln_main_called		= FALSE;

/*
 * glk_main()
 *
 * Main entry point for Glk.  Here, all startup is done, and we call our
 * function to run the game.
 */
void
glk_main (void)
{
	assert (gln_startup_called && !gln_main_called);
	gln_main_called = TRUE;

	/* Call the interpreter main function. */
	gln_main ();
}


/*---------------------------------------------------------------------*/
/*  Glk linkage relevant only to the UNIX platform                     */
/*---------------------------------------------------------------------*/
#ifdef __unix

#include "glkstart.h"

/*
 * Glk arguments for UNIX versions of the Glk interpreter.
 */
glkunix_argumentlist_t glkunix_arguments[] = {
    { (char *) "-nc", glkunix_arg_NoValue,
	(char *) "-nc        No local handling for Glk special commands" },
    { (char *) "-na", glkunix_arg_NoValue,
	(char *) "-na        Turn off abbreviation expansions" },
    { (char *) "-ni", glkunix_arg_NoValue,
	(char *) "-ni        No local handling for 'quit', 'restart'"
						" 'save', and 'restore'" },
    { (char *) "-np", glkunix_arg_NoValue,
	(char *) "-np        Turn off pictures" },
    { (char *) "-ne", glkunix_arg_NoValue,
	(char *) "-ne        Turn off additional interpreter prompt" },
    { (char *) "-nl", glkunix_arg_NoValue,
	(char *) "-nl        Turn off infinite loop detection" },
    { (char *) "", glkunix_arg_ValueCanFollow,
	(char *) "filename   game to run" },
{ NULL, glkunix_arg_End, NULL }
};


/*
 * glkunix_startup_code()
 *
 * Startup entry point for UNIX versions of Glk interpreter.  Glk will
 * call glkunix_startup_code() to pass in arguments.  On startup, we call
 * our function to parse arguments and generally set stuff up.
 */
int
glkunix_startup_code (glkunix_startup_t *data)
{
	assert (!gln_startup_called);
	gln_startup_called = TRUE;

	return gln_startup_code (data->argc, data->argv);
}
#endif /* __unix */


/*---------------------------------------------------------------------*/
/*  Glk linkage relevant only to the Mac platform                      */
/*---------------------------------------------------------------------*/
#ifdef TARGET_OS_MAC

#include "macglk_startup.h"

/* Additional Mac variables. */
static strid_t		gln_mac_gamefile	= NULL;
static short		gln_savedVRefNum	= 0;
static long		gln_savedDirID		= 0;


/*
 * gln_mac_whenselected()
 * gln_mac_whenbuiltin()
 * macglk_startup_code()
 *
 * Startup entry points for Mac versions of Glk interpreter.  Glk will
 * call macglk_startup_code() for details on what to do when the app-
 * lication is selected.  On selection, an argv[] vector is built, and
 * passed to the normal interpreter startup code, after which, Glk will
 * call glk_main().
 */
static Boolean
gln_mac_whenselected (FSSpec *file, OSType filetype)
{
	static char*		argv[2];
	assert (!gln_startup_called);
	gln_startup_called = TRUE;

	/* Set the WD to where the file is, so later fopens work. */
	if (HGetVol (0, &gln_savedVRefNum, &gln_savedDirID) != 0)
	    {
		gln_fatal ("GLK: HGetVol failed");
		return FALSE;
	    }
	if (HSetVol (0, file->vRefNum, file->parID) != 0)
	    {
		gln_fatal ("GLK: HSetVol failed");
		return FALSE;
	    }

	/* Put a CString version of the PString name into argv[1]. */
	argv[1] = gln_malloc (file->name[0] + 1);
	BlockMoveData (file->name + 1, argv[1], file->name[0]);
	argv[1][file->name[0]] = '\0';
	argv[2] = NULL;

	return gln_startup_code (2, argv);
}

static Boolean
gln_mac_whenbuiltin (void)
{
	/* Not implemented yet. */
	return TRUE;
}

Boolean
macglk_startup_code (macglk_startup_t *data)
{
	static OSType		gln_mac_gamefile_types[] = { 'LVL9' };

	data->startup_model	 = macglk_model_ChooseOrBuiltIn;
	data->app_creator	 = 'cAGL';
	data->gamefile_types	 = gln_mac_gamefile_types;
	data->num_gamefile_types = sizeof (gln_mac_gamefile_types)
					/ sizeof (*gln_mac_gamefile_types);
	data->savefile_type	 = 'BINA';
	data->datafile_type	 = 0x3F3F3F3F;
	data->gamefile		 = &gln_mac_gamefile;
	data->when_selected	 = gln_mac_whenselected;
	data->when_builtin	 = gln_mac_whenbuiltin;
	/* macglk_setprefs(); */
	return TRUE;
}
#endif /* TARGET_OS_MAC */
