// EXchess source code, (c) Daniel C. Homan  1997-2000
// Released under the GNU public license, see file license.txt

/* score.cpp */

#include "define.h"
#include "chess.h"
#include "funct.h"
#include "score.h"
#include "hash.h"

// stage of game
extern int stage; 
// taxi-cab distance table 
extern int taxi_cab[64][64];
// external hashing variables
extern h_code wtm, btm, hval[13][64], stage_code[4];
extern unsigned int phash_count;


/*------------------------ Initialize Score Routine ---------------------*/
// This routine initializes the piece square tables for the current search.
// Game stage is also determined

void init_score(position *p, int T)
{
  int wpawns = 0, bpawns = 0, wpieces = 0, bpieces = 0;
  int i,j;
  
  wpawns = p->plist[WHITE][PAWN][0]; 
  bpawns = p->plist[BLACK][PAWN][0];
  wpieces = p->pieces[WHITE]; bpieces = p->pieces[BLACK];

  // determine game stage
  if((bpawns + wpawns) > 13 && (wpieces+bpieces) > 13) stage = 0;
  else if((bpawns + wpawns > 8 && wpieces+bpieces > 8)) stage = 1; 
  else if(bpawns + wpawns > 6 && (wpieces+bpieces) > 6) stage = 2;
  else stage = 3;

  for(i = 0; i < 64; i++) {
    piece_sq[WHITE][PAWN][i] =
       proto_piece_sq[stage][PAWN][whitef[i]];
    piece_sq[WHITE][KNIGHT][i] =
       proto_piece_sq[stage][KNIGHT][whitef[i]];
    piece_sq[WHITE][BISHOP][i] =
       proto_piece_sq[stage][BISHOP][whitef[i]];
    piece_sq[WHITE][ROOK][i] =
       proto_piece_sq[stage][ROOK][whitef[i]];
    piece_sq[WHITE][QUEEN][i] =
       proto_piece_sq[stage][QUEEN][whitef[i]];
    piece_sq[WHITE][KING][i] =
       proto_piece_sq[stage][KING][whitef[i]];

    piece_sq[BLACK][PAWN][i] =
       proto_piece_sq[stage][PAWN][i];
    piece_sq[BLACK][KNIGHT][i] =
       proto_piece_sq[stage][KNIGHT][i];
    piece_sq[BLACK][BISHOP][i] =
       proto_piece_sq[stage][BISHOP][i];
    piece_sq[BLACK][ROOK][i] =
       proto_piece_sq[stage][ROOK][i];
    piece_sq[BLACK][QUEEN][i] =
       proto_piece_sq[stage][QUEEN][i];
    piece_sq[BLACK][KING][i] =
       proto_piece_sq[stage][KING][i];
  }

}

/*--------------------------- Score position ------------------------*/
// Position is scored from point of view of white to move.  Score is
// currently based on a number of factors: Open files, king proximity,
// good/bad bishops, bishop pairs, pawn structure, king safety, etc....
int score_pawns(position *p, pawn_rec *pawns);
int score_king(position *p, pawn_rec *pawns);
int bishop_mobility(position *p, int sq);
extern unsigned long PAWN_SIZE;
extern pawn_rec *pawn_table;

int score_pos(position *p, int alpha, int beta)
{
   register int score = 0, rscore = 0, i, file, rank, sq;
   register int wksq = p->plist[WHITE][KING][1];
   register int bksq = p->plist[BLACK][KING][1];
   pawn_rec *pawns; h_code pcode;


/*+++++++++++++++++++++++++++++++
|
| Score the material on the board
|
+++++++++++++++++++++++++++++++++*/
   score = (p->wtm ? p->material : -p->material); 

/*+++++++++++++++++++++++++++++++
|
| Building Pawn Hash Code
|
+++++++++++++++++++++++++++++++++*/
   pcode = stage_code[stage];
   for(i=1;i<=p->plist[WHITE][PAWN][0];i++)
    or(pcode, hval[WPAWN][p->plist[WHITE][PAWN][i]]);
   for(i=1;i<=p->plist[BLACK][PAWN][0];i++)
    or(pcode, hval[BPAWN][p->plist[BLACK][PAWN][i]]);

/*+++++++++++++++++++++++++++++++
|
| Probing Pawn Hash Table
|
+++++++++++++++++++++++++++++++++*/
   pawns = pawn_table+(pcode.address&(PAWN_SIZE-1));
   if(pawns->key == pcode.key) { 
    score += pawns->score;
    phash_count++;
   } else {
    pawns->key = pcode.key;
    score += score_pawns(p, pawns); 
   } 

/*+++++++++++++++++++++++++++++++
|
| Examining Passed Pawns
|
+++++++++++++++++++++++++++++++++*/
   // For White
   if(pawns->passed_w) {
    for(file=0;file<8;file++) {
     if(bitmask[file]&pawns->passed_w) { 
      for(i=file+48;i>7;i-=8) {
       if(p->sq[i].type==PAWN) {
        rank = RANK(i);
	     /*+++++++++++++++++++++++++++++++++
        |
        | Outside passed pawns
        |
        +++++++++++++++++++++++++++++++++++*/ 
        if(file > FILE(bksq)+3 || file < FILE(bksq)-3)
	      score += PASSED_PAWN*stage*(rank-1)>>1;
	     /*+++++++++++++++++++++++++++++++++
        |
        | If no material: Can King Catch?
        |
        +++++++++++++++++++++++++++++++++++*/ 
        if(p->pieces[BLACK] == 1) {
         if((7-rank < taxi_cab[file+56][bksq]+1 && p->wtm)
            || (7-rank < taxi_cab[file+56][bksq]+2))
	       score += PASSED_PAWN*stage*(rank-1)<<1;
        }           
        break;      
       }
      }
     }
    }
   } 
   // For Black
   if(pawns->passed_b) {
    for(file=0;file<8;file++) {
     if(bitmask[file]&pawns->passed_b) {
      for(i=file+8;i<56;i+=8) {
       if(p->sq[i].type==PAWN) {
        rank = RANK(i);
     	  /*+++++++++++++++++++++++++++++++++
        |
        | Outside passed pawns
        |
        +++++++++++++++++++++++++++++++++++*/ 
        if(file > FILE(wksq)+3 || file < FILE(wksq)-3)
	      score -= PASSED_PAWN*stage*(6-rank)>>1;
	     /*+++++++++++++++++++++++++++++++++
        |
        | If no material: Can King Catch?
        |
        +++++++++++++++++++++++++++++++++++*/ 
        if(p->pieces[WHITE] == 1) {
         if((rank < taxi_cab[file][wksq]+1 && (p->wtm^1))
            || (rank < taxi_cab[file][wksq]+2))
	       score -= PASSED_PAWN*stage*(6-rank)<<1; 
        }   
        break;      
       }
      }
     }
    }
   }    

/*+++++++++++++++++++++++++++++++
|
| Evaluate Bishop Trap
|
+++++++++++++++++++++++++++++++++*/
    if(ID(p->sq[48]) == WBISHOP 
       && ID(p->sq[41]) == BPAWN && ID(p->sq[50]) == BPAWN)
       score -= value[BISHOP] - 100;
    if(ID(p->sq[55]) == WBISHOP 
       && ID(p->sq[46]) == BPAWN && ID(p->sq[53]) == BPAWN)
       score -= value[BISHOP] - 100;
    if(ID(p->sq[8]) == BBISHOP 
       && ID(p->sq[17]) == WPAWN && ID(p->sq[10]) == WPAWN)
       score += value[BISHOP] - 100;
    if(ID(p->sq[15]) == BBISHOP 
       && ID(p->sq[22]) == WPAWN && ID(p->sq[13]) == WPAWN)
       score += value[BISHOP] - 100;

/*+++++++++++++++++++++++++++++++
|
| Examining King Safety
|
+++++++++++++++++++++++++++++++++*/
  
   score += score_king(p, pawns);

/*+++++++++++++++++++++++++++++++
|
| Early Exit ?
|
+++++++++++++++++++++++++++++++++*/
   rscore = score;
   if(p->pieces[WHITE] < 3 && !p->plist[WHITE][PAWN][0] && !p->plist[WHITE][QUEEN][0]
      && !p->plist[WHITE][ROOK][0]) rscore = MIN(rscore,0);
   if(p->pieces[BLACK] < 3 && !p->plist[BLACK][PAWN][0] && !p->plist[BLACK][QUEEN][0]
      && !p->plist[BLACK][ROOK][0]) rscore = MAX(rscore,0);  

  if(p->wtm) { 
    if(beta < rscore-EARLY_EXIT) return(rscore);
    else if(alpha  > rscore+EARLY_EXIT) return(rscore);   
  } else {
    if(beta < -rscore-EARLY_EXIT) return(-rscore);
    else if(alpha > -rscore+EARLY_EXIT) return(-rscore); 
  }

/*+++++++++++++++++++++++++++++++
|
| Evaluate Knights
|
+++++++++++++++++++++++++++++++++*/
   for(i=1;i<=p->plist[WHITE][KNIGHT][0];i++) {
    sq = p->plist[WHITE][KNIGHT][i]; 
    score += piece_sq[WHITE][KNIGHT][sq];
   }
   for(i=1;i<=p->plist[BLACK][KNIGHT][0];i++) { 
    sq = p->plist[BLACK][KNIGHT][i];
    score -= piece_sq[BLACK][KNIGHT][sq];
   }

/*+++++++++++++++++++++++++++++++
|
| Evaluate Bishops
|
+++++++++++++++++++++++++++++++++*/
   for(i=1;i<=p->plist[WHITE][BISHOP][0];i++) {
    sq = p->plist[WHITE][BISHOP][i];
    score += piece_sq[WHITE][BISHOP][sq];
    if(i==2) score += BISHOP_PAIR;
    score += bishop_mobility(p,sq)>>1;
   }
   for(i=1;i<=p->plist[BLACK][BISHOP][0];i++) {
    sq = p->plist[BLACK][BISHOP][i];
    score -= piece_sq[BLACK][BISHOP][sq];
    if(i==2) score -= BISHOP_PAIR;
    score -= bishop_mobility(p,sq)>>1;
   }

/*+++++++++++++++++++++++++++++++
|
| Evaluate Outposts for Knights
|                   and Bishops
|  - Outposts are stored as bits
|    of a character.  Specific bits
|    correspond to particular useful
|    outpost squares.
|
+++++++++++++++++++++++++++++++++*/
   if(pawns->wposts) {
     for(i=0;i<8;i++) {
       if(pawns->wposts&bitmask[i]) {
         if(ID(p->sq[white_outpost[i]]) == WKNIGHT) 
          score += KNIGHT_OUTPOST*outpost_score[i];
         else if(ID(p->sq[white_outpost[i]]) == WBISHOP) 
          score += BISHOP_OUTPOST*outpost_score[i];
          pawns->wposts ^= bitmask[i];
         if(!pawns->wposts) i=8;
       }
     }
   }  
   if(pawns->bposts) {
     for(i=0;i<8;i++) {
       if(pawns->bposts&bitmask[i]) {
         if(ID(p->sq[black_outpost[i]]) == BKNIGHT) 
          score -= KNIGHT_OUTPOST*outpost_score[i];
         else if(ID(p->sq[black_outpost[i]]) == BBISHOP) 
          score -= BISHOP_OUTPOST*outpost_score[i]; 
          pawns->bposts ^= bitmask[i];
         if(!pawns->bposts) i=8;
       }
     }
   }

/*+++++++++++++++++++++++++++++++
|
| Evaluate Rooks
|
+++++++++++++++++++++++++++++++++*/
   for(i=1;i<=p->plist[WHITE][ROOK][0];i++) {
    sq = p->plist[WHITE][ROOK][i]; 
    score += piece_sq[WHITE][ROOK][sq];
    rank = RANK(sq); file = FILE(sq);
    /*+++++++++++++++++++++++++++++++
    |
    | Rooks on open and half-open files
    |
    +++++++++++++++++++++++++++++++++*/
    if(pawns->open_files&bitmask[file]) {
     score += ROOK_OPEN_FILE;
     // Are the rooks connected?
     if(p->plist[WHITE][ROOK][0] > i && i == 1 &&
         FILE(p->plist[WHITE][ROOK][i+1])==file)
      score += ROOK_CONN_OPEN;
    } else if(pawns->half_open_files_w&bitmask[file]) {
     score += ROOK_HALF_OPEN_FILE;
     // Are the rooks connected?
     if(p->plist[WHITE][ROOK][0] > i && i == 1 &&
         FILE(p->plist[WHITE][ROOK][i+1])==file)
      score += ROOK_CONN_HALF_OPEN;
    }
    /*+++++++++++++++++++++++++++++++
    |
    | Rooks boxed in by the king
    |
    +++++++++++++++++++++++++++++++++*/
    if(stage < 3 && rank < 2 && !RANK(wksq) 
        &&((file > 4 && FILE(wksq) > 4 && FILE(wksq) < file)
           || (file < 3 && FILE(wksq) < 3 && FILE(wksq) > file))) 
     score -= BOXED_IN_ROOK;
   }
   for(i=1;i<=p->plist[BLACK][ROOK][0];i++) {
    sq = p->plist[BLACK][ROOK][i]; 
    score -= piece_sq[BLACK][ROOK][sq];
    rank = RANK(sq); file = FILE(sq);
    /*+++++++++++++++++++++++++++++++
    |
    | Rooks on open and half-open files
    |
    +++++++++++++++++++++++++++++++++*/
    if(pawns->open_files&bitmask[file]) {
     score -= ROOK_OPEN_FILE;
     // Are the rooks connected?
     if(p->plist[BLACK][ROOK][0] > i && i == 1 &&
         FILE(p->plist[BLACK][ROOK][i+1])==file)
      score -= ROOK_CONN_OPEN;
    } else if(pawns->half_open_files_b&bitmask[file]) {
     score -= ROOK_HALF_OPEN_FILE;
     // Are the rooks connected?
     if(p->plist[BLACK][ROOK][0] > i && i == 1 &&
         FILE(p->plist[BLACK][ROOK][i+1])==file)
      score -= ROOK_CONN_HALF_OPEN;
    }
    /*+++++++++++++++++++++++++++++++
    |
    | Rooks boxed in by the king
    |
    +++++++++++++++++++++++++++++++++*/
    if(stage < 3 && rank > 5 && RANK(bksq) == 7 
        &&((file > 4 && FILE(bksq) > 4 && FILE(bksq) < file)
           || (file < 3 && FILE(bksq) < 3 && FILE(bksq) > file))) 
     score += BOXED_IN_ROOK;
   }

/*+++++++++++++++++++++++++++++++
|
| Evaluate Queens
|
+++++++++++++++++++++++++++++++++*/
   for(i=1;i<=p->plist[WHITE][QUEEN][0];i++) {
    sq = p->plist[WHITE][QUEEN][i]; 
    score += piece_sq[WHITE][QUEEN][sq];
   }
   for(i=1;i<=p->plist[BLACK][QUEEN][0];i++) { 
    sq = p->plist[BLACK][QUEEN][i];
    score -= piece_sq[BLACK][QUEEN][sq];
   }

   if(p->pieces[WHITE] < 3 && !p->plist[WHITE][PAWN][0] && !p->plist[WHITE][QUEEN][0]
      && !p->plist[WHITE][ROOK][0]) score = MIN(score,0);
   if(p->pieces[BLACK] < 3 && !p->plist[BLACK][PAWN][0] && !p->plist[BLACK][QUEEN][0]
      && !p->plist[BLACK][ROOK][0]) score = MAX(score,0);  

   if(p->wtm) return(score);
   else return(-score);

}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
|
|
|  Score King:  Function to Examine the King and King Safety 
|
|
|
++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int score_king(position *p, pawn_rec *pawns)
{
   register int score = 0, j, i;
   register int wksq = p->plist[WHITE][KING][1];
   register int bksq = p->plist[BLACK][KING][1];
   register int black_king_vuln, white_king_vuln;

   /*+++++++++++++++++++++++++++++++
   |
   | Add values from piece-square table
   |
   +++++++++++++++++++++++++++++++++*/   
   score += piece_sq[WHITE][KING][wksq];   
   score -= piece_sq[BLACK][KING][bksq];
   
   /*+++++++++++++++++++++++++++++++
   |
   | Examine pawn structure 
   |
   +++++++++++++++++++++++++++++++++*/   
   if(stage < 3) {
     /*+++++++++++++++++++++++++++++++
     |
     | Examine White Castling Structure 
     |
     +++++++++++++++++++++++++++++++++*/
     if(!p->has_castled[WHITE]) {
       score -= CASTLED;      
       if(!stage && !(p->castle&3)) score -= NO_POSSIBLE_CASTLE;
     }
     if(FILE(wksq) >= 4 || !p->has_castled[WHITE]) {
       score -= pawns->ks_defects_w;
       if(pawns->ks_defects_w > 25) 
        white_king_vuln = 2;
       else if(pawns->ks_defects_w > 12)
        white_king_vuln = 1;
       else white_king_vuln = 0;
     }
     if(FILE(wksq) < 4 || !p->has_castled[WHITE]) {  
       score -= pawns->qs_defects_w;
       if(pawns->qs_defects_w > 25) 
        white_king_vuln = 2;
       else if(pawns->qs_defects_w > 12)
        white_king_vuln = 1;
       else white_king_vuln = 0;
     } 
     /*+++++++++++++++++++++++++++++++
     |
     | Examine Black Castling Structure 
     |
     +++++++++++++++++++++++++++++++++*/   
     if(!p->has_castled[BLACK]) { 
       score += CASTLED;
       if(!stage && !(p->castle&12)) score += NO_POSSIBLE_CASTLE;
     }  
     if(FILE(bksq) >= 4 || !p->has_castled[BLACK]) {
       score += pawns->ks_defects_b;
       if(pawns->ks_defects_b > 25) 
        black_king_vuln = 2;
       else if(pawns->ks_defects_b > 12)
        black_king_vuln = 1;
       else black_king_vuln = 0;
     }
     if(FILE(bksq) < 4 || !p->has_castled[BLACK]) {  
       score += pawns->qs_defects_b;
       if(pawns->qs_defects_b > 25) 
        black_king_vuln = 2;
       else if(pawns->qs_defects_b > 12)
        black_king_vuln = 1;
       else black_king_vuln = 0;
     } 
   } else {
     white_king_vuln = 0;
     black_king_vuln = 0;
   }

   
   /*+++++++++++++++++++++++++++++++
   |
   | Attacks Near King
   |
   +++++++++++++++++++++++++++++++++*/

 if(white_king_vuln > 1 && p->plist[BLACK][QUEEN][0] 
     && taxi_cab[wksq][p->plist[BLACK][QUEEN][1]] < 3) {  
   if(RANK(wksq)) {
     score -= attacks(wksq-8,p,BLACK,0)*KING_ATTACKS;
     if(FILE(wksq)) 
       score -= attacks(wksq-9,p,BLACK,0)*KING_ATTACKS;
     if(FILE(wksq)< 7)
       score -= attacks(wksq-7,p,BLACK,0)*KING_ATTACKS;
   }
   if(RANK(wksq) < 7) {
     score -= attacks(wksq+8,p,BLACK,0)*KING_ATTACKS;
     if(FILE(wksq)) 
       score -= attacks(wksq+7,p,BLACK,0)*KING_ATTACKS;
     if(FILE(wksq) < 7) 
       score -= attacks(wksq+9,p,BLACK,0)*KING_ATTACKS;
   }
   if(FILE(wksq) < 7) 
     score -= attacks(wksq+1,p,BLACK,0)*KING_ATTACKS;
   if(FILE(wksq))  
     score -= attacks(wksq-1,p,BLACK,0)*KING_ATTACKS;
 }
    
 if(black_king_vuln > 1 && p->plist[WHITE][QUEEN][0] 
     && taxi_cab[bksq][p->plist[WHITE][QUEEN][1]] < 3) {  
   if(RANK(bksq)) {
     score += attacks(bksq-8,p,WHITE,0)*KING_ATTACKS;
     if(FILE(bksq)) 
       score += attacks(bksq-9,p,WHITE,0)*KING_ATTACKS;
     if(FILE(bksq)< 7)
       score += attacks(bksq-7,p,WHITE,0)*KING_ATTACKS;
   }
   if(RANK(bksq) < 7) {
     score += attacks(bksq+8,p,WHITE,0)*KING_ATTACKS;
     if(FILE(bksq)) 
       score += attacks(bksq+7,p,WHITE,0)*KING_ATTACKS;
     if(FILE(bksq) < 7) 
       score += attacks(bksq+9,p,WHITE,0)*KING_ATTACKS;
   }
   if(FILE(bksq) < 7) 
     score += attacks(bksq+1,p,WHITE,0)*KING_ATTACKS;
   if(FILE(bksq))  
     score += attacks(bksq-1,p,WHITE,0)*KING_ATTACKS;
 }

   return score;

}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
|
|
|  Score Pawns:  Function to Examine Pawn structure 
|                and hash the results. 
|
|
|
|
++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define wpawn  p->plist[WHITE][PAWN]
#define bpawn  p->plist[BLACK][PAWN]

int score_pawns(position *p, pawn_rec *pawns)
{
   register int score = 0, i, j;
   register int wpawn_count = p->plist[WHITE][PAWN][0];
   register int bpawn_count = p->plist[BLACK][PAWN][0];
   register int psq, psq2, file, rank;
   register int weak, isle, passed, last_passed;

/*+++++++++++++++++++++++++++++++
|
| Setting the pawn hash parameters
|
+++++++++++++++++++++++++++++++++*/
   pawns->wposts = 0;
   pawns->bposts = 0;
   pawns->ks_defects_w = 0;
   pawns->qs_defects_w = 0;
   pawns->ks_defects_b = 0;
   pawns->qs_defects_b = 0;
   pawns->open_files = 255;
   pawns->half_open_files_w = 255;
   pawns->half_open_files_b = 255;
   pawns->passed_w = 0;
   pawns->passed_b = 0;
   pawns->light_pawns_w = 0;
   pawns->light_pawns_b = 0;   

/*+++++++++++++++++++++++++++++++
|
| First Loop Through White Pawns
|
+++++++++++++++++++++++++++++++++*/
    last_passed = 0;
    for(i = 1; i <= wpawn_count; i++) {
      /*+++++++++++++++++++++++++++++++
      |
      | Setting up variables and flags
      | -General strategy with the flags
      |  is to assume that a pawn is weak,
      |  on an island, or passed until
      |  proven otherwise.
      |
      +++++++++++++++++++++++++++++++++*/
      psq = wpawn[i];
      file = FILE(psq); rank = RANK(psq);
      if((file^rank)&1) pawns->light_pawns_w++;
      isle = 1; passed = 1; weak = 1;     
      /*+++++++++++++++++++++++++++++++
      |
      | Add value from piece-square table
      |
      +++++++++++++++++++++++++++++++++*/
      score += piece_sq[WHITE][PAWN][psq];
      /*+++++++++++++++++++++++++++++++
      |
      | This file is not open.
      |  - It is also not half-open for white
      |
      +++++++++++++++++++++++++++++++++*/
      if(pawns->open_files&bitmask[file])
        pawns->open_files ^= bitmask[file];
      if(pawns->half_open_files_w&bitmask[file])
        pawns->half_open_files_w ^= bitmask[file];
      /*+++++++++++++++++++++++++++++++
      |
      | Look for an immediate neighbor 
      | If one exists, add a bonus for
      | a pawn duo.
      |
      +++++++++++++++++++++++++++++++++*/
      if(file < 7 && ID(p->sq[psq+1]) == WPAWN) score += PAWN_DUO;
      /*+++++++++++++++++++++++++++++++
      |
      | Look at my own pawns to see if we 
      | are doubled or have an island
      | also look for weak pawns 
      |
      +++++++++++++++++++++++++++++++++*/
      for(j = 1; j <= wpawn_count; j++) {
       psq2 = wpawn[j];
       if(j > i) {
        if(FILE(psq2) == file) { 
         score -= DOUBLED_PAWN; 
         if(file < 4) pawns->qs_defects_w++;
         else pawns->ks_defects_w++;
        }
       }
       if(j != i) {
         if(RANK(psq2) <= rank && weak &&
            (FILE(psq2) == file-1 || FILE(psq2) == file+1)) weak = 0;
         if(isle && (FILE(psq2) == file+1 || FILE(psq2) == file-1)) 
              isle = 0;
       }
      }
      /*+++++++++++++++++++++++++++++++
      |
      | If the pawn is weak, score it
      | and see if it is backward. If the
      | pawn is not weak, this file is
      | not half open for black 
      |  - Note: I need to think about
      |          doubled pawn case.
      |  - Note: Open files are automatically
      |           half open - take care with
      |           this in scoring them
      |
      +++++++++++++++++++++++++++++++++*/
      if(weak) {
       score -= WEAK_PAWN;
       if(file < 4) pawns->qs_defects_w++;
       else pawns->ks_defects_w++;
       if((rank < 6 && file && ID(p->sq[psq+15]) == BPAWN) || 
          (rank < 6 && file < 7 && ID(p->sq[psq+17]) == BPAWN)
           || ID(p->sq[psq+8]) == BPAWN) {
          score -= BACKWARD_PAWN;
          if(file < 4) pawns->qs_defects_w++;
          else pawns->ks_defects_w++;
       }
      } else if(pawns->half_open_files_b&bitmask[file])
               pawns->half_open_files_b ^= bitmask[file];
      /*+++++++++++++++++++++++++++++++
      |
      | If the pawn is isolated, score it
      |
      +++++++++++++++++++++++++++++++++*/
      if(isle) {
        score -= PAWN_ISLAND;
        if(file < 4) pawns->qs_defects_w++;
        else pawns->ks_defects_w++;
      }
      /*+++++++++++++++++++++++++++++++
      |
      | Look at enemy pawns to see if 
      | this pawn is passed.
      |
      +++++++++++++++++++++++++++++++++*/
      for(j = 1; j <= bpawn_count; j++) {
       if(RANK(bpawn[j]) > rank && (FILE(bpawn[j]) == file ||
           FILE(bpawn[j]) == file-1 || FILE(bpawn[j]) == file+1))
         { passed = 0; break; }
      }
      /*+++++++++++++++++++++++++++++++
      |
      | If the pawn is passed, score it
      | and look for connected passers.
      | Add the file of this pawn to 
      | the passed pawn list.
      |
      +++++++++++++++++++++++++++++++++*/
      if(passed) {
        score += PASSED_PAWN*stage*(rank-1);
        pawns->passed_w |= bitmask[file];
        // is it protected? - if so, score as if stage = 3
        if(stage < 3) {
         if((file < 7 && ID(p->sq[psq-7]) == WPAWN) ||
            (file && ID(p->sq[psq-9]) == WPAWN))
           score += PASSED_PAWN*(3-stage)*(rank-1); 
        } 
        // detect connected passers
        if(stage > 1) {
          if(rank > 4) {
           if(last_passed) {
             if((file && (bitmask[file-1]&last_passed)) ||
                (file < 7 && (bitmask[file+1]&last_passed)))
               score += CON_PASSED_PAWNS*(stage - 1);
           }
           last_passed |= bitmask[file];
          }
        }
      }
    }

/*+++++++++++++++++++++++++++++++
|
| Loop Through Black Pawns
|
+++++++++++++++++++++++++++++++++*/
    last_passed = 0;
    for(i = 1; i <= bpawn_count; i++) {
      /*+++++++++++++++++++++++++++++++
      |
      | Setting up variables and flags
      | -General strategy with the flags
      |  is to assume that a pawn is weak,
      |  on an island, or passed until
      |  proven otherwise.
      |
      +++++++++++++++++++++++++++++++++*/
      psq = bpawn[i];
      file = FILE(psq); rank = RANK(psq);
      if((file^rank)&1) pawns->light_pawns_b++;
      isle = 1; passed = 1; weak = 1;     
      /*+++++++++++++++++++++++++++++++
      |
      | Add value from piece-square table
      |
      +++++++++++++++++++++++++++++++++*/
      score -= piece_sq[BLACK][PAWN][psq];
      /*+++++++++++++++++++++++++++++++
      |
      | This file is not open
      |  - It is also not half open for black
      |
      +++++++++++++++++++++++++++++++++*/
      if(pawns->open_files&bitmask[file])
       pawns->open_files ^= bitmask[file];
      if(pawns->half_open_files_b&bitmask[file])
        pawns->half_open_files_b ^= bitmask[file];
      /*+++++++++++++++++++++++++++++++
      |
      | Look for an immediate neighbor 
      | If one exists, add a bonus for
      | a pawn duo.
      |
      +++++++++++++++++++++++++++++++++*/
      if(file < 7 && ID(p->sq[psq+1]) == BPAWN) score -= PAWN_DUO;
      /*+++++++++++++++++++++++++++++++
      |
      | Look at my own pawns to see if we 
      | are doubled or have an island
      | also look for weak pawns 
      |
      +++++++++++++++++++++++++++++++++*/
      for(j = 1; j <= bpawn_count; j++) {
       psq2 = bpawn[j];
       if(j > i) {
        if(FILE(psq2) == file) { 
         score += DOUBLED_PAWN; 
         if(file < 4) pawns->qs_defects_b++;
         else pawns->ks_defects_b++;
        }
       }
       if(j != i) {
         if(RANK(psq2) >= rank && weak &&
            (FILE(psq2) == file-1 || FILE(psq2) == file+1)) weak = 0;
         if(isle && (FILE(psq2) == file+1 || FILE(psq2) == file-1)) 
               isle = 0;
       }
      }
      /*+++++++++++++++++++++++++++++++
      |
      | If the pawn is weak, score it
      | and see if it is backward.  If
      | the pawn is not weak, this file
      | is not half open for white.
      |  - Note: Think about doubled
      |          pawn case.
      |
      +++++++++++++++++++++++++++++++++*/
      if(weak) {
       score += WEAK_PAWN;
       if(file < 4) pawns->qs_defects_b++;
       else pawns->ks_defects_b++;
       if((rank > 1 && file < 7 && ID(p->sq[psq-15]) == WPAWN) || 
          (rank > 1 && file && ID(p->sq[psq-17]) == WPAWN)
           || ID(p->sq[psq-8]) == WPAWN) {
          score += BACKWARD_PAWN;
          if(file < 4) pawns->qs_defects_b++;
          else pawns->ks_defects_b++;
       }
      } else if(pawns->half_open_files_w&bitmask[file])
               pawns->half_open_files_w ^= bitmask[file];
      /*+++++++++++++++++++++++++++++++
      |
      | If the pawn is isolated, score it
      |
      +++++++++++++++++++++++++++++++++*/
      if(isle) {
        score += PAWN_ISLAND;
        if(file < 4) pawns->qs_defects_b++;
        else pawns->ks_defects_b++;
      }
      /*+++++++++++++++++++++++++++++++
      |
      | Look at enemy pawns to see if 
      | this pawn is passed.
      |
      +++++++++++++++++++++++++++++++++*/
      for(j = 1; j <= wpawn_count; j++) {
       if(RANK(wpawn[j]) < rank && (FILE(wpawn[j]) == file ||
           FILE(wpawn[j]) == file-1 || FILE(wpawn[j]) == file+1))
         { passed = 0; break; }
      }
      /*+++++++++++++++++++++++++++++++
      |
      | If the pawn is passed, score it
      | and look for connected passers.
      | Add the file of this pawn to 
      | the passed pawn list.
      |
      +++++++++++++++++++++++++++++++++*/
      if(passed) {
        score -= PASSED_PAWN*stage*(6-rank);
        pawns->passed_b |= bitmask[file];
        // is it protected? - if so, score as if stage = 3
        if(stage < 3) {
         if((file < 7 && ID(p->sq[psq+9]) == BPAWN) ||
            (file && ID(p->sq[psq+7]) == BPAWN))
           score -= PASSED_PAWN*(3-stage)*(6-rank); 
        } 
        // detect connected passers
        if(stage > 1) {
          if(rank < 3) {
           if(last_passed) {
             if((file && (bitmask[file-1]&last_passed)) ||
                (file < 7 && (bitmask[file+1]&last_passed)))
               score -= CON_PASSED_PAWNS*(stage - 1);
           }
           last_passed |= bitmask[file];
          }
        }
      }
    }

    // Set possible outpost squares for white

    for(i = 0; i < 8; i++) {
     if(p->sq[white_outpost[i]].type == PAWN) continue;
     file = FILE(white_outpost[i]);
     if((file && ID(p->sq[white_outpost[i]-9]) == WPAWN) ||
        (file < 7 && ID(p->sq[white_outpost[i]-7]) == WPAWN)) {
      passed = 1;
      for(j = 1; j <= bpawn_count; j++) {
         if(RANK(bpawn[j]) > RANK(white_outpost[i]) &&
            (FILE(bpawn[j]) == file-1 || FILE(bpawn[j]) == file+1))
           { passed = 0; break; }
      }
      if(passed) pawns->wposts |= bitmask[i];
     }
    }

    // Set possible outpost squares for black

    for(i = 0; i < 8; i++) {
     if(p->sq[black_outpost[i]].type == PAWN) continue;
     file = FILE(black_outpost[i]);
     if((file && ID(p->sq[black_outpost[i]+7]) == BPAWN) ||
        (file < 7 && ID(p->sq[black_outpost[i]+9]) == BPAWN)) {
      passed = 1;
      for(j = 1; j <= wpawn_count; j++) {
         if(RANK(wpawn[j]) < RANK(black_outpost[i]) &&
            (FILE(wpawn[j]) == file-1 || FILE(wpawn[j]) == file+1))
           { passed = 0; break; }
      }
      if(passed) pawns->bposts |= bitmask[i];
     }
    }

   if(stage < 3) {
     /*+++++++++++++++++++++++++++++++
     |
     | Examine White Castling Structure 
     |
     +++++++++++++++++++++++++++++++++*/
      // King Side
       if(ID(p->sq[13]) != WPAWN) { 
         pawns->ks_defects_w += MISSING_F_PAWN;
         if(ID(p->sq[21]) != WPAWN) {
	   pawns->ks_defects_w += (MISSING_F_PAWN>>1);
         }
       } 
       if(ID(p->sq[14]) != WPAWN) { 
         pawns->ks_defects_w += MISSING_G_PAWN;
         if(ID(p->sq[22]) != WPAWN) {
	   pawns->ks_defects_w += (MISSING_G_PAWN>>1);
         }
       } 
       if(ID(p->sq[15]) != WPAWN) { 
         pawns->ks_defects_w += MISSING_H_PAWN;
         if(ID(p->sq[23]) != WPAWN) {
	   pawns->ks_defects_w += (MISSING_H_PAWN>>1);
         }
       } 
      // Queen Side  
       if(ID(p->sq[8]) != WPAWN) { 
         pawns->qs_defects_w += MISSING_A_PAWN;
         if(ID(p->sq[16]) != WPAWN) {
	   pawns->qs_defects_w += (MISSING_A_PAWN>>1);
         }
       } 
       if(ID(p->sq[10]) != WPAWN) { 
         pawns->qs_defects_w += MISSING_C_PAWN;
         if(ID(p->sq[18]) != WPAWN) {
	   pawns->qs_defects_w += (MISSING_C_PAWN>>1);
         }
       } 
       if(ID(p->sq[9]) != WPAWN) { 
         pawns->qs_defects_w += MISSING_B_PAWN;
         if(ID(p->sq[17]) != WPAWN) {
	   pawns->qs_defects_w += (MISSING_B_PAWN>>1);
         }
       }       
     /*+++++++++++++++++++++++++++++++
     |
     | Examine Black Castling Structure 
     |
     +++++++++++++++++++++++++++++++++*/   
      // King Side 
       if(ID(p->sq[53]) != BPAWN) { 
         pawns->ks_defects_b += MISSING_F_PAWN;
         if(ID(p->sq[45]) != BPAWN) {
	   pawns->ks_defects_b += (MISSING_F_PAWN>>1);
         }
       } 
       if(ID(p->sq[54]) != BPAWN) { 
         pawns->ks_defects_b += MISSING_G_PAWN;
         if(ID(p->sq[46]) != BPAWN) {
	   pawns->ks_defects_b += (MISSING_G_PAWN>>1);
         }
       } 
       if(ID(p->sq[55]) != BPAWN) { 
         pawns->ks_defects_b += MISSING_H_PAWN;
         if(ID(p->sq[47]) != BPAWN) {
	   pawns->ks_defects_b += (MISSING_H_PAWN>>1);
         }
       } 
      // Queen Side 
       if(ID(p->sq[48]) != BPAWN) { 
         pawns->qs_defects_b += MISSING_A_PAWN;
         if(ID(p->sq[40]) != BPAWN) {
	   pawns->qs_defects_b += (MISSING_A_PAWN>>1);
         }
       } 
       if(ID(p->sq[50]) != BPAWN) { 
         pawns->qs_defects_b += MISSING_C_PAWN;
         if(ID(p->sq[42]) != BPAWN) {
	   pawns->qs_defects_b += (MISSING_C_PAWN>>1);
         }
       } 
       if(ID(p->sq[49]) != BPAWN) { 
         pawns->qs_defects_b += MISSING_B_PAWN;
         if(ID(p->sq[41]) != BPAWN) {
	   pawns->qs_defects_b += (MISSING_B_PAWN>>1);
         }
       } 
      
   }

   pawns->score = score;

   return score;

}

/*------------------------------- Bishop Moves --------------------------*/
int bishop_mobility(position *p, int sq)
{
  int mm, nn, ii, self, tsq, count = 0;

  mm = FILE(sq); nn = RANK(sq);
  self = p->sq[sq].side;

  ii = 1;
  while (mm + ii <= 7 && nn + ii <= 7)
  {
   tsq = SQR((mm+ii),(nn+ii));
   if (p->sq[tsq].type != PAWN)
   { if(tsq > 55 || tsq < 8 ||
        (self && !((mm && ID(p->sq[tsq+7]) == BPAWN) 
          || (mm < 7 && ID(p->sq[tsq+9]) == BPAWN))) || 
        (!self && !((mm && ID(p->sq[tsq-9]) == WPAWN) 
          || (mm < 7 && ID(p->sq[tsq-7]) == WPAWN)))) 
     count++; }
   else if (p->sq[tsq].side != self)
   { if(tsq > 55 || tsq < 8 ||
        (self && !((mm && ID(p->sq[tsq+7]) == BPAWN) 
          || (mm < 7 && ID(p->sq[tsq+9]) == BPAWN))) || 
        (!self && !((mm && ID(p->sq[tsq-9]) == WPAWN) 
          || (mm < 7 && ID(p->sq[tsq-7]) == WPAWN)))) 
     count++; break; }
   else break;
   ii++;
  }

  ii = 1;
  while (mm - ii >= 0 && nn + ii <= 7)
  {
   tsq = SQR((mm-ii),(nn+ii));
   if (p->sq[tsq].type != PAWN)
   { if(tsq > 55 || tsq < 8 ||
        (self && !((mm && ID(p->sq[tsq+7]) == BPAWN) 
          || (mm < 7 && ID(p->sq[tsq+9]) == BPAWN))) || 
        (!self && !((mm && ID(p->sq[tsq-9]) == WPAWN) 
          || (mm < 7 && ID(p->sq[tsq-7]) == WPAWN)))) 
     count++; }
   else if (p->sq[tsq].side != self)
   { if(tsq > 55 || tsq < 8 ||
        (self && !((mm && ID(p->sq[tsq+7]) == BPAWN) 
          || (mm < 7 && ID(p->sq[tsq+9]) == BPAWN))) || 
        (!self && !((mm && ID(p->sq[tsq-9]) == WPAWN) 
          || (mm < 7 && ID(p->sq[tsq-7]) == WPAWN)))) 
     count++; break; }
   else break;
   ii++;
  }

  ii = 1;
  while(mm + ii <= 7 && nn - ii >= 0)
  {
   tsq = SQR((mm+ii),(nn-ii));
   if (p->sq[tsq].type != PAWN)
   { if(tsq > 55 || tsq < 8 ||
        (self && !((mm && ID(p->sq[tsq+7]) == BPAWN) 
          || (mm < 7 && ID(p->sq[tsq+9]) == BPAWN))) || 
        (!self && !((mm && ID(p->sq[tsq-9]) == WPAWN) 
          || (mm < 7 && ID(p->sq[tsq-7]) == WPAWN)))) 
     count++; }
   else if (p->sq[tsq].side != self)
   { if(tsq > 55 || tsq < 8 ||
        (self && !((mm && ID(p->sq[tsq+7]) == BPAWN) 
          || (mm < 7 && ID(p->sq[tsq+9]) == BPAWN))) || 
        (!self && !((mm && ID(p->sq[tsq-9]) == WPAWN) 
          || (mm < 7 && ID(p->sq[tsq-7]) == WPAWN)))) 
     count++; break; }
   else break;
   ii++;
  }

  ii = 1;
  while (mm - ii >= 0 && nn - ii >= 0)
  {
   tsq = SQR((mm-ii),(nn-ii));
   if (p->sq[tsq].type != PAWN)
   { if(tsq > 55 || tsq < 8 ||
        (self && !((mm && ID(p->sq[tsq+7]) == BPAWN) 
          || (mm < 7 && ID(p->sq[tsq+9]) == BPAWN))) || 
        (!self && !((mm && ID(p->sq[tsq-9]) == WPAWN) 
          || (mm < 7 && ID(p->sq[tsq-7]) == WPAWN)))) 
     count++; }
   else if (p->sq[tsq].side != self)
   { if(tsq > 55 || tsq < 8 ||
        (self && !((mm && ID(p->sq[tsq+7]) == BPAWN) 
          || (mm < 7 && ID(p->sq[tsq+9]) == BPAWN))) || 
        (!self && !((mm && ID(p->sq[tsq-9]) == WPAWN) 
          || (mm < 7 && ID(p->sq[tsq-7]) == WPAWN)))) 
     count++; break; }
   else break;
   ii++;
  }

 return count;

}







