package org.apache.texen.test;

/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

import java.io.File;


/**
 * Miscellaneous routines to help with testing
 *
 * @version $Id: TestUtil.java 525422 2007-04-04 05:54:17Z wglass $
 */
public class TestUtil
{
    
    /**
     * Do not instantiate.
     */
    private TestUtil()
    {        
    }

    /**
     * Assures that the results directory exists.  If the results directory
     * cannot be created, fails the test.
     */
    public static void assureResultsDirectoryExists (String resultsDirectory)
    {
        File dir = new File(resultsDirectory);
        if (!dir.exists())
        {
            Log.log("Template results directory does not exist");
            if (dir.mkdirs())
            {
                Log.log("Created template results directory");
            }
            else
            {
                String errMsg = "Unable to create template results directory";
                Log.log(errMsg);
                throw new RuntimeException(errMsg);
            }
        }
    }

    /**
     * Normalizes lines to account for platform differences.  Macs use
     * a single \r, DOS derived operating systems use \r\n, and Unix
     * uses \n.  Replace each with a single \n.
     *
     * @author <a href="mailto:rubys@us.ibm.com">Sam Ruby</a>
     * @return source with all line terminations changed to Unix style
     */
    public static String normalizeNewlines (String source)
    {
        return source.replaceAll("\r[\r]?[\n]","\n");
    }

    /**
     * Returns whether the processed template matches the
     * content of the provided comparison file.
     *
     * @return Whether the output matches the contents
     *         of the comparison file.
     *
     * @exception Exception Test failure condition.
     */
    public static boolean compareFiles(String compareFileName, String resultsFileName)
        throws Exception
    {
        Log.log("Comparing result file '" + resultsFileName + "' with compare file '" + compareFileName + "'"); 
        String resultText = org.apache.velocity.util.StringUtils.fileContentsToString(resultsFileName);

        return compareTextWithFile(resultText, compareFileName);
    }

    
    /**
     * Returns whether the processed template matches the
     * content of the provided comparison file.
     *
     * @return Whether the output matches the contents
     *         of the comparison file.
     *
     * @exception Exception Test failure condition.
     */
    public static boolean compareTextWithFile(String resultText, String compareFileName)
        throws Exception
    {
        String compareText = org.apache.velocity.util.StringUtils.fileContentsToString(compareFileName);

        /*
         *  normalize each wrt newline
         */

        String normalizedResultText = normalizeNewlines(resultText);
        String normalizedCompareText = normalizeNewlines(compareText);
        return normalizedResultText.equals( normalizedCompareText );
    }

    
}
