/**
 * @author Jorge Bay Gondra
 */
import Bytecode from '../process/bytecode.cjs';
import { TraversalStrategy } from '../process/traversal-strategy.cjs';
import { Traversal, Traverser } from '../process/traversal.cjs';
import type { ConnectionOptions } from './connection.cjs';
export type RemoteConnectionOptions = ConnectionOptions & {
    session?: string;
};
/**
 * Represents an abstraction of a "connection" to a "server" that is capable of processing a traversal and
 * returning results.
 */
export declare abstract class RemoteConnection {
    url: string;
    protected readonly options: RemoteConnectionOptions;
    /**
     * @param {String} url The resource uri.
     * @param {RemoteConnectionOptions} [options] The connection options.
     */
    constructor(url: string, options?: RemoteConnectionOptions);
    /**
     * Opens the connection, if its not already opened.
     * @returns {Promise}
     */
    abstract open(): Promise<void>;
    /**
     * Returns true if connection is open
     * @returns {Boolean}
     */
    abstract get isOpen(): boolean;
    /**
     * Determines if the connection is already bound to a session. If so, this indicates that the
     * <code>#createSession()</code> cannot be called so as to produce child sessions.
     * @returns {boolean}
     */
    get isSessionBound(): boolean;
    /**
     * Submits the <code>Bytecode</code> provided and returns a <code>RemoteTraversal</code>.
     * @param {Bytecode} bytecode
     * @returns {Promise} Returns a <code>Promise</code> that resolves to a <code>RemoteTraversal</code>.
     */
    abstract submit(bytecode: Bytecode | null): Promise<RemoteTraversal>;
    /**
     * Create a new <code>RemoteConnection</code> that is bound to a session using the configuration from this one.
     * If the connection is already session bound then this function should throw an exception.
     * @returns {RemoteConnection}
     */
    abstract createSession(): RemoteConnection;
    /**
     * Submits a <code>Bytecode.GraphOp.commit</code> to the server and closes the connection.
     * @returns {Promise}
     */
    abstract commit(): Promise<void>;
    /**
     * Submits a <code>Bytecode.GraphOp.rollback</code> to the server and closes the connection.
     * @returns {Promise}
     */
    abstract rollback(): Promise<void>;
    /**
     * Closes the connection where open transactions will close according to the features of the graph provider.
     * @returns {Promise}
     */
    abstract close(): Promise<void>;
}
/**
 * Represents a traversal as a result of a {@link RemoteConnection} submission.
 */
export declare class RemoteTraversal extends Traversal {
    traversers: Traverser<any>[];
    sideEffects?: any[] | undefined;
    constructor(traversers: Traverser<any>[], sideEffects?: any[] | undefined);
}
export declare class RemoteStrategy extends TraversalStrategy {
    connection: RemoteConnection;
    /**
     * Creates a new instance of RemoteStrategy.
     * @param {RemoteConnection} connection
     */
    constructor(connection: RemoteConnection);
    /** @override */
    apply(traversal: Traversal): Promise<void>;
}
