/*
 * Decompiled with CFR 0.152.
 */
package org.apache.tinkerpop.gremlin.driver;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.stream.Collectors;
import org.apache.tinkerpop.gremlin.driver.Client;
import org.apache.tinkerpop.gremlin.driver.Cluster;
import org.apache.tinkerpop.gremlin.driver.Connection;
import org.apache.tinkerpop.gremlin.driver.ConnectionFactory;
import org.apache.tinkerpop.gremlin.driver.Host;
import org.apache.tinkerpop.gremlin.driver.ResultSet;
import org.apache.tinkerpop.gremlin.driver.Settings;
import org.apache.tinkerpop.gremlin.driver.exception.ConnectionException;
import org.apache.tinkerpop.gremlin.util.ExceptionHelper;
import org.apache.tinkerpop.gremlin.util.TimeUtil;
import org.apache.tinkerpop.gremlin.util.message.RequestMessage;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class ConnectionPool {
    private static final Logger logger = LoggerFactory.getLogger(ConnectionPool.class);
    public static final int MIN_POOL_SIZE = 2;
    public static final int MAX_POOL_SIZE = 8;
    public static final int MIN_SIMULTANEOUS_USAGE_PER_CONNECTION = 8;
    public static final int MAX_SIMULTANEOUS_USAGE_PER_CONNECTION = 16;
    private static final int CONNECTION_SETUP_TIME_DELTA = 25;
    public final Host host;
    private final Cluster cluster;
    private final Client client;
    private final List<Connection> connections;
    private final AtomicInteger open;
    private final Set<Connection> bin = new CopyOnWriteArraySet<Connection>();
    private final int minPoolSize;
    private final int maxPoolSize;
    private final int minSimultaneousUsagePerConnection;
    private final int maxSimultaneousUsagePerConnection;
    private final int minInProcess;
    private final String poolLabel;
    private final AtomicInteger scheduledForCreation = new AtomicInteger();
    private final AtomicReference<ConnectionResult> latestConnectionResult = new AtomicReference<ConnectionResult>(new ConnectionResult());
    private final AtomicReference<CompletableFuture<Void>> closeFuture = new AtomicReference();
    private volatile int waiter = 0;
    private final Lock waitLock = new ReentrantLock(true);
    private final Condition hasAvailableConnection = this.waitLock.newCondition();
    ConnectionFactory connectionFactory;

    public ConnectionPool(Host host, Client client) {
        this(host, client, Optional.empty(), Optional.empty());
    }

    public ConnectionPool(Host host, Client client, Optional<Integer> overrideMinPoolSize, Optional<Integer> overrideMaxPoolSize) {
        this(host, client, overrideMinPoolSize, overrideMaxPoolSize, new ConnectionFactory.DefaultConnectionFactory());
    }

    ConnectionPool(Host host, Client client, Optional<Integer> overrideMinPoolSize, Optional<Integer> overrideMaxPoolSize, ConnectionFactory connectionFactory) {
        this.host = host;
        this.client = client;
        this.cluster = client.cluster;
        this.connectionFactory = connectionFactory;
        this.poolLabel = "Connection Pool {host=" + host + "}";
        Settings.ConnectionPoolSettings settings = this.settings();
        this.minPoolSize = overrideMinPoolSize.orElse(settings.minSize);
        this.maxPoolSize = overrideMaxPoolSize.orElse(settings.maxSize);
        this.minSimultaneousUsagePerConnection = settings.minSimultaneousUsagePerConnection;
        this.maxSimultaneousUsagePerConnection = settings.maxSimultaneousUsagePerConnection;
        this.minInProcess = settings.minInProcessPerConnection;
        this.connections = new CopyOnWriteArrayList<Connection>();
        this.open = new AtomicInteger();
        try {
            ArrayList<CompletableFuture<Void>> connectionCreationFutures = new ArrayList<CompletableFuture<Void>>();
            for (int i = 0; i < this.minPoolSize; ++i) {
                connectionCreationFutures.add(CompletableFuture.runAsync(() -> {
                    ConnectionResult result = new ConnectionResult();
                    try {
                        this.connections.add(connectionFactory.create(this));
                        this.open.incrementAndGet();
                    }
                    catch (ConnectionException e) {
                        result.setFailureCause(e);
                        throw new CompletionException(e);
                    }
                    finally {
                        result.setTimeNow();
                        if (this.latestConnectionResult.get() == null || this.latestConnectionResult.get().getTime() < result.getTime()) {
                            this.latestConnectionResult.set(result);
                        }
                    }
                }, this.cluster.connectionScheduler()));
            }
            CompletableFuture.allOf(connectionCreationFutures.toArray(new CompletableFuture[0])).join();
        }
        catch (CancellationException ce) {
            logger.warn("Initialization of connections cancelled for {}", (Object)this.getPoolInfo(), (Object)ce);
            throw ce;
        }
        catch (CompletionException ce) {
            if (this.connections.isEmpty()) {
                this.closeAsync();
                String errMsg = "Could not initialize " + this.minPoolSize + " (minPoolSize) connections in pool for " + this.host + ".  Successful connections=" + this.connections.size() + ". Closing the connection pool.";
                Throwable result = ce;
                Throwable cause = result.getCause();
                if (null != cause) {
                    result = cause;
                }
                throw new CompletionException(errMsg, result);
            }
            logger.warn("ConnectionPool for " + this.host + " initialized with " + this.connections.size() + " expected minPoolSize was " + this.minPoolSize + " - will attempt to recover", (Throwable)ce);
        }
        logger.info("Opening connection pool on {} with core size of {}", (Object)host, (Object)this.minPoolSize);
    }

    public Settings.ConnectionPoolSettings settings() {
        return this.cluster.connectionPoolSettings();
    }

    public Connection borrowConnection(long timeout, TimeUnit unit) throws TimeoutException, ConnectionException {
        logger.debug("Borrowing connection from pool on {} - timeout in {} {}", new Object[]{this.host, timeout, unit});
        if (this.isClosed()) {
            throw new ConnectionException(this.host.getHostUri(), this.host.getAddress(), "Pool is shutdown");
        }
        if (this.connections.isEmpty()) {
            logger.debug("Tried to borrow connection but the pool was empty for {} - scheduling pool creation and waiting for connection", (Object)this.host);
            for (int i = 0; i < this.minPoolSize; ++i) {
                if (this.scheduledForCreation.get() >= this.minPoolSize) continue;
                this.scheduledForCreation.incrementAndGet();
                this.newConnection();
            }
            return this.waitForConnection(timeout, unit);
        }
        Connection leastUsedConn = this.getLeastUsedValidConnection();
        if (null == leastUsedConn) {
            if (this.isClosed()) {
                throw new ConnectionException(this.host.getHostUri(), this.host.getAddress(), "Pool is shutdown");
            }
            logger.debug("Pool was initialized but a connection could not be selected earlier - waiting for connection on {}", (Object)this.host);
            return this.waitForConnection(timeout, unit);
        }
        if (logger.isDebugEnabled()) {
            logger.debug("Return least used {} on {}", (Object)leastUsedConn.getConnectionInfo(), (Object)this.host);
        }
        return leastUsedConn;
    }

    public void returnConnection(Connection connection) throws ConnectionException {
        logger.debug("Attempting to return {} on {}", (Object)connection, (Object)this.host);
        if (this.isClosed()) {
            throw new ConnectionException(this.host.getHostUri(), this.host.getAddress(), "Pool is shutdown");
        }
        int borrowed = connection.borrowed.decrementAndGet();
        if (connection.isDead()) {
            logger.debug("Marking {} as dead", (Object)this.host);
            this.replaceConnection(connection);
        } else {
            if (this.bin.contains(connection) && borrowed == 0) {
                logger.debug("{} is already in the bin and it has no inflight requests so it is safe to close", (Object)connection);
                if (this.bin.remove(connection)) {
                    connection.closeAsync();
                }
                return;
            }
            int poolSize = this.connections.size();
            int availableInProcess = connection.availableInProcess();
            if (poolSize > this.minPoolSize && borrowed <= this.minSimultaneousUsagePerConnection) {
                if (logger.isDebugEnabled()) {
                    logger.debug("On {} pool size of {} > minPoolSize {} and borrowed of {} <= minSimultaneousUsagePerConnection {} so destroy {}", new Object[]{this.host, poolSize, this.minPoolSize, borrowed, this.minSimultaneousUsagePerConnection, connection.getConnectionInfo()});
                }
                this.destroyConnection(connection);
            } else if (availableInProcess < this.minInProcess && this.maxPoolSize > 1) {
                if (logger.isDebugEnabled()) {
                    logger.debug("On {} availableInProcess {} < minInProcess {} so replace {}", new Object[]{this.host, availableInProcess, this.minInProcess, connection.getConnectionInfo()});
                }
                this.replaceConnection(connection);
            } else {
                this.announceAvailableConnection();
            }
        }
    }

    Client getClient() {
        return this.client;
    }

    Cluster getCluster() {
        return this.cluster;
    }

    public boolean isClosed() {
        return this.closeFuture.get() != null;
    }

    public synchronized CompletableFuture<Void> closeAsync() {
        if (this.closeFuture.get() != null) {
            return this.closeFuture.get();
        }
        logger.info("Signalled closing of connection pool on {} with core size of {}", (Object)this.host, (Object)this.minPoolSize);
        this.announceAllAvailableConnection();
        CompletableFuture<Void> future = this.killAvailableConnections();
        this.closeFuture.set(future);
        return future;
    }

    int numConnectionsWaitingToCleanup() {
        return this.bin.size();
    }

    private CompletableFuture<Void> killAvailableConnections() {
        ArrayList<CompletableFuture<Void>> futures = new ArrayList<CompletableFuture<Void>>(this.connections.size() + this.bin.size());
        for (Connection connection : this.connections) {
            CompletableFuture<Void> future = connection.closeAsync();
            future.thenRun(this.open::decrementAndGet);
            futures.add(future);
        }
        for (Connection connection : this.bin) {
            futures.add(connection.closeAsync());
        }
        return CompletableFuture.allOf(futures.toArray(new CompletableFuture[0]));
    }

    void replaceConnection(Connection connection) {
        logger.info("Replace {}", (Object)connection);
        if (connection.isBeingReplaced.getAndSet(true) || this.isClosed()) {
            return;
        }
        this.considerNewConnection();
        this.definitelyDestroyConnection(connection);
    }

    private void considerNewConnection() {
        int inCreation;
        logger.debug("Considering new connection on {} where pool size is {}", (Object)this.host, (Object)this.connections.size());
        do {
            inCreation = this.scheduledForCreation.get();
            logger.debug("There are {} connections scheduled for creation on {}", (Object)inCreation, (Object)this.host);
            if (inCreation < 1) continue;
            return;
        } while (!this.scheduledForCreation.compareAndSet(inCreation, inCreation + 1));
        this.newConnection();
    }

    private void newConnection() {
        this.cluster.connectionScheduler().submit(() -> {
            this.scheduledForCreation.decrementAndGet();
            this.addConnectionIfUnderMaximum();
            return null;
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean addConnectionIfUnderMaximum() {
        int opened;
        do {
            if ((opened = this.open.get()) < this.maxPoolSize) continue;
            return false;
        } while (!this.open.compareAndSet(opened, opened + 1));
        int openCountToActOn = opened;
        if (this.isClosed()) {
            this.open.decrementAndGet();
            return false;
        }
        ConnectionResult result = new ConnectionResult();
        try {
            this.connections.add(this.connectionFactory.create(this));
        }
        catch (Exception ex) {
            this.open.decrementAndGet();
            logger.error(String.format("Connections[%s] were under maximum allowed[%s], but there was an error creating a new connection", openCountToActOn, this.maxPoolSize), (Throwable)ex);
            this.considerHostUnavailable();
            result.setFailureCause(ex);
            boolean bl = false;
            return bl;
        }
        finally {
            result.setTimeNow();
            if (this.latestConnectionResult.get().getTime() < result.getTime()) {
                this.latestConnectionResult.set(result);
            }
        }
        this.announceAllAvailableConnection();
        return true;
    }

    private boolean destroyConnection(Connection connection) {
        int opened;
        do {
            if ((opened = this.open.get()) > this.minPoolSize) continue;
            return false;
        } while (!this.open.compareAndSet(opened, opened - 1));
        this.definitelyDestroyConnection(connection);
        return true;
    }

    public void definitelyDestroyConnection(Connection connection) {
        if (!this.bin.contains(connection) && !connection.isClosing()) {
            this.bin.add(connection);
            this.connections.remove(connection);
            this.open.decrementAndGet();
        }
        if ((connection.isDead() || connection.borrowed.get() == 0) && this.bin.remove(connection)) {
            CompletableFuture<Void> closeFuture = connection.closeAsync();
            closeFuture.whenComplete((v, t) -> logger.debug("Destroyed {}{}{}", new Object[]{connection.getConnectionInfo(), System.lineSeparator(), this.getPoolInfo()}));
        }
    }

    private Connection waitForConnection(long timeout, TimeUnit unit) throws TimeoutException, ConnectionException {
        long start = System.nanoTime();
        long remaining = timeout;
        long to = timeout;
        do {
            try {
                this.awaitAvailableConnection(remaining, unit);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                to = 0L;
            }
            if (this.isClosed()) {
                throw new ConnectionException(this.host.getHostUri(), this.host.getAddress(), "Pool is shutdown");
            }
            Connection leastUsed = this.getLeastUsedValidConnection();
            if (leastUsed != null) {
                if (logger.isDebugEnabled()) {
                    logger.debug("Return least used {} on {} after waiting", (Object)leastUsed.getConnectionInfo(), (Object)this.host);
                }
                return leastUsed;
            }
            remaining = to - TimeUtil.timeSince((long)start, (TimeUnit)unit);
            logger.debug("Continue to wait for connection on {} if {} > 0", (Object)this.host, (Object)remaining);
        } while (remaining > 0L);
        StringBuilder cause = new StringBuilder("Potential Cause: ");
        ConnectionResult res = this.latestConnectionResult.get();
        if (System.currentTimeMillis() - res.getTime() < (long)(this.cluster.getMaxWaitForConnection() + 25) && res.getFailureCause() != null) {
            cause.append(ExceptionHelper.getRootCause((Throwable)res.getFailureCause()).getMessage());
        } else if (this.open.get() >= this.maxPoolSize) {
            cause.append("Number of active requests exceeds pool size. Consider increasing the value for maxConnectionPoolSize.");
        } else {
            cause.setLength(0);
        }
        String timeoutErrorMessage = String.format("Timed-out (%s %s) waiting for connection on %s. %s%s%s", new Object[]{timeout, unit, this.host, cause.toString(), System.lineSeparator(), this.getPoolInfo()});
        logger.error(timeoutErrorMessage);
        TimeoutException timeoutException = new TimeoutException(timeoutErrorMessage);
        this.considerHostUnavailable();
        throw timeoutException;
    }

    public void considerHostUnavailable() {
        boolean maybeUnhealthy = this.connections.stream().allMatch(Connection::isDead);
        if (maybeUnhealthy) {
            this.host.tryReconnectingImmediately(this::tryReconnect);
            if (!this.host.isAvailable()) {
                this.connections.forEach(this::definitelyDestroyConnection);
                this.cluster.loadBalancingStrategy().onUnavailable(this.host);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean tryReconnect(Host h) {
        logger.debug("Trying to re-establish connection on {}", (Object)h);
        Connection connection = null;
        try {
            connection = this.connectionFactory.create(this);
            RequestMessage ping = this.client.buildMessage(this.cluster.validationRequest()).create();
            CompletableFuture<ResultSet> f = new CompletableFuture<ResultSet>();
            connection.write(ping, f);
            f.get().all().get();
            this.cluster.loadBalancingStrategy().onAvailable(h);
            boolean bl = true;
            return bl;
        }
        catch (Exception ex) {
            logger.error(String.format("Failed reconnect attempt on %s%s%s", h, System.lineSeparator(), this.getPoolInfo()), (Throwable)ex);
            boolean bl = false;
            return bl;
        }
        finally {
            if (connection != null) {
                connection.closeAsync();
            }
        }
    }

    private void announceAvailableConnection() {
        logger.debug("Announce connection available on {}", (Object)this.host);
        if (this.waiter == 0) {
            return;
        }
        this.waitLock.lock();
        try {
            this.hasAvailableConnection.signal();
        }
        finally {
            this.waitLock.unlock();
        }
    }

    private synchronized Connection getLeastUsedValidConnection() {
        int minInFlight = Integer.MAX_VALUE;
        Connection leastBusy = null;
        for (Connection connection : this.connections) {
            int inFlight = connection.borrowed.get();
            if (connection.isDead() || inFlight >= minInFlight || inFlight >= this.maxSimultaneousUsagePerConnection) continue;
            minInFlight = inFlight;
            leastBusy = connection;
        }
        if (leastBusy != null) {
            if (leastBusy.borrowed.incrementAndGet() >= this.maxSimultaneousUsagePerConnection && this.connections.size() < this.maxPoolSize) {
                if (logger.isDebugEnabled()) {
                    logger.debug("Least used {} on {} reached maxSimultaneousUsagePerConnection but pool size {} < maxPoolSize - consider new connection", new Object[]{leastBusy.getConnectionInfo(), this.host, this.connections.size()});
                }
                this.considerNewConnection();
            }
        } else if (this.connections.size() < this.maxPoolSize) {
            this.considerNewConnection();
        }
        return leastBusy;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void awaitAvailableConnection(long timeout, TimeUnit unit) throws InterruptedException {
        logger.debug("Wait {} {} for an available connection on {} with {}", new Object[]{timeout, unit, this.host, Thread.currentThread()});
        this.waitLock.lock();
        ++this.waiter;
        try {
            this.hasAvailableConnection.await(timeout, unit);
        }
        finally {
            --this.waiter;
            this.waitLock.unlock();
        }
    }

    private void announceAllAvailableConnection() {
        if (this.waiter == 0) {
            return;
        }
        this.waitLock.lock();
        try {
            this.hasAvailableConnection.signalAll();
        }
        finally {
            this.waitLock.unlock();
        }
    }

    Set<String> getConnectionIDs() {
        return this.connections.stream().map(Connection::getChannelId).collect(Collectors.toSet());
    }

    public String getPoolInfo() {
        return this.getPoolInfo(null);
    }

    public String getPoolInfo(Connection connectionToCallout) {
        StringBuilder sb = new StringBuilder("ConnectionPool (");
        sb.append(this.host.toString());
        sb.append(")");
        if (this.connections.isEmpty()) {
            sb.append("- no connections in pool");
        } else {
            int connectionCount = this.connections.size();
            sb.append(System.lineSeparator());
            sb.append(String.format("Connection Pool Status (size=%s max=%s min=%s toCreate=%s bin=%s)", connectionCount, this.maxPoolSize, this.minPoolSize, this.scheduledForCreation.get(), this.bin.size()));
            sb.append(System.lineSeparator());
            this.appendConnections(sb, connectionToCallout, (CopyOnWriteArrayList)this.connections);
            sb.append(System.lineSeparator());
            sb.append("-- bin --");
            sb.append(System.lineSeparator());
            this.appendConnections(sb, connectionToCallout, new CopyOnWriteArrayList<Connection>(this.bin));
        }
        return sb.toString().trim();
    }

    private void appendConnections(StringBuilder sb, Connection connectionToCallout, CopyOnWriteArrayList<Connection> connections) {
        Iterator<Connection> it = connections.iterator();
        while (it.hasNext()) {
            Connection c = it.next();
            if (c.equals(connectionToCallout)) {
                sb.append("==> ");
            } else {
                sb.append("> ");
            }
            sb.append(c.getConnectionInfo(false));
            if (!it.hasNext()) continue;
            sb.append(System.lineSeparator());
        }
    }

    public String toString() {
        return this.poolLabel;
    }

    public class ConnectionResult {
        private long timeOfConnectionAttempt;
        private Throwable failureCause;

        public Throwable getFailureCause() {
            return this.failureCause;
        }

        public long getTime() {
            return this.timeOfConnectionAttempt;
        }

        public void setFailureCause(Throwable cause) {
            this.failureCause = cause;
        }

        public void setTimeNow() {
            this.timeOfConnectionAttempt = System.currentTimeMillis();
        }
    }
}

