/*
 * Decompiled with CFR 0.152.
 */
package org.apache.gobblin.ingestion.google.webmaster;

import avro.shaded.com.google.common.base.Joiner;
import com.google.api.client.googleapis.batch.json.JsonBatchCallback;
import com.google.api.client.googleapis.json.GoogleJsonError;
import com.google.api.client.http.HttpHeaders;
import com.google.api.client.repackaged.com.google.common.base.Preconditions;
import com.google.api.services.webmasters.model.ApiDimensionFilter;
import com.google.api.services.webmasters.model.SearchAnalyticsQueryResponse;
import com.google.common.base.Optional;
import java.io.IOException;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Deque;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.TimeUnit;
import org.apache.gobblin.configuration.WorkUnitState;
import org.apache.gobblin.ingestion.google.AsyncIteratorWithDataSink;
import org.apache.gobblin.ingestion.google.webmaster.GoogleWebmasterDataFetcher;
import org.apache.gobblin.ingestion.google.webmaster.GoogleWebmasterFilter;
import org.apache.gobblin.ingestion.google.webmaster.ProducerJob;
import org.apache.gobblin.ingestion.google.webmaster.ProgressReporter;
import org.apache.gobblin.ingestion.google.webmaster.TrieBasedProducerJob;
import org.apache.gobblin.ingestion.google.webmaster.UrlTrie;
import org.apache.gobblin.ingestion.google.webmaster.UrlTriePrefixGrouper;
import org.apache.gobblin.util.ExecutorsUtils;
import org.apache.gobblin.util.limiter.RateBasedLimiter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class GoogleWebmasterExtractorIterator
extends AsyncIteratorWithDataSink<String[]> {
    private static final Logger log = LoggerFactory.getLogger(GoogleWebmasterExtractorIterator.class);
    private final RateBasedLimiter LIMITER;
    private final int ROUND_TIME_OUT;
    private final int BATCH_SIZE;
    private final int TRIE_GROUP_SIZE;
    private final boolean APPLY_TRIE_ALGO;
    private final int MAX_RETRY_ROUNDS;
    private final int ROUND_COOL_DOWN;
    private final int PAGE_LIMIT;
    private final int QUERY_LIMIT;
    private final GoogleWebmasterDataFetcher _webmaster;
    private final String _startDate;
    private final String _endDate;
    private final String _country;
    private final Map<GoogleWebmasterFilter.Dimension, ApiDimensionFilter> _filterMap;
    private final List<GoogleWebmasterFilter.Dimension> _requestedDimensions;
    private final List<GoogleWebmasterDataFetcher.Metric> _requestedMetrics;
    private final WorkUnitState _wuState;
    private boolean _failed = false;

    public GoogleWebmasterExtractorIterator(GoogleWebmasterExtractorIterator iterator) {
        this(iterator._webmaster, iterator._startDate, iterator._endDate, iterator._requestedDimensions, iterator._requestedMetrics, iterator._filterMap, iterator._wuState);
    }

    public GoogleWebmasterExtractorIterator(GoogleWebmasterDataFetcher webmaster, String startDate, String endDate, List<GoogleWebmasterFilter.Dimension> requestedDimensions, List<GoogleWebmasterDataFetcher.Metric> requestedMetrics, Map<GoogleWebmasterFilter.Dimension, ApiDimensionFilter> filterMap, WorkUnitState wuState) {
        super(wuState.getPropAsInt("source.async_iterator.blocking_queue_size", 2000), wuState.getPropAsInt("source.async_iterator.poll_blocking_time", 1));
        this._wuState = wuState;
        Preconditions.checkArgument((!filterMap.containsKey((Object)GoogleWebmasterFilter.Dimension.PAGE) ? 1 : 0) != 0, (Object)"Doesn't support filters for page for the time being. Will implement support later. If page filter is provided, the code won't take the responsibility of get all pages, so it will just return all queries for that page.");
        this._webmaster = webmaster;
        this._startDate = startDate;
        this._endDate = endDate;
        this._requestedDimensions = requestedDimensions;
        this._requestedMetrics = requestedMetrics;
        this._filterMap = filterMap;
        this._country = GoogleWebmasterFilter.countryFilterToString(filterMap.get((Object)GoogleWebmasterFilter.Dimension.COUNTRY));
        this.PAGE_LIMIT = wuState.getPropAsInt("source.google_webmasters.request.page_limit", 5000);
        Preconditions.checkArgument((this.PAGE_LIMIT >= 1 ? 1 : 0) != 0, (Object)"Page limit must be at least 1.");
        this.QUERY_LIMIT = wuState.getPropAsInt("source.google_webmasters.request.query_limit", 5000);
        Preconditions.checkArgument((this.QUERY_LIMIT >= 1 ? 1 : 0) != 0, (Object)"Query limit must be at least 1.");
        this.ROUND_TIME_OUT = wuState.getPropAsInt("source.google_webmasters.request.tuning.get_queries.time_out", 120);
        Preconditions.checkArgument((this.ROUND_TIME_OUT > 0 ? 1 : 0) != 0, (Object)"Time out must be positive.");
        this.MAX_RETRY_ROUNDS = wuState.getPropAsInt("source.google_webmasters.request.tuning.get_queries.max_reties", 40);
        Preconditions.checkArgument((this.MAX_RETRY_ROUNDS >= 0 ? 1 : 0) != 0, (Object)"Retry rounds cannot be negative.");
        this.ROUND_COOL_DOWN = wuState.getPropAsInt("source.google_webmasters.request.tuning.get_queries.cool_down_time", 250);
        Preconditions.checkArgument((this.ROUND_COOL_DOWN >= 0 ? 1 : 0) != 0, (Object)"Initial cool down time cannot be negative.");
        double batchesPerSecond = wuState.getPropAsDouble("source.google_webmasters.request.tuning.get_queries.batches_per_second", 2.0);
        Preconditions.checkArgument((batchesPerSecond > 0.0 ? 1 : 0) != 0, (Object)"Requests per second must be positive.");
        this.BATCH_SIZE = wuState.getPropAsInt("source.google_webmasters.request.tuning.get_queries.batch_size", 2);
        Preconditions.checkArgument((this.BATCH_SIZE >= 1 ? 1 : 0) != 0, (Object)"Batch size must be at least 1.");
        this.LIMITER = new RateBasedLimiter(batchesPerSecond, TimeUnit.SECONDS);
        this.TRIE_GROUP_SIZE = wuState.getPropAsInt("source.google_webmasters.request.tuning.get_queries.trie_group_size", 500);
        Preconditions.checkArgument((this.TRIE_GROUP_SIZE >= 1 ? 1 : 0) != 0, (Object)"Group size must be at least 1.");
        this.APPLY_TRIE_ALGO = wuState.getPropAsBoolean("source.google_webmasters.request.tuning.get_queries.apply_trie", false);
        if (this.APPLY_TRIE_ALGO) {
            Preconditions.checkArgument((this.PAGE_LIMIT == 5000 ? 1 : 0) != 0, (Object)"Page limit must be set at 5000 if you want to use the advanced algorithm. This indicates that you understand what you are doing.");
        }
    }

    @Override
    protected Runnable getProducerRunnable() {
        try {
            log.info("Start getting all pages for " + this.toString());
            Collection<ProducerJob> allJobs = this._webmaster.getAllPages(this._startDate, this._endDate, this._country, this.PAGE_LIMIT);
            return new ResponseProducer(allJobs);
        }
        catch (Exception e) {
            log.info(this.toString() + " failed while creating a ResponseProducer", (Throwable)e);
            this._failed = true;
            GoogleWebmasterExtractorIterator.sleepBeforeRetry();
            throw new RuntimeException(e);
        }
    }

    public boolean isFailed() {
        return this._failed;
    }

    public String getCountry() {
        return this._country;
    }

    public String getProperty() {
        return this._webmaster.getSiteProperty();
    }

    public String toString() {
        return String.format("GoogleWebmasterExtractorIterator{property=%s, startDate=%s, endDate=%s, country=%s}", this.getProperty(), this._startDate, this._endDate, this._country);
    }

    private static void sleepBeforeRetry() {
        try {
            log.info("Sleep 20 seconds before task level retry");
            Thread.sleep(20000L);
        }
        catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
    }

    private class ResponseProducer
    implements Runnable {
        private Deque<ProducerJob> _jobsToProcess;

        ResponseProducer(Collection<ProducerJob> jobs) {
            int size = jobs.size();
            if (size == 0) {
                this._jobsToProcess = new ArrayDeque<ProducerJob>();
                return;
            }
            if (GoogleWebmasterExtractorIterator.this.APPLY_TRIE_ALGO) {
                ArrayList<String> pages = new ArrayList<String>(size);
                for (ProducerJob job : jobs) {
                    pages.add(job.getPage());
                }
                UrlTrie trie = new UrlTrie(GoogleWebmasterExtractorIterator.this._webmaster.getSiteProperty(), pages);
                UrlTriePrefixGrouper grouper = new UrlTriePrefixGrouper(trie, GoogleWebmasterExtractorIterator.this.TRIE_GROUP_SIZE);
                this._jobsToProcess = new ArrayDeque<ProducerJob>(size);
                while (grouper.hasNext()) {
                    this._jobsToProcess.add(new TrieBasedProducerJob(GoogleWebmasterExtractorIterator.this._startDate, GoogleWebmasterExtractorIterator.this._endDate, grouper.next(), grouper.getGroupSize()));
                }
            } else {
                this._jobsToProcess = jobs.getClass().equals(ArrayDeque.class) ? (ArrayDeque<Object>)jobs : new ArrayDeque<ProducerJob>(jobs);
            }
        }

        @Override
        public void run() {
            try {
                int r;
                for (r = 0; r <= GoogleWebmasterExtractorIterator.this.MAX_RETRY_ROUNDS; ++r) {
                    int totalPages = 0;
                    for (ProducerJob job : this._jobsToProcess) {
                        totalPages += job.getPagesSize();
                    }
                    if (r > 0) {
                        log.info(String.format("Starting #%d round retries of size %d for %s", r, totalPages, GoogleWebmasterExtractorIterator.this._country));
                    }
                    ProgressReporter reporter = new ProgressReporter(log, totalPages);
                    ConcurrentLinkedDeque<ProducerJob> retries = new ConcurrentLinkedDeque<ProducerJob>();
                    ExecutorService es = Executors.newFixedThreadPool(10, ExecutorsUtils.newDaemonThreadFactory((Optional)Optional.of((Object)log), (Optional)Optional.of((Object)this.getClass().getSimpleName())));
                    ArrayList<ProducerJob> batch = new ArrayList<ProducerJob>(GoogleWebmasterExtractorIterator.this.BATCH_SIZE);
                    while (!this._jobsToProcess.isEmpty()) {
                        ProducerJob job = this._jobsToProcess.poll();
                        if (batch.size() < GoogleWebmasterExtractorIterator.this.BATCH_SIZE) {
                            batch.add(job);
                        }
                        if (batch.size() != GoogleWebmasterExtractorIterator.this.BATCH_SIZE) continue;
                        es.submit(this.getResponses(batch, retries, GoogleWebmasterExtractorIterator.this._dataSink, reporter));
                        batch = new ArrayList(GoogleWebmasterExtractorIterator.this.BATCH_SIZE);
                    }
                    if (!batch.isEmpty()) {
                        es.submit(this.getResponses(batch, retries, GoogleWebmasterExtractorIterator.this._dataSink, reporter));
                    }
                    log.info(String.format("Submitted all jobs at round %d.", r));
                    es.shutdown();
                    boolean terminated = es.awaitTermination(GoogleWebmasterExtractorIterator.this.ROUND_TIME_OUT, TimeUnit.MINUTES);
                    if (!terminated) {
                        es.shutdownNow();
                        throw new RuntimeException(String.format("Timed out while downloading query data for country-%s at round %d. Next round now has size %d.", GoogleWebmasterExtractorIterator.this._country, r, retries.size()));
                    }
                    if (retries.isEmpty()) break;
                    this._jobsToProcess = retries;
                    Thread.sleep(GoogleWebmasterExtractorIterator.this.ROUND_COOL_DOWN);
                }
                if (r == GoogleWebmasterExtractorIterator.this.MAX_RETRY_ROUNDS + 1) {
                    log.error(String.format("Exceeded maximum retries. There are %d unprocessed jobs.", this._jobsToProcess.size()));
                    StringBuilder sb = new StringBuilder();
                    sb.append("You can add as hot start jobs to continue: ").append(System.lineSeparator()).append(System.lineSeparator());
                    sb.append(ProducerJob.serialize(this._jobsToProcess));
                    sb.append(System.lineSeparator());
                    log.error(sb.toString());
                }
                log.info(String.format("ResponseProducer finishes for %s from %s to %s at retry round %d", GoogleWebmasterExtractorIterator.this._country, GoogleWebmasterExtractorIterator.this._startDate, GoogleWebmasterExtractorIterator.this._endDate, r));
            }
            catch (InterruptedException e) {
                log.info(GoogleWebmasterExtractorIterator.this.toString() + " failed while executing the ResponseProducer");
                GoogleWebmasterExtractorIterator.this._failed = true;
                GoogleWebmasterExtractorIterator.sleepBeforeRetry();
                throw new RuntimeException(e);
            }
        }

        private Runnable getResponse(final ProducerJob job, final ConcurrentLinkedDeque<ProducerJob> retries, final LinkedBlockingDeque<String[]> responseQueue, final ProgressReporter reporter) {
            return new Runnable(){

                @Override
                public void run() {
                    try {
                        ArrayList<ApiDimensionFilter> filters = new ArrayList<ApiDimensionFilter>();
                        filters.addAll(GoogleWebmasterExtractorIterator.this._filterMap.values());
                        filters.add(GoogleWebmasterFilter.pageFilter(job.getOperator(), job.getPage()));
                        GoogleWebmasterExtractorIterator.this.LIMITER.acquirePermits(1L);
                        List<String[]> results = GoogleWebmasterExtractorIterator.this._webmaster.performSearchAnalyticsQuery(job.getStartDate(), job.getEndDate(), GoogleWebmasterExtractorIterator.this.QUERY_LIMIT, GoogleWebmasterExtractorIterator.this._requestedDimensions, GoogleWebmasterExtractorIterator.this._requestedMetrics, filters);
                        ResponseProducer.this.onSuccess(job, results, responseQueue, retries);
                        reporter.report(job.getPagesSize(), GoogleWebmasterExtractorIterator.this._country);
                    }
                    catch (IOException e) {
                        ResponseProducer.this.onFailure(e.getMessage(), job, retries);
                    }
                    catch (InterruptedException e) {
                        log.error(String.format("Interrupted while trying to get queries for job %s. Current retry size is %d.", job, retries.size()));
                    }
                }
            };
        }

        private Runnable getResponses(final List<ProducerJob> jobs, final ConcurrentLinkedDeque<ProducerJob> retries, final LinkedBlockingDeque<String[]> responseQueue, final ProgressReporter reporter) {
            final int size = jobs.size();
            if (size == 1) {
                return this.getResponse(jobs.get(0), retries, responseQueue, reporter);
            }
            final ResponseProducer producer = this;
            return new Runnable(){

                @Override
                public void run() {
                    try {
                        ArrayList<ArrayList<ApiDimensionFilter>> filterList = new ArrayList<ArrayList<ApiDimensionFilter>>(size);
                        ArrayList<JsonBatchCallback<SearchAnalyticsQueryResponse>> callbackList = new ArrayList<JsonBatchCallback<SearchAnalyticsQueryResponse>>(size);
                        ArrayList<String> jobPages = new ArrayList<String>();
                        for (ProducerJob j : jobs) {
                            jobPages.add(j.getPage());
                            final ProducerJob job = j;
                            String page = job.getPage();
                            ArrayList<Object> filters = new ArrayList<Object>();
                            filters.addAll(GoogleWebmasterExtractorIterator.this._filterMap.values());
                            filters.add(GoogleWebmasterFilter.pageFilter(job.getOperator(), page));
                            filterList.add(filters);
                            callbackList.add(new JsonBatchCallback<SearchAnalyticsQueryResponse>(){

                                public void onFailure(GoogleJsonError e, HttpHeaders responseHeaders) throws IOException {
                                    producer.onFailure(e.getMessage(), job, retries);
                                    log.debug(job.getPage() + " failed");
                                }

                                public void onSuccess(SearchAnalyticsQueryResponse searchAnalyticsQueryResponse, HttpHeaders responseHeaders) throws IOException {
                                    List<String[]> results = GoogleWebmasterDataFetcher.convertResponse(GoogleWebmasterExtractorIterator.this._requestedMetrics, searchAnalyticsQueryResponse);
                                    producer.onSuccess(job, results, responseQueue, retries);
                                    log.debug(job.getPage() + " succeeded");
                                }
                            });
                        }
                        log.debug("Submitting jobs: " + Arrays.toString(jobPages.toArray()));
                        GoogleWebmasterExtractorIterator.this.LIMITER.acquirePermits(1L);
                        GoogleWebmasterExtractorIterator.this._webmaster.performSearchAnalyticsQueryInBatch(jobs, filterList, callbackList, GoogleWebmasterExtractorIterator.this._requestedDimensions, GoogleWebmasterExtractorIterator.this.QUERY_LIMIT);
                        int processed = 0;
                        for (final ProducerJob job : jobs) {
                            processed += job.getPagesSize();
                        }
                        reporter.report(processed, GoogleWebmasterExtractorIterator.this._country);
                    }
                    catch (IOException e) {
                        log.warn("Batch request failed. Jobs: " + Joiner.on((String)",").join((Iterable)jobs));
                        for (ProducerJob job : jobs) {
                            retries.add(job);
                        }
                    }
                    catch (InterruptedException e) {
                        log.error(String.format("Interrupted while trying to get queries for jobs %s. Current retry size is %d.", Joiner.on((String)",").join((Iterable)jobs), retries.size()));
                    }
                }
            };
        }

        private void onFailure(String errMsg, ProducerJob job, ConcurrentLinkedDeque<ProducerJob> retries) {
            log.debug(String.format("OnFailure: will retry job %s.%sReason:%s", job, System.lineSeparator(), errMsg));
            retries.add(job);
        }

        private void onSuccess(ProducerJob job, List<String[]> results, LinkedBlockingDeque<String[]> responseQueue, ConcurrentLinkedDeque<ProducerJob> pagesToRetry) {
            int size = results.size();
            if (size == 5000) {
                List<? extends ProducerJob> granularJobs = job.partitionJobs();
                if (granularJobs.isEmpty()) {
                    log.warn(String.format("There might be more query data for your job %s. Currently, downloading more than the Google API limit '%d' is not supported.", job, 5000));
                } else {
                    log.info(String.format("Partition current job %s", job));
                    pagesToRetry.addAll(granularJobs);
                    return;
                }
            }
            log.debug(String.format("Finished %s. Current Queue size: %d. Record size: %d.", job, responseQueue.size(), size));
            try {
                for (String[] r : results) {
                    responseQueue.put(r);
                }
            }
            catch (InterruptedException e) {
                log.error(e.getMessage());
                throw new RuntimeException(e);
            }
        }
    }
}

