/*
 * Decompiled with CFR 0.152.
 */
package com.linkedin.parseq.retry;

import com.linkedin.parseq.Context;
import com.linkedin.parseq.Priority;
import com.linkedin.parseq.Task;
import com.linkedin.parseq.function.Function1;
import com.linkedin.parseq.internal.ArgumentUtil;
import com.linkedin.parseq.promise.Promise;
import com.linkedin.parseq.promise.Promises;
import com.linkedin.parseq.promise.SettablePromise;
import com.linkedin.parseq.retry.ErrorClassification;
import com.linkedin.parseq.retry.RetryPolicy;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class RetriableTask<T> {
    private static final Logger LOGGER = LoggerFactory.getLogger(RetriableTask.class);
    private final String _name;
    private final Function1<Integer, Task<T>> _taskFunction;
    private final RetryPolicy _policy;
    private long _startedAt;

    private RetriableTask(String name, RetryPolicy policy, Function1<Integer, Task<T>> taskFunction) {
        ArgumentUtil.requireNotNull(name, "name");
        ArgumentUtil.requireNotNull(policy, "policy");
        ArgumentUtil.requireNotNull(taskFunction, "taskFunction");
        this._name = name;
        this._policy = policy;
        this._taskFunction = taskFunction;
    }

    public static <U> Task<U> withRetryPolicy(String name, RetryPolicy policy, Function1<Integer, Task<U>> taskFunction) {
        RetriableTask retriableTask = new RetriableTask(name, policy, taskFunction);
        return Task.async(name + " retriableTask", retriableTask::run);
    }

    private Task<T> wrap(int attempt) {
        return Task.async(this._policy.getName() + ", attempt " + attempt, context -> {
            SettablePromise result = Promises.settable();
            Task<T> task = this._taskFunction.apply(attempt);
            Task recovery = Task.async(this._name + " recovery", recoveryContext -> {
                SettablePromise recoveryResult = Promises.settable();
                if (task.isFailed()) {
                    ErrorClassification errorClassification = this._policy.getErrorClassifier().apply(task.getError());
                    this.retry(attempt + 1, task.getError(), errorClassification, (Context)recoveryContext, recoveryResult);
                } else {
                    recoveryResult.done(task.get());
                }
                return recoveryResult;
            });
            recovery.setPriority(Priority.MAX_PRIORITY);
            recovery.getShallowTraceBuilder().setSystemHidden(true);
            Promises.propagateResult(recovery, result);
            context.after(task).run(recovery);
            context.run(task);
            return result;
        });
    }

    private void retry(int attempt, Throwable error, ErrorClassification errorClassification, Context recoveryContext, SettablePromise<T> recoveryResult) {
        long backoffTime = this._policy.getBackoffPolicy().nextBackoff(attempt, error);
        if (errorClassification == ErrorClassification.UNRECOVERABLE) {
            LOGGER.debug(String.format("Attempt %s of %s interrupted: %s", attempt, this._name, error.getMessage()));
            recoveryResult.fail(error);
        } else if (this._policy.getTerminationPolicy().shouldTerminate(attempt, System.currentTimeMillis() - this._startedAt + backoffTime)) {
            LOGGER.debug(String.format("Too many exceptions after attempt %s of %s, aborting: %s", attempt, this._name, error.getMessage()));
            recoveryResult.fail(error);
        } else {
            LOGGER.debug(String.format("Attempt %s of %s failed and will be retried after %s millis: %s", attempt, this._name, backoffTime, error.getMessage()));
            Task<T> retryTask = this.wrap(attempt);
            Promises.propagateResult(retryTask, recoveryResult);
            recoveryContext.createTimer(backoffTime, TimeUnit.MILLISECONDS, retryTask);
        }
    }

    private Promise<? extends T> run(Context context) {
        this._startedAt = System.currentTimeMillis();
        Task<T> task = this.wrap(0);
        context.run(task);
        return task;
    }
}

