//  Copyright 2006 Sergei Steshenko 
//  (
//   sergei_steshenko _AT_ pop3.ru
//   sergei_steshenko _AT_ softhome.net
//  ).
//
//  This program is free software; you can redistribute it and/or modify it
//  under the terms of the GNU General Public License as published by the Free
//  Software Foundation; either version 2 of the License, or (at your option)
//  any later version.
//
//  This program is distributed in the hope that it will be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
//  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
//  for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  If someone modifies the program to run with LGPL-compatible FFT, the
//  he/she can also release the derived program under the terms of LGPL, as
//  published by Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//  Boston, MA 02111-1307, USA.

// PERL_BEGIN
#define MALLOC_AND_CHECK(type, ptr, number_of_bytes, line_number) \
if((ptr = (type)malloc(number_of_bytes)) == NULL) \
  {\
  fprintf(stderr, "mbeq_119700: !!! ERROR !!! could not allocate memory at line number %d of %s file\n", line_number, __FILE__);\
  exit(1);\
  }

#define CALLOC_AND_CHECK(type, ptr, number_of_items, item_size, line_number) \
if((ptr = (type)calloc(number_of_items, item_size)) == NULL) \
  {\
  fprintf(stderr, "mbeq_119700: !!! ERROR !!! could not allocate memory at line number %d of %s file\n", line_number, __FILE__);\
  exit(1);\
  }

// PERL_END


#define MIN_DESIRED_FFT_LENGTH 4096

#include <stdlib.h>
#include <string.h>
#include "config.h"

#include <stdio.h>

#ifdef ENABLE_NLS
#include <libintl.h>
#endif

#define         _ISOC9X_SOURCE  1
#define         _ISOC99_SOURCE  1
#define         __USE_ISOC99    1
#define         __USE_ISOC9X    1

#include <math.h>

#include "ladspa.h"

#ifdef FFTW3

#include <fftw3.h>

typedef fftwf_plan fft_plan;
typedef float fftw_real;

#else

#ifdef EXPLICIT_S
#include <srfftw.h>
#else
#include <rfftw.h>
#endif //EXPLICIT_S

typedef rfftw_plan fft_plan;

#endif //FFTW3

#include "ladspa-util.h"

#define OVERSAMP 2 // for the Hanning (and Hamming) windows it can't be
                   // anything else if only before FFT input or
                   // only after FFT output is windowed



// PERL_BEGIN
#define MIN_GAIN ((double)(-12)) // min gain in db
#define MAX_GAIN ((double)(12)) // max gain in db

#define NUMBER_OF_DB_TABLE_ENTRIES 256
#define NUMBER_OF_DB_TABLE_ENTRIES_MINUS_1 255
#define DB_GAIN_PER_ENTRY 0.09375
#define ENTRIES_PER_DB_GAIN 10.6666666666667
// PERL_END

// DB_GAIN_PER_ENTRY is calculated as
// DB_GAIN_PER_ENTRY == ((MAX_GAIN - MIN_GAIN) / NUMBER_OF_DB_TABLE_ENTRIES);
// decibels will be represented with DB_GAIN_PER_ENTRY step
// ENTRIES_PER_DB_GAIN is (1 / DB_GAIN_PER_ENTRY)

// PERL_BEGIN

#define MBEQ_BAND_LOW ((unsigned)0)
#define MBEQ_BAND_00 ((unsigned)1)
#define MBEQ_BAND_01 ((unsigned)2)
#define MBEQ_BAND_02 ((unsigned)3)
#define MBEQ_BAND_03 ((unsigned)4)
#define MBEQ_BAND_04 ((unsigned)5)
#define MBEQ_BAND_05 ((unsigned)6)
#define MBEQ_BAND_06 ((unsigned)7)
#define MBEQ_BAND_07 ((unsigned)8)
#define MBEQ_BAND_08 ((unsigned)9)
#define MBEQ_BAND_09 ((unsigned)10)
#define MBEQ_BAND_10 ((unsigned)11)
#define MBEQ_BAND_11 ((unsigned)12)
#define MBEQ_BAND_12 ((unsigned)13)
#define MBEQ_BAND_13 ((unsigned)14)
#define MBEQ_BAND_14 ((unsigned)15)
#define MBEQ_BAND_15 ((unsigned)16)
#define MBEQ_BAND_16 ((unsigned)17)
#define MBEQ_BAND_17 ((unsigned)18)
#define MBEQ_BAND_18 ((unsigned)19)
#define MBEQ_BAND_19 ((unsigned)20)
#define MBEQ_BAND_20 ((unsigned)21)
#define MBEQ_BAND_21 ((unsigned)22)
#define MBEQ_BAND_22 ((unsigned)23)
#define MBEQ_BAND_23 ((unsigned)24)
#define MBEQ_BAND_24 ((unsigned)25)
#define MBEQ_BAND_25 ((unsigned)26)
#define MBEQ_BAND_26 ((unsigned)27)
#define MBEQ_BAND_27 ((unsigned)28)
#define MBEQ_BAND_28 ((unsigned)29)
#define MBEQ_BAND_29 ((unsigned)30)
#define MBEQ_BAND_HIGH ((unsigned)31)


#define MBEQ_INPUT_L ((unsigned)32)
#define MBEQ_INPUT_R ((unsigned)33)
#define MBEQ_OUTPUT_L ((unsigned)34)
#define MBEQ_OUTPUT_R ((unsigned)35)

#define NUMBER_OF_PORTS 36
// PERL_END

static LADSPA_Descriptor *MChMBEqDescriptor = NULL;

typedef struct
  {
  // PERL_BEGIN
  LADSPA_Data   *band_gain_ptr_low;
  LADSPA_Data   db_gain_low;
  LADSPA_Data   true_gain_low;
  LADSPA_Data   *band_gain_ptr_00;
  LADSPA_Data   db_gain_00;
  LADSPA_Data   true_gain_00;
  LADSPA_Data   *band_gain_ptr_01;
  LADSPA_Data   db_gain_01;
  LADSPA_Data   true_gain_01;
  LADSPA_Data   *band_gain_ptr_02;
  LADSPA_Data   db_gain_02;
  LADSPA_Data   true_gain_02;
  LADSPA_Data   *band_gain_ptr_03;
  LADSPA_Data   db_gain_03;
  LADSPA_Data   true_gain_03;
  LADSPA_Data   *band_gain_ptr_04;
  LADSPA_Data   db_gain_04;
  LADSPA_Data   true_gain_04;
  LADSPA_Data   *band_gain_ptr_05;
  LADSPA_Data   db_gain_05;
  LADSPA_Data   true_gain_05;
  LADSPA_Data   *band_gain_ptr_06;
  LADSPA_Data   db_gain_06;
  LADSPA_Data   true_gain_06;
  LADSPA_Data   *band_gain_ptr_07;
  LADSPA_Data   db_gain_07;
  LADSPA_Data   true_gain_07;
  LADSPA_Data   *band_gain_ptr_08;
  LADSPA_Data   db_gain_08;
  LADSPA_Data   true_gain_08;
  LADSPA_Data   *band_gain_ptr_09;
  LADSPA_Data   db_gain_09;
  LADSPA_Data   true_gain_09;
  LADSPA_Data   *band_gain_ptr_10;
  LADSPA_Data   db_gain_10;
  LADSPA_Data   true_gain_10;
  LADSPA_Data   *band_gain_ptr_11;
  LADSPA_Data   db_gain_11;
  LADSPA_Data   true_gain_11;
  LADSPA_Data   *band_gain_ptr_12;
  LADSPA_Data   db_gain_12;
  LADSPA_Data   true_gain_12;
  LADSPA_Data   *band_gain_ptr_13;
  LADSPA_Data   db_gain_13;
  LADSPA_Data   true_gain_13;
  LADSPA_Data   *band_gain_ptr_14;
  LADSPA_Data   db_gain_14;
  LADSPA_Data   true_gain_14;
  LADSPA_Data   *band_gain_ptr_15;
  LADSPA_Data   db_gain_15;
  LADSPA_Data   true_gain_15;
  LADSPA_Data   *band_gain_ptr_16;
  LADSPA_Data   db_gain_16;
  LADSPA_Data   true_gain_16;
  LADSPA_Data   *band_gain_ptr_17;
  LADSPA_Data   db_gain_17;
  LADSPA_Data   true_gain_17;
  LADSPA_Data   *band_gain_ptr_18;
  LADSPA_Data   db_gain_18;
  LADSPA_Data   true_gain_18;
  LADSPA_Data   *band_gain_ptr_19;
  LADSPA_Data   db_gain_19;
  LADSPA_Data   true_gain_19;
  LADSPA_Data   *band_gain_ptr_20;
  LADSPA_Data   db_gain_20;
  LADSPA_Data   true_gain_20;
  LADSPA_Data   *band_gain_ptr_21;
  LADSPA_Data   db_gain_21;
  LADSPA_Data   true_gain_21;
  LADSPA_Data   *band_gain_ptr_22;
  LADSPA_Data   db_gain_22;
  LADSPA_Data   true_gain_22;
  LADSPA_Data   *band_gain_ptr_23;
  LADSPA_Data   db_gain_23;
  LADSPA_Data   true_gain_23;
  LADSPA_Data   *band_gain_ptr_24;
  LADSPA_Data   db_gain_24;
  LADSPA_Data   true_gain_24;
  LADSPA_Data   *band_gain_ptr_25;
  LADSPA_Data   db_gain_25;
  LADSPA_Data   true_gain_25;
  LADSPA_Data   *band_gain_ptr_26;
  LADSPA_Data   db_gain_26;
  LADSPA_Data   true_gain_26;
  LADSPA_Data   *band_gain_ptr_27;
  LADSPA_Data   db_gain_27;
  LADSPA_Data   true_gain_27;
  LADSPA_Data   *band_gain_ptr_28;
  LADSPA_Data   db_gain_28;
  LADSPA_Data   true_gain_28;
  LADSPA_Data   *band_gain_ptr_29;
  LADSPA_Data   db_gain_29;
  LADSPA_Data   true_gain_29;
  LADSPA_Data   *band_gain_ptr_high;
  LADSPA_Data   db_gain_high;
  LADSPA_Data   true_gain_high;


  fftw_real     *comp_L;
  fftw_real     *real_L;

  LADSPA_Data   *input_L;
  LADSPA_Data   *output_L;
  LADSPA_Data   *in_fifo_L;
  LADSPA_Data   *out_fifo_L;
  LADSPA_Data   *out_accum_L;



  fftw_real     *comp_R;
  fftw_real     *real_R;

  LADSPA_Data   *input_R;
  LADSPA_Data   *output_R;
  LADSPA_Data   *in_fifo_R;
  LADSPA_Data   *out_fifo_R;
  LADSPA_Data   *out_accum_R;

  // PERL_END

#ifdef FFTW3
  // PERL_BEGIN
  fft_plan      plan_cr_L;
  fft_plan      plan_rc_L;
  fft_plan      plan_cr_R;
  fft_plan      plan_rc_R;
  // PERL_END
#else
  fft_plan      plan_cr;
  fft_plan      plan_rc;
#endif

  unsigned long fft_length;
  unsigned long fft_latency;
  unsigned long half_fft_length;
  unsigned long half_fft_length_minus_1;
  unsigned long step_size;
  unsigned long number_of_bytes_in_step;
  unsigned long number_of_bytes_in_fft_length;
  unsigned long number_of_bytes_in_fft_latency;

  float         *cos_times_gain;

  // bins probably should not be part of the structure...
  // PERL_BEGIN
  unsigned long bin_low;
  unsigned long bin_00;
  unsigned long bin_01;
  unsigned long bin_02;
  unsigned long bin_03;
  unsigned long bin_04;
  unsigned long bin_05;
  unsigned long bin_06;
  unsigned long bin_07;
  unsigned long bin_08;
  unsigned long bin_09;
  unsigned long bin_10;
  unsigned long bin_11;
  unsigned long bin_12;
  unsigned long bin_13;
  unsigned long bin_14;
  unsigned long bin_15;
  unsigned long bin_16;
  unsigned long bin_17;
  unsigned long bin_18;
  unsigned long bin_19;
  unsigned long bin_20;
  unsigned long bin_21;
  unsigned long bin_22;
  unsigned long bin_23;
  unsigned long bin_24;
  unsigned long bin_25;
  unsigned long bin_26;
  unsigned long bin_27;
  unsigned long bin_28;
  unsigned long bin_29;
  unsigned long bin_high;

  LADSPA_Data   reverse_delta_00_low;
  unsigned long ul_delta_00_low;

  LADSPA_Data   reverse_delta_01_00;
  unsigned long ul_delta_01_00;

  LADSPA_Data   reverse_delta_02_01;
  unsigned long ul_delta_02_01;

  LADSPA_Data   reverse_delta_03_02;
  unsigned long ul_delta_03_02;

  LADSPA_Data   reverse_delta_04_03;
  unsigned long ul_delta_04_03;

  LADSPA_Data   reverse_delta_05_04;
  unsigned long ul_delta_05_04;

  LADSPA_Data   reverse_delta_06_05;
  unsigned long ul_delta_06_05;

  LADSPA_Data   reverse_delta_07_06;
  unsigned long ul_delta_07_06;

  LADSPA_Data   reverse_delta_08_07;
  unsigned long ul_delta_08_07;

  LADSPA_Data   reverse_delta_09_08;
  unsigned long ul_delta_09_08;

  LADSPA_Data   reverse_delta_10_09;
  unsigned long ul_delta_10_09;

  LADSPA_Data   reverse_delta_11_10;
  unsigned long ul_delta_11_10;

  LADSPA_Data   reverse_delta_12_11;
  unsigned long ul_delta_12_11;

  LADSPA_Data   reverse_delta_13_12;
  unsigned long ul_delta_13_12;

  LADSPA_Data   reverse_delta_14_13;
  unsigned long ul_delta_14_13;

  LADSPA_Data   reverse_delta_15_14;
  unsigned long ul_delta_15_14;

  LADSPA_Data   reverse_delta_16_15;
  unsigned long ul_delta_16_15;

  LADSPA_Data   reverse_delta_17_16;
  unsigned long ul_delta_17_16;

  LADSPA_Data   reverse_delta_18_17;
  unsigned long ul_delta_18_17;

  LADSPA_Data   reverse_delta_19_18;
  unsigned long ul_delta_19_18;

  LADSPA_Data   reverse_delta_20_19;
  unsigned long ul_delta_20_19;

  LADSPA_Data   reverse_delta_21_20;
  unsigned long ul_delta_21_20;

  LADSPA_Data   reverse_delta_22_21;
  unsigned long ul_delta_22_21;

  LADSPA_Data   reverse_delta_23_22;
  unsigned long ul_delta_23_22;

  LADSPA_Data   reverse_delta_24_23;
  unsigned long ul_delta_24_23;

  LADSPA_Data   reverse_delta_25_24;
  unsigned long ul_delta_25_24;

  LADSPA_Data   reverse_delta_26_25;
  unsigned long ul_delta_26_25;

  LADSPA_Data   reverse_delta_27_26;
  unsigned long ul_delta_27_26;

  LADSPA_Data   reverse_delta_28_27;
  unsigned long ul_delta_28_27;

  LADSPA_Data   reverse_delta_29_28;
  unsigned long ul_delta_29_28;

  LADSPA_Data   reverse_delta_high_29;
  unsigned long ul_delta_high_29;

  // PERL_END

  LADSPA_Data         *window_for_input;
  LADSPA_Data         *db_table;
  unsigned long fifo_pos;
  } MChMBEq;

const LADSPA_Descriptor *ladspa_descriptor
  (
  unsigned long idx
  )
  {
  switch(idx)
    {
    case 0:
      fprintf(stderr, "ladspa_descriptor: returning MChMBEqDescriptor=%08lx\n", (unsigned long)MChMBEqDescriptor);
      return MChMBEqDescriptor;
    default:
      return NULL;
    }
  } // const LADSPA_Descriptor *ladspa_descriptor

static void activateMChMBEq(LADSPA_Handle instance)
  {
  fprintf(stderr, "activateMChMBEq: instance=%08lx\n", (unsigned long)instance);


  (*((MChMBEq *)instance)).fifo_pos = (*((MChMBEq *)instance)).fft_latency;
  } // static void activateMChMBEq(LADSPA_Handle instance)


static void cleanupMChMBEq(LADSPA_Handle instance)
  {
  MChMBEq *plugin_data = (MChMBEq *)instance;

  // PERL_BEGIN

  free((*plugin_data).out_fifo_L);
  free((*plugin_data).out_accum_L);
  free((*plugin_data).comp_L);
  free((*plugin_data).real_L);

  free((*plugin_data).input_L);
  free((*plugin_data).output_L);
  free((*plugin_data).in_fifo_L);


  free((*plugin_data).out_fifo_R);
  free((*plugin_data).out_accum_R);
  free((*plugin_data).comp_R);
  free((*plugin_data).real_R);

  free((*plugin_data).input_R);
  free((*plugin_data).output_R);
  free((*plugin_data).in_fifo_R);

  // PERL_END

  free((*plugin_data).window_for_input);
  free((*plugin_data).db_table);
  free((*plugin_data).cos_times_gain);
  free(instance);
  } // static void cleanupMChMBEq(LADSPA_Handle instance)


static void connectPortMChMBEq
  (
  LADSPA_Handle instance,
  unsigned long port,
  LADSPA_Data *data
  )
  {
  switch(port)
    {
    // PERL_BEGIN
    case MBEQ_BAND_LOW:
      (*(MChMBEq *)instance).band_gain_ptr_low = data;
      break;

    case MBEQ_BAND_00:
      (*(MChMBEq *)instance).band_gain_ptr_00 = data;
      break;

    case MBEQ_BAND_01:
      (*(MChMBEq *)instance).band_gain_ptr_01 = data;
      break;

    case MBEQ_BAND_02:
      (*(MChMBEq *)instance).band_gain_ptr_02 = data;
      break;

    case MBEQ_BAND_03:
      (*(MChMBEq *)instance).band_gain_ptr_03 = data;
      break;

    case MBEQ_BAND_04:
      (*(MChMBEq *)instance).band_gain_ptr_04 = data;
      break;

    case MBEQ_BAND_05:
      (*(MChMBEq *)instance).band_gain_ptr_05 = data;
      break;

    case MBEQ_BAND_06:
      (*(MChMBEq *)instance).band_gain_ptr_06 = data;
      break;

    case MBEQ_BAND_07:
      (*(MChMBEq *)instance).band_gain_ptr_07 = data;
      break;

    case MBEQ_BAND_08:
      (*(MChMBEq *)instance).band_gain_ptr_08 = data;
      break;

    case MBEQ_BAND_09:
      (*(MChMBEq *)instance).band_gain_ptr_09 = data;
      break;

    case MBEQ_BAND_10:
      (*(MChMBEq *)instance).band_gain_ptr_10 = data;
      break;

    case MBEQ_BAND_11:
      (*(MChMBEq *)instance).band_gain_ptr_11 = data;
      break;

    case MBEQ_BAND_12:
      (*(MChMBEq *)instance).band_gain_ptr_12 = data;
      break;

    case MBEQ_BAND_13:
      (*(MChMBEq *)instance).band_gain_ptr_13 = data;
      break;

    case MBEQ_BAND_14:
      (*(MChMBEq *)instance).band_gain_ptr_14 = data;
      break;

    case MBEQ_BAND_15:
      (*(MChMBEq *)instance).band_gain_ptr_15 = data;
      break;

    case MBEQ_BAND_16:
      (*(MChMBEq *)instance).band_gain_ptr_16 = data;
      break;

    case MBEQ_BAND_17:
      (*(MChMBEq *)instance).band_gain_ptr_17 = data;
      break;

    case MBEQ_BAND_18:
      (*(MChMBEq *)instance).band_gain_ptr_18 = data;
      break;

    case MBEQ_BAND_19:
      (*(MChMBEq *)instance).band_gain_ptr_19 = data;
      break;

    case MBEQ_BAND_20:
      (*(MChMBEq *)instance).band_gain_ptr_20 = data;
      break;

    case MBEQ_BAND_21:
      (*(MChMBEq *)instance).band_gain_ptr_21 = data;
      break;

    case MBEQ_BAND_22:
      (*(MChMBEq *)instance).band_gain_ptr_22 = data;
      break;

    case MBEQ_BAND_23:
      (*(MChMBEq *)instance).band_gain_ptr_23 = data;
      break;

    case MBEQ_BAND_24:
      (*(MChMBEq *)instance).band_gain_ptr_24 = data;
      break;

    case MBEQ_BAND_25:
      (*(MChMBEq *)instance).band_gain_ptr_25 = data;
      break;

    case MBEQ_BAND_26:
      (*(MChMBEq *)instance).band_gain_ptr_26 = data;
      break;

    case MBEQ_BAND_27:
      (*(MChMBEq *)instance).band_gain_ptr_27 = data;
      break;

    case MBEQ_BAND_28:
      (*(MChMBEq *)instance).band_gain_ptr_28 = data;
      break;

    case MBEQ_BAND_29:
      (*(MChMBEq *)instance).band_gain_ptr_29 = data;
      break;

    case MBEQ_BAND_HIGH:
      (*(MChMBEq *)instance).band_gain_ptr_high = data;
      break;

    case MBEQ_INPUT_L:
      (*(MChMBEq *)instance).input_L = data;
      break;
    case MBEQ_INPUT_R:
      (*(MChMBEq *)instance).input_R = data;
      break;
    case MBEQ_OUTPUT_L:
      (*(MChMBEq *)instance).output_L = data;
      break;
    case MBEQ_OUTPUT_R:
      (*(MChMBEq *)instance).output_R = data;
      break;
    // PERL_END
    } // switch(port)
  } // static void connectPortMChMBEq


static LADSPA_Handle instantiateMChMBEq
  (
  const LADSPA_Descriptor *descriptor,
  unsigned long s_rate
  )
  {
  MChMBEq *plugin_data;
  MALLOC_AND_CHECK(MChMBEq *, plugin_data, sizeof(MChMBEq), __LINE__)

  // PERL_BEGIN
  unsigned long min_necessary_fft_length =
  (unsigned long)((double)s_rate / (double)20);
  // PERL_END

 
  unsigned long fft_length;

  if(OVERSAMP < MIN_DESIRED_FFT_LENGTH)
    {
    fft_length = MIN_DESIRED_FFT_LENGTH;
    }
  else
    {
    fft_length = OVERSAMP;
    }

  while(fft_length <= min_necessary_fft_length)
    {
    fft_length *= 2;
    }

  unsigned long half_fft_length = fft_length / 2;
  unsigned long half_fft_length_minus_1  = half_fft_length - 1;

  (*plugin_data).fft_length = fft_length;
  (*plugin_data).half_fft_length = half_fft_length;
  (*plugin_data).half_fft_length_minus_1 = half_fft_length_minus_1;

  (*plugin_data).step_size = fft_length / OVERSAMP;

  (*plugin_data).fft_latency =
  fft_length - (*plugin_data).step_size;


  (*plugin_data).number_of_bytes_in_step =
  (*plugin_data).step_size * sizeof(LADSPA_Data);

  (*plugin_data).number_of_bytes_in_fft_length =
  fft_length * sizeof(LADSPA_Data);

  (*plugin_data).number_of_bytes_in_fft_latency =
  (*plugin_data).fft_latency * sizeof(LADSPA_Data);


  // PERL_BEGIN
  (*plugin_data).db_gain_low = (LADSPA_Data)0;

  (*plugin_data).db_gain_00 = (LADSPA_Data)0;

  (*plugin_data).db_gain_01 = (LADSPA_Data)0;

  (*plugin_data).db_gain_02 = (LADSPA_Data)0;

  (*plugin_data).db_gain_03 = (LADSPA_Data)0;

  (*plugin_data).db_gain_04 = (LADSPA_Data)0;

  (*plugin_data).db_gain_05 = (LADSPA_Data)0;

  (*plugin_data).db_gain_06 = (LADSPA_Data)0;

  (*plugin_data).db_gain_07 = (LADSPA_Data)0;

  (*plugin_data).db_gain_08 = (LADSPA_Data)0;

  (*plugin_data).db_gain_09 = (LADSPA_Data)0;

  (*plugin_data).db_gain_10 = (LADSPA_Data)0;

  (*plugin_data).db_gain_11 = (LADSPA_Data)0;

  (*plugin_data).db_gain_12 = (LADSPA_Data)0;

  (*plugin_data).db_gain_13 = (LADSPA_Data)0;

  (*plugin_data).db_gain_14 = (LADSPA_Data)0;

  (*plugin_data).db_gain_15 = (LADSPA_Data)0;

  (*plugin_data).db_gain_16 = (LADSPA_Data)0;

  (*plugin_data).db_gain_17 = (LADSPA_Data)0;

  (*plugin_data).db_gain_18 = (LADSPA_Data)0;

  (*plugin_data).db_gain_19 = (LADSPA_Data)0;

  (*plugin_data).db_gain_20 = (LADSPA_Data)0;

  (*plugin_data).db_gain_21 = (LADSPA_Data)0;

  (*plugin_data).db_gain_22 = (LADSPA_Data)0;

  (*plugin_data).db_gain_23 = (LADSPA_Data)0;

  (*plugin_data).db_gain_24 = (LADSPA_Data)0;

  (*plugin_data).db_gain_25 = (LADSPA_Data)0;

  (*plugin_data).db_gain_26 = (LADSPA_Data)0;

  (*plugin_data).db_gain_27 = (LADSPA_Data)0;

  (*plugin_data).db_gain_28 = (LADSPA_Data)0;

  (*plugin_data).db_gain_29 = (LADSPA_Data)0;

  (*plugin_data).db_gain_high = (LADSPA_Data)0;


  CALLOC_AND_CHECK(LADSPA_Data *, (*plugin_data).out_accum_L, fft_length * 2, sizeof(LADSPA_Data), __LINE__)
  CALLOC_AND_CHECK(LADSPA_Data *, (*plugin_data).out_fifo_L, fft_length, sizeof(LADSPA_Data), __LINE__)
  CALLOC_AND_CHECK(LADSPA_Data *, (*plugin_data).in_fifo_L, fft_length, sizeof(LADSPA_Data), __LINE__)
  CALLOC_AND_CHECK(fftw_real *, (*plugin_data).real_L, fft_length, sizeof(fftw_real), __LINE__)
  CALLOC_AND_CHECK(fftw_real *, (*plugin_data).comp_L, fft_length, sizeof(fftw_real), __LINE__)


  CALLOC_AND_CHECK(LADSPA_Data *, (*plugin_data).out_accum_R, fft_length * 2, sizeof(LADSPA_Data), __LINE__)
  CALLOC_AND_CHECK(LADSPA_Data *, (*plugin_data).out_fifo_R, fft_length, sizeof(LADSPA_Data), __LINE__)
  CALLOC_AND_CHECK(LADSPA_Data *, (*plugin_data).in_fifo_R, fft_length, sizeof(LADSPA_Data), __LINE__)
  CALLOC_AND_CHECK(fftw_real *, (*plugin_data).real_R, fft_length, sizeof(fftw_real), __LINE__)
  CALLOC_AND_CHECK(fftw_real *, (*plugin_data).comp_R, fft_length, sizeof(fftw_real), __LINE__)

  // PERL_END

  CALLOC_AND_CHECK(LADSPA_Data *, (*plugin_data).window_for_input, fft_length, sizeof(LADSPA_Data), __LINE__)
  CALLOC_AND_CHECK(LADSPA_Data *, (*plugin_data).cos_times_gain, half_fft_length, sizeof(LADSPA_Data), __LINE__)
  MALLOC_AND_CHECK(LADSPA_Data *, (*plugin_data).db_table, NUMBER_OF_DB_TABLE_ENTRIES * sizeof(LADSPA_Data), __LINE__)
	
#ifdef FFTW3
  // PERL_BEGIN
  (*plugin_data).plan_rc_L = fftwf_plan_r2r_1d(fft_length, (*plugin_data).real_L, (*plugin_data).comp_L, FFTW_R2HC, FFTW_MEASURE);
  (*plugin_data).plan_cr_L = fftwf_plan_r2r_1d(fft_length, (*plugin_data).comp_L, (*plugin_data).real_L, FFTW_HC2R, FFTW_MEASURE);
  (*plugin_data).plan_rc_R = fftwf_plan_r2r_1d(fft_length, (*plugin_data).real_R, (*plugin_data).comp_R, FFTW_R2HC, FFTW_MEASURE);
  (*plugin_data).plan_cr_R = fftwf_plan_r2r_1d(fft_length, (*plugin_data).comp_R, (*plugin_data).real_R, FFTW_HC2R, FFTW_MEASURE);
  // PERL_END
#else
  (*plugin_data).plan_rc = rfftw_create_plan(fft_length, FFTW_REAL_TO_COMPLEX, FFTW_ESTIMATE);
  (*plugin_data).plan_cr = rfftw_create_plan(fft_length, FFTW_COMPLEX_TO_REAL, FFTW_ESTIMATE);
#endif

    {
    // Create db->coefficient lookup table
    unsigned i;
    for(i=0; i < NUMBER_OF_DB_TABLE_ENTRIES; i++)
      {
      (*plugin_data).db_table[i] =
      (float)pow(10.0, ((DB_GAIN_PER_ENTRY * (double)i) + MIN_GAIN)/20.0);
      // fprintf(stderr, "(*plugin_data).db_table[%d]=%f\n", (int)i, (double)(*plugin_data).db_table[i]);
      }
    }

  // PERL_BEGIN
  (*plugin_data).bin_low = 0;


  // calculate bin <-> band stuff (band borders in terms of bins, band
  // deltas in terms of bins, pointers to band borders, etc.)

  (*plugin_data).bin_low = (unsigned long)0;

  (*plugin_data).bin_00 =
  (unsigned long)(((double)0.5) + ((double)20) * ((double)fft_length) / ((double)s_rate));

  (*plugin_data).ul_delta_00_low =
  (*plugin_data).bin_00 - (*plugin_data).bin_low;

  if(((unsigned)0) == (*plugin_data).ul_delta_00_low)
    {
    fprintf(stderr, "mbeq_119700: !!! ERROR !!! 20Hz band is too close to previous one (gets into the same FFT bin)\n");
    fprintf(stderr, "mbeq_119700: either change the frequency or increase number of point is FFT\n");
    exit(1);
    }

  (*plugin_data).reverse_delta_00_low =
  (LADSPA_Data)((double)1 / (double)(*plugin_data).ul_delta_00_low);

  //fprintf(stderr, "instantiateMChMBEq - __LINE__=%d (*plugin_data).bin_00=%d\n", __LINE__, (int)(*plugin_data).bin_00);
  fprintf
    (
    stderr,
    "instantiateMChMBEq :INFO: actual 00 band bin number: %d frequency: %GHz\n",
    (int)(*plugin_data).bin_00,
    ((double)(*plugin_data).bin_00) * ((double)s_rate) / ((double)fft_length)
    );
  (*plugin_data).bin_01 =
  (unsigned long)(((double)0.5) + ((double)28.2842712474619) * ((double)fft_length) / ((double)s_rate));

  (*plugin_data).ul_delta_01_00 =
  (*plugin_data).bin_01 - (*plugin_data).bin_00;

  if(((unsigned)0) == (*plugin_data).ul_delta_01_00)
    {
    fprintf(stderr, "mbeq_119700: !!! ERROR !!! 28.2842712474619Hz band is too close to previous one (gets into the same FFT bin)\n");
    fprintf(stderr, "mbeq_119700: either change the frequency or increase number of point is FFT\n");
    exit(1);
    }

  (*plugin_data).reverse_delta_01_00 =
  (LADSPA_Data)((double)1 / (double)(*plugin_data).ul_delta_01_00);

  //fprintf(stderr, "instantiateMChMBEq - __LINE__=%d (*plugin_data).bin_01=%d\n", __LINE__, (int)(*plugin_data).bin_01);
  fprintf
    (
    stderr,
    "instantiateMChMBEq :INFO: actual 01 band bin number: %d frequency: %GHz\n",
    (int)(*plugin_data).bin_01,
    ((double)(*plugin_data).bin_01) * ((double)s_rate) / ((double)fft_length)
    );
  (*plugin_data).bin_02 =
  (unsigned long)(((double)0.5) + ((double)40) * ((double)fft_length) / ((double)s_rate));

  (*plugin_data).ul_delta_02_01 =
  (*plugin_data).bin_02 - (*plugin_data).bin_01;

  if(((unsigned)0) == (*plugin_data).ul_delta_02_01)
    {
    fprintf(stderr, "mbeq_119700: !!! ERROR !!! 40Hz band is too close to previous one (gets into the same FFT bin)\n");
    fprintf(stderr, "mbeq_119700: either change the frequency or increase number of point is FFT\n");
    exit(1);
    }

  (*plugin_data).reverse_delta_02_01 =
  (LADSPA_Data)((double)1 / (double)(*plugin_data).ul_delta_02_01);

  //fprintf(stderr, "instantiateMChMBEq - __LINE__=%d (*plugin_data).bin_02=%d\n", __LINE__, (int)(*plugin_data).bin_02);
  fprintf
    (
    stderr,
    "instantiateMChMBEq :INFO: actual 02 band bin number: %d frequency: %GHz\n",
    (int)(*plugin_data).bin_02,
    ((double)(*plugin_data).bin_02) * ((double)s_rate) / ((double)fft_length)
    );
  (*plugin_data).bin_03 =
  (unsigned long)(((double)0.5) + ((double)50.3968419957949) * ((double)fft_length) / ((double)s_rate));

  (*plugin_data).ul_delta_03_02 =
  (*plugin_data).bin_03 - (*plugin_data).bin_02;

  if(((unsigned)0) == (*plugin_data).ul_delta_03_02)
    {
    fprintf(stderr, "mbeq_119700: !!! ERROR !!! 50.3968419957949Hz band is too close to previous one (gets into the same FFT bin)\n");
    fprintf(stderr, "mbeq_119700: either change the frequency or increase number of point is FFT\n");
    exit(1);
    }

  (*plugin_data).reverse_delta_03_02 =
  (LADSPA_Data)((double)1 / (double)(*plugin_data).ul_delta_03_02);

  //fprintf(stderr, "instantiateMChMBEq - __LINE__=%d (*plugin_data).bin_03=%d\n", __LINE__, (int)(*plugin_data).bin_03);
  fprintf
    (
    stderr,
    "instantiateMChMBEq :INFO: actual 03 band bin number: %d frequency: %GHz\n",
    (int)(*plugin_data).bin_03,
    ((double)(*plugin_data).bin_03) * ((double)s_rate) / ((double)fft_length)
    );
  (*plugin_data).bin_04 =
  (unsigned long)(((double)0.5) + ((double)63.496042078728) * ((double)fft_length) / ((double)s_rate));

  (*plugin_data).ul_delta_04_03 =
  (*plugin_data).bin_04 - (*plugin_data).bin_03;

  if(((unsigned)0) == (*plugin_data).ul_delta_04_03)
    {
    fprintf(stderr, "mbeq_119700: !!! ERROR !!! 63.496042078728Hz band is too close to previous one (gets into the same FFT bin)\n");
    fprintf(stderr, "mbeq_119700: either change the frequency or increase number of point is FFT\n");
    exit(1);
    }

  (*plugin_data).reverse_delta_04_03 =
  (LADSPA_Data)((double)1 / (double)(*plugin_data).ul_delta_04_03);

  //fprintf(stderr, "instantiateMChMBEq - __LINE__=%d (*plugin_data).bin_04=%d\n", __LINE__, (int)(*plugin_data).bin_04);
  fprintf
    (
    stderr,
    "instantiateMChMBEq :INFO: actual 04 band bin number: %d frequency: %GHz\n",
    (int)(*plugin_data).bin_04,
    ((double)(*plugin_data).bin_04) * ((double)s_rate) / ((double)fft_length)
    );
  (*plugin_data).bin_05 =
  (unsigned long)(((double)0.5) + ((double)80) * ((double)fft_length) / ((double)s_rate));

  (*plugin_data).ul_delta_05_04 =
  (*plugin_data).bin_05 - (*plugin_data).bin_04;

  if(((unsigned)0) == (*plugin_data).ul_delta_05_04)
    {
    fprintf(stderr, "mbeq_119700: !!! ERROR !!! 80Hz band is too close to previous one (gets into the same FFT bin)\n");
    fprintf(stderr, "mbeq_119700: either change the frequency or increase number of point is FFT\n");
    exit(1);
    }

  (*plugin_data).reverse_delta_05_04 =
  (LADSPA_Data)((double)1 / (double)(*plugin_data).ul_delta_05_04);

  //fprintf(stderr, "instantiateMChMBEq - __LINE__=%d (*plugin_data).bin_05=%d\n", __LINE__, (int)(*plugin_data).bin_05);
  fprintf
    (
    stderr,
    "instantiateMChMBEq :INFO: actual 05 band bin number: %d frequency: %GHz\n",
    (int)(*plugin_data).bin_05,
    ((double)(*plugin_data).bin_05) * ((double)s_rate) / ((double)fft_length)
    );
  (*plugin_data).bin_06 =
  (unsigned long)(((double)0.5) + ((double)100.79368399159) * ((double)fft_length) / ((double)s_rate));

  (*plugin_data).ul_delta_06_05 =
  (*plugin_data).bin_06 - (*plugin_data).bin_05;

  if(((unsigned)0) == (*plugin_data).ul_delta_06_05)
    {
    fprintf(stderr, "mbeq_119700: !!! ERROR !!! 100.79368399159Hz band is too close to previous one (gets into the same FFT bin)\n");
    fprintf(stderr, "mbeq_119700: either change the frequency or increase number of point is FFT\n");
    exit(1);
    }

  (*plugin_data).reverse_delta_06_05 =
  (LADSPA_Data)((double)1 / (double)(*plugin_data).ul_delta_06_05);

  //fprintf(stderr, "instantiateMChMBEq - __LINE__=%d (*plugin_data).bin_06=%d\n", __LINE__, (int)(*plugin_data).bin_06);
  fprintf
    (
    stderr,
    "instantiateMChMBEq :INFO: actual 06 band bin number: %d frequency: %GHz\n",
    (int)(*plugin_data).bin_06,
    ((double)(*plugin_data).bin_06) * ((double)s_rate) / ((double)fft_length)
    );
  (*plugin_data).bin_07 =
  (unsigned long)(((double)0.5) + ((double)126.992084157456) * ((double)fft_length) / ((double)s_rate));

  (*plugin_data).ul_delta_07_06 =
  (*plugin_data).bin_07 - (*plugin_data).bin_06;

  if(((unsigned)0) == (*plugin_data).ul_delta_07_06)
    {
    fprintf(stderr, "mbeq_119700: !!! ERROR !!! 126.992084157456Hz band is too close to previous one (gets into the same FFT bin)\n");
    fprintf(stderr, "mbeq_119700: either change the frequency or increase number of point is FFT\n");
    exit(1);
    }

  (*plugin_data).reverse_delta_07_06 =
  (LADSPA_Data)((double)1 / (double)(*plugin_data).ul_delta_07_06);

  //fprintf(stderr, "instantiateMChMBEq - __LINE__=%d (*plugin_data).bin_07=%d\n", __LINE__, (int)(*plugin_data).bin_07);
  fprintf
    (
    stderr,
    "instantiateMChMBEq :INFO: actual 07 band bin number: %d frequency: %GHz\n",
    (int)(*plugin_data).bin_07,
    ((double)(*plugin_data).bin_07) * ((double)s_rate) / ((double)fft_length)
    );
  (*plugin_data).bin_08 =
  (unsigned long)(((double)0.5) + ((double)160) * ((double)fft_length) / ((double)s_rate));

  (*plugin_data).ul_delta_08_07 =
  (*plugin_data).bin_08 - (*plugin_data).bin_07;

  if(((unsigned)0) == (*plugin_data).ul_delta_08_07)
    {
    fprintf(stderr, "mbeq_119700: !!! ERROR !!! 160Hz band is too close to previous one (gets into the same FFT bin)\n");
    fprintf(stderr, "mbeq_119700: either change the frequency or increase number of point is FFT\n");
    exit(1);
    }

  (*plugin_data).reverse_delta_08_07 =
  (LADSPA_Data)((double)1 / (double)(*plugin_data).ul_delta_08_07);

  //fprintf(stderr, "instantiateMChMBEq - __LINE__=%d (*plugin_data).bin_08=%d\n", __LINE__, (int)(*plugin_data).bin_08);
  fprintf
    (
    stderr,
    "instantiateMChMBEq :INFO: actual 08 band bin number: %d frequency: %GHz\n",
    (int)(*plugin_data).bin_08,
    ((double)(*plugin_data).bin_08) * ((double)s_rate) / ((double)fft_length)
    );
  (*plugin_data).bin_09 =
  (unsigned long)(((double)0.5) + ((double)201.58736798318) * ((double)fft_length) / ((double)s_rate));

  (*plugin_data).ul_delta_09_08 =
  (*plugin_data).bin_09 - (*plugin_data).bin_08;

  if(((unsigned)0) == (*plugin_data).ul_delta_09_08)
    {
    fprintf(stderr, "mbeq_119700: !!! ERROR !!! 201.58736798318Hz band is too close to previous one (gets into the same FFT bin)\n");
    fprintf(stderr, "mbeq_119700: either change the frequency or increase number of point is FFT\n");
    exit(1);
    }

  (*plugin_data).reverse_delta_09_08 =
  (LADSPA_Data)((double)1 / (double)(*plugin_data).ul_delta_09_08);

  //fprintf(stderr, "instantiateMChMBEq - __LINE__=%d (*plugin_data).bin_09=%d\n", __LINE__, (int)(*plugin_data).bin_09);
  fprintf
    (
    stderr,
    "instantiateMChMBEq :INFO: actual 09 band bin number: %d frequency: %GHz\n",
    (int)(*plugin_data).bin_09,
    ((double)(*plugin_data).bin_09) * ((double)s_rate) / ((double)fft_length)
    );
  (*plugin_data).bin_10 =
  (unsigned long)(((double)0.5) + ((double)253.984168314912) * ((double)fft_length) / ((double)s_rate));

  (*plugin_data).ul_delta_10_09 =
  (*plugin_data).bin_10 - (*plugin_data).bin_09;

  if(((unsigned)0) == (*plugin_data).ul_delta_10_09)
    {
    fprintf(stderr, "mbeq_119700: !!! ERROR !!! 253.984168314912Hz band is too close to previous one (gets into the same FFT bin)\n");
    fprintf(stderr, "mbeq_119700: either change the frequency or increase number of point is FFT\n");
    exit(1);
    }

  (*plugin_data).reverse_delta_10_09 =
  (LADSPA_Data)((double)1 / (double)(*plugin_data).ul_delta_10_09);

  //fprintf(stderr, "instantiateMChMBEq - __LINE__=%d (*plugin_data).bin_10=%d\n", __LINE__, (int)(*plugin_data).bin_10);
  fprintf
    (
    stderr,
    "instantiateMChMBEq :INFO: actual 10 band bin number: %d frequency: %GHz\n",
    (int)(*plugin_data).bin_10,
    ((double)(*plugin_data).bin_10) * ((double)s_rate) / ((double)fft_length)
    );
  (*plugin_data).bin_11 =
  (unsigned long)(((double)0.5) + ((double)320) * ((double)fft_length) / ((double)s_rate));

  (*plugin_data).ul_delta_11_10 =
  (*plugin_data).bin_11 - (*plugin_data).bin_10;

  if(((unsigned)0) == (*plugin_data).ul_delta_11_10)
    {
    fprintf(stderr, "mbeq_119700: !!! ERROR !!! 320Hz band is too close to previous one (gets into the same FFT bin)\n");
    fprintf(stderr, "mbeq_119700: either change the frequency or increase number of point is FFT\n");
    exit(1);
    }

  (*plugin_data).reverse_delta_11_10 =
  (LADSPA_Data)((double)1 / (double)(*plugin_data).ul_delta_11_10);

  //fprintf(stderr, "instantiateMChMBEq - __LINE__=%d (*plugin_data).bin_11=%d\n", __LINE__, (int)(*plugin_data).bin_11);
  fprintf
    (
    stderr,
    "instantiateMChMBEq :INFO: actual 11 band bin number: %d frequency: %GHz\n",
    (int)(*plugin_data).bin_11,
    ((double)(*plugin_data).bin_11) * ((double)s_rate) / ((double)fft_length)
    );
  (*plugin_data).bin_12 =
  (unsigned long)(((double)0.5) + ((double)403.174735966359) * ((double)fft_length) / ((double)s_rate));

  (*plugin_data).ul_delta_12_11 =
  (*plugin_data).bin_12 - (*plugin_data).bin_11;

  if(((unsigned)0) == (*plugin_data).ul_delta_12_11)
    {
    fprintf(stderr, "mbeq_119700: !!! ERROR !!! 403.174735966359Hz band is too close to previous one (gets into the same FFT bin)\n");
    fprintf(stderr, "mbeq_119700: either change the frequency or increase number of point is FFT\n");
    exit(1);
    }

  (*plugin_data).reverse_delta_12_11 =
  (LADSPA_Data)((double)1 / (double)(*plugin_data).ul_delta_12_11);

  //fprintf(stderr, "instantiateMChMBEq - __LINE__=%d (*plugin_data).bin_12=%d\n", __LINE__, (int)(*plugin_data).bin_12);
  fprintf
    (
    stderr,
    "instantiateMChMBEq :INFO: actual 12 band bin number: %d frequency: %GHz\n",
    (int)(*plugin_data).bin_12,
    ((double)(*plugin_data).bin_12) * ((double)s_rate) / ((double)fft_length)
    );
  (*plugin_data).bin_13 =
  (unsigned long)(((double)0.5) + ((double)507.968336629824) * ((double)fft_length) / ((double)s_rate));

  (*plugin_data).ul_delta_13_12 =
  (*plugin_data).bin_13 - (*plugin_data).bin_12;

  if(((unsigned)0) == (*plugin_data).ul_delta_13_12)
    {
    fprintf(stderr, "mbeq_119700: !!! ERROR !!! 507.968336629824Hz band is too close to previous one (gets into the same FFT bin)\n");
    fprintf(stderr, "mbeq_119700: either change the frequency or increase number of point is FFT\n");
    exit(1);
    }

  (*plugin_data).reverse_delta_13_12 =
  (LADSPA_Data)((double)1 / (double)(*plugin_data).ul_delta_13_12);

  //fprintf(stderr, "instantiateMChMBEq - __LINE__=%d (*plugin_data).bin_13=%d\n", __LINE__, (int)(*plugin_data).bin_13);
  fprintf
    (
    stderr,
    "instantiateMChMBEq :INFO: actual 13 band bin number: %d frequency: %GHz\n",
    (int)(*plugin_data).bin_13,
    ((double)(*plugin_data).bin_13) * ((double)s_rate) / ((double)fft_length)
    );
  (*plugin_data).bin_14 =
  (unsigned long)(((double)0.5) + ((double)640) * ((double)fft_length) / ((double)s_rate));

  (*plugin_data).ul_delta_14_13 =
  (*plugin_data).bin_14 - (*plugin_data).bin_13;

  if(((unsigned)0) == (*plugin_data).ul_delta_14_13)
    {
    fprintf(stderr, "mbeq_119700: !!! ERROR !!! 640Hz band is too close to previous one (gets into the same FFT bin)\n");
    fprintf(stderr, "mbeq_119700: either change the frequency or increase number of point is FFT\n");
    exit(1);
    }

  (*plugin_data).reverse_delta_14_13 =
  (LADSPA_Data)((double)1 / (double)(*plugin_data).ul_delta_14_13);

  //fprintf(stderr, "instantiateMChMBEq - __LINE__=%d (*plugin_data).bin_14=%d\n", __LINE__, (int)(*plugin_data).bin_14);
  fprintf
    (
    stderr,
    "instantiateMChMBEq :INFO: actual 14 band bin number: %d frequency: %GHz\n",
    (int)(*plugin_data).bin_14,
    ((double)(*plugin_data).bin_14) * ((double)s_rate) / ((double)fft_length)
    );
  (*plugin_data).bin_15 =
  (unsigned long)(((double)0.5) + ((double)806.349471932719) * ((double)fft_length) / ((double)s_rate));

  (*plugin_data).ul_delta_15_14 =
  (*plugin_data).bin_15 - (*plugin_data).bin_14;

  if(((unsigned)0) == (*plugin_data).ul_delta_15_14)
    {
    fprintf(stderr, "mbeq_119700: !!! ERROR !!! 806.349471932719Hz band is too close to previous one (gets into the same FFT bin)\n");
    fprintf(stderr, "mbeq_119700: either change the frequency or increase number of point is FFT\n");
    exit(1);
    }

  (*plugin_data).reverse_delta_15_14 =
  (LADSPA_Data)((double)1 / (double)(*plugin_data).ul_delta_15_14);

  //fprintf(stderr, "instantiateMChMBEq - __LINE__=%d (*plugin_data).bin_15=%d\n", __LINE__, (int)(*plugin_data).bin_15);
  fprintf
    (
    stderr,
    "instantiateMChMBEq :INFO: actual 15 band bin number: %d frequency: %GHz\n",
    (int)(*plugin_data).bin_15,
    ((double)(*plugin_data).bin_15) * ((double)s_rate) / ((double)fft_length)
    );
  (*plugin_data).bin_16 =
  (unsigned long)(((double)0.5) + ((double)1015.93667325965) * ((double)fft_length) / ((double)s_rate));

  (*plugin_data).ul_delta_16_15 =
  (*plugin_data).bin_16 - (*plugin_data).bin_15;

  if(((unsigned)0) == (*plugin_data).ul_delta_16_15)
    {
    fprintf(stderr, "mbeq_119700: !!! ERROR !!! 1015.93667325965Hz band is too close to previous one (gets into the same FFT bin)\n");
    fprintf(stderr, "mbeq_119700: either change the frequency or increase number of point is FFT\n");
    exit(1);
    }

  (*plugin_data).reverse_delta_16_15 =
  (LADSPA_Data)((double)1 / (double)(*plugin_data).ul_delta_16_15);

  //fprintf(stderr, "instantiateMChMBEq - __LINE__=%d (*plugin_data).bin_16=%d\n", __LINE__, (int)(*plugin_data).bin_16);
  fprintf
    (
    stderr,
    "instantiateMChMBEq :INFO: actual 16 band bin number: %d frequency: %GHz\n",
    (int)(*plugin_data).bin_16,
    ((double)(*plugin_data).bin_16) * ((double)s_rate) / ((double)fft_length)
    );
  (*plugin_data).bin_17 =
  (unsigned long)(((double)0.5) + ((double)1280) * ((double)fft_length) / ((double)s_rate));

  (*plugin_data).ul_delta_17_16 =
  (*plugin_data).bin_17 - (*plugin_data).bin_16;

  if(((unsigned)0) == (*plugin_data).ul_delta_17_16)
    {
    fprintf(stderr, "mbeq_119700: !!! ERROR !!! 1280Hz band is too close to previous one (gets into the same FFT bin)\n");
    fprintf(stderr, "mbeq_119700: either change the frequency or increase number of point is FFT\n");
    exit(1);
    }

  (*plugin_data).reverse_delta_17_16 =
  (LADSPA_Data)((double)1 / (double)(*plugin_data).ul_delta_17_16);

  //fprintf(stderr, "instantiateMChMBEq - __LINE__=%d (*plugin_data).bin_17=%d\n", __LINE__, (int)(*plugin_data).bin_17);
  fprintf
    (
    stderr,
    "instantiateMChMBEq :INFO: actual 17 band bin number: %d frequency: %GHz\n",
    (int)(*plugin_data).bin_17,
    ((double)(*plugin_data).bin_17) * ((double)s_rate) / ((double)fft_length)
    );
  (*plugin_data).bin_18 =
  (unsigned long)(((double)0.5) + ((double)1612.69894386544) * ((double)fft_length) / ((double)s_rate));

  (*plugin_data).ul_delta_18_17 =
  (*plugin_data).bin_18 - (*plugin_data).bin_17;

  if(((unsigned)0) == (*plugin_data).ul_delta_18_17)
    {
    fprintf(stderr, "mbeq_119700: !!! ERROR !!! 1612.69894386544Hz band is too close to previous one (gets into the same FFT bin)\n");
    fprintf(stderr, "mbeq_119700: either change the frequency or increase number of point is FFT\n");
    exit(1);
    }

  (*plugin_data).reverse_delta_18_17 =
  (LADSPA_Data)((double)1 / (double)(*plugin_data).ul_delta_18_17);

  //fprintf(stderr, "instantiateMChMBEq - __LINE__=%d (*plugin_data).bin_18=%d\n", __LINE__, (int)(*plugin_data).bin_18);
  fprintf
    (
    stderr,
    "instantiateMChMBEq :INFO: actual 18 band bin number: %d frequency: %GHz\n",
    (int)(*plugin_data).bin_18,
    ((double)(*plugin_data).bin_18) * ((double)s_rate) / ((double)fft_length)
    );
  (*plugin_data).bin_19 =
  (unsigned long)(((double)0.5) + ((double)2031.8733465193) * ((double)fft_length) / ((double)s_rate));

  (*plugin_data).ul_delta_19_18 =
  (*plugin_data).bin_19 - (*plugin_data).bin_18;

  if(((unsigned)0) == (*plugin_data).ul_delta_19_18)
    {
    fprintf(stderr, "mbeq_119700: !!! ERROR !!! 2031.8733465193Hz band is too close to previous one (gets into the same FFT bin)\n");
    fprintf(stderr, "mbeq_119700: either change the frequency or increase number of point is FFT\n");
    exit(1);
    }

  (*plugin_data).reverse_delta_19_18 =
  (LADSPA_Data)((double)1 / (double)(*plugin_data).ul_delta_19_18);

  //fprintf(stderr, "instantiateMChMBEq - __LINE__=%d (*plugin_data).bin_19=%d\n", __LINE__, (int)(*plugin_data).bin_19);
  fprintf
    (
    stderr,
    "instantiateMChMBEq :INFO: actual 19 band bin number: %d frequency: %GHz\n",
    (int)(*plugin_data).bin_19,
    ((double)(*plugin_data).bin_19) * ((double)s_rate) / ((double)fft_length)
    );
  (*plugin_data).bin_20 =
  (unsigned long)(((double)0.5) + ((double)2560) * ((double)fft_length) / ((double)s_rate));

  (*plugin_data).ul_delta_20_19 =
  (*plugin_data).bin_20 - (*plugin_data).bin_19;

  if(((unsigned)0) == (*plugin_data).ul_delta_20_19)
    {
    fprintf(stderr, "mbeq_119700: !!! ERROR !!! 2560Hz band is too close to previous one (gets into the same FFT bin)\n");
    fprintf(stderr, "mbeq_119700: either change the frequency or increase number of point is FFT\n");
    exit(1);
    }

  (*plugin_data).reverse_delta_20_19 =
  (LADSPA_Data)((double)1 / (double)(*plugin_data).ul_delta_20_19);

  //fprintf(stderr, "instantiateMChMBEq - __LINE__=%d (*plugin_data).bin_20=%d\n", __LINE__, (int)(*plugin_data).bin_20);
  fprintf
    (
    stderr,
    "instantiateMChMBEq :INFO: actual 20 band bin number: %d frequency: %GHz\n",
    (int)(*plugin_data).bin_20,
    ((double)(*plugin_data).bin_20) * ((double)s_rate) / ((double)fft_length)
    );
  (*plugin_data).bin_21 =
  (unsigned long)(((double)0.5) + ((double)3225.39788773088) * ((double)fft_length) / ((double)s_rate));

  (*plugin_data).ul_delta_21_20 =
  (*plugin_data).bin_21 - (*plugin_data).bin_20;

  if(((unsigned)0) == (*plugin_data).ul_delta_21_20)
    {
    fprintf(stderr, "mbeq_119700: !!! ERROR !!! 3225.39788773088Hz band is too close to previous one (gets into the same FFT bin)\n");
    fprintf(stderr, "mbeq_119700: either change the frequency or increase number of point is FFT\n");
    exit(1);
    }

  (*plugin_data).reverse_delta_21_20 =
  (LADSPA_Data)((double)1 / (double)(*plugin_data).ul_delta_21_20);

  //fprintf(stderr, "instantiateMChMBEq - __LINE__=%d (*plugin_data).bin_21=%d\n", __LINE__, (int)(*plugin_data).bin_21);
  fprintf
    (
    stderr,
    "instantiateMChMBEq :INFO: actual 21 band bin number: %d frequency: %GHz\n",
    (int)(*plugin_data).bin_21,
    ((double)(*plugin_data).bin_21) * ((double)s_rate) / ((double)fft_length)
    );
  (*plugin_data).bin_22 =
  (unsigned long)(((double)0.5) + ((double)4063.74669303859) * ((double)fft_length) / ((double)s_rate));

  (*plugin_data).ul_delta_22_21 =
  (*plugin_data).bin_22 - (*plugin_data).bin_21;

  if(((unsigned)0) == (*plugin_data).ul_delta_22_21)
    {
    fprintf(stderr, "mbeq_119700: !!! ERROR !!! 4063.74669303859Hz band is too close to previous one (gets into the same FFT bin)\n");
    fprintf(stderr, "mbeq_119700: either change the frequency or increase number of point is FFT\n");
    exit(1);
    }

  (*plugin_data).reverse_delta_22_21 =
  (LADSPA_Data)((double)1 / (double)(*plugin_data).ul_delta_22_21);

  //fprintf(stderr, "instantiateMChMBEq - __LINE__=%d (*plugin_data).bin_22=%d\n", __LINE__, (int)(*plugin_data).bin_22);
  fprintf
    (
    stderr,
    "instantiateMChMBEq :INFO: actual 22 band bin number: %d frequency: %GHz\n",
    (int)(*plugin_data).bin_22,
    ((double)(*plugin_data).bin_22) * ((double)s_rate) / ((double)fft_length)
    );
  (*plugin_data).bin_23 =
  (unsigned long)(((double)0.5) + ((double)5120) * ((double)fft_length) / ((double)s_rate));

  (*plugin_data).ul_delta_23_22 =
  (*plugin_data).bin_23 - (*plugin_data).bin_22;

  if(((unsigned)0) == (*plugin_data).ul_delta_23_22)
    {
    fprintf(stderr, "mbeq_119700: !!! ERROR !!! 5120Hz band is too close to previous one (gets into the same FFT bin)\n");
    fprintf(stderr, "mbeq_119700: either change the frequency or increase number of point is FFT\n");
    exit(1);
    }

  (*plugin_data).reverse_delta_23_22 =
  (LADSPA_Data)((double)1 / (double)(*plugin_data).ul_delta_23_22);

  //fprintf(stderr, "instantiateMChMBEq - __LINE__=%d (*plugin_data).bin_23=%d\n", __LINE__, (int)(*plugin_data).bin_23);
  fprintf
    (
    stderr,
    "instantiateMChMBEq :INFO: actual 23 band bin number: %d frequency: %GHz\n",
    (int)(*plugin_data).bin_23,
    ((double)(*plugin_data).bin_23) * ((double)s_rate) / ((double)fft_length)
    );
  (*plugin_data).bin_24 =
  (unsigned long)(((double)0.5) + ((double)6450.79577546175) * ((double)fft_length) / ((double)s_rate));

  (*plugin_data).ul_delta_24_23 =
  (*plugin_data).bin_24 - (*plugin_data).bin_23;

  if(((unsigned)0) == (*plugin_data).ul_delta_24_23)
    {
    fprintf(stderr, "mbeq_119700: !!! ERROR !!! 6450.79577546175Hz band is too close to previous one (gets into the same FFT bin)\n");
    fprintf(stderr, "mbeq_119700: either change the frequency or increase number of point is FFT\n");
    exit(1);
    }

  (*plugin_data).reverse_delta_24_23 =
  (LADSPA_Data)((double)1 / (double)(*plugin_data).ul_delta_24_23);

  //fprintf(stderr, "instantiateMChMBEq - __LINE__=%d (*plugin_data).bin_24=%d\n", __LINE__, (int)(*plugin_data).bin_24);
  fprintf
    (
    stderr,
    "instantiateMChMBEq :INFO: actual 24 band bin number: %d frequency: %GHz\n",
    (int)(*plugin_data).bin_24,
    ((double)(*plugin_data).bin_24) * ((double)s_rate) / ((double)fft_length)
    );
  (*plugin_data).bin_25 =
  (unsigned long)(((double)0.5) + ((double)8127.49338607718) * ((double)fft_length) / ((double)s_rate));

  (*plugin_data).ul_delta_25_24 =
  (*plugin_data).bin_25 - (*plugin_data).bin_24;

  if(((unsigned)0) == (*plugin_data).ul_delta_25_24)
    {
    fprintf(stderr, "mbeq_119700: !!! ERROR !!! 8127.49338607718Hz band is too close to previous one (gets into the same FFT bin)\n");
    fprintf(stderr, "mbeq_119700: either change the frequency or increase number of point is FFT\n");
    exit(1);
    }

  (*plugin_data).reverse_delta_25_24 =
  (LADSPA_Data)((double)1 / (double)(*plugin_data).ul_delta_25_24);

  //fprintf(stderr, "instantiateMChMBEq - __LINE__=%d (*plugin_data).bin_25=%d\n", __LINE__, (int)(*plugin_data).bin_25);
  fprintf
    (
    stderr,
    "instantiateMChMBEq :INFO: actual 25 band bin number: %d frequency: %GHz\n",
    (int)(*plugin_data).bin_25,
    ((double)(*plugin_data).bin_25) * ((double)s_rate) / ((double)fft_length)
    );
  (*plugin_data).bin_26 =
  (unsigned long)(((double)0.5) + ((double)10240) * ((double)fft_length) / ((double)s_rate));

  (*plugin_data).ul_delta_26_25 =
  (*plugin_data).bin_26 - (*plugin_data).bin_25;

  if(((unsigned)0) == (*plugin_data).ul_delta_26_25)
    {
    fprintf(stderr, "mbeq_119700: !!! ERROR !!! 10240Hz band is too close to previous one (gets into the same FFT bin)\n");
    fprintf(stderr, "mbeq_119700: either change the frequency or increase number of point is FFT\n");
    exit(1);
    }

  (*plugin_data).reverse_delta_26_25 =
  (LADSPA_Data)((double)1 / (double)(*plugin_data).ul_delta_26_25);

  //fprintf(stderr, "instantiateMChMBEq - __LINE__=%d (*plugin_data).bin_26=%d\n", __LINE__, (int)(*plugin_data).bin_26);
  fprintf
    (
    stderr,
    "instantiateMChMBEq :INFO: actual 26 band bin number: %d frequency: %GHz\n",
    (int)(*plugin_data).bin_26,
    ((double)(*plugin_data).bin_26) * ((double)s_rate) / ((double)fft_length)
    );
  (*plugin_data).bin_27 =
  (unsigned long)(((double)0.5) + ((double)12901.5915509235) * ((double)fft_length) / ((double)s_rate));

  (*plugin_data).ul_delta_27_26 =
  (*plugin_data).bin_27 - (*plugin_data).bin_26;

  if(((unsigned)0) == (*plugin_data).ul_delta_27_26)
    {
    fprintf(stderr, "mbeq_119700: !!! ERROR !!! 12901.5915509235Hz band is too close to previous one (gets into the same FFT bin)\n");
    fprintf(stderr, "mbeq_119700: either change the frequency or increase number of point is FFT\n");
    exit(1);
    }

  (*plugin_data).reverse_delta_27_26 =
  (LADSPA_Data)((double)1 / (double)(*plugin_data).ul_delta_27_26);

  //fprintf(stderr, "instantiateMChMBEq - __LINE__=%d (*plugin_data).bin_27=%d\n", __LINE__, (int)(*plugin_data).bin_27);
  fprintf
    (
    stderr,
    "instantiateMChMBEq :INFO: actual 27 band bin number: %d frequency: %GHz\n",
    (int)(*plugin_data).bin_27,
    ((double)(*plugin_data).bin_27) * ((double)s_rate) / ((double)fft_length)
    );
  (*plugin_data).bin_28 =
  (unsigned long)(((double)0.5) + ((double)16254.9867721544) * ((double)fft_length) / ((double)s_rate));

  (*plugin_data).ul_delta_28_27 =
  (*plugin_data).bin_28 - (*plugin_data).bin_27;

  if(((unsigned)0) == (*plugin_data).ul_delta_28_27)
    {
    fprintf(stderr, "mbeq_119700: !!! ERROR !!! 16254.9867721544Hz band is too close to previous one (gets into the same FFT bin)\n");
    fprintf(stderr, "mbeq_119700: either change the frequency or increase number of point is FFT\n");
    exit(1);
    }

  (*plugin_data).reverse_delta_28_27 =
  (LADSPA_Data)((double)1 / (double)(*plugin_data).ul_delta_28_27);

  //fprintf(stderr, "instantiateMChMBEq - __LINE__=%d (*plugin_data).bin_28=%d\n", __LINE__, (int)(*plugin_data).bin_28);
  fprintf
    (
    stderr,
    "instantiateMChMBEq :INFO: actual 28 band bin number: %d frequency: %GHz\n",
    (int)(*plugin_data).bin_28,
    ((double)(*plugin_data).bin_28) * ((double)s_rate) / ((double)fft_length)
    );
  (*plugin_data).bin_29 =
  (unsigned long)(((double)0.5) + ((double)20480) * ((double)fft_length) / ((double)s_rate));

  (*plugin_data).ul_delta_29_28 =
  (*plugin_data).bin_29 - (*plugin_data).bin_28;

  if(((unsigned)0) == (*plugin_data).ul_delta_29_28)
    {
    fprintf(stderr, "mbeq_119700: !!! ERROR !!! 20480Hz band is too close to previous one (gets into the same FFT bin)\n");
    fprintf(stderr, "mbeq_119700: either change the frequency or increase number of point is FFT\n");
    exit(1);
    }

  (*plugin_data).reverse_delta_29_28 =
  (LADSPA_Data)((double)1 / (double)(*plugin_data).ul_delta_29_28);

  //fprintf(stderr, "instantiateMChMBEq - __LINE__=%d (*plugin_data).bin_29=%d\n", __LINE__, (int)(*plugin_data).bin_29);
  fprintf
    (
    stderr,
    "instantiateMChMBEq :INFO: actual 29 band bin number: %d frequency: %GHz\n",
    (int)(*plugin_data).bin_29,
    ((double)(*plugin_data).bin_29) * ((double)s_rate) / ((double)fft_length)
    );
  (*plugin_data).bin_high = fft_length - 1;

  (*plugin_data).ul_delta_high_29 =
  (*plugin_data).bin_high
- (*plugin_data).bin_29;

  (*plugin_data).reverse_delta_high_29 =
  (float)((double)1 / (double)(*plugin_data).ul_delta_high_29);

  // PERL_END

    {
    float *cos_times_gain_ptr = (*plugin_data).cos_times_gain;
    unsigned bin_number = 0;

    fprintf(stderr, "instantiateMChMBEq - __LINE__=%d (*plugin_data).half_fft_length=%d\n", __LINE__, (int)(*plugin_data).half_fft_length);

    while(bin_number++ < (*plugin_data).half_fft_length)
      {
      *cos_times_gain_ptr++ = (LADSPA_Data)(1);
      } // while(bin_number++ < (*plugin_data).half_fft_length)
    }

    {
    // Create raised cosine window table

    unsigned i;
    for(i=0; i < fft_length; i++)
      {
      (*plugin_data).window_for_input[i] =
      (LADSPA_Data)(
                     (
                       (double)1.0
                     - cos
                         (
                           (double)2.0
                         * M_PI
                         * (double)i
                         / (double)fft_length
                         )
                     )/((double)2 * (double)fft_length)
                   );

      //fprintf(stderr, "(*plugin_data).window_for_input[%d]=%G\n", (int)i, (*plugin_data).window_for_input[i]);
      }
    }

  (*plugin_data).fifo_pos = (*plugin_data).fft_latency;

  return (LADSPA_Handle)plugin_data;
  } // static LADSPA_Handle instantiateMChMBEq



static void runMChMBEq(LADSPA_Handle instance, unsigned long sample_count)
  {
  MChMBEq *plugin_data = (MChMBEq *)instance;

  float *db_table = (*plugin_data).db_table;
  long fifo_pos = (*plugin_data).fifo_pos;
  float *window_for_input = (*plugin_data).window_for_input;
  float *cos_times_gain = (*plugin_data).cos_times_gain;

  unsigned long fft_length = (*plugin_data).fft_length;
  unsigned long half_fft_length = (*plugin_data).half_fft_length;

  // PERL_BEGIN
  const LADSPA_Data * const input_L = (*plugin_data).input_L;
  LADSPA_Data * const output_L = (*plugin_data).output_L;
  fftw_real *comp_L = (*plugin_data).comp_L;
  fftw_real *real_L = (*plugin_data).real_L;
  LADSPA_Data *in_fifo_L = (*plugin_data).in_fifo_L;
  LADSPA_Data *out_accum_L = (*plugin_data).out_accum_L;
  LADSPA_Data *out_fifo_L = (*plugin_data).out_fifo_L;

  const LADSPA_Data * const input_R = (*plugin_data).input_R;
  LADSPA_Data * const output_R = (*plugin_data).output_R;
  fftw_real *comp_R = (*plugin_data).comp_R;
  fftw_real *real_R = (*plugin_data).real_R;
  LADSPA_Data *in_fifo_R = (*plugin_data).in_fifo_R;
  LADSPA_Data *out_accum_R = (*plugin_data).out_accum_R;
  LADSPA_Data *out_fifo_R = (*plugin_data).out_fifo_R;

  // PERL_END

#ifdef FFTW3
  // PERL_BEGIN
  fft_plan plan_cr_L = (*plugin_data).plan_cr_L;
  fft_plan plan_rc_L = (*plugin_data).plan_rc_L;
  fft_plan plan_cr_R = (*plugin_data).plan_cr_R;
  fft_plan plan_rc_R = (*plugin_data).plan_rc_R;
  // PERL_END
#else
  fft_plan plan_cr = (*plugin_data).plan_cr;
  fft_plan plan_rc = (*plugin_data).plan_rc;
#endif

  unsigned long step_size = (*plugin_data).step_size;

  unsigned long fft_latency = (*plugin_data).fft_latency;

  unsigned long number_of_bytes_in_step =
  (*plugin_data).number_of_bytes_in_step;

  unsigned long number_of_bytes_in_fft_length =
  (*plugin_data).number_of_bytes_in_fft_length;
  
  unsigned long number_of_bytes_in_fft_latency =
  (*plugin_data).number_of_bytes_in_fft_latency;


  // PERL_BEGIN


  LADSPA_Data db_gain_low = *((*plugin_data).band_gain_ptr_low);

  LADSPA_Data db_gain_00 = *((*plugin_data).band_gain_ptr_00);

  LADSPA_Data db_gain_01 = *((*plugin_data).band_gain_ptr_01);

  LADSPA_Data db_gain_02 = *((*plugin_data).band_gain_ptr_02);

  LADSPA_Data db_gain_03 = *((*plugin_data).band_gain_ptr_03);

  LADSPA_Data db_gain_04 = *((*plugin_data).band_gain_ptr_04);

  LADSPA_Data db_gain_05 = *((*plugin_data).band_gain_ptr_05);

  LADSPA_Data db_gain_06 = *((*plugin_data).band_gain_ptr_06);

  LADSPA_Data db_gain_07 = *((*plugin_data).band_gain_ptr_07);

  LADSPA_Data db_gain_08 = *((*plugin_data).band_gain_ptr_08);

  LADSPA_Data db_gain_09 = *((*plugin_data).band_gain_ptr_09);

  LADSPA_Data db_gain_10 = *((*plugin_data).band_gain_ptr_10);

  LADSPA_Data db_gain_11 = *((*plugin_data).band_gain_ptr_11);

  LADSPA_Data db_gain_12 = *((*plugin_data).band_gain_ptr_12);

  LADSPA_Data db_gain_13 = *((*plugin_data).band_gain_ptr_13);

  LADSPA_Data db_gain_14 = *((*plugin_data).band_gain_ptr_14);

  LADSPA_Data db_gain_15 = *((*plugin_data).band_gain_ptr_15);

  LADSPA_Data db_gain_16 = *((*plugin_data).band_gain_ptr_16);

  LADSPA_Data db_gain_17 = *((*plugin_data).band_gain_ptr_17);

  LADSPA_Data db_gain_18 = *((*plugin_data).band_gain_ptr_18);

  LADSPA_Data db_gain_19 = *((*plugin_data).band_gain_ptr_19);

  LADSPA_Data db_gain_20 = *((*plugin_data).band_gain_ptr_20);

  LADSPA_Data db_gain_21 = *((*plugin_data).band_gain_ptr_21);

  LADSPA_Data db_gain_22 = *((*plugin_data).band_gain_ptr_22);

  LADSPA_Data db_gain_23 = *((*plugin_data).band_gain_ptr_23);

  LADSPA_Data db_gain_24 = *((*plugin_data).band_gain_ptr_24);

  LADSPA_Data db_gain_25 = *((*plugin_data).band_gain_ptr_25);

  LADSPA_Data db_gain_26 = *((*plugin_data).band_gain_ptr_26);

  LADSPA_Data db_gain_27 = *((*plugin_data).band_gain_ptr_27);

  LADSPA_Data db_gain_28 = *((*plugin_data).band_gain_ptr_28);

  LADSPA_Data db_gain_29 = *((*plugin_data).band_gain_ptr_29);

  LADSPA_Data db_gain_high = *((*plugin_data).band_gain_ptr_high);

  LADSPA_Data reverse_delta_00_low;
  unsigned long ul_delta_00_low;

  LADSPA_Data reverse_delta_01_00;
  unsigned long ul_delta_01_00;

  LADSPA_Data reverse_delta_02_01;
  unsigned long ul_delta_02_01;

  LADSPA_Data reverse_delta_03_02;
  unsigned long ul_delta_03_02;

  LADSPA_Data reverse_delta_04_03;
  unsigned long ul_delta_04_03;

  LADSPA_Data reverse_delta_05_04;
  unsigned long ul_delta_05_04;

  LADSPA_Data reverse_delta_06_05;
  unsigned long ul_delta_06_05;

  LADSPA_Data reverse_delta_07_06;
  unsigned long ul_delta_07_06;

  LADSPA_Data reverse_delta_08_07;
  unsigned long ul_delta_08_07;

  LADSPA_Data reverse_delta_09_08;
  unsigned long ul_delta_09_08;

  LADSPA_Data reverse_delta_10_09;
  unsigned long ul_delta_10_09;

  LADSPA_Data reverse_delta_11_10;
  unsigned long ul_delta_11_10;

  LADSPA_Data reverse_delta_12_11;
  unsigned long ul_delta_12_11;

  LADSPA_Data reverse_delta_13_12;
  unsigned long ul_delta_13_12;

  LADSPA_Data reverse_delta_14_13;
  unsigned long ul_delta_14_13;

  LADSPA_Data reverse_delta_15_14;
  unsigned long ul_delta_15_14;

  LADSPA_Data reverse_delta_16_15;
  unsigned long ul_delta_16_15;

  LADSPA_Data reverse_delta_17_16;
  unsigned long ul_delta_17_16;

  LADSPA_Data reverse_delta_18_17;
  unsigned long ul_delta_18_17;

  LADSPA_Data reverse_delta_19_18;
  unsigned long ul_delta_19_18;

  LADSPA_Data reverse_delta_20_19;
  unsigned long ul_delta_20_19;

  LADSPA_Data reverse_delta_21_20;
  unsigned long ul_delta_21_20;

  LADSPA_Data reverse_delta_22_21;
  unsigned long ul_delta_22_21;

  LADSPA_Data reverse_delta_23_22;
  unsigned long ul_delta_23_22;

  LADSPA_Data reverse_delta_24_23;
  unsigned long ul_delta_24_23;

  LADSPA_Data reverse_delta_25_24;
  unsigned long ul_delta_25_24;

  LADSPA_Data reverse_delta_26_25;
  unsigned long ul_delta_26_25;

  LADSPA_Data reverse_delta_27_26;
  unsigned long ul_delta_27_26;

  LADSPA_Data reverse_delta_28_27;
  unsigned long ul_delta_28_27;

  LADSPA_Data reverse_delta_29_28;
  unsigned long ul_delta_29_28;

  LADSPA_Data reverse_delta_high_29;
  unsigned long ul_delta_high_29;

  (*plugin_data).true_gain_low =
  db_table
    [
    LIMIT
      (
        (int)(ENTRIES_PER_DB_GAIN * (db_gain_low - MIN_GAIN)),
      0,
      NUMBER_OF_DB_TABLE_ENTRIES_MINUS_1
      )
    ];

  if(
    db_gain_low != (*plugin_data).db_gain_low 
 || db_gain_00 != (*plugin_data).db_gain_00
    )
    {
    (*plugin_data).db_gain_low = db_gain_low;

    reverse_delta_00_low = (*plugin_data).reverse_delta_00_low;
    ul_delta_00_low = (*plugin_data).ul_delta_00_low;
 
    unsigned long bin_low = (*plugin_data).bin_low;

    LADSPA_Data true_gain_low = (*plugin_data).true_gain_low;

    LADSPA_Data true_gain_00 =
    (*plugin_data).true_gain_00 =
    db_table
      [
      LIMIT
        (
          (int)(ENTRIES_PER_DB_GAIN * (db_gain_00 - MIN_GAIN)),
        0,
        NUMBER_OF_DB_TABLE_ENTRIES_MINUS_1
        )
      ];


    float  *cos_times_gain_ptr = cos_times_gain + bin_low;

    float gain_angular_coefficient_00_low =
    (true_gain_00 - true_gain_low) * reverse_delta_00_low;

    unsigned long ctr = 0;
    while(ctr < ul_delta_00_low)
      {
      float  f_ctr = (float)ctr;

      float gain = true_gain_low + gain_angular_coefficient_00_low * f_ctr;

      // fprintf(stderr, "ctr=%d true_gain_low=%f true_gain_00=%f gain=%f\n", (int)ctr, (double)true_gain_low, (double)true_gain_00, (double)gain);

      *cos_times_gain_ptr++ = gain;

      ctr++;
      } // while(ctr++ < ul_delta_00_low)
    } // if(
      //   db_gain_low != (*plugin_data).db_gain_low 
      //|| db_gain_00 != (*plugin_data).db_gain_00
      //   )


  if(
    db_gain_00 != (*plugin_data).db_gain_00 
 || db_gain_01 != (*plugin_data).db_gain_01
    )
    {
    (*plugin_data).db_gain_00 = db_gain_00;

    reverse_delta_01_00 = (*plugin_data).reverse_delta_01_00;
    ul_delta_01_00 = (*plugin_data).ul_delta_01_00;
 
    unsigned long bin_00 = (*plugin_data).bin_00;

    LADSPA_Data true_gain_00 = (*plugin_data).true_gain_00;

    LADSPA_Data true_gain_01 =
    (*plugin_data).true_gain_01 =
    db_table
      [
      LIMIT
        (
          (int)(ENTRIES_PER_DB_GAIN * (db_gain_01 - MIN_GAIN)),
        0,
        NUMBER_OF_DB_TABLE_ENTRIES_MINUS_1
        )
      ];


    float  *cos_times_gain_ptr = cos_times_gain + bin_00;

    float gain_angular_coefficient_01_00 =
    (true_gain_01 - true_gain_00) * reverse_delta_01_00;

    unsigned long ctr = 0;
    while(ctr < ul_delta_01_00)
      {
      float  f_ctr = (float)ctr;

      float gain = true_gain_00 + gain_angular_coefficient_01_00 * f_ctr;

      // fprintf(stderr, "ctr=%d true_gain_00=%f true_gain_01=%f gain=%f\n", (int)ctr, (double)true_gain_00, (double)true_gain_01, (double)gain);

      *cos_times_gain_ptr++ = gain;

      ctr++;
      } // while(ctr++ < ul_delta_01_00)
    } // if(
      //   db_gain_00 != (*plugin_data).db_gain_00 
      //|| db_gain_01 != (*plugin_data).db_gain_01
      //   )


  if(
    db_gain_01 != (*plugin_data).db_gain_01 
 || db_gain_02 != (*plugin_data).db_gain_02
    )
    {
    (*plugin_data).db_gain_01 = db_gain_01;

    reverse_delta_02_01 = (*plugin_data).reverse_delta_02_01;
    ul_delta_02_01 = (*plugin_data).ul_delta_02_01;
 
    unsigned long bin_01 = (*plugin_data).bin_01;

    LADSPA_Data true_gain_01 = (*plugin_data).true_gain_01;

    LADSPA_Data true_gain_02 =
    (*plugin_data).true_gain_02 =
    db_table
      [
      LIMIT
        (
          (int)(ENTRIES_PER_DB_GAIN * (db_gain_02 - MIN_GAIN)),
        0,
        NUMBER_OF_DB_TABLE_ENTRIES_MINUS_1
        )
      ];


    float  *cos_times_gain_ptr = cos_times_gain + bin_01;

    float gain_angular_coefficient_02_01 =
    (true_gain_02 - true_gain_01) * reverse_delta_02_01;

    unsigned long ctr = 0;
    while(ctr < ul_delta_02_01)
      {
      float  f_ctr = (float)ctr;

      float gain = true_gain_01 + gain_angular_coefficient_02_01 * f_ctr;

      // fprintf(stderr, "ctr=%d true_gain_01=%f true_gain_02=%f gain=%f\n", (int)ctr, (double)true_gain_01, (double)true_gain_02, (double)gain);

      *cos_times_gain_ptr++ = gain;

      ctr++;
      } // while(ctr++ < ul_delta_02_01)
    } // if(
      //   db_gain_01 != (*plugin_data).db_gain_01 
      //|| db_gain_02 != (*plugin_data).db_gain_02
      //   )


  if(
    db_gain_02 != (*plugin_data).db_gain_02 
 || db_gain_03 != (*plugin_data).db_gain_03
    )
    {
    (*plugin_data).db_gain_02 = db_gain_02;

    reverse_delta_03_02 = (*plugin_data).reverse_delta_03_02;
    ul_delta_03_02 = (*plugin_data).ul_delta_03_02;
 
    unsigned long bin_02 = (*plugin_data).bin_02;

    LADSPA_Data true_gain_02 = (*plugin_data).true_gain_02;

    LADSPA_Data true_gain_03 =
    (*plugin_data).true_gain_03 =
    db_table
      [
      LIMIT
        (
          (int)(ENTRIES_PER_DB_GAIN * (db_gain_03 - MIN_GAIN)),
        0,
        NUMBER_OF_DB_TABLE_ENTRIES_MINUS_1
        )
      ];


    float  *cos_times_gain_ptr = cos_times_gain + bin_02;

    float gain_angular_coefficient_03_02 =
    (true_gain_03 - true_gain_02) * reverse_delta_03_02;

    unsigned long ctr = 0;
    while(ctr < ul_delta_03_02)
      {
      float  f_ctr = (float)ctr;

      float gain = true_gain_02 + gain_angular_coefficient_03_02 * f_ctr;

      // fprintf(stderr, "ctr=%d true_gain_02=%f true_gain_03=%f gain=%f\n", (int)ctr, (double)true_gain_02, (double)true_gain_03, (double)gain);

      *cos_times_gain_ptr++ = gain;

      ctr++;
      } // while(ctr++ < ul_delta_03_02)
    } // if(
      //   db_gain_02 != (*plugin_data).db_gain_02 
      //|| db_gain_03 != (*plugin_data).db_gain_03
      //   )


  if(
    db_gain_03 != (*plugin_data).db_gain_03 
 || db_gain_04 != (*plugin_data).db_gain_04
    )
    {
    (*plugin_data).db_gain_03 = db_gain_03;

    reverse_delta_04_03 = (*plugin_data).reverse_delta_04_03;
    ul_delta_04_03 = (*plugin_data).ul_delta_04_03;
 
    unsigned long bin_03 = (*plugin_data).bin_03;

    LADSPA_Data true_gain_03 = (*plugin_data).true_gain_03;

    LADSPA_Data true_gain_04 =
    (*plugin_data).true_gain_04 =
    db_table
      [
      LIMIT
        (
          (int)(ENTRIES_PER_DB_GAIN * (db_gain_04 - MIN_GAIN)),
        0,
        NUMBER_OF_DB_TABLE_ENTRIES_MINUS_1
        )
      ];


    float  *cos_times_gain_ptr = cos_times_gain + bin_03;

    float gain_angular_coefficient_04_03 =
    (true_gain_04 - true_gain_03) * reverse_delta_04_03;

    unsigned long ctr = 0;
    while(ctr < ul_delta_04_03)
      {
      float  f_ctr = (float)ctr;

      float gain = true_gain_03 + gain_angular_coefficient_04_03 * f_ctr;

      // fprintf(stderr, "ctr=%d true_gain_03=%f true_gain_04=%f gain=%f\n", (int)ctr, (double)true_gain_03, (double)true_gain_04, (double)gain);

      *cos_times_gain_ptr++ = gain;

      ctr++;
      } // while(ctr++ < ul_delta_04_03)
    } // if(
      //   db_gain_03 != (*plugin_data).db_gain_03 
      //|| db_gain_04 != (*plugin_data).db_gain_04
      //   )


  if(
    db_gain_04 != (*plugin_data).db_gain_04 
 || db_gain_05 != (*plugin_data).db_gain_05
    )
    {
    (*plugin_data).db_gain_04 = db_gain_04;

    reverse_delta_05_04 = (*plugin_data).reverse_delta_05_04;
    ul_delta_05_04 = (*plugin_data).ul_delta_05_04;
 
    unsigned long bin_04 = (*plugin_data).bin_04;

    LADSPA_Data true_gain_04 = (*plugin_data).true_gain_04;

    LADSPA_Data true_gain_05 =
    (*plugin_data).true_gain_05 =
    db_table
      [
      LIMIT
        (
          (int)(ENTRIES_PER_DB_GAIN * (db_gain_05 - MIN_GAIN)),
        0,
        NUMBER_OF_DB_TABLE_ENTRIES_MINUS_1
        )
      ];


    float  *cos_times_gain_ptr = cos_times_gain + bin_04;

    float gain_angular_coefficient_05_04 =
    (true_gain_05 - true_gain_04) * reverse_delta_05_04;

    unsigned long ctr = 0;
    while(ctr < ul_delta_05_04)
      {
      float  f_ctr = (float)ctr;

      float gain = true_gain_04 + gain_angular_coefficient_05_04 * f_ctr;

      // fprintf(stderr, "ctr=%d true_gain_04=%f true_gain_05=%f gain=%f\n", (int)ctr, (double)true_gain_04, (double)true_gain_05, (double)gain);

      *cos_times_gain_ptr++ = gain;

      ctr++;
      } // while(ctr++ < ul_delta_05_04)
    } // if(
      //   db_gain_04 != (*plugin_data).db_gain_04 
      //|| db_gain_05 != (*plugin_data).db_gain_05
      //   )


  if(
    db_gain_05 != (*plugin_data).db_gain_05 
 || db_gain_06 != (*plugin_data).db_gain_06
    )
    {
    (*plugin_data).db_gain_05 = db_gain_05;

    reverse_delta_06_05 = (*plugin_data).reverse_delta_06_05;
    ul_delta_06_05 = (*plugin_data).ul_delta_06_05;
 
    unsigned long bin_05 = (*plugin_data).bin_05;

    LADSPA_Data true_gain_05 = (*plugin_data).true_gain_05;

    LADSPA_Data true_gain_06 =
    (*plugin_data).true_gain_06 =
    db_table
      [
      LIMIT
        (
          (int)(ENTRIES_PER_DB_GAIN * (db_gain_06 - MIN_GAIN)),
        0,
        NUMBER_OF_DB_TABLE_ENTRIES_MINUS_1
        )
      ];


    float  *cos_times_gain_ptr = cos_times_gain + bin_05;

    float gain_angular_coefficient_06_05 =
    (true_gain_06 - true_gain_05) * reverse_delta_06_05;

    unsigned long ctr = 0;
    while(ctr < ul_delta_06_05)
      {
      float  f_ctr = (float)ctr;

      float gain = true_gain_05 + gain_angular_coefficient_06_05 * f_ctr;

      // fprintf(stderr, "ctr=%d true_gain_05=%f true_gain_06=%f gain=%f\n", (int)ctr, (double)true_gain_05, (double)true_gain_06, (double)gain);

      *cos_times_gain_ptr++ = gain;

      ctr++;
      } // while(ctr++ < ul_delta_06_05)
    } // if(
      //   db_gain_05 != (*plugin_data).db_gain_05 
      //|| db_gain_06 != (*plugin_data).db_gain_06
      //   )


  if(
    db_gain_06 != (*plugin_data).db_gain_06 
 || db_gain_07 != (*plugin_data).db_gain_07
    )
    {
    (*plugin_data).db_gain_06 = db_gain_06;

    reverse_delta_07_06 = (*plugin_data).reverse_delta_07_06;
    ul_delta_07_06 = (*plugin_data).ul_delta_07_06;
 
    unsigned long bin_06 = (*plugin_data).bin_06;

    LADSPA_Data true_gain_06 = (*plugin_data).true_gain_06;

    LADSPA_Data true_gain_07 =
    (*plugin_data).true_gain_07 =
    db_table
      [
      LIMIT
        (
          (int)(ENTRIES_PER_DB_GAIN * (db_gain_07 - MIN_GAIN)),
        0,
        NUMBER_OF_DB_TABLE_ENTRIES_MINUS_1
        )
      ];


    float  *cos_times_gain_ptr = cos_times_gain + bin_06;

    float gain_angular_coefficient_07_06 =
    (true_gain_07 - true_gain_06) * reverse_delta_07_06;

    unsigned long ctr = 0;
    while(ctr < ul_delta_07_06)
      {
      float  f_ctr = (float)ctr;

      float gain = true_gain_06 + gain_angular_coefficient_07_06 * f_ctr;

      // fprintf(stderr, "ctr=%d true_gain_06=%f true_gain_07=%f gain=%f\n", (int)ctr, (double)true_gain_06, (double)true_gain_07, (double)gain);

      *cos_times_gain_ptr++ = gain;

      ctr++;
      } // while(ctr++ < ul_delta_07_06)
    } // if(
      //   db_gain_06 != (*plugin_data).db_gain_06 
      //|| db_gain_07 != (*plugin_data).db_gain_07
      //   )


  if(
    db_gain_07 != (*plugin_data).db_gain_07 
 || db_gain_08 != (*plugin_data).db_gain_08
    )
    {
    (*plugin_data).db_gain_07 = db_gain_07;

    reverse_delta_08_07 = (*plugin_data).reverse_delta_08_07;
    ul_delta_08_07 = (*plugin_data).ul_delta_08_07;
 
    unsigned long bin_07 = (*plugin_data).bin_07;

    LADSPA_Data true_gain_07 = (*plugin_data).true_gain_07;

    LADSPA_Data true_gain_08 =
    (*plugin_data).true_gain_08 =
    db_table
      [
      LIMIT
        (
          (int)(ENTRIES_PER_DB_GAIN * (db_gain_08 - MIN_GAIN)),
        0,
        NUMBER_OF_DB_TABLE_ENTRIES_MINUS_1
        )
      ];


    float  *cos_times_gain_ptr = cos_times_gain + bin_07;

    float gain_angular_coefficient_08_07 =
    (true_gain_08 - true_gain_07) * reverse_delta_08_07;

    unsigned long ctr = 0;
    while(ctr < ul_delta_08_07)
      {
      float  f_ctr = (float)ctr;

      float gain = true_gain_07 + gain_angular_coefficient_08_07 * f_ctr;

      // fprintf(stderr, "ctr=%d true_gain_07=%f true_gain_08=%f gain=%f\n", (int)ctr, (double)true_gain_07, (double)true_gain_08, (double)gain);

      *cos_times_gain_ptr++ = gain;

      ctr++;
      } // while(ctr++ < ul_delta_08_07)
    } // if(
      //   db_gain_07 != (*plugin_data).db_gain_07 
      //|| db_gain_08 != (*plugin_data).db_gain_08
      //   )


  if(
    db_gain_08 != (*plugin_data).db_gain_08 
 || db_gain_09 != (*plugin_data).db_gain_09
    )
    {
    (*plugin_data).db_gain_08 = db_gain_08;

    reverse_delta_09_08 = (*plugin_data).reverse_delta_09_08;
    ul_delta_09_08 = (*plugin_data).ul_delta_09_08;
 
    unsigned long bin_08 = (*plugin_data).bin_08;

    LADSPA_Data true_gain_08 = (*plugin_data).true_gain_08;

    LADSPA_Data true_gain_09 =
    (*plugin_data).true_gain_09 =
    db_table
      [
      LIMIT
        (
          (int)(ENTRIES_PER_DB_GAIN * (db_gain_09 - MIN_GAIN)),
        0,
        NUMBER_OF_DB_TABLE_ENTRIES_MINUS_1
        )
      ];


    float  *cos_times_gain_ptr = cos_times_gain + bin_08;

    float gain_angular_coefficient_09_08 =
    (true_gain_09 - true_gain_08) * reverse_delta_09_08;

    unsigned long ctr = 0;
    while(ctr < ul_delta_09_08)
      {
      float  f_ctr = (float)ctr;

      float gain = true_gain_08 + gain_angular_coefficient_09_08 * f_ctr;

      // fprintf(stderr, "ctr=%d true_gain_08=%f true_gain_09=%f gain=%f\n", (int)ctr, (double)true_gain_08, (double)true_gain_09, (double)gain);

      *cos_times_gain_ptr++ = gain;

      ctr++;
      } // while(ctr++ < ul_delta_09_08)
    } // if(
      //   db_gain_08 != (*plugin_data).db_gain_08 
      //|| db_gain_09 != (*plugin_data).db_gain_09
      //   )


  if(
    db_gain_09 != (*plugin_data).db_gain_09 
 || db_gain_10 != (*plugin_data).db_gain_10
    )
    {
    (*plugin_data).db_gain_09 = db_gain_09;

    reverse_delta_10_09 = (*plugin_data).reverse_delta_10_09;
    ul_delta_10_09 = (*plugin_data).ul_delta_10_09;
 
    unsigned long bin_09 = (*plugin_data).bin_09;

    LADSPA_Data true_gain_09 = (*plugin_data).true_gain_09;

    LADSPA_Data true_gain_10 =
    (*plugin_data).true_gain_10 =
    db_table
      [
      LIMIT
        (
          (int)(ENTRIES_PER_DB_GAIN * (db_gain_10 - MIN_GAIN)),
        0,
        NUMBER_OF_DB_TABLE_ENTRIES_MINUS_1
        )
      ];


    float  *cos_times_gain_ptr = cos_times_gain + bin_09;

    float gain_angular_coefficient_10_09 =
    (true_gain_10 - true_gain_09) * reverse_delta_10_09;

    unsigned long ctr = 0;
    while(ctr < ul_delta_10_09)
      {
      float  f_ctr = (float)ctr;

      float gain = true_gain_09 + gain_angular_coefficient_10_09 * f_ctr;

      // fprintf(stderr, "ctr=%d true_gain_09=%f true_gain_10=%f gain=%f\n", (int)ctr, (double)true_gain_09, (double)true_gain_10, (double)gain);

      *cos_times_gain_ptr++ = gain;

      ctr++;
      } // while(ctr++ < ul_delta_10_09)
    } // if(
      //   db_gain_09 != (*plugin_data).db_gain_09 
      //|| db_gain_10 != (*plugin_data).db_gain_10
      //   )


  if(
    db_gain_10 != (*plugin_data).db_gain_10 
 || db_gain_11 != (*plugin_data).db_gain_11
    )
    {
    (*plugin_data).db_gain_10 = db_gain_10;

    reverse_delta_11_10 = (*plugin_data).reverse_delta_11_10;
    ul_delta_11_10 = (*plugin_data).ul_delta_11_10;
 
    unsigned long bin_10 = (*plugin_data).bin_10;

    LADSPA_Data true_gain_10 = (*plugin_data).true_gain_10;

    LADSPA_Data true_gain_11 =
    (*plugin_data).true_gain_11 =
    db_table
      [
      LIMIT
        (
          (int)(ENTRIES_PER_DB_GAIN * (db_gain_11 - MIN_GAIN)),
        0,
        NUMBER_OF_DB_TABLE_ENTRIES_MINUS_1
        )
      ];


    float  *cos_times_gain_ptr = cos_times_gain + bin_10;

    float gain_angular_coefficient_11_10 =
    (true_gain_11 - true_gain_10) * reverse_delta_11_10;

    unsigned long ctr = 0;
    while(ctr < ul_delta_11_10)
      {
      float  f_ctr = (float)ctr;

      float gain = true_gain_10 + gain_angular_coefficient_11_10 * f_ctr;

      // fprintf(stderr, "ctr=%d true_gain_10=%f true_gain_11=%f gain=%f\n", (int)ctr, (double)true_gain_10, (double)true_gain_11, (double)gain);

      *cos_times_gain_ptr++ = gain;

      ctr++;
      } // while(ctr++ < ul_delta_11_10)
    } // if(
      //   db_gain_10 != (*plugin_data).db_gain_10 
      //|| db_gain_11 != (*plugin_data).db_gain_11
      //   )


  if(
    db_gain_11 != (*plugin_data).db_gain_11 
 || db_gain_12 != (*plugin_data).db_gain_12
    )
    {
    (*plugin_data).db_gain_11 = db_gain_11;

    reverse_delta_12_11 = (*plugin_data).reverse_delta_12_11;
    ul_delta_12_11 = (*plugin_data).ul_delta_12_11;
 
    unsigned long bin_11 = (*plugin_data).bin_11;

    LADSPA_Data true_gain_11 = (*plugin_data).true_gain_11;

    LADSPA_Data true_gain_12 =
    (*plugin_data).true_gain_12 =
    db_table
      [
      LIMIT
        (
          (int)(ENTRIES_PER_DB_GAIN * (db_gain_12 - MIN_GAIN)),
        0,
        NUMBER_OF_DB_TABLE_ENTRIES_MINUS_1
        )
      ];


    float  *cos_times_gain_ptr = cos_times_gain + bin_11;

    float gain_angular_coefficient_12_11 =
    (true_gain_12 - true_gain_11) * reverse_delta_12_11;

    unsigned long ctr = 0;
    while(ctr < ul_delta_12_11)
      {
      float  f_ctr = (float)ctr;

      float gain = true_gain_11 + gain_angular_coefficient_12_11 * f_ctr;

      // fprintf(stderr, "ctr=%d true_gain_11=%f true_gain_12=%f gain=%f\n", (int)ctr, (double)true_gain_11, (double)true_gain_12, (double)gain);

      *cos_times_gain_ptr++ = gain;

      ctr++;
      } // while(ctr++ < ul_delta_12_11)
    } // if(
      //   db_gain_11 != (*plugin_data).db_gain_11 
      //|| db_gain_12 != (*plugin_data).db_gain_12
      //   )


  if(
    db_gain_12 != (*plugin_data).db_gain_12 
 || db_gain_13 != (*plugin_data).db_gain_13
    )
    {
    (*plugin_data).db_gain_12 = db_gain_12;

    reverse_delta_13_12 = (*plugin_data).reverse_delta_13_12;
    ul_delta_13_12 = (*plugin_data).ul_delta_13_12;
 
    unsigned long bin_12 = (*plugin_data).bin_12;

    LADSPA_Data true_gain_12 = (*plugin_data).true_gain_12;

    LADSPA_Data true_gain_13 =
    (*plugin_data).true_gain_13 =
    db_table
      [
      LIMIT
        (
          (int)(ENTRIES_PER_DB_GAIN * (db_gain_13 - MIN_GAIN)),
        0,
        NUMBER_OF_DB_TABLE_ENTRIES_MINUS_1
        )
      ];


    float  *cos_times_gain_ptr = cos_times_gain + bin_12;

    float gain_angular_coefficient_13_12 =
    (true_gain_13 - true_gain_12) * reverse_delta_13_12;

    unsigned long ctr = 0;
    while(ctr < ul_delta_13_12)
      {
      float  f_ctr = (float)ctr;

      float gain = true_gain_12 + gain_angular_coefficient_13_12 * f_ctr;

      // fprintf(stderr, "ctr=%d true_gain_12=%f true_gain_13=%f gain=%f\n", (int)ctr, (double)true_gain_12, (double)true_gain_13, (double)gain);

      *cos_times_gain_ptr++ = gain;

      ctr++;
      } // while(ctr++ < ul_delta_13_12)
    } // if(
      //   db_gain_12 != (*plugin_data).db_gain_12 
      //|| db_gain_13 != (*plugin_data).db_gain_13
      //   )


  if(
    db_gain_13 != (*plugin_data).db_gain_13 
 || db_gain_14 != (*plugin_data).db_gain_14
    )
    {
    (*plugin_data).db_gain_13 = db_gain_13;

    reverse_delta_14_13 = (*plugin_data).reverse_delta_14_13;
    ul_delta_14_13 = (*plugin_data).ul_delta_14_13;
 
    unsigned long bin_13 = (*plugin_data).bin_13;

    LADSPA_Data true_gain_13 = (*plugin_data).true_gain_13;

    LADSPA_Data true_gain_14 =
    (*plugin_data).true_gain_14 =
    db_table
      [
      LIMIT
        (
          (int)(ENTRIES_PER_DB_GAIN * (db_gain_14 - MIN_GAIN)),
        0,
        NUMBER_OF_DB_TABLE_ENTRIES_MINUS_1
        )
      ];


    float  *cos_times_gain_ptr = cos_times_gain + bin_13;

    float gain_angular_coefficient_14_13 =
    (true_gain_14 - true_gain_13) * reverse_delta_14_13;

    unsigned long ctr = 0;
    while(ctr < ul_delta_14_13)
      {
      float  f_ctr = (float)ctr;

      float gain = true_gain_13 + gain_angular_coefficient_14_13 * f_ctr;

      // fprintf(stderr, "ctr=%d true_gain_13=%f true_gain_14=%f gain=%f\n", (int)ctr, (double)true_gain_13, (double)true_gain_14, (double)gain);

      *cos_times_gain_ptr++ = gain;

      ctr++;
      } // while(ctr++ < ul_delta_14_13)
    } // if(
      //   db_gain_13 != (*plugin_data).db_gain_13 
      //|| db_gain_14 != (*plugin_data).db_gain_14
      //   )


  if(
    db_gain_14 != (*plugin_data).db_gain_14 
 || db_gain_15 != (*plugin_data).db_gain_15
    )
    {
    (*plugin_data).db_gain_14 = db_gain_14;

    reverse_delta_15_14 = (*plugin_data).reverse_delta_15_14;
    ul_delta_15_14 = (*plugin_data).ul_delta_15_14;
 
    unsigned long bin_14 = (*plugin_data).bin_14;

    LADSPA_Data true_gain_14 = (*plugin_data).true_gain_14;

    LADSPA_Data true_gain_15 =
    (*plugin_data).true_gain_15 =
    db_table
      [
      LIMIT
        (
          (int)(ENTRIES_PER_DB_GAIN * (db_gain_15 - MIN_GAIN)),
        0,
        NUMBER_OF_DB_TABLE_ENTRIES_MINUS_1
        )
      ];


    float  *cos_times_gain_ptr = cos_times_gain + bin_14;

    float gain_angular_coefficient_15_14 =
    (true_gain_15 - true_gain_14) * reverse_delta_15_14;

    unsigned long ctr = 0;
    while(ctr < ul_delta_15_14)
      {
      float  f_ctr = (float)ctr;

      float gain = true_gain_14 + gain_angular_coefficient_15_14 * f_ctr;

      // fprintf(stderr, "ctr=%d true_gain_14=%f true_gain_15=%f gain=%f\n", (int)ctr, (double)true_gain_14, (double)true_gain_15, (double)gain);

      *cos_times_gain_ptr++ = gain;

      ctr++;
      } // while(ctr++ < ul_delta_15_14)
    } // if(
      //   db_gain_14 != (*plugin_data).db_gain_14 
      //|| db_gain_15 != (*plugin_data).db_gain_15
      //   )


  if(
    db_gain_15 != (*plugin_data).db_gain_15 
 || db_gain_16 != (*plugin_data).db_gain_16
    )
    {
    (*plugin_data).db_gain_15 = db_gain_15;

    reverse_delta_16_15 = (*plugin_data).reverse_delta_16_15;
    ul_delta_16_15 = (*plugin_data).ul_delta_16_15;
 
    unsigned long bin_15 = (*plugin_data).bin_15;

    LADSPA_Data true_gain_15 = (*plugin_data).true_gain_15;

    LADSPA_Data true_gain_16 =
    (*plugin_data).true_gain_16 =
    db_table
      [
      LIMIT
        (
          (int)(ENTRIES_PER_DB_GAIN * (db_gain_16 - MIN_GAIN)),
        0,
        NUMBER_OF_DB_TABLE_ENTRIES_MINUS_1
        )
      ];


    float  *cos_times_gain_ptr = cos_times_gain + bin_15;

    float gain_angular_coefficient_16_15 =
    (true_gain_16 - true_gain_15) * reverse_delta_16_15;

    unsigned long ctr = 0;
    while(ctr < ul_delta_16_15)
      {
      float  f_ctr = (float)ctr;

      float gain = true_gain_15 + gain_angular_coefficient_16_15 * f_ctr;

      // fprintf(stderr, "ctr=%d true_gain_15=%f true_gain_16=%f gain=%f\n", (int)ctr, (double)true_gain_15, (double)true_gain_16, (double)gain);

      *cos_times_gain_ptr++ = gain;

      ctr++;
      } // while(ctr++ < ul_delta_16_15)
    } // if(
      //   db_gain_15 != (*plugin_data).db_gain_15 
      //|| db_gain_16 != (*plugin_data).db_gain_16
      //   )


  if(
    db_gain_16 != (*plugin_data).db_gain_16 
 || db_gain_17 != (*plugin_data).db_gain_17
    )
    {
    (*plugin_data).db_gain_16 = db_gain_16;

    reverse_delta_17_16 = (*plugin_data).reverse_delta_17_16;
    ul_delta_17_16 = (*plugin_data).ul_delta_17_16;
 
    unsigned long bin_16 = (*plugin_data).bin_16;

    LADSPA_Data true_gain_16 = (*plugin_data).true_gain_16;

    LADSPA_Data true_gain_17 =
    (*plugin_data).true_gain_17 =
    db_table
      [
      LIMIT
        (
          (int)(ENTRIES_PER_DB_GAIN * (db_gain_17 - MIN_GAIN)),
        0,
        NUMBER_OF_DB_TABLE_ENTRIES_MINUS_1
        )
      ];


    float  *cos_times_gain_ptr = cos_times_gain + bin_16;

    float gain_angular_coefficient_17_16 =
    (true_gain_17 - true_gain_16) * reverse_delta_17_16;

    unsigned long ctr = 0;
    while(ctr < ul_delta_17_16)
      {
      float  f_ctr = (float)ctr;

      float gain = true_gain_16 + gain_angular_coefficient_17_16 * f_ctr;

      // fprintf(stderr, "ctr=%d true_gain_16=%f true_gain_17=%f gain=%f\n", (int)ctr, (double)true_gain_16, (double)true_gain_17, (double)gain);

      *cos_times_gain_ptr++ = gain;

      ctr++;
      } // while(ctr++ < ul_delta_17_16)
    } // if(
      //   db_gain_16 != (*plugin_data).db_gain_16 
      //|| db_gain_17 != (*plugin_data).db_gain_17
      //   )


  if(
    db_gain_17 != (*plugin_data).db_gain_17 
 || db_gain_18 != (*plugin_data).db_gain_18
    )
    {
    (*plugin_data).db_gain_17 = db_gain_17;

    reverse_delta_18_17 = (*plugin_data).reverse_delta_18_17;
    ul_delta_18_17 = (*plugin_data).ul_delta_18_17;
 
    unsigned long bin_17 = (*plugin_data).bin_17;

    LADSPA_Data true_gain_17 = (*plugin_data).true_gain_17;

    LADSPA_Data true_gain_18 =
    (*plugin_data).true_gain_18 =
    db_table
      [
      LIMIT
        (
          (int)(ENTRIES_PER_DB_GAIN * (db_gain_18 - MIN_GAIN)),
        0,
        NUMBER_OF_DB_TABLE_ENTRIES_MINUS_1
        )
      ];


    float  *cos_times_gain_ptr = cos_times_gain + bin_17;

    float gain_angular_coefficient_18_17 =
    (true_gain_18 - true_gain_17) * reverse_delta_18_17;

    unsigned long ctr = 0;
    while(ctr < ul_delta_18_17)
      {
      float  f_ctr = (float)ctr;

      float gain = true_gain_17 + gain_angular_coefficient_18_17 * f_ctr;

      // fprintf(stderr, "ctr=%d true_gain_17=%f true_gain_18=%f gain=%f\n", (int)ctr, (double)true_gain_17, (double)true_gain_18, (double)gain);

      *cos_times_gain_ptr++ = gain;

      ctr++;
      } // while(ctr++ < ul_delta_18_17)
    } // if(
      //   db_gain_17 != (*plugin_data).db_gain_17 
      //|| db_gain_18 != (*plugin_data).db_gain_18
      //   )


  if(
    db_gain_18 != (*plugin_data).db_gain_18 
 || db_gain_19 != (*plugin_data).db_gain_19
    )
    {
    (*plugin_data).db_gain_18 = db_gain_18;

    reverse_delta_19_18 = (*plugin_data).reverse_delta_19_18;
    ul_delta_19_18 = (*plugin_data).ul_delta_19_18;
 
    unsigned long bin_18 = (*plugin_data).bin_18;

    LADSPA_Data true_gain_18 = (*plugin_data).true_gain_18;

    LADSPA_Data true_gain_19 =
    (*plugin_data).true_gain_19 =
    db_table
      [
      LIMIT
        (
          (int)(ENTRIES_PER_DB_GAIN * (db_gain_19 - MIN_GAIN)),
        0,
        NUMBER_OF_DB_TABLE_ENTRIES_MINUS_1
        )
      ];


    float  *cos_times_gain_ptr = cos_times_gain + bin_18;

    float gain_angular_coefficient_19_18 =
    (true_gain_19 - true_gain_18) * reverse_delta_19_18;

    unsigned long ctr = 0;
    while(ctr < ul_delta_19_18)
      {
      float  f_ctr = (float)ctr;

      float gain = true_gain_18 + gain_angular_coefficient_19_18 * f_ctr;

      // fprintf(stderr, "ctr=%d true_gain_18=%f true_gain_19=%f gain=%f\n", (int)ctr, (double)true_gain_18, (double)true_gain_19, (double)gain);

      *cos_times_gain_ptr++ = gain;

      ctr++;
      } // while(ctr++ < ul_delta_19_18)
    } // if(
      //   db_gain_18 != (*plugin_data).db_gain_18 
      //|| db_gain_19 != (*plugin_data).db_gain_19
      //   )


  if(
    db_gain_19 != (*plugin_data).db_gain_19 
 || db_gain_20 != (*plugin_data).db_gain_20
    )
    {
    (*plugin_data).db_gain_19 = db_gain_19;

    reverse_delta_20_19 = (*plugin_data).reverse_delta_20_19;
    ul_delta_20_19 = (*plugin_data).ul_delta_20_19;
 
    unsigned long bin_19 = (*plugin_data).bin_19;

    LADSPA_Data true_gain_19 = (*plugin_data).true_gain_19;

    LADSPA_Data true_gain_20 =
    (*plugin_data).true_gain_20 =
    db_table
      [
      LIMIT
        (
          (int)(ENTRIES_PER_DB_GAIN * (db_gain_20 - MIN_GAIN)),
        0,
        NUMBER_OF_DB_TABLE_ENTRIES_MINUS_1
        )
      ];


    float  *cos_times_gain_ptr = cos_times_gain + bin_19;

    float gain_angular_coefficient_20_19 =
    (true_gain_20 - true_gain_19) * reverse_delta_20_19;

    unsigned long ctr = 0;
    while(ctr < ul_delta_20_19)
      {
      float  f_ctr = (float)ctr;

      float gain = true_gain_19 + gain_angular_coefficient_20_19 * f_ctr;

      // fprintf(stderr, "ctr=%d true_gain_19=%f true_gain_20=%f gain=%f\n", (int)ctr, (double)true_gain_19, (double)true_gain_20, (double)gain);

      *cos_times_gain_ptr++ = gain;

      ctr++;
      } // while(ctr++ < ul_delta_20_19)
    } // if(
      //   db_gain_19 != (*plugin_data).db_gain_19 
      //|| db_gain_20 != (*plugin_data).db_gain_20
      //   )


  if(
    db_gain_20 != (*plugin_data).db_gain_20 
 || db_gain_21 != (*plugin_data).db_gain_21
    )
    {
    (*plugin_data).db_gain_20 = db_gain_20;

    reverse_delta_21_20 = (*plugin_data).reverse_delta_21_20;
    ul_delta_21_20 = (*plugin_data).ul_delta_21_20;
 
    unsigned long bin_20 = (*plugin_data).bin_20;

    LADSPA_Data true_gain_20 = (*plugin_data).true_gain_20;

    LADSPA_Data true_gain_21 =
    (*plugin_data).true_gain_21 =
    db_table
      [
      LIMIT
        (
          (int)(ENTRIES_PER_DB_GAIN * (db_gain_21 - MIN_GAIN)),
        0,
        NUMBER_OF_DB_TABLE_ENTRIES_MINUS_1
        )
      ];


    float  *cos_times_gain_ptr = cos_times_gain + bin_20;

    float gain_angular_coefficient_21_20 =
    (true_gain_21 - true_gain_20) * reverse_delta_21_20;

    unsigned long ctr = 0;
    while(ctr < ul_delta_21_20)
      {
      float  f_ctr = (float)ctr;

      float gain = true_gain_20 + gain_angular_coefficient_21_20 * f_ctr;

      // fprintf(stderr, "ctr=%d true_gain_20=%f true_gain_21=%f gain=%f\n", (int)ctr, (double)true_gain_20, (double)true_gain_21, (double)gain);

      *cos_times_gain_ptr++ = gain;

      ctr++;
      } // while(ctr++ < ul_delta_21_20)
    } // if(
      //   db_gain_20 != (*plugin_data).db_gain_20 
      //|| db_gain_21 != (*plugin_data).db_gain_21
      //   )


  if(
    db_gain_21 != (*plugin_data).db_gain_21 
 || db_gain_22 != (*plugin_data).db_gain_22
    )
    {
    (*plugin_data).db_gain_21 = db_gain_21;

    reverse_delta_22_21 = (*plugin_data).reverse_delta_22_21;
    ul_delta_22_21 = (*plugin_data).ul_delta_22_21;
 
    unsigned long bin_21 = (*plugin_data).bin_21;

    LADSPA_Data true_gain_21 = (*plugin_data).true_gain_21;

    LADSPA_Data true_gain_22 =
    (*plugin_data).true_gain_22 =
    db_table
      [
      LIMIT
        (
          (int)(ENTRIES_PER_DB_GAIN * (db_gain_22 - MIN_GAIN)),
        0,
        NUMBER_OF_DB_TABLE_ENTRIES_MINUS_1
        )
      ];


    float  *cos_times_gain_ptr = cos_times_gain + bin_21;

    float gain_angular_coefficient_22_21 =
    (true_gain_22 - true_gain_21) * reverse_delta_22_21;

    unsigned long ctr = 0;
    while(ctr < ul_delta_22_21)
      {
      float  f_ctr = (float)ctr;

      float gain = true_gain_21 + gain_angular_coefficient_22_21 * f_ctr;

      // fprintf(stderr, "ctr=%d true_gain_21=%f true_gain_22=%f gain=%f\n", (int)ctr, (double)true_gain_21, (double)true_gain_22, (double)gain);

      *cos_times_gain_ptr++ = gain;

      ctr++;
      } // while(ctr++ < ul_delta_22_21)
    } // if(
      //   db_gain_21 != (*plugin_data).db_gain_21 
      //|| db_gain_22 != (*plugin_data).db_gain_22
      //   )


  if(
    db_gain_22 != (*plugin_data).db_gain_22 
 || db_gain_23 != (*plugin_data).db_gain_23
    )
    {
    (*plugin_data).db_gain_22 = db_gain_22;

    reverse_delta_23_22 = (*plugin_data).reverse_delta_23_22;
    ul_delta_23_22 = (*plugin_data).ul_delta_23_22;
 
    unsigned long bin_22 = (*plugin_data).bin_22;

    LADSPA_Data true_gain_22 = (*plugin_data).true_gain_22;

    LADSPA_Data true_gain_23 =
    (*plugin_data).true_gain_23 =
    db_table
      [
      LIMIT
        (
          (int)(ENTRIES_PER_DB_GAIN * (db_gain_23 - MIN_GAIN)),
        0,
        NUMBER_OF_DB_TABLE_ENTRIES_MINUS_1
        )
      ];


    float  *cos_times_gain_ptr = cos_times_gain + bin_22;

    float gain_angular_coefficient_23_22 =
    (true_gain_23 - true_gain_22) * reverse_delta_23_22;

    unsigned long ctr = 0;
    while(ctr < ul_delta_23_22)
      {
      float  f_ctr = (float)ctr;

      float gain = true_gain_22 + gain_angular_coefficient_23_22 * f_ctr;

      // fprintf(stderr, "ctr=%d true_gain_22=%f true_gain_23=%f gain=%f\n", (int)ctr, (double)true_gain_22, (double)true_gain_23, (double)gain);

      *cos_times_gain_ptr++ = gain;

      ctr++;
      } // while(ctr++ < ul_delta_23_22)
    } // if(
      //   db_gain_22 != (*plugin_data).db_gain_22 
      //|| db_gain_23 != (*plugin_data).db_gain_23
      //   )


  if(
    db_gain_23 != (*plugin_data).db_gain_23 
 || db_gain_24 != (*plugin_data).db_gain_24
    )
    {
    (*plugin_data).db_gain_23 = db_gain_23;

    reverse_delta_24_23 = (*plugin_data).reverse_delta_24_23;
    ul_delta_24_23 = (*plugin_data).ul_delta_24_23;
 
    unsigned long bin_23 = (*plugin_data).bin_23;

    LADSPA_Data true_gain_23 = (*plugin_data).true_gain_23;

    LADSPA_Data true_gain_24 =
    (*plugin_data).true_gain_24 =
    db_table
      [
      LIMIT
        (
          (int)(ENTRIES_PER_DB_GAIN * (db_gain_24 - MIN_GAIN)),
        0,
        NUMBER_OF_DB_TABLE_ENTRIES_MINUS_1
        )
      ];


    float  *cos_times_gain_ptr = cos_times_gain + bin_23;

    float gain_angular_coefficient_24_23 =
    (true_gain_24 - true_gain_23) * reverse_delta_24_23;

    unsigned long ctr = 0;
    while(ctr < ul_delta_24_23)
      {
      float  f_ctr = (float)ctr;

      float gain = true_gain_23 + gain_angular_coefficient_24_23 * f_ctr;

      // fprintf(stderr, "ctr=%d true_gain_23=%f true_gain_24=%f gain=%f\n", (int)ctr, (double)true_gain_23, (double)true_gain_24, (double)gain);

      *cos_times_gain_ptr++ = gain;

      ctr++;
      } // while(ctr++ < ul_delta_24_23)
    } // if(
      //   db_gain_23 != (*plugin_data).db_gain_23 
      //|| db_gain_24 != (*plugin_data).db_gain_24
      //   )


  if(
    db_gain_24 != (*plugin_data).db_gain_24 
 || db_gain_25 != (*plugin_data).db_gain_25
    )
    {
    (*plugin_data).db_gain_24 = db_gain_24;

    reverse_delta_25_24 = (*plugin_data).reverse_delta_25_24;
    ul_delta_25_24 = (*plugin_data).ul_delta_25_24;
 
    unsigned long bin_24 = (*plugin_data).bin_24;

    LADSPA_Data true_gain_24 = (*plugin_data).true_gain_24;

    LADSPA_Data true_gain_25 =
    (*plugin_data).true_gain_25 =
    db_table
      [
      LIMIT
        (
          (int)(ENTRIES_PER_DB_GAIN * (db_gain_25 - MIN_GAIN)),
        0,
        NUMBER_OF_DB_TABLE_ENTRIES_MINUS_1
        )
      ];


    float  *cos_times_gain_ptr = cos_times_gain + bin_24;

    float gain_angular_coefficient_25_24 =
    (true_gain_25 - true_gain_24) * reverse_delta_25_24;

    unsigned long ctr = 0;
    while(ctr < ul_delta_25_24)
      {
      float  f_ctr = (float)ctr;

      float gain = true_gain_24 + gain_angular_coefficient_25_24 * f_ctr;

      // fprintf(stderr, "ctr=%d true_gain_24=%f true_gain_25=%f gain=%f\n", (int)ctr, (double)true_gain_24, (double)true_gain_25, (double)gain);

      *cos_times_gain_ptr++ = gain;

      ctr++;
      } // while(ctr++ < ul_delta_25_24)
    } // if(
      //   db_gain_24 != (*plugin_data).db_gain_24 
      //|| db_gain_25 != (*plugin_data).db_gain_25
      //   )


  if(
    db_gain_25 != (*plugin_data).db_gain_25 
 || db_gain_26 != (*plugin_data).db_gain_26
    )
    {
    (*plugin_data).db_gain_25 = db_gain_25;

    reverse_delta_26_25 = (*plugin_data).reverse_delta_26_25;
    ul_delta_26_25 = (*plugin_data).ul_delta_26_25;
 
    unsigned long bin_25 = (*plugin_data).bin_25;

    LADSPA_Data true_gain_25 = (*plugin_data).true_gain_25;

    LADSPA_Data true_gain_26 =
    (*plugin_data).true_gain_26 =
    db_table
      [
      LIMIT
        (
          (int)(ENTRIES_PER_DB_GAIN * (db_gain_26 - MIN_GAIN)),
        0,
        NUMBER_OF_DB_TABLE_ENTRIES_MINUS_1
        )
      ];


    float  *cos_times_gain_ptr = cos_times_gain + bin_25;

    float gain_angular_coefficient_26_25 =
    (true_gain_26 - true_gain_25) * reverse_delta_26_25;

    unsigned long ctr = 0;
    while(ctr < ul_delta_26_25)
      {
      float  f_ctr = (float)ctr;

      float gain = true_gain_25 + gain_angular_coefficient_26_25 * f_ctr;

      // fprintf(stderr, "ctr=%d true_gain_25=%f true_gain_26=%f gain=%f\n", (int)ctr, (double)true_gain_25, (double)true_gain_26, (double)gain);

      *cos_times_gain_ptr++ = gain;

      ctr++;
      } // while(ctr++ < ul_delta_26_25)
    } // if(
      //   db_gain_25 != (*plugin_data).db_gain_25 
      //|| db_gain_26 != (*plugin_data).db_gain_26
      //   )


  if(
    db_gain_26 != (*plugin_data).db_gain_26 
 || db_gain_27 != (*plugin_data).db_gain_27
    )
    {
    (*plugin_data).db_gain_26 = db_gain_26;

    reverse_delta_27_26 = (*plugin_data).reverse_delta_27_26;
    ul_delta_27_26 = (*plugin_data).ul_delta_27_26;
 
    unsigned long bin_26 = (*plugin_data).bin_26;

    LADSPA_Data true_gain_26 = (*plugin_data).true_gain_26;

    LADSPA_Data true_gain_27 =
    (*plugin_data).true_gain_27 =
    db_table
      [
      LIMIT
        (
          (int)(ENTRIES_PER_DB_GAIN * (db_gain_27 - MIN_GAIN)),
        0,
        NUMBER_OF_DB_TABLE_ENTRIES_MINUS_1
        )
      ];


    float  *cos_times_gain_ptr = cos_times_gain + bin_26;

    float gain_angular_coefficient_27_26 =
    (true_gain_27 - true_gain_26) * reverse_delta_27_26;

    unsigned long ctr = 0;
    while(ctr < ul_delta_27_26)
      {
      float  f_ctr = (float)ctr;

      float gain = true_gain_26 + gain_angular_coefficient_27_26 * f_ctr;

      // fprintf(stderr, "ctr=%d true_gain_26=%f true_gain_27=%f gain=%f\n", (int)ctr, (double)true_gain_26, (double)true_gain_27, (double)gain);

      *cos_times_gain_ptr++ = gain;

      ctr++;
      } // while(ctr++ < ul_delta_27_26)
    } // if(
      //   db_gain_26 != (*plugin_data).db_gain_26 
      //|| db_gain_27 != (*plugin_data).db_gain_27
      //   )


  if(
    db_gain_27 != (*plugin_data).db_gain_27 
 || db_gain_28 != (*plugin_data).db_gain_28
    )
    {
    (*plugin_data).db_gain_27 = db_gain_27;

    reverse_delta_28_27 = (*plugin_data).reverse_delta_28_27;
    ul_delta_28_27 = (*plugin_data).ul_delta_28_27;
 
    unsigned long bin_27 = (*plugin_data).bin_27;

    LADSPA_Data true_gain_27 = (*plugin_data).true_gain_27;

    LADSPA_Data true_gain_28 =
    (*plugin_data).true_gain_28 =
    db_table
      [
      LIMIT
        (
          (int)(ENTRIES_PER_DB_GAIN * (db_gain_28 - MIN_GAIN)),
        0,
        NUMBER_OF_DB_TABLE_ENTRIES_MINUS_1
        )
      ];


    float  *cos_times_gain_ptr = cos_times_gain + bin_27;

    float gain_angular_coefficient_28_27 =
    (true_gain_28 - true_gain_27) * reverse_delta_28_27;

    unsigned long ctr = 0;
    while(ctr < ul_delta_28_27)
      {
      float  f_ctr = (float)ctr;

      float gain = true_gain_27 + gain_angular_coefficient_28_27 * f_ctr;

      // fprintf(stderr, "ctr=%d true_gain_27=%f true_gain_28=%f gain=%f\n", (int)ctr, (double)true_gain_27, (double)true_gain_28, (double)gain);

      *cos_times_gain_ptr++ = gain;

      ctr++;
      } // while(ctr++ < ul_delta_28_27)
    } // if(
      //   db_gain_27 != (*plugin_data).db_gain_27 
      //|| db_gain_28 != (*plugin_data).db_gain_28
      //   )


  if(
    db_gain_28 != (*plugin_data).db_gain_28 
 || db_gain_29 != (*plugin_data).db_gain_29
    )
    {
    (*plugin_data).db_gain_28 = db_gain_28;

    reverse_delta_29_28 = (*plugin_data).reverse_delta_29_28;
    ul_delta_29_28 = (*plugin_data).ul_delta_29_28;
 
    unsigned long bin_28 = (*plugin_data).bin_28;

    LADSPA_Data true_gain_28 = (*plugin_data).true_gain_28;

    LADSPA_Data true_gain_29 =
    (*plugin_data).true_gain_29 =
    db_table
      [
      LIMIT
        (
          (int)(ENTRIES_PER_DB_GAIN * (db_gain_29 - MIN_GAIN)),
        0,
        NUMBER_OF_DB_TABLE_ENTRIES_MINUS_1
        )
      ];


    float  *cos_times_gain_ptr = cos_times_gain + bin_28;

    float gain_angular_coefficient_29_28 =
    (true_gain_29 - true_gain_28) * reverse_delta_29_28;

    unsigned long ctr = 0;
    while(ctr < ul_delta_29_28)
      {
      float  f_ctr = (float)ctr;

      float gain = true_gain_28 + gain_angular_coefficient_29_28 * f_ctr;

      // fprintf(stderr, "ctr=%d true_gain_28=%f true_gain_29=%f gain=%f\n", (int)ctr, (double)true_gain_28, (double)true_gain_29, (double)gain);

      *cos_times_gain_ptr++ = gain;

      ctr++;
      } // while(ctr++ < ul_delta_29_28)
    } // if(
      //   db_gain_28 != (*plugin_data).db_gain_28 
      //|| db_gain_29 != (*plugin_data).db_gain_29
      //   )


  if(
    db_gain_29 != (*plugin_data).db_gain_29 
 || db_gain_high != (*plugin_data).db_gain_high
    )
    {
    (*plugin_data).db_gain_29 = db_gain_29;

    reverse_delta_high_29 = (*plugin_data).reverse_delta_high_29;
    ul_delta_high_29 = (*plugin_data).ul_delta_high_29;
 
    unsigned long bin_29 = (*plugin_data).bin_29;

    LADSPA_Data true_gain_29 = (*plugin_data).true_gain_29;

    LADSPA_Data true_gain_high =
    (*plugin_data).true_gain_high =
    db_table
      [
      LIMIT
        (
          (int)(ENTRIES_PER_DB_GAIN * (db_gain_high - MIN_GAIN)),
        0,
        NUMBER_OF_DB_TABLE_ENTRIES_MINUS_1
        )
      ];


    float  *cos_times_gain_ptr = cos_times_gain + bin_29;

    float gain_angular_coefficient_high_29 =
    (true_gain_high - true_gain_29) * reverse_delta_high_29;

    unsigned long ctr = 0;
    while(ctr < ul_delta_high_29)
      {
      float  f_ctr = (float)ctr;

      float gain = true_gain_29 + gain_angular_coefficient_high_29 * f_ctr;

      // fprintf(stderr, "ctr=%d true_gain_29=%f true_gain_high=%f gain=%f\n", (int)ctr, (double)true_gain_29, (double)true_gain_high, (double)gain);

      *cos_times_gain_ptr++ = gain;

      ctr++;
      } // while(ctr++ < ul_delta_high_29)
    } // if(
      //   db_gain_29 != (*plugin_data).db_gain_29 
      //|| db_gain_high != (*plugin_data).db_gain_high
      //   )


  (*plugin_data).db_gain_high = db_gain_high;
  // PERL_END

  unsigned long input_buffer_pos = 0;

  //fprintf(stderr, "sample_count=%d\n", (int)sample_count);

  while(input_buffer_pos < sample_count)
    {
    unsigned long number_of_remaining_samples = sample_count - input_buffer_pos;

    unsigned long number_of_steps_until_next_fft = fft_length - fifo_pos;

    if(number_of_remaining_samples < number_of_steps_until_next_fft)
      // just copy number_of_remaining_samples samples into in_fifo* and into
      // output* buffers
      {
      unsigned long number_of_remaining_bytes =
      number_of_remaining_samples * sizeof(LADSPA_Data);

      // PERL_BEGIN

      //fprintf(stderr, "(1) input_buffer_pos=%d *(input_L + input_buffer_pos)=%G\n", (int)input_buffer_pos, (LADSPA_Data)(*(input_L + input_buffer_pos)));
      //fprintf(stderr, "(1) input_buffer_pos=%d *(output_L + input_buffer_pos)=%G\n", (int)input_buffer_pos, (LADSPA_Data)(*(output_L + input_buffer_pos)));
      
      memcpy
        (
        in_fifo_L + fifo_pos, // dst
        input_L + input_buffer_pos, // src
        number_of_remaining_bytes // number_of_bytes
        );

      memcpy
        (
        output_L + input_buffer_pos, // dst
        out_fifo_L + fifo_pos - fft_latency, // src
        number_of_remaining_bytes // number_of_bytes
        );

      //fprintf(stderr, "(1) input_buffer_pos=%d *(input_R + input_buffer_pos)=%G\n", (int)input_buffer_pos, (LADSPA_Data)(*(input_R + input_buffer_pos)));
      //fprintf(stderr, "(1) input_buffer_pos=%d *(output_R + input_buffer_pos)=%G\n", (int)input_buffer_pos, (LADSPA_Data)(*(output_R + input_buffer_pos)));
      
      memcpy
        (
        in_fifo_R + fifo_pos, // dst
        input_R + input_buffer_pos, // src
        number_of_remaining_bytes // number_of_bytes
        );

      memcpy
        (
        output_R + input_buffer_pos, // dst
        out_fifo_R + fifo_pos - fft_latency, // src
        number_of_remaining_bytes // number_of_bytes
        );
      // PERL_END

      fifo_pos += number_of_remaining_samples;
      input_buffer_pos += number_of_remaining_samples;
      }
    else
      // copy number_of_steps_until_next_fft samples into in_fifo* and into
      // output buffers and execute FFT
      {
      unsigned long number_of_bytes_to_copy =
      number_of_steps_until_next_fft * sizeof(LADSPA_Data);

      // PERL_BEGIN

      //fprintf(stderr, "(2) input_buffer_pos=%d *(input_L + input_buffer_pos)=%G\n", (int)input_buffer_pos, (LADSPA_Data)(*(input_L + input_buffer_pos)));
      //fprintf(stderr, "(2) input_buffer_pos=%d *(output_L + input_buffer_pos)=%G\n", (int)input_buffer_pos, (LADSPA_Data)(*(output_L + input_buffer_pos)));

      memcpy
        (
        in_fifo_L + fifo_pos, // dst
        input_L + input_buffer_pos, // src
        number_of_bytes_to_copy // number_of_bytes
        );

      memcpy
        (
        output_L + input_buffer_pos,// dst
        out_fifo_L + fifo_pos - fft_latency, // src
        number_of_bytes_to_copy // number_of_bytes
        );

      //fprintf(stderr, "(2) input_buffer_pos=%d *(input_R + input_buffer_pos)=%G\n", (int)input_buffer_pos, (LADSPA_Data)(*(input_R + input_buffer_pos)));
      //fprintf(stderr, "(2) input_buffer_pos=%d *(output_R + input_buffer_pos)=%G\n", (int)input_buffer_pos, (LADSPA_Data)(*(output_R + input_buffer_pos)));

      memcpy
        (
        in_fifo_R + fifo_pos, // dst
        input_R + input_buffer_pos, // src
        number_of_bytes_to_copy // number_of_bytes
        );

      memcpy
        (
        output_R + input_buffer_pos,// dst
        out_fifo_R + fifo_pos - fft_latency, // src
        number_of_bytes_to_copy // number_of_bytes
        );
      // PERL_END

      // multiply input fifos by window coefficients

        {
        float *window_for_input_ptr = window_for_input;

        // PERL_BEGIN
        fftw_real *real_ptr_L = real_L;
        LADSPA_Data *in_fifo_ptr_L = in_fifo_L;

        fftw_real *real_ptr_R = real_R;
        LADSPA_Data *in_fifo_ptr_R = in_fifo_R;

        // PERL_END

        unsigned long ctr = 0;
        while(ctr++ < fft_length)
          {
          LADSPA_Data window_element = *window_for_input_ptr++;
          // PERL_BEGIN
          *real_ptr_L++ = *in_fifo_ptr_L++ * window_element;
          *real_ptr_R++ = *in_fifo_ptr_R++ * window_element;
          // PERL_END
	  } // while(ctr++ < fft_length)
        }

      // Run the real->complex transform
#ifdef FFTW3
      // PERL_BEGIN
      //fprintf(stderr, "*(real_L + input_buffer_pos)=%G\n", (LADSPA_Data)(*(real_L + input_buffer_pos)));

      fftwf_execute(plan_rc_L);
      //fprintf(stderr, "*(real_R + input_buffer_pos)=%G\n", (LADSPA_Data)(*(real_R + input_buffer_pos)));

      fftwf_execute(plan_rc_R);
      // PERL_END
#else
      // PERL_BEGIN
      rfftw_one(plan_rc, real_L, comp_L);
      rfftw_one(plan_rc, real_R, comp_R);
      // PERL_END
#endif
	
      // Multiply the bins magnitudes by the coefficients

      // PERL_BEGIN
      // the *_rp_ptr* and *_ip_ptr* pointers will be running in the opposite
      // directions
      //fprintf(stderr, "just after direct FFT: *(comp_L + input_buffer_pos)=%G\n", (LADSPA_Data)(*(comp_L + input_buffer_pos)));

      fftw_real *comp_rp_ptr_L = comp_L + 1;
      // the above is real part pointer - the first element after the DC
      // in comp_L array - DC is left intact

      fftw_real *comp_ip_ptr_L = comp_L + fft_length - 1;
      // the above is imaginary part pointer - end of comp_L array
      //fprintf(stderr, "just after direct FFT: *(comp_R + input_buffer_pos)=%G\n", (LADSPA_Data)(*(comp_R + input_buffer_pos)));

      fftw_real *comp_rp_ptr_R = comp_R + 1;
      // the above is real part pointer - the first element after the DC
      // in comp_R array - DC is left intact

      fftw_real *comp_ip_ptr_R = comp_R + fft_length - 1;
      // the above is imaginary part pointer - end of comp_R array
      // PERL_END

      float *cos_times_gain_ptr = cos_times_gain;

        {
        unsigned long ctr = 1;
        while(ctr++ < half_fft_length)
          {
          float cos_times_gain = *cos_times_gain_ptr++;

          // PERL_BEGIN
          *comp_rp_ptr_L++ = *comp_rp_ptr_L * cos_times_gain;
          *comp_ip_ptr_L-- = *comp_ip_ptr_L * cos_times_gain;

          *comp_rp_ptr_R++ = *comp_rp_ptr_R * cos_times_gain;
          *comp_ip_ptr_R-- = *comp_ip_ptr_R * cos_times_gain;

          // PERL_END
	  } // while(ctr++ < half_fft_length)
        }
	
      // Run the complex->real transform
#ifdef FFTW3
      // PERL_BEGIN
      //fprintf(stderr, "just after multiplying by gain: *(comp_L + input_buffer_pos)=%G\n", (LADSPA_Data)(*(comp_L + input_buffer_pos)));

      fftwf_execute(plan_cr_L);
      //fprintf(stderr, "just after multiplying by gain: *(comp_R + input_buffer_pos)=%G\n", (LADSPA_Data)(*(comp_R + input_buffer_pos)));

      fftwf_execute(plan_cr_R);
      // PERL_END
#else
      // PERL_BEGIN
      rfftw_one(plan_cr, comp_L, real_L);
      rfftw_one(plan_cr, comp_R, real_R);
      // PERL_END
#endif
      // multiply results by the window coefficients and put them
      // into output accumulator
      // PERL_BEGIN
      LADSPA_Data *out_accum_ptr_L = out_accum_L;
      fftw_real *real_ptr_L = real_L;
      LADSPA_Data *out_accum_ptr_R = out_accum_R;
      fftw_real *real_ptr_R = real_R;
      // PERL_END

        {
        unsigned long ctr = 0;
        while(ctr++ < fft_length)
          {
          // PERL_BEGIN
          *out_accum_ptr_L++ += *real_ptr_L++;
          *out_accum_ptr_R++ += *real_ptr_R++;
          // PERL_END
          } // while(ctr++ < fft_length)
        }

      // PERL_BEGIN
      //fprintf(stderr, "just after multiplying by output window: *(out_accum_L + input_buffer_pos)=%G\n", (LADSPA_Data)(*(out_accum_L + input_buffer_pos)));

      memcpy(out_fifo_L, out_accum_L, number_of_bytes_in_step);
      //fprintf(stderr, "just after multiplying by output window: *(out_accum_R + input_buffer_pos)=%G\n", (LADSPA_Data)(*(out_accum_R + input_buffer_pos)));

      memcpy(out_fifo_R, out_accum_R, number_of_bytes_in_step);
      // PERL_END


      // Shift output accumulator
      // PERL_BEGIN
      memmove
        (
        out_accum_L,
        out_accum_L + step_size,
        number_of_bytes_in_fft_length
        );
      memmove
        (
        out_accum_R,
        out_accum_R + step_size,
        number_of_bytes_in_fft_length
        );
      // PERL_END

      // Shift input fifo
      // PERL_BEGIN
      memmove
        (
        in_fifo_L,
        in_fifo_L + step_size,
        number_of_bytes_in_fft_latency
        );
      memmove
        (
        in_fifo_R,
        in_fifo_R + step_size,
        number_of_bytes_in_fft_latency
        );
      // PERL_END

      fifo_pos = fft_latency;
      input_buffer_pos += number_of_steps_until_next_fft;
      }
    } // while(input_buffer_pos < sample_count)


  // store back positions

  (*plugin_data).fifo_pos = fifo_pos;
  } // static void runMChMBEq(LADSPA_Handle instance, unsigned long sample_count)

void _init()
  {
#ifdef ENABLE_NLS
#define D_(s) dgettext(PACKAGE, s)
  setlocale(LC_ALL, "");
  bindtextdomain(PACKAGE, PACKAGE_LOCALE_DIR);
#else
#define D_(s) (s)
#endif


  MALLOC_AND_CHECK(LADSPA_Descriptor *, MChMBEqDescriptor, sizeof(LADSPA_Descriptor), __LINE__)

  //fprintf(stderr,"MChMBEqDescriptor=%08lx\n", (unsigned long)MChMBEqDescriptor);

  if(MChMBEqDescriptor)
    {
    // PERL_BEGIN
    (*MChMBEqDescriptor).UniqueID = 119700;
    (*MChMBEqDescriptor).Label = "MChMBEq";
    (*MChMBEqDescriptor).Name = D_("Multichannel Multiband EQ");
    // PERL_END
    (*MChMBEqDescriptor).Properties = LADSPA_PROPERTY_HARD_RT_CAPABLE;
    (*MChMBEqDescriptor).Maker = "Sergei Steshenko (sergei_steshenko@pop3.ru)";
    (*MChMBEqDescriptor).Copyright = "GPL, may be reduced to LGPL";
    (*MChMBEqDescriptor).PortCount = NUMBER_OF_PORTS;

    LADSPA_PortDescriptor *port_descriptors;
    CALLOC_AND_CHECK(LADSPA_PortDescriptor *, port_descriptors, (*MChMBEqDescriptor).PortCount, sizeof(LADSPA_PortDescriptor), __LINE__)

    (*MChMBEqDescriptor).PortDescriptors =
    (const LADSPA_PortDescriptor *)port_descriptors;

    LADSPA_PortRangeHint *port_range_hints;
    CALLOC_AND_CHECK(LADSPA_PortRangeHint *, port_range_hints, (*MChMBEqDescriptor).PortCount, sizeof(LADSPA_PortRangeHint), __LINE__)

    (*MChMBEqDescriptor).PortRangeHints =
    (const LADSPA_PortRangeHint *)port_range_hints;

    char **port_names;
    CALLOC_AND_CHECK(char **, port_names, (*MChMBEqDescriptor).PortCount, sizeof(char *), __LINE__)

    (*MChMBEqDescriptor).PortNames = (const char **)port_names;

    // PERL_BEGIN
    port_descriptors[MBEQ_BAND_LOW] = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    port_names[MBEQ_BAND_LOW] = D_("low Hz gain");
    port_range_hints[MBEQ_BAND_LOW].HintDescriptor =
    LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE | LADSPA_HINT_DEFAULT_0;
    port_range_hints[MBEQ_BAND_LOW].LowerBound = MIN_GAIN;
    port_range_hints[MBEQ_BAND_LOW].UpperBound = MAX_GAIN;


    port_descriptors[MBEQ_BAND_00] = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    port_names[MBEQ_BAND_00] = D_("20Hz gain");
    port_range_hints[MBEQ_BAND_00].HintDescriptor =
    LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE | LADSPA_HINT_DEFAULT_0;
    port_range_hints[MBEQ_BAND_00].LowerBound = MIN_GAIN;
    port_range_hints[MBEQ_BAND_00].UpperBound = MAX_GAIN;


    port_descriptors[MBEQ_BAND_01] = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    port_names[MBEQ_BAND_01] = D_("28.2842712474619Hz gain");
    port_range_hints[MBEQ_BAND_01].HintDescriptor =
    LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE | LADSPA_HINT_DEFAULT_0;
    port_range_hints[MBEQ_BAND_01].LowerBound = MIN_GAIN;
    port_range_hints[MBEQ_BAND_01].UpperBound = MAX_GAIN;


    port_descriptors[MBEQ_BAND_02] = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    port_names[MBEQ_BAND_02] = D_("40Hz gain");
    port_range_hints[MBEQ_BAND_02].HintDescriptor =
    LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE | LADSPA_HINT_DEFAULT_0;
    port_range_hints[MBEQ_BAND_02].LowerBound = MIN_GAIN;
    port_range_hints[MBEQ_BAND_02].UpperBound = MAX_GAIN;


    port_descriptors[MBEQ_BAND_03] = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    port_names[MBEQ_BAND_03] = D_("50.3968419957949Hz gain");
    port_range_hints[MBEQ_BAND_03].HintDescriptor =
    LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE | LADSPA_HINT_DEFAULT_0;
    port_range_hints[MBEQ_BAND_03].LowerBound = MIN_GAIN;
    port_range_hints[MBEQ_BAND_03].UpperBound = MAX_GAIN;


    port_descriptors[MBEQ_BAND_04] = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    port_names[MBEQ_BAND_04] = D_("63.496042078728Hz gain");
    port_range_hints[MBEQ_BAND_04].HintDescriptor =
    LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE | LADSPA_HINT_DEFAULT_0;
    port_range_hints[MBEQ_BAND_04].LowerBound = MIN_GAIN;
    port_range_hints[MBEQ_BAND_04].UpperBound = MAX_GAIN;


    port_descriptors[MBEQ_BAND_05] = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    port_names[MBEQ_BAND_05] = D_("80Hz gain");
    port_range_hints[MBEQ_BAND_05].HintDescriptor =
    LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE | LADSPA_HINT_DEFAULT_0;
    port_range_hints[MBEQ_BAND_05].LowerBound = MIN_GAIN;
    port_range_hints[MBEQ_BAND_05].UpperBound = MAX_GAIN;


    port_descriptors[MBEQ_BAND_06] = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    port_names[MBEQ_BAND_06] = D_("100.79368399159Hz gain");
    port_range_hints[MBEQ_BAND_06].HintDescriptor =
    LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE | LADSPA_HINT_DEFAULT_0;
    port_range_hints[MBEQ_BAND_06].LowerBound = MIN_GAIN;
    port_range_hints[MBEQ_BAND_06].UpperBound = MAX_GAIN;


    port_descriptors[MBEQ_BAND_07] = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    port_names[MBEQ_BAND_07] = D_("126.992084157456Hz gain");
    port_range_hints[MBEQ_BAND_07].HintDescriptor =
    LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE | LADSPA_HINT_DEFAULT_0;
    port_range_hints[MBEQ_BAND_07].LowerBound = MIN_GAIN;
    port_range_hints[MBEQ_BAND_07].UpperBound = MAX_GAIN;


    port_descriptors[MBEQ_BAND_08] = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    port_names[MBEQ_BAND_08] = D_("160Hz gain");
    port_range_hints[MBEQ_BAND_08].HintDescriptor =
    LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE | LADSPA_HINT_DEFAULT_0;
    port_range_hints[MBEQ_BAND_08].LowerBound = MIN_GAIN;
    port_range_hints[MBEQ_BAND_08].UpperBound = MAX_GAIN;


    port_descriptors[MBEQ_BAND_09] = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    port_names[MBEQ_BAND_09] = D_("201.58736798318Hz gain");
    port_range_hints[MBEQ_BAND_09].HintDescriptor =
    LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE | LADSPA_HINT_DEFAULT_0;
    port_range_hints[MBEQ_BAND_09].LowerBound = MIN_GAIN;
    port_range_hints[MBEQ_BAND_09].UpperBound = MAX_GAIN;


    port_descriptors[MBEQ_BAND_10] = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    port_names[MBEQ_BAND_10] = D_("253.984168314912Hz gain");
    port_range_hints[MBEQ_BAND_10].HintDescriptor =
    LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE | LADSPA_HINT_DEFAULT_0;
    port_range_hints[MBEQ_BAND_10].LowerBound = MIN_GAIN;
    port_range_hints[MBEQ_BAND_10].UpperBound = MAX_GAIN;


    port_descriptors[MBEQ_BAND_11] = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    port_names[MBEQ_BAND_11] = D_("320Hz gain");
    port_range_hints[MBEQ_BAND_11].HintDescriptor =
    LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE | LADSPA_HINT_DEFAULT_0;
    port_range_hints[MBEQ_BAND_11].LowerBound = MIN_GAIN;
    port_range_hints[MBEQ_BAND_11].UpperBound = MAX_GAIN;


    port_descriptors[MBEQ_BAND_12] = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    port_names[MBEQ_BAND_12] = D_("403.174735966359Hz gain");
    port_range_hints[MBEQ_BAND_12].HintDescriptor =
    LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE | LADSPA_HINT_DEFAULT_0;
    port_range_hints[MBEQ_BAND_12].LowerBound = MIN_GAIN;
    port_range_hints[MBEQ_BAND_12].UpperBound = MAX_GAIN;


    port_descriptors[MBEQ_BAND_13] = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    port_names[MBEQ_BAND_13] = D_("507.968336629824Hz gain");
    port_range_hints[MBEQ_BAND_13].HintDescriptor =
    LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE | LADSPA_HINT_DEFAULT_0;
    port_range_hints[MBEQ_BAND_13].LowerBound = MIN_GAIN;
    port_range_hints[MBEQ_BAND_13].UpperBound = MAX_GAIN;


    port_descriptors[MBEQ_BAND_14] = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    port_names[MBEQ_BAND_14] = D_("640Hz gain");
    port_range_hints[MBEQ_BAND_14].HintDescriptor =
    LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE | LADSPA_HINT_DEFAULT_0;
    port_range_hints[MBEQ_BAND_14].LowerBound = MIN_GAIN;
    port_range_hints[MBEQ_BAND_14].UpperBound = MAX_GAIN;


    port_descriptors[MBEQ_BAND_15] = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    port_names[MBEQ_BAND_15] = D_("806.349471932719Hz gain");
    port_range_hints[MBEQ_BAND_15].HintDescriptor =
    LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE | LADSPA_HINT_DEFAULT_0;
    port_range_hints[MBEQ_BAND_15].LowerBound = MIN_GAIN;
    port_range_hints[MBEQ_BAND_15].UpperBound = MAX_GAIN;


    port_descriptors[MBEQ_BAND_16] = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    port_names[MBEQ_BAND_16] = D_("1015.93667325965Hz gain");
    port_range_hints[MBEQ_BAND_16].HintDescriptor =
    LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE | LADSPA_HINT_DEFAULT_0;
    port_range_hints[MBEQ_BAND_16].LowerBound = MIN_GAIN;
    port_range_hints[MBEQ_BAND_16].UpperBound = MAX_GAIN;


    port_descriptors[MBEQ_BAND_17] = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    port_names[MBEQ_BAND_17] = D_("1280Hz gain");
    port_range_hints[MBEQ_BAND_17].HintDescriptor =
    LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE | LADSPA_HINT_DEFAULT_0;
    port_range_hints[MBEQ_BAND_17].LowerBound = MIN_GAIN;
    port_range_hints[MBEQ_BAND_17].UpperBound = MAX_GAIN;


    port_descriptors[MBEQ_BAND_18] = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    port_names[MBEQ_BAND_18] = D_("1612.69894386544Hz gain");
    port_range_hints[MBEQ_BAND_18].HintDescriptor =
    LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE | LADSPA_HINT_DEFAULT_0;
    port_range_hints[MBEQ_BAND_18].LowerBound = MIN_GAIN;
    port_range_hints[MBEQ_BAND_18].UpperBound = MAX_GAIN;


    port_descriptors[MBEQ_BAND_19] = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    port_names[MBEQ_BAND_19] = D_("2031.8733465193Hz gain");
    port_range_hints[MBEQ_BAND_19].HintDescriptor =
    LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE | LADSPA_HINT_DEFAULT_0;
    port_range_hints[MBEQ_BAND_19].LowerBound = MIN_GAIN;
    port_range_hints[MBEQ_BAND_19].UpperBound = MAX_GAIN;


    port_descriptors[MBEQ_BAND_20] = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    port_names[MBEQ_BAND_20] = D_("2560Hz gain");
    port_range_hints[MBEQ_BAND_20].HintDescriptor =
    LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE | LADSPA_HINT_DEFAULT_0;
    port_range_hints[MBEQ_BAND_20].LowerBound = MIN_GAIN;
    port_range_hints[MBEQ_BAND_20].UpperBound = MAX_GAIN;


    port_descriptors[MBEQ_BAND_21] = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    port_names[MBEQ_BAND_21] = D_("3225.39788773088Hz gain");
    port_range_hints[MBEQ_BAND_21].HintDescriptor =
    LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE | LADSPA_HINT_DEFAULT_0;
    port_range_hints[MBEQ_BAND_21].LowerBound = MIN_GAIN;
    port_range_hints[MBEQ_BAND_21].UpperBound = MAX_GAIN;


    port_descriptors[MBEQ_BAND_22] = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    port_names[MBEQ_BAND_22] = D_("4063.74669303859Hz gain");
    port_range_hints[MBEQ_BAND_22].HintDescriptor =
    LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE | LADSPA_HINT_DEFAULT_0;
    port_range_hints[MBEQ_BAND_22].LowerBound = MIN_GAIN;
    port_range_hints[MBEQ_BAND_22].UpperBound = MAX_GAIN;


    port_descriptors[MBEQ_BAND_23] = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    port_names[MBEQ_BAND_23] = D_("5120Hz gain");
    port_range_hints[MBEQ_BAND_23].HintDescriptor =
    LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE | LADSPA_HINT_DEFAULT_0;
    port_range_hints[MBEQ_BAND_23].LowerBound = MIN_GAIN;
    port_range_hints[MBEQ_BAND_23].UpperBound = MAX_GAIN;


    port_descriptors[MBEQ_BAND_24] = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    port_names[MBEQ_BAND_24] = D_("6450.79577546175Hz gain");
    port_range_hints[MBEQ_BAND_24].HintDescriptor =
    LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE | LADSPA_HINT_DEFAULT_0;
    port_range_hints[MBEQ_BAND_24].LowerBound = MIN_GAIN;
    port_range_hints[MBEQ_BAND_24].UpperBound = MAX_GAIN;


    port_descriptors[MBEQ_BAND_25] = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    port_names[MBEQ_BAND_25] = D_("8127.49338607718Hz gain");
    port_range_hints[MBEQ_BAND_25].HintDescriptor =
    LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE | LADSPA_HINT_DEFAULT_0;
    port_range_hints[MBEQ_BAND_25].LowerBound = MIN_GAIN;
    port_range_hints[MBEQ_BAND_25].UpperBound = MAX_GAIN;


    port_descriptors[MBEQ_BAND_26] = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    port_names[MBEQ_BAND_26] = D_("10240Hz gain");
    port_range_hints[MBEQ_BAND_26].HintDescriptor =
    LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE | LADSPA_HINT_DEFAULT_0;
    port_range_hints[MBEQ_BAND_26].LowerBound = MIN_GAIN;
    port_range_hints[MBEQ_BAND_26].UpperBound = MAX_GAIN;


    port_descriptors[MBEQ_BAND_27] = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    port_names[MBEQ_BAND_27] = D_("12901.5915509235Hz gain");
    port_range_hints[MBEQ_BAND_27].HintDescriptor =
    LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE | LADSPA_HINT_DEFAULT_0;
    port_range_hints[MBEQ_BAND_27].LowerBound = MIN_GAIN;
    port_range_hints[MBEQ_BAND_27].UpperBound = MAX_GAIN;


    port_descriptors[MBEQ_BAND_28] = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    port_names[MBEQ_BAND_28] = D_("16254.9867721544Hz gain");
    port_range_hints[MBEQ_BAND_28].HintDescriptor =
    LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE | LADSPA_HINT_DEFAULT_0;
    port_range_hints[MBEQ_BAND_28].LowerBound = MIN_GAIN;
    port_range_hints[MBEQ_BAND_28].UpperBound = MAX_GAIN;


    port_descriptors[MBEQ_BAND_29] = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    port_names[MBEQ_BAND_29] = D_("20480Hz gain");
    port_range_hints[MBEQ_BAND_29].HintDescriptor =
    LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE | LADSPA_HINT_DEFAULT_0;
    port_range_hints[MBEQ_BAND_29].LowerBound = MIN_GAIN;
    port_range_hints[MBEQ_BAND_29].UpperBound = MAX_GAIN;


    port_descriptors[MBEQ_BAND_HIGH] = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    port_names[MBEQ_BAND_HIGH] = D_("high Hz gain");
    port_range_hints[MBEQ_BAND_HIGH].HintDescriptor =
    LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE | LADSPA_HINT_DEFAULT_0;
    port_range_hints[MBEQ_BAND_HIGH].LowerBound = MIN_GAIN;
    port_range_hints[MBEQ_BAND_HIGH].UpperBound = MAX_GAIN;


    // PERL_END


    // PERL_BEGIN

    port_descriptors[MBEQ_INPUT_L] = LADSPA_PORT_INPUT | LADSPA_PORT_AUDIO;
    port_names[MBEQ_INPUT_L] = D_("Input_L");
    port_range_hints[MBEQ_INPUT_L].HintDescriptor = 0;  

    port_descriptors[MBEQ_INPUT_R] = LADSPA_PORT_INPUT | LADSPA_PORT_AUDIO;
    port_names[MBEQ_INPUT_R] = D_("Input_R");
    port_range_hints[MBEQ_INPUT_R].HintDescriptor = 0;  

    port_descriptors[MBEQ_OUTPUT_L] = LADSPA_PORT_OUTPUT | LADSPA_PORT_AUDIO;
    port_names[MBEQ_OUTPUT_L] = D_("Output_L");
    port_range_hints[MBEQ_OUTPUT_L].HintDescriptor = 0;

    port_descriptors[MBEQ_OUTPUT_R] = LADSPA_PORT_OUTPUT | LADSPA_PORT_AUDIO;
    port_names[MBEQ_OUTPUT_R] = D_("Output_R");
    port_range_hints[MBEQ_OUTPUT_R].HintDescriptor = 0;
    // PERL_END

    (*MChMBEqDescriptor).activate = activateMChMBEq;
    (*MChMBEqDescriptor).cleanup = cleanupMChMBEq;
    (*MChMBEqDescriptor).connect_port = connectPortMChMBEq;
    (*MChMBEqDescriptor).deactivate = NULL;
    (*MChMBEqDescriptor).instantiate = instantiateMChMBEq;
    (*MChMBEqDescriptor).run = runMChMBEq;
    (*MChMBEqDescriptor).run_adding = NULL;
    (*MChMBEqDescriptor).set_run_adding_gain = NULL;
    }
  } // void _init()

void _fini()
  {
  if(MChMBEqDescriptor)
    {
    free((LADSPA_PortDescriptor *)(*MChMBEqDescriptor).PortDescriptors);
    free((char **)(*MChMBEqDescriptor).PortNames);
    free((LADSPA_PortRangeHint *)(*MChMBEqDescriptor).PortRangeHints);
    free(MChMBEqDescriptor);
    }
  } // void _fini()
