//*****************************************************************************
//* Name:
//*      1212api.h
//*
//* Project:
//*      1212 I/O VxD
//*
//* Author:
//*      Bill Jenkins
//*
//* Description:
//*      This file contains declarations required for a Win95 application to
//*      communicate with the Korg 1212 I/O driver.
//*
//* Modification Log:
//*
//*      1.5   08/28/97 Bill
//*      Added k1212CardDebugInfo structure..
//*
//*      1.4   08/11/97 Bill
//*      Changed k1212MaxCards from 16 to 4.
//*
//*      1.3   05/01/97 Bill
//*      Added wave device type enumeration.
//*
//*      1.2   04/14/97 Bill
//*      Added wave device constants.  
//*
//*      1.1   11/15/96 Bill
//*      Initial version.  
//*
//*
//* Copyright (c) 1996 Korg, Inc.
//* All rights reserved.
//*
//* This program is protected as an unpublished work under the U.S.
//* copyright laws.  The above copyright notice is not intended to
//* effect a publication of this work.
//*
//* This program is the confidential and proprietary information of
//* Korg and all its subsidiaries.
//*****************************************************************************

#ifndef K1212API_H
#define K1212API_H

#ifndef K1212STAT_H
#include "1212stat.h"
#endif

// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------
// constants, enums, and type declarations
// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------
#define KORG1212_ID_STRING "KORG1212"

typedef DWORD k1212CardRef;                  // a card reference number is used to identify
                                             //    the target card for the API calls
typedef DWORD CardHandle;                    // a CardHandle is returned by the 
                                             //    open card function
typedef DWORD APIHandle;                     // when an application opens an API channel to
                                             //    the driver, that channel is registered so
                                             //    that the driver can keep track of which 
                                             //    API channels have opened particular cards.
typedef short          K1212Sample;          // channels 0-9 use 16 bit samples
typedef unsigned long  K1212SpdifSample;     // channels 10-11 use 32 bits - only 20 are sent
                                             //  across S/PDIF.
typedef unsigned long  K1212TimeCodeSample;  // holds the ADAT timecode value

#define k1212MaxCards       4
#define k1212NumWaveDevices 6
#define k16BitChannels      10
#define k32BitChannels      2
#define kAudioChannels      (k16BitChannels + k32BitChannels)
#define kPlayBufferFrames   1024

#define k1212MinADCSens     0x7f
#define k1212MaxADCSens     0x00
#define k1212MaxVolume      0x7fff
#define k1212MaxWaveVolume  0xffff
#define k1212MinVolume      0x0000
#define k1212MaxVolInverted 0x8000

// ---------------------------------------------------------------
// The KorgAudioFrame structure contains a single sample for all
// 12 channels.
// ---------------------------------------------------------------
typedef struct KorgAudioFrame {
   K1212Sample          frameData16[k16BitChannels];        
   K1212SpdifSample     frameData32[k32BitChannels];
   K1212TimeCodeSample  timeCodeVal;
} KorgAudioFrame;


// ---------------------------------------------------------------
// The following constants define byte offsets into the audio
// frame for each of the channels.
// ---------------------------------------------------------------
#define K1212Adat1Offset         (0 * sizeof(K1212Sample))
#define K1212Adat2Offset         (K1212Adat1Offset      + sizeof(K1212Sample))
#define K1212Adat3Offset         (K1212Adat2Offset      + sizeof(K1212Sample))
#define K1212Adat4Offset         (K1212Adat3Offset      + sizeof(K1212Sample))
#define K1212Adat5Offset         (K1212Adat4Offset      + sizeof(K1212Sample))
#define K1212Adat6Offset         (K1212Adat5Offset      + sizeof(K1212Sample))
#define K1212Adat7Offset         (K1212Adat6Offset      + sizeof(K1212Sample))
#define K1212Adat8Offset         (K1212Adat7Offset      + sizeof(K1212Sample))
#define K1212AnlgLeftOffset      (K1212Adat8Offset      + sizeof(K1212Sample))
#define K1212AnlgRightOffset     (K1212AnlgLeftOffset   + sizeof(K1212Sample))
#define K1212SPDIFLeftOffset     (K1212AnlgRightOffset  + sizeof(K1212Sample))
#define K1212SPDIFRightOffset    (K1212SPDIFLeftOffset  + sizeof(K1212SpdifSample))
#define K1212AdatTimecodeOffset  (K1212SPDIFRightOffset + sizeof(K1212SpdifSample))

// ---------------------------------------------------------------
// The KorgAudioBuffer structure contains kPlayBufferFrames 
// samples worth of KorgAudioFrames.
// ---------------------------------------------------------------
typedef struct KorgAudioBuffer {
   KorgAudioFrame  bufferData[kPlayBufferFrames];     /* buffer definition */
} KorgAudioBuffer;

typedef KorgAudioBuffer* PKorgAudioBuffer;


// ----------------------------------------------------------------------------
// The following enumeration defines the constants used to select the clock
// source and rate for the card in the SetClockSourceRate command.  
// ----------------------------------------------------------------------------
typedef enum {
   K1212_CLKIDX_AdatAt44_1K = 0,    // selects source as ADAT at 44.1 kHz
   K1212_CLKIDX_AdatAt48K,          // selects source as ADAT at 48 kHz
   K1212_CLKIDX_WordAt44_1K,        // selects source as S/PDIF at 44.1 kHz
   K1212_CLKIDX_WordAt48K,          // selects source as S/PDIF at 48 kHz
   K1212_CLKIDX_LocalAt44_1K,       // selects source as local clock at 44.1 kHz
   K1212_CLKIDX_LocalAt48K,         // selects source as local clock at 48 kHz
   K1212_CLKIDX_Invalid             // used to check validity of the index
} ClockSourceIndex;


// ----------------------------------------------------------------------------
// The following enumeration defines the constants used to select the monitor
// mode for the card in the SetMonitorMode command.
// ----------------------------------------------------------------------------
typedef enum {
   K1212_MONMODE_Off  = 0,     // tells card to turn off monitor mode
   K1212_MONMODE_On            // tells card to turn on monitor mode
} MonitorModeSelector;


// ----------------------------------------------------------------------------
// The following enumeration defines the constants used to select the stop
// mode for the card.
// ----------------------------------------------------------------------------
typedef enum {
   K1212_STOP_NormalStop   = 0,  // tells the card not use the fast Adat stop mode
   K1212_STOP_FastAdatStop       // tells card to use the fast Adat stop mode
} StopModeSelector;


// ----------------------------------------------------------------------------
// The following enumeration defines return codes for DeviceIoControl() calls
// to the Korg 1212 I/O driver.
// ----------------------------------------------------------------------------
typedef enum {
   K1212_CMDRET_Success         = 0,   // command was successfully placed
   K1212_CMDRET_DIOCFailure,           // the DeviceIoControl call failed
   K1212_CMDRET_PMFailure,             // the protected mode call failed
   K1212_CMDRET_FailUnspecified,       // unspecified failure
   K1212_CMDRET_FailBadState,          // the specified command can not be given in
                                       //    the card's current state. (or the wave device's 
                                       //    state)
   K1212_CMDRET_CardUninitialized,     // the card is uninitialized and cannot be used
   K1212_CMDRET_BadIndex,              // an out of range card index was specified
   K1212_CMDRET_BadHandle,             // an invalid card handle was specified
   K1212_CMDRET_NoFillRoutine,         // a play request has been made before a fill routine set
   K1212_CMDRET_FillRoutineInUse,      // can't set a new fill routine while one is in use
   K1212_CMDRET_NoAckFromCard,         // the card never acknowledged a command
   K1212_CMDRET_BadParams,             // bad parameters were provided by the caller

   // --------------------------------------------------------------
   // the following return errors are specific to the wave device
   // driver interface.  These will not be encountered by users of
   // the 32 bit DIOC interface (a.k.a. custom or native API).
   // --------------------------------------------------------------
   K1212_CMDRET_BadDevice,             // the specified wave device was out of range
   K1212_CMDRET_BadFormat              // the specified wave format is unsupported
} K1212CmdRet;


// ------------------------------------------------------------------------
// The following struct is used for reporting card information from the 
// GetCardInfo call.  An array of these structs is also used to report
// all cards configured in the system for the QueryAllCards call. 
// ------------------------------------------------------------------------
typedef struct k1212PublicCardInfo {
   DWORD            busNum;            // PCI bus number on which the card resides
   DWORD            deviceNum;         // device number of the card on the PCI bus
   DWORD            irqNum;            // interrupt level for the card
   CardState        state;             // current state of the card
} k1212PublicCardInfo;

   // ------------------------------------------------------------------------
   // The following struct is used for reporting card debug information.
   // ------------------------------------------------------------------------
typedef struct k1212CardDebugInfo {
   CardState        state;             // current state of the card
   ULONG            irqCount;          // number of IRQs received by the card
   DWORD            cmdRetryCount;     // how many times we had to resend a command
   DWORD            openCount;         // how many times the card has been opened
   ClockSourceIndex clkSrcRate;        // current clock source/rate setting on the card
   KorgAudioBuffer* playBuf0Address;   // address of the 1st playback buffer
   DWORD            devNode;           // device node of the card from the config mgr.
} k1212CardDebugInfo;


// ----------------------------------------------------------------------------
// The following enumeration defines the wave device types.
// ----------------------------------------------------------------------------
typedef enum {
   K1212_WAVETYPE_INPUT = 0,        // specifies a wave input (recording) device
   K1212_WAVETYPE_OUTPUT            // specifies a wave output (playback) device
} K1212WaveDeviceType;


#endif // K1212API_H
