//*****************************************************************************
//* Name:
//*      KORG1212.c - main module for VxD KORG1212
//*
//* Project:
//*      1212 I/O VxD
//*
//* Author:
//*      Bill Jenkins
//*
//* Description:
//*      This file processes all the control messages sent by the VMM.
//*      This file also implements the DeviceIoControl handling function to provide
//*      the interface for Win32 applications to call this VxD.  Intermediate
//*      functions for the DIOC interface are defined here as well.
//*
//*
//* Modification Log:
//*
//*      1.10  10/16/97 Bill
//*      (v 1.0)  Removed date timeout - applied only to beta versions.
//*
//*      1.9   09/15/97 Bill
//*      (v 1.0B7)  Added K1212_CLEAR_API and related calls.  Made new function
//*      for clearing the fill routine.
//*
//*      1.8   08/20/97 Bill
//*      (v 1.0B6)  Added K1212_ENABLE_AUTO_WAVE_SYNC and related calls.
//*
//*      1.7   08/08/97 Bill
//*      (v 1.0B5)  Changed stop card and monitor off processing to use the new
//*      command register in the shared buffer, rather than sending a message to
//*      the card.
//*
//*      1.6   06/08/97 Bill
//*      (v 1.08)  Added K1212PM_BREAKLOOP call.
//*
//*      1.5   06/04/97 Bill
//*      (v 1.06)  Added k1212WaveVolUpdateFuncPtr global member plus processing of
//*      registration messages from K1212Wav.exe for wave device volume updates.
//*
//*      1.4   06/03/97 Bill
//*      (v 1.05)  Added K1212PM_POSTMSG_TO_FILE call for posting messages to a file.
//*
//*      1.3   05/23/97 Bill
//*      (v 1.04)  Moved setCurFillBufNum() out of SetupForPlay() in preparation
//*      for the implementation of the wave device pre-fill mechanism.
//*
//*      1.2   12/27/96 Bill
//*      Modified to support multiple cards.
//*
//*      1.1   12/10/96 Bill
//*      Initial version created.
//*
//*
//* Copyright (c) 1996 Korg, Inc.
//* All rights reserved.
//*
//* This program is protected as an unpublished work under the U.S.
//* copyright laws.  The above copyright notice is not intended to
//* effect a publication of this work.
//*
//* This program is the confidential and proprietary information of
//* Korg and all its subsidiaries.
//*****************************************************************************

#include <vtoolsc.h>
#include <vmm.h>
#include <debug.h>
#include <vmmreg.h>

#define   DEVICE_MAIN
#include  "korg1212.h"
#undef    DEVICE_MAIN

#ifndef  K1212API_H
#include "1212api.h"
#endif
#ifndef  K1212APIP_H
#include "1212apip.h"
#endif
#ifndef  K1212INTF_H
#include "1212intf.h"
#endif
#ifndef  K1212CM_H
#include "1212cm.h"
#endif
#ifndef  K1212CFG_H
#include "1212cfg.h"
#endif
#ifndef  K1212CARD_H
#include "1212card.h"
#endif
#ifndef  K1212DNLD_H
#include "1212dnld.h"
#endif
#ifndef  K1212IFPR_H
#include "1212ifpr.h"
#endif
#ifndef  K1212IFPR_H
#include "1212sens.h"
#endif
#ifndef  K1212PCI_H
#include "1212pci.h"       // needed for accessing the PCI configuration space
#endif
#ifndef  K1212WAVEAPI_H
#include "1212WaveApi.h"
#endif
#ifndef  K1212WAVE_H
#include "1212wave.h"
#endif


// --------------------------------------------------------------
// global declarations
// --------------------------------------------------------------
DWORD apiCount = 0;     // tracks the number of API objects that have
                        //    registered with the driver.

PVOID                k1212WaveVolUpdateFuncPtr = 0;   // set when K1212Wav.exe registers.  used for
                                                      //    notifying of wave volume updates via the
                                                      //    wave device interface
THREADHANDLE         k1212WavThreadHandle      = 0;   // also needed for the APC used to notify
                                                      //    K1212wav.exe
k1212WavVolUpdate*   wavVolUpdateDataPtr       = 0;   // used for passing data to k1212Wav.exe
PVOID                wavVolUpdateThisPtr       = 0;   // used to store the this pointer for k1212Wav.exe object


// --------------------------------------------------------------
// function declarations
// --------------------------------------------------------------
void  K1212FillInCardInfo  (k1212CardInfo* allCards);
BOOL  K1212OpenCard        (DWORD     cardIndex,
                            APIHandle apiHandle
      );
void  CleanupAfterAPIClose (APIHandle apiHandle);
void  ClearAPI             (DWORD cardIndex);
void  ClearFillRoutine     (DWORD cardIndex);
BOOL  K1212CloseCard       (DWORD cardIndex);
BOOL  K1212SetMonitorMode  (DWORD cardIndex,
                            MonitorModeSelector mode
      );
// VOID __stdcall PM_Vendor_Api_Handler(VMHANDLE hVM, PVOID pRefdata, PCLIENT_STRUCT pcrs);

BOOL MapPMtoDIOC(IOCTLPARAMS* IoCtl, k1212PmReq* pReq);
BOOL MapDIOCtoPM(IOCTLPARAMS* IoCtl, k1212PmReq* pReq);

extern void InitWaveInRetBuffers(void);
extern void InitWaveOutRetBuffers(void);

// ----------------------------------------------------------------
// virtual device declaration and control handler/dispatcher stuff
// ----------------------------------------------------------------
Declare_Virtual_Device(KORG1212)
CHAR VendorString[] = KORG1212_ID_STRING;

DefineControlHandler (INIT_COMPLETE,           OnInitComplete);
DefineControlHandler (SYS_DYNAMIC_DEVICE_INIT, OnSysDynamicDeviceInit);
DefineControlHandler (SYS_DYNAMIC_DEVICE_EXIT, OnSysDynamicDeviceExit);
DefineControlHandler (PNP_NEW_DEVNODE,         OnPnpNewDevnode);
DefineControlHandler (W32_DEVICEIOCONTROL,     OnW32Deviceiocontrol);
DefineControlHandler (BEGIN_PM_APP,            OnBeginPmApp);
DefineControlHandler (END_PM_APP,              OnEndPmApp);


BOOL __cdecl ControlDispatcher(
   DWORD dwControlMessage,
   DWORD EBX,
   DWORD EDX,
   DWORD ESI,
   DWORD EDI,
   DWORD ECX)
{
   START_CONTROL_DISPATCH

      ON_INIT_COMPLETE           (OnInitComplete);
      ON_SYS_DYNAMIC_DEVICE_INIT (OnSysDynamicDeviceInit);
      ON_SYS_DYNAMIC_DEVICE_EXIT (OnSysDynamicDeviceExit);
      ON_PNP_NEW_DEVNODE         (OnPnpNewDevnode);
      ON_W32_DEVICEIOCONTROL     (OnW32Deviceiocontrol);

   END_CONTROL_DISPATCH

   return TRUE;
}


// ----------------------------------------------------------
// the following array is used to convert the clock
// source/rate index from the application to the actual value
// sent to the card.  The order of these must match the
// order of the ClockSourceIndex enumeration defined in the
// 1212api.h header file.  This array is used in the
// SET_CLOCK_SOURCE_RATE call.
// ----------------------------------------------------------
WORD  ClockSourceSelector[] = {0x8000,   // selects source as ADAT at 44.1 kHz
                               0x0000,   // selects source as ADAT at 48 kHz
                               0x8001,   // selects source as S/PDIF at 44.1 kHz
                               0x0001,   // selects source as S/PDIF at 48 kHz
                               0x8002,   // selects source as local clock at 44.1 kHz
                               0x0002    // selects source as local clock at 48 kHz
                              };



//*************************************************************************
//*
//* OnInitComplete - Handler for the VMM init complete message
//*
//*************************************************************************
BOOL OnInitComplete(VMHANDLE hVM, PCHAR CommandTail)
{
   DWORD cardIndex;

   #ifdef DEBUG
      Debug_Printf_Service("Korg 1212 Driver: Received the Init Complete message\n");
   #endif // DEBUG

   // ---------------------------------------------------------------------
   // if the DSP download hasn't been started, now is the time to do it.
   // for all possible cards, if the card is configured, and the microcode
   // has not been downloaded, it is now initiated.
   // ---------------------------------------------------------------------
   for (cardIndex = 0; cardIndex < k1212MaxCards; cardIndex++) {
      if (getDevNode(cardIndex)) {
         if (getCardState(cardIndex) == K1212_STATE_UNINITIALIZED) {
            downloadDSPCode(cardIndex);
         }
      }
   }
   return TRUE;
}



//**************************************************************************
//*
//* OnSysDynamicDeviceInit -  This procedure gets called the first time the
//*                           DLVxD is loaded.
//*
//*
//**************************************************************************
BOOL OnSysDynamicDeviceInit()
{
   DWORD cardIndex;
   DWORD date;

   // ------------------------------------------------------------------------
   // Check today's date.  For version 1.0B7, we expire after Oct. 31, 1997
   // ------------------------------------------------------------------------
//   VTD_Get_Date_And_Time(&date);
//   #ifdef DEBUG
//      Debug_Printf_Service("Korg 1212 Driver: Current date is: %08x\n", date);
//   #endif // DEBUG
//   if (date > 6513) {
//      return FALSE;
//   }

   // ------------------------------------------------------------------
   // Initialize the card data array to indicate no cards configured.
   // ------------------------------------------------------------------
   for (cardIndex = 0; cardIndex < k1212MaxCards; cardIndex++) {
      initCardData(cardIndex);
   }
   InitWaveInRetBuffers();
   InitWaveOutRetBuffers();

   #ifdef DEBUG
      Debug_Printf_Service("Korg 1212 Driver: Received SysDynDevInit message\n");
   #endif // DEBUG

   return TRUE;
}



//***************************************************************************
//*
//* OnSysDynamicDeviceExit -  This procedure gets called when the DLVxD is
//*                           unloading
//*
//*
//***************************************************************************
BOOL OnSysDynamicDeviceExit()
{
   // ------------------------------------------------------------------
   // This procedure must deallocate any global data allocated by
   // Korg1212_SysDynDevInit.
   // Korg1212.vxd will be unloaded by Configuration Manager when
   // the ConfigHandler or the EnumHandler or the NewDevNode return
   // an error code that is "OR"ed with CR_UNLOAD.
   // ------------------------------------------------------------------
   #ifdef DEBUG
      Debug_Printf_Service("Korg 1212 Driver: Received SysDynDevExit message\n");
   #endif // DEBUG
   return(TRUE);
}



//***************************************************************************
//*
//* OnPnpNewDevnode - This procedure gets called when a new devnode
//*                   has been created
//*
//* ENTRY:   DevNode is the new devnode that has just been created.
//*
//*          LoadType is the type of functionality configuration manager
//*          (or a devloader) wants this DLVxD to handle. It can be
//*          devloader, driver or enumerator.
//*
//* EXIT:    Standard config manager return value.
//*
//***************************************************************************
CONFIGRET OnPnpNewDevnode(DEVNODE Devnode, DWORD LoadType)
{
   // This function only handles the device loader case.  In this
   // case, it registers the 1212io's configuration handler as the
   // device driver.
   //
   switch (LoadType) {

     case DLVXD_LOAD_DEVLOADER:
     case DLVXD_LOAD_DRIVER:

         MMDEVLDR_Register_Device_Driver(Devnode,
                                         Korg1212_ConfigHandler,
                                         0
         );
         return CR_SUCCESS;

      default:
         return CR_DEFAULT;
   }
}




//*************************************************************************
//*
//* PM_Api_Handler - Handler for the PM API.
//*                  This routine takes the structure from the caller, and
//*                  simulates a device IO ctl call.
//*
//*************************************************************************
VOID __cdecl PM_Api_Handler(VMHANDLE hVM, PCLIENT_STRUCT pcrs)
{
   k1212PmReq*  pReq;
   IOCTLPARAMS  IoCtl;
   k1212DIOCIn  diocIn;
   k1212DIOCOut diocOut;
   DWORD        byteReturned;

   pReq = Map_Flat(CLIENT_ES, CLIENT_DI);

   if (pReq == NULL) {
      return;
   }

   // ----------------------------------------------------------------------------
   // make sure the card index and device ID are valid
   // ----------------------------------------------------------------------------
   if (!VerifyCardIndex(pReq->inBuf.cardIndex)) {
      pReq->outBuf.returnValue = K1212_CMDRET_BadIndex;
      return;
   }
   if (!VerifyDeviceIndex(pReq->inBuf.deviceID)) {
      pReq->outBuf.returnValue = K1212_CMDRET_BadDevice;
      return;
   }

   // ----------------------------------------------------------------------------
   // first determine whether the requested function requires wave device specific
   // handling, or whether it is to be mapped to the 32 bit DIOC form and handled
   // there.
   // ----------------------------------------------------------------------------
   switch (pReq->functionCode) {

      case K1212PM_ENABLE_WAVEOUTDEVS:
         K1212EnableWaveOutputDevices(&(pReq->inBuf), &(pReq->outBuf));
         return;

      case K1212PM_ENABLE_WAVEINDEVS:
         K1212EnableWaveInputDevices(&(pReq->inBuf), &(pReq->outBuf));
         return;

      case K1212PM_DISABLE_WAVEOUTDEVS:
         K1212DisableWaveOutputDevices(&(pReq->inBuf), &(pReq->outBuf));
         return;

      case K1212PM_DISABLE_WAVEINDEVS:
         K1212DisableWaveInputDevices(&(pReq->inBuf), &(pReq->outBuf));
         return;

      case K1212PM_OPEN_WAVEOUTDEV:
         K1212OpenWaveOutputDevice(&(pReq->inBuf), &(pReq->outBuf));
         return;

      case K1212PM_OPEN_WAVEINDEV:
         K1212OpenWaveInputDevice(&(pReq->inBuf), &(pReq->outBuf));
         return;

      case K1212PM_CLOSE_WAVEOUTDEV:
         K1212CloseWaveOutputDevice(&(pReq->inBuf), &(pReq->outBuf));
         return;

      case K1212PM_CLOSE_WAVEINDEV:
         K1212CloseWaveInputDevice(&(pReq->inBuf), &(pReq->outBuf));
         return;

      case K1212PM_CHECK_INPUT_FORMAT:
         K1212CheckInputFormat(&(pReq->inBuf), &(pReq->outBuf));
         return;

      case K1212PM_CHECK_OUTPUT_FORMAT:
         K1212CheckOutputFormat(&(pReq->inBuf), &(pReq->outBuf));
         return;

      case K1212PM_ADD_RECORD_BUFFER:
         K1212AddRecordBuffer(&(pReq->inBuf), &(pReq->outBuf));
         return;

      case K1212PM_ADD_PLAY_BUFFER:
         K1212AddPlayBuffer(&(pReq->inBuf), &(pReq->outBuf));
         return;

      case K1212PM_START_WAVEIN_DEV:
         K1212StartWaveInDevice(&(pReq->inBuf), &(pReq->outBuf));
         return;

      case K1212PM_STOP_WAVEIN_DEV:
         K1212StopWaveInDevice(&(pReq->inBuf), &(pReq->outBuf));
         return;

      case K1212PM_PAUSE_WAVEOUT_DEV:
         K1212PauseWaveOutDevice(&(pReq->inBuf), &(pReq->outBuf));
         return;

      case K1212PM_RESUME_WAVEOUT_DEV:
         K1212ResumeWaveOutDevice(&(pReq->inBuf), &(pReq->outBuf));
         return;

      case K1212PM_RESET_WAVE_DEV:
         K1212ResetWaveDevice(&(pReq->inBuf), &(pReq->outBuf));
         return;

      case K1212PM_SET_MASTER_VOLUME:
         K1212SetMasterVolume(&(pReq->inBuf), &(pReq->outBuf));
         return;

      case K1212PM_SET_WAVEOUT_VOLUME:
         K1212SetWaveOutDeviceVolume(&(pReq->inBuf), &(pReq->outBuf));
         return;

      case K1212PM_QUERY_WAVEOUT_DEVICE_MUTE:
         K1212QueryWaveOutDeviceMute(&(pReq->inBuf), &(pReq->outBuf));
         return;

      case K1212PM_MUTE_WAVEOUT_DEVICE:
         K1212MuteWaveOutDevice(&(pReq->inBuf), &(pReq->outBuf));
         return;

      case K1212PM_UNMUTE_WAVEOUT_DEVICE:
         K1212UnmuteWaveOutDevice(&(pReq->inBuf), &(pReq->outBuf));
         return;

      case K1212PM_GET_MASTER_VOLUME:
         K1212GetMasterVolume(&(pReq->inBuf), &(pReq->outBuf));
         return;

      case K1212PM_GET_WAVEOUT_VOLUME:
         K1212GetWaveOutDeviceVolume(&(pReq->inBuf), &(pReq->outBuf));
         return;

      case K1212PM_GET_WAVE_POSITION:
         K1212GetWavePosition(&(pReq->inBuf), &(pReq->outBuf));
         return;

      case K1212PM_GET_WAVEIN_STATE:
         K1212GetWaveInputDeviceState(&(pReq->inBuf), &(pReq->outBuf));
         return;

      case K1212PM_GET_WAVEOUT_STATE:
         K1212GetWaveOutputDeviceState(&(pReq->inBuf), &(pReq->outBuf));
         return;

      case K1212PM_POSTMSG_TO_FILE: // v1.05
         #ifdef DEBUG_MSG
            K1212PostMessageToFile(&(pReq->inBuf), &(pReq->outBuf));
         #endif // DEBUG_MSG
         return;

      case K1212PM_BREAKLOOP:       // v1.08
         K1212BreakLoop(&(pReq->inBuf), &(pReq->outBuf));
         return;

      default:
         // ----------------------------------------------------------------------
         // This function request either is not supported, or can be handled by
         // mapping to the corresponding DIOC function.
         // ----------------------------------------------------------------------
         break;

   }  //    end of switch functionCode

   IoCtl.dioc_IOCtlCode = pReq->functionCode;
   IoCtl.dioc_InBuf     = &(diocIn);
   IoCtl.dioc_cbInBuf   = sizeof(k1212DIOCIn);
   IoCtl.dioc_OutBuf    = &(diocOut);
   IoCtl.dioc_cbOutBuf  = sizeof(k1212DIOCOut);
   IoCtl.dioc_bytesret  = &(byteReturned);

   if (MapPMtoDIOC(&IoCtl, pReq)) {
      OnW32Deviceiocontrol(&IoCtl);
      MapDIOCtoPM(&IoCtl, pReq);
   }
   else {
      pReq->outBuf.returnValue = K1212_CMDRET_PMFailure;
   }
}


// defined to avoid macro problems below
void k1212EndCritical(void)
{
   End_Critical_Section();
   LeaveMustComplete();
}



//*************************************************************************
//*
//* OnW32Deviceiocontrol - Handler of Device I/O control messages
//*
//* The macros preceding the function definition are used to check the
//* input and output buffers.
//*
//*   !!!NOTE!!! If any additions or deletions of message types are made
//*               here, be sure to update the protected mode interface
//*               functions also.
//*
//*************************************************************************
#define CHECK_OUTPUT_BUFSIZE(sizeReq)                                       \
               if ( (!p->dioc_OutBuf) || (p->dioc_cbOutBuf < (sizeReq)) ) { \
                  k1212EndCritical();                                       \
                  return ERROR_INVALID_PARAMETER;                           \
               }

#define CHECK_INPUT_BUFSIZE(sizeReq)                                        \
               if ( (!p->dioc_InBuf) || (p->dioc_cbInBuf < (sizeReq)) ) {   \
                  k1212EndCritical();                                       \
                  return ERROR_INVALID_PARAMETER;                           \
               }

#define SET_BYTES_RETURNED(numBytes)                           \
               if (p->dioc_bytesret) {                         \
                  *(PDWORD)(p->dioc_bytesret) = (numBytes);    \
               }

#define VERIFY_STATE_EQ(state)                                     \
               if (getCardState(cardIndex) != state) {             \
                  outBuf->returnValue = K1212_CMDRET_FailBadState; \
                  k1212EndCritical();                              \
                  return 0;                                        \
               }

#define VERIFY_STATE_GE(state)                                     \
               if (getCardState(cardIndex) < state) {              \
                  outBuf->returnValue = K1212_CMDRET_FailBadState; \
                  k1212EndCritical();                              \
                  return 0;                                        \
               }

   // ----------------------------------------------------------
   // only the low word is checked to avoid hassles with both
   // 16 and 32 bit callers.
   // -----------------------------------------------------------
#define VERIFY_HANDLE                                                      \
               if ( (inBuf->cardHandle & 0x0000ffff) !=                    \
                    ((getDevNode(cardIndex) + getOpenCount(cardIndex))     \
                     & 0x0000ffff)                                         \
                  ) {                                                      \
                  outBuf->returnValue = K1212_CMDRET_BadHandle;            \
                  k1212EndCritical();                                      \
                  return 0;                                                \
               }


DWORD OnW32Deviceiocontrol(PIOCTLPARAMS p)
{
   DWORD          cardIndex;
   k1212DIOCIn*   inBuf;                                       // working pointers to the DIOC
   k1212DIOCOut*  outBuf;                                      //    input and output buffers

   #ifdef DEBUG
      PCI_CONFIG_HEADER_0 pciConfigSpace;
   #endif // DEBUG

   START_CRITICAL;

   inBuf  = (k1212DIOCIn*)(p->dioc_InBuf);
   outBuf = (k1212DIOCOut*)(p->dioc_OutBuf);

   SET_BYTES_RETURNED(sizeof(k1212DIOCOut))                    // broad stroke for all cases

   cardIndex = inBuf->cardIndex;
   if (cardIndex >= k1212MaxCards) {                           // make sure the index is within
      outBuf->returnValue = K1212_CMDRET_BadIndex;             //    the allowable range.
      END_CRITICAL;
      return 0;
   }

   switch (p->dioc_IOCtlCode)                                  // process based on request value
   {
      case DIOC_GETVERSION:
         END_CRITICAL;
         return 0;
         break;

      case K1212_GET_VERSION:
         outBuf->version     = K1212_VERSION;
         outBuf->returnValue = K1212_CMDRET_Success;
         break;

      case K1212_REGISTER_API:
         apiCount++;
         outBuf->apiInfo.apiHandle         = apiCount;
         outBuf->apiInfo.sharedBufferCount = getBufferCount();
         outBuf->returnValue = K1212_CMDRET_Success;
         #ifdef DEBUG
            Debug_Printf_Service("Korg 1212 driver: API channel %d opened.\n", apiCount);
         #endif // DEBUG
         break;

      case K1212_GET_CARD_INFO:
         outBuf->cardInfo.state           = getCardState(cardIndex);
         outBuf->cardInfo.irqNum          = getIrqNum(cardIndex);
         outBuf->cardInfo.busNum          = getBusNum(cardIndex);
         outBuf->cardInfo.deviceNum       = getDeviceNum(cardIndex);
         outBuf->cardInfo.devNode         = getDevNode(cardIndex);
         outBuf->cardInfo.playBuf0Address = getPlayBuf0Address(cardIndex);
         outBuf->cardInfo.recBuf0Address  = getRecBuf0Address(cardIndex);
         outBuf->cardInfo.volBufAddress   = getVolumeTable(cardIndex);
         outBuf->cardInfo.rteBufAddress   = getRoutingTable(cardIndex);
         outBuf->returnValue = K1212_CMDRET_Success;
         break;

      case K1212_GET_CARD_DEBUG_INFO:
         outBuf->cardDebugInfo.state           = getCardState(cardIndex);
         outBuf->cardDebugInfo.irqCount        = getIrqCount(cardIndex);
         outBuf->cardDebugInfo.cmdRetryCount   = getCmdRetryCount(cardIndex);
         outBuf->cardDebugInfo.openCount       = getOpenCount(cardIndex);
         outBuf->cardDebugInfo.clkSrcRate      = getClkSrcRate(cardIndex);
         outBuf->cardDebugInfo.playBuf0Address = getPlayBuf0Address(cardIndex);
         outBuf->cardDebugInfo.devNode         = getDevNode(cardIndex);
         outBuf->returnValue = K1212_CMDRET_Success;
         break;

      case K1212_WRITE_CARD:
         VERIFY_HANDLE
         outBuf->returnValue = Send1212Command(cardIndex,
                                               inBuf->regVals.doorbellVal,
                                               inBuf->regVals.mailbox0Val,
                                               inBuf->regVals.mailbox1Val,
                                               inBuf->regVals.mailbox2Val,
                                               inBuf->regVals.mailbox3Val
                               );
         #ifdef DEBUG
            Debug_Printf_Service("Korg 1212 Card %d: write to card.\n",
                                 cardIndex
            );
         #endif // DEBUG
         break;

      case K1212_READ_CARD:
         VERIFY_HANDLE
         outBuf->regVals.doorbellVal = readInDoorbell(cardIndex);
         outBuf->regVals.mailbox0Val = readMailbox0(cardIndex);
         outBuf->regVals.mailbox1Val = readMailbox1(cardIndex);
         outBuf->regVals.mailbox2Val = readMailbox2(cardIndex);
         outBuf->regVals.mailbox3Val = readMailbox3(cardIndex);
         outBuf->returnValue = K1212_CMDRET_Success;
         #ifdef DEBUG
            Debug_Printf_Service("Korg 1212 Card %d: read from card.\n",
                                 cardIndex
            );
         #endif // DEBUG
         break;

      case K1212_PRINT_DEBUG_INFO:
         VERIFY_HANDLE
         #ifdef DEBUG
            GetPCIConfigData(getDevNode(cardIndex),   // our device
                             0,                       // start at offset 0
                             (DWORD*)&pciConfigSpace, // pointer to receiving buffer
                             pciConfigSpaceSize       // get the whole thing
            );

            PrintPCIConfigData(pciConfigSpace);

            SetPCIConfigData(getDevNode(cardIndex),   // our device
                             0,                       // start at offset 0
                             (DWORD*)&pciConfigSpace, // pointer to receiving buffer
                             pciConfigSpaceSize       // set the whole thing
            );
            Debug_Printf_Service("Korg 1212 Card %d: number of interrupts = %d\n",
                                 cardIndex,
                                 getIrqCount(cardIndex)
            );
            Debug_Printf_Service("Korg 1212 Card %d: number of command retries = %d\n",
                                 cardIndex,
                                 getCmdRetryCount(cardIndex)
            );
         #endif // DEBUG
         outBuf->returnValue = K1212_CMDRET_Success;
         break;

      case K1212_OPEN_CARD:
         VERIFY_STATE_EQ(K1212_STATE_READY)
         K1212OpenCard(cardIndex,
                       inBuf->apiHandle
         );
         outBuf->cardHandle  = (getDevNode(cardIndex) + getOpenCount(cardIndex));
         outBuf->returnValue = K1212_CMDRET_Success;
         #ifdef DEBUG
            Debug_Printf_Service("Korg 1212 Card %d: API channel %d has opened the card.\n",
                                 cardIndex,
                                 inBuf->apiHandle
            );
         #endif // DEBUG
         break;

      case K1212_SET_FILL_ROUTINE:
         VERIFY_STATE_EQ(K1212_STATE_OPEN)
         VERIFY_HANDLE
         setFillParam(cardIndex,
                      inBuf->fillFuncData.fillParam
         );
         setFillFunc(cardIndex,
                     inBuf->fillFuncData.fillFunc
         );
         setFillThreadHdl(cardIndex,
                          Get_Cur_Thread_Handle()
         );
         outBuf->returnValue = K1212_CMDRET_Success;
         #ifdef DEBUG
            Debug_Printf_Service("Korg 1212 Card %d: Fill routine = 0x%x; param = 0x%x; thread = 0x%x.\n",
                                 cardIndex,
                                 getFillFunc(cardIndex),
                                 getFillParam(cardIndex),
                                 getFillThreadHdl(cardIndex)
            );
         #endif // DEBUG
         break;

      case K1212_CLEAR_FILL_ROUTINE:
         VERIFY_STATE_EQ(K1212_STATE_OPEN)
         VERIFY_HANDLE
         ClearFillRoutine(cardIndex);
         outBuf->returnValue = K1212_CMDRET_Success;
         break;

      case K1212_CLOSE_CARD:
         VERIFY_STATE_GE(K1212_STATE_OPEN)
         VERIFY_HANDLE
         K1212CloseCard(cardIndex);
         outBuf->returnValue = K1212_CMDRET_Success;
         #ifdef DEBUG
            Debug_Printf_Service("Korg 1212 Card %d: The card has been closed. Open count = %d\n",
                                 cardIndex,
                                 getOpenCount(cardIndex)
            );
         #endif // DEBUG
         break;

      case K1212_CLEAR_API:
         ClearAPI(cardIndex);
         outBuf->returnValue = K1212_CMDRET_Success;
         break;

      case K1212_CLOSE_API:
         CleanupAfterAPIClose(inBuf->apiHandle);
         outBuf->returnValue = K1212_CMDRET_Success;
         #ifdef DEBUG
            Debug_Printf_Service("Korg 1212 Driver: API channel %d has been closed.\n",
                                 inBuf->apiHandle
            );
         #endif // DEBUG
         break;

      case K1212_GET_PACKET_SIZE:
         outBuf->size        = kPlayBufferFrames;
         outBuf->returnValue = K1212_CMDRET_Success;
         break;

      case K1212_GET_NUMBER_OF_BUFFERS:
         outBuf->bufferCount = getBufferCount();
         outBuf->returnValue = K1212_CMDRET_Success;
         break;

      case K1212_GET_SAMPLE_SIZE:
         outBuf->size        = sizeof(K1212Sample);
         outBuf->returnValue = K1212_CMDRET_Success;
         break;

      case K1212_SET_CLOCK_SOURCE_RATE:
         VERIFY_STATE_EQ(K1212_STATE_OPEN)
         VERIFY_HANDLE
         if (setClkSrcRate(cardIndex,
                           inBuf->clkSrcIdx
             )) {
            outBuf->returnValue = Send1212Command(cardIndex,
                                                  K1212_DB_SetClockSourceRate,
                                                  ClockSourceSelector[getClkSrcRate(cardIndex)],
                                                  DUMMY_PARAMETER,
                                                  DUMMY_PARAMETER,
                                                  DUMMY_PARAMETER
                                  );
            #ifdef DEBUG
               Debug_Printf_Service("Korg 1212 Card %d: Clock set with value %d.\n",
                                    cardIndex,
                                    getClkSrcRate(cardIndex)
               );
            #endif // DEBUG
         }
         else {
            outBuf->returnValue = K1212_CMDRET_BadParams;
            #ifdef DEBUG
               Debug_Printf_Service("Korg 1212 Card %d: Clock set failed with value %d.\n",
                                    cardIndex,
                                    inBuf->clkSrcIdx
               );
            #endif // DEBUG
         }
         break;

      case K1212_GET_CLOCK_SOURCE_RATE:
         outBuf->curClkSrc   = getClkSrcRate(cardIndex);
         outBuf->returnValue = K1212_CMDRET_Success;
         break;

      case K1212_GET_PLAY_CHANNEL_COUNT:
         outBuf->channelCount = kAudioChannels;
         outBuf->returnValue  = K1212_CMDRET_Success;
         break;

      case K1212_GET_RECORD_CHANNEL_COUNT:
         outBuf->channelCount = kAudioChannels;
         outBuf->returnValue  = K1212_CMDRET_Success;
         break;

      case K1212_GET_CHANNEL_VOLUME:
         VERIFY_STATE_GE(K1212_STATE_READY)
         outBuf->chanVol.channel = inBuf->channel;
         outBuf->chanVol.volume  = getChannelVolume(cardIndex,
                                                    inBuf->channel
                                   );
         outBuf->returnValue = K1212_CMDRET_Success;
         break;

      case K1212_SET_CHANNEL_VOLUME:
         VERIFY_STATE_GE(K1212_STATE_READY)
         VERIFY_HANDLE
         setChannelVolume(cardIndex,
                          inBuf->chanVol.channel,
                          inBuf->chanVol.volume
         );
         outBuf->returnValue = K1212_CMDRET_Success;
         break;

      case K1212_GET_CHANNEL_ROUTE:
         VERIFY_STATE_GE(K1212_STATE_READY)
         outBuf->chanRte.channel = inBuf->channel;
         outBuf->chanRte.route   = getChannelRoute(cardIndex,
                                                   inBuf->channel
                                   );
         outBuf->returnValue = K1212_CMDRET_Success;
         break;

      case K1212_SET_CHANNEL_ROUTE:
         VERIFY_STATE_GE(K1212_STATE_READY)
         VERIFY_HANDLE
         setChannelRoute(cardIndex,
                         inBuf->chanRte.channel,
                         inBuf->chanRte.route
         );
         outBuf->returnValue = K1212_CMDRET_Success;
         break;

      case K1212_SET_INPUT_SENSITIVITY:
         VERIFY_STATE_EQ(K1212_STATE_READY)
         VERIFY_HANDLE
         setLeftADCInSens(cardIndex,
                          inBuf->lrVals.leftValue
         );
         setRightADCInSens(cardIndex,
                           inBuf->lrVals.rightValue
         );
         WriteADCSensitivity(cardIndex);
         outBuf->returnValue = K1212_CMDRET_Success;
         break;

      case K1212_GET_INPUT_SENSITIVITY:
         VERIFY_STATE_GE(K1212_STATE_READY)
         outBuf->lrVals.leftValue  = getLeftADCInSens(cardIndex);
         outBuf->lrVals.rightValue = getRightADCInSens(cardIndex);
         outBuf->returnValue       = K1212_CMDRET_Success;
         break;

      case K1212_GET_PLAY_BUFFER_ADDRESSES:
         VERIFY_STATE_GE(K1212_STATE_READY)
         outBuf->bufAddress  = (char*)(getSharedBufferPtr(cardIndex)->playDataBufs);
         outBuf->returnValue = K1212_CMDRET_Success;
         break;

      case K1212_GET_RECORD_BUFFER_ADDRESSES:
         VERIFY_STATE_GE(K1212_STATE_READY)
         outBuf->bufAddress  = (char*)(getSharedBufferPtr(cardIndex)->recordDataBufs);
         outBuf->returnValue = K1212_CMDRET_Success;
         break;

      case K1212_GET_CHANNEL_VOLUME_BUFFER_ADDRESS:
         VERIFY_STATE_GE(K1212_STATE_READY)
         outBuf->bufAddress   = (char*)(getSharedBufferPtr(cardIndex)->volumeData);
         outBuf->returnValue  = K1212_CMDRET_Success;
         break;

      case K1212_GET_CHANNEL_ROUTING_BUFFER_ADDRESS:
         VERIFY_STATE_GE(K1212_STATE_READY)
         outBuf->bufAddress   = (char*)(getSharedBufferPtr(cardIndex)->routeData);
         outBuf->returnValue  = K1212_CMDRET_Success;
         break;

      case K1212_GET_ADAT_TIMECODE_ADDRESS:
         VERIFY_STATE_GE(K1212_STATE_READY)
         outBuf->bufAddress   = (char*)getAdatTimeCodePtr(cardIndex);
         outBuf->returnValue  = K1212_CMDRET_Success;
         break;

      case K1212_GET_ADAT_TIMECODE:
         VERIFY_STATE_EQ(K1212_STATE_OPEN)
         VERIFY_HANDLE
         switch (getClkSrcRate(cardIndex)) {
            case K1212_CLKIDX_LocalAt44_1K:
            case K1212_CLKIDX_LocalAt48K:
               outBuf->returnValue = K1212_CMDRET_FailBadState;
               break;

            default:
               if (Send1212Command(cardIndex,
                                   K1212_DB_RequestAdatTimecode,
                                   DUMMY_PARAMETER,
                                   DUMMY_PARAMETER,
                                   DUMMY_PARAMETER,
                                   DUMMY_PARAMETER
                   ) != K1212_CMDRET_Success) {
                  outBuf->returnValue = K1212_CMDRET_NoAckFromCard;
                  break;
               }
               outBuf->dwordValue = readMailbox2(cardIndex);
               outBuf->returnValue  = K1212_CMDRET_Success;
               break;
         }
         break;

      case K1212_SETUP_FOR_PLAY:
         VERIFY_STATE_EQ(K1212_STATE_OPEN)
         VERIFY_HANDLE
         setCurFillBufNum(cardIndex, (kNumBuffers - 1));    // prepare to start playback
         K1212SetupForPlay(cardIndex);
         outBuf->returnValue  = K1212_CMDRET_Success;
         break;

      case K1212_TRIGGER_PLAY:
         VERIFY_STATE_EQ(K1212_STATE_SETUP)
         VERIFY_HANDLE
         K1212TriggerPlay(cardIndex);
         outBuf->returnValue  = K1212_CMDRET_Success;
         break;

      case K1212_TRIGGER_PLAY_FROM_ADAT:
         VERIFY_STATE_EQ(K1212_STATE_SETUP)
         VERIFY_HANDLE
         K1212TriggerPlayFromAdat(cardIndex,
                                  inBuf->adatTrigData.timecodeVal,
                                  inBuf->adatTrigData.stopModeSel
         );
         outBuf->returnValue  = K1212_CMDRET_Success;
         break;

      case K1212_STOP_PLAY:
         VERIFY_STATE_GE(K1212_STATE_SETUP)
         VERIFY_HANDLE
         K1212StopPlay(cardIndex);
         outBuf->returnValue  = K1212_CMDRET_Success;
         #ifdef DEBUG
            Debug_Printf_Service("Korg 1212 Card %d: Stopping play\n", cardIndex);
         #endif // DEBUG
         break;

      case K1212_SET_MONITOR_MODE:
         VERIFY_HANDLE
         if (K1212SetMonitorMode(cardIndex, inBuf->monMode)) {
            outBuf->returnValue  = K1212_CMDRET_Success;
         }
         else {
            outBuf->returnValue  = K1212_CMDRET_FailBadState;
         }
         break;

      case K1212_GET_RECORD_PLAY_LATENCY:
         outBuf->recPlyLatency = k1212RecPlayLatency;
         outBuf->returnValue   = K1212_CMDRET_Success;
         break;

      case K1212_REBOOT_CARD:
         VERIFY_STATE_GE(K1212_STATE_UNINITIALIZED)
         #ifdef DEBUG
            Debug_Printf_Service("Korg 1212 Card %d: Rebooting the card...\n", cardIndex);
         #endif // DEBUG
         RebootCard(cardIndex);
         outBuf->returnValue   = K1212_CMDRET_Success;
         break;

      case K1212_SAVE_CARD_SETTINGS:
         VERIFY_HANDLE
         #ifdef DEBUG
            Debug_Printf_Service("Korg 1212 Card %d: Request to save settings received.\n", cardIndex);
         #endif // DEBUG
         SaveCardSettings(cardIndex);
         outBuf->returnValue   = K1212_CMDRET_Success;
         break;

      case K1212_RETRIEVE_CARD_SETTINGS:
         VERIFY_HANDLE
         #ifdef DEBUG
            Debug_Printf_Service("Korg 1212 Card %d: Request to retrieve settings received.\n", cardIndex);
         #endif // DEBUG
         RetrieveCardSettings(cardIndex);
         outBuf->returnValue   = K1212_CMDRET_Success;
         break;


      // ---------------------------------------------------------------------------
      // wave controls we allow to be set by anybody at anytime.  This is okay,
      // since the use of the wave devices is mutually exclusive with the native API
      // ---------------------------------------------------------------------------
      case K1212_DISABLE_AUTO_WAVE_SYNC:
         outBuf->returnValue = K1212SetAutoWaveSync(cardIndex,
                                                    FALSE
                               );
         break;

      case K1212_ENABLE_AUTO_WAVE_SYNC:
         outBuf->returnValue = K1212SetAutoWaveSync(cardIndex,
                                                    TRUE
                               );
         break;

      case K1212_GET_AUTO_WAVE_SYNC:
         outBuf->boolValue   = getAutoWaveSyncMode(cardIndex);
         outBuf->returnValue = K1212_CMDRET_Success;
         break;

      case K1212_DISABLE_WAVE_VOL_CTRL:
         outBuf->returnValue = disableWaveVolControl(cardIndex,
                                                     inBuf->deviceId
                               );
         break;

      case K1212_ENABLE_WAVE_VOL_CTRL:
         outBuf->returnValue = enableWaveVolControl(cardIndex,
                                                    inBuf->deviceId
                               );
         break;

      case K1212_GET_WAVE_VOL_CTRL_FLAG:
         outBuf->returnValue = getWaveVolEnaFlag(cardIndex,
                                                 inBuf->deviceId,
                                                 &(outBuf->boolValue)
                               );
         break;

      case K1212_SET_WAVEDEV_SYNC_START_FLAG:
         outBuf->returnValue = setWaveDevSyncFlag(cardIndex,
                                                  (K1212WaveDeviceType)inBuf->waveSync.deviceType,
                                                  inBuf->waveSync.deviceId,
                                                  inBuf->waveSync.flagSetting
                               );
         break;

      case K1212_GET_WAVEDEV_SYNC_START_FLAG:
         outBuf->returnValue = getWaveDevSyncFlag(cardIndex,
                                                  (K1212WaveDeviceType)inBuf->waveSync.deviceType,
                                                  inBuf->waveSync.deviceId,
                                                  &(outBuf->dwordValue)
                               );
         break;

      case K1212_GET_WAVE_VOLUME:
         outBuf->returnValue = getWaveVolume(cardIndex,
                                             inBuf->deviceId,
                                             &(outBuf->lrVals)
                               );
         break;

      case K1212_SET_WAVE_VOLUME:
         outBuf->returnValue = setWaveVolume(cardIndex,
                                             inBuf->waveVol
                               );
         break;

      case K1212_SAVE_CARD_WAVE_SETTINGS:
         SaveWaveSettings(cardIndex);
         outBuf->returnValue = K1212_CMDRET_Success;
         break;

      case K1212_RETRIEVE_CARD_WAVE_SETTINGS:
         RetrieveWaveSettings(cardIndex);
         outBuf->returnValue = K1212_CMDRET_Success;
         break;

      case K1212_REGISTER_FOR_WAVE_VOLUPDATES:
         wavVolUpdateDataPtr       = HeapAllocate(sizeof(k1212WavVolUpdate),
                                                  HEAPLOCKEDIFDP
                                     );
         k1212WaveVolUpdateFuncPtr = inBuf->fillFuncData.fillFunc;
         wavVolUpdateThisPtr       = (PVOID)(inBuf->fillFuncData.fillParam);
         k1212WavThreadHandle      = Get_Cur_Thread_Handle();
         outBuf->returnValue = K1212_CMDRET_Success;
         break;

      case K1212_DEREGISTER_FOR_WAVE_VOLUPDATES:
         k1212WaveVolUpdateFuncPtr = 0;
         k1212WavThreadHandle      = 0;
         wavVolUpdateThisPtr       = 0;
         HeapFree(wavVolUpdateDataPtr, 0);
         outBuf->returnValue = K1212_CMDRET_Success;
         break;

      default:
         #ifdef DEBUG
            Debug_Printf_Service("Korg 1212 Card %d: Received Dev I/O Ctrl message 0x%x\n",
                                 cardIndex,
                                 p->dioc_IOCtlCode
            );
         #endif // DEBUG
         break;
   }
   END_CRITICAL;
   return 0;
}


// --------------------------------------------------------------------------
// K1212OpenCard
//
//    This function processes the open card command.
// --------------------------------------------------------------------------
BOOL  K1212OpenCard(DWORD     cardIndex,
                    APIHandle apiHandle)

{
   setCardState(cardIndex, K1212_STATE_OPEN);
   setApiHandle(cardIndex, apiHandle);
   incOpenCount(cardIndex);
   return TRUE;
}


// --------------------------------------------------------------------------
// CleanupAfterAPIClose
//
//    This function closes any cards that the closed API may have left open.
// --------------------------------------------------------------------------
void  CleanupAfterAPIClose (APIHandle apiHandle)
{
   DWORD cardIndex;

   for (cardIndex = 0; cardIndex < k1212MaxCards; cardIndex++) {
      if (getApiHandle(cardIndex) == apiHandle) {
         K1212CloseCard(cardIndex);
      }
   }
}


// --------------------------------------------------------------------------
// ClearFillRoutine
//
//    This function clears the fill routine for the specified card.
// --------------------------------------------------------------------------
void  ClearFillRoutine(DWORD cardIndex)
{
   setFillParam(cardIndex,
                0
   );
   setFillFunc(cardIndex,
               NULL
   );
   setFillThreadHdl(cardIndex,
                    0
   );
   #ifdef DEBUG
      Debug_Printf_Service("Korg 1212 Card %d: Fill routine cleared.\n",
                           cardIndex
      );
   #endif // DEBUG
}


// --------------------------------------------------------------------------
// ClearAPI
//
//    This function forces anyone off the card who may be on it.
// --------------------------------------------------------------------------
void  ClearAPI(DWORD cardIndex)
{
   WORD deviceId;

   if (getApiMode(cardIndex) != K1212_NATIVE_API_MODE) {
      for (deviceId = 0; deviceId < k1212NumWaveDevices; deviceId++) {
         ResetWaveInDevice ((WORD)cardIndex, deviceId);
         ResetWaveOutDevice((WORD)cardIndex, deviceId);
      }
   }
   K1212CloseCard(cardIndex);
}


// --------------------------------------------------------------------------
// K1212CloseCard
//
//    This function processes the close card command.  If the card is
//    currently in playback or monitor modes, it is stopped.  The fill
//    routine event is closed, and the card is placed into the ready state.
// --------------------------------------------------------------------------
BOOL  K1212CloseCard(DWORD cardIndex)
{
   if (getCardState(cardIndex) == K1212_STATE_SETUP) {
      if (Send1212Command(cardIndex,
                          K1212_DB_SelectPlayMode,
                          K1212_MODE_StopPlay,
                          DUMMY_PARAMETER,
                          DUMMY_PARAMETER,
                          DUMMY_PARAMETER
          ) != K1212_CMDRET_Success) {
         return FALSE;
      }
   }
   else if (getCardState(cardIndex) > K1212_STATE_SETUP) {
      setCardCommand(cardIndex, 0xffffffff);
      WaitForCardStopAck(cardIndex);
   }

   ClearFillRoutine(cardIndex);
   setApiHandle(cardIndex, 0);
   setApiMode(cardIndex, K1212_NATIVE_API_MODE);

   if (getCardState(cardIndex) > K1212_STATE_READY) {
      setCardState(cardIndex, K1212_STATE_READY);
   }

   return TRUE;
}


// --------------------------------------------------------------------------
// K1212TriggerPlay
//
//    This function processes the trigger play command.
// --------------------------------------------------------------------------
BOOL  K1212TriggerPlay(DWORD     cardIndex)
{
   #ifdef DEBUG
      Debug_Printf_Service("Korg 1212 Card %d: Triggering play\n", cardIndex);
   #endif // DEBUG
   setCardState(cardIndex, K1212_STATE_PLAYING);
   if (Send1212Command(cardIndex,
                       K1212_DB_TriggerPlay,
                       DUMMY_PARAMETER,
                       DUMMY_PARAMETER,
                       DUMMY_PARAMETER,
                       DUMMY_PARAMETER
       ) != K1212_CMDRET_Success) {
      return FALSE;
   }
   return TRUE;
}


// --------------------------------------------------------------------------
// K1212TriggerPlayFromAdat
//
//    This function processes the trigger play command.
// --------------------------------------------------------------------------
BOOL  K1212TriggerPlayFromAdat
(
   DWORD             cardIndex,
   DWORD             timecodeVal,
   StopModeSelector  stopModeSel
)
{
   DWORD stopSelVal;

   #ifdef DEBUG
      Debug_Printf_Service("Korg 1212 Card %d: Triggering play from ADAT\n", cardIndex);
      Debug_Printf_Service("                   Timecode value = %d, fast stop mode = %d\n",
                           timecodeVal,
                           stopModeSel
      );
   #endif // DEBUG

   // ----------------------------------------------------------------------
   // for K1212_STOP_NormalStop send 0x0000, for K1212_STOP_FastAdatStop,
   // send 0x8000.
   // ----------------------------------------------------------------------
   if (stopModeSel == K1212_STOP_FastAdatStop) {
      stopSelVal = 0x00008000;
   }
   else {
      stopSelVal = 0x00000000;
   }

   setCardState(cardIndex, K1212_STATE_PLAYING);
   if (Send1212Command(cardIndex,
                       K1212_DB_TriggerFromAdat,
                       timecodeVal,
                       stopSelVal,
                       DUMMY_PARAMETER,
                       DUMMY_PARAMETER
       ) != K1212_CMDRET_Success) {
      return FALSE;
   }
   return TRUE;
}


// --------------------------------------------------------------------------
// K1212SetupForPlay
//
//    This function processes the setup for play command.
// --------------------------------------------------------------------------
BOOL  K1212SetupForPlay(DWORD     cardIndex)
{
   #ifdef DEBUG
      Debug_Printf_Service("Korg 1212 Card %d: Setting up for play\n", cardIndex);
   #endif // DEBUG

   // ---------------------------------------------------------------
   // initialize the current fill buffer to the end, so at the
   // first fill interrupt from the card, it gets incremented to
   // zero, which will be the first buffer needing a refill.
   // Then update the card state, and send the command to the card.
   // ---------------------------------------------------------------
   setCardState(cardIndex, K1212_STATE_SETUP);
   if (Send1212Command(cardIndex,
                       K1212_DB_SelectPlayMode,
                       K1212_MODE_SetupPlay,
                       DUMMY_PARAMETER,
                       DUMMY_PARAMETER,
                       DUMMY_PARAMETER
       ) != K1212_CMDRET_Success) {
      return FALSE;
   }
   return TRUE;
}


// --------------------------------------------------------------------------
// K1212StopPlay
//
//    This function processes the stop command.
// --------------------------------------------------------------------------
BOOL  K1212StopPlay(DWORD     cardIndex)
{
   if (getCardState(cardIndex) != K1212_STATE_ERRORSTOP) {
      setCardCommand(cardIndex, 0xffffffff);
      WaitForCardStopAck(cardIndex);
   }
   setCardState(cardIndex, K1212_STATE_OPEN);
   return TRUE;
}



// --------------------------------------------------------------------------
// K1212TurnOnIdleMonitor
//
//    This function is called whenever the card goes into the ready state.
//    The purpose of this function is to place the card in monitor mode while
//    it is not in use, so that ADAT devices on the bus will not be hindered
//    by the 1212 I/O card.
// --------------------------------------------------------------------------
void  K1212TurnOnIdleMonitor(DWORD cardIndex)
{
   WaitRTCTicks(INTERCOMMAND_DELAY);
   setIdleMonitorState(cardIndex, TRUE);
   Send1212Command(cardIndex,
                   K1212_DB_SelectPlayMode,
                   K1212_MODE_MonitorOn,
                   DUMMY_PARAMETER,
                   DUMMY_PARAMETER,
                   DUMMY_PARAMETER
   );
}


// --------------------------------------------------------------------------
// K1212TurnOffIdleMonitor
//
//    This function is called whenever the card goes into the open state.
//    This function takes the card out of idle monitor mode.
// --------------------------------------------------------------------------
void  K1212TurnOffIdleMonitor(DWORD cardIndex)
{
   if (getIdleMonitorState(cardIndex) == TRUE) {
      setCardCommand(cardIndex, 0xffffffff);
      WaitForCardStopAck(cardIndex);
      setIdleMonitorState(cardIndex, FALSE);
   }
}


// --------------------------------------------------------------------------
// K1212SetMonitorMode
//
//    This function checks the card's state against the requested monitor
//    mode selection and tells the card to enter the new monitor mode.
// --------------------------------------------------------------------------
BOOL  K1212SetMonitorMode(DWORD               cardIndex,
                          MonitorModeSelector mode)
{
   switch (mode) {

      case K1212_MONMODE_Off:
         if (getCardState(cardIndex) != K1212_STATE_MONITOR) {
            return FALSE;
         }
         else {
            setCardCommand(cardIndex, 0xffffffff);    // tells the card to turn off
            WaitForCardStopAck(cardIndex);
            setCardState(cardIndex, K1212_STATE_OPEN);
            #ifdef DEBUG
               Debug_Printf_Service("Korg 1212 Card %d: Turning off monitor mode\n", cardIndex);
            #endif // DEBUG
         }
         break;

      case K1212_MONMODE_On:
         if (getCardState(cardIndex) != K1212_STATE_OPEN) {
            return FALSE;
         }
         else {
            setCardState(cardIndex, K1212_STATE_MONITOR);
            if (Send1212Command(cardIndex,
                                K1212_DB_SelectPlayMode,
                                K1212_MODE_MonitorOn,
                                DUMMY_PARAMETER,
                                DUMMY_PARAMETER,
                                DUMMY_PARAMETER
                ) != K1212_CMDRET_Success) {
               return FALSE;
            }
            #ifdef DEBUG
               Debug_Printf_Service("Korg 1212 Card %d: Turning on monitor mode\n", cardIndex);
            #endif // DEBUG
         }
         break;

      default:
         return FALSE;
   }

   return TRUE;
}



// ---------------------------------------------------------------------------------------
// The following functions map between the 32 bit DIOC structure and the 16 bit PM
// interface structure.  All pointers must be converted.  The DIOC structure contains
// unions whose member selection is based on the function request code.  Therefore,
// these functions must preprocess the function codes.
//
// Note - these functions specifically map from the inBuf of the pReq to the
//        dioc_InBuf of the IoCtl, and from the dioc_OutBuf of the IoCtl to the
//        outBuf of the pReq.
// ---------------------------------------------------------------------------------------
BOOL MapPMtoDIOC(IOCTLPARAMS* IoCtl, k1212PmReq* pReq)
{
   // ----------------------------------------------------------------
   // set up pointers to the respective input buffers
   // ----------------------------------------------------------------
   k1212PMIn*   pmInBufPtr;
   k1212DIOCIn* diocInBufPtr;
   pmInBufPtr   = &(pReq->inBuf);
   diocInBufPtr = IoCtl->dioc_InBuf;

   // ----------------------------------------------------------------
   // copy the api handle, card index, and card handle for all cases
   // ----------------------------------------------------------------
   diocInBufPtr->apiHandle  = (APIHandle) (pmInBufPtr->apiHandle);
   diocInBufPtr->cardIndex  = (DWORD)     (pmInBufPtr->cardIndex);

   // ----------------------------------------------------------------
   // Now do function specific mapping
   // ----------------------------------------------------------------
   switch (IoCtl->dioc_IOCtlCode) {

         // ------------------------------------------------------
         // the following requests require mapping
         // ------------------------------------------------------
      case K1212PM_GET_VERSION:
         IoCtl->dioc_IOCtlCode = K1212_GET_VERSION;
         break;

      case K1212PM_REGISTER_API:
         IoCtl->dioc_IOCtlCode = K1212_REGISTER_API;
         break;

      case K1212PM_GET_CARD_INFO:
         IoCtl->dioc_IOCtlCode = K1212_GET_CARD_INFO;
         break;

      case K1212PM_CLOSE_API:
         IoCtl->dioc_IOCtlCode = K1212_CLOSE_API;
         break;

      case K1212PM_GET_CLOCK_SOURCE_RATE:
         IoCtl->dioc_IOCtlCode = K1212_GET_CLOCK_SOURCE_RATE;
         break;

      case K1212PM_GET_INPUT_SENSITIVITY:
         IoCtl->dioc_IOCtlCode = K1212_GET_INPUT_SENSITIVITY;
         break;

      case K1212PM_GET_RECORD_PLAY_LATENCY:
         IoCtl->dioc_IOCtlCode = K1212_GET_RECORD_PLAY_LATENCY;
         break;

         // -------------------------------------------------------------
         // an unsupported request has been specified
         // -------------------------------------------------------------
      default:
         return FALSE;
   }
   return TRUE;
}

BOOL MapDIOCtoPM(IOCTLPARAMS* IoCtl, k1212PmReq* pReq)
{
   // ----------------------------------------------------------------
   // set up pointers to the respective output buffers
   // ----------------------------------------------------------------
   k1212PMOut*   pmOutBufPtr;
   k1212DIOCOut* diocOutBufPtr;
   pmOutBufPtr   = &(pReq->outBuf);
   diocOutBufPtr = IoCtl->dioc_OutBuf;

   // ----------------------------------------------------------------
   // copy the return value for all cases
   // ----------------------------------------------------------------
   pmOutBufPtr->returnValue = (WORD)diocOutBufPtr->returnValue;

   // ----------------------------------------------------------------
   // Now do function specific mapping.  Note that the dioc_IOCtlCode
   // now contains the k1212FunctionCodes instead of the protected
   // mode codes since it has gone through the input buffer mapping.
   // ----------------------------------------------------------------
   switch (IoCtl->dioc_IOCtlCode) {

         // ------------------------------------------------------
         // the following requests require mapping
         // ------------------------------------------------------
      case K1212_GET_VERSION:
         pmOutBufPtr->version = diocOutBufPtr->version;
         break;

      case K1212_REGISTER_API:
         pmOutBufPtr->apiInfo.apiHandle         = (WORD)diocOutBufPtr->apiInfo.apiHandle;
         pmOutBufPtr->apiInfo.sharedBufferCount =
                                 (WORD)diocOutBufPtr->apiInfo.sharedBufferCount;
         break;

      case K1212_GET_CARD_INFO:
         pmOutBufPtr->cardInfo.busNum      = diocOutBufPtr->cardInfo.busNum;
         pmOutBufPtr->cardInfo.deviceNum   = diocOutBufPtr->cardInfo.deviceNum;
         pmOutBufPtr->cardInfo.irqNum      = diocOutBufPtr->cardInfo.irqNum;
         pmOutBufPtr->cardInfo.state       = diocOutBufPtr->cardInfo.state;
         pmOutBufPtr->cardInfo.devNode     = diocOutBufPtr->cardInfo.devNode;
         break;

      case K1212_GET_CLOCK_SOURCE_RATE:
         pmOutBufPtr->curClkSrc = (WORD)diocOutBufPtr->curClkSrc;
         break;

      case K1212_GET_INPUT_SENSITIVITY:
         pmOutBufPtr->lrVals.leftValue  = diocOutBufPtr->lrVals.leftValue;
         pmOutBufPtr->lrVals.rightValue = diocOutBufPtr->lrVals.rightValue;
         break;

      case K1212_GET_RECORD_PLAY_LATENCY:
         pmOutBufPtr->recPlyLatency = (WORD)diocOutBufPtr->recPlyLatency;
         break;

         // -------------------------------------------------------------
         // don't need to do any more mapping for the following requests
         // -------------------------------------------------------------
      case K1212_CLOSE_API:
         break;
   }
   return TRUE;
}


