//*****************************************************************************
//* Name:
//*      KorgWave.c - Wave device module for Korg Audio Card VxD's
//*
//* Project:
//*      OASYS PCI VxD
//*
//* Author:
//*      Bill Jenkins
//*
//* Description:
//*      This file implements the wave device processing functions.
//*
//*
//* Modification Log:
//*
//*      1.8   08/20/97 Bill
//*      (v 1.0B6) Added auto sync rec/play handling for starting the card.
//*
//*      1.7   07/03/97 Bill
//*      (v 1.0B4) Added code to change the clock rate based on the opening of a wave
//*      device.  Note that this mechanism still allows the user to screw himself up.
//*      Added variables recBufferCount and playBufferCount to track the number of
//*      buffers queued.  These counts are used to determine whether we have sufficient
//*      resources to queue the buffers.
//*
//*      1.6   06/16/97 Bill
//*      (v 1.10) Commented out post message to file components.
//*
//*      1.8   06/13/97 Bill
//*      (v 1.09) Trying VM priority event in place of appy time event for returning
//*      buffers to the wave driver.  Moved marking of wave buf done and no longer
//*      queued to the wave driver.
//*
//*      1.8   06/09/97 Bill
//*      (v 1.08) Modified add and free playback buffer routines to handle looping.  Also,
//*      corrected close device/card handling to avoid problems with idle monitor mode.
//*      Releasing a wave input buffer now posts messages.
//*
//*      1.7   06/03/97 Bill
//*      (v 1.07) Modified wave out close to check whether the device is still playing.  
//*      If it is, then the close fails.
//*
//*      1.6   06/04/97 Bill
//*      (v 1.06) Modified K1212SetWaveOutDeviceVolume and K1212GetWaveOutDeviceVolume
//*      to use the same calls as the DIOC interface uses.  This is for the readdition
//*      of volume control settings from the wave driver.  Setting the wave volume via
//*      the wave driver interface also notifies K1212Wav.exe if it has registered.
//*
//*      1.5   06/03/97 Bill
//*      (v 1.05) Added K1212PostMessageToFile for debugging purposes.  
//*
//*      1.4   05/28/97 Bill
//*      (v 1.04) Added functions for handling wave output buffer prefilling.  
//*
//*      1.3   05/15/97 Bill
//*      (v 1.03) Made corrections to the handling of pause messages.  Record/play
//*      sync mechanism corrected to start the card if a setting changes that now
//*      allows the card to play.
//*
//*      1.2   05/15/97 Bill
//*      (v 1.02) Added initialization to wave output device filter members 
//*               when a wave output device is opened.  
//*
//*      1.1   03/14/97 Bill
//*      Initial version created.  
//*
//*
//* Copyright (c) 1997 Korg, Inc.
//* All rights reserved.
//*
//* This program is protected as an unpublished work under the U.S.
//* copyright laws.  The above copyright notice is not intended to
//* effect a publication of this work.
//*
//* This program is the confidential and proprietary information of
//* Korg and all its subsidiaries.
//*****************************************************************************

#include <vtoolsc.h>
#include <vmm.h>
#include <debug.h>
#include <vmmreg.h>

#ifndef  K1212CFG_H
#include "1212cfg.h"
#endif
#ifndef  K1212CARD_H
#include "1212card.h"
#endif
#ifndef  K1212INTF_H
#include "1212intf.h"
#endif
#ifndef  K1212WAVE_H
#include "1212wave.h"
#endif
#ifndef  K1212WAVEAPI_H
#include "1212WaveApi.h"
#endif
#ifndef  K1212APPYTIME_H
#include "1212AppyTime.h"
#endif



// -------------------------------------------------------------------------------
// externs defined in korg1212.c for making APCs to the K1212Wav.exe application.
// -------------------------------------------------------------------------------
extern PVOID               k1212WaveVolUpdateFuncPtr;    // v1.06
extern THREADHANDLE        k1212WavThreadHandle;
extern k1212WavVolUpdate*  wavVolUpdateDataPtr;
extern PVOID               wavVolUpdateThisPtr;
extern WORD                ClockSourceSelector[];        // v1.0B4 defined in korg1212.c


// ----------------------------------------------------------------------------
// thunk for the VM priority event and forward declaration of the event
// handler.  (v1.09)
// ----------------------------------------------------------------------------
PriorityVMEvent_THUNK  RlsBufferOutThunk;
PriorityVMEvent_THUNK  RlsBufferInThunk;
VOID  __stdcall        RlsWaveOutBufferCallback(VMHANDLE       hVM, 
                                                PVOID          Refdata, 
                                                PCLIENT_STRUCT pcrs, 
                                                DWORD          Flags
                       );
VOID  __stdcall        RlsWaveInBufferCallback (VMHANDLE       hVM, 
                                                PVOID          Refdata, 
                                                PCLIENT_STRUCT pcrs, 
                                                DWORD          Flags
                       );


// ----------------------------------------------------------------------------
// globals
// ----------------------------------------------------------------------------
int    recBuffersQueued  = 0;     // tracks total buffers queued - used to determine
                                  //    whether we have the resources to add another
                                  //    buffer to the queue.
int    playBuffersQueued = 0;     // same as above except for playback - not record.

extern ISRErrorCode;


// ----------------------------------------------------------------------------
// forward declarations...
// ----------------------------------------------------------------------------
void MakeSureTheCardIsInPlayMode
(
   WORD  cardIndex
);


// ----------------------------------------------------------------------------
// ****************************************************************************
// ----------------------------------------------------------------------------
// The following functions are for performing prefilling of the wave output
// device buffers. (v1.04)
// ----------------------------------------------------------------------------
// ****************************************************************************
// ----------------------------------------------------------------------------
DWORD FormatToBytesPerSample
(
   K1212WaveFormat format
)
{
   // -------------------------------------------------------------------------
   // this function determines the total number of bytes that are required for
   // each sample of playback by a wave device using the given format.  
   // -------------------------------------------------------------------------
   switch (format) {
      case WAVFMT_48000_MONO_8:
      case WAVFMT_44100_MONO_8:
      case WAVFMT_22050_MONO_8:
      case WAVFMT_11025_MONO_8:
         return 1;

      case WAVFMT_48000_STEREO_8:
      case WAVFMT_44100_STEREO_8:
      case WAVFMT_22050_STEREO_8:
      case WAVFMT_11025_STEREO_8:
      case WAVFMT_48000_MONO_16:
      case WAVFMT_44100_MONO_16:
      case WAVFMT_22050_MONO_16:
      case WAVFMT_11025_MONO_16:
         return 2;

      case WAVFMT_48000_STEREO_16:
      case WAVFMT_44100_STEREO_16:
      case WAVFMT_22050_STEREO_16:
      case WAVFMT_11025_STEREO_16:
      default:
         return 4;
   }
}


DWORD NumBufsOfPrefillDataInQueue
(
   k1212WaveOutDevice*  waveOutDevPtr
)
{
   DWORD       totalBytesInQueue;
   DWORD       totalSamplesInQueue;
   DWORD       numCompleteBufsOfData;
   LPWAVEHDR   waveHdrPtr;

   totalBytesInQueue = 0;     // initialize our byte count

   // -------------------------------------------------------------------------
   // loop through all of the buffers queued up for this device, tallying
   // the total number of bytes.  Then do integer division and discard the
   // remainder to get the number of complete card buffers that can be 
   // filled.
   // -------------------------------------------------------------------------
   for (waveHdrPtr = waveOutDevPtr->waveBufferQueue;
        waveHdrPtr != 0;
        waveHdrPtr = waveHdrPtr->lpNext) {

      totalBytesInQueue += waveHdrPtr->dwBufferLength;
   }
   totalSamplesInQueue   = (totalBytesInQueue / FormatToBytesPerSample(waveOutDevPtr->format));
   numCompleteBufsOfData = (totalSamplesInQueue / kPlayBufferFrames);
   return numCompleteBufsOfData;
}



void CardStartPrefill
(
   DWORD    cardIndex
)
{
   WORD                deviceId;
   k1212WaveOutDevice* waveOutDevPtr;
   DWORD               numBufsToPrefill;
   BOOL                spdifDevice;

   // ------------------------------------------------------------------
   // v1.11 - initialize new record buffer number member
   // ------------------------------------------------------------------
   setCurRecBufNum(cardIndex, (kNumBuffers - 1));

   // ------------------------------------------------------------------
   // This function prefills as much data as possible prior to starting
   // up the card.  It also initializes the numBufsPrefilled data member
   // for all devices that are not currently playing.
   //
   // Watch for paused devices.  In these cases, we cannot prefill until
   // they are unpaused.
   // ------------------------------------------------------------------
   for (deviceId = 0; deviceId < k1212NumWaveDevices; deviceId++) {

      if (deviceId == K1212_DEVICE_SPDIF) {
         spdifDevice = TRUE;
      }
      else {
         spdifDevice = FALSE;
      }

      waveOutDevPtr = getWaveOutDevPtr(cardIndex,
                                       deviceId
                      );
      if (waveOutDevPtr != 0) {

         // -------------------------------------------------------------
         // if the device is playing, and it has data, do the prefill
         // -------------------------------------------------------------
         if ((waveOutDevPtr->state == K1212_WAVEOUTSTATE_PLAYING) &&
             (waveOutDevPtr->waveBufferQueue != 0)) {

            // ----------------------------------------------------------
            // find out how much data there is to prefill.  This must be
            // limited to the number of buffers the card supports.  Init
            // the buffer index for the wave device, and then prefill as
            // many buffers as we can.
            // ----------------------------------------------------------
            numBufsToPrefill = NumBufsOfPrefillDataInQueue(waveOutDevPtr);
            
            if (numBufsToPrefill > kNumBuffers) {
               numBufsToPrefill = kNumBuffers;
            }
            
            waveOutDevPtr->numBufsPrefilled = 0;

            if (numBufsToPrefill != 0) {
               setCurFillBufNum(cardIndex, 0);
               while (numBufsToPrefill > 0) {
                  waveOutDevPtr->numBufsPrefilled++;
                  (*(BufXlateWaveOutFuncPtr)waveOutDevPtr->translateFunctionPtr)((WORD)cardIndex,
                                                                                 waveOutDevPtr,
                                                                                 spdifDevice
                                                           );
                  incCurFillBufNum(cardIndex);
                  numBufsToPrefill--;
               }
               // --------------------------------------------------------
               // now zero out the number of samples played field.  This
               // should not start counting until the card has actually
               // started.  The numBufsPrefilled field keeps track of how
               // much data is already in the shared buffers.
               // --------------------------------------------------------
               waveOutDevPtr->numSamplesPrefilled   = waveOutDevPtr->numSamplesPlayed;
               waveOutDevPtr->numSamplesPlayed      = 0;
               waveOutDevPtr->lastEstimatedPosition = 0;            // v 1.11
            }
         }
         else {
            // --------------------------------------------------------
            // the device is either paused, or does not have any data
            // to play.
            // --------------------------------------------------------
            waveOutDevPtr->numSamplesPrefilled = 0;
            waveOutDevPtr->numBufsPrefilled    = 0;
         }
      }
   }
}


BOOL  CardIsInPlayMode
(
   DWORD    cardIndex
)
{
   if (getCardState(cardIndex) == K1212_STATE_PLAYING) {
      return TRUE;
   }
   else {
      return FALSE;
   }
}




// ----------------------------------------------------------------------------
// ****************************************************************************
// ----------------------------------------------------------------------------
// The following functions are AppyTime callbacks for various requests to be
// made of the wave device driver.  (v1.09) VM priority event callbacks added.
// ----------------------------------------------------------------------------
// ****************************************************************************
// ----------------------------------------------------------------------------
VOID __stdcall RlsWaveOutBufferCallback
(
   VMHANDLE       hVM, 
   PVOID          Refdata, 
   PCLIENT_STRUCT pcrs, 
   DWORD          Flags
)
{
   #ifdef DEBUG_MSG
      k1212PMIn             reqInputBuf;
   #endif // DEBUG_MSG

   CLIENT_STRUCT         saveRegs;
   WaveOutBufReturnInfo* retBufInfoPtr;

   // -----------------------------------------------------------------------
   // for this callback, RefData is a pointer to a structure containing the
   // ring 3 function address to call, and a pointer to the wave buffer
   // header to be returned to the wave device driver.
   // -----------------------------------------------------------------------
   retBufInfoPtr = (WaveOutBufReturnInfo*)Refdata;
   if (!retBufInfoPtr) {
      return;
   }

   if (retBufInfoPtr->inUse == FALSE) {
      return;        // buffer has already been freed
   }

   // -----------------------------------------------------------------------
   // log a message to a file for debugging/timing info
   // -----------------------------------------------------------------------
   #ifdef DEBUG_MSG
      reqInputBuf.cardIndex = 0;          // can't get real index from here - only
                                          //   one debug file anyway...
      sprintf(reqInputBuf.string, "Sending WOM_DONE for %08x \n", retBufInfoPtr->waveHeaderPtr);
      K1212PostMessageToFile(&reqInputBuf, NULL);
   #endif // DEBUG_MSG

   // -----------------------------------------------------------------------
   // do the nested execution to invoke the wave driver's callback function
   // -----------------------------------------------------------------------
   Save_Client_State(&saveRegs);
   Begin_Nest_Exec();
   
   // -----------------------------------------------------------------------
   // Now make the call that frees the buffer.
   // -----------------------------------------------------------------------
   Simulate_Push(HIWORD((DWORD)K1212_RETURN_WAVEOUTBUF));
   Simulate_Push(LOWORD((DWORD)K1212_RETURN_WAVEOUTBUF));
   Simulate_Push(HIWORD(retBufInfoPtr->lpClient));
   Simulate_Push(LOWORD(retBufInfoPtr->lpClient));
   Simulate_Push(HIWORD((DWORD)retBufInfoPtr->waveHeaderPtr));
   Simulate_Push(LOWORD((DWORD)retBufInfoPtr->waveHeaderPtr));

   Simulate_Far_Call(HIWORD(retBufInfoPtr->ring3CallbackAddress),
                     LOWORD(retBufInfoPtr->ring3CallbackAddress)
   );

   Resume_Exec();
   
   End_Nest_Exec();
   Restore_Client_State(&saveRegs);
   
   FreeWaveOutRetInfo(retBufInfoPtr);
   playBuffersQueued--;
}


VOID __stdcall RlsWaveInBufferCallback
(
   VMHANDLE       hVM, 
   PVOID          Refdata, 
   PCLIENT_STRUCT pcrs, 
   DWORD          Flags
)
{
   #ifdef DEBUG_MSG
      k1212PMIn             reqInputBuf;
   #endif // DEBUG_MSG

   CLIENT_STRUCT         saveRegs;
   WaveInBufReturnInfo*  retBufInfoPtr;

   // -----------------------------------------------------------------------
   // for this callback, RefData is a pointer to a structure containing the
   // ring 3 function address to call, and a pointer to the wave buffer
   // header to be returned to the wave device driver.
   // -----------------------------------------------------------------------
   retBufInfoPtr = (WaveInBufReturnInfo*)Refdata;
   if (!retBufInfoPtr) {
      return;
   }

   if (retBufInfoPtr->inUse == FALSE) {
      return;        // buffer has already been freed
   }

   // -----------------------------------------------------------------------
   // log a message to a file for debugging/timing info
   // -----------------------------------------------------------------------
   #ifdef DEBUG_MSG
      reqInputBuf.cardIndex = 0;          // can't get real index from here - only
                                          //   one debug file anyway...
      sprintf(reqInputBuf.string, "Sending WIM_DONE for %08x \n", retBufInfoPtr->waveHeaderPtr);
      K1212PostMessageToFile(&reqInputBuf, NULL);
   #endif // DEBUG_MSG

   // -----------------------------------------------------------------------
   // do the nested execution to invoke the wave driver's callback function
   // -----------------------------------------------------------------------
   Save_Client_State(&saveRegs);
   Begin_Nest_Exec();
   
   Simulate_Push(HIWORD((DWORD)K1212_RETURN_WAVEINBUF));
   Simulate_Push(LOWORD((DWORD)K1212_RETURN_WAVEINBUF));
   Simulate_Push(HIWORD(retBufInfoPtr->lpClient));
   Simulate_Push(LOWORD(retBufInfoPtr->lpClient));
   Simulate_Push(HIWORD((DWORD)retBufInfoPtr->waveHeaderPtr));
   Simulate_Push(LOWORD((DWORD)retBufInfoPtr->waveHeaderPtr));

   Simulate_Far_Call(HIWORD(retBufInfoPtr->ring3CallbackAddress),
                     LOWORD(retBufInfoPtr->ring3CallbackAddress)
   );
   Resume_Exec();
   
   End_Nest_Exec();
   Restore_Client_State(&saveRegs);
   
   FreeWaveInRetInfo(retBufInfoPtr);
   recBuffersQueued--;
}


// ----------------------------------------------------------------------------
// ****************************************************************************
// ----------------------------------------------------------------------------
// The following functions are utilities for the wave device driver request
// handling functions defined after this section
// ----------------------------------------------------------------------------
// ****************************************************************************
// ----------------------------------------------------------------------------
K1212CmdRet disableWaveVolControl
(
   DWORD          cardIndex,
   WORD           deviceId
)
{
   DWORD                tempId;
   k1212WaveOutDevice*  waveOutPtr;

   if (deviceId > k1212NumWaveDevices) {
      return K1212_CMDRET_BadDevice;
   }
   
   if (deviceId == k1212NumWaveDevices) {
      for (tempId = 0; tempId < k1212NumWaveDevices; tempId++) {
         waveOutPtr = getWaveOutDevPtr(cardIndex,
                                       tempId
                      );
         if (waveOutPtr != 0) {
            waveOutPtr->volumeControlEnabled = FALSE;
         }
      }
      return K1212_CMDRET_Success;
   }
   
   waveOutPtr = getWaveOutDevPtr(cardIndex,
                                 (DWORD)deviceId
                );
   if (waveOutPtr != 0) {
      waveOutPtr->volumeControlEnabled = FALSE;
      return K1212_CMDRET_Success;
   }
   else {
      return K1212_CMDRET_BadDevice;
   }
}

K1212CmdRet enableWaveVolControl
(
   DWORD          cardIndex,
   WORD           deviceId
)
{
   DWORD                tempId;
   k1212WaveOutDevice*  waveOutPtr;

   if (deviceId > k1212NumWaveDevices) {
      return K1212_CMDRET_BadDevice;
   }
   
   if (deviceId == k1212NumWaveDevices) {
      for (tempId = 0; tempId < k1212NumWaveDevices; tempId++) {
         waveOutPtr = getWaveOutDevPtr(cardIndex,
                                       tempId
                      );
         if (waveOutPtr != 0) {
            waveOutPtr->volumeControlEnabled = TRUE;
         }
      }
      return K1212_CMDRET_Success;
   }
   
   waveOutPtr = getWaveOutDevPtr(cardIndex,
                                 (DWORD)deviceId
                );
   if (waveOutPtr != 0) {
      waveOutPtr->volumeControlEnabled = TRUE;
      return K1212_CMDRET_Success;
   }
   else {
      return K1212_CMDRET_BadDevice;
   }
}

K1212CmdRet getWaveVolEnaFlag
(
   DWORD          cardIndex,
   WORD           deviceId,
   BOOL*          boolPtr
)
{
   k1212WaveOutDevice*  waveOutPtr;

   if (deviceId >= k1212NumWaveDevices) {
      return K1212_CMDRET_BadDevice;
   }

   waveOutPtr = getWaveOutDevPtr(cardIndex,
                                 (DWORD)deviceId
                );
   if (waveOutPtr != 0) {
      *boolPtr = waveOutPtr->volumeControlEnabled;
      return K1212_CMDRET_Success;
   }
   else {
      return K1212_CMDRET_BadDevice;
   }
}


K1212CmdRet K1212SetAutoWaveSync
(
   DWORD     cardIndex,
   BOOL      newSetting
)
{
   setAutoWaveSyncMode(cardIndex,
                       newSetting
   );
   MakeSureTheCardIsInPlayMode((WORD)cardIndex);   // see if the card should start now
                                                   // that the sync settings have changed.
   return K1212_CMDRET_Success;   
}


K1212CmdRet setWaveDevSyncFlag
(
   DWORD                cardIndex,
   K1212WaveDeviceType  deviceType,
   DWORD                deviceId,
   DWORD                newSetting
)
{
   k1212WaveOutDevice*  waveOutPtr;
   k1212WaveInDevice*   waveInPtr;

   if (deviceId >= k1212NumWaveDevices) {
      return K1212_CMDRET_BadDevice;
   }
   
   if (deviceType == K1212_WAVETYPE_OUTPUT) {
      waveOutPtr = getWaveOutDevPtr(cardIndex,
                                    deviceId
                   );
      if (waveOutPtr != 0) {
         waveOutPtr->chainLinkSync = (BOOL)newSetting;
         MakeSureTheCardIsInPlayMode((WORD)cardIndex);   // v1.03 - checks whether the
                                                         //    card should now start
         return K1212_CMDRET_Success;
      }
      else {
         return K1212_CMDRET_BadDevice;
      }
   }
   else if (deviceType == K1212_WAVETYPE_INPUT) {
      waveInPtr = getWaveInDevPtr(cardIndex,
                                  deviceId
                  );
      if (waveInPtr != 0) {
         waveInPtr->chainLinkSync = (BOOL)newSetting;
         MakeSureTheCardIsInPlayMode((WORD)cardIndex);   // v1.03 - checks whether the
                                                         //    card should now start
         return K1212_CMDRET_Success;
      }
      else {
         return K1212_CMDRET_BadDevice;
      }
   }
   else {
      return K1212_CMDRET_BadDevice;
   }
}


K1212CmdRet getWaveDevSyncFlag
(
   DWORD                cardIndex,
   K1212WaveDeviceType  deviceType,
   DWORD                deviceId,
   DWORD*               valPtr
)
{
   k1212WaveOutDevice*  waveOutPtr;
   k1212WaveInDevice*   waveInPtr;

   if (deviceId >= k1212NumWaveDevices) {
      return K1212_CMDRET_BadDevice;
   }

   if (deviceType == K1212_WAVETYPE_OUTPUT) {
      waveOutPtr = getWaveOutDevPtr(cardIndex,
                                    deviceId
                   );
      if (waveOutPtr != 0) {
         *valPtr = (DWORD)waveOutPtr->chainLinkSync;
         return K1212_CMDRET_Success;
      }
      else {
         return K1212_CMDRET_BadDevice;
      }
   }
   else if (deviceType == K1212_WAVETYPE_INPUT) {
      waveInPtr = getWaveInDevPtr(cardIndex,
                                  deviceId
                  );
      if (waveInPtr != 0) {
         *valPtr = (DWORD)waveInPtr->chainLinkSync;
         return K1212_CMDRET_Success;
      }
      else {
         return K1212_CMDRET_BadDevice;
      }
   }
   else {
      return K1212_CMDRET_BadDevice;
   }
}


K1212CmdRet getWaveVolume
(
   DWORD           cardIndex,
   WORD            deviceId,
   k1212LeftRight* lrValsPtr
)
{
   k1212WaveOutDevice*  waveOutPtr;

   if (deviceId >= k1212NumWaveDevices) {
      return K1212_CMDRET_BadDevice;
   }

   waveOutPtr = getWaveOutDevPtr(cardIndex,
                                 (DWORD)deviceId
                );
   if (waveOutPtr != 0) {
      lrValsPtr->leftValue  = waveOutPtr->leftVolumeLevel;
      lrValsPtr->rightValue = waveOutPtr->rightVolumeLevel;
      return K1212_CMDRET_Success;
   }
   else {
      return K1212_CMDRET_BadDevice;
   }
}


K1212CmdRet setWaveVolume
(
   DWORD           cardIndex,
   k1212WaveVolume waveVol
)
{
   k1212WaveOutDevice*  waveOutPtr;

   if (waveVol.deviceId >= k1212NumWaveDevices) {
      return K1212_CMDRET_BadDevice;
   }

   waveOutPtr = getWaveOutDevPtr(cardIndex,
                                 (DWORD)waveVol.deviceId
                );
   if (waveOutPtr != 0) {
      waveOutPtr->leftVolumeLevel  = waveVol.volume.leftValue;
      waveOutPtr->rightVolumeLevel = waveVol.volume.rightValue;
      return K1212_CMDRET_Success;
   }
   else {
      return K1212_CMDRET_BadDevice;
   }
}


BOOL DevicesStillActive
(
   WORD cardIndex
)
{
   WORD                 deviceId;
   k1212WaveInDevice*   waveInDevPtr;
   k1212WaveOutDevice*  waveOutDevPtr;

   
   for (deviceId = 0; deviceId < k1212NumWaveDevices; deviceId++) {
      waveInDevPtr  = getWaveInDevPtr(cardIndex,
                                      deviceId
                      );
      waveOutDevPtr = getWaveOutDevPtr(cardIndex,
                                       deviceId
                      );
      if ((waveInDevPtr->state  > K1212_WAVEINSTATE_OPEN) ||
          (waveOutDevPtr->state > K1212_WAVEOUTSTATE_OPEN)) 
      {
         return TRUE;
      }
   }
   return FALSE;
}


BOOL DevicesStillOpen
(
   WORD cardIndex
)
{
   WORD                 deviceId;
   k1212WaveInDevice*   waveInDevPtr;
   k1212WaveOutDevice*  waveOutDevPtr;

   
   for (deviceId = 0; deviceId < k1212NumWaveDevices; deviceId++) {
      waveInDevPtr  = getWaveInDevPtr(cardIndex,
                                      deviceId
                      );
      waveOutDevPtr = getWaveOutDevPtr(cardIndex,
                                       deviceId
                      );
      if ((waveInDevPtr->state  >= K1212_WAVEINSTATE_OPEN) ||
          (waveOutDevPtr->state >= K1212_WAVEOUTSTATE_OPEN)) 
      {
         return TRUE;
      }
   }
   return FALSE;
}


BOOL CardShouldStart
(
   WORD  cardIndex
)
{
   DWORD                deviceId;
   k1212WaveOutDevice*  waveOutDevPtr;
   k1212WaveInDevice*   waveInDevPtr;
   BOOL                 returnVal;

   returnVal = FALSE;      // someone's got to prove that we should start the card

   // ------------------------------------------------------------
   // first, see if the card is in auto sync mode.  If so, go
   // through all the devices and only start the card if all of
   // the open devices have buffers.  The auto sync setting
   // overrides the normal checking that follows if not in auto
   // sync mode.  (v 1.0B6)
   // ------------------------------------------------------------
   if (getAutoWaveSyncMode((DWORD)cardIndex) == TRUE) {
      for (deviceId = 0; deviceId < k1212NumWaveDevices; deviceId++) {

         waveOutDevPtr = getWaveOutDevPtr(cardIndex,
                                          deviceId
                         );
         if (waveOutDevPtr != 0) {
            // ----------------------------------------------------
            // if the output device is paused, or open and does not
            // yet have a buffer, then the card shouldn't start
            // ----------------------------------------------------
            if (waveOutDevPtr->state == K1212_WAVEOUTSTATE_PAUSED) {
               return FALSE;
            }
            else if ((waveOutDevPtr->state >= K1212_WAVEOUTSTATE_OPEN) &&
                     (waveOutDevPtr->curWaveSamplePtr == 0)) {
               return FALSE;
            }
            if ((waveOutDevPtr->state >= K1212_WAVEOUTSTATE_OPEN) &&
                (waveOutDevPtr->curWaveSamplePtr != 0)) {
               returnVal = TRUE;
            }
         }

         waveInDevPtr = getWaveInDevPtr(cardIndex,
                                        deviceId
                        );
         if (waveInDevPtr != 0) {
            // ----------------------------------------------------
            // if the input device is open, has an input buffer,
            // but has not yet been started, then the card should
            // not start.
            // ----------------------------------------------------
            if (waveInDevPtr->state >= K1212_WAVEINSTATE_OPEN) {
               if (waveInDevPtr->curWaveSamplePtr != 0) {
                  if (waveInDevPtr->state != K1212_WAVEINSTATE_RECORDING) {
                     return FALSE;
                  }
                  else {
                     returnVal = TRUE;
                  }
               }
            }
         }
      }  // for all devices

      // ----------------------------------------------------------
      // if we've made it here, then all devices meet the criteria
      // for starting the card in auto sync mode.
      // ----------------------------------------------------------
      return returnVal;
   }

   // ------------------------------------------------------------
   // Check whether any of the playback devices have buffers -
   // if there is a sync start chain, make sure all of the
   // devices have received their first buffer, otherwise, don't
   // start the card yet...
   // ------------------------------------------------------------
   for (deviceId = 0; deviceId < k1212NumWaveDevices; deviceId++) {
      waveOutDevPtr = getWaveOutDevPtr(cardIndex,
                                       deviceId
                      );
      if (waveOutDevPtr != 0) {
         if (waveOutDevPtr->chainLinkSync == TRUE) {
            if ((waveOutDevPtr->state            != K1212_WAVEOUTSTATE_PLAYING) ||
                (waveOutDevPtr->curWaveSamplePtr == 0)) {
               return FALSE;
            }
            else {
               returnVal = TRUE;    // note that we've found a sync'd device that is ready to go
                                    // only another sync'd device that's not ready can stop us now.
            }
         }
         else if (waveOutDevPtr->curWaveSamplePtr != 0) {
            // ------------------------------------------------------------------------
            // v1.03 if the device is paused, then don't start the card on its account
            // ------------------------------------------------------------------------
            if (waveOutDevPtr->state != K1212_WAVEOUTSTATE_PAUSED) {
               returnVal = TRUE;     // note that we've found something possibly worth starting the
                                     //  card for - we'll keep going to see if nothing else  prevents
                                     //  starting the card.
            }
         }
      }
   }

   // ---------------------------------------------------------------
   // If there's no reason to start the card from the playback side,
   // check the record devices...Note, for the sync start cases, we
   // assume that the caller has already given buffers to the device
   // before starting it.  If they haven't, then all bets are off as
   // far as synchronizing the start times goes....
   // ---------------------------------------------------------------
   for (deviceId = 0; deviceId < k1212NumWaveDevices; deviceId++) { 
      waveInDevPtr = getWaveInDevPtr(cardIndex,
                                     deviceId
                     );
      if (waveInDevPtr != 0) {
         if (waveInDevPtr->state >= K1212_WAVEINSTATE_RECORDING) {
            returnVal = TRUE;    // found a reason to start it - continue, in case something else
                                 //  prevents us from starting it...
         }
         else if (waveInDevPtr->chainLinkSync == TRUE) {
             return FALSE;        // found a synced device that hasn't been started yet.
         }
      }
   }
   return returnVal;
}


void TimeStampActiveDevices
(
   WORD  cardIndex
)
{
   DWORD                deviceId;
   k1212WaveOutDevice*  waveOutDevPtr;
   k1212WaveInDevice*   waveInDevPtr;
   DWORD                timeStamp;

   timeStamp = Get_System_Time();

   // --------------------------------------------------------------
   // go through all devices - timestamp the lastIsrTime field of
   // all active devices found.
   // --------------------------------------------------------------
   for (deviceId = 0; deviceId < k1212NumWaveDevices; deviceId++) { 
      waveInDevPtr = getWaveInDevPtr(cardIndex,
                                     deviceId
                     );
      if (waveInDevPtr != 0) {
         if (waveInDevPtr->state >= K1212_WAVEINSTATE_RECORDING) {
            waveInDevPtr->lastIsrTime = timeStamp;
         }
         else {
            waveInDevPtr->lastIsrTime = 0;
         }
      }

      waveOutDevPtr = getWaveOutDevPtr(cardIndex,
                                       deviceId
                      );
      if (waveOutDevPtr != 0) {
         if (waveOutDevPtr->state >= K1212_WAVEOUTSTATE_PLAYING) {
            waveOutDevPtr->lastIsrTime = timeStamp;
         }
         else {
            waveOutDevPtr->lastIsrTime = 0;
         }
      }
   }
}


void MakeSureTheCardIsInPlayMode
(
   WORD  cardIndex
)
{
   WORD                 deviceId;

   // ------------------------------------------------------------
   // If the card is already playing, no need to do anything.
   // ------------------------------------------------------------
   if (getCardState(cardIndex) <= K1212_STATE_OPEN) {

      // ------------------------------------------------------------
      // The card is not playing, so check to see if it should start.
      // If it should, then start it.
      // ------------------------------------------------------------
      if (CardShouldStart(cardIndex)) {
         START_CRITICAL
            PrepareCardForWavePlay(cardIndex);
            CardStartPrefill      (cardIndex);
            setCurFillBufNum      (cardIndex, 
                                   (kNumBuffers - 1)
            );
            setApiMode(cardIndex, 
                       K1212_WAVEDEVICE_API_MODE
            );
            K1212SetupForPlay     (cardIndex);
            K1212TriggerPlay      (cardIndex);
            TimeStampActiveDevices(cardIndex);
         END_CRITICAL
      }
   }
}


void StopCardIfLastDevice
(
   WORD  cardIndex
)
{
   START_CRITICAL
      if (!DevicesStillActive(cardIndex)) {
         if (getCardState(cardIndex) > K1212_STATE_OPEN) {
            K1212StopPlay(cardIndex);
         }
      }
   END_CRITICAL
}


void SetAsCurrentWaveInBuffer
(
   PWAVEHDR           waveHdrPtr,
   k1212WaveInDevice* waveInDevPtr
)
{
   // ------------------------------------------------------------------------
   // WARNING! This function assumes that the buffer that was current has
   // already been given back to the owning application, or never existed.
   // The wave device's queue head pointer is blindly overwritten.
   // ------------------------------------------------------------------------
   START_CRITICAL
      waveInDevPtr->waveBufferQueue = waveHdrPtr;
      if (waveHdrPtr) {
         waveInDevPtr->curWaveSamplePtr = MapPointerToFlat(HIWORD(waveHdrPtr->lpData),
                                                           (DWORD)LOWORD(waveHdrPtr->lpData),
                                                           waveInDevPtr->vmHandle
                                           );
         waveInDevPtr->bufLength        = waveHdrPtr->dwBufferLength;
      }
      else {
         waveInDevPtr->curWaveSamplePtr = 0;
         waveInDevPtr->bufLength        = 0;
      }
      waveInDevPtr->numBytesRecordedInBuf = 0;
   END_CRITICAL
}


void SetAsCurrentWaveOutBuffer
(
   PWAVEHDR            waveHdrPtr,
   k1212WaveOutDevice* waveOutDevPtr
)
{
   // ------------------------------------------------------------------------
   // WARNING! This function assumes that the buffer that was current has
   // already been given back to the owning application, or never existed.
   // The wave device's queue head pointer is blindly overwritten.
   // ------------------------------------------------------------------------
   START_CRITICAL
      waveOutDevPtr->waveBufferQueue = waveHdrPtr;
      if (waveHdrPtr) {
         waveOutDevPtr->curWaveSamplePtr = MapPointerToFlat(HIWORD(waveHdrPtr->lpData),
                                                            (DWORD)LOWORD(waveHdrPtr->lpData), 
                                                            waveOutDevPtr->vmHandle
                                           );
         waveOutDevPtr->remainingBytesInBuf = waveHdrPtr->dwBufferLength;
      }
      else {
         waveOutDevPtr->curWaveSamplePtr    = 0;
         waveOutDevPtr->remainingBytesInBuf = 0;
      }
   END_CRITICAL
}


void PrepareCardForWavePlay
(
   WORD  cardIndex
)
{
   // -----------------------------------------------------------------------
   // fill the playback and record buffers with silence in preparation for
   // the play command to the card.
   // -----------------------------------------------------------------------
   char* recordPointer;
   char* playbackPointer;
   DWORD recordBuffersSize;
   DWORD playbackBuffersSize;
   
   recordPointer       = (char*)getRecBuf0Address (cardIndex);
   playbackPointer     = (char*)getPlayBuf0Address(cardIndex);
   recordBuffersSize   = (getBufferCount() * sizeof(KorgAudioBuffer));
   playbackBuffersSize = (getBufferCount() * sizeof(KorgAudioBuffer));
   
   if ((recordPointer != 0) && (playbackPointer != 0)) {
      memset(recordPointer,
             0,
             recordBuffersSize
      );
      memset(playbackPointer,
             0,
             playbackBuffersSize
      );
   }
}


void ReleaseWaveInBuffer
(
   WORD                 cardIndex,
   k1212WaveInDevice*   waveInDevPtr
)
{
   // ---------------------------------------------------------------
   // release the buffer at the head of the queue back to the owning
   // application.  The next buffer in the list is then setup for
   // continued record operation.
   // ---------------------------------------------------------------
   PWAVEHDR             waveHdrPtr;
   PWAVEHDR             tempWaveHdrPtr;
   WaveInBufReturnInfo* bufReturnInfoPtr;

   START_CRITICAL
      if (waveInDevPtr != 0) {
   
         waveHdrPtr = waveInDevPtr->waveBufferQueue;
   
         if (waveHdrPtr != 0) {

            // ---------------------------------------------------------
            // setup an AppyTime call to give the buffer back to the
            // wave device driver.  The buffer pointer must be a 16:16
            // address.  Get it from the reserved field of the buffer.
            // Return the client handle so the ring 3 driver will know 
            // where to return the buffer.
            // ---------------------------------------------------------
            bufReturnInfoPtr = AllocWaveInRetInfo();
            bufReturnInfoPtr->ring3CallbackAddress = waveInDevPtr->ring3Callback;
            bufReturnInfoPtr->waveHeaderPtr        = (PWAVEHDR)waveHdrPtr->reserved;
            bufReturnInfoPtr->lpClient             = (LPVOID)waveInDevPtr->clientHandle;

            // ---------------------------------------------------------
            // move the next buffer in the list to the head of the queue
            // and setup the record variables for this buffer
            // ---------------------------------------------------------
            tempWaveHdrPtr                = waveHdrPtr;
            waveHdrPtr                    = waveHdrPtr->lpNext;
            tempWaveHdrPtr->lpNext        = 0;        // zero out before returning to app
            waveInDevPtr->waveBufferQueue = waveHdrPtr;
            
            if (waveHdrPtr != 0) {
               waveInDevPtr->curWaveSamplePtr = MapPointerToFlat(HIWORD(waveHdrPtr->lpData),
                                                                 (DWORD)LOWORD(waveHdrPtr->lpData),
                                                                 waveInDevPtr->vmHandle
                                                );
               waveInDevPtr->bufLength = waveHdrPtr->dwBufferLength;
            }
            else {
               waveInDevPtr->curWaveSamplePtr = 0;
               waveInDevPtr->bufLength        = 0;
            }
            waveInDevPtr->numBytesRecordedInBuf = 0;

            // -----------------------------------------------------------------
            // make the AppyTime call.  (v1.09) VM priority event used instead.
            // -----------------------------------------------------------------
            Call_Priority_VM_Event(HIGH_PRI_DEVICE_BOOST,
                                   waveInDevPtr->vmHandle,
                                   (PEF_ALWAYS_SCHED | PEF_WAIT_FOR_STI | PEF_WAIT_NOT_CRIT),
                                   bufReturnInfoPtr,
                                   RlsWaveInBufferCallback,
                                   0,                          // no timeout specified
                                   &RlsBufferInThunk
            );
         }
      }
   END_CRITICAL
}


void ReleaseWaveOutBuffer
(
   WORD                 cardIndex,
   k1212WaveOutDevice*  waveOutDevPtr
)
{
   // ---------------------------------------------------------------
   // release the buffer at the head of the queue back to the owning
   // application.  The next buffer in the list is then setup for
   // continued playback operation.
   // ---------------------------------------------------------------
   PWAVEHDR              waveHdrPtr;
   PWAVEHDR              tempWaveHdrPtr;
   WaveOutBufReturnInfo* bufReturnInfoPtr;
   K1212MemWaveOutRet*   bufReturnBufPtr;

   START_CRITICAL

      if (waveOutDevPtr != 0) {

         waveHdrPtr = waveOutDevPtr->waveBufferQueue;

         if (waveHdrPtr != 0) {

            // ---------------------------------------------------------
            // v1.08  - Check whether we are in a loop.  There are a few
            // things to consider if we are:  if this is a middle block,
            // then the block does not get freed and we setup the next
            // block.  If this is the last block of the loop, we -- the
            // dwLoops count in the first buffer of the loop and setup
            // the first buffer in the loop for playback.  If the dwLoops
            // count is at one, we clear the inLoop flag so that the
            // buffers are freed one by one the last time through.
            // ---------------------------------------------------------
            if (waveOutDevPtr->inLoop == TRUE) {

               if (waveHdrPtr->dwFlags & WHDR_ENDLOOP) {
                  waveOutDevPtr->loopStartBuffer->dwLoops--;
                  tempWaveHdrPtr = waveOutDevPtr->loopStartBuffer;

                  if (waveOutDevPtr->loopStartBuffer->dwLoops <= 1) {
                     waveOutDevPtr->inLoop          = FALSE;
                     waveOutDevPtr->loopStartBuffer = 0;
                  }
               }
               else {      // middle or start block - just go to the next one
                  tempWaveHdrPtr = waveHdrPtr->lpNext;
               }

               waveOutDevPtr->curWaveSamplePtr = MapPointerToFlat(HIWORD(tempWaveHdrPtr->lpData),
                                                                  (DWORD)LOWORD(tempWaveHdrPtr->lpData), 
                                                                  waveOutDevPtr->vmHandle
                                                 );
               waveOutDevPtr->remainingBytesInBuf = tempWaveHdrPtr->dwBufferLength;

               END_CRITICAL
               return;
            }

            // ---------------------------------------------------------
            // setup an AppyTime call to give the buffer back to the
            // wave device driver.  The buffer pointer must be a 16:16
            // address.  Get it from the reserved field of the buffer.
            // Return the client handle so the ring 3 driver will know 
            // where to return the buffer.
            // ---------------------------------------------------------
            bufReturnInfoPtr = AllocWaveOutRetInfo();
            bufReturnInfoPtr->ring3CallbackAddress = waveOutDevPtr->ring3Callback;
            bufReturnInfoPtr->waveHeaderPtr        = (LPWAVEHDR)waveHdrPtr->reserved;
            bufReturnInfoPtr->lpClient             = (LPVOID)waveOutDevPtr->clientHandle;
            bufReturnInfoPtr->delayCount           = waveOutDevPtr->numBufsPrefilled;

            // ---------------------------------------------------------
            // move the next buffer in the list to the head of the queue
            // and setup the playback variables for this buffer
            // ---------------------------------------------------------
            tempWaveHdrPtr                 = waveHdrPtr;
            waveHdrPtr                     = waveHdrPtr->lpNext;
            tempWaveHdrPtr->lpNext         = 0;       // zero out before returning to app
            waveOutDevPtr->waveBufferQueue = waveHdrPtr;
         
            // ---------------------------------------------------------
            // now that the next buffer is in the list, set it up, and
            // check whether it is the start of a loop.
            // ---------------------------------------------------------
            if (waveHdrPtr != 0) {
               waveOutDevPtr->curWaveSamplePtr =
                                      MapPointerToFlat(HIWORD(waveHdrPtr->lpData),
                                                       (DWORD)LOWORD(waveHdrPtr->lpData),
                                                       waveOutDevPtr->vmHandle
                                      );
               waveOutDevPtr->remainingBytesInBuf = waveHdrPtr->dwBufferLength;
               
               if (waveHdrPtr->dwFlags & WHDR_BEGINLOOP) {
                  if (waveHdrPtr->dwLoops > 1) {   // if it's only one, then it's really not a loop
                     waveOutDevPtr->inLoop          = TRUE;
                     waveOutDevPtr->loopStartBuffer = waveHdrPtr;
                  }
               }
            }
            else {
               waveOutDevPtr->curWaveSamplePtr    = 0;
               waveOutDevPtr->remainingBytesInBuf = 0;
            }


            // ------------------------------------------------------------------
            // make the AppyTime call.  (v1.09) Priority VM event used instead.
            // (v1.11) queue up the buffer return event if prefill has offset
            // the actual playback of the buffer.
            // -----------------------------------------------------------------
            if (bufReturnInfoPtr->delayCount == 0) {
               Call_Priority_VM_Event(HIGH_PRI_DEVICE_BOOST,
                                      waveOutDevPtr->vmHandle,
                                      (PEF_ALWAYS_SCHED | PEF_WAIT_FOR_STI | PEF_WAIT_NOT_CRIT),
                                      bufReturnInfoPtr,
                                      RlsWaveOutBufferCallback,
                                      0,                          // no timeout specified
                                      &RlsBufferOutThunk
               );
            }
            else {
               bufReturnBufPtr = (K1212MemWaveOutRet*)((char*)bufReturnInfoPtr - sizeof(K1212MemWaveOutRet*));
               bufReturnBufPtr->next       = waveOutDevPtr->retBufQueue;
               waveOutDevPtr->retBufQueue  = bufReturnBufPtr;
            }
         }
      }   
   END_CRITICAL
}


void FreeWaveInDeviceBuffers
(
   WORD    cardIndex,
   WORD    deviceId
)
{
   // -----------------------------------------------------------------
   // This function takes all buffers in the wave queue and returns 
   // them to the application.
   // -----------------------------------------------------------------
   k1212WaveInDevice*   waveInDevPtr;
   waveInDevPtr = getWaveInDevPtr(cardIndex,
                                  deviceId
                  );
   if (waveInDevPtr != 0) {
      while (waveInDevPtr->waveBufferQueue != 0) {
         ReleaseWaveInBuffer(cardIndex,
                             waveInDevPtr
         );
      }
   }
}


void FreeCurrentWaveInDeviceBuffer
(
   WORD    cardIndex,
   WORD    deviceId
)
{
   // -----------------------------------------------------------------
   // This function takes all buffers in the wave queue and returns 
   // them to the application.
   // -----------------------------------------------------------------
   k1212WaveInDevice*   waveInDevPtr;
   waveInDevPtr = getWaveInDevPtr(cardIndex,
                                  deviceId
                  );
   if (waveInDevPtr != 0) {
      if (waveInDevPtr->waveBufferQueue != 0) {
         ReleaseWaveInBuffer(cardIndex,
                             waveInDevPtr
         );
      }
   }
}


void FreeWaveOutDeviceBuffers
(
   WORD    cardIndex,
   WORD    deviceId
)
{
   // -----------------------------------------------------------------
   // This function takes all buffers in the wave queue and returns 
   // them to the application.  (v1.11) make sure to check the return
   // queue first! (in case any are pending release)
   // -----------------------------------------------------------------
   k1212WaveOutDevice*   waveOutDevPtr;
   K1212MemWaveOutRet*   bufRetPtr;

   waveOutDevPtr = getWaveOutDevPtr(cardIndex,
                                    deviceId
                   );
   if (waveOutDevPtr != 0) {

      // --------------------------------------------------------------------
      // first, clear out the prefill count since it will no longer apply to
      // this device, and it ensures that buffers are freed immediately to
      // the owning application.
      // --------------------------------------------------------------------
      waveOutDevPtr->numBufsPrefilled    = 0;
      waveOutDevPtr->numSamplesPrefilled = 0;

      // --------------------------------------------------------------------
      // now, immediately release any buffers that are in the "to be freed"
      // queue.
      // --------------------------------------------------------------------
      START_CRITICAL
         bufRetPtr = waveOutDevPtr->retBufQueue;
         while (bufRetPtr != 0) {
            // --------------------------------------------------------------------
            // free the buffer and remove it from the linked list.
            // --------------------------------------------------------------------
            Call_Priority_VM_Event(HIGH_PRI_DEVICE_BOOST,
                                   waveOutDevPtr->vmHandle,
                                   (PEF_ALWAYS_SCHED | PEF_WAIT_FOR_STI | PEF_WAIT_NOT_CRIT),
                                   &(bufRetPtr->buffer),
                                   RlsWaveOutBufferCallback,
                                   0,                          // no timeout specified
                                   &RlsBufferOutThunk
            );
            waveOutDevPtr->retBufQueue = bufRetPtr->next;
            bufRetPtr                  = bufRetPtr->next;     // goto next in the list
         }
      END_CRITICAL

      // --------------------------------------------------------------------
      // finally, release any buffers that are in the "to be played"
      // queue.
      // --------------------------------------------------------------------
      while (waveOutDevPtr->waveBufferQueue != 0) {
         ReleaseWaveOutBuffer(cardIndex,
                              waveOutDevPtr
         );
      }
   }
}


BOOL ResetWaveInDevice
(
   WORD      cardIndex,
   WORD      deviceId
)
{
   // ------------------------------------------------------------------
   // stop record for this device.  Release all buffers in the
   // queue to the owning application.  This function gets called when
   // either the reset message has been received, or when the device is
   // disabled or closed to ensure proper buffer clean up.
   // When stopping the device, check all the other devices to see if
   // we need to send the stop command to the card.
   // ------------------------------------------------------------------
   k1212WaveInDevice*  waveInDevPtr;
   
   waveInDevPtr = getWaveInDevPtr(cardIndex,
                                  deviceId
                  );
   if (waveInDevPtr != 0) {
      if (waveInDevPtr->state > K1212_WAVEINSTATE_OPEN) {
         waveInDevPtr->state = K1212_WAVEINSTATE_OPEN;
      }
      waveInDevPtr->numSamplesRecorded    = 0;
      waveInDevPtr->lastIsrTime           = 0;
      waveInDevPtr->lastEstimatedOffset   = 0;        // v 1.11
   }
   FreeWaveInDeviceBuffers(cardIndex,
                           deviceId
   );
   StopCardIfLastDevice(cardIndex);
   return TRUE;
}


BOOL ResetWaveOutDevice
(
   WORD      cardIndex,
   WORD      deviceId
)
{
   // ------------------------------------------------------------------
   // same as ResetWaveInDevice except for a wave output device
   // ------------------------------------------------------------------
   k1212WaveOutDevice* waveOutDevPtr;
   
   waveOutDevPtr = getWaveOutDevPtr(cardIndex,
                                    deviceId
                   );

   if (waveOutDevPtr != 0) {

      waveOutDevPtr->inLoop = FALSE;
      FreeWaveOutDeviceBuffers(cardIndex,
                               deviceId
      );

      if (waveOutDevPtr->state > K1212_WAVEOUTSTATE_OPEN) {
         waveOutDevPtr->state = K1212_WAVEOUTSTATE_OPEN;
      }
      waveOutDevPtr->numSamplesPlayed      = 0;
      waveOutDevPtr->lastIsrTime           = 0;
      waveOutDevPtr->lastEstimatedPosition = 0;            // v 1.11
   }
   StopCardIfLastDevice(cardIndex);
   return TRUE;
}


BOOL VerifyWaveInFormat
(
   WORD                 cardIndex,
   WORD                 deviceId,
   k1212WaveFormatData* formatData
)
{
   if ((formatData->nChannels == 1) || 
       (formatData->nChannels == 2))
   {
      if ((formatData->bitsPerSample == 8) ||
          (formatData->bitsPerSample == 16))
      {
         switch (formatData->sampleRate) {
            case 48000:
            case 44100:
            case 22050:
            case 11025:
               return TRUE;
               
            default:
               return FALSE;
         }
      }
   }
   return FALSE;
}


BOOL VerifyWaveOutFormat
(
   WORD                 cardIndex,
   WORD                 deviceId,
   k1212WaveFormatData* formatData
)
{
   if ((formatData->nChannels == 1) || 
       (formatData->nChannels == 2))
   {
      if ((formatData->bitsPerSample == 8) ||
          (formatData->bitsPerSample == 16))
      {
         switch (formatData->sampleRate) {
            case 48000:
            case 44100:
            case 22050:
            case 11025:
               return TRUE;
               
            default:
               return FALSE;
         }
      }
   }
   return FALSE;
}


DWORD FormatToSampleRate
(
   K1212WaveFormat format
)
{
   switch (format) {
      case WAVFMT_48000_STEREO_16:
      case WAVFMT_48000_STEREO_8:
      case WAVFMT_48000_MONO_16:
      case WAVFMT_48000_MONO_8:
         return 48000;

      case WAVFMT_44100_STEREO_16:
      case WAVFMT_44100_STEREO_8:
      case WAVFMT_44100_MONO_16:
      case WAVFMT_44100_MONO_8:
         return 44100;

      case WAVFMT_22050_STEREO_16:
      case WAVFMT_22050_STEREO_8:
      case WAVFMT_22050_MONO_16:
      case WAVFMT_22050_MONO_8:
         return 22050;

      case WAVFMT_11025_STEREO_16:
      case WAVFMT_11025_STEREO_8:
      case WAVFMT_11025_MONO_16:
      case WAVFMT_11025_MONO_8:
         return 11025;

      default:
         return 0;
   }
}


K1212WaveFormat  K1212FormatParamsToId
(
   k1212WaveFormatData* formatData
)
{
   if (formatData->nChannels == 1) {                // mono IDs
      if (formatData->bitsPerSample == 8) {           // 8 bit PCM IDs
         switch (formatData->sampleRate) {
            case 48000:
               return WAVFMT_48000_MONO_8;
               
            case 44100:
               return WAVFMT_44100_MONO_8;
               
            case 22050:
               return WAVFMT_22050_MONO_8;
               
            case 11025:
               return WAVFMT_11025_MONO_8;               
            
            default:
               return WAVFMT_UNSUPPORTED;
         }
      }
      else if (formatData->bitsPerSample == 16) {     // 16 bit PCM IDs
         switch (formatData->sampleRate) {
            case 48000:
               return WAVFMT_48000_MONO_16;
               
            case 44100:
               return WAVFMT_44100_MONO_16;
               
            case 22050:
               return WAVFMT_22050_MONO_16;
               
            case 11025:
               return WAVFMT_11025_MONO_16;               
            
            default:
               return WAVFMT_UNSUPPORTED;
         }
      }
      else {
         return WAVFMT_UNSUPPORTED;
      }
   }
   else if (formatData->nChannels == 2) {          // stereo IDs
      if (formatData->bitsPerSample == 8) {           // 8 bit PCM IDs
         switch (formatData->sampleRate) {
            case 48000:
               return WAVFMT_48000_STEREO_8;
               
            case 44100:
               return WAVFMT_44100_STEREO_8;
               
            case 22050:
               return WAVFMT_22050_STEREO_8;
               
            case 11025:
               return WAVFMT_11025_STEREO_8;               
            
            default:
               return WAVFMT_UNSUPPORTED;
         }
      }
      else if (formatData->bitsPerSample == 16) {     // 16 bit PCM IDs
         switch (formatData->sampleRate) {
            case 48000:
               return WAVFMT_48000_STEREO_16;
               
            case 44100:
               return WAVFMT_44100_STEREO_16;
               
            case 22050:
               return WAVFMT_22050_STEREO_16;
               
            case 11025:
               return WAVFMT_11025_STEREO_16;               
            
            default:
               return WAVFMT_UNSUPPORTED;
         }
      }
      else {
         return WAVFMT_UNSUPPORTED;
      }
   }
   else {
      return WAVFMT_UNSUPPORTED;
   }
}     // K1212FormatParamsToId()


void SetCardClockRate
(
   DWORD             cardIndex,
   K1212WaveFormat   format
)
{
   ClockSourceIndex clkSrcIndex;
   WORD             clkSrcSel;

   // ------------------------------------------------------------------------------
   // note: THIS FUNCTION SHOULD ONLY BE CALLED WHEN THE CARD IS IN THE OPEN STATE!
   // ------------------------------------------------------------------------------
   clkSrcIndex = getClkSrcRate(cardIndex);
   clkSrcSel   = ClockSourceSelector[clkSrcIndex];

   switch (format) {
      case WAVFMT_48000_STEREO_16:
      case WAVFMT_48000_STEREO_8:
      case WAVFMT_48000_MONO_16:
      case WAVFMT_48000_MONO_8:
         // ------------------------------------------------------------------
         // the wave device needs a 48KHz clock.  If the card is operating at 
         // 44.1KHz, change the rate to 48KHz.
         // ------------------------------------------------------------------
         switch (clkSrcIndex) {
            case K1212_CLKIDX_AdatAt44_1K:
               setClkSrcRate(cardIndex,
                             K1212_CLKIDX_AdatAt48K
               );
               clkSrcSel = ClockSourceSelector[getClkSrcRate(cardIndex)];
               break;
               
            case K1212_CLKIDX_WordAt44_1K:
               setClkSrcRate(cardIndex,
                             K1212_CLKIDX_WordAt48K
               );
               clkSrcSel = ClockSourceSelector[getClkSrcRate(cardIndex)];
               break;
               
            case K1212_CLKIDX_LocalAt44_1K:
               setClkSrcRate(cardIndex,
                             K1212_CLKIDX_LocalAt48K
               );
               clkSrcSel = ClockSourceSelector[getClkSrcRate(cardIndex)];
               break;
               
            default:
               // -------------------------------------------------------------
               // the current clock setting is for 48KHz, so it should be kept
               // there.  We can now just return.
               // -------------------------------------------------------------
               return;
               
         } // switch clkSrcIndex

      default:
         // ------------------------------------------------------------------
         // the wave device needs a 44.1KHz clock.  If the card is operating  
         // at 48KHz, change the rate to 44.1KHz.
         // ------------------------------------------------------------------
         switch (clkSrcIndex) {
            case K1212_CLKIDX_AdatAt48K:
               setClkSrcRate(cardIndex,
                             K1212_CLKIDX_AdatAt44_1K
               );
               clkSrcSel = ClockSourceSelector[getClkSrcRate(cardIndex)];
               break;
               
            case K1212_CLKIDX_WordAt48K:
               setClkSrcRate(cardIndex,
                             K1212_CLKIDX_WordAt44_1K
               );
               clkSrcSel = ClockSourceSelector[getClkSrcRate(cardIndex)];
               break;
               
            case K1212_CLKIDX_LocalAt48K:
               setClkSrcRate(cardIndex,
                             K1212_CLKIDX_LocalAt44_1K
               );
               clkSrcSel = ClockSourceSelector[getClkSrcRate(cardIndex)];
               break;
               
            default:
               // ---------------------------------------------------------------
               // The card is set for 44.1KHz, and that's where we want it, so
               // we'll just return.
               // ---------------------------------------------------------------
               return;
               
         } // switch clkSrcIndex

   }  // switch format

   Send1212Command(cardIndex,
                   K1212_DB_SetClockSourceRate,
                   clkSrcSel,
                   DUMMY_PARAMETER,
                   DUMMY_PARAMETER,
                   DUMMY_PARAMETER
   );
}



// ----------------------------------------------------------------------------
// ****************************************************************************
// ----------------------------------------------------------------------------
// The following functions are utilities for estimating the current playback/
// record position of the card based on the current system time, and the last
// time the card's ISR was serviced.
// ----------------------------------------------------------------------------
// ****************************************************************************
// ----------------------------------------------------------------------------
DWORD FramesPerSample
(
   K1212WaveFormat format
)
{
   switch (format) {
      case WAVFMT_48000_STEREO_16:
      case WAVFMT_48000_STEREO_8:
      case WAVFMT_48000_MONO_16:
      case WAVFMT_48000_MONO_8:
      case WAVFMT_44100_STEREO_16:
      case WAVFMT_44100_STEREO_8:
      case WAVFMT_44100_MONO_16:
      case WAVFMT_44100_MONO_8:
      default:
         return 1;

      case WAVFMT_22050_STEREO_16:
      case WAVFMT_22050_STEREO_8:
      case WAVFMT_22050_MONO_16:
      case WAVFMT_22050_MONO_8:
         return 2;

      case WAVFMT_11025_STEREO_16:
      case WAVFMT_11025_STEREO_8:
      case WAVFMT_11025_MONO_16:
      case WAVFMT_11025_MONO_8:
         return 4;
   }
}

DWORD BytesPerSample    // note: this function factors in whether it's mono or stereo
(
   K1212WaveFormat format
)
{
   switch (format) {
      case WAVFMT_48000_MONO_8:
      case WAVFMT_44100_MONO_8:
      case WAVFMT_22050_MONO_8:
      case WAVFMT_11025_MONO_8:
         return 1;

      case WAVFMT_48000_STEREO_8:
      case WAVFMT_48000_MONO_16:
      case WAVFMT_44100_STEREO_8:
      case WAVFMT_44100_MONO_16:
      case WAVFMT_22050_STEREO_8:
      case WAVFMT_22050_MONO_16:
      case WAVFMT_11025_STEREO_8:
      case WAVFMT_11025_MONO_16:
         return 2;

      case WAVFMT_48000_STEREO_16:
      case WAVFMT_44100_STEREO_16:
      case WAVFMT_22050_STEREO_16:
      case WAVFMT_11025_STEREO_16:
      default:
         return 4;
   }
}


DWORD GetNumUnRecordedSamples
(
   k1212WaveInDevice*   waveInDevPtr
)
{
   DWORD     bytesPerSample;
   DWORD     numUnrecordedBytes;
   LPWAVEHDR tempWaveHdrPtr;

   bytesPerSample = BytesPerSample(waveInDevPtr->format);

   tempWaveHdrPtr     = waveInDevPtr->waveBufferQueue;
   if (tempWaveHdrPtr != 0) {
      numUnrecordedBytes = waveInDevPtr->bufLength - waveInDevPtr->numBytesRecordedInBuf;   

      tempWaveHdrPtr = tempWaveHdrPtr->lpNext;
      while (tempWaveHdrPtr != 0) {
         numUnrecordedBytes += tempWaveHdrPtr->dwBufferLength;
         tempWaveHdrPtr      = tempWaveHdrPtr->lpNext;
      }
   }
   else {
      numUnrecordedBytes = 0;   
   }

   return (numUnrecordedBytes / bytesPerSample);   
}


DWORD GetWaveInEstimatedPosition
(
   DWORD                cardIndex,
   k1212WaveInDevice*   waveInDevPtr
)
{
   DWORD dwPosition;
   DWORD timeDifference;
   DWORD numMsPerInt;
   DWORD samplesPerBuffer;
   DWORD estimatedOffset;
   DWORD maxPossibleOffset;

   START_CRITICAL
      dwPosition      = waveInDevPtr->numSamplesRecorded;
      estimatedOffset = 0;

      // -------------------------------------------------------------------------
      // if the device is playing, and we have a valid IsrTime, then adjust the
      // position based on the estimated number of samples recorded since the
      // last interrupt from the card.
      // -------------------------------------------------------------------------
      if ( (waveInDevPtr->state           == K1212_WAVEINSTATE_RECORDING) &&
           (waveInDevPtr->lastIsrTime     != 0)                           &&
           (waveInDevPtr->waveBufferQueue != 0)                           ) {
   
         timeDifference    = (Get_System_Time() - waveInDevPtr->lastIsrTime);
         numMsPerInt       = getNumMsPerInt(cardIndex);
         samplesPerBuffer  = (kPlayBufferFrames / FramesPerSample(waveInDevPtr->format));
         if (timeDifference > numMsPerInt) {
            timeDifference = numMsPerInt;
         }
         
         // ----------------------------------------------------------------------
         // now that we have the number of ms since the last interrupt, calculate
         // the number of samples we think have happened since then...
         // Make sure we have enough room in the remaining buffer(s), otherwise
         // scale the offset back to what we can actually accommodate.
         // ----------------------------------------------------------------------
         estimatedOffset   = ((timeDifference * samplesPerBuffer) / numMsPerInt);
         maxPossibleOffset = GetNumUnRecordedSamples(waveInDevPtr);
         if (estimatedOffset > maxPossibleOffset) {
            estimatedOffset = maxPossibleOffset;
         }
      }
      dwPosition                        += estimatedOffset;
      waveInDevPtr->lastEstimatedOffset  = estimatedOffset;        // v 1.11
   END_CRITICAL
   return dwPosition;
}


DWORD GetWaveOutEstimatedPosition
(
   DWORD               cardIndex,
   k1212WaveOutDevice* waveOutDevPtr
)
{
   DWORD dwPosition;
   DWORD timeDifference;
   DWORD numMsPerInt;
   DWORD samplesPerBuffer;
   DWORD estimatedOffset;

   START_CRITICAL
      dwPosition      = waveOutDevPtr->numSamplesPlayed;
      estimatedOffset = 0;

      // -------------------------------------------------------------------------
      // if the device is playing, and we have a valid IsrTime, then adjust the
      // position based on the estimated number of samples recorded since the
      // last interrupt from the card.
      // -------------------------------------------------------------------------
      if (waveOutDevPtr->state == K1212_WAVEOUTSTATE_PLAYING) {
   
         if (waveOutDevPtr->lastIsrTime != 0) {
            samplesPerBuffer  = (kPlayBufferFrames / FramesPerSample(waveOutDevPtr->format));
            timeDifference    = (Get_System_Time() - waveOutDevPtr->lastIsrTime);
            numMsPerInt       = getNumMsPerInt(cardIndex);
            if (timeDifference > numMsPerInt) {
               timeDifference = numMsPerInt;    // avoid errors at the boundary
            }
         
            // ----------------------------------------------------------------------
            // now that we have the number of ms since the last interrupt, calculate
            // the number of samples we think have happened since then...
            // ----------------------------------------------------------------------
            estimatedOffset = ((timeDifference * samplesPerBuffer) / numMsPerInt);
         }

         // -------------------------------------------------------------------------
         // If there are no buffers left, we may be playing back residual data from
         // prefilling the buffers.  If this is the case, make sure we don't exceed
         // the upper limit.
         // -------------------------------------------------------------------------
         if (waveOutDevPtr->waveBufferQueue == 0) {
            if (estimatedOffset > waveOutDevPtr->numSamplesPrefilled) {
               estimatedOffset = waveOutDevPtr->numSamplesPrefilled;
            }
         }
         
         dwPosition += estimatedOffset;      // set our estimated position
      }

      if (dwPosition < waveOutDevPtr->lastEstimatedPosition) {
         dwPosition = waveOutDevPtr->lastEstimatedPosition;
      }
      else {
         waveOutDevPtr->lastEstimatedPosition = dwPosition;
      }
   END_CRITICAL
   return dwPosition;
}



// ----------------------------------------------------------------------------
// ****************************************************************************
// ----------------------------------------------------------------------------
// The following functions handle requests from the wave device driver received
// across the protected mode interface.
// ----------------------------------------------------------------------------
// ****************************************************************************
// ----------------------------------------------------------------------------
void K1212GetWaveInputDeviceState
(
   k1212PMIn*  reqInputBufPtr,
   k1212PMOut* reqOutputBufPtr
)
{
   WORD               cardIndex;
   WORD               deviceId;
   k1212WaveInDevice* waveInDevPtr;

   cardIndex = reqInputBufPtr->cardIndex;
   deviceId  = reqInputBufPtr->deviceID;

   // -----------------------------------------------------------------
   // if we can get a pointer to the device, get the device's state
   // -----------------------------------------------------------------
   waveInDevPtr = getWaveInDevPtr(cardIndex,
                                  deviceId
                  );
   if (waveInDevPtr != 0) {
      reqOutputBufPtr->deviceState = waveInDevPtr->state;
      reqOutputBufPtr->returnValue = K1212_CMDRET_Success;
   }
   else {
      reqOutputBufPtr->deviceState = K1212_WAVEINSTATE_DISABLED;
      reqOutputBufPtr->returnValue = K1212_CMDRET_BadDevice;
   }
}


void K1212GetWaveOutputDeviceState
(
   k1212PMIn*  reqInputBufPtr,
   k1212PMOut* reqOutputBufPtr
)
{
   WORD                cardIndex;
   WORD                deviceId;
   k1212WaveOutDevice* waveOutDevPtr;

   cardIndex = reqInputBufPtr->cardIndex;
   deviceId  = reqInputBufPtr->deviceID;

   // -----------------------------------------------------------------
   // if we can get a pointer to the device, get the device's state
   // -----------------------------------------------------------------
   waveOutDevPtr = getWaveOutDevPtr(cardIndex,
                                    deviceId
                   );
   if (waveOutDevPtr != 0) {
      reqOutputBufPtr->deviceState = waveOutDevPtr->state;
      reqOutputBufPtr->returnValue = K1212_CMDRET_Success;
   }
   else {
      reqOutputBufPtr->deviceState = K1212_WAVEOUTSTATE_DISABLED;
      reqOutputBufPtr->returnValue = K1212_CMDRET_BadDevice;
   }
}


void K1212EnableWaveInputDevices
(
   k1212PMIn*  reqInputBufPtr,
   k1212PMOut* reqOutputBufPtr
)
{
   DWORD              deviceId;
   k1212WaveInDevice* waveInDevPtr;
   
   for (deviceId = 0; deviceId < k1212NumWaveDevices; deviceId++) {
      waveInDevPtr = getWaveInDevPtr(reqInputBufPtr->cardIndex,
                                     deviceId
                     );
      if (waveInDevPtr != 0) {
         if (waveInDevPtr->state <= K1212_WAVEINSTATE_ENABLED) {
            waveInDevPtr->state = K1212_WAVEINSTATE_ENABLED;
         }
      }
      else {
         reqOutputBufPtr->returnValue = K1212_CMDRET_BadDevice;
         return;
      }
   }
   reqOutputBufPtr->returnValue = K1212_CMDRET_Success;
}


void K1212DisableWaveInputDevices
(
   k1212PMIn*  reqInputBufPtr,
   k1212PMOut* reqOutputBufPtr
)
{
   WORD               deviceId;
   k1212WaveInDevice* waveInDevPtr;
   
   for (deviceId = 0; deviceId < k1212NumWaveDevices; deviceId++) {
      ResetWaveInDevice(reqInputBufPtr->cardIndex,
                        deviceId
      );
      waveInDevPtr = getWaveInDevPtr(reqInputBufPtr->cardIndex,
                                     deviceId
                     );
      if (waveInDevPtr != 0) {
         waveInDevPtr->state = K1212_WAVEINSTATE_DISABLED;
      }
      else {
         reqOutputBufPtr->returnValue = K1212_CMDRET_BadDevice;
         return;
      }
   }
   reqOutputBufPtr->returnValue = K1212_CMDRET_Success;
}


void K1212EnableWaveOutputDevices
(
   k1212PMIn*  reqInputBufPtr,
   k1212PMOut* reqOutputBufPtr
)
{
   WORD                deviceId;
   k1212WaveOutDevice* waveOutDevPtr;
   
   for (deviceId = 0; deviceId < k1212NumWaveDevices; deviceId++) {
      waveOutDevPtr = getWaveOutDevPtr(reqInputBufPtr->cardIndex,
                                       deviceId
                      );
      if (waveOutDevPtr != 0) {
         if (waveOutDevPtr->state <= K1212_WAVEOUTSTATE_ENABLED) {
            waveOutDevPtr->state = K1212_WAVEOUTSTATE_ENABLED;
         }
      }
      else {
         reqOutputBufPtr->returnValue = K1212_CMDRET_BadDevice;
         return;
      }
   }
   reqOutputBufPtr->returnValue = K1212_CMDRET_Success;
}


void K1212DisableWaveOutputDevices
(
   k1212PMIn*  reqInputBufPtr,
   k1212PMOut* reqOutputBufPtr
)
{
   WORD                deviceId;
   k1212WaveOutDevice* waveOutDevPtr;
   
   for (deviceId = 0; deviceId < k1212NumWaveDevices; deviceId++) {
      ResetWaveOutDevice(reqInputBufPtr->cardIndex,
                         deviceId
      );
      waveOutDevPtr = getWaveOutDevPtr(reqInputBufPtr->cardIndex,
                                       deviceId
                      );
      if (waveOutDevPtr != 0) {
         waveOutDevPtr->state = K1212_WAVEOUTSTATE_DISABLED;
      }
      else {
         reqOutputBufPtr->returnValue = K1212_CMDRET_BadDevice;
      }
   }
   reqOutputBufPtr->returnValue = K1212_CMDRET_Success;
}


void K1212OpenWaveInputDevice
(
   k1212PMIn*  reqInputBufPtr,
   k1212PMOut* reqOutputBufPtr
)
{
   WORD               cardIndex;
   WORD               deviceId;
   k1212WaveInDevice* waveInDevPtr;

   // -----------------------------------------------------------------
   // verify the card index, device index, and requested formats first
   // -----------------------------------------------------------------
   cardIndex = reqInputBufPtr->cardIndex;
   deviceId  = reqInputBufPtr->deviceID;

   if (!VerifyWaveInFormat(cardIndex,
                           deviceId,
                           &(reqInputBufPtr->devOpenData.waveFormatParams)
        )) {
      reqOutputBufPtr->returnValue = K1212_CMDRET_BadFormat;
      return;
   }
   
   // -----------------------------------------------------------------
   // make sure the card is available for a wave device open. 
   // -----------------------------------------------------------------
   if ((getCardState(cardIndex) >= K1212_STATE_OPEN) &&
       (getApiMode(cardIndex)   != K1212_WAVEDEVICE_API_MODE)) {
      reqOutputBufPtr->returnValue = K1212_CMDRET_FailBadState;
      return;
   }
   if (getCardState(cardIndex) < K1212_STATE_READY) {
      reqOutputBufPtr->returnValue = K1212_CMDRET_FailBadState;
      return;
   }
   
   // -----------------------------------------------------------------
   // if we can get a pointer to the device, set the card's state to 
   // open with a wave device and initialize the device's data.
   // -----------------------------------------------------------------
   waveInDevPtr = getWaveInDevPtr(cardIndex,
                                  deviceId
                  );
   if (waveInDevPtr != 0) {

      // --------------------------------------------------------------
      // make sure this device is available to be opened
      // --------------------------------------------------------------
      if (waveInDevPtr->state >= K1212_WAVEINSTATE_OPEN) {
         reqOutputBufPtr->returnValue = K1212_CMDRET_FailBadState;
         return;
      }
      
      // --------------------------------------------------------------
      // it's available.  open it, init data, and select the appropriate
      // buffer fill function.
      // --------------------------------------------------------------
      waveInDevPtr->format         = K1212FormatParamsToId(
                                          &(reqInputBufPtr->devOpenData.waveFormatParams)
                                     ); 

      setApiMode(cardIndex, K1212_WAVEDEVICE_API_MODE);
      if (getCardState(cardIndex) < K1212_STATE_OPEN) {
         K1212OpenCard((DWORD)cardIndex,
                       (APIHandle)reqInputBufPtr->apiHandle
         );
         SetCardClockRate((DWORD)cardIndex,
                          waveInDevPtr->format
         );
      }
      
      waveInDevPtr->ring3Callback  = reqInputBufPtr->devOpenData.driverCallbackAddress;
      waveInDevPtr->clientHandle   = reqInputBufPtr->devOpenData.clientHandle;
      waveInDevPtr->vmHandle       = Get_Cur_VM_Handle();

      waveInDevPtr->translateFunctionPtr  = recordXlateFunctions[waveInDevPtr->format];
      waveInDevPtr->state                 = K1212_WAVEINSTATE_OPEN;
      waveInDevPtr->numSamplesRecorded    = 0;
      waveInDevPtr->lastEstimatedOffset   = 0;        // v 1.11

      reqOutputBufPtr->returnValue = K1212_CMDRET_Success;
      return;
   }
   else {
      reqOutputBufPtr->returnValue = K1212_CMDRET_BadDevice;
      return;
   }
}

void K1212CloseWaveInputDevice
(
   k1212PMIn*  reqInputBufPtr,
   k1212PMOut* reqOutputBufPtr
)
{
   k1212WaveInDevice* waveInDevPtr;

   waveInDevPtr = getWaveInDevPtr(reqInputBufPtr->cardIndex,
                                  reqInputBufPtr->deviceID
                  );
   if (waveInDevPtr != 0) {

      // ------------------------------------------------------
      // v 1.07 Make sure there aren't any buffers still in
      // the input queue.  If there are, the close fails.
      // ------------------------------------------------------
      if (waveInDevPtr->waveBufferQueue != 0) {         
         reqOutputBufPtr->returnValue = K1212_CMDRET_FailBadState;
         return;
      }

      waveInDevPtr->state = K1212_WAVEINSTATE_ENABLED;
   }

   ResetWaveInDevice(reqInputBufPtr->cardIndex,
                     reqInputBufPtr->deviceID
   );

   if (!DevicesStillOpen(reqInputBufPtr->cardIndex)) {
      K1212CloseCard(reqInputBufPtr->cardIndex);
   }

   reqOutputBufPtr->returnValue = K1212_CMDRET_Success;
}

void K1212OpenWaveOutputDevice
(
   k1212PMIn*  reqInputBufPtr,
   k1212PMOut* reqOutputBufPtr
)
{
   WORD                cardIndex;
   WORD                deviceId;
   WORD                sampleNum;
   k1212WaveOutDevice* waveOutDevPtr;

   // -----------------------------------------------------------------
   // verify the requested format first
   // -----------------------------------------------------------------
   cardIndex = reqInputBufPtr->cardIndex;
   deviceId  = reqInputBufPtr->deviceID;

   if (!VerifyWaveOutFormat(cardIndex,
                            deviceId,
                            &(reqInputBufPtr->devOpenData.waveFormatParams)
        )) 
   {
      reqOutputBufPtr->returnValue = K1212_CMDRET_BadFormat;
      return;
   }
   
   // -----------------------------------------------------------------
   // make sure the card is available for a wave device open. 
   // -----------------------------------------------------------------
   if ((getCardState(cardIndex) >= K1212_STATE_OPEN) &&
       (getApiMode(cardIndex)   != K1212_WAVEDEVICE_API_MODE))
   {
      reqOutputBufPtr->returnValue = K1212_CMDRET_FailBadState;
      return;
   }
   if (getCardState(cardIndex) < K1212_STATE_READY) {
      reqOutputBufPtr->returnValue = K1212_CMDRET_FailBadState;
      return;
   }
   
   // -----------------------------------------------------------------
   // if we can get a pointer to the device, set the card's state to 
   // open with a wave device and initialize the device's data.
   // -----------------------------------------------------------------
   waveOutDevPtr = getWaveOutDevPtr(cardIndex,
                                    deviceId
                   );
   if (waveOutDevPtr != 0) {

      // --------------------------------------------------------------
      // make sure this device is available to be opened
      // --------------------------------------------------------------
      if (waveOutDevPtr->state >= K1212_WAVEOUTSTATE_OPEN) {
         reqOutputBufPtr->returnValue = K1212_CMDRET_FailBadState;
         return;
      }
      
      // --------------------------------------------------------------
      // it's available.  open it, init data, and select the appropriate
      // buffer fill function.
      // --------------------------------------------------------------
      waveOutDevPtr->format        = K1212FormatParamsToId(
                                          &(reqInputBufPtr->devOpenData.waveFormatParams)
                                     ); 

      setApiMode(cardIndex, K1212_WAVEDEVICE_API_MODE);
      if (getCardState(cardIndex) < K1212_STATE_OPEN) {
         K1212OpenCard((DWORD)cardIndex,
                       (APIHandle)reqInputBufPtr->apiHandle
         );
         SetCardClockRate((DWORD)cardIndex,
                          waveOutDevPtr->format
         );
      }
      
      waveOutDevPtr->ring3Callback = reqInputBufPtr->devOpenData.driverCallbackAddress;
      waveOutDevPtr->clientHandle  = reqInputBufPtr->devOpenData.clientHandle;
      waveOutDevPtr->vmHandle      = Get_Cur_VM_Handle();

      waveOutDevPtr->translateFunctionPtr 
                                   = playXlateFunctions[waveOutDevPtr->format];

      waveOutDevPtr->numSamplesPlayed      = 0;            // v 1.03
      waveOutDevPtr->curSampleSubscript    = 0;            // v 1.02
      waveOutDevPtr->lastEstimatedPosition = 0;            // v 1.11

      // ---------------------------------------------------------------------------
      // (v1.02) zero out low pass filter previous sample arrays.
      // ---------------------------------------------------------------------------
      for (sampleNum = 0; sampleNum < MAX_NUM_WAVEOUT_FILTER_SAMPLES; sampleNum++) {
         waveOutDevPtr->prevLeftSample[sampleNum] = 0;
         waveOutDevPtr->prevRightSample[sampleNum] = 0;
      }
      
      waveOutDevPtr->state         = K1212_WAVEOUTSTATE_OPEN;

      reqOutputBufPtr->returnValue = K1212_CMDRET_Success;
      return;
   }
   else {
      reqOutputBufPtr->returnValue = K1212_CMDRET_BadDevice;
      return;
   }
}

void K1212CloseWaveOutputDevice
(
   k1212PMIn*  reqInputBufPtr,
   k1212PMOut* reqOutputBufPtr
)
{
   k1212WaveOutDevice* waveOutDevPtr;

   waveOutDevPtr = getWaveOutDevPtr(reqInputBufPtr->cardIndex,
                                    reqInputBufPtr->deviceID
                   );
   if (waveOutDevPtr != 0) {

      // ---------------------------------------------------------------
      // first make sure it's not still playing data...
      // ---------------------------------------------------------------
      if (waveOutDevPtr->waveBufferQueue != 0) {
         reqOutputBufPtr->returnValue = K1212_CMDRET_FailBadState;
         return;
      }

      ResetWaveOutDevice(reqInputBufPtr->cardIndex,      // for good measure
                         reqInputBufPtr->deviceID
      );
   
      waveOutDevPtr->state = K1212_WAVEOUTSTATE_ENABLED;

      if (!DevicesStillOpen(reqInputBufPtr->cardIndex)) {
         K1212CloseCard(reqInputBufPtr->cardIndex);
      }

      reqOutputBufPtr->returnValue = K1212_CMDRET_Success;
   }
   else {
      reqOutputBufPtr->returnValue = K1212_CMDRET_BadDevice;
   }
}

void K1212CheckInputFormat
(
   k1212PMIn*  reqInputBufPtr,
   k1212PMOut* reqOutputBufPtr
)
{
   if (VerifyWaveInFormat(reqInputBufPtr->cardIndex,
                          reqInputBufPtr->deviceID,
                          &(reqInputBufPtr->waveFormatData)
       )) {
      reqOutputBufPtr->returnValue = K1212_CMDRET_Success;
   }
   else {
      reqOutputBufPtr->returnValue = K1212_CMDRET_BadFormat;
   }
}

void K1212CheckOutputFormat
(
   k1212PMIn*  reqInputBufPtr,
   k1212PMOut* reqOutputBufPtr
)
{
   if (VerifyWaveOutFormat(reqInputBufPtr->cardIndex,
                           reqInputBufPtr->deviceID,
                           &(reqInputBufPtr->waveFormatData)
       )) {
      reqOutputBufPtr->returnValue = K1212_CMDRET_Success;
   }
   else {
      reqOutputBufPtr->returnValue = K1212_CMDRET_BadFormat;
   }
}

void K1212AddRecordBuffer
(
   k1212PMIn*  reqInputBufPtr,
   k1212PMOut* reqOutputBufPtr
)
{
   LPWAVEHDR          waveHdrPtr, tempWaveHdrPtr;
   k1212WaveInDevice* waveInDevPtr;

   // --------------------------------------------------------
   // make sure we can return this buffer if we queue it...
   // --------------------------------------------------------
   if (recBuffersQueued >= NUM_WAVE_RETURN_ELEMENTS) {
      reqOutputBufPtr->returnValue = K1212_CMDRET_FailUnspecified;
      return;
   }
   else {
      recBuffersQueued++;
   }

   // --------------------------------------------------------
   // get a pointer to the wave input device
   // --------------------------------------------------------
   waveInDevPtr = getWaveInDevPtr(reqInputBufPtr->cardIndex,
                                  reqInputBufPtr->deviceID
                  );
   if (waveInDevPtr != 0) {

      // --------------------------------------------------------
      // make sure the device is in a state to receive buffers
      // --------------------------------------------------------
      if (waveInDevPtr->state <= K1212_WAVEINSTATE_ENABLED) {
         reqOutputBufPtr->returnValue = K1212_CMDRET_FailBadState;
         return;
      }

      // --------------------------------------------------------
      // place the buffer into the record queue - go to the end
      // of the queue.  Do this in a critical section to avoid
      // interference with fill routine execution.
      // --------------------------------------------------------
      START_CRITICAL
         waveHdrPtr = reqInputBufPtr->waveBufferInfo.waveBufferPtr;
         waveHdrPtr = MapPointerToFlat(HIWORD(waveHdrPtr),
                                       (DWORD)LOWORD(waveHdrPtr),
                                       waveInDevPtr->vmHandle
                      );

         // -----------------------------------------------------
         // before adding it to the queue, make sure it belongs
         // to the client that opened this device.
         // -----------------------------------------------------
         if (reqInputBufPtr->waveBufferInfo.lpClient != (LPVOID)waveInDevPtr->clientHandle) {
            reqOutputBufPtr->returnValue = K1212_CMDRET_BadParams;
            END_CRITICAL
            return;
         }
         
         // ---------------------------------------------------------
         // The buffer does belong to our client.  Use the reserved
         // field to store the 16:16 address of the buffer, then add
         // the buffer to the queue.  Also, zero out the lpNext
         // field to ensure the integrity of the linked list.
         // ---------------------------------------------------------
         waveHdrPtr->reserved = (DWORD)reqInputBufPtr->waveBufferInfo.waveBufferPtr;
         waveHdrPtr->lpNext   = 0;
         if (waveInDevPtr->waveBufferQueue == 0) {
            SetAsCurrentWaveInBuffer(waveHdrPtr,
                                     waveInDevPtr
            );
         }
         else {
            for (tempWaveHdrPtr = waveInDevPtr->waveBufferQueue;
                 tempWaveHdrPtr->lpNext != 0;
                 tempWaveHdrPtr = tempWaveHdrPtr->lpNext) {
            }
            tempWaveHdrPtr->lpNext = waveHdrPtr;
         }
         waveHdrPtr->dwFlags         |= WHDR_INQUEUE;
         waveHdrPtr->dwFlags         &= ~WHDR_DONE ;
         waveHdrPtr->dwBytesRecorded  = 0 ;
      END_CRITICAL
      reqOutputBufPtr->returnValue = K1212_CMDRET_Success;
   }
   else {
      reqOutputBufPtr->returnValue = K1212_CMDRET_BadDevice;
   }
}

void K1212AddPlayBuffer
(
   k1212PMIn*  reqInputBufPtr,
   k1212PMOut* reqOutputBufPtr
)
{
   LPWAVEHDR           waveHdrPtr, tempWaveHdrPtr;
   k1212WaveOutDevice* waveOutDevPtr;
   WORD                cardIndex;
   WORD                deviceId;
   
   // --------------------------------------------------------
   // make sure we can return this buffer if we queue it...
   // --------------------------------------------------------
   if (playBuffersQueued >= NUM_WAVE_RETURN_ELEMENTS) {
      reqOutputBufPtr->returnValue = K1212_CMDRET_FailUnspecified;
      return;
   }
   else {
      playBuffersQueued++;
   }

   // --------------------------------------------------------
   // make sure they are passing us a real buffer
   // --------------------------------------------------------
   if (reqInputBufPtr->waveBufferInfo.waveBufferPtr == 0) {
      reqOutputBufPtr->returnValue = K1212_CMDRET_BadParams;
      return;
   }

   // --------------------------------------------------------
   // get a pointer to the wave output device
   // --------------------------------------------------------
   cardIndex     = reqInputBufPtr->cardIndex;
   deviceId      = reqInputBufPtr->deviceID;
   waveOutDevPtr = getWaveOutDevPtr(cardIndex,
                                    deviceId
                   );
   if (waveOutDevPtr != 0) {

      // --------------------------------------------------------
      // make sure the device is in a state to receive buffers
      // --------------------------------------------------------
      if (waveOutDevPtr->state <= K1212_WAVEOUTSTATE_ENABLED) {
         reqOutputBufPtr->returnValue = K1212_CMDRET_FailBadState;
         return;
      }

      // --------------------------------------------------------
      // if the device hasn't started playing, update the state
      // and initialize the samples played variable
      // --------------------------------------------------------
      if (waveOutDevPtr->state == K1212_WAVEOUTSTATE_OPEN) {
         waveOutDevPtr->numSamplesPlayed      = 0;
         waveOutDevPtr->lastEstimatedPosition = 0;            // v 1.11
         waveOutDevPtr->state                 = K1212_WAVEOUTSTATE_PLAYING;
      }

      // --------------------------------------------------------
      // place the buffer into the playback queue - go to the
      // end of the queue.  Do this in a critical section to
      // avoid interference with fill routine execution.
      // --------------------------------------------------------
      START_CRITICAL
         waveHdrPtr = reqInputBufPtr->waveBufferInfo.waveBufferPtr;
         waveHdrPtr = MapPointerToFlat(HIWORD(waveHdrPtr),
                                       (DWORD)LOWORD(waveHdrPtr),
                                       waveOutDevPtr->vmHandle
                      );

         // -----------------------------------------------------
         // before adding it to the queue, make sure it belongs
         // to the client that opened this device.
         // -----------------------------------------------------
         if (reqInputBufPtr->waveBufferInfo.lpClient != (LPVOID)waveOutDevPtr->clientHandle) {
            reqOutputBufPtr->returnValue = K1212_CMDRET_BadParams;
            END_CRITICAL
            return;
         }
         
         // ------------------------------------------------------------------------
         // The buffer does belong to our client.  Use the reserved field to store 
         // the 16:16 address of the buffer, then add the buffer to the queue.  
         // Also, make sure the lpNext variable is nulled out so we don't have any 
         // linked list errors.
         // ------------------------------------------------------------------------
         waveHdrPtr->reserved = (DWORD)reqInputBufPtr->waveBufferInfo.waveBufferPtr;
         waveHdrPtr->lpNext   = 0;
         if (waveOutDevPtr->waveBufferQueue == 0) {

            // ---------------------------------------------------------------------
            // We are about to add this buffer at the start of the queue.  Check
            // first whether it is the start of a loop.  If it is, setup the loop,
            // then add it to the list.  (v1.08)
            // ---------------------------------------------------------------------
            if (waveHdrPtr->dwFlags & WHDR_BEGINLOOP) {
               if (waveHdrPtr->dwLoops > 1) {   // if it's only one, then it's really not a loop
                  waveOutDevPtr->inLoop          = TRUE;
                  waveOutDevPtr->loopStartBuffer = waveHdrPtr;
               }
            }
            SetAsCurrentWaveOutBuffer(waveHdrPtr,
                                      waveOutDevPtr
            );
         }
         else {
            for (tempWaveHdrPtr = waveOutDevPtr->waveBufferQueue;
                 tempWaveHdrPtr->lpNext != 0;
                 tempWaveHdrPtr = tempWaveHdrPtr->lpNext) {
            }
            tempWaveHdrPtr->lpNext = waveHdrPtr;
         }
         waveHdrPtr->dwFlags |= WHDR_INQUEUE;      // it's now queued
         waveHdrPtr->dwFlags &= ~WHDR_DONE;        // it ain't done yet
      END_CRITICAL
   }
   else {
      reqOutputBufPtr->returnValue = K1212_CMDRET_BadDevice;
      return;
   }

   // -----------------------------------------------------------------------
   // If we got here, then we successfully added the buffer.  The thing to 
   // do now is to start the card if it hasn't already been started.
   // -----------------------------------------------------------------------
   // if the card is closed, we should open it.  This may happen
   // if something catastrophic occurs to the card and we try to
   // auto recover.  Another instance is clearing the API. (1.0B7)
   // ------------------------------------------------------------
   if (getCardState(cardIndex) < K1212_STATE_OPEN) {
      K1212OpenCard(cardIndex,
                    reqInputBufPtr->apiHandle
      );
   }
   MakeSureTheCardIsInPlayMode(cardIndex);
   reqOutputBufPtr->returnValue = K1212_CMDRET_Success;
}


void K1212StartWaveInDevice
(
   k1212PMIn*  reqInputBufPtr,
   k1212PMOut* reqOutputBufPtr
)
{
   // ----------------------------------------------------------------
   // Update the state of the wave device, so the fill routine can
   // begin filling in record buffers.
   // ----------------------------------------------------------------
   k1212WaveInDevice* waveInDevPtr;
   waveInDevPtr = getWaveInDevPtr(reqInputBufPtr->cardIndex,
                                  reqInputBufPtr->deviceID
                  );
   if (waveInDevPtr != 0) {

      // --------------------------------------------------------
      // make sure the device is in a valid state to be started
      // --------------------------------------------------------
      if (waveInDevPtr->state < K1212_WAVEINSTATE_OPEN) {
         reqOutputBufPtr->returnValue = K1212_CMDRET_FailBadState;
         return;
      }
      else {
         // --------------------------------------------------------
         // start the device and initialize the record variables
         // --------------------------------------------------------
         waveInDevPtr->numSamplesRecorded    = 0;
         waveInDevPtr->lastIsrTime           = 0;
         waveInDevPtr->lastEstimatedOffset   = 0;        // v 1.11
         waveInDevPtr->state                 = K1212_WAVEINSTATE_RECORDING;
      }
   }
   else {
      reqOutputBufPtr->returnValue = K1212_CMDRET_BadDevice;
      return;
   }

   // ----------------------------------------------------------------
   // Now, make sure the card is in play mode.
   // ------------------------------------------------------------
   // if the card is closed, we should open it.  This may happen
   // if something catastrophic occurs to the card and we try to
   // auto recover.  Another instance is clearing the API. (1.0B7)
   // ------------------------------------------------------------
   if (getCardState(reqInputBufPtr->cardIndex) < K1212_STATE_OPEN) {
      K1212OpenCard(reqInputBufPtr->cardIndex,
                    reqInputBufPtr->apiHandle
      );
   }
   MakeSureTheCardIsInPlayMode(reqInputBufPtr->cardIndex);
   reqOutputBufPtr->returnValue = K1212_CMDRET_Success;
}

void K1212StopWaveInDevice
(
   k1212PMIn*  reqInputBufPtr,
   k1212PMOut* reqOutputBufPtr
)
{
   k1212WaveInDevice* waveInDevPtr;

   WORD  cardIndex;
   WORD  deviceId;
   DWORD dummyPosition;

   cardIndex = reqInputBufPtr->cardIndex;
   deviceId  = reqInputBufPtr->deviceID;

   // ----------------------------------------------------------------
   // Update the state of the wave device, so the fill routine will
   // not fill in record buffers.
   // ----------------------------------------------------------------
   waveInDevPtr = getWaveInDevPtr(cardIndex,
                                  deviceId
                  );
   if (waveInDevPtr != 0) {
      if (waveInDevPtr->state > K1212_WAVEINSTATE_OPEN) {
      
         // -----------------------------------------------------------
         // update the last estimated position, so we know how many
         // samples to record on the next interrupt from the card.
         // -----------------------------------------------------------
         dummyPosition = GetWaveInEstimatedPosition(cardIndex,
                                                    waveInDevPtr
                         );
         waveInDevPtr->state = K1212_WAVEINSTATE_STOP_PENDING;
      }
   }
   else {
      reqOutputBufPtr->returnValue = K1212_CMDRET_BadDevice;
      return;
   }
   reqOutputBufPtr->returnValue = K1212_CMDRET_Success;
}


void FinishWaveInStop
(
   WORD               cardIndex,
   WORD               deviceId,
   k1212WaveInDevice* waveInDevPtr
)
{
   if (waveInDevPtr->state > K1212_WAVEINSTATE_OPEN) {
      waveInDevPtr->state = K1212_WAVEINSTATE_OPEN;
   }

   // ----------------------------------------------------------------
   // release current buffer back to the client application.
   // ----------------------------------------------------------------
   FreeCurrentWaveInDeviceBuffer(cardIndex,
                                 deviceId
   );
   
   // ----------------------------------------------------------------
   // stop the card from playing if there are no more devices.
   // ----------------------------------------------------------------
   waveInDevPtr->lastIsrTime           = 0;
   waveInDevPtr->lastEstimatedOffset   = 0;        // v 1.11
   StopCardIfLastDevice(cardIndex);
}


void K1212PauseWaveOutDevice
(
   k1212PMIn*  reqInputBufPtr,
   k1212PMOut* reqOutputBufPtr
)
{
   k1212WaveOutDevice* waveOutDevPtr;
   waveOutDevPtr = getWaveOutDevPtr(reqInputBufPtr->cardIndex,
                                    reqInputBufPtr->deviceID
                   );
   if (waveOutDevPtr != 0) {

      // --------------------------------------------------------
      // check the device state.  If it is playing, it will now
      // be paused.  otherwise, the message fails.
      // --------------------------------------------------------
      if (waveOutDevPtr->state < K1212_WAVEOUTSTATE_OPEN) {
         reqOutputBufPtr->returnValue = K1212_CMDRET_FailBadState;
      }
      else {
         waveOutDevPtr->state         = K1212_WAVEOUTSTATE_PAUSED;
         reqOutputBufPtr->returnValue = K1212_CMDRET_Success;
      }
   }
   else {
      reqOutputBufPtr->returnValue = K1212_CMDRET_BadDevice;
   }
}


void K1212ResumeWaveOutDevice
(
   k1212PMIn*  reqInputBufPtr,
   k1212PMOut* reqOutputBufPtr
)
{
   k1212WaveOutDevice* waveOutDevPtr;
   waveOutDevPtr = getWaveOutDevPtr(reqInputBufPtr->cardIndex,
                                    reqInputBufPtr->deviceID
                   );
   if (waveOutDevPtr != 0) {

      // --------------------------------------------------------
      // check the device state.  If it is paused, it will now
      // resume playing.  otherwise, the message should fail,
      // but since the MS Media player thinks it shouldn't, we
      // will return success.
      // --------------------------------------------------------
      if (waveOutDevPtr->state != K1212_WAVEOUTSTATE_PAUSED) {
         reqOutputBufPtr->returnValue = K1212_CMDRET_Success;
      }
      else {
         // ------------------------------------------------------------
         // (v1.03)   The device was paused, update the state, and then
         // make sure the card gets started if it has been held off by
         // this device.
         // ------------------------------------------------------------
         waveOutDevPtr->state = K1212_WAVEOUTSTATE_PLAYING;
         MakeSureTheCardIsInPlayMode(reqInputBufPtr->cardIndex);
         reqOutputBufPtr->returnValue = K1212_CMDRET_Success;
      }
   }
   else {
      reqOutputBufPtr->returnValue = K1212_CMDRET_BadDevice;
   }
}

void K1212ResetWaveDevice
(
   k1212PMIn*  reqInputBufPtr,
   k1212PMOut* reqOutputBufPtr
)
{
   k1212WaveOutDevice* waveOutDevPtr;
   k1212WaveInDevice*  waveInDevPtr;
   
   if (reqInputBufPtr->deviceType == K1212_WAVETYPE_INPUT) {
      waveInDevPtr = getWaveInDevPtr(reqInputBufPtr->cardIndex,
                                     reqInputBufPtr->deviceID
                     );
      if (waveInDevPtr != 0) {
         ResetWaveInDevice(reqInputBufPtr->cardIndex,
                           reqInputBufPtr->deviceID
         );
         reqOutputBufPtr->returnValue = K1212_CMDRET_Success;
      }
      else {
         reqOutputBufPtr->returnValue = K1212_CMDRET_BadDevice;
      }
   }
   else {
      waveOutDevPtr = getWaveOutDevPtr(reqInputBufPtr->cardIndex,
                                       reqInputBufPtr->deviceID
                      );
      if (waveOutDevPtr != 0) {
         ResetWaveOutDevice(reqInputBufPtr->cardIndex,
                            reqInputBufPtr->deviceID
         );
         reqOutputBufPtr->returnValue = K1212_CMDRET_Success;
      }
      else {
         reqOutputBufPtr->returnValue = K1212_CMDRET_BadDevice;
      }
   }
}


void K1212BreakLoop              // v1.08
(
   k1212PMIn*  reqInputBufPtr,
   k1212PMOut* reqOutputBufPtr
)
{
   k1212WaveOutDevice*   waveOutDevPtr;
   LPWAVEHDR             waveHdrPtr, tempWaveHdrPtr, currentWaveHdrPtr;
   BOOL                  breakLoop;
   WaveOutBufReturnInfo* bufReturnInfoPtr;
   K1212MemWaveOutRet*   bufReturnBufPtr;

   reqOutputBufPtr->returnValue = K1212_CMDRET_Success;     // this call always succeeds

   waveOutDevPtr = getWaveOutDevPtr(reqInputBufPtr->cardIndex,
                                    reqInputBufPtr->deviceID
                   );

   if (waveOutDevPtr != 0) {
   
      // ------------------------------------------------------------------
      // To break a loop, we should finish out the current iteration.  So,
      // what we do is free all the loop buffers from the start up to, but
      // not including, the current buffer, and mark the inLoop flag
      // as FALSE, so the rest of the buffers get freed as they are played.
      // ------------------------------------------------------------------

      // ------------------------------------------------------------------
      // if we are not currently in a loop, then no need to do anything...
      // ------------------------------------------------------------------
      if (waveOutDevPtr->inLoop == FALSE) {
         return;
      }
         
      START_CRITICAL
         waveHdrPtr        = waveOutDevPtr->loopStartBuffer;
         currentWaveHdrPtr = waveOutDevPtr->waveBufferQueue;  // by storing this, the
                                                              // playback engine can go
                                                              // on its merry way...
         waveOutDevPtr->inLoop = FALSE;   // from now on, buffers will be freed
         breakLoop             = FALSE;   // init flag for looping through buffers
      END_CRITICAL

      waveOutDevPtr->loopStartBuffer = 0;

      // ---------------------------------------------------------------
      // loop through and free all buffers up to the current one
      // ---------------------------------------------------------------
      while (currentWaveHdrPtr != waveHdrPtr) {
         if (waveHdrPtr->dwFlags & WHDR_ENDLOOP) {   // make sure we don't go too far
            breakLoop = TRUE;
         }

         // ---------------------------------------------------------
         // adjust the linked list.  
         // ---------------------------------------------------------
         tempWaveHdrPtr       = waveHdrPtr->lpNext;
         waveHdrPtr->lpNext   = 0;                 // zero out before returning to app
         
         // ---------------------------------------------------------
         // setup an AppyTime call to give the buffer back to the
         // wave device driver.  The buffer pointer must be a 16:16
         // address.  Get it from the reserved field of the buffer.
         // Return the client handle so the ring 3 driver will know 
         // where to return the buffer.
         // ---------------------------------------------------------
         bufReturnInfoPtr = AllocWaveOutRetInfo();
         bufReturnInfoPtr->ring3CallbackAddress = waveOutDevPtr->ring3Callback;
         bufReturnInfoPtr->waveHeaderPtr        = (LPWAVEHDR)waveHdrPtr->reserved;
         bufReturnInfoPtr->lpClient             = (LPVOID)waveOutDevPtr->clientHandle;
         bufReturnInfoPtr->delayCount           = waveOutDevPtr->numBufsPrefilled;

         // ---------------------------------------------------------
         // make the AppyTime call
         // (v1.11) queue up the buffer return event if prefill has offset
         // the actual playback of the buffer.
         // -----------------------------------------------------------------
         if (bufReturnInfoPtr->delayCount == 0) {
            Call_Priority_VM_Event(HIGH_PRI_DEVICE_BOOST,      // v1.09
                                   waveOutDevPtr->vmHandle,
                                   (PEF_WAIT_FOR_STI | PEF_WAIT_NOT_CRIT),
                                   bufReturnInfoPtr,
                                   RlsWaveOutBufferCallback,
                                   0,                          // no timeout specified
                                   &RlsBufferOutThunk
            );
         }
         else {
            START_CRITICAL
               bufReturnBufPtr = (K1212MemWaveOutRet*)((char*)bufReturnInfoPtr - sizeof(K1212MemWaveOutRet*));
               bufReturnBufPtr->next       = waveOutDevPtr->retBufQueue;
               waveOutDevPtr->retBufQueue  = bufReturnBufPtr;
            END_CRITICAL
         }

         if (breakLoop == TRUE) {
            break;
         }
         else {      // setup for the next buffer in the list
            waveHdrPtr = tempWaveHdrPtr;
         }
      } // end while

      waveOutDevPtr->inLoop = FALSE;   // from now on, buffers will be freed
   }
}


void K1212SetMasterVolume
(
   k1212PMIn*  reqInputBufPtr,
   k1212PMOut* reqOutputBufPtr
)
{
   WORD cardIndex;

   cardIndex = reqInputBufPtr->cardIndex;

   if (VerifyCardIndex(cardIndex)) {
      setMasterVolume(cardIndex,
                      reqInputBufPtr->wValue
      );   
      reqOutputBufPtr->returnValue = K1212_CMDRET_Success;
   }
   else {
      reqOutputBufPtr->returnValue = K1212_CMDRET_BadIndex;
   }
}


void K1212GetMasterVolume
(
   k1212PMIn*  reqInputBufPtr,
   k1212PMOut* reqOutputBufPtr
)
{
   WORD cardIndex;

   cardIndex = reqInputBufPtr->cardIndex;

   if (VerifyCardIndex(cardIndex)) {
      reqOutputBufPtr->wValue      = getMasterVolume(cardIndex);   
      reqOutputBufPtr->returnValue = K1212_CMDRET_Success;
   }
   else {
      reqOutputBufPtr->returnValue = K1212_CMDRET_BadIndex;
   }
}


void K1212SetWaveOutDeviceVolume
(
   k1212PMIn*  reqInputBufPtr,
   k1212PMOut* reqOutputBufPtr
)
{
   k1212WaveVolume waveVol;
   
   waveVol.deviceId             = reqInputBufPtr->deviceID;
   waveVol.volume.leftValue     = reqInputBufPtr->waveVol.leftVolume;
   waveVol.volume.rightValue    = reqInputBufPtr->waveVol.rightVolume;
   reqOutputBufPtr->returnValue = setWaveVolume(reqInputBufPtr->cardIndex,
                                                waveVol
                                  );

   // --------------------------------------------------------------------
   // notify K1212Wav.exe if it is registered.
   // --------------------------------------------------------------------
   if ((k1212WaveVolUpdateFuncPtr != 0) && (wavVolUpdateDataPtr != 0)) {
      wavVolUpdateDataPtr->cardIndex   = reqInputBufPtr->cardIndex;
      wavVolUpdateDataPtr->deviceId    = reqInputBufPtr->deviceID;
      wavVolUpdateDataPtr->leftVolume  = waveVol.volume.leftValue;
      wavVolUpdateDataPtr->rightVolume = waveVol.volume.rightValue;
      wavVolUpdateDataPtr->thisPtr     = wavVolUpdateThisPtr;
                                       
      _VWIN32_QueueUserApc(k1212WaveVolUpdateFuncPtr, 
                           (DWORD)wavVolUpdateDataPtr, 
                           k1212WavThreadHandle
      );
   }
}


void K1212QueryWaveOutDeviceMute
(
   k1212PMIn*  reqInputBufPtr,
   k1212PMOut* reqOutputBufPtr
)
{
   k1212WaveOutDevice* waveOutDevPtr;

   waveOutDevPtr = getWaveOutDevPtr(reqInputBufPtr->cardIndex,
                                    reqInputBufPtr->deviceID
                   );
   if (waveOutDevPtr != 0) {
      reqOutputBufPtr->wValue      = (WORD)waveOutDevPtr->muted;
      reqOutputBufPtr->returnValue = K1212_CMDRET_Success;
      return;
   }
   else {
      reqOutputBufPtr->returnValue = K1212_CMDRET_BadDevice;
   }
}


void K1212MuteWaveOutDevice
(
   k1212PMIn*  reqInputBufPtr,
   k1212PMOut* reqOutputBufPtr
)
{
   k1212WaveOutDevice* waveOutDevPtr;

   waveOutDevPtr = getWaveOutDevPtr(reqInputBufPtr->cardIndex,
                                    reqInputBufPtr->deviceID
                   );
   if (waveOutDevPtr != 0) {
      waveOutDevPtr->muted            = TRUE;
      reqOutputBufPtr->returnValue    = K1212_CMDRET_Success;
      return;
   }
   else {
      reqOutputBufPtr->returnValue = K1212_CMDRET_BadDevice;
   }
}


void K1212UnmuteWaveOutDevice
(
   k1212PMIn*  reqInputBufPtr,
   k1212PMOut* reqOutputBufPtr
)
{
   k1212WaveOutDevice* waveOutDevPtr;

   waveOutDevPtr = getWaveOutDevPtr(reqInputBufPtr->cardIndex,
                                    reqInputBufPtr->deviceID
                   );
   if (waveOutDevPtr != 0) {
      waveOutDevPtr->muted            = FALSE;
      reqOutputBufPtr->returnValue    = K1212_CMDRET_Success;
      return;
   }
   else {
      reqOutputBufPtr->returnValue = K1212_CMDRET_BadDevice;
   }
}


void K1212GetWaveOutDeviceVolume
(
   k1212PMIn*  reqInputBufPtr,
   k1212PMOut* reqOutputBufPtr
)
{
   k1212LeftRight lrVals;

   reqOutputBufPtr->returnValue = getWaveVolume(reqInputBufPtr->cardIndex,
                                                reqInputBufPtr->deviceID,
                                                &lrVals
                                  );

   reqOutputBufPtr->lrVals.leftValue  = lrVals.leftValue;
   reqOutputBufPtr->lrVals.rightValue = lrVals.rightValue;
}


void K1212GetWavePosition
(
   k1212PMIn*  reqInputBufPtr,
   k1212PMOut* reqOutputBufPtr
)
{
   k1212WaveOutDevice* waveOutDevPtr;
   k1212WaveInDevice*  waveInDevPtr;
   DWORD               samplePosition;
   DWORD               sampleRate;
   DWORD               bytesPerSample;

   // ----------------------------------------------------------------------------
   // first, get sample position and sample rate from the specified device
   // ----------------------------------------------------------------------------
   if (reqInputBufPtr->deviceType == K1212_WAVETYPE_INPUT) {
      waveInDevPtr = getWaveInDevPtr(reqInputBufPtr->cardIndex,
                                     reqInputBufPtr->deviceID
                     );
      if (waveInDevPtr != 0) {
//         samplePosition = waveInDevPtr->numSamplesRecorded;
         samplePosition  = GetWaveInEstimatedPosition(reqInputBufPtr->cardIndex,
                                                      waveInDevPtr
                           );
         sampleRate      = FormatToSampleRate(waveInDevPtr->format);
         bytesPerSample  = FormatToBytesPerSample(waveInDevPtr->format);
      }
      else {
         reqOutputBufPtr->returnValue = K1212_CMDRET_BadDevice;
         return;
      }
   }
   else {
      waveOutDevPtr = getWaveOutDevPtr(reqInputBufPtr->cardIndex,
                                       reqInputBufPtr->deviceID
                      );
      if (waveOutDevPtr != 0) {
//         samplePosition  = waveOutDevPtr->numSamplesPlayed;
         samplePosition  = GetWaveOutEstimatedPosition(reqInputBufPtr->cardIndex,
                                                       waveOutDevPtr
                           );
         sampleRate      = FormatToSampleRate(waveOutDevPtr->format);
         bytesPerSample  = FormatToBytesPerSample(waveOutDevPtr->format);
      }
      else {
         reqOutputBufPtr->returnValue = K1212_CMDRET_BadDevice;
         return;
      }
   }

   // ----------------------------------------------------------------------------
   // now, fill in the MMTIME structure based on the sample position, rate, and
   // time format requested by the wave driver.
   // ----------------------------------------------------------------------------
   switch (reqInputBufPtr->mmTimeFormat) {

      case TIME_MS:
         reqOutputBufPtr->mmTimeData.type  = TIME_MS;

         // ------------------------------------------------------------------------
         // To calculate this, we need to simulate floating point to a degree.  
         // Since these can be very large numbers, we need to avoid overflow; so
         // the multiplication must be done after the division.  Since we lose the
         // remainder portion, we need to recapture it and add it into the result.
         // ------------------------------------------------------------------------
         reqOutputBufPtr->mmTimeData.value  = ((samplePosition / sampleRate) * 1000);
         reqOutputBufPtr->mmTimeData.value += ( ((samplePosition % sampleRate) * 1000) / sampleRate);
         break;

      case TIME_BYTES:
         reqOutputBufPtr->mmTimeData.type  = TIME_BYTES;
         reqOutputBufPtr->mmTimeData.value = (samplePosition * bytesPerSample);
         break;

      case TIME_SAMPLES:
      case TIME_SMPTE:
      case TIME_MIDI:
      case TIME_TICKS:
      default:
         // -----------------------------------------------------------------
         // unsupported format, or number of samples format - default to
         // returning the number of samples
         // -----------------------------------------------------------------
         reqOutputBufPtr->mmTimeData.type  = TIME_SAMPLES;
         reqOutputBufPtr->mmTimeData.value = samplePosition;
         break;
   }
   reqOutputBufPtr->returnValue = K1212_CMDRET_Success;
}

// v 1.10
#ifdef DEBUG_MSG
void K1212PostMessageToFile      // v1.05
(
   k1212PMIn*  reqInputBufPtr,
   k1212PMOut* reqOutputBufPtr
)
{

   // -------------------------------------------------------------------------
   // Note:  Modify ms9.mak so that $CDEBUG also defined DEBUG_MSG in order to 
   //        enable the post message feature. (bc4.mak is in $VTOOLSD\include)
   // -------------------------------------------------------------------------
   char finalString[100];
   WORD error1, error2;

   sprintf(finalString, "%d: ", Get_System_Time());    // get current time in milliseconds
   strcat(finalString, reqInputBufPtr->string);   

   R0_WriteFile(FALSE, 
                getFileHandle(reqInputBufPtr->cardIndex), 
                finalString, 
                strlen(finalString), 
                R0_GetFileSize(getFileHandle(reqInputBufPtr->cardIndex), &error1), 
                &error2
   );      
}
#endif // DEBUG_MSG
