//*****************************************************************************
//* Name:
//*      1212sens.c - ADC sensitivity setting module for VxD KORG1212
//*
//* Project:
//*      1212 I/O VxD
//*
//* Author:
//*      Bill Jenkins
//*
//* Description:
//*      This file implements the details of the set input sensitivity command.
//*
//*
//* Modification Log:
//*
//*      1.2   08/06/97 Bill
//*      (v1.0B5) The sensitivity updates are no longer queued.
//*
//*      1.1   02/04/97 Bill
//*      Initial version created.  
//*
//*
//* Copyright (c) 1997 Korg, Inc.
//* All rights reserved.
//*
//* This program is protected as an unpublished work under the U.S.
//* copyright laws.  The above copyright notice is not intended to
//* effect a publication of this work.
//*
//* This program is the confidential and proprietary information of
//* Korg and all its subsidiaries.
//*****************************************************************************

#include <vtoolsc.h>
#include <vmm.h>
#include <debug.h>
#include <vmmreg.h>

#ifndef  K1212INTF_H
#include "1212intf.h"
#endif
#ifndef  K1212CFG_H
#include "1212cfg.h"
#endif
#ifndef  K1212CARD_H
#include "1212card.h"
#endif
#ifndef  K1212IFPR_H
#include "1212ifpr.h"
#endif
#ifndef  K1212SENS_H
#include "1212sens.h"
#endif


// --------------------------------------------------------------------------
// SetBitInWord, ClearBitInWord
//
//    These functions take WORD references and set or clear the specified
//    bit position.
// --------------------------------------------------------------------------
void  SetBitInWord(WORD* theWord,
                  int   bitPosition)
{
  *theWord = *theWord | (0x0001 << bitPosition);
}
void  SetBitInDWord(DWORD* theWord,
                    int   bitPosition)
{
  *theWord = *theWord | (0x00000001 << bitPosition);
}
                     
void  ClearBitInWord(WORD* theWord,
                     int   bitPosition)
{
   *theWord = *theWord & ~(0x0001 << bitPosition);
}
void  ClearBitInDWord(DWORD* theWord,
                      int   bitPosition)
{
   *theWord = *theWord & ~(0x00000001 << bitPosition);
}

// --------------------------------------------------------------------------
// WaitRTCTicks
//
//    This function waits the specified number of real time clock ticks.  
//    According to the DDK, each tick is ~0.8 microseconds.
// --------------------------------------------------------------------------
void  WaitRTCTicks(DWORD numTicksToWait)
{
   DWORD killTime;
   DWORD hiTime;
   DWORD hiTimeStart;
   DWORD loTimeStart;
   DWORD loTime;
   
   VTD_Get_Real_Time(&hiTimeStart, &loTimeStart);
   loTime = loTimeStart;
   
   while (loTime < (loTimeStart + numTicksToWait)) {
      VTD_Get_Real_Time(&hiTime, &loTime);
      if (hiTimeStart != hiTime) {
         return;
      }
   }
}

void  WaitAFew(DWORD numTicksToWait)
{
   DWORD loopCount;
   for (loopCount = 0; loopCount < 100; loopCount++) {
   }
}


// --------------------------------------------------------------------------
// WriteADCSensitivity
//
//    This function writes the ADC input sensitivities to the card using the
//    custom serial interface described in the K1212 driver API document.
// --------------------------------------------------------------------------
BOOL  WriteADCSensitivity(DWORD cardIndex)
{
   sensBits  sensVals;
   int       bitPosition;
   int       channel;
   BOOL      clkIs48K;
   BOOL      monModeSet;
   WORD      dataBit;
   WORD      controlValue;    // this keeps the current value to be written to
                              //  the card's eeprom control register.
   WORD      count;

   // ----------------------------------------------------------------------------
   // initialize things.  The local init bit is always set when writing to the
   // card's control register.
   // ----------------------------------------------------------------------------
   controlValue = 0;
   SetBitInWord(&controlValue, SET_SENS_LOCALINIT_BITPOS);    // init the control value

   // ----------------------------------------------------------------------------
   // make sure the card is not in monitor mode when we do this update.
   // ----------------------------------------------------------------------------
   if ((getCardState(cardIndex)        == K1212_STATE_MONITOR) ||
       (getIdleMonitorState(cardIndex) == TRUE)) {
      setCardCommand(cardIndex, 0xffffffff);
      monModeSet = TRUE;
      WaitForCardStopAck(cardIndex);
   }
   else {
      monModeSet = FALSE;
   }

   // ----------------------------------------------------------------------------
   // we are about to send new values to the card, so clear the new values queued
   // flag.  Also, clear out mailbox 3, so we don't lockup.
   // ----------------------------------------------------------------------------
   writeMailbox3(cardIndex, 0);
   WaitRTCTicks(LOADSHIFT_DELAY);
   
   // ----------------------------------------------------------------------------
   // determine whether we are running a 48K or 44.1K clock.  This info is used
   // later when setting the SPDIF FF after the volume has been shifted in.
   // ----------------------------------------------------------------------------
   switch (getClkSrcRate(cardIndex)) {
      case K1212_CLKIDX_AdatAt44_1K:
      case K1212_CLKIDX_WordAt44_1K:
      case K1212_CLKIDX_LocalAt44_1K:
         clkIs48K = FALSE;
         break;
         
      case K1212_CLKIDX_WordAt48K:
      case K1212_CLKIDX_AdatAt48K:
      case K1212_CLKIDX_LocalAt48K:
      default:
         clkIs48K = TRUE;
         break;
   }
 
   // ----------------------------------------------------------------------------
   // start the update.  Setup the bit structure and then shift the bits.
   // ----------------------------------------------------------------------------
   sensVals.leftChanId   = SET_SENS_LEFTCHANID;
   sensVals.rightChanId  = SET_SENS_RIGHTCHANID;
   sensVals.leftChanVal  = (char)getLeftADCInSens(cardIndex);
   sensVals.rightChanVal = (char)getRightADCInSens(cardIndex);

   // ----------------------------------------------------------------------------
   // now start shifting the bits in.  Start with the left channel then the right.
   // ----------------------------------------------------------------------------
   for (channel = 0; channel < 2; channel++) {

      // ----------------------------------------------------------------------------
      // Bring the load/shift line low, then wait - the spec says >150ns from load/
      // shift low to the first rising edge of the clock.
      // ----------------------------------------------------------------------------
      ClearBitInWord(&controlValue, SET_SENS_LOADSHIFT_BITPOS);
      ClearBitInWord(&controlValue, SET_SENS_DATA_BITPOS);
      writeSensReg(cardIndex, controlValue);                          // load/shift goes low
      WaitRTCTicks(LOADSHIFT_DELAY);

      for (bitPosition = 15; bitPosition >= 0; bitPosition--) {       // for all the bits
         if (channel == 0) {
            if (sensVals.leftSensBits & (0x0001 << bitPosition)) {
               SetBitInWord(&controlValue, SET_SENS_DATA_BITPOS);     // data bit set high
            }
            else {
               ClearBitInWord(&controlValue, SET_SENS_DATA_BITPOS);   // data bit set low
            }
         }
         else {
            if (sensVals.rightSensBits & (0x0001 << bitPosition)) {
               SetBitInWord(&controlValue, SET_SENS_DATA_BITPOS);     // data bit set high
            }
            else {
               ClearBitInWord(&controlValue, SET_SENS_DATA_BITPOS);   // data bit set low
            }
         }
         
         ClearBitInWord(&controlValue, SET_SENS_CLOCK_BITPOS);
         writeSensReg(cardIndex, controlValue);                       // clock goes low
         WaitRTCTicks(SENSCLKPULSE_WIDTH);
         SetBitInWord(&controlValue, SET_SENS_CLOCK_BITPOS);
         writeSensReg(cardIndex, controlValue);                       // clock goes high
         WaitRTCTicks(SENSCLKPULSE_WIDTH);      
      }
   
      // ----------------------------------------------------------------------------
      // finish up SPDIF for left.  Bring the load/shift line high, then write a one 
      // bit if the clock rate is 48K otherwise write 0.
      // ----------------------------------------------------------------------------
      ClearBitInWord(&controlValue, SET_SENS_DATA_BITPOS);
      ClearBitInWord(&controlValue, SET_SENS_CLOCK_BITPOS);
      SetBitInWord(&controlValue, SET_SENS_LOADSHIFT_BITPOS);
      writeSensReg(cardIndex, controlValue);                   // load shift goes high - clk low
      WaitRTCTicks(SENSCLKPULSE_WIDTH);      
   
      if (clkIs48K == TRUE) {
         SetBitInWord(&controlValue, SET_SENS_DATA_BITPOS);
      }
      writeSensReg(cardIndex, controlValue);                   // set/clear data bit
      WaitRTCTicks(ONE_RTC_TICK);      
      SetBitInWord(&controlValue, SET_SENS_CLOCK_BITPOS);
      writeSensReg(cardIndex, controlValue);                   // clock goes high
      WaitRTCTicks(SENSCLKPULSE_WIDTH);      
      ClearBitInWord(&controlValue, SET_SENS_CLOCK_BITPOS);
      writeSensReg(cardIndex, controlValue);                   // clock goes low
      WaitRTCTicks(SENSCLKPULSE_WIDTH);      
   }
   
   // ----------------------------------------------------------------------------
   // The update is complete.  Set a timeout.  This is the inter-update delay.
   // Also, if the card was in monitor mode, restore it.
   // ----------------------------------------------------------------------------
   for (count = 0; count < 10; count++) {      // slow the caller down a little
      WaitRTCTicks(SENSCLKPULSE_WIDTH);      
   }

   if (monModeSet == TRUE) {
      Send1212Command(cardIndex,
                      K1212_DB_SelectPlayMode,
                      K1212_MODE_MonitorOn,
                      DUMMY_PARAMETER,
                      DUMMY_PARAMETER,
                      DUMMY_PARAMETER
      );
   }
   return TRUE;
}

