//*****************************************************************************
//* Name:
//*      1212file.c
//*
//* Project:
//*      1212 I/O VxD
//*
//* Author:
//*      Bill Jenkins
//*
//* Description:
//*      This file contains the definition of file access functions.
//*
//* Modification Log:
//*
//*      1.1   11/19/96 Bill
//*      Initial version created.  
//*
//*
//* Copyright (c) 1996 Korg, Inc.
//* All rights reserved.
//*
//* This program is protected as an unpublished work under the U.S.
//* copyright laws.  The above copyright notice is not intended to
//* effect a publication of this work.
//*
//* This program is the confidential and proprietary information of
//* Korg and all its subsidiaries.
//*****************************************************************************

#include <vtoolsc.h>
#include <vmm.h>
#include <debug.h>
#include <vmmreg.h>
#include <ifs.h>
#include <winerror.h>

#ifndef  K1212FILE_H
#include "1212file.h"
#endif


// ---------------------------------------------------------------------------
// openFileRO
//
//    This function opens the file as read only using ring 0 file services.
//
//    Returns: handle to the newly opened file
// ---------------------------------------------------------------------------
HANDLE openFileRO(char* filePath     // name of the file to open (may have absolute path)
       )
{
   HANDLE fileHandle;
   BYTE   action;
   int    initState;
   WORD   fileError;       
   
   // --------------------------------------------------------------------------
   // test the state of the VMM
   // --------------------------------------------------------------------------
   initState = VMM_GetSystemInitState();
   
   #ifdef DEBUG
      Debug_Printf_Service("Korg 1212 Driver: VMM init state is 0x%x\n", 
                           initState
      );
   #endif // DEBUG
      
   // --------------------------------------------------------------------------
   // open up the file
   // --------------------------------------------------------------------------
   #ifdef DEBUG
      Debug_Printf_Service("Korg 1212 Driver: Opening file %s\n", 
                           filePath
      );
   #endif // DEBUG

   fileHandle = R0_OpenCreateFile(FALSE,                    // global context for the file
                                  filePath,                 // name of the file to open (+path)
                                  (OPEN_ACCESS_READONLY |   // read-only file
                                   OPEN_SHARE_DENYWRITE),
                                  0,                        // attribute  (specify nothing)
                                  ACTION_IFEXISTS_OPEN,     // this file must already exist
                                  0,                        // 0 means no flags specified
                                  &fileError,               // gets filled in with error code
                                  &action                   // gets filled in with the action 
                );                                          //   taken to open the file
              
   // --------------------------------------------------------------------------
   // if no errors occurred, return the file handle.  Otherwise, return NULL
   // --------------------------------------------------------------------------
   if (fileError == NO_ERROR) {
      #ifdef DEBUG
         Debug_Printf_Service("Korg 1212 Driver: Open file succeeded.  Handle is 0x%x\n",
                              fileHandle
         );
      #endif // DEBUG
      return(fileHandle);
   }
   else {   // could not open the file
      #ifdef DEBUG
         Debug_Printf_Service("Korg 1212 Driver: Could not open file\n");
         Debug_Printf_Service("Korg 1212 Driver:  failure code is 0x%x\n", 
                              fileError
         );
         Debug_Printf_Service("Korg 1212 Driver:  file name is \'%s\'\n", 
                              filePath
         );
      #endif // DEBUG
      return((HANDLE)NULL);
   }
}


// ---------------------------------------------------------------------------
// getFileSize
//
//    Returns: size of the specified file
// ---------------------------------------------------------------------------
DWORD getFileSize(HANDLE fileHandle     // specifies which file to get the size of
      )
{
   DWORD fileSize;
   WORD  retCode;
   
   #ifdef DEBUG
      Debug_Printf_Service("Korg 1212 Driver: Getting file size...\n");
   #endif // DEBUG
             
   fileSize = R0_GetFileSize(fileHandle,
                             &retCode
              );

   if (retCode == NO_ERROR) {
      #ifdef DEBUG
         Debug_Printf_Service("Korg 1212 Driver: File size was %d\n", 
                              fileSize
         );
      #endif // DEBUG
      return fileSize;
   }
   else {   // get file size failed
      #ifdef DEBUG
         Debug_Printf_Service("Korg 1212 Driver: Could not get file size\n");
         Debug_Printf_Service("Korg 1212 Driver:  failure code is 0x%x\n", 
                              retCode
         );
         Debug_Printf_Service("Korg 1212 Driver:  file handle is 0x%x\n", 
                              fileHandle
         );
      #endif // DEBUG
      return 0;
   }
}


// ---------------------------------------------------------------------------
// readFile
//
//    This function reads the specified file into a specified memory buffer
//    using ring 0 file services.
//
//    Returns: TRUE if the read was successful, FALSE if not.
// ---------------------------------------------------------------------------
BOOL  readFile(HANDLE fileHandle,    // specifies the file to read
               PVOID  memHandle,     // specifies where to read the file into
               DWORD  numBytes)      // specifies how much of the file to read
{
   WORD  retCode;
   DWORD numBytesRead;
   
   // ---------------------------------------------------------------------
   // make sure we have a valid pointer
   // ---------------------------------------------------------------------
   if (!memHandle) {
      #ifdef DEBUG
         Debug_Printf_Service("Korg 1212 Driver: Null memory pointer to read file routine\n");
      #endif // DEBUG
      return FALSE;
   }
   
   // ---------------------------------------------------------------------
   // read the file into memory
   // ---------------------------------------------------------------------
   #ifdef DEBUG
      Debug_Printf_Service("Korg 1212 Driver: Reading file...\n");
   #endif // DEBUG

   numBytesRead = R0_ReadFile(FALSE,           // not a context read
                              fileHandle,      // which file to read
                              memHandle,       // where to put the file's contents
                              numBytes,        // how many bytes to read
                              0,               // start at the beginning of the file
                              &retCode    // gets filled in with actual bytes read
                  );

   // ---------------------------------------------------------------------
   // check for errors
   // ---------------------------------------------------------------------
   if (retCode == NO_ERROR) {
      if (numBytesRead >= numBytes) {
         #ifdef DEBUG
            Debug_Printf_Service("Korg 1212 Driver: File read successful - %d bytes read\n",
                                 numBytesRead
            );
         #endif // DEBUG
         return TRUE;
      }
      else {
         #ifdef DEBUG
            Debug_Printf_Service("Korg 1212 Driver: Read %d out of %d bytes from file\n",
                                 numBytesRead, 
                                 numBytes
            );
            Debug_Printf_Service("Korg 1212 Driver:  file handle = 0x%x\n", 
                                 fileHandle
            );
         #endif // DEBUG
         return FALSE;
      }
   }
   else {   // read file failed
      #ifdef DEBUG
         Debug_Printf_Service("Korg 1212 Driver: Could not read from file handle 0x%x\n",
                              fileHandle
         );
         Debug_Printf_Service("Korg 1212 Driver:  failure code is 0x%x\n", 
                              retCode
         );
      #endif // DEBUG
      return FALSE;
   }
}


// ---------------------------------------------------------------------------
// closeFile
//
//    This function closes the specified file using ring 0 file services.
//
//    Returns: TRUE if the file closed successfully, FALSE if not.
// ---------------------------------------------------------------------------
BOOL  closeFile(HANDLE fileHandle    // specifies which file to close
      )
{
   WORD retCode;
   
   #ifdef DEBUG
      Debug_Printf_Service("Korg 1212 Driver: Closing file...\n");
   #endif // DEBUG

   R0_CloseFile(fileHandle, &retCode);
   
   if (retCode == NO_ERROR) {
      #ifdef DEBUG
         Debug_Printf_Service("Korg 1212 Driver: Close file succeeded.\n");
      #endif // DEBUG
      return TRUE;
   }
   else {   // get file size failed
      #ifdef DEBUG
         Debug_Printf_Service("Korg 1212 Driver: Failed to close file handle 0x%x\n",
                              fileHandle
         );
         Debug_Printf_Service("Korg 1212 Driver:  failure code is 0x%x\n", 
                              retCode
         );
      #endif // DEBUG
      return FALSE;
   }   
}




