;;; ergoemacs-unbind.el --- unbind keys

;; Copyright (C) 2013, 2014 Free Software Foundation, Inc.

;; Maintainer: Matthew L. Fidler
;; Keywords: convenience

;; ErgoEmacs is free software: you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published
;; by the Free Software Foundation, either version 3 of the License,
;; or (at your option) any later version.

;; ErgoEmacs is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;; General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with ErgoEmacs.  If not, see <http://www.gnu.org/licenses/>.

;;; Commentary:

;; This library define keys that we want to set/unset because they are
;; already defined by ergoemacs minor mode

;; Todo:

;; 

;;; Code:
(eval-when-compile 
  (require 'cl)
  (require 'ergoemacs-macros))

(require 'edmacro)

(defvar ergoemacs-emacs-default-bindings
  '(
    ("<C-M-down>" (down-list))
    ("<C-M-end>" (end-of-defun))
    ("<C-M-home>" (beginning-of-defun))
    ("<C-M-left>" (backward-sexp))
    ("<C-M-right>" (forward-sexp))
    ("<C-M-up>" (backward-up-list))
    ("<C-S-backspace>" (kill-whole-line))
    ("<C-backspace>" (backward-kill-word))
    ("<C-delete>" (kill-word))
    ("<C-down>" (forward-paragraph))
    ("<C-end>" (end-of-buffer))
    ("<C-home>" (beginning-of-buffer))
    ("<C-insert>" (kill-ring-save))
    ("<C-insertchar>" (kill-ring-save))
    ("<C-left>" (left-word backward-word))
    ("<C-next>" (scroll-left))
    ("<C-prior>" (scroll-right))
    ("<C-right>" (right-word forward-word))
    ("<C-up>" (backward-paragraph))
    ("<C-wheel-down>" (mwheel-scroll))
    ("<C-wheel-up>" (mwheel-scroll))
    ("<M-begin>" (beginning-of-buffer-other-window))
    ("<M-end>" (end-of-buffer-other-window))
    ("<M-home>" (beginning-of-buffer-other-window))
    ("<M-left>" (left-word backward-word))
    ("<M-next>" (scroll-other-window))
    ("<M-prior>" (scroll-other-window-down))
    ("<M-right>" (right-word forward-word))
    ("<S-delete>" (kill-region))
    ("<S-insert>" (yank))
    ("<S-insertchar>" (yank))
    ("<again>" (repeat-complex-command))
    ("<begin>" (beginning-of-buffer))
    ("<compose-last-chars>" (compose-last-chars))
    ("<delete-frame>" (handle-delete-frame))
    ("<deletechar>" (delete-forward-char delete-char))
    ("<deleteline>" (kill-line))
    ("<down>" (next-line))
    ("<end>" (end-of-buffer move-end-of-line))
    ("<execute>" (execute-extended-command))
    ("<f10>" (menu-bar-open))
    ("<f1> ." (display-local-help))
    ("<f1> 4 i" (info-other-window))
    ("<f1> 4" (prefix))
    ("<f1> <f1>" (help-for-help))
    ("<f1> <help>" (help-for-help))
    ("<f1> ?" (help-for-help))
    ("<f1> C" (describe-coding-system))
    ("<f1> C-\\" (describe-input-method))
    ("<f1> C-a" (about-emacs))
    ("<f1> C-c" (describe-copying))
    ("<f1> C-d" (view-emacs-debugging describe-distribution))
    ("<f1> C-e" (view-external-packages view-emacs-problems))
    ("<f1> C-f" (view-emacs-FAQ))
    ("<f1> C-h" (help-for-help))
    ("<f1> C-n" (view-emacs-news))
    ("<f1> C-o" (describe-distribution nil))
    ("<f1> C-p" (view-emacs-problems describe-gnu-project))
    ("<f1> C-t" (view-emacs-todo))
    ("<f1> C-w" (describe-no-warranty))
    ("<f1> F" (Info-goto-emacs-command-node))
    ("<f1> I" (describe-input-method))
    ("<f1> K" (Info-goto-emacs-key-command-node))
    ("<f1> L" (describe-language-environment))
    ("<f1> P" (nil describe-package))
    ("<f1> RET" (view-order-manuals))
    ("<f1> S" (info-lookup-symbol))
    ("<f1> a" (apropos-command))
    ("<f1> b" (describe-bindings))
    ("<f1> c" (describe-key-briefly))
    ("<f1> d" (apropos-documentation))
    ("<f1> e" (view-echo-area-messages))
    ("<f1> f" (describe-function))
    ("<f1> g" (nil describe-gnu-project))
    ("<f1> h" (view-hello-file))
    ("<f1> i" (info))
    ("<f1> k" (describe-key))
    ("<f1> l" (view-lossage))
    ("<f1> m" (describe-mode))
    ("<f1> n" (view-emacs-news))
    ("<f1> p" (finder-by-keyword))
    ("<f1> q" (help-quit))
    ("<f1> r" (info-emacs-manual))
    ("<f1> s" (describe-syntax))
    ("<f1> t" (help-with-tutorial))
    ("<f1> v" (describe-variable))
    ("<f1> w" (where-is))
    ("<f1>" (prefix))
    ("<f2> 2" (2C-two-columns))
    ("<f2> <f2>" (2C-two-columns))
    ("<f2> b" (2C-associate-buffer))
    ("<f2> s" (2C-split))
    ("<f2>" (prefix))
    ("<f3>" (kmacro-start-macro-or-insert-counter))
    ("<f4>" (kmacro-end-or-call-macro))
    ("<find>" (search-forward))
    ("<header-line>" (prefix))
    ("<help> ." (display-local-help))
    ("<help> 4 i" (info-other-window))
    ("<help> 4" (prefix))
    ("<help> <f1>" (help-for-help))
    ("<help> <help>" (help-for-help))
    ("<help> ?" (help-for-help))
    ("<help> C" (describe-coding-system))
    ("<help> C-\\" (describe-input-method))
    ("<help> C-a" (about-emacs))
    ("<help> C-c" (describe-copying))
    ("<help> C-d" (describe-distribution view-emacs-debugging))
    ("<help> C-e" (view-emacs-problems view-external-packages))
    ("<help> C-f" (view-emacs-FAQ))
    ("<help> C-h" (help-for-help))
    ("<help> C-n" (view-emacs-news))
    ("<help> C-o" (nil describe-distribution))
    ("<help> C-p" (describe-gnu-project view-emacs-problems))
    ("<help> C-t" (view-emacs-todo))
    ("<help> C-w" (describe-no-warranty))
    ("<help> F" (Info-goto-emacs-command-node))
    ("<help> I" (describe-input-method))
    ("<help> K" (Info-goto-emacs-key-command-node))
    ("<help> L" (describe-language-environment))
    ("<help> P" (nil describe-package))
    ("<help> RET" (view-order-manuals))
    ("<help> S" (info-lookup-symbol))
    ("<help> a" (apropos-command))
    ("<help> b" (describe-bindings))
    ("<help> c" (describe-key-briefly))
    ("<help> d" (apropos-documentation))
    ("<help> e" (view-echo-area-messages))
    ("<help> f" (describe-function))
    ("<help> g" (nil describe-gnu-project))
    ("<help> h" (view-hello-file))
    ("<help> i" (info))
    ("<help> k" (describe-key))
    ("<help> l" (view-lossage))
    ("<help> m" (describe-mode))
    ("<help> n" (view-emacs-news))
    ("<help> p" (finder-by-keyword))
    ("<help> q" (help-quit))
    ("<help> r" (info-emacs-manual))
    ("<help> s" (describe-syntax))
    ("<help> t" (help-with-tutorial))
    ("<help> v" (describe-variable))
    ("<help> w" (where-is))
    ("<help>" (prefix))
    ("<home>" (beginning-of-buffer move-beginning-of-line))
    ("<iconify-frame>" (ignore-event))
    ("<insert>" (overwrite-mode))
    ("<insertchar>" (overwrite-mode))
    ("<insertline>" (open-line))
    ("<left>" (left-char backward-char))
    ("<make-frame-visible>" (ignore-event))
    ("<menu>" (execute-extended-command))
    ("<mode-line>" (prefix))
    ("<next>" (scroll-up-command scroll-up))
    ("<open>" (find-file))
    ("<prior>" (scroll-down scroll-down-command))
    ("<redo>" (repeat-complex-command))
    ("<right>" (right-char forward-char))
    ("<select-window>" (handle-select-window))
    ("<switch-frame>" (handle-switch-frame))
    ("<undo>" (undo))
    ("<up>" (previous-line))
    ("C--" (negative-argument))
    ("C-/" (undo))
    ("C-0" (digit-argument))
    ("C-1" (digit-argument))
    ("C-2" (digit-argument))
    ("C-3" (digit-argument))
    ("C-4" (digit-argument))
    ("C-5" (digit-argument))
    ("C-6" (digit-argument))
    ("C-7" (digit-argument))
    ("C-8" (digit-argument))
    ("C-9" (digit-argument))
    ("C-@" (set-mark-command))
    ("C-M-%" (query-replace-regexp))
    ("C-M--" (negative-argument))
    ("C-M-." (find-tag-regexp))
    ("C-M-/" (dabbrev-completion))
    ("C-M-0" (digit-argument))
    ("C-M-1" (digit-argument))
    ("C-M-2" (digit-argument))
    ("C-M-3" (digit-argument))
    ("C-M-4" (digit-argument))
    ("C-M-5" (digit-argument))
    ("C-M-6" (digit-argument))
    ("C-M-7" (digit-argument))
    ("C-M-8" (digit-argument))
    ("C-M-9" (digit-argument))
    ("C-M-@" (mark-sexp))
    ("C-M-S-v" (scroll-other-window-down))
    ("C-M-SPC" (mark-sexp))
    ("C-M-\\" (indent-region))
    ("C-M-a" (beginning-of-defun))
    ("C-M-b" (backward-sexp))
    ("C-M-c" (exit-recursive-edit))
    ("C-M-d" (down-list))
    ("C-M-e" (end-of-defun))
    ("C-M-f" (forward-sexp))
    ("C-M-h" (mark-defun))
    ("C-M-i" (complete-symbol))
    ("C-M-j" (indent-new-comment-line))
    ("C-M-k" (kill-sexp))
    ("C-M-l" (reposition-window))
    ("C-M-n" (forward-list))
    ("C-M-o" (split-line))
    ("C-M-p" (backward-list))
    ("C-M-r" (isearch-backward-regexp))
    ("C-M-s" (isearch-forward-regexp))
    ("C-M-t" (transpose-sexps))
    ("C-M-u" (backward-up-list))
    ("C-M-v" (scroll-other-window))
    ("C-M-w" (append-next-kill))
    ("C-SPC" (set-mark-command))
    ("C-\\" (toggle-input-method))
    ("C-]" (abort-recursive-edit))
    ("C-_" (undo))
    ("C-a" (move-beginning-of-line))
    ("C-b" (backward-char))
    ("C-c" (prefix))
    ("C-d" (delete-char))
    ("C-e" (move-end-of-line))
    ("C-f" (forward-char))
    ("C-g" (keyboard-quit))
    ("C-h ." (display-local-help))
    ("C-h 4 i" (info-other-window))
    ("C-h 4" (prefix))
    ("C-h <f1>" (help-for-help))
    ("C-h <help>" (help-for-help))
    ("C-h ?" (help-for-help))
    ("C-h C" (describe-coding-system))
    ("C-h C-\\" (describe-input-method))
    ("C-h C-a" (about-emacs))
    ("C-h C-c" (describe-copying))
    ("C-h C-d" (view-emacs-debugging describe-distribution))
    ("C-h C-e" (view-external-packages view-emacs-problems))
    ("C-h C-f" (view-emacs-FAQ))
    ("C-h C-h" (help-for-help))
    ("C-h C-n" (view-emacs-news))
    ("C-h C-o" (describe-distribution nil))
    ("C-h C-p" (view-emacs-problems describe-gnu-project))
    ("C-h C-t" (view-emacs-todo))
    ("C-h C-w" (describe-no-warranty))
    ("C-h F" (Info-goto-emacs-command-node))
    ("C-h I" (describe-input-method))
    ("C-h K" (Info-goto-emacs-key-command-node))
    ("C-h L" (describe-language-environment))
    ("C-h P" (describe-package nil))
    ("C-h RET" (view-order-manuals))
    ("C-h S" (info-lookup-symbol))
    ("C-h a" (apropos-command))
    ("C-h b" (describe-bindings))
    ("C-h c" (describe-key-briefly))
    ("C-h d" (apropos-documentation))
    ("C-h e" (view-echo-area-messages))
    ("C-h f" (describe-function))
    ("C-h g" (describe-gnu-project nil))
    ("C-h h" (view-hello-file))
    ("C-h i" (info))
    ("C-h k" (describe-key))
    ("C-h l" (view-lossage))
    ("C-h m" (describe-mode))
    ("C-h n" (view-emacs-news))
    ("C-h p" (finder-by-keyword))
    ("C-h q" (help-quit))
    ("C-h r" (info-emacs-manual))
    ("C-h s" (describe-syntax))
    ("C-h t" (help-with-tutorial))
    ("C-h v" (describe-variable))
    ("C-h w" (where-is))
    ("C-h" (prefix))
    ("C-j" (newline-and-indent))
    ("C-k" (kill-line))
    ("C-l" (recenter recenter-top-bottom))
    ("C-n" (next-line))
    ("C-o" (open-line))
    ("C-p" (previous-line))
    ("C-q" (quoted-insert))
    ("C-r" (isearch-backward))
    ("C-s" (isearch-forward))
    ("C-t" (transpose-chars))
    ("C-u" (universal-argument))
    ("C-v" (scroll-up-command scroll-up))
    ("C-w" (kill-region))
    ("C-x $" (set-selective-display))
    ("C-x '" (expand-abbrev))
    ("C-x (" (kmacro-start-macro))
    ("C-x )" (kmacro-end-macro))
    ("C-x *" (calc-dispatch))
    ("C-x +" (balance-windows))
    ("C-x -" (shrink-window-if-larger-than-buffer))
    ("C-x ." (set-fill-prefix))
    ("C-x 0" (delete-window))
    ("C-x 1" (delete-other-windows))
    ("C-x 2" (split-window-below split-window-vertically))
    ("C-x 3" (split-window-right split-window-horizontally))
    ("C-x 4 ." (find-tag-other-window))
    ("C-x 4 0" (kill-buffer-and-window))
    ("C-x 4 C-f" (find-file-other-window))
    ("C-x 4 C-o" (display-buffer))
    ("C-x 4 a" (add-change-log-entry-other-window))
    ("C-x 4 b" (switch-to-buffer-other-window))
    ("C-x 4 c" (clone-indirect-buffer-other-window))
    ("C-x 4 d" (dired-other-window))
    ("C-x 4 f" (find-file-other-window))
    ("C-x 4 m" (compose-mail-other-window))
    ("C-x 4 r" (find-file-read-only-other-window))
    ("C-x 4" (prefix))
    ("C-x 5 ." (find-tag-other-frame))
    ("C-x 5 0" (delete-frame))
    ("C-x 5 1" (delete-other-frames))
    ("C-x 5 2" (make-frame-command))
    ("C-x 5 C-f" (find-file-other-frame))
    ("C-x 5 C-o" (display-buffer-other-frame))
    ("C-x 5 b" (switch-to-buffer-other-frame))
    ("C-x 5 d" (dired-other-frame))
    ("C-x 5 f" (find-file-other-frame))
    ("C-x 5 m" (compose-mail-other-frame))
    ("C-x 5 o" (other-frame))
    ("C-x 5 r" (find-file-read-only-other-frame))
    ("C-x 5" (prefix))
    ("C-x 6 2" (2C-two-columns))
    ("C-x 6 <f2>" (2C-two-columns))
    ("C-x 6 b" (2C-associate-buffer))
    ("C-x 6 s" (2C-split))
    ("C-x 6" (prefix))
    ("C-x 8 RET" (insert-char 2 ucs-insert))
    ("C-x 8" (nil prefix))
    ("C-x ;" (comment-set-column))
    ("C-x <" (scroll-left))
    ("C-x <C-left>" (previous-buffer))
    ("C-x <C-right>" (next-buffer))
    ("C-x <left>" (previous-buffer))
    ("C-x <right>" (next-buffer))
    ("C-x =" (what-cursor-position))
    ("C-x >" (scroll-right))
    ("C-x C-+" (text-scale-adjust nil))
    ("C-x C--" (text-scale-adjust nil))
    ("C-x C-0" (text-scale-adjust nil))
    ("C-x C-=" (text-scale-adjust nil))
    ("C-x C-@" (pop-global-mark))
    ("C-x C-SPC" (pop-global-mark))
    ("C-x C-b" (list-buffers))
    ("C-x C-c" (save-buffers-kill-terminal save-buffers-kill-emacs))
    ("C-x C-d" (list-directory))
    ("C-x C-e" (eval-last-sexp))
    ("C-x C-f" (find-file))
    ("C-x C-k C-a" (kmacro-add-counter))
    ("C-x C-k C-c" (kmacro-set-counter))
    ("C-x C-k C-d" (kmacro-delete-ring-head))
    ("C-x C-k C-e" (kmacro-edit-macro-repeat))
    ("C-x C-k C-f" (kmacro-set-format))
    ("C-x C-k C-k" (kmacro-end-or-call-macro-repeat))
    ("C-x C-k C-l" (kmacro-call-ring-2nd-repeat))
    ("C-x C-k C-n" (kmacro-cycle-ring-next))
    ("C-x C-k C-p" (kmacro-cycle-ring-previous))
    ("C-x C-k C-s" (kmacro-start-macro))
    ("C-x C-k C-t" (kmacro-swap-ring))
    ("C-x C-k C-v" (kmacro-view-macro-repeat))
    ("C-x C-k RET" (kmacro-edit-macro))
    ("C-x C-k SPC" (kmacro-step-edit-macro))
    ("C-x C-k TAB" (kmacro-insert-counter))
    ("C-x C-k b" (kmacro-bind-to-key))
    ("C-x C-k e" (edit-kbd-macro))
    ("C-x C-k l" (kmacro-edit-lossage))
    ("C-x C-k n" (kmacro-name-last-macro))
    ("C-x C-k q" (kbd-macro-query))
    ("C-x C-k r" (apply-macro-to-region-lines))
    ("C-x C-k s" (kmacro-start-macro))
    ("C-x C-k" (prefix))
    ("C-x C-l" (downcase-region))
    ("C-x C-n" (set-goal-column))
    ("C-x C-o" (delete-blank-lines))
    ("C-x C-p" (mark-page))
    ("C-x C-q" (read-only-mode toggle-read-only))
    ("C-x C-r" (find-file-read-only))
    ("C-x C-s" (save-buffer))
    ("C-x C-t" (transpose-lines))
    ("C-x C-u" (upcase-region))
    ("C-x C-v" (find-alternate-file))
    ("C-x C-w" (write-file))
    ("C-x C-x" (exchange-point-and-mark))
    ("C-x C-z" (suspend-frame iconify-or-deiconify-frame))
    ("C-x DEL" (backward-kill-sentence))
    ("C-x ESC ESC" (repeat-complex-command))
    ("C-x ESC" (prefix))
    ("C-x M-:" (repeat-complex-command))
    ("C-x RET C-\\" (set-input-method))
    ("C-x RET F" (set-file-name-coding-system))
    ("C-x RET X" (set-next-selection-coding-system))
    ("C-x RET c" (universal-coding-system-argument))
    ("C-x RET f" (set-buffer-file-coding-system))
    ("C-x RET k" (set-keyboard-coding-system))
    ("C-x RET l" (set-language-environment))
    ("C-x RET p" (set-buffer-process-coding-system))
    ("C-x RET r" (revert-buffer-with-coding-system))
    ("C-x RET t" (set-terminal-coding-system))
    ("C-x RET x" (set-selection-coding-system))
    ("C-x RET" (prefix))
    ("C-x TAB" (indent-rigidly))
    ("C-x [" (backward-page))
    ("C-x ]" (forward-page))
    ("C-x ^" (enlarge-window))
    ("C-x `" (next-error))
    ("C-x a '" (expand-abbrev))
    ("C-x a +" (add-mode-abbrev))
    ("C-x a -" (inverse-add-global-abbrev))
    ("C-x a C-a" (add-mode-abbrev))
    ("C-x a e" (expand-abbrev))
    ("C-x a g" (add-global-abbrev))
    ("C-x a i g" (inverse-add-global-abbrev))
    ("C-x a i l" (inverse-add-mode-abbrev))
    ("C-x a i" (prefix))
    ("C-x a l" (add-mode-abbrev))
    ("C-x a n" (expand-jump-to-next-slot))
    ("C-x a p" (expand-jump-to-previous-slot))
    ("C-x a" (prefix))
    ("C-x b" (switch-to-buffer))
    ("C-x d" (dired))
    ("C-x e" (kmacro-end-and-call-macro))
    ("C-x f" (set-fill-column))
    ("C-x h" (mark-whole-buffer))
    ("C-x i" (insert-file))
    ("C-x k" (kill-buffer))
    ("C-x l" (count-lines-page))
    ("C-x m" (compose-mail))
    ("C-x n b" (nil org-narrow-to-block))
    ("C-x n d" (narrow-to-defun))
    ("C-x n n" (narrow-to-region))
    ("C-x n p" (narrow-to-page))
    ("C-x n s" (nil org-narrow-to-subtree))
    ("C-x n w" (widen))
    ("C-x n" (prefix))
    ("C-x o" (other-window))
    ("C-x q" (kbd-macro-query))
    ("C-x r +" (increment-register))
    ("C-x r C-@" (point-to-register))
    ("C-x r C-SPC" (point-to-register))
    ("C-x r N" (nil rectangle-number-lines))
    ("C-x r SPC" (point-to-register))
    ("C-x r b" (bookmark-jump))
    ("C-x r c" (clear-rectangle))
    ("C-x r d" (delete-rectangle))
    ("C-x r f" (frame-configuration-to-register))
    ("C-x r g" (insert-register))
    ("C-x r i" (insert-register))
    ("C-x r j" (jump-to-register))
    ("C-x r k" (kill-rectangle))
    ("C-x r l" (bookmark-bmenu-list))
    ("C-x r m" (bookmark-set))
    ("C-x r n" (number-to-register))
    ("C-x r o" (open-rectangle))
    ("C-x r r" (copy-rectangle-to-register))
    ("C-x r s" (copy-to-register))
    ("C-x r t" (string-rectangle))
    ("C-x r w" (window-configuration-to-register))
    ("C-x r x" (copy-to-register))
    ("C-x r y" (yank-rectangle))
    ("C-x r" (prefix))
    ("C-x s" (save-some-buffers))
    ("C-x u" (undo advertised-undo))
    ("C-x v +" (nil vc-update))
    ("C-x v =" (vc-diff))
    ("C-x v D" (vc-directory vc-directory nil vc-root-diff))
    ("C-x v I" (nil vc-log-incoming))
    ("C-x v L" (nil vc-print-root-log))
    ("C-x v O" (nil vc-log-outgoing))
    ("C-x v a" (vc-update-change-log))
    ("C-x v b" (vc-switch-backend))
    ("C-x v c" (vc-cancel-version vc-rollback))
    ("C-x v d" (vc-dir))
    ("C-x v g" (vc-annotate))
    ("C-x v h" (vc-insert-headers))
    ("C-x v i" (vc-register))
    ("C-x v l" (vc-print-log))
    ("C-x v m" (vc-merge))
    ("C-x v r" (vc-retrieve-snapshot vc-retrieve-tag))
    ("C-x v s" (vc-create-snapshot vc-create-tag))
    ("C-x v u" (vc-revert-buffer vc-revert))
    ("C-x v v" (vc-next-action))
    ("C-x v ~" (vc-version-other-window vc-revision-other-window))
    ("C-x v" (prefix))
    ("C-x z" (repeat))
    ("C-x {" (shrink-window-horizontally))
    ("C-x }" (enlarge-window-horizontally))
    ("C-x" (prefix))
    ("C-x" (prefix))
    ("C-y" (yank))
    ("C-z" (suspend-frame iconify-or-deiconify-frame))
    ("ESC <C-backspace>" (backward-kill-sexp))
    ("ESC <C-delete>" (backward-kill-sexp))
    ("ESC <C-down>" (down-list))
    ("ESC <C-end>" (end-of-defun))
    ("ESC <C-home>" (beginning-of-defun))
    ("ESC <C-left>" (backward-sexp))
    ("ESC <C-right>" (forward-sexp))
    ("ESC <C-up>" (backward-up-list))
    ("ESC <begin>" (beginning-of-buffer-other-window))
    ("ESC <end>" (end-of-buffer-other-window))
    ("ESC <home>" (beginning-of-buffer-other-window))
    ("ESC <left>" (backward-word))
    ("ESC <next>" (scroll-other-window))
    ("ESC <prior>" (scroll-other-window-down))
    ("ESC <right>" (forward-word))
    ("ESC ESC" (prefix))
    ("ESC" (prefix))
    ("M-!" (shell-command))
    ("M-$" (ispell-word))
    ("M-%" (query-replace))
    ("M-&" (async-shell-command nil))
    ("M-'" (abbrev-prefix-mark))
    ("M-(" (insert-parentheses))
    ("M-)" (move-past-close-and-reindent))
    ("M-*" (pop-tag-mark))
    ("M-," (tags-loop-continue))
    ("M--" (negative-argument))
    ("M-." (find-tag))
    ("M-/" (dabbrev-expand))
    ("M-0" (digit-argument))
    ("M-1" (digit-argument))
    ("M-2" (digit-argument))
    ("M-3" (digit-argument))
    ("M-4" (digit-argument))
    ("M-5" (digit-argument))
    ("M-6" (digit-argument))
    ("M-7" (digit-argument))
    ("M-8" (digit-argument))
    ("M-9" (digit-argument))
    ("M-:" (eval-expression))
    ("M-;" (comment-dwim))
    ("M-<" (beginning-of-buffer))
    ("M-=" (count-words-region count-lines-region))
    ("M->" (end-of-buffer))
    ("M-@" (mark-word))
    ("M-DEL" (backward-kill-word))
    ("M-ESC :" (eval-expression))
    ("M-ESC ESC" (keyboard-escape-quit))
    ("M-SPC" (just-one-space))
    ("M-\\" (delete-horizontal-space))
    ("M-^" (delete-indentation))
    ("M-`" (tmm-menubar))
    ("M-a" (backward-sentence))
    ("M-b" (backward-word))
    ("M-c" (capitalize-word))
    ("M-d" (kill-word))
    ("M-e" (forward-sentence))
    ("M-f" (forward-word))
    ("M-g ESC" (prefix))
    ("M-g M-g" (goto-line))
    ("M-g M-n" (next-error))
    ("M-g M-p" (previous-error))
    ("M-g g" (goto-line))
    ("M-g n" (next-error))
    ("M-g p" (previous-error))
    ("M-g" (prefix))
    ("M-h" (mark-paragraph))
    ("M-i" (tab-to-tab-stop))
    ("M-j" (indent-new-comment-line))
    ("M-k" (kill-sentence))
    ("M-l" (downcase-word))
    ("M-m" (back-to-indentation))
    ("M-o ESC" (prefix 1))
    ("M-o M-S" (center-paragraph))
    ("M-o M-o" (font-lock-fontify-block))
    ("M-o M-s" (center-line))
    ("M-o b" (facemenu-set-bold 1))
    ("M-o d" (facemenu-set-default 1))
    ("M-o i" (facemenu-set-italic 1))
    ("M-o l" (facemenu-set-bold-italic 1))
    ("M-o o" (facemenu-set-face 1))
    ("M-o u" (facemenu-set-underline 1))
    ("M-o" (prefix))
    ("M-q" (fill-paragraph))
    ("M-r" (move-to-window-line-top-bottom move-to-window-line))
    ("M-s h f" (hi-lock-find-patterns))
    ("M-s h l" (highlight-lines-matching-regexp))
    ("M-s h p" (highlight-phrase))
    ("M-s h r" (highlight-regexp))
    ("M-s h u" (unhighlight-regexp))
    ("M-s h w" (hi-lock-write-interactive-patterns))
    ("M-s h" (prefix 1))
    ("M-s o" (occur 1))
    ("M-s w" (isearch-forward-word 1))
    ("M-s" (prefix nil))
    ("M-t" (transpose-words))
    ("M-u" (upcase-word))
    ("M-v" (scroll-down-command scroll-down))
    ("M-w" (kill-ring-save))
    ("M-x" (execute-extended-command))
    ("M-y" (yank-pop))
    ("M-z" (zap-to-char))
    ("M-{" (backward-paragraph))
    ("M-|" (shell-command-on-region))
    ("M-}" (forward-paragraph))
    ("M-~" (not-modified))
    ("RET" (newline)))
  "Default Emacs Key Bindings.")

(defvar ergoemacs-single-command-keys)
(defvar ergoemacs-shortcut-keymap)
(defvar ergoemacs-no-shortcut-keymap)
(defvar ergoemacs-keymap)
(defvar keyfreq-mode)
(defvar keyfreq-table)
(defvar ergoemacs-describe-key)
(declare-function ergoemacs-debug "ergoemacs-mode.el")
(declare-function ergoemacs-real-key-binding "ergoemacs-advices.el" (key &optional accept-default no-remap position) t)
(defun ergoemacs-undefined ()
  "Ergoemacs Undefined key, echo new key for old action."
  (interactive)
  (let* ((key-kbd (or ergoemacs-single-command-keys (this-single-command-keys)))
         tmp
         (local-fn nil))
    ;; Lookup local key, if present and then issue that
    ;; command instead...
    ;;
    ;; This way the unbound keys are just above the global
    ;; map and doesn't actually change it.
    (cond
     ((progn
        ;; See if this is present in the `ergoemacs-shortcut-keymap'
        (setq local-fn (lookup-key ergoemacs-shortcut-keymap key-kbd))
        (unless (functionp local-fn)
          ;; Lookup in ergoemacs-keymap
          (setq local-fn (lookup-key ergoemacs-keymap key-kbd)))
        (functionp local-fn))
      (ergoemacs-debug "WARNING: The command %s is undefined when if shouldn't be..." local-fn)
      (setq tmp (ergoemacs-real-key-binding key-kbd))
      (when (and tmp (not (equal tmp 'ergoemacs-undefined)))
        (setq local-fn tmp))
      (when (featurep 'keyfreq)
        (when keyfreq-mode
          (let ((command 'ergoemacs-undefined) count)
            (setq count (gethash (cons major-mode command) keyfreq-table))
            (cond
             ((not count))
             ((= count 1)
              (remhash (cons major-mode command) keyfreq-table))
             (count
              (puthash (cons major-mode command) (- count 1)
                       keyfreq-table)))
            ;; Add local-fn to counter.
            (setq command local-fn)
            (setq count (gethash (cons major-mode command) keyfreq-table))
            (puthash (cons major-mode command) (if count (+ count 1) 1)
                     keyfreq-table))))
      (setq this-command local-fn)
      (condition-case err
          (call-interactively local-fn)
        (error (beep) (message "%s" err))))
     ((progn
        ;; Local map present.  Use it, if there is a key
        ;; defined there.
        (setq local-fn (get-char-property (point) 'local-map))
        (if (and local-fn
                 (ignore-errors (keymapp local-fn)))
            (setq local-fn (lookup-key local-fn key-kbd))
          (if (current-local-map)
              (setq local-fn (lookup-key (current-local-map) key-kbd))
            (setq local-fn nil)))
        (functionp local-fn))
      (setq this-command local-fn) ; Don't record this command.
      (when (featurep 'keyfreq)
        (when keyfreq-mode
          (let ((command 'ergoemacs-undefined) count)
            (setq count (gethash (cons major-mode command) keyfreq-table))
            (cond
             ((not count))
             ((= count 1)
              (remhash (cons major-mode command) keyfreq-table))
             (count
              (puthash (cons major-mode command) (- count 1)
                       keyfreq-table)))
            ;; Add local-fn to counter.
            (setq command local-fn)
            (setq count (gethash (cons major-mode command) keyfreq-table))
            (puthash (cons major-mode command) (if count (+ count 1) 1)
                     keyfreq-table))))
      (condition-case err
          (call-interactively local-fn)
        (error (beep) (message "%s" err))))
     (t
      ;; Not locally defined, complain.
      (beep)
      (ergoemacs-where-is-old-binding key-kbd))))
  (setq ergoemacs-describe-key nil))


(defvar ergoemacs-where-is-global-hash (make-hash-table :test 'equal)
  "Hash for ergoemacs lookup of global functions.")

(defun ergoemacs-reset-global-where-is ()
  "Reset `ergoemacs-where-is-global-hash'."
  (setq ergoemacs-where-is-global-hash (make-hash-table :test 'equal))
  (dolist (x ergoemacs-emacs-default-bindings)
    (let ((key (read-kbd-macro (nth 0 x))))
      (dolist (fn (nth 1 x))
        (let ((keys (gethash fn ergoemacs-where-is-global-hash)))
          (pushnew key keys :test 'equal)
          (puthash fn keys ergoemacs-where-is-global-hash))))))


;;;###autoload
(defun ergoemacs-ignore-prev-global ()
  "Ignore previously defined global keys."
  (setq ergoemacs-emacs-default-bindings
        (mapcar
         (lambda(elt)
           (let ((first (nth 0  elt))
                 (last (nth 1 elt))
                 fn)
             (setq fn (lookup-key global-map (read-kbd-macro first)))
             (if (not (functionp fn))
                 elt
               (pushnew fn last :test 'equal)
               `(,first ,last))))
         ergoemacs-emacs-default-bindings))
  (ergoemacs-reset-global-where-is))

(defun ergoemacs-translate-current-key (key)
  "Translate the current KEY."
  (cond
   ((string= (key-description key) "<backspace>")
    (read-kbd-macro "DEL" t))
   (t key)))

(defun ergoemacs-translate-current-function (curr-fn)
  "Translate the current function CURR-FN."
  (cond
   ((eq 'delete-horizontal-space curr-fn)
    'ergoemacs-shrink-whitespaces)
   ((eq 'left-char curr-fn) 'backward-char)
   ((eq 'right-char curr-fn) 'forward-char)
   (t curr-fn)))

(defvar ergoemacs-global-not-changed-cache '()
  "Cache of global variables that have not changed")

(defvar ergoemacs-global-changed-cache '()
  "Cache of global variables that have changed.")

(defvar ergoemacs-dir)
(defun ergoemacs-global-fix-defualt-bindings (kbd-code function)
  "Helper function to fix `ergoemacs-emacs-default-bindings' based on currently running emacs."
  (interactive)
  (with-temp-buffer
    (insert-file-contents (expand-file-name "ergoemacs-unbind.el" ergoemacs-dir))
    (goto-char (point-min))
    (when (search-forward (format "(\"%s\"" kbd-code))
      (skip-chars-forward " (")
      (insert (format "%s " function)))
    (write-file (expand-file-name "ergoemacs-unbind.el" ergoemacs-dir))))

(declare-function ergoemacs-kbd "ergoemacs-translate.el")
(defun ergoemacs-global-changed-p (key &optional is-variable complain fix)
  "Returns if a global key has been changed.  If IS-VARIABLE is
true and KEY is a string, then lookup the keyboard equivalent
based on current layout.

If COMPLAIN is true, complain about keys that have changed.

If FIX is true, open ergoemacs-unbind and fix the variables.
This should only be run when no global keys have been set.
"
  (let* ((key-code
          (cond
           ((eq (type-of key) 'string)
            (if is-variable
                (ergoemacs-kbd key)
              (or (ignore-errors (read-kbd-macro key))
                  (read-kbd-macro
                   (encode-coding-string
                    key locale-coding-system)))))
           (t key)))
         (key-kbd (key-description key-code)))
    (if (string-match "\\(mouse\\|wheel\\)" key-kbd)
        nil
      (if (member key-kbd ergoemacs-global-changed-cache)
          (progn
            (when (or fix complain)
              (let* ((key-function (lookup-key (current-global-map) key-code t))
                     (old-bindings (assoc key-kbd ergoemacs-emacs-default-bindings))
                     (trans-function (if (ignore-errors (keymapp key-function))
                                         'prefix
                                       key-function)))
                (message "Warning %s has been set globally. It is bound to %s not in %s." key-kbd
                         trans-function old-bindings)))
            t)
        (if (member key-kbd ergoemacs-global-not-changed-cache)
            nil
          (let* ((key-function (lookup-key (current-global-map) key-code t))
                 (old-bindings (assoc key-kbd ergoemacs-emacs-default-bindings))
                 (trans-function (if (ignore-errors (keymapp key-function))
                                     'prefix
                                   key-function))
                 (has-changed nil))
            (cond
             ((not trans-function)) ; trans-function is undefined.
                                        ; Assume not globally changed.
             ((integerp trans-function) ; Over defined. See if previous key is globally-changed.
              (let ((key-as-vector (read-kbd-macro (key-description key-code) t))
                    (prefix-vector (make-vector trans-function nil))
                    (i 0))
                (while (< i trans-function)
                  (aset prefix-vector i (elt key-as-vector i))
                  (setq i (+ 1 i)))
                ;; If it is a prefix vector, assume not globally
                ;; changed
                (unless (ignore-errors (keymapp (lookup-key (current-global-map) prefix-vector)))
                  ;; Not a prefix, see if the key had actually changed
                  ;; by recursively calling `ergoemacs-global-changed-p'
                  (setq has-changed
                        (ergoemacs-global-changed-p
                         prefix-vector is-variable complain fix)))))
             (old-bindings ; Trans function is defined, not an integer
              (unless (member trans-function (nth 1 old-bindings))
                (setq has-changed t)))
             (t
              (setq has-changed t)) ; Not found in old bindings, but bound globally
             )
            (if has-changed
                (progn
                  (when (or fix complain)
                    (message "Warning %s has been set globally. It is bound to %s not in %s." key-kbd
                             trans-function old-bindings)
                    (when fix
                      (unless (integerp trans-function)
                        (ergoemacs-global-fix-defualt-bindings key-kbd trans-function))))
                  (pushnew key-kbd ergoemacs-global-changed-cache :test 'equal))
              (pushnew key-kbd ergoemacs-global-not-changed-cache :test 'equal))
            has-changed))))))

(declare-function ergoemacs-get-fixed-layout "ergoemacs-translate.el")
(declare-function ergoemacs-get-variable-layout "ergoemacs-translate.el")
(defun ergoemacs-warn-globally-changed-keys (&optional fix)
  "Warns about globally changed keys. If FIX is true, fix the ergoemacs-unbind file."
  (interactive)
  (dolist (x ergoemacs-emacs-default-bindings)
    (ergoemacs-global-changed-p (nth 0 x) nil t fix))
  (message "Ergoemacs Keys warnings for this layout:")
  (dolist (x (symbol-value (ergoemacs-get-fixed-layout)))
    (and (eq 'string (type-of (nth 0 x)))
         (ergoemacs-global-changed-p (nth 0 x) nil t fix)))
  (dolist (x (symbol-value (ergoemacs-get-variable-layout)))
    (and (eq 'string (type-of (nth 0 x)))
         (ergoemacs-global-changed-p (nth 0 x) t t fix))))



;; Based on describe-key-briefly
(declare-function ergoemacs-key-fn-lookup "ergoemacs-translate.el")
(declare-function ergoemacs-pretty-key "ergoemacs-translate.el")
(defvar yank-menu)
(defun ergoemacs-where-is-old-binding (&optional key only-new-key)
  "Print the name of the function KEY invoked before to start ErgoEmacs minor mode."
  (interactive
   (let ((enable-disabled-menus-and-buttons t)
	 (cursor-in-echo-area t)
	 saved-yank-menu)
     (unwind-protect
	 (let (key)
	   ;; If yank-menu is empty, populate it temporarily, so that
	   ;; "Select and Paste" menu can generate a complete event.
	   (when (null (cdr yank-menu))
	     (setq saved-yank-menu (copy-sequence yank-menu))
	     (menu-bar-update-yank-menu "(any string)" nil))
	   (setq key (read-key-sequence "Describe old key (or click or menu item): "))
	   ;; If KEY is a down-event, read and discard the
	   ;; corresponding up-event.  Note that there are also
	   ;; down-events on scroll bars and mode lines: the actual
	   ;; event then is in the second element of the vector.
	   (and (vectorp key)
		(let ((last-idx (1- (length key))))
		  (and (eventp (aref key last-idx))
		       (memq 'down (event-modifiers (aref key last-idx)))))
		(read-event))
	   (list key))
       ;; Put yank-menu back as it was, if we changed it.
       (when saved-yank-menu
	 (setq yank-menu (copy-sequence saved-yank-menu))
	 (fset 'yank-menu (cons 'keymap yank-menu))))))
  
  (let* ((old-cmd (lookup-key (current-global-map) key))
         message-log-max
         (key-desc (key-description key))
         (new-key (key-description (ergoemacs-key-fn-lookup old-cmd))))
    (unless old-cmd
      (setq old-cmd (lookup-key
                     (current-global-map)
                     (ergoemacs-translate-current-key key)))
      (setq new-key (key-description (ergoemacs-key-fn-lookup old-cmd))))
    (cond
     ((and new-key only-new-key)
      (read-kbd-macro new-key t))
     (only-new-key
      nil)
     ((and old-cmd new-key)
      (message "%s keybinding%s%s (%s)"
               (ergoemacs-pretty-key key-desc)
               (if (called-interactively-p  'any)
                   " is changed to "
                 " is disabled! Use ")
               (ergoemacs-pretty-key new-key)
               old-cmd))
     (old-cmd
      (message "Key %s was bound to `%s' which is not bound any longer"
               (ergoemacs-pretty-key key-desc)
               old-cmd))
     (t
      (message "Key %s was not bound to any command (%s)"
               (ergoemacs-pretty-key key-desc)
               old-cmd)))))

(provide 'ergoemacs-unbind)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; ergoemacs-unbind.el ends here
;; Local Variables:
;; coding: utf-8-emacs
;; End:
