;;; eev-tlinks.el --- hyperlinks to temporary buffers generated by templates  -*- lexical-binding: nil; -*-

;; Copyright (C) 2013-2024 Free Software Foundation, Inc.
;;
;; This file is part of GNU eev.
;;
;; GNU eev is free software: you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.
;;
;; GNU eev is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs.  If not, see <http://www.gnu.org/licenses/>.
;;
;; Author:     Eduardo Ochs <eduardoochs@gmail.com>
;; Maintainer: Eduardo Ochs <eduardoochs@gmail.com>
;; Version:    20241202
;; Keywords:   e-scripts
;;
;; Latest version: <http://anggtwu.net/eev-current/eev-tlinks.el>
;;       htmlized: <http://anggtwu.net/eev-current/eev-tlinks.el.html>
;;       See also: <http://anggtwu.net/eev-current/eev-beginner.el.html>
;;                 <http://anggtwu.net/eev-intros/find-eev-intro.html>
;;                 <http://anggtwu.net/eev-intros/find-links-conv-intro.html>
;;                                               (find-eev-intro)
;;                                               (find-links-conv-intro)

;;; Commentary:

;; The functions defined here - `find-latex-links' and other ones -
;; produce buffers made of a series of hyperlinks followed by a
;; templated text. The templated text is usually preceded by a line
;; like `(ee-copy-rest NLINES CODE)'; see the explanation of
;; `ee-copy-rest' below to understand how this is used.
;;
;; The functions here are one complexity step above the functions in:
;;   (find-eev "eev-elinks.el")
;;
;; See:
;;   (find-links-conv-intro "3. Classification")
;;   (find-links-conv-intro "3. Classification" "c)")
;;   (find-links-conv-intro "3. Classification" "find-elinks")
;;   (find-links-conv-intro "3. Classification" "eev-tlinks.el")
;;
;; Many of the functions in this file are "five-minute hacks" whose
;; first versions were written in two steps. In the first step I ran
;; `M-x find-find-links-links-new', adjusted the arguments in the
;; first line of the temporary buffer and regenerated the buffer; this
;; gave me a "skeleton" for the real function. In the second step I
;; adjusted the arguments of the `find-elinks' and the string inside
;; the `ee-template0'. Each function here that was written in this way
;; has a "Skel:" line in the comments about its definition, and if you
;; run the sexp in the "Skel:" line you get its skeleton. There is a
;; video explaining this here:
;;
;;   (to "find-find-links-links-new")
;;
;; Some of the functions in this file were written with an ancestor of
;; `find-find-links-links-new' that generated uglier code; they need
;; to be rewritten. Also, some functions here are very old, and I
;; haven't used or tested them in ages; they deserve to be deleted.

;; Note that all the `find-*-links' functions in this file have very
;; small docstrings. This is intentional: when a function doesn't have
;; a docstring, or has a docstring that is insufficient, this always
;; means "look at the source", and the `find-*-links' functions do
;; very little besides generating a temporary buffer from a
;; template...
;;
;; So: the best way to understand what the `find-*-links' functions do
;; is by looking at their source code, and by executing the sexps in
;; comments - both the "See:"s and the "Test:"s.




;; «.ee-copy-rest»			(to "ee-copy-rest")
;; «.ee-copy-rest-3-intro»		(to "ee-copy-rest-3-intro")
;; «.ee-copy-rest-3-tests»		(to "ee-copy-rest-3-tests")
;; «.ee-copy-rest-3»			(to "ee-copy-rest-3")
;;
;; «.find-find-links-links»		(to "find-find-links-links")
;; «.find-find-links-links-new»		(to "find-find-links-links-new")
;;   «.ee-ffll-functions»		(to "ee-ffll-functions")
;;   «.find-let*-macro-links»		(to "find-let*-macro-links")
;;
;; «.find-intro-links»			(to "find-intro-links")
;; «.find-eev-header-links»		(to "find-eev-header-links")
;;
;; «.find-debpkg-links»			(to "find-debpkg-links")
;; «.find-pacman-links»			(to "find-pacman-links")
;; «.find-macports-links»		(to "find-macports-links")
;; «.find-homebrew-links»		(to "find-homebrew-links")
;; «.find-eev-reload-links»		(to "find-eev-reload-links")
;; «.find-eev-install-links»		(to "find-eev-install-links")
;; «.find-eev-update-links»		(to "find-eev-update-links")
;; «.find-youtubedl-links»		(to "find-youtubedl-links")
;;   «.ee-youtubedl-command»		(to "ee-youtubedl-command")
;;
;; «.find-psne-links»			(to "find-psne-links")
;;   «.ee-find-psne-echo-options»	(to "ee-find-psne-echo-options")
;; «.find-psne-eevvideo-links»		(to "find-psne-eevvideo-links")
;; «.find-psne-1stclassvideo-links»	(to "find-psne-1stclassvideo-links")
;; «.find-1stclassvideo-video»		(to "find-1stclassvideo-video")
;;
;; «.find-git-links»			(to "find-git-links")
;; «.find-fossil-links»			(to "find-fossil-links")
;; «.find-apt-get-source-links»		(to "find-apt-get-source-links")
;;
;; «.find-eevvideo-links»		(to "find-eevvideo-links")
;; «.find-psnevideo-links»		(to "find-psnevideo-links")
;;   «.ee-psne-if-needed»		(to "ee-psne-if-needed")
;; «.code-psnevideo»			(to "code-psnevideo")
;; «.code-eevvideo»			(to "code-eevvideo")
;; «.code-eevvideo-local»		(to "code-eevvideo-local")
;;   «.hardcoded-paths»			(to "hardcoded-paths")
;; «.find-eev-video-links»		(to "find-eev-video-links")
;; «.find-eevshortvideo-links»		(to "find-eevshortvideo-links")
;; «.find-wgeteevsubtitles-links»	(to "find-wgeteevsubtitles-links")
;;
;; «.find-latex-links»			(to "find-latex-links")
;; «.find-lua-links»			(to "find-lua-links")
;; «.find-escreenshot-links»		(to "find-escreenshot-links")
;; «.find-windows-eepitch-lua-links»	(to "find-windows-eepitch-lua-links")
;; «.find-extra-file-links»		(to "find-extra-file-links")
;; «.find-emacs-tangents-links»		(to "find-emacs-tangents-links")
;; «.find-eeit-links»			(to "find-eeit-links")
;; «.find-texlive-links»		(to "find-texlive-links")
;; «.find-newbrowser-links»		(to "find-newbrowser-links")
;; «.find-altbrowser-links»		(to "find-altbrowser-links")
;; «.find-newpdfviewer-links»		(to "find-newpdfviewer-links")
;; «.ee-0x0-upload-region»		(to "ee-0x0-upload-region")
;; «.find-0x0-links»			(to "find-0x0-links")
;; «.find-eepitch-bullet-links»		(to "find-eepitch-bullet-links")
;; «.find-red-star-links»		(to "find-red-star-links")
;;   «.ee-use-red-stars»		(to "ee-use-red-stars")
;;   «.ee-use-red-bullets»		(to "ee-use-red-bullets")
;; «.find-angg-es-links»		(to "find-angg-es-links")
;; «.find-angg-not-configured»		(to "find-angg-not-configured")
;; «.find-es-not-configured»		(to "find-es-not-configured")
;; «.find-1stclassvideo-links»		(to "find-1stclassvideo-links")
;;   «.ee-1stclassvideo-basicinfo»	(to "ee-1stclassvideo-basicinfo")
;;   «.ee-1stclassvideo-basicsexps»	(to "ee-1stclassvideo-basicsexps")
;;   «.ee-1stclassvideo-dlsubs»		(to "ee-1stclassvideo-dlsubs")
;;   «.ee-1stclassvideo-defuns»		(to "ee-1stclassvideo-defuns")
;;   «.code-1stclassvideo»		(to "code-1stclassvideo")
;;   «.code-1stclassvideos»		(to "code-1stclassvideos")
;; «.find-1stclassvideoindex»		(to "find-1stclassvideoindex")
;; «.find-1stclassvideohsubs»		(to "find-1stclassvideohsubs")
;; «.find-1stclassvideolsubs»		(to "find-1stclassvideolsubs")
;; «.find-1stclassvideodef»		(to "find-1stclassvideodef")
;; «.find-1stclassvideos»		(to "find-1stclassvideos")
;;   «.1c»				(to "1c")
;;   «.aliases»				(to "aliases")
;; «.find-dot-emacs-links»		(to "find-dot-emacs-links")
;; «.find-advicebefore-links»		(to "find-advicebefore-links")
;; «.find-osm-links»			(to "find-osm-links")
;; «.find-pip3-links»			(to "find-pip3-links")
;; «.find-yttranscript-links»		(to "find-yttranscript-links")
;; «.find-yttranscript0-links»		(to "find-yttranscript0-links")
;; «.find-importlib-links»		(to "find-importlib-links")
;; «.find-pypi-links»			(to "find-pypi-links")
;; «.find-nov-links»			(to "find-nov-links")
;; «.find-telegachat-links»		(to "find-telegachat-links")
;; «.find-eejump-links»			(to "find-eejump-links")
;; «.find-kla-links»			(to "find-kla-links")
;; «.find-rstdoc-links»			(to "find-rstdoc-links")
;; «.find-mpv-links»			(to "find-mpv-links")
;; «.find-try-sly-links»		(to "find-try-sly-links")
;; «.find-wgetrecursive-links»		(to "find-wgetrecursive-links")
;; «.find-mbe-links»			(to "find-mbe-links")
;; «.find-melpa-links»			(to "find-melpa-links")
;; «.find-emacsclient-links»		(to "find-emacsclient-links")
;; «.find-show2-links»			(to "find-show2-links")
;;   «.show2»				(to "show2")
;; «.show2-use»				(to "show2-use")
;;   «.find-luatb»			(to "find-luatb")
;; «.code-brappend»			(to "code-brappend")
;; «.find-maximamsg-links»		(to "find-maximamsg-links")
;; «.find-maximamsg»			(to "find-maximamsg")
;; «.find-qdraw-links»			(to "find-qdraw-links")
;; «.find-quicklisp-links»		(to "find-quicklisp-links")
;; «.find-sbcl-links»			(to "find-sbcl-links")
;; «.find-linki-links»			(to "find-linki-links")
;; «.find-gitdoc-links»			(to "find-gitdoc-links")
;; «.find-luainit-links»		(to "find-luainit-links")
;; «.find-luaso-links»			(to "find-luaso-links")
;; «.find-subed-mpv-links»		(to "find-subed-mpv-links")
;; «.find-debootstrap0-links»		(to "find-debootstrap0-links")
;;   «.ee-debootstrap0»			(to "ee-debootstrap0")
;; «.find-debootstrap1-links»		(to "find-debootstrap1-links")
;;   «.ee-debootstrap1»			(to "ee-debootstrap1")
;; «.find-debootstrap2-links»		(to "find-debootstrap2-links")
;; «.find-package-vc-install-links»	(to "find-package-vc-install-links")
;; «.find-ethemes-links»		(to "find-ethemes-links")
;; «.find-tryit-links»			(to "find-tryit-links")


(require 'eev-env)
(require 'eev-wrap)    ; For: (find-eev "eev-wrap.el" "ee-template0") 
(require 'cl-lib)      ; For `cl-remove-if'
(require 'subr-x)      ; For `string-join'



;;;                                                            _   
;;;   ___  ___        ___ ___  _ __  _   _       _ __ ___  ___| |_ 
;;;  / _ \/ _ \_____ / __/ _ \| '_ \| | | |_____| '__/ _ \/ __| __|
;;; |  __/  __/_____| (_| (_) | |_) | |_| |_____| | |  __/\__ \ |_ 
;;;  \___|\___|      \___\___/| .__/ \__, |     |_|  \___||___/\__|
;;;                           |_|    |___/                         
;;
;; «ee-copy-rest» (to ".ee-copy-rest")
;; Some hyperlinks buffers - for example those generated by:
;;
;;   (find-latex-links)
;;   (find-eev-install-links)
;;
;; are composed of a series of hyperlinks, then a sexp like
;; `(ee-copy-rest NLINES CODE)', then some templated text. When we run
;; the `(ee-copy-rest ...)' the templated text is flashed for a
;; fraction of a second and copied to the kill ring, and the window is
;; split in two like this:
;;
;;    ____________________________
;;   |              |             |
;;   |  hyperlinks  |  target of  |
;;   |    buffer    |    CODE     |
;;   |              |             |
;;   |______________|_____________|
;;
;; For example,
;;
;;   (find-latex-links "/tmp/foo")
;;
;; generates this hyperlinks buffer:
;;
;;    _________________________________________________________________ 
;;   |# (find-latex-links "/tmp/foo")                                  |
;;   |# (find-latex-links "/tmp/teste1")                               |
;;   |# (find-latex-links "~/2019-CLG/teste2")                         |
;;   |# (find-eev-quick-intro "`find-latex-links'")                    |
;;   |# (ee-copy-rest 1 '(find-fline "/tmp/foo.tex"))                  |
;;   |                                                                 |
;;   |% (defun c () (interactive) (find-sh "pdflatex foo.tex"))        |
;;   |% (defun d () (interactive) (find-pdf-page "/tmp/foo.pdf"))      |
;;   |% (defun e () (interactive) (find-fline "/tmp/foo.tex"))         |
;;   |%                                                                |
;;   |\documentclass{article}                                          |
;;   |\begin{document}                                                 |
;;   |                                                                 |
;;   |\end{document}                                                   |
;;   |                                                                 |
;;   |                                                                 |
;;   |                                                                 |
;;   | -:**-  *Elisp hyperlinks*   All L1     (Fundamental)  ----------|
;;   |_________________________________________________________________|
;;
;; in which CODE is `(find-fline "/tmp/foo.tex")'. When we execute the
;; `ee-copy-rest' line all the material from "% (defun c..." downwards
;; - the "templated text" - is flashed, copied to the kill ring, the
;; file "/tmp/foo.tex" is opened in the right window, and the
;; following message is displayed in the echo area:
;;
;;   "(Copied 9 lines to the kill ring - use C-y to paste)"
;;
;; If the file "/tmp/foo.tex" is empty and we want to initialize it
;; with the templated string we only need to go to the right window
;; and type `C-y'; if it is non-empty its present contents will be
;; displayed and we can decide what to do.
;;
;; TO DO: update the decumentation here:
;;   (find-eev-quick-intro "7.5. `find-latex-links'")
;;   (find-links-intro "7. The rest of the buffer")

(defvar eeflash-copy '(highlight 0.5))

(defun ee-count-lines (str)
  "Count the number of lines in STR (which should be newline-terminated)."
  (length (replace-regexp-in-string "[^\n]" "" str)))

(defun ee-copy-rest00 (skip gotoend code)
  "An internal function used by `ee-copy-rest0'."
  (setq skip (or skip 0))
  (let* ((start (save-excursion (move-beginning-of-line (+ 2 skip)) (point)))
	 (end   (eval gotoend))
	 (str   (buffer-substring start end))
	 (len   (ee-count-lines str))
	 (msg   `(Copied ,len lines to the kill ring - use C-y to paste)))
    (eeflash+ start end eeflash-copy)
    (kill-new str)
    (eval code)
    msg))

(defun ee-search-forward-before (str)
  "An internal function used by `ee-copy-rest0'."
  (save-excursion
    (search-forward str)
    (search-backward str)
    (point)))

(defun ee-copy-rest0 (skip code)
  "Copy the rest of this buffer to the kill ring and execute CODE.
The rest of this buffer is everything from the beginning of the next line -
optionally moving down SKIP lines - to the end of the buffer.

If SKIP is a list like (2 \"--snip--snip--\") then use another
notion of \"the rest of this buffer\": it will be everything from
2 lines from here downwards to the first occurrence of the string
\"--snip--snip--\" after the point."
  (if (numberp skip)
      (ee-copy-rest00 skip '(point-max) code)
    (let* ((nlines  (nth 0 skip))
	   (str     (nth 1 skip))
	   (gotoend `(ee-search-forward-before ,str)))
      (ee-copy-rest00 nlines gotoend code))))

(defun ee-copy-rest (skip code)
  "Copy the rest of this buffer to the kill ring and execute CODE.
The rest of this buffer is everything from the beginning of the next line -
optionally moving down SKIP lines - to the end of the buffer.
See: (find-eev \"eev-tlinks.el\" \"ee-copy-rest\")

If SKIP is a list like (2 \"--snip--snip--\") then use another
notion of \"the rest of this buffer\": it will be everything from
2 lines from here downwards to the first occurrence of the string
\"--snip--snip--\" after the point."
  (ee-copy-rest0 skip `(find-2a nil ',code)))


;; Obsolete:
;;
;; (defun ee-copy-after-and (nlines code)
;;   "Copy into the kill ring everything from NLINES down on, and run CODE."
;;   (move-beginning-of-line (- nlines 1))
;;   (kill-new (buffer-substring (point) (point-max)))
;;   (eval code)
;;   (let ((n (ee-count-lines (ee-last-kill))))
;;     `(Copied ,n lines to the kill ring - use C-y to paste)))
;; 
;; (defun ee-copy-after-and-2b (nlines code)
;;   "Copy into the kill ring everything from NLINES down on, and run CODE.
;; The target of the hyperlink in CODE is opened in the right-side window."
;;   (ee-copy-after-and nlines `(find-2b nil ',code)))



;;;                                                            _       _____ 
;;;   ___  ___        ___ ___  _ __  _   _       _ __ ___  ___| |_    |___ / 
;;;  / _ \/ _ \_____ / __/ _ \| '_ \| | | |_____| '__/ _ \/ __| __|____ |_ \ 
;;; |  __/  __/_____| (_| (_) | |_) | |_| |_____| | |  __/\__ \ ||_____|__) |
;;;  \___|\___|      \___\___/| .__/ \__, |     |_|  \___||___/\__|   |____/ 
;;;                           |_|    |___/                                   
;;
;; «ee-copy-rest-3-intro»  (to ".ee-copy-rest-3-intro")
;; `ee-copy-rest-3' is a variant of `ee-copy-rest' that receives three
;; arguments: where the region starts, where the region ends, and
;; where it should be copied to. Its code is be much easier to
;; understand than the code of `ee-copy-rest', and it is very
;; flexible:
;;
;;   1a) "where the region starts" is usually a number of lines to skip,
;;   1b) "where the region ends" is usually a string to search for,
;;   1c) "where it should be copied to" is usually a sexp, like:
;;         (find-fline "/tmp/foo.tex")
;;
;; The three arguments are called `skip', `gotoend', and `target',
;; and they are expanded by the functions `ee-copy-rest-skip0',
;; `ee-copy-rest-gotoend0', and `ee-copy-rest-showtarget0' into
;; code that can be `eval'ed. These functions support some
;; shorthands:
;;
;;   2a) when `skip' is nil this means "start on the next line",
;;   2b) when `gotoend' is nil this means "until the end of the buffer",
;;   2c) when `target' is a string this means a file with that name.
;;
;; «ee-copy-rest-3-tests»  (to ".ee-copy-rest-3-tests")
;; Try these low-level tests:
;;
;;                (ee-copy-rest-skip0 nil)
;;          (eval (ee-copy-rest-skip0 nil))
;;                (ee-copy-rest-skip0 1)
;;                (ee-copy-rest-skip0 2)
;;          (eval (ee-copy-rest-skip0 2))
;;                (ee-copy-rest-gotoend0 nil)
;;                (ee-copy-rest-gotoend0 "find-2a")
;;          (eval (ee-copy-rest-gotoend0 "find-2a"))
;;                (ee-copy-rest-showtarget0 '(find-ebuffer "buf"))
;;                (ee-copy-rest-showtarget0 "/tmp/o")
;;   (find-2a nil (ee-copy-rest-showtarget0 "/tmp/o"))
;;
;; and these high-level tests:
;;
;;   (ee-copy-rest-3 3   ";; \ END" '(find-ebuffer "b"))
;;   (ee-copy-rest-3 2   ";; \ END" "/tmp/o")
;;   (ee-copy-rest-3 1   ";; \ END" "/tmp/o")
;;   (ee-copy-rest-3 nil ";; \ END" "/tmp/o")
;;     foo
;;     bar
;; END

;; «ee-copy-rest-3»  (to ".ee-copy-rest-3")
;;
(defun ee-copy-rest-skip0 (&optional skip)
  "An internal function used by `ee-copy-rest-3'."
  (cond ((null    skip) '(move-beginning-of-line 2))
        ((numberp skip) `(move-beginning-of-line (+ 2 ,skip)))
	(t              skip)))

(defun ee-copy-rest-gotoend0 (&optional end)
  "An internal function used by `ee-copy-rest-3'."
  (cond ((null end)    '(end-of-buffer))
	((stringp end) `(progn (search-forward ,end) (search-backward ,end)))
	t              end))

(defun ee-copy-rest-showtarget0 (target)
  "An internal function used by `ee-copy-rest-3'."
  (cond ((stringp target) `(find-fline ,target))
	(t                target)))

(defun ee-copy-rest-3 (skip gotoend target)
  "Copy the region between SKIP and GOTOEND to the kill ring and SHOWTARGET.
SKIP       is either nil, a number of lines to skip, or a sexp.
GOTOEND    is either nil, a string to search for, or a sexp.
SHOWTARGET is either a filename or a sexp to be used in a `find-2a'.\n
See the tests in the comments to understand the details."
  (save-excursion
    (let* ((start (progn (eval (ee-copy-rest-skip0    skip))    (point)))
	   (end   (progn (eval (ee-copy-rest-gotoend0 gotoend)) (point)))
	   (show  (ee-copy-rest-showtarget0 target))
	   (str   (buffer-substring start end))
	   (len   (ee-count-lines str))
	   (msg   `(Copied ,len lines to the kill ring - use C-y to paste)))
    (eeflash+ start end eeflash-copy)
    (kill-new str)
    (find-2a nil show)
    msg)))

(defun ee-copy-rest-3m (skip gotoend target)
  "Like `ee-copy-rest-3', but runs a mkdir if TARGET is a string."
  (if (stringp target)
      (make-directory (file-name-directory target) 'make-parents))
  (ee-copy-rest-3 skip gotoend target))







;;;   __ _           _       _ _       _       /\ ____  
;;;  / _(_)_ __   __| |     | (_)_ __ | | ____|/\|___ \ 
;;; | |_| | '_ \ / _` |_____| | | '_ \| |/ / __|   __) |
;;; |  _| | | | | (_| |_____| | | | | |   <\__ \  / __/ 
;;; |_| |_|_| |_|\__,_|     |_|_|_| |_|_|\_\___/ |_____|
;;;                                                     
;;
;; «find-find-links-links» (to ".find-find-links-links")
;; Test: (find-find-links-links "u" "find-links" "k stem args")
;; THIS IS OBSOLETE - DON'T USE THIS!
;; Use `find-find-links-links-new' instead.

(defun ee-prepend-commas (str)
  (save-match-data
    (replace-regexp-in-string "\\([^ ]+\\)" ",\\1" str)))
(defun ee-if-nil-setq (str)
  (format "(setq %s (or %s \"{%s}\"))" str str str))
(defun ee-if-nil-setqs (vars sep)
  (mapconcat 'ee-if-nil-setq (save-match-data (ee-split vars)) sep))

(defalias 'find-find-links-links-old 'find-find-links-links)

(defun find-find-links-links (&optional k stem args &rest pos-spec-list)
"Visit a temporary buffer containing hyperlinks for foo."
  ;; (interactive)
  ;; ^ Commented out because this is obsolete!
  (setq k    (or k    "{k}"))
  (setq stem (or stem "{stem}"))
  (setq args (or args "{args}"))
  (apply 'find-elinks-elisp
   `((find-find-links-links ,k ,stem ,args ,@pos-spec-list)
     ;; Convention: the first sexp always regenerates the buffer.
     (find-efunction 'find-find-links-links)
     ""
     ,(ee-template0 "\
;; See: (find-links-intro)
;;      (find-templates-intro)

;; <find-{stem}-links>
;; {(ee-S `(find-find-links-links ,k ,stem ,args))}
;; A test: (find-{stem}-links ___)
\(define-key eev-mode-map \"\\M-h{k}\" 'find-{stem}-links)

\(defun find-{stem}-links (&optional {args} &rest pos-spec-list)
\"Visit a temporary buffer containing hyperlinks for foo.\"
  (interactive)
  {(ee-if-nil-setqs args \"\n  \")}
  (apply 'find-elinks
   `((find-{stem}-links {(ee-prepend-commas args)} ,@pos-spec-list)
     ;; Convention: the first sexp always regenerates the buffer.
     (find-efunction 'find-{stem}-links)
     \"\"
     ,(ee-template0 \"\\
\")
     )
   pos-spec-list))

;; Test: (find-{stem}-links ___)


;;   ,@(ee-find-{stem}-links {args})
;;
\(defun ee-find-{stem}-links ({args})
  \"Return a list of sexps and strings for {stem}.
This is an internal function used by `find-{stem}-links'.\"
  `(
    ))")
     ) pos-spec-list))






;;;   __ _           _       _ _       _       /\ ____                           
;;;  / _(_)_ __   __| |     | (_)_ __ | | ____|/\|___ \      _ __   _____      __
;;; | |_| | '_ \ / _` |_____| | | '_ \| |/ / __|   __) |____| '_ \ / _ \ \ /\ / /
;;; |  _| | | | | (_| |_____| | | | | |   <\__ \  / __/_____| | | |  __/\ V  V / 
;;; |_| |_|_| |_|\__,_|     |_|_|_| |_|_|\_\___/ |_____|    |_| |_|\___| \_/\_/  
;;;                                                                              
;; «find-find-links-links-new»  (to ".find-find-links-links-new")
;; Test: (find-find-links-links-new)
;; Many of the templated functions of eev were written using this.
;; They all have a line saying ";; Skel:" that generates their "skeleton".
;; See: (find-eevgrep "grep --color -nH --null -e Skel: *.el")
;;      (find-eevtemplvideo "0:18" "1. `find-find-links-links-new'")
;;      (find-eevtemplvideo "1:15"   "if I execute (find-emacs-tangents-links)")
;;      (find-eevtemplvideo "1:29"   "generated again with these values")
;;      (find-eevtemplvideo "1:50"   "the implementation is very simple")
;;      (find-eevtemplvideo "2:00"   "in five minutes because")
;;      (find-eevtemplvideo "2:15"   ";; Skel:")
;;      (find-eevtemplvideo "2:32"   "optional arguments, then this field")
;;      (find-eevtemplvideo "2:40"   "temporary variables set by a let*")
;;      (find-eevtemplvideo "3:25"   "to write `find-emacs-tangents-links'")
;;      (find-eevtemplvideo "3:40"   "backslashes in the right places")
;;
(defun find-find-links-links-new (&optional stem args vars &rest pos-spec-list)
"Visit a temporary buffer containing a skeleton of a find-*-links function."
  (interactive)
  (setq stem (or stem "{stem}"))
  (setq args (or args "{args}"))
  (setq vars (or vars "{vars}"))
  (apply 'find-elinks-elisp
   `((find-find-links-links-new ,stem ,args ,vars ,@pos-spec-list)
     (find-find-links-links-new)
     (find-find-links-links-new "mytaskA" "" "")
     (find-find-links-links-new "mytaskB" "foo" "")
     (find-find-links-links-new "mytaskC" "foo bar" "")
     (find-find-links-links-new "mytaskD" "foo bar baz" "plic bletch")
     ;; Convention: the first sexp always regenerates the buffer.
     (find-efunction 'find-find-links-links-new)
     ""
     ,(ee-template0 ";; <find-{stem}-links>")
     ,(concat       ";; Skel: " (ee-S `(find-find-links-links-new ,stem ,args ,vars)))
     ,(ee-template0 ";; Test: (find-{stem}-links)")
     ";;"
     ,(ee-ffll-defun   stem args vars)
     ""
     ,(ee-ffll-deftest stem args vars)
     )
   pos-spec-list))


;;;                   __  __ _ _ _            
;;;   ___  ___       / _|/ _| | | |     __/\__
;;;  / _ \/ _ \_____| |_| |_| | | |_____\    /
;;; |  __/  __/_____|  _|  _| | | |_____/_  _\
;;;  \___|\___|     |_| |_| |_|_|_|       \/  
;;;                                           
;; «ee-ffll-functions»  (to ".ee-ffll-functions")
;; Low-level functions used by find-find-links-links-new.
;;
;; Tests:
;; (ee-ffll-optional "")
;; (ee-ffll-optional "foo bar")
;; (ee-ffll-comma-args "plic bletch")
;; (find-estring (ee-ffll-setqs "  " "foo bar"))
;; (find-estring (ee-ffll-lets "  " "foo bar"))
;; (find-estring-elisp (ee-ffll-defun-without-lets "mytask" "foo bar"))
;; (find-estring-elisp (ee-ffll-defun-with-lets "mytask" "foo bar" "plc bltch"))
;; (find-estring-elisp (ee-ffll-defun "mytask" "foo bar"))
;; (find-estring-elisp (ee-ffll-defun "mytask" "foo bar" "plic bletch"))
;;
(defun ee-ffll-optional (args)
  (if (< 0 (length (split-string args)))
      (concat "&optional " args " ")
    ""))
	 
(defun ee-ffll-setqs (spaces args)
  (mapconcat (lambda (arg) (format "%s(setq %s (or %s \"{%s}\"))\n"
				   spaces arg arg arg))
	     (split-string args)
	     ""))

(defun ee-ffll-lets (spaces vars)
  (format "let* (%s)"
	  (mapconcat (lambda (var) (format "(%s \"{%s}\")" var var))
		     (split-string vars)
		     (concat "\n" spaces "       "))))

(defun ee-ffll-comma-args (args)
  (mapconcat (lambda (arg) (format ",%s " arg))
	     (split-string args)
	     ""))

(defun ee-ffll-defun-without-lets (stem args)
  (let* ((optional   (ee-ffll-optional args))
	 (setqs      (ee-ffll-setqs "  " args))
	 (comma-args (ee-ffll-comma-args args))
	 )
    (ee-template0 "\
(defun find-{stem}-links ({optional}&rest pos-spec-list)
\"Visit a temporary buffer containing hyperlinks for {stem}.\"
  (interactive)
{setqs}\
  (apply
   'find-elinks
   `((find-{stem}-links {comma-args},@pos-spec-list)
     ;; Convention: the first sexp always regenerates the buffer.
     (find-efunction 'find-{stem}-links)
     \"\"
     ,(ee-template0 \"\\
\")
     )
   pos-spec-list))
")))

(defun ee-ffll-defun-with-lets (stem args vars)
  (let* ((optional   (ee-ffll-optional   args))
	 (setqs      (ee-ffll-setqs "  " args))
	 (comma-args (ee-ffll-comma-args args))
	 (lets       (ee-ffll-lets "  "  vars))
	 )
    (ee-template0 "\
(defun find-{stem}-links ({optional}&rest pos-spec-list)
\"Visit a temporary buffer containing hyperlinks for {stem}.\"
  (interactive)
{setqs}\
  ({lets}
    (apply
     'find-elinks
     `((find-{stem}-links {comma-args},@pos-spec-list)
       ;; Convention: the first sexp always regenerates the buffer.
       (find-efunction 'find-{stem}-links)
       \"\"
       ,(ee-template0 \"\\
\")
       )
     pos-spec-list)))
")))

(defun ee-ffll-defun (stem args &optional vars)
  (if (equal vars "") (setq vars nil))
  (if vars (ee-ffll-defun-with-lets stem args vars)
     (ee-ffll-defun-without-lets stem args)))

(defun ee-ffll-deftest (stem args &optional vars)
  (ee-template0 "\
;; Functions to test changes in the template of `find-{stem}-links'.
;; To use them type `M-x tt' inside the `(defun find-{stem}-links ...)'.
;; See: (find-templates-intro \"5. Debugging the meat\")
;;
(defun ee-template-test (&rest args)
  (let ((ee-buffer-name \"*ee-template-test*\"))
    (find-2a nil `(find-{stem}-links ,@args))))

(defun tt0 () (interactive) (eek \"C-M-x\") (ee-template-test))
(defun tt  () (interactive) (eek \"C-M-x\") (ee-template-test \"A\" \"B\"))
"))



;; «find-let*-macro-links»  (to ".find-let*-macro-links")
;; Skel: (find-find-links-links-new "let*-macro" "stem args defs" "sep part1 part2")
;; Test: (find-let*-macro-links "foo" "a b c" "d e f")
;;  See: (find-templates-intro "7. let* macros")
;;
(defun find-let*-macro-links (&optional stem args defs &rest pos-spec-list)
"Visit a temporary buffer containing hyperlinks for let*-macro."
  (interactive)
  (setq stem (or stem "{stem}"))
  (setq args (or args "{args}"))
  (setq defs (or defs "{defs}"))
  (let* ((sep "\n          ")
	 (part1 (mapconcat (lambda (v) (format "(%s ,%s)" v v))
			   (ee-split args) sep))
	 (part2 (mapconcat (lambda (v) (format "(%s nil)" v v))
			   (ee-split defs) sep)))
    (apply
     'find-elinks-elisp
     `((find-let*-macro-links ,stem ,args ,defs ,@pos-spec-list)
       ;; Convention: the first sexp always regenerates the buffer.
       (find-efunction 'find-let*-macro-links)
       ""
       ,(ee-template0 "\
;; Skel: (find-let*-macro-links \"{stem}\" \"{args}\" \"{defs}\")
(defmacro ee-let*-macro-{stem} ({args} &rest code)
  \"An internal function used by `find-{stem}-links'.\"
  `(let* ({part1}{sep}{part2})
     ,@code))
")
       )
     pos-spec-list)))







;;;  _       _                   _ _       _        
;;; (_)_ __ | |_ _ __ ___       | (_)_ __ | | _____ 
;;; | | '_ \| __| '__/ _ \ _____| | | '_ \| |/ / __|
;;; | | | | | |_| | | (_) |_____| | | | | |   <\__ \
;;; |_|_| |_|\__|_|  \___/      |_|_|_| |_|_|\_\___/
;;;                                                 
;; «find-intro-links»  (to ".find-intro-links")
;; Skel: (find-find-links-links-new "intro" "stem" "")
;; Test: (find-intro-links)
;;
(defun find-intro-links (&optional stem &rest pos-spec-list)
"Visit a temporary buffer with a skeleton for defining `find-<STEM>-intro'.
All `find-*-intro' functions in eev-intro.el were written using this.
See: (find-eev \"eev-intro.el\")"
  (interactive)
  (setq stem (or stem "{stem}"))
  (apply
   'find-elinks-elisp
   `((find-intro-links ,stem ,@pos-spec-list)
     (find-intro-links "{stem}" ,@pos-spec-list)
     (find-intro-links "foo"    ,@pos-spec-list)
     ;; Convention: the first sexp always regenerates the buffer.
     (find-efunction 'find-intro-links)
     ""
     ,(ee-template0 "\
;; (ee-copy-rest 1 '(find-eev \"eev-intro.el\"))
;; (find-{stem}-intro)

;; <find-{stem}-intro>
;; Skel: (find-intro-links \"{stem}\")
;; Test: (find-{stem}-intro)

\(defun find-{stem}-intro (&rest pos-spec-list) (interactive)
  (let ((ee-buffer-name \"*(find-{stem}-intro)*\"))
    (apply 'find-eintro \"\\
\\(Re)generate: (find-{stem}-intro)
Source code:  (find-efunction 'find-{stem}-intro)
More intros:  (find-eev-quick-intro)
              (find-eev-intro)
              (find-eepitch-intro)
This buffer is _temporary_ and _editable_.
It is meant as both a tutorial and a sandbox.

Hello
=====
\" pos-spec-list)))

;; (find-{stem}-intro)
")
     )
   pos-spec-list))

;; (find-intro-links)
;; (find-intro-links "emacs")





;;;                       _                    _           
;;;   ___  _____   __    | |__   ___  __ _  __| | ___ _ __ 
;;;  / _ \/ _ \ \ / /____| '_ \ / _ \/ _` |/ _` |/ _ \ '__|
;;; |  __/  __/\ V /_____| | | |  __/ (_| | (_| |  __/ |   
;;;  \___|\___| \_/      |_| |_|\___|\__,_|\__,_|\___|_|   
;;;                                                        
;; «find-eev-header-links» (to ".find-eev-header-links")
;; (find-find-links-links "<none>" "eev-header" "stem-el date")
;; A test: (find-eev-header-links "eev-audiovideo.el")

(defun find-eev-header-links (&optional stem-el date &rest pos-spec-list)
"Visit a temporary buffer containing hyperlinks for foo."
  (interactive)
  (setq stem-el (or stem-el "{stem-el}"))
  (setq date    (or date (downcase (format-time-string "%Y%b%d"))))
  (apply 'find-elinks-elisp
   `((find-eev-header-links ,stem-el ,date ,@pos-spec-list)
     ;; Convention: the first sexp always regenerates the buffer.
     (find-efunction 'find-eev-header-links)
     (find-eev ,stem-el)
     (wrobak 2 '(find-eev ,stem-el))
     ,(ee-template0 "
;;; {stem-el} -- ???.

;; Copyright (C) 2019 Free Software Foundation, Inc.
;;
;; This file is part of GNU eev.
;;
;; GNU eev is free software: you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.
;;
;; GNU eev is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs.  If not, see <http://www.gnu.org/licenses/>.
;;
;; Author:     Eduardo Ochs <eduardoochs@gmail.com>
;; Maintainer: Eduardo Ochs <eduardoochs@gmail.com>
;; Version:    {date}
;; Keywords:   e-scripts
;;
;; Latest version: <http://anggtwu.net/eev-current/{stem-el}>
;;       htmlized: <http://anggtwu.net/eev-current/{stem-el}.html>
;;       See also: <http://anggtwu.net/eev-current/eev-readme.el.html>
;;                 <http://anggtwu.net/eev-intros/find-eev-intro.html>
;;                                               (find-eev-intro)

;;; Commentary:


;; Local Variables:
;; coding:            utf-8-unix
;; ee-anchor-format:  \"defun %s \"
;; no-byte-compile:   t
;; End:
")) pos-spec-list))




;;;   __ _           _           _      _           _         
;;;  / _(_)_ __   __| |       __| | ___| |__  _ __ | | ____ _ 
;;; | |_| | '_ \ / _` |_____ / _` |/ _ \ '_ \| '_ \| |/ / _` |
;;; |  _| | | | | (_| |_____| (_| |  __/ |_) | |_) |   < (_| |
;;; |_| |_|_| |_|\__,_|      \__,_|\___|_.__/| .__/|_|\_\__, |
;;;                                          |_|        |___/ 
;;
;; «find-debpkg-links»  (to ".find-debpkg-links")

(defun ee-links-for-debpkg (pkgname)
  "Return the three main links for the debian package PKGNAME."
  (list (ee-template0 "\
{ee-H}(find-status   \"{pkgname}\")
{ee-H}(find-vldifile \"{pkgname}.list\")
{ee-H}(find-udfile   \"{pkgname}/\")")))

(defun ee-dfs0 (pkg ext)
  "If the file /var/lib/dpkg/info/PKG.EXT exists, return a link to it."
  (let ((fname (concat pkg "." ext)))
    (if (file-exists-p (ee-vldifile fname))
	`(find-vldifile ,fname))))

(defun ee-links-for-debpkg-extra-vldi (pkg)
  "Return a list of links for files in /var/lib/dpkg/info/ belonging to PKG.
This is an internal function used by `find-debpkg-links'."
   (list (ee-dfs0 pkg "preinst")   (ee-dfs0 pkg "postinst")
	 (ee-dfs0 pkg "prerm")     (ee-dfs0 pkg "postrm")
	 (ee-dfs0 pkg "conffiles") (ee-dfs0 pkg "config")
	 (ee-dfs0 pkg "templates")
	 (ee-dfs0 pkg "md5sums")   (ee-dfs0 pkg "shlibs")
	 ))

(defun ee-debian-pooldir (pkg)
  "Used by `find-debpkg-links'; \"foo\" -> \"f\", \"libfoo\" -> \"libf\"."
  (if (string-match "^\\(lib\\)?." pkgname)
      (match-string 0 pkgname)))

;; Tests: (find-debpkg-links "bash")
;;        (find-debpkg-links "libicu")
;;
(defun find-debpkg-links (&optional pkgname &rest rest)
  "Visit a temporary buffer containing hyperlinks related to a Debian package.
Try this: (find-debpkg-links \"bash\")"
  (interactive (list (ee-debpkgname-ask)))
  (setq pkgname (or pkgname "{pkgname}"))
  (let ((p (ee-debian-pooldir pkgname)))
    (apply 'find-elinks `(
      ;; Convention: the first sexp always regenerates the buffer.
      (find-debpkg-links ,pkgname ,@rest)
      (find-efunction 'find-debpkg-links)
      (find-available ,pkgname)
      ""
      ,@(ee-links-for-debpkg pkgname)
      ""
      ,@(ee-links-for-debpkg-extra-vldi pkgname)
      ""
      ,(ee-template0 "\
{ee-H}(find-sh \"apt-file search {pkgname}\")
{ee-H}(find-sh \"apt-cache dump | grep-dctrl -P {pkgname}\")
{ee-H}(find-sh \"apt-cache showpkg {pkgname}\")
{ee-H}(find-sh \"apt-cache search {pkgname} | sort\")
{ee-H}(find-sh \"apt-cache search '{pkgname}*' | sort\")
{ee-H}(find-sh \"apt search '{pkgname}*'\")
{ee-H}(find-sh \"apt search '{pkgname}*' 2> /dev/null\")
{ee-H}(find-sh \"grep-aptavail -P {pkgname}\")
{ee-H}(find-sh \"grep-aptavail -P -s Package {pkgname}\")
{ee-H}(find-sh \"grep-aptavail -P -s Package {pkgname} | sort\")
{ee-H}(find-sh \"grep-aptavail -P -s Package,Version {pkgname}\")
{ee-H}(find-sh \"dpkg-query -W '{pkgname}*'\")

http://packages.debian.org/{pkgname}
http://packages.debian.org/sid/{pkgname}
http://packages.debian.org/source/sid/{pkgname}
http://packages.debian.org/src:{pkgname}
http://ftp.debian.org/debian/pool/main/{p}/{pkgname}/
http://backports.org/debian/pool/main/{p}/{pkgname}/
http://bugs.debian.org/cgi-bin/pkgreport.cgi?which=pkg&data={pkgname}&archive=no

http://packages.ubuntu.org/{pkgname}

 (eepitch-shell2)
sudo apt-get install    {pkgname}
sudo apt-get install -y {pkgname}

 (eepitch-shell)
 (eepitch-kill)
 (eepitch-shell)
# (find-man \"8 apt-cache\")
apt-cache show {pkgname}
apt-cache search {pkgname}
apt-cache depends {pkgname}
apt-cache rdepends {pkgname}

apt-cache showpkg {pkgname}

# (find-man \"1 dpkg-query\")
apt-file search {pkgname}
dpkg-query --search {pkgname}
")
      ) rest)))



;; «find-pacman-links»  (to ".find-pacman-links")
;; Skel: (find-find-links-links-new "pacman" "pkg" "")
;; Test: (find-pacman-links)
;;
(defun find-pacman-links (&optional pkg &rest pos-spec-list)
"Visit a temporary buffer containing hyperlinks for pacman."
  (interactive)
  (setq pkg (or pkg "{pkg}"))
  (apply
   'find-elinks
   `((find-pacman-links ,pkg ,@pos-spec-list)
     ;; Convention: the first sexp always regenerates the buffer.
     (find-efunction 'find-pacman-links)
     ""
     ,(ee-template0 "\
# https://archlinux.org/packages/?sort=&q={pkg}&maintainer=&flagged=
# https://archlinux.org/packages/extra/x86_64/{pkg}/
# https://archlinux.org/packages/extra/x86_64/{pkg}/files/

 (eepitch-shell)
 (eepitch-kill)
 (eepitch-shell)
sudo pacman -S {pkg}

")
     )
   pos-spec-list))




;;;  __  __            ____            _       
;;; |  \/  | __ _  ___|  _ \ ___  _ __| |_ ___ 
;;; | |\/| |/ _` |/ __| |_) / _ \| '__| __/ __|
;;; | |  | | (_| | (__|  __/ (_) | |  | |_\__ \
;;; |_|  |_|\__,_|\___|_|   \___/|_|   \__|___/
;;;                                            
;; «find-macports-links»  (to ".find-macports-links")
;; Skel: (find-find-links-links-new "macports" "pkg" "")
;; Test: (find-macports-links)
;;
(defun find-macports-links (&optional pkg &rest pos-spec-list)
"Visit a temporary buffer containing hyperlinks for macports."
  (interactive)
  (setq pkg (or pkg "{pkg}"))
  (apply
   'find-elinks
   `((find-macports-links ,pkg ,@pos-spec-list)
     ;; Convention: the first sexp always regenerates the buffer.
     (find-efunction 'find-macports-links)
     ""
     ,(ee-template0 "\
# (find-sh \"port contents {pkg}\")
# (find-sh \"find /opt/local | grep {pkg} | sort\")
# (find-sh \"port echo all | grep {pkg} | sort\")
# (find-sh \"port echo installed\")
# (find-sh \"port help echo\")
# (find-sh \"port help contents\")
# (find-sh \"port help provides\")
# (find-sh \"port provides /opt/local/an/existing/file\")
# (find-man \"1 port\")

# https://ports.macports.org/port/{pkg}/

 (eepitch-shell)
 (eepitch-kill)
 (eepitch-shell)
sudo port install {pkg}
sudo brew install {pkg}
")
     )
   pos-spec-list))



;;;  _   _                      _                       
;;; | | | | ___  _ __ ___   ___| |__  _ __ _____      __
;;; | |_| |/ _ \| '_ ` _ \ / _ \ '_ \| '__/ _ \ \ /\ / /
;;; |  _  | (_) | | | | | |  __/ |_) | | |  __/\ V  V / 
;;; |_| |_|\___/|_| |_| |_|\___|_.__/|_|  \___| \_/\_/  
;;;                                                     
;; «find-homebrew-links»  (to ".find-homebrew-links")
;; Skel:  (find-find-links-links-new "homebrew" "pkg" "")
;; Tests: (find-homebrew-links)
;;        (find-homebrew-links "lpeg")
;;
(defun find-homebrew-links (&optional pkg &rest pos-spec-list)
"Visit a temporary buffer containing hyperlinks for homebrew."
  (interactive)
  (setq pkg (or pkg "{pkg}"))
  (apply
   'find-elinks
   `((find-homebrew-links ,pkg ,@pos-spec-list)
     ;; Convention: the first sexp always regenerates the buffer.
     (find-efunction 'find-homebrew-links)
     ""
     ,(ee-template0 "\
# See:
#   (find-macports-links \"{pkg}\")
# https://formulae.brew.sh/formula/{pkg}
# https://formulae.brew.sh/formula/lua@5.1#default

 (eepitch-shell)
 (eepitch-kill)
 (eepitch-shell)
brew install {pkg}
")
     )
   pos-spec-list))



;;;   __ _           _           _          _           _ _     _ 
;;;  / _(_)_ __   __| |       __| |___  ___| |__  _   _(_) | __| |
;;; | |_| | '_ \ / _` |_____ / _` / __|/ __| '_ \| | | | | |/ _` |
;;; |  _| | | | | (_| |_____| (_| \__ \ (__| |_) | |_| | | | (_| |
;;; |_| |_|_| |_|\__,_|      \__,_|___/\___|_.__/ \__,_|_|_|\__,_|
;;;                                                               
;; This is a VERY EARLY prototype (hi Marc!)
;; of a rewrite of something that was very ugly.

;; (find-angg ".emacs" "find-dpkg-build-escript")

(defun ee-dsc-url-split (dsc-url)
  "Example:
  (ee-dsc-url-split
   \"http://ftp.debian.org/debian/pool/main/i/imagemagick/imagemagick_6.2.4.5.dfsg1-0.9.dsc\")
  -> (\"http\" \"ftp.debian.org/debian/pool/main/i/imagemagick/\"
      \"imagemagick\" \"6.2.4.5.dfsg1\" \"-0.9\")"
  (let ((prot://dir/ (file-name-directory dsc-url))
	(fname-dsc (file-name-nondirectory dsc-url))
	prot dir/ xxx vvv -sv)
    (if (string-match "^\\([a-z]+\\)://\\(.*\\)" prot://dir/)
	(setq prot (match-string 1 prot://dir/)
	      dir/ (match-string 2 prot://dir/)))
    (if (string-match "^\\([^_]+\\)_\\([^-]+\\)\\(-.*\\)?\\.dsc$" fname-dsc)
	(setq xxx (match-string 1 fname-dsc)
	      vvv (match-string 2 fname-dsc)
	      -sv (or (match-string 3 fname-dsc) "")))
    (list prot dir/ xxx vvv -sv)))

(defun ee-links-for-dscbuild (dsc-url)
  (apply 'ee-links-for-dscbuild0
	 (downcase (format-time-string "%Y%b%d"))
	 (ee-dsc-url-split dsc-url)))

;; Note: this is VERY OLD - it uses bounded regions,
;; that I haven't used in ages... see:
;; (find-bounded-intro)
(defun ee-links-for-dscbuild0 (date prot dir/ xxx vvv -sv)
  (ee-template
   '(date prot dir/ xxx vvv -sv) "\
#####
#
# {xxx} (from the debian sources)
# {date}
#
#####

# <{xxx}-deb-src>
# {prot}://{dir/}
# {prot}://{dir/}{xxx}_{vvv}{-sv}.dsc
# {prot}://{dir/}{xxx}_{vvv}{-sv}.diff.gz
# {prot}://{dir/}{xxx}_{vvv}.orig.tar.gz
#
rm -Rv ~/usrc/{xxx}/
mkdir  ~/usrc/{xxx}/
cd $S/{prot}/{dir/}
cp -v {xxx}_{vvv}* ~/usrc/{xxx}/
cd     ~/usrc/{xxx}/
dpkg-source -sn -x {xxx}_{vvv}{-sv}.dsc
cd     ~/usrc/{xxx}/{xxx}-{vvv}/
dpkg-buildpackage -us -uc -b -rfakeroot     2>&1 | tee odb

#
# (find-fline \"~/usrc/{xxx}/\")
 (eepitch-shell)
cd ~/usrc/{xxx}/
sudo dpkg -i *.deb

#
# (code-c-d \"{xxx}\" \"~/usrc/{xxx}/{xxx}-{vvv}/\")
# (find-{xxx}file \"\")"))




;;;                                _                 _ 
;;;   ___  _____   __     _ __ ___| | ___   __ _  __| |
;;;  / _ \/ _ \ \ / /____| '__/ _ \ |/ _ \ / _` |/ _` |
;;; |  __/  __/\ V /_____| | |  __/ | (_) | (_| | (_| |
;;;  \___|\___| \_/      |_|  \___|_|\___/ \__,_|\__,_|
;;;                                                    
;; «find-eev-reload-links»  (to ".find-eev-reload-links")
;; Skel: (find-find-links-links-new "eev-reload" "" "")
;; Test: (find-eev-reload-links)
;;
(defun find-eev-reload-links (&rest pos-spec-list)
"Visit a temporary buffer with a script for reloading eev. Experimental!"
  (interactive)
  (apply
   'find-elinks-elisp
   `((find-eev-reload-links ,@pos-spec-list)
     ;; Convention: the first sexp always regenerates the buffer.
     (find-efunction 'find-eev-reload-links)
     ""
     ,(ee-template0 "
 (eepitch-shell)
 (eepitch-kill)
 (eepitch-shell)
 (find-eevfile \"ChangeLog\")
cd {ee-eevdir}
git pull
 (find-eevfile \"ChangeLog\")


;; Based on: (find-eev \"eev-load.el\")
;;
(cl-loop for modulename in (ee-split (string-trim \"
  eev-intro
  eev-flash eev-multiwindow eev-eval eev-mode eev-anchors eev-template0
  eev-blinks eev-plinks eev-elinks eev-tlinks eev-hlinks eev-htests
  eev-brxxx
  eepitch
  eev-wrap
  eejump
  eev-anchors
  eev-code eev-pdflike
  eev-codings eev-env eev-edit eev-testblocks
  eev-kla eev-kl-here eev-strange-functions eev-helpful eev-rstdoc eev-qrl
  eev-wconfig
  eev-audiovideo
  eev-videolinks
  eev-rcirc
  eev-aliases\"))
	 do (load modulename))


(load user-init-file)
(load \"~/.emacs\")
(load \"~/.emacs.d/init.el\")


;; Compare this: (find-eev \"eev-load.el\")
;; and the output of:
(let* ((fname (ee-eevfile \"eev-load.el\"))
       (bigstr (ee-read-file fname))
       (sexps (read (format \"(\\n%s\\n)\" bigstr)))
       (f (lambda (sexp)
	    (pcase sexp
              (`(require ',feature) feature))))
       (features0 (mapcar f sexps))
       (features (seq-filter 'identity features0))
       )
  ;; (find-eppp sexps)
  (find-eppp features)
  )
")
     )
   pos-spec-list))



;;;                       _           _        _ _ 
;;;   ___  _____   __    (_)_ __  ___| |_ __ _| | |
;;;  / _ \/ _ \ \ / /____| | '_ \/ __| __/ _` | | |
;;; |  __/  __/\ V /_____| | | | \__ \ || (_| | | |
;;;  \___|\___| \_/      |_|_| |_|___/\__\__,_|_|_|
;;;                                                

;; «find-eev-install-links» (to ".find-eev-install-links")
;; (find-find-links-links "{k}" "eev-install" "dir script comment")
;; A test: (find-eev-install-links)

(defun find-eev-install-links (&optional dir script comment &rest pos-spec-list)
"Visit a temporary buffer containing an e-script for installing eev."
  (interactive)
  (setq dir (or dir "~/eev2/"))
  (setq script (or script "~/eev"))
  (setq comment (or comment ""))
  (apply 'find-elinks
   `((find-eev-install-links ,dir ,script ,comment ,@pos-spec-list)
     (find-eev-install-links "~/eev2/" "~/eev")
     (find-eev-install-links "~/eev2/" "~/eev" "#")
     (find-eev-install-links "/tmp/eev2/" "/tmp/eev")
     (find-eev-install-links "/tmp/eev2/" "/tmp/eev" "#")
     (find-eev-install-links "{dir}" "{script}" "{comment}")
     (find-eev-install-links)
     ;; Convention: the first sexp always regenerates the buffer.
     (find-efunction 'find-eev-install-links)
     ""
     (find-eev-install-intro)
     ""
     ,(ee-template0 "\
# This function is explained at:
#   (find-eev-install-intro \"2. Running `(find-eev-install-links)'\")
#
# The default way to use it to install eev in your home directory is with:
#   (find-eev-install-links \"~/eev2/\" \"~/eev\" \"#\")
#
# The default way to use it to install eev in /tmp/ for tests is with:
#   (find-eev-install-links \"/tmp/eev2/\" \"/tmp/eev\" \"#\")



# The script below downloads eev2.tgz and unpacks it into {dir}
# and creates a shell script {script}
# that can be used to start emacs+eev.

 (eepitch-shell)
 (eepitch-kill)
 (eepitch-shell)
{comment}{<}
  rm -Rv {script}
  rm -Rv {dir}
  mkdir  {dir}
  cd     {dir}
  rm -fv eev2.tgz
  wget http://anggtwu.net/eev-current/eev2.tgz
  tar -xvzf eev2.tgz
  {<}
    echo '#!/bin/sh'
    echo 'cd {dir} && emacs -l eev-beginner.el --eval=\"(find-eev-quick-intro)\" $*'
  {>} > {script}
  chmod 755 {script}
{comment}{>}

# Test:
{script}



;; If you have unpacked eev2.tgz into some permanent place you can
;; put this is your .emacs to load eev when Emacs starts. Don't
;; copy this to your .emacs if either 1) you have only unpacked
;; eev in /tmp/, or 2) if you don't know how to use Lisp well
;; enough!
;;
;; See: (find-eev-install-intro \"3. Changing your .emacs\")

;; (ee-copy-rest 0 '(find-fline \"~/.emacs\"))

;; Load eev.
;; Generated by: (find-eev-install-links \"{dir}\" \"{script}\" \"{comment}\")
;;        Check: (find-fline \"{dir}\")
;;          See: (find-eev-install-intro \"3. Changing your .emacs\")
;;
\(add-to-list 'load-path \"{dir}\")
\(require 'eev-load)                 ; (find-eev \"eev-load.el\")
\(eev-mode 1)                        ; (find-eev \"eev-mode.el\")

")
     )
   pos-spec-list))

;; A test: (find-eev-install-links)
;;         (find-eev-install-links "~/eev2/" "~/eev" "#" 60)


;;;                                       _       _       
;;;   ___  _____   __     _   _ _ __   __| | __ _| |_ ___ 
;;;  / _ \/ _ \ \ / /____| | | | '_ \ / _` |/ _` | __/ _ \
;;; |  __/  __/\ V /_____| |_| | |_) | (_| | (_| | ||  __/
;;;  \___|\___| \_/       \__,_| .__/ \__,_|\__,_|\__\___|
;;;                            |_|                        
;;
;; «find-eev-update-links» (to ".find-eev-update-links")
;; A test: (find-eev-update-links)
;; TODO: Check that this has been superseded by
;;   (find-eev-install-links)
;; and flag this as obsolete.

(defun find-eev-update-links (&optional dir script &rest pos-spec-list)
"Visit a temporary buffer with scripts for installing and updating eev."
  (interactive)
  (setq dir (or dir ee-eevdir))
  (setq script (or script "~/e"))
  (apply 'find-elinks
   `((find-eev-update-links ,dir ,script)
     (find-eev-update-links "/tmp/eev/" ,script)
     (find-eev-update-links "~/eev2/" ,script)
     (find-eev-update-links "~/eev/" ,script)
     (find-eev-update-links "{dir}" "{script}")
     (find-eev-update-links)
     ,(ee-template0 "\

# NOTE! This function is obsolete and has been superseded by:
# (find-eev-install-links)



# Download , unpack and create a script \"~/e\"
# ============================================
# See: (find-eev-quick-intro \"1. Installing eev\")

{
  rm -Rv {dir}
  mkdir  {dir}
  cd     {dir}
  rm -v eev2.tgz
  wget http://anggtwu.net/eev-current/eev2.tgz
  tar -xvzf eev2.tgz
  {
    echo '#!/bin/sh'
    echo 'cd {dir} && emacs -l eev-readme.el --eval=\"(find-eev-quick-intro)\"'
  } > {script}
  chmod 755 {script}
}



# Download / unpack the tarball (an older way)
# ============================================
# See: (find-eepitch-intro)
#      (find-eev \"eev-readme.el\")

 (eepitch-shell)
 (eepitch-kill)
 (eepitch-shell)
#rm -Rv {dir}
mkdir   {dir}
cd      {dir}
rm -v eev2.tgz
wget http://anggtwu.net/eev-current/eev2.tgz
tar -xvzf eev2.tgz

# Tests:
emacs -Q -fg bisque -bg black -l eev-readme.el eev-readme.el
emacs -Q -fg bisque -bg black                  eev-readme.el
emacs    -fg bisque -bg black                  eev-readme.el



# Tell Emacs to load eev2 by default
# ==================================
# Emacs executes the file ~/.emacs when it starts up - see:
#   (find-enode \"Init File\")
# The easiest way to make Emacs load eev2 by default is to
# open your ~/.emacs in another window with the sexp below,
# and then copy the block below to it with `C-y'.
#   (ee-copy-rest 0 '(find-fline \"~/.emacs\"))

;; Load eev2.
;; See:  (find-file \"{dir}\")
;;       (find-file \"{dir}eev-readme.el\")
;; Generated by: (find-eev-update-links \"{dir}\")
;;
\(add-to-list 'load-path \"{dir}\")
\(require 'eev2-all)                 ; (find-eev \"eev2-all.el\")
\(eev-mode 1)                        ; (find-eev \"eev-mode.el\")
")) pos-spec-list))

;; (find-eev-update-links)




;;;                    _         _                    _ _ 
;;;  _   _  ___  _   _| |_ _   _| |__   ___        __| | |
;;; | | | |/ _ \| | | | __| | | | '_ \ / _ \_____ / _` | |
;;; | |_| | (_) | |_| | |_| |_| | |_) |  __/_____| (_| | |
;;;  \__, |\___/ \__,_|\__|\__,_|_.__/ \___|      \__,_|_|
;;;  |___/                                                
;;
;; «find-youtubedl-links»  (to ".find-youtubedl-links")
;; Skel: (find-find-links-links-new "youtubedl" "dir title hash ext- stem" "")
;; Test: (find-youtubedl-links nil nil "K6LmZ0A1s9U")
;; See:  (find-audiovideo-intro "6. Youtube-dl")
;;       (find-audiovideo-intro "6.1. Downloading a local copy")
;;       (find-eevtemplvideo "19:23" "5. `find-youtubedl-links'")
;;       (find-eevtemplvideo "19:43"   "if we twist the notion of user enough")
;;       (find-eevtemplvideo "20:02"   "user-friendly to me")
;;       (find-eevtemplvideo "20:05"   "The documentation is in this tutorial:")
;;       (find-eevtemplvideo "21:36"   "its code is here")
;;       (find-eevtemplvideo "21:55"   "Let me show a demo")
;;       (find-eevtemplvideo "22:15"   "I never type these rm -Rv/mkdir/cd by hand")
;;       (find-eevtemplvideo "22:48"   "let me show how to download ... the hash is here")
;;       (find-eevtemplvideo "23:25"   "they regenerate this buffer in slightly different ways")
;;       (find-eevtemplvideo "24:03"   "a video with this hash. If I visit the directory")
;;       (find-eevtemplvideo "24:32"   "my magic functions have to")
;;       (find-eevtemplvideo "24:50"   "if I run this sexp here with nil in the title")
;;       (find-eevtemplvideo "25:00"   "it tries to guess the title")
;;       (find-eevtemplvideo "25:28"   "and if I change this {stem} here I get short links")
;;       (find-eevtemplvideo "26:14"   "has very long lines, but if I type super-w")
;;       (find-eevtemplvideo "26:28"   "I get a buffer that shows the code for")
;;       (find-eevtemplvideo "26:50"   "if I execute this I play the video")
;;       (find-eevtemplvideo "27:15"   "a copy of that points to the right position")
;;       (find-eevtemplvideo "27:53"   "summary: this is how I download videos from youtube")
;;
(defun find-youtubedl-links (&optional dir title hash ext- stem &rest pos-spec-list)
"Visit a temporary buffer containing hyperlinks for youtube-dl."
  (interactive)
  (setq dir   (or dir   ee-youtubedl-dir "{dir}"))
  (setq hash  (or hash  (ee-youtubedl-hash-around-point)    "{hash}"))
  (setq title (or title (ee-youtubedl-guess-title dir hash) "{title}"))
  (setq ext-  (or ext-  (ee-youtubedl-guess-ext- dir hash)  "{ext-}"))
  (setq stem  (or stem "{stem}"))
  (let ((ee-buffer-name (or ee-buffer-name "*find-youtubedl-links*")))
    (apply
     'find-elinks
     `((find-youtubedl-links ,dir ,title ,hash ,ext- ,stem ,@pos-spec-list)
       (find-youtubedl-links ,dir nil ,hash nil ,stem)
       ;; Convention: the first sexp always regenerates the buffer.
       (find-efunction 'find-youtubedl-links)
       ""
       ,@(ee-youtubedl-dir-links nil hash stem)
       (setq ee-youtubedl-dirs ',ee-youtubedl-dirs)
       (setq ee-youtubedl-dir   ,ee-youtubedl-dir)
       ""
       (find-audiovideo-intro "6. Youtube-dl")
       (find-eev "eev-tlinks.el" "ee-youtubedl-command")
       ""
       ,(ee-template0 "\
 (sh-mode)
 (eepitch-shell2)
 (eepitch-kill)
 (eepitch-shell2)
# http://www.youtube.com/watch?v={hash}
# http://www.youtube.com/watch?v={hash}#t=0m00s
# http://www.youtube.com/watch?v={hash}#t=0h00m00s
cd {dir}
{ee-youtubedl-command} -f 18 --restrict-filenames 'http://www.youtube.com/watch?v={hash}'

# Or:
{ee-youtubedl-command}       'http://www.youtube.com/watch?v={hash}'
{ee-youtubedl-command} -F    'http://www.youtube.com/watch?v={hash}'
{ee-youtubedl-command} -f 18 'http://www.youtube.com/watch?v={hash}'
{ee-youtubedl-command} -f 18 --restrict-filenames --all-subs 'http://www.youtube.com/watch?v={hash}'
{ee-youtubedl-command}       --restrict-filenames --all-subs 'http://www.youtube.com/watch?v={hash}'

# Play the local copy:
# (find-fline \"{dir}\" \"{hash}\")
# (find-fline \"{dir}\" \"{title}-{hash}\")
# (find-fline \"{dir}\" \"{title}-{hash}{ext-}\")
# (find-video \"{dir}{title}-{hash}{ext-}\")
# (code-video \"{stem}video\" \"{dir}{title}-{hash}{ext-}\")
# (find-{stem}video)
# (find-{stem}video \"0:00\")

# Error messages (for the player):
# (find-ebuffer \"*Messages*\")

# Play on youtube:
# (find-youtube-video \"{hash}\" \"0:00\")
# (code-youtubevideo \"{stem}\" \"{hash}\" \"{title}\")
# (find-{stem}video \"0:00\")

# Transcript:
# (find-yttranscript0-links \"{stem}\" \"{hash}\")
")
       )
     pos-spec-list)))




;; `find-youtubedl-links' uses LOTS of internal functions and
;; variables. They are defined below.

;; Code for splitting filenames of downloaded videos into components.
;; Test: (ee-youtubedl-split "~/tmp/videos/foo_bar-abF7go7RLTc.flv")
;;       --> ("~/tmp/videos/" "foo_bar" "abF7go7RLTc" ".flv" ".flv")
;;
(defvar ee-youtubedl-ext-re
  "\\(\\.[A-Za-z0-9]\\{2,5\\}\\)\\(\\.part\\)?$")

(defun ee-youtubedl-split (fname)
  "Split FNAME into (dir title hash ext ext-)."
  (string-match ee-youtubedl-ext-re fname)
  (let* ((ext-  (match-string 1 fname))
	 (ext   (match-string 0 fname))
	 (dth   (substring fname 0 (match-beginning 0)))
	 (hash  (substring dth -11))
	 (dt    (substring dth 0 -12))
	 (title (file-name-nondirectory dt))
	 (dir   (file-name-directory dt)))
    (list dir title hash ext- ext)))

(defun ee-youtubedl-dir   (fname) (nth 0 (ee-youtubedl-split fname)))
(defun ee-youtubedl-title (fname) (nth 1 (ee-youtubedl-split fname)))
(defun ee-youtubedl-hash  (fname) (nth 2 (ee-youtubedl-split fname)))
(defun ee-youtubedl-ext-  (fname) (nth 3 (ee-youtubedl-split fname)))
(defun ee-youtubedl-ext   (fname) (nth 4 (ee-youtubedl-split fname)))

;; Code for guessing the "title" and the "ext" parts of a video from
;; the "dir" and "hash" parts (in case the video has already been
;; downloaded).
;;
(defun ee-youtubedl-guess* (dir hash)
  "Return all the files in DIR containing the string HASH."
  (file-expand-wildcards (format "%s*%s*" dir hash)))

(defun ee-youtubedl-subtitle-p (fname)
  "Return true if FNAME looks like a subtitle file."
  (string-match "\\.\\(srt\\|vtt\\)$" fname))

(defun ee-youtubedl-not-subtitles (fnames)
  "Return the elements of FNAMES that don't look like subtitle files."
     (cl-remove-if 'ee-youtubedl-subtitle-p fnames))

(defun ee-youtubedl-guess (dir hash n)
  "Return a component of the first file in DIR containing the string HASH.
Files that look like subtitle files are ignored."
  (let* ((fnames0 (ee-youtubedl-guess* dir hash))
	 (fnames1 (ee-youtubedl-not-subtitles fnames0))
	 (fname (car fnames1)))
    (if fname (nth n (ee-youtubedl-split fname)))))

(defun ee-youtubedl-guess-title (dir hash) (ee-youtubedl-guess dir hash 1))
(defun ee-youtubedl-guess-ext-  (dir hash) (ee-youtubedl-guess dir hash 3))

(defun ee-youtubedl-hash-around-point ()
  (let ((hash (ee-stuff-around-point "-0-9A-Za-z_")))
    (if (>= (length hash) 11)
	(substring hash -11))))

;; «ee-youtubedl-command»  (to ".ee-youtubedl-command")
;; The default value is `ee-youtubedl-command' is (still) this one:
;;
(defvar ee-youtubedl-command "youtube-dl -t")
;;
;; but this is outdated in several ways:
;;
;;   1. when we use it we get this warning:
;;      "WARNING: --title is deprecated. Use
;;       -o "%(title)s-%(id)s.%(ext)s" instead."
;;
;;   2. youtube-dl is currently sort of dead. See:
;;      https://old.reddit.com/r/DataHoarder/comments/p9riey/youtubedl_is_possibly_dead/
;;      https://news.ycombinator.com/item?id=28289981 Youtube-dl is possibly dead
;;
;;   3. yt-dlp doesn't support the options `-t' and `--title':
;;      https://github.com/yt-dlp/yt-dlp
;;      https://github.com/yt-dlp/yt-dlp#Removed
;;
;; Suggestion (2021nov26): install yt-dlp and either put this in your .emacs,
;;
;;   ;; From: (find-eev "eev-tlinks.el" "ee-youtubedl-command")
;;   (setq ee-youtubedl-command "yt-dlp -o '%(title)s-%(id)s.%(ext)s'")
;;
;; and put a copy of this fake youtube-dl script
;;
;;   http://anggtwu.net/bin/youtube-dl.html
;;   http://anggtwu.net/bin/youtube-dl
;;
;; somewhere in your PATH, and use this:
;;
;;   (setq ee-youtubedl-command "youtube-dl -t")


;; The directories into which we usually download videos.
;; Tests: (find-elinks (ee-youtubedl-dir-links))
;;        (find-elinks (ee-youtubedl-dir-links '("DIR1" "DIR2")))
;;        (let ((ee-youtubedl-dirs '("DIR1" "DIR2"))) (find-youtubedl-links))
;;
(defvar ee-youtubedl-dir "~/videos/")

(defvar ee-youtubedl-dirs
  '("~/videos/" "~/videos/tech/" "/tmp/videos/" "/tmp/"))

(defun ee-youtubedl-dir-links (&optional dirs hash stem)
  (setq dirs (or dirs ee-youtubedl-dirs))
  (setq hash (or hash "{hash}"))
  (setq stem (or stem "{stem}"))
  (mapcar (lambda (dir) `(find-youtubedl-links ,dir nil ,hash nil ,stem))
	  dirs))

;; This is a hack (written in 2013sep10) that plays a local copy of a
;; video from its URL. To easiest way to use it is to put the point on
;; a youtube URL and type `M-x bryl'.
;;
(defun ee-youtubedl-guess** (dirs hash)
  (apply 'append (mapcar (lambda (d) (ee-youtubedl-guess* d hash)) dirs)))

(defun ee-youtubedl-hash-to-fname (hash)
  (and hash (car (ee-youtubedl-guess** ee-youtubedl-dirs hash))))

(defun ee-youtubedl-url-to-hash (url)
  (if (and url (string-match "[&?]v=\\([^&?#]+\\)" url))
      (match-string 1 url)
    url))

(defun ee-youtubedl-url-to-fname (url)
  (ee-youtubedl-hash-to-fname
   (ee-youtubedl-url-to-hash url)))

(autoload 'browse-url-url-at-point "browse-url")

;; TODO:
;; Redefine this using: (find-eevfile "eev-brxxx.el")
;; and add error handling. Tests:
;;   (setq ee-my-url "http://www.youtube.com/watch?v=tby5aMrMu6Q")
;;   (ee-youtubedl-url-to-hash ee-my-url)
;;   (ee-youtubedl-url-to-fname ee-my-url)
;;   (find-evardescr 'ee-youtubedl-dirs)
;;
;; (defun bryl (url)
;;   "Play a local copy of a video downloaded from youtube."
;;   (interactive (list (browse-url-url-at-point)))
;;   (let ((fname (and url (ee-youtubedl-url-to-fname url))))
;;     (if fname (find-video fname))))





;;;                       
;;;  _ __  ___ _ __   ___ 
;;; | '_ \/ __| '_ \ / _ \
;;; | |_) \__ \ | | |  __/
;;; | .__/|___/_| |_|\___|
;;; |_|                   
;;
;; «find-psne-links» (to ".find-psne-links")
;; Skel: (find-find-links-links-new "psne" "url wget-options echo-options" "")
;; Test: (find-psne-links "http://foo/bar")
;;       (find-psne-links "http://foo/bar" "-nc" "-N")
;;  See: (find-psne-intro "3. The new way: `M-x brep'")
;;       (find-psne-intro "3. The new way: `M-x brep'" "find-psne-links")
;;       (find-windows-beginner-intro "5.6.1. Echo in Eshell")
;;
(defun find-psne-links (&optional url wget-options echo-options &rest pos-spec-list)
"See: (find-psne-intro)"
  (interactive)
  (setq url (or url "{url}"))
  (setq wget-options (or wget-options "-nc"))
  (setq echo-options (or echo-options (ee-find-psne-echo-options)))
  (apply
   'find-elinks
   `((find-psne-links ,url ,wget-options ,echo-options ,@pos-spec-list)
     (find-psne-links ,url "-c" "" ,@pos-spec-list)
     (find-psne-links ,url "" "" ,@pos-spec-list)
     ;; Convention: the first sexp always regenerates the buffer.
     (find-efunction 'find-psne-links)
     ""
     ,(ee-adjust-red-stars " (eepitch-shell2)")
     ,(ee-find-psne-core url wget-options echo-options)
     )
   pos-spec-list))

;; Tests:
;; (find-estring (ee-find-psne-core "http://www.lua.org/start.html"))
;; (find-estring (ee-find-psne-core "http://www.lua.org/start.html" "-nc" "-N"))
;;
(defun ee-find-psne-core (url &optional wget-options echo-options)
  "This is an internal function used by `find-psne-links'."
  (let* ((localurl (ee-url-to-fname0 url))
	 (localdir (file-name-directory localurl))
	 (fname0 (file-name-nondirectory localurl))
	 (o (format "%-3s" (or wget-options "")))
	 (e (format "%-3s" (or echo-options
			       (ee-find-psne-echo-options)))))
    (ee-template0 "\
mkdir -p {localdir}
cd       {localdir}
wget {o} '{url}'
echo {e} '{url}' >> ~/.psne.log

# (find-fline \"{localdir}\")
# (find-fline \"{localdir}\" \"{fname0}\")
# (find-fline \"{localurl}\")
")))


;; «ee-find-psne-echo-options»  (to ".ee-find-psne-echo-options")
;; See: (find-windows-beginner-intro "5.6.1. Echo in Eshell")
(defun ee-find-psne-echo-options ()
  (if (eq system-type 'windows-nt) "-N" ""))




;;;                                                    _     _            
;;;  _ __  ___ _ __   ___        ___  _____   ____   _(_) __| | ___  ___  
;;; | '_ \/ __| '_ \ / _ \_____ / _ \/ _ \ \ / /\ \ / / |/ _` |/ _ \/ _ \ 
;;; | |_) \__ \ | | |  __/_____|  __/  __/\ V /  \ V /| | (_| |  __/ (_) |
;;; | .__/|___/_| |_|\___|      \___|\___| \_/    \_/ |_|\__,_|\___|\___/ 
;;; |_|                                                                   
;;
;; «find-psne-eevvideo-links»  (to ".find-psne-eevvideo-links")
;; Skel:  (find-find-links-links-new "psne-eevvideo" "stem exts time" "")
;; Tests: (find-psne-eevvideo-links "NAMEOFTHEVIDEO" ".srt .vtt")
;;        (find-psne-eevvideo-links "NAMEOFTHEVIDEO" ".srt")
;;        (find-psne-eevvideo-links "NAMEOFTHEVIDEO" "")
;;        (find-psne-eevvideo-links "NAMEOFTHEVIDEO" "" "1:23")
;; See:  (find-video-links-intro "5.1. Subtitles")
;;
;; NOTE: `find-psne-eevvideo-links', `ee-psne-eevvideo-core',
;;       `find-psne-1stclassvideo-links',
;;       `ee-psne-1stclassvideo-play', and `find-1stclassvideo-video'
;;       were written in 2022may11 and will probably replace several
;;       older functions soon... these new functions have support for
;;       subtitles, and the older ones didn't.
;;
(defun find-psne-eevvideo-links (&optional stem exts time &rest pos-spec-list)
"Visit a temporary buffer containing a script for downloading a video."
  (interactive)
  (setq stem (or stem "{stem}"))
  (setq exts (or exts "{exts}"))
  (setq time (or time "0:00"))
  (apply
   'find-elinks
   `((find-psne-eevvideo-links ,stem ,exts ,time ,@pos-spec-list)
     ;; Convention: the first sexp always regenerates the buffer.
     (find-efunction 'find-psne-eevvideo-links)
     ""
     ,(ee-psne-eevvideo-core stem exts time)
     )
   pos-spec-list))

;; Tests:
;; (find-estring (ee-psne-eevvideo-core "NAME" ".vtt .srt" "0:00"))
;; (find-estring (ee-psne-eevvideo-core "NAME" ".vtt"      "0:00"))
;; (find-estring (ee-psne-eevvideo-core "NAME" ""          "0:00"))
;; (defun ee-find-psne-echo-options () "-N")
;; (defun ee-find-psne-echo-options () "")
;;
(defun ee-psne-eevvideo-core (stem exts time)
  "An internal function used by `find-psne-eevvideo-links'."
  (let* ((dir         "$S/http/anggtwu.net/eev-videos/")
         (url- (concat "http://anggtwu.net/eev-videos/" stem))
	 (eo (format "%-3s" (ee-find-psne-echo-options)))
         (wf (lambda (ext) (format "wget -N   %s%s\n" url- ext)))
         (ef (lambda (ext) (format "echo %s  %s%s >> ~/.psne.log\n" eo url- ext)))
         (wgets (mapconcat wf (split-string exts) ""))
         (echos (mapconcat ef (cons ".mp4" (split-string exts)) ""))
	 )
    (ee-template0 "\
# (find-psne-intro \"1. Local copies of files from the internet\")
# (find-video-links-intro \"5. Local copies\")
# (find-video-links-intro \"5.1. Subtitles\")
# http://anggtwu.net/eev-videos.html

 (sh-mode)
 (eepitch-shell2)
 (eepitch-kill)
 (eepitch-shell2)
mkdir -p {dir}
cd       {dir}
wget -nc  {url-}.mp4
{wgets}\
{echos}\

# (find-fline \"{dir}\" \"{stem}.mp4\")
# (find-video \"{dir}{stem}.mp4\")
# (find-video \"{dir}{stem}.mp4\" \"{time}\")
")))


;;;                             _     _       _               
;;;  _ __  ___ _ __   ___      / |___| |_ ___| | __ _ ___ ___ 
;;; | '_ \/ __| '_ \ / _ \_____| / __| __/ __| |/ _` / __/ __|
;;; | |_) \__ \ | | |  __/_____| \__ \ || (__| | (_| \__ \__ \
;;; | .__/|___/_| |_|\___|     |_|___/\__\___|_|\__,_|___/___/
;;; |_|                                                       
;;
;; «find-psne-1stclassvideo-links»  (to ".find-psne-1stclassvideo-links")
;; Skel: (find-find-links-links-new "psne-1stclassvideo" "c time" "stem subexts")
;; Test: (find-psne-1stclassvideo-links "eev2021")
;;       (find-psne-1stclassvideo-links "2021workshop6")
;;
(defun find-psne-1stclassvideo-links (&optional c time &rest pos-spec-list)
"Visit a temporary buffer with a script for downloading the video C."
  (interactive)
  (setq c (or c "{c}"))
  (setq time (or time "0:00"))
  (let* ((stem    (ee-1stclassvideos-mp4stem c))
         (subexts (or (ee-1stclassvideos-field c :subs) "")))
    (apply
     'find-elinks
     `((find-psne-1stclassvideo-links ,c ,time ,@pos-spec-list)
       ;; Convention: the first sexp always regenerates the buffer.
       (find-efunction 'find-psne-1stclassvideo-links)
       ""
       ,(ee-psne-eevvideo-core stem subexts time)
       ,(ee-psne-1stclassvideo-play c time)
       )
     pos-spec-list)))

(defalias 'find-1stclassvideo-psne 'find-psne-1stclassvideo-links)

;; Test:
;; (find-estring (ee-psne-1stclassvideo-play "eev2021" "1:23"))
;;
(defun ee-psne-1stclassvideo-play (c time)
  "An internal function used by `find-psne-1sclassvideo-links'."
  (let* ((localmp4 (ee-1stclassvideos-localmp4 c))
	 (yt       (ee-1stclassvideos-field    c :yt))
	 (timearg  (ee-time-to-arg          time))
         (yttime   (ee-time-to-youtube-time time))
	 )
    (ee-template0 "\
# (find-1stclassvideo-video \"{c}\" \"{time}\")
# (find-{c}video \"{time}\")

# {yt}{yttime}
")))


;; «find-1stclassvideo-video»  (to ".find-1stclassvideo-video")
;; Tests: (find-1stclassvideo-video "eev2021")
;;        (find-eevvideossh0 "mv -v emacsconf2021.mp4 /tmp/")
;;        (find-eevvideossh0 "mv -v /tmp/emacsconf2021.mp4 .")
;;
(defun find-1stclassvideo-video (c &optional time &rest rest)
  "Play the first-class video C starting at TIME.
If a local copy of C is not found, run `find-psne-1stclassvideo-links'."
  (setq time (or time "0:00"))
  (if (ee-1stclassvideos-mp4found c)
      (find-video (ee-1stclassvideos-localmp4 c) time)
    (find-psne-1stclassvideo-links c time)))








;;;        _ _   
;;;   __ _(_) |_ 
;;;  / _` | | __|
;;; | (_| | | |_ 
;;;  \__, |_|\__|
;;;  |___/       

;; «find-git-links» (to ".find-git-links")
;; (find-find-links-links "g" "git" "usrc/ git/ gitname")
;; (find-find-links-links "g" "git" "url c")

(defvar ee-git-dir "~/usrc/" "See `find-git-links'.")

(defun ee-git-url-stem (url)
  (replace-regexp-in-string "^\\(.*/\\)\\([^/]+?\\)\\(\\.git\\)?$" "\\2" url))

(defun ee-git-url-at-point ()
  (require 'thingatpt)
  (let ((thing-at-point-url-regexp
	 (concat "\\<\\(https?:\\|git:\\)"
		 thing-at-point-url-path-regexp)))
    (thing-at-point 'url)))

(defun find-git-links (&optional url c &rest pos-spec-list)
"Visit a temporary buffer containing a script for downloading a git repo."
  (interactive)
  (let (gitstem dir)
    (setq url (or url (ee-git-url-at-point) "{url}"))
    (setq gitstem (or gitstem (ee-git-url-stem url)))
    (setq c (or c (replace-regexp-in-string "[-.]" "" gitstem)))
    (setq dir (format "%s%s/" ee-git-dir gitstem))
    (apply 'find-elinks
     `((find-git-links ,url ,c)
       ;; Convention: the first sexp always regenerates the buffer.
       (find-efunction 'find-git-links)
       ""
       (setq ee-git-dir ,ee-git-dir)
       (setq ee-git-dir "~/usrc/")
       (setq ee-git-dir "~/bigsrc/")
       (setq ee-git-dir "/tmp/")
       ""
       (find-fline ,ee-git-dir)
       (find-fline ,dir)
       ""
       ,(ee-template0 "\
# {url}

 (eepitch-shell)
 (eepitch-kill)
 (eepitch-shell)
# rm -Rfv {dir}
cd      {ee-git-dir}
git clone {url}
cd      {dir}

export PAGER=cat
git branch --list -a
git for-each-ref
git log --oneline --graph --all -20

# (find-fline \"{ee-git-dir}\")
# (find-fline \"{dir}\")
# (find-gitk  \"{dir}\")

# (code-c-d \"{c}\" \"{dir}\")
# (find-{c}file \"\")

# git pull --depth 1
# git pull
# git reset
# git clean -dfx
# git reset --hard
")
     )
   pos-spec-list)))

;; Test:
;; (find-git-links "https://github.com/kikito/inspect.lua" "inspectlua")




;; «find-fossil-links»  (to ".find-fossil-links")
;; Skel: (find-find-links-links-new "fossil" "url subdir c" "")
;; Test: (find-fossil-links "http://fossil.0branch.com/oorexx-mode")
;; See:  https://fossil-scm.org/
;;
(defun ee-fossil-url-stem (url) (ee-git-url-stem url))

(defun find-fossil-links (&optional url subdir c &rest pos-spec-list)
"Visit a temporary buffer containing hyperlinks for fossil."
  (interactive (list (ee-url-at-point)))
  (setq url (or url "{url}"))
  (setq subdir (or subdir (ee-fossil-url-stem url) "{subdir}"))
  (setq c (or c (replace-regexp-in-string "[-.]" "" subdir) "{c}"))
  (apply
   'find-elinks
   `((find-fossil-links ,url ,subdir ,c ,@pos-spec-list)
     (find-fossil-links "{url}" "{subdir}" "{c}")
     ;; Convention: the first sexp always regenerates the buffer.
     (find-efunction 'find-fossil-links)
     ""
     ,(ee-template0 "\
# (find-sh \"fossil help\")
# (find-sh \"fossil help clone\")
# (find-sh \"fossil help pull\")
# (find-sh \"fossil help all\")
# (find-sh \"fossil help open\")
# (find-es \"fossil\")

 (eepitch-shell)
 (eepitch-kill)
 (eepitch-shell)
# rm -fv ~/usrc/fossil-repos/{subdir}.fsl
mkdir -p ~/usrc/fossil-repos/
cd       ~/usrc/fossil-repos/

fossil clone {url}    {subdir}.fsl
fossil pull  {url} -R {subdir}.fsl
fossil all ls

# cd        ~/usrc/{subdir}/ && fossil close
# rm -Rfv   ~/usrc/{subdir}/
mkdir -p    ~/usrc/{subdir}/
cd          ~/usrc/{subdir}/
fossil open ~/usrc/fossil-repos/{subdir}.fsl

# (code-c-d \"{c}\" \"~/usrc/{subdir}/\")
# (find-{c}file \"\")
")
     )
   pos-spec-list))





;;;              _                   _                                     
;;;   __ _ _ __ | |_       __ _  ___| |_      ___  ___  _   _ _ __ ___ ___ 
;;;  / _` | '_ \| __|____ / _` |/ _ \ __|____/ __|/ _ \| | | | '__/ __/ _ \
;;; | (_| | |_) | ||_____| (_| |  __/ ||_____\__ \ (_) | |_| | | | (_|  __/
;;;  \__,_| .__/ \__|     \__, |\___|\__|    |___/\___/ \__,_|_|  \___\___|
;;;       |_|             |___/                                            
;;
;; «find-apt-get-source-links»  (to ".find-apt-get-source-links")
;; (find-find-links-links "{k}" "apt-get-source" "pkg")
;; A test: (find-apt-get-source-links)

(defun find-apt-get-source-links (&optional pkg &rest pos-spec-list)
"Visit a temporary buffer containing a script for apt-get source."
  (interactive)
  (setq pkg (or pkg "{pkg}"))
  (let ((letter (replace-regexp-in-string "^\\(\\(lib\\)?.\\).*" "\\1" pkg)))
    (apply 'find-elinks
     `((find-apt-get-source-links ,pkg ,@pos-spec-list)
       (find-apt-get-source-links "lua5.1")
       ;; Convention: the first sexp always regenerates the buffer.
       (find-efunction 'find-apt-get-source-links)
       ""
       ,(ee-template0 "\
# https://packages.debian.org/search?searchon=sourcenames&keywords={pkg}
# https://packages.debian.org/source/sid/{pkg}
# http://deb.debian.org/debian/pool/main/{letter}/{pkg}/

 (eepitch-shell)
 (eepitch-kill)
 (eepitch-shell)
# (find-sh \"apt-cache show    {pkg}\")
# (find-sh \"apt-cache showsrc {pkg}\")
rm -Rv /tmp/d/
mkdir  /tmp/d/
cd     /tmp/d/
sudo apt-get build-dep -y   {pkg}
     apt-get source         {pkg}   2>&1 | tee osource
     apt-get source --build {pkg}   2>&1 | tee osourceb

# (find-fline \"/tmp/d/\")

")
       )
     pos-spec-list)))

;; Test: (find-apt-get-source-links)






;;;             _            _        _            _   
;;;  _ __   ___| |_ ___ __ _| |_     | |_ ___  ___| |_ 
;;; | '_ \ / _ \ __/ __/ _` | __|____| __/ _ \/ __| __|
;;; | | | |  __/ || (_| (_| | ||_____| ||  __/\__ \ |_ 
;;; |_| |_|\___|\__\___\__,_|\__|     \__\___||___/\__|
;;;                                                    

;; «find-netcat-test-links» (to ".find-netcat-test-links")
;; (find-find-links-links "{k}" "netcat-test" "eesrc eetgt tgtname tgtport")
;; A test: (find-netcat-test-links)

(defun find-netcat-test-links (&optional eesrc eetgt tgtname tgtport &rest pos-spec-list)
"Visit a temporary buffer with a script to test sending data though netcat."
  (interactive)
  (setq eesrc (or eesrc "{eesrc}"))
  (setq eetgt (or eetgt "{eetgt}"))
  (setq tgtname (or tgtname "{tgtname}"))
  (setq tgtport (or tgtport "{tgtport}"))
  (apply 'find-elinks
   `((find-netcat-test-links ,eesrc ,eetgt ,tgtname ,tgtport)
     (find-netcat-test-links "shell" "shell2" "localhost" "1234")
     ;; Convention: the first sexp always regenerates the buffer.
     (find-efunction 'find-netcat-test-links)
     ""
     ,(ee-template0 "\
 (find-3EE '(eepitch-{eesrc}) '(eepitch-{eetgt}))
 (find-3ee '(eepitch-{eesrc}) '(eepitch-{eetgt}))
 (eepitch-{eetgt})
# listen on port {tgtport}
netcat -l -p {tgtport}

 (eepitch-{eesrc})
# Send things to port {tgtport} (on {tgtname})
{<}
  echo hi
  sleep 1
  echo bye
  sleep 1
{>} | netcat -q 0 {tgtname} {tgtport}

")
     )
   pos-spec-list))

;; Test: (find-netcat-test-links)






;;;                         _     _            
;;;   ___  _____   ____   _(_) __| | ___  ___  
;;;  / _ \/ _ \ \ / /\ \ / / |/ _` |/ _ \/ _ \ 
;;; |  __/  __/\ V /  \ V /| | (_| |  __/ (_) |
;;;  \___|\___| \_/    \_/ |_|\__,_|\___|\___/ 
;;;                                            
;; «find-eevvideo-links»  (to ".find-eevvideo-links")
;; Used by: (find-videos-intro "2. Short links to eev video tutorials")
;;          (find-eev "eev-audiovideo.el" "video-tutorials")
;;     See: (find-audiovideo-intro "7.2. `find-eevvideo-links'")
;;    Skel: (find-find-links-links-new "eevvideo" "c stem youtubeid time" "url")
;;    Test: (find-eevvideo-links "eevnav" "M-x-list-packages-eev-nav" "0:00")
;;
(defun find-eevvideo-links (&optional c stem youtubeid time &rest pos-spec-list)
"Visit a temporary buffer containing hyperlinks for eevvideo."
  (interactive)
  (setq c (or c "{c}"))
  (setq stem (or stem "{stem}"))
  (setq youtubeid (or youtubeid "{youtubeid}"))
  (setq time (or time "{time}"))
  (let* ((url (format "http://anggtwu.net/eev-videos/%s.mp4" stem)))
    (apply
     'find-elinks
     `((find-eevvideo-links ,c ,stem ,youtubeid ,time ,@pos-spec-list)
       ;; Convention: the first sexp always regenerates the buffer.
       ;; (find-efunction 'find-eevvideo-links)
       ,(ee-psnevideo-links c url youtubeid time)
       )
     pos-spec-list)))



;;;                             _     _            
;;;  _ __  ___ _ __   _____   _(_) __| | ___  ___  
;;; | '_ \/ __| '_ \ / _ \ \ / / |/ _` |/ _ \/ _ \ 
;;; | |_) \__ \ | | |  __/\ V /| | (_| |  __/ (_) |
;;; | .__/|___/_| |_|\___| \_/ |_|\__,_|\___|\___/ 
;;; |_|                                            
;;
;; «find-psnevideo-links»  (to ".find-psnevideo-links")
;; Skel: (find-find-links-links-new "psnevideo" "c url youtubeid time" "url")
;; Tests: (find-psnevideo-links "C" "http://foo.org/bar.mp4")
;;        (find-psnevideo-links "C" "http://foo.org/bar.mp4" nil "1:23")
;;
(defun find-psnevideo-links (&optional c url youtubeid time &rest pos-spec-list)
"Visit a temporary buffer containing an e-script for downloading and playing URL."
  (interactive)
  (setq c (or c "{c}"))
  (setq url (or url "{url}"))
  (setq youtubeid (or youtubeid "{youtubeid}"))
  (setq time (or time "{time}"))
  (apply
   'find-elinks
   `((find-psnevideo-links ,c ,url ,youtubeid ,time ,@pos-spec-list)
     ,(ee-psnevideo-links c url youtubeid time)
     )
   pos-spec-list))

;; Tests:
;; (find-estring (ee-psnevideo-links "C" "http://foo.org/bar.mp4"))
;; (find-estring (ee-psnevideo-links "C" "http://foo.org/bar.mp4" nil "1:23"))
;; (find-estring (ee-psnevideo-links "C" "http://foo.org/bar.mp4" "YID" "1:23"))
;;
(defun ee-psnevideo-links (c url &optional youtubeid time)
  (setq youtubeid (or youtubeid "{youtubeid}"))
  (setq time      (or time      "{time}"))
  (let* ((fname   (ee-shorten-file-name (ee-url-to-fname url)))
	 (argtime (ee-time-to-arg time)))
    (concat
     ;;
     ;; See:
     ;; (find-audiovideo-intro "7. `code-psnevideo'" "second line")
     ;; (find-audiovideo-intro "7. `code-psnevideo'" "redefine")
     ;; (find-audiovideo-intro "7. `code-psnevideo'" "last part")
     ;;
     (ee-template0 "\
# (find-video \"{fname}\"{argtime})

# (find-audiovideo-intro \"7. `code-psnevideo'\" \"second line\")
# (find-audiovideo-intro \"7. `code-psnevideo'\" \"redefine\")
#   (code-video \"{c}video\" \"{fname}\")
#   (find-{c}video{argtime})

")
     (ee-psnevideo-url-youtube url youtubeid time)
     "\n\n"
     "# (find-audiovideo-intro \"7. `code-psnevideo'\" \"last part\")\n"
     "\n\n"
     (ee-psne-if-needed url)
     )))



;; Tests:
;; (find-estring (ee-psnevideo-url-youtube "http://foo.org/bar.mp4"))
;; (find-estring (ee-psnevideo-url-youtube "http://foo.org/bar.mp4" nil "1:23"))
;;
(defun ee-psnevideo-url-youtube (url &optional youtubeid time)
  (setq youtubeid (or youtubeid "{youtubeid}"))
  (setq time      (or time      "{time}"))
  (let* ((fname       (ee-shorten-file-name (ee-url-to-fname url)))
         (fname0      (file-name-nondirectory fname))
         (dir         (file-name-directory fname))
	 (youtubeurl  (format "http://www.youtube.com/watch?v=%s" youtubeid))
	 (youtubetime (or (ee-time-to-youtube-time time) "")))
    (ee-template0 "\
# URL, local file, and links to the directory of the local file:
#               {url}
#              {fname}
# (find-fline \"{dir}\" \"{fname0}\")
# (find-fline \"{dir}\")

# Youtube:
# (kill-new \"{youtubeurl}{youtubetime}\")
#            {youtubeurl}{youtubetime}")
    ))




;; «ee-psne-if-needed»  (to ".ee-psne-if-needed")
;; Tests:
;;               (ee-psne-if-needed "https://www.lua.org/index.html")
;; (find-estring (ee-psne-if-needed "https://www.lua.org/index.html"))
;; (find-estring (ee-psne-if-needed "http://www.foo.org/bar.html"))
;; (find-estring (ee-psne-download  "http://www.foo.org/bar.html"))
;; (find-estring (ee-psne-download0 "http://www.foo.org/bar.html"))
;;
(defun ee-psne-if-needed (url)
  (if (ee-psne-downloaded-p url)
      "# Local file found. No need to download it again.\n"
    (ee-adjust-red-stars (ee-psne-download url))))

(defun ee-psne-downloaded-p (url)
  (file-exists-p (ee-expand (ee-url-to-fname url))))

(defun ee-psne-download (url)
  (concat "\
#  Local file not found! 
#  You need to run this: \n
" (ee-psne-download0 url)))

(defun ee-psne-download0 (url)
  (let* ((fname  (ee-shorten-file-name (ee-url-to-fname url)))
         (fname0 (file-name-nondirectory fname))
         (dir    (file-name-directory fname)))
    (ee-template0 "\
 (eepitch-shell)
 (eepitch-kill)
 (eepitch-shell)
mkdir -p {dir}
cd       {dir}
wget -nc '{url}'
echo     '{url}' >> ~/.psne.log

# (find-fline \"{dir}\")
# (find-fline \"{dir}\" \"{fname0}\")
# See: (find-psne-intro)
")))

;; Test: (find-estring (ee-psne-url-comment "http://www.foo.org/bar.html"))
(defun ee-psne-url-comment (url)
  (let* ((fname  (ee-shorten-file-name (ee-url-to-fname url)))
         (fname0 (file-name-nondirectory fname))
         (dir    (file-name-directory fname)))
    (ee-template0 "\
# URL, local file, and links to the directory of the local file:
#               {url}
#              {fname}
# (find-fline \"{dir}\" \"{fname0}\")
# (find-fline \"{dir}\")
")))







;;;                _                                       _     _            
;;;   ___ ___   __| | ___       _ __  ___ _ __   _____   _(_) __| | ___  ___  
;;;  / __/ _ \ / _` |/ _ \_____| '_ \/ __| '_ \ / _ \ \ / / |/ _` |/ _ \/ _ \ 
;;; | (_| (_) | (_| |  __/_____| |_) \__ \ | | |  __/\ V /| | (_| |  __/ (_) |
;;;  \___\___/ \__,_|\___|     | .__/|___/_| |_|\___| \_/ |_|\__,_|\___|\___/ 
;;;                            |_|                                            
;;
;; «code-psnevideo»  (to ".code-psnevideo")
;; See: (find-audiovideo-intro "7. `code-psnevideo'")

(defun      code-psnevideo (c url &optional youtubeid)
  (eval (ee-read (ee-code-psnevideo c url youtubeid))))
(defun find-code-psnevideo (&optional c url youtubeid &rest rest)
  (setq c (or c "{c}"))
  (setq url (or url "{url}"))
  (setq youtubeid (or youtubeid "{youtubeid}"))
  (find-estring-elisp (apply 'ee-code-psnevideo c url youtubeid rest)))
(defun   ee-code-psnevideo (c url youtubeid)
  (ee-template0 "\
;; (find-code-psnevideo \"{c}\" \"{url}\" \"{youtubeid}\")
;;      (code-psnevideo \"{c}\" \"{url}\" \"{youtubeid}\")
;;                 (find-{c}video \"0:00\")

(defun find-{c}video (&optional time &rest rest)
  (interactive)
  (find-psnevideo-links \"{c}\" \"{url}\" \"{youtubeid}\" time))
"))





;; The functions below were totally rewritten, and their new definions
;; are here:

;; ;;;                _                                   _     _            
;; ;;;   ___ ___   __| | ___        ___  _____   ____   _(_) __| | ___  ___  
;; ;;;  / __/ _ \ / _` |/ _ \_____ / _ \/ _ \ \ / /\ \ / / |/ _` |/ _ \/ _ \ 
;; ;;; | (_| (_) | (_| |  __/_____|  __/  __/\ V /  \ V /| | (_| |  __/ (_) |
;; ;;;  \___\___/ \__,_|\___|      \___|\___| \_/    \_/ |_|\__,_|\___|\___/ 
;; ;;;                                                                       
;; ;; «code-eevvideo»  (to ".code-eevvideo")
;; ;; TODO: rewrite this reusing parts of:
;; ;;     (find-eev "eev-videolinks.el" "find-eevlocal-links")
;; ;; See: (find-audiovideo-intro "7.1. `code-eevvideo'")
;; ;; Test: (find-code-eevvideo "eevnav" "M-x-list-packages-eev-nav")
;; ;;            (code-eevvideo "eevnav" "M-x-list-packages-eev-nav")
;; ;;                      (find-eevnavvideo "0:00")
;; ;;
;; (defun      code-eevvideo (c stem &optional youtubeid)
;;   (eval (ee-read (ee-code-eevvideo c stem youtubeid))))
;; (defun find-code-eevvideo (&optional c stem youtubeid &rest rest)
;;   (setq c (or c "{c}"))
;;   (setq stem (or stem "{stem}"))
;;   (setq youtubeid (or youtubeid "{youtubeid}"))
;;   (find-estring-elisp (apply 'ee-code-eevvideo c stem youtubeid rest)))
;; (defun   ee-code-eevvideo (c stem youtubeid)
;;   (ee-template0 "\
;; ;; (find-code-eevvideo \"{c}\" \"{stem}\" \"{youtubeid}\")
;; ;;      (code-eevvideo \"{c}\" \"{stem}\" \"{youtubeid}\")
;; ;;                (find-{c}video \"0:00\")
;; ;;
;; ;; See: (find-audiovideo-intro \"7.1. `code-eevvideo'\")
;; 
;; (defun find-{c}video (&optional time &rest rest)
;;   (interactive)
;;   (find-eevvideo-links \"{c}\" \"{stem}\" \"{youtubeid}\" time))
;; "))
;; 
;; ;; «code-eevvideo-local»  (to ".code-eevvideo-local")
;; ;; Test: (code-eevvideo-local "vlinks" "2021-video-links" "xQqWufQgzVY")
;; ;;  (find-code-eevvideo-local "vlinks" "2021-video-links" "xQqWufQgzVY")
;; ;;
;; (defun      code-eevvideo-local (c stem &optional youtubeid)
;;   (eval (ee-read (ee-code-eevvideo-local c stem youtubeid))))
;; (defun find-code-eevvideo-local (&optional c stem youtubeid &rest rest)
;;   (setq c (or c "{c}"))
;;   (setq stem (or stem "{stem}"))
;;   (setq youtubeid (or youtubeid "{youtubeid}"))
;;   (find-estring-elisp (apply 'ee-code-eevvideo-local c stem youtubeid rest)))
;; (defun   ee-code-eevvideo-local (c stem youtubeid)
;;   (ee-template0 "\
;; ;; (find-code-eevvideo-local \"{c}\" \"{stem}\" \"{youtubeid}\")
;; ;;      (code-eevvideo-local \"{c}\" \"{stem}\" \"{youtubeid}\")
;; ;;
;; ;;                    (find-fline \"$S/http/anggtwu.net/eev-videos/\" \"{stem}\")
;; ;; (find-code-video \"{c}video\" \"$S/http/anggtwu.net/eev-videos/{stem}.mp4\")
;;         (code-video \"{c}video\" \"$S/http/anggtwu.net/eev-videos/{stem}.mp4\")
;; ;;             (find-{c}video \"0:00\")
;; "))



;; «hardcoded-paths»  (to ".hardcoded-paths")
;;
;; The definitions of `find-eevvideo-links' and `code-eevvideo' above
;; have strings like "eev" and "http://anggtwu.net/eev-videos/"
;; hardcoded in several places... it is easy, but not entirely
;; trivial, to create variants of them that point to other sites that
;; stores video tutorials and presentations in ways that are easy to
;; download. If you are aware of sites like that, please get in touch
;; and I'll create functions pointing to them!

;;;                            _     _            
;;;   ___  _____   __   __   _(_) __| | ___  ___  
;;;  / _ \/ _ \ \ / /___\ \ / / |/ _` |/ _ \/ _ \ 
;;; |  __/  __/\ V /_____\ V /| | (_| |  __/ (_) |
;;;  \___|\___| \_/       \_/ |_|\__,_|\___|\___/ 
;;;                                               
;; «find-eev-video-links» (to ".find-eev-video-links")
;; Obsolete?
;; See: (find-audiovideo-intro "7.2. `find-eevvideo-links'")
;; and: (find-video-links-intro)
;;
;; Skel: (find-find-links-links "{k}" "eev-video" "c anggstem youtubehash")
;;       (find-find-links-links-new "eev-video" "c anggstem youtubehash" "")
;;
(defun find-eev-video-links (&optional c anggstem youtubehash &rest pos-spec-list)
"Visit a temporary buffer containing a script for downloading an eev video.
See: (find-videos-intro)
Examples:
  (find-eev-video-links \"eepitchvideo\" \"video4-eepitch\" \"Lj_zKC5BR64\")
  (find-eev-video-links \"eevvideo2\"    \"video2\"         \"doeyn5MOaB8\")
  (find-eev-video-links \"eevvideo2-pt\" \"video2pt\"       \"yztYD9Y7Iz4\")
Warning: the last one is in Portuguese..."
  (interactive)
  (setq c (or c "{c}"))
  (setq anggstem (or anggstem "{anggstem}"))
  (setq youtubehash (or youtubehash "{youtubehash}"))
  (let ((s (replace-regexp-in-string "." " " c)))
    (apply 'find-elinks
     `((find-eev-video-links ,c ,anggstem ,youtubehash)
       ;; Convention: the first sexp always regenerates the buffer.
       (find-efunction 'find-eev-video-links)
       ""
       ,(ee-template0 "\
 Download (or make sure we have) a local copy of the video:
 (eepitch-shell2)
 (eepitch-kill)
 (eepitch-shell2)
mkdir -p $S/http/anggtwu.net/eev-videos/
cd       $S/http/anggtwu.net/eev-videos/
wget -nc 'http://anggtwu.net/eev-videos/{anggstem}.mp4'
echo     'http://anggtwu.net/eev-videos/{anggstem}.mp4' >> ~/.psne.log

# Test:
# (find-fline  {s}  \"$S/http/anggtwu.net/eev-videos/\")
# (find-video  {s}  \"$S/http/anggtwu.net/eev-videos/{anggstem}.mp4\")
# (code-video \"{c}\" \"$S/http/anggtwu.net/eev-videos/{anggstem}.mp4\")
# (find-{c})
# Error messages:
# (find-ebuffer \"*Messages*\")

# See: (find-audiovideo-intro \"eev-avadj-mode\")
#      (find-audiovideo-intro \"The time-from-bol\")
# (eev-avadj-mode 1)
# (find-{c} t)
# 0:00

# Links to the version at youtube:
# http://www.youtube.com/watch?v={youtubehash}
# http://www.youtube.com/watch?v={youtubehash}&t=0m00s
# http://www.youtube.com/watch?v={youtubehash}&t=0h00m00s
")
       )
     pos-spec-list)))

;; Links to all the eev videos (best ones first):
;;   (find-eev-video-links "eepitchvideo" "video4-eepitch" "Lj_zKC5BR64")
;;   (find-eev-video-links "eevvideo2"    "video2"         "doeyn5MOaB8")
;;   (find-eev-video-links "eevvideo2-pt" "video2pt"       "yztYD9Y7Iz4")
;; The ones with "pt" are in Portuguese, the other ones are in English.

;; (find-eepitchvideo "0:18" "Demonstration (first time, very quickly)")



;; «find-eevshortvideo-links»  (to ".find-eevshortvideo-links")
;; Skel: (find-find-links-links-new "eevshortvideo" "c stem youtubeid" "")
;; Used in: (find-eev "eev-audiovideo.el" "video-tutorials")
;;
(defun find-eevshortvideo-links (&optional c stem youtubeid &rest pos-spec-list)
"Visit a temporary buffer containing hyperlinks for eevshortvideo."
  (interactive)
  (setq c (or c "{c}"))
  (setq stem (or stem "{stem}"))
  (setq youtubeid (or youtubeid "{youtubeid}"))
  (apply
   'find-elinks-elisp
   `((find-eevshortvideo-links ,c ,stem ,youtubeid ,@pos-spec-list)
     ;; Convention: the first sexp always regenerates the buffer.
     (find-efunction 'find-eevshortvideo-links)
     ""
     ,(ee-template0 "\
;; (find-1stclassvideo-links \"{c}\")

;; Skel: (find-eevshortvideo-links \"{c}\" \"{stem}\" \"{youtubeid}\")
;;  See: (find-videos-intro \"1. Some videos\" \"{stem}\")
;; Index: http://anggtwu.net/.emacs.videos.html#{c}
;;  Test: (find-{c}video \"0:00\")
(defun find-{c}video (&optional time &rest rest)
  \"Play one of the video tutorials of eev starting at TIME.
See: (find-videos-intro \\\"1. Some videos\\\" \\\"{stem}\\\")
     http://anggtwu.net/{stem}.html
     for more info on this particular video,
and: (find-video-links-intro \\\"7. `find-eev-video'\\\")
 or: http://anggtwu.net/eev-intros/find-video-links-intro.html#7
     for more info on these video tutorials.\"
  (interactive)
  (find-eev-video \"{stem}\" \"{youtubeid}\" time))
")
     )
   pos-spec-list))



;; «find-wgeteevsubtitles-links»  (to ".find-wgeteevsubtitles-links")
;; Skel:  (find-find-links-links-new "wgeteevsubtitles" "stem exts" "cmds")
;; Tests: (find-wgeteevsubtitles-links "emacsconf2021" ".vtt")
;;        (find-estring (ee-wgeteevsubtitles-cmds "emacsconf2021" ".vtt .srt"))
;; See:   (find-wgetnode "Download Options" "-nc" "--no-clobber")
;;        (find-wgetnode "Download Options" "-N" "--timestamping")
;;        (find-wgetnode "Time-Stamping")
;;
;; NOTE: this function was superseded by
;;       `find-psne-1stclassvideo-links' and will be deleted soon!
;;
(defun find-wgeteevsubtitles-links (&optional stem exts &rest pos-spec-list)
"Visit a temporary buffer containing a script for downloading subtitles.
The script downloads an eev video and it subtitles. For example,
if STEM is \"emacsconf2021\" and EXTS is \".vtt .srt\" then the
main part of the script will be:\n
  mkdir -p $S/http/anggtwu.net/eev-videos/
  cd       $S/http/anggtwu.net/eev-videos/
  wget -nc  http://anggtwu.net/eev-videos/emacsconf2021.mp4
  wget -N   http://anggtwu.net/eev-videos/emacsconf2021.vtt
  wget -N   http://anggtwu.net/eev-videos/emacsconf2021.srt\n
The subtitles are downloaded again if the ones in
http://anggtwu.net/ are newer than the local copy."
  (interactive)
  (setq stem (or stem "{stem}"))
  (setq exts (or exts "{exts}"))
  (let* ((cmds (ee-wgeteevsubtitles-cmds stem exts)))
    (apply
     'find-elinks
     `((find-wgeteevsubtitles-links ,stem ,exts ,@pos-spec-list)
       ;; Convention: the first sexp always regenerates the buffer.
       (find-efunction 'find-wgeteevsubtitles-links)
       (find-video-links-intro "5.1. Subtitles")
       ""
       ,(ee-template0 "\
 (eepitch-shell2)
 (eepitch-kill)
 (eepitch-shell2)
{cmds}
# (find-fline \"$S/http/anggtwu.net/eev-videos/\" \"{stem}.mp4\")
# (find-video \"$S/http/anggtwu.net/eev-videos/{stem}.mp4\")
")
       )
     pos-spec-list)))

(defun ee-wgeteevsubtitles-cmds (stem exts)
  "An internal function used by `find-wgeteevsubtitles-links'."
  (let* ((dir         "$S/http/anggtwu.net/eev-videos/")
         (url- (concat "http://anggtwu.net/eev-videos/" stem))
         (f (lambda (ext) (format "wget -N   %s%s\n" url- ext)))
         (wgets (mapconcat f (split-string exts) "")))
    (ee-template0 "\
mkdir -p {dir}
cd       {dir}
wget -nc  {url-}.mp4
{wgets}\
")))

;; 2022may10: 




;;;   __ _           _       _       _                 _ _       _        
;;;  / _(_)_ __   __| |     | | __ _| |_ _____  __    | (_)_ __ | | _____ 
;;; | |_| | '_ \ / _` |_____| |/ _` | __/ _ \ \/ /____| | | '_ \| |/ / __|
;;; |  _| | | | | (_| |_____| | (_| | ||  __/>  <_____| | | | | |   <\__ \
;;; |_| |_|_| |_|\__,_|     |_|\__,_|\__\___/_/\_\    |_|_|_| |_|_|\_\___/
;;;                                                                       
;; «find-latex-links» (to ".find-latex-links")
;; Skel: (find-find-links-links-new "latex" "stem" "stem-")
;; Tests: (find-latex-links)
;;        (find-latex-links "/tmp/foo")
;;        (find-latex-links "/tmp/foo" 2)
;;        (find-latex-links "/tmp/foo" 2 "copy-rest")
;;   See: (find-eev-quick-intro "7.5. `find-latex-links'")
;;
(defun find-latex-links (&optional stem &rest pos-spec-list)
"Visit a temporary buffer containing a template for creating a LaTeX document."
  (interactive)
  (setq stem (or stem "{stem}"))
  (let* ((stem- (file-name-nondirectory stem)))
    (apply
     'find-elinks
     `((find-latex-links ,stem ,@pos-spec-list)
       (find-latex-links "/tmp/test")
       ;; Convention: the first sexp always regenerates the buffer.
       (find-efunction 'find-latex-links)
       (find-eev-quick-intro "7.5. `find-latex-links'")
       (find-eev-quick-intro "7.5. `find-latex-links'" "change the \"{stem}\"")
       (find-eev-quick-intro "7.5. `find-latex-links'" "top line")
       (find-eev-quick-intro "7.5. `find-latex-links'" "(ee-copy-rest ...)")
       (find-eev-quick-intro "7.5. `find-latex-links'" "`C-y'")
       (find-eev-quick-intro "7.5. `find-latex-links'" "save the file foo.tex")
       (find-eev-quick-intro "7.5. `find-latex-links'" "execute the three defuns")
       ""
       (ee-copy-rest 1 '(find-fline ,(concat stem ".tex")))
       ""
       ,(ee-template0 "\
% (defun c () (interactive) (find-sh \"pdflatex {stem-}.tex\"))
% (defun d () (interactive) (find-pdf-page \"{stem}.pdf\"))
% (defun e () (interactive) (find-fline    \"{stem}.tex\"))
%
\\documentclass{<}article{>}
\\begin{<}document{>}

\\end{<}document{>}
")
       )
     pos-spec-list)))







;;;   __ _           _       _                   _ _       _        
;;;  / _(_)_ __   __| |     | |_   _  __ _      | (_)_ __ | | _____ 
;;; | |_| | '_ \ / _` |_____| | | | |/ _` |_____| | | '_ \| |/ / __|
;;; |  _| | | | | (_| |_____| | |_| | (_| |_____| | | | | |   <\__ \
;;; |_| |_|_| |_|\__,_|     |_|\__,_|\__,_|     |_|_|_| |_|_|\_\___/
;;;                                                                 
;; «find-lua-links» (to ".find-lua-links")
;; (find-find-links-links "{k}" "lua" "fname")
;;
;; Test: (find-sh0 "rm -v /tmp/foo.lua")
;;       (find-lua-links "/tmp/foo.lua")
;;
(defun find-lua-links (&optional fname &rest pos-spec-list)
"Visit a temporary buffer containing hyperlinks for foo."
  (interactive)
  (setq fname (or fname "{fname}"))
  (let ((dir    (file-name-directory    fname))
	(fname0 (file-name-nondirectory fname)))
    (find-elinks
     `((find-lua-links ,fname ,@pos-spec-list)
       (find-lua-links "~/LUA/foo.lua")
       ;; Convention: the first sexp always regenerates the buffer.
       (find-efunction 'find-lua-links)
       ""
       (find-fline ,dir ,fname0)
       (find-fline ,fname)
       (find-sh0 ,(format "touch %s && chmod 755 %s" fname fname))
       ""
       (ee-copy-rest 1 '(find-fline ,fname))
       ""
       ,(ee-template0 "\
#!/usr/bin/env lua5.1
-- (defun c () (interactive) (find-sh \"cd {dir}; ./{fname0}\"))
-- (defun d () (interactive) (find-fline \"{dir}\"))
-- (defun e () (interactive) (find-fline \"{fname}\"))
--
-- (find-sh \"./{fname0} arg1 arg2\")
--
-- (find-lua51manual \"\")
-- (find-pil2page 8 \"Contents\")
-- (find-pil2text 8 \"Contents\")
-- (find-fline \"~/LUA/lua50init.lua\")

print(\"Hello from {fname}\")

--[[
 (eepitch-lua51)
 (eepitch-kill)
 (eepitch-lua51)
dofile \"{fname0}\"

--]]\
")
     ))))




;;;                                         _           _   
;;;   ___  ___  ___ _ __ ___  ___ _ __  ___| |__   ___ | |_ 
;;;  / _ \/ __|/ __| '__/ _ \/ _ \ '_ \/ __| '_ \ / _ \| __|
;;; |  __/\__ \ (__| | |  __/  __/ | | \__ \ | | | (_) | |_ 
;;;  \___||___/\___|_|  \___|\___|_| |_|___/_| |_|\___/ \__|
;;;                                                         
;; «find-escreenshot-links» (to ".find-escreenshot-links")
;; Produce an ascii screenshot of an emacs window.
;; To test this, copy a block of lines to the kill ring and do:
;;
;;   (setq ee-ss-width 80)
;;   (find-escreenshot-links)
;;
;; Tests: (ee-ss-string-pad "" "_" " ")
;;        (ee-ss-string-pad "foo")
;;        (ee-ss-string-pad ee-ss-mode-line "-")
;;        (ee-ss-string-pad "M-x foo" "_")
;;        (find-estring (ee-ss-screenshot ";;  %s\n"))
;;        (find-escreenshot0-links)
;;        (find-escreenshot-links)
;;
(defun ee-untabify-string (string)
  (with-temp-buffer
    (insert string)
    (untabify 0 (point-max))
    (buffer-substring 0 (point-max))))

(defvar ee-ss-strings '("foo" "bar") "A list of strings, for screenshots.")
(defvar ee-ss-width 70 "The intended width of the screenshot.")
(defvar ee-ss-mode-line "-:**-" "The mode line of the screenshot.")
(defvar ee-ss-echo-area "" "The acho area or mode line of the screenshot.")
(defvar ee-ss-linefmt ";; %s\n" "??")

(defun ee-ss-string-pad (str &optional padchar barchar)
  (setq barchar (or barchar "|"))
  (let* ((padc   (if padchar (aref padchar 0) 32))
         (padlen (- ee-ss-width (length str)))
	 (padstr (make-string padlen padc)))
    (concat barchar str padstr barchar)))

(defun ee-ss-screenshot0 ()
  `(,(ee-ss-string-pad "" "_" " ")
    ,@(mapcar 'ee-ss-string-pad ee-ss-strings)
    ,(ee-ss-string-pad ee-ss-mode-line "-")
    ,(ee-ss-string-pad ee-ss-echo-area "_")
    ))

(defun ee-ss-screenshot (linefmt)
  (mapconcat (lambda (li) (format linefmt li))
	     (ee-ss-screenshot0) ""))

(defun find-escreenshot0-links (&rest pos-spec-list)
"Visit a temporary buffer containing hyperlinks for an ascii screenshot."
  ;; (interactive)
  (apply 'find-elinks
   `((find-escreenshot0-links ,@pos-spec-list)
     ;; Convention: the first sexp always regenerates the buffer.
     (find-efunction 'find-escreenshot0-links)
     (setq ee-ss-strings '(split-string (ee-last-kill) "\n"))
     (setq ee-ss-width ,ee-ss-width)
     (setq ee-ss-mode-line ,ee-ss-mode-line)
     (setq ee-ss-echo-area ,ee-ss-echo-area)
     (setq ee-ss-linefmt ,ee-ss-linefmt)
     ""
     ,(ee-ss-screenshot ee-ss-linefmt))
   pos-spec-list))

(defun find-escreenshot-links (&rest pos-spec-list)
  (interactive)
  (setq ee-ss-mode-line (format-mode-line mode-line-format 0))
  (setq ee-ss-strings (split-string (ee-last-kill) "\n"))
  (apply 'find-escreenshot0-links pos-spec-list))





;; «find-windows-eepitch-lua-links»  (to ".find-windows-eepitch-lua-links")
;; Skel: (find-find-links-links-new "windows-eepitch-lua" "dir" "")
;; Test: (find-windows-eepitch-lua-links "/tmp/")
;;
;; THIS IS OBSOLETE!
;; The new way is this one:
;;   (find-windows-beginner-intro "7.3. Lua")
;;
(defun find-windows-eepitch-lua-links (&optional dir &rest pos-spec-list)
"Visit a temporary buffer that helps in setting up `eepitch-lua52's on M$ Windows."
  (interactive)
  ;; (setq dir (or dir "{dir}"))
  (setq dir (or dir default-directory))
  (apply
   'find-elinks-elisp
   `((find-windows-eepitch-lua-links ,dir ,@pos-spec-list)
     (find-windows-eepitch-lua-links "{dir}" ,@pos-spec-list)
     ;; Convention: the first sexp always regenerates the buffer.
     ;; (find-efunction 'find-windows-eepitch-lua-links)
     ""
     ,(ee-template0 "\
;; Instructions:
;; To configure the `eepitch-lua52' function on M$ Windows, do this:
;;
;; 1) Install ZeroBrane from:
;;      https://studio.zerobrane.com/
;;
;; 2) Find the directory inside the ZeroBrane installation that
;;    has the file lua52.exe.
;;
;; 3) Open that directory in Emacs using Dired - see:
;;      (find-node \"(emacs)Dired\")
;;    Hints: use RET to enter into a subdirectory; use M-k to go back;
;;    you may start here:
;;      (find-fline \"C:/\")
;;
;; 4) Run `M-x find-windows-eepitch-lua-links' there. You should get
;;    a buffer like this one, but with that directory in the first line.
;;    This is like adjusting the parameter of `find-latex-links',
;;    but the value of \"{<}dir{>}\" is set to the current directory. See:
;;      (find-eev-quick-intro \"7.5. `find-latex-links'\")
;;
;; 5) Test if you've got the correct \"{<}dir{>}\". If you got it
;;    right then the sexp below should open that directory and go to
;;    line where the \"lua52.exe\" is:
;;      (find-fline \"{dir}\" \"lua52.exe\")
;;
;; 6) Try the sexp below - it is a low-level test to see a) if our
;;    current path to \"lua52.exe\" is the right one AND b) if we can run
;;    that \"lua52.exe\" as a \"shell-like program\" in a target buffer.
;;    These ideas are explained here,
;;      (find-eev-quick-intro \"6.1. The main key: <F8>\")
;;      (find-eepitch-intro \"1.1. Another target\")
;;    and the test sexp is:
;;      (eepitch-comint \"lua52\" \"{dir}lua52.exe\")
;;
;; 7) If you got a prompt like this one
;;
;;      Lua 5.2.4  ... 1994-2015 Lua.org, PUC-Rio
;;      > 
;;
;;    in the right window in the test in item (6) then everything is
;;    working. If you didn't get a prompt like that then you NEED to run
;;    this before performing another test:
;;      (eepitch-kill)
;;      
;; 8) Override the current definition of eepitch-lua52 by running this:

(defun eepitch-lua52 () (interactive)
  (eepitch-comint \"lua52\"
     \"{dir}lua52.exe\"))

;;    Note that it is a 3-line sexp! If you execute it with `M-e'
;;    the result in the echo area should be \"eepitch-lua52\".
;;
;; 9) Test if the eepitch block below works: run it by typing
;;    `<f8>'s on its three red star lines, and then `<f8>'s on its three
;;    non-red star lines.

 (eepitch-lua52)
 (eepitch-kill)
 (eepitch-lua52)
print(2+3)
for i=2,20,3 do print(i) end
os.exit()

")
     )
   pos-spec-list))





;; «find-extra-file-links»  (to ".find-extra-file-links")
;; Skel: (find-find-links-links-new "extra-file" "fname c" "dir fname0 fullfname")
;; Docs: (find-audiovideo-intro "4.1. `find-extra-file-links'")
;;
(defun find-extra-file-links (&optional fname c &rest pos-spec-list)
"Visit a temporary buffer containing hyperlinks for extra-file."
  (interactive (list (if (eq major-mode 'dired-mode)
			 (ee-dired-to-fname)
		       buffer-file-name)))
  (if fname (setq fname (ee-shorten-file-name fname)))
  (setq fname (or fname "{fname}"))
  (setq c (or c "{c}"))
  (let* ((dir (file-name-directory fname))
	 (fname0 (file-name-nondirectory fname))
         (fullfname (ee-expand fname)))
    (apply
     'find-elinks-elisp
     `((find-extra-file-links ,fname ,c ,@pos-spec-list)
       (find-efunction 'find-extra-file-links)
       ;; Convention: the first sexp always regenerates the buffer.
       ,(ee-template0 "\
;;
;; See: (find-audiovideo-intro \"4.1. `find-extra-file-links'\")
;;      (find-eev-quick-intro \"9.1. `code-c-d'\")

;; Links to this file and directory:
;; file:///{fullfname}
;; (find-fline {(ee-S dir)} {(ee-S fname0)})
;; (find-fline {(ee-S fname)})
;; (find-fline {(ee-S dir)})
\(code-c-d \"{c}\" \"{dir}\")
;; (find-{c}file \"\")

;; Links to a PDF file:
;; (find-pdf-page \"{fname}\")
;; (find-pdf-text \"{fname}\")
\(code-pdf-page \"{c}\" \"{fname}\")
\(code-pdf-text \"{c}\" \"{fname}\")
;; (find-{c}page)
;; (find-{c}text)

;; Links to an audio file:
;; (find-audio \"{fname}\")
\(code-audio \"{c}audio\" \"{fname}\")
;; (find-{c}audio)
;; (find-{c}audio \"0:00\")

;; Links to a video file:
;; (find-video \"{fname}\")
\(code-video \"{c}video\" \"{fname}\")
;; (find-{c}video)
;; (find-{c}video \"0:00\")
;;
;; (eev-avadj-mode 0)
;; (eev-avadj-mode)

;; Links to an shell-like program (for eepitch):
;; (eepitch-comint \"{c}\" \"{fname}\")

(defun eepitch-{c} () (interactive)
  (eepitch-comint \"{c}\"
     \"{fname}\"))

;; Test:

 (eepitch-{c})
 (eepitch-kill)
 (eepitch-{c})
")
       )
     pos-spec-list)))

;; Tests:
;; (find-extra-file-links "~/eev-videos/three-keys-2.mp4")





;; «find-emacs-tangents-links»  (to ".find-emacs-tangents-links")
;; Skel: (find-find-links-links-new "emacs-tangents" "yyyy mm dd msg txtstem" "")
;; Test: (find-emacs-tangents-links "2022" "06" "06")
;;
(defun find-emacs-tangents-links (&optional yyyy mm dd msg txtstem &rest pos-spec-list)
"Visit a temporary buffer with hyperlinks to a post in emacs-tangents."
  (interactive)
  (setq yyyy (or yyyy "{yyyy}"))
  (setq mm (or mm "{mm}"))
  (setq dd (or dd "{dd}"))
  (setq msg (or msg "{msg}"))
  (setq txtstem (or txtstem "{txtstem}"))
  (apply
   'find-elinks
   `((find-emacs-tangents-links ,yyyy ,mm ,dd ,msg ,txtstem ,@pos-spec-list)
     ;; Convention: the first sexp always regenerates the buffer.
     (find-emacs-tangents-links "2020" "10" "05" "msg00000" "txts2ETp920ql")
     (find-emacs-tangents-links "2018" "01" "29" "msg00025" "txtJ1ftXqItdm")
     (find-efunction 'find-emacs-tangents-links)
     ""
     ,(ee-template0 "\
# https://sachachua.com/blog/{yyyy}/
# https://sachachua.com/blog/{yyyy}/{mm}/
# https://sachachua.com/blog/{yyyy}/{mm}/{yyyy}-{mm}-{dd}-emacs-news/
# https://lists.gnu.org/archive/html/emacs-tangents/{yyyy}-{mm}/
# https://lists.gnu.org/archive/html/emacs-tangents/{yyyy}-{mm}/{msg}.html

 (eepitch-shell)
 (eepitch-kill)
 (eepitch-shell)
# (find-fline \"~/usrc/emacs-tangents/\")
# (find-fline \"~/usrc/emacs-tangents/{yyyy}-{mm}-{dd}-emacs-news.org\")

mkdir -p ~/usrc/emacs-tangents/
cd       ~/usrc/emacs-tangents/
# rm -v     {yyyy}-{mm}-{dd}-emacs-news.org
wget -nc -O {yyyy}-{mm}-{dd}-emacs-news.org \\
  https://lists.gnu.org/archive/html/emacs-tangents/{yyyy}-{mm}/{txtstem}.txt
")
     )
   pos-spec-list))




;; «find-eeit-links»  (to ".find-eeit-links")
;; Skel: (find-find-links-links-new "eeit" "majormode" "majormodestr eeitfunstr eeitfun")
;; See:  (find-eepitch-intro "3.1. `find-eeit-links'")
;; Test: (find-eeit-links 'lua-mode)
;;
(defun find-eeit-links (&optional majormode &rest pos-spec-list)
"Visit a temporary buffer containing hyperlinks for eeit."
  (interactive)
  (setq majormode (or majormode major-mode))
  (let* ((majormodestr (symbol-name majormode))
         (eeitfunstr (concat "ee-insert-test-" majormodestr))
         (eeitfun (intern eeitfunstr)))
    (apply
     'find-elinks-elisp
     `((find-eeit-links ',majormode ,@pos-spec-list)
       ;; Convention: the first sexp always regenerates the buffer.
       ;; ""
       ,(ee-template0 "\
;; (find-efunction 'find-eeit-links)
;; (find-eepitch-intro \"3. Test blocks\")
;; (find-eepitch-intro \"3.1. `find-eeit-links'\")
;; (find-eev \"eev-testblocks.el\" \"examples\")

;; Current definition:
;; (find-efunction              '{eeitfun})
;; (find-efunctionpp            '{eeitfun})
;; (find-epp (ee-defun-sexp-for '{eeitfun}))


;; <{eeitfunstr}>
;;
(defun {eeitfunstr} ()
  (interactive)
  (insert (ee-adjust-red-stars (format \"
--[[
 (eepitch-lua51)
 (eepitch-kill)
 (eepitch-lua51)
dofile \\\"%s\\\"

--]]
\" (buffer-name)))))

;; Test:
;; ({eeitfunstr})

")
       )
     pos-spec-list)))




;;;  _____   __  ___     _           
;;; |_   _|__\ \/ / |   (_)_   _____ 
;;;   | |/ _ \\  /| |   | \ \ / / _ \
;;;   | |  __//  \| |___| |\ V /  __/
;;;   |_|\___/_/\_\_____|_| \_/ \___|
;;;                                  
;; «find-texlive-links»  (to ".find-texlive-links")

(defun find-texlive-links (&optional date &rest pos-spec-list)
"Visit a temporary buffer containing an e-script for installing texlive from upstream."
  (interactive)
  (setq date (or date "{date}"))
  (apply 'find-elinks
   `((find-texlive-links ,date ,@pos-spec-list)
     ;; Convention: the first sexp always regenerates the buffer.
     (find-texlive-links "20210816")
     (find-efunction 'find-texlive-links)
     ""
     ,(ee-template0 "\
# https://www.tug.org/texlive/
# https://www.tug.org/texlive/acquire-netinstall.html
# https://www.tug.org/texlive/quickinstall.html
# http://mirror.ctan.org/systems/texlive/tlnet/install-tl-unx.tar.gz
#   (find-fline \"$S/http/mirror.ctan.org/systems/texlive/tlnet/\")
#   (find-fline \"$S/http/mirror.ctan.org/systems/texlive/tlnet/install-tl-unx.tar.gz\")
# https://www.tug.org/texlive/doc/install-tl.html
# (find-fline \"~/usrc/\" \"install-tl-\")
# (find-fline \"~/.texlive2018/\")
# (find-fline \"~/.texlive2019/\")
# (find-fline \"~/.texlive2021/\")
# (find-fline \"/usr/local/texlive/2018/\")
# (find-fline \"/usr/local/texlive/2019/\")
# (find-fline \"/usr/local/texlive/2019/\" \"install-tl.log\")
# (find-fline \"/usr/local/texlive/2019/release-texlive.txt\")

 (eepitch-shell2)
 (eepitch-kill)
 (eepitch-shell2)
# rm -rfv ~/.texlive2021/
# sudo rm -rfv /usr/local/texlive/2021/

mkdir -p $S/http/mirror.ctan.org/systems/texlive/tlnet/
cd       $S/http/mirror.ctan.org/systems/texlive/tlnet/
rm -fv   $S/http/mirror.ctan.org/systems/texlive/tlnet/install-tl-unx.tar.gz
wget      http://mirror.ctan.org/systems/texlive/tlnet/install-tl-unx.tar.gz
echo      http://mirror.ctan.org/systems/texlive/tlnet/install-tl-unx.tar.gz >> ~/.psne.log
# (find-fline \"$S/http/mirror.ctan.org/systems/texlive/tlnet/install-tl-unx.tar.gz\")
# ^ Check the date here and adjust the {<}date{>} parameter of the template

rm -Rfv ~/usrc/install-tl-{date}/
mkdir   ~/usrc/
tar  -C ~/usrc/ -xvzf \
  $S/http/mirror.ctan.org/systems/texlive/tlnet/install-tl-unx.tar.gz
cd      ~/usrc/install-tl-{date}/
# sudo ./install-tl
sudo ./install-tl -select-repository

# (code-c-d \"tlinstall\" \"~/usrc/install-tl-{date}/\")
# (find-tlinstallfile \"\")
# (find-tlinstallfile \"install-tl.log\")
# (find-tlinstallfile \"install-tl\")
# https://www.tug.org/texlive/doc/install-tl.html

")
     )
   pos-spec-list))

;; Test: (find-texlive-links)




;;;                      _                                     
;;;  _ __   _____      _| |__  _ __ _____      _____  ___ _ __ 
;;; | '_ \ / _ \ \ /\ / / '_ \| '__/ _ \ \ /\ / / __|/ _ \ '__|
;;; | | | |  __/\ V  V /| |_) | | | (_) \ V  V /\__ \  __/ |   
;;; |_| |_|\___| \_/\_/ |_.__/|_|  \___/ \_/\_/ |___/\___|_|   
;;;                                                            
;; «find-newbrowser-links»  (to ".find-newbrowser-links")
;; Skel: (find-find-links-links-new "newbrowser" "browser binary b" "")
;; Test: (find-newbrowser-links "googlechrome" "google-chrome" "g")
;; See:  (find-wconfig-browser-links)
;;       (find-efunction 'find-wconfig-browser-links)
;;
(defun find-newbrowser-links (&optional browser binary b &rest pos-spec-list)
"Visit a temporary buffer containing hyperlinks for newbrowser."
  (interactive)
  (setq browser (or browser "{browser}"))
  (setq binary (or binary "{binary}"))
  (setq b (or b "{b}"))
  (apply
   'find-elinks-elisp
   `((find-newbrowser-links ,browser ,binary ,b ,@pos-spec-list)
     ;; Convention: the first sexp always regenerates the buffer.
     (find-efunction 'find-newbrowser-links)
     ""
     ";; Important: this function is OBSOLETE!"
     ";; It was superseded by: (find-altbrowser-links)"
     ""
     ""
     ""
     ";; See: (find-wconfig-browser-links)"
     ""
     ,(ee-template0 "\
;; 1. This block defines `find-{browser}', `br{b}', `br{b}l', and `br{b}d'.
;; See: (find-eev-quick-intro \"3.1. Non-elisp hyperlinks\")
;;      (find-eev-quick-intro \"3.1. Non-elisp hyperlinks\" \"not free\")
;;      (find-eev-quick-intro \"3.1. Non-elisp hyperlinks\" \"find-googlechrome\")
;;      (find-eev-quick-intro \"3.1. Non-elisp hyperlinks\" \"M-x brff\")
;;      (find-brxxx-intro \"3. The `d' variants\")
;;      (find-brxxx-intro \"5. `code-brurl'\")
;;      (find-eev \"eev-brxxx.el\" \"code-brxxxs\")
;;      (find-eev \"eev-plinks.el\" \"find-googlechrome\")
;; Tests: (find-{browser} \"http://www.lua.org/start.html\")
;;        (br{b} \"http://www.lua.org/start.html\")
;;
(defun find-{browser} (url) (find-bgprocess `(\"{binary}\" ,url)))
;;
;; If you're on a Mac, use this instead:
;; (defun find-{browser} (url &rest rest)
;;   (find-bgprocess `(\"open\" \"-a\" \"{binary}\" ,url)))
;;
;; (find-code-brurl 'find-{browser}  :remote 'br{b}  :local 'br{b}l  :dired 'br{b}d)
        (code-brurl 'find-{browser}  :remote 'br{b}  :local 'br{b}l  :dired 'br{b}d)



;; 2a. This block defines `find-{browser}-page'.
;; See: (find-eev \"eev-pdflike.el\" \"find-googlechrome-page\")
;;
(defun ee-find-{browser}-page (fname &optional page)
  `(\"{binary}\" ,(ee-fname-page-to-url fname page)))
;;
;; (find-code-pdfbackend \"{browser}-page\")
        (code-pdfbackend \"{browser}-page\")

;; Run the `defalias' below if you want to make
;; `find-pdf-page' use `find-{browser}-page'.
;; See: (find-eev \"eev-pdflike.el\" \"change-default-viewer\")
;;
(defalias 'find-pdf-page 'find-{browser}-page)
;;
;; Tests:
;; (find-{browser} \"http://anggtwu.net/#eev\")
;; (find-{browser}-page \"~/Coetzee99.pdf\")
;; (find-{browser}-page \"~/Coetzee99.pdf\" 3)
;; (find-pdf-page \"~/Coetzee99.pdf\" 3)



;; 2b. Configure `find-pdf-page' to make it use the browser.
;; Use this when you don't want to set up a standalone PDF viewer.
;; See:  (find-pdf-like-intro \"2. Preparation\")
;;       (find-pdf-like-intro \"3. Hyperlinks to PDF files\")
;;       (find-pdf-like-intro \"3. Hyperlinks to PDF files\" \"If you have xpdf\")
;;       (find-eev \"eev-pdflike.el\" \"change-default-viewer\")
;;       (find-eev \"eev-pdflike.el\" \"find-firefox-page\")
;;       (find-code-pdfbackend \"{browser}-page\")
;; Test: (ee-find-{browser}-page \"~/Coetzee99.pdf\" 3)
;;          (find-{browser}-page \"~/Coetzee99.pdf\" 3)
;;                 (find-pdf-page \"~/Coetzee99.pdf\" 3)
;;
(defun ee-find-{browser}-page (fname &optional page)
  `(\"{binary}\" ,(ee-fname-page-to-url fname page)))
(code-pdfbackend \"{browser}-page\")
(defalias 'find-pdf-page 'find-{browser}-page)
;;
;; To configure `find-pdf-page' to make it use a
;; standalone PDF viewer, run one of the sexps below:
;;   (defalias 'find-pdf-page 'find-xpdf-page)
;;   (defalias 'find-pdf-page 'find-mupdf-page)
;; or see: (find-eev \"eev-pdflike.el\" \"change-default-viewer\")



;; 3. Configure `find-youtube-video'.
;; You will need this if you don't want to
;; set up a video player like mpv or vlc.
;; See:  (find-video-links-intro \"3. `find-youtube-video'\")
;;       (find-video-links-intro \"5. Local copies\")
;;       (find-eev \"eev-audiovideo.el\" \"find-youtube-video\")
;; Test: (ee-find-youtube-video \"FoAzpGzFCSE\" \"15:14\" \"nice\")
;;          (find-youtube-video \"FoAzpGzFCSE\" \"15:14\")
;;                 (find-2022findeevanggvideo \"15:14\")
(setq ee-find-youtube-video-program 'find-{browser})
")
     )
   pos-spec-list))




;;;                      _                                    ____  
;;;  _ __   _____      _| |__  _ __ _____      _____  ___ _ _|___ \ 
;;; | '_ \ / _ \ \ /\ / / '_ \| '__/ _ \ \ /\ / / __|/ _ \ '__|__) |
;;; | | | |  __/\ V  V /| |_) | | | (_) \ V  V /\__ \  __/ |  / __/ 
;;; |_| |_|\___| \_/\_/ |_.__/|_|  \___/ \_/\_/ |___/\___|_| |_____|
;;;                                                                 
;; «find-altbrowser-links»  (to ".find-altbrowser-links")
;; Skel: (find-find-links-links-new "altbrowser" "binary" "")
;; Test: (find-altbrowser-links)
;;
(defun find-altbrowser-links (&optional binary &rest pos-spec-list)
"Visit a temporary buffer with instructions for making `find-googlechrome'
use an alternate browser. More precisely, it shows how to redefine
`find-googlechrome' to make it call BINARY instead of \"google-chrome\"."
  (interactive)
  (setq binary (or binary "{binary}"))
  (apply
   'find-elinks-elisp
   `((find-altbrowser-links ,binary ,@pos-spec-list)
     (find-altbrowser-links "{binary}" ,@pos-spec-list)
     (find-altbrowser-links "google-chrome" ,@pos-spec-list)
     (find-altbrowser-links "chromium" ,@pos-spec-list)
     ;; Convention: the first sexp always regenerates the buffer.
     (find-efunction 'find-altbrowser-links)
     ""
     ,(ee-template0 "\
;; Make `{binary}' the default browser.
;; This is a hack that makes `find-googlechrome' call `{binary}'.
;; Note that this hack bypasses the variable `ee-googlechrome-program'!
;; See: (find-eev \"eev-plinks.el\" \"find-googlechrome\")


;; Try:
;;
(defun find-googlechrome (url) (find-bgprocess `(\"{binary}\" ,url)))
(find-googlechrome \"https://www.lua.org/\")
;;
;; Then run these tests:
;;   (find-bgprocess \"{binary} https://www.lua.org/\")
;;   (find-googlechrome \"https://www.lua.org/\")
;;   (brg \"https://www.lua.org/\")
;;   (find-eev-quick-intro \"3.1. Non-elisp hyperlinks\" \"M-x brg\")


;; To save this configuration, see:
;;   (find-dot-emacs-intro)
;; and run:
;;   
;;   (ee-copy-rest-3 nil \";;--end\" \"~/.emacs\")
;; From: (find-altbrowser-links \"{binary}\")
;; Test: (find-googlechrome \"https://www.lua.org/\")
(defun find-googlechrome (url) (find-bgprocess `(\"{binary}\" ,url)))
;;--end
")
     )
   pos-spec-list))





;; «find-newpdfviewer-links»  (to ".find-newpdfviewer-links")
;; Skel: (find-find-links-links-new "newpdfviewer" "short binary" "")
;; Test: (find-newpdfviewer-links "okular")
;;
(defun find-newpdfviewer-links (&optional short binary &rest pos-spec-list)
"Visit a temporary buffer containing hyperlinks for newpdfviewer."
  (interactive)
  (setq short (or short "{short}"))
  (setq binary (or binary short "{binary}"))
  (apply
   'find-elinks-elisp
   `((find-newpdfviewer-links ,short ,binary ,@pos-spec-list)
     (find-newpdfviewer-links "{short}" "{binary}" ,@pos-spec-list)
     ;; Convention: the first sexp always regenerates the buffer.
     (find-efunction 'find-newpdfviewer-links)
     ""
     ,(ee-template0 "\

;; <find-{short}-page>
;; Skel:  (find-newpdfviewer-links \"{short}\" \"{binary}\")
;; Tests: (find-pdf-like-intro \"2. Preparation\")
;;        (ee-find-{short}-page \"~/Coetzee99.pdf\" 3)
;;           (find-{short}-page \"~/Coetzee99.pdf\" 3)
;;
\(defun     find-{short}-page (fname &optional page &rest rest)
  (find-bgprocess (ee-find-{short}-page fname page)))
\(defvar ee-find-{short}-page-options '())
\(defun  ee-find-{short}-page (fname &optional page &rest rest)
  `(\"{binary}\"
    ,@ee-find-{short}-page-options
    ;; Each PDF viewer selects the initial page in a different way.
    ;; See: (find-man \"{binary}\")
    ,@(if page `(\"-p\" ,(format \"%s\" page)))
    ,fname
    ))

;; Extra definitions:
;; (find-code-pdfbackend \"mupdf-page\")
        (code-pdfbackend \"mupdf-page\")

;; Compare the code above with:
;; (find-eev \"eev-pdflike.el\" \"find-mupdfpage\")



;; To make `find-{short}-page' the default viewer for PDFs,
;; read: (find-eev \"eev-pdflike.el\" \"change-default-viewer\") 
;; and run:
(defalias 'find-pdf-page 'find-{short}-page)
;;
;; Test: (find-pdf-like-intro \"2. Preparation\")
;;       (find-pdf-page \"~/Coetzee99.pdf\" 3)
")
     )
   pos-spec-list))




;;;   ___        ___  
;;;  / _ \__  __/ _ \ 
;;; | | | \ \/ / | | |
;;; | |_| |>  <| |_| |
;;;  \___//_/\_\\___/ 
;;;                   
;; «ee-0x0-upload-region»  (to ".ee-0x0-upload-region")
;; Needs: https://melpa.org/#/0x0
;;        (find-epackage-links '0x0)
;;   See: https://lists.gnu.org/archive/html/eev/2021-08/msg00000.html
;;
(defvar ee-0x0-url nil
  "The last URL generated by a call to `ee-0x0-upload-region'.")

(defun ee-0x0-upload (b e &rest ignore)
  "Upload the current region using `0x0-dwim'.
See: https://gitlab.com/willvaughn/emacs-0x0/-/issues/2
This is a hack."
  (deactivate-mark 'force)
  (eek "C-x C-x C-x C-x     ;; reactivate the mark
        M-x 0x0-dwim RET"))

(defun ee-0x0-upload-region (b e)
  "Upload the current region to 0x0 and show some elisp hyperlinks.
I use this mainly on IRC chats, to send elisp snippets to people
who use eev. I use this function aliased to `u0', with:

  (defalias 'u0 'ee-0x0-upload-region)

If I mark the region with the snippet and run `M-x u0', `u0'
uploads the snippet to https://0x0.st/ and shows some sexps in a
temporary buffer; the two main sexps in that temporary buffer
there will be things like:

  (find-wget \"https://0x0.st/-J1W.el\")
  (find-wget-elisp \"https://0x0.st/-J1W.el\")

The one that starts with `find-wget-elisp' downloads the URL with
wget and runs `emacs-lisp-mode' on the resulting \"*wget: URL*\"
buffer. If the other person on IRC is using some IRC client that
runs in Emacs I can send the find-wget-elisp sexp to her and she
only needs to type `M-e' on it to access the elisp code that I
sent."
  (interactive "r")
  (ee-0x0-upload b e '0x0)
  (setq ee-0x0-url (ee-last-kill))
  (find-0x0-links))

;; «find-0x0-links»  (to ".find-0x0-links")
;; Skel: (find-find-links-links-new "0x0" "url" "")
;;
(defun find-0x0-links (&optional url &rest pos-spec-list)
"Visit a temporary buffer containing hyperlinks to an upload to 0x0.st.
This function is used by `ee-0x0-upload-region'."
  (interactive)
  (setq url (or url ee-0x0-url))
  (apply
   'find-elinks
   `((find-0x0-links ,url ,@pos-spec-list)
     (find-0x0-links "{url}" ,@pos-spec-list)
     ;; Convention: the first sexp always regenerates the buffer.
     (find-efunction 'find-0x0-links)
     ""
     ,(ee-template0 "\
{url}
(find-wget \"{url}\")
(find-wget-elisp \"{url}\")
")
     )
   pos-spec-list))



;;;               _       _                      __  _           _ _      _       
;;;  _ __ ___  __| |  ___| |_ __ _ _ __ ___     / / | |__  _   _| | | ___| |_ ___ 
;;; | '__/ _ \/ _` | / __| __/ _` | '__/ __|   / /  | '_ \| | | | | |/ _ \ __/ __|
;;; | | |  __/ (_| | \__ \ || (_| | |  \__ \  / /   | |_) | |_| | | |  __/ |_\__ \
;;; |_|  \___|\__,_| |___/\__\__,_|_|  |___/ /_/    |_.__/ \__,_|_|_|\___|\__|___/
;;;                                                                               
;; «find-eepitch-bullet-links»  (to ".find-eepitch-bullet-links")
;; «find-red-star-links»        (to ".find-red-star-links")
;; Skel: (find-find-links-links-new "red-star" "" "")
;; Test: (find-red-star-links)
;;
(defalias 'find-eepitch-bullet-links 'find-red-star-links)
(defun find-red-star-links (&rest pos-spec-list)
"Show code that makes eepitch use red bullets instead of red stars."
  (interactive)
  (apply
   'find-elinks-elisp
   `((find-red-star-links ,@pos-spec-list)
     ;; Convention: the first sexp always regenerates the buffer.
     (find-efunction 'find-red-star-links)
     ;; ""
     ,(ee-template0 "\
;; The code below was based on:
;;   https://lists.gnu.org/archive/html/help-gnu-emacs/2021-05/msg01080.html
;;   https://lists.gnu.org/archive/html/help-gnu-emacs/2021-05/msg01079.html
;; Note that the default regexes for eepitch accept both the red
;; stars and the bullets - chars 15 and 8226 respectively...
;; See: (find-eev \"eepitch.el\" \"glyphs\")
;;      (find-eev \"eepitch.el\" \"eepitch\" \"eepitch-regexp\")
;;      (find-eev-quick-intro \"6.3. Creating eepitch blocks: `M-T'\")
;; And try: (insert \"\\n;; \" 15 8226)



;; Run this to make the bullets use the default face:
\(eepitch-set-glyph0 ?• nil)
;;
;; Run this to make the bullets appear in the same face as the red stars:
\(eepitch-set-glyph0 ?• ?• 'eepitch-star-face)
;;
;; You'll need to force a redisplay to see the change.
;; One way to do that is with `C-l' (`recenter-top-bottom').


;; Most of the functions in eev that generate text with red stars
;; use the function `ee-adjust-red-stars' to replace the red
;; stars in the generated text by some non-default character if
;; needed. Use the first defun below to use red stars, and use
;; the second one to use bullets.
;;  See: (find-eev \"eepitch.el\" \"ee-adjust-red-stars\")
;;       (find-eval-intro \"3. What to execute, and in what order\")
;; Test: (ee-adjust-red-stars \"foobar\")
;;
\(defun ee-adjust-red-stars (str) str)
\(defun ee-adjust-red-stars (str) (replace-regexp-in-string \"\" \"•\" str))



;; To make eepitch blocks use red bullets by default, add the
;; block below to your ~/.emacs (including the line with the
;; \"See:\"). Note that this will make bullets appear in red in
;; other places too, like in info manuals - for example here:
;;   (find-enode \"Using Region\")
;;
;; See: (find-red-star-links)
\(eepitch-set-glyph0 ?• ?• 'eepitch-star-face)
\(defun ee-adjust-red-stars (str) (replace-regexp-in-string \"\" \"•\" str))



;; The high-level way to switch the default between \"use red bullets\"
;; and \"use red stars\" is to use `M-x ee-use-red-bullets' and `M-x
;; ee-use-red-stars'. Try the two sexps below...
;;
;; (progn (insert \"\\nshell\\n\") (eek \"<up> <<ee-use-red-bullets>> M-T\"))
;; (progn (insert \"\\nshell\\n\") (eek \"<up> <<ee-use-red-stars>>   M-T\"))
")
     )
   pos-spec-list))

;; «ee-use-red-stars»    (to ".ee-use-red-stars")
;; «ee-use-red-bullets»  (to ".ee-use-red-bullets")
;; See: (find-red-star-links 2 "high-level")

(defun ee-use-red-stars ()
  "See: (find-red-star-links 2 \"high-level\")"
  (interactive)
  (eepitch-set-glyph0 ?• nil)
  (defun ee-adjust-red-stars (str) str)
  (redisplay 'force)
  (message "Now `M-T' and `M-x eeit' will use red stars."))

(defun ee-use-red-bullets ()
  "See: (find-red-star-links 2 \"high-level\")"
  (interactive)
  (eepitch-set-glyph0 ?• ?• 'eepitch-star-face)
  (defun ee-adjust-red-stars (str) (replace-regexp-in-string "" "•" str))
  (redisplay 'force)
  (message "Now `M-T' and `M-x eeit' will use red bullets."))





;;;                                          
;;;   __ _ _ __   __ _  __ _        ___  ___ 
;;;  / _` | '_ \ / _` |/ _` |_____ / _ \/ __|
;;; | (_| | | | | (_| | (_| |_____|  __/\__ \
;;;  \__,_|_| |_|\__, |\__, |      \___||___/
;;;              |___/ |___/                 
;;
;; «find-angg-es-links»  (to ".find-angg-es-links")
;; Skel: (find-find-links-links-new "angg-es" "" "")
;; Test: (find-angg-es-links)
;;  See: http://anggtwu.net/eev-find-angg.html
;;
(defun find-angg-es-links (&rest pos-spec-list)
"Show an e-script for configuring `find-angg' and `find-es'."
  (interactive)
  (apply
   'find-elinks-elisp
   `((find-angg-es-links ,@pos-spec-list)
     ;; Convention: the first sexp always regenerates the buffer.
     (find-efunction 'find-angg-es-links)
     (find-eev-quick-intro "9. Shorter hyperlinks")
     (find-eev-quick-intro "9.1. `code-c-d'")
     (find-eev-quick-intro "9.2. Extra arguments to `code-c-d'" "to anchors")
     (find-2022findeevanggvideo "11:29" "(find-angg-es-links)")
     (find-2022findeevangghsubs "11:29" "(find-angg-es-links)")
     (find-2022findeevanggvideo "14:06" "the find-wgeta searches for")
     (find-2022findeevangghsubs "14:06" "the find-wgeta searches for")
     ""
     ,(ee-template0 "\


;; The `progn' below is how I (edrx) define `find-angg',
;; `find-anggfile' and `find-es' in my machine to point to local
;; files. Note that the sexps
;;
;;   (find-angg \"foo\" \"anchor\")
;;   (find-es   \"foo\" \"anchor\")
;;
;; search for anchor \"anchor\" in the files \"~/foo\" and
;; \"~/e/foo.e\" respectively... in 1996, when I wrote `find-es',
;; I had several functions like it, that appended arbitrary
;; prefixes _and suffixes_ to their first arguments. I stopped
;; using all of them except `find-es' - that is a kind of
;; living fossil.
;;
(progn

  (code-c-d \"angg\" \"~/\" :anchor :grep)
  (code-c-d \"es\"   \"$ES/\")
  (defun find-es (stem &rest rest)
    (apply 'find-anchor (ee-esfile (concat stem \".e\")) rest))

)



;; The `progn' below defines versions of `find-angg',
;; `find-anggfile' and `find-es' that use `find-wget' to access
;; the public copies of my files at anggtwu.net:
;;
(progn

  (defun find-angg (fname &rest rest)
    (apply 'find-wgeta (format \"http://anggtwu.net/%s\" fname) rest))
  (defun find-anggfile (fname &rest rest)
    (apply 'find-wget  (format \"http://anggtwu.net/%s\" fname) rest))
  (defun find-es (fname &rest rest)
    (apply 'find-wgeta (format \"http://anggtwu.net/e/%s.e\" fname) rest))

)



;; Tests:
;; (find-angg \"e/bullseye.e\")
;; (find-angg \"e/bullseye.e\" \"2021aug16\")
;; (find-es     \"bullseye\")
;; (find-es     \"bullseye\"   \"2021aug16\")
")
     )
   pos-spec-list))



;; «find-angg-not-configured»  (to ".find-angg-not-configured")
;; «find-es-not-configured»    (to ".find-es-not-configured")
;; Based on this suggestion by Quiliro Ordoñez, but simpler:
;;   https://lists.gnu.org/archive/html/eev/2022-08/msg00005.html
;; See:
;;   (find-angg-es-links)
;;
(defun find-angg-not-configured (&rest rest) (interactive)
  (error "`find-angg' not configured! Run: (find-angg-es-links)"))

(defun find-es-not-configured (&rest rest) (interactive)
  (error "`find-es' not configured! Run: (find-angg-es-links)"))

(if (not (fboundp 'find-angg)) (defalias 'find-angg 'find-angg-not-configured))
(if (not (fboundp 'find-es))   (defalias 'find-es   'find-es-not-configured))





;;;  _     _       _                     _     _            
;;; / |___| |_ ___| | __ _ ___ _____   _(_) __| | ___  ___  
;;; | / __| __/ __| |/ _` / __/ __\ \ / / |/ _` |/ _ \/ _ \ 
;;; | \__ \ || (__| | (_| \__ \__ \\ V /| | (_| |  __/ (_) |
;;; |_|___/\__\___|_|\__,_|___/___/ \_/ |_|\__,_|\___|\___/ 
;;;                                                         
;; «find-1stclassvideo-links»  (to ".find-1stclassvideo-links")
;; Skel: (find-find-links-links-new "1stclassvideo" "c" "body")
;;  See: (find-eev "eev-videolinks.el" "first-class-videos")
;;       (find-eev "eev-videolinks.el" "second-class-videos")
;; Tests: (find-1stclassvideo-links "eev2019")
;;        (find-1stclassvideo-links "foo")
;;        (find-1stclassvideo-links)
;;        (find-estring (ee-find-1stclassvideo-links "eev2019"))
;;        (find-estring (ee-find-1stclassvideo-links "foo"))
;;
(defun find-1stclassvideo-links (&optional c &rest pos-spec-list)
"Visit a temporary buffer containing links about a first-class video."
  (interactive (list (ee-1stclassvideo-around-point-ask)))
  (setq c (or c "{c}"))
  (let* ((ee-buffer-name (or ee-buffer-name "*find-1stclassvideo-links*"))
	 (body (if (member c (ee-1stclassvideos))
		   (ee-find-1stclassvideo-links c)
		 ";; Not in: (ee-1stclassvideos)")))
    (apply
     'find-elinks-elisp
     `((find-1stclassvideo-links ,c ,@pos-spec-list)
       ;; Convention: the first sexp always regenerates the buffer.
       (find-efunction 'find-1stclassvideo-links)
       (find-1stclassvideos)
       ""
       ,body
       )
     pos-spec-list)))

;; Skel: (find-let*-macro-links "1stclassvideo-c" "c" "title mp4")
;; Test: (ee-let*-macro-1stclassvideo-c "eev2021" title)
(defmacro ee-let*-macro-1stclassvideo-c (c &rest code)
  "An internal function used by `find-1stclassvideo-links'."
  `(let* ((c ,c)
	  (title     (ee-1stclassvideos-field c :title))
	  (mp4       (ee-1stclassvideos-field c :mp4))
	  (yt        (ee-1stclassvideos-field c :yt))
	  (page      (ee-1stclassvideos-field c :page))
	  (hsubs     (ee-1stclassvideos-field c :hsubs))
	  (date      (ee-1stclassvideos-field c :date))
	  (length    (ee-1stclassvideos-field c :length))
	  (comment   (ee-1stclassvideos-field c :comment))
	  (lang      (ee-1stclassvideos-field c :lang))
	  (exts      (ee-1stclassvideos-field c :subs))
	  (hasindex  (ee-1stclassvideos-field c :index))
	  (mp4stem   (ee-1stclassvideos-mp4stem c))
          (mp4found  (ee-1stclassvideos-mp4found c))
	  (hash      (ee-1stclassvideos-hash c))
	  (hassubs   exts)
	  (subsinit  (and hsubs (replace-regexp-in-string "^.*#" "" hsubs))))
     ,@code))

;; Test:
;; (find-estring-elisp (ee-find-1stclassvideo-links "eev2021"))
;; (find-estring-elisp (ee-find-1stclassvideo-links "2021ssr"))
;;
(defun ee-find-1stclassvideo-links (c)
  "An internal function used by `find-1stclassvideo-links'."
  (let* ((basicinfo   (ee-1stclassvideo-basicinfo c))
	 (basicsexps  (ee-1stclassvideo-basicsexps c))
	 (dlsubs      (ee-1stclassvideo-dlsubs c))
	 (defuns      (ee-1stclassvideo-defuns c)))
    (ee-template0 "\
{basicinfo}\
;;
{basicsexps}\
;;
{dlsubs}\

;; See:
;; (find-strange-functions-intro \"1. Introduction: videos\")
;; (find-video-links-intro \"9. First-class videos\")
;; (find-eev \"eev-videolinks.el\" \"first-class-videos\")
;; (find-eev \"eev-videolinks.el\" \"second-class-videos\")

{defuns}
")))

;; «ee-1stclassvideo-basicinfo»  (to ".ee-1stclassvideo-basicinfo")
;; Tests: (find-estring (ee-1stclassvideo-basicinfo "eev2021"))
;;        (find-estring (ee-1stclassvideo-basicinfo "eev2021" "01:45"))
;;        (find-estring (ee-1stclassvideo-basicinfo "2021ssr"))
;;
(defun ee-1stclassvideo-basicinfo (c &optional pos)
  (ee-let*-macro-1stclassvideo-c
   c
   (let* ((yttime      (if pos (ee-time-to-youtube-time pos) ""))
	  (hsubsurl    (and hsubs (ee-1stclassvideos-hsubsurl c pos)))
	  (hsubsline0  ";; HSubs: {hsubsurl}\n")
	  (hsubsline   (if hsubs (ee-template0 hsubsline0) "")))
     (ee-template0 "\
;; Title: {title}
;; MP4:   {mp4}
;; YT:    {yt}{yttime}
;; Page:  {page}
{hsubsline}\
;; Comment: {comment}
;; Date:    {date}
;; Length:  {length}
"))))

;; «ee-1stclassvideo-basicsexps»  (to ".ee-1stclassvideo-basicsexps")
;; Tests: (find-estring (ee-1stclassvideo-basicsexps "eev2021"))
;;        (find-estring (ee-1stclassvideo-basicsexps "2021ssr"))
;;
(defun ee-1stclassvideo-basicsexps (c &optional pos)
  (ee-let*-macro-1stclassvideo-c
   c
   (let* ((init        (or pos subsinit "00:00"))
	  (hsubs-t0    ";; HSubs: (find-{c}hsubs \"{init}\")\n")
	  (lsubs-t0    ";; LSubs: (find-{c}lsubs \"{init}\")\n")
	  (index-t0    ";; Index: (find-1stclassvideo-index \"{c}\")\n")
	  (hsubs-sexp  (if hsubs    (ee-template0 hsubs-t0)    ""))
	  (lsubs-sexp  (if hassubs  (ee-template0 lsubs-t0)    ""))
	  (index-sexp  (if hasindex (ee-template0 index-t0)    "")))
     (ee-template0 "\
;; Play:  (find-{c}video \"{init}\")
{hsubs-sexp}\
{lsubs-sexp}\
{index-sexp}\
;; Info:  (find-1stclassvideo-links \"{c}\")
;;        (find-1stclassvideo-def   \"{c}\")
"))))

;; «ee-1stclassvideo-dlsubs»  (to ".ee-1stclassvideo-dlsubs")
;; Tests: (find-estring (ee-1stclassvideo-dlsubs0 "eev2021" nil nil))
;;        (find-estring (ee-1stclassvideo-dlsubs0 "eev2021" nil t))
;;        (find-estring (ee-1stclassvideo-dlsubs0 "eev2021" t nil))
;;        (find-estring (ee-1stclassvideo-dlsubs0 "eev2021" t t))
;;        (find-estring (ee-1stclassvideo-dlsubs  "eev2021"))
;;        (find-estring (ee-1stclassvideo-dlsubs  "2021ssr"))
;;
(defun ee-1stclassvideo-dlsubs (c)
  (ee-let*-macro-1stclassvideo-c
   c
   (ee-1stclassvideo-dlsubs0 c mp4found hassubs)))

(defun ee-1stclassvideo-dlsubs0 (c haslocal hassubs)
  "An internal function used by `find-1stclassvideo-links'."
  (let* ((template-nolocal-nosubs "\
;; You don't have a local copy of this video.
;; To download a local copy, run this:
;;       (find-1stclassvideo-psne \"{c}\")
;; This video doesn't have subtitles.\n")
	 (template-nolocal-hassubs "\
;; You don't have a local copy of this video.
;; To download a local copy of the video with subtitles, run:
;;       (find-1stclassvideo-psne \"{c}\")\n")
	 (template-haslocal-nosubs "\
;; We have a local copy of this video.
;; This video doesn't have subtitles.\n")
	 (template-haslocal-hassubs "\
;; You have a local copy of this video.
;; The upstream copy of this video has subtitles.
;; If you don't have a local copy of its subtitles, or if you
;; want to update the local copy of the subtitles, run this:
;;        (find-1stclassvideo-psne \"{c}\")\n"))
    (if haslocal
	(if hassubs
	    (ee-template0 template-haslocal-hassubs)
	  (ee-template0   template-haslocal-nosubs))
      (if hassubs
	  (ee-template0   template-nolocal-hassubs)
	(ee-template0     template-nolocal-nosubs)))))

;; «ee-1stclassvideo-defuns»  (to ".ee-1stclassvideo-defuns")
;; Tests: (find-estring-elisp (ee-1stclassvideo-defuns "eev2021"))
;;        (find-estring-elisp (ee-1stclassvideo-defuns "2021ssr"))
;; (find-eev "eev-videolinks.el" "code-lsubs")
;;
(defun ee-1stclassvideo-defuns (c)
  (ee-let*-macro-1stclassvideo-c
   c
   (if hassubs
       (ee-template0 "\
;; From: (find-code-1stclassvideo \"{c}\")
;;  See: (find-1stclassvideo-links \"{c}\")
;;
(defun find-{c}video (&optional time &rest rest)
  (interactive)
  (find-1stclassvideo-video \"{c}\" time))

(defun find-{c}lsubs (&optional time &rest rest)
  (interactive)
  (apply 'find-1stclassvideo-lsubs \"{c}\" time rest))

(defun find-{c}hsubs (&optional time &rest rest)
  (interactive)
  (find-1stclassvideo-hsubs \"{c}\" time))
")
     (ee-template0 "\
;; From: (find-code-1stclassvideo \"{c}\")
;;  See: (find-1stclassvideo-links \"{c}\")
;;
(defun find-{c}video (&optional time &rest rest)
  (interactive)
  (find-1stclassvideo-video \"{c}\" time))
"))))



;; «code-1stclassvideo»  (to ".code-1stclassvideo")
;; Skel:  (find-code-xxx-links "1stclassvideo" "c" "")
;; Tests: (find-code-1stclassvideo "eev2021")
;;        (ee-1stclassvideo-defuns "2021ssr"))
;;
(defun      code-1stclassvideo (c)
  "Define the \"strange functions\" associated to the video C.
When C is the identifier of a first-class video with subtitles,
like \"eev2021\", define three strange functions:
`find-{c}video', `find-{c}hsubs', and `find-{c}lsubs'. When C is
the identifier of a first-class video without subtitles, define
only `find-{c}video'."
  (eval (ee-read      (ee-code-1stclassvideo c))))

(defun find-code-1stclassvideo (c)
  (find-estring-elisp (ee-code-1stclassvideo c)))
(defun   ee-code-1stclassvideo (c)
  (ee-1stclassvideo-defuns c))



;; «code-1stclassvideos»  (to ".code-1stclassvideos")
;; Skel:  (find-code-xxx-links "1stclassvideos" "cs" "")
;; Tests: (find-code-1stclassvideos '("eev2021" "2021ssr"))
;;        (find-code-1stclassvideos)
;; See:   (find-eev "eev-videolinks.el" "strange-functions")
;;
(defun      code-1stclassvideos (&optional cs)
  "Define the \"strange functions\" associated to the videos in CS.
CS must be a list of identifiers of first-class videos. When CS
is nil, use the result of (ee-1stclassvideos)."
  (eval (ee-read      (ee-code-1stclassvideos cs))))

(defun find-code-1stclassvideos (&optional cs)
  (find-estring-elisp (ee-code-1stclassvideos cs)))
(defun   ee-code-1stclassvideos (&optional cs)
  (concat
   ";; See: (find-eev \"eev-tlinks.el\" \"code-1stclassvideos\")\n"
   ";;      (find-strange-functions-intro \"2. Here\")\n\n\n"
   (mapconcat 'ee-code-1stclassvideo
	      (or cs (ee-1stclassvideos))
	      "\n\n")))






;; «find-1stclassvideoindex»  (to ".find-1stclassvideoindex")
;; «find-1stclassvideohsubs»  (to ".find-1stclassvideohsubs")
;; «find-1stclassvideolsubs»  (to ".find-1stclassvideolsubs")
;; «find-1stclassvideodef»    (to ".find-1stclassvideodef")
;; Tests: (find-1stclassvideoindex "2022findeevangg")
;;        (find-1stclassvideoindex "2022findelispintro")
;;        (find-1stclassvideohsubs "2022findeevangg")
;;        (find-1stclassvideohsubs "2022findeevangg" "15:14")
;;        (find-1stclassvideolsubs "2022findeevangg")
;;        (find-1stclassvideolsubs "2022findeevangg" "15:14")
;;        (find-1stclassvideolsubs "2022findeevangg" "nice -")
;;        (find-1stclassvideodef   "2022findelispintro")
;;
(defun find-1stclassvideoindex (c &rest pos-spec-list)
  (interactive (list (ee-1stclassvideo-around-point-ask)))
  (apply 'find-anggwgeta-elisp ".emacs.videos" c pos-spec-list))

(defun find-1stclassvideolsubs (c &rest pos-spec-list)
  (interactive (list (ee-1stclassvideo-around-point-ask)))
  (let ((mp4stem (ee-1stclassvideos-mp4stem c)))
    (apply 'find-anggwget-elisp (format "SUBTITLES/%s.lua" mp4stem)
	   pos-spec-list)))

(defun find-1stclassvideohsubs (c &optional pos &rest pos-spec-list)
  (interactive (list (ee-1stclassvideo-around-point-ask)))
  (find-googlechrome (ee-1stclassvideos-hsubsurl c pos)))

(defun find-1stclassvideodef (c &rest pos-spec-list)
  (interactive (list (ee-1stclassvideo-around-point-ask)))
  (apply 'find-eev "eev-videolinks.el" c pos-spec-list))

(defalias 'find-1stclassvideo-index 'find-1stclassvideoindex)
(defalias 'find-1stclassvideo-lsubs 'find-1stclassvideolsubs)
(defalias 'find-1stclassvideo-hsubs 'find-1stclassvideohsubs)
(defalias 'find-1stclassvideo-def   'find-1stclassvideodef)



;; «find-1stclassvideos»  (to ".find-1stclassvideos")
;; Test: (find-1stclassvideos)
;;
(defun find-1stclassvideos (&rest rest)
  "Visit a temporary buffer with a list of all first-class videos of eev."
  (interactive)
  (let* ((ee-buffer-name (or ee-buffer-name "*(find-1stclassvideos)*"))
	 (f (lambda (s) (list 'find-1stclassvideo-links s)))
	 (finds (mapcar f (ee-1stclassvideos))))
    (apply 'find-elinks-elisp
	   `((find-1stclassvideos)
	     ,@finds)
	   rest)))

;; «1c»  (to ".1c")
;; «aliases»  (to ".aliases")
;; See: (find-eev "eev-aliases.el" "1stclassvideos")



;;;      _       _                                      
;;;   __| | ___ | |_       ___ _ __ ___   __ _  ___ ___ 
;;;  / _` |/ _ \| __|____ / _ \ '_ ` _ \ / _` |/ __/ __|
;;; | (_| | (_) | ||_____|  __/ | | | | | (_| | (__\__ \
;;;  \__,_|\___/ \__|     \___|_| |_| |_|\__,_|\___|___/
;;;                                                     
;; «find-dot-emacs-links»  (to ".find-dot-emacs-links")
;; Skel: (find-find-links-links-new "dot-emacs" "opts" "ee-buffer-name")
;; Test: (find-dot-emacs-links)
;;  See: http://anggtwu.net/2024-find-dot-emacs-links.html
;;
(defun find-dot-emacs-links (&optional opts &rest pos-spec-list)
"Visit a temporary buffer containing hyperlinks for dot-emacs."
  (interactive)
  (setq opts (or opts "eev"))
  (let* ((body (ee-dot-emacs-concat opts))
         (ee-buffer-name "*(find-dot-emacs-links)*"))
    (apply
     'find-elinks-elisp
     `((find-dot-emacs-links ,opts ,@pos-spec-list)
       (find-dot-emacs-links "eev angges")
       (find-dot-emacs-links "eev angges edrxmaxima mfms")
       (find-dot-emacs-links "eev angges melpa lean4 edrxmaxima mfms")
       ;; Convention: the first sexp always regenerates the buffer.
       (find-efunction 'find-dot-emacs-links)
       ";;"
       ,(ee-template0 "\
;; See: http://anggtwu.net/2024-find-dot-emacs-links.html
;;      (find-dot-emacs-intro)


;; (ee-copy-rest-3 nil \";;--end\" \"~/.emacs\")
{body}\
;;--end
")
       )
     pos-spec-list)))

;; Test: (find-estring-elisp (ee-dot-emacs-concat "eev mfms"))
(defun ee-dot-emacs-concat (opts)
  (string-join 
   (cl-loop for opt in (ee-split opts)
            for sym = (ee-intern "ee-dot-emacs-%s" opt)
            collect (funcall sym))
   "\n"))

;; Test: (find-estring-elisp (ee-dot-emacs-eevgit))
(defun ee-dot-emacs-eevgit (&rest rest) "\
;; See: (find-package-vc-install-links \"https://github.com/edrx/eev\")
(add-to-list 'load-path \"~/.emacs.d/elpa/eev\")
")

;; Test: (find-estring-elisp (ee-dot-emacs-eev))
(defun ee-dot-emacs-eev (&rest rest) "\
;; See: (find-eev-levels-intro)
(require 'eev-load)               ; (find-eev \"eev-load.el\")
(require 'eev-aliases)            ; (find-eev \"eev-aliases.el\")
(eev-mode 1)                      ; (find-eev \"eev-mode.el\")
")

;; Test: (find-estring-elisp (ee-dot-emacs-angges))
(defun ee-dot-emacs-angges (&rest rest) "\
;; From: (find-angg-es-links)
(defun find-angg (fname &rest rest)
  (apply 'find-wgeta (format \"http://anggtwu.net/%s\" fname) rest))
(defun find-anggfile (fname &rest rest)
  (apply 'find-wget  (format \"http://anggtwu.net/%s\" fname) rest))
(defun find-es (fname &rest rest)
  (apply 'find-wgeta (format \"http://anggtwu.net/e/%s.e\" fname) rest))
")

;; Test: (find-estring-elisp (ee-dot-emacs-melpa))
(defun ee-dot-emacs-melpa (&rest rest) "\
;; From: (find-melpa-links)
(require 'package)
(add-to-list 'package-archives
  '(\"melpa\" . \"https://melpa.org/packages/\"))
")

;; Test: (find-estring-elisp (ee-dot-emacs-edrxmaxima))
(defun ee-dot-emacs-edrxmaxima (&rest rest) "\
;; From: (find-windows-beginner-intro \"8. Test Maxima with find-wget\")
;;  See: (find-windows-beginner-intro \"12. Install qdraw\")
;;       (find-fline \"~/MAXIMA/edrx-maxima.el\")
(code-c-d \"maxima\" \"/usr/share/maxima/5.47.0/\" \"maxima\")
(ignore-errors (load \"~/MAXIMA/edrx-maxima.el\"))
")

;; Test: (find-estring-elisp (ee-dot-emacs-maxima5470))
(defun ee-dot-emacs-maxima5470 (&rest rest) "\
;; From: (find-windows-beginner-intro \"8. Test Maxima with find-wget\")
;;  and: (find-windows-beginner-intro \"12. Install qdraw\")
(code-c-d \"maxima\" \"/usr/share/maxima/5.47.0/\" \"maxima\")
(add-to-list 'load-path \"~/MAXIMA/\")
(add-to-list 'auto-mode-alist '(\"\\\\.mac$\"  . maxima-mode))
(autoload 'maxima-mode \"maxima\" nil t)
")

;; Test: (find-estring-elisp (ee-dot-emacs-epl))
(defun ee-dot-emacs-epl (&rest rest) "\
;; See: (find-eepitch-intro \"3.3. `eepitch-preprocess-line'\")
(defvar eepitch-preprocess-regexp \"^\")
(setq   eepitch-preprocess-regexp \"^\")
(defun eepitch-preprocess-line (line)
  (replace-regexp-in-string eepitch-preprocess-regexp \"\" line))
")

;; Test: (find-estring-elisp (ee-dot-emacs-lean4))
(defun ee-dot-emacs-lean4 (&rest rest) "\
;; See: (find-lean4-intro)
(add-to-list 'load-path \"~/.emacs.d/elpa/lean4-mode\")
(defun fli () (interactive) (find-lean4-intro))
(defun el4 () (interactive) (find-eev \"eev-lean4.el\"))
(require 'eev-lean4)      ; (find-eev \"eev-lean4.el\")
")

;; Test: (find-estring-elisp (ee-dot-emacs-mfms))
(defun ee-dot-emacs-mfms (&rest rest) "\
;; From: (find-mpv-links)
;;  See: http://anggtwu.net/eev-videos.html#smaller-fullscreen
;;
(defun mf ()
  \"Make mpv use (real) full screen.\"
  (interactive)
  (setq ee-mpv-video-options '(\"--fs\" \"--osd-level=2\")))
;;
(defun ms ()
  \"Make mpv use a \\\"smaller full screen\\\".\"
  (interactive)
  (setq ee-mpv-video-options
	'(\"--fs\" \"--osd-level=2\"
	  \"--video-margin-ratio-bottom=0.15\"
	  \"--sub-font-size=35\")))
")

;; Test: (find-estring-elisp (ee-dot-emacs-sly))
(defun ee-dot-emacs-sly (&rest rest) "\
;; See: (find-try-sly-intro \"3. Adjust your ~/.emacs\")
(code-c-d \"ql\" \"~/quicklisp/\")
(code-c-d \"qlreleases\" \"~/quicklisp/dists/quicklisp/installed/releases/\")
(code-c-d \"qlsystems\"  \"~/quicklisp/dists/quicklisp/installed/systems/\")
(code-c-d \"qlsoftware\" \"~/quicklisp/dists/quicklisp/software/\")
(code-c-d \"sly\" (ee-locate-library \"sly.el\") \"sly\")
(code-c-d \"slynk\" (ee-slyfile \"slynk/\"))
")



;;;            _       _          
;;;   __ _  __| |_   _(_) ___ ___ 
;;;  / _` |/ _` \ \ / / |/ __/ _ \
;;; | (_| | (_| |\ V /| | (_|  __/
;;;  \__,_|\__,_| \_/ |_|\___\___|
;;;                               
;; «find-advicebefore-links»  (to ".find-advicebefore-links")
;; Skel: (find-find-links-links-new "advicebefore" "fun" "")
;; Test: (find-advicebefore-links)
;;       (find-advicebefore-links "FOO")
;; Screenshot: http://anggtwu.net/IMAGES/find-advicebefore-links.png
;;
;; I wrote this when I was trying to learn edebug and I was failing
;; miserably.
;;
(defun find-advicebefore-links (&optional fun &rest pos-spec-list)
"Visit a temporary buffer containing code for adding advice to FUN.
The temporary buffer will contain code for advice-add'ing a
logging function to FUN, for advice-remove'ing the logging
function, and for inspecting the log. See the demo to understand
how this works."
  (interactive)
  (setq fun (or fun "{fun}"))
  (apply
   'find-elinks-elisp
   `((find-advicebefore-links ,fun ,@pos-spec-list)
     ;; Convention: the first sexp always regenerates the buffer.
     (find-efunction 'find-advicebefore-links)
     ";;"
     ,(ee-template0 "\
;; See: (find-elnode \"Advice Combinators\")
;; Demo: http://anggtwu.net/elisp/find-advicebefore-links.el.html
;;       http://anggtwu.net/elisp/find-advicebefore-links.el
;; (find-wget-elisp \"http://anggtwu.net/elisp/find-advicebefore-links.el\")

(setq  ee-log nil)
(defun ee-log (f r) (setq ee-log (cons (cons f r) ee-log)))
(defun ee-log-{fun} (&rest r) (ee-log '{fun} r))
(advice-add   '{fun}  :before 'ee-log-{fun})
;;
(advice-remove '{fun}         'ee-log-{fun})
(find-2a nil '(find-eppp ee-log))
")
     )
   pos-spec-list))





;; «find-osm-links»  (to ".find-osm-links")
;; Skel: (find-find-links-links-new "osm" "lat lon zoom" "")
;; Test: (find-osm-links)
;;       (find-osm-links  43.7731  11.2562 17 "Il Duomo")
;;  See: (find-eev "eev-plinks.el" "find-osm")
;;
(defun find-osm-links (&optional lat lon zoom &rest pos-spec-list)
"Visit a temporary buffer containing hyperlinks for OpenStreetMap viewer."
  (interactive)
  (setq lat  (or lat  (ee-osm-lat)  "{lat}"))
  (setq lon  (or lon  (ee-osm-lon)  "{lon}"))
  (setq zoom (or zoom (ee-osm-zoom) "{zoom}"))
  (apply
   'find-elinks
   `((find-osm-links ,lat ,lon ,zoom ,@pos-spec-list)
     ;; Convention: the first sexp always regenerates the buffer.
     (find-efunction 'find-osm-links)
     ""
     ,@(ee-find-osm-links lat lon zoom)
     )
   pos-spec-list))

(defun ee-find-osm-links (&optional lat lon zoom)
  "An internal function used by `find-osm-links'."
  (setq lat  (or lat  (ee-osm-lat)  "{lat}"))
  (setq lon  (or lon  (ee-osm-lon)  "{lon}"))
  (setq zoom (or zoom (ee-osm-zoom) "{zoom}"))
  ;;
  `((find-osm ,lat ,lon ,zoom)
    (osm-goto ,lat ,lon ,zoom)
    (find-efunction 'find-osm)
    (find-efunction 'osm-goto)
    ""
    ,(ee-template0 "\
# (find-osm-str \"{lat},{lon},{zoom}\")
# https://www.google.com.br/maps/@{lat},{lon},{zoom}z?hl=en")
    ))

(defun ee-osm-lat  () (if (eq major-mode 'osm-mode) osm--lat))
(defun ee-osm-lon  () (if (eq major-mode 'osm-mode) osm--lon))
(defun ee-osm-zoom () (if (eq major-mode 'osm-mode) osm--zoom))





;; «find-pip3-links»  (to ".find-pip3-links")
;; Skel: (find-find-links-links-new "pip3" "pkg" "pkg_")
;; Test: (find-pip3-links "youtube-transcript-downloader")
;; See:  http://anggtwu.net/find-yttranscript-links.html
;;
(defun find-pip3-links (&optional pkg &rest pos-spec-list)
"Visit a temporary buffer containing a script for pip3."
  (interactive)
  (setq pkg (or pkg "{pkg}"))
  (let* ((pkg_ (string-replace "-" "_" pkg)))
    (apply
     'find-elinks
     `((find-pip3-links ,pkg ,@pos-spec-list)
       ;; Convention: the first sexp always regenerates the buffer.
       (find-efunction 'find-pip3-links)
       ""
       ,(ee-template0 "\
# https://pip.pypa.io/en/stable/
# (find-status   \"python3-pip\")
# (find-vldifile \"python3-pip.list\")
# (find-udfile   \"python3-pip/\")
# (find-sh \"pip3 list\" \"{pkg}\")
# (find-sh \"pip3 show {pkg}\")
# (find-importlib-links \"{pkg}\")
# (find-pypi-links \"{pkg}\")

 (eepitch-shell)
 (eepitch-kill)
 (eepitch-shell)
sudo apt-get install python3-pip

 (eepitch-shell)
 (eepitch-kill)
 (eepitch-shell)
# (find-man \"1 pip3\")
pip3 show {pkg}
pip3 install {pkg}
")
       )
     pos-spec-list)))



;; «find-yttranscript-links»  (to ".find-yttranscript-links")
;; Skel: (find-find-links-links-new "yttranscript" "c hash" "")
;; Test: (find-yttranscript-links)
;;       (find-yttranscript-links "acmetour" "dP1xVpMPn8M")
;; See:  http://anggtwu.net/find-yttranscript-links.html
;; Based on code by Bruno Macedo.
;;
(defun find-yttranscript-links (&optional c hash &rest pos-spec-list)
"Display a temporary script that downloads a transcript from youtube."
  (interactive (list nil (ee-youtubedl-hash-around-point)))
  (setq c (or c "{c}"))
  (setq hash (or hash "{hash}"))
  (apply
   'find-elinks
   `((find-yttranscript-links ,c ,hash ,@pos-spec-list)
     ;; Convention: the first sexp always regenerates the buffer.
     (find-efunction 'find-yttranscript-links)
     ""
     ,(ee-template0 "\
# (find-pip3-links \"youtube_transcript_api\")

 (python-mode)
 (eepitch-python)
 (eepitch-kill)
 (eepitch-python)
from youtube_transcript_api import YouTubeTranscriptApi
from math import floor

m       = lambda n: n  % 60
d       = lambda n: n // 60
mmss    = lambda n:      \"%d:%02d\" %           (d(n), m(n))
hhmmss  = lambda n: \"%d:%02d:%02d\" % (d(d(n)),m(d(n)),m(n))
timestr = lambda n: mmss(n) if n < 3600 else hhmmss(n)
sexp    = lambda n,text: '%s(%s \"%s\" \"%s\")' % (p, f, timestr(floor(n)), text)

p       = \"% \"
f       = \"find-{c}video\"

trlist  = YouTubeTranscriptApi.list_transcripts(\"{hash}\")
trlist1 = [x for x in trlist]
trlist2 = [str(x) for x in trlist]
trlist3 = '\\n'.join(trlist2)
print(trlist3)

tr      = YouTubeTranscriptApi.get_transcript(\"{hash}\", languages=['en-GB'])
tr      = YouTubeTranscriptApi.get_transcript(\"{hash}\")

times   = [i['start'] for i in tr]
texts   = [i['text']  for i in tr]
sexps   = [sexp(n,text) for n,text in zip(times,texts)]
print('\\n'.join(sexps))
")
     )
   pos-spec-list))



;; «find-yttranscript0-links»  (to ".find-yttranscript0-links")
;; Skel: (find-find-links-links-new "yttranscript" "c hash" "")
;; Test: (find-yttranscript0-links)
;;       (find-yttranscript0-links "acmetour" "dP1xVpMPn8M")
;; See:  http://anggtwu.net/find-yttranscript-links.html
;;
(defun find-yttranscript0-links (&optional c hash &rest pos-spec-list)
"Display a temporary script that downloads a transcript from youtube.\n
This is the original version, described in:
  http://anggtwu.net/find-yttranscript-links.html
it uses `youtube_transcript_downloader', that is broken. See:
  https://github.com/t4skmanag3r/youtube_transcript_downloader/issues/2\n
Use `find-yttranscript-links' instead."
  ;; (interactive (list nil (ee-youtubedl-hash-around-point)))
  (setq c (or c "{c}"))
  (setq hash (or hash "{hash}"))
  (let ((ee-buffer-name (or ee-buffer-name "*find-yttranscript0-links*")))
    (apply
     'find-elinks
     `((find-yttranscript0-links ,c ,hash ,@pos-spec-list)
       (find-yttranscript-links ,c ,hash ,@pos-spec-list)
       ;; Convention: the first sexp always regenerates the buffer.
       (find-efunction 'find-yttranscript0-links)
       ""
       ,(ee-template0 "\
# (find-pip3-links \"youtube-transcript-downloader\")
# (find-youtubedl-links nil nil \"{hash}\" nil \"{c}\")

 (python-mode)
 (eepitch-python)
 (eepitch-kill)
 (eepitch-python)
import youtube_transcript_downloader
url    = \"http://www.youtube.com/watch?v={hash}\"
f      = \"find-{c}video\"
tr     = youtube_transcript_downloader.get_transcript(url)
trits0 = tr.items()
trits1 = '\\n'.join((key + ' ' + text for key, text in trits0))
trits1 = '\\n'.join(('% (' + f + ' \"' + key + '\" \"' + text + '\")' for key, text in trits0))
print(trits1)

")
       )
     pos-spec-list)))





;; «find-importlib-links»  (to ".find-importlib-links")
;; Skel: (find-find-links-links-new "importlib" "pkg distr" "pkg_")
;; Tests: (find-importlib-links)
;;        (find-importlib-links "requests")
;;        (find-importlib-links "PIL")
;;        (find-importlib-links "PIL" "pillow")
;; See:   http://anggtwu.net/find-yttranscript-links.html
;;
(defun find-importlib-links (&optional pkg distr &rest pos-spec-list)
"Visit a temporary buffer containing hyperlinks for importlib."
  (interactive)
  (setq pkg (or pkg "{pkg}"))
  (setq distr (or distr "{distr}"))
  (let* ((pkg_ (string-replace "-" "_" pkg)))
    (apply
     'find-elinks
     `((find-importlib-links ,pkg ,distr ,@pos-spec-list)
       ;; Convention: the first sexp always regenerates the buffer.
       (find-efunction 'find-importlib-links)
       ""
       (find-pip3-links ,pkg)
       (find-pypi-links ,pkg)
       (find-pydoc "library/importlib")
       (find-pydoc "library/importlib.metadata")
       ""
       ,(ee-template0 "\
 (python-mode)
 (eepitch-python)
 (eepitch-kill)
 (eepitch-python)
import importlib.metadata, pprint
import {pkg_}
{pkg_}.__file__
dist = importlib.metadata.distribution('{distr}')
dist = importlib.metadata.distribution('{pkg}')
pprint.pprint(dist.files)
md = dist.metadata
print('\\n'.join(md.keys()))
md['Name']
md['Version']
md['Summary']
md['Home-Page']
")
       )
     pos-spec-list)))


;; «find-pypi-links»  (to ".find-pypi-links")
;; Skel: (find-find-links-links-new "pypi" "pkg" "pkg_")
;; Test: (find-pypi-links)
;;       (find-pypi-links "requests")
;; See:  http://anggtwu.net/find-yttranscript-links.html
;;
(defun find-pypi-links (&optional pkg &rest pos-spec-list)
"Visit a temporary buffer containing hyperlinks for pypi."
  (interactive)
  (setq pkg (or pkg "{pkg}"))
  (let* ((pkg_ (string-replace "-" "_" pkg)))
    (apply
     'find-elinks
     `((find-pypi-links ,pkg ,@pos-spec-list)
       ;; Convention: the first sexp always regenerates the buffer.
       (find-efunction 'find-pypi-links)
       ""
       ,(ee-template0 "\
# https://warehouse.pypa.io/api-reference/json.html
# https://requests.readthedocs.io/en/latest/
# (find-pip3-links \"requests\")
# (find-importlib-links \"requests\")
# (find-sh \"curl -s https://pypi.org/pypi/{pkg}/json | jq\")

 (python-mode)
 (eepitch-python)
 (eepitch-kill)
 (eepitch-python)
import requests, pprint
r = requests.get('https://pypi.org/pypi/{pkg}/json')
d = r.json()
pprint.pprint(d['info'])
pprint.pprint(d['info']['home_page'])
")
       )
     pos-spec-list)))




;;;                       _ 
;;;  _ __   _____   _____| |
;;; | '_ \ / _ \ \ / / _ \ |
;;; | | | | (_) \ V /  __/ |
;;; |_| |_|\___/ \_(_)___|_|
;;;                         
;; «find-nov-links»  (to ".find-nov-links")
;; Skel:  (find-find-links-links-new "nov" "fname" "")
;; Tests: (find-elinks-elisp (ee-find-nov-links "FOO.epub"))
;;                              (find-nov-links "FOO.epub")
;;                              (find-nov-links)
;;
(defun find-nov-links (&optional fname &rest pos-spec-list)
"Visit a temporary buffer containing hyperlinks for nov."
  (interactive (list (if (ee-nov-bufferp) buffer-file-truename)))
  (setq fname (or fname "{fname}"))
  (apply
   'find-elinks-elisp
   `((find-nov-links ,fname ,@pos-spec-list)
     ;; Convention: the first sexp always regenerates the buffer.
     (find-efunction 'find-nov-links)
     ""
     ,@(ee-find-nov-links fname)
     )
   pos-spec-list))

(defun ee-find-nov-links (&optional fname)
  (setq fname (or fname (if (ee-nov-bufferp) buffer-file-truename)))
  (setq fname (or fname "{fname}"))
  (list (ee-template0 "\
;; See: (find-epackage-links 'nov)
;;      https://depp.brause.cc/nov.el/
;; Needs: (add-to-list 'auto-mode-alist '(\"\\\\.epub\\\\'\" . nov-mode))
;;
(find-fline {(ee-S fname)})
(find-nov {(ee-S fname)})
")))

(defun find-nov (fname &rest rest)
  "Open FNAME in nov.el mode. See: (find-nov-links)
This doesn't support pos-spec lists yet."
  (find-fline fname))


;;;  _       _                  
;;; | |_ ___| | ___  __ _  __ _ 
;;; | __/ _ \ |/ _ \/ _` |/ _` |
;;; | ||  __/ |  __/ (_| | (_| |
;;;  \__\___|_|\___|\__, |\__,_|
;;;                 |___/       
;;
;; «find-telegachat-links»  (to ".find-telegachat-links")
;; Skel: (find-find-links-links-new "telegachat" "bufname" "")
;; Test: (find-telegachat-links "◀<Emacs News and Posts@emacs_posts>")
;; See:  (find-eev "eev-plinks.el" "find-telegachat")
;;
(defun find-telegachat-links (&optional bufname &rest pos-spec-list)
"Visit a temporary buffer containing hyperlinks for telegachat."
  (interactive (list (buffer-name)))
  (setq bufname (or bufname "{bufname}"))
  (apply
   'find-elinks
   `((find-telegachat-links ,bufname ,@pos-spec-list)
     ;; Convention: the first sexp always regenerates the buffer.
     (find-efunction 'find-telegachat-links)
     ""
     ,@(ee-find-telegachat-links bufname)
     )
   pos-spec-list))

(defun ee-find-telegachat-links (&optional bufname msgn)
  (setq bufname (or bufname (buffer-name)))
  (setq msgn    (or msgn (ee-telega-msgn)))
  (let* ((chat        (with-current-buffer bufname telega-chatbuf--chat))
         (username    (telega-chat-username chat))
         (atusername  (if username (format "@%s" username)))
         (atusernamen (format "%s#%s" atusername msgn))
         (title       (plist-get chat :title))
         (id          (plist-get chat :id))
         (idn         (format "%s#%s" id msgn))
	 )
    `((find-telegachat ,atusername)
      (find-telegachat ,atusername ,title)
      (find-telegachat ,atusernamen)
      (find-telegachat ,atusernamen ,title)
      (find-telegachatm ,atusernamen)
      (find-telegachat ,id)
      (find-telegachat ,id ,title)
      (find-telegachat ,idn)
      (find-telegachat ,idn ,title)
      (find-telegachatm ,idn)
      ""
      (find-ebuffer ,bufname)
      (setq tcbuf ,bufname)
      (setq tcchat (with-current-buffer tcbuf telega-chatbuf--chat))
      (setq tcmsg (with-current-buffer tcbuf (telega-msg-at (point))))
      (setq tcmsgc (plist-get (plist-get (plist-get tcmsg :content) :text) :text))
      (setq tcmsgc (ee-no-properties (telega-msg-content-text tcmsg)))
      (find-einspect tcbuf)
      (find-einspect tcchat)
      (find-einspect tcmsg)
      (find-einspect tcmsgc)
      (telega-tme-internal-link-to tcchat)
      (find-eppp tcchat)
      (plist-get tcchat :chat_id)
      (plist-get tcchat :title)
      (plist-get tcchat :id)
      (find-eppp tcmsg)
      (find-estring tcmsgc)
      )))

(defun ee-telega-msgn (&optional msg)
  (setq msg (or msg (telega-msg-for-interactive)))
  (/ (plist-get msg :id) telega-msg-id-step))




;; «find-eejump-links»  (to ".find-eejump-links")
;; Skel: (find-find-links-links-new "eejump" "n sexp" "strn strsexp def call")
;; Test: (find-eejump-links)
;;
(defun find-eejump-links (&optional n sexp &rest pos-spec-list)
"Visit a temporary buffer that (sort of) explains how to use eejump.
N should be either a number or a symbol; SEXP should be a sexp."
  (interactive)
  (setq n (or n 999))
  (setq sexp (or sexp '(find-elnode "")))
  (let* ((strn    (format "%s" n))
         (strsexp (ee-S sexp))
         (call    (if (numberp n)
		      (format "M-%sj" n)
		    (format "M-x %s"n )))
         (def     (ee-wrap-eejump strn strsexp)))
    (apply
     'find-elinks-elisp
     `((find-eejump-links ,(ee-add-quote n) ,(ee-add-quote sexp) ,@pos-spec-list)
       ;; Convention: the first sexp always regenerates the buffer.
       (find-efunction 'find-eejump-links)
       ""
       ";; Try: (find-eejump-links 999 '(find-elnode \"\"))"
       ";;      (find-eejump-links 'el '(find-elisp-intro))"
       ""
       ";; See:"
       (find-eev-quick-intro "7. Quick access to one-liners")
       (find-eev-quick-intro "7.3. Defining eejump targets")
       (find-eev-quick-intro "7.4. Commands with very short names")
       (find-elisp-intro "6. Defining functions")
       (find-eejump-intro "shorthand" "`M-123j'")
       (find-eejumps)
       (find-eeshortdefs)
       ""
       ,(ee-template0 "\
;; Typing `M-J' on the line below...
{strn} {strsexp}

;; ...converts it to:
{def}

;; After executing the defun above
;; you will be able to run the sexp
;;   {strsexp}
;; by typing `{call}'.
")
       )
     pos-spec-list)))



;; «find-kla-links»  (to ".find-kla-links")
;; Skel: (find-find-links-links-new "kla" "fname" "r c d lrcd")
;; Tests: (find-kla-links)
;;        (find-kla-links (ee-efile "array.el"))
;;
(defun find-kla-links (&optional fname &rest pos-spec-list)
"Visit a temporary buffer containing hyperlinks about eev-kla.el."
  (interactive)
  (setq fname (ee-expand (or fname (ee-kl-fname))))
  (let* ((r (ee-kl-r :fname fname))
         (c (ee-kl-c :fname fname))
         (d (ee-kl-d :fname fname))
         (ee-buffer-name (or ee-buffer-name "*find-kla-links*")))
    (apply
     'find-elinks-elisp
     `((find-kla-links ,fname ,@pos-spec-list)
       ;; Convention: the first sexp always regenerates the buffer.
       (find-efunction 'find-kla-links)
       ""
       ,(ee-template0 "\
;; See: (find-kla-intro \"3. The components\")
;;   fname  =>  {(ee-S fname)}
;;       c  =>  {(ee-S c)}
;;       d  =>  {(ee-S d)}
;;       r  =>  {(ee-S r)}
;;
;; See: (find-kla-intro \"2. `ee-code-c-d-pairs'\")
;;   (eek \"<down> M-2 M-e\")
;;     (find-eppp ee-code-c-d-pairs)
;;   (eek \"<down> M-3 M-e M-h M-h\")
;;     (find-fline {(ee-S fname)})
;;
;; See: (find-kla-intro \"4. The best `l-r-c-d'\")
;;   (eek \"<down> M-2 M-e\")
;;     (find-eppp (ee-kl-lrcds :fname {(ee-S fname)}))
;;                (ee-kl-lrcd  :fname {(ee-S fname)})
;;
;; Source code: (find-eev \"eev-kla.el\")
")
       )
     pos-spec-list)))



;; «find-rstdoc-links»  (to ".find-rstdoc-links")
;; Skel: (find-find-links-links-new "rstdoc" "kw" "bweb bhtml brst kil")
;; Tests: (find-rstdoc-links)
;;        (find-rstdoc-links :py)
;;
(defun find-rstdoc-links (&optional kw &rest pos-spec-list)
"Visit a temporary buffer containing hyperlinks for rstdoc."
  (interactive)
  (setq kw (or kw "{kw}"))
  (let* ((bweb   (or (ee-rstdoc-getfield0 kw :base-web)
		     "https://BASE-WEB/"
		     "{bweb}"))
         (bhtml  (or (ee-rstdoc-getfield0 kw :base-html)
		     "file:///BASE-HTML/"
		     "{bhtml}"))
         (brst   (or (ee-rstdoc-getfield0 kw :base-rst)
		     "/BASE-RST/"
		     "{brst}"))
	 (bhtml0 (replace-regexp-in-string "^file://" "" bhtml))
         (kil    (or (ee-rstdoc-getfield0 kw :kill)
		     "{kil}")))
    (let ((ee-buffer-name
	   (or ee-buffer-name
	       (format "*(find-rstdoc-links %S)*" kw))))
      (apply
       'find-elinks-elisp
       `((find-rstdoc-links ,kw ,@pos-spec-list)
	 ;; Convention: the first sexp always regenerates the buffer.
	 (find-efunction 'find-rstdoc-links)
	 ""
	 ,(ee-template0 "\
;; See: (find-rstdoc-intro \"7. `find-rstdoc-links'\")

;; <ee-rstdoc-{kw}>
;; Skel: (find-rstdoc-links {kw})
\(setq ee-rstdoc-{kw}
      '(:base      \"index\"
        :base-web  \"{bweb}\"
        :base-html \"{bhtml}\"
        :base-rst  \"{brst}\"
        :rst       \".rst\"
        :res       (\"#.*$\" \"\\\\?.*$\" \".html$\" \".txt$\" \".rst$\" \"^file://\"
                    \"^{bweb}\"
                    \"^{bhtml0}\"
                    \"^{brst}\")
        :kill      {kil}
	))

;; (find-code-rstdoc {kw})
        (code-rstdoc {kw})
")
	 )
       pos-spec-list))))


;; «find-mpv-links»  (to ".find-mpv-links")
;; Skel: (find-find-links-links-new "mpv" "" "")
;; Test: (find-mpv-links)
;; See: https://lists.gnu.org/archive/html/eev/2022-12/msg00019.html
;;
(defun find-mpv-links (&rest pos-spec-list)
"Visit a temporary buffer containing code for configuring mpv."
  (interactive)
  (apply
   'find-elinks-elisp
   `((find-mpv-links ,@pos-spec-list)
     ;; Convention: the first sexp always regenerates the buffer.
     (find-efunction 'find-mpv-links)
     ""
     ,(ee-template0 "\
;; See:  (find-evariable 'ee-mpv-video-options)
;;       (find-evardescr 'ee-mpv-video-options)
;; Idea: use commands like `M-x mf' and `M-x ms', defined below,
;;       to change, or (sort of) toggle, how eev calls mpv.
;;       To use this, copy the block below to your init file,
;;       modifying it to adjust the options and their values.

;; From: (find-mpv-links)
;; See:  http://anggtwu.net/eev-videos.html#smaller-fullscreen

(defun mf ()
  \"Make mpv use (real) full screen.\"
  (interactive)
  (setq ee-mpv-video-options '(\"--fs\" \"--osd-level=2\")))

(defun ms ()
  \"Make mpv use a \\\"smaller full screen\\\".\"
  (interactive)
  (setq ee-mpv-video-options
	'(\"--fs\" \"--osd-level=2\"
	  \"--video-margin-ratio-bottom=0.15\"
	  \"--sub-font-size=35\")))
")
     )
   pos-spec-list))



;; «find-try-sly-links»  (to ".find-try-sly-links")
;; Skel: (find-find-links-links-new "try-sly" "" "")
;; Test: (find-try-sly-links)
;;
(defun find-try-sly-links (&rest pos-spec-list)
"Visit a temporary buffer containing a script for trying Sly."
  (interactive)
  (apply
   'find-elinks
   `((find-try-sly-links ,@pos-spec-list)
     ;; Convention: the first sexp always regenerates the buffer.
     (find-efunction 'find-try-sly-links)
     ""
     "# Obsolete! Use the intro instead:"
     (find-try-sly-intro)
     ""
     ""
     ,(ee-template0 (ee-adjust-red-stars "\
 1. Install some Debian packages
 ===============================
 Note that the package \"hyperspec\" will install
 a local copy of the Common Lisp Hyperspec here:
   (find-fline \"/usr/share/doc/hyperspec/\")

 (eepitch-shell)
 (eepitch-kill)
 (eepitch-shell)
sudo apt-get install sbcl hyperspec maxima


 2. Install some elisp packages
 ==============================
 Here we use low-level sexps instead of `M-x list-packages'.
 Note that some of the sexps below take several seconds to run.

 (package-initialize)
 (add-to-list 'package-archives '(\"melpa\" . \"https://melpa.org/packages/\"))
 (package-refresh-contents)
 (package-install 'sly)
 (package-install 'clhs)


 3. Download quicklisp.lisp
 ==========================

 (eepitch-shell)
 (eepitch-kill)
 (eepitch-shell)
mkdir -p $S/https/beta.quicklisp.org/
cd       $S/https/beta.quicklisp.org/
ls -lAF
rm -fv quicklisp.lisp
wget -nc https://beta.quicklisp.org/quicklisp.lisp
# (find-fline \"$S/https/beta.quicklisp.org/\")
# (find-fline \"$S/https/beta.quicklisp.org/quicklisp.lisp\")


 4. Run quicklisp.lisp
 =====================
 Ask it to install slynk and to change ~/.sbclrc.

 (eepitch-shell)
 (eepitch-kill)
 (eepitch-shell)
cd       $S/https/beta.quicklisp.org/
sbcl --load quicklisp.lisp
  (quicklisp-quickstart:help)
  (quicklisp-quickstart:install)
  (ql:quickload :slynk)
  (ql:add-to-init-file)

  ;; Inspect the changes in ~/.sbclrc:
  ;;   (find-fline \"~/.sbclrc\")
  ;;   (find-fline \"~/.sbclrc\" \"added by ql:add-to-init-file:\")
  ;; Inspect the ~/quicklisp/ directory:
  ;;   (code-c-d \"ql\" \"~/quicklisp/\")
  ;;   (find-qlfile \"\")
  ;;   (find-qlsh \"find * | sort\")



 5. Change your ~/.emacs
 =======================
 Copy the block below to your ~/.emacs:

;; From: (find-try-sly-links)
;; The `code-rstdoc' below defines `find-clhsdoc':
;; (find-code-rstdoc :clhs)
        (code-rstdoc :clhs)
;;
;; See:   (find-status   \"hyperspec\")
;;        (find-vldifile \"hyperspec.list\")
;;        (find-udfile   \"hyperspec/\")
;;        (find-evardescr 'clhs-root)
;;        (find-evariable 'clhs-root)
;;        (find-ehashtable clhs-symbols)
;; Tests: (find-clhsdoc)
;;        (find-clhsdoc \"Body/m_loop\")
;;        (find-clhsdoci        \"loop\")
;;        (ee-clhs-lookup-index \"loop\")
;;
(setq clhs-root \"file:///usr/share/doc/hyperspec/\")

(defun ee-clhs-lookup-index (name)
  (require 'clhs)
  (gethash (upcase name) (clhs-symbols)))

(defun find-clhsdoci (name &rest rest)
  (let* ((str0 (ee-clhs-lookup-index name))
	 (str (replace-regexp-in-string \"\\\\.html?$\" \"\" str0)))
    (find-clhsdoc str)))

(code-c-d \"sbcl\" \"/usr/share/sbcl-source/\" \"sbcl\")
(code-c-d \"ql\" \"~/quicklisp/\")
;; (find-qlfile \"\")
;; (find-qlsh \"find * | sort\")



 6. Test Sly
 ===========
 One of its messages will (should?) be:
 [sly] Connecting to Slynk on port 45477.

 (eepitch-sly)
 (eepitch-kill)
 (eepitch-sly)
;; From: (find-es \"lisp\" \"sharpsign-P\")
(apropos \"PATHNAME\")
(make-pathname :name \"FOO\")
(inspect (make-pathname :name \"FOO\"))
(macroexpand '(defstruct mypoint x y))

(defvar o)
(setq o (macroexpand '(defstruct mypoint x y)))

 Now try `M-.' on a symbol name, and `M-,' to go back.
 Then try `C-c I <sexp>' - for example, `C-c I (list 2 3)'.
 These key sequences are explained in these pages:
   (find-node \"(sly)Finding definitions\" \"M-.\" \"sly-edit-definition\")
   (find-node \"(sly)Inspector\" \"C-c I\" \"sly-inspect\")
 Try also this low-level way to run the inspector:
 (eepitch-eval-at-target-window '(sly-inspect \"o\"))



 7. Tell Maxima how to load Sly
 ==============================
 Run this,
   (mkdir \"~/.maxima/\" t)
 and then copy the block below to:
   (find-fline \"~/.maxima/startsly.lisp\")

;; From: (find-try-sly-links)
;; Based on: (find-angg \".maxima/startsly.lisp\")
;;
(load #P\"~/quicklisp/setup.lisp\")
(ql:quickload :slynk)
(slynk:create-server :port 56789 :dont-close t)



 8. Inspect Maxima with Sly
 ==========================
 Note that here we have two eepitch targets,
 and we alternate between them...

 (eepitch-maxima)
 (eepitch-kill)
 (eepitch-maxima)
load(\"startsly\");
 (sly-connect \"localhost\" 56789)
 (eepitch-sly)
(describe '$changevar)

 Now go to the sly-mrepl buffer, put the point
 on the \"MAXIMA::$CHANGEVAR\", and type `M-.'.


"))
     )
   pos-spec-list))




;; «find-wgetrecursive-links»  (to ".find-wgetrecursive-links")
;; Skel:  (find-find-links-links-new "wgetrecursive" "url" "lurl")
;; Tests: (find-wgetrecursive-links)
;;        (find-wgetrecursive-links "https://home.csulb.edu/~woollett/")
;;
(defun find-wgetrecursive-links (&optional url &rest pos-spec-list)
"Visit a temporary buffer containing a script for using wget --recursive."
  (interactive)
  (setq url (or url "{url}"))
  (let* ((lurl (ee-url-to-fname0 url)))
    (apply
     'find-elinks
     `((find-wgetrecursive-links ,url ,@pos-spec-list)
       ;; Convention: the first sexp always regenerates the buffer.
       (find-efunction 'find-wgetrecursive-links)
       ""
       (find-node "(wget)Recursive Download")
       (find-node "(wget)Directory-Based Limits" "--no-parent")
       (find-node "(wget)Recursive Retrieval Options" "--recursive")
       (find-sh "wget --help" "--recursive")
       (find-sh "wget --help" "--no-parent")
       (find-es "wget" "recursive")
       ""
       ,(ee-H url)
       (find-fline ,lurl)
       ""
       ,(ee-template0 "\
 (eepitch-shell)
 (eepitch-kill)
 (eepitch-shell)
rm -Rv /tmp/wget-recursive.tar
rm -Rv /tmp/wget-recursive/
mkdir  /tmp/wget-recursive/
cd     /tmp/wget-recursive/

wget --recursive --no-parent \\
  {url}
find | sort
du -c
du -ch

# (find-fline     \"/tmp/wget-recursive/\")
# (find-sh-at-dir \"/tmp/wget-recursive/\" \"find * | sort\")
# (find-fline     \"{lurl}\")
# (find-sh \"rm -Rv {lurl}\")

cd     /tmp/wget-recursive/
tar              -cvf /tmp/wget-recursive.tar .
tar -C $S/https/ -xvf /tmp/wget-recursive.tar

# (find-fline     \"{lurl}\")
# (find-sh-at-dir \"{lurl}\" \"find * | sort\")
# (find-sh \"find /tmp/wget-recursive/ | sort\")
# (find-sh \"find {lurl} | sort\")
")
       )
     pos-spec-list)))




;; «find-mbe-links»  (to ".find-mbe-links")
;; Skel: (find-find-links-links-new "mbe" "ch page pos-spec" "nch")
;; Test: (find-mbe-links)
;;       (find-mbe-links 1)
;;       (find-mbe-links 1 35 "1.8.11 Taylor and Laurent")
;;
(defun find-mbe-links (&optional ch page pos-spec &rest pos-spec-list)
"Visit a temporary buffer containing hyperlinks for Maxima by Example."
  (interactive)
  (setq ch (or ch "{ch}"))
  (setq page (or page "{page}"))
  (setq pos-spec (or pos-spec "{pos-spec}"))
  (let* ((nch (if (numberp ch) (format "%02d" ch) "{nch}")))
    (apply
     'find-elinks
     `((find-mbe-links ,ch ,page ,pos-spec ,@pos-spec-list)
       ,(ee-template0 "# (find-mbe{nch}page   {page} {(ee-S pos-spec)})")
       ,(ee-template0 "# (find-mbe{nch}text   {page} {(ee-S pos-spec)})")
       ,(ee-template0 "# (find-mbefile \"\" \"mbe{nch}\")")
       ,(ee-template0 "# (find-mbefile \"\" \"mbe{ch}\")")
       ;; Convention: the first sexp always regenerates the buffer.
       (find-efunction 'find-mbe-links)
       ""
       ,(ee-H "https://home.csulb.edu/~woollett/mbe.html")
       (find-es "maxima" "maxima-by-example-eev")
       (find-es "mbe")
       ""
       (find-angg ".emacs.papers" "maxima-by-example")
       (find-mbetext 1 ,pos-spec)
       ""
       (find-eepitch-intro "3.3. `eepitch-preprocess-line'")
       (setq eepitch-preprocess-regexp "^ *(%i[0-9]+) ?")
       (defalias 'grim 'ee-grim)
       ""
       ,(ee-template0 "\
 (eepitch-maxima)
 (eepitch-kill)
 (eepitch-maxima)
")
       )
     pos-spec-list)))

;; See the explanation here:
;;   (find-es "mbe" "grim")
(defun ee-grim ()
  "<GR>ep the <I>nput lines in <M>axima code.
This greps the lines containing strings like (%i1), (%i2), etc in
the region and displays the output in a temporary buffer."
  (interactive)
  (ee-callprocessregion '("grep" "(%i")))






;; «find-melpa-links»  (to ".find-melpa-links")
;; Skel: (find-find-links-links-new "melpa" "" "")
;; Test: (find-melpa-links)
;;
(defun find-melpa-links (&rest pos-spec-list)
"Visit a temporary buffer containing hyperlinks for melpa."
  (interactive)
  (apply
   'find-elinks-elisp
   `((find-melpa-links ,@pos-spec-list)
     ;; Convention: the first sexp always regenerates the buffer.
     (find-efunction 'find-melpa-links)
     ""
     ,(ee-template0 "\
;; See:
;; https://melpa.org/#/getting-started
;; https://www.emacswiki.org/emacs/MELPA
;; (find-epp package-archives)
;; (find-eev \"eev-tlinks.el\" \"find-try-sly-links\")

;; From: (find-melpa-links)
(require 'package)
(add-to-list 'package-archives
  '(\"melpa\" . \"https://melpa.org/packages/\"))

;; Test:
;; (package-initialize)
;; (package-refresh-contents)
;; (find-epackages)
")
     )
   pos-spec-list))




;; «find-emacsclient-links»  (to ".find-emacsclient-links")
;; Skel: (find-find-links-links-new "emacsclient" "" "")
;; Test: (find-emacsclient-links)
;;
(defun find-emacsclient-links (&rest pos-spec-list)
"Visit a temporary buffer containing hyperlinks for emacsclient."
  (interactive)
  (apply
   'find-elinks
   `((find-emacsclient-links ,@pos-spec-list)
     ;; Convention: the first sexp always regenerates the buffer.
     (find-efunction 'find-emacsclient-links)
     ""
     (find-enode "Emacs Server")
     (find-enode "Invoking emacsclient")
     (find-enode "emacsclient Options" "--eval")
     (find-pdf-like-intro "7. Shorter hyperlinks to PDF files")
     (find-pdf-like-intro "code-pdf-page")
     ""
     ,(ee-template0 "\
 (require 'server)
 (server-stop)
 (server-start)
 (list-processes)

 (eepitch-shell)
 (eepitch-kill)
 (eepitch-shell)
# (find-livesofanimalspage 3)
emacsclient --eval '(+ 1 2)'
emacsclient --eval '(find-livesofanimalspage 3)'

")
     )
   pos-spec-list))




;;;  ____  _                   ____  
;;; / ___|| |__   _____      _|___ \ 
;;; \___ \| '_ \ / _ \ \ /\ / / __) |
;;;  ___) | | | | (_) \ V  V / / __/ 
;;; |____/|_| |_|\___/ \_/\_/ |_____|
;;;                                  
;; Show2.lua is used by several of the small programs that I mentioned
;; in my talk at the EmacsConf2023 - and these functions are what we
;; use to tell Show2.lua which files and directories it should use.
;;   (find-1stclassvideo-links "eev2023repls")
;;   (find-1stclassvideodef    "eev2023repls")
;;
;; «find-show2-links»  (to ".find-show2-links")
;; Skel: (find-find-links-links-new "show2" "texfile" "texdir texfile0")
;; Test: (find-show2-links)
;;
(defun find-show2-links (&optional texfile &rest pos-spec-list)
"Visit a temporary buffer containing hyperlinks for show2."
  (interactive)
  (setq texfile (or texfile "/tmp/Show2.tex"))
  (let* ((texdir   (file-name-directory    texfile))
         (texfile0 (file-name-nondirectory texfile)))
    (apply
     'find-elinks
     `((find-show2-links ,texfile ,@pos-spec-list)
       (find-show2-links "/tmp/Show2.tex" ,@pos-spec-list)
       (find-show2-links "~/LATEX/Show2.tex" ,@pos-spec-list)
       ;; Convention: the first sexp always regenerates the buffer.
       (find-efunction 'find-show2-links)
       (find-efunction 'show2)
       ""
       ,(ee-template0 "\
 (find-angg \"LUA/Show2.lua\" \"texbody\")
 (show2-use \"{texfile}\")
 (eepitch-lua51)
 (eepitch-kill)
 (eepitch-lua51)
loadshow2()

defs.repl = true
usepackages.lipsum = [=[ \\usepackage{<}lipsum{>} ]=]

body = [[
  \\section{<}SA{>} \\lipsum[1-4]
  \\section{<}SB{>} \\lipsum[5-8]
  \\section{<}SC{>} \\lipsum[9-12]
  \\condrepl
]]
= body:show0()
= body:show()
 (etv)
= Show.log
= Show.bigstr

texbody = Dang.from \"Bla\\n<<body>>\"
= show0()
= show()
 (etv)

 (eepitch-shell)
 (eepitch-kill)
 (eepitch-shell)
cd {texdir} && REPL=1 lualatex {texfile0}
printmeaning \"newpage\"
printmeaning \"@oddfoot\"
")
       )
     pos-spec-list)
    ))


;; «show2»  (to ".show2")
;; (find-eev "eev-tlinks.el" "code-show2")
;; (find-es "show2")
(defun show2 (&optional fname0)
  (interactive)
  (find-2a nil '(find-show2-links fname0)))



;;;      _                   ____                      
;;;  ___| |__   _____      _|___ \      _   _ ___  ___ 
;;; / __| '_ \ / _ \ \ /\ / / __) |____| | | / __|/ _ \
;;; \__ \ | | | (_) \ V  V / / __/_____| |_| \__ \  __/
;;; |___/_| |_|\___/ \_/\_/ |_____|     \__,_|___/\___|
;;;                                                    
;; «show2-use»  (to ".show2-use")
;; See: (find-show2-intro "3. Show2.lua")
;;      (find-show2-intro "3. Show2.lua" "show2-use")

(defun show2-use (&optional fname0)
  "Run two `setenv's and show explanations in the right window."
  (interactive)
  (eval (ee-read (ee-show2-use fname0)))
  (find-2a nil `(find-show2-use ,fname0))
  `("SHOW2DIR"  -> ,(getenv "SHOW2DIR")
    "SHOW2STEM" -> ,(getenv "SHOW2STEM")))

(defun find-show2-use (&optional fname0 &rest pos-spec-list)
  "An internal function used by `show2-use'."
  (interactive)
  (apply 'find-estring-elisp
	 (ee-show2-use fname0)
	 pos-spec-list))

;; Skel: (find-let*-macro-links "show2-use" "fname0" "fname dir stem0 stem tex pdf cmd")
(defmacro ee-let*-macro-show2-use (fname0 &rest code)
  "An internal function used by `find-show2-use'."
  `(let* ((fname0 ,fname0)
	  (fname (ee-expand (or fname0 "/tmp/Show2.tex")))
	  (dir   (file-name-directory fname))
	  (stem0 (file-name-nondirectory
		  (file-name-sans-extension fname)))
	  (stem  (if (equal stem0 "") "Show2" stem0))
	  (tex   (concat stem ".tex"))
	  (pdf   (concat stem ".pdf"))
	  (cmd   (format "cd %s && lualatex %s.tex < /dev/null" dir stem)))
     ,@code))

(defun ee-show2-use (&optional fname0)
  "An internal function used by `show2-use'."
  (ee-let*-macro-show2-use
   fname0
   (ee-template0 "\
;; (find-show2-use {(ee-S fname0)})
;;      (show2-use {(ee-S fname0)})
;; (find-efunction 'show2-use)

;; Part 1: Lua
;; ===========
;; With the argument above `show2-use' will set its local
;; variables to:
;;
;;   Arg1:  fname0 -> {(ee-S fname0)}
;;   Vars:   fname -> {(ee-S fname)}
;;             dir -> {(ee-S dir)}
;;           stem0 -> {(ee-S stem0)}
;;            stem -> {(ee-S stem)}
;;             tex -> {(ee-S tex)}
;;             pdf -> {(ee-S pdf)}
;;             cmd -> {(ee-S cmd)}
;;
;; and it will set the environment variables SHOW2DIR and
;; SHOW2STEM to:
;;
(setenv \"SHOW2DIR\"  \"{dir}\")
(setenv \"SHOW2STEM\" \"{stem}\")
;;
;; These variables will be used by Show2.lua to determine which
;; directory and which file to work on. The values above mean
;; that Show2.lua will save the TeX code in this file,
;;
;;                {dir}{stem}.tex
;;   (find-fline \"{dir}{stem}.tex\")
;;
;; and will run this command to compile that .tex:
;;
;;   {cmd}


;; Part 2: Emacs
;; =============
;; Eepitch-ing a line like this one
;;
;;    (etv)
;;
;; should create a 3-window setting like this:
;;   _________________
;;  |        |        |
;;  |        | target |
;;  |  edit  |________|
;;  |        |        |
;;  |        |  view  |
;;  |________|________|
;;
;; The defuns below configure `v', `D' and `etv' to make them
;; display the PDF produced by Show2.lua, that will be here:
;;   {dir}{stem}.pdf
;;
(defun tb  () (interactive) (find-ebuffer (eepitch-target-buffer)))
(defun v   () (interactive) (find-pdftools-page \"{dir}{stem}.pdf\"))
(defun D   () (interactive) (find-pdf-page \"{dir}{stem}.pdf\"))
(defun etv () (interactive) (find-wset \"13o2_o_o\" '(tb) '(v)))
")))




;;;   __ _           _       _             _   _     
;;;  / _(_)_ __   __| |     | |_   _  __ _| |_| |__  
;;; | |_| | '_ \ / _` |_____| | | | |/ _` | __| '_ \ 
;;; |  _| | | | | (_| |_____| | |_| | (_| | |_| |_) |
;;; |_| |_|_| |_|\__,_|     |_|\__,_|\__,_|\__|_.__/ 
;;;                                                  
;; «find-luatb»  (to ".find-luatb")
;; See: (find-angg "LUA/PrintFunction1.lua")
;; Tests: (find-luatb "~/LUA/lua50init.lua 1072 1075  ")
;;        (find-luatb "~/LUA/lua50init.lua 914 915 field atlevel")
;;
(defun find-luatb0 (&optional fname linedef linenow &rest rest)
  "An internal function used by `find-luatb'."
  (find-fline fname)
  (let* ((ld (and (stringp linedef) (string-to-number linedef)))
         (ln (and (stringp linenow) (string-to-number linenow)))
         (ldpos (progn (ee-goto-position ld) (ee-bol)))
         (pos   (progn (ee-goto-position ln) (ee-eol)))
         (lnpos (progn (ee-goto-position (+ ln 1)) (ee-bol))))
    (eeflash ldpos lnpos ee-highlight-spec)
    (goto-char pos)))

(defun find-luatb (str &rest rest)
  "See: (find-show2-intro)
Some of the Lua modules in (find-show2-intro) use `find-luatb'."
  (find-2a nil `(apply 'find-luatb0 (ee-split ,str))))



;;;  _                                          _ 
;;; | |__  _ __ __ _ _ __  _ __   ___ _ __   __| |
;;; | '_ \| '__/ _` | '_ \| '_ \ / _ \ '_ \ / _` |
;;; | |_) | | | (_| | |_) | |_) |  __/ | | | (_| |
;;; |_.__/|_|  \__,_| .__/| .__/ \___|_| |_|\__,_|
;;;                 |_|   |_|                     
;;
;; «code-brappend»  (to ".code-brappend")
;; Skel: (find-code-xxx-links "brappend" "c url f" "")
;; Test: (code-brappend "foomanual" "file:///tmp/foobarmanual.html" 'list)
;;       (find-foomanual "#section-2" "bla")

(defun      code-brappend (c url &optional f)
  (eval (ee-read      (ee-code-brappend c url f))))
(defun find-code-brappend (c url &optional f)
  (find-estring-elisp (ee-code-brappend c url f)))
(defun   ee-code-brappend (c url &optional f)
  (setq f (or f 'brg))
  (ee-template0 "\
;; (find-code-brappend \"{<}c{>}\" \"{<}url{>}\")
;; (find-code-brappend \"{<}c{>}\" \"{<}url{>}\" 'brgl)
;;
(defun find-{c} (&optional anchor &rest rest)
  (interactive)
  ({f} (concat \"{url}\"
               (or anchor \"\"))))
"))




;;;  __  __            _                 __  __           
;;; |  \/  | __ ___  _(_)_ __ ___   __ _|  \/  |___  __ _ 
;;; | |\/| |/ _` \ \/ / | '_ ` _ \ / _` | |\/| / __|/ _` |
;;; | |  | | (_| |>  <| | | | | | | (_| | |  | \__ \ (_| |
;;; |_|  |_|\__,_/_/\_\_|_| |_| |_|\__,_|_|  |_|___/\__, |
;;;                                                 |___/ 
;;
;; «find-maximamsg-links»  (to ".find-maximamsg-links")
;; Skel:  (find-find-links-links-new "maximamsg" "n yyyymm day" "")
;; Tests: (find-maximamsg-links)
;;        (find-maximamsg "37675653 202207  1" "Stavros: structural equality")
;;
(defun find-maximamsg-links (&optional n yyyymm day &rest pos-spec-list)
"Visit a temporary buffer containing hyperlinks to a Maxima message.
A \"Maxima message\" is a message in the Maxima mailing list."
  (interactive)
  (setq n (or n 999999))
  ;; (setq yyyymm (or yyyymm "{yyyymm}"))
  ;; (setq day (or day "{day}"))
  (apply
   'find-elinks
   `((find-maximamsg-links ,n ,yyyymm ,day ,@pos-spec-list)
     ;; Convention: the first sexp always regenerates the buffer.
     (find-efunction 'find-maximamsg-links)
     (find-efunction 'find-maximamsg)
     ""
     ,@(ee-find-maximamsg-links n yyyymm day)
     )
   pos-spec-list))

(defun ee-find-maximamsg-links (&optional n yyyymm day &rest pos-spec-list)
  (setq yyyymm (or yyyymm (format-time-string "%Y%m")))
  (setq day    (or day    (format-time-string "%d")))
  (setq n      (or n      99999))
  (list (ee-template0 "\
# (find-es \"maxima\" \"mailing-lists\")
# (setq last-kbd-macro (kbd \"C-x 1 M-3 M-e M-> M-e M-o\"))
# (find-maximamsg \"{n} {yyyymm} {day}\" \"\")
# https://sourceforge.net/p/maxima/mailman/message/{n}/
# https://sourceforge.net/p/maxima/mailman/maxima-discuss/?viewmonth={yyyymm}
# https://sourceforge.net/p/maxima/mailman/maxima-discuss/?viewmonth={yyyymm}&viewday={day}
# https://sourceforge.net/p/maxima/mailman/maxima-discuss/?viewmonth={yyyymm}&viewday={day}&style=threaded
# https://sourceforge.net/p/maxima/mailman/maxima-discuss/?viewmonth={yyyymm}&style=threaded&limit=250
# (find-esgrep \"grep --color=auto -nH --null -e find-maximamsg maxima.e\")
# (find-esgrep \"grep --color=auto -nH --null -e maxima-discuss maxima.e\")
# (find-esgrep \"grep --color=auto -nH --null -e maxima/mailman maxima.e\")
# (find-lynx \"https://sourceforge.net/p/maxima/mailman/maxima-discuss/?viewmonth={yyyymm}&viewday={day}\" \"Showing\")
# (find-lynx \"https://sourceforge.net/p/maxima/mailman/maxima-discuss/?viewmonth={yyyymm}&viewday={day}&style=threaded\" \"Showing\")
# (find-lynx \"https://sourceforge.net/p/maxima/mailman/maxima-discuss/?viewmonth={yyyymm}&style=threaded&limit=250\" \"Showing\")

# (find-esgrep \"grep --color=auto -nH --null -e '{yyyymm} {day}' maxima.e\")
# (find-esgrep \"grep --color=auto -nH --null -e '{n}' maxima.e\")

# (find-lynx \"https://sourceforge.net/p/maxima/mailman/message/{n}/\" 80)
")))

;; «find-maximamsg»  (to ".find-maximamsg")
;; Tests: (find-maximamsg-links)
;;        (find-maximamsg "37675653 202207  1" "Stavros: structural equality")
;;
(defun find-maximamsg (&optional str &rest rest)
  (interactive (let ((n (ee-stuff-around-point "[0-9]")))
		 (if (not (equal n ""))
		     (list n))))
  (cl-destructuring-bind
      (&optional n yyyymm day &rest rest) 
      (ee-split (or str "999999"))
    (find-maximamsg-links n yyyymm day)))



;; «find-qdraw-links»  (to ".find-qdraw-links")
;; Skel: (find-find-links-links-new "qdraw" "fs xr yr" "")
;; Test: (find-qdraw-links)
;;
(defun find-qdraw-links (&optional fs xr yr &rest pos-spec-list)
"Visit a temporary buffer containing hyperlinks for qdraw."
  (interactive)
  (setq fs (or fs "{fs}"))
  (setq xr (or xr "{xr}"))
  (setq yr (or yr "{yr}"))
  (apply
   'find-elinks
   `((find-qdraw-links ,fs ,xr ,yr ,@pos-spec-list)
     (find-qdraw-links "x,x^2,x^3,x^4" "-2,2" "-2,2")
     ;; Convention: the first sexp always regenerates the buffer.
     (find-efunction 'find-qdraw-links)
     ""
     (find-windows-beginner-intro "12. Install qdraw")
     (find-es "maxima" "flatten")
     (find-es "qdraw" "ex1")
     ;; (defalias 'qdraw 'find-qdraw-links)
     ""
     ,(ee-template0 "\
 This needs qdraw and myqdraw
 (eepitch-maxima)
 (eepitch-kill)
 (eepitch-maxima)
load_myqdraw();
[xmin,xmax, ymin,ymax] : [{xr}, {yr}];
colors : [red, orange, forest_green, blue, dark_violet]$
fs : [{fs}];

myexj(j) := myex1(fs[j], lc(colors[j]));
myexj(j) := myex1(fs[j], lc(colors[j]), lk(fs[j]));
myexs()  := makelist(myexj(j), j,1,length(fs))$
myqdraw(xyrange0(), myexs());
myqdraw(myexs());



 This only needs qdraw
 (eepitch-maxima)
 (eepitch-kill)
 (eepitch-maxima)
load_qdraw();
fs : [{fs}];
/* \"fs\" is a list of functions of x. For an ugly plot, do:
   qdraw(xr({xr}),yr({yr}), ex(fs,x,{xr}));
*/

colors : [red, orange, forest_green, blue, dark_violet]$
myqdraw([dra]) := apply('qdraw, flatten([dra]));
myex1(j)       := ex1(fs[j], x,{xr}, lc(colors[j]))$
myex1(j)       := ex1(fs[j], x,{xr}, lc(colors[j]), lk(fs[j]))$
myexs()        := makelist(myex1(j), j,1,length(fs))$
myexs();

myqdraw([xr({xr}),yr({yr})], myexs());


")
     )
   pos-spec-list))



;; «find-quicklisp-links»  (to ".find-quicklisp-links")
;; Skel: (find-find-links-links-new "quicklisp" "pkg" "")
;; Test: (find-quicklisp-links)
;;       (find-quicklisp-links "cffi")
;;
(defun find-quicklisp-links (&optional pkg &rest pos-spec-list)
"Visit a temporary buffer containing hyperlinks for quicklisp."
  (interactive (list (ee-lisp-symbol-around-point-ask "Package name: ")))
  (setq pkg (or pkg "{pkg}"))
  (apply
   'find-elinks
   `((find-quicklisp-links ,pkg ,@pos-spec-list)
     ;; Convention: the first sexp always regenerates the buffer.
     (find-efunction 'find-quicklisp-links)
     ""
     (find-try-sly-intro "3. Adjust your ~/.emacs")
     (find-try-sly-intro "4. Download quicklisp.lisp")
     (find-fline "~/.sbclrc")
     ""
     ,(ee-template0 "\
# (find-qlfile \"\")
# (find-qlsh \"find * | sort\")
# (find-qlsh \"find * | sort | grep {pkg}\")
# (find-qlreleasesfile \"\" \"{pkg}.txt\")
# (find-qlreleasesfile \"{pkg}.txt\")
# (find-qlsoftwarefile \"\" \"{pkg}\")
# https://{pkg}.common-lisp.dev/

 (eepitch-sbcl)
 (eepitch-kill)
 (eepitch-sbcl)
(load #P\"~/quicklisp/setup.lisp\")

;; https://www.quicklisp.org/beta/#basic-commands
;; https://www.quicklisp.org/beta/#installation
(ql:system-apropos \"{pkg}\")
(ql:quickload \"{pkg}\")
(ql:install \"{pkg}\")
(ql:uninstall \"{pkg}\")


 More actions:
 (find-es \"lisp\" \"quicklisp\")
 (find-es \"lisp\" \"eev-quicklisp\")

")
     )
   pos-spec-list))



;; «find-sbcl-links»  (to ".find-sbcl-links")
;; Skel: (find-find-links-links-new "sbcl" "name" "")
;; Test: (find-sbcl-links 'require)
;;
(defun find-sbcl-links (&optional name &rest pos-spec-list)
"Visit a temporary buffer containing hyperlinks for sbcl."
  (interactive (list (ee-lisp-symbol-around-point-ask)))
  (setq name (or name "{name}"))
  (apply
   'find-elinks-elisp
   `((find-sbcl-links ',name ,@pos-spec-list)
     ;; Convention: the first sexp always regenerates the buffer.
     (find-efunction 'find-sbcl-links)
     ""
     ,(ee-template0 "\
;; (find-clhsdoci \"{name}\")
;; (find-sbclnode \"Function Index\" \" {name}:\")
;; (find-sbclgrep \"grep --color=auto -niRH --null -e '{name}' *\")
;; (find-sbclgrep \"grep --color=auto -niRH --null -e 'defun {name}' *\")
;; (find-sbclsh \"find * | sort\")
;; (find-sbclsh \"find * | sort | grep {name}\")
;; (find-quicklisp-links \"{name}\")
;; (find-try-sly-intro)

 (eepitch-sbcl)
 (eepitch-kill)
 (eepitch-sbcl)
(apropos \"{name}\")
(describe '{name})

 (eepitch-sly)
 (eepitch-kill)
 (eepitch-sly)
(apropos \"{name}\")
(describe '{name})
")
     )
   pos-spec-list))






;; «find-linki-links»  (to ".find-linki-links")
;; Used in: (find-eev "eev-kl-here.el" "ee-find-linkis")
;;    Skel: (find-find-links-links-new "linki" "stem" "")
;;    Test: (find-linki-links "intro")
;;
(defun find-linki-links (&optional stem &rest pos-spec-list)
"See: (find-eev \"eev-kl-here.el\" \"ee-find-linkis\")"
  (interactive)
  (setq stem (or stem "{stem}"))
  (apply
   'find-elinks-elisp
   `((find-linki-links ,stem ,@pos-spec-list)
     ;; Convention: the first sexp always regenerates the buffer.
     (find-efunction 'find-linki-links)
     ""
     ,(ee-template0 "\
;; (find-eev \"eev-htests.el\" \"tests\" \"ee-find-{stem}-links\")
;; (find-eev \"eev-htests.el\" \"tests\" \"ee-find-{stem}-linki\")
;; Tests:
;;   (find-tlhs '(?)     '(ee-find-{stem}-links))
;;   (find-tlh1 '(?)     '(ee-find-{stem}-linki))

;; (find-eev \"eev-hlinks.el\"  \"hprog\")
;; (find-eev \"eev-hlinks.el\"  \"hprog\" \"ee-find-{stem}-links\")
;; (find-eev \"eev-kl-here.el\" \"hprog\")
;;   (:if (ee-{stem}-bufferp)  (ee-find-{stem}-linki))

;; (find-efunction 'ee-find-{stem}-links)
;; (find-efunction 'ee-find-{stem}-linki)
;; (find-eev \"eev-kl-here.el\" \"ee-find-linkis\")

;; Skel: (find-linki-links \"{stem}\")
(defun ee-find-{stem}-linki ()
  )
")
     )
   pos-spec-list))




;; «find-gitdoc-links»  (to ".find-gitdoc-links")
;; Skel: (find-find-links-links-new "gitdoc" "stem" "")
;; Test: (find-gitdoc-links)
;;
(defun find-gitdoc-links (&optional stem &rest pos-spec-list)
"Visit a temporary buffer containing hyperlinks for gitdoc."
  (interactive)
  (setq stem (or stem "{stem}"))
  (apply
   'find-elinks
   `((find-gitdoc-links ,stem ,@pos-spec-list)
     (find-gitdoc-links "branch")
     ;; Convention: the first sexp always regenerates the buffer.
     (find-efunction 'find-gitdoc-links)
     ""
     (find-git-intro)
     (code-c-d "gitdoc" "/usr/share/doc/git-doc/")
     ""
     ,(ee-template0 "\
# (find-gitdoc-links               \"{stem}\")
# (find-es \"git\"               \"git-{stem}\")
# (find-man                    \"git-{stem}\")
# (find-man                  \"1 git-{stem}\")
# (find-gitdocfile             \"git-{stem}.txt\")
# (find-gitdocfile          \"\" \"git-{stem}.txt\")
# file:///usr/share/doc/git-doc/git-{stem}.html
# (find-sh                \"git help {stem}\")

# (find-vldifile \"git-doc.list\")
# (find-udfile   \"git-doc/\")
# (find-gitdocfile \"\")
# (find-gitdocfile \"\" \"gitcore-tutorial.txt\")
# (find-gitdocfile \"\" \"gittutorial.txt\")
# (find-gitdocfile \"\" \"revisions.txt\")
# (find-gitdocfile \"revisions.txt\")
")
     )
   pos-spec-list))



;; «find-luainit-links»  (to ".find-luainit-links")
;; Skel: (find-find-links-links-new "luainit" "dir" "dir0")
;; Test: (find-luainit-links)
;;
(defun find-luainit-links (&optional dir &rest pos-spec-list)
"Visit a temporary buffer containing a script for testing lua50init.lua."
  (interactive)
  (setq dir (or dir "{dir}"))
  (let* ((dir0 (ee-expand dir)))
    (apply
     'find-elinks
     `((find-luainit-links ,dir ,@pos-spec-list)
       (find-luainit-links "/tmp/")
       (find-luainit-links "~/LUA/")
       ;; Convention: the first sexp always regenerates the buffer.
       (find-efunction 'find-luainit-links)
       ""
       ,(ee-template0 "\
 Download/update
 (eepitch-shell)
 (eepitch-kill)
 (eepitch-shell)
rm  -fv {dir}lua50init.lua
wget -P {dir} -N http://anggtwu.net/LUA/lua50init.lua
ls -lAF {dir}lua50init.lua

 Test
 (eepitch-shell)
 (eepitch-kill)
 (eepitch-shell)
export LUA_INIT=
lua5.1
  PP({20,\"30\"})   -- should fail
  os.exit()

export LUA_INIT=@{dir0}lua50init.lua
lua5.1
  PP({20,\"30\"})   -- should work
  os.exit()

 (setenv \"LUA_INIT\" nil)
 (eepitch-lua51)
 (eepitch-kill)
 (eepitch-lua51)
  PP({20,\"30\"})   -- should fail

 (setenv \"LUA_INIT\" \"@{dir0}lua50init.lua\")
 (eepitch-lua51)
 (eepitch-kill)
 (eepitch-lua51)
  PP({20,\"30\"})   -- should work


 If you want to use an init file that loads my
 lua50init.lua and then does something else, use this:

 (setenv \"LUA_INIT\" \"@{dir0}lua50init_my.lua\")
 (ee-copy-rest 2 '(find-fline \"{dir0}lua50init_my.lua\"))

-- See: (find-luainit-links \"{dir0}\")
--      (find-fline \"{dir}lua50init.lua\")
dofile \"{dir0}lua50init.lua\"
-- Put your something elses here
")
       )
     pos-spec-list)))



;; «find-luaso-links»  (to ".find-luaso-links")
;; Skel: (find-find-links-links-new "luaso" "fname funname" "fnameshort fnameangg fnamebase dir edir")
;; Test: (find-luaso-links "/tmp/dummy2.c" "foo")
;;  See: (find-angg "LUA/CLua1.lua")
;;       (find-angg "CLUA/foo.c")
;;       (find-es "lua5" "CLua1.lua")
;;
(defun find-luaso-links (&optional fname funname &rest pos-spec-list)
"Visit a temporary buffer containing hyperlinks for luaso."
  (interactive)
  (setq fname (or fname "{fname}"))
  (setq funname (or funname "{funname}"))
  (let* ((fnameshort (ee-shorten-file-name fname))
         (fnameangg  (ee-replace-prefix "~/" "" fnameshort))
         (fnamebase  (file-name-base fname))
         (dir        (or (file-name-directory fname) ""))
         (edir       (ee-expand dir)))
    (apply
     'find-elinks
     `((find-luaso-links ,fname ,funname ,@pos-spec-list)
       ;; Convention: the first sexp always regenerates the buffer.
       (find-efunction 'find-luaso-links)
       ""
       ,(ee-template0 "\
// (c-mode)
// See: (find-lua-tutorial-intro \"3. The C API\")
// All lines with \"angg\" are angg-isms!
//   (kill-new \"  {fnameangg}\")
//   (find-blogme3 \"anggmake.lua\" \"anggtranslate\")
//   (find-blogme3 \"anggmake.lua\" \"anggtranslate\" \"LUA/\")
// (ee-copy-rest 2 '(find-fline \"{fname}\"))


// This file:
//   http://anggtwu.net/{fnameangg}.html
//   http://anggtwu.net/{fnameangg}
//          (find-angg \"{fnameangg}\")
//    Skel: (find-luaso-links \"{fname}\" \"{funname}\")
//  Author: Eduardo Ochs <eduardoochs@gmail.com>
//
// (defun e () (interactive) (find-angg \"{fnameangg}\"))

#include \"lauxlib.h\"
#include <stdio.h>

static int my_{funname}(lua_State* L) {<}
  lua_pushnumber(L, 33);
  lua_pushnumber(L, 333);
  return 2;
{>}

static const struct luaL_reg {fnamebase}_lib[] = {<}
  {<}\"{funname}\", my_{funname}{>},
  {<}NULL,  NULL{>}
{>};

LUALIB_API int luaopen_{fnamebase}(lua_State *L) {<}
  lua_pushvalue(L, LUA_GLOBALSINDEX);
  luaL_openlib(L, NULL, {fnamebase}_lib, 0);
  return 0;
{>}

/*
 (eepitch-bash)
 (eepitch-kill)
 (eepitch-bash)
# MacOS with MacPorts:
CFLAGS=\"-g -Wall -shared -undefined dynamic_lookup\"
LUADIR=/opt/local/include/lua5.1

# Debian:
CFLAGS=\"-g -Wall -shared\"
LUADIR=/usr/include/lua5.1

echo gcc $CFLAGS -I$LUADIR -o {fnamebase}.so {fnamebase}.c
     gcc $CFLAGS -I$LUADIR -o {fnamebase}.so {fnamebase}.c
ls -lAF {fnamebase}*

 (eepitch-lua51)
 (eepitch-kill)
 (eepitch-lua51)
package.loadlib(\"{edir}{fnamebase}.so\", \"luaopen_{fnamebase}\")()
print({funname}(42))

*/\
")
       )
     pos-spec-list)))




;; «find-subed-mpv-links»  (to ".find-subed-mpv-links")
;; Skel: (find-find-links-links-new "subed-mpv" "emacs-width" "ee-buffer-name")
;; Test: (find-subed-mpv-links)
;;
(defun find-subed-mpv-links (&optional emacs-width &rest pos-spec-list)
"Visit a temporary buffer containing hyperlinks for subed-mpv."
  (interactive)
  (setq emacs-width (or emacs-width 140))
  (let* ((ee-buffer-name "*find-subed-mpv-links*"))
    (apply
     'find-elinks
     `((find-subed-mpv-links ,emacs-width ,@pos-spec-list)
       (find-subed-mpv-links "{emacs-width}")
       (find-efunction 'find-subed-mpv-links)
       (find-efunction 'ee-insert-test-subed-vtt-mode)
       (find-eevfile "eev-testblocks.el" "ee-insert-test-subed-vtt-mode")
       "# http://anggtwu.net/2024-find-subed-mpv-links.html"
       ""
       ,(ee-template0 "\

1. Get the video
================
Make sure that you have a local copy of this video:
  (find-1stclassvideo-links \"eev2021\")
  (find-eev2021video)



2. Prepare /tmp/subed-test/
===========================
The default for `ee-git-dir' is:

  (setq ee-git-dir \"~/usrc/\")

Here we will use \"/tmp/subed-test/\".

# (setq ee-git-dir \"/tmp/subed-test/\")
# (find-git-links \"https://github.com/sachac/subed\" \"subed\")
# https://github.com/sachac/subed

 (eepitch-shell)
 (eepitch-kill)
 (eepitch-shell)
rm -Rfv /tmp/subed-test/
mkdir   /tmp/subed-test/
cd      /tmp/subed-test/
cp -v $S/http/anggtwu.net/eev-videos/emacsconf2021.mp4 a.mp4
cp -v $S/http/anggtwu.net/eev-videos/emacsconf2021.vtt a.vtt
cd      /tmp/subed-test/
git clone https://github.com/sachac/subed
cd      /tmp/subed-test/subed/

# (setq ee-git-dir \"~/usrc/\")



3. Load subed
=============
Run the red star lines below.
To load subed from another place, skip the
`add-to-list' in the first line.

 (add-to-list 'load-path \"/tmp/subed-test/subed/subed/\")
 (ee-locate-library \"subed.el\")

 (require 'subed)
 (require 'subed-mpv)
 (require 'subed-vtt)
 (code-c-d \"subed\"  \"/tmp/subed-test/subed/\")
 (code-c-d \"subeds\" \"/tmp/subed-test/subed/subed/\")
 (code-c-d \"subeds\" (ee-locate-library \"subed.el\"))

 (find-subedfile \"\")
 (find-subedsfile \"\")



4. A test block in a .vtt
=========================
If everything is right then opening the \"a.vtt\" with the
last sexp below will put it in subed-vtt-mode. Many of the
sexps below are optional.

 (setq subed-auto-find-video t)
 (setq subed-auto-find-video nil)
 (setq subed-mpv-arguments '(\"--osd-level=2\" \"--geometry=320x200-0+0\"))
 (set-frame-parameter (selected-frame) 'fullscreen 'fullheight)
 (set-frame-position (selected-frame) 0 0)
 (set-frame-parameter (selected-frame) 'width {emacs-width})
 (find-fline \"/tmp/subed-test/\")
 (find-fline \"/tmp/subed-test/a.vtt\")

Typing `M-x eeit' between two subtitles will insert a big test block
in the .vtt. To run that test block, just type <f8>s on its red star
lines.
")
     )
   pos-spec-list)))



;;;      _      _                 _       _                   
;;;   __| | ___| |__   ___   ___ | |_ ___| |_ _ __ __ _ _ __  
;;;  / _` |/ _ \ '_ \ / _ \ / _ \| __/ __| __| '__/ _` | '_ \ 
;;; | (_| |  __/ |_) | (_) | (_) | |_\__ \ |_| | | (_| | |_) |
;;;  \__,_|\___|_.__/ \___/ \___/ \__|___/\__|_|  \__,_| .__/ 
;;;                                                    |_|    
;; See: (find-debootstrap-intro)

;; «find-debootstrap0-links»  (to ".find-debootstrap0-links")
;; Skel: (find-find-links-links-new "debootstrap0" "container n" "ee-buffer-name")
;; Test: (find-debootstrap0-links)
;;
(defun find-debootstrap0-links (&optional container n &rest pos-spec-list)
"See: (find-debootstrap-intro)"
  (interactive)
  (setq container (or container "subdebian"))
  (setq n         (or n         "0"))
  (let* ((ee-buffer-name "*find-debootstrap0-links*"))
    (apply
     'find-elinks
     `((find-debootstrap0-links ,container ,n ,@pos-spec-list)
       (find-debootstrap0-links "{container}" "{n}")
       (find-debootstrap0-links)
       ;; Convention: the first sexp always regenerates the buffer.
       (find-efunction 'find-debootstrap0-links)
       ""
       ,(ee-debootstrap0 container n)
       ,(ee-template0 "\
")
       )
     pos-spec-list)))

(defvar ee-debootstrap0-pkgs-a
  "systemd,dbus,sudo,openssh-server,xauth,xterm")

(defvar ee-debootstrap0-pkgs-b
  "psmisc")


;; «ee-debootstrap0»  (to ".ee-debootstrap0")
;; Tests: (find-estring-2a (ee-debootstrap0 "subdebian" "0"))
;;        (find-estring-2a (ee-debootstrap0 "{container}" "{n}"))
;;
(defun ee-debootstrap0 (&optional container n)
  "An internal function used by `find-debootstrap0-links'."
  (setq container (or container "subdebian"))
  (setq n         (or n         "0"))
  (ee-template0 "\
 From: (find-estring-2a (ee-debootstrap0 \"{container}\" \"{n}\"))
 See: https://wiki.debian.org/nspawn#Host_Preparation

 (sh-mode)
 (eepitch-shell)
 (eepitch-kill)
 (eepitch-shell)
sudo bash
  echo 'kernel.unprivileged_userns_clone=1' > /etc/sysctl.d/nspawn.conf
  systemctl restart systemd-sysctl.service
exit

 Create a /var/lib/machines/{container}/ with debootstrap.
 This takes several minutes.

 (eepitch-shell3)
 (eepitch-kill)
 (eepitch-shell3)
sudo ls -lAFh /var/lib/machines/
sudo rm -Rf   /var/lib/machines/{container}/
sudo debootstrap \\
  --include={ee-debootstrap0-pkgs-a},{ee-debootstrap0-pkgs-b} \\
  stable /var/lib/machines/{container}

 Save /var/lib/machines/{container}/ into {container}{n}.tar.
 This is very quick.

sudo ls -lAFh /var/lib/machines/
sudo du -ch   /var/lib/machines/
sudo machinectl export-tar {container} /var/lib/machines/{container}{n}.tar
sudo ls -lAFh /var/lib/machines/

"))


;; «ee-debootstrap1»  (to ".ee-debootstrap1")
;; Tests: (find-estring-2a (ee-debootstrap1 "{container}" "{user}" "{passwd}"))
;;        (find-estring-2a (ee-debootstrap1 "subdebian" "edrx" "edrx"))
;;
(defun ee-debootstrap1 (&optional container user passwd)
  "An internal function used by `find-debootstrap1-links'."
  (setq container (or container "subdebian"))
  (setq user      (or user   user-login-name))
  (setq passwd    (or passwd user))
  (let* ((key (find-sh0 "cat ~/.ssh/id_rsa.pub")))
    (ee-template0 "\
 From: (find-estring-2a (ee-debootstrap1 \"{container}\" \"{user}\" \"{passwd}\"))
 (sh-mode)

echo \"{passwd}/{passwd}/Full Name/////Y/\" | tr / '\\n' | adduser {user}

ls -lAF /etc/sudoers
echo '
  # Edrx:
  # (find-es \"sudo\")
  User_Alias FULLTIMERS = {user}
  FULLTIMERS ALL = NOPASSWD: ALL
' | tee -a /etc/sudoers
ls -lAF    /etc/sudoers

cat /etc/hosts
echo '127.0.1.1   {container}' | tee -a /etc/hosts
cat /etc/hosts

mkdir -p            /home/{user}/.ssh/
mkdir -p                 /root/.ssh/
echo '{key}' > /home/edrx/.ssh/authorized_keys
echo '{key}' >      /root/.ssh/authorized_keys
chmod 600           /home/{user}/.ssh/authorized_keys
chmod 700           /home/{user}/.ssh/
chmod 600                /root/.ssh/authorized_keys
chmod 700                /root/.ssh/
chown -Rv {user}:{user} /home/{user}/.ssh/

touch                    /root/.Xauthority
touch               /home/{user}/.Xauthority
chown     {user}:{user} /home/{user}/.Xauthority
")))


;; «find-debootstrap1-links»  (to ".find-debootstrap1-links")
;; Skel: (find-find-links-links-new "debootstrap1" "container user passwd" "ee-buffer-name key")
;; Test: (find-debootstrap1-links)
;;
(defun find-debootstrap1-links (&optional container user passwd &rest pos-spec-list)
"See: (find-debootstrap-intro)"
  (interactive)
  (setq container (or container "subdebian"))
  (setq user      (or user      user-login-name))
  (setq passwd    (or passwd    user))
  (let* ((ee-buffer-name "*find-debootstrap1-links*")
	 (key (find-sh0 "cat ~/.ssh/id_rsa.pub")))
    (apply
     'find-elinks
     `((find-debootstrap1-links ,container ,user ,passwd ,@pos-spec-list)
       (find-debootstrap1-links "{container}" "{user}" "{passwd}")
       (find-debootstrap1-links)
       ;; Convention: the first sexp always regenerates the buffer.
       (find-efunction 'find-debootstrap1-links)
       (find-efunction 'ee-debootstrap1)
       ""
       ,(ee-template0 "\
 (setq truncate-lines t)
 (sh-mode)
 (eepitch-shell)
 (eepitch-kill)
 (eepitch-shell)

 Restore /var/lib/machines/{container}/ from {container}0.tar.
 This is very quick.
sudo rm -Rf                /var/lib/machines/{container}/
sudo ls -lAFh              /var/lib/machines/
sudo machinectl import-tar /var/lib/machines/{container}0.tar {container}
sudo ls -lAFh              /var/lib/machines/
sudo du -ch                /var/lib/machines/{container}/




sudo systemd-nspawn -U --machine {container}

{(ee-debootstrap1 container user passwd)}\

exit





 Save /var/lib/machines/{container}/ into {container}1.tar.
 This is very quick.
sudo ls -lAFh /var/lib/machines/
sudo du -ch   /var/lib/machines/
sudo machinectl export-tar {container} /var/lib/machines/{container}1.tar
sudo ls -lAFh /var/lib/machines/
")
       )
     pos-spec-list)))


;; «find-debootstrap2-links»  (to ".find-debootstrap2-links")
;; Skel: (find-find-links-links-new "debootstrap2" "container user" "ee-buffer-name")
;; Test: (find-debootstrap2-links)
;;
(defun find-debootstrap2-links (&optional container user &rest pos-spec-list)
"See: (find-debootstrap-intro)"
  (interactive)
  (setq container (or container "subdebian"))
  (setq user      (or user user-login-name))
  (let* ((ee-buffer-name "*find-debootstrap2-links*"))
    (apply
     'find-elinks
     `((find-debootstrap2-links ,container ,user ,@pos-spec-list)
       ;; Convention: the first sexp always regenerates the buffer.
       (find-efunction 'find-debootstrap2-links)
       ""
       ,(ee-template0 "\
 Export the ssh keys of the container to the host.
 The tests in the \"Try:\" are optional.

 (find-3EE '(eepitch-shell) '(eepitch-shell2))
 (find-3ee '(eepitch-shell) '(eepitch-shell2))
 (eepitch-shell)
 (eepitch-shell2)
sudo systemd-nspawn -U --machine {container}
  mkdir -p /run/sshd     ; ps ax | grep ssh
  /usr/sbin/sshd -p 4444 ; ps ax | grep ssh

 (eepitch-shell)
ssh-keygen -f ~/.ssh/known_hosts -R \"[localhost]:4444\"
ssh -p 4444 -o StrictHostKeyChecking=accept-new root@localhost echo Ok
ssh -p 4444 -o StrictHostKeyChecking=accept-new {user}@localhost echo Ok

 Try:
 (find-fline \"/scp:root@localhost#4444:\")
 (find-fline \"/scp:{user}@localhost#4444:\")
 (find-bgprocess \"ssh -p 4444 -X -f {user}@localhost xterm\")

 (eepitch-shell2)
  killall sshd;   sleep 1; ps ax | grep ssh
exit
")
       )
     pos-spec-list)))



;;;                 _           _        _ _ 
;;; __   _____     (_)_ __  ___| |_ __ _| | |
;;; \ \ / / __|____| | '_ \/ __| __/ _` | | |
;;;  \ V / (_|_____| | | | \__ \ || (_| | | |
;;;   \_/ \___|    |_|_| |_|___/\__\__,_|_|_|
;;;                                          
;; «find-package-vc-install-links»  (to ".find-package-vc-install-links")
;; Skel: (find-find-links-links-new "package-vc-install" "giturl" "name dir0 dir")
;; Test: (find-package-vc-install-links)
;;  See: (find-lean4-intro "5. Install lean4-mode")
;;
(defun find-package-vc-install-links (&optional giturl &rest pos-spec-list)
"Show a script that does part of the work of `package-vc-install'.
GITURL must be a git url - other version control systems are not
supported."
  (interactive)
  (setq giturl (or giturl "{giturl}"))
  (let* ((name (file-name-nondirectory giturl))
         (dir0 (ee-expand "$HOME/.emacs.d/elpa/"))
         (dir (concat dir0 name)))
    (apply
     'find-elinks-elisp
     `((find-package-vc-install-links ,giturl ,@pos-spec-list)
       (find-package-vc-install-links "https://github.com/leanprover-community/lean4-mode")
       (find-package-vc-install-links "https://github.com/edrx/eev")
       ;; Convention: the first sexp always regenerates the buffer.
       (find-efunction 'find-package-vc-install-links)
       ""
       ,(ee-template0 "\
;; Check the version of your Emacs:
;;   (emacs-version)
;;
;; If you're on emacs 29.1 or newer then the sexp with
;; `package-vc-install' below should work. See:
;;   (find-enode \"Fetching Package Sources\")
;;   (find-efunctiondescr 'package-vc-install)
;;
(package-vc-install \"{giturl}\")


;; Otherwise try the script below.
;; It will try to download the package from:
;;   {giturl}
;; and install it into:
;;   {dir}
;;   (find-fline \"{dir0}\")
;;   (find-fline \"{dir0}\" \"{name}\")
;;   (find-fline \"{dir}\")
;; ...BUT IT WILL NOT INSTALL ITS DEPENDENCIES!

 (eepitch-shell)
 (eepitch-kill)
 (eepitch-shell)
rm -Rfv  {dir}
mkdir -p {dir}
cd       {dir}
git clone {giturl} .

 (add-to-list 'load-path \"{dir}\")
 (find-eppp load-path)
")
       )
     pos-spec-list)))



;;;  _   _                              
;;; | |_| |__   ___ _ __ ___   ___  ___ 
;;; | __| '_ \ / _ \ '_ ` _ \ / _ \/ __|
;;; | |_| | | |  __/ | | | | |  __/\__ \
;;;  \__|_| |_|\___|_| |_| |_|\___||___/
;;;                                     
;; «find-ethemes-links»  (to ".find-ethemes-links")
;; Skel: (find-find-links-links-new "ethemes" "" "")
;; Test: (find-ethemes-links)
;;
(defun find-ethemes-links (&rest pos-spec-list)
"Visit a temporary buffer with an alternative interface to `customize-themes'."
  (interactive)
  (apply
   'find-elinks-elisp
   `((find-ethemes-links ,@pos-spec-list)
     ;; Convention: the first sexp always regenerates the buffer.
     (find-efunction 'find-ethemes-links)
     ""
     ,(ee-template0 "\
;; (eek \"M-h M-k <menu-bar> <options> <customize> <customize-themes>\")
;; (eek         \"<menu-bar> <options> <customize> <customize-themes>\")
;; (find-efunction-links 'customize-themes)
;; (find-efunction       'customize-themes)
;; (customize-themes)
;;
;; (find-2a nil '(find-efaces))
;;
{(ee-load-themes)}\

;; (find-eppp       custom-theme-load-path)
;; (find-eppp      (custom-theme--load-path))
;; (find-efunction 'custom-theme--load-path)
;;
{(ee-custom-theme-load-path)}\
")
     )
   pos-spec-list))

;; Test: (find-estring (ee-load-themes))
;;       (find-estring (ee-custom-theme-load-path))
;;
(defun ee-load-themes ()
  "An internal function used by `find-ethemes-links'."
  (cl-loop for o in (ee-sort-symbols (custom-available-themes))
  	   concat (ee-pp0 `(load-theme ',o 'no-confirm))
           concat "\n"))

(defun ee-custom-theme-load-path ()
  "An internal function used by `find-ethemes-links'."
  (cl-loop for p in (custom-theme--load-path)
	   concat (ee-pp0 `(find-fline ,p))
	   concat "\n"))



;;;   __ _           _       _              _ _        _ _       _        
;;;  / _(_)_ __   __| |     | |_ _ __ _   _(_) |_     | (_)_ __ | | _____ 
;;; | |_| | '_ \ / _` |_____| __| '__| | | | | __|____| | | '_ \| |/ / __|
;;; |  _| | | | | (_| |_____| |_| |  | |_| | | ||_____| | | | | |   <\__ \
;;; |_| |_|_| |_|\__,_|      \__|_|   \__, |_|\__|    |_|_|_| |_|_|\_\___/
;;;                                   |___/                               
;;
;; «find-tryit-links»  (to ".find-tryit-links")
;; See: http://anggtwu.net/2024-find-tryit-links.html
;;      (find-lean4-intro "6. Install lean4-mode")
;;
(defvar ee-tryit-compact nil
  "When this is non-nil `ee-tryit-progn' produces a single-line sexp.")
(defvar ee-tryit-config  ""
  "See `ee-tryit-body'.")

;; (find-tryit-links)
;; (find-tryit-links "e 1e")
;; (find-tryit-links "e 1e" '((foo) (bar)))
;; (find-tryit-links "ikeb")
;; (find-tryit-links "ikeb" '((find-eev-quick-intro "2. Evaluating Lisp")))
;;
(defun find-tryit-links (&optional configs sexps &rest pos-spec-list)
"Visit a temporary buffer containing hyperlinks for tryit."
  (interactive (list nil (ee-tryit-read-sexps)))
  (setq configs (or configs "ire 1ire"))
  (let* ((body (ee-tryit-progns configs sexps)))
    (apply
     'find-elinks-elisp
     `((find-tryit-links ,configs ',sexps)
       (find-tryit-links "irue 1irue" ',sexps)
       (find-tryit-links "ikeb" ',sexps)
       (find-tryit-links "<SE>b" ',sexps)
       (find-tryit-links "<SG>b" ',sexps)
       (find-tryit-links "m<LI><2M><LSM><LP>" ',sexps)
       (find-efunction 'ee-tryit-body)
       (find-efunction 'find-tryit-links)
       ""
       ,(ee-template0 "\
;; See: http://anggtwu.net/2024-find-tryit-links.html
;;      (find-1stclassvideo-links \"2024trywithasexp\")
;;      https://bpa.st/

{body}
")
       )
     pos-spec-list)))

(defun ee-tryit-read-sexps ()
  (when current-prefix-arg
    (ee-goto-eol)
    (list (read (ee-last-sexp)))))

;; (ee-tryit-tostring 1)
;; (ee-tryit-tostring 's)
;; (ee-tryit-tostring '(eev-beginner))
;; (ee-tryit-tostring ";; See: link")
(defun ee-tryit-tostring (o)
  (cond ((stringp o) o)
	((consp   o) (ee-S o))
	(t           (format "%s" o))))

;; (let ((ee-tryit-config "abC")) (ee-tryit-has 'b))
;; (let ((ee-tryit-config "abC")) (ee-tryit-has 'B))
;; (let ((ee-tryit-config "abC")) (ee-tryit-has 'c))
;; (let ((ee-tryit-config "abC")) (ee-tryit-has 'C))
(defun ee-tryit-has (c)
  (let ((case-fold-search nil))
    (string-match (ee-tryit-tostring c)
		  ee-tryit-config)))

;; (let ((ee-tryit-compact t  )) (ee-tryit-line '(eev-beginner)))
;; (let ((ee-tryit-compact nil)) (ee-tryit-line '(eev-beginner)))
(defun ee-tryit-line (sexp)
  (if sexp
      (format (if ee-tryit-compact " %s" "\n  %s")
	      (ee-tryit-tostring sexp))
    ""))

;; (let ((ee-tryit-config "b")) (ee-tryit-if 'b '(eev-beginner)))
;; (let ((ee-tryit-config "x")) (ee-tryit-if 'b '(eev-beginner)))
(defun ee-tryit-if (c &rest sexps)
  (declare (indent 1))
  (if (ee-tryit-has c)
      (mapconcat 'ee-tryit-line sexps "")
    ""))
  
;; (let ((ee-tryit-config "im"))   (ee-tryit-body))
;; (let ((ee-tryit-config "ireb")) (ee-tryit-body))
;; (let ((ee-tryit-config "ikeb")) (ee-tryit-body))
(defun ee-tryit-body ()
  (concat
   (ee-tryit-if "i" '(package-initialize))
   (ee-tryit-if "m" '(add-to-list
		      'package-archives
		      '("melpa" . "https://melpa.org/packages/")))
   ;;
   (ee-tryit-if "k"
     ";;"
     ";; See: http://anggtwu.net/2024-no-public-key.html"
     '(setq package-check-signature nil)
     '(package-refresh-contents)
     '(package-install 'gnu-elpa-keyring-update)
     '(setq package-check-signature 'allow-unsigned)
     ";;")
   ;;
   (ee-tryit-if "r" '(package-refresh-contents)) ; also in "k"
   (ee-tryit-if "e" '(package-install 'eev))
   (ee-tryit-if "<SE>" '(straight-use-package 'eev))
   (ee-tryit-if "<SG>" '(straight-use-package
			 '(eev :type git :host github :repo "edrx/eev")))
   (ee-tryit-if "b" '(eev-beginner))
   ;;
   ;; For Lean.
   ;; See: (find-lean4-intro "6. Install lean4-mode")
   (ee-tryit-if "<LI>" ";; See: (find-lean4-intro \"6. Install lean4-mode\")")
   (ee-tryit-if "<2M>" '(find-2a nil '(find-ebuffer "*Messages*")))
   (ee-tryit-if "<LSM>"
     ";; https://emacs.stackexchange.com/questions/80871/how-to-provide-updated-seq-package-to-magit"
     '(setq package-install-upgrade-built-in t)
     '(package-install 'compat)
     '(package-install 'seq)
     '(progn (unload-feature 'seq t) (require 'seq))
     '(package-install 'magit))
   (ee-tryit-if "<LP>" 
     '(package-install 'company)
     '(package-install 'dash)
     '(package-install 'flycheck)
     '(package-install 'lsp-mode)
     '(package-install 'magit-section))
   ))

;; (ee-tryit-progn "1ire")
;; (ee-tryit-progn  "ire")
;; (ee-tryit-progn  "ire" '((foo) (bar)))
(defun ee-tryit-progn (config &optional sexps)
  (let* ((ee-tryit-config config)
	 (ee-tryit-compact (ee-tryit-has "1"))
	 (body (ee-tryit-body))
	 (lastsexps (mapconcat 'ee-tryit-line sexps "")))
    (format "(progn%s%s)" body lastsexps)))

;; (ee-tryit-progns "ike 1ire")
;; (ee-tryit-progns "ike 1ire" nil)
;; (ee-tryit-progns "ike 1ire" '((foo) (bar)))
(defun ee-tryit-progns (configs &optional sexps)
  (cl-loop for config in (ee-split configs)
	   concat (ee-tryit-progn config sexps)
	   concat "\n\n"))



(provide 'eev-tlinks)



;; Local Variables:
;; coding:            utf-8-unix
;; no-byte-compile:   t
;; End:
