/*****************************************************************************
 *
 * Original Author:  Fredrik Hbinette <hubbe@hubbe.net>
 *
 * Current Authors: Louis Bavoil <bavoil@cs.utah.edu>
 *                  Peter Leese <hubbe@hubbe.net>
 *
 * This code is based on and is a branch of plugger written 
 * by Fredrik Hbinette <hubbe@hubbe.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.
 *
 *****************************************************************************/

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <stdarg.h>
#include <sys/wait.h>

#include "mozplugger.h"

/*****************************************************************************
 * Debug function
 *****************************************************************************/
#ifdef DEBUG

static FILE *debug_output;
static char * debug_path = "";

/*******************************************************************************/
/**
 * @brief Get debug output file
 *
 * Open the debug file for appending to
 *
 * @return The FILE pointer
 *
 ******************************************************************************/
static FILE *getout(void)
{
     char debug_filename[128];
     char *tmpdir;

     if (debug_output)
          return debug_output;

     if (!(tmpdir = getenv("TMPDIR")))
     {
          /* If (as on default Fedora 7 install) the environment variable
           * TMPDIR is not defined, just assume $HOME/tmp */
          char * homedir = getenv("HOME");
          if(!homedir)
          {
               fprintf(stderr, "Failed to open debug file - "
                               "neither HOME or TMPDIR defined\n");
               return NULL;
          }
          debug_path = "$HOME/tmp";        /* For display purposes */
          snprintf(debug_filename, sizeof(debug_filename),
               "%s/tmp/%s", homedir, DEBUG_FILENAME);
     }
     else
     {
          debug_path = "$TMDIR";          /* For display purposes */
          snprintf(debug_filename, sizeof(debug_filename),
                 "%s/%s", tmpdir, DEBUG_FILENAME);
     }

     if (!(debug_output = fopen(debug_filename,"a+")))
     {
          fprintf(stderr,"Failed to open debug file at \'%s'\n", debug_filename);
          return NULL;
     }
     else
     {
          fprintf(stderr,"Debug file opened at \'%s\'\n", debug_filename);
     }

     fprintf(debug_output, "------------\n");
     return debug_output;
}

/*******************************************************************************/
/**
 * @brief Close debug  file
 *
 * Close the debug file
 *
 ******************************************************************************/
void close_debug(void)
{
     FILE *f;
     if ((f = getout()))
          fclose(f);
     debug_output = NULL;
}

/*******************************************************************************/
/**
 * @brief Print debug message
 *
 * Take the standard printf type arguments and write the output to the debug
 * file
 *
 * @param[in] fmt The printf format args
 * @param[in] ... The printf style arguments
 *
 ******************************************************************************/
void D(char *fmt, ...)
{
     FILE *f;
     char buffer[9999];
     va_list ap;

     va_start(ap,fmt);
     vsnprintf(buffer,sizeof(buffer),fmt,ap);
     va_end(ap);

     if ((f = getout()))
     {
          fprintf(f,"PID%4d: %s",(int) getpid(), buffer);
          fflush(f);
     }
}

/*******************************************************************************/
/**
 * @brief Get debug path
 *
 * Get path to the debug file (for display purposes only!)
 *
 * @return String containing debug path name
 *
 ******************************************************************************/
char * get_debug_path(void)
{
    if(debug_path[0] == '\0')
        getout();

    return debug_path;
}

#else

void D(char *fmt, ...) { }

#endif

/*****************************************************************************/
/**
 * @brief Wrapper for kill
 *
 * Adaptive kill(), keep calling kill with higher and higher signal level,
 *
 * NOTE: kill return zero on successfully sending the signal so the code
 * only exits the nested ifs when kill no longer can send the signal!
 *
 * @param[in] pid The process ID of process to kill
 *
 *****************************************************************************/
void my_kill(pid_t pid)
{
     int status;

     D("Killing PID %d with SIGTERM\n", pid);
     if (!kill(pid, SIGTERM))             /* '!kill' == true if signal sent! */
     {
          usleep(KILL_TIMEOUT_USEC);
          D("Killing PID %d with SIGTERM\n", pid);
          if (!kill(pid, SIGTERM))
          {
               usleep(KILL_TIMEOUT_USEC);
               D("Killing PID %d with SIGTERM\n", pid);
               if (!kill(pid, SIGTERM))
               {
                    D("Killing PID %d with SIGKILL\n", pid);
                    kill(pid, SIGKILL);
               }
          }
     }

     D("Waiting for sons\n");
     while (waitpid(-1, &status, WNOHANG) > 0);
}
