#include "HRCov.h"
#include "nsStringAPI.h"
#include "common.h"
#include "jsapi.h"
#include "jsdbgapi.h"
#include "nsIJSRuntimeService.h"
#include "nsCOMPtr.h"
#include "nsServiceManagerUtils.h"
#include "nsInterfaceHashtable.h"
#include "nsIJSContextStack.h"
#include "nsIPrefService.h"
#include "HRBitset.h"
#include "jsscript.h"
#include "jsarena.h"
#include "jsemit.h"
#include "jsfun.h"
#include "jsstr.h"
#include "nsIObserverService.h"
#include "nsCycleCollector.h"
#include "nsThreadUtils.h"

#define nsAString_h___ // work around bad header
#include "jsdIDebuggerService.h"

#define QTMSTR_DEBUG 0

#define XPC_RUNTIME_CONTRACTID "@mozilla.org/js/xpc/RuntimeService;1"

class HRCovResult : public IHRCovResult, nsIClassInfo
{
public:
  NS_DECL_ISUPPORTS
  NS_DECL_NSICLASSINFO
  NS_DECL_IHRCOVRESULT

  HRCovResult(const char* uri);

  inline void MarkExecutedLine(PRUint32 line) {
    mExecutedLines->SetBit(line);
  }

  inline void MarkPotentialLine(PRUint32 line) {
    mPotentialLines->SetBit(line);
  }

private:
  ~HRCovResult();

  nsCString mUri;
  nsCOMPtr<IHRBitset> mExecutedLines;
  nsCOMPtr<IHRBitset> mPotentialLines;
};

NS_IMPL_ISUPPORTS2(HRCovResult, IHRCovResult, nsIClassInfo)
NS_IMPL_CI_INTERFACE_GETTER1(HRCovResult, IHRCovResult)
IMPL_CI(HRCovResult, "HRCovResult", 0)

HRCovResult::HRCovResult(const char* uri)
: mUri(uri), mExecutedLines(new HRBitset()),
mPotentialLines(new HRBitset())
{}

HRCovResult::~HRCovResult() {}

/* readonly attribute string uri; */
NS_IMETHODIMP HRCovResult::GetUri(char * *aUri)
{
  *aUri = nsstrdup(mUri.get());
  return NS_OK;
}

/* readonly attribute IHRBitset executedLines; */
NS_IMETHODIMP HRCovResult::GetExecutedLines(IHRBitset * *aExecutedLines)
{
  *aExecutedLines = mExecutedLines;
  NS_ADDREF(*aExecutedLines);
  return NS_OK;
}

/* readonly attribute IHRBitset executedLines; */
NS_IMETHODIMP HRCovResult::GetPotentialLines(IHRBitset * *aPotentialLines)
{
  *aPotentialLines = mPotentialLines;
  NS_ADDREF(*aPotentialLines);
  return NS_OK;
}


class HRCov : public IHRCov
{
public:
  NS_DECL_ISUPPORTS
  NS_DECL_IHRCOV

  HRCov();

private:
  nsInterfaceHashtable<nsCharPtrHashKey, IHRCovResult> mResults;
  ptr_set mTracedObjects;
  PRBool mWasOn;

  ~HRCov();

  inline HRCovResult* GetResult(const char* filename);

  JSTrapStatus OnStep(JSContext* cx, JSScript* script,
                      jsbytecode* pc, jsval* rval);

  void OnNewScript(JSContext* cx, const char* filename,
                   uintN lineno, JSScript* script,
                   JSFunction* fun);

  void OnTrace(JSTracer* trc, void* thing, uint32 kind);

  static JS_DLL_CALLBACK JSTrapStatus
  OnStepHook(JSContext *cx, JSScript *script,
             jsbytecode *pc, jsval *rval, void *closure);

  static JS_DLL_CALLBACK void
  OnNewScriptHook(JSContext  *cx,
                  const char *filename,  /* URL of script */
                  uintN      lineno,     /* first line */
                  JSScript   *script,
                  JSFunction *fun,
                  void       *callerdata);

  static JS_DLL_CALLBACK void
  TraceNotify(JSTracer* trc, void* thing, uint32 kind);
};

NS_IMPL_ISUPPORTS1_CI(HRCov, IHRCov)

NS_IMETHODIMP HRCovSingletonConstructor(nsISupports *aOuter,
                                        const nsIID& aIID,
                                        void **aResult)
{
  if(unlikely(aOuter != NULL)) {
    return ((nsresult) 0x80040110L);;
  }

  static HRCov* inst = NULL;
  if(!inst) {
    inst = new HRCov();
    NS_ADDREF(inst);
  }

  return inst->QueryInterface(aIID, aResult);
}

HRCov::HRCov() : mWasOn(false)
{
  mResults.Init(1);
  mTracedObjects.Init(1);
}

HRCov::~HRCov()
{
  /* destructor code */
}

inline HRCovResult* HRCov::GetResult(const char* filename)
{
  HRCovResult* result =
    static_cast<HRCovResult*>(mResults.GetWeak(filename, NULL));

  if(!result) {
    result = new HRCovResult(filename);
    NS_ADDREF(result);
    mResults.Put(filename, result);
  }

  return result;
}

struct HRJSTracer {
  JSTracer base;
  HRCov* thisp;
  JSFunction* found;
};

static void do_CC(JSContext* cx)
{
#if QTMSTR_DEBUG
  // nsCycleCollector_collect() will run a ::JS_GC() indirectly
  int collected_objects = nsCycleCollector_collect();
  int suspected_objects = nsCycleCollector_suspectedCount();
  fprintf(stderr, "Collected %u objects, %u suspected objects\n",
          collected_objects, suspected_objects);
#else
  JS_GC(cx);
#endif
}

/* void start(); */
NS_IMETHODIMP HRCov::Start()
{
  JSRuntime* runtime;
  nsresult rv;
  nsCOMPtr<nsIJSRuntimeService> runtimeService = do_GetService(
    XPC_RUNTIME_CONTRACTID, &rv);
  NS_ENSURE_SUCCESS(rv, rv);

  rv = runtimeService->GetRuntime(&runtime);
  NS_ENSURE_SUCCESS(rv, rv);

  nsCOMPtr<nsIThreadJSContextStack> stack =
    do_GetService("@mozilla.org/js/xpc/ContextStack;1");

  NS_ENSURE_TRUE(stack, NS_OK);

  JSContext* cx = NULL;
  rv = stack->GetSafeJSContext(&cx);
  NS_ENSURE_TRUE(cx, NS_ERROR_UNEXPECTED);

  nsCOMPtr<jsdIDebuggerService> debugger_service =
    do_GetService("@mozilla.org/js/jsd/debugger-service;1");
  if(debugger_service) {
    rv = debugger_service->GetIsOn(&mWasOn);
    NS_ENSURE_SUCCESS(rv, rv);

    rv = debugger_service->Off();
    NS_ENSURE_SUCCESS(rv, rv);
  }

  JSDebugHooks* hooks = JS_GetGlobalDebugHooks(runtime);
  if(hooks->interruptHandler || hooks->newScriptHook) {
    return NS_ERROR_NOT_AVAILABLE;
  }

  /***********************************************************************
   * Flush *EVERYTHING* we can from memory so that when we walk the JS   *
   * heap, we only see the scripts that are actually being used.         *
   **********************************************************************/

  {
    /* Purge Forward-Backward Cache */

    nsCOMPtr<nsIPrefService> pref_service =
      do_GetService("@mozilla.org/preferences-service;1", &rv);
    NS_ENSURE_SUCCESS(rv, rv);

    nsCOMPtr<nsIPrefBranch> branch;
    rv = pref_service->GetBranch(NULL, getter_AddRefs(branch));
    NS_ENSURE_SUCCESS(rv, rv);

    const char* max_views_pref = "browser.sessionhistory.max_total_viewers";

    PRInt32 old;
    rv = branch->GetIntPref(max_views_pref, &old);
    NS_ENSURE_SUCCESS(rv, rv);

    rv = branch->SetIntPref(max_views_pref, 0);
    NS_ENSURE_SUCCESS(rv, rv);

    rv = branch->SetIntPref(max_views_pref, old);
    NS_ENSURE_SUCCESS(rv, rv);
  }

  /* Trigger an XPCOM cycle collection and JS GC */
  do_CC(cx);
  do_CC(cx);

  /**
   * Trace everything so we can get a complete picture of our starting
   * set of scripts. Cross our fingers and hope we don't race here.
   */
  {
    JSAutoRequest autoreq(cx);
    HRJSTracer tracer;
    JS_TRACER_INIT(&tracer.base, cx, HRCov::TraceNotify);
    tracer.thisp = this;
    tracer.found = NULL;

    JS_TraceRuntime(&tracer.base);
    mTracedObjects.Clear();

#if QTMSTR_DEBUG
    if(tracer.found) {
      JS_DumpHeap(cx, stderr, NULL, 0, tracer.found, 2000, NULL);
    }
#endif

  }

  JS_SetInterrupt(runtime, HRCov::OnStepHook, this);
  JS_SetNewScriptHook(runtime, HRCov::OnNewScriptHook, this);
  return NS_OK;
}

/* void Stop(); */
NS_IMETHODIMP HRCov::Stop()
{
  nsresult rv;
  JSRuntime* runtime;

  nsCOMPtr<nsIJSRuntimeService> runtimeService = do_GetService(
    XPC_RUNTIME_CONTRACTID, &rv);

  NS_ENSURE_SUCCESS(rv, rv);

  rv = runtimeService->GetRuntime(&runtime);
  NS_ENSURE_SUCCESS(rv, rv);

  JSDebugHooks* hooks = JS_GetGlobalDebugHooks(runtime);
  if(hooks->interruptHandler != HRCov::OnStepHook) {
    return NS_ERROR_UNEXPECTED;
  }

  JS_SetInterrupt(runtime, NULL, NULL);
  JS_SetNewScriptHook(runtime, NULL, NULL);

  nsCOMPtr<jsdIDebuggerService> debugger_service =
    do_GetService("@mozilla.org/js/jsd/debugger-service;1");
  if(debugger_service) {
    if(mWasOn) {
      PRBool isOn;
      rv = debugger_service->GetIsOn(&isOn);
      NS_ENSURE_SUCCESS(rv, rv);

      if(isOn) {
        NS_WARNING("DEBUGGER SHOULD NOT HAVE BEEN RE-ENABLED");
      } else {
        rv = debugger_service->OnForRuntime(runtime);
        NS_ENSURE_SUCCESS(rv, rv);
      }
    }
  }


  return NS_OK;
}

static PLDHashOperator clear_data_helper(const char* uri,
                                       IHRCovResult* result,
                                       void* arg)
{
  NS_RELEASE(result);
  return PL_DHASH_NEXT;
}

/* void clearData (); */
NS_IMETHODIMP HRCov::ClearData()
{
  mResults.EnumerateRead(clear_data_helper, NULL);
  mResults.Clear();
  return NS_OK;
}

static PLDHashOperator get_data_helper(const char* uri,
                                       IHRCovResult* result,
                                       void* arg)
{
  IHRCovResult*** pit = (IHRCovResult***)arg;
  NS_ADDREF(result);
  **pit = result;
  *pit = *pit + 1;

  return PL_DHASH_NEXT;
}

/* void getData (out unsigned long num, [array, size_is (num), retval] out IHRCovResult results); */
NS_IMETHODIMP HRCov::GetData(PRUint32 *num, IHRCovResult ***results)
{
  *num = mResults.Count();
  *results = (IHRCovResult**)nsMemory::Alloc(
    sizeof(**results) * mResults.Count());

  if(!*results) {
    return NS_ERROR_OUT_OF_MEMORY;
  }

  IHRCovResult** it = *results;
  // addrefs each result and copies the pointer into ‘it’
  mResults.EnumerateRead(get_data_helper, &it);

  return NS_OK;
}


JSTrapStatus HRCov::OnStep(JSContext* cx, JSScript* script,
                           jsbytecode* pc, jsval* rval)
{
  const char* filename = JS_GetScriptFilename(cx, script);
  if(likely(filename != NULL)) {
    uintN lineno = JS_PCToLineNumber(cx, script, pc);

    HRCovResult* result = GetResult(filename);
    result->MarkExecutedLine(lineno);
  }

  return JSTRAP_CONTINUE;
}

#ifndef NDEBUG

static void
UpdateSwitchTableBounds(JSScript *script, uintN offset,
                        uintN *start, uintN *end)
{
    jsbytecode *pc;
    JSOp op;
    ptrdiff_t jmplen;
    jsint low, high, n;

    pc = script->code + offset;
    op = (JSOp) *pc;
    switch (op) {
      case JSOP_TABLESWITCHX:
        jmplen = JUMPX_OFFSET_LEN;
        goto jump_table;
      case JSOP_TABLESWITCH:
        jmplen = JUMP_OFFSET_LEN;
      jump_table:
        pc += jmplen;
        low = GET_JUMP_OFFSET(pc);
        pc += JUMP_OFFSET_LEN;
        high = GET_JUMP_OFFSET(pc);
        pc += JUMP_OFFSET_LEN;
        n = high - low + 1;
        break;

      case JSOP_LOOKUPSWITCHX:
        jmplen = JUMPX_OFFSET_LEN;
        goto lookup_table;
      case JSOP_LOOKUPSWITCH:
        jmplen = JUMP_OFFSET_LEN;
      lookup_table:
        pc += jmplen;
        n = GET_INDEX(pc);
        pc += INDEX_LEN;
        jmplen += JUMP_OFFSET_LEN;
        break;

      default:
        /* [condswitch] switch does not have any jump or lookup tables. */
        JS_ASSERT(op == JSOP_CONDSWITCH);
        return;
    }

    *start = (uintN)(pc - script->code);
    *end = *start + (uintN)(n * jmplen);
}

static void
DumpSrcNotes(JSContext *cx, JSScript *script, FILE* out)
{
    uintN offset, delta, caseOff, switchTableStart, switchTableEnd;
    jssrcnote *notes, *sn;
    JSSrcNoteType type;
    const char *name;
    uint32 index;
    JSAtom *atom;
    JSString *str;

    fprintf(out, "\nSource notes:\n");
    offset = 0;
    notes = SCRIPT_NOTES(script);
    switchTableEnd = switchTableStart = 0;
    for (sn = notes; !SN_IS_TERMINATOR(sn); sn = SN_NEXT(sn)) {
        delta = SN_DELTA(sn);
        offset += delta;
        type = (JSSrcNoteType) SN_TYPE(sn);
        name = js_SrcNoteSpec[type].name;
        if (type == SRC_LABEL) {
            /* Check if the source note is for a switch case. */
            if (switchTableStart <= offset && offset < switchTableEnd) {
                name = "case";
            } else {
                JS_ASSERT(script->code[offset] == JSOP_NOP);
            }
        }
        fprintf(out, "%3u: %5u [%4u] %-8s",
                PTRDIFF(sn, notes, jssrcnote), offset, delta, name);
        switch (type) {
          case SRC_SETLINE:
            fprintf(out, " lineno %u", (uintN) js_GetSrcNoteOffset(sn, 0));
            break;
          case SRC_FOR:
            fprintf(out, " cond %u update %u tail %u",
                   (uintN) js_GetSrcNoteOffset(sn, 0),
                   (uintN) js_GetSrcNoteOffset(sn, 1),
                   (uintN) js_GetSrcNoteOffset(sn, 2));
            break;
          case SRC_IF_ELSE:
            fprintf(out, " else %u elseif %u",
                   (uintN) js_GetSrcNoteOffset(sn, 0),
                   (uintN) js_GetSrcNoteOffset(sn, 1));
            break;
          case SRC_COND:
          case SRC_WHILE:
          case SRC_PCBASE:
          case SRC_PCDELTA:
          case SRC_DECL:
          case SRC_BRACE:
            fprintf(out, " offset %u", (uintN) js_GetSrcNoteOffset(sn, 0));
            break;
          case SRC_LABEL:
          case SRC_LABELBRACE:
          case SRC_BREAK2LABEL:
          case SRC_CONT2LABEL:
            index = js_GetSrcNoteOffset(sn, 0);
            JS_GET_SCRIPT_ATOM(script, index, atom);
            JS_ASSERT(ATOM_IS_STRING(atom));
            str = ATOM_TO_STRING(atom);
            fprintf(out, " atom %u (", index);
            js_FileEscapedString(out, str, 0);
            putc(')', out);
            break;
          case SRC_FUNCDEF: {
            const char *bytes;
            JSObject *obj;
            JSFunction *fun;

            index = js_GetSrcNoteOffset(sn, 0);
            JS_GET_SCRIPT_OBJECT(script, index, obj);
            fun = (JSFunction *) JS_GetPrivate(cx, obj);
            str = JS_DecompileFunction(cx, fun, JS_DONT_PRETTY_PRINT);
            bytes = str ? JS_GetStringBytes(str) : "N/A";
            fprintf(out, " function %u (%s)", index, bytes);
            break;
          }
          case SRC_SWITCH:
            fprintf(out, " length %u", (uintN) js_GetSrcNoteOffset(sn, 0));
            caseOff = (uintN) js_GetSrcNoteOffset(sn, 1);
            if (caseOff)
                fprintf(out, " first case offset %u", caseOff);
            UpdateSwitchTableBounds(script, offset,
                                    &switchTableStart, &switchTableEnd);
            break;
          case SRC_CATCH:
            delta = (uintN) js_GetSrcNoteOffset(sn, 0);
            if (delta) {
                if (script->main[offset] == JSOP_LEAVEBLOCK)
                    fprintf(out, " stack depth %u", delta);
                else
                    fprintf(out, " guard delta %u", delta);
            }
            break;
          default:;
        }
        fputc('\n', out);
    }
}

#endif // NDEBUG

/**
 * Copied from jsopcode.cpp. Use private copy of jsopcode.tbl so we can backport
 * it from more recent versions of Mozilla than we're compiling against
 */
struct HRJSCodeSpec {
    int8                length;         /* length including opcode byte */
};

static const HRJSCodeSpec HRjs_CodeSpec[] = {
#define OPDEF(op,val,name,token,length,nuses,ndefs,prec,format) \
    {length},
#include "jsopcode.tbl"
#undef OPDEF
};

static uintN HRjs_NumCodeSpecs = JS_ARRAY_LENGTH(HRjs_CodeSpec);

static uintN
GetVariableBytecodeLength(jsbytecode *pc)
{
  JSOp op;
  uintN jmplen, ncases;
  jsint low, high;

  op = (JSOp) *pc;
  JS_ASSERT(HRjs_CodeSpec[op].length == -1);
  switch (op) {
    case JSOP_TABLESWITCHX:
      jmplen = JUMPX_OFFSET_LEN;
      goto do_table;
    case JSOP_TABLESWITCH:
      jmplen = JUMP_OFFSET_LEN;
    do_table:
      /* Structure: default-jump case-low case-high case1-jump ... */
      pc += jmplen;
      low = GET_JUMP_OFFSET(pc);
      pc += JUMP_OFFSET_LEN;
      high = GET_JUMP_OFFSET(pc);
      ncases = (uintN)(high - low + 1);
      return 1 + jmplen + INDEX_LEN + INDEX_LEN + ncases * jmplen;

    case JSOP_LOOKUPSWITCHX:
      jmplen = JUMPX_OFFSET_LEN;
      goto do_lookup;
    default:
      JS_ASSERT(op == JSOP_LOOKUPSWITCH);
      jmplen = JUMP_OFFSET_LEN;
    do_lookup:
      /* Structure: default-jump case-count (case1-value case1-jump) ... */
      pc += jmplen;
      ncases = GET_UINT16(pc);
      return 1 + jmplen + INDEX_LEN + ncases * (INDEX_LEN + jmplen);
  }
}



void HRCov::OnNewScript(JSContext  *cx,
                        const char *filename,
                        uintN      decl_lineno,
                        JSScript   *script,
                        JSFunction *fun)
{
  if(unlikely(!filename)) {
    return;
  }

  HRCovResult* result = GetResult(filename);

  jsbytecode* end = script->code + script->length;
  const HRJSCodeSpec* cs;
  ptrdiff_t oplen;

  for(jsbytecode* pc = script->main; pc < end ; pc += oplen) {
    PR_ASSERT(*pc < HRjs_NumCodeSpecs);
    cs = &HRjs_CodeSpec[*pc];
    oplen = cs->length;
    if(oplen < 0) {
      oplen = GetVariableBytecodeLength(pc);
    }

    uintN lineno = JS_PCToLineNumber(cx, script, pc);

    /*
    if(strstr(filename, "jslib") &&
       lineno == 4986)
    {
      fprintf(stderr, "ARRRRRRRRGH!\n");
      js_Disassemble(cx, script, true, stderr);
    }
    */

    if(lineno) {
      result->MarkPotentialLine(lineno);
    }
  }
}

void HRCov::OnTrace(JSTracer* trc, void* thing, uint32 kind)
{
  if(mTracedObjects.GetEntry(thing)) {
    return;
  }

  HRJSTracer* hjs = (HRJSTracer*)trc;

  if(kind == JSTRACE_OBJECT) {
    JSContext* cx = trc->context;
    if(VALUE_IS_FUNCTION(cx, (jsval)thing)) {
      JSFunction* fun = JS_ValueToFunction(cx, (jsval)thing);
      JSScript* script = FUN_SCRIPT(fun);
      if(script) {
        const char* filename = JS_GetScriptFilename(cx, script);

        /* Use the block below if you think you're seeing a stale script */
#if 0
        if(filename && strstr(filename, "jslib")
           && JS_GetFunctionName(fun))
        {
          fprintf(stderr, "FOUND JSLIB (func %s in \"%s\" at %lu)\n",
                  JS_GetFunctionName(fun),
                  filename,
                  (unsigned long)JS_GetScriptBaseLineNumber(cx, script));
          if(!hjs->found) {
            hjs->found = fun;
          }
        }
#endif

        OnNewScript(trc->context,
                    filename,
                    JS_GetScriptBaseLineNumber(cx, script),
                    script,
                    fun);
      }
    }
  }

  mTracedObjects.PutEntry(thing);
  JS_TraceChildren(trc, thing, kind);
}

JS_DLL_CALLBACK JSTrapStatus
HRCov::OnStepHook(JSContext *cx, JSScript *script,
              jsbytecode *pc, jsval *rval, void *closure)
{
  if(!NS_IsMainThread()) {
    return JSTRAP_CONTINUE;
  }

  return ((HRCov*)closure)->OnStep(cx, script, pc, rval);
}

JS_DLL_CALLBACK void
HRCov::OnNewScriptHook(JSContext  *cx,
                       const char *filename,
                       uintN      lineno,
                       JSScript   *script,
                       JSFunction *fun,
                       void       *callerdata)
{
  if(!NS_IsMainThread()) {
    return;
  }

  ((HRCov*)callerdata)->OnNewScript(cx, filename, lineno, script, fun);
}

JS_DLL_CALLBACK void
HRCov::TraceNotify(JSTracer* trc, void* thing, uint32 kind)
{
  ((HRJSTracer*)trc)->thisp->OnTrace(trc, thing, kind);
}
