
/*
 * Copyright (C) 2006 Jan Wrobel <wrobel@blues.ath.cx>
 * Copyright (C) 2005 Sourcefire,Inc.
 */

#include <sys/types.h>
#include <stdlib.h>
#include <ctype.h>

#include "str_search.h"
#include "acsmx.h"

typedef struct tag_search
{
	ACSM_STRUCT *acsm;
	unsigned int max_len;
} t_search;

static t_search *_acsm = NULL;
static unsigned int  _num_acsm;

/*
  Init multi-pattern search engine
  @num: number of pattern sets to use (Firekeeper uses 3 different sets - 
  for searching for url_content, headers_content and body_content)
 */
int SearchInit(unsigned int num)
{
	unsigned int i;

	_num_acsm = num;

	_acsm = (t_search *)malloc(sizeof(t_search) * num);
	if ( _acsm == NULL )
		return -1;

	for ( i = 0; i < num; i++ )
		{
			_acsm[i].acsm = acsmNew();			
			if ( !_acsm[i].acsm )
				return -1;
			acsmCompile(_acsm[i].acsm);
			_acsm[i].max_len = 0;
		}
	return 0;
}

/*
  Free old pattern sets and init new ones.
 */
int SearchReInit(unsigned int i)
{
	if ( _acsm[i].acsm != NULL )
		acsmFree(_acsm[i].acsm);
	_acsm[i].acsm = acsmNew();
	_acsm[i].max_len = 0;
    
	if ( !_acsm[i].acsm )
		return -1;

	return 0;
}

/*
  Free all pattern sets.
 */
void SearchFree()
{
	unsigned int i;

	if ( _acsm != NULL )
		{
			for ( i = 0; i < _num_acsm; i++ )
				{
					if ( _acsm[i].acsm != NULL )
						acsmFree(_acsm[i].acsm);
				}
			free(_acsm);
		}
}


/*  
    Do efficient search of data 
    @acsm_id    specify in which patterns set search
    @str        string to search
    @str_len    length of string to search
    @Match      function callback when string match found
*/
int SearchFindString(unsigned int acsm_id, unsigned char *str, 
		     unsigned int str_len, int (*Match) (void *, int, void *),
		     void *ctx)
{

	int num = acsmSearch(_acsm[acsm_id].acsm, str, str_len, Match, ctx);
	return num;
}


/*
  Add pattern to search for.
  @acsm_id    specify to which patterns set add this pattern
  @pat        pattern to add
  @pat_len    length of pattern to add
  @nocase     if true do case insensitive search
  @id         pointer to a data that will identify this pattern when there is
              a match
 */
void SearchAdd(unsigned int acsm_id, unsigned char *pat, unsigned int pat_len, 
	       int nocase, void* id)
{
	acsmAddPattern(_acsm[acsm_id].acsm, pat, pat_len, nocase, 0, 0, id, 0);
	acsmCompile(_acsm[acsm_id].acsm);
	
	if ( pat_len > _acsm[acsm_id].max_len )
		_acsm[acsm_id].max_len = pat_len;
}


/* API exported by this module */
static SearchAPI _searchAPI =
	{
		SearchInit,
		SearchReInit,
		SearchFree,
		SearchAdd,
		SearchFindString
	};

/*Global search API*/
SearchAPI *searchAPI = &_searchAPI;

