/*$Id: fkLoadGroup.cpp,v 1.7 2007/08/18 08:07:53 jwrobel Exp $*/
/* ***** BEGIN LICENSE BLOCK *****
 *  This file is part of Firekeeper.
 *
 *  Copyright (C) 2007 Jan Wrobel <wrobel@blues.ath.cx>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * ***** END LICENSE BLOCK ***** */

#include "nsISupportsImpl.h"
#include "nsCOMPtr.h"
#include "fkLoadGroup.h"
#include "fkHttpHandler.h"

fkLoadGroup::fkLoadGroup(nsILoadGroup *group)
{
	TRACE("constructor %08x", this);
	nsresult rv;

	FK_ASSERT(group);
	origLoadGroup = group;
	callbacksProxy = nsnull;

	groupHashTable->Put(origLoadGroup, this);
#ifdef DEBUG
	nsCOMPtr<fkILoadGroupProxy> proxy = do_QueryInterface(origLoadGroup, &rv);
	FK_ASSERT(NS_FAILED(rv));
	nsCOMPtr<nsIRequest> tmp;
	origLoadGroup->GetDefaultLoadRequest(getter_AddRefs(tmp));
	if (tmp){
		TRACE("default load request set %08x", &*tmp);
	}
	else
		TRACE("default load request not set");
#endif

	origRequest = do_QueryInterface(origLoadGroup, &rv);
	FK_ASSERT(!NS_FAILED(rv));

	origLoadGroup->GetGroupObserver(getter_AddRefs(origObserver));

	origRequests = nsnull;
	proxyRequests = nsnull;
	//origLoadGroup->SetGroupObserver(this);
}


fkLoadGroup::~fkLoadGroup()
{
	TRACE("destructor %08x", this);
	FK_ASSERT(groupHashTable->Get(origLoadGroup, nsnull));

	groupHashTable->Remove(origLoadGroup);
	TRACE("HashTable count %d", groupHashTable->Count());

	if (origRequests){
		origRequests->Clear();
		delete origRequests;
	}

	if (proxyRequests){
		proxyRequests->Clear();
		delete proxyRequests;
	}
}

NS_IMETHODIMP
fkLoadGroup::init()
{
	nsresult rv = NS_ERROR_OUT_OF_MEMORY;
	origRequests = new nsInterfaceHashtable<nsISupportsHashKey, nsIRequest>;
	if (!origRequests || NS_FAILED(rv = origRequests->Init()))
		return rv;

	rv = NS_ERROR_OUT_OF_MEMORY;
	proxyRequests = new nsInterfaceHashtable<nsISupportsHashKey, nsIRequest>;
	if (!proxyRequests || NS_FAILED(rv = proxyRequests->Init()))
		return rv;

	nsCOMPtr<nsIInterfaceRequestor> cb;
	origLoadGroup->GetNotificationCallbacks(getter_AddRefs(cb));
	if (cb)
		rv = SetNotificationCallbacks(cb);
	else
		rv = NS_OK;

	return rv;
}


NS_IMPL_ADDREF(fkLoadGroup)
NS_IMPL_RELEASE(fkLoadGroup)


/*Only interfaces implemented by origLoadGroup are returned.*/
NS_IMETHODIMP
fkLoadGroup::QueryInterface(const nsIID &aIID, void **aResult)
{
	if (aResult == NULL) {
		return NS_ERROR_NULL_POINTER;
	}
	nsISupports *foundInterface = 0;
	*aResult = NULL;

	if (aIID.Equals(NS_GET_IID(nsISupports))){
		foundInterface = NS_STATIC_CAST(nsISupports*,
						NS_STATIC_CAST(nsISupportsWeakReference*, this));
	}else if (aIID.Equals(NS_GET_IID(nsISupportsWeakReference))){
		foundInterface = NS_STATIC_CAST(nsISupportsWeakReference*, this);
	}else if (aIID.Equals(NS_GET_IID(fkILoadGroupProxy))){
		TRACE("returning load group proxy");
		foundInterface = NS_STATIC_CAST(fkILoadGroupProxy*, this);
	}else if (aIID.Equals(NS_GET_IID(nsILoadGroup)) && origLoadGroup){
		foundInterface = NS_STATIC_CAST(nsILoadGroup*, this);
	}else if (aIID.Equals(NS_GET_IID(nsIRequestObserver)) && origObserver){
		foundInterface = NS_STATIC_CAST(nsIRequestObserver*, this);
	}else if (aIID.Equals(NS_GET_IID(nsIRequest)) && origRequest){
		foundInterface = NS_STATIC_CAST(nsIRequest*, this);
	}

	if (!foundInterface) {
		TRACE("fkLoadGroup: interface not implemented %s .", aIID.ToString());
#ifdef DEBUG
		if (origLoadGroup){
			void *dummy = nsnull;
			/*this code trigers assertion when aIID is nsIInterfaceRequestor (?)*/

			/*This leaks memory but only in debug mode when FK_ASSERT will be
			  triggered anyway*/
			nsresult rv = origLoadGroup->QueryInterface(aIID, &dummy);
			if (!NS_FAILED(rv) && dummy){
				TRACE("origLoadGroup implements this!!!!");
				FK_ASSERT(!dummy);
			}
		}
#endif
		return NS_NOINTERFACE;
	}

	TRACE("interface implemented %s .", aIID.ToString());
	NS_ADDREF(foundInterface);
	*aResult = foundInterface;
	return NS_OK;
}


//-----------------------------------------------------------------------------
// fkLoadGroup::fkILoadGroupProxy
//-----------------------------------------------------------------------------
NS_IMETHODIMP
fkLoadGroup::GetOrigLoadGroup(nsILoadGroup **aLoadGroup)
{
	TRACE("fkLoadGroup::fkILoadGroupProxy");
	NS_ENSURE_ARG_POINTER(aLoadGroup);
	NS_IF_ADDREF(*aLoadGroup = origLoadGroup);
	return NS_OK;
}

//-----------------------------------------------------------------------------
// fkLoadGroup::nsILoadGroup
//-----------------------------------------------------------------------------
NS_IMETHODIMP
fkLoadGroup::GetGroupObserver(nsIRequestObserver **aGroupObserver)
{
	TRACE("fkLoadGroup::nsILoadGroup");
	FK_ASSERT(false);
	return NS_ERROR_NOT_IMPLEMENTED;

	NS_ENSURE_ARG_POINTER(aGroupObserver);

	if (!origObserver){
		TRACE("origObserver not set");
		*aGroupObserver = nsnull;
	}else{
		/*Return this object as a group observer in order to
		  intercept calls to OnStartRequest and OnStopReqeust*/
		NS_IF_ADDREF(*aGroupObserver = this);
	}
	return NS_OK;
}

NS_IMETHODIMP
fkLoadGroup::SetGroupObserver(nsIRequestObserver *aGroupObserver)
{
	TRACE("fkLoadGroup::nsILoadGroup");
	FK_ASSERT(false);
	return NS_ERROR_NOT_IMPLEMENTED;

	origObserver = aGroupObserver;
	if (!origObserver)
		return origLoadGroup->SetGroupObserver(nsnull);
	return origLoadGroup->SetGroupObserver(this);
}

NS_IMETHODIMP
fkLoadGroup::GetDefaultLoadRequest(nsIRequest **aDefaultLoadRequest)
{
	TRACE("fkLoadGroup::nsILoadGroup");
	FK_ASSERT(false);
	return NS_ERROR_NOT_IMPLEMENTED;

	NS_ENSURE_ARG_POINTER(aDefaultLoadRequest);
	if (!origLoadGroup){
		*aDefaultLoadRequest = nsnull;
		return NS_ERROR_FAILURE;
	}
	return origLoadGroup->GetDefaultLoadRequest(aDefaultLoadRequest);
}

NS_IMETHODIMP
fkLoadGroup::SetDefaultLoadRequest(nsIRequest *aDefaultLoadRequest)
{
	TRACE("fkLoadGroup::nsILoadGroup");
	FK_ASSERT(false);
	return NS_ERROR_NOT_IMPLEMENTED;

	if (!origLoadGroup){
		return NS_ERROR_FAILURE;
	}
	return origLoadGroup->SetDefaultLoadRequest(aDefaultLoadRequest);
}

NS_IMETHODIMP
fkLoadGroup::AddRequest(nsIRequest *aRequest, nsISupports *aContext)
{
	TRACE("fkLoadGroup::nsILoadGroup arg %08x", &*aRequest);
	nsCOMPtr<nsIRequest> proxy = getRequestProxy(aRequest);
	if (!proxy){
		TRACE("PROXY NOT SET");
		FK_ASSERT(false);

		return origLoadGroup->AddRequest(aRequest, aContext);
	}
	origRequests->Put(proxy, aRequest);
	proxyRequests->Put(aRequest, proxy);
	TRACE("Adding proxy %08x", &*proxy);
	nsresult rv =  origLoadGroup->AddRequest(proxy, aContext);
	FK_ASSERT(!NS_FAILED(rv));
	return rv;
}

NS_IMETHODIMP
fkLoadGroup::RemoveRequest(nsIRequest *aRequest, nsISupports *aContext, nsresult aStatus)
{
	NS_ENSURE_ARG_POINTER(aRequest);
	TRACE("fkLoadGroup::nsILoadGroup arg %08x", &*aRequest);

	nsCOMPtr<nsIRequest> orig, proxy;

	if (origRequests->Get(aRequest, getter_AddRefs(orig))){
		FK_ASSERT(false);
		proxy = aRequest;
		FK_ASSERT(proxyRequests->Get(orig, nsnull));
		origRequests->Remove(proxy);
		proxyRequests->Remove(orig);
	}else{
		if (!proxyRequests->Get(aRequest, getter_AddRefs(proxy))){
			TRACE("WARNING: the same request removed twice from its loadgroup");
			return NS_ERROR_FAILURE;
		}
		orig = aRequest;
		FK_ASSERT(origRequests->Get(proxy, nsnull));
		origRequests->Remove(proxy);
		proxyRequests->Remove(orig);
	}

	TRACE("Removing proxy %08x %08x", &*proxy, &*origLoadGroup);

	nsresult rv = origLoadGroup->RemoveRequest(proxy, aContext, aStatus);
#ifdef DEBUG
	if (NS_FAILED(rv)){
		TRACE("remove request failed");
	}
#endif
	return rv;
}

NS_IMETHODIMP
fkLoadGroup::GetRequests(nsISimpleEnumerator **aRequests)
{
	TRACE("fkLoadGroup::nsILoadGroup");
	FK_ASSERT(false);
	return NS_ERROR_NOT_IMPLEMENTED;

	if (!origLoadGroup)
		return NS_ERROR_FAILURE;
	return origLoadGroup->GetRequests(aRequests);
}

NS_IMETHODIMP
fkLoadGroup::GetActiveCount(PRUint32 *aActiveCount)
{
	TRACE("fkLoadGroup::nsILoadGroup");
	if (!origLoadGroup)
		return NS_ERROR_FAILURE;
	return origLoadGroup->GetActiveCount(aActiveCount);
}

NS_IMETHODIMP
fkLoadGroup::GetNotificationCallbacks(nsIInterfaceRequestor **callbacks)
{
	TRACE("fkLoadGroup::nsILoadGroup %08x", this);
	NS_ENSURE_ARG_POINTER(callbacks);

#ifdef DEBUG
	nsCOMPtr<nsIInterfaceRequestor> tmp;
	origLoadGroup->GetNotificationCallbacks(getter_AddRefs(tmp));
	if (tmp != callbacksProxy){
		TRACE("callbacks mismatch %08x %08x", &*tmp, &*callbacksProxy);
		FK_ASSERT(false);
	}
#endif

	if (!callbacksProxy){
		FK_ASSERT(false);
		*callbacks = nsnull;
		return NS_OK;
	}
	nsresult rv;
	nsCOMPtr<fkICallbacksProxy> p = do_QueryInterface(callbacksProxy, &rv);
	FK_ASSERT(!NS_FAILED(rv));

	return p->GetOrigCallbacks(callbacks);
}

NS_IMETHODIMP
fkLoadGroup::SetNotificationCallbacks(nsIInterfaceRequestor *callbacks)
{
	nsresult rv;
	TRACE("fkLoadGroup::nsILoadGroup setting callbacks %08x this = %08x",
	      callbacks, this);
	callbacksProxy = nsnull;

	if (callbacks){
		nsCOMPtr<fkICallbacksProxy> tmp = do_QueryInterface(callbacks, &rv);
		if (!NS_FAILED(rv)){
			TRACE("proxy already used");
			callbacksProxy = callbacks;
		}else{
			fkCallbacks *proxy;
			if (!callbacksHashTable->Get(callbacks, &proxy)){
				TRACE("callbacks not found");
				/* TODO: pass load group */
				proxy = new fkCallbacks(nsnull, callbacks);

				if (!proxy)
					return NS_ERROR_OUT_OF_MEMORY;
			}
			else{
				TRACE("callbacks found");
			}
			callbacksProxy = proxy;
		}
	}
	return origLoadGroup->SetNotificationCallbacks(callbacksProxy);

}

//-----------------------------------------------------------------------------
// fkLoadGroup::nsIRequestObserver
//-----------------------------------------------------------------------------
NS_IMETHODIMP
fkLoadGroup::OnStartRequest(nsIRequest *aRequest, nsISupports *aContext)
{
	TRACE("fkLoadGroup::nsIRequestObserver");
	if (!origObserver){
		TRACE("Observer not set");
		return NS_OK;
	}
	return origObserver->OnStartRequest(tryGetRequestProxy(aRequest), aContext);
}

NS_IMETHODIMP
fkLoadGroup::OnStopRequest(nsIRequest *aRequest, nsISupports *aContext, nsresult aStatusCode)
{
	TRACE("fkLoadGroup::nsIRequestObserver");
	if (!origObserver){
		TRACE("Observer not set");
		return NS_OK;
	}
	return origObserver->OnStopRequest(tryGetRequestProxy(aRequest), aContext, aStatusCode);
}


//-----------------------------------------------------------------------------
// fkLoadGroup::nsIRequest
//-----------------------------------------------------------------------------
NS_IMETHODIMP
fkLoadGroup::GetName(nsACString &aName)
{
	TRACE("fkLoadGroup::nsIRequest");
	if (!origRequest)
		return NS_ERROR_FAILURE;
	return origRequest->GetName(aName);
}

NS_IMETHODIMP
fkLoadGroup::IsPending(PRBool *value)
{
	TRACE("fkLoadGroup::nsIRequest");
	if (!origRequest)
		return NS_ERROR_FAILURE;
	return origRequest->IsPending(value);
}

NS_IMETHODIMP
fkLoadGroup::GetStatus(nsresult *aStatus)
{
	TRACE("fkLoadGroup::nsIRequest");
	if (!origRequest)
		return NS_ERROR_FAILURE;
	return origRequest->GetStatus(aStatus);
}

NS_IMETHODIMP
fkLoadGroup::Cancel(nsresult status)
{
	TRACE("fkLoadGroup::nsIRequest");
	if (!origRequest)
		return NS_ERROR_FAILURE;
	return origRequest->Cancel(status);
}

NS_IMETHODIMP
fkLoadGroup::Suspend()
{
	TRACE("fkLoadGroup::nsIRequest");
	if (!origRequest)
		return NS_ERROR_FAILURE;
	return origRequest->Suspend();
}

NS_IMETHODIMP
fkLoadGroup::Resume()
{
	TRACE("fkLoadGroup::nsIRequest");
	if (!origRequest)
		return NS_ERROR_FAILURE;
	return origRequest->Resume();
}

NS_IMETHODIMP
fkLoadGroup::GetLoadGroup(nsILoadGroup **aLoadGroup)
{
	TRACE("fkLoadGroup::nsIRequest");
	FK_ASSERT(false);
	return NS_ERROR_NOT_IMPLEMENTED;
}

NS_IMETHODIMP
fkLoadGroup::SetLoadGroup(nsILoadGroup *aLoadGroup)
{
	TRACE("fkLoadGroup::nsIRequest group %08x %08x",
	      this, aLoadGroup);
	FK_ASSERT(false);
	return NS_ERROR_NOT_IMPLEMENTED;
}

NS_IMETHODIMP
fkLoadGroup::GetLoadFlags(nsLoadFlags *aLoadFlags)
{
	if (!origRequest)
		return NS_ERROR_FAILURE;
	return origRequest->GetLoadFlags(aLoadFlags);
}

NS_IMETHODIMP
fkLoadGroup::SetLoadFlags(nsLoadFlags aLoadFlags)
{
	if (!origRequest){
		return NS_ERROR_FAILURE;
	}
	return origRequest->SetLoadFlags(aLoadFlags);
}
