
function nsContextMenu( xulMenu ) {
    this.target            = null;
    this.menu              = null;
    this.onTextInput       = false;
    this.onKeywordField    = false;
    this.onImage           = false;
    this.onLoadedImage     = false;
    this.onLink            = false;
    this.onMailtoLink      = false;
    this.onSaveableLink    = false;
    this.onMetaDataItem    = false;
    this.onMathML          = false;
    this.link              = false;
    this.linkURL           = "";
    this.linkURI           = null;
    this.linkProtocol      = null;
    this.inFrame           = false;
    this.hasBGImage        = false;
    this.isTextSelected    = false;
    this.isContentSelected = false;
    this.inDirList         = false;
    this.shouldDisplay     = true;
    this.isDesignMode      = false;
    this.possibleSpellChecking = false;

    // Initialize new menu.
    this.initMenu( xulMenu );
}

// Prototype for nsContextMenu "class."
nsContextMenu.prototype = {
    // onDestroy is a no-op at this point.
    onDestroy : function () {
    },
    // Initialize context menu.
    initMenu : function ( popup ) {
        // Save menu.
        this.menu = popup;

        // Get contextual info.
        this.setTarget( document.popupNode, document.popupRangeParent,
                        document.popupRangeOffset );

        this.isTextSelected = this.isTextSelection();
        this.isContentSelected = this.isContentSelection();

        // Initialize (disable/remove) menu items.
        this.initItems();
    },
    initItems : function () {
        this.initOpenItems();
        this.initNavigationItems();
        this.initViewItems();
        this.initMiscItems();
        this.initSpellingItems();
        this.initSaveItems();
        this.initClipboardItems();
        this.initMetadataItems();
    },
    initOpenItems : function () {
        this.showItem( "context-openlinkintab", this.onSaveableLink || ( this.inDirList && this.onLink ) );

        this.showItem( "context-sep-open", this.onSaveableLink || ( this.inDirList && this.onLink ) );
    },
    initNavigationItems : function () {
        // Back determined by canGoBack broadcaster.
        this.setItemAttrFromNode( "context-back", "disabled", "canGoBack" );

        // Forward determined by canGoForward broadcaster.
        this.setItemAttrFromNode( "context-forward", "disabled", "canGoForward" );

        this.showItem( "context-back", !( this.isContentSelected || this.onTextInput ) );
        this.showItem( "context-forward", !( this.isContentSelected || this.onTextInput ) );

        this.showItem( "context-reload", !( this.isContentSelected || this.onTextInput ) );

        this.showItem( "context-stop", !( this.isContentSelected || this.onTextInput ) );
        this.showItem( "context-sep-stop", !( this.isContentSelected || this.onTextInput) );

        // XXX: Stop is determined in navigator.js; the canStop broadcaster is broken
        //this.setItemAttrFromNode( "context-stop", "disabled", "canStop" );
    },
    initSaveItems : function () {
    },
    initViewItems : function () {
    },
    initMiscItems : function () {
    },
    initSpellingItems : function () {
    },
    initClipboardItems : function () {

        // Copy depends on whether there is selected text.
        // Enabling this context menu item is now done through the global
        // command updating system
        // this.setItemAttr( "context-copy", "disabled", !this.isTextSelected() );

        goUpdateGlobalEditMenuItems();

        this.showItem( "context-undo", this.onTextInput );
        this.showItem( "context-sep-undo", this.onTextInput );
        this.showItem( "context-cut", this.onTextInput );
        this.showItem( "context-copy", this.isContentSelected || this.onTextInput );
        this.showItem( "context-paste", this.onTextInput );
        this.showItem( "context-delete", this.onTextInput );
        this.showItem( "context-sep-paste", this.onTextInput );
        this.showItem( "context-selectall", !( this.onLink || this.onImage ) || this.isDesignMode );
        this.showItem( "context-sep-selectall", this.isContentSelected );

        // Copy link location depends on whether we're on a link.
        this.showItem( "context-copylink", this.onLink );
        this.showItem( "context-sep-copylink", this.onLink);

    },
    initMetadataItems : function () {
    },
    // Set various context menu attributes based on the state of the world.
    setTarget : function ( node, rangeParent, rangeOffset ) {
        const xulNS = "http://www.mozilla.org/keymaster/gatekeeper/there.is.only.xul";
        if ( node.namespaceURI == xulNS ) {
          this.shouldDisplay = false;
          return;
        }

        // Initialize contextual info.
        this.onImage           = false;
        this.onLoadedImage     = false;
        this.onStandaloneImage = false;
        this.onMetaDataItem    = false;
        this.onTextInput       = false;
        this.onKeywordField    = false;
        this.imageURL          = "";
        this.onLink            = false;
        this.linkURL           = "";
        this.linkURI           = null;
        this.linkProtocol      = "";
        this.onMathML          = false;
        this.inFrame           = false;
        this.hasBGImage        = false;
        this.bgImageURL        = "";
        this.possibleSpellChecking = false;

        // Remember the node that was clicked.
        this.target = node;
        
        // Remember the URL of the document containing the node
        // for referrer header and for security checks.
        this.docURL = node.ownerDocument.location.href;

        // First, do checks for nodes that never have children.
        if ( this.target.nodeType == Node.ELEMENT_NODE ) {
            // See if the user clicked on an image.
            if ( this.target instanceof Components.interfaces.nsIImageLoadingContent && this.target.currentURI  ) {
                this.onImage = true;
                this.onMetaDataItem = true;
                        
                var request = this.target.getRequest( Components.interfaces.nsIImageLoadingContent.CURRENT_REQUEST );
                if (request && (request.imageStatus & request.STATUS_SIZE_AVAILABLE))
                    this.onLoadedImage = true;
                this.imageURL = this.target.currentURI.spec;

                if ( this.target.ownerDocument instanceof ImageDocument)
                   this.onStandaloneImage = true;
            } else if ( this.target instanceof HTMLInputElement ) {
               this.onTextInput = this.isTargetATextBox(this.target);
               // allow spellchecking UI on all writable text boxes except passwords
               if (this.onTextInput && ! this.target.readOnly && this.target.type != "password") {
                   this.possibleSpellChecking = true;
               }
               this.onKeywordField = this.isTargetAKeywordField(this.target);
            } else if ( this.target instanceof HTMLTextAreaElement ) {
                 this.onTextInput = true;
                 if (! this.target.readOnly) {
                     this.possibleSpellChecking = true;
                 }
            } else if ( this.target instanceof HTMLHtmlElement ) {
               // pages with multiple <body>s are lame. we'll teach them a lesson.
               var bodyElt = this.target.ownerDocument.getElementsByTagName("body")[0];
               if ( bodyElt ) {
                 var computedURL = this.getComputedURL( bodyElt, "background-image" );
                 if ( computedURL ) {
                   this.hasBGImage = true;
                   this.bgImageURL = makeURLAbsolute( bodyElt.baseURI,
                                                      computedURL );
                 }
               }
            } else if ( "HTTPIndex" in content &&
                        content.HTTPIndex instanceof Components.interfaces.nsIHTTPIndex ) {
                this.inDirList = true;
                // Bubble outward till we get to an element with URL attribute
                // (which should be the href).
                var root = this.target;
                while ( root && !this.link ) {
                    if ( root.tagName == "tree" ) {
                        // Hit root of tree; must have clicked in empty space;
                        // thus, no link.
                        break;
                    }
                    if ( root.getAttribute( "URL" ) ) {
                        // Build pseudo link object so link-related functions work.
                        this.onLink = true;
                        this.link = { href : root.getAttribute("URL"),
                                      getAttribute: function (attr) {
                                          if (attr == "title") {
                                              return root.firstChild.firstChild.getAttribute("label");
                                          } else {
                                              return "";
                                          }
                                      }
                                    };
                        // If element is a directory, then you can't save it.
                        if ( root.getAttribute( "container" ) == "true" ) {
                            this.onSaveableLink = false;
                        } else {
                            this.onSaveableLink = true;
                        }
                    } else {
                        root = root.parentNode;
                    }
                }
            }
        }

        // Second, bubble out, looking for items of interest that can have childen.
        // Always pick the innermost link, background image, etc.
        
        const XMLNS = "http://www.w3.org/XML/1998/namespace";
        var elem = this.target;
        while ( elem ) {
            if ( elem.nodeType == Node.ELEMENT_NODE ) {
            
                // Link?
                if ( !this.onLink &&
                     ( (elem instanceof HTMLAnchorElement && elem.href) ||
                        elem instanceof HTMLAreaElement ||
                        elem instanceof HTMLLinkElement ||
                        elem.getAttributeNS( "http://www.w3.org/1999/xlink", "type") == "simple" ) ) {
                    
                    // Target is a link or a descendant of a link.
                    this.onLink = true;
                    this.onMetaDataItem = true;

                    // xxxmpc: this is kind of a hack to work around a Gecko bug (see bug 266932)
                    // we're going to walk up the DOM looking for a parent link node,
                    // this shouldn't be necessary, but we're matching the existing behaviour for left click
                    var realLink = elem;
                    var parent = elem.parentNode;
                    while (parent) {
                      try {
                        if ( (parent instanceof HTMLAnchorElement && elem.href) ||
                             parent instanceof HTMLAreaElement ||
                             parent instanceof HTMLLinkElement ||
                             parent.getAttributeNS( "http://www.w3.org/1999/xlink", "type") == "simple")
                          realLink = parent;
                      } catch (e) {}
                      parent = parent.parentNode;
                    }
                    
                    // Remember corresponding element.
                    this.link = realLink;
                    this.linkURL = this.getLinkURL();
                    this.linkURI = this.getLinkURI();
                    this.linkProtocol = this.getLinkProtocol();
                    this.onMailtoLink = (this.linkProtocol == "mailto");
                    this.onSaveableLink = this.isLinkSaveable( this.link );
                }

                // Metadata item?
                if ( !this.onMetaDataItem ) {
                    // We display metadata on anything which fits
                    // the below test, as well as for links and images
                    // (which set this.onMetaDataItem to true elsewhere)
                    if ( ( elem instanceof HTMLQuoteElement && elem.cite)    ||
                         ( elem instanceof HTMLTableElement && elem.summary) ||
                         ( elem instanceof HTMLModElement &&
                             ( elem.cite || elem.dateTime ) )                ||
                         ( elem instanceof HTMLElement &&
                             ( elem.title || elem.lang ) )                   ||
                         elem.getAttributeNS(XMLNS, "lang") ) {
                        this.onMetaDataItem = true;
                    }
                }

                // Background image?  Don't bother if we've already found a
                // background image further down the hierarchy.  Otherwise,
                // we look for the computed background-image style.
                if ( !this.hasBGImage ) {
                    var bgImgUrl = this.getComputedURL( elem, "background-image" );
                    if ( bgImgUrl ) {
                        this.hasBGImage = true;
                        this.bgImageURL = makeURLAbsolute( elem.baseURI,
                                                           bgImgUrl );
                    }
                }
            }
            elem = elem.parentNode;
        }
        
        // See if the user clicked on MathML
        const NS_MathML = "http://www.w3.org/1998/Math/MathML";
        if ((this.target.nodeType == Node.TEXT_NODE &&
             this.target.parentNode.namespaceURI == NS_MathML)
             || (this.target.namespaceURI == NS_MathML))
          this.onMathML = true;

        // See if the user clicked in a frame.
        if ( this.target.ownerDocument != window.content.document ) {
            this.inFrame = true;
        }

        // if the document is editable, show context menu like in text inputs
        var win = this.target.ownerDocument.defaultView;
        if (win) {
          var editingSession = win.QueryInterface(Components.interfaces.nsIInterfaceRequestor)
                                  .getInterface(Components.interfaces.nsIWebNavigation)
                                  .QueryInterface(Components.interfaces.nsIInterfaceRequestor)
                                  .getInterface(Components.interfaces.nsIEditingSession);
          if (editingSession.windowIsEditable(win)) {
            this.onTextInput       = true;
            this.onKeywordField    = false;
            this.onImage           = false;
            this.onLoadedImage     = false;
            this.onMetaDataItem    = false;
            this.onMathML          = false;
            this.inFrame           = false;
            this.hasBGImage        = false;
            this.isDesignMode      = true;
            this.possibleSpellChecking = true;
          }
        }
    },
    // Returns the computed style attribute for the given element.
    getComputedStyle: function( elem, prop ) {
         return elem.ownerDocument.defaultView.getComputedStyle( elem, '' ).getPropertyValue( prop );
    },
    // Returns a "url"-type computed style attribute value, with the url() stripped.
    getComputedURL: function( elem, prop ) {
         var url = elem.ownerDocument.defaultView.getComputedStyle( elem, '' ).getPropertyCSSValue( prop );
         return ( url.primitiveType == CSSPrimitiveValue.CSS_URI ) ? url.getStringValue() : null;
    },
    // Returns true if clicked-on link targets a resource that can be saved.
    isLinkSaveable : function ( link ) {
        // We don't do the Right Thing for news/snews yet, so turn them off
        // until we do.
        return this.linkProtocol && !(
                 this.linkProtocol == "mailto"     ||
                 this.linkProtocol == "javascript" ||
                 this.linkProtocol == "news"       ||
                 this.linkProtocol == "snews"      );
    },

    // Open linked-to URL in a new window.
    openLink : function () {
        openNewWindowWith(this.linkURL, this.docURL, null, false);
    },
    // Open linked-to URL in a new tab.
    openLinkInTab : function () {
        openNewTabWith(this.linkURL, this.docURL, null, null, false);
    },
    // Open frame in a new tab.
    openFrameInTab : function () {
        openNewTabWith(this.target.ownerDocument.location.href, null, null, null, false);
    },
    // Reload clicked-in frame.
    reloadFrame : function () {
        this.target.ownerDocument.location.reload();
    },
    // Open clicked-in frame in its own window.
    openFrame : function () {
        openNewWindowWith(this.target.ownerDocument.location.href, null, null, false);
    },
    // Open clicked-in frame in the same window.
    showOnlyThisFrame : function () {
      const nsIScriptSecMan = Components.interfaces.nsIScriptSecurityManager;
      var frameURL = this.target.ownerDocument.location.href;

      try {
        urlSecurityCheck(frameURL, gBrowser.currentURI.spec,
                         nsIScriptSecMan.DISALLOW_SCRIPT);
        window.loadURI(frameURL, null, null, false);
      } catch(e) {}
    },
    // View Partial Source
    viewPartialSource : function ( context ) {
        var focusedWindow = document.commandDispatcher.focusedWindow;
        if (focusedWindow == window)
          focusedWindow = content;
        var docCharset = null;
        if (focusedWindow)
          docCharset = "charset=" + focusedWindow.document.characterSet;

        // "View Selection Source" and others such as "View MathML Source"
        // are mutually exclusive, with the precedence given to the selection
        // when there is one
        var reference = null;
        if (context == "selection")
          reference = focusedWindow.getSelection();
        else if (context == "mathml")
          reference = this.target;
        else
          throw "not reached";

        var docUrl = null; // unused (and play nice for fragments generated via XSLT too)
        window.openDialog("chrome://global/content/viewPartialSource.xul",
                          "_blank", "scrollbars,resizable,chrome,dialog=no",
                          docUrl, docCharset, reference, context);
    },
    // Open new "view source" window with the frame's URL.
    viewFrameSource : function () {
        BrowserViewSourceOfDocument(this.target.ownerDocument);
    },
    viewInfo : function () {
      BrowserPageInfo();
    },
    viewFrameInfo : function () {
      BrowserPageInfo(this.target.ownerDocument);
    },
    // Change current window to the URL of the image.
    viewImage : function (e) {
        const nsIScriptSecMan = Components.interfaces.nsIScriptSecurityManager;
        urlSecurityCheck( this.imageURL, gBrowser.currentURI.spec,
                          nsIScriptSecMan.DISALLOW_SCRIPT );
        openUILink( this.imageURL, e );
    },
    // Change current window to the URL of the background image.
    viewBGImage : function (e) {
        const nsIScriptSecMan = Components.interfaces.nsIScriptSecurityManager;
        urlSecurityCheck( this.bgImageURL, gBrowser.currentURI.spec,
                          nsIScriptSecMan.DISALLOW_SCRIPT );
        openUILink( this.bgImageURL, e );
    },
    disableSetDesktopBackground: function() {
        // Disable the Set as Desktop Background menu item if we're still trying
        // to load the image or the load failed.

        const nsIImageLoadingContent = Components.interfaces.nsIImageLoadingContent;
        if (!(this.target instanceof nsIImageLoadingContent))
            return true;

        if (("complete" in this.target) && !this.target.complete)
            return true;

        if (this.target.currentURI.schemeIs("javascript"))
            return true;

        var request = this.target.QueryInterface(nsIImageLoadingContent)
                                 .getRequest(nsIImageLoadingContent.CURRENT_REQUEST);
        if (!request)
            return true;

        return false;
    },
    setDesktopBackground: function() {
      // Paranoia: check disableSetDesktopBackground again, in case the
      // image changed since the context menu was initiated.
      if (this.disableSetDesktopBackground())
        return;

      urlSecurityCheck(this.target.currentURI.spec, this.docURL);

      // Confirm since it's annoying if you hit this accidentally.
      const kDesktopBackgroundURL = 
                    "chrome://browser/content/setDesktopBackground.xul";
//@line 4935 "/cygdrive/c/builds/tinderbox/Fx-Mozilla1.8-Release/WINNT_5.2_Depend/mozilla/browser/base/content/browser.js"
      // On non-Mac platforms, the Set Wallpaper dialog is modal.
      openDialog(kDesktopBackgroundURL, "",
                 "centerscreen,chrome,dialog,modal,dependent",
                 this.target);
//@line 4940 "/cygdrive/c/builds/tinderbox/Fx-Mozilla1.8-Release/WINNT_5.2_Depend/mozilla/browser/base/content/browser.js"
    },
    // Save URL of clicked-on frame.
    saveFrame : function () {
        saveDocument( this.target.ownerDocument );
    },
    // Save URL of clicked-on link.
    saveLink : function () {
        urlSecurityCheck(this.linkURL, this.docURL);
        saveURL( this.linkURL, this.linkText(), null, true, false,
                 makeURI(this.docURL, this.target.ownerDocument.characterSet) );
    },
    sendLink : function () {
        MailIntegration.sendMessage( this.linkURL, "" ); // we don't know the title of the link so pass in an empty string
    },
    // Save URL of clicked-on image.
    saveImage : function () {
        urlSecurityCheck(this.imageURL, this.docURL);
        saveImageURL( this.imageURL, null, "SaveImageTitle", false,
                      false, makeURI(this.docURL) );
    },
    sendImage : function () {
        MailIntegration.sendMessage(this.imageURL, "");
    },
    toggleImageBlocking : function (aBlock) {
      var nsIPermissionManager = Components.interfaces.nsIPermissionManager;
      var permissionmanager =
        Components.classes["@mozilla.org/permissionmanager;1"]
                  .getService(nsIPermissionManager);

      var uri = this.target.QueryInterface(Components.interfaces.nsIImageLoadingContent).currentURI;

      permissionmanager.add(uri, "image",
                            aBlock ? nsIPermissionManager.DENY_ACTION : nsIPermissionManager.ALLOW_ACTION);

      var savedmenu = this;
      function undoImageBlock() {
        savedmenu.toggleImageBlocking(!aBlock);
      }

      var brandBundle = document.getElementById("bundle_brand");
      var app = brandBundle.getString("brandShortName");
      var bundle_browser = document.getElementById("bundle_browser");
      var message;
      if (aBlock)
        message = bundle_browser.getFormattedString("imageBlockedWarning",
                                                    [app, uri.host]);
      else 
        message = bundle_browser.getFormattedString("imageAllowedWarning",
                                                    [app, uri.host]);

      var notificationBox = gBrowser.getNotificationBox();
      var notification = notificationBox.getNotificationWithValue("images-blocked");

      if (notification)
        notification.label = message;
      else {
        var buttons = [{
          label: bundle_browser.getString("undo"),
          accessKey: bundle_browser.getString("undo.accessKey"),
          callback: undoImageBlock
         }];
         const priority = notificationBox.PRIORITY_WARNING_MEDIUM;
         notificationBox.appendNotification(message, "images-blocked",
                                            "chrome://browser/skin/Info.png",
                                             priority, buttons);
      }

      // Reload the page to show the effect instantly
      BrowserReload();
    },
    isImageBlocked : function() {
      var nsIPermissionManager = Components.interfaces.nsIPermissionManager;
      var permissionmanager =
        Components.classes["@mozilla.org/permissionmanager;1"]
          .getService(Components.interfaces.nsIPermissionManager);

      var uri = this.target.QueryInterface(Components.interfaces.nsIImageLoadingContent).currentURI;

      return permissionmanager.testPermission(uri, "image") == nsIPermissionManager.DENY_ACTION;
    },
    // Generate email address and put it on clipboard.
    copyEmail : function () {
        // Copy the comma-separated list of email addresses only.
        // There are other ways of embedding email addresses in a mailto:
        // link, but such complex parsing is beyond us.
        var url = this.linkURL;

        var qmark = url.indexOf( "?" );
        var addresses;

        if ( qmark > 7 ) {                   // 7 == length of "mailto:"
            addresses = url.substring( 7, qmark );
        } else {
            addresses = url.substr( 7 );
        }

        // Let's try to unescape it using a character set
        // in case the address is not ASCII.
        try {
          var characterSet = this.target.ownerDocument.characterSet;
          const textToSubURI = Components.classes["@mozilla.org/intl/texttosuburi;1"]
                                         .getService(Components.interfaces.nsITextToSubURI);
          addresses = textToSubURI.unEscapeURIForUI(characterSet, addresses);
        }
        catch(ex) {
          // Do nothing.
        }

        var clipboard = this.getService( "@mozilla.org/widget/clipboardhelper;1",
                                         Components.interfaces.nsIClipboardHelper );
        clipboard.copyString(addresses);
    },
    addBookmark : function() {
      var docshell = document.getElementById( "content" ).webNavigation;
//@line 5055 "/cygdrive/c/builds/tinderbox/Fx-Mozilla1.8-Release/WINNT_5.2_Depend/mozilla/browser/base/content/browser.js"
      BookmarksUtils.addBookmark( docshell.currentURI.spec,
                                  docshell.document.title,
                                  docshell.document.charset,
                                  BookmarksUtils.getDescriptionFromDocument(docshell.document));
//@line 5062 "/cygdrive/c/builds/tinderbox/Fx-Mozilla1.8-Release/WINNT_5.2_Depend/mozilla/browser/base/content/browser.js"
    },
    addBookmarkForFrame : function() {
//@line 5065 "/cygdrive/c/builds/tinderbox/Fx-Mozilla1.8-Release/WINNT_5.2_Depend/mozilla/browser/base/content/browser.js"
      var doc = this.target.ownerDocument;
      var uri = doc.location.href;
      var title = doc.title;
      var description = BookmarksUtils.getDescriptionFromDocument(doc);
      if ( !title )
        title = uri;
      BookmarksUtils.addBookmark(uri, title, doc.charset, description);
//@line 5075 "/cygdrive/c/builds/tinderbox/Fx-Mozilla1.8-Release/WINNT_5.2_Depend/mozilla/browser/base/content/browser.js"
    },
    // Open Metadata window for node
    showMetadata : function () {
        window.openDialog(  "chrome://browser/content/metaData.xul",
                            "_blank",
                            "scrollbars,resizable,chrome,dialog=no",
                            this.target);
    },

    ///////////////
    // Utilities //
    ///////////////

    // Create instance of component given contractId and iid (as string).
    createInstance : function ( contractId, iidName ) {
        var iid = Components.interfaces[ iidName ];
        return Components.classes[ contractId ].createInstance( iid );
    },
    // Get service given contractId and iid (as string).
    getService : function ( contractId, iidName ) {
        var iid = Components.interfaces[ iidName ];
        return Components.classes[ contractId ].getService( iid );
    },
    // Show/hide one item (specified via name or the item element itself).
    showItem : function ( itemOrId, show ) {
        var item = itemOrId.constructor == String ? document.getElementById(itemOrId) : itemOrId;
        if (item)
          item.hidden = !show;
    },
    // Set given attribute of specified context-menu item.  If the
    // value is null, then it removes the attribute (which works
    // nicely for the disabled attribute).
    setItemAttr : function ( id, attr, val ) {
        var elem = document.getElementById( id );
        if ( elem ) {
            if ( val == null ) {
                // null indicates attr should be removed.
                elem.removeAttribute( attr );
            } else {
                // Set attr=val.
                elem.setAttribute( attr, val );
            }
        }
    },
    // Set context menu attribute according to like attribute of another node
    // (such as a broadcaster).
    setItemAttrFromNode : function ( item_id, attr, other_id ) {
        var elem = document.getElementById( other_id );
        if ( elem && elem.getAttribute( attr ) == "true" ) {
            this.setItemAttr( item_id, attr, "true" );
        } else {
            this.setItemAttr( item_id, attr, null );
        }
    },
    // Temporary workaround for DOM api not yet implemented by XUL nodes.
    cloneNode : function ( item ) {
        // Create another element like the one we're cloning.
        var node = document.createElement( item.tagName );

        // Copy attributes from argument item to the new one.
        var attrs = item.attributes;
        for ( var i = 0; i < attrs.length; i++ ) {
            var attr = attrs.item( i );
            node.setAttribute( attr.nodeName, attr.nodeValue );
        }

        // Voila!
        return node;
    },
    // Generate fully qualified URL for clicked-on link.
    getLinkURL : function () {
        var href = this.link.href;
        
        if (href) {
          return href;
        }

        var href = this.link.getAttributeNS("http://www.w3.org/1999/xlink",
                                          "href");

        if (!href || !href.match(/\S/)) {
          throw "Empty href"; // Without this we try to save as the current doc, for example, HTML case also throws if empty
        }
        href = makeURLAbsolute(this.link.baseURI, href);
        return href;
    },
    
    getLinkURI : function () {
         var ioService = Components.classes["@mozilla.org/network/io-service;1"].getService(Components.interfaces.nsIIOService);
         try {
           return ioService.newURI(this.linkURL, null, null);
         } catch (ex) {
           // e.g. empty URL string
           return null;
         }
    },
    
    getLinkProtocol : function () {
        if (this.linkURI) {
            return this.linkURI.scheme; // can be |undefined|
        } else {
            return null;
        }
    },

    // Get text of link.
    linkText : function () {
        var text = gatherTextUnder( this.link );
        if (!text || !text.match(/\S/)) {
          text = this.link.getAttribute("title");
          if (!text || !text.match(/\S/)) {
            text = this.link.getAttribute("alt");
            if (!text || !text.match(/\S/)) {
              text = this.linkURL;
            }
          }
        }

        return text;
    },

    // Get selected text. Only display the first 15 chars.
    isTextSelection : function() {
      // Get 16 characters, so that we can trim the selection if it's greater
      // than 15 chars
      var selectedText = getBrowserSelection(16);

      if (!selectedText)
        return false;

      if (selectedText.length > 15)
        selectedText = selectedText.substr(0,15) + "...";

      return true;
    },

    // Returns true if anything is selected.
    isContentSelection : function() {
        return !document.commandDispatcher.focusedWindow.getSelection().isCollapsed;
    },

    toString : function () {
        return "contextMenu.target     = " + this.target + "\n" +
               "contextMenu.onImage    = " + this.onImage + "\n" +
               "contextMenu.onLink     = " + this.onLink + "\n" +
               "contextMenu.link       = " + this.link + "\n" +
               "contextMenu.inFrame    = " + this.inFrame + "\n" +
               "contextMenu.hasBGImage = " + this.hasBGImage + "\n";
    },

    isTargetATextBox : function ( node )
    {
      if (node instanceof HTMLInputElement)
        return (node.type == "text" || node.type == "password")

      return (node instanceof HTMLTextAreaElement);
    },
    isTargetAKeywordField : function ( node )
    {
      var form = node.form;
      if (!form)
        return false;
      var method = form.method.toUpperCase();

      // These are the following types of forms we can create keywords for:
      //
      // method   encoding type       can create keyword
      // GET      *                                 YES
      //          *                                 YES
      // POST                                       YES
      // POST     application/x-www-form-urlencoded YES
      // POST     text/plain                        NO (a little tricky to do)
      // POST     multipart/form-data               NO
      // POST     everything else                   YES
      return (method == "GET" || method == "") ||
             (form.enctype != "text/plain") && (form.enctype != "multipart/form-data");
    },

    // Determines whether or not the separator with the specified ID should be
    // shown or not by determining if there are any non-hidden items between it
    // and the previous separator.
    shouldShowSeparator : function ( aSeparatorID )
    {
      var separator = document.getElementById(aSeparatorID);
      if (separator) {
        var sibling = separator.previousSibling;
        while (sibling && sibling.localName != "menuseparator") {
          if (sibling.getAttribute("hidden") != "true")
            return true;
          sibling = sibling.previousSibling;
        }
      }
      return false;
    },

    addDictionaries : function()
    {
      var uri = formatURL("browser.dictionaries.download.url", true);

      var locale = "-";
      try {
        locale = gPrefService.getComplexValue("intl.accept_languages",
                                Components.interfaces.nsIPrefLocalizedString).data;
      }
      catch (e) { }

      var version = "-";
      try {
        version = Components.classes["@mozilla.org/xre/app-info;1"]
                            .getService(Components.interfaces.nsIXULAppInfo)
                            .version;
      }
      catch (e) { }

      uri = uri.replace(/%LOCALE%/, escape(locale));
      uri = uri.replace(/%VERSION%/, version);

      var newWindowPref = gPrefService.getIntPref("browser.link.open_newwindow");
      var where = newWindowPref == 3 ? "tab" : "window";

      openUILinkIn(uri, where);
    }
}

/**
 * Gets the selected text in the active browser. Leading and trailing
 * whitespace is removed, and consecutive whitespace is replaced by a single
 * space. A maximum of 150 characters will be returned, regardless of the value
 * of aCharLen.
 *
 * @param aCharLen
 *        The maximum number of characters to return.
 */
function getBrowserSelection(aCharLen) {
  // selections of more than 150 characters aren't useful
  const kMaxSelectionLen = 150;
  const charLen = Math.min(aCharLen || kMaxSelectionLen, kMaxSelectionLen);

  var focusedWindow = document.commandDispatcher.focusedWindow;
  var selection = focusedWindow.getSelection().toString();

  if (selection) {
    if (selection.length > charLen) {
      // only use the first charLen important chars. see bug 221361
      var pattern = new RegExp("^(?:\\s*.){0," + charLen + "}");
      pattern.test(selection);
      selection = RegExp.lastMatch;
    }

    selection = selection.replace(/^\s+/, "")
                         .replace(/\s+$/, "")
                         .replace(/\s+/g, " ");

    if (selection.length > charLen)
      selection = selection.substr(0, charLen);
  }
  return selection;
}

function makeURLAbsolute( base, url )
{
  // Construct nsIURL.
  var ioService = Components.classes["@mozilla.org/network/io-service;1"]
                .getService(Components.interfaces.nsIIOService);
  var baseURI  = ioService.newURI(base, null, null);

  return ioService.newURI(baseURI.resolve(url), null, null).spec;
}

// Gather all descendent text under given document node.
function gatherTextUnder ( root ) 
{
  var text = "";
  var node = root.firstChild;
  var depth = 1;
  while ( node && depth > 0 ) {
    // See if this node is text.
    if ( node.nodeType == Node.TEXT_NODE ) {
      // Add this text to our collection.
      text += " " + node.data;
    } else if ( node instanceof HTMLImageElement) {
      // If it has an alt= attribute, use that.
      var altText = node.getAttribute( "alt" );
      if ( altText && altText != "" ) {
        text = altText;
        break;
      }
    }
    // Find next node to test.
    // First, see if this node has children.
    if ( node.hasChildNodes() ) {
      // Go to first child.
      node = node.firstChild;
      depth++;
    } else {
      // No children, try next sibling.
      if ( node.nextSibling ) {
        node = node.nextSibling;
      } else {
        // Last resort is our next oldest uncle/aunt.
        node = node.parentNode.nextSibling;
        depth--;
      }
    }
  }
  // Strip leading whitespace.
  text = text.replace( /^\s+/, "" );
  // Strip trailing whitespace.
  text = text.replace( /\s+$/, "" );
  // Compress remaining whitespace.
  text = text.replace( /\s+/g, " " );
  return text;
}