/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is hultmann localization tools.
 *
 * The Initial Developer of the Original Code is
 * Jeferson Hultmann <hultmann@gmail.com>
 * Portions created by the Initial Developer are Copyright (C) 2005
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either of the GNU General Public License Version 2 or later (the "GPL"),
 * or the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

using System;
using System.Collections.Generic;
using System.Text.RegularExpressions;

namespace LocaleInpector
{

internal sealed class Inspector
{

public static int Main()
{
    int ret = MainCore();
    return ret;
}

public static Inspector Current;

private static int MainCore()
{
    string title = "Locale Inspector " + Config.GetVersion();
    //System.Console.OutputEncoding = BPIO.UTF8;
    System.Console.Title = title;
    System.Console.BackgroundColor = System.ConsoleColor.Blue;
    System.Console.ForegroundColor = System.ConsoleColor.Gray;
    Output.Write(title);
    Output.WriteLine(" - a test suite for Mozilla localizers.");
    System.Console.ResetColor();


    Output.WriteLine("Latest version: http://downloads.mozdev.org/br/locale-inspector.zip");
    Output.WriteLine("Jeferson Hultmann, pt-BR localization <hultmann@gmail.com>");

    Inspector theApp = new Inspector();
    return theApp.Parse();
}

private readonly List<string>     m_modules;
private          string           m_mergeModule;
private          string           m_originalModule;
private          string           m_translatedModule;
private          Config           m_config;

private Inspector()
{
    Current = this;
    m_config = new Config();
    m_modules = m_config.InitIni();
}

private int Parse()
{
    if (m_modules == null) {
        return WriteHeader();
    }

    bool ret;

    string fileLog = Config.Current.GetPref("general", "output");
    if (fileLog.Length > 0) {
        fileLog = System.IO.Path.GetFullPath(fileLog);
        using (System.IO.StringWriter log = new System.IO.StringWriter()) {
            System.IO.TextWriter screen = Console.Out;
            Console.Write("Saving output to " + fileLog);
            Console.SetOut(log);
            ret = this.ParseCore();
            BPIO.Save(log.ToString(), fileLog, BPIO.UTF8);
            Console.SetOut(screen);
        }
        Console.WriteLine(" OK");
        Console.WriteLine();

    } else {
        ret = this.ParseCore();
        Console.WriteLine();
    }

    return ret ? 0 : 1;
}


private bool ParseCore()
{
    m_config.Start();


    if (m_modules.Count == 0) {
        Output.WriteLine("No modules have been declared to check.");
        return true;
    }

    Status allStat = new Status();
    Status mergeStat = new Status();

    for (int idx = 0; idx < m_modules.Count; idx++) {

        Status stat = new Status();
        Status mergeStat2 = new Status();

        Output.OpenH1(m_modules[idx]);
        Output.Write("Original:  ");
        m_originalModule = m_config.ValidateDir(m_modules[idx], "original");
        Output.WriteLine(string.Empty);
        Output.Write("Localized: ");
        m_translatedModule = m_config.ValidateDir(m_modules[idx], "translated");
        Output.WriteLine(string.Empty);

        m_mergeModule = m_config.GetPref(m_modules[idx], "merge");
        if (m_mergeModule.Length > 0) {
            if (m_mergeModule[m_mergeModule.Length - 1] != System.IO.Path.DirectorySeparatorChar) {
                m_mergeModule = m_mergeModule + System.IO.Path.DirectorySeparatorChar.ToString();
            }
            Output.WriteLine("Merge:     " + m_mergeModule);
        } else {
            m_mergeModule = null;
        }

        if (m_originalModule == null || m_translatedModule == null) {
            stat.Errors++;

        } else {
            stat.Add(this.ParseModule());

            if (m_mergeModule != null) {
                mergeStat2 = Merge.Save(m_originalModule, m_translatedModule, m_mergeModule);
                if (mergeStat2.Files > 0) {
                    Output.OpenH2("TESTING MERGED FILES");
                    Output.Close();
                    Output.WriteLine("From: " + m_mergeModule);
                    mergeStat.Add(this.MakeTests(m_mergeModule));
                }
            }

            string pathGlossary = m_config.GetPref(m_modules[idx], "glossary");
            if (pathGlossary.Length > 0) {
                Glossary.Create(m_originalModule, m_translatedModule, pathGlossary);
            }
        }

        allStat.Add(stat);
        if (stat.Empty && mergeStat2.Empty) {
            if (mergeStat2.Comparisons == 0) {
                //Output.WriteLine("------------Clear");
                Output.Clear();
            }
        }
        Output.Close();
    }

    if (mergeStat.Empty) {
        m_config.End(allStat, m_modules.Count);
    } else {
        m_config.End(mergeStat, m_modules.Count);
    }
    return true;
}

private Status ParseModule()
{
    Status stat = new Status();

    stat.Add(this.MakeTests(m_translatedModule));

    if (this.DoTest("directories")) {
        stat.Add(Typos.Check(m_translatedModule));
    }
    return stat;
}

private bool DoTest(string test)
{
    //if (m_config.GetPref("general", "tests") != "true") {
    //    return false;
    //}
    if (m_config.GetPref(test, "enabled") != "true") {
        return false;
    }
    return true;
}

private Status MakeTests(string translatedModule)
{
    Status stat = new Status();

    if (this.DoTest("directories")) {
        stat.Add(Directories.Check(m_originalModule, translatedModule));
    }

    if (this.DoTest("files")) {
        stat.Add(Files.Check(m_originalModule, translatedModule));
    }

    if (this.DoTest("bom")) {
        stat.Add(Bom.Check(translatedModule));
    }

    if (this.DoTest("phrases")) {
        stat.Add(Keys.Check(m_originalModule, translatedModule));
    }

    if (this.DoTest("encoding")) {
        stat.Add(EscapeTest.Check(m_originalModule, translatedModule));
    }

    if (this.DoTest("keybinding")) {
        stat.Add(Phrases.CheckKeyBinding(m_originalModule, translatedModule));
    }

    if (this.DoTest("accesskey")) {
        stat.Add(AccessKey.Check(m_originalModule, translatedModule));
    }

    if (this.DoTest("c-format")) {
        stat.Add(Phrases.CheckCFormat(m_originalModule, translatedModule));
    }

    if (this.DoTest("entities")) {
        stat.Add(Phrases.CheckEntities(m_originalModule, translatedModule));
    }

    if (this.DoTest("placeholders")) {
        stat.Add(Phrases.CheckPlaceholders(m_originalModule, translatedModule));
    }

    if (this.DoTest("markup")) {
        stat.Add(Phrases.CheckMarkup(m_originalModule, translatedModule));
    }

    if (this.DoTest("untranslated")) {
        stat.Add(Phrases.CheckUntranslated(m_originalModule, translatedModule));
    }

    return stat;
}

private static int WriteHeader()
{
Console.Write(
@"
Usage:     locale-inspector.exe <config.ini>

");

    return 1;
}

}//class
}//ns