/*
  xpcomp.h - header file for xpcomp

  Copyright (C) 2001 Ingo K"ohne
  
  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
*/

#if !defined (_XPCOMP_H_)
#  define _XPCOMP_H_

#include "config.h"
#if ! defined(HAVE_LIBDL) || ! defined(HAVE_DLOPEN) || ! defined(HAVE_DLCLOSE) || ! defined(HAVE_DLFCN_H) || ! defined(HAVE_DLSYM)
#  error Dynamically loaded builtins not supported in bash
#endif
#if ! defined(PROGRAMMABLE_COMPLETION)
#  error Programmable Completion not configured in bash
#endif

#include "defines.h"
#include "pcomplete.h"

#include <stdio.h>
#include <assert.h>

/* identifiation string at the beginning of a cache file */
#define XPC_MAGIC "xpcomp cache:"
#define XPC_MAGIC_LEN 13

/* cache version number, backward compatibility is guarateed if the major
 version number is the same */
#define XPC_MAJOR 1
#define XPC_MINOR 0


#undef ABS
#define ABS(a) ( (a) > 0 ? (a) : -(a) )
#undef MIN
#define MIN(x,y) ( (x) < (y) ? (x) : (y) )
#undef MAX
#define MAX(x,y) ( (x) > (y) ? (x) : (y) )

/* converts the expansion of macro X to a string literal */
#define __VALUESTRING(X) __STRING(X)


typedef unsigned int XPC_COUNT;

/* cache index, describes layout of all contained data as needed to load a
 cache file */
struct xpc_index
{
  char magic[XPC_MAGIC_LEN + 1];
  char major;
  char minor;
  XPC_COUNT cmd_count;
  XPC_COUNT opt_count;
  XPC_COUNT compspec_count;
  XPC_COUNT chr_count;
};
typedef struct xpc_index XPC_INDEX;

/* completion features of an option completion */
struct xpc_opt_mode
{
  unsigned int arg:2;
  unsigned int suboptions:1;
  unsigned int varname:1;
  unsigned int noappend:1;
  unsigned int filename:1;
  unsigned int cut:1;
  unsigned int subcommand:1;
  unsigned int array:1;
  unsigned int plusopt:1;
  unsigned int rlhint:1;
  unsigned int added_info:1;
  unsigned int description_only:1;
  unsigned int glued:1;
};
typedef struct xpc_opt_mode XPC_OPT_MODE;
/* values for arg, correspond to getopt. */
enum
{
  arg_required = 3,
  arg_optional = 2,
  arg_none = 1
};

/* option completion specifier */
struct xpc_opt
{
  XPC_COUNT name;
  XPC_OPT_MODE mode;
  XPC_COUNT compspec;
  XPC_COUNT test;
  XPC_COUNT varname;
  char subsep;
};
typedef struct xpc_opt XPC_OPT;

typedef char XPC_CHR;

/* completion features for all options of a command */
struct xpc_cmd_mode
{
  unsigned int longopt1dash:1;
  unsigned int longopt2dash:1;
  unsigned int getoptparse:1;
  unsigned int getoptstop:1;
  unsigned int prefinvisible:1;
  unsigned int separateopt:1;
  unsigned int plusopt:1;
};
typedef struct xpc_cmd_mode XPC_CMD_MODE;

/* completed command specifier */
struct xpc_cmd
{
  XPC_COUNT name;
  XPC_CMD_MODE mode;
  XPC_COUNT opts;
  XPC_COUNT opt_count;
  XPC_COUNT opt_len;		/* sum of strlen()s of optnames */
};
typedef struct xpc_cmd XPC_CMD;

/* cache desciptor, describes a list of loaded cache files and their locations
 in memory */
struct xpc_cache
{
  char *filename;
  struct xpc_cache *next;
  XPC_INDEX *idx;
  XPC_CMD *cmds;
  XPC_OPT *opts;
  COMPSPEC *compspecs;
  XPC_CHR *chrs;
  unsigned long int checksum;
};
typedef struct xpc_cache XPC_CACHE;

/* the 'current' cache */
extern XPC_CACHE *cache;

/* access the contained structures by their index with additional check */
static __inline__ XPC_CMD *
GETCMD (XPC_COUNT n)
{
  assert (cache);
  assert (cache->cmds);
  assert (n <= cache->idx->cmd_count);
  return cache->cmds + n;
}

static __inline__ XPC_OPT *
GETOPT (XPC_COUNT n)
{
  assert (cache);
  assert (cache->opts);
  assert (n <= cache->idx->opt_count);
  return cache->opts + n;
}

static __inline__ COMPSPEC *
GETCOMPSPEC (XPC_COUNT n)
{
  assert (cache);
  assert (cache->compspecs);
  assert (n <= cache->idx->compspec_count);
  return cache->compspecs + n;
}

static __inline__ XPC_CHR *
GETCHR (XPC_COUNT n)
{
  assert (cache);
  assert (cache->chrs);
  assert (n <= cache->idx->chr_count);
  return cache->chrs + n;
}

/* the location of some compecs is predefined as this saves a lot of space */
enum
{
  COMPSPEC_EMPTY = 1,
  COMPSPEC_FILE = 2,
  COMPSPEC_DIRECTORY = 3,
  COMPSPEC_COMMAND = 4,
  COMPSPEC_RESERVED = 8
};

/* heap structures similar to a loaded cache file but mutable  */
extern XPC_CACHE heap;

extern XPC_CMD *xpc_cmd_find __P ((const char *cmdname));
extern XPC_CMD *xpc_cmd_find_all __P ((const char *cmdname));
extern XPC_CMD *xpc_cmd_add __P ((char *cmdname));
extern void xpc_cmd_del __P ((XPC_CMD * cmd));
extern void xpc_cmd_del_all __P (());
extern XPC_OPT *xpc_cmd_addopt __P ((XPC_CMD * cmd, char *optname));
extern void xpc_cmd_delopt __P ((XPC_CMD * cmd, XPC_OPT * opt));

extern XPC_OPT *xpc_opt_find __P ((XPC_CMD * cmd, char *optname));
extern int xpc_opt_match
__P ((XPC_CMD * cmd, const char *pattern, XPC_OPT ** opt));

extern void xpc_compspec_print
__P ((FILE * stream, COMPSPEC * cs, unsigned int local));
extern XPC_COUNT xpc_compspec_add __P ((COMPSPEC * values));
extern void xpc_compspec_del __P ((XPC_COUNT n));
extern void xpc_compspec_get __P ((XPC_COUNT n, COMPSPEC * ret));

extern void xpc_str_del __P ((XPC_COUNT str));
extern XPC_COUNT xpc_str_add __P ((char *str));

extern int xpc_libinit __P (());

extern void xpc_cache_set __P ((XPC_CACHE * c));
extern int xpc_cache_validate __P (());
extern unsigned long int xpc_cache_checksum __P (());
extern void xpc_heap_cache_delete __P (());
extern void xpc_cache_gc __P (());

/* current state of the xpcomp builtins */
enum xpc_state
{
  XPC_OK = 0,
  XPC_UNINITIALIZED,
  XPC_EXPANDING,
  XPC_DISABLED
};
typedef enum xpc_state XPC_STATE;
extern XPC_STATE xpcomp_state;

extern int xpc_debug;

extern int xpc_restrict;

extern FILE *errstream;

extern char const *xpc_errmsg;

extern jmp_buf self_destruct;

/* the macros try to return to the bash main loop in case of a fatal error,
 disableing the builtins but letting the user finish his bash session */
#if defined(DEBUG)
#  define xpc_disable_if_not assert
#else
/* redefine assert to disable all xpcomp builtins in the cases where assert
   would have aborted */
static const char *ASSERTFAIL_MSG =
  "Initialize self destruct sequence NOW!\n"
  "All memory areas will be abandoned immediately.\n"
  "Memory leaks on all decks.\n"
  "Please leave the bash as soon as you can, sorry.\n";
#  undef assert
#  define assert(expr) \
          ( (expr) ? 1 : \
	    ( fprintf(stderr, "xpcomp: In %s:%i:\nAssertion (%s) would have failed\n%s", \
		      __FILE__, __LINE__, __STRING(expr), ASSERTFAIL_MSG), \
	      longjmp(self_destruct,1), 0 ) )

static const char *FATAL_MSG =
  "This is propably due to an operating system call failure\n"
  "that is likely to crash the bash ( such as a failed memory allocation ).\n"
  "Please leave the bash as soon as you can, sorry.\n";
#  define xpc_disable_if_not(expr) \
          ( (expr) ? 1 : \
	    ( fprintf(stderr, "xpcomp: In %s:%i:\nFatal condition !(%s) occured.\n%s", \
		      __FILE__, __LINE__, __STRING(expr), FATAL_MSG ), \
	      longjmp(self_destruct,1), 0 ) )
#endif

#endif /* _XPCOMP_H_ */
