##	Things Copyright(C) 2009 Donn.C.Ingle
##
##	Contact: donn.ingle@gmail.com - I hope this email lasts.
##
##  This file is part of Things.
##
##  Things is free software: you can redistribute it and/or modify
##  it under the terms of the GNU General Public License as published by
##  the Free Software Foundation, either version 3 of the License, or
##  (at your option) any later version.
##
##  Things is distributed in the hope that it will be useful,
##  but WITHOUT ANY WARRANTY; without even the implied warranty of
##  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##  GNU General Public License for more details.
##
##  You should have received a copy of the GNU General Public License
##  along with Things.  If not, see <http://www.gnu.org/licenses/>.



"""
Things is a Retained-Mode Canvas Animation API

The idea is to create a spark, to light a
fire. This is only a prototype, and my
hope is that the Free Software boffins
out there will turn this into a tight
C/C++ library and then craft wrappers for
scripting languages like Python.

You have your orders! :)	

@author: Donn C. Ingle
@license: GPL version 3.
@contact: donn.ingle@gmail.com

"""

import depcheck


from bugs import Bugs as _Bugs
from BagOfStuff import *
from ThingObjects import *
from timeline import *
import Stack

import gobject
import gtk

_VERSION = (0,4)


class AllThings(Timeline):
	"""
	This is the primary Timeline. You should instance this class
	to begin work. After that you add other Things to it.
	
	An AllThings object has only one frame and cannot have keys.
	You should create at least one Thing object and add it to AllThings,
	then work from there.

	Note
	====
	If you want to install custom fonts (using a BagOfStuff) then you should
	do so *before* you instantiate AllThings.

	Use
	===

	 1. app = AllThings ( 400, 400, speed = 10, title = "TEST" )
	 2. app.add( SomeOtherThing )
	"""
	def __init__( self, canvas_width, canvas_height, window_width=None, window_height=None, speed = 30, hideCursor=False, title = "Things!"):
		"""
		@param window_width, window_height: Integer for the size of the window. If you exclude the canvas size, this will be the canvas size too.
		@param canvas_width, canvas_height: Integer for the size of the Canvas. This is the area you will draw into with 0,0 in the middle. 
		 It's the area that you would have used in Inkscape, the area you think of as "the drawing". The window size can be bigger (or smaller)
		 and the canvas will scale to fit it; if you resize the window this is what happens.
		@param speed: Claims to be in milliseconds, but Python ain't that fast. An integer -- This sets
		 the speed of the animation.I{ I know that animation should be in strict frames per second, but I can't figure-out how to make this
		 happen given all the other stuff (like jumps and stops) that happen in every frame. Halp!}
		@param title: What to say in the window title.
		@param hideCursor: True/False. Will, um, hide the cursor.
		"""
		Timeline.__init__( self)

		## Create a new window
		self.window = gtk.Window( gtk.WINDOW_TOPLEVEL)
		
		self.window.set_title(title)
		#self.window.set_resizable(False)

		## Here we just set a handler for delete_event that immediately
		## exits GTK.
		self.window.connect("delete_event", self._delete_event)
		
		#self.window.set_size_request(canvas_width, canvas_height)
		
		## Width of the window
		if not window_width: window_width = canvas_width
		if not window_height: window_height = canvas_height
		self.window_width = window_width
		self.window_height = window_height
		
		## Width of the canvas -- drawing area.
		self.canvas_width = canvas_width
		self.canvas_height = canvas_height

		self.speed = speed

		## I was experimenting with opening GTK Dialog boxes (say on a click of a HitThing)
		## and there was all kinds of cairo context mayhem. So this 'pauses' the app
		## by not calling _draw in the _mainLoop 
		self.pauseapp = False

		# Make the single instance of stack we use from all other modules:
		#  This also prevents GTK from accepting new fonts. I moved this instantiation
		#  into this position to give BOS.add() a chance to install fonts before
		#  the _Stack() instantiates.
		Stack.stack = Stack._Stack() # makes one stack common to all classes.

		## Put the width and height into stack
		## for access from FollowThing
		Stack.stack._windowSize = (window_width, window_height, canvas_width, canvas_height)

		self._HIDDEN_CURSOR = hideCursor
	
	def add(self, *args, **kwargs):
		"""
		Override of Timeline.add to cater for adding SceneThings to app.
		
		See B{Timeline.add}
		"""
		
		thing=args[0]
		if isinstance(thing, SceneThing):
			# We are adding a SceneThing to the app.
			# For this I keep a sep. list and we need only the thing
			# This is used in AllThings.playNextScene()
			self.scenelist.append( thing )

		Timeline.add(self, *args, **kwargs)

	def version(self): return _VERSION
	
	def pause( self ):
		"""
		This will pause the app. Go figure...
		Handy if you want to pop-up another window like a GTK dialog box.
		"""
		self.pauseapp = not(self.pauseapp)

	## Where we quit.
	def _delete_event(self, widget, event, data=None):
		Stack.stack.quitApp = True
		return False			

	def quit ( self ):
		"""
		Call this to quit your app at the next opportunity.
		"""
		Stack.stack.quitApp = True

	def getDrawingArea(self):
		"""
		If you want to do fancy stuff directly to the canvas
		this will return the gtk.DrawingArea.

		(It's simply the global instance named "Stack.stack", so you
		could use that too.)
		"""
		return Stack.stack

	def comeToLife(self):
		"""
		Start the show! After all your Things are made and added, this
		command will send the volts flowing :)

		Note:
		=====
		Always call this method B{last}. It starts a loop and nothing gets done after it.
		"""
		self.window.add(Stack.stack)
		Stack.stack._setSize()#self.width, self.height)
		Stack.stack.show()
		self.window.show_all()
		if self._HIDDEN_CURSOR: 
			Stack.stack._hideCursor()
		## left until as late as possible
		Stack.stack._connectExposeEvent(owner=self)
		gobject.timeout_add(self.speed, self._mainLoop)
		gtk.main()
		
	def mouse(self):
		"""
		An easy way to get the mouse position. It returns a tuple.

		Use
		===
		 1. x,y = app.mouse()

		"""
		return Stack.stack.mousex, Stack.stack.mousey
	
	def cursor(self,shape="ARROW",hideCursor=False):
		"""
		To change the cursor you send a string (a standard gtk name)

		Param
		=====
		 - B{hideCursor}: True/False. Use this to re-show a hidden
		 cursor (and set the shape) or simply to hide the cursor.

		 - B{shape}: One of::

			 X_CURSOR            ARROW                BASED_ARROW_DOWN
			 BASED_ARROW_UP      BOAT                 BOGOSITY
			 BOTTOM_LEFT_CORNER  BOTTOM_RIGHT_CORNER  BOTTOM_SIDE
			 BOTTOM_TEE          BOX_SPIRAL           CENTER_PTR
			 CIRCLE              CLOCK                COFFEE_MUG
			 CROSS               CROSS_REVERSE        CROSSHAIR
			 DIAMOND_CROSS       DOT                  DOTBOX
			 DOUBLE_ARROW        DRAFT_LARGE          DRAFT_SMALL
			 DRAPED_BOX          EXCHANGE             FLEUR
			 GOBBLER             GUMBY                HAND1
			 HAND2               HEART                ICON
			 IRON_CROSS          LEFT_PTR             LEFT_SIDE
			 LEFT_TEE            LEFTBUTTON           LL_ANGLE
			 LR_ANGLE            MAN                  MIDDLEBUTTON
			 MOUSE               PENCIL               PIRATE
			 PLUS                QUESTION_ARROW       RIGHT_PTR
			 RIGHT_SIDE          RIGHT_TEE            RIGHTBUTTON
			 RTL_LOGO            SAILBOAT             SB_DOWN_ARROW
			 SB_H_DOUBLE_ARROW   SB_LEFT_ARROW        SB_RIGHT_ARROW
			 SB_UP_ARROW         SB_V_DOUBLE_ARROW    SHUTTLE
			 SIZING              SPIDER               SPRAYCAN
			 STAR                TARGET               TCROSS
			 TOP_LEFT_ARROW      TOP_LEFT_CORNER      TOP_RIGHT_CORNER
			 TOP_SIDE            TOP_TEE              TREK
			 UL_ANGLE            UMBRELLA             UR_ANGLE
			 WATCH               XTERM                CURSOR_IS_PIXMAP

		The default is "ARROW".

		"""
		if hideCursor:
			Stack.stack._hideCursor()
		else:
			Stack.stack._unhideCursor(shape)

	def showGrid( self ):
		"""
		Call this to show a set of axes to get your bearings.
		"""
		Stack.stack.showaxis = True

	def export(self,paf, type="png"):
		"""
		When this is called, Things will export each frame (to the window size, not canvas size) to the given file name, with a number added.
		The number is padded into 6 zeros: 000001, 000002 etc.

		@param paf: A path and filename. If the path does not exist, there will be an error message. Make sure you create any needed directories.
		@param type: You can choose "png" or "svg". This determines what kind of file is exported per frame.
		"""
		type=type.lower()
		if type not in ["png","svg"]:
			raise _Bugs("CANT_EXPORT_TYPE", type=type)
		Stack.stack._export(paf, type)
	
	def panZoom(self, yesno=True):
		"""
		The idea is to allow you to zoom about the mouse (like Inscape does it), but I have not figured it out yet.
		For the time being, this just zooms in/out with the mouse wheel.
		"""
		Stack.stack._panZoom(yesno)

	## This gives life to the whole show.
	def _mainLoop(self): 
		"""
		Private
		=======

		Called by timeout in comeToLife. Keeps looping on timeout. This is the heart of the app.
		
		"""
		if self.pauseapp: return True
		#print "_timeout STARTS", self.speed
		self._tick()
		#print "---->>MAIN HAS TICKED<<----"
		#print
		#stack._dump() #Useful for debugging
		#Stack.stack._draw()
		if not Stack.stack.quitApp:
			Stack.stack.window.invalidate_rect(Stack.stack.allocation, True)
		#print "---->>DRAW DONE<<----"
		#print
		## Expose vs timeout mystery:
		## This is still HIGHLY problematic. I await replies from the 
		## GTK mailing list.
		## Update: It *appears* as if this function actually exits *before*
		## the expose event gets run. I just don't know how to avoid
		## this function being called again *before* the expose has finished.
		## I assume that one event finishes before another begins, but this
		## is based on no data at all and I can't get clarity on it.
		if Stack.stack.quitApp:
			gtk.main_quit()
		#print "_timeout RETURNS"
		return True # fires another timeout

	def startScene(self,start):
		"""
		Tell the app (AllThings) to start playing from a given scene.
		Besides being the way you start things going, this is nice for development
		because you can skip scenes as you need.
		"""
		if start<1:
			raise _Bugs("CANNOT_START_SCENE")
		self.scenelistindex = start-2
		self.playNextScene()

	def playNextScene( self ):
		"""
		SceneThing objects can be added to an AllThings (app) object. To get the 
		next one (they are in a sequence) to play, call this method.
		"""
		self.scenelistindex += 1
		if self.scenelistindex == len(self.scenelist):
			self.scenelistindex = 0

		S=self.scenelist[self.scenelistindex]
		S.play()

if __name__ == "__main__":
	raise _Bugs("CANT_RUN_ME")
