##	Things Copyright(C) 2009 Donn.C.Ingle
##
##	Contact: donn.ingle@gmail.com - I hope this email lasts.
##
##  This file is part of Things.
##
##  Things is free software: you can redistribute it and/or modify
##  it under the terms of the GNU General Public License as published by
##  the Free Software Foundation, either version 3 of the License, or
##  (at your option) any later version.
##
##  Things is distributed in the hope that it will be useful,
##  but WITHOUT ANY WARRANTY; without even the implied warranty of
##  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##  GNU General Public License for more details.
##
##  You should have received a copy of the GNU General Public License
##  along with Things.  If not, see <http://www.gnu.org/licenses/>.

"""
Higher-level Things like ButtonThing go here.
"""

from ThingObjects import *
from bugs import Bugs as _Bugs

class ButtonThing(HitThing):
	"""
	A ButtonThing is a simple way to create "buttons" where you have four states: 1) Normal, 
	2) Over, 3) Down and 4) Up.

	See the addStates() method for more information.
	
	HitArea
	=======
	Implement the usual drawHitarea() method to define the hit area.

	Events
	======
	You can implement these event methods:
	 1. onButtonOver()
	 2. onButtonOut()
	 3. onButtonDown()
	 4. onButtonUp()

	"""
	def __init__(self,id="ButtonThing"):
		HitThing.__init__(self,id)

		# NB: We have 4 frames to put each 'state' Thing into
		# See addStates() -- we use the parentFrame arg to do this
		# So, each frame gets a state: Normal, Over, Down & Up
		# This has the side-effect of masking the use of keys() in
		# a ButtonThing on the user-side of the API.
		# Therefore, the user should place their ButtonThing inside
		# another Thing to control the keys.

		HitThing.keys(self, "#===",Props()) #Use my super class' keys(), not my own.
		HitThing.stops(self,"^")

		class _wrapOneStateClass(Thing):
			"""This class is the two-frame solution: there, not-there.
			Each 'state' of a button is a Thing added to this class.
			We control their lifespan by moving to frame 1 or 2 of 
			this class as we need to."""
			def __init__(self):
				Thing.__init__(self,"inner_button_wrapper")
				self.keys( "#.",Props()) #This masks keys in user's ButtonThing.
				self.stops("^")

		self._wrapOneState=_wrapOneStateClass #make this visible to addStates(). Dunno why tho.

	# override to prevent use of keys() from a ButtonThing
	def keys(self,keystring,*stuff):
		"""NOTE: A ButtonThing can't have keys. Put it into another Thing in order 
		to manipulate its position etc."""
		raise _Bugs("BUTTON_THING_CANNOT_HAVE_KEYS")

	def addStates(self,statedict = None):
		"""Use, in your ButtonThing class:
		   self.addStates( {"normal":A1,"over":A2,"down":A3,"up":A4} )
		   (where A1, A2, A3 and A4 are all Things of some kind)
		   If you supply a dict, you *must* supply all those keys.
		   If not, a stock button will be made.
		"""
		if statedict is None:
			## We make our own stock states
			class StockNormal(DrawThing):
				def draw(self,ctx,fr):
					ctx.set_source_rgb(0.6,0,0)
					ctx.rectangle(-20,-10,40,20)
					ctx.fill()
			class StockOver(DrawThing):
				def draw(self,ctx,fr):
					ctx.set_source_rgb(1,0,0)
					ctx.rectangle(-20,-10,40,20)
					ctx.fill()
			class StockDown(DrawThing):
				def draw(self,ctx,fr):
					ctx.set_source_rgb(1,0,0)
					ctx.rectangle(-10,-5,20,10)
					ctx.fill()
			class StockUp(DrawThing):
				def draw(self,ctx,fr):
					ctx.set_source_rgb(1,1,0)
					ctx.rectangle(-20,-10,40,20)
					ctx.fill()
			statedict={"normal":StockNormal(),"over":StockOver(),"down":StockDown(),"up":StockUp()}
		else:
		## Check the keys in statedict
			if len(statedict) < 4:
				raise _Bugs("BUTTONTHING_MISSING_STATES")
			ld = statedict.keys()
			ld.sort()
			if ld != ["down","normal","over","up"]:
				raise _Bugs("BUTTONTHING_BAD_STATE_KEYS")

		Wnormal = self._wrapOneState()
		Wnormal.add(statedict['normal'])
		Wover = self._wrapOneState()
		Wover.add(statedict['over'] )
		Wdown = self._wrapOneState()
		Wdown.add(statedict['down'] )
		Wup = self._wrapOneState()
		Wup.add(statedict['up'])

		self.add(Wnormal,parentFrame=1)
		self.add(Wover,parentFrame=2)
		self.add(Wdown,parentFrame=3)
		self.add(Wup,parentFrame=4)

		self.stateObjects={"normal":Wnormal,"over":Wover,"down":Wdown,"up":Wup}

		self.userObjDict = self.__class__.__dict__ #What's in the user's class?

	# caller must implement: def drawHitarea(self,ctx,frame):

	def _prepState(self,state):
		pass
		self.stateObjects["normal"].goStop(2)
		self.stateObjects["down"].goStop(2)
		self.stateObjects["over"].goStop(2)
		self.stateObjects['up'].goStop(2)
		self.stateObjects[state].goStop(1)

	def _fireEvent(self,eventString):
		## Find the onX handler by name in the dict of the calling class.
		## If it's there, call it.
		if eventString in self.userObjDict:
			e = self.userObjDict[eventString]
			e(self)

	## Note: 14 April 2009
	## In order to have the onEnter() func in the user's class
	## i.e. class myButton(ButtonThing), the user would have to 
	## use super() to get *this* onEnter (in ButtonThing) to run
	## like so:
	##	def onEnter(self,x,y):
	##		print "ENTER FIRES"
	##		super(myButton,self).onEnter(x,y)	
	##
	## I reckon that's naff, so I am making new methods for 
	## a ButtonThing: onButtonOver, onButtonOut, onButtonDown, onButtonUp

	## These events make use of the events generated by HitThing
	## and called in the stack's expose loop.

	def onEnter(self,x,y):
		Stack.stack._setCursor(shape="HAND2")
		self._prepState("over")
		self.goStop(2)
		self._fireEvent("onButtonOver")

	def onLeave(self,x,y):
		Stack.stack._setCursor()
		self._prepState("normal")
		self.goStop(1)
		self._fireEvent("onButtonOut")

	def onDown(self,x,y):
		self._prepState("down")
		self.goStop(3)
		self._fireEvent("onButtonDown")

	def onRelease(self):
		self._prepState("up")
		self.goStop(4)
		self._fireEvent("onButtonUp")



class LoopThing(Thing):
	"""
	A LoopThing is a way to control the frames of a Loop that you get from your
	BagOfStuff. Normally you simply draw() an item in the bag and you can't
	set the keys at which its sub-frames appear.

	Use a LoopThing to set keys (etc.) on a loop. Each frame of your loop
	will appear once per keyframe, marching forward in time.

	Now you can tween a walk-cycle (say) from one keyframe to the next.

	Notes
	=====
	 1. You can have more keys than frames in a loop. The loop will wrap
	 per extra key it finds.
	 2. Use addLoop() to add your BagOfStuff loop, not add(). You can still add()
	 things to this Thing, but only addLoop() will control the loop. You can only use 
	 addLoop() once.

	Example
	=======
	Typical use::

	 class TweenLoopWalk(LoopThing):
	   def __init__(self):
	     LoopThing.__init__(self,"DuckWalking")
	     ## Let's assume our walkloop has three frames. Here we choose to tween it between
	     ## four keyframes, so that we get a smooth cycle.
	     self.keys  ( "#---#---#---#", Props(),Props(x=10),Props(x=10,y=10),Props(x=20,y=-20))

	     ## Now get a loop cycle from an SVG file:
	     BOOP = BOS['rsvg:walkloop'] # Where BOS is an instance of a BagOfStuff.
	     ## And add it to myself with the special method for this:
	     self.addLoop(BOOP)

	ends.

	Draw
	====
	You don't have to have a draw() method, but if you do then be B{sure} to call
	the draw() method of the LoopThing after you are done:

	Example::

	 def draw(self,ctx,fr):
	   someShape(ctx) # your own extra drawing
	   # To continue the draw, call the parent's draw method.
	   LoopThing.draw(self,ctx,fr)
	
	ends.

	"""
	def __init__(self,id="noid"):
		Thing.__init__(self,id)
		self.loopframe = 0
		self.mapping=[]
		self.BOSloop = None

	def addLoop(self, BOSloop):
		"""Adds a loop from a BagOfStuff to the LoopThing."""
		if not self.keyframelayoutstring:
			raise _Bugs("SET_KEYFRAMES_FIRST")

		LenKeys=self.keyframelayoutstring.count('#')
		LenLoop=len(BOSloop.frameKeys)

		if LenKeys < LenLoop:
			raise _Bugs("LOOPTHING_MISMATCH")
		
		li=0
		# 1    2  3      4   : li (loop index - index to list)
		# #----#--#------#
		# 1    6  9      15  : ki (key index)
		for c in self.keyframelayoutstring:
			if c=="#":
				self.mapping.append(li) 
				li += 1
			else:
				self.mapping.append(-1)
			if li == LenLoop: li=0
		#so mapping[10] gives frame within loop to draw

		# Get ref and continue call to add
		self.BOSloop = BOSloop
		BOSloop.Props(autoinc=False) #help skip inc logic in Loop.draw()
		#bounce etc. are all ignored in my draw() below
	
	def draw(self,ctx,frame):
		"""See help(LoopThing) for more info."""
		# force a sequence according to the frame of the thing.
		# this makes bounce and so forth irrelevant.
		df = self.mapping[frame-1]
		if df == -1:
			df = self.BOSloop.currFrame
		self.BOSloop.currFrame = df
		self.BOSloop.draw(ctx)



