##	Things Copyright(C) 2009 Donn.C.Ingle
##
##	Contact: donn.ingle@gmail.com - I hope this email lasts.
##
##  This file is part of Things.
##
##  Things is free software: you can redistribute it and/or modify
##  it under the terms of the GNU General Public License as published by
##  the Free Software Foundation, either version 3 of the License, or
##  (at your option) any later version.
##
##  Things is distributed in the hope that it will be useful,
##  but WITHOUT ANY WARRANTY; without even the implied warranty of
##  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##  GNU General Public License for more details.
##
##  You should have received a copy of the GNU General Public License
##  along with Things.  If not, see <http://www.gnu.org/licenses/>.

"""
	Stack
	=====

	The stack is a list of marker objects which get popped-off and drawn to the
	gtk.DrawingArea one after the other. Cairo matrices are set and alpha and clipping
	is handled according to the type of marker.

"""

import gtk
import cairo
from constants import *
from bugs import Bugs as _Bugs

class _AnEvent(object):
	"""
	Private: Holds one event in time.
	For some reason, I can't hold the 'event' arg passed into the event handlers.
	This little class gets the pertinent info and keeps it for later.
	"""
	def __init__( self, type, x, y):
		self.type = type
		self.x, self.y = x, y
		
class _Marker(object):
	"""Private: Holds a single 'state' of an object - get's pushed onto the Stack."""
	def __init__( self,type,objref=None):
		self.type = type
		self.objref = objref
		
class _Stack(gtk.DrawingArea):
	"""This object is a stack of _Marker objects. Each has a different purpose, but the 
	whole thing represents a single frame in time of all the Things to be drawn.
	The stack is also the actual gtk DrawingArea."""
	def __init__(self):
		gtk.DrawingArea.__init__( self)

		## Voodoo to create an invisible cursor.
		pix_data = """/* XPM */
 static char * invisible_xpm[] = {
 "1 1 1 1",
 "       c None",
 " "};"""
		self.invisible_color = gtk.gdk.Color()
		self.icp = gtk.gdk.pixmap_create_from_data(None, pix_data, 1, 1, 1, self.invisible_color, self.invisible_color)
		self.invisible_cursor = gtk.gdk.Cursor(self.icp, self.icp, self.invisible_color, self.invisible_color, 0, 0)

		self._HIDDEN_CURSOR = False # flag to obey when cursor is called.

		## Event voodoo
		self.add_events(gtk.gdk.BUTTON_PRESS_MASK | gtk.gdk.BUTTON_RELEASE_MASK | gtk.gdk.POINTER_MOTION_MASK)

		self.stack =[]
		self.section = 0
		
		## gtk.Widget signals
		self.connect("button_press_event", self._button_press_event_handler)
		self.connect("button_release_event", self._button_release_event_handler)
		self.connect("motion_notify_event", self._motion_notify_event_handler)
		## More GTK voodoo : unmask events
		self.add_events(gtk.gdk.BUTTON_PRESS_MASK | gtk.gdk.BUTTON_RELEASE_MASK | gtk.gdk.POINTER_MOTION_MASK)
		
		self.quitApp = False
		
		## My flags to monitor the event situation
		self.button_is_pressed_down = False
		self.button_has_just_been_released = False
		self.dragging_flag = False
		self.GLOBAL_RELEASE_FLAG = True
		
		## Put the initial mouse at some crazy unlikely position
		self.mousex = -100000
		self.mousey= -100000
		self.cx = self.cy = 100 #default size, before _setSize is run.
		self.scalex, self.scaley = 1,1

		self.mex = self.mey = 0 # Actual mousex, mousey

		self.showaxis=False # A debug thing
		
		self.EventOwner = None
		self.latestEvent = None
		self.EventBlocked = False
		
		self.hit = False
	   
		self._windowSize = () # Filled-in from AllThings.__init__

		self._GTKLOADED = False

		self._scrolled = 1

		self._panzooming = False
		self._centerPage = self._centerPageAbsolute

		self._exporting = False
		self._exportType = None
		self._exportPAF = ""
		self._outpic = 1

	def _connectExposeEvent(self,owner):
		"""Called from AllThings.comeToLife"""
		self.owner = owner # April 2009: owner is the AllThings instance.
		self.connect("expose_event", self._expose)
		self.connect('size-allocate', self._resize)

		if self._panzooming:
			self.connect("scroll-event", self._scroll)

	def _resize(self, widget, allocation):
		"""A signal handler for the resize event. It took a while to get this figured out..."""
		self._calcSize( allocation.width, allocation.height )

	def _button_press_event_handler(self,widget,event):
		## GDK_BUTTON_PRESS
		e = _AnEvent(event.type, event.x, event.y)
		self.latestEvent = e
		self.GLOBAL_RELEASE_FLAG = False
		return True
		
	def _button_release_event_handler(self,widget,event):
		## GDK_BUTTON_RELEASE
		e = _AnEvent(event.type, event.x, event.y)
		self.latestEvent = e
		self.GLOBAL_RELEASE_FLAG = True
		return True # seems to kill the event.
		
	def _motion_notify_event_handler(self,widget,event):
		## GDK_MOTION_NOTIFY
		e = _AnEvent(event.type, event.x, event.y)
		self.latestEvent = e
		##Just store them in self for app.getMousePos()
		self.mex,self.mey = event.x, event.y
		self.mousex = event.x - self.cx
		self.mousey = event.y - self.cy
		return True

	def _calcSize(self, w, h):
		"""Private: Called from _resize. This calcs vitalstatistix :D"""
		canw,canh = self._windowSize[2],self._windowSize[3] # Get the "canvas" size.

		self.scalex = float(w)/float(canw)
		self.scaley = float(h)/float(canh)

		self.cx = w/2
		self.cy = h/2

	def _setSize(self):
		"""
		Private: Called from AllThings class to set my initial window size.
		"""
		w,h=self._windowSize[0], self._windowSize[1]
		self._calcSize( w,h)
		self.set_size_request(w, h) # set the window

	def _push(self, object, MARK):
		"""Private: push a _Marker object onto the stack."""
		m = _Marker(MARK, object)
		self.stack.append(m)
		
	def _dump(self):
		print "---------------STACK DUMP------------------"
		for m in self.stack:
			print m.type, " : ", m.objref.id
		print "-------------------------------------------"
	
	def _export( self, paf, type ):
		"""
		Private: Called from AllThings to set export PNG flags.
		"""
		self._exporting = True
		self._exportType = type
		self._exportPAF = "%s%s.%s" % (paf,"%s",type)

	def _panZoom(self,yesno):
		"""
		Private: Called from AllThings to set panzoom. yesno determines which function gets called under the ref _centerPage.
		"""
		if yesno:
			self._centerPage = self._centerPageWithZoom
		else:
			self._centerPage = self._centerPageAbsolute
		self._panzooming = yesno

	def _scroll(self,widget, event):
		"""Private: Signal handler for mouse wheel."""
		if event.direction == gtk.gdk.SCROLL_UP:
			self._scrolled += 0.1
		else:
			self._scrolled -= 0.1

	def _doMatrix(self, frameProps):
		"""
		Private: Do the matrix stuff that moves, rotates and scales things.
		"""
		ThingMatrix = cairo.Matrix(1, 0, 0, 1, 0, 0)

		## Next, move the drawing to it's x,y
		cairo.Matrix.translate(ThingMatrix, frameProps.x, frameProps.y)
		# REMOVE THIS and mouse tracks page :: self.context.transform(ThingMatrix) # Changes the context to reflect that

		if frameProps.rot !=0:
			if frameProps.rx != 0 or frameProps.ry != 0:
				rx,ry=frameProps.rx,frameProps.ry
				cairo.Matrix.translate(ThingMatrix, rx, ry) # move it all to point of rotation

			cairo.Matrix.rotate(ThingMatrix, frameProps.rot) # Do the rotation
			
			if frameProps.rx != 0 or frameProps.ry != 0:
				cairo.Matrix.translate(ThingMatrix, -rx, -ry) # move it back again

		if frameProps.sx != 1 or frameProps.sy != 1:
			cairo.Matrix.scale(ThingMatrix, frameProps.sx, frameProps.sy) # Now scale it all

		## Only ONE transform command:
		self.context.transform(ThingMatrix) # and commit it to the context

	def _centerPageAbsolute(self):
		"""
		Private::

		 Shift the origin of the page to the center
		  -y | -y
		  -x | +x
		 ----0------
		  -x | +x
		  +y | +y

		.  
		"""
		matrix = cairo.Matrix(1, 0, 0, 1, self.cx, self.cy)
		cairo.Matrix.scale(matrix,self.scalex,self.scaley) #Scale to the window size.
		self.context.transform(matrix)

	def _centerPageWithZoom(self):
		"""
		Same as _centerPage, but does it where the mouse is. (Actually it doesn't, but that's the desired effect.)
		"""	
		matrix = cairo.Matrix(self._scrolled, 0, 0, self._scrolled, self.cx, self.cy)
		cairo.Matrix.scale(matrix,self.scalex,self.scaley) #Scale to the window size.
		self.context.transform(matrix)	

	def _drawAxis(self):
		"""Private: Draws axis so one can see what's where."""
		c = self.context
		m = 0
		for x in range(10,210,10):
			m += 1
			w = 10 +(m % 2 * 10)
			if x == 100: w = 50
			c.rectangle(x,-w/2,1,w)
			c.rectangle(-x, -w/2, 1, w)
			c.rectangle(-w/2, x, w, 1)
			c.rectangle(-w/2, -x , w, 1)
			
		c.set_source_rgb(0,0,0)
		c.fill()
		
	## GTK+ expose event :- Does all the actual drawing.
	def _expose(self, widget, event):
		"""Private: Expose event handler: Go throught the Stack and call all drawing methods.
		It also handles mouse (and later other) events.
		"""
		if self.owner.pauseapp: return True
		#print "**EXPOSE RUNS**"

		## If we are exporting, then we make an alpha image surface and get a context on it:
		if self._exporting:
			paf = self._exportPAF % str(self._outpic).zfill(6)
			canw,canh = self.allocation.width, self.allocation.height
			if self._exportType == "png":
				surf = cairo.ImageSurface(cairo.FORMAT_ARGB32, canw, canh)
			else:
				## SVG differs from PNG in that the save is kind of done here too.
				try:
					surf = cairo.SVGSurface(paf, canw, canh)
				except:
					raise _Bugs("CANT_WRITE_FILE", file=paf )
			self.context = cairo.Context( surf )
		else:
			## else, we just go directly to the window:
			self.context = widget.window.cairo_create()
		
		##
		## Go thru my stack and test the markers - this determines what gets drawn and how.
		## There's a lot of matrix stuff and save/restoring going on.
		##
		#self.context.identity_matrix () # VITAL LINE :: I don't know why, surely the context is new?
		self._centerPage()
		
			
		e = self.latestEvent # Fetch ONE definitive event to process.
		if e:
			usersp_mouse_x,usersp_mouse_y = self.context.device_to_user(e.x,e.y)

		self.context.save()

		while len(self.stack) > 0:
			mark = self.stack.pop(0) #Fetch from the bottom of the stack, in order of pushing.
		
			thing = mark.objref
			tf = thing.currentFrame
			frameProps = tf.props

			if mark.type == THINGSTART:
				
				myInstance = thing.dirstore
				## Discovered that thing.__class__.__dict__				
				## only supplies info for the immediate class
				## of thing (i.e what's used in a thingum)
				## If I need deeper introspection (see ButtonThing)
				## I am foobared.
				## So, my BFG is to store a dir() in the init.
				
				
				## 28 April 2009
				## Experimenting with a 'global Prop' for a Thing
				self.context.save()
				self._doMatrix(thing.globalProps)

				self.context.save() #Start a new bubble
				## If this is to be a clip, then it needs special attention ( in the else )
				if not thing._isClip:
					## START A NEW MATRIX
					self._doMatrix(frameProps)
					## From here on down, we are "under the influence" of this matrix.
		
					## If there is a 'tint' value, then we begin the double-draw trick
					if frameProps.tint != 0:
						thing.START_TINT = True
						self.context.push_group() # Save all the drawing() and so forth in a group.

					## Handle overall alpha-ing of the Thing
					if frameProps.a != 1: #Try to speed it up by only doing this if there is some alpha.
						self.context.push_group()
						thing.START_ALPHA = True

					## A chance to change stuff as the frame starts, before the draw
					if "preDraw" in myInstance:
						thing.preDraw ( tf.frame )
						
					if "draw" in myInstance:
						thing.draw(self.context, tf.frame) #get actual frame number from the frameProps
						
				else:
					## It's a CLIP:
						self.context.save()
						self._doMatrix(frameProps)
						## A chance to change stuff as the frame starts, before the draw
						if "preDraw" in myInstance:
							thing.preDraw ( tf.frame )
						thing.draw(self.context, tf.frame)
						self.context.restore()
						self.context.save()
						## Clip Start must be done after the draw
						self.context.clip()		 
					
				

				## If there is an event:
				if self.latestEvent:	

					## I decided to do this because events fire so quickly that
					## there can be changes to the state of the variables
					## DURING a vital test to them. 
					## The problem this solution has is that events(mostly down and release)
					## get missed if the user clicks too quickly...
					## And I don't know what will happen if the app runs too slowly overall...

					e = self.latestEvent # Fetch ONE definitive event to process.

					if "drawHitarea" in myInstance: #only HitThings get this done.	   
						## APRIL 2009: Changed this to use device_to_user instead.
						## Now the x,y is RELATIVE to the THING. i.e. x,y is "inside" the
						## thing, no matter it's rotation etc.
						## The coords are NOT those of the top window.
						##
						##  Note 2 : Even this picture is not complete. There are strange jumps etc.

						usersp_within_thing_x,usersp_within_thing_y = self.context.device_to_user(e.x,e.y)	

						##(Trying) to avoid drawing the hit and testing, *when* we already know about it.
						if not thing.is_being_dragged:
							## Draw it UNDER THE INFLUENCE of the ThingMatrix
							self.context.save()
							thing.drawHitarea(self.context, tf.frame) # Draw the path of the hit area.(invisible)
							self.hit = self.context.in_fill( usersp_within_thing_x, usersp_within_thing_y ) # Cairo's own hit detector.
							self.context.new_path() # "destroy" the appended path.
							self.context.restore()
						
						# decode the event
						if e.type == gtk.gdk.BUTTON_PRESS:
							self.button_is_pressed_down = True
							self.button_has_just_been_released = False
							
						elif e.type == gtk.gdk.BUTTON_RELEASE:
							self.button_has_just_been_released = True
							self.button_is_pressed_down = False
							self.dragging = False
							
						if e.type == gtk.gdk.MOTION_NOTIFY:
							if self.button_is_pressed_down: 
								self.dragging = True
							else:
								self.dragging = False							
						# Now the hard work
						
						## Decide which thing gets the focus of events:
						if self.button_is_pressed_down and self.hit and not self.EventBlocked:
							#print "Locking event to %s" % thing.id
							self.EventOwner = thing
							self.EventBlocked = True # one to a customer.
							thing.usersp_init_x, thing.usersp_init_y = usersp_within_thing_x, usersp_within_thing_y # record first coordinate, includes translation

						## Rollover is basic, but don't do it to the thing being dragged.
						##(however, do allow other things to get rolled over)
						if not thing.is_being_dragged:
							if self.EventOwner != thing:
								thing._handleRollover(self.hit, usersp_within_thing_x, usersp_within_thing_y)
								
						## One thing(and no other in the loop) is the focus of events
						if self.EventOwner == thing:
							if not self.dragging:
								#print "not drag"
								thing._handleMouseButtons( usersp_within_thing_x, usersp_within_thing_y )
							else: # we ARE dragging
								## April 2009
								##  I have failed to get dragging to work on hitThings inside other Things
								##  for now the rule is: only drag stuff that's on the root.

								## SUNDAY : 19 April
								##  Works perfectly when hitthing is dragging itself on ROOT (i.e. NOT inside another thing)
								dx,dy=usersp_mouse_x, usersp_mouse_y
								dx,dy=dx - thing.usersp_init_x, dy - thing.usersp_init_y
								thing._handleDragging(self.hit, dx, dy, usersp_within_thing_x, usersp_within_thing_y )
								
						## When button is released, it's time to see who was the owner and unlock them
						if self.button_has_just_been_released and self.EventBlocked and self.EventOwner == thing:
							self.EventOwner = None
							self.EventBlocked = False
							self.hit = False

					## This approach to events has solved many problems, but added some more.
					## I am finding that quick clicks will "lock" into a thing and they won't
					## register any more mouse releases.
					##
					## Going to try forcing the issue:
					## Hard as heck to debug. I expect "issues" :|				  
					if self.GLOBAL_RELEASE_FLAG:
						thing.fireEventOnce = False

					
				
				if self.showaxis: self._drawAxis()
				
				## Is there a function to call?
				if thing.currentFrame.func:
					#print thing.currentFrame.func.funcref, thing.currentFrame.func.args
					f = thing.currentFrame.func.funcref
					args = thing.currentFrame.func.args
					if args:
						f (*args)
					else:
						f ( )

				
			## Many Things may be drawn under the clip, and it only ends on a marker.
			## This is why it's not in the THINGSTART test above.
			elif  mark.type == MASKEND:
				self.context.restore()					   
				
			elif mark.type == THINGEND:
				## clipped things don't need alpha
				#thing = mark.objref
				if not thing._isClip:
					## Draw all those shapes from cairo's own "group" stack with
					## this Thing's alpha. ONLY IF we started an ALPHA at all.
					##  This is because we may goPlay and this THINGEND has a
					##  different alpha from the THINGSTART.
					if thing.START_ALPHA:
						self.context.pop_group_to_source()
						self.context.paint_with_alpha( frameProps.a )
						thing.START_ALPHA = False

					## If there was tinting, then let's do the double-draw trick:
					if thing.START_TINT:
						p = self.context.pop_group() # We get all that was drawn, including alpha above.
						self.context.set_source(p) # That becomes the source
						self.context.paint() # plonk it all down -- full colour, no tinting. The first 'draw'
						self.context.set_source_rgba(*frameProps.tintlist() ) # Now make a colour of my tint
						self.context.mask(p) # And squeeze that colour *through* my pattern (p) -- this is the second 'draw'.
						# As the 'tint' (really alpha) value gets closer to 0, so the second draw gets more see-through.
						# the first drawing is showing through more and more -- thus appearing to tint.
						thing.START_TINT = False

				## Restore the context so it's ready for the next Thing
				self.context.restore()
			
				##Restore after global Prop settings
				self.context.restore()

		self.context.restore()
		del(self.stack[:])
		self.stack =[] #Clear the stack, I hope.
		self.latestEvent = None

		## If we are exporting, then let's use that surface to draw the window
		if self._exporting:
			# Paint to the window
			window_ctx = widget.window.cairo_create()
			p = cairo.SurfacePattern( surf )
			window_ctx.set_source( p )
			window_ctx.paint()
	
			if self._exportType == "png":
				# Now write the PNG file
				try:
					surf.write_to_png( paf )
				except:
					raise _Bugs("CANT_WRITE_FILE", file=paf )

			self._outpic += 1

		#print "**EXPOSE ENDS**"
		return True
	
	def _setCursor(self,shape="ARROW"):
		if self._HIDDEN_CURSOR: return
		s="gtk.gdk." + shape
		c=gtk.gdk.Cursor(eval(s))
		self.window.set_cursor(c)

	def _hideCursor(self):
 		self.window.set_cursor(self.invisible_cursor)
		self._HIDDEN_CURSOR = True

	def _unhideCursor(self, shape="ARROW"):
		self._HIDDEN_CURSOR = False
 		self._setCursor( shape )


