##	Things Copyright(C) 2009 Donn.C.Ingle
##
##	Contact: donn.ingle@gmail.com - I hope this email lasts.
##
##  This file is part of Things.
##
##  Things is free software: you can redistribute it and/or modify
##  it under the terms of the GNU General Public License as published by
##  the Free Software Foundation, either version 3 of the License, or
##  (at your option) any later version.
##
##  Things is distributed in the hope that it will be useful,
##  but WITHOUT ANY WARRANTY; without even the implied warranty of
##  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##  GNU General Public License for more details.
##
##  You should have received a copy of the GNU General Public License
##  along with Things.  If not, see <http://www.gnu.org/licenses/>.

# Bag Of Stuff : 30 March 2009


# NOTE: 7 April 2009

# By looking at the gnome-python-desktop.rsvg sourcecode I found
# that render_cairo() can take an id="#blah" param! So, this code is
# going in another direction for a while.

"""
This is a Bag Of Stuff -- a place to get your graphics out of.
"""

from xml.dom import minidom 
from bugs import Bugs as _Bugs
import os
import atexit

import gtk
import rsvg
import pango

import svg_to_pycairo

class BagOfStuff(dict):
	"""
	Loads and holds stuff: Images, Fonts and SVG files.

	Example
	=======
	bos = BagOfStuff(); bos.add("apic.jpg", "logo")

	Then you can use bos["logo"].pixbuf to get the pixel
	buffer for Python-Cairo.
	
	Fonts note
	==========
	Make one bag in your app, before you create your main
	AllThings object. This way it can install your fonts 
	before GTK starts. Once GTK starts, there's no way to
	add new fonts to your app.
	  
	Use
	===
	bos = BagOfStuff()
	
	bos.add('some file','some_key')
	
	Once you have an instance, you can extract/use stuff:
	 1. Bitmaps: bos["key"].pixbuf - supplies a pixmap.
	 2. Vectors: bos["key"].draw(context)
	"""
	imagetypes =["png", "jpg", "jpeg" ]
	fonttypes =["ttf", "otf" ]
	vectortypes =["svg" ]
	def __init__( self):
		dict.__init__( self)
		self._SVGHANDLES = {} # Each svg file gets ONE handle. rsvg draws by #id within a handle.

	def __getitem__( self, key ):
		return dict.__getitem__( self, key)

	def add(self, paf, key):
		"""
		Add resources to your "bag". This is a simple way to get SVG (Inkscape) files and images
		and fonts into your animation; and then refer to them by keys.

		Use 
		===
		bos = BagOfStuff()
		
		bos.add(filename, key)

		This adds stuff to your bag of ..well.. stuff -- think of it as a repository of elements for
		your scenes and animations.

		Each item in a bag has a variety of methods, like B{bos['alien'].draw()}. See the classes
		Sprite, Loop, Path, Mask as well as  .. .. ..

		SVG files
		=========
			NB: B{Within your SVG file, ensure the main elements have their own UNIQUE ids.}

			Keys
			----
			Each id (in the SVG file) will be joined to the key (given to the add method) to form the master key for this bag.

			Example: B{bos.add('some.svg','walking')}. Thereafter, stuff within the bag will be keyed as 'walking:funnywalk', 'walking:run'
			assuming you have ids within the SVG called 'funnywalk' and 'run'.

			You set ids on objects in Inkscape by using the Object Properties dialog.

			LAYOUT OF YOUR SVG FILES
			------------------------
				This is quite important to get right. Use only these layer names and rules:

				sprites
				~~~~~~~
				A 'sprite' is a GROUP of shapes. Use as many sprites on this layer as you need. Each a B{unique} group B{with and id.}
				
				Clones: Create and use freely within each sprite. Just watch complexity because that slows things down.

				Groups: You can use sub-groups within a sprite. They do not need ids.
				
				All other Inkscape tools can be used -- effects like blur may not work.

				loops
				~~~~~
				A 'loop' is a SUB-LAYER of GROUPS. The idea is that you would draw animations like walk-cycles,
				or explosions -- frame-by-frame stuff -- in a "loop". Given a sub-layer called "walking", you would have individual groups
				I{within} that called "walk1", "walk2" and so on.
		          
				Each sub group must be given an incremental id: walk1, walk2, walk3 etc. These will be pulled-out and keyed by that id.
				All other rules the same as for sprites.

				masks
				~~~~~
				A 'mask' is a simple, B{closed} path that can be used for clipping or hit-testing.
				Do NOT group these vectors. Draw a single path (and convert to curves) and give it an id.
		
				paths
				~~~~~
				A 'path' is a simple open path (convert to curves). Do NOT group it.
				Paths are used for making Things follow along them. (You could also simply draw them to the screen.)

			SUB-LAYERS
			----------
			B{Important.}

			On each MAIN LAYER, you *must* add SUB LAYERS (use Inkscape's layer dialog) and keep each
			of your objects on their OWN LAYER. 

			B{For example:}
			In your 'sprites' layer, you have "redaliens" and "bluealiens" layers under it.
			
			In the "redalien" sub-layer, you have a group with id="redalien1".  You can also add other sprites, 
			let's add id="boss_monster".
			
			In your "bluealiens" sub-layer (hide the redaliens one) and draw a group with id="bluebug".

			Now you have::

			  sprites
			   |____redaliens
			   |       |_______redalien1
			   |       |
			   |       |_______boss_monster
			   |
			   |____bluealiens
			           |_______bluebug

			  .
			
			This has several advantages, not least of which is that you can draw many things in one svg file and use the visibility tool to focus on one thing at a time.

			HIDING THINGS
			-------------
			Suppose you use a rectangle as a means to guide your drawing, but don't want it in the final animation: In Inkscape, under the object properties dialog, give the rectangle a Label of B{hidden} and it will be.

		Adding images
		=============
		You can load png, jpg and jpeg files. 
		
		Use: bos.add('path/to/somepic.png','my_big_ugly_logo')

		See class AnImage for more info.

		Adding fonts
		============
		At the moment you can only use ttf files. 
		
		Use: bos.add('path/to/somefont.ttf','key')

		In the case of fonts, the key is not used, but you must pass it. You use fonts via pango, and that takes a family name.

		Adding sounds
		=============
		I suggest that you practice using lips, tongue and bits of shrubbery to make the sounds you expect to hear. Try to time this with the animation. This will have to do until I get a clue about how-to include sound, doo-bee-doo.
		
		The add() method
		================
		@param paf: Path and filename. The path can be relative; it's convenient to have a directory of data in the same place that your Python code is
		 running; if it were called 'repo' then your path would be 'repo/something.???' pointing to a picture, an svg, or whatever.
		@param key: A unique key for accessing the stuff you are adding to the bag. If you repeat keynames, complaints will not be scarce.
		"""

		if key in self:
			raise _Bugs('BAG_ERROR', key = key)
		if key in self._SVGHANDLES:
			raise _Bugs('BAG_ERROR',key = key)

		print "Preparing resource..."

		skip = False

		ext = paf[paf.rfind(".") + 1:].lower()


		if ext in BagOfStuff.imagetypes:
			print "  Loading image..."
			obj = AnImage( paf )

		elif ext in BagOfStuff.vectortypes:
			# pass self because that routine will stuff the dict
			# with *many* entries.
			print " Opening and munching SVG file."
			try:
				factory = _VECTORfactory( bag=self, paf=paf, nskey=key)#, self.HANDLE)
			except:
				raise

			self._SVGHANDLES[key]=rsvg.Handle(data=factory.final_xml)
			#print factory.final_xml
			del(factory)
			skip = True # bail out! because we don't add this key in the normal way.

		elif  ext in BagOfStuff.fonttypes:
			print "  Installing font..."
			obj = AFont( paf )
		
		else:
			raise _Bugs("UNKNOWN_TYPE") 
			
		## Plonk it into the dict.
		if not skip: self[key]= obj
		print "Done."

class AnImage(object):
	"""
	Holds a pixbuffer, width and height of a loaded image. Access this via a Bag Of Stuff.

	Example
	=======
	bos['somekey'].pixbuf

	@ivar width,height: Of the image
	@ivar pixbuf: The actual image data.
	"""
	def __init__( self, paf ):
		self.pixbuf = None
		self.w = 0
		self.h = 0
		self.paf = paf
		try:
			input = open(self.paf)
		except:
			raise _Bugs("FILE_ERROR", paf = self.paf)
			return False
		## From a recipe in the Cairo cookbook.
		imagebuf = input.read()
		pixbufloader = gtk.gdk.PixbufLoader()
		pixbufloader.write(imagebuf)
		pixbufloader.close()
		self.pixbuf = pixbufloader.get_pixbuf()
		self.w = self.pixbuf.get_width()
		self.h = self.pixbuf.get_height()
		input.close()
	def draw(self,ctx,x,y):
		"""
		Alternative: Use Cairo to draw a pixbuf: ctx.set_source_pixbuf(pixbuf,x,y)

		@param ctx: Your cairo context
		@param x,y: Lo, go unto and draw.
		"""
		ctx.set_source_pixbuf(self.pixbuf,x,y)
		ctx.paint()

class AFont(object):
	"""
	Makes a font, from a file, temporarily available to the GTK system. This fonts is placed in the 
	user's .fonts directory and removed when the app closes.
	
	TIP
	===
	To find the 'family name' of Comic.ttf from the command-line, try this: B{fc-list : family file | grep Comic.ttf}

	Note: The font needs to be installed first. This is truly a PITA. Try using Fonty Python to help you.
	"""
	def __init__( self, paf ):
		self.linkDestination = ""
		self.linkSource = ""
		self.paf = paf
		self.dontRemoveFont = False
		
		## Try to install this font into ~/.fonts 
		self.linkSource = os.path.realpath ( self.paf ) 
		filename = self.paf[self.paf.rfind("/")+1:]
		HOME = os.environ['HOME']
		self.linkDestination = os.path.join( HOME, ".fonts", filename ) 
		fontsFolder = os.path.join( HOME, ".fonts" )
		if not os.path.exists( fontsFolder ):
			## The user has no .fonts folder...
			## This is highly bizarre.
			## We gotta make it.
			try:
				os.mkdir( fontsFolder )
			except:
				raise _Bugs("CANNOT_MAKE_FONTS_FOLDER")
		#Link it if it ain't already there.
		if not os.path.exists( self.linkDestination ):  
			if os.path.exists( self.paf ):
				os.symlink ( self.linkSource, self.linkDestination )  #Should do the trick.
			else:
				## Cant find the font
				raise _Bugs("CANNOT_FIND_FONT", paf = paf)
		else:
			## That font, by chance,is already in .fonts
			self.dontRemoveFont = True # We don't want to remove it on close.

		atexit.register(self._cleanup)
			
	def _cleanup( self ):
		## remove the installed font -- if it's not supposed to stay there
		if os.path.exists( self.linkDestination ) and not self.dontRemoveFont:  
			os.unlink( self.linkDestination )
			print "Font uninstalled."

class _Vector(object):
	"""Private: Base class for other vector types:
	Sprite, Loop, Mask and Path
	"""
	def __init__(self, id, aNode, bag, nskey):#,HANDLE):
		self.id = "#" + id
		self.bag = bag
		self.nskey = nskey #key of the entire SVG file (bos.add)
		#print "SPRITE MADE:",self.nskey
		
		self.cairocommands = None

		self._transx, self._transy = 0,0

		self._doXML( aNode )
	
	def _resetTransform(self, element ):
		element.setAttribute('transform','translate(0.0,0.0)')


	def _AssignPyCairoCommands(self, aNode):
		self.cairocommands = svg_to_pycairo.parsePathElement(aNode)

	def _drawPathsOnly(self, ctx):
		# ctx comes into this namespace, it's used 
		# verbatim in the pycairo exec strings.
		exec( self.cairocommands['paths'] )

	def _drawPathAndStyle(self):
		"""Not used yet."""
		ctx = self.ctx # namespace this var, it's used in exec strings.
		for p in self.cairocommands: # this implies many commands, not used as of 6 april '09
			exec( p['paths']  )
			exec( p['styles'] )
		
class Loop(_Vector):
	"""
	A series of GROUPS from an SVG file that will be displayed as frames.
	
	First add the SVG file to your Bag Of Stuff. see help(BagOfStuff.add)
	B{Don't instantiate a Loop directly}, work via BagOfStuff.

	Use
	===
	bos["walking:funny"].Props(bounce=False)

	bos["walking:funny"].draw(ctx)

	See Props() method.
	"""
	def __init__(self,id, loops, bag, nskey):
		self.frameKeys = []
		
		self._bouncedir = +1
		self.Props() # set defaults.

		_Vector.__init__(self,id, loops, bag, nskey)

	def Props(self, currentFrame=0, bounce=True, autoinc=True, stretch=1):
		"""
		Sets Properties of the Loop. Call this after you have added an 
		SVG (with loops) to a BagOfStuff.

		Use example
		===========
		bos['walking:run_cycle'].Props( bounce=False )

		@param currentFrame: Set the frame number (integer) to show/start from.
		@param bounce: True is 1,2,3,2,1 and False is 1,2,3,1,2,3
		@param autoinc: False means it simply stays on frame currentFrame.
		@param stretch: An Integer. The frame will repeat this many times
		 before going on to the next one. Use it to stretch-out an animation that is too fast.
		 Be sure to leave enough room in the keys (use "=" same-frame keys) of the parent for the
		 animation to fit. If it's 3 frames and stretch=2, leave 6 frames.

		 B{Seriously:} See the LoopThing in the Thinglets module. It's a better way to do this.
		"""
		self.currFrame = currentFrame
		self.bounce = bounce
		self.autoinc = autoinc
		self.stretch = stretch
		self._repeatcount = 0
	def _doXML(self, loops ):
		# Go thru the chunk - It is a bunch of <g> tags
		for loopFrame in loops.childNodes:
			if loopFrame.nodeType != loopFrame.ELEMENT_NODE: continue
			if loopFrame.nodeName=="g":
				# At this point we have a <g> that is a FRAME
				# of the loop. It's got an id like "walk1"

				# reset transform attrib to 0,0
				# to keep all objects at same position
				# so that frames overlap.
				self._resetTransform( loopFrame )

				# Make a #id and keep it in my internal list.
				frameid = "#" + loopFrame.getAttribute('id')
				self.frameKeys.append(frameid)
		self.frameKeys.sort() # Get them predictable.
		#print "KEYS:",self.frameKeys
		
	def draw(self, ctx):
		"""
		draw() takes only a context.
		"""
		# Loop.draw() overrides the generic draw() so we have to
		# re-do some stuff.
		self.ctx = ctx
		
		# Deal with sub-frames.
		if self.autoinc:
			if self.stretch > 1:
				self._repeatcount += 1
				if self._repeatcount > self.stretch:
					self.currFrame += self._bouncedir
					self._repeatcount = 0

			if self.currFrame < 0:
				self.currFrame = 1
				self._bouncedir = +1
			if self.currFrame == len(self.frameKeys):
				if not self.bounce:
					self.currFrame = 0
					self._bouncedir = +1
				else:
					self.currFrame -= 2
					self._bouncedir = -1

			fr = self.frameKeys[self.currFrame]
		else:
			fr = self.frameKeys[self.currFrame]

		self.bag._SVGHANDLES[self.nskey].render_cairo(ctx,id=fr)
		#print self.bag._SVGHANDLES[self.nskey].get_dimension_data()

class Sprite(_Vector):
	"""
	A GROUP from an SVG file that will be displayed as a single thing.
	First add the SVG file to your Bag Of Stuff. see help(BagOfStuff.add)
	B{Don't instantiate directly.}

	Use
	===
	bos["scifi:robot"].draw(ctx)
	"""
	def _doXML(self, sprite):
		self._resetTransform( sprite )
	def draw(self, ctx):
		self.bag._SVGHANDLES[self.nskey].render_cairo(ctx,id=self.id)

class Mask(_Vector):
	"""
	A PATH from an SVG file that will be drawn by pyCairo commands.
	First add the SVG file to your Bag Of Stuff. see help(BagOfStuff.add)
	B{Don't instantiate directly.}

	Use
	===
	bos["scifi:robot_mask"].draw(ctx)

	Note
	====
	draw() does not stroke or fill the object.
	"""
	def draw(self,ctx):
		self._drawPathsOnly(ctx)
	def _doXML(self, mask):
		self._AssignPyCairoCommands( mask )

class Path(_Vector):
	"""
	A PATH from an SVG file that will be drawn by pyCairo commands.
	First add the SVG file to your Bag Of Stuff. see help(BagOfStuff.add)

	There is no ostensible difference between a 'mask' and a 'path'. The
	two are conceptually split according to end-use. However, I have not 
	written the path-following code yet, so this is all moot.

	B{Don't instantiate directly.}

	Use
	===
	bos["scifi:planet_orbit"].draw(ctx)
	
	Note
	====
	draw() does not stroke or fill the object.
	"""
	def draw(self,ctx):
		self._drawPathsOnly(ctx)
	def _doXML(self, path):
		self._AssignPyCairoCommands( path )


class _VECTORfactory():
	"""Private: This is the main XML engine that parses SVG.
	It looks for particular signatures and chops-up the SVG into various
	bits that go on to become Sprites, Loops etc.
	"""
	def __init__(self, bag, paf, nskey):
		try:
			xmldoc = minidom.parse(paf)
		except:
			raise _Bugs("SVG_FILE_ERROR",paf=paf)

		# Get the main svg element
		svg=xmldoc.getElementsByTagName('svg')[0]

		self._removeHiddenElements( svg )
		
		def isLayer( node ):
			"""
			Detect if a node is an inkscape:groupmode="layer" kind."""
			if not node.attributes: return False #It's 100% weird.
			if "inkscape:groupmode" not in node.attributes.keys(): 
				return False #It has no signal for a potential layer
			if node.attributes.get("inkscape:groupmode").nodeValue != "layer": 
				#Could also say if node.getAttribute('inkscape:groupmode') != "layer":
				return False #It's not layer.
			return True # Okay, it's a layer

		# A dict to hold Classes and flags
		constructors = {
				"loops":None,
				"sprites":Sprite,
				"masks":Mask,
				"paths":Path
			}

		# Now we walk the xml
		# We EXPECT:
		# g (MAIN LAYER : Determines Class to use)
		# |_g (SUB LAYER)
		#    |_ELEMENT 
		#    |_ELEMENT

		for layer in svg.childNodes:
			# We are expecting the MAIN LAYER
			if isLayer(layer):
				# What kind of layer is it?
				MAIN_LAYER_NAME=layer.getAttribute("inkscape:label").lower() #sprites, masks, loops or paths

				if MAIN_LAYER_NAME in constructors:
					#xmldoc.unlink()
					#raise _Bugs("BAD_SVG_FILE",paf=paf)
					for sublayer in layer.childNodes:
						if isLayer(sublayer):
							SUB_LAYER_NAME=sublayer.getAttribute("inkscape:label")
							if MAIN_LAYER_NAME == "loops":
								LOOP = Loop(SUB_LAYER_NAME,sublayer,bag,nskey)
								bagkey = nskey + ":" + SUB_LAYER_NAME
								if bagkey in bag:
									xmldoc.unlink()
									raise _Bugs("BAG_ERROR", bagkey)
								bag[bagkey] = LOOP
							else:
								for c in sublayer.childNodes:
									if c.nodeType != c.ELEMENT_NODE: continue
									
									# These c elements are either <g> or <path>
									# <g> when it's a sprite
									# <path> when it's a mask or path
									
									id = c.getAttribute('id')

									bagkey = nskey + ":" + id
									if bagkey in bag:
										xmldoc.unlink() 
										raise _Bugs("BAG_ERROR",bagkey)

									# Get the class and intantiate it, pass args:
									KLASS = constructors[MAIN_LAYER_NAME]
									bag[bagkey] = KLASS(id ,c, bag, nskey)
									# That runs _doXML() on each object
									# and then returns here for final unlink

		self.final_xml = svg.toxml() # stored for bos.add
		# Kill the darn thing -- dead.
		xmldoc.unlink()

	def _hasLabelHidden(self, element ):
		if element.hasAttribute('inkscape:label'):
			if element.getAttribute('inkscape:label')=="hidden":
				return True #it's hidden
		return False # It is NOT hidden

	def _removeHiddenElements(self,element):
		"""Private: Heavy-handed sweeping approach to finding
		some inkscape:label="hidden" elements and chopping them out."""
		l = element.getElementsByTagName('g')
		l.extend(element.getElementsByTagName('use'))
		l.extend(element.getElementsByTagName('path'))
		l.extend(element.getElementsByTagName('rect'))
		for e in l:
			if self._hasLabelHidden( e ):
				e.parentNode.removeChild(e)

#def getTransform( element ):
#	import re
#	x,y = 0,0
#	e=re.compile(r"\((.*),(.*)\)")
#	if element.hasAttribute('transform'):
#		t = element.getAttribute('transform')
#		# t is "transform(x,y)"
#		s = e.split(t)
#		x = 1-float(s[1])
#		y = 1-float(s[2])
#	return x,y
#		#['translate', '10', '22', '']	

