/* $Id: check_selfsyncio.c 659 2006-05-13 14:51:08Z jim $
   teebu - An archiving tool
   Copyright (C) 2006 Jim Farrand

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along with
   this program; if not, write to the Free Software Foundation, Inc., 51
   Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>

#include "selfsyncio.h"
#include "baseio.h"
#include "nullio.h"
#include "check_io.h"
#include "check_selfsyncio.h"

#define TEST_FILE "/tmp/bumtest.selfsyncio.tmp"

static const char short_escape[] = "B";
static const int short_escape_len = 1;

static const char medium_escape[] = "BC";
static const int medium_escape_len = 2;

static const char long_escape[] = "BCDEFGHIJK";
static const int long_escape_len = 10;

START_TEST (check_out_basic)
{
  out_stream_t base_outs = baseio_open_out (TEST_FILE, "w");
  fail_if (!base_outs);

  out_stream_t outs =
    selfsync_open_out (medium_escape_len, medium_escape, base_outs, false,
                       false);
  fail_if (!outs);

  basic_out_test (outs);
  release_out (outs);
  release_out (base_outs);
}
END_TEST static int
read_write_test_size ()
{
  switch (get_test_size ())
    {
    case QUICK_TEST:
      return 256;
    case NORMAL_TEST:
      return 1024;
    case LONG_TEST:
      return 1024 * 1024;
    default:
      fail ("Invalid test size");
      return -1;                // Not reached, prevent compiler warning
    }
}

static void
check_read_write (bool with_marks, int escape_len, const char *escape,
                  int block_size)
{
  const int
    test_size = read_write_test_size (), count = test_size / block_size;

  assert (count > 0);
  assert (test_size > 0);

  out_stream_t base_outs = baseio_open_out (TEST_FILE, "w");
  out_stream_t outs =
    selfsync_open_out (escape_len, escape, base_outs, false, false);
  ascending_output_half_test (with_marks, outs, block_size, count);
  release_out (outs);
  release_out (base_outs);

  mark_point ();

  in_stream_t base_ins = baseio_open_in (TEST_FILE, "r");
  in_stream_t ins =
    selfsync_open_in (escape_len, escape, 4096, base_ins, false, false);
  ascending_input_half_test (true, with_marks, ins, block_size, count);
  release_in (ins);
  release_in (base_ins);
}

/** If only C had closures.  Then we wouldn't have to define this repetetive
 * test in a macro. */

#define MAKE_READ_WRITE_CHECK(name, escape_len, escape, marks, block_len) \
START_TEST (name) \
{ \
  check_read_write(marks, escape_len, escape, block_len) ; \
} \
END_TEST

MAKE_READ_WRITE_CHECK (check_read_write_s_s, short_escape_len, short_escape,
                       false,
                       1) MAKE_READ_WRITE_CHECK (check_read_write_s_s_m,
                                                 short_escape_len,
                                                 short_escape, true,
                                                 1)
MAKE_READ_WRITE_CHECK (check_read_write_s_m, short_escape_len, short_escape,
                       false,
                       32) MAKE_READ_WRITE_CHECK (check_read_write_s_m_m,
                                                  short_escape_len,
                                                  short_escape, true,
                                                  32)
MAKE_READ_WRITE_CHECK (check_read_write_s_l, short_escape_len, short_escape,
                       false,
                       256) MAKE_READ_WRITE_CHECK (check_read_write_s_l_m,
                                                   short_escape_len,
                                                   short_escape, true,
                                                   256)
MAKE_READ_WRITE_CHECK (check_read_write_m_s, medium_escape_len, medium_escape,
                       false,
                       1) MAKE_READ_WRITE_CHECK (check_read_write_m_s_m,
                                                 medium_escape_len,
                                                 medium_escape, true,
                                                 1)
MAKE_READ_WRITE_CHECK (check_read_write_m_m, medium_escape_len, medium_escape,
                       false,
                       32) MAKE_READ_WRITE_CHECK (check_read_write_m_m_m,
                                                  medium_escape_len,
                                                  medium_escape, true,
                                                  32)
MAKE_READ_WRITE_CHECK (check_read_write_m_l, medium_escape_len, medium_escape,
                       false,
                       256) MAKE_READ_WRITE_CHECK (check_read_write_m_l_m,
                                                   medium_escape_len,
                                                   medium_escape, true,
                                                   256)
MAKE_READ_WRITE_CHECK (check_read_write_l_s, long_escape_len, long_escape,
                       false,
                       1) MAKE_READ_WRITE_CHECK (check_read_write_l_s_m,
                                                 long_escape_len, long_escape,
                                                 true,
                                                 1)
MAKE_READ_WRITE_CHECK (check_read_write_l_m, long_escape_len, long_escape,
                       false,
                       32) MAKE_READ_WRITE_CHECK (check_read_write_l_m_m,
                                                  long_escape_len,
                                                  long_escape, true,
                                                  32)
MAKE_READ_WRITE_CHECK (check_read_write_l_l, long_escape_len, long_escape,
                       false,
                       256) MAKE_READ_WRITE_CHECK (check_read_write_l_l_m,
                                                   long_escape_len,
                                                   long_escape, true, 256)
/* Check that writes various combinations of escapes and data and marks to try
 * and break the escaping code. */
     void
     check_pathological_data (const int escape_len, const char *escape,
                              const int run, const int padding)
{
  assert (run > 0);

  const int test_size = read_write_test_size () / (run + padding);
  assert (test_size > 0);

  {
    // Do the output
    out_stream_t base_outs = baseio_open_out (TEST_FILE, "w");
    fail_if (!base_outs);

    out_stream_t outs =
      selfsync_open_out (escape_len, escape, base_outs, false, true);
    fail_if (!outs);

    for (int i = 0; i < test_size; i++)
      {
        for (int j = 0; j < padding; j++)
          {
            char padding = '\0';

            iobuffer_t iobuf;
            init_iobuffer_with (&iobuf, 1, 1, (char *) &padding);
            fail_unless (OUTPUT_OK == output_all (outs, &iobuf));
          }

        for (int j = 0; j < run; j++)
          {
            // How much of the escape shall we output?
            int amount = 1 + (i % escape_len);  // Depends on i, to mix things up
            assert (amount <= escape_len);
            assert (amount > 0);

            iobuffer_t iobuf;
            init_iobuffer_with (&iobuf, escape_len, amount, (char *) escape);
            fail_unless (OUTPUT_OK == output_all (outs, &iobuf));
          }

        fail_unless_out_status (OUTPUT_OK, output_mark (outs));
      }

    fail_unless_out_status (OUTPUT_OK, close_and_release_out (outs));
  }

  {
    // Do the input
    in_stream_t base_ins = baseio_open_in (TEST_FILE, "r");
    fail_if (!base_ins);

    in_stream_t ins =
      selfsync_open_in (escape_len, escape, 4096, base_ins, false, true);
    fail_if (!ins);

    for (int i = 0; i < test_size; i++)
      {
        // Input the padding
        for (int j = 0; j < padding; j++)
          {
            char buf = 'Z';

            iobuffer_t iobuf;
            init_iobuffer_with (&iobuf, 1, 0, (char *) &buf);
            fail_unless_in_status (INPUT_OK, input_all (ins, &iobuf));
            fail_unless ('\0' == buf);
          }

        // Input escape fragment
        for (int j = 0; j < run; j++)
          {
            // How much of the escape shall we input?
            int amount = 1 + (i % escape_len);  // Depends on i, to mix things up
            assert (amount <= escape_len);
            assert (amount > 0);

            char buf[amount];
            iobuffer_t iobuf;
            init_iobuffer_with (&iobuf, amount, 0, buf);
            fail_unless_in_status (INPUT_OK, input_all (ins, &iobuf));
            for (int k = 0; k < amount; k++)
              fail_unless (buf[k] == escape[k]);
          }

        {
          input_err_t err = input_mark (ins);
          if (INPUT_OK != err)
            {
              char msg[256];
              snprintf (msg, 256,
                        "Expected INPUT_OK but got %s when i = %d test_size = %d",
                        INPUT_ERR_NAME (err), i, test_size);
              fail (msg);
            }
        }
      }

    {
      // Check we consumed whole file
      char buf;
      iobuffer_t iobuf;
      init_iobuffer_with (&iobuf, 1, 0, (char *) &buf);
      fail_unless_in_status (INPUT_EOF, input_all (ins, &iobuf));
    }
    fail_unless_in_status (INPUT_OK, close_and_release_in (ins));
  }
}

#define MAKE_PATHOLOGICAL_TEST(name, escape_len, escape, run, padding) \
START_TEST (name) \
{ \
    check_pathological_data(escape_len, escape, run, padding) ; \
} \
END_TEST

MAKE_PATHOLOGICAL_TEST (check_pathological_data_short_1_0, short_escape_len,
                        short_escape, 1,
                        0)
MAKE_PATHOLOGICAL_TEST (check_pathological_data_short_16_0, short_escape_len,
                        short_escape, 16,
                        0)
MAKE_PATHOLOGICAL_TEST (check_pathological_data_short_1_1, short_escape_len,
                        short_escape, 1,
                        1)
MAKE_PATHOLOGICAL_TEST (check_pathological_data_short_16_1, short_escape_len,
                        short_escape, 16,
                        1)
MAKE_PATHOLOGICAL_TEST (check_pathological_data_short_1_16, short_escape_len,
                        short_escape, 1,
                        16)
MAKE_PATHOLOGICAL_TEST (check_pathological_data_short_16_16, short_escape_len,
                        short_escape, 16,
                        16)
MAKE_PATHOLOGICAL_TEST (check_pathological_data_medium_1_0, medium_escape_len,
                        medium_escape, 1,
                        0)
MAKE_PATHOLOGICAL_TEST (check_pathological_data_medium_16_0,
                        medium_escape_len, medium_escape, 16,
                        0)
MAKE_PATHOLOGICAL_TEST (check_pathological_data_medium_1_1, medium_escape_len,
                        medium_escape, 1,
                        1)
MAKE_PATHOLOGICAL_TEST (check_pathological_data_medium_16_1,
                        medium_escape_len, medium_escape, 16,
                        1)
MAKE_PATHOLOGICAL_TEST (check_pathological_data_medium_1_16,
                        medium_escape_len, medium_escape, 1,
                        16)
MAKE_PATHOLOGICAL_TEST (check_pathological_data_medium_16_16,
                        medium_escape_len, medium_escape, 16,
                        16)
MAKE_PATHOLOGICAL_TEST (check_pathological_data_long_1_0, long_escape_len,
                        long_escape, 1,
                        0)
MAKE_PATHOLOGICAL_TEST (check_pathological_data_long_16_0, long_escape_len,
                        long_escape, 16,
                        0)
MAKE_PATHOLOGICAL_TEST (check_pathological_data_long_1_1, long_escape_len,
                        long_escape, 1,
                        1)
MAKE_PATHOLOGICAL_TEST (check_pathological_data_long_16_1, long_escape_len,
                        long_escape, 16,
                        1)
MAKE_PATHOLOGICAL_TEST (check_pathological_data_long_1_16, long_escape_len,
                        long_escape, 1,
                        16)
MAKE_PATHOLOGICAL_TEST (check_pathological_data_long_16_16, long_escape_len,
                        long_escape, 16,
                        16) START_TEST (check_null_escape_bug)
{
  char escape[] = "\0";
  out_stream_t outs =
    selfsync_open_out (1, escape, nullio_open_out (), false, true);
  fail_if (!outs);

  iobuffer_t iobuf;
  init_iobuffer_with (&iobuf, 1, 1, escape);

  fail_unless (OUTPUT_OK == output_all (outs, &iobuf)); // BUG: Infinite loop!
  close_out (outs);
}
END_TEST void
add_selfsyncio_tests (Suite * s)
{
  TCase *tc_core = tcase_create ("SelfSyncIO");
  tcase_add_test (tc_core, check_out_basic);
  tcase_add_test (tc_core, check_read_write_s_s);
  tcase_add_test (tc_core, check_read_write_s_s_m);
  tcase_add_test (tc_core, check_read_write_s_m);
  tcase_add_test (tc_core, check_read_write_s_m_m);
  tcase_add_test (tc_core, check_read_write_s_l);
  tcase_add_test (tc_core, check_read_write_s_l_m);
  tcase_add_test (tc_core, check_read_write_m_s);
  tcase_add_test (tc_core, check_read_write_m_s_m);
  tcase_add_test (tc_core, check_read_write_m_m);
  tcase_add_test (tc_core, check_read_write_m_m_m);
  tcase_add_test (tc_core, check_read_write_m_l);
  tcase_add_test (tc_core, check_read_write_m_l_m);
  tcase_add_test (tc_core, check_read_write_l_s);
  tcase_add_test (tc_core, check_read_write_l_s_m);
  tcase_add_test (tc_core, check_read_write_l_m);
  tcase_add_test (tc_core, check_read_write_l_m_m);
  tcase_add_test (tc_core, check_read_write_l_l);
  tcase_add_test (tc_core, check_read_write_l_l_m);
  tcase_add_test (tc_core, check_null_escape_bug);
  tcase_add_test (tc_core, check_pathological_data_short_1_0);
  tcase_add_test (tc_core, check_pathological_data_short_1_1);
  tcase_add_test (tc_core, check_pathological_data_short_16_0);
  tcase_add_test (tc_core, check_pathological_data_short_16_1);
  tcase_add_test (tc_core, check_pathological_data_short_1_16);
  tcase_add_test (tc_core, check_pathological_data_short_16_16);
  tcase_add_test (tc_core, check_pathological_data_medium_1_0);
  tcase_add_test (tc_core, check_pathological_data_medium_1_1);
  tcase_add_test (tc_core, check_pathological_data_medium_16_0);
  tcase_add_test (tc_core, check_pathological_data_medium_16_1);
  tcase_add_test (tc_core, check_pathological_data_medium_1_16);
  tcase_add_test (tc_core, check_pathological_data_medium_16_16);
  tcase_add_test (tc_core, check_pathological_data_long_1_0);
  tcase_add_test (tc_core, check_pathological_data_long_1_1);
  tcase_add_test (tc_core, check_pathological_data_long_16_0);
  tcase_add_test (tc_core, check_pathological_data_long_16_1);
  tcase_add_test (tc_core, check_pathological_data_long_1_16);
  tcase_add_test (tc_core, check_pathological_data_long_16_16);

  suite_add_tcase (s, tc_core);
}
