/* $Id: unixio.c 658 2006-05-13 14:50:30Z jim $
   teebu - An archiving tool
   Copyright (C) 2006 Jim Farrand

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along with
   this program; if not, write to the Free Software Foundation, Inc., 51
   Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


#include <assert.h>

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#include "unixio.h"

static output_err_t
my_output_limited (void *data, iobuffer_t * buf, size_t amount)
{
  if (!data)
    return OUTPUT_ERR_BAD;

  const ssize_t written =
    write (*(int *) data, iobuffer_data_pointer (buf), amount);

  if (-1 == written)
    return OUTPUT_ERR_FAILED;

  iobuffer_mark_taken (buf, written);

  return OUTPUT_OK;
}

static output_err_t
my_close_out (void *data)
{
  if (!data)
    return OUTPUT_ERR_BAD;

  // fprintf(stderr, "Closing out fd %d\n", *(int *)data) ;

  if (-1 == close (*(int *) data))
    return OUTPUT_ERR_FAILED;

  return OUTPUT_OK;
}

/* Currently used for both in and out streams! */
static void
my_release (void *uncast_data)
{
  assert (uncast_data);
  free (uncast_data);
}

static out_stream_type_t my_out_type = {
  .output_limited = my_output_limited,
  .output_mark = NULL,          // Unimplemented
  .flush = NULL,                // Unimplemented
  .close_out = my_close_out,
  .release_out = my_release
};

out_stream_t
unixio_open_out_fd (int fd)
{
  int *data = malloc (sizeof (int));
  if (!data)
    return NULL;

  *data = fd;
  // fprintf(stderr, "Opening out fd %d\n", *(int *)data) ;

  return open_out (&my_out_type, data);
}

out_stream_t
unixio_open_out (const char *pathname, int flags, mode_t mode)
{
  int fd = open (pathname, flags, mode);
  if (-1 == fd)
    return NULL;
  return unixio_open_out_fd (fd);
}

static input_err_t
my_input_limited (void *data, iobuffer_t * buf, size_t amount)
{
  if (!data)
    return OUTPUT_ERR_BAD;

  const ssize_t n = read (*(int *) data, iobuffer_free_pointer (buf), amount);
  if (0 == n)
    return INPUT_EOF;
  else if (-1 == n)
    return INPUT_ERR_FAILED;

  iobuffer_mark_added (buf, n);

  return INPUT_OK;
}

static input_err_t
my_close_in (void *data)
{
  if (!data)
    return INPUT_ERR_BAD;

  // fprintf(stderr, "Closing in fd %d\n", *(int *)data) ;
  if (-1 == close (*(int *) data))
    return INPUT_ERR_FAILED;

  return INPUT_OK;
}

static in_stream_type_t my_in_type = {
  .input_limited = my_input_limited,
  .input_mark = NULL,           // Unimplemented
  .close_in = my_close_in,
  .release_in = my_release
};

in_stream_t
unixio_open_in_fd (int fd)
{
  int *data = malloc (sizeof (int));
  if (!data)
    return NULL;

  *data = fd;
  // fprintf(stderr, "Opening in fd %d\n", *(int *)data) ;

  return open_in (&my_in_type, data);
}

in_stream_t
unixio_open_in (const char *pathname, int flags)
{
  int fd = open (pathname, flags);
  if (-1 == fd)
    return NULL;
  return unixio_open_in_fd (fd);
}
