/* $Id: uidcache.c 715 2006-05-29 21:02:00Z jim $
   teebu - An archiving tool
   Copyright (C) 2006 Jim Farrand

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along with
   this program; if not, write to the Free Software Foundation, Inc., 51
   Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/


#include <pwd.h>
#include <grp.h>
#include <assert.h>
#include <unistd.h>

#include "uidcache.h"

#include "hashtable.h"
#include "list.h"

/* This structure contains hashtables for looking up each type of structure by
 * id and name, and a list of memory blocks to free() when we release the
 * structure. */
struct uidcache
{
  hashtable_t uidcache_uids;
  hashtable_t uidcache_usernames;
  hashtable_t uidcache_gids;
  hashtable_t uidcache_groupnames;
  list_t      uidcache_allocations;
};


void
release_uidcache (uidcache_t cache)
{
  assert (cache);

  if (cache->uidcache_uids)
    hashtable_release (cache->uidcache_uids);
  if (cache->uidcache_usernames)
    hashtable_release (cache->uidcache_usernames);
  if (cache->uidcache_gids)
    hashtable_release (cache->uidcache_gids);
  if (cache->uidcache_groupnames)
    hashtable_release (cache->uidcache_groupnames);

  if (cache->uidcache_allocations)
    {
      iterator_t it = list_iterator (cache->uidcache_allocations);
      while (iterator_has_next (it))
        free (iterator_next (it));
      release_iterator (it);
      release_list (cache->uidcache_allocations);
    }

  free(cache);
}

uidcache_t
create_uidcache (void)
{
  uidcache_t cache = malloc (sizeof (struct uidcache));
  if (!cache)
    return NULL;

  cache->uidcache_uids = create_long_hashtable (1);
  cache->uidcache_usernames = create_string_hashtable (1, true);
  cache->uidcache_gids = create_long_hashtable (1);
  cache->uidcache_groupnames = create_string_hashtable (1, true);
  cache->uidcache_allocations = empty_list ();

  if (!cache->uidcache_uids ||
      !cache->uidcache_usernames ||
      !cache->uidcache_gids ||
      !cache->uidcache_groupnames ||
      !cache->uidcache_allocations)
    {
      release_uidcache(cache);
      free(cache);
    }

  return cache;
}

/* Add to the password cache. */
static void
add_to_pwcache(uidcache_t cache, uid_t *uid, const char *name,
             struct passwd *passwd)
{
  assert (cache);

  if (uid)
    {
      long uid_long = (long)*uid;
      assert (*uid == (uid_t)uid_long);

      hashtable_add(cache->uidcache_uids, &uid_long, passwd);
    }

  if (name)
    {
      hashtable_add(cache->uidcache_usernames, name, passwd);
    }
}

/* Get password structure by uid, either from cache or by doing a lookup. */
static struct passwd *
get_passwd_by_uid (uidcache_t cache, uid_t uid)
{
  long uid_long = (long)uid;
  assert (uid == (uid_t)uid_long);

  if (hashtable_contains_key (cache->uidcache_uids, &uid_long))
    return hashtable_find (cache->uidcache_uids, &uid_long);

  // Not cached, need to do the actual lookup
  struct passwd *pwbuf = malloc (sizeof (struct passwd));
  if (!pwbuf)
    return NULL;

  list_snoc (cache->uidcache_allocations, pwbuf);
  
  size_t bufsize = sysconf (_SC_GETPW_R_SIZE_MAX);
  char *buffer = malloc (bufsize);
  if (!buffer)
    return NULL;

  list_snoc (cache->uidcache_allocations, buffer);

  struct passwd *pwbufp;
  getpwuid_r (uid, pwbuf, buffer, bufsize, &pwbufp);

  add_to_pwcache(cache, &uid, pwbufp ? pwbufp->pw_name : NULL, pwbufp);

  return pwbufp;
}

/* Get password structure by username, either from cache or by doing a lookup.
 */
static struct passwd *
get_passwd_by_username (uidcache_t cache, const char *username)
{
  if (hashtable_contains_key (cache->uidcache_usernames, username))
    return hashtable_find (cache->uidcache_usernames, username);

  // Not cached, need to do the actual lookup
  struct passwd *pwbuf = malloc (sizeof (struct passwd));
  if (!pwbuf)
    return NULL;

  list_snoc (cache->uidcache_allocations, pwbuf);

  size_t bufsize = sysconf (_SC_GETPW_R_SIZE_MAX);
  char *buffer = malloc (bufsize);
  if (!buffer)
    return NULL;

  list_snoc (cache->uidcache_allocations, buffer);

  struct passwd *pwbufp;
  getpwnam_r (username, pwbuf, buffer, bufsize, &pwbufp);

  add_to_pwcache(cache, pwbufp ? &pwbuf->pw_uid : NULL, username, pwbufp);

  return pwbufp;
}

const char *
lookup_username_from_uid (uidcache_t cache, uid_t uid)
{
  struct passwd *passwd = get_passwd_by_uid (cache, uid);
  if (!passwd)
    return NULL;

  return passwd->pw_name;
}

bool
lookup_uid_from_username (uidcache_t cache, const char *username, uid_t *result)
{
  struct passwd *passwd = get_passwd_by_username (cache, username);
  if (!passwd)
    return false;

  *result = passwd->pw_uid;
  return true;
}

/* Add to the group cache. */
static void
add_to_grcache(uidcache_t cache, gid_t *gid, const char *name,
               struct group *group)
{
  assert (cache);

  if (gid)
    {
      long gid_long = (long)*gid;
      assert (*gid == (gid_t)gid_long);

      hashtable_add(cache->uidcache_gids, &gid_long, group);
    }

  if (name)
    {
      hashtable_add(cache->uidcache_groupnames, name, group);
    }
}

/* Get group structure by gid, either from cache or by doing a lookup. */
static struct group *
get_group_by_gid (uidcache_t cache, gid_t gid)
{
  long gid_long = (long)gid;
  assert (gid == (gid_t)gid_long);

  if (hashtable_contains_key (cache->uidcache_gids, &gid_long))
    return hashtable_find (cache->uidcache_gids, &gid_long);

  // Not cached, need to do the actual lookup
  struct group *grbuf = malloc (sizeof (struct group));
  if (!grbuf)
    return NULL;

  list_snoc (cache->uidcache_allocations, grbuf);
  
  size_t bufsize = sysconf (_SC_GETGR_R_SIZE_MAX);
  char *buffer = malloc (bufsize);
  if (!buffer)
    return NULL;

  list_snoc (cache->uidcache_allocations, buffer);

  struct group *grbufp;
  getgrgid_r (gid, grbuf, buffer, bufsize, &grbufp);

  add_to_grcache(cache, &gid, grbufp ? grbufp->gr_name : NULL, grbufp);

  return grbufp;
}

/* Get group structure by gid, either from cache or by doing a lookup. */
static struct group *
get_group_by_groupname (uidcache_t cache, const char *groupname)
{
  if (hashtable_contains_key (cache->uidcache_groupnames, groupname))
    return hashtable_find (cache->uidcache_groupnames, groupname);

  // Not cached, need to do the actual lookup
  struct group *grbuf = malloc (sizeof (struct group));
  if (!grbuf)
    return NULL;

  list_snoc (cache->uidcache_allocations, grbuf);
  
  size_t bufsize = sysconf (_SC_GETGR_R_SIZE_MAX);
  char *buffer = malloc (bufsize);
  if (!buffer)
    return NULL;

  list_snoc (cache->uidcache_allocations, buffer);

  struct group *grbufp;
  getgrnam_r (groupname, grbuf, buffer, bufsize, &grbufp);

  add_to_grcache(cache, grbufp ? &grbufp->gr_gid : NULL, groupname, grbufp);

  return grbufp;
}

const char *
lookup_groupname_from_gid (uidcache_t cache, gid_t gid)
{
  struct group *group = get_group_by_gid (cache, gid);
  if (!group)
    return NULL;

  return group->gr_name;
}

bool
lookup_gid_from_groupname (uidcache_t cache, const char *groupname, gid_t *result)
{
  struct group *group = get_group_by_groupname (cache, groupname);
  if (!group)
    return false;

  *result = group->gr_gid;
  return true;
}
