/* $Id: statio.c 695 2006-05-20 01:30:20Z jim $
   teebu - An archiving tool
   Copyright (C) 2006 Jim Farrand

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along with
   this program; if not, write to the Free Software Foundation, Inc., 51
   Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


#include <assert.h>

#include "logging.h"
#include "statio.h"

/*********
 * Stats *
 *********/

void
clear_stats (iostat_t * stats)
{
  stats->stats_created = time (NULL);
  if (stats->stats_path)
    {
      release_path (stats->stats_path);
      stats->stats_path = NULL;
    }
  stats->stats_bytes_written = 0;
  stats->stats_bytes_read = 0;
  stats->stats_open_outs = 0;
  stats->stats_close_outs = 0;
  stats->stats_open_ins = 0;
  stats->stats_close_ins = 0;
  stats->stats_output_marks = 0;
  stats->stats_input_marks = 0;

  for (int i = 0; i < OUTPUT_ERR_COUNT; i++)
    stats->stats_output_outcome[i] = 0;

  for (int i = 0; i < INPUT_ERR_COUNT; i++)
    stats->stats_input_outcome[i] = 0;
}

iostat_t *
create_stats (void)
{
  iostat_t *stats = malloc (sizeof (iostat_t));
  stats->stats_path = NULL;
  if (!stats)
    MEMFAILED ();

  clear_stats (stats);
  return stats;
}

void
merge_stats_path(iostat_t *stats, path_t path)
{
  // If this is the first path we have seen, keep it
  if (!stats->stats_path)
    {
      stats->stats_path = copy_path (path);
      return;
    }

  const char *str1 = path_str (stats->stats_path);
  const char *str2 = path_str (path);

  int keep_i = 0, check_i = 0;
  while (str1[check_i] != '\0' && str2[check_i] != '\0' && str1[check_i] == str2[check_i])
    {
      if ('/' == str1[check_i])
        keep_i = check_i;

      check_i++;
    }

  // Keep if whole thing matched
  if ('\0' == str1[check_i] || '\0' == str2[check_i])
    keep_i = check_i;

  // Keep at least the initial root '/' !
  if(0 == keep_i)
    keep_i = 1;

  set_path_length(stats->stats_path, keep_i);
}

void
clear_stats_path(iostat_t *stats)
{
  if (stats->stats_path)
    {
      release_path (stats->stats_path);
      stats->stats_path = NULL;
    }
}

void
release_stats (iostat_t * stats)
{
  if (stats->stats_path)
    release_path (stats->stats_path);
  free (stats);
}

/**********
 * Output *
 **********/

struct statio_out_data
{
  iostat_t *out_stats;
  out_stream_t out_base;
  bool out_pass_marks;
  bool out_own_base;
};


static output_err_t
my_output_limited (void *uncast_data, iobuffer_t * buf, size_t max)
{
  struct statio_out_data *data = uncast_data;
  if (!data)
    return OUTPUT_ERR_BAD;

  size_t size_before = iobuffer_data_size (buf);
  output_err_t err = output_limited (data->out_base, buf, max);
  data->out_stats->stats_bytes_written +=
    size_before - iobuffer_data_size (buf);
  data->out_stats->stats_output_outcome[err]++;
  return err;
}

static output_err_t
my_close_out (void *uncast_data)
{
  struct statio_out_data *data = uncast_data;
  if (!data)
    return OUTPUT_ERR_BAD;

  output_err_t err = close_out (data->out_base);

  data->out_stats->stats_output_outcome[err]++;
  if (OUTPUT_OK == err)
    data->out_stats->stats_close_outs++;

  return err;
}

static output_err_t
my_output_mark (void *uncast_data)
{
  struct statio_out_data *data = uncast_data;
  if (!data)
    return OUTPUT_ERR_BAD;

  output_err_t err = OUTPUT_OK;
  if (data->out_pass_marks)
    {
      err = output_mark (data->out_base);
      data->out_stats->stats_output_outcome[err]++;
    }

  if (OUTPUT_OK == err)
    data->out_stats->stats_output_marks++;
  return err;
}

static output_err_t
my_flush (void *uncast_data)
{
  struct statio_out_data *data = uncast_data;
  if (!data)
    return OUTPUT_ERR_BAD;

  output_err_t err = flush (data->out_base);
  data->out_stats->stats_output_outcome[err]++;
  return err;
}

static void
my_release_out (void *uncast_data)
{
  struct statio_out_data *data = uncast_data;
  assert (data);

  if (data->out_own_base)
    release_out (data->out_base);

  free (data);
}

static out_stream_type_t my_out_type = {
  .output_limited = my_output_limited,
  .output_mark = my_output_mark,
  .flush = my_flush,
  .close_out = my_close_out,
  .release_out = my_release_out
};

out_stream_t
statio_open_out (out_stream_t base, iostat_t * stats, bool pass_marks,
                 bool own_base)
{
  assert (base);
  assert (stats);

  struct statio_out_data *data = malloc (sizeof (struct statio_out_data));
  if (!data)
    MEMFAILED ();

  data->out_stats = stats;
  data->out_base = base;
  data->out_own_base = own_base;
  data->out_pass_marks = pass_marks;

  out_stream_t outs = open_out (&my_out_type, data);
  if (!outs)
    return NULL;

  stats->stats_open_outs++;
  return outs;
}

/*********
 * Input *
 *********/

struct statio_in_data
{
  iostat_t *in_stats;
  in_stream_t in_base;
  bool in_pass_marks;
  bool in_own_base;
};

static input_err_t
my_input_limited (void *uncast_data, iobuffer_t * buf, size_t max)
{
  struct statio_in_data *data = uncast_data;
  if (!data)
    return INPUT_ERR_BAD;

  size_t size_before = iobuffer_data_size (buf);
  input_err_t err = input_limited (data->in_base, buf, max);
  data->in_stats->stats_bytes_read += iobuffer_data_size (buf) - size_before;
  data->in_stats->stats_input_outcome[err]++;
  return err;
}

static input_err_t
my_input_mark (void *uncast_data)
{
  struct statio_in_data *data = uncast_data;
  if (!data)
    return INPUT_ERR_BAD;

  if (!data->in_pass_marks)
    {
      data->in_stats->stats_input_marks++;
      return INPUT_OK;
    }

  input_err_t err = INPUT_OK;
  err = input_mark (data->in_base);
  data->in_stats->stats_input_outcome[err]++;

  if (INPUT_OK == err)
    data->in_stats->stats_input_marks++;
  return err;
}

static input_err_t
my_close_in (void *uncast_data)
{
  struct statio_in_data *data = uncast_data;
  if (!data)
    return INPUT_ERR_BAD;

  input_err_t err = close_in (data->in_base);

  data->in_stats->stats_input_outcome[err]++;
  if (INPUT_OK == err)
    data->in_stats->stats_close_ins++;

  return err;
}

static void
my_release_in (void *uncast_data)
{
  struct statio_in_data *data = uncast_data;
  assert (data);

  if (data->in_own_base)
    release_in (data->in_base);

  free (data);
}

static in_stream_type_t my_in_type = {
  .input_limited = my_input_limited,
  .input_mark = my_input_mark,
  .close_in = my_close_in,
  .release_in = my_release_in
};

in_stream_t
statio_open_in (in_stream_t base, iostat_t * stats, bool pass_marks,
                bool own_base)
{
  assert (base);
  assert (stats);

  struct statio_in_data *data = malloc (sizeof (struct statio_in_data));
  if (!data)
    MEMFAILED ();

  data->in_stats = stats;
  data->in_base = base;
  data->in_pass_marks = pass_marks;
  data->in_own_base = own_base;

  in_stream_t ins = open_in (&my_in_type, data);
  if (!ins)
    return NULL;

  stats->stats_open_ins++;
  return ins;
}
