/* $Id: hashtable.h 710 2006-05-28 23:11:46Z jim $
   teebu - An archiving tool
   Copyright (C) 2006 Jim Farrand

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along with
   this program; if not, write to the Free Software Foundation, Inc., 51
   Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/


/*
 * An implementation of hashtables.
 */

#ifndef HASHTABLE_H
#define HASHTABLE_H

#include <stdbool.h>
#include "list.h"

typedef struct hashtable *hashtable_t;

/* A function for comparing keys in a hashtable.  The first argument is the
 * user data associated with the hashtable.  The second and third arguments are
 * pointers to the keys to compare. */
typedef bool (*is_key_equal_f) (void *, const void *, const void *);

/* A function for computing the hash value of a key in a hashtable.  The first
 * argument is the user data associated with thehashtable.  The second argument
 * is a pointer to the key to hash. */
typedef size_t (*key_hash_f) (void *, const void *);

/* A function for copying keys in a hashtable.  The first argument is the user
 * data associated with the hashtable.  The second argument is the key to copy.
 */
typedef void *(*key_copy_f) (void *,  const void *);

/* Create an empty hashtable configured for string keys.  Iff copy is true,
 * keys are copied when items are added to the hashtable. */
hashtable_t create_string_hashtable (size_t size, bool copy);

/* Create an empty hashtable configured for int keys.  Keys are copied when
 * items are added to the hashtable. */
hashtable_t create_int_hashtable (size_t size);

/* Create an empty hashtable configured for long keys.  Keys are copied when
 * items are added to the hashtable. */
hashtable_t create_long_hashtable (size_t size);

/* Create an empty hashtable. */
hashtable_t create_hashtable (size_t size,
                              void *user_data, key_copy_f key_copy,
                              is_key_equal_f is_key_equal, key_hash_f key_hash);

/* Replace all entries for the given key with the given binding. */
bool hashtable_replace(hashtable_t ht, const void *key, void *data);

/* Add an entry to list of bindings for the given key */
bool hashtable_add(hashtable_t ht, const void *key, void *data);

/* Remove and return most recent binding for key. */
void *hashtable_remove(hashtable_t ht, const void *key);

/* Release all data associated with the given hashtable. */
void hashtable_release(hashtable_t ht);

/* Return true if the given key is mapped in the given hashtable. */
bool hashtable_contains_key(hashtable_t ht, const void *key);

/* Find an entry. */
void *hashtable_find(hashtable_t ht, const void *key);

/* Find all values associated with the given key. Returns NULL if the given key
 * is not bound. Caller is responsible for calling release_list  on  the
 * returned list. */
list_t hashtable_find_all(hashtable_t ht, const void *key);

/*
 * Some useful functions for creating your own hashtables.
 */

bool is_int_equal(void *, const void *, const void *);
size_t int_hash(void *, const void *);

bool is_string_equal(void *, const void *, const void *);
size_t string_hash(void *, const void *);
#endif

