/* $Id: debugio.c 658 2006-05-13 14:50:30Z jim $
   teebu - An archiving tool
   Copyright (C) 2006 Jim Farrand

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along with
   this program; if not, write to the Free Software Foundation, Inc., 51
   Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


#include <assert.h>
#include <ctype.h>

#include "logging.h"
#include "debugio.h"

/* Shared between in and out channels.  */
struct debugio_data
{
  const char *pfix;
  unsigned per_line;
  char *line_buffer;
  unsigned line_buffer_used;    // Amount in buffer, should always be left < per_line
  bool own_base;
  size_t offset;
};

static void
debug_output_line (const char *pfix, size_t offset, char dir,
                   const char *data, unsigned per_line, unsigned amount)
{
  const unsigned buf_size = 256;
  char buf[buf_size];
  unsigned used = 0;

  used +=
    snprintf (buf + used, buf_size - used, "%s\t%08zx %c", pfix, offset, dir);
  assert (used < buf_size);

  for (int i = 0; i < per_line; i++)
    {
      if (i < amount)
        used +=
          snprintf (buf + used, buf_size - used, " %02x",
                    (unsigned char) data[i]);
      else
        used += snprintf (buf + used, buf_size - used, "   ");
      assert (used < buf_size);
    }

  used += snprintf (buf + used, buf_size - used, " |");
  assert (used < buf_size);

  for (int i = 0; i < amount; i++)
    {
      if (isprint (data[i]))
        used += snprintf (buf + used, buf_size - used, "%c", data[i]);
      else
        used += snprintf (buf + used, buf_size - used, ".");
      assert (used < buf_size);
    }
  assert (used < buf_size);

  used += snprintf (buf + used, buf_size - used, "|");
  assert (used < buf_size);

  LOGF (DEBUG, "%s", buf);
}

static void
debug_stream (struct debugio_data *debug_data, const char *buf, size_t amount)
{
  if (IS_DEBUG_ENABLED ())
    {
      if (debug_data->line_buffer)
        {
          // There is a line buffer
          size_t done = 0;
          do
            {
              // Copy some data into the buffer
              unsigned copy_amount =
                debug_data->per_line - debug_data->line_buffer_used;
              if ((amount - done) < copy_amount)
                copy_amount = amount - done;

              memcpy (debug_data->line_buffer + debug_data->line_buffer_used,
                      buf + done, copy_amount);

              done += copy_amount;
              debug_data->line_buffer_used += copy_amount;

              // Output the buffer if it is full
              if (debug_data->line_buffer_used == debug_data->per_line)
                {
                  debug_output_line (debug_data->pfix, debug_data->offset,
                                     '>', debug_data->line_buffer,
                                     debug_data->per_line,
                                     debug_data->line_buffer_used);
                  debug_data->offset += debug_data->line_buffer_used;
                  debug_data->line_buffer_used = 0;
                }
            }
          while (done < amount);

        }
      else
        {
          // There is no line buffer
          size_t done = 0;
          do
            {
              // Output up to a line worth of data
              unsigned output_amount = debug_data->per_line;
              if ((amount - done) < output_amount)
                output_amount = amount - done;

              debug_output_line (debug_data->pfix, debug_data->offset, '>',
                                 buf + done, debug_data->per_line,
                                 output_amount);
              debug_data->offset += output_amount;

            }
          while (done < amount);
        }
    }
}

static void
debug_stream_flush (struct debugio_data *debug_data)
{
  if (debug_data->line_buffer)
    {
      if (debug_data->line_buffer_used > 0)
        {
          debug_output_line (debug_data->pfix, debug_data->offset, '>',
                             debug_data->line_buffer, debug_data->per_line,
                             debug_data->line_buffer_used);
          debug_data->offset += debug_data->line_buffer_used;
          debug_data->line_buffer_used = 0;
        }
    }
}

/**********
 * Output *
 **********/

struct debugio_out_data
{
  struct debugio_data out_debug_data;
  out_stream_t out_base;
};

static output_err_t
my_output_limited (void *const uncast_data, iobuffer_t * const buf,
                   size_t amount)
{
  struct debugio_out_data *const data =
    (struct debugio_out_data *) uncast_data;
  if (!data)
    return OUTPUT_ERR_BAD;

  const char *char_data = iobuffer_data_pointer (buf);
  size_t size_before = iobuffer_data_size (buf);
  output_err_t err = output_limited (data->out_base, buf, amount);
  size_t output_done = size_before - iobuffer_data_size (buf);

  debug_stream (&(data->out_debug_data), char_data, output_done);

  if (OUTPUT_OK != err)
    {
      LOGF (DEBUG, "%s\t%08zx > Output error: %s", data->out_debug_data.pfix,
            data->out_debug_data.offset, OUTPUT_ERR_NAME (err));
    }

  return err;
}

static output_err_t
my_output_mark (void *const uncast_data)
{
  struct debugio_out_data *const data =
    (struct debugio_out_data *) uncast_data;
  if (!data)
    return OUTPUT_ERR_BAD;

  debug_stream_flush (&data->out_debug_data);

  LOGF (DEBUG, "%s\t%08zx > Outputting mark", data->out_debug_data.pfix,
        data->out_debug_data.offset);

  return output_mark (data->out_base);
}

static output_err_t
my_close_out (void *uncast_data)
{
  struct debugio_out_data *data = (struct debugio_out_data *) uncast_data;
  if (!data)
    return OUTPUT_ERR_BAD;

  debug_stream_flush (&data->out_debug_data);
  LOGF (DEBUG, "%s\t%08zx > Closing file", data->out_debug_data.pfix,
        data->out_debug_data.offset);

  return close_out (data->out_base);
}

static output_err_t
my_flush (void *uncast_data)
{
  struct debugio_out_data *data = (struct debugio_out_data *) uncast_data;
  if (!data)
    return OUTPUT_ERR_BAD;

  debug_stream_flush (&data->out_debug_data);
  LOGF (DEBUG, "%s\t%08zx > Flushed", data->out_debug_data.pfix,
        data->out_debug_data.offset);

  return flush (data->out_base);
}

static void
my_release_out (void *uncast_data)
{
  struct debugio_out_data *data = (struct debugio_out_data *) uncast_data;
  assert (data);

  if (data->out_debug_data.own_base)
    release_out (data->out_base);

  if (data->out_debug_data.line_buffer)
    free (data->out_debug_data.line_buffer);

  free (data);
}

static out_stream_type_t my_out_type = {
  .output_limited = my_output_limited,
  .output_mark = my_output_mark,
  .flush = my_flush,
  .close_out = my_close_out,
  .release_out = my_release_out
};

out_stream_t
debugio_open_out (out_stream_t base_outs,
                  const char *pfix, unsigned per_line,
                  bool buffer_lines, size_t start, size_t len, bool own_base)
{
  assert (base_outs);

  struct debugio_out_data *data = malloc (sizeof (struct debugio_out_data));
  if (!data)
    MEMFAILED ();

  if (buffer_lines)
    {
      data->out_debug_data.line_buffer = malloc (per_line);
      if (!data->out_debug_data.line_buffer)
        MEMFAILED ();
    }
  else
    {
      data->out_debug_data.line_buffer = NULL;
    }

  data->out_base = base_outs;
  data->out_debug_data.pfix = pfix;
  data->out_debug_data.per_line = per_line;
  data->out_debug_data.line_buffer_used = 0;
  data->out_debug_data.own_base = own_base;
  data->out_debug_data.offset = 0;

  return open_out (&my_out_type, data);
}

/*********
 * Input *
 *********/

struct debugio_in_data
{
  struct debugio_data in_debug_data;
  in_stream_t in_base;
};


static input_err_t
my_input_limited (void *uncast_data, iobuffer_t * buf, size_t amount)
{
  struct debugio_in_data *const data = (struct debugio_in_data *) uncast_data;
  if (!data)
    return INPUT_ERR_BAD;

  const char *char_data = iobuffer_free_pointer (buf);
  size_t size_before = iobuffer_data_size (buf);
  input_err_t err = input_limited (data->in_base, buf, amount);
  size_t input_done = iobuffer_data_size (buf) - size_before;

  debug_stream (&(data->in_debug_data), char_data, input_done);

  if (INPUT_OK != err)
    {
      LOGF (DEBUG, "%s\t%08zx < Input error: %s", data->in_debug_data.pfix,
            data->in_debug_data.offset, INPUT_ERR_NAME (err));
    }

  return err;
}

static input_err_t
my_input_mark (void *uncast_data)
{
  struct debugio_in_data *const data = (struct debugio_in_data *) uncast_data;
  if (!data)
    return INPUT_ERR_BAD;

  debug_stream_flush (&data->in_debug_data);

  input_err_t err = input_mark (data->in_base);
  if (INPUT_OK == err)
    {
      LOGF (DEBUG, "%s\t%08zx < Input mark", data->in_debug_data.pfix,
            data->in_debug_data.offset);
    }
  else
    {
      LOGF (DEBUG, "%s\t%08zx < Error inputting mark: %s",
            data->in_debug_data.pfix, data->in_debug_data.offset,
            INPUT_ERR_NAME (err));
    }
  return err;
}

static input_err_t
my_input_recover (void *uncast_data)
{
  struct debugio_in_data *const data = (struct debugio_in_data *) uncast_data;
  if (!data)
    return INPUT_ERR_BAD;

  DEBUG ("Error recovery");

  return input_recover (data->in_base);
}

static input_err_t
my_close_in (void *uncast_data)
{
  struct debugio_in_data *data = (struct debugio_in_data *) uncast_data;
  if (!data)
    return INPUT_ERR_BAD;

  debug_stream_flush (&data->in_debug_data);

  return close_in (data->in_base);
}

static void
my_release_in (void *uncast_data)
{
  struct debugio_in_data *data = (struct debugio_in_data *) uncast_data;
  assert (data);

  if (data->in_debug_data.own_base)
    release_in (data->in_base);

  if (data->in_debug_data.line_buffer)
    free (data->in_debug_data.line_buffer);

  free (data);
}

static in_stream_type_t my_in_type = {
  .input_limited = my_input_limited,
  .input_mark = my_input_mark,
  .close_in = my_close_in,
  .release_in = my_release_in,
  .input_recover = my_input_recover
};

in_stream_t
debugio_open_in (in_stream_t base_ins,
                 const char *pfix, unsigned per_line,
                 bool buffer_lines, size_t start, size_t len, bool own_base)
{
  assert (base_ins);

  struct debugio_in_data *data = malloc (sizeof (struct debugio_in_data));
  if (!data)
    MEMFAILED ();

  if (buffer_lines)
    {
      data->in_debug_data.line_buffer = malloc (per_line);
      if (!data->in_debug_data.line_buffer)
        MEMFAILED ();
    }
  else
    {
      data->in_debug_data.line_buffer = NULL;
    }

  data->in_base = base_ins;
  data->in_debug_data.pfix = pfix;
  data->in_debug_data.per_line = per_line;
  data->in_debug_data.line_buffer_used = 0;
  data->in_debug_data.own_base = own_base;
  data->in_debug_data.offset = 0;

  return open_in (&my_in_type, data);
}
