/* $Id: commandfilterio.c 658 2006-05-13 14:50:30Z jim $
   teebu - An archiving tool
   Copyright (C) 2006 Jim Farrand

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along with
   this program; if not, write to the Free Software Foundation, Inc., 51
   Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

/* Input and output channels which filter data through an external command.
 * Useful for implementing compression, encryption and checksums.
 * This module implements the nonstdio interface, documented in nonstdio.h
 */

#include <assert.h>
#include <stdio.h>

#include "logging.h"
#include "deque.h"
#include "commandfilterio.h"
#include "commandprocess.h"

struct commandfilterio_out_data
{
  comproc_t out_comproc;
  out_stream_t out_base;
  bool out_own_base;
};

static output_err_t
my_output_limited (void *uncast_data, iobuffer_t * buf, size_t amount)
{
  struct commandfilterio_out_data *data =
    (struct commandfilterio_out_data *) uncast_data;

  if (!data)
    return OUTPUT_ERR_BAD;

  char *output_data = iobuffer_data_pointer (buf);

  DEBUG ("Sending to comproc");
  // DEBUGF("Sending to comproc: 0x%x", output_data[amount-1]) ;
  if (!send_to_comproc (data->out_comproc, output_data, amount))
    {
      DEBUG ("Sending failed");
      return OUTPUT_ERR_FAILED;
    }

  DEBUG ("Send done");
  iobuffer_mark_taken (buf, amount);
  return OUTPUT_OK;
}

static output_err_t
my_close_out (void *uncast_data)
{
  struct commandfilterio_out_data *data =
    (struct commandfilterio_out_data *) uncast_data;
  if (!data)
    return OUTPUT_ERR_BAD;

  finish_comproc_input (data->out_comproc);

  return close_out (data->out_base);
}

void
out_consumer (void *uncast_data, const char *buf, size_t len)
{
  assert (uncast_data);
  struct commandfilterio_out_data *data =
    (struct commandfilterio_out_data *) uncast_data;

  iobuffer_t iobuf;
  init_iobuffer_with (&iobuf, len, len, (char *) buf);
  output_all (data->out_base, &iobuf);
}

static output_err_t
my_output_mark (void *uncast_data)
{
  struct commandfilterio_out_data *data =
    (struct commandfilterio_out_data *) uncast_data;
  if (!data)
    return OUTPUT_ERR_BAD;

  finish_comproc_input (data->out_comproc);
  output_err_t err = output_mark (data->out_base);
  if (OUTPUT_OK != err)
    return err;

  /* if(! restart_comproc(data->out_comproc))
     return OUTPUT_ERR_FAILED ; */

  return OUTPUT_OK;
}

static void
my_release_out (void *uncast_data)
{
  struct commandfilterio_out_data *data =
    (struct commandfilterio_out_data *) uncast_data;
  if (!data)
    return;

  release_comproc (data->out_comproc);
  if (data->out_own_base)
    release_out (data->out_base);

  free (data);
}

static out_stream_type_t my_out_type = {
  .output_limited = my_output_limited,
  .output_mark = my_output_mark,
  .flush = NULL,                // Unimplemented
  .close_out = my_close_out,
  .release_out = my_release_out
};

out_stream_t
commandfilter_open_out (const char *cmd,
                        size_t bufsize,
                        char *const argv[],
                        char *const envp[],
                        bool (*arg_cons) (void *, args_t argv, args_t envp),
                        void *arg_cons_data,
                        out_stream_t base_outs, bool own_base)
{

  assert (cmd);
  assert (base_outs);

  struct commandfilterio_out_data *data =
    malloc (sizeof (struct commandfilterio_out_data));
  if (!data)
    {
      return NULL;              // out of mem!
    }
  data->out_base = base_outs;
  data->out_own_base = own_base;

  data->out_comproc =
    create_comproc (out_consumer, data, bufsize, cmd, argv, envp, arg_cons,
                    arg_cons_data);
  if (!data->out_comproc)
    {
      free (data);
      return NULL;              // out of mem!
    }

  return open_out (&my_out_type, data);
}

struct commandfilterio_in_data
{
  comproc_t in_comproc;
  in_stream_t in_base;
  bool in_own_base;
  input_err_t in_base_last;
  deque_t in_pending_data;
  iobuffer_t in_iobuffer;
};

static input_err_t
refill_in_buffer (struct commandfilterio_in_data *data)
{
  assert (is_deque_empty (data->in_pending_data));

  do
    {
      // Buffer is empty, and last read returned an error
      if (INPUT_OK != data->in_base_last)
        return data->in_base_last;

      // Read data from the underlying stream
      data->in_base_last = input (data->in_base, &data->in_iobuffer);

      if (INPUT_OK == data->in_base_last)
        {
          // Feed it into the comproc
          char *char_data = iobuffer_data_pointer (&data->in_iobuffer);
          size_t data_len = iobuffer_data_size (&data->in_iobuffer);
          if (!send_to_comproc (data->in_comproc, char_data, data_len))
            {
              data->in_base_last = INPUT_ERR_DATALOSS;
              return INPUT_ERR_DATALOSS;
            }
          iobuffer_mark_taken (&data->in_iobuffer, data_len);
        }
      else if (INPUT_EOF == data->in_base_last
               || INPUT_MARK == data->in_base_last)
        {
          finish_comproc_input (data->in_comproc);
        }
    }
  while (is_deque_empty (data->in_pending_data));

  return INPUT_OK;
}

static input_err_t
my_input_limited (void *uncast_data, iobuffer_t * buf, size_t amount)
{
  struct commandfilterio_in_data *data =
    (struct commandfilterio_in_data *) uncast_data;
  if (!data)
    return INPUT_ERR_BAD;

  if (is_deque_empty (data->in_pending_data))
    {
      input_err_t err = refill_in_buffer (data);
      if (INPUT_OK != err)
        return err;
    }

  size_t n =
    remove_from_deque_start (data->in_pending_data,
                             iobuffer_free_pointer (buf), amount);
  assert (n > 0);
  iobuffer_mark_added (buf, n);
  return INPUT_OK;
}

static input_err_t
my_input_mark (void *uncast_data)
{
  struct commandfilterio_in_data *data =
    (struct commandfilterio_in_data *) uncast_data;
  if (!data)
    return INPUT_ERR_BAD;

  input_err_t err = input_mark (data->in_base);
  if (INPUT_OK != err)
    return err;

  /* if(! restart_comproc(data->in_comproc))
     return INPUT_ERR_FAILED ; */

  data->in_base_last = INPUT_OK;

  return INPUT_OK;
}

static input_err_t
my_input_recover (void *uncast_data)
{
  struct commandfilterio_in_data *data =
    (struct commandfilterio_in_data *) uncast_data;
  if (!data)
    return INPUT_ERR_BAD;

  DEBUG ("Error recovery");

  finish_comproc_input (data->in_comproc);

  data->in_base_last = input_recover (data->in_base);
  if (INPUT_OK != data->in_base_last)
    return data->in_base_last;

  // TODO: If the underlying stream is also a commandfilter:
  // It has lined up ready to read the next chunk
  // We are going to dump up to the next chunk
  // Meaning that we skip an chunk uncesessarily
  // Ideally, we would like recover on commandfilterio to line up the stream
  // to just BEFORE the mark, which would solve this problem.
  // Implementing that would be easier if we lazily start the subprocess.
  // Which is probably a good idea anyway

  LOG (WARNING, "Dumping data until end of chunk");
  iobuffer_t iobuf;
  char buf_data;
  do
    {
      init_iobuffer_with (&iobuf, 1, 0, &buf_data);
      data->in_base_last = input (data->in_base, &iobuf);
    }
  while (INPUT_OK == data->in_base_last);

  if (INPUT_MARK != data->in_base_last)
    {
      if (INPUT_EOF == data->in_base_last)
        {
          LOG (WARNING, "No more chunks");
        }
      else
        {
          LOGF (WARNING, "Couldn't find end of chunk: %s",
                INPUT_ERR_NAME (data->in_base_last));
        }

      return data->in_base_last;
    }

  LOG (VERBOSE, "Found end of chunk");

  // Pending data is probably now garbage
  iobuffer_mark_all_taken (&data->in_iobuffer);
  clear_deque (data->in_pending_data);

  return INPUT_OK;
}

static input_err_t
my_close_in (void *uncast_data)
{
  struct commandfilterio_in_data *data =
    (struct commandfilterio_in_data *) uncast_data;
  if (!data)
    return INPUT_ERR_BAD;

  if (INPUT_EOF != data->in_base_last)
    finish_comproc_input (data->in_comproc);

  return close_in (data->in_base);
}

static void
my_release_in (void *uncast_data)
{
  assert (uncast_data);
  struct commandfilterio_in_data *data =
    (struct commandfilterio_in_data *) uncast_data;

  release_comproc (data->in_comproc);
  release_deque (data->in_pending_data);

  if (data->in_own_base)
    release_in (data->in_base);

  free (iobuffer_data_block (&data->in_iobuffer));
  free (data);
}

void
in_consumer (void *uncast_data, const char *buf, size_t len)
{
  assert (uncast_data);
  struct commandfilterio_in_data *data =
    (struct commandfilterio_in_data *) uncast_data;

  add_to_deque_end (data->in_pending_data, buf, len);
}

static in_stream_type_t my_in_type = {
  .input_limited = my_input_limited,
  .input_mark = my_input_mark,
  .close_in = my_close_in,
  .release_in = my_release_in,
  .input_recover = my_input_recover
};

in_stream_t
commandfilter_open_in (const char *cmd,
                       size_t bufsize,
                       char *const argv[],
                       char *const envp[],
                       bool (*arg_cons) (void *, args_t argv, args_t envp),
                       void *arg_cons_data,
                       in_stream_t base_ins, bool own_base)
{
  assert (cmd);
  assert (base_ins);

  struct commandfilterio_in_data *data =
    malloc (sizeof (struct commandfilterio_in_data));
  if (!data)
    return NULL;                // out of mem!

  data->in_pending_data = create_deque (sizeof (char), bufsize);
  if (!data->in_pending_data)
    {
      free (data);
      return NULL;              // out of mem!
    }

  char *iobuffer_data = malloc (sizeof (char) * bufsize);
  if (!iobuffer_data)
    {
      release_deque (data->in_pending_data);
      free (data);
      return NULL;              // out of mem!
    }

  init_iobuffer_with (&data->in_iobuffer, bufsize, 0, iobuffer_data);

  data->in_comproc =
    create_comproc (in_consumer, data, bufsize, cmd, argv, envp, arg_cons,
                    arg_cons_data);
  if (!data->in_comproc)
    {
      free (iobuffer_data);
      release_deque (data->in_pending_data);
      free (data);
      return NULL;              // coudn't start process!
    }

  data->in_base = base_ins;
  data->in_own_base = own_base;
  data->in_base_last = INPUT_OK;

  return open_in (&my_in_type, data);
}
