/* $Id: baseio.c 658 2006-05-13 14:50:30Z jim $
   teebu - An archiving tool
   Copyright (C) 2006 Jim Farrand

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along with
   this program; if not, write to the Free Software Foundation, Inc., 51
   Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


#include <stdio.h>
#include <assert.h>

#include "baseio.h"

static output_err_t
baseio_output_limited (void *data, iobuffer_t * buf, size_t amount)
{
  if (!data)
    return OUTPUT_ERR_BAD;

  const int written =
    fwrite (iobuffer_data_pointer (buf), 1, amount, (FILE *) data);

  if (0 == written)
    return OUTPUT_ERR_FAILED;

  iobuffer_mark_taken (buf, written);

  return OUTPUT_OK;
}

static output_err_t
baseio_close_out (void *data)
{
  if (!data)
    return OUTPUT_ERR_BAD;

  if (0 != fclose ((FILE *) data))
    return OUTPUT_ERR_FAILED;

  return OUTPUT_OK;
}

/* Collect implementation of baseio. */
static out_stream_type_t baseio_out_type = {
  .output_limited = baseio_output_limited,
  .output_mark = NULL,          // Unsupported
  .close_out = baseio_close_out,
  .release_out = NULL           // Uneeded
};

static out_stream_t
baseio_open_out_from_file (FILE * file)
{
  return open_out (&baseio_out_type, file);
}

out_stream_t
baseio_open_out (const char *filename, const char *mode)
{
  FILE *file = fopen (filename, mode);
  if (!file)
    return NULL;

  return baseio_open_out_from_file (file);
}

static input_err_t
baseio_input_limited (void *data, iobuffer_t * buf, size_t amount)
{
  assert (buf);

  if (!data)
    return INPUT_ERR_BAD;

  const size_t free = iobuffer_free_size (buf);
  if (free < amount)
    amount = free;

  if (0 == amount)
    return INPUT_ERR_FULL;

  // printf("input len = %d\n", len) ;
  const size_t read =
    fread (iobuffer_free_pointer (buf), 1, amount, (FILE *) data);
  // printf("input read = %d\n", read) ;

  if (0 == read)
    {
      if (feof ((FILE *) data))
        {
          return INPUT_EOF;
        }
      else
        {
          return INPUT_ERR_FAILED;
        }
    }

  iobuffer_mark_added (buf, read);

  return INPUT_OK;
}

static input_err_t
baseio_close_in (void *data)
{
  if (!data)
    return INPUT_ERR_BAD;

  if (0 != fclose ((FILE *) data))
    return INPUT_ERR_FAILED;

  return INPUT_OK;
}

static in_stream_type_t baseio_in_type = {
  .input_limited = baseio_input_limited,
  .input_mark = NULL,           // Unsupported
  .close_in = baseio_close_in,
  .release_in = NULL            // Undeeded
};

static in_stream_t
baseio_open_in_from_file (FILE * file)
{
  return open_in (&baseio_in_type, file);
}

in_stream_t
baseio_open_in (const char *filename, const char *mode)
{
  FILE *file = fopen (filename, mode);
  if (!file)
    return NULL;

  return baseio_open_in_from_file (file);
}

static in_stream_t stdin_holder = NULL;

in_stream_t
baseio_stdin ()
{
  if (NULL == stdin_holder)
    stdin_holder = baseio_open_in_from_file (stdin);

  return stdin_holder;
}
