/* $Id: check_hashtable.c 718 2006-06-01 17:19:53Z jim $
   teebu - An archiving tool
   Copyright (C) 2006 Jim Farrand

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along with
   this program; if not, write to the Free Software Foundation, Inc., 51
   Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/


#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "hashtable.h"
#include "check_hashtable.h"

START_TEST (check_single_int_insert)
{
  hashtable_t ht = create_int_hashtable (1);
  fail_if (!ht);

  int key = 0;
  char *data = "foo";
  fail_unless (hashtable_add (ht, &key, data));
  fail_unless (data == hashtable_find (ht, &key));
  hashtable_release (ht);
}
END_TEST

START_TEST (check_single_string_insert)
{
  hashtable_t ht = create_string_hashtable (1, true);
  fail_if (!ht);

  char *test_key = "the answer";
  int test_data = 42;
  fail_unless (hashtable_add (ht, test_key, &test_data));
  fail_unless (test_data == *(int *)hashtable_find (ht, test_key));
  hashtable_release (ht);
}
END_TEST

START_TEST (check_many_int_insert)
{
  size_t inserts = 128*1024;

  // Create some random data
  int *int_data = malloc (inserts * sizeof(int));
  fail_if(!int_data);
  for (int i = 0; i < inserts; i++)
    int_data[i]=i;

  hashtable_t ht = create_int_hashtable(1);

  // Insert each data item with a different key
  for (int i = 0; i < inserts; i++)
      fail_unless (hashtable_add (ht, &i, &int_data[i]));

  // Retrieve each key and check we get the right data
  for (int i = 0; i < inserts; i++)
    {
      fail_unless (int_data[i] == *(int *)hashtable_find (ht, &i));
    }

  hashtable_release (ht);
}
END_TEST

START_TEST (check_many_int_insert_remove)
{
  const size_t inserts = 100*1024;
  const size_t keys = inserts / 1024;

  // Some small number of data items
  const unsigned data_count = 64;
  int data_items[data_count];
  for (int i = 0; i < data_count; i++)
    data_items[i] = i;

  // An array which records whether each key has been used
  bool *key_used = malloc (keys * sizeof (bool));
  fail_if(!key_used);
  for (int i = 0; i < keys; i++)
    key_used[i] = false;

  // An array which records which data item each key is mapped to
  // if the appropriate flag is set in key_used
  unsigned *key_data_item = malloc (keys * sizeof (unsigned));

  hashtable_t ht = create_int_hashtable(1);

  // Pick a random key
  // If that key is mapped, remove the mapping and check we get the right data
  // Otherwise, pick a random data item and map the key to that item
  for (int i = 0; i < inserts; i++)
    {
      int key_no = random () % keys;
      if (key_used[key_no])
        {

          fail_unless (data_items[key_data_item[key_no]] ==
                       *(int *)hashtable_remove(ht, &key_no));
          // printf ("Correct data removed: %d\n", data_items[key_data_item[key_no]]);
          key_used[key_no] = false;
        }
      else
        {
          key_data_item[key_no] = random () % data_count;
          fail_unless ( hashtable_add(ht, &key_no,
                                      &data_items[key_data_item[key_no]]));
          key_used[key_no] = true;
        }
    }

  hashtable_release (ht);
}
END_TEST
START_TEST (check_many_string_insert)
{
  size_t inserts = 128*1024;

  // Create some random data
  int *int_data = malloc (inserts * sizeof(int));
  fail_if(!int_data);
  for (int i = 0; i < inserts; i++)
    int_data[i]=i;

  hashtable_t ht = create_string_hashtable(1, true);

  // Insert each data item with a different key
  for (int i = 0; i < inserts; i++)
    {
      char buffer[128];
      snprintf (buffer, 128, "Key: %d", i);
      fail_unless (hashtable_add (ht, buffer, &int_data[i]));
    }

  // Retrieve each key and check we get the right data
  for (int i = 0; i < inserts; i++)
    {
      char buffer[128];
      snprintf (buffer, 128, "Key: %d", i);
      fail_unless (int_data[i] == *(int *)hashtable_find (ht, buffer));
    }

  hashtable_release (ht);
}
END_TEST

START_TEST (check_many_string_insert_remove)
{
  const size_t inserts = 100*1024;
  const size_t keys = inserts / 1024;

  // Some small number of data items
  const unsigned data_count = 64;
  int data_items[data_count];
  for (int i = 0; i < data_count; i++)
    data_items[i] = i;

  // An array which records whether each key has been used
  bool *key_used = malloc (keys * sizeof (bool));
  fail_if(!key_used);
  for (int i = 0; i < keys; i++)
    key_used[i] = false;

  // An array which records which data item each key is mapped to
  // if the appropriate flag is set in key_used
  unsigned *key_data_item = malloc (keys * sizeof (unsigned));

  hashtable_t ht = create_string_hashtable(1, true);

  // Pick a random key
  // If that key is mapped, remove the mapping and check we get the right data
  // Otherwise, pick a random data item and map the key to that item
  for (int i = 0; i < inserts; i++)
    {
      int key_no = random () % keys;
      char key_buffer[128];
      snprintf (key_buffer, 128, "Key: %d", key_no);
      // printf ("iteration %d\n", i);
      // if(44 == i)
        // printf("foo\n");
      if (key_used[key_no])
        {

          fail_unless (data_items[key_data_item[key_no]] ==
                       *(int *)hashtable_remove(ht, key_buffer));
          // printf ("Correct data removed: %d\n", data_items[key_data_item[key_no]]);
          key_used[key_no] = false;
        }
      else
        {
          key_data_item[key_no] = random () % data_count;
          fail_unless ( hashtable_add(ht, key_buffer,
                                      &data_items[key_data_item[key_no]]));
          key_used[key_no] = true;
        }
    }

  hashtable_release (ht);
}
END_TEST

void
add_hashtable_tests (Suite * s)
{
  TCase *tc_core = tcase_create ("Hashtable");
  tcase_set_timeout (tc_core, 10);
  tcase_add_test (tc_core, check_single_int_insert);
  tcase_add_test (tc_core, check_many_int_insert);
  tcase_add_test (tc_core, check_many_int_insert_remove);
  tcase_add_test (tc_core, check_single_string_insert);
  tcase_add_test (tc_core, check_many_string_insert);
  tcase_add_test (tc_core, check_many_string_insert_remove);
  suite_add_tcase (s, tc_core);
}
