/* $Id: simulateerrorio.c 658 2006-05-13 14:50:30Z jim $
   teebu - An archiving tool
   Copyright (C) 2006 Jim Farrand

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along with
   this program; if not, write to the Free Software Foundation, Inc., 51
   Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include <assert.h>
#include <stdio.h>

#include "logging.h"
#include "simulateerrorio.h"

bool
parse_failure_mode (const char *desc, failure_mode_t * fm)
{
  // TODO: How to I input off_t portably?
  unsigned long int position;
  char report;
  int matched = sscanf (desc, "%lu:%c:%zu:%zu",
                        &position, &report, &fm->failure_bytes_lost,
                        &fm->failure_bytes_garbage);
  if (4 != matched)
    {
      LOG (ERROR,
           "failure mode must have format \"pos:rep:lost:garb\" eg \"128:t:8:4\"");
      return false;
    }

  fm->failure_position = position;
  switch (report)
    {
    case 't':
      fm->failure_report = true;
      break;
    case 'f':
      fm->failure_report = false;
      break;
    default:
      LOGF (ERROR, "Field 2 of failure mode must be 't' or 'f', but is: '%c'",
            report);
      return false;
    }

  return true;
}

struct errio_in_data
{
  failure_mode_t in_fm;
  in_stream_t in_base;
  bool in_own_base;
};

static input_err_t
my_input_limited (void *uncast_data, iobuffer_t * buf, size_t max)
{
  struct errio_in_data *data = uncast_data;
  if (!data)
    return INPUT_ERR_BAD;

  if (0 == data->in_fm.failure_position)
    {
      if (data->in_fm.failure_report)
        {
          LOG (ERROR, "Simulating error");
          return INPUT_ERR_DATALOSS;
        }

      if (data->in_fm.failure_bytes_lost > 0)
        {
          LOGF (ERROR, "Simulating data loss (%zu bytes)",
                data->in_fm.failure_bytes_lost);
          input_err_t err =
            skip_in (data->in_base, data->in_fm.failure_bytes_lost);
          if (INPUT_OK != err)
            return err;
          data->in_fm.failure_bytes_lost = 0;
        }

      if (data->in_fm.failure_bytes_garbage > 0)
        {
          if (data->in_fm.failure_bytes_garbage < max)
            max = data->in_fm.failure_bytes_garbage;

          LOGF (ERROR, "Simulating garbage (%zu bytes of %zu)", max,
                data->in_fm.failure_bytes_garbage);
          char *dst = iobuffer_free_pointer (buf);
          for (int i = 0; i < max; i++)
            dst[i] = (char) rand ();
          iobuffer_mark_added (buf, max);
          data->in_fm.failure_bytes_garbage -= max;
          return INPUT_OK;
        }
    }

  if (data->in_fm.failure_position > 0 && data->in_fm.failure_position < max)
    max = data->in_fm.failure_position;

  size_t before = iobuffer_data_size (buf);
  input_err_t err = input_limited (data->in_base, buf, max);
  data->in_fm.failure_position -= iobuffer_data_size (buf) - before;
  return err;
}

static input_err_t
my_close_in (void *uncast_data)
{
  struct errio_in_data *data = uncast_data;
  if (!data)
    return INPUT_ERR_BAD;

  return close_in (data->in_base);
}

static void
my_release_in (void *uncast_data)
{
  struct errio_in_data *data = uncast_data;
  assert (data);

  if (data->in_own_base)
    release_in (data->in_base);

  free (data);
}

static input_err_t
my_input_recover (void *uncast_data)
{
  struct errio_in_data *data = uncast_data;
  if (!data)
    return INPUT_ERR_BAD;

  if (0 == data->in_fm.failure_position && data->in_fm.failure_report)
    {
      LOG (WARNING, "Recovering from simulated error");
      data->in_fm.failure_report = false;
    }

  return INPUT_OK;
}

static in_stream_type_t my_in_type = {
  .input_limited = my_input_limited,
  .input_mark = NULL,           // Unimplemented
  .close_in = my_close_in,
  .release_in = my_release_in,
  .input_recover = my_input_recover
};

in_stream_t
simulateerrorio_open_in (failure_mode_t * fm, in_stream_t base, bool own_base)
{
  assert (fm);
  assert (base);

  struct errio_in_data *data = malloc (sizeof (struct errio_in_data));
  if (!data)
    return NULL;

  data->in_fm = *fm;
  data->in_base = base;
  data->in_own_base = own_base;

  return open_in (&my_in_type, data);
}
