/* $Id: remarkio.c 658 2006-05-13 14:50:30Z jim $
   teebu - An archiving tool
   Copyright (C) 2006 Jim Farrand

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along with
   this program; if not, write to the Free Software Foundation, Inc., 51
   Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


#include <assert.h>
#include "logging.h"
#include "remarkio.h"

struct remarkio_out_data
{
  out_stream_t out_base;
  bool out_own_base;
  off_t out_chunk_size;
  off_t out_chunk_count;
};

static output_err_t
my_output_limited (void *uncast_data, iobuffer_t * buf, size_t amount)
{
  struct remarkio_out_data *data = uncast_data;
  if (!data)
    return OUTPUT_ERR_BAD;

  size_t before_amount = iobuffer_data_size (buf);
  output_err_t err = output_limited (data->out_base, buf, amount);
  size_t after_amount = iobuffer_data_size (buf);
  data->out_chunk_count += before_amount - after_amount;

  return err;
}

static output_err_t
my_output_mark (void *uncast_data)
{
  struct remarkio_out_data *data = uncast_data;
  if (!data)
    return OUTPUT_ERR_BAD;

  if (data->out_chunk_count <= data->out_chunk_size)
    return OUTPUT_OK;

  output_err_t err = output_mark (data->out_base);
  if (OUTPUT_OK != err)
    return err;

  if (0 == data->out_chunk_size)
    {
      data->out_chunk_count = 0;
    }
  else
    {
      data->out_chunk_count %= data->out_chunk_size;

      if (data->out_chunk_count > (data->out_chunk_size / 2))
        {
          // Too soon to next chunk, so we'll write an extra big chunk, instead of an extra small one
          data->out_chunk_count -= data->out_chunk_size;
        }
    }

  return err;
}


static output_err_t
my_flush (void *uncast_data)
{
  struct remarkio_out_data *data = uncast_data;
  if (!data)
    return OUTPUT_ERR_BAD;

  return flush (data->out_base);
}

static output_err_t
my_close_out (void *uncast_data)
{
  struct remarkio_out_data *data = uncast_data;
  assert (data);

  if (!data)
    return OUTPUT_ERR_BAD;

  return close_out (data->out_base);
}

static void
my_release_out (void *uncast_data)
{
  struct remarkio_out_data *data = uncast_data;
  assert (data);

  if (data->out_own_base)
    release_out (data->out_base);

  free (data);
}

static out_stream_type_t my_out_type = {
  .output_limited = my_output_limited,
  .output_mark = my_output_mark,
  .flush = my_flush,
  .close_out = my_close_out,
  .release_out = my_release_out
};

out_stream_t
remarkio_open_out (out_stream_t out_base, off_t min_dist, bool own_base)
{
  struct remarkio_out_data *data = malloc (sizeof (struct remarkio_out_data));
  if (!data)
    MEMFAILED ();

  data->out_base = out_base;
  data->out_own_base = own_base;
  data->out_chunk_size = min_dist;
  data->out_chunk_count = 0;

  return open_out (&my_out_type, data);
}
