/* $Id: path.c 695 2006-05-20 01:30:20Z jim $
   teebu - An archiving tool
   Copyright (C) 2006 Jim Farrand

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along with
   this program; if not, write to the Free Software Foundation, Inc., 51
   Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include "logging.h"
#include "size.h"
#include "path.h"

#define DEFAULT_PATH_BUFFER_SIZE 256

struct path
{
  size_t buffer_size;
  size_t path_size;
  char *path_buffer;
};

path_t
empty_path (void)
{
  path_t path = malloc (sizeof (struct path));
  if (!path)
    MEMFAILED ();

  path->buffer_size = DEFAULT_PATH_BUFFER_SIZE;
  path->path_buffer = malloc (path->buffer_size);
  if (!path->path_buffer)
    MEMFAILED ();
  path->path_size = 0;

  return path;
}

path_t
root_path (void)
{
  path_t path = empty_path ();

  path->path_buffer[0] = '/';
  path->path_buffer[1] = '\0';
  path->path_size = 1;

  return path;
}

path_t
current_dir (void)
{
  path_t path = empty_path ();

  path->path_buffer[0] = '.';
  path->path_buffer[1] = '\0';
  path->path_size = 1;

  return path;
}

const char *
path_str (path_t path)
{
  return path->path_buffer;
}

void
release_path (path_t path)
{
  assert (path);
  free (path->path_buffer);
  free (path);
}

static void
ensure_size (path_t path, size_t new_path_size)
{
  if (new_path_size >= path->buffer_size)
    {
      path->buffer_size *= 2;
      if (new_path_size >= path->buffer_size)
        path->buffer_size = new_path_size + 1;

      if (!
          (path->path_buffer =
           realloc (path->path_buffer, path->buffer_size)))
        MEMFAILED ();
    }
}

static void
path_add_part (path_t path, const char *part, size_t start, size_t len)
{
  // Add a trailing slash if necessary
  if (path->path_size > 0 && '/' != path->path_buffer[path->path_size - 1])
    {
      path->path_buffer[path->path_size] = '/';
      ensure_size (path, ++(path->path_size));
      path->path_buffer[path->path_size] = '\0';
    }

  ensure_size (path, path->path_size + len);
  strncpy (path->path_buffer + path->path_size, part + start, len);
  path->path_size += len;
  path->path_buffer[path->path_size] = '\0';
}

void
path_add (path_t path, const char *part)
{
  assert (path);
  assert (part);

  if ('/' == part[0])
    {
      // Path we are  adding starts with a slash, so we are going to replace
      // current path entirely
      path->path_buffer[0] = '/';
      path->path_buffer[1] = '\0';
      path->path_size = 1;
    }

  size_t start = 0;
  while ('\0' != part[start])
    {
      // Skip any leading slashes
      if ('/' == part[start])
        {
          start++;
          continue;
        }

      size_t len = 1;
      while ('/' != part[start + len] && '\0' != part[start + len])
        len++;

      path_add_part (path, part, start, len);

      start += len;
    }
}

path_t
make_path (const char *part)
{
  path_t path = empty_path ();
  path_add (path, part);
  return path;
}

path_t
copy_path (const path_t path)
{
  return make_path (path_str (path));
}

void
remove_last_path_part (path_t path)
{
  size_t path_len;
  for (path_len = path->path_size; path_len > 1; path_len--)    // >1 prevents us consuming first char
    if ('/' == path->path_buffer[path_len])
      break;

  path->path_buffer[path_len] = '\0';
  path->path_size = path_len;
}

void
set_path_length (path_t path, size_t size)
{
  assert (size <= path->path_size);
  path->path_size = size;
  path->path_buffer[size] = '\0';
}
